//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS STORE sector read/write and translation services, public interface
//
// Author: J. van Wijk
//
// Implementation is in dfstore.c  for generic stuff (platform neutral)
//               and in dfsector.c for I/O functions (platform specific)
//
// JvW  27-12-96   Initial version, loosely derived from QDISK and DHPFSERV
//
#ifndef    DFSTORE_H
   #define DFSTORE_H


typedef enum   dfstoretype                      // type of DFS STOREs
{                                               // system-oriented
   DFST_UNUSED,                                 // store not in use
   DFST_RAWIMAGE,                               // RAW image file, normal file
   DFST_IMZIMAGE,                               // IMZ image file, compressed file
   DFST_VOLUME,                                 // logical volume
   DFST_PHDISK,                                 // physical disk
   DFST_MEMORY,                                 // memory-disk DFS store
   DFST_PMIRAW,                                 // partitioned RAW image
   DFST_PMIMZD,                                 // partitioned IMZ disk image
   DFST_PMVBOX,                                 // partitioned VirtualBox image
   DFST_INVALID  = 0xff                         // invalid store (number)
} DFSTORETYPE;                                  // end of enum "dfstoretype"


typedef enum   dfsobject                        // type of dfsee object
{                                               // user-oriented
   DFSO_NONE   = 0,                             // no object or unknown
   DFSO_DISK   = 1,                             // partitioned medium, disk
   DFSO_PART   = 2,                             // partition on a disk
   DFSO_VOLD   = 3,                             // Volume or Device
   DFSO_IMGF   = 4,                             // Image File
   DFSO_FRSP   = 5,                             // Freespace area
   DFSO_BASE   = 6                              // Any object,  base != 0
} DFSOBJECT;                                    // end of enum "dfsobject"


#define DFSTFL_EXPLICITSIZE     0x00000001      // file (img) may NOT grow

typedef struct dfsistore                        // store-external info
{
   DFSOBJECT           object;                  // Object type
   USHORT              disknr;                  // DFSee disk id
   USHORT              partid;                  // partition-id
   TXTM                drive;                   // drive letter or filename
   TXTS                afsys;                   // FS filesystem name (mode)
   DFSPARTINFO        *p;                       // Partition info structure
   ULONG               noAccess;                // RC of reading 1st sector
   ULONG               stFlags;                 // store flags
} DFSISTORE;                                    // end of struct "dfsistore"


#define DFS_MAX_PSN         DFS64MAX            // invalid PSN, 64bit
#define DFS_BADSEC_PATTERN  0xFE                // byte pattern bad-sectors
#define DFS_BADSEC_CRC32    0x8c347018          // CRC32 for such a sector

//- DFS locking modes
#define DFS_LOCK          0
#define DFS_UNLOCK        1


#if defined (DOS32)                             // std Int 13 definitions
   #define DFS_I13_S_READ    0x02               // INT13 standard read
   #define DFS_I13_S_WRITE   0x03               // INT13 standard write
   #define DFS_I13_S_PARAM   0x08               // INT13 standard get params

   #define DFS_I13           0x13               // INT13 Dos interrupt
   #define DFS_I21           0x21               // DOS21 Dos interrupt
#endif

typedef ULONG DFST_HANDLE;                      // store handle

#define DFST_SYSTEM   0                         // reserved system store
#define DFST_ST_ONE   1                         // first  'user' store
#define DFST_ST_TWO   2                         // second 'user' store
#define DFSTORE       dfstGetDefaultStore()
#define DFSTALT     ((dfstGetDefaultStore() != DFST_ST_TWO) ? DFST_ST_TWO : DFST_ST_ONE)

typedef struct dfs_geo                          // basic DFS geometry structure
{
   ULONG  C;                                    // Cylinders
   ULONG  H;                                    // Heads
   ULONG  S;                                    // Sectors per track
   USHORT B;                                    // Bytes per sector
} DFS_GEO;                                      // end of struct "dfs_geo"


typedef struct dfsgeodisk                       // disk level geometry info
{
   DFS_GEO             Calculated;              // calculated disk geometry
   DFS_GEO             Forced;                  // forced geo, user/calculated
   TXTT                Reason;                  // Force reason, "" = calculated
} DFSGEODISK;                                   // end of struct "dfsgeodisk"

#define DFS_STD_CYLS     1024L                  // standard cylinder count
#define DFS_STD_HEAD     255L                   // standard head count
#define DFS_STD_SECT     63L                    // standard sector count (OS/2 up to 500Gb)

#define DFS_1MIB_MASK    0x7ff                  // Mask to get PSN modulo 1-MiB cylinder size
#define DFS_1MIB_ALIGNED 0x000                  // Masked value for aligned partition start/size
#define DFS_1MIB_START   0x800                  // Start of 1MIB aligned primary, and cyl size
#define DFS_1MIB_HEAD    64L                    // head count   1-MiB cyl (Win, SSD, 4Kb etc)
#define DFS_1MIB_SECT    32L                    // sector count 1-MiB cyl (Win, SSD, 4Kb etc)

#define DFS_1TB_SECT     127L                   // sector count OS/2 up to 1TB disks
#define DFS_2TB_SECT     255L                   // sector count OS/2 up to 2TB disks

#define DFS_MAX_SECT     255L                   // max sect/track (DANIS506, LVM)

#define dfstCylinderSize(s)  (dfstGeoSectors(s) * dfstGeoHeads(s))
#define dfsCylinderSize()     dfstCylinderSize(DFSTORE)
#define dfsRawDiskSize()      dfstRawDiskSize(DFSTORE)
#define dfsGetSectorSize()    dfstGetSectorSize(DFSTORE)
#define dfsGetClusterSize()   dfstGetClusterSize(DFSTORE)
#define dfsGetLogicalSize()   dfstGetLogicalSize(DFSTORE)
#define dfsRead(lsn,nrs,buf)  dfstReadLsn( DFSTORE,(lsn),(nrs),buf)
#define dfsWrite(lsn,nrs,buf) dfstWriteLsn(DFSTORE,(lsn),(nrs),buf)
#define DFSTORE_WRITE_ALLOWED dfstCheckWriteAccess(DFSTORE, FALSE)

// Get the number for the current DEFAULT store (used in most DFSee commands)
DFST_HANDLE dfstGetDefaultStore
(
   void
);

// Set the number for the current DEFAULT store (used in most DFSee commands)
ULONG dfstSetDefaultStore
(
   DFST_HANDLE         store                    // IN    new default store
);

// Reset the DEFAULT store to previously saved one (must be initialized!)
ULONG dfstRestoreDefaultStore
(
   DFST_HANDLE         store                    // IN    previously saved store
);                                              //       must be initialized!

// Issue DFSee message and set the readonly attribute for all stores
ULONG dfstSwitch2ReadOnly
(
   char               *message,                 // IN    cause of switch or NULL
   BOOL                verbose,                 // IN    give error/warnings
   BOOL                readonly                 // IN    readonly yes/no
);

// Set the readonly attribute for the specified store
ULONG dfstSetReadOnly
(
   DFST_HANDLE         store,                   // IN    store to set RO
   BOOL                verbose,                 // IN    give error/warnings
   BOOL                readonly                 // IN    readonly yes/no
);

// Get readonly status for specified store
BOOL dfstGetReadOnly
(
   DFST_HANDLE         store,                   // IN    store
   char               *descr                    // OUT   descr string or NULL
);

// Show status of all stores, and current default
void dfstShowStoreInfo
(
   DFST_HANDLE         first                    // IN    first store to show
);

// Get store-name as exact 15-character short description (no partition!)
char *dfstStoreName                             // RET   exact 15-char name
(
   DFST_HANDLE         store                    // IN    store to query
);

// Get store identification as disk/partition or volume description, length 8
char *dfstStoreDpid                             // RET   exact 8-char name
(
   DFST_HANDLE         store                    // IN    store to query
);

// Query type and optional inf and description of specified store
#define dfstStoreType(s) dfstQueryStoreType((s),NULL,NULL)
DFSTORETYPE dfstQueryStoreType
(
   DFST_HANDLE         store,                   // IN    store to query
   DFSISTORE         **info,                    // OUT   info struct, or NULL
   char               *descr                    // OUT   descr string or NULL
);

// Get complete store description of specified store in a 58 byte string
#define dfstStoreDesc1(s) dfstStoreDescription1((s),NULL)
char *dfstStoreDescription1
(
   DFST_HANDLE         store,                   // IN    store to query
   char               *buf                      // INOUT description or NULL
);                                              //      (uses static buffer)

// Get secondary store-description (like disk or image name) in 38 byte string
#define dfstStoreDesc2(s)     dfstStoreDescription2((s),NULL)
char *dfstStoreDescription2
(
   DFST_HANDLE         store,                   // IN    store to query
   char               *buf                      // INOUT description or NULL
);                                              //      (uses static buffer)

// Initialize STORE administration at startup
void dfstInitStoreInfo
(
   void
);

// Update partinfo ptr for any opened physical-disk stores (after ReadDiskInfo)
void dfstUpdateStoreInfo
(
   BOOL                verbose                  // IN    output info
);

// Open specified volume in direct and if possible, sector-read mode
ULONG dfstOpenVolume
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   char               *drive,                   // IN    Drive specification
   char               *fsys,                    // OUT   File system info
   BOOL                verbose                  // IN    Give descriptive msgs
);

// Open specified Raw or Compressed image file as source for analysis
ULONG dfstOpenImageFile
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   char               *file,                    // IN    File specification
   BOOL                append,                  // IN    Append to file allowed
   BOOL                forceRAW,                // IN    always open RAW (IMZ/VDI)
   BOOL                verbose,                 // IN    Display geometry & size
   ULONG              *fcount                   // OUT   #of files (multi) or NULL
);

// Open a partitionable medium: phys/virt/image using DFSee disk-id range 1..n
ULONG dfstOpenDisk
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   USHORT              disknr,                  // IN    phys/virt disk number
   BOOL                verbose,                 // IN    show disk geo
   BOOL                reset                    // IN    reset disk geo
);


// Get reference to disk level geometry info for specified disk number
DFSGEODISK *dfsGetDiskGeoInfo
(
   USHORT              disk                     // IN    disk number 0..n
);                                              //       0 is 'other' object


// Calculate most likely used disk-geometry based on disk-contents (sniffing)
ULONG dfstCalculateGeometry
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                verbose,                 // IN    show disk geo
   DFS_GEO            *calculated               // OUT   calculated geometry
);


// Display (and force a specific) geometry on the currently open physical disk
// Special value 0 for cyl, head or sector will keep the original value
// Special value L32_NULL for cyl will cause recalculation for Cylinder count
ULONG dfstDiskGeometry
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               base,                    // IN    bootsector offset or 0
   ULONG               cyl,                     // IN    nr of cylinders
   ULONG               hds,                     // IN    nr of heads
   ULONG               spt,                     // IN    nr of sectors/track
   USHORT              bps,                     // IN    bytes per sector  or 0
   BOOL                sync,                    // IN    sync System geo too
   BOOL                verbose                  // IN    show disk geo
);

// Close open store, and re-initialize
ULONG dfstClose
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Translate Physical sector number (PSN) to CHS Cylinder value
ULONG  dfstPsn2Cyl                              // RET   cylinder number
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    physical sector nr
);

// Translate Physical sector number (PSN) to CHS Head value
ULONG  dfstPsn2Head                             // RET   head number
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    physical sector nr
);

// Translate Physical sector number (PSN) to CHS Sector value
ULONG  dfstPsn2Sector                            // RET   sector number
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    physical sector nr
);

// Translate Physical sect-nr (PSN) to Head-Cylinder-Sector, use forced Geo
ULONG dfstPsn2Chs
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    physical sector nr
   ULONG              *cylinder,                // OUT   cylinder number
   ULONG              *head,                    // OUT   head number
   ULONG              *sector                   // OUT   first sector number
);

// Translate Physical sect-nr (PSN) to Head-Cylinder-Sector, use Original Geo
ULONG dfstPsn2Org
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    physical sector nr
   ULONG              *cylinder,                // OUT   Org cylinder number
   ULONG              *head,                    // OUT   Org head number
   ULONG              *sector                   // OUT   Org sector number
);

// Reset ForcedGeometry to be empty for specified disk-device
void dfsResetForcedGeo
(
   USHORT              did                      // IN    disk-id
);

// Translate Cylinder-Head-Sector to Physical sector number (PSN)
ULN64 dfstChs2Psn                               // RET   physical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector                   // IN    first sector number
);

// Set start and final-PSN for Read logical sector(s)
ULN64 dfstLogicalLimits                         // RET   current start
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               start,                   // IN    PSN logical sector 0
   ULN64               final                    // IN    PSN final sector
);

// Set start and final-PSN for FS-specific Area sector(s)
ULN64 dfstSetAreaLimits                         // RET   current area start
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               start,                   // IN    PSN logical sector 0
   ULN64               final                    // IN    PSN final sector
);

// Get size as defined by logical limits
ULN64 dfstGetLogicalSize
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Get exact size in bytes for the store object (useful for images mainly)
ULN64 dfstGetByteSize                           // RET   Exact size in bytes
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Set exact size in bytes for specified store
void dfstSetByteSize                            // RET   Exact size in bytes
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               size                     // IN    size in bytes
);

// Set cluster-size constant for logical partition
void dfstSetClusterSize
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   USHORT              sectors                  // IN    sectors per cluster
);

// Get cluster-size constant for logical partition
USHORT dfstGetClusterSize                       // RET   sectors per cluster
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Set sector-size mapping constant for logical partition
void dfstSetSectorSize
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   USHORT              bytes                    // IN    bytes per sector
);

// Get sector-size mapping constant for logical partition
USHORT dfstGetSectorSize                        // RET   bytes per sector
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);


// Get raw disksize, not rounded down to whole cylinders
ULN64 dfstRawDiskSize                           // RET   raw disk size, sectors
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Get Geometry, number of Cylinders
ULONG  dfstGeoCylinders
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Get Geometry, number of Heads
ULONG  dfstGeoHeads
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Get Geometry, number of Sectors
ULONG  dfstGeoSectors
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Get FILEHANDLE for opened Volume/Imagefile (vfHandle), optional reset to 0
TXHFILE dfstGetVfHandle                         // RET   handle value or 0
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                reset                    // IN    reset handle to 0
);                                              //       (steal the handle :-)

// Get ACCESS-INFO for opened store object (accessInfo), optional reset to NULL
void *dfstGetAccessInfo                         // RET   access info or NULL
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                reset                    // IN    reset ptr to NULL
);                                              //       (steal the info :-)

// Get System-Geometry, number of Cylinders
ULONG  dfstSysCylinders
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Get System-Geometry, number of Heads
ULONG  dfstSysHeads
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Get System-Geometry, number of Sectors
ULONG  dfstSysSectors
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Translate Logical-Sector-Number to Physical sector number
ULN64 dfstLSN2Psn                               // RET   Physical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn                      // IN    Logical sector nr
);

// Translate Physical-sector-number to logical-sector-number
ULN64 dfstPsn2LSN                               // RET   Logical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    phys. sector nr
);

// Translate current 'Area' full-Disk based SN to Part based sector number
ULN64 dfstAreaD2Part                            // RET   Part bootrec based SN
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               sn                       // IN    full-Disk MBR based SN
);

// Translate current 'Area' Part based SN to full-Disk based sector number
ULN64 dfstAreaP2Disk                            // RET   full-Disk MBR based SN
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               sn                       // IN    Part bootrec based SN
);

// Translate Logical-Cluster-Number to Physical sector number
ULN64 dfstLCN2Psn                               // RET   Physical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lcn                      // IN    Logical cluster nr
);

// Translate Physical-sector-number to logical-cluster-number
ULN64 dfstPsn2LCN                               // RET   Logical cluster nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    phys. sector nr
);

// Get the PSN of the last read sector (last of group if multiple)
ULN64 dfstLastPsn                               // RET   Physical sector nr
(
   DFST_HANDLE         store                    // IN    DFS store to be used
);

// Read logical sectors using LSN and nr of sectors, retry single bad sectors
// Minimizes and reports nr of unreadable sectors (filled with 0xFE pattern)
// Sectors reported as 'done' are read, and include bad ones (once, no retry)
// 'done' value is less than 'nrs' only at last (partial) buffer on an object
ULONG dfstBadSectorRead
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    logical sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG              *done,                    // OUT   sectors read, or NULL
   ULONG              *bads,                    // OUT   bad sectors in buffer
   BYTE               *buffer                   // OUT   Sector buffer
);

// Read logical sector(s) using Logical sector nr (LSN) and nr of sectors
ULONG dfstReadLsn
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    logical sector nr
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // OUT   Sector buffer
);

// Read physical sector(s) using Physical sector nr (PSN) and nr of sectors
// Will always return PSN_LIMIT when psn is beyond last sector of object now
// (previously allowed reading whole disk, even if just one partition selected)
#define dfstReadPsn(s,p,n,b) dfstPsnReadCount(s,p,n,NULL,b)
// Read sectors using PSN, and count number of sectors actually read
ULONG dfstPsnReadCount
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG              *done,                    // OUT   sectors done, or NULL
   BYTE               *buffer                   // OUT   Sector buffer
);

// Write logical sectors using LSN and nr of sectors, skipping bad sectors
// Allows clone-MERGE for large buffers with some of them bad (0xFE pattern)
ULONG dfstBadSectorWrite
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    logical sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG               bads,                    // IN    bad sectors in buffer
   BYTE               *buffer                   // OUT   Sector buffer
);

// Write logical sector(s) using Logical sector number (LSN) and nr of sectors
ULONG dfstWriteLsn
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // IN    Sector buffer
);

// Write physical sector(s) using Physical sector nr (PSN) and nr of sectors
#define dfstWritePsn(s,p,n,b) dfstPsnWriteCount(s,p,n,NULL,b)
// Write sectors using PSN, and count number of sectors actually written
ULONG dfstPsnWriteCount
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG              *done,                    // OUT   sectors done, or NULL
   BYTE               *buffer                   // IN    Sector buffer
);

// Lock or unlock physical-disk or volume, use persistent handle variables
ULONG dfstLocking
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               lockmode,                // IN    LOCK / UNLOCK
   BOOL                ignore,                  // IN    ignore failure
   BOOL                verbose                  // IN    report lock status
);

// Get and set 'ignore lock-error' status
BOOL dfstIgnoreLockErr                           // return previous status
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                ignore                   // IN    Desired ignore status
);

// Check if write-access to current device is allowed (locked/ignore) and R/W
BOOL dfstCheckWriteAccess                       // RET   Writing allowed
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                noIgnore                 // IN    Ignoring not allowed
);

#endif
