//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS STORE, all store interface and translation services except sector I/O
//
// Author: J. van Wijk
//
// JvW  27-12-96   Initial version, loosely derived from QDISK and DHPFSERV
// JvW  01-05-97   Ported to WIN32
// JvW  21-01-98   Finished port to DOS and completed set of Write API's
// JvW  01-01-99   Added extended Int13 support for DFSDOS on disks > 1024 cyl
// JvW  06-08-99   Added physical geo display for OS/2, when supported (Aurora)
// JvW  07-12-99   More selective use of ext-int13 (avoiding some BIOS bugs)
// JvW  06-01-2000 Faster CountDisks for NT (start at 1, not 31)
// JvW  02-07-2000 Major restructure, introducing DFS stores and memdisk stores
// JvW  09-10-2001 Fully implemented store concept now (TODO: UNDO/REDO :-)
// JvW  03-05-2002 Fixed passing of volume letter for DOS (in vfHandle)
// JvW  28-05-2002 Finished port to DPMI version with WATCOM/Causeway extender
// JvW  16-08-2004 Linux Open/read/write plus generic device abstraction-layer
// JvW  27-07-2006 Split off sector-IO to dfsector.c, rename this to dfstore.c
// JvW  04-04-2017 Updated for 64bit sectornumbers and disk/partition sizes
//

#include <txlib.h>                              // TX library interfaces

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfsupart.h>                           // FDISK partition functions
#include <dfstore.h>                            // Store and sector I/O
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS navigation and defs
#include <dfsmdisk.h>                           // Disk store memory disks
#include <dfsutil.h>                            // DFS utility functions
#include <dfsectio.h>                           // DFS store I/O private defs

//- Magic value for 'align1' property set for GPT guard in CalculateGeometry
#define DFST_GPT_ALIGNED       0xbadfaced


        DFSTOREINFO  sti[ DFST_MAXOPEN];        // global store administration
        DFSTOREDISK  PhysDisk[DFS_MAX_DISK +1]; // info bound to each PHYS disk

static  BOOL         sti_valid = FALSE;         // sti administration is valid
static  DFST_HANDLE  sti_default = 1;           // default store number/handle

static  DFSGEODISK dskGeoInfo[DFS_MAX_DISK +1]; // Disk level geometry info


#if !defined (OEMSB)
static  TXLN         stStoreBuf1;               // static store description
static  TXLN         stStoreBuf2;               // static 2nd   description
#endif
static  TXTT         stStoreName;               // static StoreName buffer
static  TXTT         stStoreDpid;               // static StoreDpid buffer


/*****************************************************************************/
// Get the number for the current DEFAULT store (used in most DFSee commands)
/*****************************************************************************/
DFST_HANDLE dfstGetDefaultStore
(
   void
)
{
   return (sti_default);
}                                               // end 'dfstGetDefaultStore'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the number for the current DEFAULT store (used in most DFSee commands)
/*****************************************************************************/
ULONG dfstSetDefaultStore
(
   DFST_HANDLE         store                    // IN    new default store
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRARGS(("store: %u\n", store));

   if (store < DFST_MAXOPEN)
   {
      sti_default = store;
      dfsa->is    = &(sti[ sti_default].is);    // store dependend info in dfsa
   }
   else
   {
      rc = DFS_VALUE_ERROR;
   }
   RETURN ( rc);
}                                               // end 'dfstSetDefaultStore'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset the DEFAULT store to previously saved one (must be initialized!)
/*****************************************************************************/
ULONG dfstRestoreDefaultStore
(
   DFST_HANDLE         store                    // IN    previously saved store
)                                               //       must be initialized!
{
   ULONG               rc = NO_ERROR;
   DEVICE_STATE        v_screen = TxScreenState( DEVICE_TEST);

   ENTER();

   TxScreenState( DEVICE_OFF);                  // no output at all
   dfstClose(   sti_default);                   // close object in store
   dfstSetDefaultStore( store);                 // reselect saved one
   #if !defined (OEM_BRANDED)
      dfsInitFileSystem();                      // and re-init FS (menu!)
   #endif
   TxScreenState( v_screen);                    // restore screen output

   RETURN ( rc);
}                                               // end 'dfstRestoreDefaultStore'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Issue DFSee message and set the readonly attribute for all stores
/*****************************************************************************/
ULONG dfstSwitch2ReadOnly
(
   char               *message,                 // IN    cause of switch or NULL
   BOOL                verbose,                 // IN    give error/warnings
   BOOL                readonly                 // IN    readonly yes/no
)
{
   ULONG               rc = NO_ERROR;
   DFST_HANDLE         store;

   ENTER();

   for (store = 0; store < DFST_MAXOPEN; store++)
   {
      dfstSetReadOnly( store, verbose, readonly);
   }
   if (message && readonly && (dfsa->ReadOnly == FALSE))
   {
      TxNamedMessage( !dfsa->batch, 5002, " INFO: Read-Only access ",
         "Access to ALL stores is limited to READONLY for safety now.\n\n"
         "Reason: %s\n\nYou CAN (temporarily) enable writing by using the "
         "'-R-' option on the 'store' or 'disk' command first, or use the "
         "corresponding menu-item:\n\n [Edit -> Set Read-Only, no writes]",
          message);
   }
   dfsa->ReadOnly = readonly;                   // Set as default too
   RETURN ( rc);
}                                               // end 'dfstSwitch2ReadOnly'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the readonly attribute for the specified store
/*****************************************************************************/
ULONG dfstSetReadOnly
(
   DFST_HANDLE         store,                   // IN    store to set RO
   BOOL                verbose,                 // IN    give error/warnings
   BOOL                readonly                 // IN    readonly yes/no
)
{
   ULONG               rc = NO_ERROR;
   DFSTOREINFO        *si;

   ENTER();
   TRARGS(("store: %u, RO: %s\n", store, (readonly) ? "Yes" : "No"));

   si = &(sti[store]);
   if (store < DFST_MAXOPEN)
   {
      si = &(sti[store]);
      si->ReadOnly = readonly;
      if (verbose)
      {
         TxPrint( "%sDFSee     Store %s : Set to Read%s mode.\n",
                 (store) ? "" : "\n",
                 (store) ? (store == 1) ? "A" : "B" : "S",
                 (readonly) ? "Only" : "Write");
      }
   }
   else
   {
      rc = DFS_VALUE_ERROR;
   }
   RETURN ( rc);
}                                               // end 'dfstSetReadOnly'
/*---------------------------------------------------------------------------*/


#if !defined (OEMSB)
/*****************************************************************************/
// Get readonly status for specified store
/*****************************************************************************/
BOOL dfstGetReadOnly
(
   DFST_HANDLE         store,                   // IN    store
   char               *descr                    // OUT   descr string or NULL
)
{
   BOOL                rc = FALSE;              // function return
   DFSTOREINFO        *si;

   ENTER();

   if (store < DFST_MAXOPEN)                    // valid store
   {
      si = &(sti[store]);
      rc = si->ReadOnly;
      if (descr != NULL)
      {
         if (si->Type != DFST_UNUSED)
         {
            strcpy( descr, (si->ReadOnly) ? "Read-Only" : "ReadWrite");
         }
         else
         {
            strcpy( descr, "         ");
         }
      }
   }
   else if (descr != NULL)
   {
      sprintf( descr, "Invalid! ");
   }
   BRETURN (rc);
}                                               // end 'dfstGetReadOnly'
/*---------------------------------------------------------------------------*/


#ifndef OEMSB
/*****************************************************************************/
// Show status of all stores, and current default
/*****************************************************************************/
void dfstShowStoreInfo
(
   DFST_HANDLE         first                    // IN    first store to show
)
{
   DFST_HANDLE         store;
   DFSTOREINFO        *si;
   TXLN                text;

   ENTER();

   for (store = first; store < DFST_MAXOPEN; store++)
   {
      si = &(sti[store]);

      dfstQueryStoreType( store, NULL, text);

      TxPrint("\n%s\n", text);

      if (si->Type != DFST_UNUSED)
      {
         ULN64 maxpsn = ((ULN64) si->Geo.C) * si->Geo.H * si->Geo.S -1;
         sprintf( text, "Log geometry: Cyl :%9u H:%3u S:%-3u= %4.2lf MiB ",
                              si->Geo.C,  si->Geo.H,  si->Geo.S,
                                  TXSMIB((si->Geo.H * si->Geo.S), si->Geo.B));
         TxPrint( dfstrSz64Bps( text, "Size: ", maxpsn + 1, si->Geo.B, "\n"));

         if ((si->Geo.C != si->Sys.C) ||        // forced geometry in effect
             (si->Geo.H != si->Sys.H) ||
             (si->Geo.S != si->Sys.S) ||
             (si->Geo.B != si->Sys.B))
         {
            maxpsn =    ((ULN64) si->Sys.C) * si->Sys.H * si->Sys.S -1;
            sprintf( text, "Sys geometry: Cyl :%9u H:%3u S:%-3u= %4.2lf MiB ",
                                 si->Sys.C,  si->Sys.H,  si->Sys.S,
                                     TXSMIB((si->Sys.H * si->Sys.S), si->Sys.B));
            TxPrint( dfstrSz64Bps( text, "Size: ",  maxpsn + 1, si->Sys.B, "\n"));
         }
         dfsX10(     "Last accessed LSN : ", dfstPsn2LSN( store, si->LastPSN),  CNN, "  ");
         dfsX10(     "Base :  ",             si->Start,    CBG, "");
         dfsSz64Bps(" Size: ",   si->Final - si->Start +1, si->Geo.B, "\n");

         dfsX10(     "Last accessed PSN : ", si->LastPSN,  CNN, "  ");
         dfsX10(     "Final:  ",             si->Final,    CNC, "\n");

         TxPrint("Locking   handle  : 0x%8.8x  lock-count : % 3u   ", si->LogLock, si->LockCnt);
         dfsUllDot20("size", si->ByteSize, " bytes\n");

         TxPrint("Sectors / cluster : %- 3hu         sectorsize :% 4hu   "
                 "Sector read retries : %u\n", si->SpClu, si->Geo.B, si->Retries);

         switch (si->Type)
         {
            case DFST_IMZIMAGE:
            case DFST_RAWIMAGE:
            case DFST_PMVBOX:
            case DFST_PMIMZD:
            case DFST_PMIRAW:
            case DFST_VOLUME:
               TxPrint("Image/"
                  #if defined (UNIX)
                       "Dev."
                  #else
                       "Vol."
                  #endif
                       " handle : 0x%8.8x for ", si->vfHandle);
               switch (si->Type)
               {
                  case DFST_IMZIMAGE: TxPrint( "IMZ-imgFile"); break;
                  case DFST_RAWIMAGE: TxPrint( "RAW-imgFile"); break;
                  case DFST_PMVBOX:   TxPrint( "VDIdskImage"); break;
                  case DFST_PMIMZD:   TxPrint( "IMZdskImage"); break;
                  case DFST_PMIRAW:   TxPrint( "RAWdskImage"); break;
                  case DFST_VOLUME:
                     #if defined (UNIX)
                                      TxPrint( "Device-name"); break;
                     #else
                                      TxPrint( "Driveletter"); break;
                     #endif
                  default:                                     break;
               }
               TxPrint(": '%s%s%s'\n", CBY, si->Name, CNN);
               break;

            case DFST_MEMORY:
               dfsShowMdiskStatus( dfsDid2RealDiskNr(si->DiskId));
               break;

            case DFST_PHDISK:
               TxPrint("Phys disk handle  : 0x%8.8x  Shared  %u times "
                       "on disk nr: %s%hu%s = '%s'\n",
                        DI->pdHandle, DI->useCount, CBC, si->DiskId, CNN,
                                      dfsDid2DeviceName( si->DiskId));
               break;

            default:
               break;
         }
      }
   }
   VRETURN ();
}                                               // end 'dfstShowStoreInfo'
/*---------------------------------------------------------------------------*/
#endif

/*****************************************************************************/
// Query type and short description of specified store
/*****************************************************************************/
DFSTORETYPE dfstQueryStoreType
(
   DFST_HANDLE         store,                   // IN    store to query
   DFSISTORE         **info,                    // OUT   info struct, or NULL
   char               *descr                    // OUT   descr string or NULL
)
{
   DFSTORETYPE         rc = DFST_INVALID;       // function return
   DFSTOREINFO        *si;

   ENTER();

   if (store < DFST_MAXOPEN)                    // valid store
   {
      si = &(sti[store]);
      rc = si->Type;
      if (descr != NULL)
      {
         if      (store == DFST_SYSTEM) strcpy( descr, "System  = ");
         else if (store == DFSTORE)     strcpy( descr, "Current = ");
         else                           strcpy( descr, "Alternate ");
         strcat( descr, dfstStoreDesc1( store));

         if (si->Type != DFST_UNUSED)
         {
            strcat( descr, (si->ReadOnly) ? "  Read-Only!" : "  Read/Write");
         }
         else
         {
            strcat( descr, "  Unused");
         }
      }
      if (info != NULL)
      {
         *info  = &(si->is);
      }
   }
   else if (descr != NULL)
   {
      sprintf( descr, "Invalid store-index : %u", store);
   }
   RETURN (rc);
}                                               // end 'dfstQueryStoreType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get complete store description of specified store in a 54 byte string
/*****************************************************************************/
char *dfstStoreDescription1                     // RET   description-1
(
   DFST_HANDLE         store,                   // IN    store to query
   char               *buf                      // INOUT description or NULL
)                                               //      (uses static buffer)
{
   char               *descr = (buf) ? buf : stStoreBuf1;

   ENTER();

   if (store < DFST_MAXOPEN)                    // valid store
   {
      DFSTOREINFO     *si = &(sti[store]);
      TXLN             text;

      sprintf( descr, "Store %s : ", (store) ? (store == 1) ? "A" : "B" : "S");
      switch (si->Type)
      {
         case DFST_INVALID:
         case DFST_UNUSED:   strcpy( text, "Unused, not assigned yet. "); break;

         case DFST_RAWIMAGE:
         case DFST_IMZIMAGE:
            TRACES(("si->Name: '%s' length: %d\n", si->Name, strlen( si->Name)));
            if      (strlen( si->Name) > 26)
            {
               sprintf( text, "%6.6s..%18.18s", si->Name, si->Name + (strlen(si->Name) -18));
            }
            else if (strlen( si->Name) > 21)
            {
               sprintf( text, "%-26.26s",         si->Name);
            }
            else
            {
               sprintf( text, "%s= %-21.21s",    (si->Type == DFST_RAWIMAGE) ? "RAW" : "IMZ", si->Name);
            }
            break;

         case DFST_VOLUME:
            sprintf( text,
                     #if defined (UNIX)
                        "Dev %-14.14s"
                     #else
                        "Volume Dr-letter %2.2s"
                     #endif
                     " %-6.6s", si->is.drive, si->is.afsys);
            break;

         case DFST_PMIRAW:
         case DFST_PMIMZD:
         case DFST_PMVBOX:
            if (si->is.partid != 0)             // it is a partition
            {
               sprintf( text, "Img-P %2.2hu type:%2.2hhx %2.2s %-6.6s", si->is.partid,
                        si->is.p->partent.PartitionType, si->is.drive,  si->is.afsys);
            }
            else
            {
               if (si->Start == 0)              // whole partitioned image
               {
                  sprintf( text, "%s ImgFile Idisk %hu %-6.6s",
                                 (si->Type == DFST_PMVBOX) ? "VBOX" :
                                 (si->Type == DFST_PMIMZD) ? "IMZ"  : "RAW",
                                  si->is.disknr, si->is.afsys);

               }
               else                             // (freespace ?) area of disk
               {
                  sprintf( text, "@0x%11.11llx Id %hu %-6.6s", si->Start, si->is.disknr, si->is.afsys);
               }
            }
            break;

         case DFST_MEMORY:
            if (si->is.partid != 0)             // it is a partition
            {
               sprintf( text, "MdskP %2.2hu type:%2.2hhx %2.2s %-6.6s", si->is.partid,
                        si->is.p->partent.PartitionType, si->is.drive,  si->is.afsys);
            }
            else
            {
               if (si->Start == 0)              // whole memory disk
               {
                  sprintf( text, "Whole Memory disk %hu %-6.6s", si->is.disknr, si->is.afsys);
               }
               else                             // (freespace ?) area of disk
               {
                  sprintf( text, "@0x%11.11llx Md %hu %-6.6s", si->Start, si->is.disknr, si->is.afsys);
               }
            }
            break;

         case DFST_PHDISK:
            if (si->is.partid != 0)             // it is a partition
            {
               sprintf( text, "Part. %2.2hu type:%2.2hhx %2.2s %-6.6s", si->is.partid,
                        si->is.p->partent.PartitionType, si->is.drive,  si->is.afsys);
            }
            else
            {
               if (si->Start == 0)              // whole hard disk
               {
                  sprintf( text, "Whole  Phys. disk %hu %-6.6s", si->is.disknr, si->is.afsys);
               }
               else                             // (freespace ?) area of disk
               {
                  sprintf( text, "@0x%11.11llx Pd %hu %-6.6s", si->Start, si->is.disknr, si->is.afsys);
               }
            }
            break;
      }
      strcat(       descr, text);
      dfstrSz64XiB( descr, " size:", dfstGetLogicalSize( store), si->Geo.B, "");
   }
   else
   {
      sprintf( descr, "Invalid store-index : %u!", store);
   }
   RETURN (descr);
}                                               // end 'dfstStoreDescription1'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get secondary store-description (disk/image name) min 38 byte, max 80 byte
/*****************************************************************************/
char *dfstStoreDescription2                     // RET   description-2
(
   DFST_HANDLE         store,                   // IN    store to query
   char               *buf                      // INOUT description or NULL
)                                               //      (uses static buffer)
{
   char               *descr = (buf) ? buf : stStoreBuf2;
   USHORT              disk  = 0;
   DFSDISKINFO        *d;
   DFSTOREINFO        *si    = NULL;

   ENTER();

   if (store < DFST_MAXOPEN)                    // valid store
   {
      si = &(sti[store]);
      switch (si->Type)
      {
         case DFST_RAWIMAGE:
            if (strlen( si->Name) > 69)
            {
               sprintf( descr, "Filename: %11.11s..%56.56s", si->Name, si->Name + (strlen(si->Name) -56));
            }
            else
            {
               sprintf( descr, "Filename: %.69s", si->Name);
            }
            break;

         case DFST_VOLUME:
            sprintf( descr, "%s ", si->is.drive);
            if (strchr( "abAB",    si->is.drive[0]) != NULL)
            {
               strcat( descr, "volume access, (simulated) diskette");
            }
            else
            {
               strcat( descr, "accessed as Volume, not Direct-Disk");
            }
            break;

         case DFST_MEMORY:
         case DFST_PMIRAW:
         case DFST_PHDISK:
            disk = si->is.disknr;
            break;

         default:
            sprintf( descr, "%38.38s", "");
            break;
      }
   }
   else                                         // could be a disk-number
   {
      disk = (USHORT) (store - DFST_MAXOPEN);
   }
   if (disk != 0)
   {
      if ((d = dfsGetDiskInfo( disk)) != NULL)
      {
         if ((si != NULL) && (si->Start != 0))  // partition or freespace
         {
            sprintf( descr, "Area of : disk %2hu name = %s", disk, d->DiskName);
         }
         else
         {
            sprintf( descr, "Disk %2hu : name = %s", disk, d->DiskName);
         }
      }
      else
      {
         strcpy( descr, "Invalid partitioned-media disk number");
      }
   }
   RETURN (descr);
}                                               // end 'dfstStoreDescription2'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Get store-name as exact 15-character short description (no partition shown)
/*****************************************************************************/
char *dfstStoreName                             // RET   exact 15-char name
(
   DFST_HANDLE         store                    // IN    store to query
)
{
   DFSTOREINFO        *si;
   char               *s;

   ENTER();

   if (store < DFST_MAXOPEN)                    // valid store
   {
      si = &(sti[store]);
      s  = strrchr( si->Name, FS_PATH_SEP);

      switch (si->Type)
      {
         case DFST_PHDISK:   sprintf( stStoreName, "Disk%2hu         ",  si->DiskId); break;
         case DFST_VOLUME:
      #if !defined (UNIX)
                             sprintf( stStoreName, "Vol %2.2s         ", si->Name);   break;
      #endif
         case DFST_PMVBOX:
         case DFST_PMIRAW:
         case DFST_PMIMZD:   sprintf( stStoreName, "%s Disk Image", dfsMediaTypeDescr( dfsDid2DiskType(si->DiskId)));
            break;

         case DFST_IMZIMAGE: sprintf( stStoreName, "%-15.15s", (s) ? s+1 : si->Name); break;
         case DFST_RAWIMAGE: sprintf( stStoreName, "%-15.15s", (s) ? s+1 : si->Name); break;
         default:            sprintf( stStoreName, "Virt%2hu         ",  si->DiskId); break;
      }
   }
   else
   {
      sprintf( stStoreName, "Invalid store%2u", store);
   }
   RETURN (stStoreName);
}                                               // end 'dfstStoreName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get store identification as disk/partition or volume description, length 8
/*****************************************************************************/
char *dfstStoreDpid                             // RET   exact 15-char name
(
   DFST_HANDLE         store                    // IN    store to query
)
{
   DFSTOREINFO        *si;
   char               *s;

   ENTER();

   if (store < DFST_MAXOPEN)                    // valid store
   {
      si = &(sti[store]);
      s  = strrchr( si->Name, FS_PATH_SEP);

      switch (si->Type)
      {
         case DFST_PMVBOX:
         case DFST_PMIRAW:
         case DFST_PMIMZD:
         case DFST_MEMORY:
         case DFST_PHDISK:
            if (si->is.partid == 0)
            {
               sprintf( stStoreDpid, "Disk: %2hu",   si->DiskId);
            }
            else
            {
               sprintf( stStoreDpid, "Part: %02hu",  si->is.partid);
            }
            break;

         case DFST_VOLUME:
      #if !defined (UNIX)
                  sprintf( stStoreDpid, "Drive %2.2s", si->Name);        break;
      #endif
         default: sprintf( stStoreDpid, "%-8.8s", (s) ? s+1 : si->Name); break;
      }
   }
   else
   {
      sprintf( stStoreDpid, "Invalid!");
   }
   RETURN (stStoreDpid);
}                                               // end 'dfstStoreDpid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize STORE administration at startup (or close all)
/*****************************************************************************/
void dfstInitStoreInfo
(
   void
)
{
   DFST_HANDLE         store;
   ULONG               i;

   ENTER();

   for (store = 0; store < DFST_MAXOPEN; store++)
   {
      if (sti_valid)                            // close open stores
      {
         dfstClose( store);                     // close and init ...
      }
      else                                      // just initialize otherwise
      {
         dfstInitSingleStore( store);
      }
   }
   for (i = 0; i <= DFS_MAX_DISK; i++)
   {
      PhysDisk[i].useCount = 0;
      PhysDisk[i].pdHandle = PDHNULL;
   }
   sti_valid = TRUE;
   dfsa->is  = &(sti[ sti_default].is);         // store dependend info in dfsa

   VRETURN ();
}                                               // end 'dfstInitStoreInfo'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Initialize a single STORE structure
/*****************************************************************************/
void dfstInitSingleStore
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();
   TRARGS(( "Initialize store: %u\n", store));

   si->Type       = DFST_UNUSED;                // type of this store
   si->Geo.C      = DFS_STD_CYLS;               // default cylinders
   si->Geo.H      = DFS_STD_HEAD;               // default nr of heads
   si->Geo.S      = DFS_STD_SECT;               // default sectors/track
   si->Geo.B      = SECTORSIZE;                 // BPS, default at 512
   si->DiskId     = 0;                          // physical disk number
   si->di         = NULL;                       // physical disk info
   si->SpClu      = 1;                          // sectors per cluster
   si->Start      = 0;                          // start PSN logical limit
   si->Final      = DFS_MAX_PSN;                // final PSN logical limit
   si->AreaStart  = 0;                          // Area start PSN logical limit
   si->AreaFinal  = DFS_MAX_PSN;                // Area final PSN logical limit
   si->LastPSN    = 0;                          // Last PSN accessed
   si->Sectors    = DFS_MAX_PSN;                // sectors on open phys disk
   si->vfHandle   = 0;                          // Volume or (image) file handle
   si->accessInfo = NULL;                       // Access info (like IMZ index)
   si->SectorIO   = FALSE;                      // no HPFS sector-based IO
   si->Extended13 = FALSE;                      // no extended Int-13 used
   si->ReadOnly   = dfsa->ReadOnly;             // no writes allowed on store?

   si->LockIgn    = dfsa->LockIgn;              // ignore lock-failures
   si->LogLock    = 0;                          // logical volume lock-handle
   si->LockCnt    = 0;                          // device lock-counter

   si->Name[0]    = 0;                          // no filename
   si->ByteSize   = 0;                          // zero size

   si->is.object  = DFSO_NONE;                  // Unknown object
   si->is.p       = NULL;                       // Partition info structure
   si->is.disknr  = 0;                          // physical disk number
   si->is.partid  = 0;                          // partition-id
   si->is.stFlags = 0;                          // flags all clear
   strcpy( si->is.drive, "--");                 // drive letter or filename
   strcpy( si->is.afsys, "FDISK");              // filesystem name (FS / mode)

   dfstGeo2Sys( si);                            // sync Original Geo with new

#if defined (DEV32)
   TxFreeMem( si->TrackLayout);                 // free existing info
   si->tlSize      = 0;
#endif

   VRETURN ();
}                                               // end 'dfstInitSingleStore'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update partinfo ptr for any opened physical-disk stores (after ReadDiskInfo)
/*****************************************************************************/
void dfstUpdateStoreInfo
(
   BOOL                verbose                  // IN    output info
)
{
   #ifndef OEMSB
   ULONG               store;
   DFSTOREINFO        *si;

   ENTER();

   for (store = 0; store < DFST_MAXOPEN; store++)
   {
      si = &(sti[store]);
      if (si->Type == DFST_PHDISK)              // physical disk
      {
         if (si->is.partid != 0)                // partition open ?
         {
            if ((si->is.p = dfsGetPartInfo( si->is.partid)) != NULL)
            {
               if (si->is.p->drive[0] != si->is.drive[0])
               {
                  //- Note: old code did reset to FDISK mode, causing partid=00
                  //-       with a non-zero base value. This causes problems!

                  strcpy( si->is.drive, si->is.p->drive);
                  if ((verbose) && (si->is.drive[0] != 0))
                  {
                     TxPrint( "\nDrive-letter for partition %2.2hx has changed "
                              "to '%s'\n", si->is.partid, si->is.drive);
                  }
                  if (store == sti_default)     // was it the current one ?
                  {
                     dfsInitFileSystem();       // re-init
                  }
               }
            }
            else                                // partition does not exist
            {                                   // anymore (ejected ?
               TxPrint( "\nDisk with partition %2.2hu may have been ejected"
                        ", closing disk %hu ...\n", si->is.partid, si->is.disknr);
               dfstClose( store);               // close this store and init FDSK
               dfsFdskInit();
            }
         }
      }
   }
   VRETURN ();
   #endif
}                                               // end 'dfstUpdateStoreInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get reference to disk level geometry info for specified disk number
/*****************************************************************************/
DFSGEODISK *dfsGetDiskGeoInfo
(
   USHORT              disk                     // IN    disk number 0..n
)                                               //       0 is 'other' object
{
   DFSGEODISK         *rc = NULL;               // function return

   if (disk <= DFS_MAX_DISK)                    // allow whole array access
   {
      rc = &(dskGeoInfo[disk]);
   }
   return (rc);
}                                               // end 'dfsGetDiskGeoInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate most likely used disk-geometry based on disk-contents (sniffing)
/*****************************************************************************/
ULONG dfstCalculateGeometry
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                verbose,                 // IN    show disk geo
   DFS_GEO            *calculated               // OUT   calculated geometry
)
{
   ULONG               rc   = NO_ERROR;
   DFSTOREINFO        *si   = &(sti[store]);
   BYTE               *sec  = NULL;
   ULONG               sect = DFS_STD_SECT;
   ULONG               head = DFS_STD_HEAD;
   static TXTT         fmtDiskLgeoMessage = "\n Disk %hu L-Geo from: ";

   ENTER();
   TRACES(( "store:%u disk:%hu size:%llx verbose:%s\n", store, si->DiskId, si->Sectors, (verbose) ? "YES" : "NO"));
   TRHEAP( 500);

   calculated->H = (si->Geo.H)  ?  si->Geo.H : head;
   calculated->S = (si->Geo.S)  ?  si->Geo.S : sect;
   calculated->B = (si->Geo.B)  ?  si->Geo.B : SECTORSIZE;

   calculated->C = (si->Geo.C)  ?  si->Geo.C : (si->Sectors) ?
                   (si->Sectors / (calculated->H * calculated->S)) : DFS_STD_SECT;
   if ((sec = TxAlloc( DFS_MAX_SECT, calculated->B)) != NULL)
   {
      if ((rc = dfstReadPsn(store, 0, DFS_MAX_SECT, sec)) == NO_ERROR)
      {
         S_BOOTR      *pr  = (S_BOOTR *) sec;   // as partition-table record(s)
         S_LVINF      *lv  = (S_LVINF *) sec;   // as LVM information record(s)
         S_LVINF      *lvm;                     // selected DLAT record
         S_LVINF      *lvm2nd = NULL;           // second DLAT record found
         TXTS          sig;
         ULN64         size = 0;                // size from tables
         ULN64         last;                    // last size seen
         ULONG         i;
         ULONG         asect  = sect;           // alternative sect-count, to override
         ULONG         amatch = 0;              // alternative sect-match, to override
         ULONG         smatch = 0;              // number of sect matches
         ULONG         hmatch = 0;              // number of head matches
         ULONG         align1 = 0;              // # of boundaries aligned to 1 MiB cyl
         ULONG         pslots = 0;              // # of used p-table slots in MBR
         DFSPARTENTRY *pe;
         BYTE          st = dfsIdentifySector(0, 0, sec);
         ULONG         firstPartOffset = L32_NULL;     // Offset for 1st partition

         TRHEAP( 500);
         if (verbose)
         {
            TxPrint( fmtDiskLgeoMessage, si->DiskId);
         }
         switch (st)
         {
            case ST_MASTR:                      // Normal MBR found
            case ST_EXTBR:                      // disk with EBR, no MBR
            case ST_MCDDM:                      // MAC-style
               TRHEAP( 500);
               //- determine most likely sectors/track and size from the MBR partition table
               //- detect presence of a GPT guard partition, and force 1 MiB alignment then
               for (i = 0, pe = &(pr->PartitionTable[0]); i < 4; i++, pe++)
               {
                  if (pe->PartitionType != 0)
                  {
                     if (pe->BootSectorOffset != 0)
                     {
                        if (pe->BootSectorOffset < firstPartOffset)
                        {
                           firstPartOffset = pe->BootSectorOffset;
                        }
                        if (pe->BootSectorOffset <= DFS_MAX_SECT)
                        {
                           sect   = pe->BootSectorOffset;  //- partition starts after MBR track
                           smatch = 2;                     //- rather strong indicator
                        }
                     }
                     if ((last = pe->BootSectorOffset + pe->NumberOfSectors) > size)
                     {
                        size = last;
                     }
                     if (pe->PartitionType == DFS_P_EFI_GPT)
                     {
                        TRACES(("GPT table guard detected, force 1 MiB alignment\n"));
                        pslots = 1;                //- fake aligned slot found
                        align1 = DFST_GPT_ALIGNED; //- force 1 MiB alignment, magic value
                     }
                  }
                  TRACES(( "pe:%u sect:%u sm:%u size:%llx\n", i, sect, smatch, size));
               }

               TRHEAP( 500);
               //- determine most likely head/sector values from the MBR partition table
               //- and collect info on the 1-MiB alignment of the involved partitions
               //- unless a GPT guard partition has already been detected
               for ( i = 0,                       pe = &(pr->PartitionTable[0]);
                    (i < 4) && (align1 != DFST_GPT_ALIGNED);
                     i++,                         pe++)
               {
                  if (pe->PartitionType != 0)
                  {
                     if ((pe->BootSectorOffset != 0) && (pe->NumberOfSectors != 0))
                     {
                        pslots++;               // used slot, and plausible contents

                        if (pe->BootSectorOffset == DFS_1MIB_START) // starting at first 1-MiB boundary
                        {
                           align1 += 3;         // aligned start on first cylinder, very strong clue
                        }
                        if ((pe->BootSectorOffset & DFS_1MIB_MASK) == DFS_1MIB_ALIGNED)
                        {
                           align1 += 2;         // aligned start found, strong clue

                           if ((pe->NumberOfSectors  & DFS_1MIB_MASK) == DFS_1MIB_ALIGNED)
                           {
                              align1++;         // end is also aligned, even stronger
                           }
                           else
                           {
                              align1--;         // end not aligned, weaker clue
                           }
                        }
                        else
                        {
                           if (((pe->BootSectorOffset + pe->NumberOfSectors) & DFS_1MIB_MASK) == DFS_1MIB_ALIGNED)
                           {
                              align1++;         // end aligned, but start was not, weak
                           }
                        }

                        if (pe->LastSector.Head < DFS_STD_HEAD)
                        {
                           if (pe->LastSector.Head == (head -1))
                           {
                              hmatch++;
                           }
                           else if (hmatch < 2)
                           {
                              hmatch = 1;
                              head   = pe->LastSector.Head +1;
                           }
                        }
                        if (DFSC2SECTOR(pe->LastSector.SecCyl) != 0)
                        {
                           if (DFSC2SECTOR(pe->LastSector.SecCyl) == sect)
                           {
                              smatch++;
                           }
                           else if (smatch < 3)
                           {
                              ULONG alternative = DFSC2SECTOR(pe->LastSector.SecCyl);

                              if (alternative == asect)
                              {
                                 amatch++;
                                 if (amatch >= smatch)  //- alternative strong enough?
                                 {
                                    smatch = amatch;    //- switch to alternative sect#
                                    sect   = asect;
                                 }
                              }
                              else if ((amatch < 2) &&  //- switch to the NEXT alternative
                                       (alternative <= firstPartOffset)) // if acceptable
                              {
                                 amatch = 1;
                                 asect  = DFSC2SECTOR(pe->LastSector.SecCyl);
                              }
                           }
                        }
                     }
                  }
                  TRACES(( "pe:%u sect:%u sm:%u head:%u hm:%u\n", i, sect, smatch, head, hmatch));
               }

               TRHEAP( 500);
               //- Check LVM locations for large disk first, than standard (63) and
               //- finally start search from the MBR to check for multiples or odd geo

               memset( sig, 0,        TXMAXTS);
               memcpy( sig, sg_lvinf, SG_LVINF);

               lvm = &(lv[ DFS_2TB_SECT -1]);
               TRACES(( "Probe lvm sect %3u, addr: %8.8x\n", DFS_2TB_SECT, lvm));
               TRHEXS( 100,  lvm, 48, "Possible LVM info DLAT candidate");
               if (memcmp( lvm->Signature, sig, SG_LVINF) != 0) // LVM not there ?
               {
                  lvm = &(lv[ DFS_1TB_SECT -1]);
                  TRACES(( "Probe lvm sect %3u, addr: %8.8x\n", DFS_1TB_SECT, lvm));
                  TRHEXS( 100,  lvm, 48, "Possible LVM info DLAT candidate");
                  if (memcmp( lvm->Signature, sig, SG_LVINF) != 0) // LVM not there ?
                  {
                     lvm = &(lv[ DFS_STD_SECT -1]);
                     TRACES(( "Probe lvm sect %3u, addr: %8.8x\n", DFS_STD_SECT, lvm));
                     TRHEXS( 100,  lvm, 48, "Possible LVM info DLAT candidate");
                     if (memcmp( lvm->Signature, sig, SG_LVINF) != 0) // LVM not there ?
                     {
                        lvm = &(lv[ sect -1]);  // as indicated by table entries
                        TRACES(( "Probe lvm sect %3u, addr: %8.8x\n", sect, lvm));
                        TRHEXS( 100,  lvm, 48, "Possible LVM info DLAT candidate");
                        if (memcmp( lvm->Signature, sig, SG_LVINF) != 0) // LVM not there ?
                        {
                           lvm = NULL;
                        }
                     }
                  }
               }

               if (lvm == NULL)                 // LVM not in any standard location ?
               {
                  BYTE *lvmpos = (BYTE *) TxMemStr( sec, sig, DFS_MAX_SECT * si->Geo.B);

                  lvm = (S_LVINF *) lvmpos;
                  if (lvm != NULL)
                  {
                     TRACES(( "lvm info for sect %3u: %8.8x\n", (lvmpos - sec) / si->Geo.B, lvm));

                     // there should not be a 2nd LVM after this first one found
                     lvm2nd = (S_LVINF *) TxMemStr(  lvmpos + 10,   sig,
                      (DFS_MAX_SECT * si->Geo.B) - ((lvmpos + 10) - sec));

                     if (((lvmpos - sec) / si->Geo.B) < DFS_STD_SECT)
                     {
                        //- LVM found within first 63 sectors, should match tables,
                        //- Beyond that, it could be for huge disks and will NEVER match CHS!
                        if ((hmatch + smatch) > 2)     //- enough MBR table info present
                        {                              //- discard the LVM DLAT info
                           if (verbose)
                           {
                              TxPrint( "LVM DLAT at 0x%8.8x IGNORED, sect# invalid, or no primaries listed",
                                                                           ((BYTE *) lvm - sec) / si->Geo.B);
                              TxPrint( fmtDiskLgeoMessage, si->DiskId);
                           }
                           lvm = NULL;
                        }
                     }
                  }
                  else
                  {
                     TRACES(("No LVM DLAT found within 1st %hu sectors\n", DFS_MAX_SECT));
                  }
               }
               else                             // check for 2nd-LVM and validate size too
               {
                  // There should not be a 2nd LVM before this standard location
                  lvm2nd = (S_LVINF *) TxMemStr( sec, sig, DFS_MAX_SECT * si->Geo.B);

                  #ifndef OEMSB
                  if ((verbose) && (lvm->geoSecs > DFS_STD_SECT))
                  {
                     ULONG sptBelow = (lvm->geoSecs > DFS_1TB_SECT) ? DFS_1TB_SECT : DFS_STD_SECT;

                     if (si->Sectors < (65535UL * lvm->geoHeads * sptBelow))
                     {
                        ULONG       os2Limit = dfs16bitCylLimitGiB( lvm->geoHeads, sptBelow);

                        TxPrint( "LVM: Sect/track %u implies disk > %u GiB, non optimal value!", lvm->geoSecs, os2Limit);
                        TxPrint( fmtDiskLgeoMessage, si->DiskId);
                     }
                  }
                  #endif

                  #if defined (OEM_BRANDED)
                     if (TxaExeSwitchUnSet(DFS_O_LVMSIZE)) // default overruled, ignore LVM SIZE
                  #else
                     if (!TxaExeSwitchSet(DFS_O_LVMSIZE)) // default not overruled, ignore LVM SIZE
                  #endif
                  {                             // ignore the LVM size, adjust L-geo to match tables
                     if (((ULN64) lvm->geoCyls * lvm->geoHeads * lvm->geoSecs) < size)
                     {
                        if (verbose)
                        {
                           TxPrint( "LVM DLAT at 0x%8.8x IGNORED, end of last partition is beyond LVM disk size",
                                                                                ((BYTE *) lvm - sec) / si->Geo.B);
                           TxPrint( fmtDiskLgeoMessage, si->DiskId);
                        }
                        lvm = NULL;
                     }
                     TRACES(("Last partition extends BEYOND the LVM disk size, IGNORE and adjust L-geo\n"));
                  }
                  else
                  {
                     TRACES(("Last partition extends BEYOND LVM disk size, don't ignore, force a warning!\n"));
                  }
               }
               if (lvm != NULL)
               {
                  S_LVINF      *lvm1st = lvm;   // original DLAT record

                  TRHEXS( 100,  lvm1st, 128, "LVM 1st DLAT");

                  if ((lvm2nd != NULL) && (lvm2nd != lvm1st)) // 2 different DLATs!
                  {
                     TRHEXS( 100,  lvm2nd, 128, "LVM 2nd  DLAT");

                     if (firstPartOffset != L32_NULL)
                     {
                        //- Check which of the two DLAT's is the 'most likely correct one,
                        //- based on 'offset 1st partition' modulo sect/track being zero

                        TRACES(("Offset-P1: 0x%8.8x  SPT-1: %u SPT2: %u\n",
                                 firstPartOffset, lvm1st->geoSecs, lvm2nd->geoSecs));

                        if ((firstPartOffset % lvm1st->geoSecs) != 0) // First seems unaligned
                        {
                           if ((firstPartOffset % lvm2nd->geoSecs) == 0) // Second IS aligned
                           {
                              lvm = lvm2nd;     // update LVM DLAT reference to use for GEO
                              lvm2nd = lvm1st;  // update '2nd' for correct user message
                           }
                        }
                        //- on exit here: 'lvm' is the accepted, and 'lvm2nd' the discarded one
                     }

                     if (verbose)               // report (possibly swapped) 2nd DLAT
                     {
                        TxPrint( "2nd DLAT at PSN 0x%8.8x IGNORED, multiple DLAT sectors!",
                                   ((BYTE *) lvm2nd - sec) / si->Geo.B);
                        TxPrint( fmtDiskLgeoMessage, si->DiskId);
                     }
                  }
                  if (verbose)
                  {
                     TxPrint(    "LVM info at PSN 0x%8.8x ACCEPTED as matching DLAT sector",
                                   ((BYTE *) lvm - sec) / si->Geo.B);
                  }
                  calculated->C = lvm->geoCyls;
                  calculated->H = lvm->geoHeads;
                  calculated->S = lvm->geoSecs;
               }
               else if ((((hmatch + smatch) != 0)               || // clues from table entries
                         ((pslots != 0) && (align1 >= pslots))) && // or likely a 64/32 geo (1 MiB)
                       (si->Geo.S <= DFS_STD_SECT)               ) // and no OS/2 large disk geo ?
               {
                  if ((pslots != 0) && (align1 >= pslots))
                  {
                     if (verbose)
                     {
                        if (align1 == DFST_GPT_ALIGNED)
                        {
                           TxPrint( "GPT table, likely 1-MiB cylinders (GPT guard present in MBR)");
                        }
                        else
                        {
                           TxPrint( "MBR table, likely 1-MiB cylinders (like SSD or 4Kb sectors)");
                        }
                     }
                     sect = DFS_1MIB_SECT;
                     head = DFS_1MIB_HEAD;
                  }
                  else
                  {
                     if (verbose)
                     {
                        TxPrint( "MBR table, matches for H:%u S:%u", hmatch, smatch);
                     }
                  }
                  if (size <= si->Sectors)      // keep largest size for geo
                  {
                     size = si->Sectors;
                  }
                  else                          // update store size ...
                  {
                     si->Sectors = size;
                  }
                  calculated->C = (ULONG) ((size -1) / (head * sect)) +1; // round UP non-aligned
                  calculated->H = head;
                  calculated->S = sect;
               }
               else                             // not able to calculate, use
               {                                // input values, unless zero
                  if (verbose)
                  {
                     #if defined (DEV32)
                     if (si->Geo.S > DFS_STD_SECT)
                     {
                        TxPrint( "Device info (OS/2 huge disk)");
                     }
                     else
                     #endif
                     {
                        TxPrint( "Device info and sanity");
                     }
                  }
               }
               break;

            case ST_BOOTR:                      // large floppy format
            case ST_BOOTX:
               //- to be refined, may interpret boot-sectors
               //- but these often have incorrect geo anyway
               if (verbose)
               {
                  TxPrint( "Device info, large floppy format!");
               }
               break;

            default:                            // empty or garbage
               if (verbose)
               {
                  TxPrint( "Device info, empty or garbage sectors!");
               }
               break;
         }
         TRHEAP( 500);
      }
      TxFreeMem( sec);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(("Calculated geo now : C:%6u  H:%3u  S:%2u  bps:%hu\n",
            calculated->C, calculated->H, calculated->S, calculated->B));
   RETURN (rc);
}                                               // end 'dfstCalculateGeometry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display (and force a specific) geometry on the currently open physical disk
// Special value 0 for cyl, head or sector will keep the original value
// Special value L32_NULL for cyl will cause recalculation for Cylinder count
/*****************************************************************************/
ULONG dfstDiskGeometry
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               base,                    // IN    bootsector offset or 0
   ULONG               cyl,                     // IN    nr of cylinders   or 0
   ULONG               hds,                     // IN    nr of heads       or 0
   ULONG               spt,                     // IN    sectors/track     or 0
   USHORT              bps,                     // IN    bytes per sector  or 0
   BOOL                sync,                    // IN    sync System geo too
   BOOL                verbose                  // IN    show disk geo
)
{
   ULONG               rc = NO_ERROR;
   DFSTOREINFO        *si = &(sti[store]);
   ULN64               maxpsn;                  // size of disk in sectors
   TXTS                who;                     // for 10/6 character descr
   DFS_GEO            *forced = &(dskGeoInfo[ si->DiskId].Forced);

   ENTER();
   TRACES(( "store:%u disk:%hu C:%9u H:%3u S:%2u B:%3hu sync:%s\n",
             store, si->DiskId, cyl, hds, spt, bps, (sync) ? "YES" : "NO"));
   TRACES(("Original Lgeometry : C:%9u  H:%3u  S:%2u  bps:%hu  Sectors: 0x%llx\n",
                         si->Geo.C, si->Geo.H, si->Geo.S, si->Geo.B, si->Sectors));

   strcpy( who, dfstStoreName( store));
   who[ 9] = 0;                                 // get 9-character name

   // JvW 20100105, fixed forcing geo back to original values (used to be ignored)
   if ((cyl != 0) && (cyl != L32_NULL) && ((cyl != si->Geo.C) || (cyl != forced->C)))
   {
      if (verbose)
      {
         TxPrint("\n%s forcing : cylinders from %9u to %9u",
                       who, (cyl != si->Geo.C) ? si->Geo.C : forced->C, cyl);
      }
      forced->C = si->Geo.C = cyl;

      #if defined (DOS32)
         if (cyl > 1024)                        // need extended int13 ?
         {
            si->Extended13 = TRUE;
         }
         else
         {
            si->Extended13 = FALSE;
         }
         if (verbose)
         {
            TxPrint("\n%s forcing : use of extended Int13 "
                    "at cylinders >= 1024: %s%s%s",
                     who, CBM, (si->Extended13) ? "YES" : "NO", CNN);
         }
      #endif
   }
   if (hds && ((hds != si->Geo.H) || (hds != forced->H)))
   {
      if (verbose)
      {
         TxPrint("\n%s forcing : heads     from %6u to %6u",
                       who, (hds != si->Geo.H) ? si->Geo.H : forced->H, hds);
      }
      forced->H = si->Geo.H = hds;
   }
   if (spt && ((spt != si->Geo.S) || (spt != forced->S)))
   {
      if (verbose)
      {
         TxPrint("\n%s forcing : sectors   from %6u to %6u",
                       who, (spt != si->Geo.S) ? si->Geo.S : forced->S, spt);
      }
      forced->S = si->Geo.S = spt;
   }
   if (bps && (bps != si->Geo.B))
   {
      USHORT           old = si->Geo.B;
      ULN64            sec = si->Sectors;

      dfstSetSectorSize( store, bps);           // check and set Geo.B
      if (si->Geo.B == bps)                     // set correctly ...
      {
         if (cyl == L32_NULL)                   // recalculate requested
         {
            if (old < bps)                      // update nr of sectors
            {
               si->Sectors  /= (bps / old);     // less sectors ...
            }
            else
            {
               si->Sectors  *= (old / bps);     // more sectors ...
            }
         }
         if (verbose)
         {
            TxPrint("\n%s forcing : bytes/sec from %6hu to %6hu",   who, old, bps);
            if (cyl == L32_NULL)                // recalculate requested
            {
               TxPrint("\n%s forcing : #sects from  %12.12llx to %12.12llx", who, sec, si->Sectors);
            }
         }
         forced->B = si->Geo.B = bps;
      }
   }
   if (cyl == L32_NULL)                         // recalculate requested
   {
      ULN64            cc;                      // calculated cylinders

      if (si->Sectors != 0)
      {
         cc = ((si->Sectors -1) / (si->Geo.H * si->Geo.S)) +1;
         if (cc > ((ULN64) 0xffffffff))         // too large for ULONG
         {
            cc = 0xffffffff;
         }
         if (verbose)
         {
            TxPrint("\n%s  adjust : cylinders from %9u to %9u", who, si->Geo.C, (ULONG) cc);
         }
         forced->C = si->Geo.C = (ULONG) cc;
      }
   }

   //- Make sure actual geometry (L-Geo) always matches forced one, when set
   //- Might get out of sync on memdisks otherwise (see Mensys trace 20110604)
   if (forced->C != 0)
   {
      si->Geo.C = forced->C;
   }
   if (forced->H != 0)
   {
      si->Geo.H = forced->H;
   }
   if (forced->S != 0)
   {
      si->Geo.S = forced->S;
   }
   if (forced->B != 0)
   {
      si->Geo.B = forced->B;
   }

   maxpsn = (ULN64) si->Geo.C * si->Geo.H * si->Geo.S -1;
   if ((maxpsn < si->Sectors) && (sync == FALSE))    //- unless we want to set S-geo too
   {
      maxpsn = si->Sectors -1;                       //- keep max (physical) sector count!
   }
   if ((base != 0) || (cyl != 0) || (hds != 0) || (spt != 0)) // any change ?
   {
      TRACES(("Logical limits reset by Diskgeometry()!\n"));
      dfstLogicalLimits( store, base, maxpsn);  // set the updated actual size
   }
   if (sync)                                    // sync S-Geo to new geometry
   {
      dfstGeo2Sys( si);                         // sync Original Geo with new
      if (maxpsn != 0)
      {
         si->Sectors = maxpsn;                  // sync size too
      }
   }
   #ifndef OEMSB
   if (verbose)
   {
      TXLN             geo;

      who[6] = 0;
      sprintf( geo, "\n L-Geo %s%s%s Cyl :%9u H:%3u S:%-3u Bps:%-4hu  Size:",
               CBG, who, CNN,  si->Geo.C,  si->Geo.H,  si->Geo.S, si->Geo.B);
      dfsSz64Bps( geo, (ULN64) si->Geo.C * si->Geo.H * si->Geo.S, si->Geo.B, "\n");

      if (TxaExeSwitch('g') &&                  // most relevant to STDOUT too
          (si->is.noAccess == NO_ERROR))        // and MBR OK (dummy removeable)
      {
         TxStripAnsiCodes( geo);
         printf( "%s", geo);
         fflush( stdout);
      }

      if ((si->Geo.C != si->Sys.C) ||
          (si->Geo.H != si->Sys.H) ||
          (si->Geo.S != si->Sys.S) ||           // forced geometry in effect
          (si->Geo.B != si->Sys.B) || sync)     // or newly synchronized
      {
         sprintf( geo, " S-Geo %s%s%s Cyl :%9u H:%3u S:%-3u Bps:%-4hu  Size:",
                  CBG, who, CNN,  si->Sys.C,  si->Sys.H,  si->Sys.S, si->Sys.B);
         dfsSz64Bps( geo, (ULN64) si->Sys.C * si->Sys.H * si->Sys.S, si->Sys.B, "\n");
      }
   }
   TRACES(("Forced geometry now: C:%9u  H:%3u  S:%2u  bps:%hu\n",
                         forced->C, forced->H, forced->S, forced->B));
   TRACES(("System geometry now: C:%9u  H:%3u  S:%2u  bps:%hu\n",
                         si->Sys.C, si->Sys.H, si->Sys.S, si->Sys.B));
   TRACES(("Resulting geometry : C:%9u  H:%3u  S:%2u  bps:%hu  Sectors: 0x%llx\n",
                         si->Geo.C, si->Geo.H, si->Geo.S, si->Geo.B, si->Sectors));
   #endif
   RETURN (rc);
}                                               // end 'dfstDiskGeometry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Physical sector number (PSN) to CHS Cylinder value
/*****************************************************************************/
ULONG  dfstPsn2Cyl                              // RET   cylinder number
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    physical sector nr
)
{
   ULONG               rc = 0xffffffff;         // max cylinder value
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               ulSpc;                   // sectors per cylinder

   if (psn != DFS_MAX_PSN)
   {
      ulSpc = si->Geo.H * si->Geo.S;
      if (ulSpc)
      {
         rc = psn / ulSpc;
      }
   }
   return (rc);
}                                               // end 'dfstPsn2Cyl'
/*---------------------------------------------------------------------------*/

#if !defined (OEMSB)
/*****************************************************************************/
// Translate Physical sector number (PSN) to CHS Head value
/*****************************************************************************/
ULONG  dfstPsn2Head                             // RET   head number
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    physical sector nr
)
{
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               rc = si->Geo.H -1;       // max head value
   ULONG               ulSpc;                   // sectors per cylinder
   ULONG               ulHeadSec;               // sectors per cylinder

   if (psn != DFS_MAX_PSN)
   {
      ulSpc     = si->Geo.H * si->Geo.S;
      ulHeadSec = psn % ulSpc;
      if (si->Geo.S)
      {
         rc = ulHeadSec / si->Geo.S;
      }
   }
   return (rc);
}                                               // end 'dfstPsn2Head'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Physical sector number (PSN) to CHS Sector value
/*****************************************************************************/
ULONG  dfstPsn2Sector                           // RET   sector number
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    physical sector nr
)
{
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               rc = si->Geo.S;          // max sector value
   ULONG               ulSpc;                   // sectors per cylinder
   ULONG               ulHeadSec;               // sectors per cylinder

   if (psn != DFS_MAX_PSN)
   {
      ulSpc = si->Geo.H * si->Geo.S;
      if (ulSpc)
      {
         ulHeadSec = psn % ulSpc;
         if (si->Geo.S)
         {
            rc = (ulHeadSec % si->Geo.S) +1;
         }
      }
   }
   return (rc);
}                                               // end 'dfstPsn2Sector'
/*---------------------------------------------------------------------------*/
#endif

/*****************************************************************************/
// Translate Physical sect-nr (PSN) to Head-Cylinder-Sector, use forced Geo
/*****************************************************************************/
ULONG dfstPsn2Chs
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    physical sector nr
   ULONG              *cylinder,                // OUT   cylinder number
   ULONG              *head,                    // OUT   head number
   ULONG              *sector                   // OUT   first sector number
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return( dfsGeoPsn2Chs( psn, si->Geo.H, si->Geo.S, cylinder, head, sector));
}                                               // end 'dfstPsn2Chs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Physical sect-nr (PSN) to Head-Cylinder-Sector, use Original Geo
/*****************************************************************************/
ULONG dfstPsn2Sys
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    physical sector nr
   ULONG              *cylinder,                // OUT   cylinder number
   ULONG              *head,                    // OUT   head number
   ULONG              *sector                   // OUT   first sector number
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return( dfsGeoPsn2Chs( psn, si->Sys.H, si->Sys.S, cylinder, head, sector));
}                                               // end 'dfstPsn2Sys'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make system Geo equal to (forced) Geo
/*****************************************************************************/
void dfstGeo2Sys
(
   DFSTOREINFO        *si                       // IN    DFS storeinfo pointer
)
{
   ENTER();
   TRACES(("C:%u H:%u S:%u B:%u\n", si->Geo.C, si->Geo.H, si->Geo.S, si->Geo.B));

   si->Sys.C = si->Geo.C;
   si->Sys.H = si->Geo.H;
   si->Sys.S = si->Geo.S;
   si->Sys.B = si->Geo.B;

   VRETURN();
}                                               // end 'dfstGeo2Sys'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset ForcedGeometry to be empty for specified disk-device
/*****************************************************************************/
void dfsResetForcedGeo
(
   USHORT              did                      // IN    disk-id
)
{
   ENTER();

   dskGeoInfo[ did].Forced.C = 0;
   dskGeoInfo[ did].Forced.H = 0;
   dskGeoInfo[ did].Forced.S = 0;
   dskGeoInfo[ did].Forced.B = 0;

   strcpy( dskGeoInfo[ did].Reason, "");

   VRETURN();
}                                               // end 'dfsResetForcedGeo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Cylinder-Head-Sector to Physical sector number (PSN)
/*****************************************************************************/
ULN64 dfstChs2Psn                               // RET   physical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector                   // IN    first sector number
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return( dfsGeoChs2Psn( si->Geo.H, si->Geo.S, cylinder, head, sector));
}                                               // end 'dfstChs2Psn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set start and final-PSN for Read logical sector(s)
/*****************************************************************************/
ULN64 dfstLogicalLimits                         // RET   current start
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               start,                   // IN    PSN logical sector 0
   ULN64               final                    // IN    PSN final sector
)
{
   DFSTOREINFO        *si = &(sti[store]);
   ULN64               current = si->Start;     // current Base PSN

   ENTER();

   TRARGS(("Set base to: %llx, end:%llx => size:%llx\n", start, final, final +1 - start));

   si->Start     = start;
   si->AreaStart = start;                       // sync Area to object
   si->Final     = final;
   si->AreaFinal = final;

   if (si->Type != DFST_RAWIMAGE)               // has exact size set
   {                                            // in OpenFile()
      si->ByteSize = (ULN64) (final - start +1) * si->Geo.B;
   }
   RETN64 (current);
}                                               // end 'dfstLogicalLimits'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set start and final-PSN for FS-specific Area sector(s)
/*****************************************************************************/
ULN64 dfstSetAreaLimits                         // RET   current area start
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               start,                   // IN    PSN logical sector 0
   ULN64               final                    // IN    PSN final sector
)
{
   DFSTOREINFO        *si = &(sti[store]);
   ULN64               current;                 // current Base PSN

   ENTER();

   TRARGS(("Set base to: %llx, end:%llx => size:%llx\n", start, final, final +1 - start));

   current       = si->AreaStart;
   si->AreaStart = start;                       // set Area to new limits
   si->AreaFinal = final;
   RETN64 (current);
}                                               // end 'dfstSetAreaLimits'
/*---------------------------------------------------------------------------*/

#if !defined (OEMSB)
/*****************************************************************************/
// Get size as defined by logical limits
/*****************************************************************************/
ULN64 dfstGetByteSize                           // RET   Exact size in bytes
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();

   RETN64 (si->ByteSize);
}                                               // end 'dfstGetByteSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set exact size in bytes for specified store
/*****************************************************************************/
void dfstSetByteSize                            // RET   Exact size in bytes
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               size                     // IN    size in bytes
)
{
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();
   TRACES(( "Set size to %llu bytes\n", size));

   si->ByteSize = size;

   VRETURN();
}                                               // end 'dfstSetByteSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get size as defined by logical limits
/*****************************************************************************/
ULN64 dfstGetLogicalSize
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   ULN64               rc;
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();

   rc = si->Final - si->Start;
   if (rc != DFS_MAX_PSN)
   {
      rc++;                                     // add one, when possible
   }
   TRACES(("size: 0x%llx\n", rc));
   RETN64 (rc);
}                                               // end 'dfstGetLogicalSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cluster-size constant for logical partition
/*****************************************************************************/
USHORT dfstGetClusterSize                       // RET   sectors per cluster
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return( si->SpClu);
}                                               // end 'dfstGetClusterSize'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Set cluster-size constant for logical partition
/*****************************************************************************/
void dfstSetClusterSize
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   USHORT              sectors                  // IN    sectors per cluster
)
{
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();

   si->SpClu = (sectors) ? sectors : 1;
   TRACES(("Logical mapping, sectors per cluster: %u\n", sectors));
   VRETURN();
}                                               // end 'dfstSetClusterSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get sector-size mapping constant for logical partition
/*****************************************************************************/
USHORT dfstGetSectorSize                        // RET   bytes per sector
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   USHORT              rc = SECTORSIZE;         // sane value
   DFSTOREINFO        *si = &(sti[store]);

   if (si->Geo.B != 0)
   {
      rc = si->Geo.B;
   }
   return( rc);
}                                               // end 'dfstGetSectorSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set sector-size mapping constant for logical partition, validate BPS value
/*****************************************************************************/
void dfstSetSectorSize
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   USHORT              bytes                    // IN    bytes per sector
)
{
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();
   TRACES(( "Bytes per sector, raw value: %hu\n", bytes));

   switch (bytes)                               // allow valid values only
   {
      case  512:
      case 1024:
      case 2048:
      case 4096:
         si->Geo.B = bytes;
         break;

      default:
         si->Geo.B = SECTORSIZE;
         // TxPrint( "\nWARNING: invalid bytes/sect value %hu, using 512!\n", bytes);
         break;
   }
   TRACES(("Geometry, bytes/sector: %u\n", si->Geo.B));
   VRETURN();
}                                               // end 'dfstSetSectorSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get raw disksize, not rounded down to whole cylinders
/*****************************************************************************/
ULN64 dfstRawDiskSize                           // RET   raw disk size, sectors
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();
   RETN64 ( si->Sectors);
}                                               // end 'dfstRawDiskSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get Geometry, number of Cylinders
/*****************************************************************************/
ULONG dfstGeoCylinders
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return ( si->Geo.C);
}                                               // end 'dfstGeoCylinders'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get Geometry, number of Heads
/*****************************************************************************/
ULONG dfstGeoHeads
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return ( si->Geo.H);
}                                               // end 'dfstGeoHeads'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get Geometry, number of Sectors
/*****************************************************************************/
ULONG dfstGeoSectors
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return ( si->Geo.S);
}                                               // end 'dfstGeoSectors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Logical-Sector-Number to Physical sector number
/*****************************************************************************/
ULN64 dfstLSN2Psn                               // RET   Physical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn                      // IN    Logical sector nr
)
{
   DFSTOREINFO        *si = &(sti[store]);
   ULN64               rc = si->Start + lsn;    // out of range PSN forces error
                                                // except on RAWIMAGE to allow extend
   if ((rc > si->Final) &&                      // Now works for WRITE and READ,
       (si->Type != DFST_RAWIMAGE))             // while READ may need protection
   {                                            // (see P1484)
      si->LastPSN = rc;                         // keep last-accessed actual PSN
      rc = DFS_MAX_PSN;                         // when returning the LIMIT
   }
   return (rc);
}                                               // end 'dfstLSN2Psn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Physical-sector-number to logical-sector-number
/*****************************************************************************/
ULN64 dfstPsn2LSN                               // RET   Logical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    phys. sector nr
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return (psn - si->Start);
}                                               // end 'dfstPsn2LSN'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate current 'Area' full-Disk based SN to Part based sector number
/*****************************************************************************/
ULN64 dfstAreaD2Part                            // RET   Part bootrec based SN
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               sn                       // IN    full-Disk MBR based SN
)
{
   DFSTOREINFO        *si  = &(sti[store]);

   return ( sn + si->Start - si->AreaStart);
}                                               // end 'dfstAreaD2Part'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate current 'Area' Part based SN to full-Disk based sector number
/*****************************************************************************/
ULN64 dfstAreaP2Disk                            // RET   full-Disk MBR based SN
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               sn                       // IN    Part bootrec based SN
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return ( sn + si->AreaStart - si->Start);
}                                               // end 'dfstAreaP2Disk'
/*---------------------------------------------------------------------------*/


#if !defined (OEMSB)
/*****************************************************************************/
// Translate Logical-Cluster-Number to Physical sector number
/*****************************************************************************/
ULN64 dfstLCN2Psn                               // RET   Physical sector nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lcn                      // IN    Logical cluster nr
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return dfstLSN2Psn( store, lcn * si->SpClu);
}                                               // end 'dfstLCN2Psn'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Translate Physical-sector-number to logical-cluster-number
/*****************************************************************************/
ULN64 dfstPsn2LCN                               // RET   Logical cluster nr
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn                      // IN    phys. sector nr
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return( dfstPsn2LSN( store, psn) / si->SpClu);
}                                               // end 'dfstPsn2LCN'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get FILEHANDLE for opened Volume/Imagefile (vfHandle), optional reset to 0
/*****************************************************************************/
TXHFILE dfstGetVfHandle                         // RET   handle value or 0
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                reset                    // IN    reset handle to 0
)                                               //       (steal the handle :-)
{
   DFSTOREINFO        *si = &(sti[store]);
   TXHFILE             rc = si->vfHandle;

   if (reset)
   {
      si->vfHandle = 0;                         // avoid Close handle when
   }                                            // store is closed
   return ( rc);
}                                               // end 'dfstGetVfHandle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get ACCESS-INFO for opened store object (accessInfo), optional reset to NULL
/*****************************************************************************/
void *dfstGetAccessInfo                         // RET   access info or NULL
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                reset                    // IN    reset ptr to NULL
)                                               //       (steal the info :-)
{
   DFSTOREINFO        *si = &(sti[store]);
   void               *rc = si->accessInfo;

   if (reset)
   {
      si->accessInfo = NULL;                    // avoid FREE of info when
   }                                            // store is closed
   return ( rc);
}                                               // end 'dfstGetVfHandle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get System-Geometry, number of Cylinders
/*****************************************************************************/
ULONG dfstSysCylinders
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return ( si->Sys.C);
}                                               // end 'dfstSysCylinders'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Get System-Geometry, number of Heads
/*****************************************************************************/
ULONG dfstSysHeads
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return ( si->Sys.H);
}                                               // end 'dfstSysHeads'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Get System-Geometry, number of Sectors
/*****************************************************************************/
ULONG dfstSysSectors
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return ( si->Sys.S);
}                                               // end 'dfstSysSectors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read logical sectors using LSN and nr of sectors, retry single bad sectors
// Minimizes and reports nr of unreadable sectors (filled with 0xFE pattern)
// Sectors reported as 'done' are read, and include bad ones (once, no retry)
// 'done' value is less than 'nrs' only at last (partial) buffer on an object
/*****************************************************************************/
ULONG dfstBadSectorRead
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    logical sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG              *done,                    // OUT   sectors read, or NULL
   ULONG              *bads,                    // OUT   bad sectors in buffer
   BYTE               *buffer                   // OUT   Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // function return code
   ULN64               firstPsn   = dfstLSN2Psn(store, lsn);
   ULONG               badsectors = 0;
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               rc1;                     // single-sector rc
   ULONG               i;

   ENTER();
   switch (rc = dfstPsnReadCount( store, firstPsn, nrs, done, buffer))
   {
      case NO_ERROR:                            // OK, or
      case DFS_NO_DEVICE:                       // bad-sector retry useless
      case DFS_PSN_LIMIT:
         break;

      default:
         if ((nrs > 1) &&                       // on multiple-read, try
             (dfsa->eStrategy != TXAE_QUIT))    // to get the best result
         {                                      // read one sector at a time
            TXTM    status;

            sprintf( status, "Sector: 0x%12.12llx - retrying BAD SECTORS! ", lsn);
            txwSetSbviewStatus( status, TXWSCHEME_ERRMSG);

            for (i = 0; i < nrs; i++)
            {
               rc1 = dfstReadPsn( store, firstPsn +i, 1, buffer + (i * si->Geo.B));
               if (rc1 == DFS_PSN_LIMIT)        // rest is out of bounds
               {
                  break;                        // stop, keep previous rc
               }
               else if (rc1 != NO_ERROR)        // some read error, BAD
               {
                  rc = rc1;                     // remember cause
                  badsectors++;
               }
            }
         }
         else
         {
            badsectors = nrs;
         }
         break;
   }
   if (bads != NULL)
   {
      *bads = badsectors;
   }
   TRACES(( "Read %u sectors from %llx, %u BAD ones\n", nrs, lsn, badsectors));
   RETURN (rc);
}                                               // end 'dfstBadSectorRead'
/*---------------------------------------------------------------------------                    */
#endif


/*****************************************************************************/
// Get the PSN of the last read sector (last of group if multiple)
/*****************************************************************************/
ULN64 dfstLastPsn                               // RET   Physical sector nr
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   DFSTOREINFO        *si = &(sti[store]);

   return( si->LastPSN);
}                                               // end 'dfstLastPsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read logical sector(s) using Logical sector nr (LSN) and nr of sectors
/*****************************************************************************/
ULONG dfstReadLsn
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    logical sector nr
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // OUT   Sector buffer
)
{
   return dfstReadPsn( store, dfstLSN2Psn(store, lsn), nrs, buffer);
}                                               // end 'dfstReadLsn'
/*---------------------------------------------------------------------------*/


#if !defined (OEMSB)
/*****************************************************************************/
// Write logical sectors using LSN and nr of sectors, skipping bad sectors
// Allows clone-MERGE for large buffers with some of them bad (0xFE pattern)
/*****************************************************************************/
ULONG dfstBadSectorWrite
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    logical sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG               bads,                    // IN    bad sectors in buffer
   BYTE               *buffer                   // OUT   Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // function return code
   ULN64               firstPsn = dfstLSN2Psn(store, lsn);
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               i;

   ENTER();
   TRACES(( "Store:%u lsn:%llx sectors:%u bads:%u\n", store, lsn, nrs, bads));

   if ((nrs == 1) || (bads == 0))               // the common case ...
   {
      rc = dfstWritePsn( store, firstPsn, nrs, buffer);
   }
   else                                         // write sector by sector
   {
      TXTM    status;

      sprintf( status, "Sector: 0x%12.12llx - merge %2u BAD SECTORS! ", lsn, bads);
      txwSetSbviewStatus( status, TXWSCHEME_ERRMSG);

      for (i = 0; (i < nrs) && (rc == NO_ERROR); i++)
      {
         if (TxCrc32( buffer + (i * si->Geo.B), si->Geo.B) != DFS_BADSEC_CRC32)
         {
            rc = dfstWritePsn( store, firstPsn +i, 1, buffer + (i * si->Geo.B));
         }
         else
         {
            TRACES(( "Skipped BAD sector at LSN %llx\n", lsn +i));
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfstBadSectorWrite'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Write logical sector(s) using Logical sector number (LSN) and nr of sectors
/*****************************************************************************/
ULONG dfstWriteLsn
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // IN    Sector buffer
)
{
   return dfstWritePsn( store, dfstLSN2Psn(store, lsn), nrs, buffer);
}                                               // end 'dfstWriteLsn'
/*---------------------------------------------------------------------------*/


#if !defined (OEMSB)
/*****************************************************************************/
// Get and set 'ignore lock-error' status
/*****************************************************************************/
BOOL dfstIgnoreLockErr                          // return previous status
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                ignore                   // IN    Desired ignore status
)
{
   DFSTOREINFO        *si = &(sti[store]);
   BOOL                rc = si->LockIgn;        // function return

   si->LockIgn = ignore;                        // Set desired new status

   return (rc);                                 // return previous status
}                                               // end 'dfstIgnoreLockErr'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Check if write-access to current device is allowed (locked/ignore) and R/W
/*****************************************************************************/
BOOL dfstCheckWriteAccess                       // RET   Writing allowed
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   BOOL                noIgnore                 // IN    Ignoring not allowed
)
{
   BOOL                wa = TRUE;               // return value

   #if !defined (OEMSB)
   DFSTOREINFO        *si = &(sti[store]);

   ENTER();

   if (si->ReadOnly)
   {
      wa = FALSE;                               // no writing allowed for store
   }
   else                                         // allow, based on locking state
   {
      if ((si->LockCnt == 0) && (si->Type != DFST_MEMORY))
      {
         if (dfstLocking( store, DFS_LOCK, FALSE, noIgnore) != NO_ERROR)
         {
            if (noIgnore || (si->LockIgn == FALSE)) // Prompt for permission
            {
               wa = (dfsa->batch || ( TxConfirm( 5121, "Can not lock destination "
                          "for writing (open files ?).  Write anyway ? [Y/N] : ")));
               if (wa)
               {
                  si->LockIgn = TRUE;           // ignore for 2nd time
               }
            }
         }
      }
   }
   #endif
   BRETURN (wa);
}                                               // end 'dfstCheckWriteAccess'
/*---------------------------------------------------------------------------*/
