//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFS Sector Lookup Table and Name-lookup preparation
//
// Author: J. van Wijk
//
// JvW  21-05-97   Initial version, derived from DFSLHPFS
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface

#include <dfstable.h>                           // SLT utility functions


typedef struct s_lsni                           // info-block for LSN array
{
   ULN64               start;                   // start sector of area
   ULN64               size;                    // size of area in sectors
   ULN64               ref;                     // referencing sector
   ULONG               flags;                   // ERROR signal flags
   BYTE                type;                    // type BYTE
   USHORT              info;                    // additional (ref) info
} LSNI;                                         // end of struct "s_lsninfo"

typedef struct s_slt
{
   LSNI               *area;                    // array of LSN info-blocks
   ULONG               size;                    // total entries in array
   ULONG               used;                    // nr of entries in array
   ULN64               todo;                    // sectors todo in total
   ULN64               done;                    // sectors done sofar
   SLT_STATUS          status;                  // valid indicator
} S_SLT;                                        // end of struct "s_slt"

static S_SLT slt =
{
   NULL,                                        // array of LSN info-blocks
   0,                                           // total entries in array
   0,                                           // nr of entries in array
   0,                                           // sectors todo in total
   0,                                           // sectors done sofar
   SLT_EMPTY                                    // valid indicator
};


// Walk SLT, mark alloc-no-link block (0002) and link-no-alloc (0001) blocks
static void dfsSlTMarkAllocErrors
(
   void
);

// Walk SLT, split every alloc-no-link block (0002) into head, bad and tail
static void dfsSlTSplitAllocErrors
(
   void
);

// Validate free-block from SLT and copy to SPACE structure
static BOOL dfsSlTableCopyFree                  // RET   space OK and copied
(
   ULN64               size,                    // IN    maximum needed size
   ULN64               clsize,                  // IN    cluster size
   ULONG               index,                   // IN    index in SLT
   S_SPACE            *extent                   // OUT   one extent structure
);

// Compare S_SPACE elements for qsort
static int dfsSpaceCompare
(
   const void         *one,
   const void         *two
);


// Compare LSNI info elements for qsort
static int dfsSlTableCompare
(
   const void         *one,
   const void         *two
);


/*****************************************************************************/
// Prepare filesystem for name-lookup, using SLT build, Name-cache or nothing
/*****************************************************************************/
ULONG dfsSltPrepareNameLookup                   // ABORTED when <Esc> used
(
   void
)
{
   ULONG               rc = NO_ERROR;
   ENTER();

   TxPrint("\n");                               // create one empty seperator line
   switch (dfsa->FsModeId)                      // Some filesystems need the SLT for file info
   {
      case DFS_FS_E_FAT:
      case DFS_FS_FAT:
      case DFS_FS_JFS:
      case DFS_FS_HFS:
      case DFS_FS_EXT:
         if (dfsa->FsNpBuild != NULL)           // specific NP-cache builder ?
         {
            TxPrint( "Update cache for Filename/Parent-dir lookup (<Esc> aborts), please wait ...\n");
            rc = DFSFNCALL( dfsa->FsNpBuild,0,0,NULL,NULL);
         }
         else                                   // need to build full SLT
         {
            if (dfsSlTableStatus( NULL) != SLT_READY)
            {
               TxPrint( "Building SLT for filename lookup, please wait ...\n");
               dfsa->sltProgress = TRUE;
               dfsSlTableCreate(dfsa->Fsi);     // build SLT now
               TxPrint( "\n\n");                // terminate progress bars + empty line
            }
         }
         TxCancelAbort();                       // might have aborted
         break;

      default:                                  // others don't need the SLT for file info
         break;
   }
   RETURN(rc);
}                                               // end 'dfsSltPrepareNameLookup'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Intialize SLT building for FS-specific phase (progress etc)
/*****************************************************************************/
ULONG dfsSlTableFsInit                          // RET   SLT allocated size
(
   ULN64               todo                     // IN    approx sectors todo
)
{
   ENTER();

   slt.done = 0;
   slt.todo = todo;
   dfsProgressInit( 0, max( todo, 1), 0, "Sector:", "in SLT", DFSP_BARS, 0);

   RETURN (slt.size);
}                                               // end 'dfsSlTableFsInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create sector-lookup-table, calling Filesystem specific functions
/*****************************************************************************/
void dfsSlTableCreate
(
   void               *param                    // IN    parameter to pass on
)
{
   ULONG               rc   = DFS_ALLOC_ERROR;
   ULONG               size = max(dfsa->FsSltRecSize, 0x1000);
   ULONG               si;
   ULONG               us;                      // nr of used entries
   ULN64               lsn;                     // adjacent LSN
   ULN64               start;                   // start LSN of area
   ULN64               next;                    // first LSN after area
   ULONG               nr;
   ULONG               ef = 0;                  // error flag

   ENTER();

   txwSetSbviewStatus( "Check : Creating Sector Lookup Table ...", cSchemeColor);

   TxFreeMem( slt.area);                        // free SLT array
   slt.size = size;

   TRACES(( "SLT entry size: %u   entries: %u\n", sizeof(LSNI), size));

   for (nr = 0; (nr < 88) && (slt.size > 100); nr++, (slt.size = ((slt.size * 6)/7)))
   {
      TRACES(("Sizeof LSNI: %u, slt.size = %u\n", (ULONG) sizeof(LSNI), slt.size));
      if ((slt.area = TxAlloc( slt.size, sizeof(LSNI))) != NULL)
      {
         break;
      }
   }
   if (slt.area != NULL)
   {
      slt.status = SLT_UPDATE;                  // SLT being updated

      TRACES(( "DirtyStatus now: %u\n", dfsa->FsDirtyStatus));

      if (dfsa->FsDirtyStatus == DFSTAT_DIRTY)  // filesystem seems to be
      {                                         // mounted or is dirty
         ef = EF_FS_DIRTYMOUNT;
      }
      dfsSlTableAdd( LSN_BOOTR, 0x01, ST_BOOTR, L64_NULL, 0, ef);

      TRACES(("added bootsector, FS-specific now in low priority ...\n"));
      TxThreadPrioMin();                        // Lowest priority thread

      rc = DFSFNCALL(dfsa->FsSltBuild,0,0,NULL,param);

      dfsProgressTerm();                        // finish BAR and clear status
      TRACES(( "\nDone by FS : 0x%llx todo:0x%llx\n", slt.done, slt.todo));
      slt.todo = 0;                             // stop FS-specific progress

      txwAllowUserStatusMessages( TRUE);

      txwSetSbviewStatus( "Check : Sorting dir+files info ...", cSchemeColor);
      qsort( slt.area, (size_t) slt.used, sizeof(LSNI), dfsSlTableCompare);
      TRACES(("1st Sort finished, add freespace area's\n"));
      if (rc == NO_ERROR)
      {                                         // add Free-space area's
         for ( si = 0, us = slt.used, lsn = 0;
              (si < us) && (slt.status == SLT_UPDATE);
               si++)
         {
            start = slt.area[si].start;
            next  = slt.area[si].size + start;
            if (start > lsn)
            {
               dfsSlTableAdd( lsn, start - lsn, ST_FREES, L64_NULL, 0, 0);
            }
            if (next > lsn)
            {
               lsn = next;
            }
         }
         if (slt.status == SLT_UPDATE)          // no error conditions
         {
            txwSetSbviewStatus( "Check : Sorting incl freespace ...", cSchemeColor);
            qsort( slt.area, (size_t) slt.used, sizeof(LSNI), dfsSlTableCompare);
            TRACES(("2nd Sort finished, check allocation\n"));

            if ((TxaOption('c')) || (TxaOptSet('m'))) // explicit check, or show errors
            {
               dfsSlTMarkAllocErrors();

               txwSetSbviewStatus( "Check : Analysing allocation errors ...", cSchemeColor);

               dfsSlTSplitAllocErrors();        // split alloc-no-link blocks

               qsort( slt.area, (size_t) slt.used, sizeof(LSNI), dfsSlTableCompare);
               TRACES(("3rd Sort finished, done\n"));
            }
            TxCancelAbort();                    // might have canceled checking
            slt.status = SLT_READY;             // SLT valid and ready
            TRACES(("SLT ready\n"));
         }
      }
      if (slt.status == SLT_STOPPING)           // aborting SLT thread
      {
         slt.status = SLT_EMPTY;                // signal done
      }
      else
      {
         slt.status = SLT_READY;                // SLT ready for use
      }
      TRACES(("SLT build ending, entries: %u, status: %u\n", slt.used, slt.status));
      txwSetSbviewStatus( "Check : SLT-build finished", cSchemeColor);
      #if defined (HAVETHREADS)
         txwSignalEventHook( DFS_EHK_SLTBUILD, NULL);
      #endif
      txwAllowUserStatusMessages( FALSE);       // resume default status text
   }
   else
   {
      TxPrint( "\nError allocating SLT buffer memory\n");
   }
   VRETURN();
}                                               // end 'dfsSlTableCreate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Walk SLT, mark alloc-no-link block (0002) and link-no-alloc (0001) blocks
/*****************************************************************************/
static void dfsSlTMarkAllocErrors
(
   void
)
{
   ULONG               ClustSize = dfsGetClusterSize();
   ULN64               storeSize = dfsGetLogicalSize();
   ULN64               scTodo = 0;              // sector-checks todo
   ULN64               scDone = 0;              // sector-checks done
   ULONG               si;
   ULN64               lsn;                     // adjacent LSN
   ULN64               start;                   // adjacent LSN
   ULN64               nr;
   BYTE                st;                      // sector type wanted

   ENTER();

   if (dfsa->sltProgress)
   {
      TxPrint(" Checking allocation for all %u areas (Use <Esc> to skip the allocation check)\n", slt.used);
      for ( si = 0; (si < slt.used); si++)
      {
         scTodo += slt.area[si].size;
      }
      dfsProgressInit( 0, scTodo, 0, "Sector:", "ALLOC checked", DFSP_BARS, 0);
   }
   for (si = 0; (si < slt.used) && (slt.status == SLT_UPDATE) && !TxAbort(); si++)
   {
      st    = slt.area[si].type;
      nr    = slt.area[si].size;
      start = slt.area[si].start;

      TRACES(("Check alloc entry %9lu, start:0x%llX size:0x%llx, ClustSize: %u\n", si, start, nr, ClustSize));

      if ((st != ST_FINAL) && (nr != DFS64MAX) && (ClustSize != 0))
      {
         for ( lsn  = start;      (nr > 0) && (slt.status == SLT_UPDATE) && !TxAbort() &&
              (lsn  < storeSize);
               lsn += ClustSize)
         {
            if (DFSFNCALL(dfsa->FsLsnAllocated,lsn,0,NULL,NULL))
            {
               if (st == ST_FREES)
               {
                  slt.area[si].flags |= EF_ALLOC_NO_LINK;
                  TRACES(("ALLOC_NO_LINK LSN:%llX on area:%llx size:%llx\n", lsn, start, slt.area[si].size));
               }
            }
            else
            {
               if (st != ST_FREES)
               {
                  slt.area[si].flags |= EF_LINK_NO_ALLOC;
                  TRACES(("LINK_NO_ALLOC LSN:%llX on area:%llx size:%llx\n", lsn, start, slt.area[si].size));
               }
            }
            scDone += ClustSize;
            if (TxTmrTimerExpired( dfsa->statusTimer))
            {
               dfsProgressShow( scDone, 0, si + 1, "#:");
               dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
            }
            nr = (nr > ClustSize) ? nr - ClustSize : 0;
         }
      }
   }
   if (dfsa->sltProgress)
   {
      dfsProgressTerm();                        // finish BAR and clear status
      if (TxAbort())                            // aborting alloc checking
      {
         dfsX10( "Abort! Checked ALLOC upto : ",  scDone, CBG, ", ");
         dfsX10( "last used sector is : ", dfsa->FsTruncPoint, CBR, "\n");
      }
   }
   VRETURN();
}                                               // end 'dfsSlTMarkAllocErrors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Walk SLT, split every alloc-no-link block (0002) into head, bad and tail
/*****************************************************************************/
static void dfsSlTSplitAllocErrors
(
   void
)
{
   ULONG               ClustSize = dfsGetClusterSize();
   ULN64               storeSize = dfsGetLogicalSize();
   ULONG               si;
   ULN64               lsn;                     // adjacent LSN
   ULN64               nr;
   BYTE                st;                      // sector type wanted

   ENTER();

   txwSetSbviewStatus( "Check : Analysing allocation errors ...", cSchemeColor);

   for ( si = 0;
        (si < slt.used) && (slt.status == SLT_UPDATE) && !TxAbort();
         si++)
   {
      st = slt.area[si].type;
      nr = slt.area[si].size;

      if ((slt.area[si].type == ST_FREES) && (slt.area[si].flags & EF_ALLOC_NO_LINK))
      {
         ULN64         headstart = slt.area[si].start;
         ULN64         bad_start;

         slt.area[si].flags &= ~EF_ALLOC_NO_LINK; // reset error on tail-block

         TRACES(("Split entry %6u, 0x%llX size:0x%llx\n", si, headstart, nr));
         for ( lsn = headstart; (slt.status == SLT_UPDATE) && !TxAbort() &&
              (lsn < storeSize) && (nr) && (nr != DFS64MAX);
               lsn += ClustSize)
         {
            if (DFSFNCALL(dfsa->FsLsnAllocated,lsn,0,NULL,NULL))
            {
               if (lsn > headstart)            // don't add empty elements
               {
                  //- add new clean-free element to table (head)
                  dfsSlTableAdd( headstart, lsn - headstart, ST_FREES, L64_NULL, 0, 0);
               }
               bad_start = lsn;                 // start of allocated area
               do
               {                                // find end of bad area
                  lsn++;                        // check next ones ...
                  nr--;
               } while (nr && DFSFNCALL(dfsa->FsLsnAllocated,lsn,0,NULL,NULL));

               //- add new alloc-error element to table (bad)
               dfsSlTableAdd( bad_start, lsn - bad_start, ST_UDATA, L64_NULL, 0, EF_ALLOC_NO_LINK);

               headstart = lsn;                 // start next clean area (tail)
            }
            nr = (nr > ClustSize) ? nr - ClustSize : 0;
         }
      }
   }
}                                               // end 'dfsSlTSplitAllocErrors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset sector-lookup-table to empty state
/*****************************************************************************/
void dfsSlTableReset
(
   void
)
{
   ENTER();
   if (slt.status == SLT_UPDATE)
   {
      TRACES(("Stopping running SLT thread ...\n"));
      slt.status = SLT_STOPPING;
      while (slt.status == SLT_STOPPING)
      {
         TxSleep( 500);                         // wait until thread has
      }                                         // stopped updating
   }
   slt.used = 0;                                // empty SLT array
   slt.status = SLT_EMPTY;                      // SLT invalid
   if (slt.area != NULL)
   {
      TxFreeMem(slt.area);
      slt.area = NULL;
   }
   VRETURN();
}                                               // end 'dfsSlTableReset'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// return SLT status information
/*****************************************************************************/
SLT_STATUS dfsSlTableStatus
(
   ULONG              *entries                  // OUT   nr of entries
)                                               //       or NULL
{
   ENTER();

   if (entries)
   {
      *entries = slt.used;
   }
   RETURN (slt.status);
}                                               // end 'dfsSlTableStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show status of SLT array, for any Filesystem, max 20 characters
/*****************************************************************************/
char *dfsSlTableStatusString                    // RET   output string
(
   char               *txt                      // OUT   SLT status string
)
{
   if (slt.status != SLT_EMPTY)
   {
      sprintf(txt, "SLT: %u", slt.used);
      switch (slt.status)
      {
         case SLT_UPDATE:   strcat(txt, " UPD"); break;
         case SLT_OVERFLOW: strcat(txt, " ERR"); break;
         default:                                break;
      }
   }
   else
   {
      strcpy( txt, "");
   }
   return(txt);
}                                               // end 'dfsSlTableStatusString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// verbose status of SLT array, for any Filesystem, max 66 characters
/*****************************************************************************/
char *dfsGetSltStatusString                     // RET   output string
(
   char               *txt,                     // OUT   SLT status string
   char               *message                  // IN    Message for index == 0
)                                               //       or NULL (max 16 chars)
{
   ENTER()

   if ((slt.status != SLT_READY) && (message))  // alternate message available
   {
      sprintf( txt, "SLT Areas: %-10u %-16.16s  status: ", slt.used, message);
   }
   else
   {
      sprintf( txt, "SLT Areas: %-10u index: %-9u  status: ", slt.used, nav.index);
   }
   switch (slt.status)
   {
      case SLT_READY    : strcat(txt, "Ready for action"); break;
      case SLT_UPDATE   : strcat(txt, "Being built now");  break;
      case SLT_OVERFLOW : strcat(txt, "Table Overflow!");  break;
      case SLT_STOPPING : strcat(txt, "Aborting build!");  break;
      case SLT_EMPTY    : strcat(txt, "To be built ...");  break;
      default:            strcat(txt, "Unknown!");         break;
   }
   RETURN(txt);
}                                               // end 'dfsGetSltStatusString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show all info in  SLT array
/*****************************************************************************/
ULONG dfsSlTableDisplay                         // RET   next index in array
(
   ULONG               start,                   // IN    start-index for dump
   ULONG               size,                    // IN    Nr of entries to dump
   BYTE                stype,                   // IN    type of sector(s)
   ULONG               ef_mask,                 // IN    error selection mask
   BOOL                verbose,                 // IN    verbose, add file-info
   BOOL                oneliner,                // IN    verbose on same line
   ULONG              *errorcount               // OUT   detected errors
)
{
   ULONG               i;
   BYTE                ct;                      // current type of sector
   BOOL                match;
   ULONG               errors = 0;              // total errors
   ULONG               listed = 0;              // listed errors
   ULONG               lost_areas   = 0;        // Areas allocated but not linked
   ULONG               lost_sectors = 0;        // lost areas total size
   ULONG               link_areas   = 0;        // Areas linked but NOT allocated
   ULONG               link_sectors = 0;        // link areas total size
   ULONG               flags;
   TXLN                text;                    // collecting text buffer
   TXTM                tbuf;                    // intermediate text buffer

   ENTER();
   TRARGS(("start: %u, size: %u, stype: '%c'\n", start, size, stype));
   if (start == 0)
   {
      TxPrint("\nDisplay Sector Lookup Table in %s format.\n", (verbose) ? "verbose"  : "compact");
      TxPrint("%s%-10u%s of %10u ", CBY, slt.used, CNN, slt.size);
   }
   else
   {
      TxPrint( "%25.25s", "");
   }
   TxPrint("at %s%-10u%s   Showing: %s", CBW, start, CNN, CBM);

   if (verbose && (ef_mask != 0))
   {
      TxPrint("Errors matching: ");
   }
   else
   {
      dfsShowSectorType(stype);
   }
   if (ef_mask != 0)
   {
      TxPrint("%s0x%s%8.8X%s", CNZ, CBR, ef_mask, CNN);
   }
   TxPrint("\n");
   TxPrint("Sector-nr  Reference +info Sectors in area, size    Containing         %s\n",
              (ef_mask) ? "" : (oneliner) ? "Error or Path+Filename" : "Error codes");
   TxPrint("========== =============== ======================== =================  %s\n",
              (ef_mask) ? "" : (oneliner) ? "======================" : "===========");
   for (i = start; i < slt.used && !TxAbort() && size; i++)
   {
      if ((flags = slt.area[i].flags) != 0)     // count all errors for stats
      {
         errors++;
      }
      if (ef_mask && ((flags & ef_mask) == 0))
      {
         match = FALSE;
      }
      else
      {
         ct = slt.area[i].type;
         switch (stype)
         {
            case ST_WILDC: match = TRUE;                        break;
            case ST_KNOWN: match = ( ct != ST_UDATA);           break;
            default:
               match = ((BYTE)(ct & ~ST__INFO) == stype);
               break;
         }
         TRACES(( "ct: %2.2hx  stype: %2.2hx  match: %s\n",
                   ct,         stype,        (match) ? "YES" : "NO"));
      }
      if (match)
      {
         strcpy( text, "");
         dfstrX10( text, "",  slt.area[i].start, CBG, " ");
         if (slt.area[i].ref == L64_NULL)
         {
            strcat( text, "none           ");
         }
         else
         {
            dfstrX10( text, "",                 slt.area[i].ref, CBC, "");
            if (slt.area[i].info & DFSSNINFO)
            {
               sprintf( tbuf, " %s%4.4hx%s", CNC, DFSSNIGET(slt.area[i].info), CNN);
               strcat( text, tbuf);
            }
            else                                // no extra info available
            {
               strcat( text, "     ");
            }
         }
         dfstrSz64( text, " ",  slt.area[i].size, " ");
         dfsSectorTypeAsAscii((BYTE) (ct & ~ST__INFO), tbuf);
         strcat( text, tbuf);
         if (flags != 0)                        // this entry has an error
         {
            sprintf( tbuf, "  %s%8.8X%s", CBR, flags, CNN);
            strcat( text, tbuf);
            listed++;

            if (flags & EF_ALLOC_NO_LINK)     // link areas summary statistics
            {
               lost_areas++;
               lost_sectors += slt.area[i].size;

            }
            if (flags & EF_LINK_NO_ALLOC)     // link areas summary statistics
            {
               link_areas++;
               link_sectors += slt.area[i].size;
            }
         }
         if (verbose)
         {
            BOOL       haveNewline = FALSE;

            TxPrint( "%s%s", text, (oneliner) ? " " : "\n");
            strcpy(          text, (oneliner) ? ":" : "     LSN info: ");
            if (DFSFNCALL( dfsa->FsFileInformation, slt.area[i].start, 0, "*", text) == NO_ERROR)
            {
               haveNewline = TRUE;
            }
            if ((slt.area[i].ref != L64_NULL) && // info on referer available and
                ((!oneliner) || (!haveNewline))) // multi-line or no direct info
            {
               strcpy( text, (oneliner) ? ":" : "     Ref info: ");
               if (DFSFNCALL( dfsa->FsFileInformation, slt.area[i].ref, slt.area[i].info, "*", text) == NO_ERROR)
               {
                  haveNewline = TRUE;
               }
            }
            if (oneliner && !haveNewline)
            {
               TxPrint( "\n");
            }
            if (!oneliner || ef_mask)           // verbose or errors wanted
            {
               sprintf( text, "       - ");
               if (flags & EF_LINK_NO_ALLOC)
               {
                  TxPrint("%sArea in FS-administration but allocation not set\n", text);
               }
               if (flags & EF_ALLOC_NO_LINK)
               {
                  TxPrint("%sAllocation set but area not in FS-administration\n", text);
               }
               if (flags & EF_FS_DIRTYMOUNT)
               {
                  TxPrint("%sFilesystem is marked DIRTY or MOUNTED (in use)\n", text);
               }
               DFSFNCALL( dfsa->FsDisplayError, flags, 0, text, NULL);
            }
         }
         else
         {
            TxPrint( "%s\n", text);
         }
         TRACES(("type: %2.2x, size: 0x%llX\n", ct, slt.area[i].size));
         size--;
      }
   }
   if (i >= slt.used)
   {
      if (verbose)
      {
         TxPrint("\nTotal errors/warnings detected:   %u", errors);
         if (listed < errors)
         {
            TxPrint(",  listed %u", listed);
         }
         TxPrint("\n");
         if ((ef_mask & EF_ALLOC_NO_LINK) && (lost_areas != 0))
         {
            sprintf( text, "Areas allocated but not linked:   %u", lost_areas);
            dfstrSz64( text, ", size ",  lost_sectors, "\n");
            TxPrint(text);
         }
         if ((ef_mask & EF_LINK_NO_ALLOC) && (link_areas != 0))
         {
            sprintf( text, "Areas linked but not allocated:   %u", link_areas);
            dfstrSz64( text, "size ",  link_sectors, "\n");
            TxPrint(text);
         }
      }
      i = 0;                                    // wrap next index
   }
   *errorcount = listed;                        // report only listed errors
   RETURN(i);
}                                               // end 'dfsSlTableDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find sector in SLT array
/*****************************************************************************/
BOOL dfsSlTableFind
(
   ULN64               snr,                     // IN    sectornumber to find
   ULN64              *fns,                     // OUT   reference LSN  or NULL
   USHORT             *info,                    // OUT   info, FATentry or NULL
   BYTE               *stp,                     // OUT   type of sector or NULL
   ULONG              *slh                      // OUT   SLT index
)
{
   BOOL                found = FALSE;
   LSNI               *ce;                      // current element
   LSNI               *be = NULL;               // best element
   ULONG               th = 0;                  // table handle

   ENTER();

   if ((slt.area != NULL) && (snr != L64_NULL))
   {
      for ( ce = slt.area,            th = 0,     be = NULL;
           (ce->start <= snr) && (th < slt.used);
            ce++,                 th++)
      {
         if (ce->start + ce->size > snr)        // hit
         {
            if (be == NULL || (ce->size < be->size))
            {
               be = ce;
            }
         }
      }
   }
   if (be != NULL)                              // found
   {
      if (fns)
      {
         *fns = be->ref;
      }
      if (info)
      {
         *info = be->info;
      }
      if (stp)
      {
         *stp = be->type;
      }
      found = TRUE;
   }
   *slh = (LONG) th -1;
   BRETURN(found);
}                                               // end 'dfsSlTableFind'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search for free disk-space in the SLT, using a 'best-fit' algorithm
/*****************************************************************************/
BOOL dfsSlTableSearchFree                       // RET   requested space found
(
   ULN64               size,                    // IN    number of sectors
   ULN64               clsize,                  // IN    minimal extent size
   ULONG               maxext,                  // IN    maximal nr of extents
   ULONG              *extents,                 // OUT   nr of extents
   S_SPACE            *space                    // OUT   extent array
)
{
   BOOL                found = FALSE;
   ULN64               rest;
   ULONG               i;
   S_SPACE            *le = &(space[maxext-1]); // last SPACE element

   ENTER();
   TRARGS(("Size: 0x%llx, clsize: 0x%llx, maxext: %u\n", size, clsize, maxext));

   memset( space, 0, (size_t) (maxext * sizeof(S_SPACE))); // initialize array

   for (i = 0; i < slt.used && !TxAbort(); i++)
   {
      if ((slt.area[i].type == ST_FREES) &&     // Freespace block found
          (slt.area[i].size >= clsize) )
      {
         TRACES(( "Candidate at:0x%llx size 0x%llx\n", slt.area[i].start, slt.area[i].size));
         if (space->size >= size)               // having a single extent
         {
            if ((slt.area[i].size >= size) &&   // better fit, replace
                (slt.area[i].size < space->size))
            {
               TRACES(( "Replace only chunk with this better fit\n"));
               dfsSlTableCopyFree( size, clsize, i, space);
            }
         }
         else                                   // new candidate to add
         {
            if (slt.area[i].size > le->size)    // larger than smallest
            {
               TRACES(( "Replace last chunk, then sort on size ...\n"));
               if (dfsSlTableCopyFree( size, clsize, i, le))
               {
                  qsort( space, (size_t) maxext, sizeof(S_SPACE), dfsSpaceCompare);
               }
               TRINIT(70);
                  dfsSspaceDisplay( SD_DEFAULT, maxext, space);
               TREXIT();
            }
            else
            {
               TRACES(( "Worse than any chunk sofar, reject ...\n"));
            }
         }
      }
   }
   for ( i = 0, rest = size;                    // check if extents have enough
        (i < maxext) && !found;                 // space and set exact size
         i++)
   {                                            // cluster aligned extent size
      ULONG            this = (space[i].size - (space[i].size % clsize));

      if (this >= rest)
      {
         this     = rest;                       // make it an exact fit
         *extents = i+1;
         found    = TRUE;
      }
      else
      {
         rest    -= this;                       // use whole aligned extent
      }
      space[i].size = this;                     // exact needed size in extent
   }
   TRACES(("Used Extents: %u, from total SPACE:\n", *extents));
   TRINIT(70);
      dfsSspaceDisplay( SD_DEFAULT, maxext, space);
   TREXIT();
   BRETURN(found);
}                                               // end 'dfsSlTableSearchFree'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Validate free-block from SLT and copy to SPACE structure
/*****************************************************************************/
static BOOL dfsSlTableCopyFree                  // RET   space OK and copied
(
   ULN64               size,                    // IN    maximum needed size
   ULN64               clsize,                  // IN    cluster size
   ULONG               index,                   // IN    index in SLT
   S_SPACE            *extent                   // OUT   extent structure
)
{
   BOOL                spok = TRUE;
   ULN64               lsn  = slt.area[index].start;
   ULONG               nr;

   ENTER();
   TRACES(( "size:0x%llx extent *:%8.8x\nSLT[%u] start:0x%llx  size:0x%llx\n",
             size, extent, index, slt.area[index].start, slt.area[index].size));

   for ( nr  = 0;                               // double-check allocation
        (nr  < min(size, slt.area[index].size)) && spok; // check upto needed size
         nr += clsize, lsn += clsize)
   {
      if (DFSFNCALL(dfsa->FsLsnAllocated,lsn,0,NULL,NULL))
      {
         TRACES(("Failing free-check LSN: 0x%llX\n", lsn));
         spok = FALSE;
      }
   }
   if (spok)
   {
      extent->size  = slt.area[index].size;
      extent->start = slt.area[index].start;
      extent->index = index;
   }
   TRACES(( "extent start:0x%llx size:0x%llx\n"));
   BRETURN(spok);
}                                               // end 'dfsSlTableCopyFree'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate specified free disk-space in the SLT and FS allocation-structure
/*****************************************************************************/
void dfsSlTableAllocate                         // RET   requested space found
(
   ULN64               size,                    // IN    number of sectors
   ULN64               clsize,                  // IN    minimal extent size
   ULONG               extents,                 // IN    nr of extents
   BYTE                type,                    // IN    type of sectors
   ULN64               ref,                     // IN    reference LSN for SLT
   USHORT              info,                    // IN    related INFO, FS specific
   S_SPACE            *space                    // IN    extent array
)
{
   ULONG               i;                       // index in extent array
   ULONG               index;                   // index in extent lsn's
   ULONG               entry;                   // index in SLT
   ULN64               this;                    // size for this extent
   ULN64               rest;                    // remaining size

   ENTER();

   TRACES(( "size:0x%llx  clsize:0x%llx  extents:%u\n", size, clsize, extents));
   TRINIT(70);
      dfsSspaceDisplay( SD_DEFAULT, extents, space);
   TREXIT();
   for (i = 0, rest = size; i < extents; i++)   // walk specified extents
   {
      if (space[i].size != 0)                   // sanity check
      {
         entry = space[i].index;                // corresponding SLT entry
         if (rest < slt.area[entry].size)       // freespace larger than needed
         {
            this = rest;                        // only use needed part
            dfsSlTableAdd( slt.area[entry].start + this, // add leftover to table-end
                          (slt.area[entry].size  - this),
                           ST_FREES, L64_NULL, info, 0);
         }
         else                                   // exact match, or more to do
         {
            this = (slt.area[entry].size - (slt.area[entry].size % clsize));
            if (this < slt.area[entry].size)    // not whole extent used
            {
               dfsSlTableAdd( slt.area[entry].start + this, // add leftover to table-end
                             (slt.area[entry].size  - this),
                              ST_FREES, L64_NULL, info, 0);
            }
            rest -= this;                       // size left to do
         }
         slt.area[entry].size = this;           // set the allocated block
         slt.area[entry].type = type;           // properties in the SLT
         slt.area[entry].ref  = ref;

         //- set allocation bits in FS for this chunk, use cluster sized steps
         for (index = 0; index < this; index += clsize)
         {
            DFSFNCALL( dfsa->FsLsnSetAlloc,
                       space[i].start + index,
                       0, (char *) TRUE, NULL);
         }
      }
   }
   qsort( slt.area, (size_t) slt.used, sizeof(LSNI), dfsSlTableCompare);
   VRETURN();
}                                               // end 'dfsSlTableAllocate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add list of 32bit sector-numbers plus info to SLT array
/*****************************************************************************/
void dfsSlTableListSn32
(
   ULONG              *list,                    // IN    ptr to list of LSN's
   ULONG               items,                   // IN    #LSN's in list
   ULONG               sects,                   // IN    #sectors per LSN
   BYTE                stype,                   // IN    type of sectors
   ULONG               ref,                     // IN    referencing LSN
   USHORT              info,                    // IN    related INFO
   ULONG               flags                    // IN    ERROR signal flags
)
{
   ULONG              *lsn;                     // LSN item in list
   ULONG               l;                       // index in list
   ULONG               start;                   // start LSN for chunk
   ULONG               last;                    // last  LSN added
   ULONG               size;                    // current chunk size

   ENTER();
   lsn    = list;
   start  = *lsn;
   last   = start - sects;
   size   = 0;
   for (l = 0; l < items; l++, lsn++)
   {
      if (*lsn != last + sects)                 // not consecutive
      {
         dfsSlTableAdd(start, size + sects, stype, ref, info, flags);
         start = *lsn;
         size  = 0;
      }
      else
      {
         size += sects;
      }
      last = *lsn;
   }
   if (size > 0)                                // final (or only) chunk
   {
      dfsSlTableAdd(start, size, stype, ref, info, flags);
   }
   VRETURN();
}                                               // end 'dfsSlTableListSn32'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add info-entry to  SLT array
/*****************************************************************************/
ULONG dfsSlTableAdd
(
   ULN64               start,                   // IN    start LSN of area
   ULN64               size,                    // IN    nr of sectors in area
   BYTE                type,                    // IN    sector-type
   ULN64               ref,                     // IN    related LSN (fnode)
   USHORT              info,                    // IN    related INFO, FS specific
   ULONG               flags                    // IN    ERROR signal flags
)
{
   #if defined (HAVETHREADS) || defined (DUMP)
      static TXTM         status;
   #endif
   #if defined (HAVETHREADS)
      static TXTIMER   nextSignal = TXTMR_INIT_EXPIRED;
   #endif

   if (slt.todo)
   {
      slt.done += size;                         // sectors done sofar
      dfsProgressShow( slt.done, 0, slt.used, "#:");
   }
   #if defined (HAVETHREADS)
   if (slt.todo && TxTmrTimerExpired( nextSignal))
   {
      txwSignalEventHook( DFS_EHK_SLTBUILD, status);
      nextSignal = TxTmrSetTimer( TMR_SEC( 1)); // just once per second to
   }                                            // allow user-input too :-)
   #endif

   #if defined (DUMP)
   if (TxTrLevel >= 70)                         // along with TRACES
   {
      dfsSectorTypeAsAscii((BYTE)(type & ~ST__INFO), status);
      TRACES(("slt:%8u: 0x0%llX, R:0x0%llX I:%4.4hx T:%2.2hx = %s\n",
               slt.used, start, ref, info, type, status));
   }
   #endif
   if ((slt.used +1) < slt.size)                // don't overflow
   {
      slt.area[slt.used].start = start;
      slt.area[slt.used].size  = size;
      slt.area[slt.used].type  = type;
      slt.area[slt.used].ref   = ref;
      slt.area[slt.used].info  = info;
      slt.area[slt.used].flags = flags;
      slt.used++;
   }
   else
   {
      slt.status = SLT_OVERFLOW;
   }
   slt.area[slt.used].start = L64_NULL;         // sentinel value on next elem

   return( slt.used);
}                                               // end 'dfsSlTableAdd'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add info-entries for S_SPACE structure to SLT array
/*****************************************************************************/
ULONG dfsSlTableSpaceAdd
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   BYTE                type,                    // IN    sector-type
   ULN64               ref,                     // IN    related LSN (fnode)
   USHORT              info,                    // IN    related INFO, FS-specific
   ULONG               flags                    // IN    ERROR signal flags
)
{
   ULONG               chunk;

   ENTER();
   TRARGS(("%u chunks at:%8.8x type:%2.2hx='%c' ref:0x0%llx info:%4.4hx flags:%8.8x\n",
            chunks, space, type, (type & ~ST__INFO), ref, info, flags));

   if (space != NULL)
   {
      TRACES(( "start adding %u chunks\n", chunks));
      TRINIT(70);
         dfsSspaceDisplay( SD_DEFAULT, chunks, space);
      TREXIT();
      for (chunk = 0; chunk < chunks; chunk++)
      {
         TRACES(( "adding chunk %u of %u\n", chunk, chunks));
         if (space[chunk].start != L64_NULL)    // not a sparse entry ?
         {
            TRACES(( "start:0x0%llx size:0x0%llx\n",
                          space[chunk].start, space[chunk].size));

            dfsSlTableAdd(space[chunk].start, space[chunk].size,
                  (BYTE)((type != ST_SINFO) ?   // to be refined for DFSSPACE
                  (BYTE ) type : (BYTE) space[chunk].info),
                          ref, info, flags);
         }
      }
   }
   RETURN( slt.used);
}                                               // end 'dfsSlTableSpaceAdd'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare LSNI info elements for qsort, ascending on start location
/*****************************************************************************/
static int dfsSlTableCompare
(
   const void         *one,
   const void         *two
)
{
   int                 rc = 0;

   if      ((((LSNI *) one)->start) < (((LSNI *) two)->start))
   {
      rc = -1;
   }
   else if ((((LSNI *) one)->start) > (((LSNI *) two)->start))
   {
      rc = +1;
   }
   return(rc);
}                                               // end 'dfsSlTableCompare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare S_SPACE elements for qsort, DESCENDING on size
/*****************************************************************************/
static int dfsSpaceCompare
(
   const void         *one,
   const void         *two
)
{
   int                 rc = 0;

   if      ((((S_SPACE *) one)->size) < (((S_SPACE *) two)->size))
   {
      rc = +1;
   }
   else if ((((S_SPACE *) one)->size) > (((S_SPACE *) two)->size))
   {
      rc = -1;
   }
   return(rc);
}                                               // end 'dfsSpaceCompare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add selected sectors from SLT areas to the sector list
/*****************************************************************************/
ULONG dfsSltRange2List                          // RET   result
(
   ULN64               asize,                   // IN    #sectors per area
   ULN64               start,                   // IN    start sector
   ULN64               rsize,                   // IN    size of the range
   ULONG               mask,                    // IN    error selection mask
   BOOL                refsect,                 // IN    add reference sectors
   BYTE                stype                    // IN    type selection
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               i;
   BYTE                ct;                      // current type of sector
   ULONG               flags;
   BOOL                match;
   ULN64               sn;

   ENTER();
   TRACES(( "asize:%llu start:0x%llx rsize:0x%llx mask:0x%8.8x stype:%s\n",
             asize,     start,       rsize,       mask,        stype));

   for (i = start; i < slt.used; i++)
   {
      sn = slt.area[i].start;
      if ((sn >= start) && (sn < (start + rsize)))
      {
         flags = slt.area[i].flags;
         if (mask && ((flags & mask) == 0))
         {
            match = FALSE;
         }
         else
         {
            ct = slt.area[i].type;
            switch (stype)
            {
               case ST_WILDC: match = TRUE;                        break;
               case ST_KNOWN: match = ( ct != ST_UDATA);           break;
               default:
                  match = ((BYTE)(ct & ~ST__INFO) == stype);
                  break;
            }
            TRACES(( "ct: %2.2hx  stype: %2.2hx  match: %s\n",
                      ct,         stype,        (match) ? "YES" : "NO"));
         }
         if (match)
         {
            ULONG      a;

            if (refsect)                        // replace target by reference
            {
               sn = slt.area[i].ref;
            }
            for (a = 0; a < asize; a++)
            {
               dfsAdd2SectorList( sn + a);      // add to list
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsSltRange2List'
/*---------------------------------------------------------------------------*/


