//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee file-space allocation structure definitions and handling functions
//
// Author: J. van Wijk
//
// JvW  01-04-2017 Updated to use 64-bit sectornumbers
// JvW  08-08-2005 Initial version, split off from dfsutil
//
#ifndef    DFSSPACE
   #define DFSSPACE

//- Special value to identify sparse extents in SSPACE, and sparse LSN from Rsn2Lsn
#define LSN_SPARSE     0xfffffffffffffffaULL

// Basic space structure element, used in arrays of #extents elements
typedef struct s_space                          // disk space allocation
{
   ULN64               start;                   // LSN of first sector
   ULN64               size;                    // number of sectors
   ULONG               index;                   // SLT entry or exact size
   ULONG               info;                    // additional info (data type)
} S_SPACE;                                      // end of struct "s_space"


#define DFSXA_NONE     0x0000                   // xattr not present
#define DFSXA_UNIX     0xFEA0                   // xattr Unix, Linux, OSX
#define DFSXA_FEA2     0xFEA2                   // xattr as OS/2 FEA2-list ptr

// integrated self-contained space structure, includes additional information
typedef struct dfsispace
{
   ULONG               chunks;                  // number of extents  (0 if resident)
   S_SPACE            *space;                   // space entries[] (data if resident)
   ULN64               byteSize;                // bytesize (space-array or resident)
   ULONG               allocated;               // #extents in SPACE tx-alloc or 0
   ULONG               clsize;                  // clustersize, if any
   ULN64               meta;                    // meta info, like MFT-nr, Inode etc
   USHORT              spid;                    // further identification, meta-ID
   void               *xattrs;                  // extended attributes  (OS2-EA or ...)
   ULONG               xasize;                  // XATTR size in bytes
   USHORT              xatype;                  // XATTR identification (OS2-EA or ...)
   USHORT              offset;                  // offset (example: FS resident data)
} DFSISPACE;                                    // end of struct "dfsispace"



//- Used by S_SPACE and SLT logic, to get more info (like NTFS attribute type)
#define ST_SINFO  ((BYTE) 0xff)                 //  I* Use type in S_SPACE info

//- Space-display
#define SD_DEFAULT   0x0000                     // no special options
#define SD_TOPLINE   0x0001                     // display header lines
#define SD_BOTLINE   0x0002                     // display bottom lines > 20
#define SD_NAVDOWN   0x0004                     // assign 1st to nav.down
#define SD_CLUSTER   0x0008                     // report in clusters
#define SD_BLOCKS    0x0108                     // report in blocks (use clsize)
#define SD_SUMMARY   0x0010                     // add summary line
#define SD_ALLINFO   0x0020                     // incl index and info fields
#define SD_CR_LINE   0x0040                     // force EOL char at end
#define SD_LIMIT32   0x0080                     // limit display to first 32


// Return short descriptive string for an XATTR Sspace attachement, length 12
char *dfsXattrDescription
(
   USHORT              xatype                   // IN    XATTR type
);

// Display allocation info for an S_SPACE array
ULN64 dfsSspaceDisplay                          // RET   nr of sectors
(
   ULONG               options,                 // IN    SD_ display options
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Get number of sectors represented by S_SPACE array
ULN64 dfsSspaceSectors                          // RET   nr of sectors or 0
(
   BOOL                ondisk,                  // IN    ondisk only, no sparse
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Copy data sectors from 1st SSPACE to space described by 2nd SSPACE
// Note: Destroys current .info, which is replaced by start RSN for chunk!
ULONG dfsSspaceCopyData
(
   ULONG               ch1,                     // IN    nr space entries 1st
   S_SPACE            *sp1,                     // IN    space allocation 1st
   ULONG               ch2,                     // IN    nr space entries 2nd
   S_SPACE            *sp2                      // IN    space allocation 2nd
);

// Enhance SSPACE structure by writing the start RSN for each chunk into .info
void dfsSspaceRsn2Info
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Read part of a file using relative-LSN and S_SPACE array
ULONG dfsSspaceReadFilePart
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn,                     // IN    relative sector nr
   ULN64               size,                    // IN    number of sectors
   BYTE               *buf                      // OUT   data buffer
);

// Write part of a file using relative-LSN and S_SPACE array
ULONG dfsSspaceWriteFilePart
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn,                     // IN    relative sector nr
   ULN64               size,                    // IN    number of sectors
   BYTE               *buf                      // IN    data buffer
);

// Wipe part of a file using relative-LSN and S_SPACE array, specified pattern
ULONG dfsSspaceWipeFilePart
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn,                     // IN    relative sector nr
   ULN64               size,                    // IN    number of sectors
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize                  // IN    pattern size
);

// Calculate real-LSN for relative-LSN in a S_SPACE array
ULN64 dfsSspaceRsn2Lsn                          // RET   Real LSN, _NULL or _SPARSE
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn                      // IN    relative sector nr
);

// Calculate relative-LSN for real-LSN in a S_SPACE array
ULN64 dfsSspaceLsn2Rsn                          // RET   Relative LSN
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               lsn                      // IN    real sector nr
);

// Truncate/extend an S_SPACE structure to specified size in sectors
ULONG dfsSspaceUpdateSize                       // RET   result
(
   DFSISPACE          *isp,                     // INOUT Integrated space
   ULN64               size,                    // IN    new size in sectors
   BOOL                extend                   // IN    allow extend last chunk
);

// Set allocation bits for range of sectors in SPACE, relative-start to END
ULONG dfsSetAllocForSpace
(
   DFSISPACE          *isp,                     // IN    Integrated space
   ULN64               start,                   // IN    relative start sector
   BOOL                set                      // IN    SET alloc (or reset)
);

// Wipe Sspace file with specified pattern, from, relative-start to END
ULONG dfsWipeTailForSpace
(
   DFSISPACE          *isp,                     // IN    Integrated space
   ULN64               start,                   // IN    relative start sector
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize                  // IN    pattern size
);

// Check allocation bits for range of sectors in SPACE, relative-start to END
ULONG dfsCheckAllocForSpace                     // RET   NO_ERROR if alloc OK
(
   DFSISPACE          *isp,                     // IN    Integrated space
   ULN64               start,                   // IN    relative start sector
   BOOL                set,                     // IN    CHECK if SET (or free)
   ULN64              *bads                     // OUT   Nr of failing sectors
);

// Create joined S_SPACE structure from 2 lists, free the two input S_SPACE's
ULONG dfsSspaceJoin                             // RET   nr of clusters
(
   ULONG               ch1,                     // IN    nr space entries 1st
   S_SPACE            *sp1,                     // IN    space allocation 1st
   ULONG               ch2,                     // IN    nr space entries 2nd
   S_SPACE            *sp2,                     // IN    space allocation 2nd
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// DFS universal Sspace write-file to disk (SaveTo to file)
// 20161204: Support SPARSE file chunks (NTFS/JFS) substituting ZEROED sectors
// 20161211: Support 'memory' based areas, and changed interface to DFSISPACE
ULONG dfsSspaceFileSaveAs
(
   DFSISPACE          *isp,                     // IN    Integrated space/mem
   BOOL                isDirectory,             // IN    Directory
   BOOL                isDeleted,               // IN    deleted / alive
   BOOL                noAllocCheck,            // IN    no check (faster/required)
   BOOL                force83,                 // IN    force 8.3 fname
   char               *path,                    // IN    destination path
   char               *fname,                   // IN    base name
   char               *rname                    // OUT   resulting filename
);

// Convert SectorList contents to a range of areas in an S_SPACE (bad sectors)
// The areas will be aligned to cluster/block size (not perfect for FAT!)
ULONG dfsSectorList2Space
(
   ULONG               maxAreas,                // IN    max areas to convert
   DFSISPACE          *is                       // OUT   Integrated S_SPACE
);

#endif
