//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee file-space handling functions, implementation
//
// Author: J. van Wijk
//
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
// JvW  01-04-2017 Updated to use 64-bit sectornumbers
// JvW  08-08-2005 Initial version, split off from dfsutil
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfswipe.h>                            // wipe functions


/*****************************************************************************/
// Return short descriptive string for an XATTR Sspace attachement, length 12
/*****************************************************************************/
char *dfsXattrDescription
(
   USHORT              xatype                   // IN    XATTR type
)
{
   switch (xatype)
   {
      case DFSXA_UNIX: return( "Linux, macOS"); break;
      case DFSXA_FEA2: return( "OS/2 EA data"); break;
      default:         return( "Not present!"); break;
   }
}                                               // end 'dfsXattrDescription'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display allocation info for an S_SPACE array (including SPARSE support)
/*****************************************************************************/
ULN64 dfsSspaceDisplay                          // RET   nr of sectors
(
   ULONG               options,                 // IN    SD_ display options
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULN64               used = 0;                // used sectors
   ULONG               chunk;                   // index in space-area
   TXTM                htext;                   // header text fragment
   TXTM                hline;                   // header underline
   TXTM                atext;                   // intermediate text
   TXLN                aline;                   // allocation line
   BOOL                navdown = FALSE;         // nav->down link set
   BOOL                sdai    = ((options & SD_ALLINFO) != 0);
   USHORT              clsize  = dfsGetClusterSize();

   ENTER();
   TRACES(("Options: %8.8x  Chunks: %u, space: %8.8X\n", options, chunks, space));

   sprintf( htext,      "Rel%s Size of allocated Extent   Data-LSN",
           ((options & SD_BLOCKS) == SD_BLOCKS) ? "Block  " :
            (options & SD_CLUSTER) ? "Cluster"  : "Sector ");
   sprintf( hline, "========== ======================== ==========");

   if (sdai)                                    // all info, single lines
   {
      strcat(  htext, "  Index     Info hex  ch");
      strcat(  hline, "  ========  ============");
   }
   if (options & SD_TOPLINE)
   {
      TxPrint( "\n %s  %s",   htext, ((chunks > 1) && !sdai) ? htext : "");
      TxPrint( "\n %s  %s\n", hline, ((chunks > 1) && !sdai) ? hline : "");
   }

   if (space != NULL)
   {
      for (chunk = 0; (chunk < chunks)  && (!TxAbort()); chunk++)
      {
         if ((chunk < 32) || ((options & SD_LIMIT32) == 0))
         {
            if (sdai)                           // all info, single lines
            {
               sprintf( aline, "%s ", (chunk) ? "\n" : "");
            }
            else
            {
               sprintf( aline, "%s ", (chunk % 2) ? " " : (chunk) ? "\n" : "");
            }
            strcpy( atext, "");
            if ((options & SD_CLUSTER) && (dfsa->boot != NULL)   && (clsize     != 0))
            {
               dfstrX10( atext, "", used / clsize, CNN, " ");
            }
            else                                // report in relative sectors
            {
               dfstrX10( atext, "", used, CNN, " ");
            }
            dfstrSz64( aline, atext,  space[chunk].size, "");
            if (space[chunk].start != LSN_SPARSE) // not a sparse element ?
            {
               if ((navdown == FALSE) && (options & SD_NAVDOWN))
               {
                  dfstrX10( aline, " ", space[chunk].start, CBG, "");
                  nav.down_sninfo = 0;          // avoid INFO interpretation
                  nav.down = space[chunk].start;
                  navdown  = TRUE;              // just assign once
               }
               else
               {
                  dfstrX10( aline, " ", space[chunk].start, CNN, "");
               }
            }
            else
            {
               sprintf( atext, " %s-sparse-%s", CNC, CNN);
               strcat(  aline, atext);
            }
            if (sdai)                           // all info, single lines
            {
               sprintf( atext, "  %8.8x  %8.8x  %s%c",
                        space[chunk].index, space[chunk].info,  CBC,
                                    (char) (space[chunk].info & 0x7f));
               strcat(  aline, atext);
               strcat(  aline, CNN);
            }
            TxPrint( "%s", aline);
         }
         used += space[chunk].size;             // RelSec of next chunk
      }
      if ((chunks > 32) && (options & SD_LIMIT32))
      {
         TxPrint("\n %u remaining chunks not displayed for brevity.", chunks - 32);
      }
      else
      {
         if (options & SD_BOTLINE)
         {
            TxPrint( "\n %s  %s",     hline, ((chunks > 1) && !sdai) ? hline : "");
            TxPrint( "\n %s  %s\n",   htext, ((chunks > 1) && !sdai) ? htext : "");
         }
      }
      if (options & SD_SUMMARY)
      {
         dfsSz64("\n\n Total allocation : ", dfsSspaceSectors( TRUE, chunks, space), "      ");
         if (chunks > 1)
         {
            TxPrint("in %s%u%s extents", CBM, chunks, CNN);
         }
      }
      if ((options & SD_TOPLINE) || (options & SD_CR_LINE))
      {
         TxPrint("\n");
      }
   }
   else                                         // resident by convention, NTFS
   {
      TxPrint( "  00000000 Resident, at byte: %u", chunks);
   }
   TxPrint("\n");
   RETN64 (used);
}                                               // end 'dfsSspaceDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get number of sectors represented by S_SPACE array
/*****************************************************************************/
ULN64 dfsSspaceSectors                          // RET   nr of sectors or 0
(
   BOOL                ondisk,                  // IN    ondisk only, no sparse
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc = 0;
   ULONG               chunk;                   // index in space-area

   ENTER();

   TRACES(("Chunks: %u, space: %8.8X\n", chunks, space));

   if (space != NULL)
   {
      for ( chunk = 0; (chunk < chunks); chunk++)
      {
         if ((ondisk == FALSE) || (space[chunk].start != LSN_SPARSE))
         {
            rc  += space[chunk].size;           // total size sofar
         }
      }
   }
   RETN64 (rc);
}                                               // end 'dfsSspaceSectors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy data sectors from 1st SSPACE to space described by 2nd SSPACE
// Note: Destroys current .info, which is replaced by start RSN for chunk!
/*****************************************************************************/
ULONG dfsSspaceCopyData
(
   ULONG               ch1,                     // IN    nr space entries 1st
   S_SPACE            *sp1,                     // IN    space allocation 1st
   ULONG               ch2,                     // IN    nr space entries 2nd
   S_SPACE            *sp2                      // IN    space allocation 2nd
)
{
   ULONG               rc  = NO_ERROR;          // function result
   ULN64               thisRsn;                 // Current Relative SN
   ULN64               thisSize;                // Current copy-chunk size
   ULN64               copySize;                // Sectors to be copied
   ULN64               todo1;                   // left todo chunk 1st
   ULN64               todo2;                   // left todo chunk 2nd
   ULONG               e1  = 0;                 // extent index in 1st
   ULONG               e2  = 0;                 // extent index in 2nd
   ULONG               bs;                      // blocksize in sectors
   BYTE               *data;
   USHORT              bps = dfsGetSectorSize(); // bytes per sector

   ENTER();
   TRACES(( "1st: %u @ %8.8x, 2nd: %u @ %8.8x\n", ch1, sp1, ch2, sp2));

   if (DFSTORE_WRITE_ALLOWED)
   {
      bs = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);
      if (bs <= RBUFSECTORS)                    // fits in the standard I/O buffer
      {
         data = rbuf;                           // always used with DOS!
      }
      else
      {
         data = TxAlloc( bps, bs);              // large buffer, non-DOS only!
      }
      if (data != NULL)
      {
         copySize = dfsSspaceSectors( FALSE, ch1, sp1);        //- size in source
         if (copySize <= dfsSspaceSectors( FALSE, ch2, sp2))   //- size in destination
         {
            TRACES(("buffer:0x%8.8x, copySize:0x%llx\n", bs, copySize));

            //- Enhance the SSPACE structure by writing the start RSN
            //- into the corresponding .info field in each SSPACE chunk
            dfsSspaceRsn2Info( ch1, sp1);
            dfsSspaceRsn2Info( ch2, sp2);

            //- Now copy in chunks that are as large as possible, but limited by
            //- remaining chunk-sizes in source, destination and the buffer-size
            //- Size could even be 0 if there are source/dest chunks that are 0
            for ( thisRsn = 0, e1 = 0, e2 = 0;
                 (thisRsn < copySize) && (rc == NO_ERROR);
                  thisRsn += thisSize)
            {
               todo1 = sp1[ e1].size + sp1[ e1].info - thisRsn;
               todo2 = sp2[ e2].size + sp2[ e2].info - thisRsn;
               thisSize = min( bs, min( todo1, todo2));

               TRACES(("RSN:%llx, todo1:%llu, todo2:%llu, thisSize:%llu\n",
                        thisRsn,  todo1,      todo2,      thisSize));

               rc = dfsSspaceReadFilePart( ch1, sp1, thisRsn, thisSize, data);
               if (rc == NO_ERROR)
               {
                  rc = dfsSspaceWriteFilePart( ch2, sp2, thisRsn, thisSize, data);
                  if (rc == NO_ERROR)
                  {
                     if (thisSize >= todo1)     // copied whole source chunk now
                     {
                        e1++;                   // to next chunk in source
                     }
                     if (thisSize >= todo2)     // copied whole destination chunk now
                     {
                        e2++;                   // to next chunk in destination
                     }
                  }
               }
            }
         }
         else
         {
            rc = DFS_BAD_STRUCTURE;
         }
         if (data != rbuf)
         {
            TxFreeMem( data);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN (rc);
}                                               // end 'dfsSspaceCopyData'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Enhance SSPACE structure by writing the start RSN for each chunk into .info
/*****************************************************************************/
void dfsSspaceRsn2Info
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               chunk;                   // index in space-area
   ULN64               rsn = 0;                 // RSN for this chunk

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X\n", chunks, space));

   for ( chunk = 0;
        (chunk < chunks) && (space != NULL);
         chunk++)
   {
      space[ chunk].info = rsn;
      rsn += space[ chunk].size;
   }
   VRETURN();
}                                               // end 'dfsSspaceRsn2Info'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read part of a file using relative-LSN and S_SPACE array, Area aware
/*****************************************************************************/
ULONG dfsSspaceReadFilePart
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn,                     // IN    relative sector nr
   ULN64               size,                    // IN    number of sectors
   BYTE               *buf                      // OUT   data buffer
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sec;                     // relative sector read
   ULN64               lsn;
   USHORT              bps = dfsGetSectorSize();

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, rsn: 0x%llX, sectors: 0x%llx, bps:%hu  buf:%8.8x\n",
            chunks, space, rsn, size, bps, buf));

   for (sec = 0; (sec < size) && (rc == NO_ERROR); sec++)
   {
      lsn = dfsSspaceRsn2Lsn( chunks, space, rsn + sec);
      if (lsn != LSN_SPARSE)
      {
         rc  = dfsRead( lsn, 1, buf + (sec * bps));
      }
      else
      {
         memset( buf, 0, bps);
      }
   }
   RETURN (rc);
}                                               // end 'dfsSspaceReadFilePart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write part of a file using relative-LSN and S_SPACE array, Area aware
/*****************************************************************************/
ULONG dfsSspaceWriteFilePart
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn,                     // IN    relative sector nr
   ULN64               size,                    // IN    number of sectors
   BYTE               *buf                      // IN    data buffer
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sec;                     // relative sector read
   ULN64               lsn;
   USHORT              bps = dfsGetSectorSize();

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, rsn: 0x%llX, sectors: 0x%llx, bps:%hu  buf:%8.8x\n",
            chunks, space, rsn, size, bps, buf));

   for (sec = 0; (sec < size) && (rc == NO_ERROR); sec++)
   {
      lsn = dfsSspaceRsn2Lsn( chunks, space, rsn + sec);
      if (lsn != LSN_SPARSE)
      {
         rc  = dfsWrite( lsn, 1, buf + (sec * bps));
      }
      else
      {
         rc = ERROR_WRITE_FAULT;
      }
   }
   RETURN (rc);
}                                               // end 'dfsSspaceWriteFilePart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Wipe part of a file using relative-LSN and S_SPACE array, specified pattern
/*****************************************************************************/
ULONG dfsSspaceWipeFilePart
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn,                     // IN    relative sector nr
   ULN64               size,                    // IN    number of sectors
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize                  // IN    pattern size
)
{
   ULONG               rc = NO_ERROR;
   ULN64               sec;                     // relative sector read
   ULN64               lsn;

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, rsn: 0x%llX, sectors: 0x%llx\n",
            chunks, space, rsn, size));

   for (sec = 0; (sec < size) && (rc == NO_ERROR); sec++)
   {
      lsn = dfsSspaceRsn2Lsn( chunks, space, rsn + sec);
      if (lsn != LSN_SPARSE)
      {
         rc  = dfsWipeArea( lsn, 1, DFSP_NONE, NULL, FALSE, pattern, patsize);
         TxCancelAbort();                       // reset pending abort status
      }
      else
      {
         rc = ERROR_WRITE_FAULT;
      }
   }
   RETURN (rc);
}                                               // end 'dfsSspaceWipeFilePart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate real-LSN for relative-LSN in a S_SPACE array
/*****************************************************************************/
ULN64 dfsSspaceRsn2Lsn                          // RET   Real LSN, _SPARSE or _NULL
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               rsn                      // IN    relative sector nr
)
{
   ULONG               chunk;                   // index in space-area
   ULN64               lsn = L64_NULL;
   ULN64               sec;                     // nr of sectors from start
   static ULN64        lastSec   = 0;           // cached last-RSN start sn
   static ULONG        lastChunk = 0;           // cached last chunk/index
   static S_SPACE     *lastSpace = NULL;        // cached SPACE

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, rsn: 0x%llX, lastSec: 0x%llX\n",
            chunks,      space,         rsn,           lastSec));

   if (space != NULL)
   {
      if ((space == lastSpace) && (rsn >= lastSec)) // use cache
      {
         chunk = lastChunk;
         sec   = rsn - lastSec;
      }
      else                                      // start from start of SPACE
      {
         chunk = 0;
         sec   = rsn;                           // sectors to travel
      }

      while (chunk < chunks)
      {
         if (sec < space[chunk].size)
         {
            //- set up cache variables for next call
            lastChunk = chunk;
            lastSec   = rsn - sec;
            lastSpace = space;

            lsn = space[chunk].start;           // Start of chunk, or sparse
            if (lsn != LSN_SPARSE)
            {
               lsn += sec;                      // real allocated LSN
            }
            break;                              // out of loop
         }
         else
         {
            sec -= space[chunk].size;           // sectors left to travel
            chunk++;                            // to next chunk
         }
      }
   }
   RETN64 (lsn);
}                                               // end 'dfsSspaceRsn2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate relative-LSN for real-LSN in a S_SPACE array
/*****************************************************************************/
ULN64 dfsSspaceLsn2Rsn                          // RET   Relative LSN
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64               lsn                      // IN    real sector nr
)
{
   ULONG               chunk;                   // index in space-area
   ULN64               rsn = L64_NULL;
   ULN64               sec = 0;                 // nr of sectors from start

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, lsn: 0x%llX\n", chunks, space, lsn));

   for ( chunk = 0;
        (chunk < chunks) && (rsn == L64_NULL) && (space != NULL);
         chunk++)
   {
      if ((space[chunk].start != LSN_SPARSE) &&
          (lsn >= space[chunk].start)        &&
          (lsn < (space[chunk].start + space[chunk].size)))
      {
         rsn = lsn - space[chunk].start + sec;  // found target rsn
      }
      else
      {
         sec += space[chunk].size;              // sectors traveled sofar
      }
   }
   RETN64 (rsn);
}                                               // end 'dfsSspaceLsn2Rsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Truncate/extend an S_SPACE structure to specified size in sectors
/*****************************************************************************/
ULONG dfsSspaceUpdateSize                       // RET   result
(
   DFSISPACE          *isp,                     // INOUT Integrated space
   ULN64               size,                    // IN    new size in sectors
   BOOL                extend                   // IN    allow extend last chunk
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULN64               sec = 0;                 // nr of sectors from start

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, size: %llX\n",
            isp->chunks, isp->space, size));

   if (isp->space != NULL)
   {
      for ( chunk = 0; chunk < isp->chunks; chunk++)
      {
         if ((sec + isp->space[chunk].size) >= size) // truncation point found
         {
            isp->space[chunk].size = size - sec; // leave in this chunk
            isp->chunks = chunk +1;             // remaining nr of entries
            sec  = size;                        // resulting total size
            break;
         }
         else
         {
            sec += isp->space[chunk].size;      // sectors traveled sofar
         }
      }
      if ((size > sec) && extend)               // size larger, extend allowed
      {
         isp->space[ isp->chunks -1].size += (size - sec); // adjust last block
      }
   }
   RETURN (rc);
}                                               // end 'dfsSspaceUpdateSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocation bits for range of sectors in SPACE, relative-start to END
/*****************************************************************************/
ULONG dfsSetAllocForSpace
(
   DFSISPACE          *isp,                     // IN    Integrated space
   ULN64               start,                   // IN    relative start sector
   BOOL                set                      // IN    SET alloc (or reset)
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               chunk;                   // index in space-area
   ULN64               sec = start;             // nr of sectors from start

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, start: %llX, clsize:%8.8x\n",
            isp->chunks, isp->space, start, isp->clsize));

   for ( chunk = 0;
        (chunk < isp->chunks) && (rc == NO_ERROR) && (isp->space != NULL);
         chunk++)
   {
      if (sec < isp->space[chunk].size)         // start is in this chunk
      {
         rc = dfsSetAllocForRange( isp->space[chunk].start + sec,
                                   isp->space[chunk].size  - sec,
                                  (isp->clsize) ? isp->clsize :
                                   dfsGetClusterSize(), set);
         sec = 0;
      }
      else
      {
         sec -= isp->space[chunk].size;         // sectors left to travel
      }
   }
   RETURN (rc);
}                                               // end 'dfsSetAllocForSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Wipe Sspace file with specified pattern, from, relative-start to END
/*****************************************************************************/
ULONG dfsWipeTailForSpace
(
   DFSISPACE          *isp,                     // IN    Integrated space
   ULN64               start,                   // IN    relative start sector
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize                  // IN    pattern size
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               chunk;                   // index in space-area
   ULN64               sec = start;             // nr of sectors from start

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, start: %llX\n",
            isp->chunks, isp->space, start));

   for ( chunk = 0;
        (chunk < isp->chunks) && (rc == NO_ERROR) && (isp->space != NULL);
         chunk++)
   {
      if (sec < isp->space[chunk].size)         // start is in this chunk
      {
         rc = dfsWipeArea( isp->space[chunk].start + sec,
                           isp->space[chunk].size  - sec,
                           DFSP_NONE, NULL, FALSE, pattern, patsize);
         TxCancelAbort();                       // reset pending abort status
         sec = 0;
      }
      else
      {
         sec -= isp->space[chunk].size;         // sectors left to travel
      }
   }
   RETURN (rc);
}                                               // end 'dfsWipeTailForSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check allocation bits for range of sectors in SPACE, relative-start to END
/*****************************************************************************/
ULONG dfsCheckAllocForSpace
(
   DFSISPACE          *isp,                     // IN    Integrated space
   ULN64               start,                   // IN    relative start sector
   BOOL                set,                     // IN    CHECK if SET (or free)
   ULN64              *bads                     // OUT   Nr of failing sectors
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               chunk;                   // index in space-area
   ULN64               sec = start;             // nr of sectors from start

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X, start: %llX, clsize:%8.8x\n",
            isp->chunks, isp->space, start, isp->clsize));

   if (bads != NULL)
   {
      *bads = 0;
   }
   for ( chunk = 0;
        (chunk < isp->chunks) && (rc == NO_ERROR) && (isp->space != NULL);
         chunk++)
   {
      if (sec < isp->space[chunk].size)         // start is in this chunk
      {
         rc = dfsCheckAllocForRange( isp->space[chunk].start + sec,
                                     isp->space[chunk].size  - sec,
                                    (isp->clsize) ? isp->clsize :
                                     dfsGetClusterSize(), set, bads);
         sec = 0;
      }
      else
      {
         sec -= isp->space[chunk].size;         // sectors left to travel
      }
   }
   RETURN (rc);
}                                               // end 'dfsCheckAllocForSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create joined S_SPACE structure from 2 lists, free the two input S_SPACEs
/*****************************************************************************/
ULONG dfsSspaceJoin                             // RET   nr of clusters
(
   ULONG               ch1,                     // IN    nr space entries 1st
   S_SPACE            *sp1,                     // IN    space allocation 1st
   ULONG               ch2,                     // IN    nr space entries 2nd
   S_SPACE            *sp2,                     // IN    space allocation 2nd
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               extents = ch1 + ch2;     // nr of extents, most likely
   S_SPACE            *sp      = NULL;          // joined space structure

   ENTER();

   TRACES(( "1st: %u @ %8.8x, 2nd: %u @ %8.8x\n", ch1, sp1, ch2, sp2));
   if ((sp1 != NULL) && (ch1 != 0))             // sp1 not empty
   {
      TRINIT(700);
         dfsSspaceDisplay( SD_DEFAULT, ch1, sp1);
         dfsSspaceDisplay( SD_DEFAULT, ch2, sp2);
      TREXIT();
      sp = TxAlloc(extents, sizeof(S_SPACE));
      if (sp != NULL)
      {
         ULONG         ch;
         ULONG         offset = 0;

         sp->index = sp1->index + sp2->index;   // total exact size
         TRACES(("memcopy %u extents in 1st\n", ch1));
         memcpy( &(sp[ 0 ]), sp1, (size_t) ch1 * sizeof(S_SPACE));

         for (ch = 0; ch < ch2; ch++)           // copy second S_SPACE, optimized
         {
            if (ch == 0)                        // first one, check if adjacent
            {
               ULONG   last = ch1 -1;           // last extent of first space
               if ((sp[last].start + sp[last].size) == sp2[0].start)
               {
                  TRACES(("Merge new with last: %u\n", last));
                  extents--;                    // there will be one less ...
                  sp[last].size += sp2[0].size; // join the two extents
                  offset  = last;               // first extent of 2nd is here
               }
               else
               {
                  TRACES(("Add new after last at: %u\n", ch1));
                  offset  = ch1;                // first extent of 2nd is here
                  sp[ offset] = sp2[ch];        // copy one extent over
               }
            }
            else                                // not first extent, just copy
            {
               TRACES(("Add new %u after last at: %u\n", ch, offset));
               sp[ch + offset] = sp2[ch];       // copy one extent over
            }
         }
         TxFreeMem( sp1);
         TxFreeMem( sp2);                       // free input arrays
         TRINIT(700);
            dfsSspaceDisplay( SD_DEFAULT, extents, sp);
         TREXIT();
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // return sp2 as join
   {
      TRACES(( "First list is empty, return second as joined ...\n"));
      sp      = sp2;
      extents = ch2;
   }
   *space  = sp;
   *chunks = extents;
   RETURN (rc);
}                                               // end 'dfsSspaceJoin'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS universal Sspace write-file to disk (SaveTo recovery file)
// 20161204: Support SPARSE file chunks (NTFS/JFS) substituting ZEROED sectors
// 20161211: Support 'memory' based areas, and changed interface to DFSISPACE
/*****************************************************************************/
ULONG dfsSspaceFileSaveAs
(
   DFSISPACE          *isp,                     // IN    Integrated space/mem
   BOOL                isDirectory,             // IN    Directory
   BOOL                isDeleted,               // IN    deleted / alive
   BOOL                noAllocCheck,            // IN    no check (faster/required)
   BOOL                force83,                 // IN    force 8.3 fname
   char               *fpath,                   // IN    destination path
   char               *fname,                   // IN    base name
   char               *rname                    // OUT   resulting filename
)
{
   ULONG               rc = NO_ERROR;
   S_SPACE            *space = isp->space;      // space or 'mem' allocation
   ULN64               fsize = isp->byteSize;   // exact filesize
   TXLN                path;                    // path to create
   TXLN                name;                    // file to create
   TXLN                fspec;                   // full, original filespec
   TXHFILE             fh = 0;                  // filehandle
   ULONG               chunk;                   // index in space-area
   ULN64               sect;                    // sectors to handle
   BYTE               *data;                    // sector
   ULN64               sn;                      // Data LSN
   USHORT              col = 0;
   ULONG               written;                 // nr of bytes written per write
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
      int              openflag = O_WRONLY | O_CREAT | O_TRUNC  | O_LARGEFILE; // purpose NOW
      int              openmode = TX_DEFAULT_OPEN_MODE;                        // permissions AFTER
   #else
      ULONG            action;                  // action taken on open
      EAOP2            ea;                      // ea operations structure
      EAOP2           *eap = NULL;              // ptr to ea structure
   #endif

   ENTER();
   TRARGS(("Chunks: %u  size : %llu, to path: '%s'  basename: '%s'\n", isp->chunks, fsize, fpath, fname));
   TRARGS(("isDirectory:%u isDeleted:%u force83:%u noAllocCheck:%u\n", isDirectory, isDeleted, force83, noAllocCheck));

   if ((data = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
   {
      ULONG            invalids = 0;            // invalid allocation count
      BOOL             alok = FALSE;            // allocation OK
      ULN64            progressSectors = 0;     // set when progress wanted
      ULN64            copiedSectors   = 0;     // running count while copying

      strcpy( name, fname);
      strcpy( path, fpath);                     // Copy full-length components
      TxRepl( path, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      if (( force83) ||                          // explicit 8.3 requested
          ((strlen(path) + strlen(name) +1) >= dfsa->maxPath) || // too long
          ((TxMakePath( path)) != NO_ERROR) )   // or make long-path failed
      {
         TxMake8dot3( fpath, path);             // convert fpath to 8.3
         TxMakePath( path);                     // retry with 8.3 path
      }
      if (( force83) ||                         // explicit 8.3 requested
          ((strlen(path) + strlen(name) +1) >= dfsa->maxPath)) // still too long
      {
         TxMake8dot3( fname, name);             // convert fname to 8.3
      }
      if ((strlen(path) + strlen(name) +1) < dfsa->maxPath) // length OK now
      {
         sprintf( fspec, "%s%s%s", path, (path[strlen(path)-1] != FS_PATH_SEP) ? FS_PATH_STR : "", name);

         #if defined (DEV32)
            ea.fpGEA2List = NULL;
            ea.oError     = 0;
            if ((isp->xatype == DFSXA_FEA2) && (isp->xattrs != NULL))
            {
               ea.fpFEA2List = (PFEA2LIST) isp->xattrs;
               if (dfsa->verbosity >= TXAO_MAXIMUM)
               {
                  col++;
                  if (++col >= dfsGetDisplayMargin())
                  {
                     TxPrint("\n");
                     col = 2;
                  }
                  TxPrint("%sEA%s", CBY, CNN);  // File includes EA's
               }
               eap = &ea;
            }
            else
            {
               eap = NULL;
            }
         #endif

         if (!isDirectory)                      // regular file
         {
            strcpy( path, fspec);
            #if defined (WIN32)
               fh = CreateFile( path,
                                GENERIC_WRITE,  // write access only
                                0,              // deny-write
                                NULL,           // default security info
                                CREATE_ALWAYS,  // always new file
                                FILE_ATTRIBUTE_NORMAL,
                                NULL);
               if (fh == INVALID_HANDLE_VALUE)
               {
                  TxMake8dot3( fspec, path);
                  fh = CreateFile( path,
                                   GENERIC_WRITE, // write access only
                                   0,           // deny-write
                                   NULL,        // default security info
                                   CREATE_ALWAYS, // always new file
                                   FILE_ATTRIBUTE_NORMAL,
                                   NULL);
                  if (fh == INVALID_HANDLE_VALUE)
                  {
                     TRACES(( "CreateFile error on '%s': %s\n", path, txNtLastError()));
                     rc = TX_INVALID_FILE;
                  }
               }
            #elif defined (DOS32)
               fh = fopen( path, "wb");         // open binary write
               if (fh == 0)
               {
                  TxMake8dot3( fspec, path);
                  fh = fopen( path, "wb");      // open binary write 8.3
                  if (fh == 0)
                  {
                     rc = TX_INVALID_FILE;
                  }
               }
            #elif defined (UNIX)
               if ((fh = open( fspec, openflag, openmode)) == -1)
               {
                  fh = 0;
               }
               if (fh == 0)
               {
                  TxMake8dot3( fspec, path);
                  if ((fh = open( fspec, openflag, openmode)) == -1)
                  {
                     fh = 0;
                     rc = TxRcFromErrno( errno);
                  }
               }
            #else
               if (dfsa->os2api->DosOpenL)
               {
                  rc = (dfsa->os2api->DosOpenL)(
                                path, &fh,
                                &action,
                                fsize,
                                FILE_NORMAL,
                                FILE_TRUNCATE | FILE_CREATE |
                                OPEN_ACTION_CREATE_IF_NEW |
                                OPEN_ACTION_REPLACE_IF_EXISTS,
                                OPEN_SHARE_DENYWRITE |
                                OPEN_ACCESS_WRITEONLY,
                                eap);           // possible EA information
               }
               else
               {
                  rc = DosOpen( path, &fh,
                                &action,
                                fsize,
                                FILE_NORMAL,
                                FILE_TRUNCATE | FILE_CREATE |
                                OPEN_ACTION_CREATE_IF_NEW |
                                OPEN_ACTION_REPLACE_IF_EXISTS,
                                OPEN_SHARE_DENYWRITE |
                                OPEN_ACCESS_WRITEONLY,
                                eap);           // possible EA information
               }
               if (rc != NO_ERROR)
               {
                  TxMake8dot3( fspec, path);
                  if (dfsa->os2api->DosOpenL)
                  {
                     rc = (dfsa->os2api->DosOpenL)(
                                   path, &fh,
                                   &action,
                                   fsize,
                                   FILE_NORMAL,
                                   FILE_TRUNCATE | FILE_CREATE |
                                   OPEN_ACTION_CREATE_IF_NEW |
                                   OPEN_ACTION_REPLACE_IF_EXISTS,
                                   OPEN_SHARE_DENYWRITE |
                                   OPEN_ACCESS_WRITEONLY,
                                   eap);        // possible EA information
                  }
                  else
                  {
                     rc = DosOpen( path, &fh,
                                   &action,
                                   fsize,
                                   FILE_NORMAL,
                                   FILE_TRUNCATE | FILE_CREATE |
                                   OPEN_ACTION_CREATE_IF_NEW |
                                   OPEN_ACTION_REPLACE_IF_EXISTS,
                                   OPEN_SHARE_DENYWRITE |
                                   OPEN_ACCESS_WRITEONLY,
                                   eap);        // possible EA information
                  }
               }
            #endif
         }
         else
         {
            strcpy( path, fspec);
            if ((force83) || ((TxMakePath( path)) != NO_ERROR))
            {
               TxMake8dot3( fspec, path);
               TxMakePath( path);               // retry with 8.3 path
            }
            if (rname)
            {
               strcpy( rname, path);
            }
            #if defined (DEV32)
               if (eap != NULL)
               {
                  rc = DosSetPathInfo( path,    // Add the EA's
                                       FIL_QUERYEASIZE,
                                       eap,
                                       sizeof(EAOP2),
                                       0);
                  TRACES(("Add EA's for '%s', rc:%u\n", path, rc));
                  if (rc != NO_ERROR)
                  {
                     if (dfsa->verbosity >= TXAO_NORMAL)
                     {
                        TxPrint("Setting Extended-Attributes (EA) failed!\n");
                     }
                     rc = NO_ERROR;             // don't abort due to failed EA
                  }
               }
            #endif
         }
         TRACES(("Open: '%s', handle %u, rc: %d\n", path, fh, (LONG)rc));

         if (dfsa->verbosity >= TXAO_NORMAL)
         {
            TxPrint("SaveTo '%s'", path);       // report opened path
            if (dfsa->verbosity >= TXAO_VERBOSE)
            {
               TxPrint("\n");                   // make it a separate line
            }
         }

         //- read+write the data
         if (fh != 0)                           // only for files (DIR is create only)
         {
            if (rname)
            {
               strcpy( rname, path);
            }

            if ((isp->chunks != 0) && (fsize != 0)) // no memory area, and no empty file
            {
               if (fsize > (DFS_COPY_THRESHOLD_SECTORS * dfsGetSectorSize()))
               {
                  progressSectors = (fsize / dfsGetSectorSize()) + 1;
                  dfsProgressInit( 0, progressSectors, 0, "Sector:", "Copied", DFSP_STAT, 0);
               }
               else                             // no ProgressInit() to anounce next item
               {
                  dfsProgressItemsNext();       // explicit indicate next 'item'
               }
               for (chunk = 0; (chunk < isp->chunks) && !TxAbort(); chunk++)
               {
                  if (dfsa->verbosity >= TXAO_MAXIMUM)
                  {
                     if (++col >= dfsGetDisplayMargin())
                     {
                        TxPrint("\n");
                        col = 1;
                     }
                     TxPrint("");              // start of allocation extent
                  }
                  for (sect = 0; (sect < space[chunk].size) && !TxAbort(); sect++)
                  {
                     if (space[chunk].start == LSN_SPARSE) // sparse chunk
                     {
                        if (sect == 0)          // first of the chunk
                        {
                           TRACES(("SPARSE chunk, size: %u sectors\n", space[chunk].size));
                           memset( data, 0, dfsGetSectorSize());
                        }
                        alok = TRUE;            // always OK
                     }
                     else                       // regular, read the sector
                     {
                        sn = space[chunk].start + sect;
                        rc = dfsRead(sn, 1, data);

                        if (noAllocCheck)
                        {
                           alok = TRUE;         // always OK
                        }
                        else                    // check if ALLOC status matches
                        {
                           if (DFSFNCALL( dfsa->FsLsnAllocated, sn, 0, NULL, NULL))
                           {
                              alok = (isDeleted == FALSE);
                           }
                           else                 // must be a deleted file
                           {
                              alok = (isDeleted);
                           }
                        }
                     }
                     if (rc == NO_ERROR)
                     {
                        rc = TxWrite( fh, data, dfsGetSectorSize(), &written);
                        TRACES(("SaveTo: TxWrite rc: %d,  alok:%u\n", (LONG) rc, alok));
                        if (rc == NO_ERROR)
                        {
                           if (dfsa->verbosity >= TXAO_MAXIMUM)
                           {
                              if (++col >= dfsGetDisplayMargin())
                              {
                                 TxPrint("\n");
                                 col = 1;
                              }
                              TxPrint("%s%s", alok ? CBG : CBR, CNN);
                           }
                           if (!alok)
                           {
                              invalids++;
                           }
                           if (progressSectors != 0)
                           {
                              dfsProgressShow( copiedSectors++, 0, 0, NULL);
                           }
                        }
                        else
                        {
                           TxPrint("\nWrite failed, rc: %d on '%s'\n", (LONG) rc, path);
                        }
                     }
                  }
               }
               if (progressSectors != 0)
               {
                  dfsProgressTerm();
               }
            }
            else                                // Memory area SaveAs
            {
               ULONG   at;                      // position in area
               BYTE   *area = (BYTE *) isp->space;

               for ( at = 0; at < fsize; at += dfsGetSectorSize())
               {
                  memcpy( data, area + at,
                         (size_t) (min( (ULONG) (fsize - at),
                                        (ULONG)  dfsGetSectorSize())));
                  rc = TxWrite( fh, data, dfsGetSectorSize(), &written);
                  TRACES(("SaveTo: TxWrite rc: %d\n", (LONG) rc));
                  if (rc == NO_ERROR)
                  {
                     if (dfsa->verbosity >= TXAO_MAXIMUM)
                     {
                        TxPrint("%s%s", CBG, CNN);
                     }
                  }
                  else
                  {
                     TxPrint("\nWrite failed, rc: %d on '%s'\n", (LONG) rc, path);
                  }
               }
            }
         }
         if (dfsa->verbosity >= TXAO_MAXIMUM)
         {
            TxPrint("\n");
         }

         //- Set size and close the file
         if (fh != 0)
         {
            if (rc == NO_ERROR)
            {
               TRACES(( "Set length of new file to: %lld\n", fsize));
               if (fsize != 0)                  // avoid clipping a written but
               {                                // still length zero file
                  rc = TxSetFileSize( fh, fsize);
               }
               TRACES(("SaveTo: TxSetFileSize rc: %d\n", (LONG) rc));
               if (invalids)
               {
                  if (dfsa->verbosity >= TXAO_VERBOSE)
                  {
                     TxPrint("Allocation error detected on %5u sectors, "
                             "saved data is unreliable!\n", invalids);
                  }
                  else
                  {
                     TxPrint(" Allocation (bitmap) inconsistent, saved data is unreliable! ");
                  }
                  rc = DFS_CMD_WARNING;
               }
               else if (rc == NO_ERROR)
               {
                  if (dfsa->verbosity >= TXAO_VERBOSE)
                  {
                     TxPrint("No errors detected while saving the filedata.\n");
                  }
               }
               else
               {
                  TxPrint("Error %d setting filesize\n", (LONG) rc);
               }
            }
            TxClose( fh);
            fh = 0;
         }
         else if (rc != NO_ERROR)
         {
            TxPrint("Create failure, rc:%d on '%s'\n", (LONG) rc, path);
         }
      }
      else
      {
         rc = ERROR_FILENAME_EXCED_RANGE;
      }
      TxFreeMem( data);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsSspaceFileSaveAs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert SectorList contents to a range of areas in an S_SPACE (bad sectors)
// The areas will be aligned to cluster/block size (not perfect for FAT!)
/*****************************************************************************/
ULONG dfsSectorList2Space
(
   ULONG               maxAreas,                // IN    max areas to convert
   DFSISPACE          *is                       // OUT   Integrated S_SPACE
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64              *list   = dfsa->snlist;   // sectorlist
   ULN64               rangCl = dfsSn2Cl(list[1]); //- start of cluster range
   ULN64               lastCl = rangCl;         // last cluster in this range
   ULN64               clust;                   // cluster value
   ULONG               nr;                      // index in sector list
   ULONG               area = 0;                // area index being converted

   ENTER();

   memset( is, 0, sizeof(DFSISPACE));           // initialize to all zero
   is->allocated = (maxAreas) ? maxAreas : 1024;
   is->clsize = dfsGetClusterSize();

   if (*list != 0)                              // any entries in the list?
   {
      S_SPACE   *sp = (S_SPACE *) TxAlloc( is->allocated, sizeof(S_SPACE));

      if (sp != NULL)
      {
         is->space  = sp;

         for (nr = 1; nr <= *list; nr++)
         {
            clust = dfsSn2Cl( list[nr]);        // current cluster/block number

            if ((clust <  lastCl    ) ||        // begin of a new range, output current
                (clust >  lastCl + 1)  )
            {
               sp[ area].start = dfsCl2Sn( rangCl);
               sp[ area].size  = (lastCl - rangCl + 1) * dfsGetClusterSize();

               is->chunks = ++area;             // advance to next area

               rangCl = clust;                  // start of new range
            }
            lastCl = clust;

            if ((nr == *list) && (area < is->allocated)) // at end, output last range
            {
               sp[ area].start = dfsCl2Sn( rangCl);
               sp[ area].size  = (lastCl - rangCl + 1) * dfsGetClusterSize();

               is->chunks = ++area;             // set final area count
            }
            else if (area == is->allocated)     // some snlist entries NOT handled
            {
               rc = DFS_CMD_WARNING;
               break;                           // exit the loop
            }
         }
         TRINIT(700);
            dfsSspaceDisplay( SD_DEFAULT, is->chunks, is->space);
         TREXIT();
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsSectorList2Space'
/*---------------------------------------------------------------------------*/
