//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee major functions: shifting data within sectors
//
// Author: J. van Wijk
//
// JvW  27-01-2006   Initial version, split off from DFSMAJOR.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsshift.h>                           // sector data shifting
#include <dfsutil.h>                            // DFS  utility functions


/*****************************************************************************/
// Move data in sectors over specified number of bytes towards start (Left)
/*****************************************************************************/
ULONG dfsByteShiftLeftSector
(
   ULONG               shift,                   // IN    shift amount 0 .. 511
   ULN64               start,                   // IN    starting lsn
   ULN64               size                     // IN    number of sectors todo
)
{
   ULONG               rc  = NO_ERROR;          // function result
   ULN64               lsn;                     // Current LSN
   USHORT              bps = dfsGetSectorSize(); // bytes per sector
   BYTE               *mbp = rbuf;              // memory byte ptr (1st sect)
   ULONG               bs;                      // blocksize in sectors
   ULN64               sects;                   // nr of sectors todo now
   ULN64               i;

   ENTER();

   bs = dfsGetBufferSize( DFSOPTIMALBUF, (RBUFBYTES / bps) / 2);
   if (DFSTORE_WRITE_ALLOWED)                   // lock or ask to ignore
   {
      dfsX10("\nShift startsector : ", start, "", " ");
      dfsSz64Bps("size : ", size, bps, "\n");
      TxPrint(  "Sector byte shift : 0x%2.2lx = %3lu bytes LEFT, buffer% 6lu sect", shift, shift, bs);
      dfsSize( " = ", bs, "\n");

      TRACES(( "start:%llx size:%llx bs:%8.8lx\n", start, size, bs));
      dfsProgressInit( start, size, 0, "Sector:", "shifted", DFSP_BARS, 0);

      for ( i  = 0,   lsn = start, sects = bs;
           (i < size) && (rc == NO_ERROR) && !TxAbort();
            i += bs,  lsn += bs)
      {
         if ((i + sects) > size)                // incomplete block
         {
            sects = size - i;
         }
         TRACES(( "i:%llx lsn:%llx sects:%8.8lx\n", i, lsn, sects));

         //- read one extra sector, data to be shifted into last real one ...
         rc = dfstBadSectorRead( DFSTORE, lsn, sects +1, NULL, NULL, mbp);
         if (rc == NO_ERROR)
         {
            memmove( mbp, mbp + shift, sects * bps);
            rc = dfsWrite( lsn, sects, mbp);    // write shifted sectors back
         }
         dfsProgressShow( lsn + sects, 0, 0, NULL);
      }
      dfsProgressTerm();
      dfsProgressThroughput( bps);
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN (rc);
}                                               // end 'dfsByteShiftLeftSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Move data in sectors over specified number of bytes towards end (Right)
/*****************************************************************************/
ULONG dfsByteShiftRightSector
(
   ULONG               shift,                   // IN    shift amount 0 .. 511
   ULN64               start,                   // IN    starting lsn
   ULN64               size                     // IN    number of sectors todo
)
{
   ULONG               rc  = NO_ERROR;          // function result
   ULN64               lsn;                     // Current LSN
   USHORT              bps = dfsGetSectorSize(); // bytes per sector
   BYTE               *mbp = rbuf + bps;        // memory byte ptr (2nd sect)
   ULONG               bs;                      // blocksize in sectors
   ULN64               sects;                   // nr of sectors todo now
   ULN64               i;
   BYTE               *prev;

   ENTER();

   if ((prev = TxAlloc( 1, bps)) != NULL)       // ZEROED initial prev
   {
      bs = dfsGetBufferSize( DFSOPTIMALBUF, (RBUFBYTES / bps) / 2);
      if (DFSTORE_WRITE_ALLOWED)                // lock or ask to ignore
      {
         dfsX10("\nShift startsector : ", start, "", " ");
         dfsSz64Bps("size : ", size, bps, "\n");
         TxPrint(  "Sector byte shift : 0x%2.2lx = %3lu bytes RIGHT, buffer% 6lu sect", shift, shift, bs);
         dfsSize( " = ", bs, "\n");

         TRACES(( "start:%llx size:%llx bs:%8.8lx\n", start, size, bs));
         dfsProgressInit( start, size, 0, "Sector:", "shifted", DFSP_BARS, 0);

         for ( i  = 0,   lsn = start, sects = bs;
              (i < size) && (rc == NO_ERROR) && !TxAbort();
               i += bs,  lsn += bs)
         {
            if ((i + sects) > size)             // incomplete block
            {
               sects = size - i;
            }
            TRACES(( "i:%llx lsn:%llx sects:%8.8lx\n", i, lsn, sects));

            //- read in 2nd sect and put 'previous' sector in first
            //- save last sector for next round, and shift right ...
            rc = dfstBadSectorRead(  DFSTORE, lsn, sects, NULL, NULL, mbp);
            memcpy( rbuf, prev,            bps); // use from last
            memcpy( prev, rbuf + bs * bps, bps); // save for next
            if (rc == NO_ERROR)
            {
               memmove( mbp + shift, mbp, sects * bps);
               rc = dfsWrite( lsn, sects, mbp); // write shifted back
            }
            dfsProgressShow( lsn + sects, 0, 0, NULL);
         }
         dfsProgressTerm();
         dfsProgressThroughput( bps);
      }
      else
      {
         rc = DFS_READ_ONLY;
      }
      TxFreeMem( prev);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsByteShiftRightSector'
/*---------------------------------------------------------------------------*/

