//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// Sub-command / script variable handling
//
// Author: J. van Wijk
//

#include <txlib.h>                              // TX library interface

#include <dfsver.h>                             // DFS version info
#include <dfsscvar.h>                           // Subcommand/script handling
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions

// Read specified sector, convert to allocated hex-string (1024 chars)
static ULONG dfsGetSectorHexString
(
   ULONG               lsn,                     // IN    sectornumber
   TXSVALUE           *v                        // INOUT new variable value
);

// Read specified sector, keep the RAW binary format
static ULONG dfsGetSectorBinString
(
   ULONG               lsn,                     // IN    sectornumber
   TXSVALUE           *v                        // INOUT new variable value
);


/*****************************************************************************/
// Get host (dfsee) value for named variable and optional (array) index
/*****************************************************************************/
ULONG dfsGetVariable
(
   char               *name,                    // IN    variable name
   ULONG               ord,                     // IN    index or TXS...
   TXSVALUE           *v                        // INOUT new variable value
)                                               //       initialized to ZERO
{
   ULONG               rc    = NO_ERROR;        // function return
   char               *h     = name +1;
   BOOL                stype = FALSE;           // it is a string
   TXLN                str;                     // string output
   DFSDISKINFO        *d;
   DFSPARTINFO        *p;

   ENTER();

   if (ord == TXSCONSTANT)                      // resolve constant value
   {
      if      (strcasecmp(  name, "rc_value_error" ) == 0) v->num = DFS_VALUE_ERROR;
      else if (strcasecmp(  name, "rc_not_found"   ) == 0) v->num = DFS_NOT_FOUND;
      else if (strcasecmp(  name, "rc_list_empty"  ) == 0) v->num = DFS_LIST_EMPTY;
      else if (strcasecmp(  name, "rc_no_script"   ) == 0) v->num = DFS_NO_SCRIPT;
      else if (strcasecmp(  name, "rc_no_device"   ) == 0) v->num = DFS_NO_DEVICE;
      else if (strcasecmp(  name, "rc_psn_limit"   ) == 0) v->num = DFS_PSN_LIMIT;
      else if (strcasecmp(  name, "rc_st_mismatch" ) == 0) v->num = DFS_ST_MISMATCH;
      else if (strcasecmp(  name, "rc_reboot_req"  ) == 0) v->num = DFS_REBOOT_REQ;
      else if (strcasecmp(  name, "rc_user_abort"  ) == 0) v->num = DFS_USER_ABORT;
      else if (strcasecmp(  name, "rc_no_change"   ) == 0) v->num = DFS_NO_CHANGE;
      else if (strcasecmp(  name, "rc_cmd_failed"  ) == 0) v->num = DFS_CMD_FAILED;
      else if (strcasecmp(  name, "rc_cmd_warning" ) == 0) v->num = DFS_CMD_WARNING;
      else
      {
         rc = TX_FAILED;
      }
   }
   else if (name[0] == '_')                     // resolve host variable
   {
      memset( str, 0, TXMAXLN);
      if (ord != TXSNOINDEX)
      {
         h = name +3;
         if ((strlen(name) > 2) && (name[2] != '_')) // intended as a scalar var
         {
            rc = TX_INVALID_HANDLE;             // index on scalar variable
         }
         else if (toupper(name[1]) == 'D')      // array of disk-level info
         {
            if ((d = dfsGetDiskInfo( ord)) != NULL)
            {
               if      (!strcasecmp( h, "creatr"   )) {strcpy(str, d->creatr);         stype = TRUE;}
               else if (!strcasecmp( h, "drive"    )) {strcpy(str, d->ddrive);         stype = TRUE;}
               else if (!strcasecmp( h, "fsform"   )) {strcpy(str, d->fsform);         stype = TRUE;}
               else if (!strcasecmp( h, "label"    )) {strcpy(str, d->dlabel);         stype = TRUE;}
               else if (!strcasecmp( h, "pstyle"   )) {strcpy(str, d->pStyle);         stype = TRUE;}
               else if (!strcasecmp( h, "lvmname"  )) {strcpy(str, d->DiskName);       stype = TRUE;}
               else if (!strcasecmp( h, "unixdev"  )) {strcpy(str, d->UnixDeviceDisk); stype = TRUE;}
               else if (!strcasecmp( h, "access"   )) v->num =(d->OpenError) ? 0 : 1;
               else if (!strcasecmp( h, "cylsize"  )) v->num = d->cSC;
               else if (!strcasecmp( h, "disknr"   )) v->num = d->disknr;
               else if (!strcasecmp( h, "firstpart")) v->num = d->firstpart;
               else if (!strcasecmp( h, "flags"    )) v->num = d->flags;
               else if (!strcasecmp( h, "frareas"  )) v->num = d->freeSpace;
               else if (!strcasecmp( h, "frsects"  )) v->num = d->freeSects;
               else if (!strcasecmp( h, "geobps"   )) v->num = d->bpsector;
               else if (!strcasecmp( h, "geocyls"  )) v->num = d->geoCyls;
               else if (!strcasecmp( h, "geohead"  )) v->num = d->geoHeads;
               else if (!strcasecmp( h, "geosect"  )) v->num = d->geoSecs;
               else if (!strcasecmp( h, "largeflop")) v->num =(d->flags & DFS_F_FSYSTEMONLY) ? 1 : 0;
               else if (!strcasecmp( h, "cryptdisk")) v->num =(d->flags & DFS_F_CRYPTO_DISK) ? 1 : 0;
               else if (!strcasecmp( h, "lastpart" )) v->num = d->lastpart;
               else if (!strcasecmp( h, "lvmparts" )) v->num = d->lvmPartitions;
               else if (!strcasecmp( h, "ntsig"    )) v->num = d->NTsignature;
               else if (!strcasecmp( h, "parts"    )) v->num = d->totalpart;
               else if (!strcasecmp( h, "primaries")) v->num = d->primaries;
               else if (!strcasecmp( h, "removable")) v->num =(d->Removable) ? 1 : 0;
               else if (!strcasecmp( h, "size"     )) v->num = d->sectors;
               else
               {
                  rc = TX_FAILED;
               }
            }
            else                                // invalid disknr, var not found
            {
               rc = TX_FAILED;
            }
         }
         else if (toupper(name[1]) == 'P')      // array of partition  info
         {
            if ((p = dfsGetPartInfo( ord)) != NULL)
            {
               if      (!strcasecmp( h, "bmname"   )) {strcpy(str, p->bmname); stype = TRUE;}
               else if (!strcasecmp( h, "creatr"   )) {strcpy(str, p->creatr); stype = TRUE;}
               else if (!strcasecmp( h, "descr"    )) {strcpy(str, p->descr);  stype = TRUE;}
               else if (!strcasecmp( h, "drive"    )) {strcpy(str, p->drive);  stype = TRUE;}
               else if (!strcasecmp( h, "fsform"   )) {strcpy(str, p->fsform); stype = TRUE;}
               else if (!strcasecmp( h, "label"    )) {strcpy(str, p->plabel); stype = TRUE;}
               else if (!strcasecmp( h, "unixdev"  )) {strcpy(str, p->UnixDevicePart); stype = TRUE;}
               else if (!strcasecmp( h, "lvmpart"  )) {strcpy(str, p->lvm.PartName); stype = TRUE;}
               else if (!strcasecmp( h, "lvmvolume")) {strcpy(str, p->lvm.VoluName); stype = TRUE;}
               else if (!strcasecmp( h, "lvmletter")) {str[0] =(p->lvm.letter) ? p->lvm.letter:'-'; stype = TRUE;}
               else if (!strcasecmp( h, "active"   )) v->num =(p->partent.Status & 0x80) ? 1 : 0;
               else if (!strcasecmp( h, "basepsn"  )) v->num = p->basePsn;
               else if (!strcasecmp( h, "checksum" )) v->num = p->checksum;
               else if (!strcasecmp( h, "cluster"  )) v->num = p->scluster;
               else if (!strcasecmp( h, "cylsize"  )) v->num = p->cSC;
               else if (!strcasecmp( h, "disknr"   )) v->num = p->disknr;
               else if (!strcasecmp( h, "diskpart" )) v->num = p->diskPart;
               else if (!strcasecmp( h, "flags"    )) v->num = p->flags | ((LLONG)(p->flag2) << 32);
               else if (!strcasecmp( h, "geobps"   )) v->num = p->bpsector;
               else if (!strcasecmp( h, "geocyls"  )) v->num = p->geoCyls;
               else if (!strcasecmp( h, "geohead"  )) v->num = p->geoHeads;
               else if (!strcasecmp( h, "geosect"  )) v->num = p->geoSecs;
               else if (!strcasecmp( h, "lastpsn"  )) v->num = p->lastPsn;
               else if (!strcasecmp( h, "unixnr"   )) v->num = p->UnixDevNr;
               else if (!strcasecmp( h, "lvminfo"  )) v->num =(p->lvmPresent)            ? 1 : 0;
               else if (!strcasecmp( h, "lvminstal")) v->num =(p->lvm.Installable)       ? 1 : 0;
               else if (!strcasecmp( h, "lvmonmenu")) v->num =(p->lvm.OnBmMenu)          ? 1 : 0;
               else if (!strcasecmp( h, "partpsn"  )) v->num = p->partPsn;
               else if (!strcasecmp( h, "primary"  )) v->num =(p->primary)               ? 1 : 0;
               else if (!strcasecmp( h, "size"     )) v->num = p->sectors;
               else if (!strcasecmp( h, "tableslot")) v->num = p->partnr;
               else if (!strcasecmp( h, "type"     )) v->num = p->partent.PartitionType;
               else
               {
                  rc = TX_FAILED;
               }
            }
            else                                // invalid partnr, var not found
            {
               rc = TX_FAILED;
            }
         }
         else if (toupper(name[1]) == 'F')      // array of freespace info
         {
            if ((p = dfsGetFreeInfo( ord)) != NULL)
            {
               if      (!strcasecmp( h, "descr"    )) {strcpy(str, p->descr); stype = TRUE;}
               else if (!strcasecmp( h, "basepsn"  )) v->num = p->basePsn;
               else if (!strcasecmp( h, "cylsize"  )) v->num = p->cSC;
               else if (!strcasecmp( h, "disknr"   )) v->num = p->disknr;
               else if (!strcasecmp( h, "geobps"   )) v->num = p->bpsector;
               else if (!strcasecmp( h, "geocyls"  )) v->num = p->geoCyls;
               else if (!strcasecmp( h, "geohead"  )) v->num = p->geoHeads;
               else if (!strcasecmp( h, "geosect"  )) v->num = p->geoSecs;
               else if (!strcasecmp( h, "lastpsn"  )) v->num = p->lastPsn;
               else if (!strcasecmp( h, "size"     )) v->num = p->sectors;
               else if (!strcasecmp( h, "type"     )) v->num = p->partent.PartitionType;
               else
               {
                  rc = TX_FAILED;
               }
            }
            else
            {
               rc = TX_FAILED;
            }
         }
         else if ((toupper(name[1]) == 'S') && !strcasecmp( h, "list"))
         {
            if (ord < dfsa->snlist[0])
            {
               v->num = dfsa->snlist[ord +1];
            }
            else
            {
               rc = TX_FAILED;
            }
         }
         else if ((toupper(name[1]) == 'H') && !strcasecmp( h, "sector"))
         {
            switch (rc = dfsGetSectorHexString( ord, v))
            {
               case NO_ERROR:
               case TX_ALLOC_ERROR:
                  break;

               default:
                  TxFreeMem( v->str);
                  v->mem = 0;
                  rc = TX_FAILED;
                  break;
            }
         }
         else if ((toupper(name[1]) == 'B') && !strcasecmp( h, "sector"))
         {
            switch (rc = dfsGetSectorBinString( ord, v))
            {
               case NO_ERROR:
               case TX_ALLOC_ERROR:
                  //- to be refined, make TXS string value
                  break;

               default:
                  TxFreeMem( v->str);
                  v->mem = 0;
                  rc = TX_FAILED;
                  break;
            }
         }
      }
      else                                      // scalar variables
      {
         if      (!strcasecmp( h, "secdesc"   )) {dfsSectorTypeAsAscii( dfsIdentifySector( nav.this, nav.this_sninfo, rbuf), str); stype = TRUE;}
         else if (!strcasecmp( h, "cmdsep"    )) {strcpy(str, dfsa->cmdSeparator); stype = TRUE;}
         else if (!strcasecmp( h, "dirsep"    )) {strcpy(str, FS_PATH_STR); stype = TRUE;}
         else if (!strcasecmp( h, "exedir"    )) {strcpy(str, dfsa->dfsExeDir);  stype = TRUE;}
         else if (!strcasecmp( h, "exename"   )) {strcpy(str, dfsa->dfsExeName); stype = TRUE;}
         else if (!strcasecmp( h, "drive"     )) {strcpy(str, SINF->drive); stype = TRUE;}
         else if (!strcasecmp( h, "fsys"      )) {strcpy(str, SINF->afsys); stype = TRUE;}
         else if (!strcasecmp( h, "platform"  )) {strcpy(str, DFS_NO); stype = TRUE;}
         else if (!strcasecmp( h, "program"   )) {strcpy(str, DFS_NP); stype = TRUE;}
         else if (!strcasecmp( h, "sectype"   )) {str[0] = dfsIdentifySector( nav.this, nav.this_sninfo, rbuf); stype = TRUE;}
         else if (!strcasecmp( h, "base"      )) v->num = dfstLSN2Psn(DFSTORE, 0);
         else if (!strcasecmp( h, "batch"     )) v->num = dfsa->batch;
         else if (!strcasecmp( h, "columns"   )) v->num = dfsa->sbWidth;
         else if (!strcasecmp( h, "disk"      )) v->num = SINF->disknr;
         else if (!strcasecmp( h, "disks"     )) v->num = dfsPartitionableDisks();
         else if (!strcasecmp( h, "down"      )) v->num = nav.down;
         else if (!strcasecmp( h, "fsdirty"   )) v->num =(dfsa->FsDirtyStatus) ? 1 : 0;
         else if (!strcasecmp( h, "fsfree"    )) v->num = dfsa->FsUnallocated;
         else if (!strcasecmp( h, "fsmaxsize" )) v->num = dfsa->FsExpandSize;
         else if (!strcasecmp( h, "fsminsize" )) v->num = dfsa->FsTruncPoint;
         else if (!strcasecmp( h, "geobps"    )) v->num = dfsGetSectorSize();
         else if (!strcasecmp( h, "geocyls"   )) v->num = dfstGeoCylinders(DFSTORE);
         else if (!strcasecmp( h, "geohead"   )) v->num = dfstGeoHeads(DFSTORE);
         else if (!strcasecmp( h, "geosect"   )) v->num = dfstGeoSectors(DFSTORE);
         else if (!strcasecmp( h, "lines"     )) v->num = dfsa->sbLength;
         else if (!strcasecmp( h, "listed"    )) v->num = dfsa->snlist[0];
         else if (!strcasecmp( h, "lvm"       )) v->num = dfsa->lvmPresent;
         else if (!strcasecmp( h, "magic"     )) v->num = DFS_V_EYE;
         else if (!strcasecmp( h, "object"    )) v->num = SINF->object;
         else if (!strcasecmp( h, "parts"     )) v->num = dfsPartitions();
         else if (!strcasecmp( h, "pid"       )) v->num = SINF->partid;
         else if (!strcasecmp( h, "retc"      )) v->num = dfsa->retc;
         else if (!strcasecmp( h, "readonly"  )) v->num = dfsa->ReadOnly;
         else if (!strcasecmp( h, "size"      )) v->num = dfsGetLogicalSize();
         else if (!strcasecmp( h, "store"     )) v->num = DFSTORE;
         else if (!strcasecmp( h, "this"      )) v->num = nav.this;
         else if (!strcasecmp( h, "up"        )) v->num = nav.up;
         else if (!strcasecmp( h, "xtra"      )) v->num = nav.xtra;
         else if (!strcasecmp( h, "version"   ))
         {
            //- note: Upto 14.9 'version' was 4 digits, so up to 1490
            //-       starting with 14.10 it is 5 digits: 14100, and 15.0 will be 15000
            v->num  = (LLONG) (atol( DFS_VN    ) * 1000); // major part 10 .. 26
            v->num += (LLONG) (atol( DFS_VN + 3) *   10); // minor part  1 .. 99
         }
         else
         {
            if ((strlen(name) > 2) && (name[2] == '_')) // intended as an array var
            {
               rc = TX_INVALID_HANDLE;
            }
            else
            {
               rc = TX_FAILED;
            }
         }
      }
      if (stype)                                // value is a 'C' string now
      {
         TxStrip( str, str, ' ', ' ');
         v->mem = TXSVSIZE(strlen(str));
         if ((v->str = TxAlloc( 1, v->mem)) != NULL)
         {
            strcpy( v->str, str);
            v->len = strlen(str);
         }
         else
         {
            rc = TX_ALLOC_ERROR;
         }
      }
   }
   else
   {
      rc = TX_FAILED;
   }
   if (rc == NO_ERROR)                          // did we find it ?
   {
      v->defined = TRUE;
   }
   RETURN (rc);
}                                               // end 'dfsGetVariable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read specified sector, convert to allocated hex-string (1024 chars)
/*****************************************************************************/
static ULONG dfsGetSectorHexString
(
   ULONG               lsn,                     // IN    sectornumber
   TXSVALUE           *v                        // INOUT new variable value
)
{
   ULONG               rc  = NO_ERROR;          // function return
   ULONG               bps = dfsGetSectorSize();
   ULONG               num = TXSVSIZE( bps * 2);
   BYTE               *sector = NULL;
   char               *hexstr = NULL;
   char               *s;
   int                 i;

   ENTER();



   if (((sector = TxAlloc( 1, bps)) != NULL) &&
       ((hexstr = TxAlloc( 1, num)) != NULL)  )
   {
      if ((rc = dfsRead( lsn, 1, sector)) == NO_ERROR)
      {
         for (i = 0, s = hexstr; i < bps; i++, s += 2)
         {
            sprintf( s, "%2.2hhx", sector[i]);
         }
         v->str = hexstr;
         v->mem = num;
         v->len = bps * 2;
      }
   }
   else
   {
      rc = TX_ALLOC_ERROR;
   }
   TxFreeMem( sector);
   RETURN (rc);
}                                               // end 'dfsGetSectorHexString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read specified sector, keep the RAW binary format
/*****************************************************************************/
static ULONG dfsGetSectorBinString
(
   ULONG               lsn,                     // IN    sectornumber
   TXSVALUE           *v                        // INOUT new variable value
)
{
   ULONG               rc  = NO_ERROR;          // function return
   ULONG               bps = dfsGetSectorSize();
   ULONG               num = TXSVSIZE( bps * 1);
   BYTE               *sector = NULL;

   ENTER();

   if ((sector = TxAlloc( 1, num)) != NULL)
   {
      if ((rc = dfsRead( lsn, 1, sector)) == NO_ERROR)
      {
         v->str = (char *) sector;
         v->mem = num;
         v->len = bps;
      }
   }
   else
   {
      rc = TX_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsGetSectorBinString'
/*---------------------------------------------------------------------------*/


#if defined (DEV32)
/*****************************************************************************/
// Execute a DFSee command as REXX Subcommand handler
/*****************************************************************************/
ULONG _System dfs_subcom
(
   PRXSTRING          cmd,                      /* command string (input)    */
   PUSHORT            err_flag,                 /* ERROR/FAILURE  (output)   */
   PRXSTRING          rv                        /* string retcode (output)   */
)
{
   ULONG               rc;
   TXLN                stbuf;
   TXTS                stem;
   ULONG               i;

   ENTER();
   TRARGS(("command: '%s'\n", cmd->strptr));
   if (cmd->strptr[0] == '@')                   // no echo for cmd
   {
      rc = dfsMultiCommand( cmd->strptr +1, 0, FALSE, FALSE, TRUE);
   }
   else
   {
      rc = dfsMultiCommand( cmd->strptr,    0, TRUE, TRUE, TRUE);
      TxPrint("\nOpened object: %s%s %s%s%s  "
              "u=0x%s0%llX%s x=0x%s0%llX%s "
              "this=0x%s0%llX%s  Base=0x%s0%llX%s\n",
              CBG, SINF->drive,
              CBM, SINF->afsys,                   CNN,
              CBC, nav.up,                        CNN,
              CBY, nav.xtra,                      CNN,
              CBM, nav.this,                      CNN,
              CNG, dfstLSN2Psn( DFSTORE, 0),      CNN);
   }

   (void) TxsRexxSetnum( "dfs_this",    nav.this , "%llX");
   (void) TxsRexxSetnum( "dfs_down",    nav.down , "%llX");
   (void) TxsRexxSetnum( "dfs_up",      nav.up   , "%llX");

   (void) TxsRexxSetnum( "dfs_number",  dfsa->number , "%lu");
   (void) TxsRexxSetnum( "dfs_disknr",  SINF->disknr , "%u");
   (void) TxsRexxSetnum( "dfs_partid",  SINF->partid , "%u");

   (void) TxsRexxSetvar( "dfs_drive", SINF->drive, strlen(SINF->drive));
   (void) TxsRexxSetvar( "dfs_afsys", SINF->afsys, strlen(SINF->afsys));

   (void) TxsRexxSetnum( "dfs_sn.0",  dfsa->snlist[0], "%lu");
   for (i = 1; i <= dfsa->snlist[0]; i++)
   {
      sprintf( stem, "dfs_sn.%lu", i);
      (void) TxsRexxSetnum( stem,  dfsa->snlist[i], "%8.8lX");
   }

   stbuf[0] = (char) dfsIdentifySector(nav.this, nav.this_sninfo, rbuf);
   stbuf[1] = ' ';
   dfsSectorTypeAsAscii( stbuf[0], &stbuf[2]);
   (void) TxsRexxSetvar( "dfs_type", stbuf, strlen(stbuf));
   (void) TxsRexxSetvar( "dfs_sect", rbuf,  4 * SECTORSIZE);

   sprintf(rv->strptr, "%ld", (LONG) rc);
   rv->strlength = strlen(rv->strptr);
   *err_flag = RXSUBCOM_OK;
   RETURN (0);
}                                               // end 'dfs_subcom'
/*---------------------------------------------------------------------------*/
#endif

