//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee major functions: bad-sector scanning, speed measurements ...
//
// Author: J. van Wijk
//
// JvW  07-01-2004   Initial version, split off from DFSUTIL.C
// JvW  27-07-2006   Split off IMAGE, CLONE, FIND, WIPE, SHIFT and RSIZE
// JvW  27-07-2006   Renamed rest of dfsmajor to dfsscan
// JvW  09-10-2017   Added 'continue' confirmation on write-error at start
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsscan.h>                            // bad-sector & speed scan
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfsupart.h>                           // PART utility functions
#include <dfstable.h>                           // SLT  utility functions


/*****************************************************************************/
// Find bad sectors on the disk-surface by reading and/or writing
/*****************************************************************************/
ULONG dfsBadSectorScan
(
   BOOL                write,                   // IN    do write test too
   BOOL                autoid,                  // IN    automatic Id display
   ULN64               start                    // IN    starting lsn
)
{
   ULONG               rc  = NO_ERROR;          // function result
   ULN64               lsn = start;             // Current LSN
   ULN64               slt = L64_NULL;          // last SLT related LSN
   ULONG               i;
   char               *rp;                      // read string pointer
   char               *ip;                      // inverse string pointer
   TXTM                text;
   ULONG               bs      = 0;             // buffersize in sectors
   ULONG               sz;                      // buffersize in bytes
   ULONG               bads;                    // bad sectors read
   USHORT              bps = dfsGetSectorSize(); // bytes per sector
   BYTE               *data;                    // read in data
   BYTE               *ibuf;                    // inverse to ibuf
   ULONG               skiparea = 0;            // skip size on bad-sectors
   ULONG               sectorsRead;             // number of sectors actually read
   BOOL                writeErrorCont = FALSE;  // continue on write errors

   ENTER();

   #if defined (DOS32)
      if (write)
      {
         //- need TWO DPMI compatible buffers for invert-write, so split the largest buffer (rbuf) in half
         bs = dfsGetBufferSize( DFSOPTIMALBUF, RBUFSECTORS / 2);
      }
      else                                      // use largest preallocated buffer as a whole
      {
         bs = dfsGetBufferSize( DFSOPTIMALBUF, RBUFSECTORS);
      }
      data = rbuf;                              // always used with DOS!
   #else
      bs = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);
      if (write)
      {
         if ((bs * 2) <= RBUFSECTORS)           // 2 buffers fit in standard I/O
         {
            data = rbuf;                        // use the preallocated buffer
         }
         else
         {
            data = TxAlloc( bps, bs * 2);       // large buffer, non-DOS only!
         }
      }
      else                                      // need one buffer only
      {
         if (bs <=  RBUFSECTORS)                // fits in the standard I/O buffer
         {
            data = rbuf;                        // use the preallocated buffer
         }
         else
         {
            data = TxAlloc( bps, bs);           // large buffer, non-DOS only!
         }
      }
   #endif

   if (data != NULL)
   {
      if (TxaOptSet( DFS_O_SKBADS))
      {
         skiparea = TxaOptNum( DFS_O_SKBADS, NULL, 2048);
      }
      nav.xtra = start;                         // remember start position

      sz   = bs * bps;                          // size single FULL buffer
      ibuf = data + sz;                         // second buffer for write

      if ((!write) || DFSTORE_WRITE_ALLOWED)    // lock or ask to ignore
      {
         TXTIMER       stime = TxTmrGetNanoSecFromStart();

         TxPrint( "SectorScan method : %s Read%s", (bs == 1) ? "Precise" : "Fast",
                                                   (write)   ? "/Write"  : "-only ");
         dfsX10( " At: ", lsn, "", ", ");
         TxPrint( "buffer %lu sect", bs);
         dfsSize( " = ", bs, "\n");
         if (skiparea != 0)
         {
            dfsSiz8( "Size to skip on each BAD-sector area  : ", skiparea, "\n");
         }
         TxPrint( "(Use the <Esc> key at any time to abort the scan)\n");
         dfsProgressInit( start, dfsGetLogicalSize() - start, 0, "Sector:", "scanned", DFSP_BARS, 0);
         TxFsAutoFailCriticalErrors( TRUE);     // avoid Not-ready pop-ups
         dfsInitList( 0, "-w", "-d");           // with default list options
         do
         {
            strcpy( text, "SectorRead ");
            rc = dfstBadSectorRead( DFSTORE, lsn, bs, &sectorsRead, &bads, data);
            if (rc == NO_ERROR)
            {
               bads = sectorsRead;                       // errors are all sectors read
               sz   = sectorsRead * bps;                 // size buffer,  actually read
               if (write)
               {
                  for (i = 0,  rp = (char *) data, ip = (char *) ibuf;
                       i < sz;
                       i++,    rp++,      ip++)
                  {
                     *ip = (char) ~(*rp);       // inverse copy char
                  }
                  strcpy( text,          "InvertWrite");
                  rc = dfsWrite( lsn, sectorsRead, ibuf);
                  if (rc == NO_ERROR)
                  {
                     strcpy( text,       "VerifyRead ");
                     memset( ibuf, 0xee, sz);   // clear buffer before read
                     rc = dfsRead( lsn, sectorsRead, ibuf);
                     if (rc ==  NO_ERROR)
                     {
                        for (i = 0,  ip = (char *) ibuf;
                             i < sz;
                             i++,    ip++)
                        {
                           *ip = (char) ~(*ip); // inverse in-place
                        }
                        rc = memcmp( data, ibuf, sz);
                        strcpy( text, "VerifyComp ");
                     }
                  }
                  else if (writeErrorCont == FALSE)        // not confirmed yet
                  {
                     if (TxConfirm( 5220, "Sectors could be read, but not written, "
                                          "disk/partition might be write-protected"
                                          "\n\nContinue the SCAN ? [Y/N]: ", text))
                     {
                        writeErrorCont = TRUE;
                     }
                     else
                     {
                        TxSetPendingAbort();    // user aborted
                        break;                  // end the loop
                     }
                  }
                  (void) dfsWrite( lsn, sectorsRead, data);  //- Write original data back
               }                                             //- may fail after write error
            }
            if (rc == NO_ERROR)
            {
               dfsProgressShow( lsn + sectorsRead, 0, dfsa->snlist[0], "Bad:");
            }
            else if (rc != DFS_PSN_LIMIT)       // bad-sector found
            {
               ULONG      c,h,s;                // chs numbering
               ULONG      psn;

               dfsProgressSuspend();            // allow regular screen output
               dfsX10("Bad spot found at : ", lsn, CBW, TREOLN);
               TxPrint("  %2lu sectors %s RC:%3lu ", bads, text, rc);

               if (SINF->disknr)                // physical access
               {
                  psn = dfstLSN2Psn( DFSTORE, lsn);
                  dfstPsn2Chs( DFSTORE, psn, &c, &h, &s);
                  TxPrint("%s CHS:%7lu %3lu %-3lu", CNY, c,  h,  s);
                  dfsX10( " = ", psn, CNG, "");
               }
               TxPrint("\n");

               if ((autoid) && (dfsSlTableStatus(&i) == SLT_READY))
               {
                  DFSNL_ACTIVE(FALSE);          // protect the list!
                  dfsDisplaySltIdInfo( lsn, &slt);
                  DFSNL_ACTIVE(TRUE);           // allow updating again
               }
               dfsProgressResume("");
               for (i = 0; i < sectorsRead; i++)
               {
                  if ((bads == sectorsRead) ||           // whole buffer (like write)
                      (TxCrc32( data + i * bps, bps) == DFS_BADSEC_CRC32))
                  {
                     dfsAdd2SectorList( lsn+i); // add to list
                  }
               }
               lsn += skiparea;                 // extra area to skip on bads
            }
            lsn += sectorsRead;
         } while (!TxAbort() && DFSNL_ROOM && (rc != DFS_PSN_LIMIT) && (sectorsRead == bs));
         dfsProgressTerm();
         dfsDisplayThroughput( stime, lsn - start +1, dfsGetSectorSize());

         if (!DFSNL_ROOM)
         {
            TxPrint("Maximum number of bad-sectors reached\n");
         }
         TxFsAutoFailCriticalErrors( FALSE);    // enable criterror handler
         if (TxAbort())
         {
            rc = DFS_USER_ABORT;
            nav.down = lsn;                     // remember Abort position
         }
         else if (rc == DFS_PSN_LIMIT)
         {
            //- rc = NO_ERROR;
         }
         if (data != rbuf)
         {
            TxFreeMem( data);
         }
      }
      else
      {
         rc = DFS_READ_ONLY;
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsBadSectorScan'
/*---------------------------------------------------------------------------*/


#define DFS_MAX_SPEEDSIZE  (0xA00000 * SECTORSIZE / dfsGetSectorSize()) // allow 500 MiB/sec for 100%
/*****************************************************************************/
// Measure I/O speed for Read-only or Read + Write same area (buffer)
/*****************************************************************************/
void dfsMeasureIoSpeed
(
   BOOL                write,                   // IN    Read/Write
   ULONG               start                    // IN    start sector
)
{
   ULONG               rc  = NO_ERROR;          // function result
   ULN64               lsn = start;             // Current LSN
   ULONG               bs;                      // blocksize in sectors
   ULN64               msecs;                   // number of millisecond intervals
   ULONG               interval;                // buffers per report interval
   TXTIMER             now;                     // momentary timestamp value
   int                 i;                       // buffer counter
   BYTE               *data;
   USHORT              bps   = dfsGetSectorSize(); // bytes per sector
   TXTIMER             stime;
   TXTIMER             mtime = TxTmrSetTimer( TMR_SEC(6)); // total duration timer

   ENTER();

   if ((!write) || DFSTORE_WRITE_ALLOWED)
   {
      bs = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);
      if (bs <= RBUFSECTORS)                    // fits in the standard I/O buffer
      {
         data = rbuf;                           // always used with DOS!
      }
      else
      {
         data = TxAlloc( bps, bs);              // large buffer, non-DOS only!
      }
      TRACES(("bs:%lu RUBUFSECTORS:%lu bps:%hu  data:%p  rbuf:%p\n", bs, RBUFSECTORS, bps, data, rbuf));
      if (data != NULL)
      {
         switch (dfstStoreType(DFSTORE))
         {
            case DFST_MEMORY:  interval = 9;  break;
            case DFST_PHDISK:  interval = 3;  break;
            default:           interval = 1;  break;
         }
         if      (!write)      interval *=  5;  // reading faster than writing
         if      (bs <  10)    interval *=  3;  // small buffer
         else if (bs >  19999) interval  =  1;  // huge  buffer > 10 MiB
         else if (bs >= RBUFS) interval /=  5;  // large buffer
         interval++;                            // avoid a divide by zero, minimum interval is 1

         TRACES(( "buffersize: %lu  Interval: %u\n", bs, interval));
         TxPrint( "\nI/O speed for max : 6 seconds, testing Read%s, buffer% 7lu sect", (write) ? "/Write"  : "-only", bs);
         dfsSize( " = ", bs, "\n");
         #if defined (DOS32) || defined (DEV32)
            if (bs != dfstGeoSectors( DFSTORE))
            {
               TxPrint( "Buffersize does NOT equal tracksize, using the '-b' option may be faster.\n");
            }
         #else
            if (bs <= dfstGeoSectors( DFSTORE))
            {
               TxPrint( "Buffersize less than, or equal to tracksize, using a larger buffer may be faster,\n"
                        "you can try the -b:N option with values up to %lu sectors (64 MiB)\n",  DFSMAXBUFSIZE);
            }
         #endif

         //- Progress in milliseconds, over a maximum of 6 seconds
         dfsProgressInit( 0, 6000, 0, "milliSecond:", "I/O speed tested", DFSP_BARS | DFSP_VDEC | DFSP_NO_ETA, 0);

         TxFsAutoFailCriticalErrors( TRUE);     // avoid Not-ready pop-ups

         stime = TxTmrGetNanoSecFromStart();    // for throughput
         for (i = 0; ((i < 3) || (!TxTmrTimerExpired( mtime))) && (rc != DFS_PSN_LIMIT) && !TxAbort(); i++)
         {
            rc = dfsRead( lsn, bs, data);       // normal/fast read
            if ((write) && (rc == NO_ERROR))
            {                                   // Write original data back
               dfsWrite( lsn, bs, data);        // ignore write RC
            }
            lsn += bs;                          // sectors done sofar
            if ((i % interval) == 0)            // read time every interval
            {
               now   = TxTmrGetNanoSecFromStart();
               msecs = ((ULN64) (now - stime) / 1000000);
               dfsProgressShow( msecs, 0, 0, NULL);
            }
         }
         dfsProgressTerm();
         dfsDisplayThroughput( stime, lsn - start +1, bps);

         TxFsAutoFailCriticalErrors( FALSE);    // enable criterror handler
         if (data != rbuf)
         {
            TxFreeMem( data);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   VRETURN();
}                                               // end 'dfsMeasureIoSpeed'
/*---------------------------------------------------------------------------*/

