//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee major functions: partition resizing
//
// Author: J. van Wijk
//
// JvW  27-01-2006   Initial version, split off from DFSMAJOR.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsrsize.h>                           // partition resize/expand
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfsupart.h>                           // PART utility functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfscfdsk.h>                           // FDSK command functions
#include <dfsufgpt.h>                           // FDSK GPT utility and defs



static  char       *cmd_resize[] =
{
   "",
   "",
   "Resize a filesystem inside a partition, update partition table to new size",
   "",
   " Usage: RESIZE  size | -s:remain | -s | -f:size | -f",
   "",
   "   size         : new size as mcs-number (see DFSTERMS.TXT)",
   "",
   "   -c:[0|1|2]   : CHS dummy style: 1=PQmagic, 2=MS, default 0=IBM/DFSee",
   "   -f:size      : size to free-up as mcs-number",
   "   -f           : free-up half the available size",
   "   -m           : maximum size, resize FS to current partition size",
   "   -s:remain    : remaining size as mcs-number (see DFSTERMS.TXT)",
   "   -s           : split partition in half",
   "",
   " A partition MUST be selected using 'part' before using this command!",
   "",
   " The mcs-number format is [0x]nnnn[,g|m|k|c|t|s] a HEX or decimal value,",
   " in GiB, MiB, KIb, Cylinders, Tracks or Sectors. Default is decimal MiB.",
   "",
   " Note: The new size must be equal or smaller than the partition size, this",
   "       means you must delete+recreate the partition first to the new size,",
   "       or use the 'EXPAND' command if you want to make it larger (expand).",
   NULL
};

static  char       *cmd_expand[] =
{
   "",
   "Expand a partition and filesystem, using freespace AFTER the partition",
   "",
   " Usage: EXPAND  [size] [options]",
   "",
   "   size         : new size as mcs-number (see DFSTERMS.TXT)",
   "",
   "   -F           : Force resize even if larger than indicated",
   "                  maximum size. Can be useful if allocation",
   "                  information is not correct.",
   "   -s           : maximum size, use ALL freespace",
   "   -size:new    : new size as mcs-number (see DFSTERMS.TXT)",
   "",
   " A partition MUST be selected using 'part' before using this command!",
   "",
   " The mcs-number format is [0x]nnnn[,g|m|k|c|t|s] a HEX or decimal value,",
   " in GiB, MiB, KIb, Cylinders, Tracks or Sectors. Default is decimal MiB.",
   "",
   " Examples: 'EXPAND 1500'     Expand to new size of 1500 MiB",
   "           'EXPAND -s:5000'  Expand to 5000 MiB",
   "           'EXPAND -s'       Expand to maximum size, using ALL freespace",
   "           'EXPAND'          Prompt for new size, then expand ...",
   "",
   NULL
};

static  char        cleanResize[] =
         "Resizing any filesystem should only be done "
         "when the filesystem is not in use and CLEAN. "
         "This usually means you need to resize when booted "
         "from a diskette or another (maintenance) partition, "
         "and the filesystems had a regular shutdown or a CHKDSK\n\n"
         "Filesystem status as seen by DFSee now is %s!\n";

static  char        ntfsResize[] =
         "If this NTFS partition contains a BOOTABLE Windows system, make sure "
         "you use 'CHKDSK /f' or the similar function from the properties->tools "
         "menu in Windows before resizing! Defragmenting is recommended as well.\n\n"
         "When possible, resize from ANOTHER OS like a boot-diskette or CDROM";

static  char        doneResize[] =
         "of partition completed.\n\n"
         "The partition needs to be rediscovered by the operating-system "
         "that wants to use it. The safest way to do that is (re)booting\n\n";

static  char        needChkdsk[] =
         "After reboot a full CHKDSK needs to be run, most likely it will "
         "be done automatically upon booting the OS that uses the partition.\n\n"
         "When NOT, schedule a 'CHKDSK /f' on the partition manually.\n"
         "Failing to do this MAY result in a non-accessible partition!";

static  char        wantChkdsk[] =
         "After reboot you may want to run a full 'CHKDSK /f' on the partition, "
         "allthough the resizing leaves the filesystem without any errors ...";


/*****************************************************************************/
// Resize current partition to make it SMALLER, update Ptables and LVM
/*****************************************************************************/
ULONG dfsResizePartition
(
   char               *param                    // IN    size param or NULL
)
{
   ULONG               rc = NO_ERROR;           // function return
   BOOL                ok = TRUE;               // arg/option combinations OK
   TXA_OPTION         *opt;                     // option pointer

   ENTER();

   if (((param) && (param[0] == '?')) || TxaOption('?'))
   {
      TxShowTxt( cmd_resize);
   }
   else if (SINF->p != NULL)                    // partition selected ?
   {
      DFSPARTINFO  *p  = SINF->p;
      ULN64         fs = p->sectors;
      TXTS          a1;                         // resulting argument values

      ok &= TxaOptMutEx((param != NULL), "fsm", "if 'size' is specified!", 0);

      if (dfsa->FsTruncPoint == 0)
      {
         DFSFNCALL( dfsa->FsAllocDisplay, 0,0, "@", NULL);
      }
      if (param != NULL)                         // explicit size
      {
         TxCopy( a1, param, TXMAXTS);
         TxPrint( "\nRESIZE to a remaining size of %s\n", a1);
      }
      else if ((opt = TxaOptValue('m')) != NULL) // maximum size
      {
         sprintf( a1, "%.0lf", TXSMIB( fs, p->bpsector) - (double) 0.5);
         TxPrint( "\nRESIZE to the maximum (partition) size ="
                  " %.0lf MiB\n", TXSMIB( fs, p->bpsector) - (double) 0.5);
      }
      else if ((opt = TxaOptValue('s')) != NULL) // remaining size
      {
         switch (opt->type)
         {
            case TXA_NO_VAL:                    // split in half
               sprintf( a1, "%.0lf", TXSMIB((fs / 2), p->bpsector));
               TxPrint( "\nRESIZE to half the current size ="
                        " %.0lf MiB\n", TXSMIB((fs / 2), p->bpsector));
               break;

            case TXA_STRING:
               TxPrint( "\n* Need a NUMBER value for '-s' size option!");
               ok = FALSE;
               break;

            default:                            // remaining size
               sprintf(a1, "%llu,%c", opt->value.number, opt->unit);
               TxPrint( "\nRESIZE to a remaining size %s\n", a1);
               break;
         }
      }
      else if ((opt = TxaOptValue('f')) != NULL) // freespace wanted
      {
         ULONG   value;

         switch (opt->type)
         {
            case TXA_NO_VAL:                    // option only, half freespace
               value = (fs - dfsa->FsTruncPoint) / 2 + dfsa->FsTruncPoint;
               sprintf( a1, "%.0lf", TXSMIB(value, p->bpsector));
               TxPrint( "\nRESIZE to free-up half the available "
                        "space, remaining ="
                        " %.0lf MiB\n", TXSMIB((value), p->bpsector));
               break;

            case TXA_STRING:
               TxPrint( "\n* Need a NUMBER value for '-f' freespace option!");
               ok = FALSE;
               break;

            default:
               switch (opt->unit)
               {
                  case 's':                     // sectors, OK as is
                     value = opt->value.number;
                     break;

                  case 'h': case 't':           // heads (or tracks :-)
                     value = opt->value.number * p->geoSecs;
                     break;

                  case 'c':                     // cylinders
                     value = opt->value.number * p->cSC;
                     break;

                  case 'k':                     // Kilobytes
                     value = DFSKB2SECTS( opt->value.number, p->bpsector);
                     break;

                  default:                      // Megabytes or Gigabytes
                     value = DFSMB2SECTS( opt->value.number, p->bpsector);
                     if (opt->unit == 'g')
                     {
                        value *= 1024;          // Giga correction
                     }
                     break;
               }
               value += (p->cSC -1);            // round UP, almost a cylinder
               if (value < fs)
               {
                  sprintf( a1, "%.0lf", TXSMIB((fs - value), p->bpsector));
                  TxPrint( "\nRESIZE to get a freespace of almost "
                           " %.0lf MiB, remaining: %.0lf MiB\n",
                           TXSMIB((value), p->bpsector),
                           TXSMIB((fs - value), p->bpsector));
               }
               else
               {
                  TxPrint( "\n* Specified freespace too large for partition!");
                  ok = FALSE;
               }
               break;
         }
      }
      else if ((!TxaOptUnSet('P')) &&           // Interactive, prompt for size
               (!dfsa->batch)       )           // unless batchmode active
      {
         ULN64 pvalue = (p->lastPsn - p->basePsn + dfsa->FsTruncPoint) / 2;
         sprintf( a1, "%.0lf", TXSMIB( pvalue, p->bpsector));

         if (TxPrompt( 5541, 15, a1,
                      "Partition id: %2.2hu, type %2.2hu %2.2s %-6.6s\n"
                      "New partition size in megabytes (MiB)\n\n"
                      "Minimum: %.0lf MiB, maximum: %.0lf MiB",
               p->id, p->partent.PartitionType, p->drive, SINF->afsys,
               TXSMIB( (dfsa->FsTruncPoint), p->bpsector) +1.0,
               TXSMIB(((dfsa->FsExpandSize == 0) ? p->sectors : dfsa->FsExpandSize), p->bpsector) -0.5))
         {
            TxStrip( a1, a1, ' ', ' ');
            if (a1[0] == '*')                   // use half of maximum
            {
               sprintf( a1, "%.0lf", TXSMIB( pvalue, p->bpsector));
               TxPrint( "\nRESIZE to half the current size = %.0lf MiB\n", TXSMIB( pvalue, p->bpsector));
            }
         }
         else                                   // aborted from Prompt dialog
         {
            ok = FALSE;
            rc = DFS_NO_CHANGE;
         }
      }
      else                                      // wrong nr of params / options
      {
         ok = FALSE;
      }
      TRACES(( "ok: %s  rc:%u\n", (ok) ? "TRUE" : "FALSE", rc));

      if (ok)                                   // parameters OK sofar
      {
         FDSK_CB_INFO  cbi;                     // callback parameter struct
         FDSK_NUMVAL   size = FDSK_NUMZERO;     // size param info
         TXTM          dc;
         USHORT        pid;
         char          targetKind = (p->tablenr == GPT_STYLE) ? 'G' : (p->primary) ? 'P' : 'L';

         if (!TxaOption('X'))                   // unless called by other cmd
         {
            if (dfsa->FsModeId == DFS_FS_NTFS)
            {
               TxNamedMessage( !dfsa->batch, 5139, " INFO: NTFS resize hint ", ntfsResize);
            }
            TxNamedMessage( !dfsa->batch, 5019, " INFO: FS clean resize ", cleanResize, (dfsa->FsDirtyStatus) ? "DIRTY/IN-USE" : "CLEAN");
            TxPrint("\nSize partition %s%2u%s : ", CBG, p->id, CNN);
            dfsSz64Bps( "", fs, p->bpsector, ", ");
            TxPrint("type %s%s%s\n", CBM, p->descr, CNN);
         }

         dfsParseNewSize( a1, targetKind, p, NULL, &size);

         if (!TxaOptUnSet( DFS_O_ALIGN))        // unless alignment disabled
         {
            if (targetKind == 'G')              // GPT style
            {
               ULONG   kmgAlign = dfsAlignmentSectors( 'G', p);

               if ((p->basePsn % kmgAlign) == 0) // start is aligned now
               {
                  //- align the size value as well, so end will be aligned
                  size.value = (((size.value -1) / kmgAlign) + 1) * kmgAlign;
               }
            }
            else
            {
               ULONG         c, h, s;

               dfsGeoPsn2Chs( (p->basePsn + size.value -1), p->geoHeads, p->geoSecs, &c, &h, &s);
               if (h != (p->geoHeads -1))
               {
                  size.value -= ((h +1) * p->geoSecs); // head alignment
               }
               if (s != p->geoSecs)
               {
                  size.value -= s;              // adjust sector alignment
               }
            }
         }

         memset( &cbi, 0, sizeof(cbi));         // initial callback info
         cbi.disknr = p->disknr;
         cbi.flag   = p->primary;
         cbi.partnr = p->partnr;
         cbi.size   = &size;                    // attach to callback info
         cbi.number = size.value;
         cbi.sn     = p->basePsn;
         cbi.ntype  = p->partent.PartitionType;

         if (!TxaOption('X'))                   // unless called by other cmd
         {
            if (targetKind == 'G')              // GPT style
            {
               TxPrint("\nResize GPT style  : new size = %8.1lf MiB, type '%s'  on disk %u\n",
                          TXSMIB(cbi.number, p->bpsector), p->descr,  cbi.disknr);
            }
            else
            {
               TxPrint("\nResize : %s  : new size = %8.1lf MiB, type %02.2X = %s  on disk %u\n",
                (cbi.flag) ? "Primary" : "Logical", TXSMIB(cbi.number, p->bpsector), cbi.ntype, p->descr,  cbi.disknr);
               dfsGeoDispTransPsn("Part-table offset :", p->geoHeads, p->geoSecs, p->partPsn);
            }
            dfsGeoDispTransPsn(   "First part sector :", p->geoHeads, p->geoSecs, cbi.sn);
            dfsGeoDispTransPsn(   "Last  part sector :", p->geoHeads, p->geoSecs, cbi.sn -1 + cbi.number);
            dfsSz64Bps( "Size after update : ", size.value, p->bpsector, "\n\n");
         }

         if (size.value > fs)
         {
            TxPrint( "\n* Remaining size bigger than the current partition size!");
            rc = DFS_VALUE_ERROR;
         }
         else if ((dfsa->FsTruncPoint < size.value) || (TxaOption('F'))) // forced resize
         {
            TRACES(("Resize partinfo: 0x%8.8x, bpsector:%hu\n", p, p->bpsector));
            rc = DFSFNCALL( dfsa->FsTruncateSize, size.value, 0, NULL, p);
            switch (rc)
            {
               case NO_ERROR:                   // truncate partition-table
                  pid = SINF->partid;           // save partition id
                  if (size.value < fs)          // update needed ?
                  {
                     if (targetKind == 'G')     // GPT style
                     {
                        rc = dfsFdskResizeGptEntry( &cbi);
                        if (rc == NO_ERROR)
                        {
                           dfsReadDiskInfo( FDSK_QUIET); // re-read partition info
                        }
                     }
                     else
                     {
                        rc = dfsExecOnBootRec( p, cbi.disknr, dfsFdskTruncateMbr, &cbi);
                        if ((rc == NO_ERROR) && (p->primary == FALSE))
                        {
                           dfsReadDiskInfo( FDSK_QUIET); // re-read partition info
                           dfsCleanupExtChain( p->disknr, TRUE);
                        }
                        if ((dfsa->lvmPresent) && (rc == NO_ERROR))
                        {
                           sprintf( dc, "lvm -B -q %hu -sync", pid);
                           dfsMultiCommand( dc, 0, TRUE, FALSE, FALSE);
                           TxPrint(  "Updating LVM info : synchronize to partition size\n");
                        }
                     }
                  }
                  TxPrint( "\nResulting partition after the update:\n");
                  sprintf( dc, "part -r -q -a %hu", pid);
                  dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);

                  if (!TxaOption('X'))          // unless called by other cmd
                  {
                     TxPrint("\nResize filesystem : %s\n", (rc) ? "not completed" : "successful");
                     if (rc == NO_ERROR)
                     {
                        TxNamedMessage( !dfsa->batch, 5141, " INFO: Resize completed ", "RESIZE %s%s", doneResize,
                                   (dfsa->FsModeId == DFS_FS_HPFS)  ? needChkdsk
                                                                    : wantChkdsk);
                     }
                  }
                  break;

               case DFS_PENDING:
                  TxPrint( "\nResizing not yet supported in mode " "%s%s%s\n", CBM, SINF->afsys, CNN);
                  rc = DFS_CMD_FAILED;
                  break;

               default:
                  TxPrint( "\nFilesystem is NOT resized!\n");
                  break;
            }
         }
         else                                   // truncation not possible
         {
            TxPrint( "\nThe specified new size is too small to "
                     "contain the existing data!\n");
            rc = DFS_VALUE_ERROR;
         }
      }
      else if (rc != DFS_NO_CHANGE)             // unless canceled ...
      {
         TxPrint("\n");
         TxShowTxt( cmd_resize);                // give usage
         rc = DFS_VALUE_ERROR;
      }
   }
   else
   {
      TxPrint( "\n* Resizing is only supported on harddisk partitions.\n"
               "  Use the 'part' command to select the partition first!");
      TxShowTxt( cmd_resize);
      rc = DFS_CMD_FAILED;
   }
   RETURN (rc);
}                                               // end 'dfsResizePartition'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Resize current partition to make it LARGER, update Ptables and LVM
// Calls 'ResizeParition' through the 'resize' command to resize the FS!
/*****************************************************************************/
ULONG dfsExpandPartition
(
   char               *param                    // IN    size param or NULL
)
{
   ULONG               rc = NO_ERROR;           // function return
   BOOL                ok = TRUE;               // arg/option combinations OK
   TXA_OPTION         *opt;                     // option pointer

   ENTER();

   if (((param) && (param[0] == '?')) || TxaOption('?'))
   {
      TxShowTxt( cmd_expand);
   }
   else if (SINF->p != NULL)                    // partition selected ?
   {
      DFSPARTINFO     *p  = SINF->p;
      ULONG            fs = p->sectors;
      TXTS             a1;                      // resulting argument values
      char             targetKind = (p->tablenr == GPT_STYLE) ? 'G' : (p->primary) ? 'P' : 'L';
      ULONG            kmgAlign = dfsAlignmentSectors( targetKind, p);
      ULONG            alignSz  = (dfsa->FsExpandSize / kmgAlign) * kmgAlign;
      double           fslimit  = TXSMIB( alignSz, p->bpsector) - 0.5;

      if (param != NULL)                        // explicit size
      {
         TxCopy( a1, param, TXMAXTS);
         TxPrint( "\nEXPAND to a size of %s\n", a1);
      }
      else if ((opt = TxaOptValue('s')) != NULL) // remaining size
      {
         switch (opt->type)
         {
            case TXA_NO_VAL:                    // use maximum size
               sprintf( a1, "%.0lf", fslimit);
               TxPrint( "\nEXPAND to maximum size %s MiB, using ALL freespace when possible\n", a1);
               break;

            case TXA_STRING:
               TxPrint( "\n* Need a NUMBER value for '-s' size option!");
               ok = FALSE;
               break;

            default:                            // new explicit size
               sprintf(a1, "%llu,%c", opt->value.number, opt->unit);
               TxPrint( "\nEXPAND to a size of %s\n", a1);
               break;
         }
      }
      else if ((!TxaOptUnSet('P')) &&           // Interactive, prompt for size
               (!dfsa->batch)       )           // unless batchmode active
      {
         if (alignSz > p->sectors)
         {
            sprintf( a1, "%.0lf", fslimit);

            if (TxPrompt( 5541, 15, a1,
                         "Partition id: %2.2hu, type '%s' %2.2s %-6.6s\n"
                         "New partition size in megabytes (MiB)\n\n"
                         "Minimum: %.0lf MiB, maximum: %.0lf MiB",
                  p->id, p->descr, p->drive, SINF->afsys,
                  TXSMIB((p->sectors), p->bpsector) -0.5, fslimit))
            {
               TxStrip( a1, a1, ' ', ' ');
               if ((a1[0] == 0) || (a1[0] == '*')) // use maximum
               {
                  sprintf( a1, "%.0lf", fslimit);
                  TxPrint( "\nEXPAND to maximum size %s MiB, using ALL freespace when possible\n", a1);
               }
            }
            else                                // aborted from Prompt dialog
            {
               ok = FALSE;
               rc = DFS_NO_CHANGE;
            }
         }
         else
         {
            TxPrint( "\nPartition cannot be expanded at the moment!\n"
                     "Reported max-size does not allow increase in size.\n");
            ok = FALSE;
            rc = DFS_NO_CHANGE;
         }
      }
      else                                      // wrong nr of params / options
      {
         ok = FALSE;
      }
      TRACES(( "ok: %s  rc:%u\n", (ok) ? "TRUE" : "FALSE", rc));

      if (ok)                                   // parameters OK sofar
      {
         FDSK_CB_INFO  cbi;                     // callback parameter struct
         FDSK_NUMVAL   size = FDSK_NUMZERO;     // size param info
         double        newsize = 0.0;

         sscanf( a1, "%lf", &newsize);
         if ((newsize > fslimit) || (newsize == 0.0))
         {
            sprintf( a1, "%.0lf", fslimit);
            TxPrint( "\nLIMIT the new size to the maximum size of %s MiB for this filesystem\n", a1);
         }

         if (dfsa->FsModeId == DFS_FS_NTFS)
         {
            TxNamedMessage( !dfsa->batch, 5139, " INGFO: NTFS resize hint ", ntfsResize);
         }
         TxNamedMessage( !dfsa->batch, 5019, " INFO: FS clean resize ", cleanResize, (dfsa->FsDirtyStatus) ? "DIRTY/IN-USE" : "CLEAN");
         TxPrint("\nSize partition %s%2u%s : ", CBG, p->id, CNN);
         dfsSz64Bps( "", fs, p->bpsector, ", ");
         TxPrint("type %s%s%s\n", CBM, p->descr, CNN);

         memset( &cbi, 0, sizeof(cbi));         // initial callback info
         cbi.disknr = p->disknr;
         cbi.flag   = p->primary;

         dfsParseNewSize( a1, targetKind, p, NULL, &size);

         if (!TxaOptUnSet( DFS_O_ALIGN))        // unless alignment disabled
         {
            if (targetKind == 'G')              // GPT style
            {
               ULONG   kmgAlign = dfsAlignmentSectors( 'G', p);

               if ((p->basePsn % kmgAlign) == 0) // start is aligned now
               {
                  //- align the size value as well, so end will be aligned
                  size.value = (((size.value -1) / kmgAlign) + 1) * kmgAlign;
               }
            }
            else
            {
               ULONG         c, h, s;

               dfsGeoPsn2Chs( (p->basePsn + size.value -1), p->geoHeads, p->geoSecs, &c, &h, &s);
               if (h != (p->geoHeads -1))
               {
                  size.value -= ((h +1) * p->geoSecs); // head alignment
               }
               if (s != p->geoSecs)
               {
                  size.value -= s;              // adjust sector alignment
               }
            }
         }

         cbi.size   = &size;                    // attach to callback info
         cbi.number = size.value;
         cbi.partnr = p->partnr;
         cbi.sn     = p->basePsn;
         cbi.ntype  = p->partent.PartitionType;

         if (targetKind == 'G')                 // GPT style
         {
            TxPrint("\nExpand GPT style  : new size = %8.1lf MiB, type '%s'  on disk %u PTA entry:%hu\n",
                       TXSMIB(cbi.number, p->bpsector), p->descr,  cbi.disknr, cbi.partnr);
         }
         else
         {
            TxPrint("\nExpand : %s  : new size = %8.1lf MiB, type %02.2X = %s  on disk %u\n",
                      (cbi.flag) ? "Primary" : "Logical",
                       TXSMIB(cbi.number, p->bpsector), cbi.ntype, p->descr,  cbi.disknr);
            dfsGeoDispTransPsn( "Part-table offset :", p->geoHeads, p->geoSecs, p->partPsn);
         }
         dfsGeoDispTransPsn(    "First part sector :", p->geoHeads, p->geoSecs, cbi.sn);
         dfsGeoDispTransPsn(    "Last  part sector :", p->geoHeads, p->geoSecs, cbi.sn -1 + cbi.number);
         dfsSz64Bps( "Size after update : ", size.value, p->bpsector, "\n\n");

         if (size.value < fs)
         {
            TxPrint( "\n* Expanded size smaller than the current partition size!");
            rc = DFS_VALUE_ERROR;
         }
         else if (size.value > p->expandSize)
         {
            TxPrint( "\n* Expanded (cylinder aligned) size larger than available space!");
            rc = DFS_VALUE_ERROR;
         }
         else
         {
            BOOL       isActive = (p->partent.Status & 0x80);
            ULONG      startsec = (cbi.flag) ? p->basePsn : p->partPsn;
            USHORT     disknr   = p->disknr;
            USHORT     pid      = SINF->partid;
            TXLN       text;
            TXTM       dc;                      // command string

            strcpy( text, "");
            dfstrSizeBps( text, "to a new size of: ", size.value, p->bpsector, "");

            if ((dfsa->batch) || (TxConfirm( 5543,
                "Expand this %5.5s filesystem %s ? [Y/N] : ",
                 SINF->afsys, text)))
            {
               if (targetKind == 'G')           // GPT style
               {
                  rc = dfsFdskResizeGptEntry( &cbi);
                  if (rc == NO_ERROR)
                  {
                     dfsReadDiskInfo( FDSK_QUIET); // re-read partition info

                     //- quiet reselect partition, ResizeGptEntry selects whole disk!
                     sprintf( dc, "part -r -q -a- %hu", pid);
                     dfsMultiCommand( dc, 0, FALSE, FALSE, FALSE); // quiet reselect
                  }
               }
               else
               {
                  sprintf( dc, "fdisk -B delete %hu %hu -L-", pid, disknr);
                  rc = dfsMultiCommand( dc, 0, TRUE, FALSE, FALSE);
                  if (rc == NO_ERROR)
                  {
                     sprintf( dc, "cr -B -M %s 0x%2.2hhx %.0lf -a:0x%x,s -d:%hu%s -o",
                              (cbi.flag) ? "pri" : "log", cbi.ntype,
                              TXSMIB( size.value, p->bpsector) -0.5,
                              startsec, disknr, (isActive) ? " -F" : "");
                     rc = dfsMultiCommand( dc, 0, TRUE, FALSE, FALSE);
                  }
               }
               if (rc == NO_ERROR)
               {
                  if ((dfsa->FsTruncateSize != NULL) && // RESIZE the FS to MAX if truncation is
                      (dfsa->FsAllocDisplay != NULL)  ) // supported including ALLOC information
                  {
                     sprintf( dc, "resize -B -max -X");
                     rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE);
                  }
                  else
                  {
                     TxNamedMessage( !dfsa->batch, 0, " WARNING: filesystem not resized ",
                                "The partition has been expanded in size, but the contained filesystem "
                                "has not been touched (RESIZE not supported inside DFSee). You need to "
                                "use some other filesystem tool to complete the expand, or FORMAT the "
                                "partition to use its full size.");
                  }
               }
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
            TxPrint("\nExpand filesystem : %s\n", (rc) ? "not completed" : "successful");
            if (rc == NO_ERROR)
            {
               TxNamedMessage( !dfsa->batch, 5141, " INFO: Resize completed ", "EXPAND %s%s", doneResize,
                          (dfsa->FsModeId == DFS_FS_HPFS)  ? needChkdsk : wantChkdsk);
            }
         }
      }
      else if (rc != DFS_NO_CHANGE)            // unless canceled somewhere ...
      {
         TxPrint("\n");
         TxShowTxt( cmd_expand);                // give usage
         rc = DFS_VALUE_ERROR;
      }
   }
   else
   {
      TxPrint( "\nExpanding is only supported on harddisk partitions.\n"
               "Use the 'part' command to select the partition first!\n");
      TxShowTxt( cmd_expand);
      rc = DFS_CMD_FAILED;
   }
   RETURN (rc);
}                                               // end 'dfsExpandPartition'
/*---------------------------------------------------------------------------*/

