//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFSee registration functions
//
// 19-07-2009 JvW Updated to suppurt major versions > 9     (DFSee 10.0)
// 23-06-2005 JvW Moved to DFSee to allow 'open' txlib      (DFSee 7.07)
// 29-09-2001 JvW First released version                    (DFSee 4.04)

#include <txlib.h>                              // TxLib interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsrgkey.h>                           // Registration interface
#include <dfswin.h>                             // Help ID

#if defined (REGISTRATION)

#define DFSR_LASTDATE       22222222L           // latest accepted reg date and
                                                // PERMANENT key expiration!!!

static  char        fake_email[]  = "@DFSee.Fsys";

static BOOL            registred = FALSE;
static TXLN            keystring;               // RAW key contents, as read from file
static TXLN            regstring;               // decoded registry string, for feedback
static TXLN            regstatus;
static TXTS            prodname;
static BOOL            cracked = FALSE;
static char           *tmpkeyIndicator = NULL;  // temporary, keep nagging/informing

// Validate encrypted registration-date against limits
static BOOL dfsValidateRegdate
(
   char               *registration,            // IN    registration text
   int                 major,                   // IN    regfile major ver
   long                introduction,            // IN    introduction date
   char               *failuretext              // OUT   failure reason
);

// calculate number of days represented by (DFSRG) date (resembles Julian date)
static int dfsRegDays                           // RET   linear number of days
(
   int                 rgdate                   // IN    REG date, YYYYMMDD
);

// calculate DFSRG date from DFSRG number of days (resembles Julian date)
static int dfsRegDate                           // RET   REG date, YYYYMMDD
(
   int                 rgdays                   // IN    nr of days
);


// Get date of compilation in DFSRG long format (resembles Julian date)
static long dfsCompileDate                      // RET   REG date, YYYYMMDD
(
   char               *compdate                 // IN    compile date __DATE__
);

/*****************************************************************************/
// Get registration name text in a string, length TXMAXTM
/*****************************************************************************/
char *dfsRegistrationString
(
   char               *string                   // OUT   registration text
)
{
   strcpy( string, regstring);
   return( string);
}                                               // end 'dfsRegistrationString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get registration status text in a string, length TXMAXTM
/*****************************************************************************/
char *dfsRegistrationStatus
(
   char               *string                   // OUT   registration text
)
{
   strcpy( string, regstatus);
   return( string);
}                                               // end 'dfsRegistrationStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Print registration info to the screen and request confirmation
/*****************************************************************************/
void dfsRegistrationInfo
(
   DFSRG               mode,                    // IN    display/confirm mode
   char               *major,                   // IN    major version
   BOOL                confirm                  // IN    confirmation required
)
{
   TXTS                alead;

   ENTER();
   TRACES(("mode: %d  ver:%s  confirm: %s  registred: %s  tmpkeyIndicator: %s\n",
            mode, major, (confirm) ? "yes" : "no", (registred) ? "yes" : "no",
                          (tmpkeyIndicator) ? tmpkeyIndicator : "none"));

   #if defined (USEWINDOWING)
   if (txwIsWindow( TXHWND_DESKTOP))
   {
      strcpy( alead, "     ");
   }
   else
   #endif                                       // USEWINDOWING
   {
      strcpy( alead, " ");
      #if defined (DOS32)
         TxPrintInit( TxaExeSwitch('a'),        // make sure ANSI support
                      TxaExeSwitch('7'));       // is set correctly (probe)
      #endif
   }

   if (((mode != DFSRG_WHEN_NOT_REGISTRED) && (mode != DFSRG_CONFIRM_ONLY)) ||
        (registred == FALSE) || (tmpkeyIndicator != NULL))
   {
      TxPrint(  "\n%sRegistration : %s\n", alead, regstring);
      if (strlen(regstatus))
      {
         TxPrint(  "%sRgKey status : %s%s%s\n", alead, (registred) ? CNN : CBR, regstatus, CNN);
      }
   }

   if (((mode == DFSRG_DISPLAY_CONFIRM) || (mode == DFSRG_CONFIRM_ONLY)) &&
       ((registred == FALSE) || (tmpkeyIndicator != NULL))  && (confirm)  )
   {
      DEVICE_STATE     screen = TxScreenState( DEVICE_TEST);
      BOOL             batch  = TxaExeSwitch('b');

      TxScreenState( DEVICE_ON);

      if (batch)                                // batch mode
      {
         TxSetPendingAbort();                   // don't wait for key, abort
      }
      if (tmpkeyIndicator != NULL)              // valid, but expiring, warning
      {
         TxNamedMessage( !batch, DFS_H_REGIST, " INFO: Temporary registration key ",
           "\nThe registration you are using is an %s temporary key\n"
           "Make sure to replace it by a PERMANENT one before it expires!\n\n%s",
            tmpkeyIndicator, regstatus);
      }
      else if (cracked)                         // invalid key!
      {
         TxNamedMessage( !batch, DFS_H_REGIST, " ERROR: Invalid registration key ",
           "\nIt looks like the registration key you are attempting to use "
           "is an illegal one. If so, do you want to risk the contents of "
           "your hard-disks to a possibly cracked utility?\n\n"
           "If the key was supplied to you by a regular DFSee retailer,\n"
           "contact them and also inform: 'info@dfsee.com'\n\n");
      }
      else
      {
         TxPrint(
           "\n\nPlease install the registration 'dfsee.key' in a directory in the PATH.\n"
           "To get a registration, follow the registration links from:\n\n"
           "            %s https://www.dfsee.com/dfsee/buynow.php %s\n\n", CBC, CNN);

         TxNamedMessage( !batch, DFS_H_REGIST, " ERROR: No registration key ",
           "            NO VALID 'dfsee.key' REGISTRATION FOUND!\n\n"
           "Without a valid key, %s should be used for limited time "
           "evaluation of DFSee functionality only!\n\n"
           "Please copy 'dfsee.key' to a directory in the PATH; Purchase through:\n\n"
           "            https://www.dfsee.com/dfsee/buynow.php\n\n%s",
            TxaExeArgv(0), txwIsWindow( TXHWND_DESKTOP) ?
               "\nUse the <Enter> key or the mouse to confirm the message." : "" );
      }
      if ((batch) && (confirm))                 // batch mode, continue slow!
      {
         TxSleep( 5000);                        // allow reading and slowdown
      }                                         // crack analysis loops
      TxScreenState( screen);                   // restore screen state
      TxCancelAbort();                          // Cancel pending abort!
   }
   VRETURN ();
}                                               // end 'dfsRegistrationInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check registration availability and acceptance, and evaluation expiration
/*****************************************************************************/
BOOL dfsRegistrationTest                        // RET   not expired/cracked
(
   char               *product,                 // IN    name product
   char               *regfile,                 // IN    regfile if non-std
   char               *major,                   // IN    major version
   long                introduction,            // IN    introduction date
   char               *ccDATEcc,                // IN    compile date __DATE__
   long                expiredays               // IN    expiration date offset
)                                               //       days from compile date
{
   BOOL                rc = FALSE;              // assume expired
   FILE               *fp = NULL;
   TXLN                buf;
   TXTS                keyname;
   TXLN                text;                    // large text buffer
   TXLN                regTxt;
   int                 keyVer;

   ENTER();
   TRACES(("product: '%s' ver: '%s' intro: %d  expire: %d days - compiled on '%s'\n",
            product, major, introduction, expiredays, ccDATEcc));

   registred = FALSE;
   strcpy(  regstring, "none");
   strcpy(  prodname, product);
   sprintf( keyname, "%3.3sreg=", product);     // keyname

   if (regfile && (strlen(regfile)))            // explicit regfile ?
   {
      strcpy( buf, regfile);
   }
   else
   {
      strcpy( buf, product);                    // use product as reg filename
      TxStrToLower( buf);                       // force lowercase!
   }
   TxFnameExtension( buf, DFSRG_EXT);           // append default extension

   if ((    !TxaExeSwitchUnSet('r')) &&         // Unless -r- (check expiration)
       (fp = TxFindAndOpenFile( buf, "PATH", text)) != NULL)
   {
      cracked = TRUE;                           // assume cracked
      strcpy( regTxt, "invalid, damaged or cracked");
      if (fgets( keystring, TXMAXLN, fp) != NULL)
      {
         TRACES(("prod: '%s'  key: '%s'  reg:'%s'\n", product, keyname, keystring));
         if (strncmp( keystring, keyname, strlen(keyname)) == 0)
         {
            ULONG        dr;
            USHORT       ml;
            DFSUL_HANDLE ulh;                     // compression handle
            ULONG        size = 0;

            ml = TxFormatMixedStr( keystring + strlen(keyname), buf);
            if (ml != 0)
            {
               if (dfsUlRegister(&ulh) == NO_ERROR)
               {
                  if ((dr = dfsUlBufUncompress( ulh,  DFSUL_MLZW_14,
                                                (BYTE *) buf, (ULONG) ml,
                                                (BYTE *) text, TXMAXLN,
                                                &size)) == NO_ERROR)
                  {
                     text[ size] = '\0';
                     {
                        int checkchars = TxRepl( text, '\t', ' ');
                        int majorpos   = strlen( prodname) +1;

                        strcpy( regstring, text); // RAW format of regname
                        if (strncmp( text, prodname, strlen(prodname)) == 0)
                        {
                           keyVer = atoi(text + majorpos);

                           TRACES(("prodname: '%s'  majorpos = %d\n", prodname, majorpos));
                           TRACES(("keyver: %u, prodver: %u\n", keyVer, atoi(major)));
                           if ((keyVer == atoi(major)   ) || // key same version
                               (keyVer == atoi(major) +1)  ) // or 1 release newer
                           {      //- check mandatory regstring fragments
                              if ((strchr( text, '@'          ) != NULL) &&
                                  (strchr( text, '.'          ) != NULL) &&
                                  (strchr( text, '-'          ) != NULL) &&
                                  (strstr( text, "': "        ) != NULL) &&
                                  (strstr( text, ".x"         ) != NULL)  )
                              {
                                 strcpy( regstatus, "");
                                 if (dfsValidateRegdate( text, keyVer, introduction, regstatus))
                                 {
                                    char    *fake  = strstr( text, fake_email);

                                    if (fake != NULL)
                                    {
                                       memcpy( fake, fake + strlen(fake_email),
                                       strlen(       fake + strlen(fake_email)) +1);
                                       TxRepl(       text + majorpos + 4, '.', ' ');
                                    }
                                    strcpy( regstring, text);
                                    if (checkchars == 1) // right number ?
                                    {
                                       cracked   = FALSE;
                                       registred = TRUE; // no nagging
                                       rc        = TRUE; // and continue ...
                                    }
                                 }
                                 else
                                 {
                                    if (strlen(buf))
                                    {
                                       strcpy( regTxt, regstatus);
                                       cracked = FALSE;
                                       rc      = TRUE; // continue, but nag :-)
                                    }
                                 }
                              }
                           }
                           else if (keyVer < atoi(major))
                           {
                              rc = TRUE;        // allow continuation
                              cracked = FALSE;
                              sprintf( regTxt, "Key for old DFSee version (%u%s). "
                                               "Renew or upgrade",
                                                keyVer, (keyVer > 9) ? ".x" : ".xx");
                           }
                        }
                     }
                  }
                  dfsUlTerminate( ulh);
               }
            }
         }
         else                                   // test for DFSPUPPY environment
         {
            #if defined (LINUX)
               if (TxFileExists( "/root/dfsee/linux/dfsee")) // PUPPY environment
            #else
               if (TxFileExists( "\\c\\dfs\\makefile.mif" )) // development, test
            #endif
            {

               rc = TRUE;
               cracked = FALSE;
               sprintf( regTxt, "Initial EVALUATION-ONLY key for DFSPUPPY USB stick");
            }
         }
      }
      fclose( fp);
   }
   else                                         // check no-key EXE expiration
   {
      TXTS             now;                     // current date YYYYMMDD
      time_t           tt = time( &tt);         // current date/time
      int              nowdate;
      int              expiration;

      expiration = dfsRegDate( dfsRegDays( dfsCompileDate(ccDATEcc)) + expiredays);
      strcpy( regTxt, "'dfsee.key' not present!  ");

      strftime( now, TXMAXTS, "%Y%m%d", localtime( &tt));
      nowdate = (int) atol( now);

      TRACES(( "current time: '%s' = %u, expiration: %u\n", now, nowdate, expiration));

      if (nowdate > expiration)
      {
         strcat( regTxt, "Evaluation period expired");
         rc = FALSE;
      }
      else
      {
         sprintf( text, "%d days left for evaluation",
                  dfsRegDays(expiration) - dfsRegDays(nowdate));
         strcat( regTxt, text);
         rc = TRUE;                             // allow continuation
      }
   }
   if (registred == FALSE)                      // update regstatus
   {
      sprintf( regstatus, "%s!", regTxt);
   }
   BRETURN (rc);
}                                               // end 'dfsRegistrationTest'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get registration status and description
/*****************************************************************************/
BOOL dfsRegistrationValid                       // RET   registration valid
(
   char               *regstr,                  // OUT   registration id string
   char               *status                   // OUT   registration status
)
{
   ENTER();

   if (regstr != NULL)
   {
      strcpy( regstr, regstring);
   }
   if (status != NULL)
   {
      strcpy( status, regstatus);
   }
   BRETURN ( registred);
}                                               // end 'dfsRegistrationValid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Patch DFSee RAW registration key into current sector, replace existing key
/*****************************************************************************/
ULONG dfsRegistrationPatch                      // RET   keyvalue replaced OK
(
   void
)
{
   ULONG               rc = DFS_NOT_FOUND;      // no registration available
   ENTER();

   if (registred)
   {
      if (dfsRead( nav.this, 1, rbuf) == NO_ERROR)
      {
         if (memcmp( rbuf, keystring, 7) == 0)  // current sector looks like a key
         {
            strcpy((char *) rbuf, keystring);

            rc = dfsWrite( nav.this, 1, rbuf);  // write back patched sector
         }
      }
   }
   RETURN ( rc);
}                                               // end 'dfsRegistrationValid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Validate encrypted registration-date against limits, build registration msg
/*****************************************************************************/
static BOOL dfsValidateRegdate                  // RET   registration is OK
(
   char               *registration,            // IN    registration text
   int                 major,                   // IN    regfile major ver
   long                introduction,            // IN    introduction date
   char               *failuretext              // OUT   failure reason when
)                                               //       continue allowed
{
   BOOL                rc = FALSE;              // function return
   char                buf[ SECTORSIZE];
   ULONG               crc1 = 0;
   ULONG               crc2 = 0;
   ULONG               crct = 0;
   ULONG               regc;                    // key date
   char               *s;
   char               *n;
   char                text[ SECTORSIZE];
   time_t              tt = time( &tt);         // current date/time
   int                 tmpdate = DFSR_LASTDATE; // expiry date temporary key
   int                 regdate;
   int                 nowdate;
   int                 days = 100;              // default 100 days temporary

   ENTER();
   TRACES(("reg:'%s' intro:%d\n", registration, introduction));

   strcpy( text, " reg-nr for '");
   if ((s = strstr( registration, text)) != NULL)
   {
      if ((n = strstr( s, "': ")) != NULL)
      {
         s += strlen( text);
         memset( buf, 0, SECTORSIZE);
         memcpy( buf, s, (size_t) (n - s));
         crct = (ULONG) atoll( n + 3);

         TRACES(("addr:'%s'  crct: %u\n", buf, crct));

         crc1 = TxHpfsCheckSum( buf);

         crc2 = TxCalculateLvmCrc((BYTE *) buf, SECTORSIZE);

         regc = crc1 ^ crc2 ^ crct;

         TRACES(("regc = %8.8x = %u\n", regc, regc));

         if ((major % 2) == 1)                  // uneven, swap DMY to YMD
         {
            sprintf( text, "%4.4x%2.2x%2.2x",
                     (ULONG)  (regc        & 0xffff),
                     (ULONG) ((regc >> 16) & 0xff),
                     (ULONG) ((regc >> 24) & 0xff) );
         }
         else
         {
            sprintf( text, "%8.8x", regc);
         }
         TRACES(("date string: '%s'\n", text));
         regdate = atol( text);

         dfsa->regTypeCh = 'S';                 // standard registration type
         if (strstr( registration, "EVALUAT") != NULL)
         {
            tmpkeyIndicator = "EVALUATION";     // inform on startup/exit
            days = 999;                         // extreme evaluation period
         }
         if ((s = strchr( registration, '#')) != NULL)
         {
            dfsa->regTypeCh = toupper( *( s + 1)); // save the registration type, if any

            if ((dfsa->regTypeCh == 'U') || (dfsa->regTypeCh == 'T'))
            {
               if (dfsa->regTypeCh == 'U')
               {
                  tmpkeyIndicator = "UPGRADE";  // inform on startup/exit
               }
               else
               {
                  tmpkeyIndicator = "EVALUATION"; // inform on startup/exit
               }
               if (isdigit(*( s + 2)))          // number following it ?
               {
                  days = atol( s + 2);
                  if (days <= 999)              // max 9 months, 99 days, 1-year temp key ...
                  {
                     days = dfsRegDays( days);
                  }
                  else
                  {
                     days = 0;
                  }
               }
               TRACES(("Nr of days tmp is valid: %d\n", days));
               tmpdate = dfsRegDate( dfsRegDays( regdate) + days);
            }
         }

         strftime( text, SECTORSIZE, "%Y%m%d", localtime( &tt));
         nowdate = atol( text);

         TRACES(("reg: %d  intro: %d  final: %d  now: %d  tmp: %d\n",
                  regdate, introduction, DFSR_LASTDATE, nowdate, tmpdate));

         if ((  regdate >= introduction)  &&    // made after introduction
             (  regdate <= DFSR_LASTDATE) &&    // before extinction
             (( regdate <= (nowdate +1))  ||    // in the past (1 day slack)
              ((nowdate %  10000) == 101) ))    // or any 1st of januari
         {
            if (tmpdate != DFSR_LASTDATE)       // temporary key ?
            {
               if (nowdate > tmpdate)           // already expired ?
               {
                  sprintf( failuretext,
                     "TEMPORARY, expired %d days ago! Contact 'regs@dfsee.com'",
                           dfsRegDays(nowdate) - dfsRegDays(tmpdate));
               }
               else
               {
                  sprintf( failuretext,
                     "TEMPORARY key, expiring on: %d, %d days left.",
                           tmpdate, dfsRegDays(tmpdate) - dfsRegDays(nowdate));
                  rc = TRUE;                    // no nagging, just warn!
               }
            }
            else                                // PERMANENT key
            {
               sprintf( failuretext, "This is a permanent key for %u%s, it will not expire.",
                                      major, (major > 9) ? ".x" : ".xx");

               rc = TRUE;                       // all is OK
            }
         }
      }
   }
   BRETURN (rc);
}                                               // end 'dfsValidateRegdate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// calculate number of days represented by (DFSRG) date (resembles Julian date)
/*****************************************************************************/
static int dfsRegDays                           // RET   linear number of days
(
   int                 rgdate                   // IN    REG date, YYYYMMDD
)
{
   int                 rc = 0;                  // function return
   int                 md = 0;                  // days in year upto this month
   int                 y,m,d;                   // year, day and month numbers
   int                 i;
   BOOL                leap;

   ENTER();

   d  = (rgdate        ) % 100;
   m  = (rgdate / 100  ) % 100;
   y  = (rgdate / 10000);

   leap = (((y % 4) == 0) && ((y % 100) != 0)); // leap year

   for (i = 1; i <= m; i++)
   {
      switch (i)
      {
         case 2:                           md += (leap) ? 29 : 28; break;
         case 4: case 6: case 9: case 11:  md +=               30; break;
         default:                          md +=               31; break;
      }
   }
   TRACES(( "y:%d m:%d d:%d md:%d for rgdate:%d\n", y, m, d, md, rgdate));
   rc = y * 365 + md + d -1;
   RETURN (rc);
}                                               // end 'dfsRegDays'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// calculate DFSRG date from DFSRG number of days (resembles Julian date)
/*****************************************************************************/
static int dfsRegDate                           // RET   REG date, YYYYMMDD
(
   int                 rgdays                   // IN    nr of days
)
{
   int                 rc = 0;                  // function return
   int                 md = 0;                  // days difference
   int                 y,m,d;                   // year, day and month numbers
   BOOL                leap;
   int                 month = 0;

   ENTER();

   y = (rgdays / 365);
   d = (rgdays % 365);

   leap = (((y % 4) == 0) && ((y % 100) != 0)); // leap year

   for (m = 1; m <= 12; m++)
   {
      switch (m)
      {
         case 2:                           md = (leap) ? 29 : 28; break;
         case 4: case 6: case 9: case 11:  md =               30; break;
         default:                          md =               31; break;
      }
      if (d >= md)
      {
         d  -= md;
      }
      else
      {
         month = m -1;                          // m == months
         break;                                 // d == day in month
      }
   }
   if (month == 0)                              // map 0..11 to 1..12 months
   {
      month = 12;
      y--;
   }
   TRACES(( "y:%d m:%d d:%d for rgdays:%d\n", y, month, d, rgdays));
   rc = y * 10000 + month * 100 + d +1;
   RETURN (rc);
}                                               // end 'dfsRegDate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get date of compilation in DFSRG long format (resembles Julian date)
/*****************************************************************************/
static long dfsCompileDate                      // RET   REG date, YYYYMMDD
(
   char               *compdate                 // IN    compile date __DATE__
)
{
   int                 rc;                      // function return
   int                 day;
   int                 year;
   TXTS                month;

   ENTER();
   TRACES(("cc date: '%s'\n", compdate));

   sscanf( compdate, "%s %d %d", month, &day, &year);
   if      (strncasecmp( month, "Jan", 3) == 0)  rc =  100;
   else if (strncasecmp( month, "Feb", 3) == 0)  rc =  200;
   else if (strncasecmp( month, "Mar", 3) == 0)  rc =  300;
   else if (strncasecmp( month, "Apr", 3) == 0)  rc =  400;
   else if (strncasecmp( month, "May", 3) == 0)  rc =  500;
   else if (strncasecmp( month, "Jun", 3) == 0)  rc =  600;
   else if (strncasecmp( month, "Jul", 3) == 0)  rc =  700;
   else if (strncasecmp( month, "Aug", 3) == 0)  rc =  800;
   else if (strncasecmp( month, "Sep", 3) == 0)  rc =  900;
   else if (strncasecmp( month, "Oct", 3) == 0)  rc = 1000;
   else if (strncasecmp( month, "Nov", 3) == 0)  rc = 1100;
   else                                          rc = 1200;

   rc +=  (year * 10000L) + day;
   TRACES(( "Compiledate: %d\n", rc))

   RETURN (rc);
}                                               // end 'dfsCompileDate'
/*---------------------------------------------------------------------------*/

#endif                                          // REGISTRATION
