//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS  utility services, Progress reporting
//
// Author: J. van Wijk
//
// JvW  16-03-2006 Initial version, split off from DFSUTIL.C
//
// Note: Interface in dfsutil.h

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // Partition definitions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsutil.h>                            // DFS utility & display functions

#define DFSP_ABORT           0x80000            // progress aborted, no BAR

#define DFSP_BAR_CH1         ''
#define DFSP_BAR_CH2         ''


static ULN64   dfspBaseNum  = 0;                // default Progress base number
static ULN64   dfspSizeNum  = 0;                // size to process, number range
static ULN64   dfspDoneNum  = 0;                // number 'done' sofar
static TXTT    dfspOperText;                    // operation, like 'wiped'
static TXTT    dfspItemText;                    // item, like 'pass' or 'partition'
static TXTT    dfspLeadText;                    // Number descr. like "Sector:"
static TXTT    dfspLeadPbar;                    // Lead for BAR, like "Sectors"
static ULONG   dfspItemTodo = 0;                // number of separate items todo
static ULONG   dfspItemDone = 0;                // number of separate items done
static ULONG   dfspFlags    = DFSP_STAT;
static ULONG   dfspBarwidth = 75;
static ULONG   dfspBardone  = 0;
static TXLN    dfspBar;
static TXTIMER dfspStart;                       // High-res start timestamp
static TXTT    dfspStrEta;
static double  dfspSecEta1;                     // last reported ETA seconds
static ULONG   dfspFastFact;                    // Fast area speed factor
static ULN64   dfspFastSize;                    // Size of the 'fast' area
static ULN64   dfspUnifSize;                    // Unified size, ETA calculation

#if defined (USEWINDOWING)
 static TXTIMER  dfspLastEta;
#endif


/*****************************************************************************/
// Init progress indicator for multiple items/passes
/*****************************************************************************/
void dfsProgressItemsTodo
(
   ULONG               todo,                    // IN    total items to
   char               *itemname                 // IN    item-kind indicator
)
{
   ENTER();
   TRACES(( "Todo: %u of things with itemname: '%s'\n", todo, itemname));

   dfspItemTodo = todo;
   dfspItemDone = 0;
   TxCopy( dfspItemText, itemname, TXMAXTT);
   VRETURN();
}                                               // end 'dfsProgressItemsTodo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Increment items done for progress indicator (when NOT using ProgressInit!)
/*****************************************************************************/
void dfsProgressItemsNext
(
   void
)
{
   ENTER();

   dfspItemDone++;
   TRACES(( "Starting %s %u of %u\n", dfspItemText, dfspItemDone, dfspItemTodo));
   VRETURN();
}                                               // end 'dfsProgressItemsNext'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Init progress indicator for BAR and percentage/ETA display in status-line
// 20170526: Added 'fast' area-progress, for more accurate ETA calculations
/*****************************************************************************/
void dfsProgressInit
(
   ULN64               base,                    // IN    base number (like LSN)
   ULN64               size,                    // IN    size todo   (like sectors)
   ULN64               fastSize,                // IN    fast 'smart' size or 0
   char               *lead,                    // IN    lead text like Sector:
   char               *operation,               // IN    operation indicator
   ULONG               flags,                   // IN    progress style/options
   ULONG               fastFactor               // IN    Speed factor, fast area
)                                               //       being faster than other
{
   ENTER();

   TRACES(( "base:0x%llx size:0x%llx lead:'%s' operation:'%s' flags:%4.4x\n",
             base, size, lead, (operation) ? operation : "Done", flags));

   dfspFlags = flags;                           // remember init flags
   dfspStart = TxTmrGetNanoSecFromStart();      // start-time throughput/ETA
   strcpy( dfspLeadText, lead);
   strcpy( dfspLeadPbar, lead);
   if (dfspLeadPbar[strlen(dfspLeadPbar)-1] == ':') // singular on statusline
   {
       dfspLeadPbar[strlen(dfspLeadPbar) -1] = 's'; // plural on progress-bar
   }
   if ((dfspItemTodo != 0) && (dfspItemDone >= dfspItemTodo))
   {
      dfspItemDone = 0;                         // set of items done
      dfspItemTodo = 0;
   }
   else                                         // indicate next in set
   {
      dfspItemDone++;
      TRACES(( "Starting %s %u of %u\n", dfspItemText, dfspItemDone, dfspItemTodo));
   }
   TxCopy( dfspOperText, (operation) ? operation : "Done", TXMAXTT);

   if (dfspFlags & DFSP_BAR)                    // display as progress bar
   {
      dfspBarwidth = dfsGetDisplayMargin() -2;
      dfspBardone  = 0;

      memset(  dfspBar, DFSP_BAR_CH1, TXMAXLN); // full width empty BAR
      dfspBar[ dfspBarwidth] = 0;

      if (dfsa->stdOutProgBars)                 // force (only) BARS to STDOUT
      {
         printf("\n %s %s\n %s\n %s", dfspLeadPbar, dfspOperText, dfspBar, CU1);
         fflush( stdout);
      }
      else if (!dfsGuiStdOut())                 // standard scroll-buffer too
      {
         TxPrint("\n %s %s\n %s\n %s", dfspLeadPbar, dfspOperText, dfspBar, CU1);
      }
   }

   if (dfsGuiStdOut())                          // single-line to STDOUT only
   {
      printf("\n");
   }

   dfspBaseNum  = base;
   dfspSizeNum  = size;
   dfspFastSize = (fastSize < size) ? fastSize   : 0; // sane defaults
   dfspFastFact = (fastFactor != 0) ? fastFactor : 1;

   if (dfspSizeNum == 0)                               // avoid later divide by ZERO!
   {
      dfspSizeNum = 1;
      TRACES(("dfspSizeNum corrected from 0 to 1, to avoid divide by ZERO!\n"));
   }

   //- Total 'unified' size todo, fast+slow combined, used for better ETA calculation
   dfspUnifSize = dfspFastSize + ((ULN64) dfspFastFact * (size - dfspFastSize));
   TRACES(("fastSize:0x%llx  Unified size: 0x%llx\n", dfspFastSize, dfspUnifSize));

   dfspSecEta1       = 0;
   dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
   if (flags & DFSP_NO_ETA)
   {
      strcpy( dfspStrEta, "");
   }
   else
   {
      strcpy( dfspStrEta, "  Estimating ETA");
   }
   txwAllowUserStatusMessages( TRUE);
   VRETURN();
}                                               // end 'dfsProgressInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Resume progress session for BAR
/*****************************************************************************/
void dfsProgressResume
(
   char               *text                     // IN    resume text or ""
)
{
   int                 i;
   ENTER();

   if ((dfspFlags & DFSP_BAR) && (dfspFlags & DFSP_ABORT)) // BAR, aborted
   {
      dfspFlags &= ~DFSP_ABORT;                 // clear abort flag

      if (dfsa->stdOutProgBars)                 // force (only) BARS to STDOUT
      {
         printf(" %s %s  %s\n %s\n %s", dfspLeadPbar, dfspOperText,
                (text && *text)        ?   text          :
                (dfspFlags & DFSP_RES) ?  "resuming ..." : "", dfspBar, CU1);

         for (i = 0; i < dfspBardone; i++)      // repaint BAR to current pos
         {
            printf( "%c", DFSP_BAR_CH2);
         }
         fflush( stdout);
      }
      else if (!dfsGuiStdOut())                 // standard scroll-buffer too
      {
         TxPrint(" %s %s  %s\n %s\n %s", dfspLeadPbar, dfspOperText,
                (text && *text)        ?   text          :
                (dfspFlags & DFSP_RES) ?  "resuming ..." : "", dfspBar, CU1);

         for (i = 0; i < dfspBardone; i++)      // repaint BAR to current pos
         {
            TxPrint( "%c", DFSP_BAR_CH2);
         }
      }
   }
   VRETURN();
}                                               // end 'dfsProgressResume'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Suspend progress session (end BAR painting) but keep status for resume
/*****************************************************************************/
void dfsProgressSuspend
(
   void
)
{
   ENTER();

   if ((dfspFlags & DFSP_BAR) && !(dfspFlags & DFSP_ABORT)) // BAR, not aborted
   {
      dfspFlags |= DFSP_ABORT;
      if (dfsa->stdOutProgBars)                 // force (only) BARS to STDOUT
      {
         printf( "\n\n");                       // out of bar, one empty line
         fflush( stdout);
      }
      else if (!dfsGuiStdOut())                 // standard scroll-buffer too
      {
         TxPrint( "\n\n");                      // out of bar, one empty line
      }
   }
   VRETURN();
}                                               // end 'dfsProgressSuspend'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Terminate progress indicator, may be called multiple times (later ignored)
/*****************************************************************************/
void dfsProgressTerm
(
   void
)
{
   TXLN                remaining = {0};         // remaining bar
   int                 i;

   ENTER();

   if ((dfspFlags & DFSP_BAR) && !(dfspFlags & DFSP_ABORT)) // BAR, not aborted
   {
      for (i = 0; i < (dfspBarwidth - dfspBardone); i++)
      {
         remaining[ i] = DFSP_BAR_CH2;
      }
      remaining[ i] = 0;
      if (dfsa->stdOutProgBars)                 // force (only) BARS to STDOUT
      {
         printf( "%s\n\n", remaining);          // out of bar, plus empty line
         fflush( stdout);
      }
      else if (!dfsGuiStdOut())                 // standard scroll-buffer too
      {
         TxPrint( "%s\n\n", remaining);         // out of bar, plus empty line
      }
   }
   dfspFlags = DFSP_NONE;
   txwAllowUserStatusMessages( FALSE);          // resume default status text
   VRETURN();
}                                               // end 'dfsProgressTerm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS show progress in SB-status line (percentage) and/or progress BAR in SB
// 20170526: Added 'smart' area-progress, for more accurate ETA calculations
/*****************************************************************************/
void dfsProgressShow
(
   ULN64               progress,                // IN    actual progress, rel to base
   ULN64               fastProgress,            // IN    progress fast area, rel to 0
   ULN64               addValue,                // IN    additional displ value
   char               *addText                  // IN    additional displ text
)                                               //       addText hex/dec value
{
   TXTM                text;
   BOOL                newstatus = FALSE;

   if ((progress == 0) && (addValue == 0))
   {
      dfspDoneNum = 0;
   }
   else
   {
      dfspDoneNum = progress - dfspBaseNum;     // progress sofar ...
   }

   if (dfsa->stdOutProgBars)                    // force (only) BARS to STDOUT
   {
      if ((dfspFlags & DFSP_BAR) && !(dfspFlags & DFSP_ABORT)) // BAR, not aborted
      {                                         // round down avoids overshoot
         ULONG curBar = (ULONG) ((((double) (dfspDoneNum) / (double) dfspSizeNum)
                                                       * dfspBarwidth) - 0.5);
         while ((dfspBardone < curBar) && (dfspBardone < dfspBarwidth))
         {
            dfspBardone++;                      // extend to current pos
            printf( "%c", DFSP_BAR_CH2);
            fflush( stdout);
         }
      }
   }
   else if (!dfsGuiStdOut())                    // standard scroll-buffer too
   {
      if ((dfspFlags & DFSP_BAR) && !(dfspFlags & DFSP_ABORT)) // BAR, not aborted
      {                                         // round down avoids overshoot
         ULONG curBar = (ULONG) ((((double) (dfspDoneNum) / (double) dfspSizeNum)
                                                       * dfspBarwidth) - 0.5);
         while ((dfspBardone < curBar) && (dfspBardone < dfspBarwidth))
         {
            dfspBardone++;                      // extend to current pos
            TxPrint( "%c", DFSP_BAR_CH2);
         }
      }
   }
   else
   {
      if (dfspDoneNum >= dfspSizeNum)         // at 100% ?
      {
         newstatus = TRUE;
      }
   }
   if ((dfspFlags & DFSP_PERC) && (newstatus || TxTmrTimerExpired( dfsa->statusTimer)))
   {
      TXTM    extras;                           // extra string to display
      double  perc;                             // TIME based percentage
      ULN64   unifiedDone;                      // Unified 'done' number

      //- 'unified' size DONE, fast+slow combined, used for better ETA and time-based percentage
      unifiedDone     = fastProgress;
      if (dfspDoneNum > fastProgress)
      {
         unifiedDone += ((ULN64) dfspFastFact * (dfspDoneNum - fastProgress));
      }
      if (dfspUnifSize > unifiedDone)           // normal case
      {
         perc = ((double)( 100.0 * ((double) (unifiedDone) / (double) dfspUnifSize)));
         if (perc      > 70.0)                  // round DOWN in final part
         {
            perc -= 0.05;                       // so even 99.99 will be rounded down ...
         }
         else if (perc < 30.0)                  // round UP on low percentage values
         {
            perc += 0.04;                       // so even 0.01 will be shown as 0.1
         }
      }
      else                                      // may happen with extreme fragmented
      {                                         // 'fast' areas, too small to work
         perc = 99.9;                           // well with smart skipping, limit ...
      }
      if (dfspItemTodo != 0)                    // multiple items todo
      {
         sprintf( extras, " %s %u of %u", dfspItemText, dfspItemDone, dfspItemTodo);
      }
      else if (addText && addValue)
      {
         if (addValue == DFSP_STRONLY)
         {
            sprintf( extras, ", %s", addText);
         }
         else
         {
            if (dfspFlags & DFSP_ADEC)
            {
               sprintf( extras, ", %s %3llu", addText, addValue);
            }
            else                                // default, additional HEX
            {
               sprintf( extras, ", %s 0x%10.10llx", addText, addValue);
            }
         }
      }
      else
      {
         strcpy( extras, "");
      }

      if (dfsGuiStdOut())                       // single-line to STDOUT only
      {
         //- to be refined, can be made TX generic by making stdOutInterval
         //- a paramater for the ProgressShow function and test above as
         //- (stdOutInterval != 0) && (txIsWindow( TXHWND_DESKTOP))
         //- could also put the stdOutInterval in txwa, with set and get
         //- to avoid passing it on all Show functions ...

         sprintf( text, "%.1lf%%", perc);       // real stdout, no logfile
         printf( "%s%-6.6s%s\n", (dfsa->stdOutInterval > 10) ? CU1 : "", text, extras);
         fflush( stdout);

         dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL  * (dfsa->stdOutInterval % 10));
      }
      #if defined (USEWINDOWING)
      else if (txwIsWindow( TXHWND_DESKTOP))    // to TXWIN status-bar
      {
         TXLN    status;

         if (dfspDoneNum && (dfspDoneNum < dfspSizeNum))
         {
            TXTIMER  now         = TxTmrGetNanoSecFromStart();
            ULONG    secondsDone = (ULONG) TxTmrDifferenceSeconds( now, dfspStart);
            ULONG    secTodo;                   // ETA calculated for now

            if (!(dfspFlags & DFSP_NO_ETA)       && // unless not wanted
                 (secondsDone > DFSP_ETA_1STGUESS)) // after a little while
            {
               if (now >= (dfspLastEta + DFSP_ETA_INTERVAL)) // update periodically
               {
                  dfspLastEta = now;

                  if (dfspUnifSize > unifiedDone)
                  {
                     secTodo = (ULONG) (((double) (dfspUnifSize  - unifiedDone) /
                                         (double)  unifiedDone)  * secondsDone);
                  }
                  else                          // can happen with much unallocated areas smaller
                  {                             // than the used buffersize (not counted here!)
                     secTodo = 1;               // not done, unknown time left ...
                     TRACES(("secTodo unknown, unallocated space too fragmented!\n"));
                  }
                  TRACES(("Pshow P:%12llx fastP:%12llx  uDone:%12llx  secDone:%7u secTodo:%7u Total:%7u\n",
                           dfspDoneNum, fastProgress, unifiedDone, secondsDone, secTodo, secondsDone + secTodo));

                  if ((secTodo > DFSP_ETA_REMAINING) || //- enough left todo or
                      (dfspSecEta1 != 0))               //- already showing ETA
                  {
                     if (dfspSecEta1 == 0)
                     {
                        dfspSecEta1 = (double) secTodo; // first time, no history
                     }
                     else                       // average with last reported
                     {
                        dfspSecEta1 = ((double) (secTodo + dfspSecEta1) / 2.0) +1.0;
                     }
                     strcpy(       dfspStrEta, "");
                     txStrSec2hms( dfspStrEta, "  ETA ", (ULONG) dfspSecEta1, " ");
                     if (dfsa->sbWidth > 82) // maybe enough room for time done ...
                     {
                        txStrSec2hms( dfspStrEta, "done:", secondsDone, " ");
                     }
                  }
               }
            }                                   // but display ETA each time!
         }
         sprintf( status, (dfspFlags & DFSP_VDEC) ?
                          "%s%5llu of %llu %s |%5.1lf%%%s%s" :
                          "%s%4llx of %llx %s |%5.1lf%%%s%s",
                           dfspLeadText, dfspDoneNum, dfspSizeNum,
                           dfspOperText, perc, extras, dfspStrEta);

         txwSetSbviewStatus( status, cSchemeColor);
         dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
      }
      #endif
   }
}                                               // end 'dfsProgressShow'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display (ALWAYS) elapsed time and leading/trailing text for progress-session
/*****************************************************************************/
double dfsProgressElapsed                       // RET   elapsed time, seconds
(
   char               *lead,                    // IN    lead text
   char               *trail                    // IN    trailing text
)
{
   BOOL                rc  = FALSE;
   double              el;
   TXTT                elapsed = {0};
   TXTIMER             now = TxTmrGetNanoSecFromStart();

   if ((el = TxTmrDifferenceSeconds( now, dfspStart)) >= 60.0)
   {
      txStrSec2hms( elapsed, lead, (ULONG) el, " (h:m:sec)");
      rc = TRUE;
   }
   else
   {
      sprintf( elapsed, "%s%8.3lf (seconds)", lead, el);
   }
   TxPrint( "%s%s", elapsed, trail);
   return (rc);
}                                               // end 'dfsProgressElapsed'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display elapsed time and optional throughput for progress-session
/*****************************************************************************/
void dfsProgressThroughput
(
   USHORT              bps                      // IN    bytes per sector
)
{
   dfsDisplayThroughput( dfspStart, dfspDoneNum, bps);
}                                               // end 'dfsProgressThroughput'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display elapsed time and sector throughput based on starttime and #sectors
/*****************************************************************************/
double dfsDisplayThroughput                     // RET   elapsed seconds
(
   TXTIMER             stime,                   // IN    starttime
   ULN64               sectors,                 // IN    sectors handled
   USHORT              bps                      // IN    bytes per sector
)
{
   double              el;
   double              tp;                      // throughput value
   TXTIMER             etime = TxTmrGetNanoSecFromStart();
   TXTS                unit;
   TXTT                elapsed = {0};

   ENTER();

   if ((el = TxTmrDifferenceSeconds( etime, stime)) > 0.001)
   {
      strcpy( unit, "MiB/sec");
      tp = TXSMIB( sectors, bps) / el;
      if (tp < 1)
      {
         tp *= 1024.0;                          // convert to KiB/sec
         unit[0] = 'K';
      }
      if (el >= 60.0)                           // longer durations in H:M:S
      {
         txStrSec2hms( elapsed, "Duration  (H:M:S) : ", (ULONG) el, "");
      }
      else
      {
         sprintf( elapsed, "%7.3lf (seconds) : ", el);
      }
      TxPrint("%s  Disk-sector throughput = %8.2lf %s\n", elapsed, tp, unit);
   }
   RETURN((ULONG) el);
}                                               // end 'dfsDisplayThroughput'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display elapsed seconds and byte throughput based on #seconds and #bytes
/*****************************************************************************/
void dfsDisByteThroughput
(
   double              seconds,                 // IN    elapsed seconds
   ULN64               bytes,                   // IN    bytes handled (filesize)
   char               *spec                     // IN    Throughput spec, 22 ch
)
{
   double              tp;                      // throughput value
   TXTS                unit;

   ENTER();

   if (seconds > 0.1)                           // suppress when quite small
   {
      tp = ((double) bytes / seconds) / 1024.0; // Start with KiB/sec
      strcpy( unit, "KiB/sec");
      if (tp > 1024.0)
      {
         tp /= 1024.0;                          // convert to MiB/sec
         unit[0] = 'M';
      }
      TxPrint( "Elapsed # seconds :%10.1lf %23.23s = %8.2lf %s\n", seconds, spec, tp, unit);
   }
}                                               // end 'dfsDisByteThroughput'
/*---------------------------------------------------------------------------*/

