//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// POSIX related utility functions (mainly for Linux related FS)
//
// Author: J. van Wijk
//
// JvW  16-07-2017 Initial version, derived from DFSUJFS
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsposix.h>                           // POSIX utility functions





/*****************************************************************************/
// Convert Posix MODE to string, decoded value and optional HEX and posix-bits
// String up to 24: decoded only, 24..57 include HEX, over 57: incl posix-bits
/*****************************************************************************/
char *dfsPosixMode2String                       // RET   posix mode string
(
   ULONG               mode,                    // IN    filemode
   int                 length,                  // IN    max string length
   char               *str                      // OUT   mode string
)
{
   char               *s;

   if (length > 23)
   {
      sprintf( str, "0x%8.8x = ", mode);
   }
   else
   {
      strcpy(  str, "");
   }
   s = str + strlen( str);

   *s++ = (S_ISDIR( mode))   ? 'd' : '-';

   *s++ = (mode &   S_IRUSR) ? 'r' : '-';
   *s++ = (mode &   S_IWUSR) ? 'w' : '-';
   *s++ = (mode &   S_IXUSR) ? 'x' : '-';

   *s++ = (mode &   S_IRGRP) ? 'r' : '-';
   *s++ = (mode &   S_IWGRP) ? 'w' : '-';
   *s++ = (mode &   S_IXGRP) ? 'x' : '-';

   *s++ = (mode &   S_IROTH) ? 'r' : '-';
   *s++ = (mode &   S_IWOTH) ? 'w' : '-';
   *s++ = (mode &   S_IXOTH) ? 'x' : '-';

   *s++ = 0;

   if (length > 57)                             // add aditional Possix bits
   {
      if ((mode &  S_ISUID) ==  S_ISUID) strcat( str, " setUid");
      if ((mode &  S_ISGID) ==  S_ISGID) strcat( str, " setGid");
      if ((mode &  S_ISVTX) ==  S_ISVTX) strcat( str, " Sticky");
      if ((mode & _S_IFLNK) == _S_IFLNK) strcat( str, " symLink");
   }
   return (str);
}                                               // end 'dfsPosixMode2String'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display Posix UID+GID to 1-line txprint output, HEX + symbolic or decimal
/*****************************************************************************/
void dfsPosixUidGidPrint
(
   ULONG               uid,                     // IN    uid value
   ULONG               gid                      // IN    gid value
)
{
   TXTM                text;                    // ASCII buffer

   sprintf( text, "Owning User (Uid) : 0x%8.8x = ",    uid);
   switch (uid)
   {
      case  0:  TxPrint( "%s%-3u (root)",        text, uid); break;
      default:  TxPrint( "%s%-10u",              text, uid); break;
   }
   sprintf( text,    "    Group : 0x%8.8x = ",         gid);
   switch (gid)
   {
      case  0:  TxPrint( "%s%-3u (root)\n",      text, gid); break;
      case  1:  TxPrint( "%s%-3u (daemon)\n",    text, gid); break;
      case  2:  TxPrint( "%s%-3u (bin)\n",       text, gid); break;
      case  3:  TxPrint( "%s%-3u (sys)\n",       text, gid); break;
      case  4:  TxPrint( "%s%-3u (adm)\n",       text, gid); break;
      case  6:  TxPrint( "%s%-3u (disk)\n",      text, gid); break;
      case  7:  TxPrint( "%s%-3u (lp)\n",        text, gid); break;
      case  8:  TxPrint( "%s%-3u (mail)\n",      text, gid); break;
      case  9:  TxPrint( "%s%-3u (news)\n",      text, gid); break;
      case 10:  TxPrint( "%s%-3u (uucp)\n",      text, gid); break;
      case 27:  TxPrint( "%s%-3u (sudo)\n",      text, gid); break;
      case 34:  TxPrint( "%s%-3u (backup)\n",    text, gid); break;
      case 40:  TxPrint( "%s%-3u (src)\n",       text, gid); break;
      case 100: TxPrint( "%s%-3u (users)\n",     text, gid); break;
      default:  TxPrint( "%s%-10u\n",            text, gid); break;
   }
}                                               // end 'dfsPosixUidGidPrint'
/*---------------------------------------------------------------------------*/


