//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS partition manager
//
// Author: J. van Wijk
//
// JvW  04-01-1997 Initial version
// JvW  05-04-2001 Added LVM definitions based on Linux open-source for LVM
//
#ifndef    DFSPART_H
   #define DFSPART_H

#define DFS_INV_MBR       ((ULONG) 96)          // invalid master boot record
#define DFS_INV_BOOT      ((ULONG) 97)          // invalid boot record
#define DFS_OVERFLOW      ((ULONG) 98)          // part-list overflow

#define DFS_P_NULL        ((USHORT) -1)         // invalid partition index

#define DFS_P_ACTIVE        0x80                // standard ACTIVE flag

#define DFS_P_PHIDDEN       0x10                // hidden primary indicator
#define DFS_P_MAXHIDE       0x1f                // maximum hidden primary

#define DFS_P_EMPTY         ((BYTE) 0x00)       // unused table entry
#define DFS_P_FAT12         ((BYTE) 0x01)       // FAT12
#define DFS_P_XENIX_S       ((BYTE) 0x02)       // Xenix root partition
#define DFS_P_XENIX_U       ((BYTE) 0x03)       // Xenix user partition
#define DFS_P_FAT16         ((BYTE) 0x04)       // FAT16 < 32Mb
#define DFS_P_EXTENDED      ((BYTE) 0x05)       // Extended partion (container)
#define DFS_P_BIGDOS        ((BYTE) 0x06)       // FAT16 > 32Mb (standard I13)
#define DFS_P_INST_FS       ((BYTE) 0x07)       // HPFS or standard NTFS or ...
#define DFS_P_AIX_BOOT      ((BYTE) 0x08)       // AIX boot partition
#define DFS_P_AIX_DATA      ((BYTE) 0x09)       // AIX data partition
#define DFS_P_BOOTMGR       ((BYTE) 0x0a)       // OS/2 Bootmanager
#define DFS_P_FAT32         ((BYTE) 0x0b)       // FAT32         (standard I13)
#define DFS_P_FAT32X        ((BYTE) 0x0c)       // FAT32X        (extended I13)
#define DFS_P_FAT16X        ((BYTE) 0x0e)       // FAT16X        (extended I13)
#define DFS_P_BIGEXTEND     ((BYTE) 0x0f)       // Ext-container (extended I13)
#define DFS_P_RECOVERY      ((BYTE) 0x12)       // IBM/Compaq    (FAT16/32/NTFS)
#define DFS_P_HIDEXTEND     ((BYTE) 0x15)       // Ext-container, hidden
#define DFS_P_ASTWINSWP     ((BYTE) 0x18)       // AST Windows swapfile
#define DFS_P_OXYGENEXT     ((BYTE) 0x22)       // Oxygen extended P-table
#define DFS_P_NECDOS_3X     ((BYTE) 0x24)       // NEC DOS 3.x
#define DFS_P_IFSHIDDEN     ((BYTE) 0x27)       // Hidden IFS (Vista pre-install)
#define DFS_P_WARP_LVM      ((BYTE) 0x35)       // LVM aware container like JFS
#define DFS_P_THEOS_V32     ((BYTE) 0x38)       // Theos ver 3.2 2Gb partition
#define DFS_P_THEOS_V4S     ((BYTE) 0x39)       // Theos ver 4 spanned part
#define DFS_P_THEOS_V4P     ((BYTE) 0x3a)       // Theos ver 4   4Gb partition
#define DFS_P_THEOS_V4X     ((BYTE) 0x3b)       // Theos ver 4 extended part
#define DFS_P_PQM_RECOV     ((BYTE) 0x3c)       // PartitionMagic recovery
#define DFS_P_VENIX_286     ((BYTE) 0x40)       // Venix 80286
#define DFS_P_WIND_PREP     ((BYTE) 0x41)       // Windows PREParation type
#define DFS_P_WIN2XPLDM     ((BYTE) 0x42)       // Windows 2000/XP LDM
#define DFS_P_LINUXDRSW     ((BYTE) 0x43)       // Linux/DrDos swap
#define DFS_P_GOBACK_DR     ((BYTE) 0x44)       // GoBack monitored (Win-ME)
#define DFS_P_QNXVER4P1     ((BYTE) 0x4d)       // QNX 4.x 1st partition
#define DFS_P_QNXVER4P2     ((BYTE) 0x4e)       // QNX 4.x 2nd partition
#define DFS_P_QNXVER4P3     ((BYTE) 0x4f)       // QNX 4.x 3rd partition
#define DFS_P_ONTRACK_R     ((BYTE) 0x50)       // OnTrack read-only
#define DFS_P_ONTRACKRW     ((BYTE) 0x51)       // OnTrack read-write
#define DFS_P_CPM_RES_1     ((BYTE) 0x52)       // CP/M reserved ID 1
#define DFS_P_ONTRACK_6     ((BYTE) 0x53)       // OnTrack DM 6.0
#define DFS_P_ONTRACKDO     ((BYTE) 0x54)       // DM 6.0 Dynamic Drive Overlay
#define DFS_P_EZ_DRIVE      ((BYTE) 0x55)       // MicroHouse EZ-drive
#define DFS_P_VFEATURE      ((BYTE) 0x56)       // GB Vfeature
#define DFS_P_PRI_EDISK     ((BYTE) 0x5c)       // Priam Edisk
#define DFS_P_SPEEDSTR      ((BYTE) 0x61)       // Speedstore
#define DFS_P_SCO_SYSV      ((BYTE) 0x63)       // SCO / System V Unix
#define DFS_P_NOVELL_R2     ((BYTE) 0x64)       // Novell 2.xx
#define DFS_P_NOVELL_34     ((BYTE) 0x65)       // Novell 3.xx or 4.xx
#define DFS_P_NOVELL386     ((BYTE) 0x66)       // Novell 386
#define DFS_P_NOVELL_X1     ((BYTE) 0x67)       // Novell reserved (4 ?)
#define DFS_P_NOVELL_X2     ((BYTE) 0x68)       // Novell reserved 2
#define DFS_P_NOVELL_X3     ((BYTE) 0x69)       // Novell Storage Services 5+
#define DFS_P_APFS_CONT     ((BYTE) 0x73)       // APFS Container MBR-alias
#define DFS_P_PCIX_UNIX     ((BYTE) 0x75)       // PC/IX
#define DFS_P_XOSL_BMGR     ((BYTE) 0x78)       // XOSL OS loader (FAT FS)
#define DFS_P_VRAIDPART     ((BYTE) 0x7C)       // VRAID partition
#define DFS_P_MINIX_13      ((BYTE) 0x80)       // Minux upto 1.4a
#define DFS_P_MINIX_15      ((BYTE) 0x81)       // Minux from 1.4b / old Linux
#define DFS_P_SWAPSOLAR     ((BYTE) 0x82)       // Linux swap or Solaris
#define DFS_P_LINUXNATV     ((BYTE) 0x83)       // Linux EXT2 data partition
#define DFS_P_HIBERNATE     ((BYTE) 0x84)       // Hibernate partition APM 1.1
#define DFS_P_LINUXEXTX     ((BYTE) 0x85)       // Linux extended partition
#define DFS_P_BIGDOS_VS     ((BYTE) 0x86)       // FAT16 > 32 Mb, NT volumeset
#define DFS_P_IFS_MIRR      ((BYTE) 0x87)       // NTFS volset or HPFS mirror
#define DFS_P_AB_LINKRN     ((BYTE) 0x8a)       // Air-Boot Linux kernel part
#define DFS_P_LINUX_LVM     ((BYTE) 0x8e)       // Linux Logical Volume Manager
#define DFS_P_BIGLINEXT     ((BYTE) 0x8f)       // Ext-container (Linux big)
#define DFS_P_AMOEBASYS     ((BYTE) 0x93)       // Amoeba OS (Tannenbaum)
#define DFS_P_AMOEBA_BB     ((BYTE) 0x94)       // Amoeba OS, bad-block table
#define DFS_P_LOGIC1024     ((BYTE) 0x99)       // Mylex-DCE376 large extended
#define DFS_P_SAVE2DISK     ((BYTE) 0xa0)       // Save-2-Disk (Phoenix/Tpad)
#define DFS_P_SAVEDISK2     ((BYTE) 0xa1)       // Save-2-Disk (NEC 6000H)
#define DFS_P_FREE_BSD      ((BYTE) 0xa5)       // FreeBSD/BSD386/NetBSD disklabel
#define DFS_P_OPEN_BSD      ((BYTE) 0xa6)       // OpenBSD
#define DFS_P_NEXTSTEP      ((BYTE) 0xa7)       // NextStep
#define DFS_P_MACXDATA      ((BYTE) 0xa8)       // macOS data partition
#define DFS_P_NETBSD98      ((BYTE) 0xa9)       // NetBSD after 1998
#define DFS_P_OLISERVE      ((BYTE) 0xaa)       // Olivetti FAT service
#define DFS_P_MACXBOOT      ((BYTE) 0xab)       // macOS boot partition
#define DFS_P_AOS2GPVD      ((BYTE) 0xad)       // ArcaOS OS/2 Guid Partition Virtual Disk
#define DFS_P_MACXHFSP      ((BYTE) 0xaf)       // macOS HFS(+) partition
#define DFS_P_BSDI_FSYS     ((BYTE) 0xb7)       // BSDI filesystem
#define DFS_P_BSDI_SWAP     ((BYTE) 0xb8)       // BSDI swap partition
#define DFS_P_ACRONIS_H     ((BYTE) 0xbb)       // BootWizard/Acronis hidden
#define DFS_P_ACRONISSZ     ((BYTE) 0xbc)       // Acronis Safe Zone (backup)
#define DFS_P_SOLARBOOT     ((BYTE) 0xbe)       // Solaris boot partition
#define DFS_P_SOLAR_ZFS     ((BYTE) 0xbf)       // Solaris ZFS  partition
#define DFS_P_CTOS_FSYS     ((BYTE) 0xc0)       // CTOS filesystem/Novell NTFT
#define DFS_P_DRDOSSF12     ((BYTE) 0xc1)       // DRDOS secure FAT12 (6.0)
#define DFS_P_DRDOSSF16     ((BYTE) 0xc4)       // DRDOS secure FAT16 (6.0)
#define DFS_P_BIGDOSCVS     ((BYTE) 0xc6)       // FAT16, corrupted volumeset
#define DFS_P_IFS_DISAB     ((BYTE) 0xc7)       // Disabled HPFS mirror
#define DFS_P_CTOSMDUMP     ((BYTE) 0xcd)       // Disabled HPFS mirror
#define DFS_P_DFSNOWARN     ((BYTE) 0xd7)       // DFSee temporary, no warnings given
#define DFS_P_CPM_86        ((BYTE) 0xd8)       // CPM/86
#define DFS_P_LIN_NONFS     ((BYTE) 0xda)       // Linux non-FS data (RAID)
#define DFS_P_CPM_RES_2     ((BYTE) 0xdb)       // CPM/86 reserved 2
#define DFS_P_DELL_UTIL     ((BYTE) 0xde)       // DELL utility/maintenance FAT
#define DFS_P_DFSEETYPE     ((BYTE) 0xdf)       // DFSee temporary/BootIt
#define DFS_P_SPEEDS_12     ((BYTE) 0xe1)       // Speedstore-FAT12
#define DFS_P_DOSACCESS     ((BYTE) 0xe3)       // DOS RO / speedstore
#define DFS_P_SPEEDS_16     ((BYTE) 0xe4)       // Speedstore-FAT16
#define DFS_P_LINUXLUKS     ((BYTE) 0xe8)       // Linux LUKS encrypted
#define DFS_P_EFI_ANY       ((BYTE) 0xea)       // EFI any partition type
#define DFS_P_BEOS_FS       ((BYTE) 0xeb)       // BeOS filesystem
#define DFS_P_EFI_GPT       ((BYTE) 0xee)       // EFI GUID disk (XP LDM)
#define DFS_P_EFI_ESP       ((BYTE) 0xef)       // EFI System Partition on MBR disk
#define DFS_P_SDMAXTOR1     ((BYTE) 0xf1)       // Maxtor-1 storage dimensions
#define DFS_P_DOSSECOND     ((BYTE) 0xf2)       // DOS 3.3+ secondary part
#define DFS_P_SDMAXTOR2     ((BYTE) 0xf4)       // Maxtor-2 storage dimensions
#define DFS_P_BOCHS_VM      ((BYTE) 0xfa)       // Boch VirtualMachine
#define DFS_P_VMWARSYST     ((BYTE) 0xfb)       // VMWare system partition
#define DFS_P_VMWARSWAP     ((BYTE) 0xfc)       // VMWare swap partition
#define DFS_P_LINUXRAID     ((BYTE) 0xfd)       // Linux raid partition
#define DFS_P_PS2SYST       ((BYTE) 0xfe)       // PS/2 system partition
#define DFS_P_BADBLOCKS     ((BYTE) 0xff)       // Bad Block Tables (cyl 0)

#define DFS_M_OPCO_1        ((BYTE) 0xfa)       // MBR first instruction std
#define DFS_M_OPCO_2        ((BYTE) 0x33)       // MBR first instruction alt

#define DFS_B_OPCO_1        ((BYTE) 0xe9)       // Jump instruction
#define DFS_B_OPCO_2        ((BYTE) 0xeb)       // Jump instruction

#define DFS_B_FATOS2        ((BYTE) 0x44)       // Offset-byte OS/2  bootcode
#define DFS_B_FATVKO        ((BYTE) 0x47)       // Offset-byte Veit  bootcode
#define DFS_B_FATDOS        ((BYTE) 0x3c)       // Offset-byte DOS   bootcode
#define DFS_B_FATWNT        ((BYTE) 0x3c)       // Offset-byte wNT   bootcode

#define DFS_B_JMPW98        DFS_B_OPCO_2        // Jump instruction W98 FAT32
#define DFS_B_F32W98        ((BYTE) 0x58)       // Offset-byte w98   bootcode
#define DFS_B_OPCNOP        ((BYTE) 0x90)       // NOP  instruction  bootcode
#define DFS_B_OPCJE         ((BYTE) 0x74)       // JE   instruction  bootcode
#define DFS_B_OPCJNE        ((BYTE) 0x75)       // JNE  instruction  bootcode
#define DFS_B_OPCJMPS       ((BYTE) 0xEB)       // JMPS instruction  bootcode

// offset I13X reference in BMGR and HPFS bootsectors,  from os.RestCode
#define BMGR_I13X_INBOOT    0x8b                // os.RestCode + 0x008b = 0x00e0
#define HPFS_I13X_CHECK     0x2c                // os.RestCode + 0x2c   = 0x81

// Unpartionable space (not really FREE) used for non MBR/GPT style disks
#define DFS_FSP_UNPS        ((BYTE) 0xd0)       // unusable freespace

// GPT style disk freespaces
#define DFS_FSP_GPTM        ((BYTE) 0xd4)       // GPT PTA at start + MBR
#define DFS_FSP_GPTS        ((BYTE) 0xdc)       // GPT PTA +  MBR + freespace
#define DFS_FSP_GPTE        ((BYTE) 0xe4)       // GPT PTA at end
#define DFS_FSP_GPTC        ((BYTE) 0xec)       // GPT PTA at end + freespace
#define DFS_FSP_GPTF        ((BYTE) 0xe8)       // GPT freespace area
#define DFS_FSP_GPT_STYLE   ((BYTE) 0xef)       // GPT freespace area, max ID

// MBR style disk freespaces
#define DFS_FSP_NONE        ((BYTE) 0xf0)       // unusable freespace
#define DFS_FSP_PRIM        ((BYTE) 0xf1)       // primary-only freespace
#define DFS_FSP_LOG         ((BYTE) 0xf2)       // Log freespace, in EXT part
#define DFS_FSP_HLOG        ((BYTE) 0xf6)       // Log freespace, before EXT
#define DFS_FSP_TLOG        ((BYTE) 0xfa)       // Log freespace, after  EXT
#define DFS_FSP_ELOG        ((BYTE) 0xfc)       // Log freespace, empty  EBR
#define DFS_FSP_NEWX        ((BYTE) 0xf3)       // P/L freespace, no EXT yet
#define DFS_FSP_HEAD        ((BYTE) 0xf7)       // P/L freespace, before EXT
#define DFS_FSP_TAIL        ((BYTE) 0xfb)       // P/L freespace, after  EXT
#define DFS_FSP_ENDC        ((BYTE) 0xfe)       // End of disk partial cylinder
#define DFS_FSP_ZERO        ((BYTE) 0xff)       // Track-Zero freespace

#define DFS_FSP_LOGMASK     ((BYTE) 0x02)       // Logical
#define DFS_FSP_PRIMASK     ((BYTE) 0x01)       // Primary


//- Note: major/minor may be updated based on perceived severity :-)

#define DFS_F_M_MAJOR       0xfff81500          // show major warnings only
#define DFS_F_M_MINOR       0x0007eaff          // show minor warnings only
#define DFS_F_M_TRIVIAL     0x00008a3e          // NOT counted in warnLevel 0
#define DFS_F_M_ALIGN       0x0000ec00          // cylinder alignment
#define DFS_F_M_IBMGR       0x00070000          // IBM BootManager related
#define DFS_F_M_CHSER       0x000000f0          // CHS related
#define DFS_F_M_FORMT       0x00000008          // not-formatted
#define DFS_F_M_OVERL       0x03f00000          // size overlap related
#define DFS_F_M_EXT0F       0x04000000          // ext-container type 0x0f
#define DFS_F_M_SHOWALL     0xffffffff          // show all warnings mask

//- error definitions used in combination with the 1st set of 'flags'
#define DFS_F_LOGICALACT    0x00000001          // Logical part marked active
#define DFS_F_MINORWARN1    0x00000002          // Reserved, free for new warning
#define DFS_F_MINORWARN2    0x00000004          // used to be INT13 limit warning
#define DFS_F_BIGEXTENDF    0x00000008          // non-std ext type 0x0f used
#define DFS_F_CHSDUM_S13    0x00000010          // start CHS nonstd above int13
#define DFS_F_CHSDUM_E13    0x00000020          // end   CHS nonstd above int13
#define DFS_F_CHSLBA_DUM    0x00000030          // any   CHS-LBA er above int13
#define DFS_F_CHSLBA_S13    0x00000040          // start CHS != LBA below int13
#define DFS_F_CHSLBA_E13    0x00000080          // end   CHS != LBA below int13
#define DFS_F_CHSLBA_I13    0x000000c0          // any   CHS-LBA er below int13
#define DFS_F_CHSLBA_ERR    0x000000f0          // any   CHS-LBA mismatch
#define DFS_F_ACT_HIDDEN    0x00000100          // active partition is hidden
#define DFS_F_ACT_NOT1ST    0x00000200          // active partition on disk > 1
#define DFS_F_EXTC_NOT_0    0x00000400          // extended start not on track0
#define DFS_F_EXTC_NOT_C    0x00000800          // extended end not on last
#define DFS_F_BOOTRECBAD    0x00001000          // Bad boot record, unformatted
#define DFS_F_PRIM_NOT_0    0x00002000          // Primary-start not on track0
#define DFS_F_PLOG_NOT_1    0x00004000          // Pri/Log-start not on track1
#define DFS_F_PART_NOT_C    0x00008000          // Part-end not on last track
#define DFS_F_BMGR_OTHER    0x00010000          // a BMGR on other than 1 or 2
#define DFS_F_BMGR_NOT12    0x00020000          // only BMGR not on disk 1 or 2
#define DFS_F_BMGR_INACT    0x00040000          // BMGR present but not active
#define DFS_F_EXTEND_ACT    0x00080000          // Extended part marked active
#define DFS_F_LOG_BEFORE    0x00100000          // log starts before start cont
#define DFS_F_LOG_BEYOND    0x00200000          // log ends beyond end cont
#define DFS_F_EXT_BEFORE    0x00400000          // ext starts before start cont
#define DFS_F_EXT_BEYOND    0x00800000          // ext ends beyond end cont
#define DFS_F_PART_EDISK    0x01000000          // partition beyond end of disk
#define DFS_F_EXTC_EDISK    0x02000000          // extended  beyond end of disk
#define DFS_F_WINLDMDISK    0x04000000          // LDM style, Windows Dynamic Disk
#define DFS_F_EFIGPTDISK    0x08000000          // GPT style partitioning (XP)
#define DFS_F_MULTIP_ACT    0x10000000          // Multiple active partitions
#define DFS_F_NO_ACTIVE1    0x20000000          // No active partitions disk 1
#define DFS_F_NO_ACTIVES    0x40000000          // No active partitions at all
#define DFS_F_OVERLAPREV    0x80000000          // Overlaps previous partition


//- Note: major/minor may be updated based on perceived severity :-)
#define DFS_F2M_MAJOR       0xffffef4c          // show major warnings only
#define DFS_F2M_MINOR       0x000010b3          // show minor warnings only
#define DFS_F2M_TRIVIAL     0x00001053          // NOT counted in warnLevel 0

//- error definitions used in combination with the 2nd set of 'flags'
#define DFS_F_BMNOTW2KBM    0x00000001          // IBM BMGR, not protected
#define DFS_F_BMBADCHS      0x00000002          // IBM BMGR, bad boot/data CHS
#define DFS_F_OVERLAPMBR    0x00000004          // MBR overlapped, ISOHybrid
#define DFS_F_LVMBADSIZE    0x00000008          // LVM size or start mismatch
#define DFS_F_LVMBOOTBAD    0x00000010          // LVM boot-id   inconsistent
#define DFS_F_I13XNEEDED    0x00000020          // Bootsector requires 'I13X'
#define DFS_F_HIDDENSBAD    0x00000040          // BR HiddenSectors != ptable
#define DFS_F_BOOTGEOBAD    0x00000080          // BR GEO Head/Sect != part-geo
#define DFS_F_BRDUPLICAT    0x00000100          // Bootrecord exact duplicate
#define DFS_F_LVMBADSEQN    0x00000200          // LVM Drivelink zero SEQ-nr
#define DFS_F_F32INF16TP    0x00000400          // FAT32 FS in FAT16 parttype
#define DFS_F_F16INF32TP    0x00000800          // FAT16 FS in FAT32 parttype
#define DFS_F_LINUXHDA63    0x00001000          // Linux PATA dev number > 63
#define DFS_F_LVMBADCRCI    0x00002000          // LVM CRC bad in info sector
#define DFS_F_LVMBADCRCS    0x00004000          // LVM CRC bad in sign sector
#define DFS_F_LVMNAME_20    0x00008000          // LVM partname 20 characters
#define DFS_F_LVMNAMEBAD    0x00010000          // LVM disk-name inconsistent
#define DFS_F_LVMTHISBAD    0x00020000          // LVM this-id   inconsistent
#define DFS_F_LVM_NO_SIG    0x00040000          // LVM Type 35, Info but no Sig
#define DFS_F_SIG_NO_LVM    0x00080000          // LVM Sig found on non 35 type
#define DFS_F_EMPTYCONTN    0x00100000          // Empty extended container
#define DFS_F_NONSTDFLAG    0x00200000          // Non-std ACTIVE flag byte
#define DFS_F_GUIDDUPLIC    0x00400000          // GPT part GUID not unique
#define DFS_F_WASTECONTN    0x00800000          // Wasting extended container
#define DFS_F_EXTC_BLINK    0x01000000          // link to next is backwards
#define DFS_F_EXTC_NLINK    0x02000000          // link to this is backwards
#define DFS_F_EXTC_LLOOP    0x04000000          // extended links in a loop
#define DFS_F_EXTC_NLOOP    0x08000000          // ext is linked to in a loop
#define DFS_F_PRIM_INEXT    0x10000000          // Primary inside ext container
#define DFS_F_DLATINPART    0x20000000          // Part start overlaps LVM DLAT
#define DFS_F_NOBOOTLETT    0x40000000          // Bootable JFS/HPFS, no LVM letter
#define DFS_F_BADBPBLETT    0x80000000          // Bootable JFS/HPFS, BPB letter mismatch


//- to be refined, move some of the PART warnings to the DISK flags
//- might be attached to first partition on the disk right now

//- Note: major/minor may be updated based on perceived severity :-)
#define DFS_FDM_MAJOR       0xffeffcff          // show major warnings only
#define DFS_FDM_MINOR       0x00100300          // show minor warnings only
#define DFS_FDM_TRIVIAL     0x00000300          // NOT counted in warnLevel 0

//- error definitions used in combination with the 1st set of 'flags' for DISK
#define DFS_F_LVMDISK_20    0x00000100          // LVM diskname 20 characters
#define DFS_F_LVMCYL_64K    0x00000200          // LVM geometry Cyls exceeds 64K
#define DFS_F_PTYPE_0x00    0x00000400          // Deleted partition type 0x00
#define DFS_F_INVALIDPTE    0x00000800          // Invalid partition table entry
#define DFS_F_APMMBRPART    0x00002000          // Apple Partition Map + MBR partitions
#define DFS_F_FSYSTEMONLY   0x00004000          // FileSystem Only (large floppy)
#define DFS_F_CRYPTO_DISK   0x00008000          // Encrypted disk  (LUKS header)
#define DFS_F_MULTI_EXT     0x00010000          // Multiple extended  containers
#define DFS_F_MULTI_GPT     0x00020000          // Multiple GPT guard containers
#define DFS_F_GPT_NOGUARD   0x00040000          // GPT hdr/tables but no guard
#define DFS_F_OS2_64K_CYL   0x00100000          // More than 64K cylinders on OS/2
#define DFS_F_MBR_NO_SIG    0x00200000          // MBR has table entries, no 0x55AA
#define DFS_F_EBR_NO_SIG    0x00400000          // EBR has table entries, no 0x55AA
#define DFS_F_GPTMBRPART    0x00800000          // GPT tables and MBR partitions present
#define DFS_F_GPT_U_512     0x01000000          // Unexpected GPT at 512 for 4K sect
#define DFS_F_GPT_U_4096    0x02000000          // Unexpected GPT at 4K for 512 sect
#define DFS_F_GPTINVALID    0x04000000          // Guard partition, no valid HDR/PTA
#define DFS_F_GPT_BADCRC    0x08000000          // GPT header has incorrect CRC value
#define DFS_F_GPT_BADPRI    0x10000000          // GPT primary OK, alternate missing
#define DFS_F_GPT_BADALT    0x20000000          // GPT alternate OK, primary missing
#define DFS_F_GPTSMGUARD    0x40000000          // GPT guard partition smaller than disk

#define DFS_MAX_VOLS          30                // max volumes
#define DFS_MAX_PART        1000                // max partitions
#define DFS_MAX_DISK          24                // max physical disks
                                                // conforming to BMGR limits
typedef enum dfs_pi_type
{
   DFS_PI_PARTITION,
   DFS_PI_EBR_CHAIN,
   DFS_PI_FREESPACE
} DFS_PI_TYPE;                                  // end of enum "dfs_pi_type"


//- Definitions for the LVM info sector (last sector of 1st track MBR/EBR)
//- LVM signature/feature sectors (at end of non-compatibility partitions)

#define LVM_NAME_L     20                       // LVM name length
#define LVM_COMM_L     81                       // LVM comment length

typedef struct s_lvpart                         // LVM partition info
{
   ULONG               VolumeId;                // LVM-id, unique for a volume
   ULONG               PartitId;                // LVM-id, unique for partition
   ULONG               sectors;                 // nr of sectors in partition
   ULONG               partPsn;                 // start Psn related partition
   BYTE                OnBmMenu;                // partition is on the BMGR menu
   BYTE                Installable;             // has been set installable
   BYTE                letter;                  // uppercase volume letter
   BYTE                bzero;                   // value '00'
   char                VoluName[LVM_NAME_L];    // Logical volume name
   char                PartName[LVM_NAME_L];    // partition name
} S_LVPART;                                     // end of struct "s_lvpart"


                                                // max nr of slots per LVM info
#define MAXLVINF       4
#define SG_LVINF       8
#define SV_LVINF        (char)0x02,(char)0x52,(char)0x4d,(char)0x42, \
                        (char)0x50,(char)0x4d,(char)0x46,(char)0x44
typedef struct s_lvinf                          // LVM information sector
{
   char                Signature[SG_LVINF];     // Signature RMBPMFD
   ULONG               LvmCRC;                  // LVM info sector checksum
   ULONG               ThisDiskId;              // Current disk id (serial)
   ULONG               BootDiskId;              // Boot    disk id (serial)
   ULONG               InstallFlags;            // Installation flags
   ULONG               geoCyls;                 // Number of cylinders
   ULONG               geoHeads;                // Number of Heads
   ULONG               geoSecs;                 // Number of sectors per track
   char                DiskName[LVM_NAME_L];    // physical disk name
   BYTE                RebootFlag;              // Reboot indicator
   BYTE                Reserved[3];             // unused  '00 00 00'
   S_LVPART            part[MAXLVINF];          // info on max 4 partitions
   char                fill[212];               // upto 512 bytes
} S_LVINF;                                      // end of struct "s_lvinf"


                                                // known feature ID's
#define LVMF_DRIVE_LINK     100
#define LVMF_BAD_BLOCKR     101

typedef struct s_lvfeat                         // LVM feature info
{
   ULONG               FeatureId;               // LVM-id, unique for a feature
   ULONG               priPsn;                  // primary   feature location
   ULONG               secPsn;                  // secondary feature location
   ULONG               sectors;                 // nr of sectors in feature
   USHORT              majVers;                 // Feature major version nr
   USHORT              minVers;                 // Feature minor version nr
   BYTE                active;                  // feature active flag
   BYTE                fill[3];                 // filler to 24 bytes
} S_LVFEAT;                                     // end of struct "s_lvfeat"

#define LVM_FEATURES   10                       // nr of feature slots

#define SG_LVSIG       8
#define SV_LVSIG        (char)0x32,(char)0x53,(char)0x43,(char)0x4a, \
                        (char)0x4b,(char)0x44,(char)0x52,(char)0x42
#define SV_LVREM        (char)0x42,(char)0x42,(char)0x52,(char)0x65, \
                        (char)0x6d,(char)0x6f,(char)0x76,(char)0x65
typedef struct s_lvsig                          // LVM signature sector
{
   char                Signature[SG_LVSIG];     // Signature 2SCJKDRB/BBRemove
   ULONG               LvmCRC;                  // LVM sign sector checksum
   ULONG               PartitId;                // LVM-id, unique for partition
   ULONG               firstPsn;                // first LBA for partition
   ULONG               lastPsn;                 // last  LBA for partition
   ULONG               sectors;                 // total sectors in partition
   ULONG               reserved;                // sectors reserved for LVM use
   ULONG               reported;                // sectors usable by FS
   ULONG               BootDiskId;              // Boot    disk id (serial)
   ULONG               VolumeId;                // LVM-id, unique for a volume
   ULONG               fakeEbrPsn;              // fake EBR LBA (on 1st ?)
   USHORT              majVers;                 // LVM major version nr
   USHORT              minVers;                 // LVM minor version nr
   char                PartName[LVM_NAME_L];    // partition name
   char                VoluName[LVM_NAME_L];    // Logical volume name
   S_LVFEAT            feature[ LVM_FEATURES];  // info on max 10 LVM features
   BYTE                letter;                  // uppercase volume letter
   BYTE                fakeEbrSet;              // fake EBR present
                                                // next not seen on disk yet!
   char                pComment[LVM_COMM_L];    // partition comment string
   char                DiskName[LVM_NAME_L];    // physical disk name
   ULONG               SequenceNr;              // part sequence nr (JFS ?)
   ULONG               NextAggregate;           // next aggregate   (JFS ?)
} S_LVSIG;                                      // end of struct "s_lvsig"



typedef struct s_lvdlink
{
   ULONG               DriveSerialNr;
   ULONG               PartitionSerialNr;
} S_LVDLINK;                                    // end of struct "s_lvdlink"

#define LVDLT_SIZE1    60
#define LVDLT_SIZEN    62

#define LVM_CLASS_PARTITION  0
#define LVM_CLASS_AGGREGATE  1
#define LVM_CLASS_VOLUME     2

#define SG_LVDLF       4
#define SV_LVDLF        (char)0x57,(char)0x41,(char)0x4e,(char)0x43
typedef struct s_lvdlf                          // LVM drive-link feature
{
   char                Signature[SG_LVDLF];     // Signature WANC
   ULONG               LvmCRC;                  // LVM DLT sector checksum
   ULONG               SequenceNr;              //
   ULONG               LinksInUse;              //
   S_LVDLINK           LinkTable[ LVDLT_SIZE1]; // links in first table sector
   ULONG               AggregateSerNr;          // aggregate serial nr
   ULONG               ActualClass;             // feature class (aggregate)
   BOOL                TopOfClass;              //
   ULONG               FeatureSequenceNr;
} S_LVDLF;                                      // end of struct "s_lvdlf"

#define SG_LVDLT       4
#define SV_LVDLT        (char)0x52,(char)0x56,(char)0x4d,(char)0x4d
typedef struct s_lvdlt                          // LVM drive-link table
{
   char                Signature[SG_LVDLT];     // Signature RVMM
   ULONG               LvmCRC;                  // LVM DLT sector checksum
   ULONG               SequenceNr;              //
   S_LVDLINK           LinkTable[ LVDLT_SIZEN]; // links in next table sector
} S_LVDLT;                                      // end of struct "s_lvdlt"



typedef struct s_lvbadbl
{
   ULONG               Badsector;
   ULONG               Replacement;
} S_LVBADBL;                                    // end of struct "s_lvbadbl"

#define LVBBT_SIZE     62

#define SG_LVBBF       4
#define SV_LVBBF        (char)0x62,(char)0x6d,(char)0x72,(char)0x00
typedef struct s_lvbbf                          // LVM bad-block-rel feature
{
   char                Signature[SG_LVBBF];     // Signature bmr.
   ULONG               LvmCRC;                  // LVM BBF sector checksum
   ULONG               SequenceNr;              //
   ULONG               TableSize;               //
   ULONG               TableEntriesInUse;       //
   ULONG               SectorsPerTable;
   ULONG               FirstReplSector;         // location 1st repl sector
   ULONG               LastReplSector;          // location last repl sector
   ULONG               ReplSectorCount;         // nr of repl sectors
   ULONG               Flags;                   // global to BB feature
   ULONG               ActualClass;             // feature class (partition)
   BOOL                TopOfClass;
   ULONG               FeatureSequenceNr;
} S_LVBBF;                                      // end of struct "s_lvbbf"


#define SG_LVBBT       4
#define SV_LVBBT        (char)0x76,(char)0x62,(char)0x72,(char)0x01
typedef struct s_lvbbt                          // LVM bad-block-rel table
{
   char                Signature[SG_LVBBT];     // Signature vbr.
   ULONG               LvmCRC;                  // LVM BBT sector checksum
   ULONG               SequenceNr;              //
   S_LVBADBL           BbrTable[ LVBBT_SIZE];   // links in badbl-rel sector
   ULONG               Reserved;                // Usually zero ...
} S_LVBBT;                                      // end of struct "s_lvbbt"


//- Linux Unified Key Setup structures, header at start partition or disk
//- Note that the header is LARGER than one sector (when using > 6 keys)
#define SG_LUKSH       6
#define SV_LUKSH        (char)0x4C,(char)0x55,(char)0x4B,(char)0x53,(char)0xBA,(char)0xBE
#define LUKS_KEY_ENABLED      0x00AC71F3        // enabled key slot indicator
#define LUKS_KEY_DISABLED     0x0000DEAD        // unused  key slot indicator
#define LUKS_STR_L    32                        // length name strings
#define LUKS_MKC_L    20                        // length master key checksum
#define LUKS_SALTL    32                        // length 'salt' sequence
#define LUKS_UUIDL    40                        // reserved length ascii UUID
#define LUKS_NKEYS     8                        // total number of key structures
#define LUKS_1KEYS     6                        // number of keys in 1st sector (512)

// LUKS key
typedef struct s_luks_key
{
   ULONG               active;                  // activation status for key
   ULONG               pwdIterations;
   char                pwdSalt[ LUKS_SALTL];    // name of cipher algorithm
   ULONG               keyMaterialOffset;       // offset to key contents
   ULONG               antiForensicStripes;     // stripes count for AF
} S_LUKS_KEY;                                   // end of struct "s_luks_key"

// LUKS crypto header
typedef struct s_luksh
{
   char                Signature[SG_LUKSH];     // Signature LUKS = 'LUKS'babe
   USHORT              version;
   char                cipherName[ LUKS_STR_L]; // name of cipher algorithm
   char                cipherMode[ LUKS_STR_L]; // mode of cipher algorithm
   char                hashSpec[   LUKS_STR_L]; // used hash specification
   ULONG               payloadOffset;
   ULONG               keyBytes;
   char                digestMkc[  LUKS_MKC_L]; // name of cipher algorithm
   char                digestSalt[ LUKS_SALTL]; // name of cipher algorithm
   ULONG               digestIterations;
   char                uuidString[ LUKS_UUIDL]; // UUID as ascii (36 + 0x00)
   S_LUKS_KEY          keyblock[   LUKS_NKEYS]; // array of key info blocks
} S_LUKSH;                                      // end of struct "s_luksh"


//- Apple macOS Core Storage (FileVault/HFS) header at start GPT partition
#define CORE_ST_SIG           0x5343            // Signature 'CS' as ushort
#define CORE_ST_PAD           0xffffffff        // CRC32 padding to 64 bits
// VAULT crypto header
typedef struct s_core_st                        // Apple Core Storage header
{
   ULONG               crc32;                   // 32-bit CRC over 8..end
   ULONG               crcPad;                  // padding to 64 bits
   USHORT              version;                 // header version number   (1)
   char                fill1[38];               // Filler up to sectorsize
   ULONG               sectorSize;              // usually 512
   char                fill2[36];               // Filler up to signature
   USHORT              Signature;               // signature 'CS' = 0x5343
   char                fill3[6];                // Filler up to blocksize
   ULONG               blockSize;               // blocksize, usually 4096
   ULONG               metaSize;                // disklabel metablock size
   ULN64               metaOffset;              // disklabel block offset
   char                fill4[56];               // Filler up to key length
   ULONG               keyLength;               // length of key (bytes?)
   ULONG               cryptoVersion;           // version crypto algorithm (2)
   DFS_GUID            aesXtsKey;               // AES-XTS-Key1 (plist/meta)
   char                fill5[112];              // Filler up to Phys UUIDs
   DFS_GUID            physVolumeUuid;          // AES-XTS-Key2 (tweak key)
   DFS_GUID            logVolGroupUuid;         // Logical Volume Group UUID
   char                fill6[176];              // Filler up to end sector (512)
} S_CORE_ST;                                    // end of struct "s_core_st"


// Apple Macintosh driver-description-record (in MBR sector)
typedef struct s_mcdriver
{
   ULONG               start;                   // first PSN for driver
   USHORT              size;                    // driver size in sectors
   USHORT              type;                    // driver type, 0001 = MacOS
} S_MCDRIVER;                                   // end of struct "s_mcdriver"


#define SG_MCDDM       2
#define SV_MCDDM        (char)0x45,(char)0x52
typedef struct s_mcddm                          // MAC driver description
{                                               // Values are BIG-Endian!
   char                Signature[SG_MCDDM];     // 00 Signature DDM = 'ER'
   USHORT              bpsector;                // 02 blocksize, bytes per sector
   ULONG               sectors;                 // 04 nr of sectors (sometimes LITTLE Endian!)
   USHORT              sbDevType;               // 08 reserved (typical 0001)
   USHORT              sbDevId;                 // 0a reserved (typical 0001)
   ULONG               sbData;                  // 0c reserved (typical 00000080)
   USHORT              entries;                 // 10 nr of driver descriptions
   S_MCDRIVER          driver[1];               // 12 driver description array
   USHORT              fill[242];               // 1a filler to 510 bytes
   USHORT              pcSignature;             // fe PC-style signature AA55
} S_MCDDM;                                      // end of struct "s_mcddm"


// Apple Macintosh disk-partition-record (1 per partition, starting at sector 1)
#define MAC_NAME       32
#define SG_MCDPM       2
#define SV_MCDPM        (char)0x50,(char)0x4d
typedef struct s_mcdpm                          // MAC partition map
{                                               // Values are BIG-Endian!
   char                Signature[SG_MCDPM];     // Signature DDM = 'PM'
   USHORT              fill1;                   // reserved
   ULONG               mapSectors;              // #sectors in map = #partitions!
   ULONG               basePsn;                 // partition base PSN
   ULONG               sectors;                 // partition size in sectors
   char                partName[MAC_NAME];      // partition name
   char                partType[MAC_NAME];      // partition type
   ULONG               logDataStart;            // 1st block in data area
   ULONG               logDataSize;             // size in sectors data area
   ULONG               status;                  // partition status
   ULONG               bootStart;               // 1st block boot code
   ULONG               bootSize;                // boot code size in bytes
   ULONG               bootLoadAddres;          // 1st block boot code
   ULONG               breserved1;              // reserved
   ULONG               bootEntryPoint;          // boot code entrypoint
   ULONG               breserved2;              // reserved
   ULONG               bootChecksum;            // boot code checksum
   char                processorType[16];       // processor type
   USHORT              fill[188];               // filler to 512 bytes
} S_MCDPM;                                      // end of struct "s_mcdpm"


#define BMSYS_TIME_NO  0xffff                   // timeout set to none
#define BMSYS_TIMEOUT  18                       // timeout divider (clock-tick)
#define BMSYS_DEFAULT  0                        // default BM system
#define BMSYS_FORCEDB  4                        // forced boot system (/iba:)
#define BMSYS_LASTONE  5                        // last booted system

#define BMSYS_SHORT    8                        // length short BM name
#define BMSYS_LONG     20                       // length long  BM name

typedef struct bmisystem
{
   ULONG               unknown;                 // filler to 12 byte size
   char                BmSysShort[BMSYS_SHORT]; // BootMgr name (short, old)
} BMISYSTEM;                                    // end of struct "bmisystem"

typedef struct bmisyslong
{
   ULONG               size;                    // 0 or sizeof name (20)
   char                BmSysLong[BMSYS_LONG];   // BootMgr name (long, lvm)
} BMISYSLONG;                                   // end of struct "bmisystem"

// BM system 1..4 are 'fallback' systems when used as boot-system with X:i
// BM system 0 is the default boot system (default in BM-menu)
// BM system 4 is the forced-boot system  (set when using /IBA:name)
// BM system 5 is the last-booted system  (default when system-0 not set)
// BM system 0..3 and 5 can be set using std OS/2 command: 'SETBOOT n:name'
//           where the 'n' is 0..4 (the forced-boot index 4 is skipped)

// Bootmgr-setup-info is in 2nd sector of the Bootmgr partition
typedef struct bminfosec
{
   BYTE                diskid;                  // disk id (value 0x80)
   DFSC                FirstSector;             // First sector (of BMGR ?)
   USHORT              unknown1;                // unknown (value 0x0004)
   USHORT              timeout;                 // menu timeout in timer-ticks
   BYTE                defsys;                  // index of default system
   BYTE                advanced;                // advanced flag
   BYTE                forceboot;               // forced boot letter (/ibd:)
   BYTE                unknown2;                // unknown (value 0x0000)
   ULONG               unknown3;                // unknown (value 0x00000000)
   BMISYSTEM           system[6];               // array of classic  BM names
   BMISYSLONG          syslng[6];               // array of LVM type BM names
} BMINFOSEC;                                    // end of struct "bminfosec"

// Array of Bootmgr-info is in 4th sector of the Bootmgr partition
typedef struct bmprimary                        // Bootmgr primary part info
{                                               // old BMGR only (no LVM)
   char                BootDisk;                // Bootable-flag and disknr
   DFSC                FirstSector;             // First sector of partition
   char                BootMgrFlag;             // Part in this in BootMgr menu
   char                BootMgrName[9];          // Null terminated BootMgr name
   char                fill1[2];                // filler upto 16 bytes
} BMPRIMARY;                                    // end of struct "bmprimary"

//- Note: in LVM version still maintained, except BootMgrName which
//-       is replaced with the fixed string: "--> LVM "

typedef BMPRIMARY S_BMPRIM[DFS_MAX_DISK][4];    // room for 24 Physical disks
                                                // in 3 sectors (trial&error)

#define BMINFOSECTN    1                        // sector 1 in BM partition
#define BMINFOSECTS    1                        // number of info sectors
#define BMPRIMSECTN    3                        // sector 3 in BM partition
#define BMPRIMSECTS    3                        // number of prim sectors

#define FS_UUID_LENGTH  DFS_GUID_LENGTH         // length of UUID fields

#define APM_STYLE      ((USHORT) 0xAAAA)        // APM special 'tablenr' indicator
#define GPT_STYLE      ((USHORT) 0xEEEE)        // GPT special 'tablenr' indicator

//- Note: first field 'id' is used by PSAVE/PRESTORE to identify the partition!
typedef struct dfspartinfo
{
   USHORT              id;                      // Id for partinfo (= index +1)
   USHORT              diskPart;                // relative partition on disk
   DFS_PI_TYPE         pitype;                  // partition info type
   DFSPARTENTRY        partent;                 // copy of partion-table entry
   struct dfspartinfo *ebrChain;                // related/next EBR info
   struct dfspartinfo *fspChain;                // freespace BEFORE, or NEXT link
   struct dfspartinfo *fspAfter;                // freespace AFTER this partition
   S_BOOTR            *pbrSect;                 // copy of the bootsector
   ULN64               partPsn;                 // PSN of MBR/EBR with table
   ULN64               basePsn;                 // PSN of boot sector (LSN 0)
   ULN64               lastPsn;                 // PSN of last sector in part
   ULN64               sectors;                 // size in sectors (redundant)
   ULN64               expandSize;              // size incl freespace AFTER
   ULONG               cSC;                     // nr of sectors per cylinder
   USHORT              scluster;                // nr of sectors per cluster
   USHORT              bpsector;                // nr of bytes per sector
   USHORT              disknr;                  // Physical disk number
   USHORT              tablenr;                 // Table nr, 0=MBR..n or GPT_STYLE
   USHORT              partnr;                  // Part nr in part table 0..3
   USHORT              UnixDevNr;               // Unix device nr 1..n 0 is invalid
   TXTT                UnixDevicePart;          // /dev/hdXnn or /dev/rdiskX
   ULONG               geoCyls;                 // Geometry; nr of cylinders
   ULONG               geoHeads;                //                 heads
   ULONG               geoSecs;                 //         sectors/track
   USHORT              relSize;                 // Relative size of partition
   USHORT              relChars;                // #variable chars this part.
   BOOL                primary;                 // primary/logical partition
   ULONG               flags;                   // 1st set warning/error flags
   ULONG               flag2;                   // 2nd set warning/error flags
   ULONG               checksum;                // checksum of boot record
   ULN64               secFree;                 // reported free sectors by OS
   TXTS                drive;                   // drive letter (string!)
   TXTS                plabel;                  // part/bootr/superblock label
   TXTS                creatr;                  // creator string / Related
   TXTS                fsform;                  // FS format info
   char                blabel[LVM_NAME_L +1];   // BMGR label (FDISK or LVM)
   char                bmname[BMSYS_SHORT +1];  // BMGR label (classic FDISK)
   BOOL                lvmPresent;              // LVM partition info present
   BOOL                lvmSigPres;              // LVM signature info present
   S_LVPART            lvm;                     // related LVM partition info
   TXTM                descr;                   // textual description
   ULONG               lvmDriveLinks;           // number of drives in volume
   ULONG               lvmDriveIndex;           // index this drive in multi
   ULONG               lvmReserved;             // size of reserved area
   ULN64               attrFlags;               // Attribute flags for GPT
   BOOL                uuidPresent;             // Valid fs UUID is present
   DFS_GUID            fsUuid;                  // 128-bit uuid (4-2-2-2-6)
   DFS_GUID            typeGuid;                // gpt style type GUID
   DFS_GUID            partGuid;                // gpt style part GUID
} DFSPARTINFO;                                  // end of struct "DFSPARTINFO"


typedef enum dfspstyle                          // DFS partitioning style
{                                               // matches first char of DFSTR_*
   DFSP_MBR     = 'M',                          // regular MBR partitioning style
   DFSP_GPT     = 'G',                          // GPT partitioning style
   DFSP_APM     = 'A',                          // Apple Partition Map (MCDDM)
   DFSP_FSO     = 'F',                          // FileSystem Only disk, Large-Floppy
   DFSP_CRP     = 'C',                          // Encrypted disk, LUKS or ...
   DFSP_UNK     = 'U'                           // Unknown, non-zero first sector
} DFSPSTYLE;                                    // end of enum "dfspstyle"

#define DPSTR_MBR        "MBR"                  // regular MBR partitioning style
#define DPSTR_GPT        "GPT"                  // GPT partitioning style
#define DPSTR_APM        "APM"                  // Apple Partition Map (MCDDM)
#define DPSTR_FSO        "FSO"                  // FileSystem Only disk
#define DPSTR_CRP        "CRP"                  // Encrypted disk, LUKS or ...
#define DPSTR_UNK        "UNK"                  // Unknown, but SOME contents ...
//-
//- Note:     sectors, geoCyls/Heads/Secs, cSC, bpsector are used by PSAVE binary files!
//-           should stay at the same address. Determine if sectors is a 64bit value
//-           by checking a special MAGIC value in the first 8 bytes (write in PSAVE)
//-
//- 20170407: Structure updated for 64bit sn and sizes, stay compatible with PSAVE/PRESTORE
//- 20170812: Structure updated with pStyle 3-character terminated string MBR/GPT/FLP/LKS/UNK
//-
typedef struct dfsdiskinfo
{
   DFSPARTINFO        *ebrHead;                 // 000 primary extended, head
   DFSPARTINFO        *fspHead;                 // 004 freespace area list, head
   USHORT              primaries;               // 008 number of primaries on disk
   USHORT              totalpart;               // 00A number of partitions on disk
   USHORT              firstpart;               // 00C PID of first partition
   USHORT              lastpart;                // 00E PID of last  partition
   ULONG               geoCyls;                 // 010 Geometry; nr of cylinders
   ULONG               geoHeads;                // 014                heads
   ULONG               geoSecs;                 // 018        sectors/track
   USHORT              relTotal;                // 01C Total relative sizes
   USHORT              relChars;                // 01E #variable chars in mapline
   USHORT              relFrees;                // 020 Freespace areas to display
   USHORT              relParts;                // 022 Partition areas to display
   USHORT              freeSpace;               // 024 number of freespace area's
   ULONG               reservedF;               // 026 old freeSects ULONG, unused
   ULONG               cSC;                     // 02A nr of sectors per cylinder
   ULN64               sectors;                 // 02E nr of sectors on disk
   //- ULONG           bmgrPsn;                 // 032 PSN bmgr, pre-14.5 (32bit SN)
   ULONG               flags;                   // 036 1st set warning/error flags
   ULONG               NTsignature;             // 03A Possible NT signature (MBR)
   ULONG               lvmDiskId;               // 03E lvm physical disk id
   ULONG               lvmBootId;               // 042 lvm ph. boot disk id
   char                DiskName[LVM_NAME_L +1]; // 046 disk name, LVM/constructed
   USHORT              bpsector;                // 05B nr of bytes per sector
   BOOL                OpenError;               // 05D disk cannot be opened
   BYTE                align1;                  // 061 alignment for following
   TXTS                UnixDeviceDisk;          // 062 device name for disk (12)
   USHORT              disknr;                  // 082 Physical disk number
   ULONG               lvmPartitions;           // 084 #partitions with LVM info
   BOOL                Removable;               // 088 Disk is a removable
   ULN64               bmgrPsn;                 // 08C PSN of bootmgr partition
   ULN64               lastUsedPsn;             // 094 last sector of last partition
   ULN64               freeSects;               // 09C number of free sectors, 64bit
   TXTS                fsform;                  // 0A4 FS format info (largeFloppy)
   TXTS                dlabel;                  // 0C4 Bootsect label (largeFloppy)
   TXTS                ddrive;                  // 0E4 Driveletter    (largeFloppy)
   TXTS                creatr;                  // 104 FS creator     (largeFloppy)
   char                pStyle[4];               // 124 Partitioning Style, 3 characters!
   TXTT                dxInfo;                  // 128 Extra disk information, max 64
   BOOL                AirBootPresent;          // 168 Air-Boot bootmanager present
   ULONG               gptPartitions;           // 16C #partitions defined in GPT
   void               *gptHeader;               // 170 optional GPT header data
   void               *gptArray;                // 174 optional GPT partition array
} DFSDISKINFO;                                  // 178 end of struct "dfsdiskinfo"

// Return number of partitionable disks administred by the partition-manager
USHORT dfsPartitionableDisks
(
   void
);


// Get reference to diskinfo for specified physical disk
DFSDISKINFO *dfsGetDiskInfo
(
   USHORT              disk                     // IN    disk number 1..n
);

// Test accessability for specified physical disk (from last Open attempt)
BOOL dfsDiskAccessible
(
   USHORT              disk                     // IN    disk number 1..n
);

// Get number of accessible disks (non-dummy)
USHORT  dfsAccessibleDisks                      // RET   nr of non-dummy disks
(
   void
);

// Return number of partitions administred by the partition-manager
USHORT dfsPartitions
(
   void
);


// Get reference to partinfo for specified DFSee partition
DFSPARTINFO *dfsGetPartInfo                     // RET   partition info or NULL
(
   USHORT              part                     // IN    part number 1..n
);

// Get reference to freespace info for specified freespace area
DFSPARTINFO *dfsGetFreeInfo                     // RET   freespace pointer
(
   USHORT              fsid                     // IN    freespace id 1..n
);

// Get reference to freespace AFTER specified partition, if any
DFSPARTINFO *dfsGetFreeAfter                    // RET   freespace pointer
(
   USHORT              pid                      // IN    partition id
);

// Read basic physical disk information and add partition info for each disk
ULONG dfsReadDiskInfo
(
   USHORT              geodisk                  // IN    nr/FDSK_ANY/FDSK_QUIET
);

// Discard disk-info structures and free any associated resources
void dfsFreeDiskInfo
(
   void
);

// Read all partition info for opened disk
ULONG dfsReadPartInfo
(
   USHORT              disknr                   // IN    physical disk nr
);


// Detect presence of a Unix filesystem when no bootsector present
BOOL dfsDetectUnixFileSystem                    // RET   Filesystem detected
(
   ULN64               basePsn,                 // IN    Base PSN possible FS
   BYTE               *fsect,                   // IN    FS sectors (64KiB)
   char               *label,                   // OUT   label string   or NULL
   char               *creatr,                  // OUT   creator string or NULL
   char               *fsform                   // OUT   FileSystem name string
);

// Set FS-format and related creatr + label description of MAC (0xAF) type
BOOL dfsMacOsNativeInfo                         // macOS filesystem found
(
   S_BOOTR            *boot,                    // IN    bootsector (BBUFBYTES)
   DFSPARTINFO        *p                        // INOUT partition info
);

// Set FS-format and related creatr + label description of Linux (0x83) type
BOOL dfsLinuxNativeInfo                         // RET   Linux filesystem found
(
   S_BOOTR            *boot,                    // IN    bootsector
   DFSPARTINFO        *p                        // INOUT partition info
);

// Allocate one new EBR info structure (used from AddEbr and fdskCreate)
DFSPARTINFO *dfsNewEbrInfo
(
   DFSPARTINFO        *template                 // IN    template structure
);


// Select partition number for Logical reading (aka 'open logical')
ULONG dfsSelectPartition
(
   USHORT              index                    // IN    partition id 1..n
);

// Translate drive-letter to partition-index, usable in Select
USHORT dfsDrive2PartId                          // RET   partition-ID or 0
(
   char               *drive                    // IN    drive letter for part
);

// Translate Linux partition number (NN in /dev/xdaNN) to DFSee PID
USHORT dfsDevNr2PartId                          // RET   RET partition-ID or 0
(
   USHORT              disknr,                  // IN    disk number
   USHORT              devNr                    // IN    Linux device number
);

// Get DFSee partition-ID for specified LVM-partition-Serial-number (lvmsnp)
USHORT dfsLvmSnP2PartId                         // RET   partition-ID or 0
(
   USHORT              disknr,                  // IN    disk number
   ULONG               lvmsnp                   // IN    LVM partition-id
);

// Get DFSee partition-ID for specified disk plus PSN value (within partition)
// Note: For logical partitions, the EBR track is considered to be 'inside'
USHORT dfsDiskPsn2PartId                        // RET   partition-ID or 0
(
   USHORT              disknr,                  // IN    disk number
   ULN64               psn                      // IN    PSN value in partition
);

// Show partition-info table for selection format based on TxaOptions
ULONG dfsShowPartitions                         // RET   nr of partitions
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
);

// Show disk and partition warnings using -w option and masking for a disk
void dfsShowWarnings
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
);

// Show very compact (space) separated list with partition-info (DFSee query)
void dfsShowPartQueryList
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
);

#endif
