//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS partition manager
//
// Author: J. van Wijk
//
// JvW  04-01-97   Initial version
//

#define  INCL_DOSDEVIOCTL                       // device definitions
#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT defs and functions

#if !defined (OEMSB)
#include <dfsspace.h>                           // DFS file-space interface
#include <dfscfdsk.h>                           // FDISK command functions
#include <dfsufat.h>                            // DFS FAT  utility functions
#include <dfsuext.h>                            // DFS EXT  utility functions
#include <dfsursr.h>                            // DFS RSR  utility functions
#include <dfsuxfs.h>                            // DFS XFS  utility functions
#include <dfsuhfs.h>                            // DFS HFS  utility functions
#include <dfsapfs.h>                            // DFS APFS structure definitions
#include <dfsuapfs.h>                           // DFS APFS utility functions
#include <dfsajfs.h>                            // DFS JFS  analysis interface
#include <dfsujfs.h>                            // DFS JFS  utility functions
#include <dfsntfs.h>                            // DFS NTFS definitions
#include <dfsuntfs.h>                           // DFS NTFS utility functions
#include <dfsuefat.h>                           // DFS EFAT utility functions
#include <dfsadump.h>                           // DFS DUMPFS functions, OS/2
#include <dfswin.h>                             // DFS menu and windowing defs
#include <dfsntreg.h>                           // DFS NT registry/devices
#endif


// To be refined, ebrHead has a FIRST object that represents the outer container
// This references the MBR partition-table entry and causes a problem with
// delete of a logical, see P#1345 (fixed in another way, problem still there!)
// The IteratePebr function may behave different from expectations ...

//------------------ Diagram of connected disk/part/free information ----------
//
// Ŀ
//                                 DISKINFO                                  
//                                                                           
//               (PARTINFO in one large array, indexed by id)                
//                                                                           
//                    GPT info             lastpart  (id)                    
// ebrHead             Hdr PTA             firstpart (id)             fspHead
// 
//                                               |                     
//        Ŀ        Ŀ   |                     
//        GPT Header<      type:PARTITION<---                     
//                           Ĵ                         
//        #entries           PRIM                                   
//        entry-size                              Ŀ  
//                        type:FREESPACE<
//                                                    Ĵ
//        Ŀ                                ebrChain      
//         GPT PTA  <   Ŀ                     
//                            type:PARTITION<------id            
//        partition           Ĵ                     
//          array             PRIM                               
//                        fspChain>      fspChainĿ
//                                                     
//                              ebrChain                               
//                                                       
//       Ŀ                                                
//     >type:EBR_CHAIN                                                
//        Ĵ       Ŀ                         
//                    id------>type:PARTITION                         
//                             Ĵ                         
//                             LOG   fspChain                         
//                                                                    
//     ĳebrChain      <ĳebrChain                               
//                     Ŀ  
//                                                     type:FREESPACE<
//       Ŀ                              Ĵ
//     >type:EBR_CHAIN<ĳebrChain      
//        Ĵ       Ŀ                     
//                    id------>type:PARTITION<------id            
//                             Ĵ                     
//                             LOG   fspChain>      fspChainĿ
//                                                    
//     ĳebrChain      <ĳebrChain                               
//                     Ŀ  
//                                                     type:FREESPACE<
//       Ŀ                              Ĵ
//     >type:EBR_CHAIN<ĳebrChain      
//        Ĵ       Ŀ                     
//                    id------>type:PARTITION<------id            
//                             Ĵ                     
//                             LOG   fspChain>      fspChainĿ
//                                                    
//        ebrChain      <ĳebrChain                               
//                      Ŀ  
//                                                      type:FREESPACE<
//   The example shows ONE disk, with 2 primaries,      Ĵ
//   three logicals, and three areas of freespace:      ebrChain      
//                                                                    
//         PRI fsp PRI LOG fsp LOG LOG fsp              id            
//                                                            fspChain
//------------------------------------------------      


static DFSDISKINFO *dfsDiskInfo = NULL;         // detected physical disk info
static DFSPARTINFO *dfsPartInfo = NULL;         // detected partitions info
static USHORT       dfsNrPart = 0;              // number of partitions

static USHORT       dfsDisks  = 0;              // number of physical disks


#define             NO_LVM_GPT_LUKS   "-"       // No LVM/GPT/Crypt/additional info

#define             MACOS_CREATOR "macOS"

#ifndef OEMSB

#define TYPE_DESC_LEN  17                       // Note: hardcoded in part sprintf!

static  char        TableFrameTop[] = "";
static  char        TableFrameTxt[] = "ID %sDrType, description";
static  char        TableFrameMid[] = "[%s%s%s %s%s %sdisk %2hu%s]";
static  char        TableFrameBot[] = "";

static  char        TableFrameTol[] = "%-*.*sĿ\n";
static  char        TableFrameTxl[] = "Format  Related VolumeLabel%-*.*s  Size MiB \n";
static  char        TableFrameMil[] = "";
static  char        TableFrameBol[] = "%-*.*s\n";
static  char        TableFrameVtl[] = "OS2-LVM/BM / GPT / Crypt / additional info";

static  char        TableFrameTon[] = "%-*.*sĿ\n";
static  char        TableFrameTxn[] = "Begin SectEnd sector   Cylinder range  %-*.*s  Size MiB \n";
static  char        TableFrameMin[] = "";
static  char        TableFrameBon[] = "%-*.*s\n";
static  char        TableFrameVtn[] = "  Sectors ";

static  char        TableFrameTob[] = "%-*.*sĿ\n";
static  char        TableFrameTxb[] = " Fmt Related   Cylinder range   Begin Block   End Block%-*.*s  Size MiB \n";
static  char        TableFrameMib[] = "";
static  char        TableFrameBob[] = "%-*.*s\n";
static  char        TableFrameVtb[] = "  Blocks  ";

//- end of the FrameMi* lines, dynamically truncated from the end
static  char        TableFrameEnd[] =
   "Ĵ\n";

//- line fragment printed with variable length
static  char        TableFrameLin[] =
   "";

#endif

// Allocate disk-info structures
static ULONG dfsAllocDiskInfo
(
   BOOL               verbose                   // IN    Report to screen too
);

// Display hints about needing admin/root priviliges to see disk info
static void dfsShowAdminHint
(
   char               *leadtext                 // IN    upto two lines of leading text
);


// Read GPT partition info for opened disk, validate and add GPT partitions
static ULONG dfsReadGptPartitionInfo
(
   USHORT              disknr                   // IN    DFSee disk id
);

// Read APM partition info for opened disk, validate and add APM partitions
static ULONG dfsReadApmPartitionInfo            // Apple Partition Map
(
   USHORT              disknr                   // IN    DFSee disk id
);

// Calculate drive-letter based on PartInfo and industry-standard algorithm
static void dfsCalculateDriveLetters
(
   void
);

// Set relative partition-numbers as used by NT \Device\HardDiskX\PartitionY
static void dfsSetRelDiskPart
(
   void
);

// Correlate the current operating-system volume-map with found partitions
static void dfsCorrelateDriveVolumes
(
   void
);

// Correlate the current operating-system volume-map with specific bootsector
static void dfsCorrelateBootsector
(
   S_BOOTR            *bsect,                   // IN    boootsector to match
   char               *bdrive                   // OUT   driveletter, or ""
);

// Determine freespace for each partition as reported by the OS
static void dfsSetOsFreeSpace
(
   void
);

// Set freespace-AFTER references and maximum expandSize values
static void dfsSetExpandSize
(
   void
);

// Gather additional info on partitions and freespace on each physical disk
static void dfsAdditionalDiskInfo
(
   void
);

// Add Freespace info to a partition in table and in freeSpace list
static DFSPARTINFO *dfsAddFreeSpaceInfo
(
   DFSPARTINFO        *p,                       // IN    related partition
   USHORT              disknr,                  // IN    physical disknr
   ULN64               first,                   // IN    first free sector
   ULN64               last                     // IN    last  free sector
);

// Add partition info to table
static ULONG dfsAddPartInfo
(
   USHORT              disknr,                  // IN    physical disk nr
   USHORT              tablenr,                 // IN    part-table nr 0..n
   USHORT              partnr,                  // IN    index in part-table
   USHORT              arcIndex,                // IN    index in ARC style
   ULN64               thisPSN,                 // IN    PSN of this MBR/EBR
   S_BOOTR            *br,                      // IN    rec with part-table
   void               *info                     // IN    additional partition info
);

// Set FS-format and related (creatr) description of Linux SWAP (0x82) type
static void dfsLinuxSwapInfo
(
   S_BOOTR            *boot,                    // IN    bootsector
   DFSPARTINFO        *p                        // INOUT partition info
);

// Add EBR partition info to table
static DFSPARTINFO *dfsAddEbrInfo
(
   USHORT              disknr,                  // IN    physical disk nr
   USHORT              tablenr,                 // IN    part-table nr 0..n
   USHORT              partnr,                  // IN    index in part-table
   ULONG               thisPSN,                 // IN    PSN of this boot-rec
   S_BOOTR            *br                       // IN    rec with part-table
);

// Show (comma) separated list with partition-info
static void dfsShowPartSepList
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
);

// Create an XML style output (to a file) with detailed partition info
static void dfsShowPartXmlList
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
);

// Show table with partition-info depending on detail/type
static void dfsShowPartTable
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
);


// Correlate and copy LVM disk/volume information to owning partition
// Note: this is a callback function to be used with dfsIteratePlvm
//       For LVMSIG sectors, the cbp.more will have the partition info
static ULONG dfsPartLvmMatch                    // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM info
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
);


// Merge collected LVM info into regular part-info (drive-letter / volume)
static void dfsMergeLvmDriveInfo
(
   BOOL                letters                  // IN    assign letters too
);


/*****************************************************************************/
// Return number of partitionable disks administred by the partition-manager
/*****************************************************************************/
USHORT dfsPartitionableDisks
(
   void
)
{
   ENTER();
   if (dfsDiskInfo == NULL)
   {
      dfsReadDiskInfo( FDSK_QUIET);             // read all partition info
   }
   RETURN (dfsDisks);
}                                               // end 'dfsPartitionableDisks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get reference to diskinfo for specified physical disk
/*****************************************************************************/
DFSDISKINFO *dfsGetDiskInfo
(
   USHORT              disk                     // IN    disk number 1..n
)
{
   DFSDISKINFO        *rc = NULL;               // function return

   if ((disk <= dfsDisks) && (disk != 0))
   {
      rc = &(dfsDiskInfo[disk]);
   }
   return (rc);
}                                               // end 'dfsGetDiskInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return number of partitions administred by the partition-manager
/*****************************************************************************/
USHORT dfsPartitions
(
   void
)
{
   ENTER();
   if (dfsDiskInfo == NULL)
   {
      dfsReadDiskInfo( FDSK_QUIET);             // read all partition info
   }
   RETURN (dfsNrPart);
}                                               // end 'dfsPartitions'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get reference to partition info for specified partition number (PID)
/*****************************************************************************/
DFSPARTINFO *dfsGetPartInfo                     // RET   partition info or NULL
(
   USHORT              part                     // IN    part number 1..n
)
{
   DFSPARTINFO        *rc = NULL;               // function return

   if ((part <= dfsNrPart) && (part != 0))
   {
      rc = &(dfsPartInfo[part -1]);
   }
   return (rc);
}                                               // end 'dfsGetPartInfo'
/*---------------------------------------------------------------------------*/

#if !defined (OEMSB)
/*****************************************************************************/
// Test accessability for specified physical disk (from last Open attempt)
/*****************************************************************************/
BOOL dfsDiskAccessible
(
   USHORT              disk                     // IN    disk number 1..n
)
{
   BOOL                rc = FALSE;              // function return

   ENTER();
   TRACES(( "dfsDisks: %hu  disk: %hu\n", dfsDisks, disk));

   if ((disk <= dfsDisks) && (disk != 0))
   {
      rc = (dfsDiskInfo[disk].OpenError == FALSE);
   }
   BRETURN (rc);
}                                               // end 'dfsDiskAccessible'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get number of accessible disks (non-dummy)
/*****************************************************************************/
USHORT  dfsAccessibleDisks                      // RET   nr of non-dummy disks
(
   void
)
{
   USHORT              rc = 0;
   DFSDISKINFO        *d;                       // ptr to phys disk info
   USHORT              disk;

   ENTER();

   for (disk = 1; disk <= dfsDisks; disk++)
   {
      d = &(dfsDiskInfo[disk]);
      if (d && (d->OpenError == FALSE))
      {
         rc++;
      }
   }
   RETURN (rc);
}                                               // end 'dfsAccessibleDisks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get reference to freespace info for specified freespace area
/*****************************************************************************/
DFSPARTINFO *dfsGetFreeInfo                     // RET   freespace pointer
(
   USHORT              fsid                     // IN    freespace id 1..n
)
{
   DFSPARTINFO        *rc = NULL;               // function return
   DFSPARTINFO        *f;                       // ptr to freespace info
   DFSDISKINFO        *d;                       // ptr to phys disk info
   USHORT              disk;

   ENTER();
   TRARGS(("fsid: %hu\n", fsid));

   for (disk = 1; (disk <= dfsDisks) && (rc == NULL); disk++)
   {
      d = &(dfsDiskInfo[disk]);
      for (f = d->fspHead; f &&  (rc == NULL); f = f->fspChain)
      {
         if (f->id == fsid)                     // wanted one ?
         {
            rc = f;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsGetFreeInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get reference to freespace AFTER specified partition, if any
/*****************************************************************************/
DFSPARTINFO *dfsGetFreeAfter                    // RET   freespace pointer
(
   USHORT              pid                      // IN    partition id
)
{
   DFSPARTINFO        *rc = NULL;               // function return
   DFSPARTINFO        *f;                       // ptr to freespace info
   DFSDISKINFO        *d;                       // ptr to phys disk info
   USHORT              disk;

   ENTER();
   TRARGS(("pid: %u\n", pid));

   if ((f = dfsGetPartInfo( pid)) != NULL)
   {
      disk = f->disknr;

      if ((d = &(dfsDiskInfo[disk])) != NULL)
      {
         for (f = d->fspHead; f &&  (rc == NULL); f = f->fspChain)
         {
            if      (f->id == (pid +1))
            {
               rc = f;                          // fsp in between ...
            }
            else if ((f->id > pid) && (pid == d->lastpart))
            {
               rc = f;                          // fsp at end disk ...
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsGetFreeAfter'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Read basic physical disk information and add partition info for each disk
/*****************************************************************************/
ULONG dfsReadDiskInfo
(
   USHORT              geodisk                  // IN    nr/FDSK_ANY/0 (quiet)
)
{
   ULONG               rc = NO_ERROR;           // API rc
   USHORT              disk;
   USHORT              pbm = DFS_MAX_PART;      // index Bootmgr partition
   S_BMPRIM           *bmpi;                    // Bootmgr part-info sectors
   USHORT              index;
   FDSK_CB_INFO       *cbi = NULL;              // callback parameter struct
   DFST_HANDLE         cst = dfstGetDefaultStore();
   BOOL                verbose = (geodisk != 0);
   DFSPARTINFO        *p;

   ENTER();

   TRHEAP( 500);
   TxCancelAbort();                             // might have an aborted action
   dfstSetDefaultStore( DFST_SYSTEM);           // select system store

   dfsa->OS2FsPresent = FALSE;                  // determine from new info
   if ((rc = dfsAllocDiskInfo( verbose)) == NO_ERROR) // allocate memory
   {
      dfsa->mbr1I13X = FALSE;                   // no I13X capability known
      dfsa->winCandidates = 0;                  // no Windows detected yet

      for (disk = 1; (disk <= dfsDisks) && (rc == NO_ERROR); disk++)
      {
         BOOL   showgeo = ((disk == geodisk) || (geodisk == FDSK_ANY));

         // Devicename needs to be 12 positions for the MAC '/dev/disk0' .. 99 naming
         sprintf( dfsDiskInfo[ disk].UnixDeviceDisk, "%-12.12s", dfsDid2DeviceName( disk));
         #if defined (UNIX)                     // check just once per disk for UNIX ...
            if (TxFsIsRemovable( dfsDiskInfo[ disk].UnixDeviceDisk))
            {
               dfsDiskInfo[ disk].Removable = TRUE;
            }
         #elif defined (WIN32)                  // attempt to get disk-level removable info
         {
            TXTS     diskdevice;

            sprintf( diskdevice, "%hu", disk - 1); // '0' is disk-1
            if (TxFsIsRemovable( diskdevice))
            {
               dfsDiskInfo[ disk].Removable = TRUE;
            }
         }
         #endif

         TRHEAP( 500);
         rc = dfstOpenDisk( DFSTORE, disk, showgeo, FALSE);
         dfsReadPartInfo( disk);                // always read, ignore errors
         if (rc != NO_ERROR)
         {
            switch (rc)                         // open-disk result
            {
               case ERROR_NOT_READY:
                  strcpy( dfsDiskInfo[disk].DiskName, "not ready removable");
                  break;

               case ERROR_ACCESS_DENIED:
                  strcpy( dfsDiskInfo[disk].DiskName, "no access, locked");
                  break;

               default:
                  sprintf(dfsDiskInfo[disk].DiskName, "access error:%3.3u!", rc);
                  break;
            }
            dfsDiskInfo[disk].OpenError = TRUE; // disk cannot be opened
            rc = NO_ERROR;
         }

         TRACES(("Handled disk %hu\n", disk));
         TRHEAP( 500);
      }
      TRACES(("Sort PartInfo table for %hu partitions\n", dfsNrPart));
      qsort(dfsPartInfo, (size_t) dfsNrPart, sizeof(DFSPARTINFO), dfsComparePartInfo);

      TRHEAP( 500);
      TRACES(("Initialize drive-names and indexes\n"));
      for (index = 0; index < dfsNrPart; index++)
      {
         p = &dfsPartInfo[index];
         strcpy( p->drive,"--");                // initialize drive names
         p->id = index +1;                      // reference id for part
         if (p->ebrChain != NULL)
         {
            p->ebrChain->id = index +1;
         }
      }
      TRHEAP( 500);

      dfsa->lvmPresent = FALSE;                 // init to no LVM
      if ((cbi = TxAlloc( 1, sizeof( FDSK_CB_INFO))) != NULL)
      {
         cbi->disknr       = FDSK_ANY;          // to their default values
         cbi->flag         = FALSE;             // only visit LVM sectors
         cbi->option       = TRUE;              // including Signature ones
         cbi->doAllBBR     = TRUE;              // non 0x35 too (warnings)
         cbi->nowrite      = TRUE;
         rc = dfsIteratePlvm( dfsPartLvmMatch, cbi);
         TxFreeMem( cbi);
      }

      #if !defined (DFS_PHYSDISK_ONLY)
         if       (TxaOption(DFS_O_STD))        // Force industry-standard ...
         {
            dfsCalculateDriveLetters();         // Use algorithm to assign
         }
         else if (!TxaOption('L'))              // Unless specials only ...
         {
            dfsCorrelateDriveVolumes();         // using bootsector CRC's
         }
      #elif !defined (OEMSB)                    // no driveletters for setboot
         dfsCalculateDriveLetters();            // Use algorithm to assign
      #endif

      TRHEAP( 500);
      if (!TxaOptUnSet('L'))                    // Unless specials suppressed
      {
         #if defined (WIN32) && !defined (OEMSB)
            ULONG merger = TxaOptNum('L', NULL, 2); // Win default is DosDevice

            if (merger == 2)
            {
               dfsMergeDosDeviceMapping();      // adds prefered driveletters
            }                                   // from NT DosDevice mapping
         #else
            ULONG merger = TxaOptNum('L', NULL, 1); // others use LVM
         #endif

         dfsMergeLvmDriveInfo(merger == 1);     // add BMGR labels from LVM
      }                                         // and driveletters if TRUE
      TRHEAP( 500);

      for (index = 0; index < dfsNrPart; index++)
      {
         p = &dfsPartInfo[index];
         if (strcmp( p->drive,"--") == 0)
         {
            strcpy( p->drive, "");              // remove non-assigned ones
         }                                      // for better readability
         else if (isalpha( p->drive[0]))
         {
            if (TxFsIsRemovable( p->drive))
            {
               dfsDiskInfo[ p->disknr].Removable = TRUE;
            }
         }

         // Set LVM-signature sector missing flags
         if (p->partent.PartitionType == DFS_P_WARP_LVM)
         {
            if ((p->lvmPresent) && (!p->lvmSigPres))
            {
               p->flag2 |= DFS_F_LVM_NO_SIG;
            }
         }
         else                                   // should NOT have a signature
         {
            if ((p->lvmPresent) && (p->lvmSigPres))
            {
               p->flag2 |= DFS_F_SIG_NO_LVM;
            }
         }

         // Set linux device number out-of-range flags
         if (p->UnixDevNr > 63)
         {
            p->flag2 |= DFS_F_LINUXHDA63;
         }

         // Set DLAT-overlap flag if LBA-offset too small on MBR style disks
         if ((p->tablenr != GPT_STYLE) &&
             (p->tablenr != APM_STYLE) &&
             (p->partent.BootSectorOffset < p->geoSecs))
         {
            p->flag2 |= DFS_F_DLATINPART;
         }
      }
      TRHEAP( 500);

      dfsa->bmgrDisk = 0;                       // no BMGR identified yet
      dfsa->bmgrI13X = DFSI13X_NOTTHERE;        // no I13X capability known
      for (index = 0; index < dfsNrPart; index++) // find bootmanager info
      {
         p = &dfsPartInfo[index];
         if (p->primary)
         {
            if (p->partent.PartitionType == DFS_P_BOOTMGR)
            {
               TRACES(("ReadDisk BMGR on disk %hu is at:%llx\n", p->disknr, p->basePsn));

               disk = p->disknr;
               dfsDiskInfo[ disk].bmgrPsn = p->basePsn;
               if (dfsa->bmgrDisk != 1)         // first has preference
               {                                // (and no warning yet)
                  dfsa->bmgrDisk = disk;        // remember the disk
                  pbm = index;                  // found the BM partition
               }
               else if ((disk != 1) && (disk != 2))
               {
                  p->flags |= DFS_F_BMGR_OTHER;
               }
            }
         }
      }
      TRHEAP( 500);

      if (pbm != DFS_MAX_PART)                  // Bootmgr present ?
      {
         p = &dfsPartInfo[pbm];
         if (strncasecmp( p->plabel, "I13X", 4) == 0)
         {
            if (strncasecmp( p->plabel, "I13Xopt", 7) == 0)
            {
               dfsa->bmgrI13X = DFSI13X_OPTIONAL; // bmgr has I13X capability
            }
            else
            {
               dfsa->bmgrI13X = DFSI13X_REQUIRED; // bmgr has I13X requirement
            }
            TRACES(("BMGR I13X status: %s\n", p->plabel));
         }
         if ((dfsa->bmgrDisk != 1) && (dfsa->bmgrDisk != 2))
         {
            p->flags |= DFS_F_BMGR_NOT12;
         }
         else if (p->partent.Status == 0)
         {
            p->flags |= DFS_F_BMGR_INACT;
         }
         rc = dfstOpenDisk( DFSTORE, p->disknr, FALSE, FALSE);
         if (rc == NO_ERROR)
         {
            if ((bmpi = TxAlloc( BMPRIMSECTS, dfsGetSectorSize())) != NULL)
            {
               BMPRIMARY *bmi = (BMPRIMARY *) bmpi; // Bootmgr info, first partition

               rc = dfstReadPsn( DFSTORE, p->basePsn + BMPRIMSECTN, BMPRIMSECTS, (BYTE   *) bmpi);
               if ((rc == NO_ERROR) &&          // assign BM-labels for prim
                   (dfsa->lvmPresent == FALSE)) // when not done by LVM info
               {
                  TRHEXS(70, bmpi, 0xC0, "BMPRIMARY");
                  for (index = 0; index < dfsNrPart; index++)
                  {
                     DFSPARTINFO *pi = &dfsPartInfo[index];

                     if (pi->primary)
                     {
                        ULONG  bi;

                        if (dfsFdskGetBmiIndex( pi->disknr, &(pi->partent), bmi, &bi) != DFS32MAX)
                        {
                           if ((bmi[bi].BootMgrFlag == BT_BM_BOOT) && (bmi[bi].BootMgrName[8] == 0)) // properly terminated
                           {
                              TRACES(( "BootMgrName: '%s'\n", bmi[bi].BootMgrName));
                              if (bmi[bi].BootMgrName[0] != '-' ) // not --> LVM
                              {
                                 strcpy( pi->blabel, bmi[bi].BootMgrName);

                                 TRACES(("Assigned '%s' to blabel for (PRI) partition: %hu\n", pi->blabel, index +1));
                              }
                              strcpy( pi->bmname, bmi[bi].BootMgrName);
                           }
                        }
                     }
                  }
               }
               if (rc == NO_ERROR)
               {
                  BMINFOSEC *bis = (BMINFOSEC *) bmpi; // Global BMGR info

                  rc = dfstReadPsn( DFSTORE, p->basePsn + BMINFOSECTN, BMINFOSECTS, (BYTE   *) bmpi);
                  if (rc == NO_ERROR)           // get BMGR timeout value
                  {
                     dfsa->bmgrTimeout = bis->timeout / BMSYS_TIMEOUT;
                  }
               }
               TxFreeMem( bmpi);
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
      }
      TRHEAP( 500);
      #if !defined (DFS_PHYSDISK_ONLY)
         dfsSetOsFreeSpace();                   // determine OS FS-freespace
      #endif
      #if !defined (OEMSB)                      // info not needed for setboot
         dfsAdditionalDiskInfo();               // count partitions, primaries
         dfsSetRelDiskPart();                   // assign NT part numbers
         dfsSetFreeSpaceType();                 // identify freespace areas
         dfsSetExpandSize();                    // set expandSize and fspAfter
      #endif
   }
   #if defined (USEWINDOWING)                   // always, even if read failed
      txSelDestroy( &(dfsa->slBootabl));
      txSelDestroy( &(dfsa->slFreeCCr));
      txSelDestroy( &(dfsa->slFreeMBm));
      txSelDestroy( &(dfsa->slFreeGPr));
      txSelDestroy( &(dfsa->slFreeLog));
      txSelDestroy( &(dfsa->slFreeAll));
      txSelDestroy( &(dfsa->slPartAll));
      txSelDestroy( &(dfsa->slPartLog));
      txSelDestroy( &(dfsa->slPartOne));        // remove current list contents
      txSelDestroy( &(dfsa->slDiskOne));        // and create new ones
      txSelDestroy( &(dfsa->slDiskAll));
      txSelDestroy( &(dfsa->slDiskMnt));
      txSelDestroy( &(dfsa->slDiskSam));

      dfsa->slDiskSam = dfsDiskSelist( TRUE,  TRUE); // start with the 'ALL disks' list
      if ((dfsa->slDiskSam != 0) && (dfsa->slDiskSam->count > 0))
      {
         TRACES(("Replacing item/desc texts for slDiskSam\n"));
         strcpy( dfsa->slDiskSam->items[0]->text, "   SAME disk 'x' as the .PDx file");
         strcpy( dfsa->slDiskSam->items[0]->desc, "Restore partition info to SAME disk, the 'x' from the .PDx file extension");
      }
      dfsa->slDiskAll = dfsDiskSelist( TRUE,  TRUE);
      dfsa->slDiskOne = dfsDiskSelist( FALSE, TRUE);
      dfsa->slDiskMnt = dfsDiskSelist( FALSE, FALSE);
      dfsa->slPartAll = dfsPartSelist( TRUE,  FDSK_ANY, 'A');
      dfsa->slPartOne = dfsPartSelist( FALSE, FDSK_ANY, 'A');
      dfsa->slPartLog = dfsPartSelist( FALSE, FDSK_ANY, 'L');
      dfsa->slFreeAll = dfsFreeSelist( );
      dfsa->slFreeCCr = dfsFreeSelist( );
      dfsa->slFreeMBm = dfsFreeSelist( );
      dfsa->slFreeGPr = dfsFreeSelist( );
      dfsa->slFreeLog = dfsFreeSelist( );
      dfsa->slBootabl = dfsBmgrSelist( );
      TRHEAP( 500);
   #endif                                       // USEWINDOWING
   if (rc == NO_ERROR)                          // re-select current disk
   {
      dfstUpdateStoreInfo( verbose);            // update PID/PARTINFO *
   }
   else if (verbose)
   {
      TxPrint("Error %u in ReadDiskInfo\n", rc);
      dfsExplainRC( rc);
   }
   dfstClose(   DFST_SYSTEM);                   // close last used disk
   dfstSetDefaultStore( cst);                   // reselect current store

   if (verbose &&    (dfsNrPart == 0) &&        // no partitions found, and
       (dfsGetDiskCount( FALSE) != 0) &&        // there ARE partitionable media
       (!TxaExeSwitchUnSet('d')     )  )        // and we did want real disks ...
   {
      dfsShowAdminHint( "WARNING: No disk partitions found, disks seem to be empty!\n\n"
                        "         The first sector (MBR) might be invalid or cleared, or\n");
   }
   RETURN (rc);
}                                               // end 'dfsReadDiskInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display hints about needing admin/root priviliges to see disk info
/*****************************************************************************/
static void dfsShowAdminHint
(
   char               *leadtext                 // IN    upto two lines of leading text
)
{
   static BOOL         needAck = TRUE;          // let user acknowledge just once
   char               *at;

   if ((at = TxAlloc( 1, TXMAX1K)) != NULL)
   {
      strcpy( at, leadtext);

      #if defined (DEV32)
         strcat( at, "         another FDISK, LVM, PQMagic or DFSee program is active, or\n");
      #endif
         strcat( at, "         there is no working driver for the main harddisk,\n"
                     "         or no access to that disk is permitted.\n\n"
                     "         (Or there really are no disks/partitions present :-)\n");
      #if   defined (DOS32)
      #elif defined (WIN32)
         strcat( at, "\n Win-XP..Win-10: You need to be logged on with an administrator account.\n\n");
         if (TxOsVersion( NULL) >= TXVER_WINNT_VISTA) // need 'start as admin'
         {
            strcat( at, "         On Windows (Vista, 7,8,10) you may need to do that by using the"
                      "\n         'Run as administrator' option (right mouse button). Also make"
                      "\n         sure the 'User Access Control (UAC) is NOT set to the maximum"
                      "\n         security level in 'Control Panel -> Security Settings'\n\n");
         }
      #elif defined (UNIX)
         strcat( at, "\n   Unix: Your userid must be a member of the 'disk' group, you\n"
                       "         must login as 'root', or use a 'su' or 'sudo' command.\n\n");
      #else
         strcat( at, "\n   OS/2: If HPFS386 local-security is active, logon with\n"
                       "         an admin userid or use a privileged session.\n\n");
      #endif

      if (needAck)
      {
         TxNamedMessage( TRUE, 0, " Info: Admin required hint ", at);
         needAck = FALSE;
      }
      TxPrint( "%s", at);                       // echo to scrollbuffer/log too
      TxFreeMem( at);
   }
   else
   {
      TxPrint( "\nOut of memory!\n");
   }

}                                               // end 'dfsShowAdminHint'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Discard disk-info structures and free any associated resources
/*****************************************************************************/
void dfsFreeDiskInfo
(
   void
)
{
   USHORT              index;
   DFSPARTINFO        *this;
   DFSPARTINFO        *next;

   ENTER();
   for (index = 1; index <= dfsDisks; index++)
   {
      for (this = dfsDiskInfo[ index].ebrHead; this; this = next)
      {
         next = this->ebrChain;
         TxFreeMem( this);
      }
      for (this = dfsDiskInfo[ index].fspHead; this; this = next)
      {
         next = this->fspChain;
         TxFreeMem( this);
      }
      TxFreeMem( dfsDiskInfo[ index].gptHeader);
      TxFreeMem( dfsDiskInfo[ index].gptArray);
   }
   TxFreeMem(  dfsDiskInfo);
   dfsDisks  = 0;                               // reset disk info table

   for (index = 0; index < dfsNrPart; index++)
   {
      TxFreeMem( dfsPartInfo[index].pbrSect);
   }
   TxFreeMem(  dfsPartInfo);
   dfsNrPart = 0;                               // reset part info table
   VRETURN();
}                                               // end 'dfsFreeDiskInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate disk-info structures
/*****************************************************************************/
static ULONG dfsAllocDiskInfo
(
   BOOL               verbose                   // IN    Report to screen too
)
{
   ULONG               rc = DFS_ALLOC_ERROR;    // function return

   ENTER();
   dfsFreeDiskInfo();
   dfsDisks = dfsGetDiskCount( verbose);
   dfsPartInfo = TxAlloc(DFS_MAX_PART +1, sizeof(DFSPARTINFO));
   if (dfsPartInfo != NULL)
   {
      dfsDiskInfo = TxAlloc( dfsDisks +1, sizeof(DFSDISKINFO));
      if (dfsDiskInfo != NULL)
      {
         if (dfsDisks == 0)
         {
            if (!TxaExeSwitchUnSet('d'))        // no -d-
            {
               dfsShowAdminHint( "WARNING: No partitionable disks found at all!\n\n"
                                 "         Either there are no disks, you have insufficient rights, or\n");
            }
            else
            {
               TxPrint("INFO: Started with explicit switch to ignore physical disks.\n\n");

            }
            rc = DFS_NO_DEVICE;                 // OK, but no devices present
         }                                      // will avoid much work :-)
         else
         {
            rc = NO_ERROR;                      // OK to read Partition info
         }
      }
      else
      {
         TxFreeMem( dfsPartInfo);
      }
   }
   RETURN (rc);
}                                               // end 'dfsAllocDiskInfo'
/*---------------------------------------------------------------------------*/

#if !defined (OEMSB)
/*****************************************************************************/
// Calculate drive-letter based on PartInfo and industry-standard algorithm
/*****************************************************************************/
static void dfsCalculateDriveLetters
(
   void
)
{
   USHORT              index;
   char                drive = 'B';             // first drive-letter minus one
   BYTE                type;

   ENTER();
   TRACES(("NrPart: %hu\n", dfsNrPart));

   //- to be refined, assign FAT32 or not, based on current OS or option letter
   //- assign primaries driveletter, upto 'Z'
   for (index = 0; (index < dfsNrPart) && (drive < 'Z'); index++)
   {
      type = dfsPartInfo[index].partent.PartitionType;
      if (dfsPartInfo[index].primary)
      {
         switch (type)
         {
            case DFS_P_FAT12     : case DFS_P_FAT16     : case DFS_P_FAT16X    : case DFS_P_BIGDOS    :
            case DFS_P_INST_FS   : case DFS_P_IFSHIDDEN :
            case DFS_P_FAT32     : case DFS_P_FAT32X    :
            case DFS_P_ACRONISSZ :
               sprintf(dfsPartInfo[index].drive, "%c:", ++drive);
               TRACES(("PRIM part %2d ==> %s%s%s\n", index +1, CBG, dfsPartInfo[index].drive, CNN));
               break;

            default:
               break;
         }
      }
   }

   //- assign logicals driveletter, upto 'Z'
   for (index = 0; (index < dfsNrPart) && (drive < 'Z'); index++)
   {
      type = dfsPartInfo[index].partent.PartitionType;
      if (dfsPartInfo[index].primary == FALSE)
      {
         switch (type)
         {
            case DFS_P_FAT12     : case DFS_P_FAT16     : case DFS_P_FAT16X    : case DFS_P_BIGDOS    :
            case DFS_P_INST_FS   : case DFS_P_IFSHIDDEN :
            case DFS_P_FAT32     : case DFS_P_FAT32X    :
            case DFS_P_ACRONISSZ :
               sprintf(dfsPartInfo[index].drive, "%c:", ++drive);
               TRACES(("LOG  part %2d ==> %s%s%s\n", index +1, CBG, dfsPartInfo[index].drive, CNN));
               break;

            default:
               break;
         }
      }
   }
   VRETURN();
}                                             // end 'dfsCalculateDriveLetters'
/*---------------------------------------------------------------------------*/
#endif


#if !defined (DFS_PHYSDISK_ONLY)
/*****************************************************************************/
// Determine freespace for each partition as reported by the OS
/*****************************************************************************/
static void dfsSetOsFreeSpace
(
   void
)
{
   USHORT              index;                   // partition index
   USHORT              bps;                     // bytes per sector
   ULN64               frees;                   // free  sectors
   ULN64               total;                   // total sectors
   DFSPARTINFO        *p;                       // ptr to partition info

   ENTER();

   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups
   for (index = 0; index < dfsNrPart; index++)
   {
      p = &(dfsPartInfo[index]);
      p->secFree = 0;

      if ((p->flags & DFS_F_BOOTRECBAD) == 0)   // valid bootrec ?
      {
         if (TxFsSpace(p->drive, &frees, &total, &bps) == NO_ERROR)
         {
            p->secFree = frees;
         }
      }
   }
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   VRETURN();
}                                               // end 'dfsSetOsFreeSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Correlate the current operating-system volume-map with found partitions
/*****************************************************************************/
static void dfsCorrelateDriveVolumes            // nr of correlated drives
(
   void
)
{
   ULONG               checksum;                // boot checksum
   ULN64               dummy;
   USHORT              index;
   TXTS                drive;
   TXTS                fsys;
   TXTS                volumes;
   S_BOOTR            *boot;                    // boot record buffer
   char               *s;

   ENTER();

   TxFsVolumes( TXFSV_HD, volumes);             // no CDROM, LAN or FLOPPY
   TRACES(("Volumes: '%s', NrPart: %hu\n", volumes, dfsNrPart));

   for (s = volumes; *s; s++)                   // walk all volume letters
   {
      sprintf(drive, "%c:", toupper(*s));
      TxFsAutoFailCriticalErrors( TRUE);        // avoid Not-ready pop-ups
      TRACES(( "Start for drive '%s'\n", drive));
      if (TxFsSpace(drive, &dummy, &dummy, &index) == NO_ERROR) // valid and Ready ?
      {
         TxFsType( drive, fsys, NULL);
         strcat(          fsys, " ");           // terminate with a space
         TxStrToUpper(    fsys);
         TRACES(("fsys: '%s'  skipFs: '%s'\n", fsys, dfsa->skipFs));
         if ((strlen( dfsa->skipFs) <= 1) ||    // no skipFs defined ?
             (strstr( dfsa->skipFs, fsys) == NULL))
         {
            if (dfstOpenVolume( DFSTORE, drive, fsys, FALSE) == NO_ERROR)
            {
               if ((boot = TxAlloc( 1, dfsGetSectorSize())) != NULL)
               {
                  //- DUMPFS has a volumeletter, and first sectors do match, but no SIG!
                  //- if ((dfsRead( LSN_BOOTR, 1, (BYTE   *) boot) == NO_ERROR) && (boot->Signature == SV_BOOTR))

                  if (dfsRead( LSN_BOOTR, 1, (BYTE   *) boot) == NO_ERROR)
                  {
                     checksum = dfsCheckSum((BYTE *) boot);

                     for (index = 0; index < dfsNrPart; index++)
                     {
                        DFSPARTINFO *p = &(dfsPartInfo[index]);

                        TRACES(("Corr: Index: %hu of: %hu = '%s'\n", index, dfsNrPart, p->drive));
                        if ( (p->checksum == checksum) &&
                            ((p->flags & DFS_F_BOOTRECBAD) == 0))
                        {
                           TRACES(("part %2d drive: %s%s%s matching %s%s%s\n", index +1, CBC, drive, CNN, CBY, p->drive, CNN));
                           if (strcmp(p->drive, "--") == 0)
                           {
                              strcpy( p->drive, drive);
                           }
                           else                 // multiple match, Duplicate
                           {
                              p->drive[1] = drive[0];
                              p->flag2 |= DFS_F_BRDUPLICAT; // BR not unique
                           }
                        }
                     }
                  }
                  TxFreeMem( boot);
               }
               else
               {
                  TRACES(("Alloc fail %hu bytes for drive %s\n", dfsGetSectorSize(), drive));
               }
               dfstClose( DFSTORE);
            }
         }
         else
         {
            TRACES(( "Skipped '%s', being a '%s' type filesystem\n", drive, fsys));
         }
      }
      TRACES(( "Done with drive '%s'\n", drive));
      TxFsAutoFailCriticalErrors( FALSE);       // enable criterror handler
   }
   VRETURN();
}                                             // end 'dfsCorrelateDriveVolumes'
/*---------------------------------------------------------------------------                    */
#endif

#if !defined (OEMSB)
/*****************************************************************************/
// Correlate the current operating-system volume-map with specific bootsector
/*****************************************************************************/
static void dfsCorrelateBootsector
(
   S_BOOTR            *bsect,                   // IN    boootsector to match
   char               *bdrive                   // OUT   driveletter, or ""
)
{
   ULONG               csb = dfsCheckSum((BYTE *) bsect);
   ULONG               csv;
   ULN64               dummy;
   USHORT              index;
   TXTS                drive;
   TXTS                fsys;
   TXTS                volumes;
   S_BOOTR            *boot;                    // boot record buffer
   char               *s;

   ENTER();

   TxFsVolumes( TXFSV_HD, volumes);             // no CDROM, LAN or FLOPPY
   TRACES(("Volumes: '%s'\n", volumes));

   for (s = volumes; *s; s++)                   // walk all volume letters
   {
      sprintf(drive, "%c:", toupper(*s));
      TxFsAutoFailCriticalErrors( TRUE);        // avoid Not-ready pop-ups
      TRACES(( "Start for drive '%s'\n", drive));
      if (TxFsSpace(drive, &dummy, &dummy, &index) == NO_ERROR) // valid and Ready ?
      {
         TxFsType( drive, fsys, NULL);
         strcat(   fsys, " ");                  // terminate with a space
         if ((strlen( dfsa->skipFs) >  1) ||    // more than one space there ?
             (strstr( dfsa->skipFs, fsys) == NULL))
         {
            if (dfstOpenVolume( DFSTALT, drive, fsys, FALSE) == NO_ERROR)
            {
               if ((boot = TxAlloc( 1, dfsGetSectorSize())) != NULL)
               {
                  if (dfstReadLsn( DFSTALT, LSN_BOOTR, 1, (BYTE   *) boot) == NO_ERROR)
                  {
                     csv = dfsCheckSum((BYTE *) boot);
                     TRACES(( "Drive %s cs:%8.8x, bsect-cs:%8.8x\n", drive, csv, csb));
                     if (csv == csb)
                     {
                        strcpy( bdrive, drive);
                     }
                  }
                  TxFreeMem( boot);
               }
               dfstClose( DFSTALT);
            }
         }
      }
      TRACES(( "Done with drive '%s'\n", drive));
      TxFsAutoFailCriticalErrors( FALSE);       // enable criterror handler
   }
   VRETURN();
}                                             // end 'dfsCorrelateBootsector'
/*---------------------------------------------------------------------------                    */


/*****************************************************************************/
// Set relative partition-numbers as used by NT \Device\HardDiskX\PartitionY
// Note: Updated to use the real ARC numbering order starting in ver 6.07
// Primaries ordered based on their PTable-index, not physical order on disk
/*****************************************************************************/
static void dfsSetRelDiskPart
(
   void
)
{
   USHORT              index;                   // partition index
   USHORT              disk;
   USHORT              rdpart = 0;              // relative disk partition
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSDISKINFO        *d;                       // ptr to phys disk info

   ENTER();

   for (disk = 0, index = 0; index < dfsNrPart; index++) // assign LOGICALS
   {
      p = &(dfsPartInfo[index]);

      if (disk != p->disknr)
      {
         disk  = p->disknr;
         d = &(dfsDiskInfo[disk]);
         rdpart = d->primaries + 1;             // number for first logical
      }
      if (p->primary == FALSE)
      {
         p->diskPart = rdpart++;
      }
      TRACES(( "L Part: %2u NT HD:%hu P:%hu\n", index+1, disk, p->diskPart));
   }
   VRETURN();
}                                               // end 'dfsSetRelDiskPart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set freespace-AFTER references and maximum expandSize values
/*****************************************************************************/
static void dfsSetExpandSize
(
   void
)
{
   USHORT              index;                   // partition index
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSPARTINFO        *f;                       // ptr to freespace area AFTER

   ENTER();

   for (index = 0; index < dfsNrPart; index++)
   {
      p = &(dfsPartInfo[index]);
      if ((f = dfsGetFreeAfter( p->id)) != NULL)
      {
         p->fspAfter   = f;
         p->expandSize = p->sectors + f->sectors;
         TRACES(( "expandSize pid %hu + fsid %hu: %8.8x = %11.1lf MiB\n",
                   p->id, f->id, p->expandSize, TXSMIB( p->expandSize, p->bpsector)));
      }
   }
   VRETURN();
}                                               // end 'dfsSetExpandSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Gather additional info on partitions and freespace on each physical disk
/*****************************************************************************/
static void dfsAdditionalDiskInfo
(
   void
)
{
   USHORT              index;                   // partition index
   USHORT              disk;                    // partition index
   ULN64               endsec;                  // last sector prev part
   ULN64               startsec;                // startsector new part
   ULONG               actives = 0;             // nr of actives on all disks
   ULONG               actived;                 // nr of actives on this disk
   DFSPARTINFO        *warning = NULL;          // ptr to general warning part
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSPARTINFO        *e;                       // ptr to extended container
   DFSDISKINFO        *d;                       // ptr to phys disk info
   ULN64               extBase = 0;             // start of Ext-container
   ULN64               extLast = 0;             // end   of Ext-container
   BOOL                AirBoot = FALSE;         // there is an AirBoot on disk-1 or 2

   ENTER();

   for (disk = 1; disk <= dfsDisks; disk++)     // visit every physical disk
   {
      d = &(dfsDiskInfo[disk]);
      d->primaries   = 0;
      d->totalpart   = 0;
      d->firstpart   = 0;
      d->lastpart    = 0;
      d->lastUsedPsn = 0;
      endsec         = 0;
      actived        = 0;
      if (d->ebrHead)
      {
         extBase = d->ebrHead->basePsn;         // start of Ext-container
         extLast = d->ebrHead->lastPsn;         // end   of Ext-container
      }

      TRACES(("d->AirBootPresent for disk %hu: %s\n", disk, (d->AirBootPresent) ? "YES" : "NO"));
      if (d->AirBootPresent && ((disk == 1) || (disk == 2))) // to be refined ? (all disks ?)
      {
         AirBoot = TRUE;
      }

      TRACES(("disk:%hu extBase: 0x%llx extLast: 0x%llx  disksects: %llx\n", disk, extBase, extLast, d->sectors));
      for (index = 0; index < dfsNrPart; index++)
      {
         p = &(dfsPartInfo[index]);

         if (p->disknr == disk)                 // partition on this disk
         {
            if ((p->partent.Status & DFS_P_ACTIVE) && p->primary)
            {
               actives++;                       // an active partition
               actived++;
               if (actived > 1)                 // but violating a rule!
               {
                  p->flags |= DFS_F_MULTIP_ACT; // multiple active primaries
               }
            }
            if (((e = p->ebrChain)  != NULL) && // It is a logical partition
                 (e->tablenr        == 0)    && // and the outer container
                 (e->partent.Status != 0)     ) // ans set ACTIVE
            {
               actives++;                       // an active partition
               actived++;
               if (actived > 1)                 // but violating a rule!
               {
                  p->flags |= DFS_F_MULTIP_ACT; // multiple active primaries
               }
            }
            if (p->partent.Status & ~DFS_P_ACTIVE)
            {
               p->flag2 |= DFS_F_NONSTDFLAG;    // non standard flags set
            }
            if (d->firstpart == 0)
            {
               d->firstpart = p->id;
            }
            d->lastpart    = p->id;             // keep assigning :-)
            d->lastUsedPsn = p->lastPsn;

            d->totalpart++;                     // count the partition

            TRACES(("Beyond check, disk size: 0x%llx  lastPsn: 0x%llx\n", d->sectors, p->lastPsn));
            if (p->lastPsn >= d->sectors )
            {
               p->flags |= DFS_F_PART_EDISK;    // beyond end of disk
            }
            if (p->ebrChain != NULL)
            {
               strcpy( p->ebrChain->drive, p->drive);
               if (p->ebrChain->lastPsn >= d->sectors )
               {
                  p->flags |= DFS_F_EXTC_EDISK; // beyond end of disk
               }
               if (p->basePsn < p->ebrChain->basePsn)
               {
                  p->flags |= DFS_F_LOG_BEFORE; // log before OWN container
               }
               if (p->lastPsn > p->ebrChain->lastPsn)
               {
                  p->flags |= DFS_F_LOG_BEYOND; // log beyond OWN container
               }
               if (p->ebrChain->basePsn < extBase)
               {
                  p->flags |= DFS_F_EXT_BEFORE; // EBR before MBR container
               }
               if (p->ebrChain->lastPsn > extLast)
               {
                  p->flags |= DFS_F_EXT_BEYOND; // EBR beyond MBR container
               }
               startsec = p->ebrChain->basePsn; // start of extended
            }
            else                                // must be a primary
            {
               startsec = p->basePsn;           // start of primary
               d->primaries++;
               if (d->ebrHead)                  // are there logicals ?
               {
                  if ((startsec >= extBase) && (startsec < extLast))
                  {
                     p->flag2 |= DFS_F_PRIM_INEXT; // Primary inside container
                  }
               }
            }
            TRACES(("index:%2.2hu basePsn: %llx lastPsn: %llx\n", index, p->basePsn, p->lastPsn));
            TRACES((    "        startsec: %llx  endsec: %llx\n", startsec, endsec));
            if (startsec > endsec +1)           // freespace before this part
            {
               //- first area starts at 0! others at last-end+1
               dfsAddFreeSpaceInfo( p, disk, (endsec) ? endsec   +1 : 0, startsec -1);
            }
            else if (startsec <= endsec)
            {
               if (startsec == 0)               // MBR overlap, perhaps ISOHybrid
               {
                  p->flag2 |= DFS_F_OVERLAPMBR; // overlaps MBR
               }
               else
               {
                  p->flags |= DFS_F_OVERLAPREV; // overlaps previous partition
               }
            }
            endsec = p->lastPsn;                // new last used sector

            // Check if the partion is bootable JFS/HPFS without driveletter
            if ((p->partent.Status && p->primary) || (strlen(p->blabel) > 0))
            {
                                                // active primary or logical on BM menu
               if ((dfsa->lvmPresent) && (p->pbrSect != NULL)) // LVM and BR available
               {
                  if (p->partent.PartitionType == DFS_P_INST_FS)  // type 07 IFS
                  {
                     //- check in actual bootsector, so skips any GRUB or other bootsectors
                     if ((strncmp( p->pbrSect->os.Type, "HPFS",  4) == 0) ||
                         (strncmp( p->pbrSect->os.Type, "JFS",   3) == 0)  )
                     {
                        if ((p->lvmPresent == FALSE) || (p->lvm.letter == 0))
                        {
                           //- warning, no driveletter on bootable/startable
                           p->flag2 |= DFS_F_NOBOOTLETT;
                        }
                        else if (p->lvmPresent == TRUE)
                        {
                           if ((p->pbrSect->os.FsysValue - 0x80 + 'C') != p->lvm.letter)
                           {
                              //- warning, BPB driveletter mismatch on bootable/startable
                              p->flag2 |= DFS_F_BADBPBLETT;
                           }
                        }
                     }
                  }
               }
            }
         }
      }
      if (d->sectors > endsec +1)               // freespace at end of disk
      {
         dfsAddFreeSpaceInfo( NULL, disk, (endsec) ? endsec   +1 : 0, d->sectors -1);
      }
      if ((warning == NULL) && (actives == 0) && (d->totalpart != 0))
      {
         if ((p = dfsGetPartInfo( d->firstpart)) != NULL)
         {
            warning = p;                        // partition receives warnings
         }
      }
   }
   if ((warning          != NULL)      &&       // there is a partition
       (warning->tablenr != GPT_STYLE) &&       // disk must be an MBR style
       (warning->tablenr != APM_STYLE) &&
       (AirBoot          == FALSE)      )      // and there is no AirBoot
   {
      TRACES(( "Adding 'No active' to PID %hu, actives = %u\n",
                warning->id, actives));
      if (actives == 0)                         // no actives at all
      {
         warning->flags |= DFS_F_NO_ACTIVES;    // no active at all
      }
      else
      {
         warning->flags |= DFS_F_NO_ACTIVE1;    // no active on disk 1
      }
   }
   VRETURN();
}                                               // end 'dfsAdditionalDiskInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Freespace info to a partition in table and in freeSpace list
/*****************************************************************************/
static DFSPARTINFO *dfsAddFreeSpaceInfo
(
   DFSPARTINFO        *p,                       // IN    related partition
   USHORT              disknr,                  // IN    physical disknr
   ULN64               first,                   // IN    first free sector
   ULN64               last                     // IN    last  free sector
)
{
   DFSPARTINFO        *f = NULL;                // new freespace element
   DFSPARTINFO        *tail;
   DFSDISKINFO        *d = &(dfsDiskInfo[ disknr]);

   ENTER();
   TRACES(("first: %llx  last:%llx\n", first, last));

   if ((d != NULL) && (f = TxAlloc( 1, sizeof(DFSPARTINFO))) != NULL)
   {
      f->pitype     = DFS_PI_FREESPACE;
      f->disknr     = disknr;
      f->geoCyls    = d->geoCyls;
      f->geoHeads   = d->geoHeads;
      f->geoSecs    = d->geoSecs;
      f->primary    = FALSE;                    // to be refined
      f->partPsn    = 0;
      f->basePsn    = first;
      f->lastPsn    = last;
      f->sectors    = last - first +1;
      f->relSize    = dfsLogarithmicSize( f->sectors);
      f->cSC        = d->cSC;
      f->bpsector   = d->bpsector;

      if (p != NULL)
      {
         p->fspChain  = f;                      // link from related partition
         f->id        = p->id;                  // link to   related partition
         f->ebrChain  = p->ebrChain;            // link to   related EBR info
         f->tablenr   = p->tablenr;
         f->partnr    = p->partnr;
         f->partPsn   = p->partPsn;
         f->flags     = p->flags;               // copy flags too
         f->flag2     = p->flag2;               // (EMPTYCONTN)
      }
      else                                      // freespace at end of disk
      {                                         // use 'derived' id
         f->id        = dfsNrPart + disknr;     // unique for each disk ...
      }
      if (d->fspHead == NULL)                   // first freespace area
      {
         d->fspHead   = f;                      // attach as head of list
      }
      else
      {
         for (tail = d->fspHead; tail->fspChain; tail = tail->fspChain)
         {
         }
         tail->fspChain = f;                    // attach to last element
      }
      TRACES(("FreeSpace %hu, size: 0x%llx bpsector: 0x%4.4hx\n", f->id, f->sectors, f->bpsector));
   }
   RETURN ( f);
}                                               // end 'dfsAddFreeSpaceInfo'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Read all partition info for opened disk, MBR-style first, GPT if present
/*****************************************************************************/
ULONG dfsReadPartInfo
(
   USHORT              disknr                   // IN    DFSee disk id
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   BYTE               *boot = NULL;             // boot sector buffer
   USHORT              ti = 0;                  // part-table index
   USHORT              pi;                      // partition index, straight
   USHORT              arcIndex;                // partition index, ARC style
   BYTE                pt;                      // partition type
   ULONG               ePSN = L32_NULL;         // first EBR, base for others
   ULONG               xPSN;                    // current EBR, based on ePSN
   ULONG               rPSN = 0;                // refering PSN, points to xPSN
   ULONG               rEnt = 0;                // refering entry in table
   ULONG               offs = 0;                // next EBR offset
   DFSPARTINFO        *ebrThis = NULL;          // EBR info for this logical
   DFSPARTINFO        *ebrNext = NULL;          // EBR info for next logical
   DFSPARTINFO        *ebrPrev = NULL;          // Previous element in chain
   DFSDISKINFO        *d;                       // diskinfo for this disk
   DFSPARTENTRY       *pe;                      // partition-table entry
   BOOL                guardPresent = FALSE;    // GPT guard partition 0xEE/0xEF
   BOOL                gptblPresent = FALSE;    // GPT primary tables present
   BOOL                apmapPresent = FALSE;    // APM tables are present
   USHORT              mbrPartitionsAdded = 0;  // MBR style partitions on THIS disk

   ENTER();

   if ((boot = TxAlloc( BBUFSECTORS, dfsGetSectorSize())) != NULL)
   {
      d                 = &(dfsDiskInfo[ disknr]);
      d->disknr         = disknr;
      d->sectors        = dfstRawDiskSize(   DFSTORE);
      d->geoCyls        = dfstGeoCylinders(  DFSTORE);
      d->geoHeads       = dfstGeoHeads(      DFSTORE);
      d->geoSecs        = dfstGeoSectors(    DFSTORE);
      d->bpsector       = dfsGetSectorSize();
      d->cSC            = d->geoHeads * d->geoSecs;
      d->bmgrPsn        = L64_NULL;
      d->lvmPartitions  = 0;
      d->gptPartitions  = 0;
      d->AirBootPresent = FALSE;

      strcpy( d->pStyle, DPSTR_UNK);            // default UNKNOWN partitioning style
      strcpy( d->dxInfo, "-");                  // default info, none

      TRACES(( "store:%u disk:%hu set GEO C:%6u H:%4u S:%3u B:%hu\n",
              DFSTORE, disknr, d->geoCyls, d->geoHeads, d->geoSecs, d->bpsector));

      // Recognize disks > 500GB with cyls > 65535, incompatible with OS/2 LVM
      #if !defined (DEV32)
         if (dfsa->lvmPresent)                  // LVM system, non-OS/2 booted
      #endif
      {
         TRACES(("OS/2 or eCS with > 64K cylinders: %u\n", d->geoCyls));
         if (d->geoCyls > 65535)
         {
            d->flags |= DFS_F_OS2_64K_CYL;
         }
      }

      if (dfsGetDiskIdentification( disknr, d->UnixDeviceDisk, LVM_NAME_L, d->DiskName) == FALSE)
      {
         //- Initialize the Diskname to some usable description, including size
         sprintf( d->DiskName, "%s D%hu ", dfsMediaTypeDescr( dfsDid2DiskType(disknr)), d->disknr);
         dfstrSz64XiB( d->DiskName, "", d->sectors, d->bpsector, "");
      }

      //- Do NOT retry with single-sector reading if any bad sectors (too often, way to slow!)
      //-  dfstBadSectorRead( DFSTORE, 0, BBUFSECTORS, NULL, &bads, boot);
      rc = dfstReadPsn( DFSTORE, 0, BBUFSECTORS, boot);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   if (rc == NO_ERROR)
   {
      S_BOOTR *pr = (S_BOOTR *) boot;          // partition-table record
      BYTE     st = dfsIdentifySector(0, 0, boot);
      BOOL     primaryEntryOK;                  // primary table entry should be added

      if (dfsIdentifySector( 1, 0, boot + dfsGetSectorSize()) == ST_GPTHD)
      {
         TRACES(("There are GPT tables\n"));
         gptblPresent = TRUE;                   // there IS a GPT table, so add the partitions
      }
      else if (dfsIdentifySector( 1, 0, boot + dfsGetSectorSize()) == ST_MCDPM)
      {
         TRACES(("There are APM tables\n"));
         apmapPresent = TRUE;                   // there IS an APM table
      }
      if (memcmp( boot + 2, "AiRBOOT", 7) == 0) // Air-Boot bootmanager present
      {
         TRACES(("Airboot present on disk\n"));
         d->AirBootPresent = TRUE;
      }
      if (pr->Signature == SV_BOOTR)            // is PC-style parttable
      {
         strcpy( d->pStyle, DPSTR_MBR);         // default partitioning style
         switch (st)
         {
            case ST_MCDDM:
               TxPrint( "WARNING disk nr %hu : Contains MAC driver description record (DDM) and\n"
                        "                    PC-style partition-tables, disk is NOT bootable on a PC!\n", disknr);
               if (dfsValidXbrTableEntries( boot))    // valid MBR style entries found
               {
                  st = ST_MASTR;                // MBR partitions prefered
               }
               break;

            case ST_EXTBR:
               TxPrint( "WARNING disk nr %hu : First sector has no MBR " "bootcode, disk not BIOS-bootable\n", disknr);
            default:
               TRACES(("checking I13X capability MBR on disk: %hu\n", disknr));
               if (disknr == 1)                 // check I13X capability
               {
                  if (TxMemStr( boot, "I13X", 500) != NULL)
                  {
                     dfsa->mbr1I13X = TRUE;    // I13X capable MBR code
                     TRACES(("Disk 1 MBR has I13X capable IBM/DFSee code\n"));
                  }
                  else if (memcmp( boot +2, "AiRBOOT", 7) == 0)
                  {
                     dfsa->mbr1I13X = TRUE;     // I13X capable MBR code
                     TRACES(("Disk 1 MBR has I13X capable AirBOOT code\n"));
                  }
               }
               break;
         }
      }
      else                                      // no valid 0x55AA signature
      {
         if (dfsValidXbrTableEntries( boot))    // but valid entries found
         {
            d->flags |= DFS_F_MBR_NO_SIG;
            st = ST_MASTR;                      // try to interpret further
         }
      }
      switch (st)
      {
         case ST_EXTBR:                         // disk with EBR, no MBR
         case ST_MASTR:                         // Normal MBR found
            TRHEXS(70,&(pr->PartitionTable[0]), 0x40, "pr->PartitionTable");
            d->NTsignature = pr->NTsignature;   // copy possible NT signature
            arcIndex = 1;                       // next ARC index to assign
            for ( pi = 0; (pi < 4) && (rc == NO_ERROR); pi++)
            {
               pe = &( pr->PartitionTable[pi]);
               pt = pe->PartitionType;

               if (dfsa->pt0Show == TRUE)       // allow type 0x00 with valid start/size ?
               {
                  primaryEntryOK = ((pe->BootSectorOffset != 0) && (pe->NumberOfSectors  != 0));
               }
               else                             // only no 0x00 types, but start may be 0
               {
                  //- Note: ZERO bootsector offset allowed (used in ISOHybrid MBR's)
                  primaryEntryOK = ((pt != DFS_P_EMPTY) && (pe->NumberOfSectors  != 0));
               }

               if (primaryEntryOK)
               {
                  if (dfsIsExtendedType( pe->PartitionType))
                  {
                     if (ePSN != L32_NULL)      // already set!  multiple ...
                     {
                        d->flags |= DFS_F_MULTI_EXT;
                        d->flags |= (pi << 4);
                     }
                     if ((ePSN == L32_NULL) ||  // first container is OK
                         (!TxaOption('F')))     // not FIRST prefered
                     {
                        ePSN = pe->BootSectorOffset;
                        rEnt = pi;              // refering entry
                        ebrNext = dfsAddEbrInfo( disknr, ti, pi, 0, pr);
                        dfsDiskInfo[ disknr].ebrHead = ebrNext;
                        ebrNext->UnixDevNr = pi +1;
                        sprintf( ebrNext->UnixDevicePart, "%8.8s%-2u", d->UnixDeviceDisk, pi +1);
                     }
                  }
                  else                          // valid primary
                  {
                     if ((pt == DFS_P_EFI_GPT) && (dfsa->gptAuto)) // auto add GPT ones
                     {
                        if (guardPresent)       // multiple 0xEE guard partitions
                        {
                           d->flags |= DFS_F_MULTI_GPT;
                           d->flags |= (pi << 4);
                        }
                        else
                        {
                           TRACES(("d->sectors: 0x%llx  pe->#: 0x%x\n, d->sectors, pe->NumberOfSectors"));
                           if ((d->sectors    <= ((ULN64)  0xffffffff)) &&
                               (pe->NumberOfSectors < (d->sectors - 1)) &&
                               (pt == DFS_P_EFI_GPT))
                           {
                              d->flags |= DFS_F_GPTSMGUARD;
                           }
                           guardPresent = TRUE; // analyse GPT data afterwards
                        }
                        if (gptblPresent == FALSE) // no genuine GPT tables, add MBR style anyway
                        {
                           mbrPartitionsAdded++; // count for MBR/GPT style determination
                           rc = dfsAddPartInfo( disknr, ti, pi, arcIndex++, 0, pr, NULL);
                        }
                     }
                     else if ((gptblPresent == FALSE) || (dfsa->gptAuto == FALSE)) // only when GPT unlikely/not wanted
                     {
                        mbrPartitionsAdded++;   // count for MBR/GPT style determination
                        rc = dfsAddPartInfo( disknr, ti, pi, arcIndex++, 0, pr, NULL);
                     }
                  }
               }
               else if (pt != DFS_P_EMPTY)
               {
                  d->flags |= DFS_F_INVALIDPTE;
                  d->flags |= (pi << 2);
               }
               else if (dfsa->pt0Show == FALSE) // type 'deleted' check for
               {                                // non-zero other fields
                  if ((pe->BootSectorOffset   != 0) ||
                      (pe->NumberOfSectors    != 0) ||
                      (pe->Status             != 0) ||
                      (pe->FirstSector.Head   != 0) ||
                      (pe->FirstSector.SecCyl != 0) ||
                      (pe->LastSector.Head    != 0) ||
                      (pe->LastSector.SecCyl  != 0)  )
                  {
                     d->flags |= DFS_F_PTYPE_0x00;
                     d->flags |= pi;
                  }
               }
            }
            break;

         case ST_BOOTR:
         case ST_BOOTX:
            if (dfsDetectUnixFileSystem( 0, boot, d->dlabel, d->creatr, d->fsform) == TRUE)
            {
               strcpy( d->pStyle, DPSTR_FSO);   // specific partitioning style
               d->flags |= DFS_F_FSYSTEMONLY;   // FS-Only without bootsector
            }
            else                                // not a UNIX compatible filesystem
            {
               if ((strncasecmp((char *) &(pr->Instruction[3]), "MSWIN4", 6) == 0) &&
                   (strncasecmp((char *) &(pr->os.Type[0]),     "FAT16",  5) != 0))
               {
                  strcpy( d->fsform, "FAT32");  // likely FAT32
               }
               else
               {
                  if (strncmp(pr->f2.Type, "FAT32", 5) == 0)
                  {
                     strcpy( d->fsform, "FAT32"); // probably a FAT32
                  }
                  else
                  {
                     TxCopy( d->fsform, pr->os.Type, BT_TYPE_L);
                     if (d->fsform[0] == '\0')  // no FS-type (NTFS/EFAT!)
                     {
                        TxCopy( d->fsform, pr->OpSys, BT_SYST_L);
                     }                   \
                     else if (strncmp(pr->os.Type +1, "3м", 5) == 0)
                     {
                        strcpy( d->fsform, "FAT16"); // probably DOS 3.3 FAT
                     }
                  }
               }
               if (isalpha( pr->OpSys[0]))      // could be a creator name
               {
                  TxCopy( d->creatr, pr->OpSys, BT_SYST_L +1);
               }
               if ((strncmp( d->fsform, "HPFS",  4) == 0) ||
                   (strncmp( d->fsform, "JFS",   3) == 0)  )
               {
                  TxCopy( d->dlabel, pr->os.Label, BT_LABL_L +1);
               }
               #ifndef OEMSB
               else if (strncmp( d->fsform, "FAT",   3) == 0)
               {
                  dfsFatRootLabel( pr, 0, d->dlabel);
               }
               else if (strncmp( d->fsform, "NTFS",  4) == 0)
               {
                  dfsa->winCandidates += 2;
                  dfsNtfsVolumeLabel( pr, 0, d->dlabel);
               }
               else if (strncmp( d->fsform, "EXFAT",  5) == 0)
               {
                  strcpy( d->fsform, "EFAT");   // Enhanced FAT32
                  if (pr->ex.CodeEnd == DFSEX_MACOSX_CODE_END)
                  {
                     strcpy( d->creatr, "Mac OSX");
                  }
                  else
                  {
                     strcpy( d->creatr, "Windows");
                  }
                  dfsEfatRootLabel( pr, 0, d->dlabel);
               }
               dfsCorrelateBootsector( pr, d->ddrive);
               if (isalpha( d->ddrive[0]) && TxFsIsRemovable( d->ddrive))
               {
                  d->Removable = TRUE;          // for later reporting ...
               }
               #endif
               strcpy( d->pStyle, DPSTR_FSO);   // specific partitioning style
               d->flags |= DFS_F_FSYSTEMONLY;   // MBR and FS-Only with a bootsector
            }
            rc = DFS_INV_MBR;
            break;

         case ST_MCDDM:                         // MAC-style Apple Partition Map
            strcpy( d->dlabel, "ApplePMap!"); // set descriptive fields
            strcpy( d->creatr, MACOS_CREATOR);
            strcpy( d->fsform, "MCDPM");
            strcpy( d->dxInfo, "Possible Disk-Image");
            strcpy( d->pStyle, DPSTR_APM);      // specific partitioning style

            //- Try to find any APM style partitions, and add them to the partition list
            rc = dfsReadApmPartitionInfo( disknr);
            break;

         case ST_LUKSH:
            {
               TXTM          string;
               S_LUKSH      *sd = (S_LUKSH *) boot;

               strcpy( d->dlabel, "Encrypted!"); // set descriptive fields
               strcpy( d->creatr, "Linux");
               strcpy( d->fsform, "LUKS");

               sprintf( string, "%u bit", TxBE32( sd->keyBytes) * 8);
               strcpy(  d->dxInfo, sd->cipherName);
               TxStrip( d->dxInfo, d->dxInfo, 0, ' ');
               if (strlen( d->dxInfo) < (TXMAXTT - LUKS_STR_L - 2))
               {
                  strcat(  d->dxInfo, " ");
                  strcat(  d->dxInfo, sd->cipherMode);
                  TxStrip( d->dxInfo, d->dxInfo, 0, ' ');
               }
               if (strlen( d->dxInfo) < (TXMAXTT - LUKS_STR_L - 2))
               {
                  strcat(  d->dxInfo, " ");
                  strcat(  d->dxInfo, sd->hashSpec);
                  TxStrip( d->dxInfo, d->dxInfo, 0, ' ');
               }
               if (strlen( d->dxInfo) < (TXMAXTT - 8 - 2))
               {
                  strcat(  d->dxInfo, " ");
                  strcat(  d->dxInfo, string);  // add key bit count
               }
            }
            strcpy( d->pStyle, DPSTR_CRP);      // specific partitioning style
            d->flags |= DFS_F_CRYPTO_DISK;
            rc = DFS_INV_MBR;
            break;

         default:                               // no MBR, FileSystemOnly without bootsec
            if (dfsDetectUnixFileSystem( 0, boot, d->dlabel, d->creatr, d->fsform) == TRUE)
            {
               strcpy( d->pStyle, DPSTR_FSO);   // specific partitioning style
               d->flags |= DFS_F_FSYSTEMONLY;   // FS-Only without bootsector
            }
            else
            {
               switch (st)
               {
                  case ST_FDISK:                // Fdisk cleared,   0xf6
                     strcpy( d->dlabel, "F6 Cleared!"); // set descriptive fields
                     strcpy( d->creatr, "Fdisk");
                     strcpy( d->pStyle, DPSTR_MBR); // allow partitioning, like NEWMBR
                     break;

                  case ST_EMPTY:                // completely empy, 0x00
                     strcpy( d->dlabel, "0x00 Empty!"); // set descriptive fields
                     strcpy( d->pStyle, DPSTR_MBR); // allow partitioning, like NEWMBR
                     break;

                  case ST_BADFE:                // marked with 0xFE pattern, BAD
                     strcpy( d->dlabel, "0xFE BadSec"); // set descriptive fields
                     strcpy( d->pStyle, DPSTR_MBR); // allow partitioning, like NEWMBR
                     break;

                  default:
                     if (dfsa->warnLevel > 1)
                     {
                        TxPrint( "WARNING disk nr %hu : Master boot record is not "
                                 "valid, no partitions recognized\n", disknr);
                     }
                     rc = DFS_INV_MBR;
                     break;
               }
            }
            break;
      }
      if ((ePSN != L32_NULL) && (rc == NO_ERROR))
      {
         ULONG         ebrs     = 0;            // sanity count
         ULONG         nPSN     = 0;            // next EBR PSN
         ULONG         backlink = 0;            // 1st backward link seen
         BOOL          emptyContainer = FALSE;  // previous freespace has
                                                // and empty EBR container
         xPSN = ePSN;
         do
         {
            TRACES(( "Next EBR link (ebrThis) now: %8.8x\n", ebrNext));
            ebrPrev = ebrNext;                  // connect prepared ebr info
            ebrThis = ebrNext;                  // connect prepared ebr info
            ti++;                               // indicate next table-nr
            rc = dfstReadPsn( DFSTORE, xPSN, 2, boot);
            if (rc == NO_ERROR)
            {
               if (pr->Signature != SV_BOOTR)   // no valid signature, check tables
               {
                  if (dfsValidXbrTableEntries( boot)) // but valid entries found
                  {
                     d->flags |= DFS_F_EBR_NO_SIG;
                     TxPrint( "WARNING disk nr %hu : "
                              "%s at 0x%8.8x, Cyl:%u, entry %hu points to an invalid\n"
                              "                    "
                              "EBR at 0x%8.8x, Cyl:%u, no %4hx signature present!\n",
                               disknr, (rPSN) ? "EBR" : "MBR",
                               rPSN, dfstPsn2Cyl(DFSTORE, rPSN), rEnt,
                               xPSN, dfstPsn2Cyl(DFSTORE, xPSN), SV_BOOTR);
                  }
                  else
                  {
                     TxPrint( "ERROR   disk nr %hu : "
                              "%s at 0x%8.8x, Cyl:%u, entry %hu points to an invalid\n"
                              "                    "
                              "EBR at 0x%8.8x, Cyl:%u, no %4hx sig, no valid table!\n",
                               disknr, (rPSN) ? "EBR" : "MBR",
                               rPSN, dfstPsn2Cyl(DFSTORE, rPSN), rEnt,
                               xPSN, dfstPsn2Cyl(DFSTORE, xPSN), SV_BOOTR);
                     rc = DFS_INV_BOOT;         // abort logical/EBR iteration
                  }

               }
               if (rc == NO_ERROR)
               {
                  ULONG     links = 0;          // #links to next (must be 0 or 1)
                  ULONG     parts = 0;          // #parts defined (must be 1)

                  offs = 0;                     // init next link offset
                  TRHEXS(70,&(pr->PartitionTable[0]), 0x40, "pr->PartitionTable");
                  for ( pi = 0; (pi < 4) && (rc == NO_ERROR); pi++)
                  {
                     pe = &( pr->PartitionTable[pi]);
                     pt = pe->PartitionType;

                     if ((pt                   != DFS_P_EMPTY) &&
                         (pe->BootSectorOffset != 0          ) &&
                         (pe->NumberOfSectors  != 0)) // valid ptable entry
                     {
                        if (dfsIsExtendedType( pe->PartitionType))
                        {
                           if (++links != 1)    // not first & only ?
                           {
                              TxPrint( "ERROR   disk nr %hu : EBR at 0x%8.8x, Cyl:%u "
                                       "has multiple next-links  (%u)!\n",
                                        disknr, xPSN, dfstPsn2Cyl(DFSTORE, xPSN), links);
                           }
                           else                 // first EBR link
                           {
                              offs    = pe->BootSectorOffset;
                              ebrNext = dfsAddEbrInfo( disknr, ti, pi, xPSN, pr);
                              if (ebrPrev != NULL)
                              {
                                 ebrPrev->ebrChain = ebrNext;
                              }
                              ebrPrev = ebrNext;
                              ebrNext->UnixDevNr = ti +5;
                              sprintf( ebrNext->UnixDevicePart, "%8.8s%-2u", d->UnixDeviceDisk, ti +5);

                              rPSN = xPSN;      // refering EBR (error msg)
                              rEnt = pi;        // refering entry

                              nPSN = ePSN + offs;                       //- PSN of next
                              if ((nPSN <= xPSN) && ebrThis && ebrNext) // is illegal ?
                              {
                                 if (nPSN == xPSN) // direct looping EBR
                                 {
                                    ebrThis->flag2 |= DFS_F_EXTC_LLOOP;
                                    ebrNext->flag2 |= DFS_F_EXTC_NLOOP;
                                    TxPrint( "ERROR   disk nr %hu : EBR at sector 0x%8.8x "
                                             "links to SAME sector, illegal loop!\n",
                                              disknr, xPSN);
                                    rc = DFS_BAD_STRUCTURE;
                                 }
                                 else           // backward linking!
                                 {
                                    if (nPSN == backlink) // same link, 2nd time
                                    {
                                       ebrThis->flag2 |= DFS_F_EXTC_LLOOP;
                                       ebrNext->flag2 |= DFS_F_EXTC_NLOOP;
                                       TxPrint( "ERROR   disk nr %hu : EBR at 0x%8.8x, Cyl:%u "
                                                "links BACKWARDS, illegal loop!\n",
                                                 disknr, xPSN, dfstPsn2Cyl(DFSTORE, xPSN));
                                       rc = DFS_BAD_STRUCTURE;
                                    }
                                    else
                                    {
                                       ebrThis->flag2 |= DFS_F_EXTC_BLINK;
                                       ebrNext->flag2 |= DFS_F_EXTC_NLINK;
                                       TxPrint( "WARNING disk nr %hu : EBR at 0x%8.8x, Cyl:%u "
                                                "links BACKWARDS, illegal order!\n",
                                                 disknr, xPSN, dfstPsn2Cyl(DFSTORE, xPSN));
                                       if (backlink == 0) // not set yet
                                       {
                                          backlink = nPSN; // remember first backlink
                                       }
                                    }
                                 }
                              }
                           }
                        }
                        else                    // logical volume
                        {
                           if (++parts != 1)    // not first & only ?
                           {
                              TxPrint( "ERROR   disk nr %hu : EBR at 0x%8.8x, Cyl:%u "
                                       "has multiple logicals (%u)!\n",
                                        disknr, xPSN, dfstPsn2Cyl(DFSTORE, xPSN), parts);
                           }
                           if (pe->BootSectorOffset > d->cSC) // only if significant space
                           {
                              //- Logical EBR-to-BOOTSEC area is LARGER than one cylinder!
                              dfsPartInfo[dfsNrPart].flag2 |= DFS_F_WASTECONTN;
                           }
                           if ((pr->BootMgrFlag == BT_BM_BOOT) &&
                               (pr->BootMgrName[8] == 0)) // properly terminated
                           {
                              if ((pr->BootMgrName[0] != '-' ) && // not --> LVM
                                  isprint( pr->BootMgrName[0]) && // printable ASCII
                                  isprint( pr->BootMgrName[1]) && // printable ASCII
                                  isprint( pr->BootMgrName[2]) && // printable ASCII
                                  isprint( pr->BootMgrName[3]) && // printable ASCII
                                  isprint( pr->BootMgrName[4]) && // printable ASCII
                                  isprint( pr->BootMgrName[5]) && // printable ASCII
                                  isprint( pr->BootMgrName[6]) && // printable ASCII
                                  isprint( pr->BootMgrName[7])  ) // printable ASCII
                              {
                                 TxCopy( dfsPartInfo[ dfsNrPart].blabel, pr->BootMgrName, 9);

                                 TRACES(("Assigned '%s' to blabel for (LOG) partition: %hu\n",
                                          dfsPartInfo[ dfsNrPart].blabel, dfsNrPart +1));
                              }
                              TxCopy( dfsPartInfo[ dfsNrPart].bmname, pr->BootMgrName, 9);
                           }
                           rc = dfsAddPartInfo( disknr, ti, pi, pi +1, xPSN,   pr, ebrThis);
                           if ((rc == NO_ERROR) && emptyContainer)
                           {
                              //- NrPart already incremented by AddPartInfo!
                              dfsPartInfo[dfsNrPart-1].flag2 |= DFS_F_EMPTYCONTN;
                              emptyContainer = FALSE;
                           }
                        }
                     }
                     else if (pt != DFS_P_EMPTY)
                     {
                        TxPrint( "WARNING disk nr %hu : EBR at 0x%8.8x, Cyl:%u has invalid entry nr %hu\n",
                                                   disknr, xPSN, dfstPsn2Cyl(DFSTORE, xPSN), pi);
                     }
                  }
                  if (parts == 0)               // no logical defined in EXT ?
                  {
                     emptyContainer = TRUE;     // propagate to next logical
                     TxPrint( "WARNING disk nr %hu : EBR at 0x%8.8x, Cyl:%u has NO logical partition defined!\n",
                                                disknr, xPSN, dfstPsn2Cyl(DFSTORE, xPSN));
                  }
                  if (rc == NO_ERROR)           // fine sofar
                  {
                     xPSN = nPSN;               // move to next EBR
                     if (ebrs++ > DFS_MAX_PART)
                     {
                        TxPrint( "Too many EBRs now : %u, possible loop! Last EBR at %8.8X, Cyl:%u\n",
                                                     ebrs, xPSN, dfstPsn2Cyl(DFSTORE, xPSN));
                        rc = DFS_BAD_STRUCTURE;
                     }
                  }
               }
            }
            else                                // cannot read EBR!
            {
               TxPrint( "ERROR   disk nr %hu : EBR at 0x%8.8x, Cyl:%u cannot be read! RC:%03.3u\n",
                                          disknr, xPSN, dfstPsn2Cyl(DFSTORE, xPSN), rc);
            }
         } while ((offs != 0) && (rc == NO_ERROR));
      }

      #ifndef OEMSB
      if ((gptblPresent) || (guardPresent))     // we have GPT tables, guard or EFI-system
      {
         if (mbrPartitionsAdded == 0)           // no MBR style added yet
         {
            if (guardPresent == FALSE)
            {
               d->flags |= DFS_F_GPT_NOGUARD;
            }

            //- stay in MBR-style if no guard and no automatic GPT wanted (allow MBR edits)
            if ((dfsa->gptAuto == TRUE) || (guardPresent))
            {
               strcpy( d->pStyle, DPSTR_GPT);   // specific partitioning style

               //- Analyse possible GPT header and related partition array, add as primaries
               //- When just MBR-style GPT guard or EFI-system partition is present, it has NOT been added!
               rc = dfsReadGptPartitionInfo( disknr);
            }
         }
         else
         {
            if (gptblPresent)
            {
               //- warning GPT tables present, and MBR-style partitions
               d->flags |= DFS_F_GPTMBRPART;
            }
            else
            {
               //- Try to find GPT-tables at start or end, add partitions as primaries, issue warnings
               rc = dfsReadGptPartitionInfo( disknr);
            }
         }
      }
      if ((apmapPresent) && (st != ST_MCDDM))   // hybrid, handled as MBR style!
      {
         d->flags |= DFS_F_APMMBRPART;
      }
      #endif
   }
   TxFreeMem( boot);
   RETURN (rc);
}                                               // end 'dfsReadPartInfo'
/*---------------------------------------------------------------------------*/


#ifndef OEMSB
/*****************************************************************************/
// Read GPT partition info for opened disk, validate and add GPT partitions
/*****************************************************************************/
static ULONG dfsReadGptPartitionInfo
(
   USHORT              disknr                   // IN    DFSee disk id
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   S_GPTHDR           *gptHdr = NULL;
   DFSDISKINFO        *d;                       // diskinfo for this disk
   USHORT              arcIndex = 1;            // partition index, ARC style
   USHORT              i;
   BYTE               *ep;                      // BYTE aligned entry pointer
   BOOL                priGptValid = TRUE;
   BOOL                altGptValid = TRUE;

   ENTER();

   d  = &(dfsDiskInfo[ disknr]);

   rc = dfsGptReadAlternate( d, d->sectors -1, &priGptValid, &altGptValid);
   if ((rc != NO_ERROR) && (rc != DFS_BAD_STRUCTURE))
   {
      //- GPT sectors are variable size (512..4096), but disk could be accessed with
      //- the wrong sectorsize (and the GPT header will NOT be found in that case).
      if ((gptHdr = TxAlloc( 1, dfsGetSectorSize())) != NULL) // fixed size
      {
         d->gptHeader = gptHdr;
         rc = dfstReadPsn( DFSTORE, 0, 10, rbuf); // read at least 5 KiB into rbuf
         if (rc == NO_ERROR)
         {
            //- Copy start of 2nd sector (512..4096) into the GPT header structure
            memcpy((BYTE *) gptHdr, rbuf + dfsGetSectorSize(), sizeof( S_GPTHDR));

            rc = dfsGptValidateHdr( gptHdr, (S_GPTENTRY **) &( d->gptArray));
            if ((rc != NO_ERROR) && (rc != DFS_BAD_STRUCTURE)) // ignore CRC errors
            {                                   // check for GPT header at other sector size
               if (dfsGetSectorSize() == 512)   // check at 4096 byte boundary
               {
                  memcpy((BYTE *) gptHdr, rbuf + 4096, sizeof( S_GPTHDR));
                  if ((gptHdr->Signature1 == SV_GPTHDR1) && (gptHdr->Signature2 == SV_GPTHDR2))
                  {
                     //- GPT header found at unexpected sector offset 4K, for 512 sectorsize
                     d->flags |= DFS_F_GPT_U_4096;
                  }
                  else
                  {
                     //- GPT header or the attached partition array is invalid in some way
                     d->flags |= DFS_F_GPTINVALID;
                  }
               }
               else                             // check at the 512 byte boundary
               {
                  memcpy((BYTE *) gptHdr, rbuf +  512, sizeof( S_GPTHDR));
                  if ((gptHdr->Signature1 == SV_GPTHDR1) && (gptHdr->Signature2 == SV_GPTHDR2))
                  {
                     //- GPT header found at unexpected sector offset 512, for 4K sectorsize
                     d->flags |= DFS_F_GPT_U_512;
                  }
                  else
                  {
                     //- GPT header or the attached partition array is invalid in some way
                     d->flags |= DFS_F_GPTINVALID;
                  }
               }
            }
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }

   gptHdr = (S_GPTHDR *) d->gptHeader;          // current valid header info
   if (rc == DFS_BAD_STRUCTURE)                 // CRC error, hint at rewrite
   {
      d->flags |= DFS_F_GPT_BADCRC;             // primay bad CRC, alt is BAD

      //- Do some sanity checks, and keep bad RC when too bad
      //- with an extra message telling partitions are ignored for this disk

      if ((gptHdr->ptaEntrySize < sizeof( S_GPTENTRY)) || // sanity check on HDR
          (gptHdr->ptaEntrySize > 4096)                ||
          (gptHdr->ptaEntries   < 4)                   ||
          (gptHdr->firstPSN     < 3)                   ||
          (gptHdr->ptaPSN       < 2)                   ||
          (gptHdr->ptaEntries   > 65536)               ||
          (gptHdr->hdrSize      < sizeof( S_GPTHDR))   ||
          (gptHdr->hdrSize      > dfsGetSectorSize())   )
      {
         TxPrint( "WARNING disk nr %hu : GPT header structure not usable, GPT partitions NOT shown!\n", disknr);
      }
      else
      {
         rc = NO_ERROR;
      }
   }
   if (rc == NO_ERROR)                          // add GPT partitions to DFSee
   {
      S_GPTENTRY      *ge;

      if (altGptValid == FALSE)
      {
         d->flags |= DFS_F_GPT_BADALT;
      }
      else if (priGptValid == FALSE)
      {
         d->flags |= DFS_F_GPT_BADPRI;          // signal only when alt is OK
      }

      TRACES(("gptHdr now: %8.8x\n", gptHdr));

      //- Read the GPT partition info from the array, and add to PartInfo
      for ( i = 0,                      ep  = (BYTE *) d->gptArray;
           (i < gptHdr->ptaEntries) && (rc == NO_ERROR);
            i++,                        ep += gptHdr->ptaEntrySize)
      {
         if (!TxAreaEmpty( ep, sizeof(S_GPTENTRY), 0))
         {
            ge = (S_GPTENTRY *) ep;

            TRACES(("PTA slot: %hu, ep:%8.8x   first PSN: 0x%llx\n", i, ep, ge->firstPSN));
            if ((ge->lastPSN          > gptHdr->lastPSN)            ||
                (TxAreaEmpty( ge->typeGuid, sizeof(S_GPTENTRY), 0)) ||
                (TxAreaEmpty( ge->partGuid, sizeof(S_GPTENTRY), 0))  )
            {
               TxPrint( "WARNING disk nr %hu : GPT partition array, ignored invalid entry: %hu\n", disknr, i);
            }
            else
            {
               d->gptPartitions++;
               rc = dfsAddPartInfo( disknr, GPT_STYLE, i, arcIndex++, gptHdr->ptaPSN, NULL, ep);
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsReadGptPartitionInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read APM partition info for opened disk, validate and add APM partitions
/*****************************************************************************/
static ULONG dfsReadApmPartitionInfo            // Apple Partition Map
(
   USHORT              disknr                   // IN    DFSee disk id
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSDISKINFO        *d;                       // diskinfo for this disk
   USHORT              arcIndex = 1;            // partition index, ARC style
   BYTE               *sec = NULL;              // sector buffer

   ENTER();

   d  = &(dfsDiskInfo[ disknr]);

   if ((sec = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
   {
      ULONG            last = dfstGeoSectors( DFSTORE);
      ULONG            dpms;
      ULONG            macs = 0;                // nr of MAC partitions seen
      S_MCDPM         *sd = (S_MCDPM *) sec;    // as MCDPM sector

      for (dpms = 1; (dpms <= last) && (rc == NO_ERROR); dpms++)
      {
         rc = dfstReadPsn( DFSTORE, dpms, 1, sec);
         if (rc == NO_ERROR)
         {
            if (dfsIdentifySector( dpms, 0, sec) == ST_MCDPM)
            {                                   // index in Ptable is 0 for 1st one
               rc = dfsAddPartInfo( disknr, APM_STYLE, macs, arcIndex++, 0, NULL, sec);

               macs++;
               if (macs >= txSwapUL( sd->mapSectors)) // reached nr of partitions listed
               {
                  break;
               }
            }
         }
      }
      TxFreeMem( sec);                          // free the memory
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsReadApmPartitionInfo'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Add partition info to table
/*****************************************************************************/
static ULONG dfsAddPartInfo
(
   USHORT              disknr,                  // IN    physical disk nr
   USHORT              tablenr,                 // IN    part-table nr 0..n
   USHORT              partnr,                  // IN    index in part-table
   USHORT              arcIndex,                // IN    index in ARC style
   ULN64               thisPSN,                 // IN    PSN of this MBR/EBR
   S_BOOTR            *br,                      // IN    rec with part-table
   void               *info                     // IN    additional partition info
)
{
   ULONG               rc   = 0;                // DOS rc
   S_BOOTR            *boot = (S_BOOTR *) rbuf;
   DFSPARTINFO        *p    = &(dfsPartInfo[ dfsNrPart]);
   DFSDISKINFO        *d    = &(dfsDiskInfo[ disknr]);
   DFSPARTENTRY       *ent  = NULL;
   #if !defined (OEMSB)
      BYTE             st   = ST_UDATA;         // sector type
      S_GPTENTRY      *gpe  = NULL;
      S_MCDPM         *ape  = NULL;
   #endif
   #if defined (DARWIN)
      TXTT             text;
   #endif

   ENTER();
   TRACES(( "disk:%hu table:%hu part:%hu arc:%hu PSN:0x0%llx br:%8.8x info:%8.8x BBUFSECTORS:%u\n",
             disknr,  tablenr,  partnr, arcIndex, thisPSN,   br, info, BBUFSECTORS));

   p->pitype     = DFS_PI_PARTITION;
   p->flags      = 0;
   p->flag2      = 0;
   p->disknr     = disknr;
   p->tablenr    = tablenr;
   p->partnr     = partnr;
   p->diskPart   = arcIndex;

#if !defined (OEMSB)
   if (tablenr == GPT_STYLE)                    // primary from GPT table array
   {
      gpe = (S_GPTENTRY *) info;

      p->ebrChain    = NULL;
      p->basePsn     = gpe->firstPSN;
      p->lastPsn     = gpe->lastPSN;
      p->sectors     = p->lastPsn - p->basePsn + 1;
      p->attrFlags   = gpe->attrFlags;
      p->uuidPresent = TRUE;
      memcpy( p->fsUuid,   gpe->partGuid, sizeof(DFS_GUID));
      memcpy( p->typeGuid, gpe->typeGuid, sizeof(DFS_GUID));
      memcpy( p->partGuid, gpe->partGuid, sizeof(DFS_GUID));

      if (dfsGuid2Pid( p->partGuid) != 0)
      {
         p->flag2 |= DFS_F_GUIDDUPLIC;          // part GUID not unique
      }
      TRACES(("typeGuid: %s\n", dfsFsUuidValueString( p->typeGuid)));
      TRACES(("partGuid: %s\n", dfsFsUuidValueString( p->partGuid)));

      //- For default map colors, and as default MBR-style type for MOVE/COPY
      p->partent.PartitionType = DFS_P_EFI_ANY;
      p->primary = TRUE;
   }
   else if (tablenr == APM_STYLE)               // primary from GPT table array
   {
      ape = (S_MCDPM *) info;

      p->ebrChain    = NULL;
      p->basePsn     = txSwapUL( ape->basePsn); // APM values are big-endian
      p->sectors     = txSwapUL( ape->sectors);
      p->lastPsn     = p->basePsn + p->sectors - 1;

      //- For default map colors, and as default MBR-style type for MOVE/COPY
      p->partent.PartitionType = DFS_P_MACXHFSP;
      p->primary = TRUE;
   }
   else                                         // regular MBR style partition
#endif
   {
      ent   = &(br->PartitionTable[ partnr]);

      p->ebrChain    = (DFSPARTINFO *) info;
      p->basePsn     = ent->BootSectorOffset + thisPSN;
      p->sectors     = ent->NumberOfSectors;
      if (p->sectors == DFS_MAX_PSN)            // size in table is 0xffffffff!
      {
         p->sectors  = d->sectors - p->basePsn; // correct using disksize/start
      }
      p->lastPsn     = p->basePsn + p->sectors -1;
      p->partent     = *ent;
      p->attrFlags   = 0;
      p->primary     = (thisPSN == 0) ? TRUE : FALSE;
   }

   p->geoCyls    = dfstGeoCylinders( DFSTORE);
   p->geoHeads   = dfstGeoHeads(     DFSTORE);
   p->geoSecs    = dfstGeoSectors(   DFSTORE);
   p->partPsn    = thisPSN;
   p->relSize    = dfsLogarithmicSize( p->sectors);
   p->cSC        = ((ULONG) p->geoHeads *  (ULONG) p->geoSecs);
   p->bpsector   = dfsGetSectorSize();
   p->scluster   = 0;                           // filled in from bootrec later, or defaulted
   p->drive[ 0]  = '\0';
   p->plabel[0]  = '\0';
   p->creatr[0]  = '\0';
   p->lvmPresent = FALSE;
   p->lvmSigPres = FALSE;

   TRACES(( "partinfo PID:%u at address:%8.8x, bps:%hu\n", dfsNrPart+1, p, p->bpsector));

#ifndef OEMSB
   if (tablenr == GPT_STYLE)                    // primary from GPT table array
   {
      //- Store partition name in lvm VoluName (with excess length in PartName, total 40 bytes)
      strcpy( p->lvm.VoluName, "");
      TxUnicAsciiAppend( gpe->partName, p->lvm.VoluName, GPT_PNAME_LEN * 2);

      //- descr is used to show a more verbose type description, not based on partent
      strcpy( p->descr, dfsGptGuidDescription( gpe->typeGuid));
   }
   else if (tablenr == APM_STYLE)               // primary from APM table array
   {
      //- Store partition name in lvm VoluName (with excess length in PartName, total 40 bytes)
      TxCopy( p->lvm.VoluName, ape->partName, MAC_NAME + 1);

      TxCopy( p->descr, ape->partType, TYPE_DESC_LEN + 1);
   }
   else                                         // regular MBR style partition
#endif
   {
      if ((p->primary == FALSE) && (ent->Status != 0))
      {
         p->flags |= DFS_F_LOGICALACT;          // logical marked active!
      }
      if (p->cSC != 0)                          // geometry is known
      {
         ULONG            chsc, chsh, chss;     // CHS check values
         ULONG            lbac, lbah, lbas;     // LBA check values

         chsc = DFSC2CYLIND( ent->FirstSector.SecCyl);
         chsh =              ent->FirstSector.Head;
         chss = DFSC2SECTOR( ent->FirstSector.SecCyl);
         dfstPsn2Chs( DFSTORE, p->basePsn, &lbac, &lbah, &lbas);

         TRACES(("Start LBA Cyl:%5u H:%3u S:%2u   CHS Cyl:%5u H:%3u S:%2u\n",
                            lbac, lbah, lbas,             chsc, chsh, chss));

         TRACES(("CHS check GEO Cyl:%5u H:%3u S:%3u\n", p->geoCyls, p->geoHeads, p->geoSecs));
         if (p->geoSecs <= 63)                  // NO warnings when #sectors
         {                                      // can NEVER be correct in CHS
            if ((chsc !=  lbac) ||              // exact cyl and head match
                (chsh !=  lbah) ||              // sectors limited to 6 bits
                (chss != (lbas & 0x3F)))        // to allow 64 sectors as 00
            {
               if (lbac <= 1023)
               {
                  TRACES(("bad CHS values signalled on start < 1024\n"));
                  p->flags |= DFS_F_CHSLBA_S13; // start CHS != LBA below int13
               }
               else if (!((  chsc == 1023) &&
                          (((chsh == lbah)           && (chss == lbas      )) ||
                           ((chsh == p->geoHeads -1) && (chss == p->geoSecs)))))
               {
                  TRACES(("bad dummy CHS signalled on start > 1024  (h:%u)\n", p->geoHeads));
                  p->flags |= DFS_F_CHSDUM_S13; // start CHS nonstd above int13
               }
            }
         }

         chsc = DFSC2CYLIND(ent->LastSector.SecCyl);
         chsh = ent->LastSector.Head;
         chss = DFSC2SECTOR(ent->LastSector.SecCyl);
         dfstPsn2Chs( DFSTORE, p->lastPsn, &lbac, &lbah, &lbas);

         TRACES(("End   LBA Cyl:%5u H:%3u S:%2u   CHS Cyl:%5u H:%3u S:%2u\n",
                            lbac, lbah, lbas,             chsc, chsh, chss));

         if (p->geoSecs <= 63)                  // NO warnings when #sectors
         {                                      // can NEVER be correct in CHS
            if ((chsc !=  lbac) ||              // exact cyl and head match
                (chsh !=  lbah) ||              // sectors limited to 6 bits
                (chss != (lbas & 0x3F)))        // to allow 64 sectors as 00
            {
               if (lbac <= 1023)
               {
                  TRACES(("bad CHS values signalled on end   < 1024\n"));
                  p->flags |= DFS_F_CHSLBA_E13; // end   CHS != LBA below int13
               }
               else if (!((  chsc == 1023) &&
                          (((chsh == lbah)           && (chss == lbas      )) ||
                           ((chsh == p->geoHeads -1) && (chss == p->geoSecs)))))
               {
                  TRACES(("bad dummy CHS signalled on end   > 1024  (h:%u)\n", p->geoHeads));
                  p->flags |= DFS_F_CHSDUM_E13; // end   CHS nonstd above int13
               }
            }
         }

         if ((p->primary == FALSE) ||           // logical partition, or
             (p->basePsn < p->cSC)  )           // primary in first cylinder
         {
            if (dfsOnTrackBoundary(p->basePsn, p->geoHeads, p->geoSecs) != 1)
            {
               p->flags |= DFS_F_PLOG_NOT_1;    // start not on Cyl boundary
            }
         }
         else                                   // other primaries
         {
            if (dfsOnTrackBoundary(p->basePsn, p->geoHeads, p->geoSecs) != 0)
            {
               p->flags |= DFS_F_PRIM_NOT_0;    // start not on Cyl boundary
            }
         }
         if (dfsOnTrackBoundary(p->lastPsn +1, p->geoHeads, p->geoSecs) != 0)
         {
            p->flags |= DFS_F_PART_NOT_C;       // end   not on Cyl boundary
         }
      }
      dfsPartTypeDescription(ent->PartitionType, p->descr);
   }

   rc = dfstReadPsn( DFSTORE, p->basePsn, BBUFSECTORS, rbuf); // enough for BMGR/JFS-super
   if (rc != NO_ERROR)
   {
      TxPrint( "ERROR   disk nr %hu : read %u sectors at PSN 0x0%llx, "
               "retry 1 sector ...\n", disknr, BBUFSECTORS, p->basePsn);
      //- read just the single sector to minimize read-errors and still allow most analysis
      rc = dfstReadPsn( DFSTORE, p->basePsn, 1, rbuf);
   }
   if (rc == NO_ERROR)
   {
      TRACES(( "bootsec from Psn:%llx into buffer at %llx\n", p->basePsn, boot));
      p->checksum = dfsCheckSum((BYTE *) rbuf);
      strcpy( p->creatr, "unknown");

#if !defined (OEMSB)
      if ((st = dfsIdentifySector( 0, 0, rbuf)) == ST_LUKSH) // LUKS encrypted partition
      {
         TXTM          string;
         S_LUKSH      *sd = (S_LUKSH *) rbuf;

         strcpy( p->plabel, "Encrypted!");      // set descriptive fields
         strcpy( p->creatr, "Linux");
         strcpy( p->fsform, "LUKS");

         sprintf( string, "{%36.36s}", sd->uuidString);
         dfsUidStringToBinary( string, p->fsUuid, TRUE);
         p->uuidPresent = TRUE;

         TxCopy(  p->lvm.VoluName, sd->cipherName, LVM_NAME_L);
         TxStrip( p->lvm.VoluName, p->lvm.VoluName, 0, ' ');
         strcat(  p->lvm.VoluName, " ");
         TxCopy(  string, sd->cipherMode, LVM_NAME_L - strlen( p->lvm.VoluName));
         strcat(  p->lvm.VoluName, string);

         TxCopy(  p->lvm.PartName, sd->hashSpec, LVM_NAME_L);
         TxStrip( p->lvm.PartName, p->lvm.PartName, 0, ' ');
         strcat(  p->lvm.PartName, " ");
         sprintf( string, "%u bit", TxBE32( sd->keyBytes) * 8);
         strcat(  p->lvm.PartName, string);
      }
      #if defined (NEVER)
      else if (st == ST_CORES)                  // macOS FileVault encrypted partition
      {

         TXTM          string;
         S_CORE_ST    *sd = (S_CORE_ST *) rbuf;

         strcpy( p->plabel, "Encrypted!");      // set descriptive fields
         strcpy( p->creatr, MACOS_CREATOR);
         strcpy( p->fsform, "Fvault");

         //- to be refined, print more details, set label as LVM-name (like GPT)
      }
      #endif
      else if (tablenr == APM_STYLE)            // primary from APM table array
      {
         strcpy( p->creatr, MACOS_CREATOR);
         if (strstr( ape->partType, "HFS") != NULL)
         {
            p->partent.PartitionType = DFS_P_MACXHFSP;
            strcpy( p->fsform, "HFS");
         }                                      // to be refined, may want to recognize more
      }
      else if (tablenr == GPT_STYLE)            // primary from GPT table array
      {
         S_GPTENTRY    *gpte = NULL;
         S_GPTENTRY    *gptArray  = (S_GPTENTRY *) d->gptArray;
         S_GPTHDR      *gptHeader = (S_GPTHDR   *) d->gptHeader;

         if ((gptHeader) && (gptArray) && (partnr < gptHeader->ptaEntries))
         {
            gpte = &(gptArray[ partnr]);
         }

         TRHEXS(70, boot, 0x80, "Boot sector start");

         if (boot->Signature == SV_BOOTR)       // GPT style with valid bootsector
         {
            TRACES(("GPT analysis, bootsector contents\n"));

            //- First try to determine based on bootsector contents only
            if      ((strncasecmp( boot->f2.Type, "FAT32", 5) != 0) && // FAT16
                     (strncasecmp( boot->os.Type, "FAT",   3) == 0)  )
            {
               TxCopy( p->creatr, boot->OpSys,    BT_SYST_L +1);
               TxCopy( p->plabel, boot->os.Label, BT_LABL_L +1);
               TxCopy( p->fsform, boot->os.Type,  BT_TYPE_L +1);
               dfsa->winCandidates += 2;
               dfsFatRootLabel( boot, p->basePsn, p->plabel);
               p->partent.PartitionType = DFS_P_FAT16;
            }
            else if ((strncasecmp( boot->f2.Type, "FAT32", 5) == 0) && // FAT32
                     (strncasecmp( boot->os.Type, "FAT",   3) != 0)  )
            {
               if (strncasecmp( boot->OpSys, "-FVE-FS-", 7) == 0) // BitLocker encrypted
               {
                  strcpy( p->plabel, "Encrypted!"); // set descriptive fields
                  strcpy( p->creatr, "Windows");
                  strcpy( p->fsform, "BITLOCKR");
               }
               else
               {
                  p->scluster = (USHORT) boot->eb.ClustSize;
                  TxCopy( p->creatr, boot->OpSys,    BT_SYST_L +1);
                  TxCopy( p->plabel, boot->f2.Label, BT_LABL_L +1);
                  TxCopy( p->fsform, boot->f2.Type,  BT_TYPE_L +1);
                  dfsa->winCandidates += 2;
                  dfsFatRootLabel( boot, p->basePsn, p->plabel);
                  p->partent.PartitionType = DFS_P_FAT32;
               }
            }
            else
            {
               if (((boot->eb.SectSize % SECTORSIZE) == 0) && // 512 multiple
                   ((boot->eb.SectSize / SECTORSIZE) <= 8)  ) // upto 4096
               {
                  p->bpsector = boot->eb.SectSize;
               }
               if (strncasecmp( boot->OpSys, "NTFS", 4) == 0)
               {
                  p->scluster = (USHORT) boot->eb.ClustSize;
                  TxCopy( p->fsform, boot->OpSys, BT_SYST_L +1);
                  strcpy( p->creatr, "Win NT");
                  dfsa->winCandidates += 2;
                  #ifndef OEMSB
                     dfsNtfsVolumeLabel( boot, p->basePsn, p->plabel);
                  #endif
               }
               else if (strncasecmp( boot->OpSys, "EXFAT", 5) == 0)
               {
                  strcpy( p->fsform, "EFAT");  // Enhanced FAT32
                  p->scluster = (USHORT) (1 << boot->ex.SpcShift);
                  p->bpsector = (USHORT) (1 << boot->ex.BpsShift);
                  if (boot->ex.CodeEnd == DFSEX_MACOSX_CODE_END)
                  {
                     strcpy( p->creatr, MACOS_CREATOR);
                  }
                  else
                  {
                     strcpy( p->creatr, "Windows");
                  }
                  dfsEfatRootLabel( boot, p->basePsn, p->plabel);
               }
               else if (strncasecmp(p->fsform, "JFS", 3) == 0)
               {
                  dfsJfsFsIdentify( boot, p); // Add UUID value, if any ...
               }
               p->partent.PartitionType = DFS_P_INST_FS;
            }
            if (!isprint(p->fsform[0]))
            {
               strcpy( p->fsform, "unknown");
            }
            if (!isprint(p->creatr[0]))
            {
               strcpy( p->creatr, "unknown");
            }

            if (gpte != NULL)                   // we have a GPT entry
            {
               if      (GptGuidMatch( gpte->typeGuid, dfsGuid_EFI_SYSTEM))
               {
                  p->partent.PartitionType = DFS_P_EFI_ESP;
               }
               else if ((GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_DATA))        ||
                        (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_ROOT_X86_32)) ||
                        (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_ROOT_X86_64))  )
               {
                  dfsLinuxNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_LINUXNATV;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_SWAP))
               {
                  dfsLinuxSwapInfo( boot, p);
                  p->partent.PartitionType = DFS_P_SWAPSOLAR;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_WINDOWS_LDM_DATA))
               {
                  p->partent.PartitionType = DFS_P_WIN2XPLDM;
               }
               else
               {
                  //- to be refined, other known ones with a bootsector ?
               }
            }
         }
         else                                   // no bootsector
         {
            TRACES(("GPT analysis, on GUID and FS-identification function using gpte:%p\n", gpte));

            if (gpte != NULL)                   // we have a GPT entry
            {
               if      (GptGuidMatch( gpte->typeGuid, dfsGuid_EFI_SYSTEM))
               {
                  //- Should be FAT32, so force that, even without a bootsector
                  strcpy( p->creatr, "GPT/EFI");
                  strcpy( p->fsform, "FAT32");
                  p->partent.PartitionType = DFS_P_EFI_ESP;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_OS2_ARCAOS_TYPE_1))
               {
                  //- Should be an embedded 'MBR-disk' container for ArcaOS on GPT
                  strcpy( p->creatr, "ArcaOS");
                  strcpy( p->fsform, "GPVD");
                  p->partent.PartitionType = DFS_P_AOS2GPVD;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_WINDOWS_BASIC_DATA))
               {
                  if (dfsDetectUnixFileSystem( p->basePsn, (BYTE *) boot, p->plabel, p->creatr, p->fsform) == TRUE)
                  {
                     if (strcmp( p->creatr, MACOS_CREATOR) == 0)
                     {
                        p->partent.PartitionType = DFS_P_MACXHFSP;
                     }
                     else                       // Linux native, older distros use Windows type!
                     {
                        p->partent.PartitionType = DFS_P_LINUXNATV;
                     }
                  }
                  else                          // regular case, a Windows IFS
                  {
                     strcpy( p->creatr, "Windows");
                     p->partent.PartitionType = DFS_P_INST_FS;
                  }
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_WINDOWS_RECOVERY))
               {
                  strcpy( p->creatr, "Windows");
                  p->partent.PartitionType = DFS_P_IFSHIDDEN;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_MICROSOFT_RESERVED))
               {
                  strcpy( p->creatr, "Windows");
                  p->partent.PartitionType = DFS_P_FAT32X;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_WINDOWS_LDM_DATA))
               {
                  strcpy( p->creatr, "Windows");
                  p->partent.PartitionType = DFS_P_WIN2XPLDM;
               }
               else if ((GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_DATA))        ||
                        (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_ROOT_X86_32)) ||
                        (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_ROOT_X86_64))  )
               {
                  dfsLinuxNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_LINUXNATV;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_SWAP))
               {
                  dfsLinuxSwapInfo( boot, p);
                  p->partent.PartitionType = DFS_P_SWAPSOLAR;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_RAID))
               {
                  dfsLinuxNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_LINUXRAID;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_LINUX_LVM))
               {
                  dfsLinuxNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_LINUX_LVM;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_MAC_OS_X_HFS_PLUS  ))
               {
                  dfsMacOsNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_MACXHFSP;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_APPLE_CORE_STORAGE ))
               {
                  dfsMacOsNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_MACXDATA;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_APFS_CONTAINER ))
               {
                  if (dfsMacOsNativeInfo( boot, p) == FALSE)
                  {
                     strcpy( p->fsform, "APFS"); // default 'format' for container
                  }
                  //- force 4 KiB blocksize for APFS (better Brec display)
                  p->scluster = (p->bpsector) ? (4096 / p->bpsector) : 8;
                  p->partent.PartitionType = DFS_P_APFS_CONT;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_APPLE_BOOT         ))
               {
                  dfsMacOsNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_MACXBOOT;
               }
               else if (GptGuidMatch( gpte->typeGuid, dfsGuid_SOLARIS_BOOT       ))
               {
                  dfsMacOsNativeInfo( boot, p);
                  p->partent.PartitionType = DFS_P_SOLARBOOT;
               }
               else
               {
                  TRACES(("No GUID match found\n"));
               }
            }
         }
         TRACES(("Nr of WIN candidates: %hu\n", dfsa->winCandidates));
      }
      else                                      // MBR style partitioning
#endif

      {
         if (boot->Signature == SV_BOOTR)       // with valid bootsector
         {
            BYTE   NonHiddenType = ent->PartitionType;

            TRACES(("bootsec has VALID signature\n"));

            if (dfsPartTypeHidable(NonHiddenType)) // only allowed on hidable!
            {                                   // otherwise values like 35
               NonHiddenType &= ~DFS_P_PHIDDEN; // for LVM will be masked out
            }

            TRHEXS(70, boot, 0x80, "Boot sector start");
            TxCopy( p->fsform, boot->os.Type, BT_TYPE_L +1);
            TRACES(("NhType: %2.2hx FS:%s Hiddens:%8.8x Offset:%8.8x\n",
                NonHiddenType, p->fsform, boot->eb.HiddenSectors, ent->BootSectorOffset));

            if (dfsPartTypeHidable(NonHiddenType)  || // regular types like FAT/HPFS
                ((NonHiddenType == DFS_P_WARP_LVM) && // or LVM, not being JFS
                  (strncasecmp(p->fsform, "JFS", 3) != 0))) // JFS has fixed values ...
            {
               if ((boot->eb.HiddenSectors != 0) || // and not partly empty (GRUB)
                   (boot->eb.LogGeoHead    != 0) ||
                   (boot->eb.LogGeoSect    != 0)  )
               {
                  if (boot->eb.HiddenSectors != ent->BootSectorOffset)
                  {
                     p->flag2 |= DFS_F_HIDDENSBAD; // HiddenSectors incorrect
                  }
                  if ((boot->eb.LogGeoHead != p->geoHeads) ||
                      (boot->eb.LogGeoSect != p->geoSecs )  )
                  {
                     p->flag2 |= DFS_F_BOOTGEOBAD; // PBR geometry incorrect
                  }
               }
            }
            if (TxMemStr( boot, "I13X", 500) != NULL) // 'I13X' referenced in boot ?
            {
               if ((strncmp( boot->os.Type, "HPFS", 4) != 0) || // unless DFSee patched HPFS
                           ( boot->os.RestCode[ HPFS_I13X_CHECK] != DFS_B_OPCNOP))
               {
                  p->flag2 |= DFS_F_I13XNEEDED; // Requires 'I13X' set by MBR
               }
            }
            switch (NonHiddenType)
            {
               case DFS_P_BOOTMGR:
                  if (memcmp( boot->OpSys, "APJ&WN", 6) == 0) // IBM BM present ?
                  {
                     if ((boot->eb.FatOffset   <  DFSFDSK_BMGRSEC) ||
                         (boot->eb.RootEntries != 0)) // Not W2KBM protected ?
                     {
                        p->flag2 |= DFS_F_BMNOTW2KBM; // Warning: not protected
                     }
                     #if !defined (OEMSB)
                     if (dfsFixChsI13Changed( NULL, FALSE, p->basePsn, &boot->os.BmBootSecCyl, &boot->os.BmBootHead))
                     {
                        p->flag2 |= DFS_F_BMBADCHS; // Warning: bad BOOT CHS
                     }
                     if (dfsFixChsI13Changed( NULL, FALSE, p->basePsn +1, &boot->os.BmDataSecCyl, &boot->os.BmDataHead))
                     {
                        p->flag2 |= DFS_F_BMBADCHS; // Warning: bad DATA CHS
                     }
                     #endif

                     if (p->flag2 & DFS_F_I13XNEEDED) // I13X aware BootMgr ?
                     {
                        if (boot->OpSys[6] >= DFSFDSK_BMGRLVM)
                        {
                           strcpy( p->creatr, "LVM");
                        }
                        else
                        {
                           strcpy( p->creatr, "FDISK");
                        }
                        if (boot->os.RestCode[ BMGR_I13X_INBOOT] != DFS_B_OPCJMPS)
                        {
                           strcpy( p->plabel, "I13Xneeded");
                        }
                        else
                        {
                           strcpy( p->plabel, "I13Xoption");
                        }
                     }
                     else                       // must be classic BootMgr
                     {
                        strcpy( p->creatr, "FDISK");
                        strcpy( p->plabel, "MaxCyl:1023");
                     }
                  }
                  else
                  {
                     strcpy( p->creatr, "unknown");
                     strcpy( p->plabel, "NotPresent!");
                  }
                  strcpy( p->fsform, "BMGR");   // 'fs' is alias for FDISK
                  break;

               case DFS_P_WARP_LVM:             // assume JFS, uses spc
                  dfsa->OS2FsPresent = TRUE;
                  p->scluster = (USHORT) boot->eb.ClustSize;
                  TRACES(("Assigning %hu to scluster for LVM 2\n", p->scluster));
                  if (isalpha( boot->OpSys[0]))
                  {
                     TxCopy( p->creatr, boot->OpSys, BT_SYST_L +1);
                  }
                  TxCopy( p->plabel, boot->os.Label, BT_LABL_L +1);
                  #ifndef OEMSB
                     dfsJfsFsIdentify( boot, p);   // Add UUID value, if any ...
                  #endif
                  break;

               case DFS_P_MACXHFSP:             // valid bootsector (unlikely)
                  dfsMacOsNativeInfo( boot, p);
                  break;

               case DFS_P_LINUXNATV:            // valid bootsector, Lilo/Grub
               case DFS_P_LIN_NONFS:
               case DFS_P_LINUXRAID:
               case DFS_P_LINUX_LVM:
                  dfsLinuxNativeInfo( boot, p);
                  break;

               case DFS_P_SWAPSOLAR:            // Linux SWAP or Solaris
                  dfsLinuxSwapInfo( boot, p);
                  break;

               case DFS_P_FAT32:
               case DFS_P_FAT32X:
               case DFS_P_ACRONISSZ:
                  if ((strncmp( boot->f2.Type, "FAT32", 5) != 0) && //- FAT16 in
                      (strncmp( boot->os.Type, "FAT",   3) == 0)  ) //- F32 type
                  {
                     p->flag2 |= DFS_F_F16INF32TP; // FAT16 in F32 type
                     TxCopy( p->creatr, boot->OpSys,    BT_SYST_L +1);
                     TxCopy( p->plabel, boot->os.Label, BT_LABL_L +1);
                  }
                  else
                  {
                     p->scluster = (USHORT) boot->eb.ClustSize;
                     TxCopy( p->creatr, boot->OpSys,    BT_SYST_L +1);
                     TxCopy( p->plabel, boot->f2.Label, BT_LABL_L +1);
                     TxCopy( p->fsform, boot->f2.Type,  BT_TYPE_L +1);
                     dfsa->winCandidates++;
                     if (p->primary)
                     {
                        dfsa->winCandidates++;
                     }
                     TRACES(("Nr of WIN candidates: %hu\n", dfsa->winCandidates));
                  }
                  #ifndef OEMSB
                     dfsFatRootLabel( boot, p->basePsn, p->plabel);
                  #endif
                  break;

               case DFS_P_EFI_GPT:              // EFI spec, GUID (64 bit Intel)
                  strcpy( p->creatr, "GPT/EFI ");
                  strcpy( p->fsform, "Prot-MBR");
                  p->flags |= DFS_F_EFIGPTDISK; // Warning: GPT disk
                  break;

               case DFS_P_FREE_BSD:
               case DFS_P_OPEN_BSD:
                  strcpy( p->creatr, "fdisk");
                  strcpy( p->fsform, "BSD-FS");
                  strcpy( p->plabel, "");
                  break;

               case DFS_P_BEOS_FS:
                  strcpy( p->creatr, "BeOS");
                  strcpy( p->fsform, "BeFS");
                  TxCopy( p->plabel, ((char *) (rbuf)) + dfsGetSectorSize(), BT_LABL_L +1);
                  break;

               case DFS_P_WIN2XPLDM:            // Windows 2000 XP-LDM
                  p->flags |= DFS_F_WINLDMDISK; // Warning: Windows Dynamic disk
               default:
                  TxCopy( p->creatr, boot->OpSys,    BT_SYST_L +1);
                  if ((strncmp( boot->f2.Type, "FAT32", 5) == 0) && //- FAT32 in
                      (strncmp( boot->os.Type, "FAT",   3) != 0)  ) //- F16 type
                  {
                     switch (NonHiddenType)
                     {
                        case DFS_P_FAT12:
                        case DFS_P_FAT16:      //- FAT32 in F16 type
                        case DFS_P_BIGDOS:
                        case DFS_P_BIGDOS_VS:
                        case DFS_P_BIGDOSCVS:
                           p->flag2 |= DFS_F_F32INF16TP; // issue warning
                           break;

                        default:               //- types like laptop recovery, 0x12
                           break;              //- FAT32 is OK for these ...
                     }
                     p->scluster = (USHORT) boot->eb.ClustSize;
                     TxCopy( p->plabel, boot->f2.Label, BT_LABL_L +1);
                     TxCopy( p->fsform, boot->f2.Type,  BT_TYPE_L +1);
                     #ifndef OEMSB
                        dfsFatRootLabel( boot, p->basePsn, p->plabel);
                     #endif
                  }
                  else                          // regular types
                  {
                     TxCopy( p->plabel, boot->os.Label, BT_LABL_L +1);
                  }
                  break;
            }
            switch (NonHiddenType)
            {
               case DFS_P_FAT12:
               case DFS_P_FAT16:
               case DFS_P_BIGDOS:
               case DFS_P_BIGDOS_VS:
               case DFS_P_BIGDOSCVS:
                  if (((boot->eb.SectSize % SECTORSIZE) == 0) && // 512 multiple
                      ((boot->eb.SectSize / SECTORSIZE) <= 8)  ) // upto 4096
                  {
                     p->bpsector =          boot->eb.SectSize;
                     p->scluster = (USHORT) boot->eb.ClustSize;
                     #ifndef OEMSB
                        dfsFatRootLabel( boot, p->basePsn, p->plabel);
                     #endif
                     if (p->primary)
                     {
                        dfsa->winCandidates++;
                        TRACES(("Nr of WIN candidates: %hu\n", dfsa->winCandidates));
                     }
                  }
                  break;

               case DFS_P_BOOTMGR:
                  break;

               case DFS_P_WARP_LVM:             // assume JFS, uses spc
                  p->scluster = (USHORT) boot->eb.ClustSize;
                  TRACES(("Assigning %hu to scluster for LVM 2\n", p->scluster));
                  break;

               case DFS_P_WIN2XPLDM:            // Dynamic disk, usually NTFS or FAT32
               case DFS_P_RECOVERY:             // Recovery, could be FAT or NTFS
               case DFS_P_IFSHIDDEN:            // hidden IFS, Vista pre-install
               case DFS_P_IFS_MIRR:             // mirrored HPFS / NTFS volset
               case DFS_P_IFS_DISAB:            // disabled/corrupted HPFS/NTFS
               case DFS_P_INST_FS:              // HPFS, NTFS. JFS etc
                  if (strncasecmp((char *) &(boot->Instruction[2]), "LILO", 4) == 0)
                  {
                     strcpy( p->creatr, "LILO");
                     strcpy( p->fsform, "LinuxNatv");
                     strcpy( p->plabel, "");
                     break;
                  }
                  else if (strncasecmp( boot->f2.Type, "FAT32", 5) == 0)
                  {
                     if (strncasecmp( boot->OpSys, "-FVE-FS-", 7) == 0) // BitLocker encrypted
                     {
                        strcpy( p->plabel, "Encrypted!"); // set descriptive fields
                        strcpy( p->creatr, "Windows");
                        strcpy( p->fsform, "BITLOCKR");
                     }
                     else
                     {
                        p->scluster = (USHORT) boot->eb.ClustSize;
                        TxCopy( p->creatr, boot->OpSys,    BT_SYST_L +1);
                        TxCopy( p->plabel, boot->f2.Label, BT_LABL_L +1);
                        TxCopy( p->fsform, boot->f2.Type,  BT_TYPE_L +1);
                     }
                     dfsa->winCandidates++;
                     if (p->primary)
                     {
                        dfsa->winCandidates++;
                     }
                     TRACES(("Nr of WIN candidates: %hu\n", dfsa->winCandidates));
                     break;
                  }
                  else
                  {
                     if (((boot->eb.SectSize % SECTORSIZE) == 0) && // 512 multiple
                         ((boot->eb.SectSize / SECTORSIZE) <= 8)  ) // upto 4096
                     {
                        p->bpsector = boot->eb.SectSize;
                     }
                     if (strncasecmp(p->creatr, "NTFS", 4) == 0)
                     {
                        p->scluster = (USHORT) boot->eb.ClustSize;
                        strcpy(p->fsform, p->creatr);
                        strcpy(p->creatr, "Win NT");
                        dfsa->winCandidates++;
                        if (p->primary)
                        {
                           dfsa->winCandidates++;
                        }
                        TRACES(("Nr of WIN candidates: %hu\n", dfsa->winCandidates));
                        #ifndef OEMSB
                           dfsNtfsVolumeLabel( boot, p->basePsn, p->plabel);
                        #endif
                        break;                  // note: rbuf destroyed here!
                     }
                     else if (strncasecmp(p->creatr, "EXFAT", 5) == 0)
                     {
                        p->scluster = (USHORT) (1 << boot->ex.SpcShift);
                        p->bpsector = (USHORT) (1 << boot->ex.BpsShift);
                        strcpy( p->fsform, "EFAT"); // Enhanced FAT32
                        if (boot->ex.CodeEnd == DFSEX_MACOSX_CODE_END)
                        {
                           strcpy( p->creatr, "Mac OSX");
                        }
                        else
                        {
                           strcpy( p->creatr, "Windows");
                        }
                        #ifndef OEMSB
                           dfsEfatRootLabel( boot, p->basePsn, p->plabel);
                        #endif
                        break;
                     }
                     else if (strncasecmp(p->fsform, "JFS", 3) == 0)
                     {
                        dfsa->OS2FsPresent = TRUE;
                        #ifndef OEMSB
                           dfsJfsFsIdentify( boot, p); // Add UUID value, if any ...
                        #endif
                        break;
                     }
                     else if (strncasecmp(p->fsform, "HPFS", 4) == 0)
                     {
                        dfsa->OS2FsPresent = TRUE;
                     }
                     //- non JFS/NTFS ones fall-through here! might be unknown
                  }
               default:                         // keep bps/spc defaults
                                                // last-resort, try UNIX supported FS
                  dfsDetectUnixFileSystem( p->basePsn, (BYTE *) boot, p->plabel, p->creatr, p->fsform);

                  if (!isprint(p->fsform[0]))   // remove garbage strings from BR
                  {
                     strcpy( p->fsform, "unknown");
                  }
                  if (!isprint(p->creatr[0]))
                  {
                     strcpy( p->creatr, "unknown");
                  }
                  break;
            }
         }
         else                                   // MBR style, no valid Bootrec
         {
            TRACES(("bootsec has INVALID signature\n"));
            switch (ent->PartitionType)
            {
               case DFS_P_HIBERNATE:
                  strcpy( p->creatr, "system--");
                  strcpy( p->fsform, "MEM-dump");
                  break;

               case DFS_P_WIN2XPLDM:            // Windows 2000 XP-LDM
                  strcpy( p->creatr, "Win-LDM ");
                  strcpy( p->fsform, "Dyn-Disk");
                  p->flags |= DFS_F_WINLDMDISK; // Warning: Windows Dynamic disk
                  break;

               case DFS_P_EFI_GPT:              // EFI spec, GUID (64 bit Intel)
                  strcpy( p->creatr, "GPT/EFI ");
                  strcpy( p->fsform, "Prot-MBR");
                  p->flags |= DFS_F_EFIGPTDISK; // Warning: GPT disk
                  break;

               case DFS_P_APFS_CONT:            // APFS-container, MBR style
               case DFS_P_MACXHFSP:             // macOS HFS+, MBR style
                  dfsMacOsNativeInfo( boot, p); // without valid bootsector
                  break;

               case DFS_P_LINUXNATV:            // without valid bootsector
               case DFS_P_LIN_NONFS:
               case DFS_P_LINUXRAID:
               case DFS_P_LINUX_LVM:
                  dfsLinuxNativeInfo( boot, p);
                  break;

               case DFS_P_SWAPSOLAR:            // Linux SWAP or Solaris
                  dfsLinuxSwapInfo( boot, p);
                  break;

               default:
                  #ifndef OEMSB
                  if (dfsBootsecIsDumpFS( (BYTE *) boot, p->plabel))
                  {
                     strcpy( p->creatr, "OS/2");
                     strcpy( p->fsform, "DUMPFS");
                  }
                  else
                  if (dfsBootAreaIsJFS(( BYTE *) boot, BBUFSECTORS, p->plabel))
                  {
                     strcpy( p->creatr, "Linux");
                     strcpy( p->fsform, "JFS");
                  }
                  else
                  #endif
                  {
                     strcpy( p->creatr, "unknown");
                     strcpy( p->fsform, "-");
                     p->flags |= DFS_F_BOOTRECBAD;
                  }
                  break;
            }
         }
         if ((ent->Status) && (p->primary))     // bootable primary
         {
            if ((dfsPartTypeHidable(ent->PartitionType)) && (ent->PartitionType & DFS_P_PHIDDEN))
            {
               p->flags |= DFS_F_ACT_HIDDEN;    // hidden bootable
            }
            if (disknr > 1)
            {
               p->flags |= DFS_F_ACT_NOT1ST;    // active on disk > 1
            }
         }
      }
   }
   else                                         // cannot read bootsector!
   {
      TxPrint( "ERROR   disk nr %hu : Bootsector at 0x0%llx cannot be read", disknr, p->basePsn);
      #if defined (WIN32)
         TxPrint( ": %s\n", txNtLastError());
      #else
         TxPrint( "! RC:%3.3u\n", rc);
      #endif
      sprintf( p->plabel,  "-ERROR:%3.3u-", rc);
      strcpy(  p->creatr, "-SECTOR-");
      strcpy(  p->fsform, "-BAD-");
      p->flags |= DFS_F_BOOTRECBAD;

      rc = NO_ERROR;                            // continue with rest!
   }
   if (p->tablenr == GPT_STYLE)
   {
      //- to be refined! Probably valid only when sorted (on start PSN ?)
      p->UnixDevNr = (USHORT) (p->partnr +1);
   }
   else
   {
      p->UnixDevNr = (p->primary) ? (USHORT) (p->partnr +1) : (USHORT) (p->tablenr +4);
   }
   #if defined (DARWIN)
      TxCopy(  text, dfsDiskInfo[disknr].UnixDeviceDisk + 5, TXMAXTT);
      TxStrip( text, text, ' ', ' ');
      sprintf( p->UnixDevicePart, "%ss%hu", text,  p->UnixDevNr);
   #else
      sprintf( p->UnixDevicePart, "%8.8s%-2hu", dfsDiskInfo[disknr].UnixDeviceDisk, p->UnixDevNr);
   #endif

   if ((p->pbrSect = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      memcpy( p->pbrSect, boot, dfsGetSectorSize()); // independant copy
   }
   TRARGS(("#%d %s\n", dfsNrPart, p->descr));
   #ifndef OEMSB
   TRINIT(30);
   {
      dfsX10("   StartLSN:", p->basePsn, CBG, "");
      dfsSz64("  size:",     p->sectors, "\n");
   }
   TREXIT();
   #endif
   if (++dfsNrPart == DFS_MAX_PART)             // next will overflow
   {
      dfstSwitch2ReadOnly( "Internal tables reached maximum number of partitions", TRUE, TRUE);
      rc = DFS_OVERFLOW;
   }
   if (p->bpsector == 0)
   {
      p->bpsector = SECTORSIZE;                 // sane value, avoid div by zero ...
   }
   if (p->scluster == 0)
   {
      p->scluster = 1;                          // sane value, avoid div by zero ...
   }
   TxStrPrintable( p->plabel);                  // make sure there are no strange characters
   TxStrPrintable( p->creatr);                  // that would cause table display corruption
   TxStrPrintable( p->fsform);

   TRACES(("scluster value now %hu\n", p->scluster));
   RETURN (rc);
}                                               // end 'dfsAddPartInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Detect presence of a Unix filesystem when no bootsector present
/*****************************************************************************/
BOOL dfsDetectUnixFileSystem                    // RET   Filesystem detected
(
   ULN64               basePsn,                 // IN    Base PSN possible FS
   BYTE               *fsect,                   // IN    FS sectors (64KiB)
   char               *label,                   // OUT   label string   or NULL
   char               *creatr,                  // OUT   creator string or NULL
   char               *fsform                   // OUT   FileSystem name string
)
{
   BOOL                rc = FALSE;              // function return
   DFSPARTINFO         partInfo;

   ENTER();
   TRACES(("basePsn: 0x%llx\n", basePsn));

   if (fsect != NULL)
   {
      //- Try to recognize a (Linux/MacOS) filesystem when no bootrec
      memset( &partInfo, 0, sizeof( DFSPARTINFO));
      partInfo.basePsn = basePsn;               // needed by identify functions!

      //- Favor 'macOS' as creatr/related info when running on macOS (both can have HFS)
      #if defined (DARWIN)
         if ((dfsMacOsNativeInfo( (S_BOOTR *) fsect, &partInfo) == TRUE) ||
             (dfsLinuxNativeInfo( (S_BOOTR *) fsect, &partInfo) == TRUE)  )
      #else
         if ((dfsLinuxNativeInfo( (S_BOOTR *) fsect, &partInfo) == TRUE) ||
             (dfsMacOsNativeInfo( (S_BOOTR *) fsect, &partInfo) == TRUE)  )
      #endif
      {
         if (label)
         {
            strcpy( label, partInfo.plabel);
         }
         if (creatr)
         {
            strcpy( creatr, partInfo.creatr);
         }
         strcpy( fsform, partInfo.fsform);
         rc = TRUE;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsDetectUnixFileSystem'
/*---------------------------------------------------------------------------*/



#define MAC_XML_LEAD_STR "lvg.name</key><string>"
/*****************************************************************************/
// Set FS-format and related creatr + label description of MAC (0xAF) type
/*****************************************************************************/
BOOL dfsMacOsNativeInfo                         // macOS filesystem found
(
   S_BOOTR            *boot,                    // IN    bootsector (BBUFBYTES)
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   BOOL                rc = TRUE;

   ENTER();

   TRACES(( "boot:%8.8x  p:%8.8x\n", boot, p));
   strcpy( p->creatr, MACOS_CREATOR);
   #ifndef OEMSB
   if ((dfsHfsFsIdentify(  boot, p) == DFS_FS_UNKNOWN) && // not a FileSystemOnly disk ?
       (dfsApfsFsIdentify( boot, p) == DFS_FS_UNKNOWN)  ) // either HFS+ or APFS ...
   {
      char            *xml;
      TXTM             name;

      if ((xml = TxMemStr( boot, MAC_XML_LEAD_STR, BBUFBYTES)) != NULL)
      {
         TxCopy( name, xml + strlen( MAC_XML_LEAD_STR), TXMAXTM);
         if ((xml = (strchr( name, '<'))) != NULL)
         {
            *xml = 0;                           // terminate name at next <
         }

         if (strlen( p->lvm.VoluName) == 0)     // no LVM/GPT volumename yet
         {
            TxCopy(  p->lvm.VoluName, name, LVM_NAME_L);
            strcpy(  p->plabel, "Encrypted!");  // indicator for encryption
         }
         else                                   // use label column for (truncated) name
         {
            TxCopy(  p->plabel, name, BT_LABL_L +1);
         }
         strcpy( p->fsform, "FlVault");
      }
      else
      {
         strcpy( p->fsform, "unknown");
         rc = FALSE;
      }
   }
   #endif
   TRACES(( "creatr:'%s' fsform: '%s' label: '%s'\n", p->creatr, p->fsform, p->plabel));
   BRETURN( rc);
}                                               // end 'dfsMacOsNativeInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set FS-format and related creatr + label description of Linux (0x83) type
/*****************************************************************************/
BOOL dfsLinuxNativeInfo                         // RET   Linux filesystem found
(
   S_BOOTR            *boot,                    // IN    bootsector
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   BOOL                rc = TRUE;

   ENTER();

   TRACES(( "boot:%8.8x  p:%8.8x\n", boot, p));
   if      (TxMemStr( boot, "LILO", SECTORSIZE) != NULL)
   {
      strcpy( p->creatr, "LILO ");
   }
   else if (TxMemStr( boot, "GRUB", SECTORSIZE) != NULL)
   {
      strcpy( p->creatr, "GRUB ");
   }
   else
   {
      strcpy( p->creatr, "Linux");
   }
   #ifndef OEMSB
   if (dfsExtFsIdentify( boot, p) == DFS_FS_UNKNOWN)
   {
      if (dfsRsrFsIdentify( boot, p) == DFS_FS_UNKNOWN)
      {
         if (dfsJfsFsIdentify( boot, p) == DFS_FS_UNKNOWN)
         {
            if (dfsXfsFsIdentify( boot, p) == DFS_FS_UNKNOWN)
            {
               if (dfsHfsFsIdentify( boot, p) == DFS_FS_UNKNOWN)
               {
                  strcpy( p->fsform, "unknown");
                  rc = FALSE;
               }
            }
         }
      }
   }
   #endif
   TRACES(( "creatr:'%s' fsform: '%s' label: '%s'\n", p->creatr, p->fsform, p->plabel));
   BRETURN( rc);
}                                               // end 'dfsLinuxNativeInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set FS-format and related (creatr) description of Linux SWAP (0x82) type
/*****************************************************************************/
static void dfsLinuxSwapInfo
(
   S_BOOTR            *boot,                    // IN    bootsector
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   DFSWAPBLOCK1 *bb = (DFSWAPBLOCK1 *) boot;    // SWAP boot block

   ENTER();

   if      (TxMemStr( boot, "LILO", SECTORSIZE) != NULL)
   {
      strcpy( p->creatr, "LILO ");
   }
   else if (TxMemStr( boot, "GRUB", SECTORSIZE) != NULL)
   {
      strcpy( p->creatr, "GRUB ");
   }
   else
   {
      strcpy( p->creatr, "Linux");
   }
   if (bb->swSignature[9] == '2')
   {
      switch (bb->swVersion)
      {
         case 0:  strcat( p->creatr, "V0"); break;
         case 1:  strcat( p->creatr, "V1"); break;
         default: strcat( p->creatr, "V?"); break;
      }
   }
   else
   {
      strcat( p->creatr, "T1");
   }
   if (!memcmp( bb->swSignature, "SWAP", 4))
   {
      TxCopy( p->plabel, (char *) bb->swSignature, 11);
   }
   else
   {
      strcpy( p->plabel, "");                   // unknown/unused
   }
   strcpy( p->fsform, "SWAP");

   TRACES(( "creatr:'%s' fsform: '%s'\n", p->creatr, p->fsform));
   VRETURN ();
}                                               // end 'dfsLinuxSwapInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add EBR partition info to table
/*****************************************************************************/
static DFSPARTINFO *dfsAddEbrInfo
(
   USHORT              disknr,                  // IN    physical disk nr
   USHORT              tablenr,                 // IN    part-table nr 0..n
   USHORT              partnr,                  // IN    index in part-table
   ULONG               thisPSN,                 // IN    PSN of this boot-rec
   S_BOOTR            *br                       // IN    rec with part-table
)
{
   DFSPARTINFO        *e;
   DFSPARTENTRY       *ent = &(br->PartitionTable[ partnr]);
   DFSDISKINFO        *d;                       // diskinfo for this disk

   ENTER();
   TRARGS(("#%d type %2.2x\n", dfsNrPart, ent->PartitionType));
   TRARGS(("disknr:%hu  tablenr:%hu  partnr:%hu  BrPsn:%8.8x  br:%8.8x\n",
            disknr, tablenr, partnr, thisPSN, br));
   TRACES(("ent*:%8.8x  ->type:%hu  ->offset:%8.8x  ->sect:%8.8x\n", ent,
            ent->PartitionType, ent->BootSectorOffset, ent->NumberOfSectors));

   if ((e = dfsNewEbrInfo(NULL)) != NULL)       // create new element
   {
      d = &(dfsDiskInfo[ disknr]);

      e->disknr     = disknr;
      e->tablenr    = tablenr;
      e->partnr     = partnr;
      e->geoCyls    = d->geoCyls;
      e->geoHeads   = d->geoHeads;
      e->geoSecs    = d->geoSecs;
      e->primary    = (thisPSN == 0) ? TRUE : FALSE;
      e->partent    = *ent;
      e->partPsn    = thisPSN;
      e->basePsn    = ent->BootSectorOffset;
      if (tablenr  != 0)                        // not head of chain
      {
         e->basePsn += dfsEbrBase();            // add EBR base offset
      }
      e->lastPsn    = e->basePsn + ent->NumberOfSectors -1;
      e->sectors    = ent->NumberOfSectors;
      e->cSC        = d->cSC;
      e->lvmPresent = FALSE;
      e->lvmSigPres = FALSE;
      e->bpsector   = d->bpsector;

      if ((ent->PartitionType == DFS_P_BIGEXTEND) ||
          (ent->PartitionType == DFS_P_BIGLINEXT) ||
          (ent->PartitionType == DFS_P_LINUXEXTX)  )
      {
         e->flags |= DFS_F_BIGEXTENDF;          // non-std extended type used
      }
      if (ent->Status != 0)
      {
         e->flags |= DFS_F_EXTEND_ACT;          // extended marked active!
      }                                         // May cause MBR BOOT failure!

#ifndef OEMSB
      //- Add description of the bootcode in the EBR, if any (creator/Related)
      dfsGetBootcodeDescription( e->basePsn, e->creatr);
#endif

      if (e->cSC != 0)
      {
         ULONG            chsc, chsh, chss;     // CHS check values
         ULONG            lbac, lbah, lbas;     // LBA check values

         chsc = DFSC2CYLIND(ent->FirstSector.SecCyl);
         chsh = ent->FirstSector.Head;
         chss = DFSC2SECTOR(ent->FirstSector.SecCyl);
         dfstPsn2Chs( DFSTORE, e->basePsn, &lbac, &lbah, &lbas);

         TRACES(("Start LBA Cyl:%5u H:%3u S:%2u   CHS Cyl:%5u H:%3u S:%2u\n",
                            lbac, lbah, lbas,             chsc, chsh, chss));

         TRACES(("CHS check GEO Cyl:%5u H:%3u S:%3u\n", e->geoCyls, e->geoHeads, e->geoSecs));
         if (e->geoSecs <= 63)                  // NO warnings when #sectors
         {                                      // can NEVER be correct in CHS
            if ((chsc !=  lbac) ||              // exact cyl and head match
                (chsh !=  lbah) ||              // sectors limited to 6 bits
                (chss != (lbas & 0x3F)))        // to allow 64 sectors as 00
            {
               if (lbac <= 1023)                // below int-13, should match
               {
                  TRACES(("bad CHS values signalled on start < 1024\n"));
                  e->flags |= DFS_F_CHSLBA_S13; // start CHS != LBA below int13
               }
               else if (!((  chsc == 1023) &&
                          (((chsh == lbah)           && (chss == lbas      )) ||
                           ((chsh == e->geoHeads -1) && (chss == e->geoSecs)))))
               {
                  TRACES(("bad dummy CHS signalled on start > 1024  (h:%u)\n", e->geoHeads));
                  e->flags |= DFS_F_CHSDUM_S13; // start CHS nonstd above int13
               }
            }
         }

         chsc = DFSC2CYLIND(ent->LastSector.SecCyl);
         chsh = ent->LastSector.Head;
         chss = DFSC2SECTOR(ent->LastSector.SecCyl);
         dfstPsn2Chs( DFSTORE, e->lastPsn, &lbac, &lbah, &lbas);

         TRACES(("End   LBA Cyl:%5u H:%3u S:%2u   CHS Cyl:%5u H:%3u S:%2u\n",
                            lbac, lbah, lbas,             chsc, chsh, chss));

         if (e->geoSecs <= 63)                  // NO warnings when #sectors
         {                                      // can NEVER be correct in CHS
            if ((chsc !=  lbac) ||              // exact cyl and head match
                (chsh !=  lbah) ||              // sectors limited to 6 bits
                (chss != (lbas & 0x3F)))        // to allow 64 sectors as 00
            {
               if (lbac <= 1023)                // below int-13, should match
               {
                  TRACES(("bad CHS values signalled on end   < 1024\n"));
                  e->flags |= DFS_F_CHSLBA_E13; // end   CHS != LBA below int13
               }
               else if (!((  chsc == 1023) &&
                          (((chsh == lbah)           && (chss == lbas      )) ||
                           ((chsh == e->geoHeads -1) && (chss == e->geoSecs)))))
               {
                  TRACES(("bad dummy CHS signalled on end   > 1024  (h:%u)\n", e->geoHeads));
                  e->flags |= DFS_F_CHSDUM_E13; // end   CHS nonstd above int13
               }
            }
         }
         if (dfsOnTrackBoundary(e->basePsn, e->geoHeads, e->geoSecs) != 0)
         {
            e->flags |= DFS_F_EXTC_NOT_0;       // start not on Cyl boundary
         }
         if (dfsOnTrackBoundary(e->lastPsn +1, e->geoHeads, e->geoSecs) != 0)
         {
            e->flags |= DFS_F_EXTC_NOT_C;       // end   not on Cyl boundary
         }
      }
      dfsPartTypeDescription(ent->PartitionType, e->descr);


      TRARGS(("#%d type %2.2x %s\n", dfsNrPart, ent->PartitionType, e->descr));
      #ifndef OEMSB
      TRINIT(30);
      {
         dfsX10("   StartLSN:", e->basePsn, CBG, "");
         dfsSz64("  size:",     e->sectors, "\n");
      }
      TREXIT();
      #endif
   }
   RETURN ( e);
}                                               // end 'dfsAddEbrInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate one new EBR info structure (used from AddEbr and fdskCreate)
/*****************************************************************************/
DFSPARTINFO *dfsNewEbrInfo
(
   DFSPARTINFO        *template                 // IN    template structure
)
{
   DFSPARTINFO        *e;

   ENTER();

   if ((e = TxAlloc( 1, sizeof(DFSPARTINFO))) != NULL)
   {
      if (template)
      {
         *e = *template;                        // make shallow copy ...
      }
      e->pitype = DFS_PI_EBR_CHAIN;             // make sure type is correct
      TRACES(("New EBR info structure allocated at: %8.8x\n", e));
   }
   RETURN ( e);
}                                               // end 'dfsNewEbrInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get DFSee partition-ID for specified LVM-partition-Serial-number (lvmsnp)
/*****************************************************************************/
USHORT dfsLvmSnP2PartId                         // RET   partition-ID or 0
(
   USHORT              disknr,                  // IN    disk number
   ULONG               lvmsnp                   // IN    LVM partition S#
)
{
   USHORT              rc = 0;                  // function return
   DFSPARTINFO        *p;                       // partition info
   USHORT              pid   = 1;               // partition ID

   ENTER();

   TRACES(( "lvmsnp: %8.8x, partitions: %hu\n", lvmsnp, dfsNrPart));

   for (pid = 1; pid <= dfsNrPart; pid++)
   {
      p = &(dfsPartInfo[pid -1]);
      if ((p) && (p->disknr == disknr) && p->lvmPresent)
      {
         if (p->lvm.PartitId == lvmsnp)
         {
            rc = pid;
            break;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsLvmSnP2PartId'
/*---------------------------------------------------------------------------*/


#ifndef OEMSB
/*****************************************************************************/
// Select partition number for Logical reading (aka 'open logical')
/*****************************************************************************/
ULONG dfsSelectPartition
(
   USHORT              index                    // IN    partition id 1..n
)
{
   ULONG               rc = 0;                  // DOS rc
   USHORT              pi = index-1;

   ENTER();

   if (dfsDiskInfo == NULL)
   {
      dfsReadDiskInfo( FDSK_QUIET);             // read all partition info
   }
   if ((pi < dfsNrPart) && (pi != DFS_P_NULL))  // valid partition number ?
   {
      TRACES(("scluster: %hu  bpsector: %hu\n", dfsPartInfo[ pi].scluster, dfsPartInfo[ pi].bpsector));

      rc = dfstOpenDisk(  DFSTORE, dfsPartInfo[ pi].disknr, FALSE, FALSE);
      dfstLogicalLimits(  DFSTORE, dfsPartInfo[ pi].basePsn,
                                   dfsPartInfo[ pi].lastPsn);
      dfstSetSectorSize(  DFSTORE, dfsPartInfo[ pi].bpsector);
      if (dfsPartInfo[ pi].scluster)
      {
         dfstSetClusterSize( DFSTORE, dfsPartInfo[ pi].scluster);
      }
      else if (dfsPartInfo[ pi].bpsector)
      {
         dfstSetClusterSize( DFSTORE, (4096 / dfsPartInfo[ pi].bpsector));
      }
   }
   else
   {
      rc = DFS_NO_DEVICE;                       // return error
   }
   RETURN (rc);
}                                               // end 'dfsSelectPartition'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate drive-letter to partition-index, usable in Select
/*****************************************************************************/
USHORT dfsDrive2PartId                          // RET   RET partition-ID or 0
(
   char               *drive                    // IN    drive letter for part
)
{
   USHORT              rc = 0;                  // return value
   USHORT              pi;
   char                driveletter = drive[0];
   DFSPARTINFO        *p;                       // partition info

   ENTER();

   if (dfsDiskInfo == NULL)
   {
      dfsReadDiskInfo( FDSK_QUIET);             // read all partition info
   }
   for (pi = dfsNrPart; pi > 0; pi--)           // search from last
   {
      p = &(dfsPartInfo[ pi -1]);               // PID starts from 1, array from 0
      TRACES(( "Candidate:'%s' for letter:'%c'\n", p->drive, *drive));
      if ((p->drive[0] == toupper(driveletter)) ||
          (p->drive[1] == tolower(driveletter))  )
      {
         rc = pi;                               // found a partition match
         if ((p->flags & DFS_F_BOOTRECBAD) == 0) // valid bootrec ?
         {
            break;                              // exit when valid
         }                                      // otherwise keep searching
      }
   }
   RETURN (rc);
}                                               // end 'dfsDrive2PartId'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Linux partition number (NN in /dev/xdaNN) to DFSee PID
/*****************************************************************************/
USHORT dfsDevNr2PartId                          // RET   RET partition-ID or 0
(
   USHORT              disknr,                  // IN    disk number
   USHORT              devNr                    // IN    Linux device number
)
{
   USHORT              rc = 0;                  // return value
   USHORT              pid;
   DFSPARTINFO        *p;                       // partition info

   ENTER();

   if (dfsDiskInfo == NULL)
   {
      dfsReadDiskInfo( FDSK_QUIET);             // read all partition info
   }
   for (pid = 1; pid <= dfsNrPart; pid++)
   {
      p = &(dfsPartInfo[pid -1]);
      TRACES(( "Candidate:'%s' for devNr:%hu\n", p->UnixDevicePart, devNr));
      if ((p->UnixDevNr == devNr) &&
          (p->disknr    == disknr) )
      {
         rc = pid;                              // found a partition match
         break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsDevNr2PartId'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get DFSee partition-ID for specified disk plus PSN value (within partition)
// Note: For logical partitions, the EBR track is considered to be 'inside'
//       For primary partitions, the MBR track belongs to the FIRST partition
/*****************************************************************************/
USHORT dfsDiskPsn2PartId                        // RET   partition-ID or 0
(
   USHORT              disknr,                  // IN    disk number
   ULN64               psn                      // IN    PSN value in partition
)
{
   USHORT              rc = 0;                  // function return
   DFSPARTINFO        *p;                       // partition info
   USHORT              pid   = 1;               // partition ID
   ULN64               first;

   for (pid = 1; pid <= dfsNrPart; pid++)
   {
      p = &(dfsPartInfo[pid -1]);
      if (p->disknr == disknr)
      {
         first = (p->primary) ? p->basePsn : p->partPsn;
         if ((psn >= first) && (psn <= p->lastPsn))
         {
            rc = pid;
            break;
         }
      }
   }
   return (rc);
}                                               // end 'dfsDiskPsn2PartId'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show partition-info for selection depending on detail/format level
/*****************************************************************************/
ULONG dfsShowPartitions                         // RET   nr of partitions
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   USHORT              index;                   // partition index
   DFSPARTINFO        *p;                       // partition info

   ENTER();
   TRACES(("disknr: %4.4hx\n", disknr));
   if (dfsDiskInfo == NULL)
   {
      dfsReadDiskInfo( FDSK_ANY);               // read all partition info
   }
   if ((dfsa->verbosity >= TXAO_VERBOSE) || TxaOption('c') || TxaOption('a'))
   {
      for (index = 0; (index < dfsNrPart) && !TxAbort(); index++)
      {
         p = &dfsPartInfo[ index];
         if ((disknr == FDSK_ANY) || (disknr == p->disknr))
         {
            TXTM          cmd;

            if (TxaOption(dfsa->verbosity == TXAO_VERBOSE)) // simple verbose mode
            {
               dfsVerboseShowPart( &dfsPartInfo[ index]);
            }
            else                                // extended verbose modes
            {
               sprintf( cmd, "part %u", index +1);
               if (dfsa->verbosity <= TXAO_QUIET) // less verbose selection
               {
                  strcat( cmd, " -O:q");        // shortened part-detail
               }
               if (TxaOptSet('a'))              // don't show alloc twice!
               {                                // and supress for -a-
                  strcat( cmd, " -a-");
               }
               strcat( cmd, " #0#d");           // default bootsect + next
               if (dfsa->verbosity >= TXAO_MAXIMUM) // verbose + contents
               {
                  if (strncasecmp( p->fsform, "HPFS", 4) == 0)
                  {
                     strcat( cmd, "#d#0x11#d"); // add ROOT, spare & codepage
                  }
                  else if (strncasecmp( p->fsform, "FAT", 3) != 0)
                  {
                     strcat( cmd, "#d#d");      // add another two
                  }
                  else
                  {
                     strcat( cmd, "#d");        // just one more ...
                  }
               }
               if (TxaOption('c'))              // check filesystem
               {
                  strcat( cmd, "#check");
               }
               if (TxaOption('a'))              // show allocation
               {
                  strcat( cmd, "#alloc");
               }
               rc = dfsMultiCommand( cmd, 0, TRUE, FALSE, TRUE); // exec with echo
            }
         }
      }
   }
   else
   {
      dfsShowWarnings( disknr);                 // show selected warnings

      if (!TxaOptUnSet('t'))                    // unless table suppressed
      {
         if (TxaOption('s'))                    // (comma) separated verbose
         {
            dfsShowPartSepList( disknr);
         }
         else if (TxaOption(TXA_O_XML))         // XML style verbose info
         {
            dfsShowPartXmlList( disknr);
         }
         else if (TxaOption('C'))               // (space) separated Compact
         {
            dfsShowPartQueryList( disknr);
         }
         else                                   // tight table
         {
            dfsShowPartTable( disknr);          // show tight table
         }
      }
   }
   RETURN(dfsNrPart);
}                                               // end 'dfsShowPartitions'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show disk and partition warnings using -w option and masking for a disk
/*****************************************************************************/
void dfsShowWarnings
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
)
{
   USHORT              index;                   // partition index
   DFSPARTINFO        *p;                       // partition info
   DFSDISKINFO        *d;                       // disk info
   DFSPARTINFO        *e;                       // related EBR info
   ULONG               mask = DFS_F_M_MAJOR;    // warning mask filter, part
   ULONG               masd = DFS_FDM_MAJOR;    // warning mask filter, disk
   TXA_OPTION         *opt;                     // option pointer
   #if defined (LINUX)
      static BOOL      LinuxNoteGiven = FALSE;
   #endif

   ENTER();

   if ((opt = TxaOptValue('w')) != NULL)        // specific warning option
   {
      mask = DFS_F_M_SHOWALL;                   // default ALL part warnings
      masd = DFS_F_M_SHOWALL;                   // default ALL disk warnings
      switch (opt->type)                        // warning catagory selection
      {
         case TXA_STRING:
            switch (opt->value.string[0])
            {
               case 't':                        // minor non-trivial warnings
                  mask = DFS_F_M_MINOR & ~DFS_F_M_TRIVIAL;
                  masd = DFS_FDM_MINOR & ~DFS_FDM_TRIVIAL;
                  break;

               case 'm':                        // all minor warnings
                  mask = DFS_F_M_MINOR;
                  masd = DFS_FDM_MINOR;
                  break;

               case 'M': mask = DFS_F_M_MAJOR; break; // major warnings
               case 'a': mask = DFS_F_M_ALIGN; break; // cylinder alignment
               case 'b': mask = DFS_F_M_IBMGR; break; // IBM BootManager related
               case 'c': mask = DFS_F_M_CHSER; break; // CHS related
               case 'f': mask = DFS_F_M_FORMT; break; // not-formatted
               case 's': mask = DFS_F_M_OVERL; break; // size overlap related
               case 'x': mask = DFS_F_M_EXT0F; break; // ext-container type 0x0f
               default:                        break; // ALL warnings
            }
            break;

         case TXA_NUMBER:                       // specific mask specified
            mask = opt->value.number;
            break;

         default:
            break;
      }
   }
   if (mask != 0)
   {
      ULONG            suppressedMinors = 0;

      for (index = 1; index <= dfsDisks; index++)
      {
         if ((disknr == FDSK_ANY) || (disknr == index))
         {
            TRACES(("Warnings for disk: %hu\n", index));
            if ((d = dfsGetDiskInfo(index)) != NULL)
            {
               dfsShowDiskWarnings( d, masd);   // disk level warnings
            }
         }
      }
      #if defined (LINUX)
         if (!LinuxNoteGiven)
         {
            for (index = 0; index < dfsNrPart; index++)
            {
               p = &dfsPartInfo[ index];
               if ((disknr == FDSK_ANY) || (disknr == p->disknr))
               {
                  if ((p->flags & (DFS_F_M_ALIGN | DFS_F_M_CHSER)) != 0) // CHS/alignment problems ?
                  {
                     TxPrint( "\nNote: \n"
                       "There are some (minor) CHS/ALIGNMENT issues! (use 'part -w -t-' for details)\n"
                       "CHS and 'cyl boundary' warnings on Linux may be caused by an incorrect\n"
                       "disk-geometry being used. Newer kernels tend to use a 16-head geometry,\n"
                       "while most other OS's use 255 (or 240) heads. If so, a command like:\n\n"
                       "  'geo ? 255'  may result in less warnings being displayed\n\n");
                     break;
                  }
               }
            }
            LinuxNoteGiven = TRUE;
         }
      #endif
      dfsShowPartWarnings( NULL, 0, &suppressedMinors); // signal start of sequence
      for (index = 0; index < dfsNrPart; index++)
      {
         p = &dfsPartInfo[ index];
         if ((disknr == FDSK_ANY) || (disknr == p->disknr))
         {
            TRACES(("Warnings for part %hu PID: %hu at %8.8x\n", index, p->id, p));
            if ((e = p->ebrChain) != NULL)      // related ebr info
            {
               TRACES(("ebrChain at %8.8x\n", e));
               dfsShowPartWarnings( e, mask, &suppressedMinors);
            }
            dfsShowPartWarnings( p, mask, &suppressedMinors);
         }
      }
      if ((suppressedMinors != 0) &&            // some minors supressed
          (mask == DFS_F_M_MAJOR) )             // shown are only MAJOR
      {
         TxPrint( "\nAt least %u %s minor warning%s not shown. Use 'part -w%s -t-' for details.\n",
             suppressedMinors,                          (dfsa->warnLevel == 0) ? "non-trivial" : "trivial only",
            (suppressedMinors > 1) ? "s were" : " was", (dfsa->warnLevel == 0) ? ":t" : "");
      }
   }
   VRETURN();
}                                               // end 'dfsShowWarnings'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show (comma) separated list with partition-info
//
// Note:   used in SVISTA/eCS/ArcaOS OEM Engine interface, parsed by position
//         Positions/format has changed for 64bit values in version 14.5!
//         Header must start with '=' in 1st position
//         Each line must start with a digit in 1st position
// -g      EXE switch now allows ouput of JUST the partition lines to STDOUT
// -x[:fs] Adds extra columns for FS indo (resizing and status) for eCS OEM
/*****************************************************************************/
static void dfsShowPartSepList
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
)
{
   USHORT              index;                   // partition index
   USHORT              disk;                    // disk number
   USHORT              partype;
   char                active;                  // boot/start active indicator
   TXTS                adescr;                  // active descriptor
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSPARTINFO        *e;                       // ptr to extended part info
   DFSPARTINFO        *f;                       // related freespace area
   char               *line = NULL;             // full line of output
   TXTT                name;                    // one word text buffer
   TXTT                text;                    // one word text buffer
   TXTT                plvm;                    // one word text buffer
   TXTT                h;                       // separator string in header
   USHORT              startDisk  = (disknr != FDSK_ANY) ? disknr : 1;
   USHORT              finalDisk  = (disknr != FDSK_ANY) ? disknr : dfsDisks;
   char               *x = TxaOptStr( 's', NULL, ",");
   TXTS                dsp;                     // doublespaced line-end
   BOOL                uCols = TxaOption('u');  // extra UUID column at end
   BOOL                xCols = TxaOption('x');  // extra FS columns wanted
   char               *xFsys = TxaOptStr('x', NULL, "*"); // default all FS
   DFST_HANDLE         cst   = dfstGetDefaultStore();
   char                eActive;                 // ACTIVE indicator for the extended

   ENTER();

   strcpy( dsp, (TxaOption('S')) ? "\n\n" : "\n");
   memset(   h,  ' ', TXMAXTS);
   h[ min( strlen(x), TXMAXTS -1)] = 0;         // terminate the string
   if (TxaOption('e'))
   {
      strcpy( adescr, "/ Ptable ");
   }
   else
   {
      strcpy( adescr, "-descr.  ");
   }
   TxPrint( "\nId%sPD%sActive%s%sDr%sType%s0x%sDescription%sDeviceName%sBI%sFormat  %sRelated %s"
              "Label info %sGPT or LVM part-name%sLVM volume name     %sBM-name %sBegin sector%s Begin Hex%s  End sector %s  End Hex%s"
              "Begin-cyl%s  End-cyl%sSector count%s Sect# Hex %s  Size MiB",
               h, h, h, adescr, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h);
   if (uCols)
   {
      TxPrint( "%sUUID, when supported by filesystem", h);
   }
   if (xCols)
   {
      TxPrint( "%sMin-Size%sMax-Size%s    Free%sStatus", h, h, h, h);
   }
   TxPrint(   "%s", dsp);
   TxPrint(   "==%s==%s=-=============%s==%s====%s==%s===========%s==========%s==%s========%s"
              "========%s===========%s====================%s====================%s========%s============%s==========%s============%s==========%s"
              "=========%s=========%s============%s==========%s===========",
               h, h, h,         h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h, h);
   if (uCols)
   {
      TxPrint( "%s======================================", h);
   }
   if (xCols)
   {
      TxPrint( "%s========%s========%s========%s======", h, h, h, h);
      dfstSetDefaultStore( DFST_SYSTEM);        // select system store
   }
   TxPrint(   "%s", dsp);

   line = TxAlloc( 1, TXMAX1K);                 // too large for the stack
   for (disk = startDisk; (disk <= finalDisk) && (line != NULL); disk++)
   {
      f = NULL;                                 // no freespace displayed yet
      d = &(dfsDiskInfo[disk]);

      for (index = 0; index < dfsNrPart; index++)
      {
         p = &(dfsPartInfo[index]);
         if (p->disknr == disk)                 // partition on current disk
         {
            BOOL       brbad = (BOOL) (p->flags & DFS_F_BOOTRECBAD);

            if (p->fspChain != NULL)
            {
               f = p->fspChain;                 // last shown freespace area
               dfsShowFreeSpaceSep( f, x, dsp, line);
            }
            strcpy( adescr, "");
            partype = p->partent.PartitionType;
            if (p->partent.Status && p->primary)
            {
               active = (char)(brbad) ? (char) '!' : (char) '>';
               strcpy( adescr, "Active (BIOS)");
            }
            else
            {
               if (strlen(p->blabel) > 0)       // bootable by bootmgr
               {
                  active = (char)(brbad) ? (char) '-' : (char) '*';
                  strcpy( adescr, "BMGRstartable");
               }
               else
               {
                  active = ' ';                 // default no indicator
               }
            }
            if (d->Removable)
            {
               if (active == ' ') active = 'r';
               else               active = 'R';
            }
            if (((e = p->ebrChain)  != NULL) && // there is an extended container
                 (e->partent.Status != 0)     ) // and it is marked ACTIVE
            {
               if (e->tablenr       == 0)       // It is the 1st one (MBR)
               {
                  eActive = '>';
               }
               else                             // other EBR marked ACTIVE, weird
               {
                  eActive = '!';
               }
            }
            else
            {
               eActive = ' ';
            }
            if ((TxaOption('e')) && (e != NULL))
            {
               strcpy( text, "");
               strcpy( plvm, "");

               sprintf( line, "%2.2hu%s%2hu%s%c %s Ptable   %s  %s    %s"
                       "%2.2x%s%-*s%s%-10s%s%s%s%-8.8s%s%-8.8s%s%-11.11s%s%-20.20s%s"
                       "%-20.20s%s%-8.8s%s%12llu%s%10llx%s%12llu%s%10llx%s"
                       "%9u%s%9u%s%12llu%s%10llx%s%11.1lf",
                                 e->id,      x, e->disknr,       x,
                       eActive, (e->primary) ? "MBR" : "EBR",    x,
                                             x,                  x,
                                 e->partent.PartitionType,       x,
                                 #if defined (OEM_BRANDED)
                                    9,
                                 #else
                                   11,
                                 #endif
                                 e->descr,                       x,
                                 e->UnixDevicePart,              x,
                                 "  ",                           x,
                                 "",         x, e->creatr,       x,
                                 "",         x, plvm,            x,
                                 text,       x, "",              x,
                                 e->basePsn, x, e->basePsn,      x,
                                 e->lastPsn, x, e->lastPsn,      x,
                        (ULONG) (e->basePsn / e->cSC),  x,
                        (ULONG) (e->lastPsn / e->cSC),  x,
                                 e->sectors, x, e->sectors,      x,
                         TXSMIB( e->sectors, e->bpsector));
               strcat( line, dsp);
               TxPrint( "%s", line);
               if (TxaExeSwitch('g'))
               {
                  printf( "%s", line);
                  fflush( stdout);
               }
            }
            if (!TxaOptUnSet('p'))              // unless partitions blocked
            {
               if ((p->lvmPresent)  ||                //- real LVM, use volume, partition name
                   (strcmp(p->fsform, "LUKS") == 0))  //- LUKS, use cipher and hash info
               {
                  strncpy( text, p->lvm.VoluName, LVM_NAME_L);
                  text[LVM_NAME_L] = '\0';
                  strncpy( plvm, p->lvm.PartName, LVM_NAME_L);
                  plvm[LVM_NAME_L] = '\0';
               }
               else if (p->tablenr == GPT_STYLE) // GPT, name is stored in lvm info
               {
                  strncpy( plvm, p->lvm.VoluName, LVM_NAME_L); // first 20 characters
                  plvm[LVM_NAME_L] = '\0';
                  strncpy( text, p->lvm.PartName, LVM_NAME_L); // second part of name
                  text[LVM_NAME_L] = '\0';
               }
               else
               {
                  strcpy( text, NO_LVM_GPT_LUKS);
                  strcpy( plvm, text);
               }
               if ((p->tablenr == GPT_STYLE) ||  // use raw description as type
                   (p->tablenr == APM_STYLE)  )
               {
                  sprintf( name, "%-17.17s%s%s", p->descr, x, x);
               }
               else                             // MBR style
               {
                  sprintf( name, "%s%s%2.2x%s%-*s",
                          ((dfsPartTypeHidable((BYTE) partype)) &&
                                     (partype & DFS_P_PHIDDEN))  ?
                                     (p->primary) ? "Hide" : "Lhid" :
                                     (p->primary) ? "Prim" : "Log ",  x,
                                      partype,    x,
                                      #if defined (OEM_BRANDED)
                                                9,
                                      #else
                                                11,
                                      #endif
                                      p->descr);
               }

               sprintf( line, "%2.2hu%s%2hu%s%c %-13.13s%s%-2.2s%s%s"
                       "%s%-10s%s%2hu%s%-8.8s%s%-8.8s%s%-11.11s%s%-20.20s%s"
                       "%-20.20s%s%-8.8s%s%12llu%s%10llx%s%12llu%s%10llx%s"
                       "%9u%s%9u%s%12llu%s%10llx%s%11.1lf",
                                 p->id,      x, p->disknr,       x,
                         active, adescr,     x, p->drive,        x,
                                 name,                           x,
                                 p->UnixDevicePart,              x,
                                 p->diskPart,                    x,
                                 p->fsform,  x, p->creatr,       x,
                                 p->plabel,  x, plvm,            x,
                                 text,       x, p->bmname,       x,
                                 p->basePsn, x, p->basePsn,      x,
                                 p->lastPsn, x, p->lastPsn,      x,
                        (ULONG) (p->basePsn / p->cSC),  x,
                        (ULONG) (p->lastPsn / p->cSC),  x,
                                 p->sectors, x, p->sectors,      x,
                         TXSMIB( p->sectors, p->bpsector));
               if (uCols)
               {
                  if (p->uuidPresent)
                  {
                     strcpy( text, x);
                     dfstrFsUuidValue( text, p->fsUuid, TRUE);
                  }
                  else
                  {
                     sprintf( text, "%s%38.38s", x, "");
                  }
                  strcat(  line, text);
               }
               if (xCols)
               {
                  TxStrip( text, p->fsform, ' ', ' ');
                  if ((xFsys[0] == '*') || (strstr( xFsys, text)))
                  {
                     //- switch screen AND logfile off during select (silent)
                     sprintf( text, "part -Q:3 %hu -q -a", p->id);
                     dfsMultiCommand(  text, 0, FALSE, FALSE, FALSE);

                     sprintf( text, "%s%11.1lf%s%11.1lf%s%11.1lf%s",        x,
                              TXSMIB( dfsa->FsTruncPoint,  p->bpsector), x,
                              TXSMIB( dfsa->FsExpandSize,  p->bpsector), x,
                              TXSMIB( dfsa->FsUnallocated, p->bpsector), x);
                     switch (dfsa->FsDirtyStatus)
                     {
                        case DFSTAT_CLEAN: strcat( text, "Clean");   break;
                        case DFSTAT_DIRTY: strcat( text, "Dirty");   break;
                        default:           strcat( text, "Unknown"); break;
                     }
                  }
                  else
                  {
                     sprintf( text, "%s        %s        %s        %s", x, x, x, x);
                  }
                  strcat(  line, text);
               }
               strcat( line, dsp);
               TxPrint( "%s", line);
               if (TxaExeSwitch('g'))
               {
                  printf( "%s", line);
                  fflush( stdout);
               }
            }
         }
      }
      if (f != NULL)                            // at least one area shown
      {
         if (f->fspChain != NULL)               // not last freespace area
         {
            dfsShowFreeSpaceSep( f->fspChain, x, dsp, line);
         }
      }
      else if (d->fspHead != NULL)              // one and only (whole disk)
      {
         if (d->OpenError == FALSE)
         {
            dfsShowFreeSpaceSep( d->fspHead, x, dsp, line);
         }
      }
   }
   TxFreeMem( line);
   if (xCols)                                   // part may have been selected
   {
      dfstRestoreDefaultStore( cst);            // reselect current store
   }
   VRETURN();
}                                               // end 'dfsShowPartSepList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create an XML style output (to a file) with detailed partition info
//
// Note:   used in eCS 2.x installer for resizing GUI (OEMSV)
/*****************************************************************************/
static void dfsShowPartXmlList
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
)
{
   USHORT              index;                   // partition index
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSPARTINFO        *p;                       // ptr to partition info
   TXLN                fname;                   // XML filename
   TXTM                text;                    // one line text buffer
   BOOL                xCols = TxaOption('x');  // extra FS columns wanted
   char               *xFsys = TxaOptStr('x', NULL, "*"); // default all FS
   DFST_HANDLE         cst   = dfstGetDefaultStore();
   FILE               *xml   = stdout;          // default output to STDOUT
   char               *xid   = TxaOptStr( TXA_O_LABEL, NULL,
                               #if defined (OEMSV)
                                                      "ecs");
                               #elif defined (OEMAN)
                                                      "aos");
                               #else
                                                      "dfs");
                               #endif
   ENTER();

   if (!TxaOptUnSet( TXA_O_FILE))               // -file- keeps STDOUT
   {
      sprintf( fname, "%sdisks", xid);          // default filename
      if (TxaOptSet( TXA_O_FILE))               // can be overruled
      {
         strcpy( fname, TxaOptStr( TXA_O_FILE, NULL, fname));
      }
      TxFnameExtension( fname, "rdf");
      if ((xml = fopen( fname, "w")) == NULL)
      {
         TxPrint("Error opening  file  : '%s' for write\n", fname);
      }
   }
   if (xml != NULL)
   {
      sprintf( text, "http://xml.%s.com/installer/disks#", xid);
      TxPrint( "\nGenerating DFSee XML overview to: '%s', please wait ...\n", (xml == stdout) ? "STDOUT" : fname);

      fprintf( xml, "<?xml version=\"1.0\"?>\n");
      fprintf( xml, "<RDF:RDF xmlns:RDF=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\"\n"
                    "         xmlns:%sdisks=\"%s\">\n", xid, TxaOptStr( TXA_O_URL, NULL, text));
      if (xCols)
      {
         dfstSetDefaultStore( DFST_SYSTEM);     // select system store
      }

      for (index = 0; index < dfsNrPart; index++)
      {
         p = &(dfsPartInfo[index]);
         d = dfsGetDiskInfo( p->disknr);
         if ((d != NULL) && ((disknr == FDSK_ANY) || (disknr == p->disknr)))
         {
            TxPrint( "Adding partition %02.2hu, %-8.8s", index +1, p->fsform);
            dfsSiz8( " size: ", p->sectors, "\n");

            TxStrip( text, p->fsform, ' ', ' ');
            if ((xFsys[0] == '*') || (strstr( xFsys, text)))
            {
               //- switch screen AND logfile off during select (silent)
               sprintf( text, "part -Q:3 %hu -q -a", p->id);
               dfsMultiCommand(  text, 0, FALSE, FALSE, FALSE);

               fprintf( xml, "    <RDF:Description RDF:about=\"urn:%s:part%02hu\"\n",
                                                                       xid, (USHORT) (index +1));
               fprintf( xml, "         %sdisks:number=\"%hu\"\n",      xid, (USHORT) (index +1));
               fprintf( xml, "         %sdisks:disknr=\"%hu\"\n",      xid, p->disknr);
               fprintf( xml, "         %sdisks:diskstyle=\"%s\"\n",    xid, d->pStyle);
               fprintf( xml, "         %sdisks:letter=\"%s\"\n",       xid, p->drive);
               fprintf( xml, "         %sdisks:type=\"%s %2.2hhx\"\n",  xid,
                        (p->primary) ? "primary" : "logical", p->partent.PartitionType);
               fprintf( xml, "         %sdisks:descr=\"%s\"\n",        xid, p->descr);
               fprintf( xml, "         %sdisks:bootstatus=\"%s\"\n",   xid,
                       (p->partent.Status && p->primary) ? "BIOS active" :
                       (strlen( p->blabel) == 0)    ? "" : "BMGR startable");
               fprintf( xml, "         %sdisks:removable=\"%s\"\n",    xid,
                       (d->Removable) ? "yes" : "no");
               fprintf( xml, "         %sdisks:related=\"%s\"\n",      xid, p->creatr);
               fprintf( xml, "         %sdisks:device=\"%8.8s%hu\"\n", xid, d->UnixDeviceDisk,
                  (USHORT) (((p->primary == FALSE) && (d->ebrHead != NULL)) ?
                  (USHORT) (p->tablenr +4) : (USHORT) (p->partnr +1)));
               strcpy(  text, p->plabel);
               TxStrip( text, text, ' ', ' ');
               fprintf( xml, "         %sdisks:label=\"%s\"\n",          xid, text);
               if (p->uuidPresent)
               {
                  strcpy( text, "");
                  dfstrFsUuidValue( text, p->fsUuid, TRUE);
                  fprintf( xml, "         %sdisks:uuid=\"%s\"\n",        xid, text);
               }
               if (p->lvmPresent)
               {
                  TxCopy(  text, p->lvm.VoluName, LVM_NAME_L +1);
                  TxStrip( text, text, ' ', ' ');
                  fprintf( xml, "         %sdisks:lvmvolume=\"%s\"\n",   xid, text);
                  TxCopy(  text, p->lvm.PartName,  LVM_NAME_L +1);
                  TxStrip( text, text, ' ', ' ');
                  fprintf( xml, "         %sdisks:lvmpart=\"%s\"\n",     xid, text);
                  fprintf( xml, "         %sdisks:installable=\"%s\"\n", xid,
                                (p->lvm.Installable)  ? "true" : "false");
               }
               else if (p->tablenr == GPT_STYLE) // name is stored in lvm info
               {
                  fprintf( xml, "         %sdisks:gptpart=\"%s\"\n",   xid, p->lvm.VoluName);
               }
               fprintf( xml, "         %sdisks:flags1=\"%8.8x\"\n",     xid, p->flags);
               fprintf( xml, "         %sdisks:flags2=\"%8.8x\"\n",     xid, p->flag2);
               fprintf( xml, "         %sdisks:fixpbr=\"%s\"\n",         xid,
                                      ((p->flag2 & DFS_F_HIDDENSBAD) ||
                                       (p->flag2 & DFS_F_BOOTGEOBAD))      ? "yes" : "no");
               fprintf( xml, "         %sdisks:fixchs=\"%s\"\n",         xid,
                                      ((p->flags & DFS_F_CHSLBA_ERR) != 0) ? "yes" : "no");
               fprintf( xml, "         %sdisks:lvmcrcseq=\"%s\"\n",      xid,
                                      ((p->flag2 & DFS_F_LVMBADCRCI) ||
                                       (p->flag2 & DFS_F_LVMBADCRCS) ||
                                       (p->flag2 & DFS_F_LVMBADSEQN) != 0) ? "yes" : "no");
               fprintf( xml, "         %sdisks:cleanup=\"%s\"\n",        xid,
                                      ((p->flags & DFS_F_EXTEND_ACT) ||
                                       (p->flags & DFS_F_MULTIP_ACT) ||
                                       (p->flag2 & DFS_F_NONSTDFLAG))      ? "yes" : "no");
               fprintf( xml, "         %sdisks:beyondenddisk=\"%s\"\n",  xid,
                                      ((p->flags & DFS_F_PART_EDISK) != 0) ? "yes" : "no");
               fprintf( xml, "         %sdisks:dlatinsidepart=\"%s\"\n",  xid,
                                      ((p->flag2 & DFS_F_DLATINPART) != 0) ? "yes" : "no");
               fprintf( xml, "         %sdisks:wastedextcont=\"%s\"\n",  xid,
                                      ((p->flag2 & DFS_F_WASTECONTN) != 0) ? "yes" : "no");
               fprintf( xml, "         %sdisks:cylinder=\"%u\"\n",      xid,
                                                      (ULONG) (p->basePsn / p->cSC));
               fprintf( xml, "         %sdisks:minsize=\"%u\"\n",
                        xid, DFSECT2MIB( dfsa->FsTruncPoint,        p->bpsector));
               fprintf( xml, "         %sdisks:maxsize=\"%u\"\n",
                        xid, DFSECT2MIB((dfsa->FsExpandSize == 0) ? p->sectors :
                                         dfsa->FsExpandSize,        p->bpsector));
               fprintf( xml, "         %sdisks:size=\"%u\"\n",
                        xid, DFSECT2MIB( p->sectors, p->bpsector));
               if (dfsa->FsUnallocated != 0)
               {
                  fprintf( xml, "         %sdisks:free=\"%u\"\n",
                           xid, DFSECT2MIB( dfsa->FsUnallocated,    p->bpsector));
               }
               if ((strncasecmp( p->fsform, "FAT",  3) == 0) ||
                   (strncasecmp( p->fsform, "NTFS", 4) == 0) ||
                   (strncasecmp( p->fsform, "HPFS", 4) == 0)  )
               {
                  fprintf( xml, "         %sdisks:resize=\"%s\"\n", xid,
                     ( (p->sectors > p->cSC) && (dfsa->FsTruncPoint != 0) &&
                      ((p->sectors - p->cSC)  >  dfsa->FsTruncPoint))  ? "true" : "false");
               }
               else
               {
                  fprintf( xml, "         %sdisks:resize=\"false\"\n", xid);
               }
               fprintf( xml, "         %sdisks:status=\"", xid);
               switch (dfsa->FsDirtyStatus)
               {
                  case DFSTAT_CLEAN: fprintf( xml, "clean\"\n");   break;
                  case DFSTAT_DIRTY: fprintf( xml, "dirty\"\n");   break;
                  default:           fprintf( xml, "unknown\"\n"); break;
               }
               TxStrip( text, p->fsform, ' ', ' ');
               fprintf( xml, "         %sdisks:filesystem=\"%s\"/>\n",
                        xid, text);
            }
         }
      }

      fprintf( xml, "    <RDF:Seq RDF:about=\"urn:%s:disklist\">\n", xid);
      for (index = 0; index < dfsNrPart; index++)
      {
         p = &(dfsPartInfo[index]);
         if ((disknr == FDSK_ANY) || (disknr == p->disknr))
         {
            TxStrip( text, p->fsform, ' ', ' ');
            if ((xFsys[0] == '*') || (strstr( xFsys, text)))
            {
               fprintf( xml, "        <RDF:li RDF:resource=\"urn:%s:part%02hu\"/>\n",
                        xid, (USHORT) (index +1));
            }
         }
      }
      fprintf( xml, "    </RDF:Seq>\n");
      fprintf( xml, "</RDF:RDF>\n");
      if (xCols)                                // part may have been selected
      {
         dfstRestoreDefaultStore( cst);         // reselect current store
      }
      if (TxaOptSet( TXA_O_FILE))
      {
         fclose( xml);                          // close outputfile
      }
      TxPrint( "DFSee XML overview ready.\n\n");
   }
   VRETURN();
}                                               // end 'dfsShowPartXmlList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show very compact (space) separated list with partition-info (DFSee query)
// -g    EXE switch now allows ouput of JUST the partition lines to STDOUT
/*****************************************************************************/
void dfsShowPartQueryList
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
)
{
   USHORT              index;                   // partition index
   USHORT              disk;                    // disk number
   USHORT              partype;
   char                active;                  // boot/start active indicator
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSPARTINFO        *e;                       // ptr to extended part info
   DFSPARTINFO        *f;                       // ptr to freespace info
   TXTM                line;                    // full line of output, max 128!
   TXTS                text;                    // one word text buffer
   TXTS                h;                       // separator string in header
   USHORT              startDisk  = (disknr != FDSK_ANY) ? disknr : 1;
   USHORT              finalDisk  = (disknr != FDSK_ANY) ? disknr : dfsDisks;
   char               *x = TxaOptStr( 'C', NULL, " ");
   TXTS                dsp;                     // doublespaced line-end

   ENTER();

   strcpy( dsp, (TxaOption('S')) ? "\n\n" : "\n");
   memset(   h,  ' ', TXMAXTS);
   h[ min( strlen(x), TXMAXTS -1)] = 0;         // terminate the string

   TxPrint( "\n Id%sD%sA%sDr%sType%sForm%sLabel info %sLVM vol,part %sBegin-cyl%s  End-cyl%s Size MiB%s",
                h, h, h, h, h, h, h, h, h, h, dsp);

   for (disk = startDisk; disk <= finalDisk; disk++)
   {
      f = NULL;                                 // no freespace displayed yet
      d = &(dfsDiskInfo[disk]);

      if (d && (d->OpenError == FALSE))
      {
         TxPrint(   " --%s-%s-%s--%s----%s----%s-----------%s-------------%s---------%s---------%s-----------%s",
                      h, h, h, h, h, h, h, h, h, h, dsp);

         for (index = 0; index < dfsNrPart; index++)
         {
            p = &(dfsPartInfo[index]);
            if (p->disknr == disk)              // partition on current disk
            {
               BOOL       brbad = (BOOL) (p->flags & DFS_F_BOOTRECBAD);

               if (p->fspChain != NULL)
               {
                  f = p->fspChain;              // last shown freespace area
                  dfsShowFreeSpaceQuery( f, x, dsp, line);
               }
               partype = p->partent.PartitionType;
               if (p->partent.Status && p->primary)
               {
                  active = (char)(brbad) ? (char) '!' : (char) '>';
               }
               else
               {
                  if (strlen(p->blabel) > 0)    // bootable by bootmgr
                  {
                     active = (char)(brbad) ? (char) '-' : (char) '*';
                  }
                  else
                  {
                     active = ' ';              // default no indicator
                  }
               }
               if (d->Removable)
               {
                  if (active == ' ') active = 'r';
                  else               active = 'R';
               }

               if ((TxaOption('e')) && ((e = p->ebrChain) != NULL))
               {
                  strcpy(  text, "");
                  sprintf( line, " %2.2d%s%d%s%c%s%-2.2s%s%s "
                          "%2.2x%s%-4.4s%s%-11.11s%s%-13.13s%s"
                          "%9u%s%9u%s%11.1lf%s",
                                    e->id,      x, e->disknr,       x,
                               ' ',             x, "",              x,
                                   (e->primary) ? "M" : "E",
                                    e->partent.PartitionType,       x,
                                    "",         x,
                                    "",         x,
                                    "",         x,
                           (ULONG) (e->basePsn / e->cSC),  x,
                           (ULONG) (e->lastPsn / e->cSC),  x,
                            TXSMIB( e->sectors, e->bpsector), dsp);
                  TxPrint( "%s", line);
                  if (TxaExeSwitch('g'))
                  {
                     printf( "%s", line);
                     fflush( stdout);
                  }
               }
               if (!TxaOptUnSet('p'))           // unless partitions blocked
               {
                  TXTM    bootlabel;

                  if (p->lvmPresent)            // use volume, partition name
                  {
                     strncpy( bootlabel, p->lvm.VoluName, LVM_NAME_L);
                     bootlabel[LVM_NAME_L] = '\0';
                     strncpy( text,      p->lvm.PartName, LVM_NAME_L);
                     text[LVM_NAME_L] = '\0';
                     if (strlen(text) > 0)
                     {
                        bootlabel[8] = '\0';    // truncate volume-name
                        strcat( bootlabel, (*x == ',') ? "/" : ",");
                        strcat( bootlabel, text);
                     }
                  }
                  else if (p->tablenr == GPT_STYLE) // name is stored in lvm info
                  {
                     strcpy( bootlabel, p->lvm.VoluName);
                  }
                  else
                  {
                     if ((dfsa->lvmPresent) && (strlen(p->bmname) == 0))
                     {
                        strcpy( bootlabel, "-No-LVM-info-");
                     }
                     else
                     {
                        if (strlen(p->bmname))
                        {
                           sprintf( bootlabel, "BM:%s", p->bmname);
                        }
                        else
                        {
                           strcpy( bootlabel, "");
                        }
                     }
                  }
                  sprintf( line, " %2.2d%s%d%s%c%s%-2.2s%s%s "
                          "%2.2x%s%-4.4s%s%-11.11s%s%-13.13s%s"
                          "%9u%s%9u%s%11.1lf%s",
                                    p->id,      x, p->disknr,       x,
                            active,             x, p->drive,        x,
                        ((dfsPartTypeHidable((BYTE) partype)) &&
                                   (partype & DFS_P_PHIDDEN))  ?
                                    "H" : (p->primary) ? "P" : "L",
                                    partype,    x,
                                    p->fsform,  x,
                                    p->plabel,  x,
                                    bootlabel,  x,
                           (ULONG) (p->basePsn / p->cSC),  x,
                           (ULONG) (p->lastPsn / p->cSC),  x,
                            TXSMIB( p->sectors, p->bpsector), dsp);
                  TxPrint( "%s", line);
                  if (TxaExeSwitch('g'))
                  {
                     printf( "%s", line);
                     fflush( stdout);
                  }
               }
            }
         }
         if (f != NULL)                         // at least one area shown
         {
            if (f->fspChain != NULL)            // not last freespace area
            {
               dfsShowFreeSpaceQuery( f->fspChain, x, dsp, line);
            }
         }
         else if (d->fspHead != NULL)           // one and only (whole disk)
         {
            if (d->OpenError == FALSE)
            {
               if ((d->pStyle[0] != DFSP_MBR) &&
                   (d->pStyle[0] != DFSP_GPT))  // only MBR/GPT style has whole disk freespace
               {
                  dfsShowNonMbrQuery( d, x, dsp, line);
               }
               else
               {
                  dfsShowFreeSpaceQuery( d->fspHead, x, dsp, line);
               }
            }
         }
      }
   }
   TxPrint(   " --%s-%s-%s--%s----%s----%s-----------%s-------------%s---------%s---------%s-----------%s",
                h, h, h, h, h, h, h, h, h, h, dsp);
   TxPrint( "      > = Active, * = Bootable, r = Removable, R = Removable+Active/Bootable (possibly)\n");
   VRETURN();
}                                               // end 'dfsShowPartQueryList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show table with partition-info depending on detail/type
/*****************************************************************************/
static void dfsShowPartTable
(
   USHORT              disknr                   // IN    disknr or FDSK_ANY
)
{
   USHORT              index;                   // partition index
   USHORT              disk;                    // disk number
   USHORT              partype;
   char                active;                  // boot/start active indicator
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSPARTINFO        *e;                       // related ebr partition
   DFSPARTINFO        *f;                       // related freespace area
   TXTS                text;                    // one word text buffer
   TXTS                name;                    // one word text buffer
   ULONG               cyl;                     // cylinder count
   int                 nl;                      // variable column width
   USHORT              startDisk = (disknr != FDSK_ANY) ? disknr : 1;
   USHORT              finalDisk = (disknr != FDSK_ANY) ? disknr : dfsDisks;
   ULONG               blocksize = 1;
   BOOL                blocks    = TxaOption('b');
   BOOL                numeric   = TxaOption('n');
   BOOL                partinfo  = !TxaOptUnSet('p');
   BOOL                freespace = !TxaOptUnSet('f');
   char               *top;
   char               *txt;
   char               *mid;
   char               *bot;
   char               *vtx;
   ULONG               extended;                // nr of extended/EBR to show per disk
   char                eActive;                 // ACTIVE indicator for the extended

   ENTER();

   if (numeric)
   {
      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            nl = 10;
         }
         else
      #endif
      {
         nl = 7;
      }
      top = TableFrameTon;
      txt = TableFrameTxn;
      mid = TableFrameMin;
      bot = TableFrameBon;
      vtx = TableFrameVtn;
   }
   else if (blocks)
   {
      blocksize = TxaOptNum('b', NULL, 1);
      nl  = 10;
      top = TableFrameTob;
      txt = TableFrameTxb;
      mid = TableFrameMib;
      bot = TableFrameBob;
      vtx = (blocksize != 1) ? TableFrameVtb : TableFrameVtn;
   }
   else
   {
      if (dfsGetDisplayMargin() > 72)
      {
         nl = min( TXMAXTT - 1, dfsGetDisplayMargin() - 72); // size max 63 chars (dxInfo minus terminator)
      }
      else
      {
         #if defined (USEWINDOWING)
            if (txwIsWindow( TXHWND_DESKTOP))
            {
               nl = 10;
            }
            else
         #endif
         {
            nl = 7;
         }
      }
      top = TableFrameTol;
      txt = TableFrameTxl;
      mid = TableFrameMil;
      bot = TableFrameBol;
      vtx = TableFrameVtl;
   }

   TxPrint( TableFrameTop);
   TxPrint( top, nl, nl, TableFrameLin);
   sprintf( text, "%sux%s", CNY, CNN);
   TxPrint( TableFrameTxt, text);
   TxPrint( txt, nl, nl, vtx);

   for (disk = startDisk; disk <= finalDisk; disk++)
   {
      if (TxaOptSet('e'))                       // explicit set ...
      {
         extended  = TxaOptNum('e', NULL, DFS_MAX_PART);
      }
      else                                      // allow -e:1 for each disk
      {                                         // to show outer container
         extended = 0;
      }
      f = NULL;                                 // no freespace displayed yet

      if (((d = &(dfsDiskInfo[disk])) != NULL) && (d->OpenError == FALSE))
      {
         TxPrint( TableFrameMid, CNY,  d->UnixDeviceDisk, CNN, CBM,  d->pStyle, CBC,  disk, CNN);
         TxPrint( "%s[%s%s%s]%s", mid, CBC,    d->DiskName, CNN,
                  TableFrameEnd + TXMAXTM + strlen( d->DiskName) - nl);

         for (index = 0; index < dfsNrPart; index++)
         {
            p = &(dfsPartInfo[index]);
            if (p->disknr == disk)              // partition on current disk
            {
               BOOL       brbad = (BOOL) (p->flags & DFS_F_BOOTRECBAD);

               if (p->fspChain != NULL)
               {
                  f = p->fspChain;              // last shown freespace area
                  if (freespace)
                  {
                     dfsShowFreeSpaceArea( f, nl);
                  }
               }
               partype = p->partent.PartitionType;

               if (p->partent.Status && p->primary)
               {
                  active = (char)(brbad) ? (char) '!' : (char) '>';
               }
               else
               {
                  if (strlen(p->blabel) > 0)    // bootable by bootmgr
                  {
                     active = (char)(brbad) ? (char) '-' : (char) '*';
                  }
                  else
                  {
                     active = ' ';
                  }
               }
               if (d->Removable)
               {
                  if (active == ' ') active = 'r';
                  else               active = 'R';
               }

               if (((e = p->ebrChain)  != NULL) && // there is an extended container
                    (e->partent.Status != 0)     ) // and it is marked ACTIVE
               {
                  if (e->tablenr       == 0)       // It is the 1st one (MBR)
                  {
                     if (extended      == 0)    // But it would NOT be displayed
                     {
                        extended++;             // force display of an ACTIVE outer container
                     }
                     eActive = '>';
                  }
                  else                          // other EBR marked ACTIVE, weird
                  {
                     eActive = '!';
                  }
               }
               else
               {
                  eActive = ' ';
               }
               if (numeric)
               {
                  if (e && extended)            // is logical, and need to display
                  {
                     extended--;                // one less to display ...
                     TxPrint((e->id < 100) ? "%s%02.2d%s" : "%s%03.3d%s", CBZ, e->id, CNN);
                     TxPrint("%s%c%s%2hu", CBC, eActive, CNN, e->UnixDevNr);
                     if (partinfo)              // unless partitions blocked
                     {
                        TxPrint("  ");          // no letters on EBR's
                     }
                     else
                     {
                        TxPrint("%-2.2s", p->drive);
                     }
                     cyl = e->basePsn / e->cSC;
                     sprintf( text, (cyl > 99999999) ? "%9u" : "%8u ", cyl);
                     TxPrint("%s %2.2x %s%10llx% 10llx%s-%9u% *llx",
                                   (e->primary) ? "mbr " : "ebr ",
                                    e->partent.PartitionType,
                                    e->descr,
                                    e->basePsn,  e->lastPsn,
                                    text,
                           (ULONG) (e->lastPsn / e->cSC),
                                nl, e->sectors);
                     dfsShowMiBsize(e->sectors,  e->bpsector);
                  }
                  if (partinfo)                 // unless partitions blocked
                  {
                     if ((p->tablenr == GPT_STYLE) || // use raw description as type
                         (p->tablenr == APM_STYLE)  )
                     {
                        sprintf( name, "%-17.17s", p->descr);
                     }
                     else
                     {
                        sprintf( name, "%s %2.2x %s",
                                       ((dfsPartTypeHidable((BYTE) partype)) &&
                                          (partype & DFS_P_PHIDDEN))  ?
                                          (p->primary) ? "Hide" : "Lhid" :
                                          (p->primary) ? "Prim" : "Log ", partype, p->descr);
                     }
                     cyl = p->basePsn / p->cSC;
                     sprintf( text, (cyl > 99999999) ? "%9u" : "%8u ", cyl);
                     TxPrint((p->id < 100) ? "%02.2d" : "%03.3d", p->id);
                     TxPrint("%s%c%s%s%2hu%s%-2.2s%s% 10llx% 10llx%s-%9u% *llx",
                                   CBC, active, CNN,
                                   CNY, p->UnixDevNr, CNN,
                                    p->drive,
                                    name,
                                    p->basePsn,  p->lastPsn,
                                    text,
                           (ULONG) (p->lastPsn / p->cSC),
                                nl, p->sectors);
                     dfsShowMiBsize(p->sectors,  p->bpsector);
                  }
               }
               else if (blocks)                 // mixed label / blocks
               {
                  if (e && extended)            // is logical, and need to display
                  {
                     extended--;                // one less to display ...
                     TxPrint((e->id < 100) ? "%s%02.2d%s" : "%s%03.3d%s", CBZ, e->id, CNN);
                     TxPrint("%s%c%s%2hu", CBC, eActive, CNN, e->UnixDevNr);
                     if (partinfo)              // no driveletters on EBR
                     {                          // when partitions shown too
                        TxPrint("  ");
                     }
                     else
                     {
                        TxPrint("%-2.2s", p->drive);
                     }
                     TxPrint("%s %2.2x %s", (e->primary) ? "mbr " : "ebr ",
                                                    e->partent.PartitionType,
                                                    e->descr);
                     TxPrint("%-5.5s%-7.7s", (partinfo) ? ""          : p->fsform, e->creatr);
                     cyl = e->basePsn / e->cSC;
                     sprintf( text, (cyl > 99999999) ? "%9u" : "%8u ", cyl);
                     TxPrint("%s-%9u%12llu%12llu%*llu",
                                    text, (ULONG) (e->lastPsn / e->cSC),
                                    e->basePsn,  e->lastPsn, nl, (e->sectors / blocksize));
                     dfsShowMiBsize(e->sectors,  e->bpsector);
                  }
                  if (partinfo)                 // unless partitions blocked
                  {
                     if ((p->tablenr == GPT_STYLE) || // use raw description as type
                         (p->tablenr == APM_STYLE)  )
                     {
                        sprintf( name, "%-17.17s", p->descr);
                     }
                     else
                     {
                        sprintf( name, "%s %2.2x %s",
                                       ((dfsPartTypeHidable((BYTE) partype)) &&
                                          (partype & DFS_P_PHIDDEN))  ?
                                          (p->primary) ? "Hide" : "Lhid" :
                                          (p->primary) ? "Prim" : "Log ", partype, p->descr);
                     }
                     cyl = p->basePsn / p->cSC;
                     sprintf( text, (cyl > 99999999) ? "%9u" : "%8u ", cyl);
                     TxPrint((p->id < 100) ? "%02.2d" : "%03.3d", p->id);
                     TxPrint("%s%c%s%s%2hu%s%-2.2s%s%-5.5s%-7.7s%s-%9u%12llu%12llu%*llu",
                               CBC, active, CNN,
                               CNY, p->UnixDevNr, CNN,
                                    p->drive,
                                    name,
                                    p->fsform,   p->creatr, text,
                           (ULONG) (p->lastPsn / p->cSC),
                                    p->basePsn / blocksize,
                                    p->lastPsn / blocksize, nl,
                                   (p->sectors / blocksize));
                     dfsShowMiBsize(p->sectors,  p->bpsector);
                  }
               }
               else                             // labels, no numbers
               {
                  TXTT    varColumn;            // at most 2 * LVM-name + 2

                  if ((p->lvmPresent)  ||                //- real LVM, use volume, partition name
                      (strcmp(p->fsform, "LUKS") == 0))  //- LUKS, use cipher and hash info
                  {
                     strncpy( name, p->lvm.VoluName, LVM_NAME_L);
                     name[LVM_NAME_L] = '\0';
                     strncpy( text,      p->lvm.PartName, LVM_NAME_L);
                     text[LVM_NAME_L] = '\0';
                     if (strlen(text) > 0)
                     {
                        sprintf( varColumn, (nl > 25) ? "%-20.20s %s"
                                                      : "%s, %s", name, text);
                     }
                     else
                     {
                        strcpy( varColumn, name); // just a volumename
                     }
                  }
                  else if ((p->tablenr == GPT_STYLE) || // name is stored in lvm info
                           (p->tablenr == APM_STYLE)  ) // Volume+Part names
                  {
                     strcpy( varColumn, p->lvm.VoluName);
                  }
                  else
                  {
                     if ((dfsa->lvmPresent) && (strlen(p->bmname) == 0))
                     {
                        strcpy( varColumn, NO_LVM_GPT_LUKS);
                     }
                     else
                     {
                        if (dfsa->bmgrDisk != 0)
                        {
                           if (strlen(p->bmname))
                           {
                              sprintf( varColumn, "BM:%s", p->bmname);
                           }
                           else
                           {
                              strcpy( varColumn, "");
                           }
                        }
                        else                    // HEX number of sectors
                        {
                           sprintf( varColumn, "%10llx", p->sectors);
                        }
                     }
                  }
                  if (e && extended)            // is logical, and need to display
                  {
                     extended--;                // one less to display ...
                     TxPrint((e->id < 100) ? "%s%02.2d%s" : "%s%03.3d%s", CBZ, e->id, CNN);
                     TxPrint("%s%c%s%2hu", CBC, eActive, CNN, e->UnixDevNr);
                     if (partinfo)              // no driveletters on EBR
                     {                          // when partitions shown too
                        TxPrint("  ");
                     }
                     else
                     {
                        TxPrint("%-2.2s", p->drive);
                     }
                     TxPrint("%s %2.2x %s", (e->primary) ? "mbr " : "ebr ",
                                               e->partent.PartitionType,
                                               e->descr);
                     if (partinfo)              // no labels on EBR lines
                     {                          // when partition shown too
                        TxPrint("%-8.8s%-8.8s%-11.11s%-*.*s", "",
                                       e->creatr, "",    nl, nl, "");
                        dfsShowMiBsize(e->sectors,  e->bpsector);
                     }
                     else                       // labels and size
                     {
                        TxPrint("%-8.8s%-8.8s%-11.11s%-*.*s", p->fsform,
                                       e->creatr,   p->plabel, nl, nl, varColumn);
                        dfsShowMiBsize(e->sectors,  e->bpsector);
                     }
                  }
                  if (partinfo)                 // unless partitions blocked
                  {
                     if ((p->tablenr == GPT_STYLE) || // use raw description as type
                         (p->tablenr == APM_STYLE)  )
                     {
                        sprintf( name, "%-17.17s", p->descr);
                     }
                     else
                     {
                        sprintf( name, "%s %2.2x %s",
                                       ((dfsPartTypeHidable((BYTE) partype)) &&
                                          (partype & DFS_P_PHIDDEN))  ?
                                          (p->primary) ? "Hide" : "Lhid" :
                                          (p->primary) ? "Prim" : "Log ", partype, p->descr);
                     }
                     TxPrint((p->id < 100) ? "%02.2d" : "%03.3d", p->id);
                     TxPrint("%s%c%s%s%2hu%s%-2.2s%s%-8.8s%-8.8s%-11.11s%-*.*s",
                               CBC, active, CNN,
                               CNY, p->UnixDevNr, CNN,
                                    p->drive,
                                    name,
                                    p->fsform,   p->creatr,
                                    p->plabel,   nl, nl, varColumn);
                     dfsShowMiBsize(p->sectors,  p->bpsector);
                  }
               }
            }
         }
         if (f != NULL)                         // at least one area shown
         {
            if (f->fspChain != NULL)            // not last freespace area
            {
               if (freespace)
               {
                  dfsShowFreeSpaceArea( f->fspChain, nl);
               }
            }
         }
         else if (d->fspHead != NULL)           // one and only (whole disk)
         {
            if ((d->pStyle[0] != DFSP_MBR) &&
                (d->pStyle[0] != DFSP_GPT))     // only MBR/GPT style has whole disk freespace
            {
               dfsShowNonMbrDiskArea( d, nl);
            }
            else
            {
               dfsShowFreeSpaceArea( d->fspHead, nl);
            }
         }
      }
   }
   TxPrint( TableFrameBot);
   TxPrint( bot, nl, nl, TableFrameLin);
   TxPrint( "   %s>%s = Active/Startable %s*%s = Bootable"
              " %sr%s = Removable %sR%s = Removable+Active/Bootable (possibly)\n",
                CBC, CNN, CBC, CNN, CBC, CNN, CBC, CNN);
   VRETURN();
}                                               // end 'dfsShowPartTable'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Correlate and copy LVM disk/volume information to owning partition
// Note: this is a callback function to be used with dfsIteratePlvm
//       Non 0x35 type BBR sectors may be checked too (generate WARNING)
/*****************************************************************************/
static ULONG dfsPartLvmMatch                    // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM info
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTINFO        *p;                       // ptr to partition info

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if ((sec != NULL) && (phase == FDSK_PERFORM))
   {
      BYTE             st  = dfsIdentifySector( cbp->sn, 0, sec);
      S_LVINF         *sd  = (S_LVINF *) sec;
      ULONG            crc = sd->LvmCRC;        // CRC found in sector

      sd->LvmCRC = 0;                           // prepare for calculation
      sd->LvmCRC = TxCalculateLvmCrc( sec, SECTORSIZE);

      if (st == ST_LVINF)                       // only found in FDISK mode
      {
         S_LVPART     *lp = NULL;               // LVM part info
         int           pi;                      // partition index in LVM
         DFSDISKINFO  *d = &(dfsDiskInfo[cbp->disknr]);

         for ( pi = 0; pi < 4; pi++)            // maximum of 4 per LVM sec
         {
            lp = &sd->part[pi];
            if (lp->sectors != 0)               // valid entry
            {
               TRACES(( "LVM DLAT info sector for lvmsnp: %8.8x in entry %d\n", lp->PartitId, pi));
               if ((p = dfsFdskDlatEntry2Part( lp, cbp->disknr)) != NULL)
               {
                  TRACES(("Match '%-*.*s' to part: %hu\n", LVM_NAME_L, LVM_NAME_L, lp->PartName, p->id));
                  if (crc != sd->LvmCRC)
                  {
                     p->flag2 |= DFS_F_LVMBADCRCI; // CRC value is bad
                  }

                  //- Test for, and set a WARNING if the size/position is different
                  if ((p->sectors != lp->sectors) || (p->basePsn != lp->partPsn)  )
                  {
                     p->flag2 |= DFS_F_LVMBADSIZE; // Size/position mismatch
                  }

                  d->lvmPartitions++;           // #partition with LVM info
                  p->lvmPresent = TRUE;         // present for this partition
                  p->lvm = sd->part[pi];        // assign LVM info to part

                  TRHEXS(70, &(p->lvm), sizeof(S_LVPART), "LVM info for part");

                  if (d->lvmDiskId == 0)        // disk-level info not set ...
                  {
                     d->lvmBootId = sd->BootDiskId;
                     d->lvmDiskId = sd->ThisDiskId;
                     memcpy( d->DiskName, sd->DiskName, LVM_NAME_L);
                     if (d->DiskName[LVM_NAME_L -1] != 0)
                     {
                        d->flags |= DFS_F_LVMDISK_20; // 20 characters long!
                     }
                     if (sd->geoCyls > 65535)
                     {
                        d->flags |= DFS_F_LVMCYL_64K; // GEO Cyl > 65535
                     }
                  }
                  else                          // check disk properties
                  {
                     if (memcmp( d->DiskName, sd->DiskName, LVM_NAME_L) != 0)
                     {
                        p->flag2 |= DFS_F_LVMNAMEBAD; // different diskname!
                     }
                     if (d->lvmDiskId != sd->ThisDiskId)
                     {
                        p->flag2 |= DFS_F_LVMTHISBAD; // different THIS id!
                     }
                     if (d->lvmBootId != sd->BootDiskId)
                     {
                        p->flag2 |= DFS_F_LVMBOOTBAD; // different BOOT id!
                     }
                  }
                  dfsa->lvmPresent = TRUE;      // some valid LVM info present
                  if (dfsa->lvmBootId == 0)
                  {
                     dfsa->lvmBootId = sd->BootDiskId;
                  }
               }
               else                             // obsolete DLAT entry!
               {
                  TxPrint( "WARNING disk nr %hu : Unmatched entry "
                           "%d in LVM DLAT sector at 0x0%llX\n"
                           "%20.20sEITHER the LVM info is obsolete, OR the corresponding MBR\n"
                           "%20.20spartition-table entries are deleted (perhaps by accident)\n"
                           "%20.20sOnly when really obsolete, remove using command: 'lvm -c'\n",
                            cbp->disknr, pi, cbp->thisSn, "", "", "");
               }
            }
         }
         if (d->lvmDiskId == 0)                 // disk-level info STILL not set ...
         {
            //- Set diskname even if no partitions have any info (diskname by DFSee only)
            memcpy( d->DiskName, sd->DiskName, LVM_NAME_L);
         }
      }
      else if (st == ST_LVSIG)                  // check if multiple-partition
      {                                         // and INF/SIG consistency
         S_LVSIG      *sg = (S_LVSIG *) sec;

         TRACES(( "LVM signature sector for lvmsnp: %8.8x\n", sg->PartitId));

         p = dfsGetPartInfo( dfsLvmSnP2PartId( cbp->disknr, sg->PartitId));
         if (p != NULL)                         // real and valid LVM sig
         {
            p->lvmSigPres  = TRUE;              // signature is present

            if (p->partent.PartitionType == DFS_P_WARP_LVM) // and it IS an LVM one
            {
               S_LVFEAT     *lf;                // LVM feature info
               ULONG         fi;                // feature index

               if (crc != sg->LvmCRC)
               {
                  p->flag2 |= DFS_F_LVMBADCRCS; // CRC value is bad
               }
               p->lvmReserved = sg->reserved;   // size of reserved area

               for (fi = 0; (fi < LVM_FEATURES) && (p->lvm.letter != 0); fi++)
               {
                  lf = &(sg->feature[fi]);
                  TRACES(( "Feature %u (%u) with %u sectors, PID: %02.2hx\n", fi, lf->FeatureId, lf->sectors, p->id));
                  if (lf->sectors != 0)
                  {
                     if (lf->FeatureId == LVMF_DRIVE_LINK)
                     {
                        BYTE   *dls = NULL;     // DriveLink sector buffer

                        if ((dls = (TxAlloc( 2, dfsGetSectorSize()))) != NULL)
                        {
                           if (dfstReadPsn( DFSTORE, lf->priPsn, 2, dls) == NO_ERROR)
                           {
                              S_LVDLF      *sd = (S_LVDLF *)  dls;
                              S_LVDLT      *lt = (S_LVDLT *) (dls + dfsGetSectorSize());
                              ULONG         li;

                              p->lvmDriveLinks = sd->LinksInUse;
                              for (li = 0; (li < sd->LinksInUse) && (li < LVDLT_SIZE1); li++)
                              {
                                 if (sd->LinkTable[ li].PartitionSerialNr == sg->PartitId)
                                 {
                                    p->lvmDriveIndex = li +1; // relative for multi
                                    TRACES(( "DriveIndex for PID:%02.2hu set to %u\n", p->id, p->lvmDriveIndex));
                                 }
                                 //- to be refined, else see if existing partition-ID
                                 //- and issue a warning if not (multi-part incomplete)
                              }
                              if (lt->SequenceNr == 0) // not initialized (P#1262)
                              {
                                 p->flag2 |= DFS_F_LVMBADSEQN; // SEQ-nr value is bad
                              }
                           }
                           TxFreeMem( dls);     // free the memory
                        }
                        break;                  // abort feature loop
                     }
                  }
               }
               //- to be refined, check consistency between LVMSIG values and
               //- the corresponding ones in the p->lvm structure.
            }
            else
            {
               //- no further checks desirable, non 0x35 should have no signature sector!
               //- Setting lvmSigPress will lead to a WARNING later ...
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsPartLvmMatch'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Merge collected LVM info into regular part-info (drive-letter / volume)
/*****************************************************************************/
static void dfsMergeLvmDriveInfo
(
   BOOL                letters                  // IN    assign letters too
)
{
   USHORT        index;                         // partition index in table
   DFSPARTINFO  *p;                             // ptr to partition info

   ENTER();

   for (index = 0; index < dfsNrPart; index++)
   {
      p = &(dfsPartInfo[index]);
      if (p->lvmPresent)
      {
         if ((p->lvm.VoluName[LVM_NAME_L -1] != 0) || (p->lvm.PartName[LVM_NAME_L -1] != 0))
         {
            p->flag2 |= DFS_F_LVMNAME_20;       // 20 characters long!
         }

         if (letters)
         {
            if ((toupper(p->drive[0])) != toupper(p->lvm.letter)) // not actual
            {
               #if defined (DUMP)
                  if (p->drive[0] != '-')
                  {
                     TRACES(( "PID:%02.2hx letter '%s' is not actual\n", p->id, p->drive));
                  }
               #endif
               if (p->drive[0] == 0)            // no letter yet
               {
                  p->drive[0] = '-';
               }
               if ((p->drive[0]   == '-') &&    // just a dummy assigned now
                   (p->lvmDriveLinks > 1) &&    // multiple LVM partition
                   (p->lvmDriveIndex > 1))      // and not first one
               {
                  p->drive[0] = tolower(p->lvm.letter); // prefered letter
                  p->drive[1] = p->lvmDriveIndex + '0'; // multi-part index
               }
               else
               {
                  if (p->lvm.letter == 0)
                  {
                     p->drive[1] = '-';
                  }
                  else
                  {
                     p->drive[1] = tolower(p->lvm.letter); // add/merge prefered
                  }
               }
               p->drive[2] = 0;
               #if defined (DUMP)
                  if (p->drive[0] != '-')
                  {
                     TRACES(("part %2d ==> %s%s%s, other letter prefered!\n", p->id, CBR, p->drive, CNN));
                  }
               #endif
            }
            else
            {
               TRACES(("part %2d ==> %s%s%s, actual same as prefered\n", p->id, CBG, p->drive, CNN));
            }
         }
         else
         {
            TRACES(("part %2d ==> %s%s%s, keep correlated letter\n", p->id, CBG, p->drive, CNN));
         }

         if (p->lvm.OnBmMenu)
         {
            if (p->lvm.VolumeId != 0)           // real Volume
            {
               TxCopy( p->blabel, p->lvm.VoluName, LVM_NAME_L +1);
            }
            else
            {
               TxCopy( p->blabel, p->lvm.PartName, LVM_NAME_L +1);
            }
            TRACES(("Assigned '%s' to blabel for (LVM) partition: %hu\n", p->blabel, p->id));
         }
         else
         {
            TRACES(("Partition %hu NOT OnBmMenu\n", p->id));
         }
      }
   }
   VRETURN();
}                                               // end 'dfsMergeLvmDriveInfo'
/*---------------------------------------------------------------------------*/
