//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFSee OS specific interfaces
//
//   for OS2: LVM Engine interface, dynamically attached
//
// Author: J. van Wijk
//
// JvW  08-01-2006 Initial version, derived from TXOSAPI

#include <txlib.h>

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsosapi.h>                           // OS specific stuff

#if   defined (DOS32)
// no DOS specific functions yet


#elif defined (WIN32)
// no WIN specific functions yet

#elif defined (UNIX)
// no Unix specific functions yet


#else

#define LVME_REBOOT_REQ  44

typedef unsigned char  BOOLEAN;
typedef void *         LVMEREF;                 // engine reference (address)


typedef struct lbmvcr
{
   ULONG               vSerial;
   LVMEREF             vHandle;
   BOOLEAN             Compatible;
   BYTE                DeviceType;              // 0=HD 1=PRM 2=CD 3=NETW 4=??
   BYTE                Fill[2];
} LVMVCR;

typedef struct lvmvca
{
   LVMVCR             *VcData;                  // array, must be freed!
   ULONG               Count;                   // number of records
} LVMVCA;


typedef struct lvmvinfo                         // 60 bytes
{
   ULONG      Size;                             // nr of sectors
   ULONG      Partitions;                       // parts in volume
   ULONG      ConflictFlag;                     // 0=OK 1=gotit 2=moved 4=hide
   BOOLEAN    Compatible;                       // compatibility volume
   BOOLEAN    Bootable;                         // on the BM menu
   char       lPrefered;                        // preferred driveletter
   char       lCurrent;                         // current driveletter
   char       lInitial;                         // initial driveletter
   BOOLEAN    New_Volume;                       // did not exist before Open
   BYTE       Status;                           // 0=none 1=boot 2=start 3=inst
   BYTE       Fill1;
   char       VoluName[LVM_NAME_L];             // Logical volume name
   char       FsysName[LVM_NAME_L];             // FileSystem name
} LVMVINFO;


typedef void      (_System * DFSLVM_OPEN)
(
   BOOLEAN             ignoreCHS,
   ULONG              *error
);

typedef void      (_System * DFSLVM_CLOSE)
(
   void
);

typedef void      (_System * DFSLVM_HIDE)
(
   LVMEREF             vHandle,
   ULONG              *error
);

typedef void      (_System * DFSLVM_LETTER)
(
   LVMEREF             vHandle,
   char                letter,
   ULONG              *error
);

typedef BOOLEAN   (_System * DFSLVM_COMMIT)
(
   ULONG              *error
);

typedef void      (_System * DFSLVM_FREE)
(
   LVMEREF             object
);

typedef LVMVCA    (_System * DFSLVM_VOLS)
(
   ULONG              *error
);

typedef LVMVINFO  (_System * DFSLVM_VINFO)
(
   LVMEREF             vHandle,
   ULONG              *error
);

typedef void      (_System * DFSLVM_REPRMS)
(
   ULONG              *error
);



typedef struct dfs_lvmapi                       // entrypoint structure
{
   DFSLVM_OPEN         Open;                    // Open LVM engine
   DFSLVM_CLOSE        Close;                   // Close LVM engine
   DFSLVM_HIDE         Hide;                    // Hide a volume
   DFSLVM_LETTER       Letter;                  // Assign driveletter
   DFSLVM_COMMIT       Commit;                  // Commit changes
   DFSLVM_FREE         FreeMem;                 // Free Engine memory
   DFSLVM_VOLS         Volumes;                 // Get Volume control data
   DFSLVM_VINFO        VolInfo;                 // Get Volume info one volume
   DFSLVM_REPRMS       RePrms;                  // Refresh removable media
} DFS_LVMAPI;                                   // end of struct "dfs_lvmapi"


static  BOOL       dfsLvmEngineAPItested =   FALSE;
static  DFS_LVMAPI lEngine = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};

#define DFS_LVMCALLS      ((PSZ) "lvm")         // name of dll with LVM calls
#define DFS_LVM_OPEN      ((PSZ) "Open_LVM_Engine")
#define DFS_LVM_CLOSE     ((PSZ) "Close_LVM_Engine")
#define DFS_LVM_HIDE      ((PSZ) "Hide_Volume")
#define DFS_LVM_LETTER    ((PSZ) "Assign_Drive_Letter")
#define DFS_LVM_COMMIT    ((PSZ) "Commit_Changes")
#define DFS_LVM_FREE      ((PSZ) "Free_Engine_Memory")
#define DFS_LVM_VOLS      ((PSZ) "Get_Volume_Control_Data")
#define DFS_LVM_VINFO     ((PSZ) "Get_Volume_Information")
#define DFS_LVM_REPRMS    ((PSZ) "Rediscover_PRMs")



// Get volume reference handle for specified volume name
static LVMEREF dfsLvmEngineVolRef
(
   char               *vname                    // IN    volume name
);


/*****************************************************************************/
// Test if LVM Engine Sync is wanted for disk partition specified
/*****************************************************************************/
BOOL dfsLvmSyncDesired                          // RET   Sync desired for part
(
   DFSPARTINFO        *p                        // IN    partition info or NULL
)
{
   BOOL                rc = FALSE;              // function return

   ENTER();

   if (p != NULL)
   {
      TRACES(( "p:%8.8lx id:%hu lvm:%s\n", p, p->id, (p->lvmPresent) ? "Y" : "N"));

      if (p->lvmPresent && (dfsLvmEnginePresent()))
      {
         if (TxaOptSet( DFS_O_SYNCLA))          // local forced override ?
         {
            rc = TxaOption( DFS_O_SYNCLA);
         }
         else
         {
            DFSDISKINFO *d = dfsGetDiskInfo( p->disknr);

            if (!d->Removable)
            {
               rc = dfsa->lvmSyncApi;           // use global default
            }
         }
      }
   }
   BRETURN( rc);
}                                               // end 'dfsLvmSyncDesired'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test if LVM Engine support is available by trying to load/attach the LVM DLL
/*****************************************************************************/
BOOL dfsLvmEnginePresent                        // RET   LVM Engine API present
(
   void
)
{
   BOOL                rc = FALSE;              // function return
   TXLN                dlmerror;                // one line of data
   HMODULE             engine = 0;              // handle LVM engine dll
   ULONG               dr;                      // API return code

   ENTER();

   if (dfsLvmEngineAPItested == FALSE)
   {
      if ((DosLoadModule( dlmerror, TXMAXLN, DFS_LVMCALLS, &engine)) == NO_ERROR)
      {
         TRACES(("LVM-engine module handle: %lu\n", engine));

         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_OPEN  ,(PFN *) &(lEngine.Open));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_OPEN  ,dr,       lEngine.Open));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_CLOSE ,(PFN *) &(lEngine.Close));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_CLOSE ,dr,       lEngine.Close));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_HIDE  ,(PFN *) &(lEngine.Hide));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_HIDE  ,dr,       lEngine.Hide));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_LETTER,(PFN *) &(lEngine.Letter));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_LETTER,dr,       lEngine.Letter));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_COMMIT,(PFN *) &(lEngine.Commit));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_COMMIT,dr,       lEngine.Commit));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_FREE  ,(PFN *) &(lEngine.FreeMem));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_FREE  ,dr,       lEngine.FreeMem));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_VOLS  ,(PFN *) &(lEngine.Volumes));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_VOLS  ,dr,       lEngine.Volumes));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_VINFO ,(PFN *) &(lEngine.VolInfo));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_VINFO ,dr,       lEngine.VolInfo));
         dr = DosQueryProcAddr( engine, 0,      DFS_LVM_REPRMS,(PFN *) &(lEngine.RePrms));
         TRACES(("%-25.25s rc:%lu fn:%8.8lx\n", DFS_LVM_REPRMS,dr,       lEngine.RePrms));
      }
      else
      {
         TxPrint( "\nLVM Engine DLL not available, no dynamic updates possible\n");
         TRACES(("Error %lu loading LVM DLL: '%s'\n", rc, DFS_LVMCALLS));
      }
      dfsLvmEngineAPItested = TRUE;
   }
   if ((lEngine.Open     != NULL) &&
       (lEngine.Close    != NULL) &&
       (lEngine.Hide     != NULL) &&
       (lEngine.Letter   != NULL) &&
       (lEngine.Commit   != NULL) &&
       (lEngine.FreeMem  != NULL) &&
       (lEngine.Volumes  != NULL) &&
       (lEngine.VolInfo  != NULL) &&
       (lEngine.RePrms   != NULL)  )
   {
      rc = TRUE;
   }
   BRETURN( rc);
}                                               // end 'dfsLvmEnginePresent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set driveletter for an LVM volume, or hide it; using the real LVM engine
// Note: DFSee should not have any disks opened (locked) so close all stores
// Empty volumename will execute just a RediscoverPRMs (Refresh removables)
/*****************************************************************************/
ULONG dfsLvmEngineExecute
(
   BOOL                verbose,                 // IN    display LVM command
   char               *volume,                  // IN    LVM volume name
   char                letter                   // IN    new letter or 0
)
{
   ULONG               rc = DFS_CMD_FAILED;     // function return
   ULONG               eRC;                     // engine RC
   LVMEREF             vol = 0;                 // volume reference handle

   ENTER();

   if (dfsLvmEnginePresent())                   // safety first, valid pointers
   {
      if (volume && strlen(volume))
      {
         (lEngine.Open)( TRUE, &eRC);
         if (eRC == NO_ERROR)
         {
            if ((vol = dfsLvmEngineVolRef( volume)) != 0)
            {
               if ((letter) && (letter != '-')) // assign a driveletter
               {
                  TRACES(( "Assign '%c:' for volume '%s'\n", letter, volume));
                  if (verbose)
                  {
                     TxPrint( "Sync - LVM Engine : Volume '%s', "
                              "assign new driveletter %c:\n", volume, letter);
                  }
                  (lEngine.Letter)( vol, toupper(letter), &eRC);
               }
               else                             // hide the volume
               {
                  TRACES(( "Hide volume '%s'\n", volume));
                  if (verbose)
                  {
                     TxPrint( "Sync - LVM Engine : Volume '%s', "
                              "removed driveletter (HIDE)\n", volume);
                  }
                  (lEngine.Hide)( vol, &eRC);
               }
               if (eRC == NO_ERROR)
               {
                  rc = NO_ERROR;
               }
               else if (eRC == LVME_REBOOT_REQ)
               {
                  if (verbose)
                  {
                     TxPrint( "Sync - LVM Engine : A reboot is required!\n", volume);
                  }
                  rc = DFS_REBOOT_REQ;          // LVM engine requires reboot
               }                                // may trap when ignored!
            }
            else
            {
               if (verbose)
               {
                  TxPrint( "Sync - LVM Engine : volume '%s' not found in LVM\n", volume);
               }
               rc = DFS_NOT_FOUND;
            }
            if (rc == NO_ERROR)
            {
               (lEngine.Commit)( &eRC);
               TRACES(( "LVM Engine Commit  RC: %lu\n", eRC));
            }
            (lEngine.Close)( );
         }
         else
         {
            if (verbose)
            {
               TxPrint( "Sync - LVM Engine : unable to open engine (LVM.DLL)\n");
            }
            TRACES(( "LVM Engine Open    RC: %lu\n", eRC));
         }
      }
      else                                      // refresh removables
      {
         if (verbose)
         {
            TxPrint( "Sync - LVM Engine : rediscover removable media ...\n");
         }
         (lEngine.RePrms)( &eRC);
         TRACES(( "LVM Engine RediscoverPRMs RC: %lu\n", eRC));
         if (eRC == NO_ERROR)
         {
            rc = NO_ERROR;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsLvmEngineExecute'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Refresh removable media, passed on to OS2LVM/OS2DASD; Engine must be closed
// Note: DFSee should not have any disks opened (locked) so close all stores
/*****************************************************************************/
ULONG dfsLvmRefreshRemovables
(
   void
)
{
   ULONG               rc = DFS_CMD_FAILED;     // function return
   ULONG               eRC;                     // engine RC

   ENTER();

   if (dfsLvmEnginePresent())                   // safety first, valid pointers
   {
      (lEngine.RePrms)( &eRC);
      TRACES(( "LVM Engine RediscoverPRMs RC: %lu\n", eRC));
      if (eRC == NO_ERROR)
      {
         rc = NO_ERROR;
      }
      else
      {
         TRACES(( "LVM Engine Open    RC: %lu\n", eRC));
      }
   }
   RETURN (rc);
}                                               // end 'dfsLvmRefreshRemovables'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get volume reference handle for specified volume name
/*****************************************************************************/
static LVMEREF dfsLvmEngineVolRef
(
   char               *vname                    // IN    volume name
)
{
   LVMEREF             rc = 0;                  // function return
   LVMVCA              va;                      // volume array
   LVMVINFO            volinfo;                 // info one volume
   ULONG               eRC;                     // engine RC
   ULONG               i;

   ENTER();

   va = (lEngine.Volumes)( &eRC);
   if (rc == NO_ERROR)
   {
      for (i = 0; i < va.Count; i++)
      {
         volinfo = (lEngine.VolInfo)( va.VcData[i].vHandle, &eRC);
         if (eRC == NO_ERROR)
         {
            TRACES(( "Vol:%-20.20s fs:%-8.8s p:%c c:%c i:%c size: %9.1lf\n",
                      volinfo.VoluName,    volinfo.FsysName,
                     (volinfo.lPrefered) ? volinfo.lPrefered : '-',
                     (volinfo.lCurrent)  ? volinfo.lCurrent  : '-',
                     (volinfo.lInitial)  ? volinfo.lInitial  : '-',
                                   TXSMIB( volinfo.Size, SECTORSIZE)));
            if (strcasecmp( vname, volinfo.VoluName) == 0)
            {
               rc = va.VcData[i].vHandle;
               TRACES(( "Found handle %8.8lx for '%s'\n", rc, vname));
               break;
            }
         }
         else
         {
            TRACES(( "VolInfo for handle %2lu = %8.8lx failed, RC:%lu\n",
                                     i, va.VcData[i].vHandle, eRC));
         }
      }
      (lEngine.FreeMem)( va.VcData);            // Free volume array
   }
   else
   {
      TRACES(( "LVM Engine Volumes RC: %lu\n", eRC));
   }
   RETURN (rc);
}                                               // end 'dfsLvmEngineVolRef'
/*---------------------------------------------------------------------------*/

#endif
