//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFSee OS/2 EA utility functions
//
// Author: J. van Wijk
//
// JvW  2006-03-27   Initial version, split off from DFSUHPFS/DFSUDIS2
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions

#include <dfsahpfs.h>                           // HPFS analysis functions
#include <dfsuhpfs.h>                           // HPFS utility functions
#include <dfsos2ea.h>                           // HPFS/JFS/FAT EA handling


// EA type indicators (EAT) as used by OS/2 on HPFS and FAT EA data
// Values 0xFFFE thru 0x8000 are reserved.          (for 'IBM' use)
// Values 0x0000 thru 0x7fff are user definable.    (but hardly seen)
// Value  0xFFFC is not used

#define DFS_EA_BINARY      0xFFFE       //- Single EA binary
#define DFS_EA_ASCII       0xFFFD       //- Single EA ASCII
#define DFS_EA_BITMAP      0xFFFB       //- Single EA bitmap
#define DFS_EA_METAFILE    0xFFFA       //- Single EA metafile
#define DFS_EA_ICON        0xFFF9       //- Single EA icon
#define DFS_EA_EA          0xFFEE       //- Single EA ASCII
                                        //- name of associated data (#include)
#define DFS_EA_MVMT        0xFFDF       //- multi-valued, multi-typed  EA
#define DFS_EA_MVST        0xFFDE       //- multi-valued, single-typed EA
#define DFS_EA_ASN1        0xFFDD       //- ASN.1 ISO multi-value data stream


typedef struct s_ea_atom
{
   USHORT              Length;                  // Length of single EA
   char                Data[1];                 // Actual EA data
} S_EA_ATOM;                                    // end of struct "s_ea_atom"

typedef struct s_easingle                       // normal EA
{
   USHORT              Type;                    // Type of the EA, length
   S_EA_ATOM           Ea;                      // Length and data of EA
} S_EASINGLE;                                   // end of struct "s_easingle"

typedef struct s_ea_mvmt                        // multi-value multi-type combo
{
   USHORT              Type;                    // Type of the EA, length
   S_EA_ATOM           Ea;                      // Length and data of EA
} S_EA_MVMT;                                    // end of struct "s_ea_mvmt"

typedef struct s_ea_mvst                        // multi-value single-type combo
{
   USHORT              Type;                    // Type of the EA, length
   S_EA_ATOM           Ea[1];                   // Repeated Length and data
} S_EA_MVST;                                    // end of struct "s_ea_mvst"

typedef struct s_eamulti
{
   USHORT              Type;                    // Multi/Single typed indicator
   USHORT              CodePage;                // CodePage value
   USHORT              Count;                   // Number of entries
   union
   {
      S_EA_MVMT        Mt[1];                   // Repeated multi-typed
      S_EA_MVST        St;                      // Single typed
   }                   Tp;                      // end of union multi-ea
} S_EAMULTI;                                    // end of struct "s_eamulti"

typedef union s_eadata
{
   S_EASINGLE          Sv;                      // single value
   S_EAMULTI           Mv;                      // multi  value
} S_EADATA;                                     // end of union "s_eadata"



// Get EA value for a single extended attribute
static ULONG dfsEaSingleGetValue
(
   S_EABLK            *data,                    // IN    data
   ULONG              *len,                     // OUT   EA data length
   void              **eaData                   // OUT   EA data (allocated)
);

// Display a single extended attribute
static USHORT dfsEaSingleDisplay                // RET   net size of EA
(
   char              *text,                     // IN    leading text
   S_EABLK           *data                      // IN    data
);

// Display a single atom-part for an extended attribute
static USHORT dfsEaAtomDisplay                  // RET   net size of EA
(
   USHORT              type,                    // IN    EA atom type
   S_EA_ATOM          *atom                     // IN    EA atom data
);


/*****************************************************************************/
// Display extended attributes from a memory-buffer
/*****************************************************************************/
void dfsEaBlockDisplay
(
   char               *text,                    // IN    leading text
   S_EABLK            *first,                   // IN    first EA block
   ULONG               size                     // IN    (max) size of EA area
)
{
   S_EABLK            *ea = first;              // EA block
   ULONG               at = 0;

   ENTER();
   TxPrint("%s", text);
   dfsDec4("      Ea Size: ", size, "\n");
   while ((  at               < size ) &&       // stop at given limit! P0088
          ( (ea->Flag & 0x7c) == 0   ) &&       // only allow 'sane' looking
          (  ea->NameLength   != 0   ) &&       // EA blocks
          (  ea->DataLength   != 0   ) &&
          (  ea->DataLength   <= (USHORT) size) &&
          ( (ea->DataLength   == (USHORT) sizeof(SPTR)) ||
           ((ea->Flag & 0x03) == 0)))
   {
      at += dfsEaSingleDisplay( "Name ", ea);
      ea  = (S_EABLK *) (((char *) first) + at);
      TRACES(("Next S_EABLK Flag: %2.2lX  size: %u, done: %lu, limit %lu\n",
               ea->Flag, ea->DataLength, at, size));
   }
   VRETURN();
}                                               // end 'dfsEaBlockDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display a single extended attribute
/*****************************************************************************/
static USHORT dfsEaSingleDisplay                // RET   net size of EA
(
   char               *text,                    // IN    leading text
   S_EABLK            *data                     // IN    data
)
{
   USHORT              rc = data->NameLength + data->DataLength + 5;
   char               *d  = data->Name + data->NameLength +1;
   S_EADATA           *ea = (S_EADATA *) d;
   USHORT              i;
   USHORT              offset;

   ENTER();
   TxPrint("%s%s%-20.20s%s %2.2X ", text, CNG, data->Name, CNN,
                                      (USHORT) data->Flag);
   switch (data->Flag & 0x03)                   // EA type
   {
      case 0x00:                                // Actual EA data
         dfsDec4("Size: ", data->DataLength, " ");
         switch (ea->Sv.Type)
         {
            case DFS_EA_MVMT:                   // multi-value, multi-type
               {
                  char *mvmt = (char *) &ea->Mv.Tp.Mt[0];

                  TxPrint("MultiVal-MultiType,  nr:%-5u\n", ea->Mv.Count);
                  for (offset = 0, i = 0; i < ea->Mv.Count; i++)
                  {
                     TxPrint("Ea-value as ");
                     mvmt  += offset;
                     offset = dfsEaAtomDisplay(  ((S_EA_MVMT *)mvmt)->Type,
                                                &((S_EA_MVMT *)mvmt)->Ea)
                            + sizeof(USHORT);   // EA-size + sizeof Type
                  }
               }
               break;

            case DFS_EA_MVST:                   // multi-value of same type
               {
                  char *atom = (char *) &ea->Mv.Tp.St.Ea[0];

                  TxPrint("MultiVal-SingleType, nr:%-5u\n", ea->Mv.Count);
                  for (offset = 0, i = 0; i < ea->Mv.Count; i++)
                  {
                     TxPrint("Ea-value as ");
                     atom  += offset;
                     offset = dfsEaAtomDisplay( ea->Mv.Tp.St.Type,
                                               (S_EA_ATOM *) atom);
                  }
               }
               break;

            case DFS_EA_ASCII:                  // known single-types
            case DFS_EA_EA:
            case DFS_EA_BINARY:
            case DFS_EA_BITMAP:
            case DFS_EA_METAFILE:
            case DFS_EA_ICON:
               dfsEaAtomDisplay( ea->Sv.Type, &ea->Sv.Ea);
               break;

            case DFS_EA_ASN1:                   // ISO, unknown format
               TxPrint("ISO ASN.1 format\n");
               TxDisplHexDump((BYTE *) d, data->DataLength);
               break;

            default:                            // user defined, unknown format
               TxPrint("User defined\n");
               TxDisplHexDump((BYTE *) d, data->DataLength);
               break;
         }
         break;

      case 0x01:                                // SPTR to EA data
         dfsDec4("Size: ",                   ((SPTR *) d)->size, "");
         TxPrint(" EA-Data-LSN : %08.8lX\n", ((SPTR *) d)->ptr);
         nav.xtra = ((SPTR *) d)->ptr;
         break;

      default:                                  // SPTR to ALLOC sector
         dfsDec4("Size: ",                   ((SPTR *) d)->size, "");
         TxPrint(" EaAlloc-LSN : %08.8lX\n", ((SPTR *) d)->ptr);
         nav.xtra = ((SPTR *) d)->ptr;
         break;
   }
   RETURN(rc);                                  // total size of this EA block
}                                               // end 'dfsEaSingleDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display a single atom-part for an extended attribute
/*****************************************************************************/
static USHORT dfsEaAtomDisplay                  // RET   net size of EA
(
   USHORT              type,                    // IN    EA atom type
   S_EA_ATOM          *atom                     // IN    EA atom data
)
{
   ENTER();
   switch (type)
   {
      case DFS_EA_ASCII:
      case DFS_EA_EA:
         switch (type)
         {
            case DFS_EA_EA:       TxPrint("EaName"); break;
            default:              TxPrint("Ascii "); break;
         }
         TxPrint(": %-5u %s%*.*s%s\n", atom->Length, CBG,
                                       atom->Length,
                                       atom->Length,
                                       atom->Data,   CNN);
         break;

      default:
         switch (type)
         {
            case DFS_EA_BINARY:   TxPrint("Binary"); break;
            case DFS_EA_BITMAP:   TxPrint("Bitmap"); break;
            case DFS_EA_METAFILE: TxPrint("Meta-f"); break;
            case DFS_EA_ICON:     TxPrint("Icon  "); break;
            default:              TxPrint("Other "); break;
         }
         TxPrint(": %-5u\n",           atom->Length);
         TxDisplHexDump((BYTE *)       atom->Data,
                                       atom->Length);
         break;
   }
   RETURN(atom->Length + sizeof(atom->Length));
}                                               // end 'dfsEaAtomDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get specified single EA value from an EA-block in a memory-buffer
/*****************************************************************************/
ULONG dfsEaBlockGetValue
(
   char               *name,                    // IN    EA name wanted
   S_EABLK            *first,                   // IN    first EA block
   ULONG               size,                    // IN    (max) size of EA area
   ULONG              *len,                     // OUT   EA data length
   void              **eaData                   // OUT   EA data (allocated)
)
{
   ULONG               rc = DFS_NOT_FOUND;
   S_EABLK            *ea = first;              // EA block
   ULONG               at = 0;

   ENTER();
   TRACES(( "name:'%s' size:%lu  flag:%2.2hx  NameLength:%2.2hx  DataLength:%4.4hx\n",
             name, size, ea->Flag, ea->NameLength, ea->DataLength));

   while ((  at               < size ) &&       // stop at given limit! P0088
          ( (ea->Flag & 0x7c) == 0   ) &&       // only allow 'sane' looking
          (  ea->NameLength   != 0   ) &&       // EA blocks
          (  ea->DataLength   != 0   ) &&
          (  ea->DataLength   <= (USHORT) size) &&
          ( (ea->DataLength   == (USHORT) sizeof(SPTR)) ||
           ((ea->Flag & 0x03) == 0)))
   {
      at += (ea->NameLength + ea->DataLength + 5);

      TRACES(( "EA: '%*.*s'\n", ea->NameLength, ea->NameLength, ea->Name));
      if ((strncasecmp( name, ea->Name, strlen( name)) == 0) &&
                  (ea->NameLength == strlen( name)))
      {
         rc = dfsEaSingleGetValue( ea, len, eaData);
         break;
      }
      else                                      // keep searching
      {
         ea  = (S_EABLK *) (((char *) first) + at);
         TRACES(("Next S_EABLK Flag: %2.2lX  size: %u, done: %lu, limit %lu\n",
                  ea->Flag, ea->DataLength, at, size));
      }
   }
   RETURN(rc);
}                                               // end 'dfsEaBlockGetValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get EA value for a single extended attribute
/*****************************************************************************/
static ULONG dfsEaSingleGetValue
(
   S_EABLK            *data,                    // IN    data
   ULONG              *len,                     // OUT   EA data length
   void              **eaData                   // OUT   EA data (allocated)
)
{
   ULONG               rc = NO_ERROR;
   char               *d  = data->Name + data->NameLength +1;
   S_EADATA           *ea = (S_EADATA *) d;
   char               *ev = NULL;               // allocated EA value

   ENTER();
   switch (data->Flag & 0x03)                   // EA type
   {
      case 0x00:                                // Inline EA data
         switch (ea->Sv.Type)
         {                                      // get 1st of a multi-value
            case DFS_EA_MVST:                   // multi-valued, single-type EA
            case DFS_EA_BINARY:                 // all single-types
            case DFS_EA_ASCII:
            case DFS_EA_BITMAP:
            case DFS_EA_METAFILE:
            case DFS_EA_ICON:
            case DFS_EA_EA:
               TRACES(( "Allocate %lu bytes for EA value\n", ea->Sv.Ea.Length +1));
               if ((ev = TxAlloc( 1, ea->Sv.Ea.Length +1)) != NULL)
               {
                  *len = ea->Sv.Ea.Length +1;
                  memcpy( ev, ea->Sv.Ea.Data, ea->Sv.Ea.Length);

                  TRHEXS( 70, ev, *len, "EA data area");
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
               break;

            default:                            // not supported yet
               rc = DFS_ST_MISMATCH;
               break;
         }
         break;

      case 0x01:                                // SPTR to EA data
         //- to be refined, read from external data-extent (HPFS only)
         rc = DFS_NOT_FOUND;
         break;

      default:                                  // SPTR to ALLOC sector
         //- to be refined, read using ALLOC descriptions  (HPFS only)
         rc = DFS_NOT_FOUND;
         break;
   }
   *eaData = ev;
   RETURN(rc);                                  // total size of this EA block
}                                               // end 'dfsEaSingleGetValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add external-EA sector(s) info to std FEA2-list structure
/*****************************************************************************/
ULONG dfsOs2EaReadExtern
(
   ULN64               sn,                      // IN    first ext-EA LSN
   ULONG               offset,                  // IN    EABLK position
   ULONG               asize,                   // IN    ea area size
   S_FEA2LIST         *feal                     // INOUT list of FEA2's
)                                               //       alloc by caller
{
   ULONG               rc = NO_ERROR;
   BYTE               *db = NULL;
   ULONG               eaSectors = ((asize -1) / dfsGetSectorSize()) +1;

   ENTER();
   if ((db = TxAlloc( eaSectors, dfsGetSectorSize())) != NULL)
   {
      TRACES(("LSN: 0x%llX  offset:%lu  asize:%lu\n", sn, offset, asize));
      if ((rc = dfsRead( sn, eaSectors, (BYTE   *) db)) == NO_ERROR)
      {
         rc = dfsOs2EaReadArea((S_EABLK *) (db + offset), asize, feal);
      }
      TxFreeMem(db);
   }
   RETURN(rc);
}                                               // end 'dfsOs2EaReadExtern'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add extended attributes area std FEA2-list structure
/*****************************************************************************/
ULONG dfsOs2EaReadArea
(
   S_EABLK            *first,                   // IN    first EA block
   ULONG               size,                    // IN    size of EA area
   S_FEA2LIST         *feal                     // INOUT list of FEA2's
)
{
   ULONG               rc = NO_ERROR;
   S_EABLK            *ea = first;              // current EA block
   ULONG               at = 0;                  // position in EA area
   USHORT              delta;                   // move-distance in EA area

   ENTER();
   TRACES(("first:0x%8.8lx Flag: 0x%2.2hhx nl:%hhu dl:%hu\n",
            first, first->Flag, first->NameLength, first->DataLength));
   while ( ( rc == NO_ERROR          ) &&        // no read-errors
          ( (ea->Flag & 0x7c) == 0   ) &&        // only allow 'sane' looking
          (  ea->NameLength   != 0   ) &&        // EA blocks
          (  ea->DataLength   != 0   ) &&
          (  ea->DataLength   <= (USHORT) size) &&
          ( (ea->DataLength   == (USHORT) sizeof(SPTR)) ||
           ((ea->Flag & 0x03) == 0)))
   {
      if ((rc = dfsOs2EaReadSingle( ea, &delta, feal)) == NO_ERROR)
      {
         at += delta;
         ea  = (S_EABLK *) (((char *) first) + at);
      }
   }
   feal->cbList += sizeof(ULONG);               // add size for cbList itself
   RETURN(rc);
}                                               // end 'dfsOs2EaReadArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add extended attribute to std FEA2-list structure
/*****************************************************************************/
ULONG dfsOs2EaReadSingle
(
   S_EABLK            *data,                    // IN    data
   USHORT             *read,                    // OUT   read size of data
   S_FEA2LIST         *feal                     // INOUT list of FEA2's
)
{
   ULONG               rc   = NO_ERROR;
   char               *d    = data->Name + data->NameLength +1;
   S_FEA2             *fea  = (S_FEA2 *) ((char *) (feal) + feal->cbList +4);
   ULONG               next = 0;                // minimum offset to next FEA2
   ULONG               datasize;
   char               *nextdest;

   ENTER();

   //- note: EA type 00 shared by HPFS and JFS, others are HPFS only!

   *read = data->NameLength + data->DataLength + sizeof(S_EABLK);
   switch (data->Flag & 0x03)                   // EA type
   {
      case 0x00:                                // Actual EA data in Area
         memcpy( &fea->fEA, data, *read);       // copy complete S_EABLK
         datasize = data->DataLength;
         next = data->NameLength + datasize + sizeof(S_FEA2);
         break;

      case 0x01:                                // SPTR to EA data
         datasize     = ((SPTR *) d)->size;
         fea->fEA     = (BYTE)   0x00;          // make it an 'inline' EA
         fea->cbName  = (BYTE)   data->NameLength;
         fea->cbValue = (USHORT) datasize;
         strcpy( fea->szName, data->Name);
         next = data->NameLength + datasize + sizeof(S_FEA2);
         rc = dfsOs2EaReadData(((SPTR *) d)->ptr, datasize,
                                  fea->szName + fea->cbName +1);
         break;

      default:                                  // SPTR to ALLOC sector
         datasize     = ((SPTR *) d)->size;
         fea->fEA     = (BYTE)   0x00;          // make it an 'inline' EA
         fea->cbName  = (BYTE)   data->NameLength;
         fea->cbValue = (USHORT) datasize;
         strcpy( fea->szName, data->Name);
         next = data->NameLength + datasize + sizeof(S_FEA2);
         nextdest = (char *) &(fea->szName) + fea->cbName +1;
         rc = dfsOs2EaReadAlloc( ((SPTR *) d)->ptr, &datasize, &nextdest);
         break;
   }
   fea->oNextEntryOffset = (next + 3) & 0xFFFFFFFC; // Dword-align
   feal->cbList += fea->oNextEntryOffset;
   RETURN(rc);
}                                               // end 'dfsOs2EaReadSingle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add allocation-sector EA info to std FEA2-list structure (recursive)
/*****************************************************************************/
ULONG dfsOs2EaReadAlloc
(
   ULN64               sn,                      // IN    Alloc-sector LSN
   ULONG              *size,                    // INOUT size left todo
   char              **dest                     // INOUT destination ptr
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                id  = ST_UDATA;          // type of sector
   BYTE               *sb  = NULL;              // sector buffer

   ENTER();
   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead(sn, 1, sb);
      if (rc == NO_ERROR)
      {
         id = dfsIdentifySector(sn, 0, sb);     // double-check sector type
         if (id == ST_ALLOC)
         {
            S_ALLOC   *sd = (S_ALLOC *) sb;
            BYTE       af = sd->AlInfo.Flag;
            int        al = (int) sd->AlInfo.UsedEntries;
            int        i;

            for (i=0; i < al; i++)
            {
               if (af & 0x80)                   // NODE allocation block
               {
                  dfsOs2EaReadAlloc(sd->Al.Node[i].ChildBlock, size, dest);
               }
               else                             // LEAF allocation block
               {
                  ULONG thissize = min((ULONG) *size,
                                       (ULONG) sd->Al.Leaf[i].LengthData *
                                               SECTORSIZE);

                  rc   = dfsOs2EaReadData(sd->Al.Leaf[i].DataBlock, thissize,
                                                                    *dest);
                  *size -= thissize;            // left to do
                  *dest += thissize;            // next destination
               }
            }
         }
      }
      TxFreeMem(sb);
   }
   RETURN(rc);
}                                               // end 'dfsOs2EaReadAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add EA data from sector(s) to std FEA2-list structure
/*****************************************************************************/
ULONG dfsOs2EaReadData
(
   ULN64               sn,                      // IN    first ext-EA LSN
   ULONG               size,                    // IN    ea data size
   void               *dest                     // OUT   FEA2 data area
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *sb  = NULL;              // sector buffer
   ULONG               nr;                      // nr of ea sectors
   USHORT              bps = dfsGetSectorSize();

   ENTER();
   nr = ((size -1) / bps) +1;

   if ((sb = TxAlloc(nr, bps)) != NULL)
   {
      rc = dfsRead(sn, nr, sb);
      if (rc == NO_ERROR)
      {
         memcpy( dest, sb, size);
      }
      TxFreeMem(sb);
   }
   RETURN(rc);
}                                               // end 'dfsOs2EaReadData'
/*---------------------------------------------------------------------------*/

