//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// NT registry disk-key structure and Dosdevice mapping display and update
//
// Author: J. van Wijk
//
// JvW  2018-11-21   Added NtGetDiskMfgFwSerial() using CCS->disk->enum reg key
// JvW  1998-04-25   Final test and cleanup (DefineDosDevice, "" as target!)
// JvW  1998-04-23   Added DosDevice display and update
// JvW  1998-04-18   Generate registry diskkey;
// JvW  1998-04-09   Initial version, display only; based on registry analysis
//

#include <txlib.h>                              // TX library interface

#if defined (WIN32)                             // windows only source!

#include <winioctl.h>
#include <winnt.h>
#include <winreg.h>

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsupart.h>                           // FS partition utility functions
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsntreg.h>                           // DFS NT registry interface

// registry key storing disk Manufacturer, Firmware-id and Serialnumber info
#define DFSNTREG_CCS_DISK_MFS "System\\CurrentControlSet\\Services\\Disk\\Enum"


#define DFSNTREG_DISK_KEYNAME "System\\Disk"    // DISK key in LOCAL_MACHINE
#define DFSNTREG_DISK_VALNAME "Information"     // binary value with all info
#define DFSNTREG_DISK_VERSION 3                 // NT 4 disk-key version

#define DFSNTREG_PART_NONE_FT 4                 // partition not part of FT

#define DFSNTSYM_CDROM_DEVICE "\\Device\\CdRom%u"
#define DFSNTSYM_DPART_DEVICE "\\Device\\HardDisk%u\\Partition%u"

typedef struct s_ntreg_head                     // header of registry disk key
{
   ULONG               Version;
   ULONG               CheckSum;
   ULONG               Dirty;                   // last shutdown was dirty
   ULONG               InfoOffset;              // offset to regular info
   ULONG               InfoSize;                // size of regular info
   ULONG               FtOffset;                // offset to FT info
   ULONG               FtSize;                  // size of Ft info
   ULONG               FtStrWidth;              // stripe width
   ULONG               FtPoolSize;              // size of Ft Pool
   ULN64               Reserved;
} S_NTREG_HEAD;                                 // end of struct "s_ntreg_head"

typedef struct s_ntreg_part                     // one partition id disk key
{
   ULONG               FtType;
   ULONG               FtState;
   ULN64               StartSect;
   ULN64               Length;
   ULN64               FtLength;
   ULN64               Reserved;
   BYTE                DrLetter;                // drive letter character
   BYTE                DrForced;                // drive letter use forced
   USHORT              LogNumber;
   USHORT              FtGroup;
   USHORT              FtMember;
   ULONG               Modified;
} S_NTREG_PART;                                 // end of struct "s_ntreg_part"


typedef struct s_ntreg_disk                     // one disk in disk key
{
   USHORT              NrParts;
   USHORT              Reserved;
   ULONG               Signature;
   S_NTREG_PART        Part[1];
} S_NTREG_DISK;                                 // end of struct "s_ntreg_disk"


typedef struct s_ntreg_info                     // array of info on all disks
{
   USHORT              NrDisks;
   USHORT              Reserved;
   S_NTREG_DISK        Disk[1];
} S_NTREG_INFO;                                 // end of struct "s_ntreg_info"


#define IS_NT4(t) ((t) == DFS_P_FAT12  || \
                   (t) == DFS_P_FAT16  || \
                   (t) == DFS_P_BIGDOS || \
                   (t) == DFS_P_INST_FS )

#define ntregNextDisk( p ) ((S_NTREG_DISK *)((p)->Part + (p)->NrParts))

static void ntregSetDiskInfo
(
   S_NTREG_DISK       *ntrDisk,                 // IN    Phys disk info
   USHORT              Index                    // IN    one based disk index
);

//
// The disk information
//
static char           *ntrBuff = NULL;          // buffer for registry value
static DWORD           ntrSize = 0;             // size of disk registry value
static LONG            ntrStat;                 // status NT registry action



/*****************************************************************************/
// Get DISK device-ID string (like manufacturer/model/) from REG CCS\disk\enum
/*****************************************************************************/
char *dfsNtGetDiskDeviceId                      // RET    Disk info, or NULL
(                                               //        (must be freed after use)
   USHORT              diskNr                   // IN     DFSee physical disknr (1 based)
)
{
   char               *rc = NULL;
   HKEY                hDiskKey = NULL;
   TXTS                diskNrString;
   DWORD               vSize = 0;               // init to 0 to get size
   char               *value = NULL;            // registry retrieval buffer

   ENTER();

   if ((diskNr >= 1) && (RegOpenKeyEx( HKEY_LOCAL_MACHINE, DFSNTREG_CCS_DISK_MFS, 0, KEY_READ, &hDiskKey) == ERROR_SUCCESS))
   {
      sprintf( diskNrString, "%hu", diskNr - 1); // make ZERO based disk number string

      if (RegQueryValueEx( hDiskKey, diskNrString, NULL, NULL, NULL, &vSize) == ERROR_SUCCESS)
      {
         value = TxAlloc( 1, TXMAXLN);          // plenty room for formatting
         if (value != NULL)
         {
            if (RegQueryValueEx( hDiskKey, diskNrString, NULL, NULL, value, &vSize ) == ERROR_SUCCESS)
            {
               //- This is the Windows 'device instance ID' (maxlen 200) can be used with the CM_* APIs
               //- to get to device information in the device tree    (Plug and Play services, complex)
               //- Example values:
               //- IDE\DiskSamsung_SSD_850_EVO_500GB_______________EMT02B6Q\5&57df284&0&0.0.0
               //- USBSTOR\Disk&Ven_General&Prod_USB_Flash_Disk&Rev_1100\0421QP0000002103&0

               value[ vSize] = 0;               // make it ZERO terminated
               TRACES(("RAW value: '%s'\n", value));

               if (strchr( value, '\\'))        // there are valid ID components, try to format
               {
                  int       words;
                  TXTM      class    = {0};     // class, like IDE, SCSI or USBSTOR
                  TXLN      devid    = {0};     // device ID, various formats
                  TXLN      instance = {0};     // Instance info, various formats (unique)


                  //- sscanf %s modified to use '\' as separator, and read separator to get to next one!
                  words = sscanf( value, "%[^\\]\\%[^\\]\\%s", class, devid, instance);
                  TxRepl( devid,    '_', ' ');  // change _ to spaces in DeviceId string
                  TxRepl( instance, '&', ' ');  // change & to spaces in Instance string

                  TRACES(("words: %d  class: '%s' devid: '%s' instance: '%s'\n", words, class, devid, instance));

                  if (strstr( devid, "&Ven") && strstr( devid, "&Prod")) // most USB devices ...
                  {
                     TXTM   vend = {0};
                     TXTM   prod = {0};
                     TXTS   rev  = {0};

                     words = sscanf( devid, "Disk&Ven %[^&]&Prod %[^&]&Rev %s", vend, prod, rev);
                     TRACES(("words: %d  vend: '%s' prod: '%s' rev: '%s'\n", words, vend, prod, rev));

                     TxStrip( prod, prod, ' ', ' '); // strip leading/trailing spaces
                     if ((words == 0) || (strlen( prod) == 0))
                     {
                        //- Just take the RAW devid string, starting from &Prod
                        sprintf( value, "%-8.8s  %-50.50s  %s", class, strstr( devid, "&Prod ") + 6, instance);
                     }
                     else                       // combine vendor and product strings
                     {
                        sprintf( devid, "%s %s", vend, prod);
                        sprintf( value, "%-8.8s  %-39.39s %-10.10s  %s", class, devid, rev, instance);
                     }
                  }
                  else                          // use device ID string, just skip the 4 'Disk' chars
                  {
                     sprintf( value, "%-8.8s  %-50.50s  ", class, devid + 4);

                     if (strncasecmp( devid + 4, "VBOX", 4) == 0) // translate instance from 'HEX' to ASCII
                     {
                        int             iHex;
                        int             iAsc;
                        USHORT          b16;

                        memset( devid, 0, strlen( instance));   //- terminating zeroes, plenty
                        for (iAsc = 0, iHex = 0; iHex < strlen( instance); iHex += 4)
                        {
                           TxCopy( class, instance + iHex, 5);  //- copy 4 bytes, plus terminating zero
                           sscanf( class, "%hx", &b16);         //- convert 16-bits (2 chars)
                           devid[ iAsc++] = b16 & 0xff;         //- Write LSB ascii byte
                           devid[ iAsc++] = b16 >> 8;           //- Write MSB ascii byte
                        }
                        TRACES(("VBOX translated: '%s' to '%s'\n", instance, devid));
                        strcat( value, devid);
                     }
                     else                       // just add raw instance value
                     {
                        strcat( value, instance);
                     }
                  }
               }
               TRACES(("FORMATTED value: '%s'\n", value));
               rc = value;                      // return allocated (formatted) value
            }
            else
            {
               TRACES(( "QueryValue with size %d failed!\n", vSize));
            }
         }
      }
      else
      {
         TRACES(( "QueryValue size 0 failed!\n"));
      }
      RegCloseKey( hDiskKey );
   }
   RETPTR (rc);
}                                               // end 'dfsNtGetDiskDeviceId'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Correlate current NT DosDevice map with found partitions, merge as prefered
/*****************************************************************************/
ULONG dfsMergeDosDeviceMapping                  // RET nr of correlated drives
(
   void
)
{
   ULONG               drives = 0;              // DOS rc
   USHORT              dfsPi;                   // index in DFS part info
   USHORT              totalParts;              // total nr of partitions
   DFSPARTINFO        *p;                       // ptr to partition info
   TXTM                partdev;                 // partition device string
   TX1K                result;                  // resulting string array
   TX1K                current;                 // device string
   char               *s;

   ENTER();

   totalParts = dfsPartitions();
   for (dfsPi = 1; dfsPi <= totalParts; dfsPi++)
   {
      p = dfsGetPartInfo( dfsPi);
      if (p && (IS_NT4(p->partent.PartitionType)))
      {
         sprintf( partdev, DFSNTSYM_DPART_DEVICE, p->disknr -1, p->diskPart);

         TRACES(("part %2d, find '%s'\n", p->id, partdev));

         if (QueryDosDevice( NULL, result, TXMAX1K) != 0)
         {
            for (s = result; strlen(s); s += strlen(s) +1)
            {
               if (QueryDosDevice( s, current, TXMAX1K) != 0)
               {
                  if (strcasecmp( partdev, current) == 0)
                  {
                     if ((toupper(p->drive[0])) != toupper(*s)) // not actual
                     {
                        if (p->drive[0] == 0)   // no letter yet
                        {
                           p->drive[0] = '-';
                        }
                        p->drive[1] = tolower(*s); // add/merge prefered
                        p->drive[2] = 0;
                        TRACES(("part %2d ==> %s%s%s, other letter prefered!\n",
                                 p->id, CBR, p->drive, CNN));
                        drives++;
                     }
                     else
                     {
                        TRACES(("part %2d ==> %s%s%s, actual same as prefered\n",
                                 p->id, CBG, p->drive, CNN));
                     }
                  }
               }
            }
         }
      }
   }
   RETURN( drives);
}                                             // end 'dfsMergeDosDeviceMapping'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create diskkey registry key and synchronize DosDevice mappings
/*****************************************************************************/
ULONG ntregSyncDevMapDiskKey
(
   void
)
{
   ULONG              rc = NO_ERROR;            // function return

   ENTER();

   if (ntregDiskKeyFromDisk())
   {
      if (ntregDiskKeyToReg())
      {
         TxPrint( "NT registry diskkey created from DFSee info\n");
         if (ntregSetDevPartitions())           // update device mapping
         {
            TxPrint( "NT drive-letter mapping updated\n");
         }
         else
         {
            TxPrint( "NT drive-letter mapping failed\n");
         }
      }
      else
      {
         TxPrint( "NT registry diskkey create failed\n");
      }
      free( ntrBuff);                           // free registry-key info
      ntrBuff = NULL;
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'ntregSyncDevMapDiskKey'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get the DISK\Information value from the registry
/*****************************************************************************/
BOOL ntregDiskKeyFromReg
(
   void
)
{
   BOOL                rc = FALSE;
   DWORD               disktype;
   HKEY                hDiskKey = NULL;

   ENTER();

   ntrStat = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
                           DFSNTREG_DISK_KEYNAME,
                           0,
                           KEY_READ | KEY_WRITE,
                           &hDiskKey);
   TRACES(( "OpenKey returned status %ld\n", ntrStat));
   if (ntrStat == ERROR_SUCCESS)
   {
      ntrSize = 0;                              // NULL read to find size
      ntrStat = RegQueryValueEx( hDiskKey,
                                 DFSNTREG_DISK_VALNAME,
                                 NULL,
                                 &disktype,
                                 NULL, &ntrSize);
      if (ntrStat == ERROR_SUCCESS)             // zero size, filled in
      {
         ntrBuff = (char *) malloc( ntrSize );
         TRACES(( "Malloc of %lu resulting in %4.4lx\n", ntrSize, ntrBuff));
         if (ntrBuff != NULL)
         {
            ntrStat = RegQueryValueEx( hDiskKey,
                                       DFSNTREG_DISK_VALNAME,
                                       NULL,
                                       &disktype,
                                       ntrBuff, &ntrSize );
            TRACES(( "QueryValue returned status %ld\n", ntrStat));
            if (ntrStat == ERROR_SUCCESS)
            {
               rc = TRUE;
            }
         }
      }
      else
      {
         TRACES(( "QueryValue size 0 failed!\n"));
      }
      RegCloseKey( hDiskKey );
   }
   BRETURN (rc);
}                                               // end 'ntregDiskKeyFromReg'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Save the registry value
/*****************************************************************************/
BOOL ntregDiskKeyToReg
(
   void
)
{
   BOOL                rc = FALSE;
   LONG                status;
   HKEY                hDiskKey = NULL;
   DWORD               lpdwDisposition;
   char                KeyClass;

   ENTER();

   KeyClass = REG_BINARY;                       // Open the disk key
   status   = RegCreateKeyEx( HKEY_LOCAL_MACHINE,
                              DFSNTREG_DISK_KEYNAME,
                              0,
                              &KeyClass,
                              REG_OPTION_NON_VOLATILE,
                              KEY_READ | KEY_WRITE,
                              NULL,
                              &hDiskKey,
                              &lpdwDisposition);
   if (status == ERROR_SUCCESS)
   {
      status = RegSetValueEx( hDiskKey,
                              DFSNTREG_DISK_VALNAME,
                              0,
                              REG_BINARY,
                              ntrBuff,
                              ntrSize);
      if (status == ERROR_SUCCESS)
      {
         rc = TRUE;
      }
      else
      {
         TxPrint("Error writing DISK\\Information value: " );
      }
      RegCloseKey( hDiskKey );
   }
   else
   {
      TxPrint("Error opening DISK key: ");
   }
   BRETURN (rc);
}                                         // end 'ntregDiskKeyToReg'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Build the DiskKey registry value from current disk/partition info
/*****************************************************************************/
BOOL ntregDiskKeyFromDisk
(
   void
)
{
   BOOL                rc = FALSE;
   USHORT              i;
   USHORT              Disks;
   USHORT              totalParts;              // total nr of partitions
   USHORT              Parts = 0;
   S_NTREG_HEAD       *ntrHead;
   S_NTREG_INFO       *ntrInfo;
   S_NTREG_DISK       *ntrDisk;
   DFSPARTINFO        *p;                       // ptr to partition info

   ENTER();

   Disks      = dfsPartitionableDisks();        // nr of disks in system
   totalParts = dfsPartitions();                // total nr of partitions
   for (i = 1; i <= totalParts; i++)            // check all partitions
   {
      p = dfsGetPartInfo( i);
      if (p && (IS_NT4(p->partent.PartitionType)))
      {
         Parts++;                               // NT 4 recognized partition
      }
   }
   ntrSize = (DWORD) (sizeof(S_NTREG_HEAD)
           +         (sizeof(S_NTREG_INFO) - sizeof(S_NTREG_DISK))
           + Disks * (sizeof(S_NTREG_DISK) - sizeof(S_NTREG_PART))
           + Parts *  sizeof(S_NTREG_PART));

   TRACES(("Allocating %lu bytes for diskkey; disks: %hu, parts: %hu\n",
                                     ntrSize, Disks, Parts));
   ntrBuff = (char *) malloc( ntrSize );
   if (ntrBuff != NULL)
   {
      memset( ntrBuff, 0, ntrSize );            // initialize to all zero
      ntrHead = (S_NTREG_HEAD *) ntrBuff;

      ntrHead->Version    = DFSNTREG_DISK_VERSION;
      ntrHead->Dirty      = FALSE;
      ntrHead->InfoOffset = sizeof(S_NTREG_HEAD);
      ntrHead->InfoSize   = ntrSize - sizeof(S_NTREG_HEAD);
      ntrHead->FtOffset   = ntrSize;

      ntrInfo = (S_NTREG_INFO *) &ntrBuff [ ntrHead->InfoOffset];
      ntrInfo->NrDisks = Disks;

      ntrDisk = (S_NTREG_DISK *) &ntrInfo->Disk; // point to first disk
      for (i = 1; i <= Disks; i++)
      {
         ntregSetDiskInfo( ntrDisk, i);         // add info for this disk
         ntrDisk = ntregNextDisk( ntrDisk );    // point to next disk
      }
      rc = TRUE;
   }
   BRETURN (rc);
}                                               // end 'ntregDiskKeyFromDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set info from disk/partition tables in the DiskKey for one physical disk
/*****************************************************************************/
static void ntregSetDiskInfo
(
   S_NTREG_DISK       *ntrDisk,                 // IN    Phys disk info
   USHORT              Index                    // IN    one based disk index
)
{
   S_NTREG_PART       *ntrPart;                 // ptr to NT registry part
   USHORT              dfsPi;                   // index in DFS part info
   USHORT              ntrPi   = 0;             // index in NTregistry diskkey
   USHORT              totalParts;              // total nr of partitions
   DFSDISKINFO        *d;                       // corresponding diskinfo
   DFSPARTINFO        *p;                       // ptr to partition info

   ENTER();

   d = dfsGetDiskInfo( Index);
   if (d != NULL)
   {
      if (d->NTsignature == 0)                  // no signature yet ?
      {
         dfsFdskNtSignature( Index, NULL);      // set DFSee default signature
      }
      ntrDisk->Signature = d->NTsignature;      // set in registry DiskKey
   }

   totalParts = dfsPartitions();
   for (dfsPi = 1; dfsPi <= totalParts; dfsPi++)
   {
      p = dfsGetPartInfo( dfsPi);
      if (p && (p->disknr == Index) && (IS_NT4(p->partent.PartitionType)))
      {
         TRACES(("Adding partition %hu = '%s'\n", dfsPi, p->drive));
         ntrPart = &ntrDisk->Part[ ntrPi];      // point to registry part
         ntrPi++;                               // will be the next one ...

         TRACES(("basePsn: %llX\n", p->basePsn));

         ntrPart->FtType       = DFSNTREG_PART_NONE_FT;
         ntrPart->StartSect    = p->basePsn  * dfsGetSectorSize();
         ntrPart->Length       = p->sectors  * dfsGetSectorSize();
         if (p->drive[0] != '-')
         {
            ntrPart->DrLetter  = p->drive[0];
         }
         else
         {
            ntrPart->DrLetter  = 0;
         }
         ntrPart->DrForced     = TRUE;
         ntrPart->LogNumber    = p->diskPart;
         ntrPart->FtGroup      = 0xFFFF;
         ntrPart->Modified     = TRUE;
      }
   }
   ntrDisk->NrParts = ntrPi;                    // recognized partitions
   VRETURN();
}                                               // end 'ntregSetDiskInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update DosDevice mappings for NT partitions
/*****************************************************************************/
BOOL ntregSetDevPartitions
(
   void
)
{
   BOOL                rc = TRUE;               // function return
   USHORT              dfsPi;                   // index in DFS part info
   USHORT              totalParts;              // total nr of partitions
   DFSPARTINFO        *p;                       // ptr to partition info
   BOOL                assign;                  // assign required

   ENTER();

   totalParts = dfsPartitions();
   for (dfsPi = 1; dfsPi <= totalParts; dfsPi++)
   {
      p = dfsGetPartInfo( dfsPi);
      if (p && IS_NT4(p->partent.PartitionType))
      {
         TRACES(("DosDevice for disk %hu, relpart: %hu, dfspart %hu = '%s'\n",
                  p->disknr, p->diskPart, dfsPi, p->drive));

         assign = ntregFreeWhenDifferent( p->disknr, p->diskPart, p->drive);
         if (assign)
         {
            ntregFreeDriveLetter(  p->drive);   // make new letter available
            if (ntregAssignPartition( p->disknr, // make new assignment
                                      p->diskPart,
                                      p->drive) == FALSE)
            {
               rc = FALSE;                      // at least one failure
            }
         }
      }
   }
   BRETURN(rc);
}                                               // end 'ntregSetDevPartitions'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Modify CDROM driveletter assignment in registry and DosDevice tree
/*****************************************************************************/
ULONG ntregModifyCDRomLetter
(
   USHORT              cdnr,                    // IN    CDROM number
   char               *letter                   // IN    drive letter or '-'
)
{
   ULONG              rc = NO_ERROR;            // function return

   ENTER();

   ntregFreeCdrom(   cdnr);                     // detach from current letter
   ntregDelCdromKey( cdnr);                     // delete from registry

   if (*letter != '-')
   {
      ntregFreeDriveLetter(      letter);       // make new letter available
      ntregAssignCdrom(    cdnr, letter);       // make new assignment

      ntregSetCdromLetter( cdnr, letter);       // and set in registry
   }
   RETURN (rc);
}                                               // end 'ntregModifyCDRomLetter'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set specific drive-letter to use for a CDROM device, in the registry
/*****************************************************************************/
void ntregSetCdromLetter
(
   USHORT              cdnr,                    // IN    CDROM number
   char               *drive                    // IN    CDROM drive letter
)
{
   DWORD               disktype;
   HKEY                hDiskKey = NULL;
   DWORD               lpdwDisposition;
   char                KeyClass;
   LONG                status;
   TXTM                cdwanted;                // wanted device string
   TXTS                letter;                  // drive letter string

   KeyClass = REG_BINARY;
   status = RegCreateKeyEx( HKEY_LOCAL_MACHINE,
                            DFSNTREG_DISK_KEYNAME,
                            0,
                            &KeyClass,
                            REG_OPTION_NON_VOLATILE,
                            KEY_READ | KEY_WRITE,
                            NULL,
                            &hDiskKey,
                            &lpdwDisposition);
   if (status == ERROR_SUCCESS)
   {
      sprintf( cdwanted, DFSNTSYM_CDROM_DEVICE, cdnr);
      sprintf( letter, "%c:", drive[0]);
      disktype = REG_SZ;

      status = RegSetValueEx( hDiskKey,
                              cdwanted,
                              0,
                              disktype,
                              letter,
                              strlen(letter));
      if (status != ERROR_SUCCESS)
      {
         TxPrint( "Error writing DISK\\CdRom drive-letter\n");
      }
      RegCloseKey( hDiskKey );
   }
   else
   {
      TxPrint( "Error creating disk registry key\n");
   }
}                                               // end 'ntregSetCdromLetter'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get current drive-letter used for a CDROM device, from the registry
/*****************************************************************************/
BOOL ntregGetCdromLetter
(
   USHORT              cdnr,                    // IN    CDROM number
   char               *drive                    // OUT   drive letter
)
{
   BOOL                rc = FALSE;
   DWORD               disktype;
   HKEY                hDiskKey = NULL;
   LONG                status;
   TXTM                cdwanted;                // wanted device string
   ULONG               ValueSize;

   status = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
                          DFSNTREG_DISK_KEYNAME,
                          0,
                          KEY_READ | KEY_WRITE,
                          &hDiskKey);
   if (status == ERROR_SUCCESS)                 // key opened, read value
   {
      sprintf( cdwanted, DFSNTSYM_CDROM_DEVICE, cdnr);
      disktype  = REG_SZ;
      ValueSize = TXMAXTS;
      status = RegQueryValueEx( hDiskKey,
                                cdwanted,
                                0,
                                &disktype,
                               (PBYTE) drive,
                                &ValueSize );
      if (status == ERROR_SUCCESS)
      {
         rc = TRUE;                             // the value was found
      }
      RegCloseKey( hDiskKey );
   }
   return rc;
}                                               // end 'ntregGetCdromLetter'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Delete registry entry for specified CDROM device
/*****************************************************************************/
void ntregDelCdromKey
(
   USHORT              cdnr                     // IN    CDROM number
)
{
   HKEY                hDiskKey = NULL;
   DWORD               lpdwDisposition;
   char                KeyClass;
   LONG                status;
   TXTM                cdwanted;                // wanted device string

   KeyClass = REG_BINARY;
   status = RegCreateKeyEx( HKEY_LOCAL_MACHINE,
                            DFSNTREG_DISK_KEYNAME,
                            0,
                            &KeyClass,
                            REG_OPTION_NON_VOLATILE,
                            KEY_READ | KEY_WRITE,
                            NULL,
                            &hDiskKey,
                            &lpdwDisposition);
   if (status == ERROR_SUCCESS)
   {
      sprintf( cdwanted, DFSNTSYM_CDROM_DEVICE, cdnr);

      status = RegDeleteValue( hDiskKey, cdwanted);
      if (status != ERROR_SUCCESS)
      {
         TxPrint( "Error deleting '%s'\n", cdwanted);
      }
      RegCloseKey( hDiskKey );
   }
   else
   {
      TxPrint( "Error creating disk registry key\n");
   }
}                                               // end 'ntregDelCdromKey'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display complete registry DiskKey contents
/*****************************************************************************/
ULONG ntregShowDiskkey
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_NTREG_HEAD       *ntrHead;
   S_NTREG_INFO       *ntrInfo;
   S_NTREG_DISK       *ntrDisk;
   int                 i, j;

   ENTER();

   if (ntregDiskKeyFromReg())
   {
      ntrHead = (S_NTREG_HEAD *)  ntrBuff;
      ntrInfo = (S_NTREG_INFO *) &ntrBuff [ ntrHead->InfoOffset];
      ntrDisk = (S_NTREG_DISK *) &ntrInfo->Disk;

      TxPrint( "\nNumber of disks   : %u\n",     ntrInfo->NrDisks);

      for (i = 0; i < ntrInfo->NrDisks; i++)
      {
         TxPrint("\nPhysical disk nr  : %u\n", i);
         TxPrint(" Disk signature   : %8.8lX", ntrDisk->Signature);
         TxPrint("      partitions  : %u\n",   ntrDisk->NrParts);

         for (j = 0; j < ntrDisk->NrParts; j++)
         {
            TxPrint("\nPartition index   : %2.2u%22.22slog-nr: %u\n", j, "",
                              ntrDisk->Part[j].LogNumber);
            dfsX10( "   start sector   : ", (ntrDisk->Part[j].StartSect / dfsGetSectorSize()), CNN, "\n");
            dfsSz64("   partition size : ", (ntrDisk->Part[j].Length    / dfsGetSectorSize()),      "\n");
            TxPrint("   drive letter   : %c %22.22sForced: %s\n",
                              ntrDisk->Part[j].DrLetter ?
                              ntrDisk->Part[j].DrLetter : '-', "",
                              ntrDisk->Part[j].DrForced ? "True" : "False");
         }
         ntrDisk = ntregNextDisk( ntrDisk );
      }
      free( ntrBuff);                           // free registry-key info
      ntrBuff = NULL;
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'ntregShowDiskkey'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show table of assigned partition driveletters
/*****************************************************************************/
ULONG ntregShowDriveLetters
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_NTREG_HEAD       *ntrHead;
   S_NTREG_INFO       *ntrInfo;
   S_NTREG_DISK       *ntrDisk;
   int                 i, j;
   char                DrLetter[5];

   ENTER();

   if (ntregDiskKeyFromReg())
   {
      ntrHead = (S_NTREG_HEAD *)  ntrBuff;
      ntrInfo = (S_NTREG_INFO *) &ntrBuff [ ntrHead->InfoOffset];
      ntrDisk = (S_NTREG_DISK *) &ntrInfo->Disk;

      TxPrint("\nDisk  RPnr  Letter  Size Mb  DosDevice\n");
      TxPrint(  "====  ====  ======  =======  ============================\n");
      for (i = 0; i < ntrInfo->NrDisks; i++)
      {
         for (j = 0; j < ntrDisk->NrParts; j++)
         {
            if (ntrDisk->Part[j].DrForced == FALSE)
            {
               TxPrint("PD%2i   %2i   none    %7lu\n", i+1, j+1, (ntrDisk->Part[j].Length)>>20);
            }
            else
            {
               if (ntrDisk->Part[j].DrLetter != 0)
               {
                  TXTS letter;                  // drive letter string
                  TX1K device;                  // resulting string array

                  sprintf( letter, "%c:", ntrDisk->Part[j].DrLetter);
                  if (QueryDosDevice( letter, device, TXMAX1K) == 0)
                  {
                     strcpy( device, "not defined");
                  }

                  TxPrint("PD%2i   %2i     %c:    %7lu  %s\n", i+1, j+1, ntrDisk->Part[j].DrLetter,
                                                                    (ntrDisk->Part[j].Length)>>20, device);

               }
               else
               {
                  TxPrint("PD%2i   %2i   dynamic %7lu\n", i+1, j+1, (ntrDisk->Part[j].Length)>>20);
               }
            }
         }
         ntrDisk = ntregNextDisk( ntrDisk );
      }
      free( ntrBuff);                           // free registry-key info
      ntrBuff = NULL;
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
                                                // 
   for (i = 0; i < 9; i++)                      // add the CD-ROM drive letters
   {
      if (ntregGetCdromLetter((USHORT) i, DrLetter))
      {
         TXTS letter;                           // drive letter string
         TX1K device;                           // resulting string array

         memset( device, 0, TXMAX1K);
         sprintf( letter, "%c:", DrLetter[0]);
         if (QueryDosDevice( letter, device, TXMAX1K) == 0)
         {
            strcpy( device, "not defined");
         }
         TxPrint("CD%2i          %c:      CDROM  %s\n",
                  i+1,  DrLetter[0],           device);
      }
   }
   RETURN (rc);
}                                               // end 'ntregShowDriveLetters'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show present DosDevices, list limited by specified wildcard(s)
/*****************************************************************************/
void ntregShowDosDevice
(
   char               *dev,                     // IN    DosDevice   wildcard
   char               *map                      // IN    dev mapping wildcard
)
{
   TX1K                result;                  // resulting string array
   TX1K                mapping;                 // mapping string array
   TXTM                dwc;                     // device wildcard
   TXTM                mwc;                     // mapping wildcard
   char               *s;

   ENTER();

   strcpy( dwc, (strlen(dev) ? dev : "?:"));    // default to drive-letters
   strcpy( mwc, (strlen(map) ? map : "*"));     // on all possible devices

   if (QueryDosDevice( NULL, result, TXMAX1K) != 0)
   {
      TxPrint( "DosDevice query result for: '%s' and '%s'\n\n", dwc, mwc);
      for (s = result; strlen(s); s += strlen(s) +1)
      {
         if (QueryDosDevice( s, mapping, TXMAX1K) != 0)
         {
            if (TxStrWicmp(  s, dwc) == 0)      // wildcard compare
            {
               if (TxStrWicmp(  mapping, mwc) == 0)
               {
                  TxPrint( " %s% 32.32s%s = %s\n",
                           (s[1] == ':') ? CBG : CNN, s, CNN, mapping);
               }
            }
         }
      }
   }
   else
   {
      TxPrint("QueryDosDevice failed\n");
   }
   VRETURN ();
}                                               // end 'ntregShowDosDevice'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free a driveletter by removing the defined DosDevice for it
/*****************************************************************************/
BOOL ntregFreeDriveLetter
(
   char               *drive                    // IN    drive letter
)
{
   BOOL                rc = TRUE;               // function return
   TXTS                letter;                  // drive letter string
   TX1K                result;                  // resulting string array

   ENTER();
   TRACES(("Free letter: '%s'\n", drive));

   sprintf( letter, "%c:", drive);
   if (QueryDosDevice( letter, result, TXMAX1K) != 0)
   {
      rc = DefineDosDevice( DDD_RAW_TARGET_PATH, letter, "");
   }
   BRETURN (rc);
}                                               // end 'ntregFreeDriveLetter'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free DosDevice mapping(s) for specified DFS partition, when different
/*****************************************************************************/
BOOL ntregFreeWhenDifferent                     // RET   (re)assign required
(
   USHORT              disk,                    // IN    disk number
   USHORT              part,                    // IN    rel partition number
   char               *letter                   // IN    wanted driveletter
)
{
   BOOL                rc = TRUE;               // function return
   TXTM                partdev;                 // partition device string
   TX1K                result;                  // resulting string array
   TX1K                current;                 // device string
   char               *s;

   ENTER();
   TRACES(("Check part: %u on disk %u, for letter: %s\n", part, disk, letter));

   sprintf( partdev, DFSNTSYM_DPART_DEVICE, disk -1, part);

   if (QueryDosDevice( NULL, result, TXMAX1K) != 0)
   {
      for (s = result; strlen(s); s += strlen(s) +1)
      {
         if (QueryDosDevice( s, current, TXMAX1K) != 0)
         {
            if (strcasecmp( partdev, current) == 0)
            {
               if (toupper(*s) != toupper(*letter))
               {
                  TRACES(( "Freeing '%s' from '%s'\n", s, partdev));

                  DefineDosDevice( DDD_RAW_TARGET_PATH, s, "");
               }
               else                             // no update needed
               {
                  TRACES(( "No update needed: '%s' -> '%s'\n", s, partdev));
                  rc = FALSE;
               }
            }
         }
      }
   }
   BRETURN (rc);
}                                               // end 'ntregFreeWhenDifferent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Assign DosDevice mapping for specified CDROM number and drive letter
/*****************************************************************************/
BOOL ntregAssignPartition
(
   USHORT              disk,                    // IN    disk number
   USHORT              part,                    // IN    rel partition number
   char               *drive                    // IN    CDROM drive letter
)
{
   BOOL                rc = TRUE;               // function return
   TXTM                partdev;                 // partition device string
   TXTS                letter;                  // drive letter string

   ENTER();

   sprintf( letter, "%c:", drive[0]);
   sprintf( partdev, DFSNTSYM_DPART_DEVICE, disk -1, part);

   TRACES(( "Assign '%s' to '%s'\n", letter, partdev));
   rc = DefineDosDevice( DDD_RAW_TARGET_PATH,
                         letter, partdev);
   BRETURN (rc);
}                                               // end 'ntregAssignPartition'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free DosDevice mapping(s) for specified CDROM number
/*****************************************************************************/
BOOL ntregFreeCdrom
(
   USHORT              cdnr                     // IN    CDROM number
)
{
   BOOL                rc = TRUE;               // function return
   TXTM                cddev;                   // CDROM device string
   TX1K                result;                  // resulting string array
   TX1K                current;                 // current device string
   char               *s;

   ENTER();

   sprintf( cddev, DFSNTSYM_CDROM_DEVICE, cdnr);

   if (QueryDosDevice( NULL, result, TXMAX1K) != 0)
   {
      for (s = result; strlen(s); s += strlen(s) +1)
      {
         if (QueryDosDevice( s, current, TXMAX1K) != 0)
         {
            if (strcasecmp( cddev, current) == 0)
            {
               TRACES(( "Freeing '%s' from '%s'\n", s, cddev));

               rc = DefineDosDevice( DDD_RAW_TARGET_PATH, s, "");
            }
         }
      }
   }
   BRETURN (rc);
}                                               // end 'ntregFreeCdrom'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Assign DosDevice mapping for specified CDROM number and drive letter
/*****************************************************************************/
BOOL ntregAssignCdrom
(
   USHORT              cdnr,                    // IN    CDROM number
   char               *drive                    // IN    CDROM drive letter
)
{
   BOOL                rc = TRUE;               // function return
   TXTM                cddev;                   // CDROM device string
   TXTS                letter;                  // drive letter string

   ENTER();

   sprintf( letter, "%c:", drive[0]);
   sprintf( cddev, DFSNTSYM_CDROM_DEVICE, cdnr);

   TRACES(( "Assign '%s' to '%s'\n", letter, cddev));
   rc = DefineDosDevice( DDD_RAW_TARGET_PATH,
                         letter, cddev);
   BRETURN (rc);
}                                               // end 'ntregAssignCdrom'
/*---------------------------------------------------------------------------*/

#endif
