//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// NTFS disk structure definitions
//
// Author: J. van Wijk
//
// JvW  22-03-1998 Initial version, derived from DFSHPFS, MFTFILE definitions
// JvW  04-04-2017 Updated LSN to 64bit, MFT-numbers to 64bit
//
#ifndef    DFSNTFS_H
   #define DFSNTFS_H

#define NTLDR_LSN        ((ULN64)  1)           // LSN of first NTLDR sector
#define NTLDR_SIZE       ((ULONG) 15)           // SIZE in sectors for NTLDR
#define NTLDR_IMGNAME       "dfsntldr"          // Default imagefile name

#define SG_S_NTLDR     12
#define SV_S_NTLDR     (char)0x05,(char)0x00,(char)0x4e,(char)0x00, \
                       (char)0x54,(char)0x00,(char)0x4c,(char)0x00, \
                       (char)0x44,(char)0x00,(char)0x52,(char)0x00
#define SV_S_BOOTM     (char)0x07,(char)0x00,(char)0x42,(char)0x00, \
                       (char)0x4f,(char)0x00,(char)0x4f,(char)0x00, \
                       (char)0x54,(char)0x00,(char)0x4d,(char)0x00

typedef struct s_s_ntldr                        // fake superblock, NTLDR 1st
{
   BYTE                Signature[SG_S_NTLDR];   // 'NTLDR' / 'BOOTMGR' unicode
   BYTE                fill[500];
} S_S_NTLDR;                                    // end of struct "s_s_ntldr"

// Macros to split-up a 64bit MFT number into a pure-MFT and a 16-bit sequence number
#define dfsMftPure(x) ((ULN64) ((x) & 0xffffffffffffULL))
#define dfsMftSqnr(x) ((USHORT)((x) >> 48))

#define NTFS_INDX_BLOCKSIZE        4096         // default size INDX/RSTR


#define FIXUP_AREA_SIZE    512                  // 2 last bytes of each block
                                                // are fixed up by table value


#define MA_END_LIST         0xffffffff
#define MA_STANDARD         0x010
#define MA_ATTRLIST         0x020
#define MA_FILENAME         0x030
#define MA_VERSION          0x040               // W2K: $OBJECT_ID
#define MA_SECURITY         0x050
#define MA_VOLNAME          0x060
#define MA_VOLINFO          0x070
#define MA_DATA             0x080
#define MA_IDXROOT          0x090
#define MA_DIRINDX          0x0a0
#define MA_BITMAP           0x0b0
#define MA_SYMLINK          0x0c0               // W2K:  $REPARSE_POINT
#define MA_EAINFO           0x0d0
#define MA_EADATA           0x0e0
#define MA_PROPSET          0x0f0               // NT:  $PROPERTY_SET
#define MA_LOGUSTR          0x100               // W2K: $LOGGED_UTILITY_STREAM

#define MA_ALLATTRS         0xfff               // special query value

#define IA_ANY     (USHORT) 0xffff              // any attribute ID matches

#define AF_REC_INUSE        0x01                // MFT in use (not deleted)
#define AF_DIRECTORY        0x02                // directory file-record

#define FN_POSIX            0x00                // Link    style filename (posix)
#define FN_UNICO            0x01                // Unicode style filename
#define FN_DOS83            0x02                // DOS 8.3 style filename
#define FN_UNDOS            0x03                // DOS & Unicode filename

#define FA_COMPR            0x800               // FAT-attribute compression

#define MFT_MFT             0x00                // MFT number MFT itself
#define MFT_MFTMIRR         0x01                // MFT number MFT mirror
#define MFT_LOGFILE         0x02                // MFT number Logfile
#define MFT_VOLUME          0x03                // MFT number Volume file
#define MFT_ATTRDEF         0x04                // MFT number attrib defs
#define MFT_ROOT            0x05                // MFT number root directory
#define MFT_BITMAPF         0x06                // MFT number bitmap file
#define MFT_BOOT            0x07                // MFT number boot-area
#define MFT_BADCLUS         0x08                // MFT number bad-cluster map
#define MFT_QUOTA           0x09                // MFT number quota file
#define MFT_EXTEND          0x0b                // MFT number quota file
#define MFT_1STUSER         0x10                // MFT number first user file

                                                // MFT attribute flags
#define AT_COMPRESSED       0x0001
#define AT_ENCRYPTED        0x4000
#define AT_SPARSE           0x8000


#define SC_LOGRSTR     8                        // sectors in restart-record
#define SG_LOGRSTR     4
#define SV_LOGRSTR     (char)0x52,(char)0x53,(char)0x54,(char)0x52
typedef struct s_logrstr                        // LOG restart record
{
   char                Signature[SG_LOGRSTR];   // 000 Signature "RSTR"
   USHORT              FixupOffset;             // 004 offset fixup pattern   (1e)
   USHORT              FixupSize;               // 006 Size of fixup-list +1  (09)
   ULN64               LogSeqNr;                // 008 Logfile sequence number
   ULONG               SysPageSize;             // 010 system page,  #bytes (1000)
   ULONG               LogPageSize;             // 014 logrec page,  #bytes (1000)
   USHORT              LraOffset;               // 018 Offset to log records  (30)
   USHORT              LogMinor;                // 01A Minor version Logfile  (01)
   USHORT              LogMajor;                // 01C Major version Logfile  (01)
   USHORT              FixupPattern;            // 01E Current fixup pattern
   USHORT              FixupList[1];            // 020 Variable-size fixup-list
   //- followed by log records, typ starting at  0x030
} S_LOGRSTR;                                    // end of struct "s_logrstr"

#define NTFSLOG_CHANGES   0x0001                // transactions pending ??
#define NTFSLOG_UNMOUNT   0x0002                // FS/LOG are clean (unmounted)

#define NTFSLOG_LISTEMPTY 0xffff                // free/used client list empty

typedef struct s_lrarea                         // LOG restart area
{
   ULN64               CurrentLog;              // 000 current log record
   USHORT              Clients;                 // 008 nr of client records
   USHORT              FreeList;                // 00A clients in free-list
   USHORT              UsedList;                // 00C clients in used-list
   USHORT              LraFlags;                // 00E LOG restart related flags
   ULONG               LogSeqBits;              // 010 bits Log Sequence Nr  (29)
   USHORT              Lralength;               // 014 restart area length   (e0)
   USHORT              ClientOffset;            // 016 offset to 1st client  (40)
   ULN64               LogFileSize;             // 018 Logfile size in bytes
   ULONG               ulFill1;                 // 020 Uncertain, data length (0)
   USHORT              LraOffset;               // 024 Lra offset from start (30)
   USHORT              CliOffset;               // 026 Cli offset from Lra   (40)
} S_LRAREA;                                     // 028 end of struct "s_lrarea"

typedef struct s_lclient                        // log client record
{
   ULN64               OldLogSeq;               // 000 Oldest LogSeqNr for client
   ULN64               ActLogSeq;               // 008 Actual LogSeqNr for client
   USHORT              PrevClient;              // 010 Prev, (offset ?), or 0xFFFF
   USHORT              NextClient;              // 012 Next, (offset ?), or 0xFFFF
   ULONG               ulFill1;                 // 014 Uncertain, flags ?     (0)
   ULONG               ulFill2;                 // 018 Uncertain, flags ?     (0)
   ULONG               Namelength;              // 01C Perhaps namelength     (8)
   USHORT              Clientname[1];           // 020 Unicode Name    (N.T.F.S.)
} S_LCLIENT;                                    // end of struct "s_lclient"


#define SG_LOGRCRD     4
#define SV_LOGRCRD     (char)0x52,(char)0x43,(char)0x52,(char)0x44
typedef struct s_logrcrd                        // LOG record page
{
   char                Signature[SG_LOGRSTR];   // 000 Signature "RSTR"
   USHORT              FixupOffset;             // 004 offset fixup pattern   (1e)
   USHORT              FixupSize;               // 006 Size of fixup-list +1  (09)
   ULN64               LogSeqNr;                // 008 Logfile sequence number
   ULONG               CrdFlags;                // 010 LOG record related flags
   //- to be refined
} S_LOGRCRD;                                    // end of struct "s_logrcrd"


#define SG_MFTFILE     4
#define SV_MFTFILE     (char)0x46,(char)0x49,(char)0x4c,(char)0x45
typedef struct s_mftfile                        // MFT File table entry
{
   char                Signature[SG_MFTFILE];   // 000 Signature "FILE"
   USHORT              FixupOffset;             // 004 offset fixup pattern  (30)
   USHORT              FixupSize;               // 006 Size of fixup-list +1 (03)
   ULN64               LogSeqNr;                // 008 Logfile sequence number
   USHORT              Sequence;                // 010 sequence nr in MFT
   USHORT              HardLinks;               // 012 Hard-link count
   USHORT              AttribOffset;            // 014 Offset to seq of Attributes
   USHORT              MftFlags;                // 016 0x01 = NonRes; 0x02 = Dir
   ULONG               RecLength;               // 018 Real size of the record
   ULONG               AllLength;               // 01c Allocated size of the record
   ULN64               BaseMftRec;              // 020 base MFT rec or 0, incl seq
   USHORT              NextAttributeId;         // 028 Next attribute Id to use
   USHORT              Align4bytes;             // 02A XP only: align to 4 bytes
   ULONG               MftRecNumber;            // 02C XP only: MFT record number
                                                //     This pos pointed to by (1)
   USHORT              FixupPattern;            // 030 Current fixup pattern
   USHORT              FixupList[1];            // 032 Variable-size fixup-list
   //- followed by resident and part of non-res attributes
} S_MFTFILE;                                    // end of struct "s_mftfile"

//- Note that attributes are aligned to 8 bytes (64bit values) by the NTFS IFS

typedef struct s_resident
{
   ULONG               Size;                    // 000 Size of resident attribute
   USHORT              Offset;                  // 004 offset specific value part
   USHORT              IndexFlag;
} S_RESIDENT;                                   // end of struct "s_resident"

typedef struct s_nonres
{
   ULN64               SegFirst;                // 000 first LCN in this segment
   ULN64               SegLast;                 // 008 last  LCN in this segment
   USHORT              Offset;                  // 010 Offset to the run-list
   USHORT              ComprUnitSize;           // 012 2^CUS is #clusters / unit
   USHORT              usFil2;                  // 014
   USHORT              usFil3;                  // 016
   ULN64               Allocated;               // 018 Allocated disk space
   ULN64               Size;                    // 020 Size of uncompressed attrib
   ULN64               Compressed;              // 028 Compressed size of attribute
} S_NONRES;                                     // 030 end of struct "s_nonres"


typedef struct s_mftattr                        // MFT attribute header part
{
   ULONG               Type;                    // 000 type of attribute MA_...
   USHORT              Length;                  // 004 length of this attribute
   USHORT              usFil2;                  // 006 (used on some dir bitmaps)
   BYTE                DataExtern;              // 008 0 = resident, 1 in Runs
   BYTE                NameLen;                 // 009 Length of name (if used)
   USHORT              Offset;                  // 00A to name or resident data
   USHORT              AttFlags;                // 00C compressed/encrypted etc
   USHORT              Ident;                   // 00E id, making it unique in
   union                                        // 010 combination with Type
   {
      S_RESIDENT       R;                       // 010 Resident attribute
      S_NONRES         N;                       // 010 Non-resident attribute
   };                                           // end of union
   USHORT              Fixup[1];                // variable-size name/res-data
} S_MFTATTR;                                    // end of struct "s_mftattr"

#define IDX_BTREE_PTR   0x01
#define IDX_BTREE_END   0x02


//- Note: Mftnr holds sequence-number in the upper two bytes! Use dfsMftPure(x)
typedef struct s_direntry
{
   ULN64               Mftnr;                   // 000 MFT record number
   USHORT              Length;                  // 008 entry size
   USHORT              Magic;                   // 00A size value, < Length
   BYTE                DirFlags;                // 00C 1 = subnode; 2 = last
   BYTE                fil2[3];                 // 00D
   ULN64               Parent;                  // 010 Parent MFT record number
   NTIME               Create;                  // 018 Creation time
   NTIME               Modify;                  // 020 Last Modify time
   NTIME               Modfil;                  // 028 Last modify of record
   NTIME               Access;                  // 030 Last Access time
   ULN64               Alloc;                   // 038 Allocated size
   ULN64               Size;                    // 040 Data real size
   ULN64               fil3;                    // 048
   BYTE                NameLen;                 // 050 Name length in words
   BYTE                FileNameType;            // 051 Type of filename 8.3 / UNI
   USHORT              Filename[1];             // 052 Variable length filename
   ULONG               BtreeOrdinal;            // relative cluster in the
                                                // index-allocation (attr A0)
                                                // when this is a Btree-ptr
                                                // (DirFlags bit 1 is set)
} S_DIRENTRY;                                   // end of struct "s_direntry"


typedef struct s_sii_sec                        // usual size is 0x28 bytes
{
   USHORT              data1;
   USHORT              data2;                   // unknown reference into data
   ULONG               data3;                   // often zero
   USHORT              Length;                  // entry size
   USHORT              Magic;                   // often 0x0004
   BYTE                DirFlags;                // 1 = subnode; 2 = last
   BYTE                fil2[3];
   ULONG               data[4];                 // core data

   ULONG               BtreeOrdinal;            // relative cluster in the
                                                // index-allocation (attr A0)
                                                // when this is a Btree-ptr
                                                // (DirFlags bit 1 is set)
} S_SII_SEC;                                    // end of struct "s_sii_sec"

typedef struct s_sdh_sec                        // usual size is 0x30 bytes
{
   USHORT              data1;
   USHORT              data2;                   // unknown reference into data
   ULONG               data3;                   // often zero
   USHORT              Length;                  // entry size
   USHORT              Magic;                   // often 0x0008
   BYTE                DirFlags;                // 1 = subnode; 2 = last
   BYTE                fil2[3];
   ULONG               data[6];                 // core data

   ULONG               BtreeOrdinal;            // relative cluster in the
                                                // index-allocation (attr A0)
                                                // when this is a Btree-ptr
                                                // (DirFlags bit 1 is set)
} S_SDH_SEC;                                    // end of struct "s_sdh_sec"

//- BtreeOrdinal can be present in 'empty' entries (size 0x10 or 0x18)
//- or real filename entries. In all cases the offset in the structure
//- for this field is entry->Length -8


//- file attribute definitions for STANDARD and FILENAME attributes
#define FA_READONLY                0x00000001
#define FA_HIDDEN                  0x00000002
#define FA_SYSTEM                  0x00000004
#define FA_ARCHIVE                 0x00000020
#define FA_DEVICE                  0x00000040
#define FA_NORMAL                  0x00000080

#define FA_TEMPORARY               0x00000100
#define FA_SPARSE_FILE             0x00000200
#define FA_REPARSE_POINT           0x00000400
#define FA_COMPRESSED              0x00000800

#define FA_OFFLINE                 0x00001000
#define FA_NOT_CNT_INDXD           0x00002000
#define FA_ENCRYPTED               0x00004000

#define FA_NAME_IDX_PRES           0x10000000   // in Filename attr only
#define FA_VIEW_IDX_PRES           0x20000000   // Fname, is a directory

typedef struct a_standard
{
   NTIME               Create;                  // Creation time
   NTIME               Modify;                  // Last Modify time
   NTIME               Modfil;                  // Last modify of record
   NTIME               Access;                  // Last Access time
   ULONG               FileAttributes;          // As FAT + 0x800 = compressed
   ULONG               ReservedField;           // unknown, could be, see below ...
} A_STANDARD;                                   // end of struct "a_standard"

//- Note: Found 'WofCompressedData' files, that have the 'ReparsePoint' and 'sparse' flag
//-       set in the standard attribute as well as the filename attribute, however,
//-       the ReparsePointTag in the filename attribute is not displayed for some reason.
//-       Perhaps it is in the standard attribute only ? Shown as a 'reserved' field) ...

typedef struct a_attrelem                       // element of an attrlist
{
   ULONG               Type;                    // type of attribute MA_...
   USHORT              RecLength;               // 0x20 if no name
   BYTE                NameLen;                 // 0 if no name present
   BYTE                unknown;                 // often 0x1a, offset to name
   ULN64               StartVcn;                // LCN for first VCN in attr
   ULN64               MftRecordNr;             // MFT rec-nr holding attrib
   USHORT              Ident;                   // id, attribute identificator
   USHORT              Name;                    // 1st Unicode name character
   USHORT              Name1;                   // 2nd Unicode name character
   USHORT              Name2;                   // 3rd Unicode name character
} A_ATTRELEM;                                   // end of struct "a_attrelem"

#define NTFS_SANE_ATTRLIST   1024               // max attra in 'sane' list
                                                // arbitray limit, largest
                                                // list seen had 130 entries

typedef struct a_attrlist                       // in base/continuation MFT
{
   A_ATTRELEM          attr[1];                 // array of attribute elements
} A_ATTRLIST;                                   // end of struct "a_attrlist"

typedef struct a_filename
{
   ULN64               MftParentMft;            // Parent-dir MFT entry
   NTIME               Create;                  // Creation time
   NTIME               Modify;                  // Last Modify time
   NTIME               Modfil;                  // Last modify of record
   NTIME               Access;                  // Last Access time
   ULN64               Allocated;               // Allocated disk space
   ULN64               RealSize;                // Size of the attribute
   ULONG               FileAttributes;          // File attribute flags
   union
   {
      USHORT           EaSize;                  // packed extended-attr size
      ULONG            ReparsePointTag;         // type of reparse point
   };                                           // end of union ""
   BYTE                FileNameLength;
   BYTE                FileNameType;            // 8.3 / Unicode
   USHORT              Filename[1];             // Name (in Unicode ?)
} A_FILENAME;                                   // end of struct "a_filename"

typedef struct a_version
{
   BYTE                contents;
}  A_VERSION;                                   // end of struct "a_version"

typedef struct a_security
{
   BYTE                contents;
} A_SECURITY;                                   // end of struct "a_security"

#define NTFS_LEN_LBL         32                 // by trial and error (on XP)
typedef struct a_volname
{
   USHORT              VolumeName[1];           // Volume name in Unicode
} A_VOLNAME;                                    // end of struct "a_volname"


                                                // Note: Dirty does NOT mean
//- Volume information flag values              // the FS is currently in use
#define VF_VOLUME_IS_DIRTY         0x0001       // force CHKDSK on next boot!
#define VF_RESIZE_LOG_FILE         0x0002
#define VF_UPGRADE_ON_MOUNT        0x0004
#define VF_MOUNTED_ON_NT4          0x0008
#define VF_DEL_USN_UNDERWAY        0x0010
#define VF_REPAIR_OBJ_ID           0x0020
#define VF_MODIF_BY_CHKDSK         0x8000
#define VF_UNKNOWN_VALUE           0x7fc0

//- Versions known: 1.2 NT4 SP6, 3.0 W2K, 3.1 XP
typedef struct a_volinfo
{
   ULONG               Fil1;                    // reserved
   ULONG               Fil2;                    // reserved
   BYTE                MajorVer;
   BYTE                MinorVer;
   USHORT              VolFlags;                // Volume flags
} A_VOLINFO;                                    // end of struct "a_volinfo"

typedef struct a_data                           // main data attribute
{
   BYTE                contents;
} A_DATA;                                       // end of struct "a_data"

#define INDX_I30_DIR   0x30
#define INDX_SDH_SEC   0x00                     // to be refined, value is 0
#define INDX_SII_SEC   0x00                     // may not be the real Id ...

#define IHDR_SDH_IND   0x08                     // value  SDH indicator
#define IHDR_SII_IND   0x04                     // value  SII indicator

typedef struct a_idxroot
{
   ULONG               Id;                      // 0x30 = DIR    00 $Secure
   ULONG               ulOne;                   // 0x01 for DIR, 00 $Secure
   ULONG               Size;
                                                // rest for DIR type only!
   ULONG               Clusters;                // clusters per index ???
   ULONG               I10;                     // allways 0x10 ???
   ULONG               EntrySize1;              // size of entry - 0x10 ???
   ULONG               EntrySize2;              // size of entry - 0x10 ???
   USHORT              usOne;                   // allways 0x01
   USHORT              IdxFlags;                // ???
   S_DIRENTRY          EntryList[1];            // list of directory entries
} A_IDXROOT;                                    // end of struct "a_idxroot"

#define SG_DIRINDX     4
#define SV_DIRINDX     (char)0x49,(char)0x4e,(char)0x44,(char)0x58
typedef struct s_dirindx
{
   BYTE                Signature[SG_DIRINDX];   // 'INDX'
   USHORT              FixupOffset;             // usualy 0x28
   USHORT              FixupSize;               // number of fixups +1
   USHORT              hf1;                     // unknown, but varying
   USHORT              hf2;                     // unknown, but varying
   ULONG               hf3;                     // unknown, usually 0
   ULN64               BufferVcn;               // unknown, usually 0
   ULONG               HeaderSize;              // size to first entry
   ULONG               InUseLength;             // excluding header ?
   ULONG               TotalLength;             // usualy 0x0fe8
   ULONG               ulFill;                  // unknown, usually 0
   USHORT              Fixup[0xc];              // Size varies! (maybe :-)
   S_DIRENTRY          EntryList[1];            // list of directory entries
} S_DIRINDX;                                    // end of struct "s_dirindx"

typedef struct a_bitmap
{
   BYTE                contents;
} A_BITMAP;                                     // end of struct "a_bitmap"

typedef struct a_symlink
{
   BYTE                contents;
} A_SYMLINK;                                    // end of struct "a_symlink"

typedef struct a_eainfo
{
   BYTE                contents;
} A_EAINFO;                                     // end of struct "a_eainfo"


typedef struct a_eadata
{
   BYTE                contents;
} A_EADATA;                                     // end of struct "a_eainfo"


typedef union u_mftattr                         // Attr specific value parts
{
   A_STANDARD       Stand;
   A_ATTRLIST       Alist;
   A_FILENAME       Fname;
   A_VERSION        Vvers;
   A_SECURITY       Secur;
   A_VOLNAME        Vname;
   A_VOLINFO        Vinfo;
   A_DATA           Data;
   A_IDXROOT        Index;
   S_DIRINDX        Idxal;
   A_BITMAP         Bitmp;
   A_SYMLINK        Slink;
   A_EAINFO         Eainfo;
   A_EADATA         Eadata;
} U_MFTATTR;                                    // end of union "u_mftattr"
#endif
