//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee Arealist, SN-list + File-recovery functions
//
// Author: J. van Wijk
//
// JvW  18-08-2005 Initial version, split off from DFSUTIL.C
// JvW  25-01-2018 Renamed to dfsnlist
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsupart.h>                           // PART utility functions
#include <dfsafdsk.h>                           // FDISK analysis functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT  utility functions
#include <dfsdgen.h>                            // DFS generic dialogs


#include <dfsaext.h>                            // EXT  superblock display
#include <dfsahpfs.h>                           // HPFS superblock display
#include <dfsahfs.h>                            // HFS  superblock display
#include <dfsaapfs.h>                           // APFS superblock display
#include <dfsajfs.h>                            // JFS  superblock display
#include <dfsantfs.h>                           // NTFS superblock display
#include <dfsarsr.h>                            // RSR  superblock display
#include <dfsaxfs.h>                            // XFS  superblock display


#if defined (WIN32)
   #include <dfsntreg.h>                        // NT error detail reporting
#endif

// Compare LSN elements for qsort
static int dfsLsnCompare
(
   const void         *one,
   const void         *two
);

// Compare LSN elements for qsort descending
static int dfsLsnCompDes
(
   const void         *one,
   const void         *two
);


// Recover multiple files, from the list of sectornumbers in the SNLIST
static ULONG dfsRecoverMulti
(
   char               *path,                    // IN    destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
);

// DFS recover single directory (contents) by recursing into it
// Saves the current SNLIST, handles DIR, then restores original SNLIST
static ULONG dfsRecursiveDirRecover
(
   ULN64               fn,                      // IN    LSN for file to recover
   ULN64               info,                    // IN    meta info for the LSN
   char               *path,                    // IN    base destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
);

// Multi-Recovery status information, mainly for progress and result reporting
static BOOL            mrAborted;               // Multi-Recover session aborted
static BOOL            mrStatus;                // New status update desirable
static ULONG           mrHandled;               // Total nr of handled   items
static ULONG           mrFailed;                // Total nr of failed    items
static ULONG           mrRecovered;             // Total nr of recovered items
                                                // (skipped is mrH - mrF - mrR)

/*****************************************************************************/
// Sort SN-table ascending or descending, optionaly make each entry unique
/*****************************************************************************/
void dfsSortSectorList
(
   ULN64              *list,                    // IN    count + list of sns
   char               *options                  // IN    sort-order, unique
)
{
   if (*list != 0)
   {
      qsort(&list[1], (size_t) *list, sizeof(ULN64),
         (strchr(options, '-') != NULL) ? dfsLsnCompDes : dfsLsnCompare);
      if (strchr(options, 'u') != NULL)
      {
         ULONG               i;                 // index complete list
         ULONG               u;                 // index unique entries

         for (i = 2, u = 1; i <= *list; i++)
         {
            if (list[i] != list[u])
            {
               list[++u] = list[i];
            }
         }
         *list = u;
      }
   }
}                                               // end 'dfsSortSectorList'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Compare LSN elements for qsort ascending
/*****************************************************************************/
static int dfsLsnCompare
(
   const void         *one,
   const void         *two
)
{
   int                 rc = 0;

   if      ((*(ULN64 *) one) < (*(ULN64 *) two))
   {
      rc = -1;
   }
   else if ((*(ULN64 *) one) > (*(ULN64 *) two))
   {
      rc = +1;
   }
   return(rc);
}                                               // end 'dfsLsnCompare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare LSN elements for qsort descending
/*****************************************************************************/
static int dfsLsnCompDes
(
   const void         *one,
   const void         *two
)
{
   int                 rc = 0;

   if      ((*(ULN64 *) one) < (*(ULN64 *) two))
   {
      rc = +1;
   }
   else if ((*(ULN64 *) one) > (*(ULN64 *) two))
   {
      rc = -1;
   }
   return(rc);
}                                               // end 'dfsLsnCompDes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Export list of sectornumbers in the SN-table to a file, incl snifsmode
/*****************************************************************************/
ULONG dfsExportSectorList                       // RET   function result
(
   char               *fname,                   // IN    Filename for LSN list
   char               *dname                    // IN    Filename for data file
)
{
   ULONG               rc   = NO_ERROR;
   ULN64              *list = dfsa->snlist;     // count + list of sns
   ULN64               sn;
   FILE               *fh;

   ENTER();

   if ((fh = fopen(fname, "w")) == 0)
   {
      TxPrint("\nError opening lsn-file : '%s' for write\n", fname);
      rc = ERROR_OPEN_FAILED;
   }
   else
   {
      TxPrint("\nWriting LSNs  to  : %s, press <Esc> to abort.\n", fname);
      fprintf( fh, "%s%s\n",  DFSNIFSMODE, dfsa->snifsmode);
      if (dfsa->snifsmcmd[0] != 0)              // when defined for this FS
      {
         fprintf( fh, "%s%s\n",  DFSNIFSMCMD, dfsa->snifsmcmd);
      }
      fprintf( fh, "%s%s\n",  DFSNLISTCOM, dfsa->sncompact);
      fprintf( fh, "%s%s\n",  DFSNLISTVER, dfsa->snverbose);
      fprintf( fh, "%s%s\n",  DFSNLISTDRV, SINF->drive);
      fprintf( fh, "%s%hu\n", DFSNLISTDSK, SINF->disknr);
      fprintf( fh, "%s%hu\n", DFSNLISTPAR, SINF->partid);
      for (sn = 1; (sn <= *list) && (!TxAbort()); sn++)
      {
         fprintf( fh, "0x0%llX 0x%4.4hX\n", list[sn], dfsa->sninfo[sn]);
      }
      fclose( fh);

      TxPrint("\nFinished writing %llu LSN(s) to %s\n", *list, fname);

      if (strlen(dname))                        // data file too
      {
         if ((fh = fopen( dname, "wb")) == 0)
         {
            TxPrint("Error opening data-file : '%s' for write\n", dname);
            rc = ERROR_OPEN_FAILED;
         }
         else
         {
            size_t        bps = (size_t) dfsGetSectorSize();
            TXTIMER       stime = TxTmrGetNanoSecFromStart();

            TxPrint("\nExporting data to : %s, press <Esc> to abort.\n", dname);
            if (dfsa->verbosity != TXAO_QUIET)
            {
               dfsProgressInit( 0, *list, 0, "Sector:", "Exported", DFSP_BARS, 0);
            }

            for (sn = 1; (sn <= *list) && (rc == NO_ERROR) && (!TxAbort()); sn++)
            {
               rc = dfsRead( list[sn], 1, rbuf);
               if (rc == NO_ERROR)
               {
                  if (fwrite( rbuf, 1, bps, fh) == bps)
                  {
                     if (dfsa->verbosity != TXAO_QUIET)
                     {
                        dfsProgressShow( sn + 1, 0, 0, NULL);
                     }
                  }
                  else
                  {
                     rc = ERROR_WRITE_FAULT;
                  }
               }
            }

            if (dfsa->verbosity != TXAO_QUIET)
            {
               dfsProgressTerm();
               dfsDisplayThroughput( stime, sn, dfsGetSectorSize());
            }
            fclose( fh);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsExportSectorList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Import list of sectornumbers to the SN-table from a file, incl snifsmode
/*****************************************************************************/
ULONG dfsImportSectorList                       // RET   function result
(
   char               *fname,                   // IN    Filename for LSN list
   char               *dname                    // IN    Filename for data file
)
{
   ULONG               rc   = NO_ERROR;
   ULN64              *list = dfsa->snlist;     // count + list of sns
   ULN64               sn = 1;
   FILE               *fh;
   TXLN                line;
   char               *lsn;                     // ptr to valid LSN in text

   ENTER();

   if ((fh = fopen(fname, "r")) == 0)
   {
      TxPrint("\nError opening lsn-file : '%s' for read\n", fname);
      rc = ERROR_OPEN_FAILED;
   }
   else
   {
      TxPrint("\nReading LSNs from : %s, press <Esc> to abort.\n", fname);
      while (!TxAbort()   &&                    // no user abort
             !feof(   fh) &&                    // no end of file
             !ferror( fh) &&                    // and no error
              DFSNL_ROOM)                       // and room in the list
      {
         if (fgets( line, TXMAXLN, fh) != NULL)
         {
            if (line[strlen(line) -1] == '\n')
            {
               line[strlen(line) -1] = 0;
            }
            switch (line[0])
            {
               case '0':                        // hex sectornr at start
                  lsn = line;                   // start from begin of line
                  break;

               case '.':
                  //- to be refined 64bit and 7 digit .NNN value, make more generic
                  if (line[29] == '=')          // standard list format
                  {
                     lsn = &line[20];
                  }
                  else                          // list -f format
                  {
                     for (lsn = line; *lsn != ' '; lsn++)
                     {
                     }
                     while (*lsn == ' ')        // find start of 2nd word on line
                     {
                        lsn++;
                     }
                  }
                  break;

               default:
                  if      (strncasecmp( line, DFSNIFSMODE, strlen(DFSNIFSMODE)) == 0)
                  {
                     strcpy( dfsa->snifsmode, line + strlen(DFSNIFSMODE));
                     TxPrint( "Original Filesystem mode : %s\n", dfsa->snifsmode);
                  }
                  else if (strncasecmp( line, DFSNIFSMCMD, strlen(DFSNIFSMCMD)) == 0)
                  {
                     strcpy( dfsa->snifsmcmd, line + strlen(DFSNIFSMCMD));
                     TxPrint( "Filesystem init command  : %s%s\n",
                                                 dfsa->snifsmcmd,
                                    (strcasecmp( dfsa->snifsmode, SINF->afsys) == 0) ?
                                           "    (auto executed now)" : "");
                     if ((strlen(     dfsa->snifsmcmd)) &&
                         (strcasecmp( dfsa->snifsmode, SINF->afsys) == 0))
                     {
                        dfsMultiCommand( dfsa->snifsmcmd, 0, FALSE, FALSE, FALSE);
                     }
                  }
                  else if (strncasecmp( line, DFSNLISTCOM, strlen(DFSNLISTCOM)) == 0)
                  {
                     strcpy( dfsa->sncompact, line + strlen(DFSNLISTCOM));
                  }
                  else if (strncasecmp( line, DFSNLISTVER, strlen(DFSNLISTVER)) == 0)
                  {
                     strcpy( dfsa->snverbose, line + strlen(DFSNLISTVER));
                  }
                  else if (strncasecmp( line, DFSNLISTDRV, strlen(DFSNLISTDRV)) == 0)
                  {
                     TxPrint( "Original IMG/driveletter : %s\n", line + strlen(DFSNLISTDRV));
                  }
                  else if (strncasecmp( line, DFSNLISTDSK, strlen(DFSNLISTDSK)) == 0)
                  {
                     TxPrint( "Original selected disknr : %s\n", line + strlen(DFSNLISTDSK));
                  }
                  else if (strncasecmp( line, DFSNLISTPAR, strlen(DFSNLISTPAR)) == 0)
                  {
                     TxPrint( "Original partition (PID) : %s\n", line + strlen(DFSNLISTPAR));
                  }
                  lsn = NULL;
                  break;
            }
            if (lsn != NULL)
            {
               dfsa->sninfo[sn] = 0;
               if (sscanf( lsn, "%llX %hX", &(list[sn]), &(dfsa->sninfo[sn])) >= 1)
               {
                  sn++;
               }
            }
         }
      }
      *list = sn -1;
      fclose( fh);
      TxPrint("\nFinished reading %llu LSN(s) from %s\n", *list, fname);
      if (!DFSNL_ROOM)
      {
         TxPrint("Import aborted, size of internal list exceeded\n");
      }

      if (strlen(dname))                        // data file too
      {
         if ((fh = fopen( dname, "rb")) == 0)
         {
            TxPrint("\nError opening data-file : '%s' for read\n", dname);
            rc = ERROR_OPEN_FAILED;
         }
         else if (dfstStoreType( DFSTORE) != DFST_UNUSED)
         {
            char   sizetext[ 32];               // text buffer

            sizetext[0] = 0;
            dfstrSz64( sizetext, "", dfsa->snlist[0], "");

            if ((dfsa->batch) ||                // forced or confirmed
                (TxConfirm( 5007,
               "Write %s sectors from data-file %s\nto %s\n"
               "using the sectornumbers from %s ? [Y/N] : ",
                sizetext, dname, dfstStoreDesc1( DFSTORE), fname)))
            {
               if (DFSTORE_WRITE_ALLOWED)
               {
                  size_t     bps = (size_t) dfsGetSectorSize();
                  TXTIMER    stime = TxTmrGetNanoSecFromStart();

                  TxPrint("\nImporting    from : %s, press <Esc> to abort.\n", dname);
                  if (dfsa->verbosity != TXAO_QUIET)
                  {
                     dfsProgressInit( 0, *list, 0, "Sector:", "Imported", DFSP_BARS, 0);
                  }

                  for (sn = 1; (sn <= *list) && (rc == NO_ERROR) && (!TxAbort()); sn++)
                  {
                     if (fread( rbuf, 1, bps, fh) == bps)
                     {
                        rc = dfsWrite( list[sn], 1, rbuf);
                        if (dfsa->verbosity != TXAO_QUIET)
                        {
                           dfsProgressShow( sn + 1, 0, 0, NULL);
                        }
                     }
                     else
                     {
                        rc = DFS_BAD_STRUCTURE; // data file too short
                     }
                  }

                  if (dfsa->verbosity != TXAO_QUIET)
                  {
                     dfsProgressTerm();
                     dfsDisplayThroughput( stime, sn, dfsGetSectorSize());
                  }
               }
               else
               {
                  rc = DFS_READ_ONLY;
               }
            }
            else
            {
               TxPrint("\nImport from '%s' aborted, no changes made\n", dname);
            }
            fclose( fh);
         }
         else
         {
            TxPrint( "\nNo volume, partition or disk opened in default "
                     "store (%hu), use the VOL, PART\nor DISK cmd to "
                     "select one before using the '-data' option on ",
                     "import/export!", DFSTORE);
            rc = DFS_CMD_FAILED;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsImportSectorList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add a value to the standard dfs sector-number list
/*****************************************************************************/
ULONG dfsAdd2List
(
   ULN64               sn,                      // IN    sector-nr to add
   ULN64               info,                    // IN    additional info
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT dummy
)
{
   ULONG               rc    = NO_ERROR;
   ULN64               index = dfsa->snlist[0] +1; // first is the count!

   ENTER();
   TRACES(("At .%07llu add 0x%llx-%x\n", dfsa->snlist[0], sn, info));

   if ((index < LSIZE) && dfsa->snlact)
   {
      dfsa->snlist[index] = sn;
      dfsa->sninfo[index] = info;
      dfsa->snlist[0]     = index;
   }
   else
   {
      rc = DFS_CMD_FAILED;
   }
   RETURN (rc);
}                                               // end 'dfsAdd2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// (Re) Initialize the standard dfs sector-number list to a specific size
/*****************************************************************************/
ULONG dfsInitList
(
   ULN64               size,                    // IN    requested size
   char               *compact,                 // IN    compact display mode
   char               *verbose                  // IN    verbose display mode
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();
   TRACES(( "Initialize SN-list to %u items, compact:%s verbose:%s\n", size, compact, verbose));

   if ((size < LSIZE) && dfsa->snlact)
   {
      TRLEVX(200,("SnListFlag now: %llu => 0\n", DFSBR_SnlistFlag));
      DFSBR_SnlistFlag = 0;                     // list flags in sninfo[0]
      dfsa->brlist = 0;                         // Reset 'this' reference (browse)
      dfsa->brinfo = 0;
      strcpy( dfsa->sncompact, compact);
      strcpy( dfsa->snverbose, verbose);        // optimal list options
      strcpy( dfsa->snifsmode, SINF->afsys);    // initial FS mode
      if (dfsa->snifsmcmd[0] != 0)
      {
         //- to be refined ???
      }
      dfsa->snlist[0] = size;
      memset( dfsa->sninfo, 0, dfsa->snsize * sizeof( USHORT)); // clear additional info
   }
   else
   {
      rc = DFS_CMD_FAILED;
   }
   RETURN (rc);
}                                               // end 'dfsInitList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display of sectornumbers in the SN-list in format specified by options
/*****************************************************************************/
void dfsDisplaySectorList
(
   char               *select                   // IN    path+filename wildcard
)
{
   ULONG               nr;
   ULN64              *list   = dfsa->snlist;   // the list itself
   BYTE                st;                      // type of sector
   ULONG               ul     = 0;              // SLT size, or #ranges
   BOOL                slt    = FALSE;          // SLT info available
   ULN64               ref    = 0;              // reference value
   ULN64               last   = list[1];        // anchor value
   ULN64               rangCl = dfsSn2Cl(last); // start of cluster range (-r)
   ULN64               lastCl = rangCl;         // last cluster in this range
   ULN64               clust;                   // cluster value
   TXLN                txt;
   TXLN                selstr;                  // select IN, str OUT
   TXTM                tbuf;                    // temporary text buffer
   ULONG               selected = 0;            // nr of selected sectors
   ULONG               rc = DFS_PENDING;        // handler result
   BOOL                newstatus = FALSE;
   TXLN                string;
   TXLN                htext;                   // header text
   TXLN                hline;                   // header underline
   TXTS                tinfo;                   // string representation info field
   USHORT              binfo;                   // number representation info field

   ENTER();

   strcpy( string, select);
   #if defined (USEWINDOWING)
   if ((TxaOption('f') || TxaOption('S')) && TxaOption('P'))        // explicit prompting
   {
      if ((dfsFileListDialog( " Specify selection critera for files to be LISTed ",
                               *list, FALSE, string)) != NO_ERROR)
      {
         VRETURN();
      }
   }
   #endif
   strcpy( htext, "");
   strcpy( hline, "");
   if (!TxaOption('w') && !TxaOption('r'))
   {
      TxPrint("\n");
      if (dfsSlTableStatus(&ul) == SLT_READY)
      {
         slt = TRUE;                            // use SLT info for sector type
      }
      if      (TxaOption('f'))                  // filename + path based
      {
         sprintf(htext, " Index Value        Info T Perc %s Size     Ea/Xa/Rf Path + Filename ...",
                ((dfsa->FsModeId != DFS_FS_E_FAT) &&
                 (dfsa->FsModeId != DFS_FS_FAT)) ? "  Location" :
                 TxaOption('I') ? TxaOption('C') ? "RelCluster" :
                                                   "  EA-index" :
                                  TxaOption('C') ? " ClusterNr" :
                                                   "  SectorNr");
         strcpy( hline, " ===== ============ ==== = ==== ========== ======== ======== ==========================");
      }
      else if (TxaOption('g'))                  // GEO translated values
      {
         strcpy( htext, " Index Translation   Value       24-Bit C-H-S      Cylinder Head Sec          PSN");
         strcpy( hline, " ===== ===========   ==========  ============   ============ === ===   ==========");

      }
      else if (TxaOption('s') ||                // show delta-sizes & type from list
               TxaOption('t')  )                // show delta-sizes & type from sector
      {
         strcpy( htext, " Index Delta-Size   Value        Sector-type        Cylinder Head Sec        PSN Block/ClusterNr");
         strcpy( hline, " ===== ===========  ==========   ================= ========== === === ========== ================");
      }
      else if (!TxaOption('b') && !TxaOption('c') && !TxaOption('d') && !TxaOption('S'))
      {
         sprintf( htext, " Index DecimalValue   HexValue   %s"        "       Cylinder Head Sec        PSN Block/ClusterNr",
                          (dfsa->sninfo[1] & DFSSNINFO) ? "Entry/Record"
                                                        : "Sector-type ");
         strcpy(  hline, " ===== ============ ==========   ================= ========== === === ========== ================");
      }
   }
   else if (TxaOption('r'))                     // can't find diff between W and R
   {                                            // but there is an extra \n in W ...
      TxPrint("\n");
   }
   if (strlen( htext) != 0)
   {
      TxPrint( "%s\n", htext);
      TxPrint( "%s\n", hline);
   }
   strcpy( txt, "");
   for ( nr = 1; (nr <= *list) && !TxAbort(); nr++)
   {
      binfo = dfsa->sninfo[nr];                 // prepare info field display value

      TRACES(("index: %4lu  sn:0x%llx  info:0x%4.4hx\n", nr, list[ nr], binfo));
      if (binfo & DFSSNINFO)
      {
         sprintf( tinfo, "%4.4hx", DFSSNIGET(binfo));
      }
      else
      {
         strcpy(  tinfo, "    ");
      }
      if      (TxaOption('w'))                  // wide format, 4 per line
      {
         if ((nr % 4) == 1)
         {
            TxPrint( "%s\n", txt);
            strcpy( txt, "");
         }
         if (binfo & DFSSNINFO)
         {
            sprintf( tbuf, "   %12.12llX.%4.4hx", list[nr], DFSSNIGET(binfo));
         }
         else
         {
            sprintf( tbuf, "   %12.12llX     ", list[nr]);
         }
         strcat(  txt, tbuf);
      }
      else if (TxaOption('r'))                  // Cl/Bl range-format (1 per line)
      {
         clust = dfsSn2Cl( list[nr]);           // current cluster/block number

         if ((clust <  lastCl    ) ||           // begin of a new range, output current
             (clust >  lastCl + 1)  )
         {
            if (dfsGetClusterSize() > 1)
            {
               sprintf( txt, " Range Cl/Block: %12.12llX .. %12.12llX = sect: %12.12llX .. %12.12llX =",
                        rangCl,  lastCl, dfsCl2Sn( rangCl), dfsCl2Sn( lastCl +1) -1);
            }
            else
            {
               sprintf( txt, " Range of sectornumbers: 0x%12.12llX .. 0x%12.12llX, size is:", rangCl, lastCl);
            }
            dfstrSz64XiB( txt, "", (lastCl - rangCl + 1) * dfsGetClusterSize(), dfsGetSectorSize(), "");
            TxPrint( "%s\n", txt);
            ul++;                               // count ranges

            rangCl = clust;                     // start of new range
         }
         lastCl = clust;

         if (nr == *list)                       // at end list, output last range
         {
            if (dfsGetClusterSize() > 1)
            {
               sprintf( txt, " Range Cl/Block: %12.12llX .. %12.12llX = sect: %12.12llX .. %12.12llX =",
                        rangCl,  lastCl, dfsCl2Sn( rangCl), dfsCl2Sn( lastCl +1) -1);
            }
            else
            {
               sprintf( txt, " Range of sectornumbers: 0x%12.12llX .. 0x%12.12llX, size is:", rangCl, lastCl);
            }
            dfstrSz64XiB( txt, "", (lastCl - rangCl + 1) * dfsGetClusterSize(), dfsGetSectorSize(), "");
            TxPrint( "%s\n", txt);
            ul++;                               // count ranges
         }
      }
      else                                      // all line based modes
      {
         ref = list[nr];
         if (TxaOption('d'))                    // sectors in default-format, maybe hex
         {
            TxaOptSetItem( "-l-");              // force non-recursive lists
            st = 0;
            if ((rc = dfsReadAnDisplay( ref, dfsa->sninfo[nr], &st)) == NO_ERROR)
            {
               strcpy( tbuf, "def");            // default option value
               TxaOptAsString('d', TXMAXTM, tbuf);

               if (((tolower( tbuf[0]) == 'h')) && (st != ST_UDATA)) // hex wanted and not done yet
               {
                  TxDisplayHex( "Hex-dump of sector:\n", rbuf, (ULONG)  dfsGetSectorSize(), 0);
               }
               TxPrint("\n");
            }
         }
         else if (TxaOption('b'))               // MBR/EBR/boot/LVM display
         {
            TxPrint("\n");
            st = 0;
            if ((rc = dfsReadAnDisplay( ref, dfsa->sninfo[nr], &st)) == NO_ERROR)
            {
               switch (st)
               {
                  case ST_LVINF:
                  case ST_LVSIG:
                  case ST_LVDLF:
                  case ST_LVDLT:
                  case ST_LVBBF:
                  case ST_LVBBT:
                  case ST_LUKSH:
                     TxDisplayHex( "\nHex-dump of first part of the sector:\n", rbuf, (ULONG ) 0x150, 0x000);
                     break;

                  case ST_UDATA:                // could be some superblock
                     if (binfo & DFSSNINFO)     // info value will be FS-type
                     {
                        dfsDisplaySnType( "\nFS superblock  at : ", list[nr], 0, DFSSNIGET(binfo));
                        switch (DFSSNIGET(binfo))
                        {
                           case ST_SHPFS: dfsHpfsDisplaySuperBlock( rbuf); break;
                           case ST_SNTFS: dfsNtfsDisplaySuperBlock( rbuf); break;
                           case ST_SJFS:  dfsJfsDisplaySuperBlock(  rbuf); break;
                           case ST_SEXT:  dfsExtDisplaySuperBlock(  rbuf); break;
                           case ST_SRSR:  dfsRsrDisplaySuperBlock(  rbuf); break;
                           case ST_SXFS:  dfsXfsDisplaySuperBlock(  rbuf); break;
                           case ST_SHFS:  dfsHfsDisplaySuperBlock(  rbuf); break;
                           case ST_SAPFS: dfsApfsDisplaySuperBlock( rbuf); break;
                           default: TxPrint("Unsupported Superblock.\n");  break;
                        }
                     }
                     break;

                  default:
                     TxDisplayHex( "\nHex-dump of last part of the sector:\n",
                                      rbuf + 0x100, (ULONG ) SECTORSIZE - 0x100,  0x100);
                     break;

               }
               TxPrint("\n");
            }
         }
         else if (TxaOption('c'))               // minimum + command exec
         {
            rc = DFS_PENDING;
            if (!TxaOptUnSet('d'))              // unless contents display suppressed
            {
               if (TxaOption('f'))              // single-line file-info too
               {                                // selecting ALL sectors/files!
                  sprintf( txt, "\n.%5.5u %12.12llX %4.4s ", nr - 1, list[nr], tinfo);
                  rc = DFSFNCALL(dfsa->FsFileInformation, list[nr], binfo, "", txt);
               }
               if (rc == DFS_PENDING)           // not handled yet
               {
                  st  = dfsReadIdentifySector(list[nr]);
                  sprintf( txt, "\n.%5.5u %12.12llX     ", nr - 1, ref);
                  dfsDisplaySnType( txt, list[nr], 0, st);
               }
            }
            if (*select != '.')
            {
               nav.this = list[nr];             // make available for cmd
               strcpy( string, select);
               strcat( string, " -l-");         // avoid list modification
               if (TxaOptUnSet('t'))            // test explicitly OFF
               {
                  strcat( string, " -t-");      // pass-trough of non-test option
               }
               else if (TxaOption('t'))         // test explicitly ON
               {
                  strcat( string, " -t");       // pass-trough of test option
               }
               rc = dfsMultiCommand( string, 0, TRUE, FALSE, TRUE);
            }
         }
         else if (TxaOption('g'))               // GEO translated values
         {
            sprintf( txt, ".%5.5u ", nr - 1);
            strcpy( tbuf, "logical");           // default option value
            TxaOptAsString('g', TXMAXTM, tbuf);

            if (tolower( tbuf[0]) == 's')       // system geometry
            {
               strcat( txt, "System GEO : ");
               dfsGeoDispTransLsn( txt, dfstSysHeads(   DFSTORE),
                                        dfstSysSectors( DFSTORE), list[nr]);
            }
            else
            {
               strcat( txt, "Normal GEO : ");
               dfsDisplayTransLsn( txt, list[nr]);
            }
         }
         else if (TxaOption('s') ||             // show delta-sizes & type from list
                  TxaOption('t')  )             // show delta-sizes & type from sector
         {
            ULN64 size = (list[nr] > last) ? (list[nr] - last) : (last -list[nr]);
            strcpy( tbuf, " ");
            if (size > 205)                     // more than 0.1 Mb
            {
               dfstrSz64XiB( tbuf, "", size, dfsGetSectorSize(), "");
               last = list[nr];                 // set last value to 'this'
            }
            sprintf( txt, ".%5.5u%12.12s: ", nr - 1, tbuf);
            if (TxaOption('t') || ((binfo & DFSSNINFO) == 0))
            {
               dfsDisplayLsnTypeChsPsn( txt, list[nr], 0); // type from sector data
            }
            else                                // use stored sector type if available
            {
               dfsDisplaySnType( txt, list[nr], 0, DFSSNIGET(binfo));
            }
         }
         else if (TxaOption('S'))               // Test String returns
         {
            strcpy( txt, "");                   // no display by FileInfo, just string return
            strcpy( selstr, string);            // selection will be destroyed by output str!

            rc = DFSFNCALL(dfsa->FsFileInformation, list[nr], dfsa->sninfo[nr], selstr, txt);
            if (rc == NO_ERROR)
            {
               TxPrint( ".%5.5u: '%s' '%s'\n", nr - 1, selstr, txt);
            }
         }
         else                                   // default format, and -f file-info-based
         {
            //- First check if the -f FileInfo gets a result, else use default format
            if ((!slt) || (!dfsSlTableFind( list[nr], &ref, &binfo, &st, &ul)))
            {
               st = dfsReadIdentifySector(list[nr]);
            }
            if (TxaOption('f'))                 // file-info requested
            {
               strcpy( tbuf, "file");           // default option value
               TxaOptAsString('f', TXMAXTM, tbuf);

               sprintf( txt, "%c%5.5u %12.12llX %4.4s ",
                         (strchr( tbuf, '8')) ? '~' : '.', nr-1, list[nr], tinfo);
               if (slt && strchr( tbuf, 'i'))       // SLT-indirection
               {
                  //- Note: binfo comes from found reference in the SLT now
                  rc = DFSFNCALL(dfsa->FsFileInformation, ref,      binfo, string, txt);
               }
               else
               {
                  rc = DFSFNCALL(dfsa->FsFileInformation, list[nr], dfsa->sninfo[nr], string, txt);
               }
            }
            switch (rc)                         // without -f, it will still be PENDING!
            {
               case NO_ERROR:                   // matched selection
                  selected++;
                  newstatus = TRUE;
                  break;

               case DFS_PENDING:                // not handled yet (default, or not a file object)
                  if (TxaOption('f'))           // file-info was requested
                  {
                     TxPrint( ".%5.5u %12.12llX %4.4s        %sNoFileInfo%s\n", nr-1, list[nr], tinfo, CBZ, CNN);
                  }
                  else                          // default format for unknown stuff
                  {
                     if (ref < 999999999999ULL)
                     {
                        sprintf( txt, ".%5.5u %12llu ", nr - 1, ref);
                     }
                     else
                     {
                        sprintf( txt, ".%5.5u              ", nr - 1);
                     }
                     dfsDisplaySnType( txt, list[nr], binfo, st);
                  }
                  newstatus = TRUE;
                  break;

               case DFS_ST_MISMATCH:            // skipped selection
               default:                         // other error
                  break;
            }
            #if defined (USEWINDOWING)
               if (txwIsWindow( TXHWND_DESKTOP))
               {
                  if (newstatus || TxTmrTimerExpired( dfsa->statusTimer))
                  {
                     TXTM     status;
                     sprintf( status, " Item : %8u of %8llu; matched now: %8u", nr, list[0], selected);
                     txwSetSbviewStatus( status, cSchemeColor);
                     dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
                     newstatus         = FALSE;
                  }
               }
            #endif
         }
         strcpy( txt, "");
      }
   }
   if (TxaOption('w'))
   {
      if (strlen( txt) != 0)
      {
         TxPrint( "%s", txt);
      }
      TxPrint("\n");
   }
   else
   {
      if (TxaOption('s'))                       // show delta-sizes
      {
         if ((last > list[1]) && (last < (dfsGetLogicalSize() -1)))
         {
            sprintf( txt, " toEnd%9.1lf Mb: ",
                           TXSMIB(( dfsGetLogicalSize()-1 - last),
                                    dfsGetSectorSize()));
            dfsDisplaySnType( txt,  dfsGetLogicalSize()-1, 0, ST_FINAL);
         }
         else if (last != 0)                    // decreasing sns
         {
            sprintf( txt, " Begin%9.1lf Mb: ",  TXSMIB( last,
                                    dfstGetSectorSize(  DFSTORE)));
            dfsDisplaySnType( txt,  0, 0, ST_FINAL);
         }
      }
   }
   if ((strlen( htext) != 0) && (*list > 20))
   {
      TxPrint( "%s\n", hline);
      TxPrint( "%s\n", htext);
   }
   TxPrint("\nSN list contains  : %llu entries", *list);
   if (TxaOption('f') && selected && strlen(string))
   {
      TxPrint(" with %u matching %s%s%s", selected, CBC, string, CNN);
   }
   else if (TxaOption('r'))
   {
      TxPrint(" in %u %s ranges", ul, (dfsGetClusterSize() > 1) ? "Cl/Block" : "sectornumber");
   }
   TxPrint("\n");
   VRETURN();
}                                               // end 'dfsDisplaySectorList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize a Multi-item recovery session, by clearing the status counters
/*****************************************************************************/
void dfsInitMultiRecover
(
   void
)
{
   mrAborted     = FALSE;                       // Multi-Recover session aborted
   mrStatus      = FALSE;                       // New status update desirable
   mrHandled     = 0;                           // Total nr of handled   items
   mrFailed      = 0;                           // Total nr of failed    items
   mrRecovered   = 0;                           // Total nr of recovered items
}                                               // end 'dfsInitMultiRecover'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get total number of handled items in a Multi-item recovery session
/*****************************************************************************/
ULONG dfsGetMrHandled
(
   void
)
{
   return mrHandled;
}                                               // end 'dfsGetMrHandled'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get total number of recovered items in a Multi-item recovery session
/*****************************************************************************/
ULONG dfsGetMrRecovered
(
   void
)
{
   return mrRecovered;
}                                               // end 'dfsGetMrRecovered'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get total number of failed items in a Multi-item recovery session
/*****************************************************************************/
ULONG dfsGetMrFailed
(
   void
)
{
   return mrFailed;
}                                               // end 'dfsGetMrFailed'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Recover multiple files using the list of sectornumbers in the DFSee SN-table
/*****************************************************************************/
ULONG dfsRecoverList
(
   char               *path,                    // IN    destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
)
{
   ULONG               rc  = NO_ERROR;
   TXTM                wildcard;

   ENTER();

   strcpy( wildcard, param->select);
   #if defined (USEWINDOWING)
   if ((TxaOption('P') || (strlen(wildcard) == 0))) // prompting
   {
      rc = dfsFileListDialog( " Specify selection critera for files to be RECOVERed ",
                               *(dfsa->snlist), FALSE, wildcard);
      if (rc == NO_ERROR)
      {
         TRACES(("savetoPath: '%s'\n", dfsa->SavetoPath));
         TRACES(("wildcard  : '%s'\n", wildcard));
         strcpy( param->select, wildcard);      // copy back into param structure!
      }
   }
   #endif
   if (rc == NO_ERROR)
   {
      TxPrint( "\nStart recovery for a maximum of %llu files to %s\n"
               "using selection: '%s'\n\n", *(dfsa->snlist), path,
                (strlen( wildcard)) ? wildcard : "*");

      dfsInitMultiRecover();

      rc = dfsRecoverMulti( path, param);

      if (mrAborted)                            // aborted on error
      {
         TxPrint("\nRecovery aborted after a failure\n");
      }
      TxPrint( "\n%u files recovered, %u UNRELIABLE out of a total of %u.\n",
                 mrRecovered, mrFailed, mrHandled);
   }
   RETURN(rc);
}                                               // end 'dfsRecoverList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Recover multiple files, from the list of sectornumbers in the SNLIST
// May be called recursively when recovering directories (guard against '..')
/*****************************************************************************/
static ULONG dfsRecoverMulti
(
   char               *path,                    // IN    destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               nr;
   ULONG               thisListSize = (ULONG) *(dfsa->snlist);
   ULONG               firstIndex   = 1;
   ULONG               itemsTodo    = thisListSize;

   ENTER();

   if (DFSBR_SnlistFlag & DFSBR_1ST_PARENT)     // first entry is the parent DIR
   {
      TRACES(("Skipping parent directory!\n"));
      firstIndex++;                             // skip the '..' entry in recovery!
      itemsTodo--;
   }
   dfsProgressItemsTodo( itemsTodo, "Recover");

   for (nr = firstIndex; (nr <= thisListSize) && (!TxAbort() && !mrAborted); nr++)
   {
      switch (rc = dfsRecoverSingle( dfsa->snlist[nr], dfsa->sninfo[nr], path, param))
      {
         case NO_ERROR:
            mrStatus = TRUE;
            break;

         case DFS_PENDING:
         case DFS_ST_MISMATCH:                  // skipped
            dfsProgressItemsNext();             // explicit indicate next 'item'
            rc = NO_ERROR;
            break;

         default:
            dfsProgressItemsNext();             // explicit indicate next 'item'
            switch (dfsa->eStrategy)
            {
               case TXAE_QUIT:
                  mrAborted = TRUE;             // abort the recovery
                  break;

               case TXAE_CONFIRM:
                  if (dfsa->batch)              // CONFIRM not possible, but do NOT ignore errors!
                  {
                     mrAborted = TRUE;          // abort the recovery
                  }
                  else
                  {
                     if (TxConfirm( 5201, "Recovery of the last file failed. Do you want to "
                                          "continue with the rest of the files ? [Y/N]: "))
                     {
                        if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                        {
                           dfsa->eStrategy = TXAE_IGNORE;
                        }
                        TxPrint("\nContinue recovering more files ...\n");
                        TxCancelAbort();        // might have aborted a
                     }                          // failing file with <Esc>
                     else
                     {
                        mrAborted = TRUE;       // abort the recovery
                     }
                  }
                  break;

               case TXAE_IGNORE:
               default:                         // ignore the error
                  break;
            }
            break;
      }
      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            if (mrStatus || TxTmrTimerExpired( dfsa->statusTimer))
            {
               TXTM                status;
               sprintf( status, "Item : %8u; recovered:%8u", mrHandled, mrRecovered);
               txwSetSbviewStatus( status, cSchemeColor);
               dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
               mrStatus          = FALSE;
            }
         }
      #endif
   }
   RETURN(rc);
}                                               // end 'dfsRecoverMulti'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS recover a single file/directory
// May be called recursively when recovering directories
/*****************************************************************************/
ULONG dfsRecoverSingle
(
   ULN64               fn,                      // IN    LSN for file to recover
   ULN64               info,                    // IN    meta info for the LSN
   char               *path,                    // IN    base destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
)
{
   ULONG               rc;
   TXLN                txt;
   TXLN                select;                  // selection string MUST be IN/OUT if txt empty!
   TXLN                fullpath;
   char               *sp;
   BOOL                isDirectory = FALSE;     // target item is a directory (recurse Y/N)
   char               *sym;                     // symlink name detection (from FileInfo)

   ENTER();
   TRACES(( "LSN:%llx info:%llx path:'%s' accept:'%s' newname:'%s'\n",
             fn, info, path, param->select, param->newname));

   mrHandled++;                                 // count EVERY item handled

   if (dfsa->rParams.noAllocCheck)              // skip allocation check (faster)
   {
      strcpy(  txt, "!");                       // set as option for FileInfo...
   }
   else if (dfsa->verbosity > TXAO_NORMAL)      // verbose, multiline, with alloc check
   {
      sprintf( txt, "Rec  %12.12llX : ",     fn);
   }
   else                                         // single line, with alloc check
   {
      strcpy(  txt, "");
   }
   strcpy( select, (param->select) ? param->select : "");

   //- check if file matches criteria, and get the full-path to target
   rc = DFSFNCALL(dfsa->FsFileInformation, fn, info, select, txt);
   switch (rc)
   {
      case NO_ERROR:                            // matched accept criteria

         if ((sym = strstr( txt, DFS_SH_LINK_STR)) != NULL) // there is a symlink part
         {
            TRACES(("Removing symlink indicator from target: '%s'\n", sym));
            *sym = 0;                           // clip txt at start of symlink
            TxStrip( txt, txt, ' ', ' ');       // and remove leading/trailing spaces
         }
         TRACES(("base destination: '%s'   target: '%s'\n", path, txt));

         if ((path[strlen(path)-1] == FS_PATH_SEP) || (path[strlen(path)-1] == FS_PALT_SEP))
         {
            path[strlen(path)-1] = '\0';        // remove trailing char
         }
         if (param->nopath)                     // don't want target path, just basename
         {
            strcpy( fullpath, path);            // use destination only, discard target path
         }                                      // (basename retrieved again, later)
         else
         {
            if (txt[strlen(txt)-1] == FS_PATH_SEP)    //- target is a directory
            {
               txt[strlen(txt)-1] = '\0';             //- remove trailing char
               isDirectory = TRUE;
            }
            if ((sp = strrchr(txt, FS_PATH_SEP)) != NULL)
            {
               *sp = '\0';                      // remove target-name
            }                                   // from the path
            else
            {
               txt[0] = '\0';                   // not even a path left!
            }
            sprintf( fullpath, "%s%s", path, txt);
         }
         rc = DFSFNCALL(dfsa->FsDirFileSaveAs, fn, info, fullpath, param);

         if (dfsa->verbosity == TXAO_NORMAL)    // non-verbose, single line
         {
            if (rc == NO_ERROR)                 // signal OK/WARNING/FAIL and end-of-line
            {
               TxPrint( " %sOK%s\n", CBG, CNN);
            }
            else if (rc == DFS_CMD_WARNING)     // File date Allocation error, warning only
            {
               TxPrint( " %sWARNING%s Saved but not reliable\n", CBY, CNN);
            }
            else
            {
               TxPrint( " %sFAIL!%s rc: %u  ", CBR, CNN, rc);
               dfsExplainRC( rc);
            }
         }
         if ((rc == NO_ERROR) || (rc == DFS_CMD_WARNING))
         {
            mrRecovered++;
            if (rc == DFS_CMD_WARNING)
            {
               mrFailed++;                      // Recovered but unreliable
            }
            if (param->recurseDir && isDirectory)
            {
               rc = dfsRecursiveDirRecover( fn, info, path, param);
            }
         }
         else
         {
            mrFailed++;
         }
         break;

      case DFS_PENDING:                         // skipped, wrong sector type
         strcpy( txt, "Not a valid sector");
      case DFS_ST_MISMATCH:                     // skipped on accept
         if (dfsa->verbosity >= TXAO_MAXIMUM)
         {
            dfsX10("Skipping ", fn, "", " : ");
            TxPrint("%s\n", txt);
         }
         break;

      default:                                  // other error, abort
         dfsX10("Abort on ", fn, "", " : error ");
         TxPrint( "%u\n", rc);
         break;
   }
   RETURN(rc);
}                                               // end 'dfsRecoverSingle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS recover single directory (contents) by recursing into it
// Saves the current SNLIST, handles DIR, then restores original SNLIST
// 20180218 JvW: Also save the 'info' BYTE aray for FAT and EXT filesystems!
/*****************************************************************************/
static ULONG dfsRecursiveDirRecover
(
   ULN64               fn,                      // IN    LSN for file to recover
   ULN64               info,                    // IN    meta info for the LSN
   char               *path,                    // IN    base destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
)
{
   ULONG               rc;
   ULN64              *savedList = NULL;        // preserved current SNLIST
   USHORT             *savedInfo = NULL;        // preserved current SNINFO
   size_t              savedSize;               // number of entries in saved list

   ENTER();
   TRACES(( "LSN:%llx info:%llx\n", fn, info));

   savedSize = (size_t) ((*(dfsa->snlist)) + 1);
   if ((savedList = TxAlloc( savedSize, sizeof( ULN64))) != NULL)
   {
      if ((savedInfo = TxAlloc( savedSize, sizeof( USHORT))) != NULL)
      {
         memcpy( savedList, dfsa->snlist, savedSize * sizeof( ULN64 )); // save snlist
         memcpy( savedInfo, dfsa->sninfo, savedSize * sizeof( USHORT)); // save sninfo

         rc = DFSFNCALL(dfsa->FsMakeBrowseList, fn, info, "L", NULL); // exclude links when configured so
         if (rc == NO_ERROR)
         {
            rc = dfsRecoverMulti( path, param);
         }
         memcpy( dfsa->snlist, savedList, savedSize * sizeof( ULN64 )); // restore snlist
         memcpy( dfsa->sninfo, savedInfo, savedSize * sizeof( USHORT)); // restore sninfo

         TxFreeMem( savedInfo);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
      TxFreeMem( savedList);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsRecursiveDirRecover'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse file-selection string: wildcard, percentage, min/max, size, timestamp
/*****************************************************************************/
BOOL dfsParseFileSelection                      // RET   more than a wildcard
(
   char               *spec,                    // IN    selection string
   TXTM                wildcard,                // OUT   wildcard part
   BOOL               *minperc,                 // OUT   percentage is minimum
   ULONG              *percent,                 // OUT   percentage value, or 0
   ULONG              *minsize,                 // OUT   min size sectors, or 0
   ULONG              *maxsize,                 // OUT   max size sectors, or 0
   time_t             *timestamp,               // OUT   Modify timestamp, or 0
   char               *type                     // OUT   D(ir) F(ile) B(rowse)
)                                               //       (unchanged if not set)
{
   BOOL                rc = FALSE;              // function return
   TXTM                text;                    // string buffer
   long                threshold;               // percentage threshold
   char               *sp;
   char               *sp2;
   char               *sp3;
   ULONG               sectors;                 // a size in sectors
   BYTE                unit;                    // unit for size (k,m,g,c,h,s)

   ENTER();
   TRACES(( "selection spec:'%s'\n", spec));

   *minperc   = TRUE;                           // initialize return values to
   *percent   = 0;                              // reflect 'all selected/matched'
   *minsize   = 0;
   *maxsize   = 0;
   *timestamp = 0;

   TxCopy( text, spec, TXMAXTM);                // copy to modifyable buffer
   if ((sp = strchr(text, '%')) != NULL)        // is there a percentage ?
   {
      *sp++ = '\0';                             // delimit wildcard

      threshold = atol( sp);
      if (threshold < 0)                        // select upto maximum
      {
         *percent = -threshold;                 // make positive ...
         *minperc = FALSE;                      // it is a maximum
      }
      else
      {
         *percent = threshold;                  // positive value
      }

      if ((sp2 = strchr( sp, '%')) != NULL)     // minsize value present ?
      {
         sp2++;                                 // advance to value string

         if ((sp = strchr( sp2, '%')) != NULL)  // maxsize value present ?
         {
            *sp++ = '\0';                       // delimit minsize

            if ((sp3 = strchr( sp, '%')) != NULL) // item-type value present ?
            {
               *sp3++ = '\0';                   // delimit maxsize
               *type  = *sp3++;                 // copy type value and advance

               if (*sp3 == '%')                 // timestamp value present ?
               {
                  *timestamp = TxaParseDateTime( ++sp3);
               }
            }
            sectors  = (ULONG) TxaParseNumber( sp, DFSRADIX_SIZE, &unit);
            *maxsize = (ULONG) dfsApplyNumberUnit( sectors,
                              (unit == TXA_DFUNIT) ? 'k' : unit, DFSTORE);
         }
         sectors  = (ULONG) TxaParseNumber( sp2, DFSRADIX_SIZE, &unit);
         *minsize = (ULONG) dfsApplyNumberUnit( sectors,
                           (unit == TXA_DFUNIT) ? 'k' : unit, DFSTORE);
      }
      rc = TRUE;                                // there was more ...
   }
   strcpy(   wildcard, text);                   // copy terminated wildcard
   TRACES(( "wildcard:'%s'  M%s %u%%  minS:%u  maxS:%u  type:%2.2hhx  timestamp:0x%8.8x\n",
             wildcard, (*minperc) ? "IN" : "AX", *percent, *minsize, *maxsize, *type, *timestamp));
   BRETURN (rc);
}                                               // end 'dfsParseFileSelection'
/*---------------------------------------------------------------------------*/

