//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee registration number/file generator
//
// Author: J. van Wijk
//
// 16-07-2001  JvW  Initial version
// 02-07-2002  JvW  Update to allow normal name input
// 03-07-2002  JvW  Update to allow specific temp/update times
// 28-12-2004  JvW  Changed default to version 7

#include <txlib.h>                              // TX library interface

#include <dfsulzw.h>                            // DFSee compression interface

#define SECTORSIZE  512

typedef unsigned long  ULONG;

static ULONG          *crcTable = NULL;


int main (int argc, char *argv[]);

/*****************************************************************************/
// Create DFSee registration number in a compressed file and stdout
/*****************************************************************************/
int main (int argc, char *argv[])
{
   int                 rc = 0;
   char                buf[ SECTORSIZE];
   char                prg[ 100];
   char               *s;
   ULONG               crc1 = 0;
   ULONG               crc2 = 0;
   ULONG               crct = 0;
   ULONG               regc = 0;
   ULONG               vmaj = 7;                // Default major version
   BOOL                fixedName = FALSE;       // no fixed name
   char                dext[ 8];                // default extension
   char                post[32];                // key postfix string like #T
   long                expire;

   strcpy( post, "");

   strcpy( prg, argv[0]);
   while ((argc > 1) && strchr("-/", argv[1][0]))
   {
      switch (argv[1][1])
      {
         case 'f':
         case 'F':
            fixedName = TRUE;
            break;

         case 'c':                              // check the value
         case 'C':
            if (*(s = argv[1] +2) == ':')
            {
               s++;
            }
            if (isdigit(*s))
            {
               sscanf( s, "%lu", &crct);
            }
            break;

         case 't':
         case 'T':                              // Temporary key
         case 'u':
         case 'U':                              // Upgrade key
            if (*(s = argv[1] +2) == ':')
            {
               s++;
            }
            if (isdigit(*s))                    // explicit period specified
            {
               expire = atol( s);
               sprintf( post, "#%c%ld", toupper(argv[1][1]), expire);
            }
            else
            {
               sprintf( post, "#%c",    toupper(argv[1][1]));
            }
            break;

         case '?':                              // give usage
            break;

         default:
            vmaj = atol(argv[1]+1);             // set major version number
            break;
      }
      argv++;
      argc--;                                   // shift rest of arguments
   }
   if (vmaj <= 4)
   {
      strcpy( dext, ".reg");
   }
   else
   {
      strcpy( dext, ".key");
   }
   if (argc > 1)
   {
      if ( (        argc > 2             ) ||   // name with spaces, or
          ((strchr( argv[1], '@') != NULL) &&   // email with dot and
           (strchr( argv[1], '.') != NULL) ))   // an at-sign
      {
         char          text[ SECTORSIZE];
         time_t        tt = time( &tt);         // current date/time
         int           i;

         memset( buf, 0, SECTORSIZE);

         strcat( buf, argv[1]);                 // copy the argument
         if (argc > 2)
         {
            for (i = 2; i < argc; i++)
            {
               strcat( buf, ".");
               strcat( buf, argv[i]);
            }
            strcat( buf, "@DFSee.Fsys");        // create fake email
         }
         strcat( buf, post);                    // Add key concatenations

         crc1 = TxHpfsCheckSum( buf);

         crc2 = TxCalculateLvmCrc((BYTE *) buf, SECTORSIZE);

         printf( "Base string: '%s'  crct: %lu\n", buf, crct);

         if (crct != 0)                          // check value
         {
            regc = crc1 ^ crc2 ^ crct;

            printf( "\nDFSee %lu.xx validation for '%s', "
                      "reg-nr %lu: %8.8lX\n", vmaj, buf, crct, regc);
         }
         else                                   // use current time
         {
            ULONG      ulh;                     // compression handle
            char       rf[ 250];                // register file

            if (fixedName)
            {
               strcpy( rf,"dfsee");
            }
            else                                // create unique reg name
            {
               strcpy( rf, buf);
               TxRepl( rf, '@', '_');
               TxRepl( rf, '.', '-');
               TxRepl( rf, ' ', '\0');
            }
            strcat( rf, dext);

            if ((vmaj % 2) == 1)                // uneven major version
            {
               strftime( text, SECTORSIZE, "%d%m%Y", localtime( &tt));
            }
            else
            {
               strftime( text, SECTORSIZE, "%Y%m%d", localtime( &tt));
            }
            sscanf(   text, "%lX", &crct);

            regc = crc1 ^ crc2 ^ crct;
            sprintf( text, "DFSee %lu.xx%sreg-nr for '%s': %10.10lu",
                            vmaj, (vmaj < 5) ? " " : "\t", buf, regc);
            printf( "\n%s\n", text);

            if (dfsUlRegister(&ulh) == NO_ERROR)
            {
               ULONG   size = 0;
               ULONG   dr;

               dr = dfsUlBufCompress(ulh, DFSUL_MLZW_14,
                                     text, (ULONG) strlen(text),
                                     buf,  SECTORSIZE, &size);
               if ((dr == NO_ERROR) || (dr == TX_NO_COMPRESS))
               {
                  FILE  *fo;

                  if ((fo = fopen( rf, "wb")) != NULL)
                  {
                     char   c;
                     int    i;

                     fprintf( fo, "DFSreg='");
                     for (i = 0; i < size; i++)
                     {
                        c = buf[i];

                        fprintf( fo, "%2.2lx", (ULONG) c);
                     }
                     fprintf( fo, "'\n");
                     fclose(  fo);
                  }
                  else
                  {
                     printf("Outputfile '%s' open failure\n", rf);
                  }
               }
               else
               {
                  printf("'%s' file encoding failed, rc: %lu\n", rf, dr);
               }
               dfsUlTerminate( ulh);
            }
            else
            {
               printf( "\n%s internal error, no '%s' file generated\n",
                       prg, rf);
            }
         }
      }
      else
      {
         printf( "\nSupplied identification does not conform to standards, use:\n"
                 "\n  - a valid email-address with at least one '.' and a '@'\n"
                   "  - a name with at least two parts separated with spaces\n\n",
                   "When a postfix like '#P' needs to be specified it must be\n",
                   "attached to the identification string WITHOUT any spaces!\n\n");
         rc = 2;
      }
   }
   else
   {
      printf( "\nUsage: %s [-f] [-c:nr] [-digit] email-address | first lastname\n", prg);
      printf( "\n  -f        = Generate file with fixed name: dfsee%s\n", dext);
      printf(   "  -digit    = MajorVersion other than the default of 7\n");
      printf(   "  -t[:Mdd]  = Make it a temporary key, default is 3 months\n");
      printf(   "  -u[:Mdd]  = Make it an upgrade key,  default is 3 months\n");
      printf( "\n  -c:nr     = validate (reg) nr, yielding generation-date\n");
      printf( "\nExample: %s  -8  jvw@dfsee.com\n",  prg);
      printf(   "         %s  -f  Jan van Wijk#P\n", prg);
      rc = 1;
   }
   return( rc);
}                                               // end 'main'
/*---------------------------------------------------------------------------*/

