//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS Partitionable Media management including Unix specific device functionality
//
// Author: J. van Wijk
//
// JvW  18-11-2004 Renamed to 'dfsmedia' to avoid confusion with 'devices'
// JvW  09-08-2004 First version, to hold the did -> Unix-devicename relation
//

#ifndef    DFSDEVS_H
   #define DFSDEVS_H


typedef enum dfsmediatype
{
   DFSD_NONE   = 0,                             // invalid type
   DFSD_PHYS   = 'p',                           // physical disk, real device
   DFSD_VIRT   = 'M',                           // in-Memory virtual disk
   DFSD_IRAW   = 'I',                           // RAW imagefile
   DFSD_IMZD   = 'Z',                           // DFSee disk image, IMZ
   DFSD_VBOX   = 'B'                            // VirtualBox compressed disk
} DFSMEDIATYPE;                                 // end of enum "dfsdisktype"


// Initialize Partitionable Media Map, optional default physical disk mappings
ULONG dfsInitMediaMap                           // RET function result
(
   USHORT              physicals                // IN    max physical disks
);

#if defined (DEV32)
// Get cached disk-ID string for IDE (DANIS506.ADD) / SATA (OS2AHCI.ADD) disks
BOOL dfsMediaOs2IdeSataId
(
   USHORT              pd,                      // IN    OS2 physical disk number
   TXTM                idString                 // OUT   Disk ID string
);

// Determine if given disk is a QSINIT Ram Disk (LVM name will be PAE_RAM_DISK)
BOOL dfsMediaOs2IsPaeRamDisk
(
   USHORT              pd,                      // IN    OS2 physical disk number
   TXTM                idString                 // OUT   Disk ID string
);
#endif

// Add another mapping to the end of the partitionable media mapping table
ULONG dfsAddMediaMapping
(
   DFSMEDIATYPE        type,                    // IN    type of this medium
   USHORT              disknr,                  // IN    phys/virt disk-nr or 0
   ULONG               files,                   // IN    image file count  or 0
   char               *devname,                 // IN    Disk devname      or NULL
   char               *desc,                    // IN    disk description  or NULL
   BOOL                verbose                  // IN    display resulting mapping
);

// Delete specified mapping from the partitionable media mapping table
ULONG dfsDelMediaMapping
(
   USHORT              did,                     // IN    disk id to delete
   BOOL                verbose                  // IN    display deleted mapping
);

// Display Partitionable Media Map contents
void dfsShowMediaMap
(
   USHORT              first,                   // IN    first did to show (0 = all)
   USHORT              last                     // IN    last  did to show
);

// Return description string for DFSee disk media type (length 4)
char *dfsMediaTypeDescr                         // RET   media type as string
(
   DFSMEDIATYPE        type                     // IN    media type
);

// Map DFSee-disk-id to physical/virtual type
DFSMEDIATYPE dfsDid2DiskType                    // RET   media type PHYS/VIRT/NONE
(
   USHORT              did                      // IN    DFSee disk id    1..n
);

// Map DFSee-disk-id to /dev/ device-name (UNIX)
char  *dfsDid2DeviceName                        // RET   device string /dev/...
(
   USHORT              did                      // IN    DFSee disk id    1..n
);

// Map DFSee-disk-id to the disk description (often the disk-identification string)
char  *dfsDid2DiskDescription                   // RET   Disk description string
(
   USHORT              did                      // IN    DFSee disk id    1..n
);

// Set explicit media disk description from specified identification string
void dfsMediaSetDiskIdDesc
(
   USHORT              did,                     // IN    DFSee disk id    1..n
   char               *desc                     // IN    media disk description
);

// Map DFSee-disk-id to real disk number for type of disk
USHORT dfsDid2RealDiskNr                        // RET   Real disk-nr 1..n or 0
(
   USHORT              did                      // IN    DFSee disk id    1..n
);

// Map real disk type+number to DFSee disk id
USHORT dfsRealDiskNr2Did                        // RET   DFSee disk-id 1..n or 0
(
   DFSMEDIATYPE        type,                    // IN    Type of disk, phys, virt, idisk
   USHORT              disknr                   // IN    Real disknr      1..n
);

// Count number of partitionable physical disks, using OS interfaces
USHORT dfsCountPhysicalDisks                    // RET   number of disks
(
   void
);

// Get number of physical disks (using DFSee cached value)
USHORT dfsGetPhysicalDisks                      // RET   number of disks
(
   void
);

// Get number of partitionable media image disks
USHORT dfsGetPmImageDisks                       // RET   number of disks
(
   void
);

// Get total number of partitionable disks (physical, images or virtual)
USHORT dfsGetDiskCount                          // RET   number of disks
(
   BOOL                verbose                  // IN    Report to screen too
);

// Get geometry, filehandle and accessinfo from 1st open of an image type disk
ULONG dfsGetImageDiskInfo
(
   USHORT              idisk,                   // IN    image disk number
   char               *name,                    // OUT   image filename
   TXHFILE            *ifh,                     // OUT   image file handle
   void              **access,                  // OUT   image access info
   ULONG              *cyl,                     // OUT   Geo cylinders
   ULONG              *hds,                     // OUT   Geo heads
   ULONG              *spt                      // OUT   Geo sectors
);

// Create Virtual disk and add as partitionable medium
ULONG dfsmCreateMemoryDisk
(
   int                 argc,                    // IN    valid parameters
   char               *argv1,                   // IN    first  argument
   char               *argv2,                   // IN    second argument
   char               *argv3,                   // IN    third  argument
   char               *argv4                    // IN    fourth argument
);


// Create RAW/Compressed Image disk and add as partitionable medium
ULONG dfsmCreateImageDisk
(
   char               *image                    // IN    image filename
);

#define DFSM_UNMOUNT_ALL     0                  // special 'all' for remove
// Remove last or all Virtual disks including partitionable medium registration
ULONG dfsmDetachMedium
(
   USHORT              did,                     // IN    DFSee disk id
   BOOL                verbose
);

#if defined (UNIX)

// Test if the specified name exists and can be opened, works for (RAW) image files too
BOOL dfsUxDeviceExists                          // RET   existing block device
(
   char               *devname,                 // IN    device like /dev/hda
   BOOL                verbose
);                                              //                or /dev/disk0

#endif

#endif
