//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS Partitionable Media management including Unix specific device functionality
//
// Author: J. van Wijk
//
// JvW  09-08-2004 First version, to hold the disknr -> Unix-devicename relation
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsutil.h>                            // other utility functions
#include <dfsimage.h>                           // DFS Image defs and  functions
#include <dfsimzci.h>                           // DFS IMZ cache/index functions
#include <dfsvdici.h>                           // DFS VDI cache/index functions

#if !defined (OEMSB)
#include <dfsmdisk.h>                           // Disk store memory disks
#endif


typedef struct dfsmedium
{
   DFSMEDIATYPE        type;                    // type of this medium
   USHORT              disknr;                  // phys/virt/image disknr
   ULONG               files;                   // filecount for images (descr)
   TXTT                devname;                 // Disk devname /dev/XdY or /dev/XdiskY
   TXLN                desc;                    // optional description / path
} DFSMEDIUM;                                    // end of struct "dfsmedium"


static DFSMEDIUM    pMediaMap[DFS_MAX_DISK +1]; // max nr of partionable media!
static USHORT       pMediaMapped = 0;           // nr of mapped disks
static USHORT       pPendingIdIndex = 0;        // first one without ID string (OS2)

#if defined   (WIN32)
#elif defined (DOS32)
#elif defined (UNIX)

#define dfsValidateAndAddDevice(d,m)                                \
           if ((dfsPhysDisks < m) && dfsUxDeviceExists((d), FALSE)) \
                dfsAddMediaMapping(DFSD_PHYS,0,0,(d), NULL, FALSE);

#else
   #define INFO_COUNT_DISKS     1               // INFO # partitionable disks

   static TXTM os2IdeSataID[ DFS_MAX_DISK];     // IDE/SATA disk-id string cache

// Create disk-ID string cache for OS/2 IDE/SATA attached disks
static void dfsMediaSetIdeSataIdCache           // allocated ID string array
(
   void
);

// Disk-ID string cache for OS/2 IDE/SATA attached disks, from CONFIG string
static void dfsMediaIdeSataFromString
(
   char               *cfgString                // INOUT CONFIG.SYS contents
);                                               // contents is uppercased!

// Add disk-id (model) strings for hard-disks listed in OS2AHCI$
static void dfsMediaAddAhciDiskIds
(
   USHORT             *index                    // INOUT Index to place ID string
);

// Add disk-id (model) strings for hard-disks listed in IBM1506$ (DANIS506 only!)
static void dfsMediaAddS506DiskIds
(
   USHORT             *index                    // INOUT Index to place ID string
);

#endif


#if !defined (DFS_PHYSDISK_ONLY)
typedef struct dfsi_disk                        // RAW or Compressed image disk
{
   DFS_GEO             Geo;                     // image geometry
   TXHFILE             handle;                  // image file handle
   void               *accessInfo;              // access data (IMZ index etc)
   TXLN                name;                    // image filename
} DFSI_DISK;                                    // end of struct "dfsi_disk"

static USHORT          dfsImageDisks = 0;       // number of active image disks


static DFSI_DISK       dfsidisk[DFS_MAX_DISK +1];

// Return first unused slot in the IDISK array
static USHORT dfsmGetFreeIdiskSlot              // RET   index free slot or 0
(
   void
);

// Make a new Image disk, using DFSee handle/accessInfo from OpenImageFile
static ULONG dfsMakeImageDisk
(
   ULONG               cyl,                     // IN    Geo cylinders
   ULONG               hds,                     // IN    Geo heads
   ULONG               spt,                     // IN    Geo sectors
   TXHFILE             ifh,                     // IN    Image file handle
   void               *access,                  // IN    Access information
   char               *name,                    // IN    Image filename
   USHORT             *idisk                    // OUT   idisk number (or NULL)
);

// Remove image disk and free all resources
static ULONG dfsRemoveImageDisk
(
   DFSMEDIATYPE        itype,                   // IN    image type, RAW, IMZ etc
   USHORT              idisk                    // IN    image disk number
);
#endif

static USHORT dfsPhysDisks = 0;                 // nr of real physical disks

/*************************************************************************************************/
// Initialize Partitionable Media Map, optional default physical disk mappings
/*************************************************************************************************/
ULONG dfsInitMediaMap                           // RET function result
(
   USHORT              physicals                // IN    max physical disks
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              did;                     // disk/card number
   #if defined (DEV32) || defined (DOS32)
      USHORT           disks;
   #endif

   ENTER();

   memset( pMediaMap, 0, sizeof(pMediaMap));
   if (physicals)
   {
      #if defined (LINUX)
         TXTM          devspec;
         USHORT        nsp;                     // namespaces for nvme devices

         strcpy( pMediaMap[0].devname, "/dev/null");

         //- JvW 20181016: Probe NVME devices first (probably SSD system/boot disks)
         for (did = 0; did < 10; did++) // probe 10 NVMe cards nvme0nN .. nvme9nN
         {
            for (nsp = 0; nsp < 10; nsp++)      // probe 10 NVMe namespaces nvmeCn0 .. nvmeCn9
            {
               sprintf( devspec, "/dev/nvme%cn%c", did + '0', nsp + '0');
               dfsValidateAndAddDevice( devspec,  physicals);

               //- also check for the Toshiba/OCZ driver names (ordering may mix with regular nvme!)
               sprintf( devspec, "/dev/ocznvme%cn%c", did + '0', nsp + '0');
               dfsValidateAndAddDevice( devspec,  physicals);
            }
         }
         for (did = 0; did < 26; did++) // probe 26 IDE devices names first ...
         {
            sprintf( devspec, "/dev/hd%c", did + 'a'); // hda .. hdz
            dfsValidateAndAddDevice( devspec,  physicals);
         }
         for (did = 0; did < 26; did++) // probe 26 SCSI (and SATA?) names next ...
         {
            sprintf( devspec, "/dev/sd%c", did + 'a'); // sda .. sdz
            dfsValidateAndAddDevice( devspec,  physicals);
         }
         for (did = 0; did < 26; did++) // probe 26 VM (para/virtio interface) names next ...
         {
            sprintf( devspec, "/dev/vd%c", did + 'a'); // vda .. vdz
            dfsValidateAndAddDevice( devspec,  physicals);
         }
         for (did = 0; did < 26; did++) // probe 26 VM (HVM virtual interface) names next ...
         {
            sprintf( devspec, "/dev/xvd%c", did + 'a'); // xvda .. xvdz
            dfsValidateAndAddDevice( devspec,  physicals);
         }
         for (did = 0; did < 10; did++) // probe 10 MMC (SD card) mmcblk0 .. mmcblk9
         {
            sprintf( devspec, "/dev/mmcblk%c", did + '0');
            dfsValidateAndAddDevice( devspec,  physicals);
         }
      #elif defined (DARWIN)
         TXTM          devspec;

         for (did = 0; did < DFS_MAX_DISK; did++) // probe up to DFSee limit
         {
            //- When probing 'diskN' devices, filters out locked 'virtual' logical volumes (root filesystem)
            //- so making them invisible, however, it also does not see some disk-images then, with the
            //- same error on open: Resource busy. So probe the rdiskN devices by default ...
            sprintf( devspec, "/dev/%sdisk%hu", TxaOptUnSet( TXA_O_RAW) ? "" : "r", did);
            if (dfsUxDeviceExists( devspec, (did == 0))) // Verbose on disk0 (SIP or no su)
            {
               //- Add the faster 'rdiskN' devices by default, or 'diskN' when -raw- switch used
               dfsValidateAndAddDevice( devspec,  physicals);
            }
         }
      #elif defined (WIN32)
         TXHFILE          hDisk;                // physical-disk-nrs are NOT always consequtive!
         TXTM             dspec;                // iterate and assign as discovered here, up to 32
         ULONG            pdnr;                 // physical disk number being probed/added

         for (did = 0, pdnr = 0; pdnr < 32; pdnr++) // up to phys-disk 32, but no more than specified
         {
            sprintf( dspec, "\\\\.\\PHYSICALDRIVE%u", pdnr);
            hDisk = CreateFile( dspec,
                                GENERIC_READ    | GENERIC_WRITE,
                                FILE_SHARE_READ | FILE_SHARE_WRITE,
                                NULL, OPEN_EXISTING,
                                FILE_ATTRIBUTE_NORMAL,
                                NULL);
            if (hDisk != INVALID_HANDLE_VALUE)
            {
               CloseHandle( hDisk);

               if (did < physicals)
               {
                  //- note that phys-disk-nr in AddMediaMapping is 1 based! (Windows is 0 beased)
                  dfsAddMediaMapping( DFSD_PHYS, pdnr + 1, 0, NULL, NULL, FALSE); // add physical disk number
                  did++;                        // count as added disk
               }
               else
               {
                  break;                        // already added maximum no of disks specified
               }
            }
         }
      #else
         disks = dfsCountPhysicalDisks();

         for (did = 1; (did <= disks) && (did <= physicals); did++)
         {
            dfsAddMediaMapping( DFSD_PHYS, did, 0, NULL, NULL, FALSE); // add physical did
         }
      #endif
   }
   if (pMediaMapped > 0)                        // any disk auto-mounted ?
   {
      dfsReadDiskInfo( FDSK_QUIET);             // read diskinfo for all disks

      #if defined (DEV32)                       // re-add disk id string (may need driveletter)

         //- Fill ID cache-array here once, after startup and each refresh (media -update)
         dfsMediaSetIdeSataIdCache();

         for (did = 1; (did <= disks) && (did <= physicals); did++)
         {
            dfsGetDiskIdentification( did, pMediaMap[did].devname, 0, pMediaMap[did].desc);
         }
      #endif

      dfsSelectDisk(   1, FALSE, FALSE);        // select the first one
   }
   RETURN (rc);
}                                               // end 'dfsInitMediaMap'
/*-----------------------------------------------------------------------------------------------*/


#if defined (DEV32)
/*****************************************************************************/
// Create disk-ID string cache for OS/2 IDE/SATA attached disks
/*****************************************************************************/
static void dfsMediaSetIdeSataIdCache        // allocated ID string array
(
   void
)
{
   char               *configSys = NULL;        // config.sys contents
   TXTS                filename;
   ULONG               sysinfo[QSV_VERSION_MINOR]; // bootdrive etc
   ULONG               sysRc = NO_ERROR;

   ENTER();

   if (DosAllocMem( (PPVOID)&configSys, 65536L, PAG_COMMIT|PAG_READ|PAG_WRITE) == NO_ERROR)
   {
      *configSys = 0;

      if ((sysRc = DosSysCtl( SYSCTL_FUNC_CONFIG, (PULONG) configSys)) == NO_ERROR)
      {
         TRACES(("CONFIG INFO from DosSysCtl (OS/4 kernel ?)\n"));
         dfsMediaIdeSataFromString( configSys + 4); // skip 32-bit leading character count
      }
      else
      {
         TRACES(("DosSysCtl failed, rc: %u\n", sysRc));
      }
      DosFreeMem( configSys);
   }

   if (sysRc != NO_ERROR)                       // bad alloc, or failed SysCtl function
   {
      if (DosQuerySysInfo( 1, QSV_VERSION_MINOR,  sysinfo,
                              QSV_VERSION_MINOR * sizeof(ULONG)) == NO_ERROR)
      {
         sprintf( filename, "%c:\\config.sys", (char) '@' + (char) sysinfo[QSV_BOOT_DRIVE-1]);

         if ((configSys = txcReadFileInMemString( filename, 0)) != NULL)
         {
            TRACES(("CONFIG INFO from CONFIG.SYS file (OS/2 kernel)\n"));
            dfsMediaIdeSataFromString( configSys);
            TxFreeMem( configSys);
         }
      }
   }
   VRETURN();
}                                               // end 'dfsMediaSetIdeSataIdCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Disk-ID string cache for OS/2 IDE/SATA attached disks, from CONFIG string
/*****************************************************************************/
static void dfsMediaIdeSataFromString
(
   char               *cfgString                // INOUT CONFIG.SYS contents
)                                               // contents is uppercased!
{
   USHORT              nextIdIndex = pPendingIdIndex - 1; // our index is ZERO based!
   char               *os2ahci   = NULL;        // location of OS2AHCI driver
   char               *ibms506   = NULL;        // location of DANIS506 or IBM1S506 driver

   ENTER();

   TxStrToUpper( cfgString);                    // make it ALL uppercase for string search

   //- Note: detect IBM1S506.ADD removed, since that TRAPS when reading from IBMS506$
   ibms506 = strstr( cfgString, "BASEDEV=DANIS506.ADD");
   os2ahci = strstr( cfgString, "BASEDEV=OS2AHCI.ADD");

   TRACES(("os2ahci: %p  ibms506: %p\n", os2ahci, ibms506));

   if ((os2ahci != NULL) && (os2ahci < ibms506)) // AHCI before IDE ?
   {
      //- OS2AHCI found in config.sys, and BEFORE the DANIS506 driver (if any)
      dfsMediaAddAhciDiskIds( &nextIdIndex);
      dfsMediaAddS506DiskIds( &nextIdIndex);
   }
   else                                         // no AHCI, or AFTER IDE
   {
      //- always handle both, even when not 'seen' in config.sys to account for
      //- syntax variants there. config.sys relevant for ORDER only, not presence
      dfsMediaAddS506DiskIds( &nextIdIndex);
      dfsMediaAddAhciDiskIds( &nextIdIndex);
   }
   VRETURN();
}                                               // end 'dfsMediaIdeSataFromString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add disk-id (model) strings for hard-disks listed in OS2AHCI$
/*****************************************************************************/
static void dfsMediaAddAhciDiskIds
(
   USHORT             *index                    // INOUT Index to place ID string
)
{
   char               *driverInfo = NULL;       // info for OS2AHCI disk driver
   int                 retryCount = 5;          // Retry when file seems empty

   ENTER();
   TRACES(("Index to add: %hu\n", *index));

   //- Pass non-zero sizelimit, since device will show as ZERO by itself!
   while ((--retryCount) && ((driverInfo = txcReadFileInMemString( "OS2AHCI$",  9999)) != NULL))
   {
      char            *this = driverInfo;       // initial parse position
      char            *next;
      char            *disksize;
      char            *diskmodel;
      ULONG            sizeMiB = 0;

      if ((strlen( this) > 0) && (strstr( this, "Drive 0:") != NULL)) // non empty
      {
         if ((next = strstr( this, "Adapter 0:")) != NULL) // first occurence (after possible error lines)
         {
            if ((next = strstr( next + 10, "Adapter 0:")) != NULL) // repeated contents
            {
               *next = 0;                       // terminate single contents
            }
         }

         //- This assumes there will only be one drive per Port (to be refined)
         next = this;                           // starting point for search
         while ((this = strstr( next, "Drive 0:")) != NULL)
         {
            if ((next = strstr( this, " Port ")) == NULL) // No next Port listed
            {
               next = this + strlen( this);     // Stop at end of file
            }
            else
            {
               *next++ = 0;                     // terminate 'this' disk string
            }                                   // and set next to start next one

            if (strstr( this, "cylinders,") != NULL) // it is a disk, no CDROM or other
            {
               TxRepl(  this, 0x0A, ' ');       // replace newlines by spaces
               TxRepl(  this, 0x0D, ' ');
               TxStrip( this, this, 0, ' ');    // then remove trailing spaces

               if ((diskmodel = strstr( this, "Model: ")) != NULL) // Model string
               {
                  diskmodel += strlen( "Model: ");
               }
               else                             // there is NO Model: string!
               {
                  diskmodel  = this + strlen( "Drive 0:") + 1; // Use Disk geometry instead
               }
               if ((disksize = strchr( this, '(')) != NULL)
               {
                  sizeMiB = atol( disksize + 1); // decimal number of Mib disksize
               }

               sprintf( os2IdeSataID[ *index], "AHCI %-42.42s %8u.0 MiB", diskmodel, sizeMiB);
               TRACES(("AHCI disk %hu id: '%s'\n", *index, os2IdeSataID[ *index]));

               (*index)++;                      // set next target ID string
            }
         }
         TxFreeMem( driverInfo);
         break;                                 // from retry loop
      }
      else
      {
         TxFreeMem( driverInfo);
         TRACES(("Retry: %d  IBMS506$ empty: '%s'\n", retryCount, this));
      }
   }
   VRETURN ();
}                                               // end 'dfsMediaAddAhciDiskIds'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add disk-id (model) strings for hard-disks listed in IBM1506$ DANIS506 only!
/*****************************************************************************/
static void dfsMediaAddS506DiskIds
(
   USHORT             *index                    // INOUT Index to place ID string
)
{
   char               *driverInfo = NULL;       // info for IBMS506 disk driver
   int                 retryCount = 5;          // Retry when file seems empty

   ENTER();
   TRACES(("Index to add: %hu\n", *index));

   //- Pass non-zero sizelimit, since device will show as ZERO by itself!
   while ((--retryCount) && ((driverInfo = txcReadFileInMemString( "IBMS506$",  9999)) != NULL))
   {
      char            *this = driverInfo;       // initial parse position
      char            *next;
      char            *disksize;
      char            *diskmodel;
      ULONG            sectors;                 // Available sectors, disksize

      if ((strlen( this) > 0) && (strstr( this, "Controller:") != NULL)) // non empty
      {
         if ((next = strstr( this, "R1.")) != NULL) // First occurence, after possible error lines
         {
            if ((next = strstr( this + 10, "R1.")) != NULL) // repeated contents
            {
               *next = 0;                       // terminate single contents
            }
         }
         next = this;                           // starting point for search
         while ((this = strstr( next, "Model:")) != NULL)
         {
            if ((next = strstr( this, " Model:")) == NULL) // No next Unit/Model listed
            {
               next = this + strlen( this);     // Stop at end of file
            }
            else
            {
               *(next - 1) = 0;                 // terminate 'this' disk string
            }                                   // and set next to start next one

            if ((strstr( this, "OS2:") != NULL) && // it is a disk, no CDROM or other
                ((disksize = strstr( this, "Avail")) != NULL))
            {
               TxRepl(  this, 0x0A, ' ');       // replace newlines by spaces
               TxRepl(  this, 0x0D, ' ');

               diskmodel = this + strlen( "Model:");
               sectors   = atol( disksize + strlen( "Avail"));

               sprintf(      os2IdeSataID[ *index], "IDE  %-42.42s ", diskmodel);
               dfstrSizeXiB( os2IdeSataID[ *index], "   ", sectors, SECTORSIZE, "");

               TRACES(("S506 disk %hu id: '%s'\n", *index, os2IdeSataID[ *index]));

               (*index)++;                      // set next target ID string
            }
         }
         TxFreeMem( driverInfo);
         break;                                 // from retry loop
      }
      else
      {
         TxFreeMem( driverInfo);
         TRACES(("Retry: %d  IBMS506$ empty: '%s'\n", retryCount, this));
      }
   }
   VRETURN ();
}                                               // end 'dfsMediaAddS506DiskIds'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cached disk-ID string for IDE (DANIS506.ADD) / SATA (OS2AHCI.ADD) disks
/*****************************************************************************/
BOOL dfsMediaOs2IdeSataId
(
   USHORT              pd,                      // IN    OS2 physical disk number
   TXTM                idString                 // OUT   Disk ID string
)
{
   BOOL                rc = FALSE;              // function return

   ENTER();

   if ((os2IdeSataID != NULL) && (pd <= DFS_MAX_DISK))
   {
      strcpy( idString, os2IdeSataID[ pd -1]);  // offset in array is zero based

      rc = (strlen( idString) != 0);
   }
   BRETURN (rc);
}                                               // end 'dfsMediaOs2IdeSataId'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine if given disk is a QSINIT Ram Disk (LVM name will be PAE_RAM_DISK)
/*****************************************************************************/
BOOL dfsMediaOs2IsPaeRamDisk
(
   USHORT              pd,                      // IN    OS2 physical disk number
   TXTM                idString                 // OUT   Disk ID string
)
{
   BOOL                rc = FALSE;              // function return
   BYTE               *sector;
   S_LVINF            *dlat;

   ENTER();

   if (pd <= DFS_MAX_DISK)
   {
      if ((sector = TxAlloc( 1, dfsGetSectorSize())) != NULL)
      {
         dfsSelectDisk( pd, FALSE, FALSE);      // quietly select the disk

         //- Read possible LVM DLAT at the standard geometry location
         if (dfstReadPsn( DFSTORE, DFS_STD_SECT - 1, 1, sector) == NO_ERROR)
         {
            dlat = (S_LVINF *) sector;

            if ((memcmp( dlat->Signature, sg_lvinf, SG_LVINF) == 0) &&
                (strcmp( dlat->DiskName,      "PAE_RAM_DISK") == 0)  )
            {
               strcpy( idString, "QS OS/2 PAE Ram Disk");
               rc = TRUE;
            }
         }
         TxFreeMem( sector);
      }
   }
   BRETURN (rc);
}                                               // end 'dfsMediaOs2IsPaeRamDisk'
/*---------------------------------------------------------------------------*/
#endif


/*************************************************************************************************/
// Add another mapping to the end of the partitionable media mapping table
/*************************************************************************************************/
ULONG dfsAddMediaMapping
(
   DFSMEDIATYPE        type,                    // IN    type of this disk
   USHORT              disknr,                  // IN    phys/virt disk-nr or 0
   ULONG               files,                   // IN    image file count  or 0
   char               *devname,                 // IN    Disk devname      or ""
   char               *desc,                    // IN    disk description  or ""
   BOOL                verbose                  // IN    display resulting mapping
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              did;                     // DFSee disk-id
   USHORT              pd;                      // constructed physical disknr
   char                devl = (char) type;      // device letter to use for
                                                // 'constructed' devicename
   ENTER();
   TRACES(("type %c disk:%hu dev:'%s' desc:'%s'\n", (char) type, disknr, devname, desc));

   if (pMediaMapped < DFS_MAX_DISK)             // room for a new one ?
   {
      did = ++pMediaMapped;                     // index for new one

      if (type == DFSD_PHYS)                    // overrule for PHYS hda or sda
      {
         //- replace 'p' designation to specified or default letter for platform
         #if defined (DARWIN)
            char  *s = TxaExeSwitchStr( 'D', NULL, "r"); // default RAW disk
         #else
            char  *s = TxaExeSwitchStr( 'D', NULL, "s"); // default SATA/SCSI = sda
         #endif
         devl = s[0];                           // constructed for non-Unix

         dfsPhysDisks++;                        // increment available PDs
         if (disknr != 0)                       // real physical disk number ?
         {
            pd = disknr;                        // use specified value
         }
         else if (strlen(devname) > 7)          // Unix device, create unique
         {                                      // disknr for handle-sharing
            #if   defined (LINUX)
               pd = devname[7] - 'a' +1;        // /dev/hda will be disk 1
               switch  (devname[5])
               {
                  case 's':            break;   // SCSI/SATA, 1st group of  12
                  case 'h': pd += 12;  break;   // IDE,       2nd group of   8
                  default:  pd += 20;  break;   // Other,     3rd group of   4
               }
            #elif defined (DARWIN)
               if (strncmp( devname, "/dev/rdisk", 10) == 0) // RAW disk device
               {
                  pd = (USHORT) atoi(devname +10) +1; // /dev/rdisk0 will be disk 1
                  TRACES(( "dev: '%s' nr: '%s' atoi: %hu\n", devname, devname + 10, pd));
               }
               else if (strncmp( devname, "/dev/disk", 9) == 0) // BLK disk device
               {
                  pd = (USHORT) atoi(devname +9) +1; // /dev/disk0 will be disk 1
                  TRACES(( "dev: '%s' nr: '%s' atoi: %hu\n", devname, devname + 9, pd));
               }
               else                             // could be anything ...
               {                                // generate 'random' nr 17..24
                  pd = (USHORT) (TxCalculateLvmCrc((BYTE *) devname, strlen( devname)) % 8) +17;
               }
            #else
               pd = (USHORT) (TxCalculateLvmCrc((BYTE *) devname, strlen( devname)) % 8) +17;
            #endif
         }
         else                                   // use last available index
         {
            pd = DFS_MAX_DISK -1;
         }
      }
      else
      {
         pd = disknr;
      }
      pMediaMap[did].disknr = (pd % DFS_MAX_DISK);
      pMediaMap[did].type   = type;
      pMediaMap[did].files  = files;

      if (devname && strlen(devname) && (strlen(devname) < TXMAXTS))
      {
         strcpy( pMediaMap[did].devname, devname);
      }
      else                                      // construct the devname
      {
         #if defined (DARWIN)
            sprintf( pMediaMap[did].devname, "/dev/%cdisk%d", devl, pd -1);
         #else
            sprintf( pMediaMap[did].devname, "/dev/%cd%c", devl, (char) ('a' + (pd % DFS_MAX_DISK) -1));
         #endif
      }
      if (desc && strlen(desc) && (strlen(desc) < TXMAXTM)) // explicit description given
      {
         strcpy( pMediaMap[did].desc, desc);
      }
      else                                      // Use diskID when present, construct one otherwise
      {
         if (dfsGetDiskIdentification(
             #if defined (WIN32)
                                        did,    // Windows (disk enum) needs consequtive disk nr (did)
             #else
                                        pd,
             #endif
                                        pMediaMap[did].devname, 0, pMediaMap[did].desc) == FALSE)
         {
            if (pPendingIdIndex == 0)           // remember first without ID (for OS2)
            {
               pPendingIdIndex = did;           // Array index is ZERO based!
            }

            //- Description will be empty, when no identification is present, except on macOS
            sprintf( pMediaMap[did].desc, "%s",
            #if   defined (LINUX)
                                                         "");
            #elif defined (DARWIN)
                    (pMediaMap[did].devname[5] == 'r') ? "(RAW [char, unbuffered] device)" :
                                                         "(Std [block,  buffered] device)");
            #else
                                                         "");
            #endif
         }
      }
      if (verbose)
      {
         dfsShowMediaMap( did, did);            // show just the new one
      }
      dfsResetForcedGeo( did);                  // make sure Force is reset
   }
   else
   {
      rc = DFS_NOT_FOUND;
   }
   TRACES(( "Number of disk-id now: %hu\n", pMediaMapped));
   RETURN (rc);
}                                               // end 'dfsAddMediaMapping'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Delete mapping from the partitionable media mapping table, including lower levels
/*************************************************************************************************/
ULONG dfsDelMediaMapping                        // RET   result
(
   USHORT              did,                     // IN    disk id to delete
   BOOL                verbose                  // IN    display deleted mapping
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if ((did != 0) && (did <= pMediaMapped))     // found at index 'did'
   {
      TRACES(( "Delete mapping for did: %hu, disknr:%hu devname:'%s'\n",
                did, pMediaMap[did].disknr, pMediaMap[did].devname));

      if (verbose)
      {
         TxPrint( "\nRemoving partitionable media mapping for:\n");
         dfsShowMediaMap( did, did);            // show just one to remove
      }
      if (did == SINF->disknr)                  // open in current store
      {
         dfstClose( DFSTORE);
      }
      switch (pMediaMap[did].type)              // lower level cleanup ?
      {
         #if !defined (DFS_PHYSDISK_ONLY)
         case DFSD_VIRT:
            dfsRemoveMemoryDisk(  pMediaMap[did].disknr);
            break;

         case DFSD_VBOX:
         case DFSD_IMZD:
         case DFSD_IRAW:
            dfsRemoveImageDisk( pMediaMap[did].type, pMediaMap[did].disknr);
            break;
         #endif

         case DFSD_PHYS:
            if (dfsPhysDisks)
            {
               dfsPhysDisks--;                  // update PHYS count
            }
            break;

         default:
            break;
      }
      if (did < pMediaMapped)                   // not the last  one
      {
         memcpy( &pMediaMap[did],               // copy tail of array down ...
                 &pMediaMap[did +1],
                 (pMediaMapped - did) * sizeof(DFSMEDIUM));
      }
      pMediaMapped--;                           // adjust number of mappings
   }
   else
   {
      rc = DFS_NOT_FOUND;
   }
   RETURN (rc);
}                                               // end 'dfsDelMediaMapping'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Display Partitionable Media Map contents
/*************************************************************************************************/
void dfsShowMediaMap
(
   USHORT              first,                   // IN    first did to show (0 = all)
   USHORT              last                     // IN    last  did to show
)
{
   USHORT              did;
   USHORT              final = (last) ? min(last, pMediaMapped) : pMediaMapped;
   DFSMEDIUM          *m;
   int                 maxd = dfsGetDisplayMargin() - 19;

   TxPrint( "\n DFSee  OS  Image filename or\n");
   TxPrint(   "nr type nr  (Unix) device plus optional partitionable media description or identification info from the OS.\n");
   TxPrint(   "== ==== ==  ===============================================================================================\n");

   for (did  = (first) ? first : 1; did <= final; did++)
   {
      m = &(pMediaMap[did]);

      TxStrip( m->desc, m->desc, ' ', ' ');

      TxPrint( "%2hu %s %2hu  ", did, dfsMediaTypeDescr( m->type), m->disknr);
      switch ( m->type)
      {
         case DFSD_PHYS:
            TxPrint( "%-13s %s\n", m->devname, m->desc);
            break;

         default:
            if (strlen( m->desc) > maxd)        // limit to visible size
            {
               TxPrint( "%12.12s..%s\n", m->desc, m->desc + strlen(m->desc) -maxd +15);
            }
            else
            {
               if (m->files != 0)
               {
                  TxPrint( "%s (%u file%s)\n", m->desc, m->files, (m->files > 1) ? "s" : "");
               }
               else
               {
                  TxPrint( "%s\n", m->desc);
               }
            }
            break;
      }
   }
   TxPrint( "\n");
}                                               // end 'dfsShowMediaMap'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Return description string for DFSee disk media type (length 4)
/*************************************************************************************************/
char *dfsMediaTypeDescr                         // RET   media type as string
(
   DFSMEDIATYPE        type                     // IN    media type
)
{
   switch (type)
   {
      default:
      case DFSD_PHYS: return "Phys";
      case DFSD_VIRT: return "Mem ";
      case DFSD_IRAW: return "Iraw";
      case DFSD_IMZD: return "IMZ ";
      case DFSD_VBOX: return "Vbox";
   }
}                                               // end 'dfsMediaTypeDescr'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Map DFSee-disk-id to physical/memory or other type
/*************************************************************************************************/
DFSMEDIATYPE dfsDid2DiskType                    // RET   disk type PHYS/VIRT/NONE
(
   USHORT              did                      // IN    DFSee disk id    1..n
)
{
   DFSMEDIATYPE        rc = DFSD_NONE;          // function return

   if (did <= pMediaMapped)
   {
      rc = pMediaMap[did].type;
   }
   return (rc);
}                                               // end 'dfsDid2DiskType'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Map DFSee-disk-id to /dev/ device-name (UNIX)
/*************************************************************************************************/
char  *dfsDid2DeviceName                        // RET   device string /dev/...
(
   USHORT              did                      // IN    DFSee disk id    1..n
)
{
   char               *rc = pMediaMap[(did <= pMediaMapped) ? did : 0].devname;

   return (rc);
}                                               // end 'dfsDid2DeviceName'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Map DFSee-disk-id to the disk description (often the disk-identification string)
/*************************************************************************************************/
char  *dfsDid2DiskDescription                   // RET   Disk description string
(
   USHORT              did                      // IN    DFSee disk id    1..n
)
{
   char               *rc = pMediaMap[(did <= pMediaMapped) ? did : 0].desc;

   return (rc);
}                                               // end 'dfsDid2DiskDescription'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Set explicit media disk description from specified identification string
/*****************************************************************************/
void dfsMediaSetDiskIdDesc
(
   USHORT              did,                     // IN    DFSee disk id    1..n
   char               *desc                     // IN    media disk description
)
{
   USHORT              safeId = (did <= pMediaMapped) ? did : 0;

   ENTER();

   TxCopy( pMediaMap[ safeId].desc, desc, TXMAXLN);

   TRACES(("did: %hu desc: '%s'\n", did, pMediaMap[ safeId].desc));

   VRETURN ();
}                                               // end 'dfsMediaSetDiskIdDesc'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Map DFSee-disk-id to real disk number for the type of the disk
/*************************************************************************************************/
USHORT dfsDid2RealDiskNr                        // RET   Actual (Memory) disk-nr 1..n or 0
(
   USHORT              did                      // IN    DFSee disk id    1..n
)
{
   USHORT              rc = 0;                  // function return

   if (did <= pMediaMapped)
   {
      rc = pMediaMap[did].disknr;
   }
   return (rc);
}                                               // end 'dfsDid2RealDiskNr'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Map real disk type+number to DFSee disk id
/*************************************************************************************************/
USHORT dfsRealDiskNr2Did                        // RET   DFSee disk-id 1..n or 0
(
   DFSMEDIATYPE        type,                    // IN    Type of disk, phys, virt, idisk
   USHORT              disknr                   // IN    Real disknr      1..n
)
{
   USHORT              rc = 0;                  // function return
   USHORT              did;

   ENTER();
   TRACES(("Real disk id: %hu, MappedDev: %hu\n", disknr, pMediaMapped));

   for (did = 1; did <= pMediaMapped; did++)
   {
      if ((pMediaMap[did].type   == type) &&
          (pMediaMap[did].disknr == disknr))
      {
         rc = did;                              // real disk found, return did
         break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsRealDiskNr2Did'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Count number of partitionable physical disks, using OS interfaces
/*****************************************************************************/
USHORT dfsCountPhysicalDisks                    // RET   number of disks
(
   void
)
{
   USHORT              disks   = 0;
   #if defined (WIN32)
      TXHFILE          hDisk;
      TXTM             dspec;
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      ULONG            DataLen = 2;
      ULONG            ParmLen = 0;
   #endif

   ENTER();

   #if defined (WIN32)
      for (disks = 0; disks < 32; disks++)
      {
         sprintf( dspec, "\\\\.\\PHYSICALDRIVE%u", disks);
         hDisk = CreateFile( dspec,
                             GENERIC_READ    | GENERIC_WRITE,
                             FILE_SHARE_READ | FILE_SHARE_WRITE,
                             NULL, OPEN_EXISTING,
                             FILE_ATTRIBUTE_NORMAL,
                             NULL);
         if (hDisk != INVALID_HANDLE_VALUE)
         {
            CloseHandle( hDisk);
         }
         else                                   // assume first invalid one
         {
            break;
         }
      }
   #elif defined (DOS32)
      disks = (USHORT) TxxBiosChar( TXX_PHYS_DISKS);
   #elif defined (UNIX)
      disks = dfsPhysDisks;                     // No API available, use DFSee
   #else                                        // cached value set at Init
      (void) DosPhysicalDisk( INFO_COUNT_DISKS, &disks,
                              DataLen, NULL, ParmLen);
   #endif
   RETURN (disks);
}                                               // end 'dfsCountPhysicalDisks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get number of physical disks (using DFSee cached value)
/*****************************************************************************/
USHORT dfsGetPhysicalDisks                      // RET   number of disks
(
   void
)
{
   ENTER();

   RETURN( dfsPhysDisks);
}                                               // end 'dfsGetPhysicalDisks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get total number of partitionable disks (physical or memory or ...)
/*****************************************************************************/
USHORT dfsGetDiskCount                          // RET   number of disks
(
   BOOL                verbose                  // IN    Report to screen too
)
{
   USHORT              pd;
   USHORT              vd = 0;
   USHORT              id = 0;

   ENTER();

   pd = dfsGetPhysicalDisks();

   #if !defined (DFS_PHYSDISK_ONLY)
   vd = dfsGetMemoryDisks();
   id = dfsGetPmImageDisks();
   #endif

   if (verbose)
   {
      TxPrint("Number of disks   : %s%hu%s", CBC, pd + vd + id, CNN);
      if ((vd != 0) || (id != 0))
      {
         TxPrint(", with %s%hu%s physical, %s%hu%s in-memory and %s%hu%s images",
                    CBG, pd, CNN, CBM, vd, CNN, CBY, id, CNN);
      }
      TxPrint("\n");
   }
   RETURN( (pd + vd + id));
}                                               // end 'dfsGetDiskCount'
/*---------------------------------------------------------------------------*/

#if !defined (DFS_PHYSDISK_ONLY)

/*****************************************************************************/
// Get number of partitionable media image disks
/*****************************************************************************/
USHORT dfsGetPmImageDisks                       // RET   number of disks
(
   void
)
{
   ENTER();

   RETURN( dfsImageDisks);
}                                               // end 'dfsGetPmImageDisks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get geometry, filehandle and accessinfo from 1st open of an image type disk
/*****************************************************************************/
ULONG dfsGetImageDiskInfo
(
   USHORT              idisk,                   // IN    image disk number
   char               *name,                    // OUT   image filename
   TXHFILE            *ifh,                     // OUT   image file handle
   void              **access,                  // OUT   image access info
   ULONG              *cyl,                     // OUT   Geo cylinders
   ULONG              *hds,                     // OUT   Geo heads
   ULONG              *spt                      // OUT   Geo sectors
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSI_DISK          *id = &dfsidisk[ idisk];  // image disk info

   ENTER();
   TRACES(( "Get info on IMAGE disk number: %hu\n", idisk));

   if ((idisk != 0) && (idisk < DFS_MAX_DISK) && (id->handle != 0)) // slot in use ?
   {
      *cyl    = id->Geo.C;
      *hds    = id->Geo.H;
      *spt    = id->Geo.S;
      *ifh    = id->handle;
      *access = id->accessInfo;

      TRACES(("handle: %u  accessinfo: 0x%8.8lx\n", id->handle, id->accessInfo));

      strcpy( name, id->name);
   }
   else
   {
      rc = DFS_NO_DEVICE;                       // not a valid idisk nr
   }
   RETURN (rc);
}                                               // end 'dfsGetImageDiskInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create Memory disk and add as partitionable medium
/*****************************************************************************/
ULONG dfsmCreateMemoryDisk
(
   int                 argc,                    // IN    valid parameters
   char               *argv1,                   // IN    first  argument
   char               *argv2,                   // IN    second argument
   char               *argv3,                   // IN    third  argument
   char               *argv4                    // IN    fourth argument
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               geoCyls  = dfstGeoCylinders( DFSTORE);
   ULONG               geoHeads = dfstGeoHeads(     DFSTORE);
   ULONG               geoSects = dfstGeoSectors(   DFSTORE);
   ULONG               geoBps   = dfstGetSectorSize(DFSTORE);
   BOOL                prestore = FALSE;
   USHORT              index    = 0;            // disk index
   TXLN                s1;
   TXLN                dc;
   FILE               *df;
   char               *s;
   BYTE                unit;                    // unit for size (k,m,g,c,h,s)
   DFSDISKINFO        *d        = dfsGetDiskInfo( SINF->disknr);
   ULN64               sectors  = (d) ? d->sectors : dfsGetLogicalSize();

   ENTER();

   sprintf( dc, "%llu,s", (d) ? d->sectors : dfsGetLogicalSize());

   strcpy( s1, "");
   if (argc > 1)
   {
      if ((*argv1 != '?') && (!isdigit( *argv1))) // argument is a filename
      {
         strcpy( s1, argv1);
         if ((df = fopen( s1, "rb")) == NULL)
         {
            sprintf( s1, "%s.pd1", argv1);
            if ((df = fopen( s1, "rb")) == NULL)
            {
               sprintf( s1, "%s.pd2", argv1);
               if ((df = fopen( s1, "rb")) == NULL)
               {
                  sprintf( s1, "%s.pd3", argv1);
                  if ((df = fopen( s1, "rb")) == NULL)
                  {
                     sprintf( s1, "%s.pd4", argv1);
                     if ((df = fopen( s1, "rb")) == NULL)
                     {
                        sprintf( s1, "%s.pd5", argv1);
                        if ((df = fopen( s1, "rb")) == NULL)
                        {
                           sprintf( s1, "%s.pd6", argv1);
                           if ((df = fopen( s1, "rb")) == NULL)
                           {
                              sprintf( s1, "%s.pd7", argv1);
                              if ((df = fopen( s1, "rb")) == NULL)
                              {
                                 sprintf( s1, "%s.pd8", argv1);
                                 df = fopen( s1, "rb");
                              }
                           }
                        }
                     }
                  }
               }
            }
         }

         if (df != NULL)                        // open binary file for read
         {
            DFSDISKINFO  pdx;                   // diskinfo from .PDx file
            DFSDISKINFO *d = &pdx;              // diskinfo, corrected pointer
            double       version = 0.0;         // invalid version

            //- FSCANF upto the first ';' char, then skip it and continue

            if (fscanf( df, "%*[^;]%*c version %lf", &version) != 1)
            {
               version = 5.06;                  // assume newer version
            }
            TRACES((".PDX version detected: %2.2lf\n", version));

            fseek( df, DFSPS_ASCII_SIZE, SEEK_SET);
            fread( &pdx, sizeof(DFSDISKINFO), 1, df);

            TRHEXS( 500,  &pdx,  sizeof(DFSDISKINFO), "diskinfo");

            TRACES(("Read Geo: H:%hu  S:%hu  cSC:%u  bps:%hu\n",
                     d->geoHeads, d->geoSecs, d->cSC, d->bpsector));

            if (((d->geoHeads * d->geoSecs) == d->cSC) &&
                 (d->geoCyls != 0) && (d->cSC != 0))
            {
               geoCyls  = d->geoCyls;
               geoHeads = d->geoHeads;
               geoSects = d->geoSecs;
               geoBps   = d->bpsector;
               sectors  = d->sectors;           // RAW disk size, 64bit

               if ((d->ebrHead != DFSPS_MAGIC1) || // diskinfo not written
                   (d->fspHead != DFSPS_MAGIC2)  ) // by 64bit aware version
               {
                  sectors &= 0xffffffff;        // clip to valid 32bits
               }                                // as used prior to 14.5
               prestore = TRUE;                 // auto RESTORE, possible GEO
            }
            else                                // pdx is incorrect
            {
               TxPrint( "Invalid GEO '%u %u * %u = %u 'in template '%s'\n",
                         d->geoCyls, d->geoHeads, d->geoSecs, d->cSC, s1);
               TxPrint( "File may be damaged in transport (CR/LF translation ?)\n");
               rc = DFS_BAD_STRUCTURE;
            }
            fclose( df);
         }
         else
         {
            TxPrint( "Memdisk template file *.PDx for '%s' not found\n", argv1);
            rc = DFS_NOT_FOUND;
         }
      }
      else                                      // manual specified size / geo
      {
         ULN64         cc;                      // cylinder count, 64bit

         if (*argv1 == '?')                     // same size as current object
         {
            sprintf( dc, "%llu,s", sectors);
         }
         else                                   // size from first argument
         {
            strcpy( dc, argv1);
         }
         if (argc > 2)
         {
            geoHeads = (ULONG)  max(dfsGetMcsNumber( argv2, 1), 1);
         }
         if (argc > 3)
         {
            geoSects = (ULONG)  max(dfsGetMcsNumber( argv3, 1), 1);
         }
         if (argc > 4)
         {
            geoBps   = (USHORT) max(dfsGetMcsNumber( argv4, 1), 1);
         }

         //- get specified size, in sectors. Will be MDISK raw size
         //- Need to set STORE geometry silently first, to honor CHS/BPS in appyNumberUnit ...
         dfstDiskGeometry( DFSTORE, 0, L32_NULL, geoHeads, geoSects, geoBps, TRUE, FALSE);
         sectors = TxaParseNumber( dc, DFSRADIX_SIZE, &unit);
         sectors = dfsApplyNumberUnit( sectors, (unit == TXA_DFUNIT) ? 'm' : unit, DFSTORE);

         //- convert specified size to a 32bit cylinder value, rounded down when needed
         cc = sectors / (geoHeads * geoSects);
         if (cc > ((ULN64) 0xffffffff))         // too large for ULONG
         {
            cc = 0xffffffff;
         }
         geoCyls = (ULONG) cc;

         sprintf( s1, "Using size/geo : 0x0%llx C:%u %s %s %s", sectors, geoCyls, argv2, argv3, argv4);
      }
   }
   else
   {
      sprintf( s1, "Like : %s", dfstStoreDesc2( DFSTORE));
   }
   if (rc == NO_ERROR)
   {
      rc = dfsMakeMemoryDisk( sectors, geoCyls, geoHeads, geoSects, geoBps, &index);
      if (rc == NO_ERROR)
      {
         USHORT vdnum = dfsGetDiskCount( FALSE); // new (last) disknr

         TxPrint( "\nNew inMemory disk : %sM%hu%s created as disknr: %hu with Geo CHS:%7lu %3lu %-3lu\n",
                   CBM, index, CNN,  vdnum, geoCyls, geoHeads, geoSects);

         dfsAddMediaMapping( DFSD_VIRT, index, 0, NULL, s1, TRUE);

         dfsReadDiskInfo( FDSK_QUIET);          // re-read diskinfo

         if (prestore)                          // from .PDx saved file
         {
            TxPrint( "Run PRESTORE from : %s%s%s to new inMemory disk %sM%hu%s (disk %hu)\n",
                      CBC, s1,    CNN, CBM, index, CNN,  vdnum);
            sprintf( dc, "prestore -c- %hu %s", vdnum, s1);
            if ((rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE)) == NO_ERROR)
            {
               TxPrint( "\nNew inMemory disk : %sV%hu%s (disk %hu) partitions initialized using %s%s%s\n",
                          CBM, index, CNN, vdnum, CBC, s1, CNN);

               if ((s = strrchr( s1, '.')) != NULL) // file extention ?
               {
                  *(++s) = 's';                 // change filename.pdX to
                  *(++s) = 'n';                 // list   filename.snX

                  if (TxFileExists( s1))        // list exists ?
                  {
                     TxPrint( "Run  IMPORT  from : %s%s%s to new inMemory disk %sM%hu%s (disk %hu)\n",
                               CBC, s1,    CNN, CBM, index, CNN,  vdnum);
                     sprintf( dc, "import %s -data", s1);
                     if ((rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE)) == NO_ERROR)
                     {
                        TxPrint( "\nNew inMemory disk : %sV%hu%s (disk %hu) imported sectors from list %s%s%s\n",
                                  CBM, index, CNN, vdnum, CBC, s1, CNN);
                     }
                  }
               }
               if ((s = strrchr( s1, '.')) != NULL) // file extention ?
               {
                  *(++s) = 'i';                 // change filename.snX to
                  *(++s) = 'm';                 // list   filename.imX

                  if (TxFileExists( s1))        // disk-image exists ?
                  {
                     TxPrint( "Run  RESTORE from : %s%s%s to new inMemory disk %sM%hu%s (disk %hu)\n",
                               CBC, s1,    CNN, CBM, index, CNN,  vdnum);
                     sprintf( dc, "restore \"%s\" 0 -S", s1);
                     if ((rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE)) == NO_ERROR)
                     {
                        TxPrint( "\nNew inMemory disk : %sM%hu%s (disk %hu) restored sectors from image %s%s%s\n",
                                  CBM, index, CNN, vdnum, CBC, s1, CNN);
                     }
                  }
               }
            }
            if (dfsa->geoCalc)
            {
               //- when geocalc ON, execute GEO command to get L-GEO from possible changed contents
               strcpy( dc, "geo");
               if ((rc = dfsMultiCommand( dc, 0, TRUE, FALSE, TRUE)) == NO_ERROR)
               {
                  TxPrint( "\nLogical geometry updated for possibly changed memory disk contents\n");
               }
            }
            TxPrint( "\n");
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsmCreateMemoryDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create RAW/Compressed Image disk and add as partitionable medium
/*****************************************************************************/
ULONG dfsmCreateImageDisk
(
   char               *image                    // IN    image filename
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXHFILE             handle   = 0;            // image file handle
   void               *access   = NULL;         // IMZ/VDI access info
   ULONG               geoCyls  = 0;
   ULONG               geoHeads = 0;
   ULONG               geoSects = 0;
   USHORT              index    = 0;            // disk index
   DFST_HANDLE         cst      = dfstGetDefaultStore();
   ULONG               files    = 1;

   ENTER();

   dfstSetDefaultStore(             DFST_SYSTEM); // select system store
   if ((rc = dfstOpenImageFile(     DFST_SYSTEM, image, TRUE, FALSE, TRUE, &files)) == NO_ERROR)
   {
      handle   = dfstGetVfHandle(   DFST_SYSTEM, TRUE); // take over the handle
      access   = dfstGetAccessInfo( DFST_SYSTEM, TRUE); // take over the info
      geoCyls  = dfstGeoCylinders(  DFST_SYSTEM);
      geoHeads = dfstGeoHeads(      DFST_SYSTEM);
      geoSects = dfstGeoSectors(    DFST_SYSTEM); // and get image geometry

      TRACES(( "Stolen  filehandle: %8.8lx  Access: %p\n", handle, access));
   }
   dfstClose( DFST_SYSTEM);                     // close last used disk
   dfstSetDefaultStore( cst);                   // reselect current store

   if (rc == NO_ERROR)
   {
      BOOL       createNewDisk = TRUE;
      DFS_GUID   this;
      DFS_GUID   link;

      //- check if it is a VDI type image file
      if (dfsGetVdiImageInfo( image, handle, &this, &link, NULL, NULL, NULL, NULL, NULL, NULL))
      {
         if (dfsVdiIsNextSnapShot( &link, &access))
         {
            TxPrint( "Add  snapshot VDI : %s in '%s'\n"
                     "to last VDI-chain : %s\n",
                      dfsFsUuidValueString( this), image, dfsFsUuidValueString( link));

            rc = dfsVdiAdd2CacheIndex( handle, image, TRUE, access, &files);
            if (rc == NO_ERROR)
            {
               USHORT  iDisk;                   // Image-Disk index 1..n
               USHORT  did;                     // DFSee disk-ID

               for (iDisk = 1; iDisk <= DFS_MAX_DISK; iDisk++)
               {
                  if (dfsidisk[ iDisk].accessInfo == access) // found 'our' Idisk
                  {
                     did = dfsRealDiskNr2Did( DFSD_VBOX, iDisk);

                     if (did != 0)              // valid disk ?
                     {
                        pMediaMap[ did].files = files; // update filecount in media
                     }
                     break;                     // should be one match only
                  }
               }
            }
            createNewDisk = FALSE;
         }
         else                                   // must be a BASE VDI file
         {
            DFS_GUID   zero;

            memset( zero, 0, DFS_GUID_LENGTH);
            if (memcmp( link, zero, DFS_GUID_LENGTH) != 0) // there IS a link
            {
               if ((dfsa->batch) || (TxConfirm( 5920,
                  "Parent VDI with UUID: %s is NOT present yet, "
                  "chain of BASE+snapshot/diff images will be incomplete.\n\n"
                  "Ignore parent chain and load snapshot VDI standalone ? [Y/N]: ",
                   dfsFsUuidValueString( link))))
               {
                  TxPrint( "Load snapshot VDI : %s in '%s'\n", dfsFsUuidValueString( this), image);
                  rc = dfsVdiInitCacheIndex( handle, image, TRUE, &access);
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
            else
            {
               TxPrint( "Loading  BASE VDI : %s in '%s'\n", dfsFsUuidValueString( this), image);
               rc = dfsVdiInitCacheIndex( handle, image, TRUE, &access);
            }
         }
      }
      //- Note: InitCacheIndex for IMZ is done in the OpenImageFile (can be partition IMZ)
      if ((rc == NO_ERROR) && (createNewDisk))  // Image NOT added to an existing disk
      {
         rc = dfsMakeImageDisk( geoCyls, geoHeads, geoSects, handle, access, image, &index);
         if (rc == NO_ERROR)
         {
            USHORT        idnum = dfsGetDiskCount( FALSE); // new (last) disknr
            DFSMEDIATYPE  mtype = DFSD_IRAW;    // default type RAW

            if (access != NULL)                 // not a RAW disk
            {
               if      (dfsGetImzImageInfo( image, handle, TRUE, NULL, NULL, NULL, NULL, NULL, NULL, NULL))
               {
                  mtype = DFSD_IMZD;
                  dfstSetReadOnly( cst, TRUE, TRUE);
               }
               else if (dfsGetVdiImageInfo( image, handle, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL))
               {
                  mtype = DFSD_VBOX;
               }
            }
            TxPrint( "\nNew %s Imagedisk: %sI%hu%s created as disknr: %hu "
                     "with Geo CHS:%7lu %3lu %-3lu\n", dfsMediaTypeDescr( mtype),
                      CBM, index, CNN,  idnum, geoCyls, geoHeads, geoSects);

            dfsAddMediaMapping( mtype, index, files, NULL, image, TRUE);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsmCreateImageDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make a new Image disk, using DFSee handle/accessInfo from OpenImageFile
/*****************************************************************************/
static ULONG dfsMakeImageDisk
(
   ULONG               cyl,                     // IN    Geo cylinders
   ULONG               hds,                     // IN    Geo heads
   ULONG               spt,                     // IN    Geo sectors
   TXHFILE             ifh,                     // IN    Image file handle
   void               *access,                  // IN    Access information
   char               *name,                    // IN    Image filename
   USHORT             *idisk                    // OUT   idisk number (or NULL)
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              ni;                      // next free disk handle
   DFSI_DISK          *id;                      // image disk info

   ENTER();

   if ((ni = dfsmGetFreeIdiskSlot()) != 0)      // free slot available
   {
      dfsImageDisks++;                          // update total Image-DISK count
      id = &dfsidisk[ ni];

      strcpy( id->name, name);                  // image filename

      id->Geo.C      = cyl;
      id->Geo.H      = hds;
      id->Geo.S      = spt;
      id->handle     = ifh;
      id->accessInfo = access;

      if (idisk != NULL)
      {
         *idisk = ni;
      }
   }
   else
   {
      rc = DFS_NO_DEVICE;                       // no room for new idisk
   }
   RETURN (rc);
}                                               // end 'dfsMakeImageDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Remove image disk and free all resources
/*****************************************************************************/
static ULONG dfsRemoveImageDisk
(
   DFSMEDIATYPE        itype,                   // IN    image type, RAW, IMZ etc
   USHORT              idisk                    // IN    image disk number
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSI_DISK          *id = &dfsidisk[ idisk];  // image disk info

   ENTER();

   if ((idisk != 0) && (idisk < DFS_MAX_DISK) && (id->handle != 0)) // slot in use ?
   {
      if (id->accessInfo != NULL)               // access info (IMZ index)
      {
         switch (itype)
         {
            case DFSD_IMZD:
               rc = dfsImzFreeCacheIndex( &id->accessInfo); // free cache/index data
               break;

            case DFSD_VBOX:
               rc = dfsVdiFreeCacheIndex( &id->accessInfo); // flush & free cache/index data
               break;

            default:
               break;
         }
      }
      TxClose( id->handle);                     // close real handle
      id->handle = 0;                           // free the used slot
      dfsImageDisks--;                          // adjust image disk count
   }
   else
   {
      rc = DFS_NO_DEVICE;
   }
   RETURN (rc);
}                                              // end 'dfsRemoveImageDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return first unused slot in the IDISK array
/*****************************************************************************/
static USHORT dfsmGetFreeIdiskSlot              // RET   index free slot or 0
(
   void
)
{
   USHORT              rc = 0;                  // 0 is non-found
   int                 i;

   ENTER();

   for (i = 1; i <= DFS_MAX_DISK; i++)
   {
      if (dfsidisk[i].handle == 0)              // slot is unused
      {
         rc = i;
         break;
      }
   }
   RETURN ( rc);
}                                               // end 'dfsmGetFreeIdiskSlot'
/*---------------------------------------------------------------------------*/
#endif


/*************************************************************************************************/
// Remove specified or all Memory disks including partitionable medium registration
/*************************************************************************************************/
ULONG dfsmDetachMedium
(
   USHORT              did,                     // IN    DFSee disk id
   BOOL                verbose
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (did == DFSM_UNMOUNT_ALL)
   {
      USHORT           disk;

      for (disk = 1; (disk < DFS_MAX_DISK) && (pMediaMapped != 0); disk++)
      {
         dfsDelMediaMapping( pMediaMapped, verbose);
      }
   }
   else                                         // remove specified
   {
      rc = dfsDelMediaMapping( did, verbose);
   }
   if (pMediaMapped != 0)
   {
      dfsReadDiskInfo( FDSK_QUIET);             // re-read all partition info
   }
   RETURN (rc);
}                                               // end 'dfsmDetachMedium'
/*-----------------------------------------------------------------------------------------------*/


#if defined (UNIX)
/*************************************************************************************************/
// Test if the specified name exists and can be opened, works for (RAW) image files too
/*************************************************************************************************/
BOOL dfsUxDeviceExists
(
   char               *devname,                 // IN    device name like /dev/hda
   BOOL                verbose
)
{
   BOOL                rc = FALSE;              // function return
   int                 stat_rc;
   int                 dh;
   struct stat         dstat;
   USHORT              st_mode;
   #if !defined (DARWIN)
      struct stat64    dstat64;
   #endif

   ENTER();
   TRACES(( "Verify device: '%s'\n", devname));

   if ((stat_rc = stat( devname, &dstat)) != -1)
   {
      TRHEXS( 70,  &dstat,  sizeof(dstat), "dstat");
      st_mode = (USHORT) dstat.st_mode;
   }

   #if !defined (DARWIN)
      else if (errno == EOVERFLOW)
      {
         // retry with 64-bit stat (latest Linux kernels fail on regular)
         TRACES(( "EOVERFLOW, retry with stat64 ...\n"));
         if ((stat_rc = stat64( devname, &dstat64)) != -1)
         {
            TRHEXS( 70,  &dstat64,  sizeof(dstat64), "dstat64");
            st_mode = (USHORT) dstat64.st_mode;
         }
      }
   #endif

   if (stat_rc != -1)                           // name exists
   {
      TRACES(( "stat OK, st_mode: %4.4lx (BLK = 0x6000)\n", st_mode));

      //- Note:   On OSX (Darwin) /dev/diskN is a BLK device, /dev/rdiskN is NOT!
      //-         But since DFSee reads/writes blocks anyway (sectors) there is no
      //-         difference in using one or the other, and using the /dev/diskN
      //-         allows us to use the same code as Linux (S_ISBLK)
      //- Update: But rdiskN device (RAW) is almost always MUCH faster than
      //-         the buffered diskN device

      //- if (S_ISBLK( st_mode))                // and is a block-device (may allow others!)
      {
         if ((dh = open( devname, O_RDONLY | O_LARGEFILE)) != -1)
         {
            TRACES(( "Open successful, handle: %hu\n", dh));
            close( dh);
            rc = TRUE;
         }
         else if (verbose)
         {
            TxPrint( "\nOpen of existing disk device '%s' failed : %s\n", devname, strerror(errno));

            #if defined (DARWIN)                // possible SIP protected disk0 (High Sierra)
            if ((strstr( devname, "disk0") != NULL) &&
                (TxOsVersion( NULL) >= TXVER_MACOS_HIGH_SIERRA))
            {
               TxPrint( "Your boot disk %s may be locked by System Integrity Protection (SIP)\n"
                        "Either boot from another (external) disk or temporarily disable SIP\n"
                        "using 'csrutil disable' after a Recovery boot (command + R)\n", devname);
            }
            else
            #endif
            {
               TxPrint( "You may need root rights to access it (su/sudo).\n");
            }
         }
         else
         {
            TRACES(( "Open fail, errno %hu = '%s'\n", errno, strerror(errno)));
         }
      }
   }
   else
   {
      TRACES(( "Stat(64) fail, errno %hu = '%s'\n", errno, strerror(errno)));
   }
   BRETURN (rc);
}                                               // end 'dfsUxDeviceExists'
/*-----------------------------------------------------------------------------------------------*/

#endif /* defined UNIX */
