//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFS STORE memory disk interface (previously called virtual disk)
//
// Author: J. van Wijk
//
// JvW  05-07-2000   Initial version, derived from DFSUFDSK
// JvW  18-10-2015   Updated to allow a size independant from GEO (partial cylinder tests)
// JvW  04-04-2017   Updated for 64bit sectornumbers and size
// JvW  07-07-2017   Updated for renaming VirtDisk to Mem-disk (and /dev/vdX to /dev/MdX)
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfsupart.h>                           // FS partition utilities
#include <dfsufdsk.h>                           // FS FDISK utilities
#include <dfs.h>                                // DFS navigation and defs
#include <dfsmdisk.h>                           // Disk store memory disks
#include <dfsutil.h>                            // DFS utility functions


typedef struct dfsm_sector
{
   struct dfsm_sector *Next;                    // next memory sector
   ULN64               Psn;                     // memory phys sector number
   BYTE                Sector[1];               // sector data, variable size
} DFSM_SECTOR;                                  // end of struct "dfsm_sector"

typedef struct dfsm_disk
{
   DFSM_SECTOR        *Mbr;                     // first sector, 0 == MBR
   DFSM_SECTOR        *Cur;                     // current sector, last access
   DFS_GEO             Geo;                     // memory geometry
   ULN64               Size;                    // RAW size in sectors
   ULN64               Used;                    // sectors in use
   ULONG               Reads;                   // nr of read operations
   ULONG               Found;                   // real sectors found
   ULONG               Writes;                  // nr of write operations
} DFSM_DISK;                                    // end of struct "dfsm_disk"


// Note: Starting with version 7.xx, free-slot management will be used
// A free slot-position (Used == 0) will be searched on Make()
// Any memory-disk can be removed at any time, not just the last one
// The memory disk-number still is the index in the table,
// with the first one (0) unused. 0 can be used as 'remove all' on Remove

static USHORT          dfsMemDisks = 0;         // number of active memory disks


static DFSM_DISK       dfsmdisk[DFS_MAX_DISK] = {{0}};


// Return first unused slot in the MDISK array
static USHORT dfsmGetFreeVirtSlot               // RET   index free slot or 0
(
   void
);

// Find specified sector in a linked list, return lower-one when not found
static BOOL dfsmFindSector                      // RET   sector found
(
   DFSM_SECTOR        *first,                   // IN    start sector in list
   ULN64               psn,                     // IN    wanted psn
   DFSM_SECTOR       **found                    // OUT   resulting sector
);

// Add a new sector (psn) to the linked list at specified insertion point (bs)
static ULONG dfsmAddSector
(
   DFSM_SECTOR        *bs,                      // IN    sector before new one
   ULN64               psn,                     // IN    target psn
   USHORT              bps,                     // IN    bytes per sector
   DFSM_SECTOR       **new                      // OUT   added sector
);


/*****************************************************************************/
// Return number of partitionable memory disks
/*****************************************************************************/
USHORT dfsGetMemoryDisks                        // RET   nr of memory disks
(
   void
)
{
   ENTER();

   RETURN ( dfsMemDisks);
}                                               // end 'dfsMemoryDisks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make a new memory disk, with an empty (0x00) first MBR sector
/*****************************************************************************/
ULONG dfsMakeMemoryDisk
(
   ULN64               size,                    // IN    Size in sectors, or 0
   ULONG               cyl,                     // IN    Geo cylinders
   ULONG               hds,                     // IN    Geo heads
   ULONG               spt,                     // IN    Geo sectors
   USHORT              bps,                     // IN    Bytes per sector
   USHORT             *mdisk                    // OUT   mdisk number (or NULL)
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              nv;                      // next free disk handle
   DFSM_DISK          *md;                      // memory disk info

   ENTER();

   if ((nv = dfsmGetFreeVirtSlot()) != 0)       // free slot available
   {
      dfsMemDisks++;                            // update total MDISK count
      md = &dfsmdisk[ nv];

      md->Geo.C = cyl;
      md->Geo.H = hds;
      md->Geo.S = spt;
      md->Geo.B = bps;

      md->Size = cyl * hds * spt;
      if (size)
      {
         if (md->Size > size)                   // cyl too large for size, adjust
         {
            md->Geo.C = size / (hds * spt);     // based on size, rounded down
         }
         md->Size = size;
      }

      if ((md->Mbr = malloc( sizeof( DFSM_SECTOR) + bps)) != NULL)
      {
         md->Cur       = md->Mbr;
         md->Mbr->Next = NULL;
         md->Mbr->Psn  = 0;

         md->Used      = 1;
         md->Reads     = 0;
         md->Writes    = 0;
         md->Found     = 0;

         memset( md->Mbr->Sector, 0, bps);

         if (mdisk   != NULL)
         {
            *mdisk    = nv;
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_NO_DEVICE;                       // no room for new mdisk
   }
   RETURN (rc);
}                                               // end 'dfsMakeMemoryDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Remove memory disk and free all resources
/*****************************************************************************/
ULONG dfsRemoveMemoryDisk
(
   USHORT              mdisk                    // IN    memory disk number
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSM_DISK          *md = &dfsmdisk[ mdisk];  // memory disk info

   ENTER();

   if ((mdisk != 0) && (mdisk < DFS_MAX_DISK) && (md->Used != 0)) // slot in use ?
   {
      DFSM_SECTOR     *ms;
      DFSM_SECTOR     *ns;                      // next memory sector

      for (ms = md->Mbr; ms != NULL; ms = ns)
      {
         ns  = ms->Next;                        // read next while still valid
         free( ms);                             // free the sector memory
      }
      md->Used = 0;                             // free the used slot
      md->Mbr  = NULL;
      dfsMemDisks--;                            // adjust memory disk count
   }
   else
   {
      rc = DFS_NO_DEVICE;
   }
   RETURN (rc);
}                                              // end 'dfsRemoveMemoryDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get geometry specified on creation of a memory disk
/*****************************************************************************/
ULONG dfsGetMdiskGeometry
(
   USHORT              mdisk,                   // IN    memory disk number
   ULN64              *size,                    // OUT   Size in sectors
   ULONG              *cyl,                     // OUT   Geo cylinders
   ULONG              *hds,                     // OUT   Geo heads
   ULONG              *spt,                     // OUT   Geo sectors
   USHORT             *bps                      // OUT   Bytes per sector
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSM_DISK          *md = &dfsmdisk[ mdisk];  // memory disk info

   ENTER();

   if ((mdisk != 0) && (mdisk < DFS_MAX_DISK) && (md->Used != 0)) // slot in use ?
   {
      *size = md->Size;
      *cyl  = md->Geo.C;
      *hds  = md->Geo.H;
      *spt  = md->Geo.S;
      *bps  = md->Geo.B;
   }
   else
   {
      rc = DFS_NO_DEVICE;                       // not a valid mdisk nr
   }
   RETURN (rc);
}                                               // end 'dfsGetMdiskGeometry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read one sector from the specified memory disk (empty when not present)
/*****************************************************************************/
ULONG dfsReadMdiskSector
(
   USHORT              mdisk,                   // IN    memory disk number
   ULN64               psn,                     // IN    wanted psn
   ULONG               bps,                     // IN    bytes per sector (buffer)
   BYTE               *data                     // OUT   sector data
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSM_DISK          *md = &dfsmdisk[ mdisk];  // memory disk info

   ENTER()

   if ((mdisk != 0) && (mdisk < DFS_MAX_DISK) && (md->Used != 0)) // slot in use ?
   {
      DFSM_SECTOR     *ms = (md->Cur->Psn <= psn) ? md->Cur : md->Mbr;

      TRACES(("PSN:%llx BPS:%u md-Geo.B:%u data:%8.8x\n", psn, bps, md->Geo.B, data));

      if (dfsmFindSector( ms, psn, &ms))
      {
         // read at most BPS or Geo.B bytes, whichever is smaller (avoid traps)
         memcpy( data, ms->Sector, min( md->Geo.B, bps));
         md->Cur = ms;                          // update current position
         md->Found++;
      }
      else
      {
         memset( data, 0,          min( md->Geo.B, bps));
      }
      md->Reads++;
   }
   else
   {
      rc = DFS_NO_DEVICE;                       // not a valid mdisk nr
   }
   RETURN (rc);
}                                               // end 'dfsReadMdiskSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write one sector to specified memory disk
/*****************************************************************************/
ULONG dfsWriteMdiskSector
(
   USHORT              mdisk,                   // IN    memory disk number
   ULN64               psn,                     // IN    wanted psn
   ULONG               bps,                     // IN    bytes per sector (buffer)
   BYTE               *data                     // IN    sector data
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSM_DISK          *md = &dfsmdisk[ mdisk];  // memory disk info

   ENTER()

   if ((mdisk != 0) && (mdisk < DFS_MAX_DISK) && (md->Used != 0)) // slot in use ?
   {
      DFSM_DISK       *md = &dfsmdisk[ mdisk];  // memory disk info
      DFSM_SECTOR     *ms = (md->Cur->Psn <= psn) ? md->Cur : md->Mbr;

      TRACES(("PSN:%llx BPS:%u md-Geo.B:%u data:%8.8x\n", psn, bps, md->Geo.B, data));

      if (dfsmFindSector( ms, psn, &ms) == FALSE)
      {
         ULONG        *u  = (ULONG *) data;
         ULONG         i;

         for (i = 0; i < (bps / sizeof( ULONG)); i++, u++) // check for non-empty sector
         {
            if (*u)                             // at least one non-zero value
            {
               rc = dfsmAddSector( ms, psn, md->Geo.B, &ms); // add when not present
               if (rc == NO_ERROR)
               {
                  memcpy( ms->Sector, data, min( md->Geo.B, bps));
                  md->Cur = ms;                 // update current position
                  md->Used++;
               }
               break;                           // only once!  exit loop
            }
         }
      }
      else                                      // existing sector
      {
         memcpy( ms->Sector, data, min( md->Geo.B, bps));
         md->Cur = ms;                          // update current position
         md->Found++;
      }
      md->Writes++;
   }
   else
   {
      rc = DFS_NO_DEVICE;                       // not a valid mdisk nr
   }
   RETURN (rc);
}                                               // end 'dfsWriteMdiskSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show status and statistics for all memory disks
/*****************************************************************************/
void dfsShowMdiskStatus
(
   USHORT              select                   // IN    Mdisk number, 0=ALL
)
{
   DFSM_DISK          *md;
   USHORT              mdisk;
   TXLN                text;
   double              perc = 100;

   ENTER();

   if (select == 0)
   {
      TxPrint("\nNumber of Mdisks  : %s%hu%s\n", CBM, dfsMemDisks, CNN);
   }
   for (mdisk = 1; mdisk < DFS_MAX_DISK; mdisk++)
   {
      if ((select == 0) || (select == mdisk))
      {
         md = &dfsmdisk[ mdisk];                // memory disk info

         if (md->Used != 0)                     // Slot in use ?
         {
            if (select == 0)                    // all mdisks, separate them
            {
               TxPrint( "\n");
            }
            sprintf( text, "Geo Mdisk %sM%-2hu%s Cyl :%9u H:%3u S:%-3u= %4.2lf MiB ",
                     CBM, mdisk, CNN, md->Geo.C,  md->Geo.H,  md->Geo.S,
                                      TXSMIB((    md->Geo.H * md->Geo.S), md->Geo.B));
            dfstrSz64( text, "Size: ", md->Size, "");
            TxPrint( "%s\n", text);

            dfsSz64("Non-zero  sectors : ",     md->Used, "  ");
            TxPrint("Last PSN : 0x%12.12llx\n", md->Cur->Psn);

            if ((md->Reads + md->Writes) != 0)
            {
               perc = ((double)(md->Found * 100) / (double)(md->Reads + md->Writes));
            }
            TxPrint("Total nr of Reads : %-11u    "
                               "Writes : %-11u  "
                             "SecFound : %5.1lf%%\n", md->Reads, md->Writes, perc);
         }
      }
   }
   if (select == 0)
   {
      TxPrint( "\n");
   }
   VRETURN ();
}                                               // end 'dfsShowMdiskStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return first unused slot in the MDISK array
/*****************************************************************************/
static USHORT dfsmGetFreeVirtSlot               // RET   index free slot or 0
(
   void
)
{
   USHORT              rc = 0;                  // 0 is non-found
   int                 i;

   ENTER();

   for (i = 1; i < DFS_MAX_DISK; i++)
   {
      if (dfsmdisk[i].Used == 0)                // slot is unused
      {
         rc = i;
         break;
      }
   }
   RETURN ( rc);
}                                               // end 'dfsmGetFreeVirtSlot'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find specified sector in a linked list, return lower-one when not found
/*****************************************************************************/
static BOOL dfsmFindSector                      // RET   sector found
(
   DFSM_SECTOR        *first,                   // IN    start sector in list
   ULN64               psn,                     // IN    wanted psn
   DFSM_SECTOR       **found                    // OUT   resulting sector
)
{
   BOOL                rc = FALSE;              // function return
   DFSM_SECTOR        *ms;

   for (ms = first; ms != NULL; ms = ms->Next)
   {
      if (( ms->Psn       == psn ) ||           // exact hit
          ( ms->Next      == NULL) ||           // at last in list
          ( ms->Next->Psn >  psn )  )           // or next has higher psn
      {
         break;
      }
   }
   if ((ms != NULL) && (ms->Psn == psn))
   {
      rc = TRUE;                                // exact hit
   }
   *found = ms;
   return (rc);
}                                               // end 'dfsmFindSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add a new sector (psn) to the linked list at specified insertion point (bs)
/*****************************************************************************/
static ULONG dfsmAddSector
(
   DFSM_SECTOR        *bs,                      // IN    sector before new one
   ULN64               psn,                     // IN    target psn
   USHORT              bps,                     // IN    bytes per sector
   DFSM_SECTOR       **new                      // OUT   added sector
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSM_SECTOR        *ms;

   ENTER();

   if ((ms = malloc( sizeof( DFSM_SECTOR) + bps)) != NULL)
   {
      TRACES(("Added: %8.8x after %8.8x for psn: %llx\n", ms, bs, psn));
      ms->Next = bs->Next;                      // link remaining sectors
      bs->Next = ms;                            // resulting new sector
      ms->Psn  = psn;

      *new = ms;
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsmAddSector'
/*---------------------------------------------------------------------------*/

