//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// NTFS dump & display Sector Lookup Table
//
// Author: J. van Wijk
//
// JvW  21-05-1997 Initial version, derived from DFSHPFS
// JvW  04-04-2017 Updated LSN to 64bit, MFT-numbers to 64bit
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsantfs.h>                           // NTFS analysis definitions
#include <dfsuntfs.h>                           // NTFS utility functions

#include <dfslntfs.h>                           // NTFS sector lookup



// Add a file from the MFT to the SLT array (callback)
static ULONG dfsNtfsAddMftFile
(
   ULN64               mftlsn,                  // IN    lsn
   ULN64               dummy,                   // IN    dummy
   char               *strdata,                 // IN    descr "mft"
   void               *ul                       // IN    ptr to MFT-nr
);

/*****************************************************************************/
// Add info to sector-lookup-table for Ntfs filesystem
/*****************************************************************************/
ULONG dfsNtfsSltBuild
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    pointer parameter
)
{
   ULONG               rc    = NO_ERROR;
   ULN64               slack = dfsNtfsClust2Lsn( ntfs->Clust);
   ULN64               last  = ntfs->SpareLsn;

   ENTER();

   if (dfsa->FsUnallocated == 0)                // probably not set yet ...
   {
      DFSFNCALL( dfsa->FsAllocDisplay, 0,0, "@", NULL);
   }
   dfsSlTableAdd( last +1, DFS64MAX,      ST_FINAL, LSN_BOOTR, 0, 0);

                                                // Approximate sectors todo:
   dfsSlTableFsInit( last -                     // RAW size of FS
                     dfsa->FsUnallocated);      // minus known freespace

   if (slack < last)                            // some sectors between last
   {                                            // cluster and final bootsec
      dfsSlTableAdd( slack, last - slack, ST_CLSLACK, LSN_BOOTR, 0, 0);
   }
   dfsSlTableAdd( last,    1,             ST_BOOTR, LSN_BOOTR, 0, 0);

   //- Note: MftIterator does a linear listing of the MFT file.
   //- It could be replaced with a real Btree-iterator using the same
   //- logic as the dfsNtfsFindPath()/dfsNtfsDirIndx2Entry() functions
   //- using recursion to read nested INDX block instead of in-line
   //- This will probably be much slower though, and is not needed
   //- for building the SLT ...

   rc = dfsNtfsMftIterator( dfsNtfsAddMftFile, 0, L64_NULL);

   RETURN (rc);
}                                               // end 'dfsNtfsSltBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display one line of error description for each bit in error-flag
/*****************************************************************************/
ULONG dfsNtfsDispError
(
   ULN64               flags,                   // IN    error flag
   ULN64               d2,                      // IN    dummy
   char               *lead,                    // IN    leading string
   void               *param                    // IN    pointer parameter
)
{
   ENTER();

   if      (flags & EF_BAD_FIXUP_MFT)
   {
      TxPrint("%sMFT record is damaged, bad fixup value in last 2 bytes\n", lead);
   }
   else if (flags & EF_BAD_MFT_STYPE)
   {
      TxPrint("%sMFT record might be damaged, unrecognized contents\n", lead);
   }
   RETURN (NO_ERROR);
}                                               // end 'dfsNtfsDispError'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add a file from the MFT to the SLT array (callback)
/*****************************************************************************/
static ULONG dfsNtfsAddMftFile
(
   ULN64               mftlsn,                  // IN    lsn
   ULN64               dummy,                   // IN    dummy
   char               *strdata,                 // IN    descr "mft"
   void               *ul                       // IN    ptr to MFT-nr
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st;                      // type of sector (MFT)
   DFSISPACE           MftSp;
   ULN64              *pMftNr = (ULN64 *) ul;   // ptr to MFT-number

   ENTER();
   TRACES(("mftlsn: %llx   mft: %llx\n", mftlsn, *pMftNr));

   //- to be refined, if the xxAlloc function with ALLATTRS now also
   //- correctly gets all external attributes in secondary MFT records
   //- through the attribute list, the MFTSEC* should be skipped to
   //- avoid double accounting in the SLT!

   if ((rc = dfsNtfsMftLsn2Alloc( mftlsn, MA_ALLATTRS, IA_ANY, &st, &MftSp)) == NO_ERROR)
   {
      TRACES(( "extents: %lu  type: '%c'\n", MftSp.chunks, st));
      if ((MftSp.chunks > 0) && ((st == ST_MFTFILE) || (st == ST_MFTDIRB)))
      {
         //- Note: secondary MFT-records and deleted ones are ignored here!
         //-       Avoids double accounting for the secondaries as well.
         //-       It DOES require MA_ALLATTRS to include attributes in
         //-       the secondary MFT-records as well!

         //- to be refined for DFSSPACE use ?
         dfsSlTableSpaceAdd( MftSp.chunks, MftSp.space, ST_SINFO, mftlsn, 0, 0);
      }
      TxFreeMem( MftSp.space);                  // free S_SPACE structure
   }
   else if (rc == DFS_BAD_STRUCTURE)            // bad fixups, or not an MFT
   {                                            // sector-type at all!
      switch (st)
      {
         case ST_MFTFILE:
         case ST_MFTSECF:
         case ST_MFTSECD:
         case ST_MFTDIRB:
         case ST_MFTEMPT:
            dfsSlTableAdd( mftlsn, ntfs->MftRecSize, st, 0, 0, EF_BAD_FIXUP_MFT);
            break;

         case ST_MFGHOST:                       // ignore fixup errors on
         case ST_MFTDELD:                       // deleted/ghost MFT records
         case ST_MFTDELF:
            break;

         default:                               // wrong type or ZEROED sector
            if (dfsNtfsMftMapped( *pMftNr))
            {
               dfsSlTableAdd( mftlsn, ntfs->MftRecSize, ST_MFTFILE, 0, 0, EF_BAD_MFT_STYPE);
            }
            break;
      }
      rc = NO_ERROR;                            // ignore error and continue!
   }
   else if (rc != DFS_ALLOC_ERROR)
   {
      dfsSlTableAdd( mftlsn, ntfs->MftRecSize, ST_MFTFILE, 0, 0, EF_BAD_MFT_STYPE);
      rc = NO_ERROR;                            // ignore error and continue!
   }
   if (dfsSlTableStatus( NULL) == SLT_STOPPING)
   {
      rc = DFS_QUIT;                            // stop the iterator
   }                                            // RC will be discarde later!
   RETURN( rc);
}                                               // end 'dfsNtfsAddMftFile'
/*---------------------------------------------------------------------------*/

