//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// JFS dump & display Sector Lookup Table
//
// Author: J. van Wijk
//
// JvW  23-02-2018 Name/Parent build now iterates over all Inodes (not root tree)
// JvW  22-02-2018 Changed recursion guard in SLT/NP build to decrement to 0
// JvW  01-02-2006 Start addition of real JFS SLT functionality
// JvW  19-04-2001 Initial version
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsjfs.h>                             // JFS disk structures
#include <dfsajfs.h>                            // JFS analysis definitions
#include <dfsujfs.h>                            // JFS utility functions
#include <dfsljfs.h>                            // JFS sector lookup

#define DFS_JFS_RECURSE_MAX   100               // recurse up to 100 levels from root

#define DFS_JFS_NO_PARENT     0                 // not a valid parent Inode#

// Walk Inodes, and for the directories, find name/inode pairs for NP cache
static ULONG dfsJfsNpWalkInodes
(
   ULONG              *inodes                   // INOUT Inodes added to cache
);

// Add Inode-map summary (IMAP) to SLT
static void dfsJfsSltIMap
(
   BYTE                stype,                   // IN    type of sector
   S_IMAP             *data,                    // IN    data
   ULONG               fn                       // IN    ref sector LSN
);

// Add Inode info + data to  SLT array, recurse into directories when allowed
static ULONG dfsJfsSltInode
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   BYTE                stype,                   // IN    type for the data
   ULONG               in,                      // IN    Inode LSN
   ULONG               parent,                  // IN    referencing inode nr
   ULN64               ref,                     // IN    referencing LSN (DIR)
   ULONG               ef                       // IN    error flag
);

// Add Inode name + parent to cache, recurse into directories when allowed
static ULONG dfsJfsNpInode
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               lsn,                     // IN    Inode LSN
   ULONG               parent,                  // IN    referencing inode nr
   ULONG              *inodes                   // INOUT Inodes added to cache
);

// Add Directory Btree+ page to SLT, add files and recurse into directories
static ULONG dfsJfsSltDtreePage                 // RET   rc = 0 if type match
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               pageLsn,                 // IN    start LSN of page
   BYTE                stype,                   // IN    type of the data
   ULONG               parent,                  // IN    referencing inode nr
   ULONG               ref                      // IN    referencing Inode LSN
);

// Handle Dir Btree+ page, adding to Inode Name + Parent cache, may recurse
static ULONG dfsJfsNpDtreePage                  // RET   rc = 0 if type match
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULN64               pageLsn,                 // IN    start LSN of page
   ULONG               parent,                  // IN    referencing inode nr
   ULONG              *inodes                   // INOUT Inodes added to cache
);

// Add Inodes for a sequence of JFS Directory entries to the SLT, may recurse
ULONG dfsJfsSltDir
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   BYTE                stype,                   // IN    type of the data
   ULONG               parent,                  // IN    referencing inode nr
   ULN64               ref,                     // IN    referencing Inode LSN
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes                // IN    tree internal nodes
);

// Add Inodes for a sequence of JFS Directory entries to NP-cache, may recurse
ULONG dfsJfsNpDir
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               parent,                  // IN    referencing inode nr
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes,               // IN    tree internal nodes
   ULONG              *inodes                   // INOUT Inodes added to cache
);


// Status for write-back Inodes with added magic name/parent on SLT build
static ULONG dfsJfsWriteMagicNames = DFS_JFS_M_OFF;

/*****************************************************************************/
// Add info to sector-lookup-table for Jfs filesystem
/*****************************************************************************/
ULONG dfsJfsSltBuild
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    not used
)
{
   ULONG               rc    = NO_ERROR;
   ULONG               slack = dfsJfsBlock2Lsn( jfs->TotalBlocks);
   ULONG               last  = jfs->Sect;
   ULONG               ef    = 0;
   ULONG               ino;                     // first aggregate inode LSN
   ULONG               sag;                     // first 2nd aggr  inode LSN
   ULONG               fs1;                     // first Fileset-1 inode LSN
   ULONG               root;                    // inode lsn to use as root
   ULONG               freeSect = jfs->Bm.FreeSect;

   ENTER();

   if (TxaOptSet('M'))                          // add magic filenames to
   {                                            // the Inodes for recovery
      if (dfsa->FsDirtyStatus != DFSTAT_DIRTY)
      {
         dfsJfsWriteMagicNames = TxaOptNum( 'M', NULL, DFS_JFS_MBOTH);
      }
      else                                      // not allowed, FS mounted
      {
         TxNamedMessage( !dfsa->batch, 5033, " ERROR: Filesystem mounted ",
            "Adding filenames to the Inodes for later recovery/undelete "
            "is not possible since the filesystem is in a MOUNTED state.\n\n"
            "Use of the '-M' option is restricted to CLEAN and UNMOUNTED "
            "filesystems");
      }
   }
   dfsSlTableAdd( jfs->Sect,              DFS64MAX,  ST_FINAL, LSN_BOOTR, 0, 0);

   if (jfs->Ic == NULL)                         // no name/parent cache yet
   {
      jfs->Ic = TxAlloc( jfs->Fs1.Max, sizeof( DFSJFSINAME));
   }

   if ((dfsa->FsUnallocated != 0) && (dfsa->FsUnallocated < freeSect))
   {
      freeSect = dfsa->FsUnallocated;           // value from bitmap analysis
   }
                                                // Approximate sectors todo:
   dfsSlTableFsInit( jfs->Sect     +            // RAW size of FS
                     jfs->Fs1.Used + 8 -        // FS1 Inodes  + 2 * Inags
                     freeSect);                 // minus known freespace

   if (slack < last)                            // some sectors between last
   {                                            // block and last sector
      dfsSlTableAdd( slack, last - slack, ST_SLACK, LSN_BOOTR, 0, 0);
   }
   dfsSlTableAdd( LSN_BOOTR +1,     (LSN_JFSUP -1),  ST_JBOOT, L64_NULL,  0, 0);

   if (jfs->sup->SignatNum != '1')
   {
      ef = EF_BAD_SUPER_SIG;
   }
   dfsSlTableAdd( LSN_JFSUP,              JFS_PSECT, ST_JFSUP, L64_NULL,  0, ef);
   dfsSlTableAdd( LSN_JFSU2,              JFS_PSECT, ST_JFSUP, L64_NULL,  0, 0);

   dfsSlTableAdd( jfs->sup->SaInodeMap.lo    * jfs->SectorsPerBlock,
                  jfs->sup->SaInodeMap.len   * jfs->SectorsPerBlock,
                                               ST_JIMA2, LSN_JFSUP, 0, 0);

   dfsSlTableAdd( jfs->sup->SaInodeTable.lo  * jfs->SectorsPerBlock,
                  jfs->sup->SaInodeTable.len * jfs->SectorsPerBlock,
                                               ST_JIXA2, LSN_JFSUP, 0, 0);

   dfsSlTableAdd( jfs->sup->InlineLog.lo     * jfs->SectorsPerBlock,
                  jfs->sup->InlineLog.len    * jfs->SectorsPerBlock,
                                               ST_JFSCK, LSN_JFSUP, 0, 0);

   dfsSlTableAdd( jfs->sup->FsckSpace.lo     * jfs->SectorsPerBlock,
                  jfs->sup->FsckSpace.len    * jfs->SectorsPerBlock,
                                               ST_JINLL, LSN_JFSUP, 0, 0);

   //- add the primary aggregate Inodes including their tree pages and data
   ino = dfsJfsInode2Lsn( 0, &jfs->Agg);        // first aggregate Inode
   dfsJfsSltInode( 0, ST_JIMAG, ino + 1, 0, ino + 1, 0); // SELF aggregate Inode
   dfsJfsSltInode( 0, ST_JBLAM, ino + 2, 0, ino + 1, 0); // Block Allocation Map
   dfsJfsSltInode( 0, ST_JINLL, ino + 3, 0, ino + 1, 0); // In-line Log file
   dfsJfsSltInode( 0, ST_BADSC, ino + 4, 0, ino + 1, 0); // Bad-blocks  file
   dfsJfsSltInode( 0, ST_JIMFS, ino +16, 0, ino + 1, 0); // SUPER Inode Fileset-1

   //- add the secondary aggregate Inodes including their tree pages (meta)
   sag = jfs->sup->SaInodeTable.lo * jfs->SectorsPerBlock; // 2nd aggr Inodes
   dfsJfsSltInode( 0, ST_JI2AG, sag + 1, 0, LSN_JFSUP, 0); // SELF aggregate Inode
   dfsJfsSltInode( 0, ST_JI2AG, sag + 2, 0, LSN_JFSUP, 0); // Block Allocation Map
   dfsJfsSltInode( 0, ST_JI2AG, sag + 3, 0, LSN_JFSUP, 0); // In-line Log file
   dfsJfsSltInode( 0, ST_JI2AG, sag + 4, 0, LSN_JFSUP, 0); // Bad-blocks  file
   dfsJfsSltInode( 0, ST_JI2AG, sag +16, 0, LSN_JFSUP, 0); // SUPER Inode Fileset-1

   //- add the allocated Inode extents for aggregate and fileset1
   dfsJfsSltIMap(ST_JIXAG, &jfs->Agg, ino + 1); // aggregate Inode extents
   dfsJfsSltIMap(ST_JIXFS, &jfs->Fs1, ino +16); // Fileset-1 Inode extents

   //- add the non-file fileset1 Inodes including their tree pages and data
   fs1 = dfsJfsInode2Lsn( 0, &jfs->Fs1);        // first fileset-1 Inode
   dfsJfsSltInode( 0, ST_JIMFS, fs1 + 1, 0, ino +16, 0); // Fs1 super extension
   dfsJfsSltInode( 0, ST_JACLD, fs1 + 3, 0, ino +16, 0); // ACL file Fileset-1

   root = TxaOptNum( 'L', NULL, fs1 + JFS_ROOT_INODE); // root inode lsn
   if (jfs->Ic != NULL)
   {
      jfs->Ic[JFS_SYS_RESV].Parent = JFS_SYS_PARENT;
      jfs->Ic[JFS_SYS_RESV].Name   = TxAlloc(1, strlen(JFS_TXT_RESV) +1);
      strcpy( jfs->Ic[JFS_SYS_RESV].Name,              JFS_TXT_RESV);

      jfs->Ic[JFS_SYS_MAPX].Parent = JFS_SYS_PARENT;
      jfs->Ic[JFS_SYS_MAPX].Name   = TxAlloc(1, strlen(JFS_TXT_MAPX) +1);
      strcpy( jfs->Ic[JFS_SYS_MAPX].Name,              JFS_TXT_MAPX);

      jfs->Ic[JFS_SYS_ACLI].Parent = JFS_SYS_PARENT;
      jfs->Ic[JFS_SYS_ACLI].Name   = TxAlloc(1, strlen(JFS_TXT_ACLI) +1);
      strcpy( jfs->Ic[JFS_SYS_ACLI].Name,              JFS_TXT_ACLI);

      jfs->Ic[JFS_ROOT_INODE].Name   = TxAlloc( 1, 1); // ROOT name ""
      jfs->Ic[JFS_ROOT_INODE].Parent = JFS_ROOT_INODE; // by convention
   }
   rc = dfsJfsSltInode( DFS_JFS_RECURSE_MAX, ST_FDATA, root, JFS_ROOT_INODE, ino +16, 0);

   RETURN (rc);
}                                               // end 'dfsJfsSltBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Build the Name+Parent cache for JFS filesystem (stripped down SLT build)
// Note: needs to be abortable when not started as a thread!
/*****************************************************************************/
ULONG dfsJfsNpBuild                             // RET   possibbly ABORTED
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    not used
)
{
   ULONG               rc    = NO_ERROR;
   ULONG               fs1;                     // first Fileset-1 inode LSN
   ULONG               root;                    // inode lsn to use as root

   ENTER();

   if (TxaOption( TXA_O_RESET))
   {
      if (jfs->Ic != NULL)
      {
         int              i;

         for (i = 0; i < jfs->Fs1.Max; i++)
         {
            TxFreeMem( jfs->Ic[i].Name);        // free cached Inode names
         }
         TxFreeMem( jfs->Ic);                   // free name/parent cache
      }
   }

   if (jfs->Ic == NULL)                         // no name/parent cache yet
   {
      jfs->Ic = TxAlloc( jfs->Fs1.Max, sizeof( DFSJFSINAME));

      if (jfs->Ic != NULL)
      {
         ULONG         iDone = 0;               // Inodes done
         ULONG         pFlag = (TxaOption(TXA_O_DEBUG)) ? DFSP_STAT : DFSP_BARS;

         jfs->Ic[JFS_SYS_RESV].Parent = JFS_SYS_PARENT;
         jfs->Ic[JFS_SYS_RESV].Name   = TxAlloc(1, strlen(JFS_TXT_RESV) +1);
         strcpy( jfs->Ic[JFS_SYS_RESV].Name,              JFS_TXT_RESV);

         jfs->Ic[JFS_SYS_MAPX].Parent = JFS_SYS_PARENT;
         jfs->Ic[JFS_SYS_MAPX].Name   = TxAlloc(1, strlen(JFS_TXT_MAPX) +1);
         strcpy( jfs->Ic[JFS_SYS_MAPX].Name,              JFS_TXT_MAPX);

         jfs->Ic[JFS_SYS_ACLI].Parent = JFS_SYS_PARENT;
         jfs->Ic[JFS_SYS_ACLI].Name   = TxAlloc(1, strlen(JFS_TXT_ACLI) +1);
         strcpy( jfs->Ic[JFS_SYS_ACLI].Name,              JFS_TXT_ACLI);

         jfs->Ic[JFS_ROOT_INODE].Name   = TxAlloc( 1, 1); // ROOT name ""
         jfs->Ic[JFS_ROOT_INODE].Parent = JFS_ROOT_INODE; // by convention

         fs1 = dfsJfsInode2Lsn( 0, &jfs->Fs1);  // first fileset-1 Inode
         root = TxaOptNum( 'L', NULL, fs1 + JFS_ROOT_INODE); // root inode lsn

         if (TxaOption( DFS_O_TREE))            // non-standard, recurse from ROOT
         {
            dfsProgressInit( 0, jfs->Fs1.Used + 10, 0, "Inode:", "cached", pFlag | DFSP_VDEC, 0);
            rc = dfsJfsNpInode( DFS_JFS_RECURSE_MAX, root, JFS_ROOT_INODE, &iDone);
         }
         else                                   // standard, linear walk ALL DIR-Inodes
         {
            dfsProgressInit( 0, jfs->Fs1.Max, 0, "Inode:", "handled", pFlag | DFSP_VDEC, 0);

            //- start with just the files in the ROOT (no tree recurse) and ignore errors!
            dfsJfsNpInode( 1, root, JFS_ROOT_INODE, &iDone);

            //- then add all of the directory Inodes present ...
            rc = dfsJfsNpWalkInodes( &iDone);
         }
         dfsProgressTerm();
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsNpBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Walk Inodes, and for the directories, find name/inode pairs for NP cache
/*****************************************************************************/
static ULONG dfsJfsNpWalkInodes
(
   ULONG              *inodes                   // INOUT Inodes added to cache
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               inode;
   ULN64               lsn;

   ENTER();

   if (jfs->Fs1.Inode)                          // mapping available ?
   {
      for ( inode = JFS_FIRSTINODE; inode < jfs->Fs1.Max; inode++)
      {
         lsn = dfsJfsInode2Lsn( inode, &jfs->Fs1);
         if ((lsn != L64_NULL) && (lsn != 0))
         {
            //- DFS_JFS_NO_PARENT will analyse directory Inodes only (not added itself)
            rc = dfsJfsNpInode( 1, lsn, DFS_JFS_NO_PARENT, inodes);
         }
         if ((inode & 0xff) == 0)               // every 256, unallocated too
         {
            if (TxAbort())
            {
               break;
            }
            else
            {
               dfsProgressShow( inode, 0, (ULN64) *inodes, "Cached:");
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsNpWalkInodes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inode-map summary (IMAP) to SLT
/*****************************************************************************/
static void dfsJfsSltIMap
(
   BYTE                stype,                   // IN    type of sector
   S_IMAP             *data,                    // IN    data
   ULONG               fn                       // IN    ref sector LSN
)
{
   ULONG               entries = (data->Max / JFS_INOEXT);
   ULONG               i;
   ULN64               extlsn;
   ULONG               ef = 0;

   ENTER();

   for (i = 0; i < entries; i++)                // walk all inode extents
   {
      extlsn = data->Inode[i];
      if (extlsn < LSN_AIMAX)                   // A few very high values seem to be used as flags ????
      {
         TRACES(("extlsn: 0x%16.16llx  jfs->Sect: 0x%16.16llx LSN_AITAB: 0x%16.16llx\n", extlsn, jfs->Sect, L32_NULL, LSN_AITAB));
         if ((extlsn < LSN_AITAB) || (extlsn > jfs->Sect))
         {
            ef |= EF_BAD_INOEXTLSN;
         }
         dfsSlTableAdd( extlsn, JFS_INOEXT, stype, fn, 0, ef); // one Inode extent
      }
   }
   VRETURN();
}                                               // end 'dfsJfsSltIMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inode info + data to SLT array, recurse into directories when allowed
/*****************************************************************************/
static ULONG dfsJfsSltInode
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   BYTE                stype,                   // IN    type for the data
   ULONG               lsn,                     // IN    Inode LSN
   ULONG               parent,                  // IN    referencing inode nr
   ULN64               ref,                     // IN    referencing inode LSN
   ULONG               ef                       // IN    error flag
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                st;
   S_JINODE           *ino = NULL;

   ENTER();
   if ((rc = dfsJfsReadChkInode( lsn, &st, &ino)) == NO_ERROR)
   {
      if ((st == ST_JINOD) || (st == ST_JIDIR) || (st == ST_JINAG))
      {
         DFSISPACE           data;              // primary data space
         S_SPACE            *tSpace   = NULL;
         ULONG               tExtents = 0;      // alloc-tree space

         if ((stype == ST_FDATA)          &&    // regular Fs1 inode
             (jfs->Ic)                    &&    // and name cache there
             (ino->Self > JFS_ROOT_INODE) &&
             (ino->Self < jfs->Fs1.Max))        // and for normal inode range
         {
            if (dfsJfsWriteMagicNames != DFS_JFS_M_OFF)
            {
               if ((st == ST_JINOD) ||
                   ((dfsJfsWriteMagicNames & DFS_JFS_MDIRS) && // Wanted and no
                    ((jfs->sup->Flag & JFS_DIR_INDEX) == 0)) ) // Linux Indexes
               {
                  ino->fMagic.magic  = DFS_JFS_MAGIC;
                  ino->fMagic.parent = parent;
                  TxCopy( ino->fMagic.fname, jfs->Ic[ ino->Self].Name, DFS_JFS_ML);

                  dfsWrite( lsn, 1, (BYTE *) ino);  //- write the Inode back
               }                                    //- (FS is not mounted)
            }
            jfs->Ic[ ino->Self].Parent = parent;

            TRACES(( "Inode Parent: %8.8lx Inode: %8.8lx Name:'%s'\n",
                            parent, ino->Self, jfs->Ic[ ino->Self].Name));
         }

         if (st == ST_JIDIR)                    // Fileset directory
         {
            if (recursionsAllowed)              // we are allowed to recurse deeper
            {
               //- guard checked and decremented in a SINGLE place in the recursion!
               rc = dfsJfsSltDir( recursionsAllowed - 1, // one level less allowed
                                  stype, ino->Self, lsn, // data type and reference
                                  ino->Slot, JFS_IN_SLOTS,
                                  ino->Dir.sTable, ino->Dir.sCount,
                                 (ino->Dir.rflag & JFS_BT_INTN));
            }
         }
         else                                   // add allocated (tree) data
         {
            if (dfsJfsInodeData2Space( ino, 0, &data, &tExtents, &tSpace) == NO_ERROR)
            {
               if ((data.chunks != 0) &&        // regular data present
                   (stype != ST_JI2AG) )        // but not a 2nd aggr inode
               {
                  //- 2nd aggr Inodes map the same data as the primary ones
                  //- so the data does not need to be added again, however
                  //- the META data like inode and tree pages are duplicated

                  dfsSlTableSpaceAdd( data.chunks, data.space, stype, lsn, 0, 0);
                  TxFreeMem( data.space);
               }
               if (tExtents != 0)               // for 2nd aggr inode too
               {                                // add meta data (tree pages)
                  dfsSlTableSpaceAdd( tExtents, tSpace, ST_JXTPG, lsn, 0, 0);
                  TxFreeMem( tSpace);
               }
            }
            else
            {
               TRACES(( "Error on Inode 0x%8.8lx, rc: %lu\n", ino->Self, rc));
               ef |= EF_BAD_INODE_ALL;          // report error on Inode
            }
         }
         if ((ino->Acl.size != 0) && (ino->Acl.pe.len != 0))
         {
            dfsSlTableAdd( ino->Acl.pe.lo  * jfs->SectorsPerBlock,
                           ino->Acl.pe.len * jfs->SectorsPerBlock,
                                             ST_JACLD, lsn, 0, 0);
         }
         if ((ino->Eax.size != 0) && (ino->Eax.pe.len != 0))
         {
            dfsSlTableAdd( ino->Eax.pe.lo   * jfs->SectorsPerBlock,
                           ino->Eax.pe.len  * jfs->SectorsPerBlock,
                                              ST_JEASD, lsn, 0, 0);
         }
      }
      else                                      // invalid Inode type
      {
         ef |= EF_BAD_INODE_LSN;
      }
      TxFreeMem(ino);
   }
   else if (rc != DFS_ALLOC_ERROR)
   {
      ef |= EF_BAD_INODE_LSN;
      rc  = NO_ERROR;                           // ignore structural problems
   }
   dfsSlTableAdd( lsn, 1, (stype == ST_JI2AG) ? stype : st, ref, 0, ef);
   RETURN(rc);
}                                               // end 'dfsJfsSltInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inode name + parent to cache, recurse into directories when allowed
/*****************************************************************************/
static ULONG dfsJfsNpInode
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               lsn,                     // IN    Inode LSN
   ULONG               parent,                  // IN    referencing inode nr
   ULONG              *inodes                   // INOUT Inodes added to cache
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                st;
   S_JINODE           *ino = NULL;

   ENTER();

   if ((rc = dfsJfsReadChkInode( lsn, &st, &ino)) == NO_ERROR)
   {
      if ((st == ST_JINOD) || (st == ST_JIDIR))
      {
         if (parent != DFS_JFS_NO_PARENT)       // only set for a KNOWN parent
         {
            if ((ino->Self >= JFS_ROOT_INODE) && (ino->Self < jfs->Fs1.Max))
            {
               jfs->Ic[ ino->Self].Parent = parent;

               if (TxaOptSet( TXA_O_DEBUG))
               {
                  TxPrint( "Set Inode Parent: %8.8lx Inode: %8.8lx Name:'%s'\n",
                                  parent, ino->Self, jfs->Ic[ ino->Self].Name);
               }
               else
               {
                  TRACES(( "Set Inode Parent: %8.8lx Inode: %8.8lx Name:'%s'\n",
                                  parent, ino->Self, jfs->Ic[ ino->Self].Name));
               }
            }
         }

         if (st == ST_JIDIR)                    // Fileset directory
         {
            //- guard checked and decremented in a SINGLE place in the recursion!
            if (recursionsAllowed)              // we are allowed to recurse deeper
            {
               rc = dfsJfsNpDir( recursionsAllowed - 1, // one level less allowed
                                 ino->Self,
                                 ino->Slot, JFS_IN_SLOTS,
                                 ino->Dir.sTable, ino->Dir.sCount,
                                (ino->Dir.rflag & JFS_BT_INTN), inodes);
            }
         }
      }
      TxFreeMem(ino);
   }
   else if ((rc == DFS_ST_MISMATCH) ||          // error reading Inode, ignore!
            (rc == DFS_PSN_LIMIT)    )          // continue with cache building
   {
      rc = NO_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsJfsNpInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Directory Btree+ page to SLT, add files and recurse into directories
/*****************************************************************************/
static ULONG dfsJfsSltDtreePage                 // RET   rc = 0 if type match
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               pageLsn,                 // IN    start LSN of page
   BYTE                stype,                   // IN    type of the data
   ULONG               parent,                  // IN    referencing inode nr
   ULONG               ref                      // IN    referencing Inode LSN
)
{
   ULONG               rc = NO_ERROR;           // rc, sector match
   BYTE                id = ST_UDATA;           // type of sector
   ULONG               ef = 0;                  // error check flag
   BYTE               *sb;                      // sector buffer
   BYTE               *st;                      // sort table location

   ENTER();

   if ((sb = TxAlloc(1, JFS_PSIZE)) != NULL)
   {
      if ((rc = dfsRead( pageLsn, JFS_PSECT, sb)) == NO_ERROR)
      {
         S_DTHEAD  *sd = (S_DTHEAD *) sb;

         id = dfsIdentifySector( pageLsn, 0, sb); // double-check sector type
         if (id == ST_JDTPG)
         {

            st = ((BYTE *) sd) + (sd->stIndex * JFS_SLOTSIZE); // sort-table

            rc = dfsJfsSltDir( recursionsAllowed,
                               stype, parent, ref,   // data type and reference
                               sd->slot,
                              (sd->self.len * jfs->SectorsPerBlock * JFS_DB_SLOTS) -1,
                               st, sd->sCount,
                              (sd->rflag & JFS_BT_INTN));
         }
         else                                   // invalid Inode type
         {
            ef |= EF_BAD_DPAGE_LSN;
         }
         dfsSlTableAdd( pageLsn, sd->self.len * jfs->SectorsPerBlock, id, ref, 0, ef);
      }
      TxFreeMem( sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsJfsSltDtreePage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle Dir Btree+ page, adding to Inode Name + Parent cache, may recurse
/*****************************************************************************/
static ULONG dfsJfsNpDtreePage                  // RET   rc = 0 if type match
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULN64               pageLsn,                 // IN    start LSN of page
   ULONG               parent,                  // IN    referencing inode nr
   ULONG              *inodes                   // INOUT Inodes added to cache
)
{
   ULONG               rc = NO_ERROR;           // rc, sector match
   BYTE                id = ST_UDATA;           // type of sector
   BYTE               *sb;                      // sector buffer
   BYTE               *st;                      // sort table location

   ENTER();

   if ((sb = TxAlloc(1, JFS_PSIZE)) != NULL)
   {
      if ((rc = dfsRead( pageLsn, JFS_PSECT, sb)) == NO_ERROR)
      {
         S_DTHEAD  *sd = (S_DTHEAD *) sb;

         id = dfsIdentifySector( pageLsn, 0, sb); // double-check sector type
         if (id == ST_JDTPG)
         {
            st = ((BYTE *) sd) + (sd->stIndex * JFS_SLOTSIZE); // sort-table

            rc = dfsJfsNpDir( recursionsAllowed,
                              parent,
                              sd->slot,
                             (sd->self.len * jfs->SectorsPerBlock * JFS_DB_SLOTS) -1,
                              st, sd->sCount,
                             (sd->rflag & JFS_BT_INTN), inodes);
         }
      }
      TxFreeMem( sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsJfsNpDtreePage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inodes for a sequence of JFS Directory entries to the SLT, recursive
/*****************************************************************************/
ULONG dfsJfsSltDir
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   BYTE                stype,                   // IN    type of the data
   ULONG               parent,                  // IN    referencing inode nr
   ULN64               ref,                     // IN    referencing Inode LSN
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes                // IN    tree internal nodes
)
{
   ULONG               rc = NO_ERROR;
   S_JDSLOT           *de;                      // single directory slot
   BYTE                si;                      // sort index
   BYTE                slotnr;

   ENTER();
   TRACES(( "Slot *:%8.8lx maxslot:%lu  sTable:%8.8lx sCount:%lu\n",
             slot, maxslot, sTable, sCount));

   for ( si = 0;
        (si < sCount) && (rc == NO_ERROR);      // used entries in sort-table
         si++)
   {
      slotnr = sTable[(int)si] -1;              // ZERO based slotnumber
      de     = slot + slotnr;                   // Slot with Inode + name

      TRACES(( "si: %lu  index: %hu = slotnr: %hu\n", si, sTable[(int) si], slotnr));

      if (treeNodes)                            // Btree split format
      {
         ULONG bdown = dfsJfsBlock2Lsn( de->t.Page.lo);

         rc = dfsJfsSltDtreePage( recursionsAllowed, bdown, stype, parent, ref);
      }
      else                                      // regular directory format
      {
         ULONG ef    = 0;
         ULONG inode = dfsJfsInode2Lsn( de->i.Inode, &jfs->Fs1); // Inode Lsn

         if ((jfs->Ic) && (de->i.Inode < jfs->Fs1.Max))
         {
            TXLN       text;                    // name for filename cache

            text[0] = 0;
            dfsJfsFileNameAppend( slot, text, slotnr, maxslot);
            if (strlen( text))
            {
               char  **name = &(jfs->Ic[ de->i.Inode].Name);

               if (*name == NULL)               // nothing there yet (1st link)
               {
                  if ((*name = TxAlloc( 1, strlen(text) +1)) != NULL)
                  {
                     strcpy( *name, text);
                     TRACES(( "Set Name for Dir-Inode %8.8lx, name:'%s'\n", de->i.Inode, *name));
                  }
               }
            }
         }
         if (dfsSlTableStatus( NULL) == SLT_STOPPING)
         {
            rc = DFS_QUIT;                      // break the recursion
         }                                      // to allow build stop
         else
         {
            switch (inode)
            {
               case 0:        ef = EF_INODE_NOALLOC; break;
               case L32_NULL: ef = EF_INODE_OORANGE; break;
            }
            //- Add Inode, possibly recurse deeper
            rc = dfsJfsSltInode( recursionsAllowed, stype, inode, parent, ref, ef);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsSltDir'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inodes for a sequence of JFS Directory entries to NP-cache, may recurse
/*****************************************************************************/
ULONG dfsJfsNpDir
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               parent,                  // IN    referencing inode nr
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes,               // IN    tree internal nodes
   ULONG              *inodes                   // INOUT Inodes added to cache
)
{
   ULONG               rc = NO_ERROR;
   S_JDSLOT           *de;                      // single directory slot
   BYTE                si;                      // sort index
   BYTE                slotnr;

   ENTER();

   for ( si = 0;
        (si < sCount) && (rc == NO_ERROR);      // used entries in sort-table
         si++)
   {
      slotnr = sTable[(int)si] -1;              // ZERO based slotnumber
      de     = slot + slotnr;                   // Slot with Inode + name

      if (treeNodes)                            // Btree split format
      {
         ULONG bdown = dfsJfsBlock2Lsn( de->t.Page.lo);

         rc = dfsJfsNpDtreePage( recursionsAllowed, bdown, parent, inodes);
      }
      else                                      // regular directory format
      {
         ULONG inode = dfsJfsInode2Lsn( de->i.Inode, &jfs->Fs1); // Inode Lsn

         if (de->i.Inode < jfs->Fs1.Max)
         {
            TXLN       text;                    // name for filename cache

            (*inodes)++;                        // increment #inodes done
            if (recursionsAllowed)              // progress here only when doing TREE!
            {
               if (((*inodes) % 100) == 0)      // don't update BAR too often
               {
                  dfsProgressShow( *inodes, 0, 0, NULL);
               }
            }
            text[0] = 0;
            dfsJfsFileNameAppend( slot, text, slotnr, maxslot);
            if (strlen( text))
            {
               char  **name = &(jfs->Ic[ de->i.Inode].Name);

               if (*name == NULL)               // nothing there yet (1st link)
               {
                  if ((*name = TxAlloc( 1, strlen(text) +1)) != NULL)
                  {
                     strcpy( *name, text);
                     TRACES(( "NpDir: Inode %8.8lx, name:'%s'\n", de->i.Inode, *name));
                  }
               }
            }
            rc = dfsJfsNpInode( recursionsAllowed, inode, parent, inodes);
         }
      }
      if (TxAbort())                            // allow user abort with <Esc>
      {
         rc = DFS_USER_ABORT;
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsNpDir'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display one line of error description for each bit in error-flag
/*****************************************************************************/
ULONG dfsJfsDispError
(
   ULN64               flags,                   // IN    error flag
   ULN64               d2,                      // IN    dummy
   char               *lead,                    // IN    leading string
   void               *param                    // IN    pointer parameter
)
{
   ENTER();

   if      (flags & EF_BAD_INODE_LSN)
   {
      TxPrint("%sSector does not contain a valid File or Directory Inode\n", lead);
   }
   if (flags & EF_BAD_INODE_ALL)
   {
      TxPrint("%sFile Inode has incorrect data allocation\n", lead);
   }
   if (flags & EF_BAD_DPAGE_LSN)
   {
      TxPrint("%sSector does not contain a valid Directory Btree+ page\n", lead);
   }
   if (flags & EF_INODE_NOALLOC)
   {
      TxPrint("%sInvalid Inode number, not allocated, not in use\n", lead);
   }
   if (flags & EF_INODE_OORANGE)
   {
      TxPrint("%sInvalid Inode number, larger than maximum for filesystem\n", lead);
   }
   if (flags & EF_BAD_SUPER_SIG)
   {
      TxPrint("%sSuperblock 'JFS1' signature has incorrect numeric value\n", lead);
   }
   if (flags & EF_BAD_INOEXTLSN)
   {
      TxPrint("%sInvalid Inode extent LSN, error in Inode allocation map\n", lead);
   }
   if (flags & EF_BAD_RECURSION)
   {
      TxPrint("%sInode directory points back to itself (recursion, loop)\n", lead);
   }
   RETURN (NO_ERROR);
}                                               // end 'dfsJfsDispError'
/*---------------------------------------------------------------------------*/


