//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// HPFS dump & display Sector Lookup Table
//
// Author: J. van Wijk
//
// JvW  22-05-97   Split-off most basic functionality to dfstable
// JvW  04-01-97   Removed HPFS volume open/close; read_sect remaps to PSN
// JvW  12-10-95   Added Free-space and EA-data to SLT
// JvW  08-09-95   Complete version, includes all significant sectors
// JvW  14-08-95   Initial version, split off from DHPFS.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsahpfs.h>                           // HPFS analysis definitions

#include <dfslhpfs.h>                           // HPFS sector lookup

static  char        msgHpfsMem[] = "\nError allocating buffer memory\n";

// Add HotFix sectors info to SLT array
static void dfsHpfsSltHotfix
(
   DFSAHPFS           *fsi                      // IN    File system info
);

// Add BitMap sectors info to SLT array
static void dfsHpfsSltBitMaps
(
   DFSAHPFS           *fsi                      // IN    File system info
);

// Add Bad sectors from list to SLT array
static void dfsHpfsSltBadSec
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               lsn,                     // IN    Bad-sector-list LSN
   ULONG               ref                      // IN    referencing LSN
);

// Add Codepage sectors info to SLT array
static void dfsHpfsSltCodePage
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               cpi,                     // IN    Codepage info LSN
   ULONG               ref                      // IN    referencing LSN
);

// Add single Fnode info to  SLT array, recursive into directories
static ULONG dfsHpfsSltFnode
(
   ULONG               guard,                   // IN    recursion guard
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               fn,                      // IN    Fnode LSN
   S_DIRENT           *de,                      // IN    directory entry
   ULONG               ref                      // IN    referencing LSN (DIR)
);

// Add external-EA sector(s) info to  SLT array
static void dfsHpfsSltEaExtern
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               sn,                      // IN    first ext-EA LSN
   ULONG               asize,                   // IN    ea area size
   ULONG               fn                       // IN    related Fnode LSN
);

// Add extended attributes to SLT
static void dfsHpfsSltEaArea
(
   DFSAHPFS           *fsi,                     // IN    File system info
   BYTE                stype,                   // IN    type of sector
   S_EABLK            *first,                   // IN    first EA block
   ULONG               size,                    // IN    size of EA area
   ULONG               fn                       // IN    ref sector LSN
);

// Add extended attribute to SLT
static USHORT dfsHpfsSltEaSingle
(
   DFSAHPFS           *fsi,                     // IN    File system info
   BYTE                stype,                   // IN    type of sector
   S_EABLK            *data,                    // IN    data
   ULONG               fn                       // IN    ref sector LSN
);


// Add allocation-sector info to  SLT array (recursive)
static ULONG dfsHpfsSltAlloc                    // RET   alloc size in sectors
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               sn,                      // IN    Alloc-sector LSN
   BYTE                stype,                   // IN    type of sector
   ULONG               fn                       // IN    related Fnode LSN
);

// Add dirblock-sector info to  SLT array (recursive)
static ULONG dfsHpfsSltDirblock
(
   ULONG               guard,                   // IN    recursion guard
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               sn,                      // IN    Dirblock LSN
   ULONG               fn                       // IN    related Fnode LSN
);


/*****************************************************************************/
// Add info to sector-lookup-table, fnode processing recursive from root
/*****************************************************************************/
ULONG dfsHpfsSltBuild
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    pointer parameter
)
{
   ULONG               rc  = DFS_BAD_STRUCTURE;
   DFSAHPFS           *fsi = (DFSAHPFS *) param; // IN    HPFS info structure
   ULONG               nr;
   ULONG               ef;

   ENTER();

   TRHEXS( 70,  fsi,  sizeof(DFSAHPFS), "hpfs global info");
   if (fsi->root != L32_NULL)                   // valid root
   {
      if (dfsa->FsUnallocated == 0)             // probably not set yet ...
      {
         DFSFNCALL( dfsa->FsAllocDisplay, 0, 0, "@", NULL);
      }
      dfsSlTableAdd(   fsi->sup->TotalSec, DFS64MAX, ST_FINAL, LSN_SUPER, 0, 0);

                                                // Approximate sectors todo:
      dfsSlTableFsInit( (ULN64) fsi->sup->TotalSec - // RAW size of FS
                        dfsa->FsUnallocated);   // minus known freespace

      dfsSlTableAdd(   LSN_BOOTR +1,           0x0f, ST_BOOTA, L64_NULL,  0, 0);

      ef = 0;
      if (fsi->spr->SuChecksum != dfsCheckSum((BYTE *) fsi->sup))
      {
         ef |= EF_BAD_CHECK_SUM;                // Checksum error on super
      }
      dfsSlTableAdd(   LSN_SUPER,             0x01, ST_SUPER, L64_NULL,  0, ef);

      ef = 0;
      if (fsi->spr->SpChecksum != dfsCheckSum((BYTE *) fsi->spr))
      {
         ef |= EF_BAD_CHECK_SUM;                // Checksum error on spare
      }
      dfsSlTableAdd(   LSN_SPARE,             0x01, ST_SPARE, L64_NULL,  0, ef);

      dfsSlTableAdd(   LSN_RESRV,        LEN_RESRV, ST_RESRV, L64_NULL,  0, 0);
      dfsSlTableAdd(   fsi->sup->UserIdTable, 0x08, ST_UITAB, LSN_SUPER, 0, 0);

      dfsHpfsSltBadSec(fsi, fsi->sup->BadList,                LSN_SUPER);

      nr = ((((fsi->sup->TotalSec >> 14) +2) * 4 / SECTORSIZE) +4) & 0xfffc;
      dfsSlTableAdd(    fsi->sup->BitmapInd,    nr, ST_BMAPI, LSN_SUPER, 0, 0);

      dfsSlTableAdd(    fsi->sup->DirBand,
                        fsi->sup->DirBandSec,       ST_DBAND, LSN_SUPER, 0, 0);

      nr = ((((fsi->sup->DirBandSec + 31) / 32) / SECTORSIZE) +4) & 0xfffc;
      dfsSlTableAdd(    fsi->sup->DirBandMap,   nr, ST_DBMAP, LSN_SUPER, 0, 0);

      dfsSlTableListSn32( &(fsi->spr->SpareDir[0]),
                            fsi->spr->SpareDirUsed,  4, ST_SPDIR, LSN_SPARE, 0, 0);

      dfsHpfsSltCodePage(fsi, fsi->spr->CodePageInfo,         LSN_SPARE);

      dfsHpfsSltHotfix(fsi);

      dfsHpfsSltBitMaps(fsi);

      rc = dfsHpfsSltFnode( 0, fsi, fsi->root, NULL, LSN_SUPER);
   }
   TRHEXS( 70,  fsi,  sizeof(DFSAHPFS), "hpfs global info");
   RETURN (rc);
}                                               // end 'dfsHpfsSltBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display one line of error description for each bit in error-flag
/*****************************************************************************/
ULONG dfsHpfsDispError
(
   ULN64               flags,                   // IN    error flag
   ULN64               d2,                      // IN    dummy
   char               *lead,                    // IN    leading string
   void               *param                    // IN    pointer parameter
)
{
   ENTER();

   if (flags & EF_NAME_FN_N_DIR)
   {
      TxPrint("%sFnode shortname does not match name in directory\n", lead);
   }
   if (flags & EF_DIRFL_NOT_SET)
   {
      TxPrint("%sDirectory flag not set on root\n", lead);
   }
   if (flags & EF_SIZE_FN_G_DIR)
   {
      TxPrint("%sFilesize in Fnode larger than in dir-entry\n", lead);
   }
   if (flags & EF_SIZE_DIR_G_FN)
   {
      TxPrint("%sFilesize in Fnode smaller than in dir-entry\n", lead);
   }
   if (flags & EF_SIZE_FN_G_ALL)
   {
      TxPrint("%sFilesize larger than allocated-space\n", lead);
   }
   if (flags & EF_SIZE_ALL_G_FN)
   {
      TxPrint("%sFilesize smaller than allocated-space\n", lead);
   }
   if (flags & EF_BAD_CHECK_SUM)
   {
      TxPrint("%sIncorrect checksum value, possibly by a FORMAT bug\n", lead);
      TxPrint("%sRun CHKDSK, or the DFSee 'FIXSPARE' command to fix\n", lead);
   }
   if (flags & EF_BAD_FNODE_LSN)
   {
      TxPrint("%sLSN does not point to a valid Fnode\n", lead);
   }
   if (flags & EF_BAD_DIRBL_LSN)
   {
      TxPrint("%sLSN does not point to a valid Directory block\n", lead);
   }
   if (flags & EF_BAD_ALLOC_LSN)
   {
      TxPrint("%sLSN does not point to a valid Allocation sector\n", lead);
   }
   if (flags & EF_BAD_RANGE_LSN)
   {
      TxPrint("%sLSN is too large for current volume\n", lead);
   }
   if (flags & EF_BAD_CPINF_LSN)
   {
      TxPrint("%sLSN points to unreadable or corrupted CP-INFO sector\n", lead);
   }
   if (flags & EF_BAD_CPDAT_LSN)
   {
      TxPrint("%sLSN points to unreadable or corrupted CP-DATA sector\n", lead);
   }
   if (flags & EF_BAD_RECURSION)
   {
      TxPrint("%sFnode directory points back to itself (recursion, loop)\n", lead);
   }
   RETURN (NO_ERROR);
}                                               // end 'dfsHpfsDispError'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add HotFix sectors info to SLT array
/*****************************************************************************/
static void dfsHpfsSltHotfix
(
   DFSAHPFS            *fsi                     // IN    File system info
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               lsn = fsi->spr->HotFixList;
   ULONG               hfs;                     // nr of HotFixtable sectors
   BYTE               *hft = NULL;              // HotFix table

   ENTER();
   hfs = (((fsi->spr->HotFixMax*3*sizeof(ULONG) -1) /
            SECTORSIZE) +4) & 0xfffffffc;

   dfsSlTableAdd( lsn, hfs, ST_HOTFT, LSN_SPARE, 0, 0);

   if ((hft = TxAlloc(hfs, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead(lsn, hfs, hft);
      if (rc == NO_ERROR)
      {
         dfsSlTableListSn32( ((ULONG *) hft) + fsi->spr->HotFixMax,
                                               fsi->spr->HotFixMax,
                                               1, ST_HOTFD,
                                               fsi->spr->HotFixList,
                                               0, 0);
      }
      TxFreeMem( hft);
   }
   else
   {
      printf( "%s", msgHpfsMem);
   }
   VRETURN();
}                                               // end 'dfsHpfsSltHotfix'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add BitMap sectors info to SLT array
/*****************************************************************************/
static void dfsHpfsSltBitMaps
(
   DFSAHPFS           *fsi                      // IN    File system info
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               lsn = fsi->sup->BitmapInd;
   ULONG               bis;                     // nr of bitmapInd sectors
   ULONG              *mp;                      // LSN array, equals 4 sectors
   BYTE               *bmt = NULL;              // BitMap table

   ENTER();
   bis = ((((fsi->sup->TotalSec >> 14) +2) * sizeof(ULONG) /
             SECTORSIZE) +4) & 0xfffffffc;

   if ((bmt = TxAlloc(bis, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead(lsn, bis, bmt);
      if (rc == NO_ERROR)
      {
         for (mp = (ULONG *) bmt; *mp != 0; mp++)
         {
            dfsSlTableAdd( *mp, 4, ST_BMAPD, fsi->sup->BitmapInd, 0, 0);
         }
      }
      TxFreeMem( bmt);
   }
   else
   {
      printf( "%s", msgHpfsMem);
   }
   VRETURN();
}                                               // end 'dfsHpfsSltBitMaps'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Bad sectors from list to SLT array
/*****************************************************************************/
static void dfsHpfsSltBadSec
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               lsn,                     // IN    Bad-sector-list LSN
   ULONG               ref                      // IN    referencing LSN
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *bsl = NULL;              // Bad-sector-list element
   ULONG               bse = lsn;

   ENTER();

   if ((bsl = TxAlloc(4, dfsGetSectorSize())) != NULL)
   {
      S_BADSEL          *sb = (S_BADSEL *) bsl;
      ULONG              ib;
      ULONG              size = 0;
      ULONG              last = 0;

      while ((bse != 0) && (rc == NO_ERROR))
      {
         dfsSlTableAdd( bse, 4, ST_BADSL, ref, 0, 0); // Add Bad-sec element

         rc = dfsRead(bse, 4, bsl);
         if (rc == NO_ERROR)
         {
            for (ib = 0; ib < HPFS_BADSECS; ib++) // iterate bad-sectors
            {
               if (sb->Bads[ib] != 0)
               {
                  if (sb->Bads[ib] != last+1)
                  {
                     if ((last != 0) && (size != 0)) // register last range
                     {
                        dfsSlTableAdd( last-size+1, size, ST_BADSC, lsn, 0, 0);
                     }
                     size = 1;
                  }
                  else
                  {
                     size++;
                  }
                  last = sb->Bads[ib];
               }
            }
            bse = sb->Next;                     // LSN ptr to next element
         }
      }
      if ((last != 0) && (size != 0))           // register last range
      {
         dfsSlTableAdd( last-size+1, size, ST_BADSC, lsn, 0, 0);
      }
      TxFreeMem( bsl);
   }
   else
   {
      printf( "%s", msgHpfsMem);
   }
   VRETURN();
}                                               // end 'dfsHpfsSltBadSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Codepage sectors info to SLT array
/*****************************************************************************/
static void dfsHpfsSltCodePage
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               cpi,                     // IN    Codepage info LSN
   ULONG               ref                      // IN    referencing LSN
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *cps = NULL;              // CodePage sector
   ULONG               ef  = 0;

   ENTER();
   TRACES(("CP-info LSN: %8.8lx  ref-LSN: 0x%llx\n", cpi, ref));

   if ((cps = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead(cpi, 1, cps);
      if (rc == NO_ERROR)
      {
         S_CPINF           *sd = (S_CPINF *) cps;
         ULONG              ib;
         ULONG              cpd;                // CpData LSN
         ULONG              last = 0;           // last added CpData LSN
         BYTE               id   = ST_UDATA;    // type of sector

         id = dfsIdentifySector(cpi, 0, cps);   // double-check sector type
         if (id == ST_CPINF)
         {
            //- to be refined, check further contents ??

            for ( ib = 0;
                 (ib < sd->CpCount) &&          // count in sector
                 (ib < S_MAX_CPIB);             // absolute sanity limit
                  ib++)                         // iterate attached CpData
            {
               cpd = sd->Cpib[ib].CpData;

               TRACES(("CpCount: %lu  ib:%lu  cpd LSN: %8.8lx, last: %8.8lx\n",
                    sd->CpCount,      ib,     cpd,             last));

               if (cpd != last)                 // don't add duplicates
               {
                  S_CPDAT  *cds = NULL;         // Codepage Data sector
                  ULONG     def = 0;            // date error flag

                  if ((cds = TxAlloc(1, dfsGetSectorSize())) != NULL)
                  {
                     rc = dfsRead( cpd, 1, (BYTE   *) cds);
                     if (rc == NO_ERROR)
                     {
                        id = dfsIdentifySector(cpd, 0, (BYTE   *) cds);
                        if (id == ST_CPDAT)
                        {
                           //- to be refined, check further contents ??
                        }
                        else
                        {
                           def |= EF_BAD_CPDAT_LSN; // bad CPDAT sector signature
                        }
                     }
                     else
                     {
                        def |= EF_BAD_CPDAT_LSN; // bad CPDAT sector
                     }
                     TxFreeMem( cds);
                  }
                  else
                  {
                     printf( "%s", msgHpfsMem);
                  }
                  dfsSlTableAdd( cpd, 1, ST_CPDAT, cpi, 0, def); // Add CpData
                  last = cpd;
               }
               fsi->cpg[ib] = sd->Cpib[ib].CodePageId; // keep Cp around
            }
            if (sd->NextCpInfo != 0)            // recurse until end of list
            {
               dfsHpfsSltCodePage( fsi, sd->NextCpInfo, cpi);
            }
         }
         else
         {
            ef |= EF_BAD_CPINF_LSN;             // bad CPINF sector signature
         }
      }
      else                                      // error reading sector
      {
         ef |= EF_BAD_CPINF_LSN;                // bad CPINF sector
      }
      TxFreeMem( cps);
   }
   else
   {
      printf( "%s", msgHpfsMem);
   }
   dfsSlTableAdd( cpi, 1, ST_CPINF, ref, 0, ef); // Add CpInfo
   VRETURN();
}                                               // end 'dfsHpfsSltCodePage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add single Fnode info to  SLT array, recursive into directories
/*****************************************************************************/
static ULONG dfsHpfsSltFnode
(
   ULONG               guard,                   // IN    recursion guard
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               fn,                      // IN    Fnode LSN
   S_DIRENT           *de,                      // IN    directory entry
   ULONG               ref                      // IN    referencing LSN (DIR)
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                id  = ST_UDATA;          // type of sector
   BYTE               *sb  = NULL;              // sector buffer
   ULONG               ef  = 0;                 // error check flag

   ENTER();
   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      TRACES(("add_fnode for LSN: %8.8lX\n", fn));
      rc = dfsRead(fn, 1, sb);
      if (rc == NO_ERROR)
      {
         id = dfsIdentifySector(fn, 0, sb);     // double-check sector type
         if ((id == ST_FNODE) || (id == ST_FNDIR))
         {
            S_FNODE            *sd = (S_FNODE *) sb;
            BYTE                af = sd->AlInfo.Flag;
            int                 al = (int) sd->AlInfo.UsedEntries;
            int                 i;
            ULONG               alsize = 0;     // allocated size in sectors
            ULONG               fnsize = 0;     // fnode size in sectors

            if (de != NULL)                     // DIR info available
            {
               if      (de->FileSize > sd->FileLength)
               {
                  ef |= EF_SIZE_DIR_G_FN;       // Dirsize greater than fnode
               }
               else if (de->FileSize < sd->FileLength)
               {
                  ef |= EF_SIZE_FN_G_DIR;       // Fnode size greater than Dir
               }
               if ((     de->NameLength != sd->NameLength) ||
                   (strncasecmp(de->FileName, sd->Name, min(15, sd->NameLength))))
               {
                  ef |= EF_NAME_FN_N_DIR;       // Short name mismatch
               }
            }
            if ((sd->DirFlag) ||                // Directory
                (sd->ParentDir == fn))          // Root Directory
            {
               al = 1;                          // Overrule Used, must be 1
            }
            for (i=0; (i < al) && (rc == NO_ERROR); i++)
            {
               if (af & HPFS_AL_NODE)           // NODE allocation block
               {
                  alsize += dfsHpfsSltAlloc(fsi, sd->Al.Node[i].ChildBlock,
                                            ST_FDATA, fn);
               }
               else                             // LEAF allocation block
               {
                  if ((sd->DirFlag) ||          // Directory
                      (sd->ParentDir == fn))    // Root Directory
                  {
                     rc = dfsHpfsSltDirblock( guard +1, fsi, sd->Al.Directory.DirBlock, fn);
                  }
                  else                          // File, multiple alloc
                  {
                     dfsSlTableAdd( sd->Al.Leaf[i].DataBlock,
                                        sd->Al.Leaf[i].LengthData,
                                        ST_FDATA, fn, 0, 0);
                     alsize += sd->Al.Leaf[i].LengthData;
                  }
               }
            }
            if (sd->FileLength > 0)
            {
               fnsize = ((sd->FileLength -1) / SECTORSIZE) +1;
            }
            if (fnsize > alsize)
            {
               ef |= EF_SIZE_FN_G_ALL;
            }
            if (fnsize < alsize)
            {
               ef |= EF_SIZE_ALL_G_FN;
            }
            if ((sd->DirFlag) ||                // Directory
                (sd->ParentDir == fn))          // Root Directory
            {
               if (sd->DirFlag == 0)            // Root without DirFlag
               {
                  ef |= EF_DIRFL_NOT_SET;
               }
               dfsSlTableAdd( fn, 1, ST_FNDIR, ref, 0, ef);
            }
            else
            {
               dfsSlTableAdd( fn, 1, ST_FNODE, ref, 0, ef);
            }

            if (sd->EAsExtern.size != 0)
            {
               dfsSlTableAdd( sd->EAsExtern.ptr,
                           ((sd->EAsExtern.size -1) / SECTORSIZE) +1,
                             ST_EDATA, fn, 0, 0);
               dfsHpfsSltEaExtern( fsi, sd->EAsExtern.ptr, sd->EAsExtern.size, fn);
            }
            if (sd->EAsFnLength != 0)
            {
               dfsHpfsSltEaArea(fsi, ST_EDATA, (S_EABLK *) sd->EasOrAcl,
                                               (ULONG) sd->EAsFnLength, fn);
            }
            if (sd->AclExtern.size != 0)
            {
               dfsSlTableAdd( sd->AclExtern.ptr,
                           ((sd->AclExtern.size -1) / SECTORSIZE) +1,
                             ST_ADATA, fn, 0, 0);
            }
            if (sd->AclFnLength != 0)
            {
               dfsHpfsSltEaArea(fsi, ST_ADATA, (S_EABLK *) sd->EasOrAcl,
                                               (ULONG) sd->AclFnLength, fn);
            }
         }
         else                                   // invalid Fnode LSN
         {
            ef |= EF_BAD_FNODE_LSN;
            dfsSlTableAdd( fn, 1, ST_FNODE, ref, 0, ef);
         }
      }
      TxFreeMem(sb);
   }
   RETURN(rc);
}                                               // end 'dfsHpfsSltFnode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add external-EA sector(s) info to  SLT array
/*****************************************************************************/
static void dfsHpfsSltEaExtern
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               sn,                      // IN    first ext-EA LSN
   ULONG               asize,                   // IN    ea area size
   ULONG               fn                       // IN    related Fnode LSN
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *sb  = NULL;              // sector buffer
   ULONG               nr;                      // nr of ea sectors
   USHORT              bps = dfsGetSectorSize();

   ENTER();
   nr = ((asize -1) / bps) +1;

   if ((sb = TxAlloc(nr, bps)) != NULL)
   {
      TRACES(("add_extea for LSN: %8.8lX\n", sn));
      rc = dfsRead(sn, nr, sb);
      if (rc == NO_ERROR)
      {
         dfsHpfsSltEaArea(fsi, ST_EDATA, (S_EABLK *) sb, asize, fn);
      }
      TxFreeMem(sb);
   }
   VRETURN();
}                                               // end 'dfsHpfsSltEaExtern'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add extended attributes / ACL area to SLT
/*****************************************************************************/
static void dfsHpfsSltEaArea
(
   DFSAHPFS           *fsi,                     // IN    File system info
   BYTE                stype,                   // IN    type of sector
   S_EABLK            *first,                   // IN    first EA block
   ULONG               size,                    // IN    size of EA area
   ULONG               fn                       // IN    ref sector LSN
)
{
   S_EABLK            *ea = first;              // EA block
   ULONG               at = 0;

   ENTER();
   while ((  at               < size ) &&       // stop at given limit! P0088
          ( (ea->Flag & 0x7c) == 0   ) &&       // only allow 'sane' looking
          (  ea->NameLength   != 0   ) &&       // EA blocks
          (  ea->DataLength   != 0   ) &&
          (  ea->DataLength   <= (USHORT) size) &&
          ( (ea->DataLength   == (USHORT) sizeof(SPTR)) ||
           ((ea->Flag & 0x03) == 0)))
   {
      at += dfsHpfsSltEaSingle( fsi, stype, (S_EABLK *) ea, fn);
      ea  = (S_EABLK *) (((char *) first) + at);
   }
   VRETURN();
}                                               // end 'dfsHpfsSltEaArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add extended attribute or ACL to SLT
/*****************************************************************************/
static USHORT dfsHpfsSltEaSingle
(
   DFSAHPFS           *fsi,                     // IN    File system info
   BYTE                stype,                   // IN    type of sector
   S_EABLK            *data,                    // IN    data
   ULONG               fn                       // IN    ref sector LSN
)
{
   char               *d  = data->Name + data->NameLength +1;
   USHORT              rc = data->NameLength + data->DataLength + 5;

   ENTER();
   switch (data->Flag & 0x03)
   {
      case 0x00:                                // Actual EA data in Fnode
         break;

      case 0x01:                                // SPTR to EA data
         dfsSlTableAdd(  ((SPTR *) d)->ptr,
                      ((((SPTR *) d)->size -1) / SECTORSIZE) +1,
                          stype, fn, 0, 0);
         break;

      default:                                  // SPTR to ALLOC sector
         dfsHpfsSltAlloc(fsi, ((SPTR *) d)->ptr, stype, fn);
         break;
   }
   RETURN(rc);                                  // total size of this EA block
}                                               // end 'dfsHpfsSltEaSingle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add allocation-sector info to  SLT array (recursive)
/*****************************************************************************/
static ULONG dfsHpfsSltAlloc                    // RET   alloc size in sectors
(
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               sn,                      // IN    Alloc-sector LSN
   BYTE                stype,                   // IN    type of sector
   ULONG               fn                       // IN    related Fnode LSN
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                id  = ST_UDATA;          // type of sector
   BYTE               *sb  = NULL;              // sector buffer
   ULONG               alsize = 0;              // allocated size in sectors
   ULONG               ef  = 0;

   ENTER();
   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      TRACES(("add_alloc for LSN: %8.8lX\n", sn));
      rc = dfsRead(sn, 1, sb);
      if (rc == NO_ERROR)
      {
         id = dfsIdentifySector(sn, 0, sb);     // double-check sector type
         if (id == ST_ALLOC)
         {
            S_ALLOC           *sd = (S_ALLOC *) sb;
            BYTE               af = sd->AlInfo.Flag;
            int                al = (int) sd->AlInfo.UsedEntries;
            int                i;

            for (i=0; i < al; i++)
            {
               if (af & HPFS_AL_NODE)           // NODE allocation block
               {
                  alsize += dfsHpfsSltAlloc(fsi, sd->Al.Node[i].ChildBlock,
                                            stype, fn);
               }
               else                             // LEAF allocation block
               {
                  dfsSlTableAdd( sd->Al.Leaf[i].DataBlock,
                                     sd->Al.Leaf[i].LengthData,
                                     stype, fn, 0, 0);
                  alsize += sd->Al.Leaf[i].LengthData;
               }
            }
         }
         else                                   // invalid Alloc-block LSN
         {
            ef |= EF_BAD_ALLOC_LSN;
         }
         dfsSlTableAdd( sn, 1, ST_ALLOC, fn, 0, 0);
      }
      TxFreeMem(sb);
   }
   RETURN(alsize);
}                                               // end 'dfsHpfsSltAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Dirblock-sector info to  SLT array (recursive)
/*****************************************************************************/
static ULONG dfsHpfsSltDirblock
(
   ULONG               guard,                   // IN    recursion guard
   DFSAHPFS           *fsi,                     // IN    File system info
   ULONG               sn,                      // IN    Dirblock LSN
   ULONG               fn                       // IN    related Fnode LSN
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                id  = ST_UDATA;          // type of sector
   BYTE               *sb  = NULL;              // sector buffer
   ULONG               ef  = 0;

   ENTER();
   if (guard > 100)                             // sanity recursion check
   {
      //- add the error to SLT and stop the recursion
      dfsSlTableAdd( sn, 1, ST_FNDIR, fn, 0, EF_BAD_RECURSION);
   }
   else
   {
      if ((sb = TxAlloc(4, dfsGetSectorSize())) != NULL)
      {
         TRACES(("add_dirbl for LSN: %8.8lX\n", sn));
         rc = dfsRead(sn, 4, sb);
         if (rc == NO_ERROR)
         {
            id = dfsIdentifySector(sn, 0, sb);  // double-check sector type
            if (id == ST_DIRBL)
            {
               S_DIRBL      *sd = (S_DIRBL *) sb;
               S_DIRENT     *de;
               BYTE          df;
               ULONG        *bdown;

               for ( de = &(sd->FirstEntry);
                    (de->Length != 0) && (rc == NO_ERROR);
                     de = (S_DIRENT *) ((char *) (de) + de->Length))
               {
                  df = de->Flag;
                  if (df & HPFS_DF_BDOWN)       // B-tree DOWN pointer
                  {
                     bdown = (ULONG *) ((char *) (de) + de->Length - sizeof(ULONG));
                     dfsHpfsSltDirblock( guard +1, fsi, *bdown, fn);
                  }
                  if ((df & HPFS_DF_BGEND) == 0) // no special entry
                  {
                     rc = dfsHpfsSltFnode( guard, fsi, de->Fnode, de, sn);
                  }
                  if (df & HPFS_DF_EODIR)       // Special "END" entry
                  {
                     break;                     // out of entry-loop
                  }
                  if (dfsSlTableStatus( NULL) == SLT_STOPPING)
                  {
                     rc = DFS_QUIT;             // break the recursion
                  }                             // RC will be discarded later!
               }
            }
            else                                // invalid Dirblock LSN
            {
               ef |= EF_BAD_DIRBL_LSN;
            }
            dfsSlTableAdd( sn, 4, ST_DIRBL, fn, 0, ef);
         }
         TxFreeMem(sb);
      }
   }
   RETURN(rc);
}                                               // end 'dfsHpfsSltDirblock'
/*---------------------------------------------------------------------------*/

