//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// HFS dump & display Sector Lookup Table
//
// Author: J. van Wijk
//
// JvW  16-07-2007 Initial version, derived from RSR
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfshfs.h>                             // HFS disk structures
#include <dfsahfs.h>                            // HFS analysis definitions
#include <dfsuhfs.h>                            // HFS utility functions
#include <dfslhfs.h>                            // HFS sector lookup


#if defined (HFS_USE_NP_CACHE)
// Walk Catalog leaf-nodes, and add cnid and (directory) name to the NP cache
// Files: Name is in record key, only add the parent CnID (to find the folder)
static ULONG dfsHfsNpWalkLeafs
(
   ULONG              *cnids                    // INOUT CnIDs added to cache
);
#endif


/*****************************************************************************/
// Add info to sector-lookup-table for HFS filesystem
/*****************************************************************************/
ULONG dfsHfsSltBuild
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    pointer parameter
)
{
   ULONG               dr  = NO_ERROR;
   DFSAHFS            *hfs = (DFSAHFS *) param; // HFS info structure
   ULONG               HfsExtents;
   S_SPACE            *HfsSpace;

   ENTER();

   dfsSlTableAdd( hfs->SectorCount,                  DFS64MAX, ST_FINAL, LSN_BOOTR, 0, 0);

   //- first superblock at fixed location
   dfsSlTableAdd( HFS_LSNSUP1,                                  1, ST_HFSSUP, L64_NULL,  0, 0);

   //- second superblock is at FS-size minus 2 sectors
   dfsSlTableAdd( (hfs->BlockCount * hfs->SectorsPerBlock) - 2, 1, ST_HFSSUP, L64_NULL,  0, 0);

   //- Special files allocation is stored in the superblock, list each of them
   if (hfs->sup->Startupfile.logicalSize != 0)
   {
      if ((dfsHfsFork2Space( &(hfs->sup->Startupfile), &HfsExtents, &HfsSpace)) == NO_ERROR)
      {
         dfsSlTableSpaceAdd( HfsExtents, HfsSpace, ST_HFSBFL, HFS_LSNSUP1, 0, 0);
         TxFreeMem( HfsSpace);                  // free S_SPACE structure
      }
   }
   if (hfs->sup->AllocationFile.logicalSize != 0)
   {
      if ((dfsHfsFork2Space( &(hfs->sup->AllocationFile), &HfsExtents, &HfsSpace)) == NO_ERROR)
      {
         dfsSlTableSpaceAdd( HfsExtents, HfsSpace, ST_HFSBMP, HFS_LSNSUP1, 0, 0);
         TxFreeMem( HfsSpace);                  // free S_SPACE structure
      }
   }
   if (hfs->sup->ExtentsFile.logicalSize != 0)
   {
      if ((dfsHfsFork2Space( &(hfs->sup->ExtentsFile), &HfsExtents, &HfsSpace)) == NO_ERROR)
      {
         dfsSlTableSpaceAdd( HfsExtents, HfsSpace, ST_HFSEXT, HFS_LSNSUP1, 0, 0);
         TxFreeMem( HfsSpace);                  // free S_SPACE structure
      }
   }
   if (hfs->sup->AttributesFile.logicalSize != 0)
   {
      if ((dfsHfsFork2Space( &(hfs->sup->AttributesFile), &HfsExtents, &HfsSpace)) == NO_ERROR)
      {
         dfsSlTableSpaceAdd( HfsExtents, HfsSpace, ST_HFSATR, HFS_LSNSUP1, 0, 0);
         TxFreeMem( HfsSpace);                  // free S_SPACE structure
      }
   }
   if (hfs->sup->CatalogFile.logicalSize != 0)
   {
      if ((dfsHfsFork2Space( &(hfs->sup->CatalogFile), &HfsExtents, &HfsSpace)) == NO_ERROR)
      {
         dfsSlTableSpaceAdd( HfsExtents, HfsSpace, ST_HFSCAT, HFS_LSNSUP1, 0, 0);
         TxFreeMem( HfsSpace);                  // free S_SPACE structure
      }
   }
   RETURN (dr);
}                                               // end 'dfsHfsSltBuild'
/*---------------------------------------------------------------------------*/

#if defined (HFS_USE_NP_CACHE)
/*****************************************************************************/
// Build the Name+Parent cache for HFS filesystem (caching 'thread records')
// Note: needs to be abortable when not started as a thread!
/*****************************************************************************/
ULONG dfsHfsNpBuild                             // RET   possibbly ABORTED
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    not used
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();

   if (TxaOption( TXA_O_RESET))
   {
      if (hfs->Ic != NULL)
      {
         int              i;

         for (i = 0; i < hfs->icSize; i++)
         {
            TxFreeMem( hfs->Ic[i].Name);        // free cached Inode names
         }
         TxFreeMem( hfs->Ic);                   // free name/parent cache
      }
   }

   if (hfs->Ic == NULL)                         // no name/parent cache yet
   {
      //- size determined by nextCatalogID, or fixed maximum when resued
      if (TxBE32( hfs->sup->Attributes) & HFS_VA_NODEID_REUSED)
      {
         hfs->icSize = HFS_NP_CACHE_MAXSIZE;
      }
      else                                      // nr of CnIDs is limited
      {
         hfs->icSize = TxBE32( hfs->sup->NextCatalogID);
      }
      hfs->Ic = TxAlloc( hfs->icSize, sizeof( DFSHFSINAME));

      if (hfs->Ic != NULL)
      {
         ULONG         iDone = 0;               // CnIDs done
         ULONG         pFlag = (TxaOption(TXA_O_DEBUG)) ? DFSP_STAT : DFSP_BARS;

         hfs->Ic[HFS_ID_RootFolder].Name   = TxAlloc( 1, 1);     //- ROOT name ""
         hfs->Ic[HFS_ID_RootFolder].Parent = HFS_ID_RootFolder;  //- by convention (check!)

         if (hfs->Catalog.Btree != NULL)        // only when properly initialized
         {
            dfsProgressInit( 0, hfs->icSize, 0, "CnID:", "handled", pFlag | DFSP_VDEC, 0);

            //- start with the firstLeaf, and walk the linked list until the last one
            rc = dfsHfsNpWalkLeafs( &iDone);

            dfsProgressTerm();
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsHfsNpBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Walk Catalog leaf-nodes, and add cnid and (directory) name to the NP cache
// Files: Name is in record key, only add the parent CnID (to find the folder)
/*****************************************************************************/
static ULONG dfsHfsNpWalkLeafs
(
   ULONG              *cnids                    // INOUT CnIDs added to cache
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               thisNodeNr;              // node nr  to be handled next (auto)
   ULN64               thisNodeLsn;             // node LSN to be handled next (auto)
   BYTE               *node;

   ENTER();

   //- read full node contents
   if ((node = TxAlloc( hfs->Catalog.spNode, dfsGetSectorSize())) != NULL)
   {
      S_BT_NODE_DESC  *desc = (S_BT_NODE_DESC *) node;

      thisNodeNr = TxBE32( hfs->Catalog.Btree->hdr.bthFirstLeaf);

      do
      {
         thisNodeLsn = dfsHfsNode2Lsn( thisNodeNr);

         if ((rc = dfsRead( thisNodeLsn, hfs->Catalog.spNode, node)) == NO_ERROR)
         {
            S_HFS_CATALOG_KEY *key;
            U_HFS_CAT_RECORD  *catData;         // catalog data record folder/file/thread
            USHORT             rec;
            USHORT             nRecs = TxBE16( desc->nRecords); // number of records
            ULONG             thisCnId;

            for (rec = 0; rec < nRecs; rec++)
            {
               USHORT     this = dfsHfsRecordOffset( rec,     node);
               USHORT     keyLength;
               USHORT     recType;              // record-type folder/file (or -thread)
               ULONG      parentId;
               char     **name = NULL;

               key       = (S_HFS_CATALOG_KEY *) (node + this);
               keyLength = TxBE16( key->keyLength);
               thisCnId  = TxBE32( key->parentId);

               catData = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
               recType = TxBE16( catData->t.crRecType);

               switch (recType)
               {
                  case HFS_CR_Dthread:          // need to cache name for folders too
                     if (catData->t.name.length != 0)
                     {
                        name = &(hfs->Ic[ thisCnId].Name); // point to name in the cache entry
                        if (*name == NULL)      // nothing there yet (1st link)
                        {
                           if ((*name = TxAlloc( 1, TxBE16( catData->t.name.length) + 1)) != NULL)
                           {
                              TxMacUniStr2Ascii( &(catData->t.name), *name);
                           }
                           else
                           {
                              rc = DFS_ALLOC_ERROR;
                           }
                        }
                     }
                  case HFS_CR_Fthread:          // cache the parent-ID and TRACE/debug
                     parentId = TxBE32( catData->t.parentId);
                     hfs->Ic[ thisCnId].Parent = parentId;

                     if (TxaOptSet( TXA_O_DEBUG))
                     {
                        TxPrint( "Set Parent: %8.8lx CnID: %8.8lx %s %s\n", parentId, thisCnId,
                           (recType == HFS_CR_Fthread) ? "File" : "Folder:", (name && *name) ? *name : "");
                     }
                     else
                     {
                        TRACES(( "Set Parent: %8.8lx CnID: %8.8lx %s %s\n", parentId, thisCnId,
                           (recType == HFS_CR_Fthread) ? "File" : "Folder:", (name && *name) ? *name : ""));
                     }
                     thisCnId++;                // count cached entry
                     break;

                  default:                      // skip other record types
                     break;
               }
            }
            dfsProgressShow( thisCnId, 0, (ULN64) *cnids, "Cached:");
         }
         thisNodeNr = TxBE32( desc->lNext);     // next leaf, or 0 (end)

      } while ((rc == NO_ERROR) && (thisNodeNr != 0) && !TxAbort());

      TxFreeMem( node);
   }
   RETURN (rc);
}                                               // end 'dfsHfsNpWalkLeafs'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Display one line of error description for each bit in error-flag
/*****************************************************************************/
ULONG dfsHfsDispError
(
   ULN64               flags,                   // IN    error flag
   ULN64               d2,                      // IN    dummy
   char               *lead,                    // IN    leading string
   void               *param                    // IN    pointer parameter
)
{
   ENTER();

   RETURN (NO_ERROR);
}                                               // end 'dfsHfsDispError'
/*---------------------------------------------------------------------------*/


