//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FAT dump & display Sector Lookup Table
//
// Author: J. van Wijk
//
// JvW  21-05-97   Initial version, derived from DFSHPFS
// JvW  05-01-2000 Work-around for AllocChain bug (size 0)
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsafat.h>                            // FAT analysis definitions
#include <dfsufat.h>                            // FAT utility functions

#include <dfslfat.h>                            // FAT sector lookup


// Add FAT directory info to SLT array
static ULONG dfsFatSltDirectory
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   BYTE                type,                    // IN    sector-type for DIR
   ULN64               ref,                     // IN    related LSN (dirsect)
   BYTE                info,                    // IN    related INFO (entry)
   ULONG               flags                    // IN    error flags
);


/*****************************************************************************/
// Add info to sector-lookup-table for Fat filesystem
/*****************************************************************************/
ULONG dfsFatSltBuild
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    pointer parameter
)
{
   ULONG               dr  = NO_ERROR;
   DFSAFAT            *fsi = (DFSAFAT *) param; // FAT info structure
   S_SPACE            *root;                    // allocation for Root
   ULONG               rootchunks;              // nr of Root fragments
   int                 fnr;
   ULONG               fsn;                     // FAT nr and sectornumber

   ENTER();

   if ((fsi->FatFaked == FALSE) &&              // no faked values (no bootrec)
       (fsi->CacheA.Value != NULL))             // valid fat structures
   {
      if (dfsa->FsUnallocated == 0)             // probably not set yet ...
      {
         DFSFNCALL( dfsa->FsAllocDisplay, 0, 0, "@", NULL);
      }
      dfsSlTableAdd( fsi->Sect,                  DFS64MAX, ST_FINAL, LSN_BOOTR, 0, 0);

                                                // Approximate sectors todo:
      dfsSlTableFsInit( (ULN64) fsi->Sect -     // RAW size of FS
                        dfsa->FsUnallocated);   // minus known freespace

      if (fsi->Fat1 > 1)                        // boot/admin space present
      {
         dfsSlTableAdd( 1, fsi->Fat1 - 1, ST_RESRV, LSN_BOOTR, 0, 0);
      }
      for ( fnr = 0; fnr < fat->NrOfFats; fnr++)
      {
         fsn = fat->Fat1 + (fnr * fat->FatSectors);
         dfsSlTableAdd( fsn, fat->FatSectors, (fnr == 0) ? ST_FAT_1 : ST_FAT_2, LSN_BOOTR, 0, 0);
      }
      dr = dfsDirLsn2Space( fsi->Root, &rootchunks, &root);
      if (dr == NO_ERROR)
      {
         dr = dfsFatSltDirectory( rootchunks, root, ST_ROOTD, LSN_BOOTR, 0, 0);
         if (root)
         {
            TxFreeMem( root);
         }
      }
   }
   else
   {
      dr = DFS_ST_MISMATCH;                     // bad structures
   }
   RETURN (dr);
}                                               // end 'dfsFatSltBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display one line of error description for each bit in error-flag
/*****************************************************************************/
ULONG dfsFatDispError
(
   ULN64               flags,                   // IN    error flag
   ULN64               d2,                      // IN    dummy
   char               *lead,                    // IN    leading string
   void               *param                    // IN    pointer parameter
)
{
   ENTER();

   if (flags & EF_ALLOC_CL_FREE)
   {
      TxPrint("%sAllocation chain ends in a free cluster\n", lead);
   }
   if (flags & EF_ALLOC_CL_BADS)
   {
      TxPrint("%sAllocation chain ends in a cluster with bad sectors\n", lead);
   }
   if (flags & EF_CLUSTER_RANGE)
   {
      TxPrint("%sCluster value is invalid for current volume\n", lead);
   }
   if (flags & EF_RUNAWAY_CHAIN)
   {
      TxPrint("%sCluster chain is too long for specified item\n", lead);
   }
   RETURN (NO_ERROR);
}                                               // end 'dfsFatDispError'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add FAT directory info to SLT array
/*****************************************************************************/
static ULONG dfsFatSltDirectory
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   BYTE                type,                    // IN    sector-type for DIR
   ULN64               ref,                     // IN    related LSN (dirsect)
   BYTE                info,                    // IN    related INFO (entry)
   ULONG               flags                    // IN    error flags
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   USHORT              entry;                   // index in fat directory
   S_FATDIR           *fatdir;                  // Fat directory sector
   ULN64               dlsn;                    // Directory sector LSN
   ULONG               ef = 0;                  // Initialize error flag!
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp = NULL;               // Alloc chunk array
   USHORT              fatEntries;              // entries per sector
   S_VFSLOT           *fl;                      // ptr to LFN entry struct

   ENTER();

   TRACES(("chunks: %lu, type %2.2x, ref: 0x%llX, flags: %8.8lX\n",
            chunks,      type,       ref,         flags));
   dfsSlTableSpaceAdd(chunks, space,            // add the DIR itself
                      type, ref, info, flags);

   if ((fatdir = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      for ( chunk = 0;
           (chunk < chunks) && (rc == NO_ERROR);
            chunk++)                            // walk all alloc chunks
      {
         for ( sect = 0;
              (sect < space[chunk].size) && (rc == NO_ERROR);
               sect++)                          // each sector in chunk
         {
            fatEntries = dfsGetSectorSize() / sizeof(S_FATDIR);
            dlsn = space[chunk].start + sect;
            rc = dfsRead( dlsn, 1, (BYTE   *) fatdir);
            switch (dfsIdentifySector( dlsn, 0, (BYTE   *) fatdir))
            {
               case ST_ROOTD:
               case ST_SUBDR:
               case ST_SUBRT:
               case ST_DIREC:
                  for ( entry = 0;
                       (entry < fatEntries) && (rc == NO_ERROR);
                        entry++)                // each dir-entry in sector
                  {
                     TRACES(("Chunk: %lu, Sect: %lu, entry %u\n", chunk, sect, entry));

                     if (dfsFatDir2Clust(&fatdir[entry]) != 0) // space allocated ?
                     {
                        switch (fatdir[entry].Name[0])
                        {
                           case FAT_DIRDEL:     // deleted entry
                           case FAT_DIRDOT:     // . or .. entry
                              TRACES(("Ignore:%5u Name: '%*.*s'\n", entry,
                                       11, 11, fatdir[entry].Name));
                           case FAT_DIRFREE:    // free entry
                              break;

                           default:             // used, but could be LFN
                              fl = (S_VFSLOT *) &(fatdir[entry]);
                              if ((fl->FatAttrib == VFAT_ATTRIB) &&
                                  (fl->clust     == VFAT_CLUSTR) )
                              {
                                 TRACES(("Entry :%5u Lfn slotId: 0x%2.2hx\n", entry, fl->SlotId));
                              }
                              else
                              {
                                 TRACES(("Entry :%5u Name: '%*.*s'\n", entry,
                                          11, 11, fatdir[entry].Name));
                                 ef = 0;
                                 dfsFatAllocChain( dfsFatDir2Clust(
                                                   &fatdir[entry]),
                                                    fatdir[entry].fsize,
                                                   &ef, &nr, &sp);
                                 if (sp != NULL)
                                 {
                                    TRINIT(70);
                                       dfsSspaceDisplay( SD_DEFAULT, nr, sp);
                                    TREXIT();
                                    if (fatdir[entry].FatAttrib & FATTR_DIRECTORY)
                                    {
                                       rc = dfsFatSltDirectory( nr, sp, ST_SUBDR,
                                                                dlsn, entry | DFSSNINFO,
                                                                ef);
                                    }
                                    else        // Add file allocation
                                    {
                                       dfsSlTableSpaceAdd( nr, sp, ST_FDATA,
                                                           dlsn, entry | DFSSNINFO,
                                                           ef);
                                    }
                                    TxFreeMem( sp);
                                 }
                              }
                              break;
                        }
                     }
                     if (dfsSlTableStatus(&nr) == SLT_STOPPING)
                     {
                        rc = DFS_QUIT;          // break the recursion
                     }                          // RC will be dicarded later!
                  }
                  break;

               default:                         // invalid DIR sector, skip
                  break;
            }
         }
      }
      TxFreeMem( fatdir);
   }
   RETURN( rc);
}                                               // end 'dfsFatSltDirectory'
/*---------------------------------------------------------------------------*/

