//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// EXT2/3 dump & display Sector Lookup Table
//
// Author: J. van Wijk
//
// JvW  24-02-2018 Name/Parent build ported from JFS (iterates over all Inodes)
// JvW  24-02-2018 Changed recursion guard in SLT/NP build to decrement to 0
// JvW  30-03-2004 Initial version
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsext.h>                             // EXT disk structures
#include <dfsaext.h>                            // EXT analysis definitions
#include <dfsuext.h>                            // EXT utility functions
#include <dfslext.h>                            // EXT sector lookup


#define DFS_EXT_RECURSE_MAX   100               // recurse up to 100 levels from root

#define DFS_EXT_NO_PARENT     0                 // not a valid parent Inode#


// Build the Name+Parent cache for EXT filesystem (stripped down SLT build)
// Note: needs to be abortable when not started as a thread!
ULONG dfsExtNpBuild                             // RET   possibbly ABORTED
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    not used
);

// Walk Inodes, and for the directories, find name/inode pairs for NP cache
static ULONG dfsExtNpWalkInodes
(
   ULONG              *inodes                   // INOUT Inodes added to cache
);

// Add single Inode info + data to  SLT array, recursive into directories
static ULONG dfsExtSltInode
(
   ULONG               guard,                   // IN    recursion guard
   BYTE                stype,                   // IN    type for the (file)data
   ULONG               this,                    // IN    Inode number to handle
   ULONG               parent,                  // IN    referencing inode nr
   ULONG               ef                       // IN    error flag
);

// Add Inode info + data to NP cache, recursive into directories when allowed
static ULONG dfsExtNpInode
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               this,                    // IN    Inode number to handle
   ULONG               parent,                  // IN    parent inode, for cache
   ULONG              *inodes                   // INOUT Inodes added to cache
);

// Add Inodes for a sequence of EXT Directory entries to the SLT, recursive
static ULONG dfsExtSltDir
(
   ULONG               guard,                   // IN    recursion guard
   ULONG               dirInode,                // IN    referencing Inode number
   DFSISPACE          *dirData                  // IN    Directory blocks
);

// Add Inodes for a sequence of EXT Directory entries to NP cache, recursive
static ULONG dfsExtNpDir
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               dirInode,                // IN    referencing Inode number
   DFSISPACE          *dirData,                 // IN    Directory blocks
   ULONG              *inodes                   // INOUT Inodes added to cache
);

// Status for write-back Inodes with added magic XATTR name/parent on SLT build
static ULONG dfsExtWriteMagicNames = DFS_EXT_M_OFF;

/*****************************************************************************/
// Add info to sector-lookup-table for Ext filesystem
/*****************************************************************************/
ULONG dfsExtSltBuild
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    pointer parameter
)
{
   ULONG               rc  = NO_ERROR;
   DFSAEXT            *ext = (DFSAEXT *) param; // EXT info structure
   ULN64               gdFirstBlk = 1;                       //- 2nd block, is block 1
   ULONG               gdReserved = ext->GdtReservedBlocks;  //- Size of reserved GDT area
   ULONG               i;
   ULN64               freeSect = ext->FreeSect; // from Group descriptors

   ENTER();

   if (TxaOptSet('M'))                          // add magic XATTR filenames to
   {                                            // the Inodes for recovery
      if (dfsa->FsDirtyStatus != DFSTAT_DIRTY)
      {
         dfsExtWriteMagicNames = TxaOptNum( 'M', NULL, DFS_EXT_MBOTH);
      }
      else                                      // not allowed, FS mounted
      {
         TxNamedMessage( !dfsa->batch, 5033, " ERROR: Filesystem mounted ",
            "Adding filename XATTRs to the Inodes for later recovery/undelete "
            "is not possible since the filesystem is in a MOUNTED state.\n\n"
            "Use of the '-M' option is restricted to CLEAN and UNMOUNTED "
            "filesystems");
      }
   }
   dfsSlTableAdd( ext->Sect,              DFS64MAX, ST_FINAL, LSN_BOOTR, 0, 0);

   if (ext->Ic == NULL)                         // no name/parent cache yet
   {
      ext->Ic = TxAlloc( ext->Groups * ext->sup->InodesPerGroup, sizeof( S_EXT_INAMECACHE));
   }

   if ((dfsa->FsUnallocated != 0) && (dfsa->FsUnallocated < freeSect))
   {
      freeSect = dfsa->FsUnallocated;           // value from bitmap analysis
   }
                                                // Approximate sectors todo:
   dfsSlTableFsInit( ext->Sect - freeSect);     // RAW size minus free sectors

   //- Handle BR, superblock, GDT and GDT-reserve separately for group 0
   //- to be able to handle small blocksize and large-sectorsize differences
   if (dfsGetSectorSize() == 512)
   {
      dfsSlTableAdd( LSN_BOOTR + 1, 1, ST_BOOTR, L64_NULL, 0, 0);
   }
   if (ext->BlockSize <= 1024)                  // 1 KiB blocks
   {
      gdFirstBlk += 1;                          // GDT will be in 3rd block, is block 2
      gdReserved -= 1;                          // adjust size of reserved GDT area too

      //- first superblock, taking a whole block
      dfsSlTableAdd( LSN_EXTSUP,     ext->SectorsPerBlock,               ST_EXTSUP, L64_NULL, 0, 0);
   }
   else                                         // block at least 2 KiB
   {
      //- first superblock, taking rest of whole block, after 1024-byte boot-record
      if (dfsGetSectorSize() >= 2048)           // Boot+Super in 1 sector
      {
         dfsSlTableAdd( LSN_EXTSUP, (ext->SectorsPerBlock - LSN_EXTSUP), ST_EXTBSU, L64_NULL, 0, 0);
      }
      else                                      // Bootsector(s) in different sector from Super
      {
         dfsSlTableAdd( LSN_EXTSUP, (ext->SectorsPerBlock - LSN_EXTSUP), ST_EXTSUP, L64_NULL, 0, 0);
      }
   }
   dfsSlTableAdd(  gdFirstBlk                   * ext->SectorsPerBlock,
                                ext->GdtBlocks  * ext->SectorsPerBlock,  ST_EXTGRP, L64_NULL, 0, 0);
   dfsSlTableAdd( (gdFirstBlk + ext->GdtBlocks) * ext->SectorsPerBlock,
                                     gdReserved * ext->SectorsPerBlock,  ST_EXTGDR, L64_NULL, 0, 0);

   for (i = 0; i < ext->Groups; i++)            // iterate over groups
   {
      if ((i > 0) && dfsExtIsBackupSuperBG( i)) // SB+GDT backup group, for all except group 0
      {
         ULONG  bgFirstBlk = i * ext->sup->BlocksPerGroup;

         dfsSlTableAdd( bgFirstBlk                       * ext->SectorsPerBlock,
                                                           ext->SectorsPerBlock,  ST_EXTBUS, L64_NULL, 0, 0);
         dfsSlTableAdd((bgFirstBlk + 1                 ) * ext->SectorsPerBlock,
                                        (ext->GdtBlocks  * ext->SectorsPerBlock), ST_EXTBGR, L64_NULL, 0, 0);
         dfsSlTableAdd((bgFirstBlk + 1 + ext->GdtBlocks) * ext->SectorsPerBlock,
                                 (ext->GdtReservedBlocks * ext->SectorsPerBlock), ST_EXTGDR, L64_NULL, 0, 0);
      }
      if ((ext->GroupDescTable[i].BlockBitmap != 0) &&
          (ext->GroupDescTable[i].InodeBitmap != 0) &&
          (ext->GroupDescTable[i].InodeTable  != 0) &&
          (ext->GroupDescTable[i].FreeBlocks  <= ext->sup->BlocksPerGroup) &&
          (ext->GroupDescTable[i].FreeInodes  <= ext->sup->InodesPerGroup)  )
      {
         S_EXT_GROUPDESCR   *gd = &(ext->GroupDescTable[i]);

         dfsSlTableAdd( gd->BlockBitmap * ext->SectorsPerBlock,   ext->SectorsPerBlock, ST_EXTBBM, L64_NULL, 0, 0);
         dfsSlTableAdd( gd->InodeBitmap * ext->SectorsPerBlock,   ext->SectorsPerBlock, ST_EXTIBM, L64_NULL, 0, 0);
         dfsSlTableAdd( gd->InodeTable  * ext->SectorsPerBlock,
                                          ext->InodeTableBlocks * ext->SectorsPerBlock, ST_EXTITB, L64_NULL, 0, 0);
      }
   }

   if (ext->Ic != NULL)
   {
      ext->Ic[EXT_I_JOURNAL].Parent = EXT_SYS_PARENT;
      ext->Ic[EXT_I_JOURNAL].Name   = TxAlloc(1, strlen(EXT_TXT_JOURNAL) +1);
      strcpy( ext->Ic[EXT_I_JOURNAL].Name,              EXT_TXT_JOURNAL);

      ext->Ic[EXT_I_RESIZE].Parent = EXT_SYS_PARENT;
      ext->Ic[EXT_I_RESIZE].Name   = TxAlloc(1, strlen(EXT_TXT_RESIZE) +1);
      strcpy( ext->Ic[EXT_I_RESIZE].Name,              EXT_TXT_RESIZE);

      ext->Ic[EXT_I_ROOT].Name   = TxAlloc( 1, 1);  //- ROOT name ""
      ext->Ic[EXT_I_ROOT].Parent = EXT_I_ROOT;      //- by convention
   }

   rc = dfsExtSltInode( 0, ST_EXTRES, EXT_I_RESIZE, EXT_SYS_PARENT, 0);

   if ((rc == NO_ERROR) && (ext->sup->FeatureCompat & EXT_FCO_HAS_JOURNAL))
   {
      rc = dfsExtSltInode( 0, ST_EXTJRN, EXT_I_JOURNAL, EXT_SYS_PARENT, 0);
   }
   if (rc == NO_ERROR)                          // add ROOT, and recurse
   {
      rc = dfsExtSltInode( DFS_EXT_RECURSE_MAX, ST_EXTDIR, EXT_I_ROOT, EXT_I_ROOT, 0);
   }
   RETURN (rc);
}                                               // end 'dfsExtSltBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Build the Name+Parent cache for EXT filesystem (stripped down SLT build)
// Note: needs to be abortable when not started as a thread!
/*****************************************************************************/
ULONG dfsExtNpBuild                             // RET   possibbly ABORTED
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    not used
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();

   if (TxaOption( TXA_O_RESET))
   {
      if (ext->Ic != NULL)
      {
         int              i;

         for (i = 0; i <= ext->InodeMax; i++)
         {
            TxFreeMem( ext->Ic[i].Name);        // free cached Inode names
         }
         TxFreeMem( ext->Ic);                   // free name/parent cache
      }
   }

   if (ext->Ic == NULL)                         // no name/parent cache yet
   {
      ext->Ic = TxAlloc( ext->Groups * ext->sup->InodesPerGroup, sizeof( S_EXT_INAMECACHE));

      if (ext->Ic != NULL)
      {
         ULONG         iDone = 0;               // Inodes done
         ULONG         pFlag = (TxaOption(TXA_O_DEBUG)) ? DFSP_STAT : DFSP_BARS;

         ext->Ic[EXT_I_JOURNAL].Parent = EXT_SYS_PARENT;
         ext->Ic[EXT_I_JOURNAL].Name   = TxAlloc(1, strlen(EXT_TXT_JOURNAL) +1);
         strcpy( ext->Ic[EXT_I_JOURNAL].Name,              EXT_TXT_JOURNAL);

         ext->Ic[EXT_I_RESIZE].Parent = EXT_SYS_PARENT;
         ext->Ic[EXT_I_RESIZE].Name   = TxAlloc(1, strlen(EXT_TXT_RESIZE) +1);
         strcpy( ext->Ic[EXT_I_RESIZE].Name,              EXT_TXT_RESIZE);

         ext->Ic[EXT_I_ROOT].Name   = TxAlloc( 1, 1);  //- ROOT name ""
         ext->Ic[EXT_I_ROOT].Parent = EXT_I_ROOT;      //- by convention

         if (TxaOption( DFS_O_TREE))            // non-standard, recurse from ROOT
         {
            dfsProgressInit( 0, ext->InodeUsed, 0, "Inode:", "cached", pFlag | DFSP_VDEC, 0);
            rc = dfsExtNpInode( DFS_EXT_RECURSE_MAX, EXT_I_ROOT, EXT_I_ROOT, &iDone);
         }
         else                                   // standard, linear walk ALL DIR-Inodes
         {
            dfsProgressInit( 0, ext->InodeMax + 1, 0, "Inode:", "handled", pFlag | DFSP_VDEC, 0);

            //- start with just the files in the ROOT (no tree recurse) and ignore errors!
            dfsExtNpInode( 1, EXT_I_ROOT, EXT_I_ROOT, &iDone);

            //- then add all of the directory Inodes present ...
            rc = dfsExtNpWalkInodes( &iDone);
         }
         dfsProgressTerm();
      }
   }
   RETURN (rc);
}                                               // end 'dfsExtNpBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Walk Inodes, and for the directories, find name/inode pairs for NP cache
/*****************************************************************************/
static ULONG dfsExtNpWalkInodes
(
   ULONG              *inodes                   // INOUT Inodes added to cache
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               ino;

   ENTER();

   for (ino = EXT_I_ROOT + 1; ino <= ext->InodeMax; ino++)
   {
      if (dfsExtInoBitmapCache( ino))           // Inode number in-use ?
      {
         //- DFS_JFS_NO_PARENT will analyse directory Inodes only (not added itself)
         rc = dfsExtNpInode( 1, ino, DFS_EXT_NO_PARENT, inodes);
         if ((ino & 0xff) == 0)                  // every 256, unallocated too
         {
            if (TxAbort())
            {
               break;
            }
            else
            {
               dfsProgressShow( ino, 0, (ULN64) *inodes, "Cached:");
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsExtNpWalkInodes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inode info + data to SLT array, recursive into directories when allowed
/*****************************************************************************/
static ULONG dfsExtSltInode
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   BYTE                stype,                   // IN    type for the (file)data
   ULONG               this,                    // IN    Inode number to handle
   ULONG               parent,                  // IN    parent inode, for cache
   ULONG               ef                       // IN    error flag
)
{
   ULONG               rc  = NO_ERROR;
   S_EXT_INODE        *inode  = NULL;           // Inode sector data
   USHORT              inoIdx;
   ULONG               inoLsn = dfsExtInode2LsnIndex( this,   &inoIdx);
   USHORT              inoInf = inoIdx | DFSSNINFO;

   ENTER();

   TRACES(("guard:%lu this:%8.8lx parent:%8.8lx\n", recursionsAllowed, this, parent));

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      if ((rc = dfsExtGetInodeRecord( inoLsn, inoIdx, inode)) == NO_ERROR)
      {
         DFSISPACE           fdata;             // file data allocation
         DFSISPACE           meta1;             // meta data L1/leaf
         DFSISPACE           meta2;             // meta data L2/index

         memset( &fdata, 0, sizeof(DFSISPACE)); // initialize to all zero
         memset( &meta1, 0, sizeof(DFSISPACE));
         memset( &meta2, 0, sizeof(DFSISPACE));


         if (ext->Ic)                           // if name cache there
         {
            ext->Ic[ this].Parent = parent;     // add parent ref to cache

            TRACES(( "Inode Parent: %8.8lx Inode: %8.8lx Name:'%s'\n", parent, this, ext->Ic[ this].Name));
         }

         //- first add the allocation  for this Inode, primary and meta
         if (dfsExtInodeData2Space( inode, &fdata, &meta1, &meta2) == NO_ERROR)
         {
            if (fdata.chunks != 0)              // regular data present
            {
               dfsSlTableSpaceAdd( fdata.chunks, fdata.space, stype, inoLsn, inoInf, 0);

               if (meta1.chunks != 0)           // meta allocation type-1 present
               {
                  TRACES(("META-1 chunks: %hu, type: %lu\n", meta1.chunks, meta1.spid));
                  dfsSlTableSpaceAdd( meta1.chunks, meta1.space, meta1.spid, inoLsn, inoInf, 0);
               }
               if (meta2.chunks != 0)           // meta allocation type-2 present
               {
                  TRACES(("META-2 chunks: %hu, type: %lu\n", meta2.chunks, meta2.spid));
                  dfsSlTableSpaceAdd( meta2.chunks, meta2.space, meta2.spid, inoLsn, inoInf, 0);
               }
               if (((inode->Flags & EXT_FL_EXTENTS) == 0) && // inode uses block-indirect alloc
                    (inode->Alloc.bi.Indirect3      != 0)  ) // and triple-indirect block there
               {
                  dfsSlTableAdd( inode->Alloc.bi.Indirect3 * ext->SectorsPerBlock,
                                                             ext->SectorsPerBlock,
                                                             ST_EXTBI3, inoLsn, inoInf, 0);
               }
            }
            else
            {
               //- possibly inlined data/symlink, nothing to add to SLT
            }
         }
         else
         {
            TRACES(( "Error on Inode 0x%8.8lx, rc: %lu\n", this, rc));
            ef |= EF_BAD_INODE_ALL;             // report error on Inode
         }
         if (inode->EaBlockLo)                  // ad EA block, when there
         {
            dfsSlTableAdd( inode->EaBlockLo * ext->SectorsPerBlock,
                                              ext->SectorsPerBlock,
                                              ST_EXTEAB, inoLsn, inoInf, 0);
         }

         //- Then, for directories, traverse the directory data recursively
         if (S_ISDIR( inode->Mode))             // Directory
         {
            if (recursionsAllowed)              // we are allowed to recurse deeper
            {
               rc = dfsExtSltDir( recursionsAllowed - 1, this, &fdata);
            }
         }

         TxFreeMem( fdata.space);               // free the primary data
         TxFreeMem( meta1.space);               // free the meta allocation
         TxFreeMem( meta2.space);
      }
      else if (rc != DFS_ALLOC_ERROR)
      {
         ef |= EF_BAD_INODE_LSN;
         rc  = NO_ERROR;                        // ignore structural problems
      }
      TxFreeMem( inode);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsExtSltInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inode info + data to NP cache, recursive into directories when allowed
/*****************************************************************************/
static ULONG dfsExtNpInode
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               this,                    // IN    Inode number to handle
   ULONG               parent,                  // IN    parent inode, for cache
   ULONG              *inodes                   // INOUT Inodes added to cache
)
{
   ULONG               rc  = NO_ERROR;
   S_EXT_INODE        *inode  = NULL;           // Inode sector data
   USHORT              inoIdx;
   ULONG               inoLsn = dfsExtInode2LsnIndex( this,   &inoIdx);

   ENTER();

   TRACES(("guard:%lu this:%8.8lx parent:%8.8lx\n", recursionsAllowed, this, parent));

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      if ((rc = dfsExtGetInodeRecord( inoLsn, inoIdx, inode)) == NO_ERROR)
      {
         if (parent != DFS_EXT_NO_PARENT)       // only set for a KNOWN parent
         {
            ext->Ic[ this].Parent = parent;     // add parent ref to cache

            if (TxaOptSet( TXA_O_DEBUG))
            {
               TxPrint( "Set Inode Parent: %8.8lx Inode: %8.8lx Name:'%s'\n",
                               parent, this, ext->Ic[ this].Name);
            }
            else
            {
               TRACES(( "Set Inode Parent: %8.8lx Inode: %8.8lx Name:'%s'\n",
                               parent, this, ext->Ic[ this].Name));
            }
         }

         if (S_ISDIR( inode->Mode))             // Directory
         {
            if (recursionsAllowed)              // we are allowed to recurse deeper
            {
               DFSISPACE           fdata;       // file data allocation

               memset( &fdata, 0, sizeof(DFSISPACE)); // initialize to all zero

               if (dfsExtInodeData2Space( inode, &fdata, NULL, NULL) == NO_ERROR)
               {
                  rc = dfsExtNpDir( recursionsAllowed - 1, this, &fdata, inodes);
               }
               TxFreeMem( fdata.space);         // free the primary data
            }
         }
      }
      else if (rc != DFS_ALLOC_ERROR)
      {
         rc  = NO_ERROR;                        // ignore structural problems
      }
      TxFreeMem( inode);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsExtNpInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inodes for a sequence of EXT Directory entries to the SLT, recursive
/*****************************************************************************/
static ULONG dfsExtSltDir
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               dirInode,                // IN    referencing Inode number
   DFSISPACE          *dirData                  // IN    Directory blocks
)
{
   ULONG               rc = NO_ERROR;
   ULONG               totalDirSects;           // sectors in SSPACE
   ULONG               done = 0;                // sectors handled
   BYTE               *dirBlock;                // one block of DIR data
   S_EXT_DIRENTRY     *de;                      // single directory slot
   ULONG               offset = 0;              // byte offset into block
   TXLN                fName;                   // filename

   ENTER();
   TRACES(( "Guard: %lu  dirinode:%8.8lx\n", recursionsAllowed, dirInode));

   if ((dirBlock = TxAlloc( 1, ext->BlockSize)) != NULL)
   {
      totalDirSects = dfsSspaceSectors( TRUE, dirData->chunks, dirData->space);

      for (done = 0; done < totalDirSects; done += ext->SectorsPerBlock)
      {
         rc = dfsSspaceReadFilePart( dirData->chunks, dirData->space,
                                     done, ext->SectorsPerBlock, dirBlock);

         offset = 0;                            // (re)start at begin block!
         while ((rc == NO_ERROR) && (offset < ext->BlockSize) && !TxAbort())
         {
            de = (S_EXT_DIRENTRY *) ((BYTE *) dirBlock + offset);

            if (de->Inode == 0)                 // unused entry, end of DIR
            {
               if (de->RecLen == 0)
               {
                  //- should not happen if all RecLen values are correct
                  TRACES(("Aborting dirwalk on de->Inode 0 and RecLen 0\n"));
                  done = totalDirSects;         // abort SSPACE-block loop too
                  break;                        // end the loop
               }
            }
            else                                // normal entry, handle it
            {
               TxCopy( fName, de->Name, de->NameLen + 1); // null-terminated name

               TRACES(("Evaluate: '%s' with ino:%8.8lx reclen:%lu\n", fName, de->Inode, de->RecLen));
               if ((strcmp( fName, ".") != 0) && (strcmp( fName, "..") != 0))
               {
                  if (ext->Ic)                  // if name cache there
                  {
                     char  **name = &(ext->Ic[ de->Inode].Name);

                     if (*name == NULL)         // nothing there yet (1st link)
                     {
                        if ((*name = TxAlloc( 1, strlen(fName) +1)) != NULL)
                        {
                           strcpy( *name, fName);
                           TRACES(( "Dir-Inode %8.8lx, name:'%s'\n", de->Inode, *name));
                        }
                     }
                  }
                  if (dfsSlTableStatus( NULL) == SLT_STOPPING)
                  {
                     rc = DFS_QUIT;             // break the recursion
                  }                             // to allow build stop
                  else
                  {
                     //- add Inode, recurse (guard decrement on each call to SltDir)
                     rc = dfsExtSltInode( recursionsAllowed,
                                         (de->FileType == EXT_FT_DIR) ? ST_EXTDIR : ST_FDATA,
                                          de->Inode, dirInode, 0);
                  }
               }
            }
            offset += de->RecLen;               // to next DIR entry, or end
         }
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtSltDir'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Inodes for a sequence of EXT Directory entries to NP cache, recursive
/*****************************************************************************/
static ULONG dfsExtNpDir
(
   ULONG               recursionsAllowed,       // IN    recursion levels allowed
   ULONG               dirInode,                // IN    referencing Inode number
   DFSISPACE          *dirData,                 // IN    Directory blocks
   ULONG              *inodes                   // INOUT Inodes added to cache
)
{
   ULONG               rc = NO_ERROR;
   ULONG               totalDirSects;           // sectors in SSPACE
   ULONG               done = 0;                // sectors handled
   BYTE               *dirBlock;                // one block of DIR data
   S_EXT_DIRENTRY     *de;                      // single directory slot
   ULONG               offset = 0;              // byte offset into block
   TXLN                fName;                   // filename

   ENTER();
   TRACES(( "Guard: %lu  dirinode:%8.8lx\n", recursionsAllowed, dirInode));

   if ((dirBlock = TxAlloc( 1, ext->BlockSize)) != NULL)
   {
      totalDirSects = dfsSspaceSectors( TRUE, dirData->chunks, dirData->space);

      for (done = 0; done < totalDirSects; done += ext->SectorsPerBlock)
      {
         rc = dfsSspaceReadFilePart( dirData->chunks, dirData->space,
                                     done, ext->SectorsPerBlock, dirBlock);

         offset = 0;                            // (re)start at begin block!
         while ((rc == NO_ERROR) && (offset < ext->BlockSize) && !TxAbort())
         {
            de = (S_EXT_DIRENTRY *) ((BYTE *) dirBlock + offset);

            if (de->Inode == 0)                 // unused entry, end of DIR
            {
               if (de->RecLen == 0)
               {
                  //- should not happen if all RecLen values are correct
                  TRACES(("Aborting dirwalk on de->Inode 0 and RecLen 0\n"));
                  done = totalDirSects;         // abort SSPACE-block loop too
                  break;                        // end the loop
               }
            }
            else                                // normal entry, handle it
            {
               TxCopy( fName, de->Name, de->NameLen + 1); // null-terminated name

               if ((strcmp( fName, ".") != 0) && (strcmp( fName, "..") != 0))
               {
                  (*inodes)++;                  // increment #inodes done
                  if (recursionsAllowed)        // progress here only when doing TREE!
                  {
                     if (((*inodes) % 100) == 0) // don't update BAR too often
                     {
                        dfsProgressShow( *inodes, 0, 0, NULL);
                     }
                  }
                  if (ext->Ic)                  // if name cache there
                  {
                     char  **name = &(ext->Ic[ de->Inode].Name);

                     if (*name == NULL)         // nothing there yet (1st link)
                     {
                        if ((*name = TxAlloc( 1, strlen(fName) +1)) != NULL)
                        {
                           strcpy( *name, fName);
                           TRACES(( "Dir-Inode %8.8lx, name:'%s'\n", de->Inode, *name));
                        }
                     }
                  }
                  //- add Inode, recurse (guard decrement on each call to SltDir)
                  rc = dfsExtNpInode( recursionsAllowed, de->Inode, dirInode, inodes);
               }
            }
            offset += de->RecLen;               // to next DIR entry, or end
         }
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtNpDir'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display one line of error description for each bit in error-flag
/*****************************************************************************/
ULONG dfsExtDispError
(
   ULN64               flags,                   // IN    error flag
   ULN64               d2,                      // IN    dummy
   char               *lead,                    // IN    leading string
   void               *param                    // IN    pointer parameter
)
{
   ENTER();

   RETURN (NO_ERROR);
}                                               // end 'dfsExtDispError'
/*---------------------------------------------------------------------------*/


