//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// EFAT dump & display Sector Lookup Table
//
// Author: J. van Wijk
//
// JvW  19-06-2014   Initial version, derived from DFSLFAT
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsaexft.h>                           // EXFT display & analysis
#include <dfsuexft.h>                           // EXFT utility functions
#include <dfslexft.h>                           // EXFT SLT functions


// Add FAT directory info to SLT array
static ULONG dfsExftSltDirectory
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   BYTE                type,                    // IN    sector-type for DIR
   LSN                 ref,                     // IN    related LSN (fnode)
   BYTE                info,                    // IN    related INFO (FAT)
   ULONG               flags                    // IN    error flags
);


/*****************************************************************************/
// Add info to sector-lookup-table for Fat filesystem
/*****************************************************************************/
ULONG dfsExftSltBuild
(
   ULONG               dummy1,                  // IN    not used
   ULONG               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    pointer parameter
)
{
   ULONG               dr  = NO_ERROR;
   DFSAEXFT           *fsi = (DFSAEXFT *) param; // EFAT info structure
   S_SPACE            *root;                    // allocation for Root
   ULONG               rootchunks;              // nr of Root fragments
   int                 fnr;
   ULONG               fsn;                     // FAT nr and sectornumber

   ENTER();

   if ((fsi->ExftFaked == FALSE) &&             // no faked values (no bootrec)
       (fsi->CacheA.Value != NULL))             // valid fat structures
   {
      if (dfsa->FsUnallocated == 0)             // probably not set yet ...
      {
         DFSFNCALL( dfsa->FsAllocDisplay, 0, 0, "@", NULL);
      }
      TRACES(("fsi->Sect: 0x%8.8lx\n", fsi->Sect));

      dfsSlTableAdd( fsi->Sect,                  DFSULMAX, ST_FINAL, LSN_BOOTR, 0, 0);

      fsn = fsi->ClHeap + (fsi->RawClust * fsi->ClustSize);
      if (fsn < fsi->Sect)                      // final cluster slack
      {
         dfsSlTableAdd( fsn, fsi->Sect - fsn, ST_RESRV, LSN_BOOTR, 0, 0);
      }

      if (fsi->Fat1 > 1)                        // FAT alignment space (reserved)
      {
         dfsSlTableAdd( 1, fsi->Fat1 - 1, ST_RESRV, LSN_BOOTR, 0, 0);
      }

      fsn = fsi->Fat1 + (fsi->NrOfFats * fsi->FatSectors);
      if (fsn < fsi->ClHeap)                    // HEAP alignment space (reserved)
      {
         dfsSlTableAdd( fsn, fsi->ClHeap - fsn, ST_RESRV, LSN_BOOTR, 0, 0);
      }
                                                // Approximate sectors todo:
      dfsSlTableFsInit( fsi->Sect -             // RAW size of FS
                        dfsa->FsUnallocated);   // minus known freespace

      for ( fnr = 0; fnr < exft->NrOfFats; fnr++)
      {
         fsn = exft->Fat1 + (fnr * exft->FatSectors);
         dfsSlTableAdd( fsn, exft->FatSectors, (fnr == 0) ? ST_FAT_1 : ST_FAT_2, LSN_BOOTR, 0, 0);
      }
      dr = dfsExftDirLsn2Space( fsi->Root, &rootchunks, &root);
      if (dr == NO_ERROR)
      {
         dr = dfsExftSltDirectory( rootchunks, root, ST_ROOTD, LSN_BOOTR, 0, 0);
         if (root)
         {
            free( root);                        // re-alloc'ed no TxFreeMem !
         }
      }
   }
   else
   {
      dr = DFS_ST_MISMATCH;                     // bad structures
   }
   RETURN (dr);
}                                               // end 'dfsExftSltBuild'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display one line of error description for each bit in error-flag
/*****************************************************************************/
ULONG dfsExftDispError
(
   ULONG               flags,                   // IN    error flag
   ULONG               d2,                      // IN    dummy
   char               *lead,                    // IN    leading string
   void               *param                    // IN    pointer parameter
)
{
   ENTER();

   if (flags & EF_ALLOC_CL_FREE)
   {
      TxPrint("%sAllocation chain ends in a free cluster\n", lead);
   }
   if (flags & EF_ALLOC_CL_BADS)
   {
      TxPrint("%sAllocation chain ends in a cluster with bad sectors\n", lead);
   }
   if (flags & EF_CLUSTER_RANGE)
   {
      TxPrint("%sCluster value is invalid for current volume\n", lead);
   }
   if (flags & EF_RUNAWAY_CHAIN)
   {
      TxPrint("%sCluster chain is too long for specified item\n", lead);
   }
   RETURN (NO_ERROR);
}                                               // end 'dfsExftDispError'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add FAT directory info to SLT array
/*****************************************************************************/
static ULONG dfsExftSltDirectory
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   BYTE                type,                    // IN    sector-type for DIR
   LSN                 ref,                     // IN    related LSN (fnode)
   BYTE                info,                    // IN    related INFO (FAT)
   ULONG               flags                    // IN    error flags
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   USHORT              entry;                   // index in fat directory
   S_EXFTDIR           prevEntryBuffer;         // Previous entry, contents
   S_EXFTDIR          *prev;                    // Previous entry, pointer
   S_EXFTDIR          *this;                    // Fat directory entry
   S_EXFTDIR          *fatdir;                  // Fat directory sector
   LSN                 dlsn;                    // Directory sector LSN
   LSN                 prevFileLsn;             // Previous  LSN,   FILE entry
   USHORT              prevFileEntry;           // Previous  entry, FILE
   ULONG               ef = 0;                  // Initialize error flag!
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp = NULL;               // Alloc chunk array
   USHORT              fatEntries;              // entries per sector

   ENTER();

   TRACES(("chunks: %lu, type %2.2x, ref: %8.8lX, flags: %8.8lX\n",
            chunks,      type,       ref,         flags));

   dfsSlTableSpaceAdd(chunks, space,            // add the DIR itself
                      type, ref, info, flags);

   if ((fatdir = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      prev   = &prevEntryBuffer;

      for ( chunk = 0;
           (chunk < chunks) && (rc == NO_ERROR);
            chunk++)                            // walk all alloc chunks
      {
         for ( sect = 0;
              (sect < space[chunk].size) && (rc == NO_ERROR);
               sect++)                          // each sector in chunk
         {
            fatEntries = dfsGetSectorSize() / sizeof(S_EXFTDIR);
            dlsn = space[chunk].start + sect;
            rc = dfsRead( dlsn, 1, (BYTE   *) fatdir);
            switch (dfsIdentifySector( dlsn, 0, (BYTE   *) fatdir))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  for ( entry = 0;
                       (entry < fatEntries) && (rc == NO_ERROR);
                        entry++)                // each dir-entry in sector
                  {
                     this = &(fatdir[entry]);
                     TRACES(("Chunk: %lu, Sect: %lu, entry %u, type:0x%2.2hu\n", chunk, sect, entry, this->EntryType));

                     switch (this->EntryType)
                     {
                        case EXFAT_DIR_FILE:
                           prevFileLsn   = dlsn;    //- remember LSN and entry
                           prevFileEntry = entry;   //- for the STREAM processing
                           break;

                        case EXFAT_DIR_STREAM:
                           if (prev->EntryType == EXFAT_DIR_FILE)
                           {
                              dfsExftAllocStream( this, &ef, &nr, &sp);

                              if (sp != NULL)
                              {
                                 TRINIT(70);
                                    dfsSspaceDisplay( SD_DEFAULT, nr, sp);
                                 TREXIT();

                                 if ((prev->fl.Attrib & FATTR_DIRECTORY) != 0)
                                 {
                                    rc = dfsExftSltDirectory( nr, sp, ST_DIREC, prevFileLsn,
                                                              prevFileEntry | DFSSNINFO, ef);
                                 }
                                 else
                                 {
                                    dfsSlTableSpaceAdd( nr, sp, ST_FDATA, prevFileLsn,
                                                        prevFileEntry | DFSSNINFO, ef);
                                 }
                                 free( sp);     // re-alloc'ed no TxFreeMem !
                              }
                           }
                           else
                           {
                              TRACES(( "EFAT stream, unexpected at this position!\n"));
                           }
                           break;

                        case EXFAT_DIR_BITMAP:
                        case EXFAT_DIR_UPCASE:
                           dfsExftAllocStream( this, &ef, &nr, &sp);

                           if (sp != NULL)
                           {
                              TRINIT(70);
                                 dfsSspaceDisplay( SD_DEFAULT, nr, sp);
                              TREXIT();

                              dfsSlTableSpaceAdd( nr, sp, ST_FDATA,
                                                  dlsn, entry | DFSSNINFO, ef);
                              free( sp);        // re-alloc'ed no TxFreeMem !
                           }
                           break;

                        default:                // other types, deleted, name etc
                           //- no action
                           break;
                     }
                     prevEntryBuffer = *this;   // make persistent copy, each time!

                     if (dfsSlTableStatus(&nr) == SLT_STOPPING)
                     {
                        rc = DFS_QUIT;          // break the recursion
                     }                          // rc will be discarded later!
                  }
                  break;

               default:                         // invalid DIR sector, skip
                  break;
            }
         }
      }
      TxFreeMem( fatdir);
   }
   RETURN( rc);
}                                               // end 'dfsExftSltDirectory'
/*---------------------------------------------------------------------------*/

