//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// JFS disk structure definitions
// Based on information from the IBM open-sourced JFS (Linux)
// Most info from jfs_types.h, jfs_superblock.h, btree.h, dtree.h
//
// Author: J. van Wijk
//
// JvW  19-04-2001   Initial version
//
#ifndef    DFSJFS_H
   #define DFSJFS_H

#define JFLDR_LSN      ((ULN64)  0x01)          // LSN of first JFS LDR sector
#define JFLDR_SIZE          31                  // SIZE in sectors for JFS LDR
#define JFLDR_IMGNAME       "dfsjfldr"          // Default imagefile name

#define JFS_PSIZE              4096             // pagesize in bytes
#define JFS_PSECT                 8             // pagesize in sectors

#define JFS_MAXAG               128             // max nr of Aggregate Groups

                                                // positions fixed for JFS 1.0
#define LSN_JFSUP    ((ULN64)  0x40)            // primary superblock
#define LSN_AIMAP    ((ULN64)  0x48)            // aggregate inode map
#define LSN_AITAB    ((ULN64)  0x58)            // aggregate inode table
#define LSN_JFSU2    ((ULN64)  0x78)            // secondary superblock

#define LSN_AIMAX ((ULN64)  0xfffffffffffffff0) // values like ..fffa seem to be used!

#define JFS_AGGR_SELF_INODE    0x01             // Aggregate self A-inode (meta)
#define JFS_BLOCKAMAP_INODE    0x02             // Allocation Map A-inode (bitmap)
#define JFS_FS1_SUPER_INODE    0x10             // Super-inode    A-inode

#define JFS_ROOT_INODE         0x02             // Filesystem ROOT Inode
#define JFS_FIRSTINODE         0x04             // First real filesystem node

#define JFS_BADBLOCK_LSN       0x5c             // Badblock special Inode LSN

#define JFS_SYS_PARENT         0x01             // Parent Inode# for sys inodes

#define JFS_SYS_RESV           0x00
#define JFS_SYS_MAPX           0x01
#define JFS_SYS_ACLI           0x03

#define JFS_TXT_RESV       "   Fs1 Reserved"
#define JFS_TXT_MAPX       "   Fs1 extension"
#define JFS_TXT_ACLI       "   Fs1 ACL file"

#define DFS_JFS_LFN_EA     ".LONGNAME"          // EA with long filename

#define JFS_MINSIZE       0x01000000            // 16Mb minimum size

                                                // super->Flag values:
#define JFS_AIX           0x80000000            // AIX, POSIX name/directory
#define JFS_OS2           0x40000000            // OS/2, case-insensitive names
#define JFS_DFS           0x20000000            // DCE DFS LFS support

#define JFS_UNICODE       0x00000001            // unicode names

#define JFS_COMMIT        0x00000f00            // commit option mask
#define JFS_GROUPCOMMIT   0x00000100            // group (of 1) commit
#define JFS_LAZYCOMMIT    0x00000200            // lazy commit
#define JFS_TMPFS         0x00000400            // temp FS, no log/commit

#define JFS_INLINELOG     0x00000800            // inline log within FS
#define JFS_INLINEMOVE    0x00001000            // inline log being moved

#define JFS_BAD_SAIT      0x00010000            // secondary ait is bad

#define JFS_DASD_ENABLED  0x00040000            // DASD limits enabled
#define JFS_DASD_PRIME    0x00080000            // Prime DASD usage on boot

#define JFS_DIR_INDEX     0x00200000            // Linux style DIR index table
                                                // used in directory trees

                                                // super->Status values:
#define FM_CLEAN          0x00000000            // FS unmounted and clean
#define FM_MOUNT          0x00000001            // FS mounted cleanly
#define FM_DIRTY          0x00000002            // FS not unmounted and clean,
                                                // commit failure while mounted
                                                // fsck() must be run to repair

#define FM_LOGREDO        0x00000004            // recovery (logredo()) failed
                                                // fsck() must be run to repair

#define FM_EXTENDFS       0x00000008            // FS extendfs() in progress


typedef struct s_pxd                            // physical extent descriptor
{
   BITFIELD32          len : 24;                // length of extent in blocks
   BITFIELD32          hi  :  8;                // upper  8 bits of address
   ULONG               lo;                      // lower 32 bits of address
} S_PXD;                                        // end of struct "s_pxd"

// Generic Btree+ definitions

#define JFS_BT_ROOT             0x01            // root of the tree
#define JFS_BT_LEAF             0x02            // leaf node of the tree
#define JFS_BT_INTN             0x04            // internal node in tree
#define JFS_BT_LEFT             0x10            // leftmost  sibling leaf
#define JFS_BT_RIGHT            0x20            // rightmost sibling leaf

// Allocation Btree specific definitions
#define JFS_DXD_INDEX           0x80            // Index for a B+-tree
#define JFS_DXD_INLINE          0x40            // data extent, inline
#define JFS_DXD_EXTENT          0x20            // data extent, external
#define JFS_DXD_FILE            0x10            // data external, Inode
#define JFS_DXD_DAMAGED         0x08            // corrupted data found

#define JFS_IDXROOTLEAF (JFS_DXD_INDEX | JFS_BT_ROOT | JFS_BT_LEAF)
#define JFS_IDXROOTINTN (JFS_DXD_INDEX | JFS_BT_ROOT | JFS_BT_INTN)

typedef struct s_dxd                            // data extent descriptor DXD
{
   BYTE                dflag;                   // data descriptor flags
   BYTE                db1;                     // data descriptor byte 1
   BYTE                db2;                     // data descriptor byte 2
   BYTE                db3;                     // data descriptor byte 3
   ULONG               size;                    // size in bytes
   S_PXD               pe;                      // physical extent on disk
} S_DXD;                                        // end of struct "s_dxd"


typedef struct s_xad                            // extent allocation descriptor
{
   BYTE                xflag;                   // extent alloc flags
   BYTE                xb1;                     // extent alloc byte 1
   BYTE                xb2;                     // extent alloc byte 2
   BYTE                rhi;                     // relative block, hi  8 bits
   ULONG               rlo;                     // relative block, lo 32 bits
   S_PXD               pe;                      // physical extent on disk
} S_XAD;                                        // end of struct "s_dxd"


#define JFS_XB_XADS                 32          // nr of XAD's per sector
#define JFS_PG_XADS                256          // max nr of XAD's in a page

typedef struct s_xthead                         // extent Btree header
{
   ULN64               next;                    // block next page in sequence
   ULN64               prev;                    // prev page
   BYTE                tflag;                   // tree flags
   BYTE                tb1;                     // reserved
   USHORT              entries;                 // nr of entries used    (+2)
   USHORT              maxentry;                // maximum nr of entries (+2)
   USHORT              ts1;                     // reserved
   S_PXD               self;                    // physical extent for self
   S_XAD               extent[1];               // XAD array in rest of page
} S_XTHEAD;                                     // end of struct "s_xthead"

typedef union s_xtpage                          // extent Btree page
{
   S_XTHEAD            header;                  // Header, incl array start
   S_XAD               xad[JFS_PG_XADS];        // XAD alias, entries 2..255
} S_XTPAGE;                                     // end of struct "s_xtpage"


typedef struct jtime                            // JFS time value (64 bits)
{
   ULONG               hi;                      // high 32 bits, seconds
   ULONG               lo;                      // low  32 bits, nanoseconds
} JTIME;                                        // end of struct "jtime"


#define JFS_LEN_UUID             FS_UUID_LENGTH
#define JFS_LEN_LBL              16

typedef struct jfs_linux
{
   //- Used with Linux, version-2 JFS filesystems; fstab mount identification etc
   DFS_GUID            lxVolUuid;               // 88 128-bit uuid (4-2-2-2-6) no-swap!
   char                lxVolName[JFS_LEN_LBL];  // 98 volume name
   DFS_GUID            lxLogUuid;               // A8 logdevice uuid
} JFS_LINUX;                                    // end of struct "jfs_linux"

typedef struct jfs_aeos2
{
   //- There seems to be some info on the ArcaOS/eCS/OS2 version of JFS (JFS1) here,
   S_PXD               UnknownPxd;              // 88 unknown, similar to XtndLog
   ULONG               zero1;                   // 90 Unknown, ZERO in OS2, not in Linux
   ULONG               unknown1;                // 94 Unknown, non-zero in OS2
   ULONG               zero2;                   // 98 Unknown, ZERO in OS2, label in Linux
   ULONG               zero3;                   // 9C Unknown, ZERO in OS2
   ULONG               unknown2;                // A0 Unknown, non-zero in OS2
   ULONG               zeroPad[5];              // A4 Unknown, ZERO in OS2
} JFS_AEOS2;                                    // end of struct "jfs_aeos2"


#define JFS_VERSION    ((ULONG) 1)              // Used on OS/2 and eCs (bootable)
                                                // Latest Linux uses 2 (and has UUID)

#define SG_JFSUP       3
#define SV_JFSUP       (char)0x4a,(char)0x46,(char)0x53

typedef struct s_jfsup                          // JFS super-block
{
   char                Signature[SG_JFSUP];     // 00 text signature "JFS"
   BYTE                SignatNum;               // 03 numeric part of signature
   ULONG               Version;                 // 04
   ULN64               TotalSec;                // 08 volume size in sectors
   ULONG               BlockSize;               // 10 size of 1 block
   USHORT              Block2Log;               // 14 Log2 of blocksize
   USHORT              FactorLog;               // 16 Log2 of blocksize/size
   ULONG               SectSize;                // 18 sector size (LVM, bytes)
   USHORT              SectszLog;               // 1c Log2 of sector size
   USHORT              Reserved;                // 1e fill for alignment
   ULONG               AllocSize;               // 20 allocation group (AG) size
   ULONG               Flag;                    // 24 aggregate attributes
   ULONG               Status;                  // 28 mount/unmount/recovery
   ULONG               Compression;             // 2c compression when > 0
   S_PXD               SaInodeTable;            // 30 1st extent Sec. aggr I-table
   S_PXD               SaInodeMap;              // 38 1st extent Sec. aggr I-map
   ULONG               LogDevAddr;              // 40 Log device address
   ULONG               LogSerial;               // 44 Log Serial nr at aggr mount
   S_PXD               InlineLog;               // 48 Inline log extent
   S_PXD               FsckSpace;               // 50 fsck workspace (CHKDSK)
   JTIME               LastUpdate;              // 58 last update time
   ULONG               FsckLogLen;              // 60 fsck log size (blocks)
   BYTE                FsckRecent;              // 64 current log: none, 1st, 2nd
   char                FsVolume[BT_LABL_L];     // 65 filesystem label (from BR)

                                                //    next if State & FM_EXTENDFS
   ULN64               XtndSize;                // 70 size to fsextend()
   S_PXD               XtndFsck;                // 78 extend fsck workspace
   S_PXD               XtndLog;                 // 80 extend Inline log extent

   union
   {
      JFS_AEOS2        os2;                     // ArcaOS/eCS/OS2, largely unknown
      JFS_LINUX        linux;                   // LINUX, UUID and 2nd Vol-name
   };

} S_JFSUP;                                      // end of struct "s_jfsup"



// NOTE on OS2/Linux compatibility:
// When traversing a Linux formatted JFS, the directory entries will be sorted
// case-sensitive, on OS/2 they will be case-insensitive. This does not affect
// the structures directly, but the DIR-traversal algorithms


#define JFS_LASTSLOT  ((BYTE) 0xFF)

typedef struct s_jdtslot                        // JFS Btree splitpoint slot
{
   S_PXD               Page;                    // Tree page, physical extent
   BYTE                Dnext;                   // Next entry or 0xff
   BYTE                NameLen;                 // Length of filename
   USHORT              FileName[11];            // FileName, variable length
} S_JDTSLOT;                                    // end of struct "s_jdtslot"


typedef struct s_jdislot                        // JFS Inode dir slot
{
   ULONG               Inode;                   // Inode number
   BYTE                Dnext;                   // Next entry or 0xff
   BYTE                NameLen;                 // Length of filename
   USHORT              FileName[11];            // FileName, variable length
   ULONG               Dindex;                  // Linux only, or 2 more chars
} S_JDISLOT;                                    // end of struct "s_jdislot"

typedef struct s_jdnslot                        // JFS name-continuation slot
{
   BYTE                Dnext;                   // Next entry or 0xff
   BYTE                Count;                   // Unknown purpose, usually 01
   USHORT              FileName[15];            // FileName, variable length
} S_JDNSLOT;                                    // end of struct "s_jdnslot"

typedef union s_jdslot                          // JFS directory slot
{
   S_JDISLOT           i;                       // Inode + name
   S_JDNSLOT           n;                       // Name continuation
   S_JDTSLOT           t;                       // Tree splitpoint
} S_JDSLOT;                                     // end of struct "s_jdslot"



typedef struct s_dasd
{
   BYTE                threshold;               // threshold percentage
   BYTE                delta;                   // alert delta percentage
   BYTE                reserve1;
   BYTE                limit_hi;                // limit in blocks (hi part)
   ULONG               limit_lo;                // limit in blocks (lo part)
   BYTE                reserve2[3];
   BYTE                used_hi;                 // used blocks (hi part)
   ULONG               used_lo;                 // used blocks (lo part)
} S_DASD;                                       // end of struct "s_dasd"


#define DFS_JFS_MAGIC  0xffe95edf               // spells DFSee9ff (almost :-)
#define DFS_JFS_ML     88
#define DFS_JFS_M_OFF  0                        // SLT magic filename OFF
#define DFS_JFS_MFILE  1                        // for filenames only
#define DFS_JFS_MDIRS  2                        // for directories
#define DFS_JFS_MBOTH (DFS_JFS_MFILE | DFS_JFS_MDIRS)

typedef struct s_fmagic                         // recovery/undelete magic
{
   ULONG               magic;                   // 80 magic signature
   ULONG               parent;                  // 84 parent inode nr
   char                fname[ DFS_JFS_ML];      // 88 DFSee undelete magic name
} S_FMAGIC;                                     //    end of struct "s_fmagic"


typedef struct s_dtroot                         // JFS directory root of tree
{
   S_DASD              dasdInfo;                // E0 DASD limit info
   BYTE                rflag;                   // F0 flags for tree root
   BYTE                sCount;                  // F1 used entries in sort-table
   BYTE                fCount;                  // F2 free count
   BYTE                fList;                   // F3 free list header index
   ULONG               parent;                  // F4 parent DIR (dotdot)
   BYTE                sTable[8];               // F8 sort-table
} S_DTROOT;                                     //    end of struct "s_dtroot"

typedef struct s_ftroot                         // JFS file root of tree
{
   ULONG               reserve[4];              // E0 unused for files
   BYTE                rflag;                   // F0 flags for tree root
   BYTE                db1;
   BYTE                xCount;                  // F2 nr of extents + 2
   BYTE                db3;
   S_PXD               pe;                      // F4 physical extent on disk
} S_FTROOT;                                     //    end of struct "s_ftroot"


#define JFS_EA_EXTERN       JFS_DXD_EXTENT      // in external extent
#define JFS_EA_INTERN       JFS_DXD_INLINE      // at offset 0x180

typedef struct s_jfsea
{
   ULONG               size;                    // size of EA area
   S_EABLK             value[1];                // EA value array
} S_JFSEA;                                      // end of struct "s_jfsea"


#define JFS_INODE_AGGREGATE       0x01          // aggregate Inode (system)
#define JFS_INODE_FILESET_1       0x10          // regular fileset Inode
#define JFS_INODE_EXTENT_SIZE     0x20          // sectors in Inode extent

#define JFS_MAXFSET               0x10
#define JFS_MAX_UID               0x1fffff
#define JFS_MAX_GID               0x1fffff
#define JFS_MAX_FSHI              512           // 512 * 4GiB yields 2TiB

#define JFS_FATTRSHIFT           25             // required shift to get FAT
                                                // type attr value from mode

//- extended Mode bits
#define JFS_SPARSE        0x00020000            // sparse regular file
#define JFS_JOURNAL       0x00010000            // journaled file
#define JFS_EA_FREE       0x00040000            // EA area free

typedef struct s_jinode                         // JFS Inode
{
   ULONG               InoStamp;                //  00 fileset mark (signature)
   ULONG               Fileset;                 //  04 fileset number
   ULONG               Self;                    //  08 number of this inode
   ULONG               Generation;              //  0C generation number
   S_PXD               Extent;                  //  10 extent descriptor
   ULN64               FileSize;                //  18 file size       (in bytes)
   ULN64               Allocated;               //  20 Allocated size  (in blocks)
   ULONG               Links;                   //  28 nr of links to this inode
   ULONG               Uid;                     //  2C User-id  of owner
   ULONG               Gid;                     //  30 Group-id of owner
   ULONG               Mode;                    //  34 file attribute/permissions
   JTIME               LastAccess;              //  38 last access         (atime)
   JTIME               LastStatus;              //  40 last status modify  (ctime)
   JTIME               LastModify;              //  48 last data modify    (mtime)
   JTIME               Creation;                //  50 Creation time       (otime)
   S_DXD               Acl;                     //  58 ACL descriptor
   S_DXD               Eax;                     //  68 EA  descriptor   (external)
   ULONG               Compression;             //  78 compression info
   ULONG               AclType;                 //  7C Type of the ACL

   union                                        //     mFilename used with magic
   {                                            //     and 1st-link parent field
      S_FMAGIC         fMagic;                  //  80 DFSee undelete magic name
      BYTE             DirTables[ 96];          //  80 Linux Dir-table slot
   };                                           //     unused on OS/2 JFS

   union                                        //     Dir/File tree header info
   {
      S_FTROOT         File;                    //  E0 File extent tree root
      S_DTROOT         Dir;                     //  E0 Directory   tree root
   };

   union                                        //     Dir/File tree data or other
   {
      S_JDSLOT         Slot[1];                 // 100 first of max  8 Dir slots
      S_XAD            Xad[1];                  // 100 first of max 16 File extents
      BYTE             SymLink[  128];          // 100 used for fast symlink name
   };                                           //     CAN extend to END of Inode!

   union
   {
      S_JFSEA          Eai;                     // 180 EA values      (internal)
      BYTE             EaData[ 128];            // 180 Inline EA, raw bytes
   };
} S_JINODE;                                     // end of struct "s_jinode"

#define JFS_IN_XADS                 16          //  16 XAD structures per Inode

#define JFS_SLOTSIZE                32          //  32 bytes per directory slot
#define JFS_IN_SLOTS                 8          //   8 dir slots per Inode
#define JFS_DB_SLOTS                16          //  xx dir slots per DB-sector
#define JFS_PG_SLOTS               128          // 128 dir slots per page
#define JFS_MAX_STIN               120          // maximum sort-table start
#define JFS_EMPYLIST               255          // no-list (freelist -1)

typedef struct s_dthead                         // directory Btree header
{
   ULN64               next;                    // block next page in sequence
   ULN64               prev;                    // prev page
   BYTE                rflag;                   // tree flags
   BYTE                sCount;                  // used entries in sort-table
   BYTE                fCount;                  // free count
   BYTE                fList;                   // free list header index
   BYTE                maxslot;                 // maximum nr of slots
   BYTE                stIndex;                 // slot nr start of sort-table
   USHORT              reserve1;                //     (must be < 120 decimal)
   S_PXD               self;                    // physical extent for self
   S_JDSLOT            slot[1];                 // slot array in rest of page
} S_DTHEAD;                                     // end of struct "s_dthead"

typedef union s_dtpage                          // Directory Btree page
{                                               // at maximum blocksize (4 KiB)
   S_DTHEAD            header;                  // Header, incl array start
   S_JDSLOT            slot[JFS_PG_SLOTS];      // SLOT alias, entries 2..127
} S_DTPAGE;                                     // end of struct "s_dtpage"

#define JFS_SMAPSZ                  4           // size of summary maps
#define JFS_IAGEXT                128           // nr of extents per IAG
#define JFS_IAGCPG                  8           // sectors in IAM control page
#define JFS_IAGSEC                  8           // sectors per IAG
#define JFS_INOEXT                 32           // inodes (sectors) per extent
#define JFS_IAGMAX           0x100000           // max nr of IAG's (0..0xfffff)

//- Free lists seem to be empty (0xffffffff)

//- Inode status 0 means extend allocated AND at least one of the 32 is free
//- Inode extend bitmap, 1 means the extend for 32 inodes is allocated

typedef struct s_iag                            // Inode Allocation Group
{
   ULN64               blAgStart;               // blocknr related Alloc Group
   ULONG               iagNumber;               // sequence number for IAG
   ULONG               inoFreeHead;             // Inode Free-list head
   ULONG               inoFreeTail;             // Inode Free-list tail
   ULONG               extFreeHead;             // Extent Free-list head
   ULONG               extFreeTail;             // Extent Free-list tail
   ULONG               iagFreeList;             // IAG    Free-list
   ULONG               inoSumMap[JFS_SMAPSZ];   // Inode status, 1 bit/extent
   ULONG               extBitMap[JFS_SMAPSZ];   // Extent bitmap 1 bit/extent
   ULONG               inoFreeCount;            // free Inodes  in IAG
   ULONG               extFreeCount;            // free Extents in IAG
   ULONG               reserved[494];           // filler to 2 KiB
   ULONG               inoBitMap[JFS_IAGEXT];   // Inode working bitmap
   ULONG               inoPrsMap[JFS_IAGEXT];   // Inode persistent bitmap
   S_PXD               inoExtent[JFS_IAGEXT];   // Inode extent descriptors
} S_IAG;                                        // end of struct "s_iag"


#define JFS_DMAPSZ                256           // ULONGs per block-bitmap
#define JFS_DMAPPED  (JFS_DMAPSZ * 32)          // blocks mapped per dmap

typedef struct s_dmap                           // DMAP page (bitmap, leaf)
{
   ULONG               blocks;                  // 000 blocks mapped in this dmap
   ULONG               bFree;                   // 004 blocks free in this dmap
   ULN64               blStart;                 // 008 first blocknumber mapped
   BYTE                dmapTree[360];           // 010 summary tree, to be refined
   BYTE                reserved[1672];          // 178 filler extending hdr to 2 kiB
   ULONG               blBitMap[JFS_DMAPSZ];    // 800 Block working bitmap    1 KiB
   ULONG               blPrsMap[JFS_DMAPSZ];    // C00 Block persistent bitmap 1 KiB
} S_DMAP;                                       // end of struct "s_dmap"

// Note: From disk analysis, it seems dmapTree summary is 358 bytes, and filler 1674!

typedef struct s_bmap                           // bitmap control page
{
   ULN64               blocks;                  // 000 blocks mapped, aggregate
   ULN64               bFree;                   // 008 blocks free in aggregate
   ULONG               l2bpPage;                // 010 log2 blocks-per-page (0 ?)
   ULONG               agCount;                 // 014 number of aggregate Groups
   ULONG               agActive;                // 018 number of active AGs
   ULONG               agMaxAct;                // 01c maximum AG active
   ULONG               agPreferred;             // 020 preferred AG on alloc
   ULONG               agLevel;                 // 024
   ULONG               agHeight;                // 028
   ULONG               agWidth;                 // 02c
   ULONG               agStart;                 // 030
   ULONG               l2bpAg;                  // 034 log2 blocks-per-AG
   ULN64               agFree[JFS_MAXAG];       // 038 Free blocks per AG ?
   ULN64               agSize;                  // 438 blocks in one AG
   BYTE                maxFreeBuddy;            // 440 max free in buddy system
   BYTE                reserved[3007];          // 441 fillers to 4 KiB pagesize
} S_BMAP;                                       // end of struct "s_bmap"


#endif
