//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// ISO disk structure definitions
//
// Based on various articles on the WEB, and reverse-engineering with DFSee
//
// Author: J. van Wijk
//
// JvW  10-01-2019 Initial version, derived from HFS
//
#ifndef    DFSISO_H
   #define DFSISO_H

#define ISO_STD_BLOCK_SIZE              0x800   // 2048 byte standard sector/block size
#define ISO_STD_CDROM_BLOCKS          0x57800   // 700 MiB / 737 MB standard CDROM size

                                                // 32 KiB system area, for OEM boot (MBR/GPT)
#define ISO_BOOTAREA_OFFSET            0x8000   // bytes from start to 1st Volume Descriptor

#define ISO_VOLUMEDESCSIZE ISO_STD_BLOCK_SIZE   // usually the same as logical-block-size!

#define ISO_LSNVD(vdi) ((ISO_BOOTAREA_OFFSET / ISO_STD_BLOCK_SIZE) + (vdi))
#define ISO_LSNSUP1 ((ISO_BOOTAREA_OFFSET / ISO_STD_BLOCK_SIZE) + 1)


//============================ VALUE STRUCTURES ===============================

typedef struct s_iso_lb16
{
   USHORT              le;                      // little endian value
   USHORT              be;                      // big    endian value
} S_ISO_LB16;                                   // end of struct "s_iso_lb16"

typedef struct s_iso_lb32
{
   ULONG               le;                      // little endian value
   ULONG               be;                      // big    endian value
} S_ISO_LB32;                                   // end of struct "s_iso_lb32"

// access macros, to be refined, may implement a function with le/be consistency check
#define dfsIsoLE16(v) ((v).le)
#define dfsIsoBE16(v) (TxBE16((v).be))

#define dfsIsoLE32(v) ((v).le)
#define dfsIsoBE32(v) (TxBE32((v).be))


//============================ DATE/TIME STRUCTS ==============================

#define ISO_VTIME_LEN  16

// data and time in ASCII format, as used in volume descriptors
typedef struct s_iso_vtime
{
   BYTE                Year[4];                 // year,  1..9999
   BYTE                Month[2];                // month, 1..12
   BYTE                Day[2];                  // day    1..31
   BYTE                Hour[2];                 // hour   0..23
   BYTE                Minute[2];               // minute 0..59
   BYTE                Second[2];               // second 0..59
   BYTE                Hundredths[2];           // sec100 0..99
   BYTE                GmtOffset;               // GMT offset, * 15 min EAST positive
} S_ISO_VTIME;                                  // end of struct "s_iso_vtime"

// date and time in binary format, as used in Directory entries
typedef struct s_iso_ftime
{
   BYTE                Year;                    // years since 1900
   BYTE                Month;                   // month, 1..12
   BYTE                Day;                     // day    1..31
   BYTE                Hour;                    // hour   0..23
   BYTE                Minute;                  // minute 0..59
   BYTE                Second;                  // second 0..59
   BYTE                GmtOffset;               // GMT offset, * 15 min EAST positive
} S_ISO_FTIME;                                  // end of struct "s_iso_ftime"


//============================ PATH TABLE ENTRY ===============================

// Variable length PathTable record, either LE or BE, lenght is multiple of 2
// Ascending order of directory level, alphabetically sorted withing each DIR
typedef struct s_iso_ptentry
{
   BYTE                RecordLength;            // 000 Length of Directory Record
   BYTE                ExtAttribLength;         // 001 Extended Attribute Length
   ULONG               Location;                // 002 LBA location of extent (LE or BE)
   USHORT              ParentDirNumber;         // 006 index in PT for Parent Directory
   BYTE                FileName[1];             // 008 variable length name (up to 220 or so)
} S_ISO_PTENTRY;                                // end of struct "s_iso_ptentry"

//============================ DIRECTORY ENTRY ================================

#define ISO_FLAG_HIDDEN                 0x01    // File not shown to user
#define ISO_FLAG_SUBDIR                 0x02    // entry is for a subdirectory
#define ISO_FLAG_ASSO_F                 0x04    // entry for 'associated file'
#define ISO_FLAG_EXATTR                 0x08    // Ext Attr has file format info
#define ISO_FLAG_OWNGRP                 0x10    // Ext Attr has owner/group info
#define ISO_FLAG_NEXT_R                 0x80    // More dir-records follow for this file
                                                // (used for large multi-extent files)

// Directory entry, sized for the ROOT entry as present in VD, name is variable
// Record size is always a multiple of 2, padded with a zero when needed
typedef struct s_iso_direntry
{
   BYTE                RecordLength;            // 000 Length of Directory Record
   BYTE                ExtAttribLength;         // 001 Extended Attribute Length (RR ?)
   S_ISO_LB32          Location;                // 002 LBA location of the extent
   S_ISO_LB32          FileSize;                // 00A Data length of the extent
   S_ISO_FTIME         FileDateTime;            // 012 Interleave Gap Size, or 0
   BYTE                FileFlags;               // 019 File Flags, see defines above
   BYTE                FileUnitSize;            // 01A Interleave Unit Size or 0
   BYTE                InterleaveGap;           // 01B Interleave Gap Size, or 0
   S_ISO_LB16          VolSequenceNr;           // 01C Volume extent is recorded on
   BYTE                NameLength;              // 020 length of filename   (ending in  ';1')
   BYTE                FileName[1];             // 021 variable length name (up to 220 or so)

   //- remainder used for SUSP extensions, like Rock Ridge interchange Protocol (up to 255)

} S_ISO_DIRENTRY;                               // end of struct "s_iso_direntry"


//============================ VOLUME DESCRIPTORS =============================

#define ISO_VDT_BOOT_RECORD             0x00    // Boot record  Volume Descriptor (like El-Torito)
#define ISO_VDT_PRIMARY                 0x01    // Primary      Volume descriptor (often 8.1)
#define ISO_VDT_SUPPLEMENTAL            0x02    // Supplemental Volume descriptor (often Joliet)
#define ISO_VDT_PARTITION               0x03    // Partition    Volume descriptor (seldom used)
#define ISO_VDT_TERMINATOR              0xff    // Terminating  Volume descriptor (always last)

#define ISO_VD_MAX                        16    // sane maximum for numver of VD's

#define ISO_SIDLEN                        32    // length of small identifier strings
#define ISO_LIDLEN                       128    // length of large identifier strings

#define ISO_VD_FLAG_ESCSEQ              0x01    // Not all escape sequences conform ISO2375

typedef struct s_iso_vd_bootrec
{
   BYTE                BtSysId[ ISO_SIDLEN];    // 007 System name for booting, e.g. Linux
   BYTE                BootId[  ISO_SIDLEN];    // 027 Id of boot system used,  e.g. El Torito
   ULONG               ElToritoBcBlock;         // 047 El Torito boot catalog block number
   BYTE                BootSystemUse[  1973];   // 04B Custom boot usage
} S_ISO_VD_BOOTREC;                             // end of struct "s_iso_vd_bootrec"

typedef struct s_iso_vd_primsup
{
   BYTE                VolumeFlags;             // 007 SUPPLEMENTAL: Volume Flags
   BYTE                SystemId[ ISO_SIDLEN];   // 008 System name that may use sects 0..0xF
   BYTE                VolumeId[ ISO_SIDLEN];   // 028 Identification of this Volume (label)
   BYTE                ResZero1[  8];           // 048
   S_ISO_LB32          LogicalBlocks;           // 050 Number of logical blocks in volume
   BYTE                EscapeSeq[ 32];          // 058 SUPPLEMENTAL: Escape seq (%\ indicates Unicode)
   S_ISO_LB16          VolSetSize;              // 078 Sets in volume, disks
   S_ISO_LB16          VolSequenceNr;           // 07C Nr of this disk in the set
   S_ISO_LB16          LogBlockSize;            // 080 Block size in bytes (2048)
   S_ISO_LB32          PathTableSize;           // 084 Path Table Size in bytes
   ULONG               LePathTableLba;          // 08C Location primary Path Table, Little Endian
   ULONG               LeAltPtableLba;          // 090 Location altern. Path Table, Little Endian
   ULONG               BePathTableLba;          // 094 Location primary Path Table, Big    Endian
   ULONG               BeAltPtableLba;          // 098 Location altern. Path Table, Big    Endian
   S_ISO_DIRENTRY      RootDirEntry;            // 09C Directory entry for the ROOT directory
   BYTE                VolumeSet[ ISO_LIDLEN];  // 0BE Identifier for the whole volume set
   BYTE                Publisher[ ISO_LIDLEN];  // 13E Id for publisher; can be 0x5f + root fname
   BYTE                Preparer[  ISO_LIDLEN];  // 1BE Id for preparer ; can be 0x5f + root fname
   BYTE                Applicat[  ISO_LIDLEN];  // 23E Id for application (CDRECORD) 0x5f + fname
   BYTE                Copyright[ 38];          // 2BE ROOT filename for copyright information
   BYTE                Abstract[  36];          // 2E4 ROOT filename for abstract file info
   BYTE                BiblioGrp[ 37];          // 308 ROOT filename for bibliographic info
   S_ISO_VTIME         VdtCreation;             // 32D Volume date/time Creation
   S_ISO_VTIME         VdtModification;         // 33E Volume date/time Modification
   S_ISO_VTIME         VdtExpiration;           // 34F Volume date/time Expiration
   S_ISO_VTIME         VdtEffective;            // 360 Volume date/time Effective
   BYTE                FileStructVersion;       // 371 Dir-record and Path-table version (always 0x01)
   BYTE                ResZero3;                // 372 Unused                            (always 0x00)
   BYTE                ApplicationUsed[ 512];   // 373 Not defined by ISO 9660
   BYTE                Reserved[ 653];          // 573 Reserved by ISO 9660
} S_ISO_VD_PRIMSUP;                             // 800 size; end of struct "s_iso_vd_primsup"


// ISO9660 signature string 'CD001'
#define SG_ISO_SIG      5
#define SV_ISO_SIG      (char)0x43,(char)0x44,(char)0x30,(char)0x30,(char)0x31

typedef struct s_iso_voldesc
{
   BYTE                vdType;                  // Volume descriptor type
   BYTE                Signature[ 5];           // Signature CD001 or CDROM
   BYTE                vdVersion;               // Version value, usually 0x01
   union
   {
      S_ISO_VD_BOOTREC b;
      S_ISO_VD_PRIMSUP p;
   };                                           // end of union ""
} S_ISO_VOLUMEDESC;


//============================ SUSP / Rock Ridge ==============================


// Signature values defined as little endian two-byte values (USHORT)
#define RRP_SUSPI_CHECKSIG            0xefbe    // 'SP' check value 'be ef'
#define RRP_SUSPI_DATA_SIG            0x5053    // 'SP' SUSP Indicator record
#define RRP_CONTA_DATA_SIG            0x5245    // 'ER' SUSP Extensions Reference
#define RRP_CONTR_DATA_SIG            0x4543    // 'CE' SUSP Continuation area
                                                //      points to block with 'ER'

#define RRP_ROCK_RIDGE_SIG            0x5252    // 'RR' Rock Ridge SUSP signature
#define RRP_POSIX_DATA_SIG            0x5850    // 'PX' Rock Ridge Posix data sig
#define RRP_TIMES_DATA_SIG            0x4654    // 'TF' Rock Ridge Date/Time  sig
#define RRP_ANAME_DATA_SIG            0x4d4e    // 'NM' Rock Ridge Alt-Fname  sig


// Primary SUSP record structure, located directly after regular Directory entry
typedef struct srr_susp_record
{
   USHORT              Signature;               // SUSP data signature, like 'RR'
   BYTE                Length;                  // Length of SUSP record
   BYTE                Version;                 // Version number SUSP, usually 1
   BYTE                Data[1];                 // Variable size SUSP data area
} SRR_SUSP_RECORD;                              // end of struct "srr_susp_record"

#define RRP_TFFLG_CREATION              0x01    // Creation       time present
#define RRP_TFFLG_MODIFY                0x02    // Modify         time present
#define RRP_TFFLG_ACCES                 0x04    // Acces          time present
#define RRP_TFFLG_ATTRIBUTES            0x08    // Attributes     time present
#define RRP_TFFLG_BACKUP                0x10    // Backup         time present
#define RRP_TFFLG_EXPIRATION            0x20    // Expiration     time present
#define RRP_TFFLG_EFFECTIVE             0x40    // Effective      time present
#define RRP_TFFLG_LONG_FORM             0x80    // 17-byte format used, 7 otherwise

typedef struct ssr_ce_record                    // continuation area reference
{
   USHORT              Signature;               // SUSP data signature, value 'CE'
   BYTE                Length;                  // Length of CE record, 0x1C = 28
   BYTE                Version;                 // Verison number SUSP, usually 1
   S_ISO_LB32          Location;                // Location of block with Cont-area
   S_ISO_LB32          Offset;                  // Offset in that block for Cont-area
   S_ISO_LB32          AreaLength;              // Length of Continuation area
} SSR_CE_RECORD;                                // end of struct "ssr_ce_record"

typedef struct ssr_tf_record                    // Alternate name structure record
{
   USHORT              Signature;               // SUSP data signature, value 'TF'
   BYTE                Length;                  // Length of TF record, variable
   BYTE                Version;                 // Version number TF    usually 1
   BYTE                TfFlags;                 // Flags indicating times present
   BYTE                Data[1];                 // Variable size date values
} SSR_TF_RECORD;                                // end of struct "ssr_nm_record"

typedef struct ssr_px_record                    // Possix attribute/owner record
{
   USHORT              Signature;               // SUSP data signature, value 'PX'
   BYTE                Length;                  // Length of TF record, 0x24 = 36
   BYTE                Version;                 // Version number TF    usually 1
   S_ISO_LB32          FileMode;                // FileMode bits, max 16 bits used
   S_ISO_LB32          Links;                   // File Links count (not shown here)
   S_ISO_LB32          UserId;                  // Owner user-id, max  6 digits shown
   S_ISO_LB32          GroupId;                 // Group user-id, max  6 digits shown
} SSR_PX_RECORD;                                // end of struct "ssr_px_record"

#define RRP_NMFLG_CONTINUE              0x01    // name continues in next 'NM' record
#define RRP_NMFLG_CURRENT               0x02    // name refers to the current DIR '.'
#define RRP_NMFLG_PARENT                0x04    // name refers to the parent DIR '..'

typedef struct ssr_nm_record                    // Alternate name structure record
{
   USHORT              Signature;               // SUSP data signature, value 'NM'
   BYTE                Length;                  // Length of NM record, variable
   BYTE                Version;                 // Version number NM    usually 1
   BYTE                NmFlags;                 // Flags detailing name properties
   BYTE                AltName[1];              // Alternate filename (Ascii or Unicode!)
} SSR_NM_RECORD;                                // end of struct "ssr_nm_record"


typedef struct ssr_sp_record                    // SUSP Indicator record (usually on '.')
{
   USHORT              Signature;               // SUSP data signature, value 'CE'
   BYTE                Length;                  // Length of CE record, 0x07 =  7
   BYTE                Version;                 // Verison number SUSP, usually 1
   USHORT              SuspSig;                 // Checkvalue 'be ef'
   BYTE                LenSkip;                 // Extra bytes between name and SUSP
} SSR_SP_RECORD;                                // end of struct "ssr_sp_record"

// Continuation area, located at offset Y in another block X (see 'CE' record)
typedef struct srr_cont_record
{
   USHORT              Signature;               // SUSP data signature, value 'ER'
   BYTE                Length;                  // Length of CONT record
   BYTE                Version;                 // Version number SUSP, must be 1
   BYTE                IdLength;                // Length of ID name string
   BYTE                DescLength;              // Length of Description string
   BYTE                SrcLength;               // Length of extension-source string
   BYTE                ExtVersion;              // Extension version number
   BYTE                Data[1];                 // Variable size ID/DESC/SRC data area
} SRR_CONT_RECORD;                              // end of struct "srr_cont_record"


//============================ EL TORITO HEADER ===============================

#define ELT_ARCH_80x86                  0x00    // Boot architecture x8086
#define ELT_ARCH_POWER                  0x01    // Boot architecture PowerPC
#define ELT_ARCH_MACOS                  0x02    // Boot architecture MAC 68000/PPC
#define ELT_ARCH_UEFI                   0xef    // Boot architecture UEFI

#define ELT_HEADER_ID                   0x01    // El Torito header ID, always 1
#define ELT_SIGNATURE                 0xAA55    // validation header signature

#define ELT_NONBOOT                     0x00    // non-bootable indication for entry
#define ELT_BOOTABLE                    0x88    // bootable indication for entry

#define ELT_MEDIA_NO_EMUL               0x00    // no-emulation mode
#define ELT_MEDIA_DSK_120               0x01    // emulation of 1.2  MB diskette
#define ELT_MEDIA_DSK_144               0x02    // emulation of 1.44 MB diskette
#define ELT_MEDIA_DSK_284               0x03    // emulation of 2.88 MB diskette
#define ELT_MEDIA_HARDISK               0x04    // emulation of a harddisk

typedef struct s_iso_el_torito
{
   BYTE                hdrId;                   // 000 Header identification, always 0x01
   BYTE                archId;                  // 001 Architectur ID, see above
   USHORT              resZero2;                // 002 reserved, ZERO
   BYTE                devId[24];               // 004 Developer/Manufacturer ID (empty!)
   USHORT              checksum;                // 01C checksum over header
   USHORT              Signature;               // 01E signature, little endian '55' 'aa'

   BYTE                bootIndicator;           // 020 Bootable indicator
   BYTE                media;                   // 021 Media indicator, usually 0
   USHORT              loadSeg;                 // 022 load segment (0x00 -> 0x07C0)
   BYTE                partType;                // 024 for media 0x04, partition table type
   BYTE                resZero1;                // 025 reserved, ZERO
   USHORT              sectCount;               // 026 sector count (512 byte sectors!)
   ULONG               loadBlock;               // 028 ISO9660 block with boot image code
} S_ISO_EL_TORITO;                              // end of struct "s_iso_el_torito"

#endif
