//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFS IMZ image data cache and indexing
//
// Implements creating and using data structures to access DFSee compressed
// image files (IMZ) with an INDEX that maps LSN-areas to file-offsets and
// and a data-cache that holds one or more recently accessed areas.
//
// The INDEX, cache and read/write functions support multiple file images
// transparently. The initial IMZ is opened outside the module, while the
// follow-ups (.I02, I03 etc) will be opened/closed on demand
//
// Author: J. van Wijk
//
// JvW  31-10-2016   Initial version, derived from DFSMDISK
// JvW  02-12-2016   Add persistent INDEX (.XMZ) to save time on large images
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsutil.h>                            // DFS utility functions
#include <dfsimage.h>                           // DFS image functions

#include <dfsimzci.h>                           // DFS IMZ cache/index functions

#define IMZCI_MAGIC            0xdf5face5       // magic identifier for imzci

#define IMZCI_CACHESIZE                 9       // number of cache buffers

#define IMZCI_INDEX_EXT        "xmz"            // file extension index file
#define IMZCI_INDEX_SIG        0x497A6D49       // Sig ImzI at start of file (old format)
#define IMZCI_SINGL_SIG        0x537A6D49       // Sig ImzS at start of file (single file)
#define IMZCI_MULTI_SIG        0x4D7A6D49       // Sig ImzM at start of file (multi  file)

//- Compression unit types SMARTBUF and RLEBYTE are used when the WHOLE buffer
//- contains the SAME bytevalue. For SMARTBUF that is always 0x00, and for the
//- RLE one it is the 'data' part of the typecode: (cuType & DFSIM_RLEDATA)


//- Obsolete index structure took up 12 bytes per compression-unit, used before 14.5
//- This limits practical use to objects not larger than 2TB (depending on memory),
//- since the index would take up more than 3 gigabytes in memory for a 4TB object.
//- Ultimate solution is to make the compression-units much larger, like 1..2 MiB


//- Direct index structure takes up 8 bytes per compression-unit, used for Multi-file
//- Limits use to about 3TB objects, unless large buffers are used (up to 2MiB)
typedef struct imzdirect
{
   USHORT              filenr;                  // file nr, 1==IMZ 2=I02 etc
   USHORT              cuType;                  // compression unit type
   ULONG               offset;                  // offset in compressed file
} IMZDIRECT;                                    // end of struct "imzdirect

// New offset-group index, with a single 64-bit file-offset for every 'gsize'
// compression units, and and additional 24-bit offset + 8-bit type per unit
// Requires 5 bytes per compression unit, allowing indexing upto at least 5TB
// unless large buffers are used (up to 2 MiB), used for Single file
#define IMZCI_GSIZE         8
typedef struct imzogroup
{
   ULN64               offset;                  // offset in compressed file
   ULONG               subUnit[ IMZCI_GSIZE];   // sub-unit type + offset bits
} IMZOGROUP;                                    // end of struct "imzogroup"


typedef struct imzcachedata                     // cache data element
{
   TXTIMER             lastUse;                 // last use timestamp or 0
   ULONG               ix;
   BYTE               *Buffer;                  // cache data buffer
} IMZCACHEDATA;                                 // end of struct "imzcachedata"


typedef struct imzciroot                        // ROOT info for IMZ cache/index
{
   ULONG               magic;                   // magic signature
   ULN64               start;                   // first sector in IMZ, offset
   ULN64               sectors;                 // nr of sectors in IMZ, size
   ULONG               bpsector;                // bytes per buffer sector
   ULONG               bufSize;                 // number of sectors per buffer
   ULONG               files;                   // filecount for multiple-file
   ULONG               lastIx;                  // last unit read (sequential test)
   ULONG               cacheCount;              // number of cachedata elements
   IMZCACHEDATA       *cache;                   // allocated array of cache elements
   ULONG               ixUsed;                  // number of index elements (compression units)
   ULONG               indexCount;              // number of index elements (direct or group)
   size_t              indexSize;               // size per index element in bytes
   void               *index;                   // allocated array of index elements
   BYTE               *readBuf;                 // Read buffer, compressed data
   DFSUL_HANDLE        ulzwHandle;              // ULZW compression session handle
   USHORT              curFileNr;               // Number of CURRENT file (1 = IMZ)
   TXHFILE             curHandle;               // Handle to CURRENT file (nnn/IMZ)
   TXHFILE             imzHandle;               // R/W filehandle first file  (IMZ)
   TXLN                imzName;                 // path+filename  first file  (IMZ)
} IMZCIROOT;                                    // end of struct "imzciroot"


// Create indexes for compressed image file, either the IMZ or subsequent .nnn
static ULONG dfsImzIndexSingleFile
(
   int                 filenr,                  // IN    sequence number for file
   TXHFILE             fh,                      // IN    handle to opened IMZ/Inn
   BOOL                verbose,                 // IN    show info to screen/log
   ULN64              *filepos,                 // INOUT imagefile size read
   IMZCIROOT          *pImz                     // INOUT IMZ info, root structure
);

// Get a (pointer to) a cache structure to be used with this Compression Unit
static IMZCACHEDATA *dfsImzGetCache
(
   ULONG               ix,                      // IN    Compression Unit index
   IMZCIROOT          *ir                       // INOUT IMZ info, root structure
);

// Read and decompress compression-unit from IMZ file into a cache element
static ULONG dfsImzFile2Cache
(
   ULONG               ix,                      // IN    Compression Unit index
   IMZCACHEDATA       *cache,                   // INOUT cache structure to use
   IMZCIROOT          *ir                       // INOUT IMZ info, root structure
);


/*****************************************************************************/
// Generate an index file for the specified IMZ imagefile
/*****************************************************************************/
ULONG dfsImzGenerateIndex
(
   char               *fname,                   // IN    path+name of the IMZ
   BOOL                verbose,                 // IN    show index file written
   BOOL                force                    // IN    delete existing index
)
{
   ULONG               rc = NO_ERROR;
   TXLN                indexName;
   TXHFILE             imzHandle;
   void               *index = NULL;

   ENTER();

   if (TxFileOpenReadOnly( fname, &imzHandle) == NO_ERROR)
   {
      strcpy( indexName, fname);
      TxStripExtension( indexName);             // remove the file extension
      TxFnameExtension( indexName, IMZCI_INDEX_EXT);

      if (force)
      {
         TxDeleteFile(  indexName);             // delete existing one
      }
      if ((dfsImzInitCacheIndex( imzHandle, fname, verbose, &index)) == NO_ERROR)
      {
         dfsImzFreeCacheIndex( &index);
      }
      TxClose( imzHandle);
   }
   RETURN ( rc);
}                                               // end 'dfsImzGenerateIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize Cache and Indexing structures to access (opened) IMZ image
/*****************************************************************************/
ULONG dfsImzInitCacheIndex
(
   TXHFILE             fh,                      // IN    handle to opened IMZ
   char               *fname,                   // IN    path+name, for .002 etc
   BOOL                verbose,                 // IN    show info to screen/log
   void              **ppImz                    // OUT   anonymised access data
)
{
   ULONG               rc = NO_ERROR;
   IMZCIROOT          *ir = NULL;               // IMZ info, root structure
   int                 i;                       // cache iterator
   ULONG               bufferBytes;
   ULN64               fileBytes;               // total IMZ size in bytes
   TXLN                subFilename;
   TXHFILE             subFh;
   ULONG               cSignature;              // signature, old or new format

   ENTER();

   if ((ir = TxAlloc( 1, sizeof( IMZCIROOT))) != NULL)
   {
      ir->magic = IMZCI_MAGIC;                  // in-memory consistency check

      if (dfsGetImzImageInfo( fname, fh, TRUE, &fileBytes, &ir->files, &ir->start,
                              &ir->sectors,  &bufferBytes, &ir->bpsector, NULL))
      {
         ULONG         unitCount;               // number of compression units

         ir->bufSize    = bufferBytes / ir->bpsector;
         ir->cacheCount = IMZCI_CACHESIZE;
         ir->ixUsed     = 0;

         unitCount = (ULONG) ((ir->sectors - 1) / ir->bufSize) + 1;

         if (ir->files > 1)                     // old format, allocate direct indexes
         {
            cSignature     = IMZCI_MULTI_SIG;
            ir->indexSize  = sizeof( IMZDIRECT);
            ir->indexCount = unitCount;         // one unit per index element
         }
         else                                   // new format, allocate index offset-groups
         {
            cSignature     = IMZCI_SINGL_SIG;
            ir->indexSize  = sizeof( IMZOGROUP);
            ir->indexCount = ((unitCount - 1) / IMZCI_GSIZE) + 1; // GSIZE units per index element
         }
         if (verbose)
         {
            dfsSz64Bps( "IMZ expanded size : ", ir->sectors, ir->bpsector, "");
            TxPrint(    " for '%s'\n", fname);
         }

         if (((ir->readBuf = TxAlloc( ir->bufSize,    ir->bpsector         )) != NULL) &&
             ((ir->index   = TxAlloc( ir->indexCount, ir->indexSize        )) != NULL) &&
             ((ir->cache   = TxAlloc( ir->cacheCount, sizeof( IMZCACHEDATA))) != NULL)  )
         {
            for (i = 0; i < ir->cacheCount; i++)
            {
               if ((ir->cache[ i].Buffer = TxAlloc( ir->bufSize, ir->bpsector)) == NULL)
               {
                  rc = DFS_ALLOC_ERROR;
                  break;
               }
            }
            if (rc == NO_ERROR)
            {
               time_t   cModifTime;
               ULONG    handled;

               TxGetFileTime( fname, NULL, NULL, &cModifTime); // date/time of IMZ file

               strcpy( ir->imzName, fname);     // original path+filename IMZ
               TxStripExtension( ir->imzName);  // remove the file extension
               ir->imzHandle = fh;              // the base IMZ handle
               ir->curHandle = fh;              // make it CURRENT too
               ir->curFileNr = 1;               // by definition

               sprintf( subFilename, "%s.%s", ir->imzName, IMZCI_INDEX_EXT);
               if (TxFileOpenReadOnly( subFilename, &subFh) == NO_ERROR) // index file present
               {
                  ULONG    rSignature = 0;      // values read from the index file
                  time_t   rModifTime = 0;
                  ULN64    rFileBytes = 0;

                  if ((TxRead( subFh, &rSignature, sizeof(ULONG),  &handled) == NO_ERROR) &&
                      (TxRead( subFh, &rModifTime, sizeof(time_t), &handled) == NO_ERROR) &&
                      (TxRead( subFh, &rFileBytes, sizeof(ULN64),  &handled) == NO_ERROR)  )
                  {
                     if ((rSignature == cSignature) && // on sig mismatch (old index) discard and regenerate
                         (rModifTime == cModifTime) &&
                         (rFileBytes ==  fileBytes)  ) // right type of index, with correct size/time info
                     {
                        if (TxRead( subFh, ir->index, ir->indexCount * ir->indexSize, &handled) == NO_ERROR)
                        {
                           if (verbose)
                           {
                              TxPrint( "Read  image index : %s%s%s\n", CBY, subFilename, CNN);
                           }
                           ir->ixUsed = unitCount; // actual compression units indexed
                        }
                     }
                  }
                  TxClose( subFh);
               }

               if (ir->ixUsed == 0)             // INDEX not there, build and write it
               {
                  ULN64   fsDone = 0;           // filesize done (progress)

                  dfsProgressInit( 0, fileBytes / dfsGetSectorSize(), 0, "Sector:", "indexed", DFSP_BARS, 0);

                  rc = dfsImzIndexSingleFile( 1, fh, verbose, &fsDone, ir);          //- base IMZ
                  for (i = 2; (i <= ir->files) && (rc == NO_ERROR); i++)
                  {
                     sprintf( subFilename, "%s.%3.3d", ir->imzName, i);
                     if ((rc = TxFileOpenReadOnly( subFilename, &subFh)) == NO_ERROR)
                     {
                        rc = dfsImzIndexSingleFile( i, subFh, verbose, &fsDone, ir); //- subsequent .nnn
                        TxClose( subFh);
                     }
                  }
                  dfsProgressTerm();            // allow regular screen output
                  TRACES(("Filled %lu indexes from total: %lu in %lu records\n", ir->ixUsed, unitCount, ir->indexCount));

                  //- write the generated INDEX to a file, to save time the next time it is opened
                  sprintf( subFilename, "%s.%s", ir->imzName, IMZCI_INDEX_EXT);
                  if (TxFileOpenForWrite( subFilename, FALSE, &subFh) == NO_ERROR) // recreate index file
                  {
                     if ((TxWrite( subFh, &cSignature, sizeof(ULONG),  &handled) == NO_ERROR) &&
                         (TxWrite( subFh, &cModifTime, sizeof(time_t), &handled) == NO_ERROR) &&
                         (TxWrite( subFh, &fileBytes,  sizeof(ULN64),  &handled) == NO_ERROR) &&
                         (TxWrite( subFh, ir->index, ir->indexCount * ir->indexSize, &handled) == NO_ERROR))
                     {
                        if (verbose)
                        {
                           TxPrint( "Write image index : %s%s%s\n", CBY, subFilename, CNN);
                        }
                        TRACES(("INDEX '%s' written\n", subFilename));
                     }
                     else
                     {
                        TxFileSeek( subFh, 0, SEEK_SET); // rewind to start, truncate file to ZERO bytes
                     }
                     TxClose( subFh);
                  }
               }
               if (rc == NO_ERROR)
               {
                  rc = dfsUlRegister( &ir->ulzwHandle); // start compression session
               }
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         rc = DFS_ST_MISMATCH;                  // unexpected, should be IMZ
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *ppImz = ir;

   if (rc != NO_ERROR)                          // free (partly) allocated memory
   {
      dfsImzFreeCacheIndex( ppImz);
   }
   RETURN ( rc);
}                                               // end 'dfsImzInitCacheIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free Cache and Indexing structures associated with an IMZ image
/*****************************************************************************/
ULONG dfsImzFreeCacheIndex
(
   void              **pImz                     // INOUT anonymised access data
)
{
   ULONG               rc = NO_ERROR;
   IMZCIROOT          *ir = (IMZCIROOT *) *pImz; // IMZ info, root structure
   int                 i;

   ENTER();
   TRACES(("info-root: 0x%8.8lx\n", ir));

   if ((ir != NULL) && (ir->magic == IMZCI_MAGIC))
   {
      if (ir->curFileNr > 1)                    // close existing .nnn file
      {
         TxClose( ir->curHandle);
      }
      dfsUlTerminate( ir->ulzwHandle);          // end compression session
      if (ir->cache != NULL)                    // if there IS an allocated cache
      {
         for (i = 0; i < ir->cacheCount; i++)
         {
            TRACES(("Free cache buffer: %d at 0x%8.8lx\n", i, ir->cache[ i].Buffer));
            TxFreeMem( ir->cache[ i].Buffer);
         }
         TRACES(("Free cache at: 0x%8.8lx\n", ir->cache));
         TxFreeMem( ir->cache);
      }
      TRACES(("Free index at: 0x%8.8lx\n", ir->index));
      TxFreeMem( ir->index);
      TRACES(("Free rdBuf at: 0x%8.8lx\n", ir->readBuf));
      TxFreeMem( ir->readBuf);
      TRACES(("Free root* at: 0x%8.8lx\n", *pImz));
      TxFreeMem( *pImz);                        // free ROOT structure, zero ref
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN ( rc);
}                                               // end 'dfsImzFreeCacheIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read ONE sector from an IMZ image file, using supplied access data
// Reading a single sector avoids having to 'map' sector-ranges on one or more
// of the possibly cached IMZ areas, at the cost of a little more overhead
/*****************************************************************************/
ULONG dfsImzReadSector
(
   ULN64               sn,                      // IN    wanted sector number
   ULONG               bps,                     // IN    bytes per sector (buffer)
   void               *pImz,                    // INOUT anonymised access data
   BYTE               *data                     // OUT   sector data
)
{
   ULONG               rc = NO_ERROR;
   IMZCIROOT          *ir = (IMZCIROOT *) pImz; // IMZ info, root structure
   ULONG               ix;                      // Compression Unit index
   ULONG               rsn;                     // Relative sector in Compression Unit
   ULONG               ixGroup;                 // index offset group number (new format)
   ULONG               ixSub;                   // sub-index in offset group
   USHORT              cuType;                  // Type of record from index
   IMZCACHEDATA       *cache;                   // pointer to cache to be used
   BYTE                databyte;                // single data byte

   ENTER();
   TRACES(("pImz:%8.8lx sn: 0x%llx  bps:%lu\n", pImz, sn, bps));

   if (ir && (ir->magic == IMZCI_MAGIC) && (bps == ir->bpsector))
   {
      TRACES(("ir: bufSize:%lu  start:0x%llx sectors:0x%llx\n", ir->bufSize, ir->start, ir->sectors));
      if ((ir->bufSize) && (sn >= ir->start) && (sn < (ir->start + ir->sectors)))
      {
         ix     = (ULONG) (sn - ir->start) / ir->bufSize;
         rsn    = (ULONG) (sn - ir->start) % ir->bufSize;

         if (ir->files > 1)                     // multiple file, direct-index format
         {
            IMZDIRECT    *directIndex = (IMZDIRECT *) ir->index;

            cuType  = directIndex[ ix].cuType;
         }
         else                                   // single file, new compact format
         {
            IMZOGROUP    *ogroupIndex = (IMZOGROUP *) ir->index;

            ixGroup = ix / IMZCI_GSIZE;
            ixSub   = ix % IMZCI_GSIZE;

            //- Get compression-type, with possible RLE/Smart subvalue
            cuType  = (ogroupIndex[ ixGroup].subUnit[ ixSub] & 0xffff0000) >> 16;
            if (cuType < DFSIM_SMARTBUF)        // non-Smart/RLE type of compression
            {
               cuType >>= 8;                    // shift compression type in lowest byte
            }                                   // with upper byte being ZERO
         }
         TRACES(("CU nr: 0x%8.8lx  rsn:0x0%lx  for type %s\n", ix, rsn, dfsImageCode2String( cuType)));

         if (cuType < DFSIM_SMARTBUF)           // NONE or LZW compressed, access cache
         {
            cache = dfsImzGetCache( ix, ir);
            if (cache->lastUse == 0)            // no valid contents, need to read
            {
               rc = dfsImzFile2Cache( ix, cache, ir);
            }

            if (rc == NO_ERROR)
            {
               //- data is in the cache now, copy the single specified sector
               memcpy( data, cache->Buffer + (rsn * ir->bpsector), ir->bpsector);
               cache->lastUse = TxTmrGetNanoSecFromStart(); // update LastUse timestamp
            }
         }
         else                                   // RLE and SMART compressed
         {
            switch (cuType & DFSIM_RLEBYTE)
            {
               case DFSIM_SMART:                // 0xEE top byte, various SMART variants
                  databyte = 0x00;
                  break;

               default:                         // other, RLE variants, use data byte
                  databyte = cuType & DFSIM_RLEDATA;
                  break;
            }
            memset( data, databyte, bps);       // set whole buffer to same databyte
         }
      }
      else
      {
         rc = DFS_PSN_LIMIT;                    // outside of the IMZ stored range
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN ( rc);
}                                               // end 'dfsImzReadSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get a (pointer to) a cache structure to be used with this Compression Unit
/*****************************************************************************/
static IMZCACHEDATA *dfsImzGetCache
(
   ULONG               ix,                      // IN    Compression Unit index
   IMZCIROOT          *ir                       // INOUT IMZ info, root structure
)
{
   IMZCACHEDATA       *rc = NULL;
   int                 i;

   //- First, see if there is a cache-element for this unit already
   for (i = 0; i < ir->cacheCount; i++)
   {
      if ((ir->cache[ i].ix == ix) && (ir->cache[ i].lastUse != 0))
      {
         TRACES(("ImZGetCache: CU already cached in nr: %d\n", i));
         rc = &(ir->cache[ i]);                 // same unit already there
         break;
      }
   }
   if (rc == NULL)                              // not present yet
   {
      //- Second, see if this unit is a sequential-read for a cache-element
      for (i = 0; i < ir->cacheCount; i++)
      {
         if ((ir->cache[ i].ix + 1 == ix) && (ir->cache[ i].lastUse != 0))
         {
            TRACES(("ImZGetCache: CU is sequential read against nr: %d\n", i));
            rc = &(ir->cache[ i]);              // unit is NEXT one for this element
            rc->lastUse = 0;                    // LastUse 0 indicates invalid data
            rc->ix      = ix;                   // set it to this next ix value
            break;
         }
      }
   }
   if (rc == NULL)                              // find a free/discardable one
   {
      TXTIMER  oldest  = TXTMR_MAX;

      TRACES(("ImZGetCache: LRU candidates: "));
      for (i = 0; i < ir->cacheCount; i++)
      {
         TXTIMER  lastUse = ir->cache[ i].lastUse;

         if      (lastUse == 0)                 // Free, use this
         {
            TRINTF(("=> Free nr: %d", i));
            rc = &(ir->cache[ i]);
            break;
         }
         else if (lastUse < oldest)             // oldest sofar
         {
            oldest = lastUse;
            TRINTF((" %d ", i));
            rc = &(ir->cache[ i]);
         }
      }
      TRINTF(("\n"));
      if (rc != NULL)
      {
         rc->lastUse = 0;                       // LastUse 0 indicates invalid data
         rc->ix      = ix;                      // set it to this ix value
      }
   }
   return ( rc);
}                                               // end 'dfsImzGetCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read and decompress compression-unit from IMZ file into a cache element
/*****************************************************************************/
static ULONG dfsImzFile2Cache
(
   ULONG               ix,                      // IN    Compression Unit index
   IMZCACHEDATA       *cache,                   // INOUT cache structure to use
   IMZCIROOT          *ir                       // INOUT IMZ info, root structure
)
{
   ULONG               rc = NO_ERROR;
   ULONG               ixGroup;                 // index offset group number (new format)
   ULONG               ixSub;                   // sub-index in offset group
   USHORT              cuType;                  // Type of record from index
   USHORT              filenr;
   TXLN                subFilename;
   ULN64               unitOffset;              // calculated unit offset in file

   ENTER();

   if (ir->files > 1)                           // multiple file, direct-index format
   {
      IMZDIRECT    *directIndex = (IMZDIRECT *) ir->index;

      cuType     = directIndex[ ix].cuType;
      filenr     = directIndex[ ix].filenr;
      unitOffset = directIndex[ ix].offset;
   }
   else                                         // single file, new compact format
   {
      IMZOGROUP    *ogroupIndex = (IMZOGROUP *) ir->index;

      ixGroup    = ix / IMZCI_GSIZE;
      ixSub      = ix % IMZCI_GSIZE;

      //- Get compression-type, with possible RLE/Smart subvalue
      cuType  = (ogroupIndex[ ixGroup].subUnit[ ixSub] & 0xffff0000) >> 16;
      if (cuType < DFSIM_SMARTBUF)              // non-Smart/RLE type of compression
      {
         cuType >>= 8;                          // shift compression type in lowest byte
      }
      filenr     = 1;                           // fixed, single file .IMZ
      unitOffset = ogroupIndex[ ixGroup].offset +
                  (ogroupIndex[ ixGroup].subUnit[ ixSub] & 0x00ffffff);
   }
   TRACES(("Read data for C-unit#: 0x%8.8lx  in file#: %lu  offset: 0x%11.11llx  cuType: %s\n",
                                   ix, filenr, unitOffset, dfsImageCode2String( cuType)));

   if (filenr != ir->curFileNr)                 // other file needed
   {
      if (ir->curFileNr > 1)                    // close existing .nnn
      {
         TxClose( ir->curHandle);
      }
      if (filenr == 1)                          // it is the base IMZ
      {
         ir->curHandle = ir->imzHandle;
      }
      else
      {
         sprintf( subFilename, "%s.%3.3hu", ir->imzName, filenr);
         rc = TxFileOpenReadOnly( subFilename, &ir->curHandle);
      }
      ir->curFileNr = filenr;
   }
   if (rc == NO_ERROR)                          // file is open, read ...
   {
      ULONG            handled;                 // nr of bytes handled
      USHORT           ssize;                   // 16-bit size value
      ULONG            rlen;                    // data/total-record length
      ULONG            bufferBytes;             // size of buffer, in bytes
      ULONG            lenSize;                 // length of size field 2 or 4
      ULONG            crc32    = 0;
      BOOL             usecrc32 = FALSE;        // CRC32 value present ?

      TxFileSeek( ir->curHandle, unitOffset + sizeof( USHORT), SEEK_SET);

      bufferBytes = ir->bufSize * ir->bpsector;

      if (bufferBytes < 65536)                  // DOS/old-version compatible small buffer
      {
         lenSize = sizeof( ssize);
         rc = TxRead( ir->curHandle, &ssize, lenSize, &handled);
         rlen = ssize;
      }
      else                                      // non-DOS or new-version large buffer
      {
         lenSize = sizeof( rlen);
         rc = TxRead( ir->curHandle, &rlen, lenSize, &handled);
      }
      TRACES(( "TxRead rc:%lu fhandle:%lu  handled: %lu rlen: 0x%lx\n", rc, ir->curHandle, handled, rlen));
      if ((rc == NO_ERROR) && (handled == lenSize))
      {
         if ((cuType == DFSIM_LZW_CRC) || (cuType == DFSIM_NON_CRC))
         {
            usecrc32 = TRUE;
            rc = TxRead( ir->curHandle, &crc32, sizeof(crc32), &handled);
            TRACES(( "TxRead rc:%lu handled: %lu first: %8.8lx\n", rc, handled, crc32));
         }
         if (rc == NO_ERROR)                    // OK sofar, read the data
         {
            if (rlen <= bufferBytes)            // not too large ?
            {
               rc = TxRead( ir->curHandle, ir->readBuf, rlen, &handled);
               TRACES(( "\nTxRead rc:%lu handled: %lx\n", rc, handled));

               if (rc == NO_ERROR)
               {
                  switch (cuType)
                  {
                     case DFSIM_NON_CRC:        // not compressed, copy to cache
                     case DFSIM_NOCOMPR:
                        memcpy( cache->Buffer, ir->readBuf, rlen);
                        break;

                     case DFSUL_MLZW_12:        // LZW compressed, decompress to cache
                     case DFSIM_LZW_CRC:
                        rc = dfsUlBufUncompress( ir->ulzwHandle, (ULONG)  DFSUL_METHOD(cuType),
                                                 ir->readBuf,    (ULONG)  rlen,
                                                 cache->Buffer,  bufferBytes, &handled);
                        break;

                     default:
                        rc = DFS_ST_MISMATCH;
                        break;
                  }
                  if ((rc == NO_ERROR) && (usecrc32)) // check CRC over uncompressed data
                  {
//                   if (crc32 != TxCrc32( cache->Buffer, bufferBytes))
                     if (crc32 != TxCrc32( cache->Buffer, handled))
                     {
                        rc = ERROR_CRC;
                     }
                  }
               }
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
      }
   }
   RETURN ( rc);
}                                               // end 'dfsImzFile2Cache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create indexes for compressed image file, either the IMZ or subsequent .nnn
/*****************************************************************************/
static ULONG dfsImzIndexSingleFile
(
   int                 filenr,                  // IN    sequence number for file
   TXHFILE             fh,                      // IN    handle to opened IMZ/Inn
   BOOL                verbose,                 // IN    show info to screen/log
   ULN64              *filepos,                 // INOUT imagefile size read
   IMZCIROOT          *ir                       // INOUT IMZ info, root structure
)
{
   ULONG               rc = NO_ERROR;
   ULONG               handled;                 // nr of bytes handled
   ULN64               imgpos = DFSIMZ_HDRSIZE; // file-position in image
   ULONG               uncompressed;            // recorded buffersize
   ULONG               error_blocks = 0;        // blocks with detected errors
   USHORT              info;                    // compress info
   USHORT              ssize;                   // 16-bit size value
   ULONG               rlen;                    // data/total-record length
   ULONG               unitCountLimit;          // max nr of CU-inits to index
   ULONG               bufferBytes;             // size of buffer, in bytes
   ULONG               lenSize;                 // length of size field 2 or 4

   ENTER();

   TxFileSeek( fh, DFSIMZ_HDRSIZE, SEEK_SET);   // skip header (already known)

   TRACES(("Check for compression header\n"));
   rlen = DFSUL_SIG_LEN;
   rc = TxRead( fh, ir->readBuf, rlen, &handled);
   TRACES(( "TxRead rc:%lu handled: %lu sig: '%6.6s'\n", rc, handled, ir->readBuf));
   if (rc == NO_ERROR)
   {
      imgpos += handled;                        // keep file position in sync
      if ((memcmp( ir->readBuf, DFSUL_FILESIG, DFSUL_SIG_LEN) == 0) ||
          (memcmp( ir->readBuf, DFSUL_FILEBIG, DFSUL_SIG_LEN) == 0)  )
      {
         bufferBytes = ir->bufSize * ir->bpsector;
         if (bufferBytes < 65536)               // DOS/old-version compatible small buffer
         {
            lenSize = sizeof( ssize);
            rc = TxRead( fh, &ssize, lenSize, &handled);
            uncompressed = ssize;
         }
         else                                   // non-DOS or new-version large buffer
         {
            lenSize = sizeof( uncompressed);
            rc = TxRead( fh, &uncompressed, lenSize, &handled);
         }
         TRACES(( "TxRead rc:%lu handled: %lu bufsiz: 0x%lx\n", rc, handled, uncompressed));
         if (rc == NO_ERROR)
         {
            imgpos += handled;                  // keep file position in sync
            if (uncompressed <= bufferBytes)
            {
               if (ir->files > 1)               // multiple file, direct-index format
               {
                  unitCountLimit = ir->indexCount;
               }
               else                             // GSIZE index-records per group
               {
                  unitCountLimit = ir->indexCount * IMZCI_GSIZE;
               }
               do
               {
                  rc = TxRead( fh, &info, sizeof(info), &handled);
                  TRACES(( "TxRead rc:%lu handled: %lu   info: %4.4hx\n", rc, handled, info));
                  if ((rc == NO_ERROR) && (handled == sizeof(info))) // ctype size read correctly
                  {
                     if (((info & 0xff00) != DFSIM_FULLBUF) && // RLE or SMART full-buffer
                          (info           != DFSIM_SMARTFB)  ) // has NO length field, no CRC!
                     {
                        if (bufferBytes < 65536) // DOS/old-version compatible small buffer
                        {
                           rc = TxRead( fh, &ssize, sizeof(ssize), &handled);
                           rlen = ssize;
                        }
                        else                    // non-DOS or new-version large buffer
                        {
                           rc = TxRead( fh, &rlen, sizeof(rlen), &handled);
                        }
                        TRACES(( "TxRead rc:%lu handled: %lu length: 0x%lx\n", rc, handled, rlen));
                     }
                     if (rc == NO_ERROR)
                     {
                        if ((info == DFSUL_MLZW_12) || // LZW compr, regular
                            (info == DFSIM_LZW_CRC) || // LZW compr, including CRC32
                            (info == uncompressed)  || // NON compr, missing 2 bytes
                            (info == DFSIM_NON_CRC) || // NON compr, including CRC32
                            (info == DFSIM_NOCOMPR)  ) // NON compr, OK or missing 1
                        {
                           if ((  info == DFSIM_NOCOMPR) && // test 'missing-bytes' bug P#928
                               (  rlen               != uncompressed) &&
                               (((rlen & 0xff) << 8) == uncompressed)  )
                           {
                              rlen = uncompressed;
                              imgpos -= 1;      // fixup for single-byte bug
                              error_blocks++;
                           }
                           else if (info == uncompressed)
                           {
                              rlen = uncompressed; // assume full buffer size
                              info = DFSIM_NOCOMPR; // reset to uncompressed type
                              imgpos -= 2;      // fixup for double-byte bug
                              error_blocks++;
                           }
                        }
                        else if (info == DFSIM_NEXTFILE)
                        {
                           TRACES(("RLE end of file marker!\n"));
                           break;
                        }
                        else
                        {
                           rlen = 0;            // no data to skip/check for others
                        }
                        rlen += sizeof(info);   // add info bytes
                        if (((info & 0xff00) != DFSIM_FULLBUF) && // RLE or SMART full-buffer
                             (info           != DFSIM_SMARTFB)  ) // has NO length field, no CRC!
                        {
                           rlen += lenSize;     // add length bytes
                        }
                        if (( info           == DFSIM_LZW_CRC)  ||
                            ( info           == DFSIM_NON_CRC)  ||
                            ( info           == DFSIM_SMARTCRC) ||
                            ((info & 0xff00) == DFSIM_RLEBCRC)   )
                        {
                           rlen += sizeof( ULONG);      //- add crc bytes
                        }

                        //- Add found information to direct/group index structure
                        if (ir->files > 1)      // multiple file, direct-index format
                        {
                           IMZDIRECT *directIndex = (IMZDIRECT *) ir->index;

                           directIndex[ ir->ixUsed].filenr = filenr; //- sequence nr for this file
                           directIndex[ ir->ixUsed].cuType = info;   //- compression type
                           directIndex[ ir->ixUsed].offset = imgpos; //- offset for cuType in the file
                        }
                        else                    // single file, new compact format
                        {
                           IMZOGROUP *ogroupIndex = (IMZOGROUP *) ir->index;
                           ULONG      ixGroup     = ir->ixUsed / IMZCI_GSIZE; // index offset group number
                           ULONG      ixSub       = ir->ixUsed % IMZCI_GSIZE; // sub-index in offset group
                           ULONG      subunit;

                           if (ixSub == 0)      // first index in group, fill in base offset
                           {
                              TRACES(("New group %lu, offset: 0x%11.11llx\n", ixGroup, imgpos));
                              ogroupIndex[ ixGroup].offset = imgpos;
                           }
                           if (info < DFSIM_SMARTBUF) // regular compression type, or NONE
                           {
                              subunit  = ((ULONG) info) << 24; // just keep actual type byte
                              subunit +=  (ULONG) (imgpos - ogroupIndex[ ixGroup].offset);
                           }
                           else                 // Smart or RLE type of compression
                           {
                              subunit = ((ULONG) info) << 16;
                           }
                           TRACES(("New sub %lu, subunit: 0x%8.8lx\n", ixSub, subunit));
                           ogroupIndex[ ixGroup].subUnit[ ixSub] = subunit;
                        }
                        TRACES(( "File %3d Cunit#:%7lu at 0x%11.11llx type: %s  size:0x%8.8lx\n",
                                  filenr, ir->ixUsed, imgpos, dfsImageCode2String( info), rlen));

                        ir->ixUsed++;           // to next index slot
                        imgpos += rlen;         // add total length, to next unit

                        if (((info & 0xff00) != DFSIM_FULLBUF) && // RLE or SMART full-buffer
                             (info           != DFSIM_SMARTFB)  ) // has NO length field, no CRC!
                        {
                           TxFileSeek( fh, imgpos, SEEK_SET); // to next unit
                        }
                        dfsProgressShow( (*filepos + imgpos) / dfsGetSectorSize(), 0, 0, NULL);
                     }
                  }
                  else
                  {
                     TRACES(("Reached end of file\n"));   //- End of IMZ file, not always an error!
                     rc = NO_ERROR;                       //- Could be truncated, uses (partial) XMZ
                     break;
                  }
               } while ((rc == NO_ERROR) && (ir->ixUsed < unitCountLimit));

               if (verbose && (error_blocks != 0))
               {
                  TxPrint( "Major warning!    : %lu blocks in the opened image had errors, data on\n"
                           "                    those areas (max %hu sectors each) is unreliable.\n",
                                                error_blocks, ir->bufSize);
               }
            }
            else                                // buffer size mismatch
            {
               if (verbose)
               {
                  TxPrint( "Error: Recorded buffersize in IMZ is inconsistent!\n");
               }
               rc = DFS_BAD_STRUCTURE;
            }
         }
      }
   }
   *filepos += imgpos;                          // total size read from image
   RETURN ( rc);
}                                               // end 'dfsImzIndexSingleFile'
/*---------------------------------------------------------------------------*/


