//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee major functions: imaging
//
// Author: J. van Wijk
//
// JvW  07-01-2004   Initial version, split off from DFSUTIL.C
// JvW  01-11-2016   Removed OS/2 named-pipe support, use generic TxFileOpenRead...
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsrgkey.h>                           // Registration interface
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsdgen.h>                            // DFS generic dialogs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsimage.h>                           // image create restore
#include <dfsimzci.h>                           // DFS IMZ cache/index functions
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfsupart.h>                           // PART utility functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfscfdsk.h>                           // FDSK command functions
#include <dfstable.h>                           // SLT  utility functions
#include <dfsfat.h>                             // FAT  directory definitions

#if defined (WIN32)
   #include <dfsntreg.h>                        // NT error detail reporting
#endif


//- definitions for new-style DFSee images
#define DFSIM_HEADERSIZE SECTORSIZE
#define DFSIM_BDATA_SIZE  64                    // size of binary data area
#define DFSIM_ASCII_SIZE (SECTORSIZE - DFSIM_BDATA_SIZE)
#define DFSIM_SSIZE       16
#define DFSIM_TSIZE      (DFSIM_ASCII_SIZE - DFSIM_SSIZE)

#define DFSIM_FIXEDCRC    0xdeafface            // fixed CRC for SMART type

#define DFSIM_SIGNATURE  "DFSeeImageFile #"     // 16 chars plus terminator,

typedef struct dfsim_header
{
   char                signature[DFSIM_SSIZE];  // 000 DFSee signature
   char                asciitext[DFSIM_TSIZE];  // 010 ascii description ~ 430
   ULONG               sequence;                // 1C0 seq in multi-file 1..n
   ULONG               offset;                  // 1C4 start-offset for this file
   ULONG               flags;                   // 1C8 flags on save (smart/compr)
   ULONG               sectors;                 // 1CC total #sectors in image
   ULONG               bufsize;                 // 1D0 buffersize used (compress)
   ULONG               objbase;                 // 1D4 base for current object
   ULONG               objsize;                 // 1D8 size for current object
   ULONG               bpsector;                // 1DC bytes per sector(8.09)
   ULONG               hi_offs;                 // 1E0 upper 32 bits of offset   (14.5)
   ULONG               hi_sect;                 // 1E4 upper 32 bits of #sectors (14.5)
   ULONG               hi_base;                 // 1E8 upper 32 bits of objbase  (14.5)
   ULONG               hi_size;                 // 1EC upper 32 bits of objsize  (14.5)
   BYTE                reserved[DFSIMZ_HDRSIZE  -0x1F0]; // upto end of sector
} DFSIM_HEADER;                                 // end of struct "dfsim_header"

// Progress related definitions and variables
#define DFSIM_SMART_FAST_FACTOR   70            // smart (unused) areas 70 times faster
                                                // than regular used areas (imaging)

static  char       *cmd_image[] =
{
   "",
   "Save specified or current store to an IMAGE file",
   "",
   " Usage: IMAGE filename  [firstsector  [size]] [options]",
   "",
   "   filename     : Name of the imagefile to create, extension is optional",
   "                  but you can specify a full path+filename",
   "",
   "   firstsector  : First sector (in Hex) of area to image, default 0",
   "   size         : Size to write to image as mcs-nr, default whole area",
   "",
   "   -6           : Do NOT use CRC32 data consistency checking (V-6.xx compat)",
   "   -8           : Do NOT use ultra compact 2-byte RLE/SMART  (V-13.x compat)",
   "   -A           : Append to existing imagefile (use with RAW only)",
   "   -b:sectors   : Number of sectors to buffer, the default is to",
   "                  use the current number of sectors per track",
   "   -D[:disk]    : FROM disk, disk number to make an image FROM, 1..max disk",
   "   -F:sector    : First sector of area to image, default 0",
   "   -I[:fname]   : Filename to make an image FROM, img filename",
   "   -L           : Exclude LVM signature area from image (LVM/JFS)",
   "   -M           : Media change prompt after each file (on removables)",
   "   -m[:mcs]     : Maximum size per imagefile as mcs-nr, default 650 MiB",
   "   -P[:pid]     : FROM partition, partition make an image FROM, PID or letter",
   "   -q           : Quiet, display minimal details, progress mostly",
   "   -s:size      : Size to write to image as mcs-nr, default whole area",
   "   -S           : Smart skip unused space in FS or disk (implies '-z')",
   "   -v           : Verify, verbose list every data block while compressing",
   "   -V[:vol]     : FROM volume,  a volume to make an image FROM, driveletter",
   "   -x           : Generate separate INDEX file for compressed image (BROWSE)",
   "   -z           : Use (LZW/advanced) compression on imagefile   (DFSee .IMZ)",
   "   -!           : Force full dialog, even when sufficient info is specified",
   "",
   NULL
};

static  char       *cmd_restore[] =
{
   "",
   "Restore an IMAGE file to specified or current store"
   "",
   " Usage: RESTORE  filename  [firstsector  [size]] [options]",
   "",
   "   filename     : Name of the imagefile to restore, extension is optional",
   "                  but you can specify a full path+filename",
   "",
   "   firstsector  : First sector (in Hex) of area to restore to, default 0",
   "   size         : Size to read from image as mcs-nr, default ALL",
   "",
   "   -b:sectors   : RAW only, default is the tracksize of the destination",
   "                  (for IMZ the number of sectors recorded in the IMZ",
   "                   is used, and the -b option is ignored completely)",
   "   -c           : Compare image data against data in current object",
   "                  stops at first difference found (implies -t)",
   "   -D[:disk]    : TO disk, disk number to restore the image TO, 1..max",
   "   -F:sector    : First sector of area to restore to, default 0",
   "   -I[:fname]   : Use option dialog to restore to an imagefile",
   "   -L           : Exclude LVM signature area from restore (LVM/JFS)",
   "   -P[:pid]     : TO partition, partition restore the image TO, PID",
   "   -q           : Quiet, display minimal details, progress mostly",
   "   -t           : Test imagefile for VERIFY, do NOT write back",
   "   -v           : Verify, verbose list every data block while uncompressing",
   "   -V[:vol]     : TO volume,  a volume to restore the image TO, driveletter",
   "   -s:size      : Size to write to image as mcs-nr, default whole area",
   "   -S           : Calculate size to restore from the imagefile for RAW,",
   "                  or get size and startsector from header info for IMZ.",
   "",
   NULL
};


// DFS save a single imagefile with optional compression
static ULONG dfsSaveSingleImageFile             // RET   result code
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    remaining sectors todo
   char               *fspec,                   // IN    destination filespec
   ULONG               flags,                   // IN    compress, smart etc
   ULONG               sequence,                // IN    file sequence number
   ULN64               limit,                   // IN    max filesize in bytes
   ULN64              *filesize,                // INOUT size of imagefile
   ULN64              *lastsn,                  // OUT   last sectornr handled
   ULN64              *failed,                  // OUT   sect with read-failure
   ULN64              *totalSkipped             // OUT   total smart skipped sectors
);


// DFS restore sectors to volume/partition from a previously saved image
static ULONG dfsRestoreSingleFile
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    nr of sectors to write
   ULONG               flags,                   // IN    imaging options
   char               *fspec,                   // IN    source filespec
   ULONG               sequence,                // IN    file sequence number
   ULONG               fileCount,               // IN    #files, same as last file#
   ULN64              *filepos,                 // INOUT imagefile size read
   ULN64              *startsn,                 // OUT   real start, from IMZ
   ULN64              *lastsn,                  // OUT   last sectornr handled
   ULN64              *totalsize                // OUT   total size  (IMZ header)
);


/*****************************************************************************/
// High level IMAGE and RESTORE command logic
/*****************************************************************************/
ULONG dfsImageRestoreCommand
(
   int                 cc,                      // IN    number of parameters
   char               *c1,                      // IN    parameter string 1
   char               *c2,                      // IN    parameter string 2
   char               *c3,                      // IN    parameter string 3
   BOOL                create                   // IN    action is create img
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               sn = 0;                  // current sector number
   ULN64               sectors = 0;
   ULN64               size;
   ULONG               ulFiles;                 // Number of files in image set
   BOOL                isCompressed;            // Compresed imagefile(s)
   TXLN                s0;                      // temporary string space
   TX1K                s1;                      // temporary huge string
   BOOL                ok = TRUE;
   TXTM                sizetext;                // text buffer
   DFSPARTINFO        *p;

   ENTER();

   if (TxaOption('?') || ((cc > 1) && (c1[0] == '?')))
   {
      TxShowTxt((create) ? cmd_image : cmd_restore);
      RETURN (rc);
   }

   rc = dfsSetObjectCurrent( "DPVI", (dfsa->verbosity > TXAO_QUIET)); // open object when specified
   if ((rc == NO_ERROR) || (rc == DFS_BAD_STRUCTURE))
   {
      rc = NO_ERROR;                            // OK for now ...
      ok &= TxaOptMutEx((cc > 2), "F",  "if 'First' is specified!", NULL);
      if (create)
      {
         ok &= TxaOptMutEx((cc > 3), "s", "if 'size'  is specified!", NULL);
      }
      else                                      // -S and size not compatible!
      {
         ok &= TxaOptMutEx((cc > 3), "sS", "if 'size'  is specified!", NULL);
      }
      if (ok)
      {
         ok &= dfsOptionSizeSectors( 'F', DFSTORE, 'm', 0,      &sn); // from
         ok &= dfsOptionSizeSectors( 's', DFSTORE, 'm', 0, &sectors); // size

         TRACES(( "sn:%llx sectors:%llx\n", sn, sectors));

         if (cc > 2)                            // regular parameters
         {
            sn = dfsGetSymbolicSN( c2, 0);
         }
         if (cc > 3)
         {
            sectors = dfsGetSymbolicSize( c3, sectors, DFSTORE, sn);
         }
         else if ((sectors == 0) && !TxaOption('S')) // default: FROM to end of object
         {
            //- Uses size of CURRENT object, not the one selected later in dialog!
            sectors = dfsGetSymbolicSize( "$", sectors, DFSTORE, sn);
         }

         p = SINF->p;
         if ((p != NULL) && TxaOption('L'))     // partition, exclude-LVM option
         {
            if (sectors > p->lvmReserved)       // valid reserved area
            {
               sectors -= p->lvmReserved;       // exclude signature-area
               TRACES(( "sectors -L: 0x%llx\n", sectors));
            }
         }
      }
      if (TxaOption('!') || (cc < 2) || (c1[strlen(c1)-1] == FS_PATH_SEP)
                                     || (c1[strlen(c1)-1] == FS_PALT_SEP))
      {
         #if defined (USEWINDOWING)
            if (dfsa->batch)
            {
               TxShowTxt((create) ? cmd_image : cmd_restore);
               rc = DFS_CMD_FAILED;
            }
            else
            {
               DFSOBJECT  object = dfsGetObjectType( DFSTORE, "PDV");

               if (cc <= 3)                     // no explicit size given yet
               {
                  sectors = 0;                  // use dialog default (could be other object!)
               }
               if ((object == DFSO_NONE) && (dfstStoreType( DFSTORE) == DFST_UNUSED))
               {
                  object = DFSO_PART;           // default when nothing open/selected yet
               }
               if (create)
               {
                  if (TxaOption('m'))
                  {
                     dfsOptionSizeSectors( 'm', DFSTORE, 'm', DFSECT_CD80, &size);
                  }
                  else
                  {
                     size = 0;                  // use dialog default
                  }
                  dfsImageDialog(  object, c1, TxaOption('M'), DFSECT2MIB( size, dfsGetSectorSize()), sn, sectors);
               }
               else
               {
                  dfsRestoreDialog( object, c1, FALSE, sn, sectors);
               }
            }
         #else
            TxShowTxt((create) ? cmd_image : cmd_restore);
            rc = DFS_CMD_FAILED;
         #endif
         ok = FALSE;
      }
      else
      {
         strcpy( s0, c1);
      }
      if (ok)
      {
         if (create)                            // create image
         {
            ULONG   flags    = DFSI_STANDARD;
            ULN64   maxfsize = 0;

            if (TxaOption('S'))                 // smart-use
            {
               flags |= DFSI_SMARTUSE;
            }
            if (TxaOption('M'))                 // media-change
            {
               flags |= DFSI_REMOVABL;
            }
            if (TxaOption('A'))                 // append to image
            {
               flags |= DFSI_APPENDTO;
            }
            if (TxaOptUnSet('p'))               // no progress item
            {
               flags |= DFSI_NOPRITEM;
            }
            if ((TxaOption('z')) ||             // compress
                (TxaOption('S'))  )             // Smart, implies COMPRESS!
            {
               flags |= DFSI_COMPRESS;
            }
            else
            {
               flags |= DFSI_RAWIMAGE;
            }
            if (TxaOption('x'))
            {
               flags |= DFSI_CR_INDEX;          // Create IMZ index
            }
            if (TxaOption('6'))
            {
               flags |= DFSI_NOCRCUSE;
            }
            if (TxaOption('8'))
            {
               flags |= DFSI_NOFULBUF;
            }
            if (TxaOption('v'))
            {
               flags |= DFSOPT_VERBOSE;
            }
            if (TxaOptSet('R'))                 // explicit RAW setting
            {
               if (TxaOption('R'))              // no header on image
               {
                  flags |=  DFSI_RAWIMAGE;
               }
               else
               {
                  flags &= ~DFSI_RAWIMAGE;
               }
            }
            if (flags & DFSI_COMPRESS)          // Use (TX) zipped format
            {
               TxFnameExtension( s0, "imz");
            }
            else
            {
               TxFnameExtension( s0, "img");
            }
            strcpy(    s1, "");
            if (TxaOptSet('m'))                 // limit filesize to maximum
            {
               dfsOptionSizeSectors( 'm', DFSTORE, 'm', DFSECT_CD80, &size);
               maxfsize = ((LLONG) size * dfstGetSectorSize( DFSTORE));

               if (TxaOption('M'))
               {
                  strcpy( s1, "Prompt for media-changes.  ");
               }
               dfstrSXiB( s1, "Files limited to : ", size, "\n");
            }
            if (TxaOptSet('A'))
            {
               strcat( s1, "Append to imagefile, when it exists.\n\n");
            }
            else
            {
               strcat( s1, "Replace imagefile, when it exists.\n\n");
            }
            if (sectors == 0)                   // interpret explicit 0
            {                                   // for size as 'whole object'
               sectors = dfsGetLogicalSize();
            }
            sizetext[0] = 0;
            dfstrSz64( sizetext, "", sectors, "");
            if ((dfsa->batch) || (TxConfirm( 5905,
               "Save from current store:\n%s\n%s\n\n%s"
               "Save %s from SN 0x0%llX to %s%simagefile %s ? [Y/N] : ",
                dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE), s1,
                sizetext, sn, (flags & DFSI_SMARTUSE) ? "Smart-"      : "",
                              (flags & DFSI_COMPRESS) ? "compressed " : "", s0)))
            {
               rc = dfsCreateImage( sn, sectors, s0, flags, maxfsize);
            }
            else
            {
               rc = DFS_NO_CHANGE;
            }
         }
         else                                   // restore image
         {
            TRACES(( "sn:%llx sectors:%llx\n", sn, sectors));
            if (!TxFileExists( s0))             // as given (perhaps no ext)
            {
               TxFnameExtension( s0, "imz");
               if (!TxFileExists( s0))
               {
                  strcpy( s0, c1);
                  TxFnameExtension( s0, "img");
                  if (!TxFileExists( s0))
                  {
                     rc = DFS_NOT_FOUND;
                  }
               }
            }
            if (rc == NO_ERROR)
            {
               ULONG         opts   = 0;
               ULONG         chelp;                   // confirmation help id

               if (TxaOption('t') || TxaOption(TXA_O_TEST))
               {
                  opts |= DFSOPT_TEST;
               }
               if (TxaOption('c'))              // Compare, implies -t
               {
                  opts |= DFSOPT_TEST;
                  opts |= DFSOPT_COMPARE;
               }
               if (TxaOption('v'))
               {
                  opts |= DFSOPT_VERBOSE;
               }
               //- to be refined, use sectors/bps value from IMZ header? (for buffers etc)
               isCompressed = dfsImageTypeIsImz( s0, TxaOption('S'), NULL, &ulFiles, &sn, &sectors, NULL, NULL, NULL);
               if (TxaOption('S'))              // Size from file info / IMZ hdr
               {
                  if (isCompressed)
                  {
                     opts |= DFSW_IMZ_SIZE;
                  }
                  strcpy( sizetext, "whole imagefile contents, to indicated START");
               }
               else
               {
                  if (sectors == 0)             // interpret explicit 0
                  {                             // for size as 'whole object'
                     sectors = dfsGetLogicalSize();
                  }
               }
               sizetext[0] = 0;
               dfstrSz64( sizetext, "", sectors, " sectors");
               dfstrX10(  sizetext, " to SN ", sn, "", "");

               if    (!(opts & DFSOPT_TEST))    // actual restore
               {
                  chelp = 5906;
                  sprintf( s1, "RESTORE %u imagefile%s %s to:\n%s\n%s\n\n"
                               "Write %s ? [Y/N] : ", ulFiles, (ulFiles) ? "s" : "", s0,
                                dfstStoreDesc1( DFSTORE),
                                dfstStoreDesc2( DFSTORE), sizetext);
               }
               else if (opts & DFSOPT_COMPARE)  // verify against object
               {
                  chelp = 5907;
                  sprintf( s1, "COMPARE %u image-file%s %s to:\n%s\n%s\n\n"
                               "Compare %s ? [Y/N] : ", ulFiles, (ulFiles) ? "s" : "", s0,
                                dfstStoreDesc1( DFSTORE),
                                dfstStoreDesc2( DFSTORE), sizetext);
               }
               else                             // verify file CRC only
               {
                  chelp = 5908;
                  sprintf( s1, "VERIFY %u imagefile%s %s on CRC values\n\n"
                               "Verify %s ? [Y/N] : ", ulFiles, (ulFiles) ? "s" : "", s0, sizetext);
               }
               if ((dfsa->batch) || (TxConfirm( 5006, s1)))
               {
                  if (DFSTORE_WRITE_ALLOWED)
                  {
                     rc = dfsRestoreImage( sn, sectors, opts, s0, ulFiles);
                     if ((rc == NO_ERROR) || (rc == DFS_PSN_LIMIT))
                     {
                        dfsReadDiskInfo( FDSK_QUIET); // re-read all partition info

                        #if !defined (OEM_BRANDED)
                           if ((dfstStoreType( DFSTORE) == DFST_PHDISK) &&
                               (!(opts & DFSOPT_TEST))) // no verify, no virtual
                           {
                              TxNamedMessage( !dfsa->batch, 0, " INFO: Reboot recommended ",
                                  "It may be required to REBOOT the system to allow "
                                  "full use of the disk/partition just restored ...");
                           }
                        #endif
                     }
                  }
                  else
                  {
                     rc = DFS_READ_ONLY;
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
            else
            {
               TxPrint( "\nImagefile '%s' not found!\n", c1);
               rc = DFS_NOT_FOUND;
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsImageRestoreCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get type of image (IMZ/RAW) and optional #files, start, size, bufsize & bps
/*****************************************************************************/
BOOL dfsImageTypeIsImz                          // RET   Image is an IMZ type
(
   char               *fname,                   // IN    Image filename
   BOOL                ssFromFile,              // IN    get start/size from files
   ULN64              *fSize,                   // OUT   Total filesize in bytes
   ULONG              *files,                   // OUT   Nr of files when multiple
   ULN64              *imStart,                 // OUT   start sector
   ULN64              *imSize,                  // OUT   Image size in sectors
   ULONG              *imzBuf,                  // OUT   IMZ buffersize (spt * bps)
   ULONG              *imzBps,                  // OUT   IMZ bytes per sector
   BYTE               *sType                    // OUT   type first sector (MBR...)
)
{
   BOOL                rc  = FALSE;             // function return
   TXHFILE             fhandle;                 // Handle for Image file

   ENTER();

   if (TxFileOpenReadOnly( fname, &fhandle) == NO_ERROR)
   {
      rc = dfsGetImzImageInfo( fname, fhandle, ssFromFile,
                               fSize, files, imStart, imSize, imzBuf, imzBps, sType);
      TxClose( fhandle);
   }
   RETURN (rc);
}                                               // end 'dfsImageTypeIsImz'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get image (IMZ/RAW) type and optional information, from OPEN filehandle
/*****************************************************************************/
BOOL dfsGetImzImageInfo                         // RET   Image is an IMZ type
(
   char               *fname,                   // IN    Image filename
   TXHFILE             fhandle,                 // IN    Handle to OPEN Image file
   BOOL                ssFromFile,              // IN    get start/size from files
   ULN64              *fSize,                   // OUT   Total filesize in bytes
   ULONG              *files,                   // OUT   Nr of files when multiple
   ULN64              *imStart,                 // OUT   start sector
   ULN64              *imSize,                  // OUT   Image size in sectors
   ULONG              *imzBuf,                  // OUT   IMZ buffersize (spt * bps)
   ULONG              *imzBps,                  // OUT   IMZ bytes per sector
   BYTE               *sType                    // OUT   type first sector (MBR...)
)
{
   BOOL                rc  = FALSE;             // function return
   DFSIM_HEADER        hdr;                     // IMZ header, 1 sector
   ULONG               handled;                 // nr of bytes handled
   ULN64               fileBytes = 0;           // size in bytes
   ULONG               fileCount;               // number of files in RAW set

   ENTER();

   TRACES(("fname: '%s'  handle: %u\n", fname, fhandle));

   TxFileSeek( fhandle, 0, SEEK_SET);           // make sure we are at start of file
   if (TxRead( fhandle, &hdr, sizeof( DFSIM_HEADER), &handled) == NO_ERROR)
   {
      if (memcmp( hdr.signature, DFSIM_SIGNATURE, DFSIM_SSIZE) == 0)
      {
         if (hdr.bpsector == 0)                 // older header version
         {
            hdr.bpsector = SECTORSIZE;          // set to fixed 512
         }
         fileBytes = TXmku64( hdr.sectors, hdr.hi_sect) * hdr.bpsector;

         fileCount = dfsMultipleImgInfo( fname, fhandle, fSize);

         if ((imStart != NULL) && (ssFromFile == TRUE))
         {
            *imStart = TXmku64( hdr.offset, hdr.hi_offs);
         }
         if (imzBuf != NULL)
         {
            *imzBuf = hdr.bufsize;
         }
         if (imzBps != NULL)
         {
            *imzBps = hdr.bpsector;
         }
         if (sType != NULL)
         {
            if ((strstr((char *) &hdr, " Whole  ") != NULL) &&
                (strstr((char *) &hdr, " FDISK  ") != NULL)  )
            {
               *sType = ST_MASTR;               // 1st will be an MBR sector
            }
            else
            {
               *sType = ST_UDATA;               // contents unknown
            }
         }
         rc = TRUE;                             // signal compression was used
      }
      else                                      // raw IMG
      {
         if ((imStart != NULL) && (ssFromFile == TRUE))
         {
            *imStart = 0;
         }
         fileCount = dfsMultipleImgInfo( fname, fhandle, &fileBytes);
      }
      if ((imSize != NULL) && (ssFromFile == TRUE))
      {
         *imSize = fileBytes / dfsGetSectorSize();
      }
      if (files != NULL)
      {
         *files = fileCount;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsGetImzImageInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show available information for an imagefile (IMZ/IMG)
/*****************************************************************************/
ULONG dfsImageInfoDisplay
(
   char               *fname                    // IN    Image filename
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSIM_HEADER        hdr;                     // IMZ header, 1 sector
   TXHFILE             fhandle = 0;
   ULONG               handled;                 // nr of bytes handled
   ULN64               fileBytes = 0;           // size in bytes
   ULN64               size;                    // size in sectors
   ULONG               fileCount;               // number of files in RAW set
   TX1K                message;
   TXTM                text;

   ENTER();

   if (TxFileOpenReadOnly( fname, &fhandle) == NO_ERROR)
   {
      if (TxRead( fhandle, &hdr, sizeof(hdr), &handled) == NO_ERROR)
      {
         if (memcmp( hdr.signature, DFSIM_SIGNATURE, DFSIM_SSIZE) == 0)
         {
            if (hdr.bpsector == 0)              // older header version
            {
               hdr.bpsector = SECTORSIZE;       // set to fixed 512
            }
            size      = TXmku64( hdr.sectors, hdr.hi_sect);
            fileBytes = size * hdr.bpsector;

            hdr.sequence = 0;                   // make sure txt is NULL terminated
            TxRepl( hdr.asciitext, 0x1a, 0);    // replace EOF by 0-terminator
            TxRepl( hdr.asciitext, 0x0d, ' ');  // replace CR  by one space

            sprintf( text, "%llu = 0x0%llx sectors\n", size, size);
            sprintf(       message, "IMZ info for '%s':\n\n%s\n", fname, hdr.signature);
            dfstrUllDot20( message, "Expanded:", fileBytes, " bytes, or ");
            strcat(        message, text);

            //- to be refined, add compatibility non-DOS/old-version for large-buffer?
         }
         else                                   // raw IMG
         {
            fileCount = dfsMultipleImgInfo( fname, fhandle, &fileBytes);
            size = fileBytes / dfsGetSectorSize();
            sprintf( text, "%llu = 0x0%llx sectors\n", size, size);
            sprintf(       message, "RAW imagefile or other filetype (no IMZ header found):\n"
                                    "                                          \n'%s'\n\n", fname);
            dfstrUllDot20( message, "Img size:", fileBytes, " bytes, or ");
            strcat(        message, text);
            sprintf( text, "in %u RAW image file%s\n", fileCount, (fileCount > 1) ? "s" : "");
            strcat(        message, text);
         }
         TxPrint( "\n%s", message);
         TxNamedMessage( !dfsa->batch, 0, " INFO: Compressed DFSee image file ", message);
      }
      else
      {
         rc = DFS_BAD_STRUCTURE;
      }
      TxClose( fhandle);
   }
   else
   {
      rc = TX_INVALID_FILE;
   }
   RETURN (rc);
}                                               // end 'dfsImageInfoDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get number of files in a set of image files, and return total size in bytes
/*****************************************************************************/
ULONG dfsMultipleImgInfo                        // RET   Nr of files in image set
(
   char               *fname,                   // IN    filename image file
   TXHFILE             fhandle,                 // IN    open handle first file
   ULN64              *size                     // OUT   filesize or NULL
)
{
   ULN64               thisSize;                // size for current file
   ULONG               filenr = 1;              // file sequence number
   TXLN                line;                    // filename and text buffer
   TXTS                fext;                    // file extension string

   ENTER();

   TxHandle2FileSize( fhandle, size);           // get size for base .IMG/.IMZ

   strcpy( line, fname);                        // first filename
   do
   {
      sprintf( fext, ".%3.3u", ++filenr);       // next file number/extension
      TxStripExtension( line);
      strcat( line, fext);                      // new full filename

      if (TxFileSize( line, &thisSize))         // next in sequence exists
      {
         if (size != NULL)
         {
            *size += thisSize;                  // add its size to total
         }
      }
      else
      {
         break;
      }
   } while (TRUE);
   RETURN( filenr - 1);                         // last tried number, minus one
}                                               // end 'dfsMultipleImgInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate IMAGE/RESTORE image block-codes to short description string
/*****************************************************************************/
char *dfsImageCode2String
(
   USHORT              code                     // IN    block code
)
{
   static TXTS         str;

   switch (code)
   {
      case DFSIM_NOCOMPR : return "Not compressed";
      case DFSUL_MLZW_12 : return "LZW compressed";
      case DFSIM_LZW_CRC : return "LZW compr, CRC";
      case DFSIM_NON_CRC : return "Not compr, CRC";
      case DFSIM_SMARTBUF: return "Smart-4-comprs";
      case DFSIM_SMARTCRC: return "Smart-8-cp-CRC";
      case DFSIM_SMARTFB:  return "Smart-2-FulBuf";
      case DFSIM_NEXTFILE: return "Next-File-Mark";
      default:
         if      ((code & 0xff00) == DFSIM_RLEBCRC)
         {
            sprintf( str, "RLE-8 0x%2.2hhx CRC", (BYTE) (code & 0xff));
            return ( str);
         }
         else if ((code & 0xff00) == DFSIM_RLEBYTE)
         {
            sprintf( str, "RLE-4 0x%2.2hhx cmp", (BYTE) (code & 0xff));
            return ( str);
         }
         else if ((code & 0xff00) == DFSIM_FULLBUF)
         {
            sprintf( str, "RLE-2 0x%2.2hhx FB ", (BYTE) (code & 0xff));
            return ( str);
         }
         else
         {
            return "Invalid code! ";
         }
   }
}                                               // end 'dfsImageCode2String'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS write sectors to disk-file(s) with optional compression
/*****************************************************************************/
ULONG dfsCreateImage                            // RET   result code
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    nr of sectors to save
   char               *fspec,                   // IN    destination filespec
   ULONG               flags,                   // IN    compress, smart etc
   ULN64               limit                    // IN    max filesize in bytes
)
{
   ULONG               rc = NO_ERROR;
   BOOL                smartuse = (flags & DFSI_SMARTUSE); // zero unused sects
   BOOL                removabl = (flags & DFSI_REMOVABL); // do media change
   ULONG               filenr   = 1;            // file sequence number
   ULN64               todo     = size;         // max size todo for next file
   ULN64               start    = fn;           // next sector to start with
   ULN64               skipped  = 0;            // total smart-skipped sectors
   ULN64               fail     = 0;            // total read-errors
   ULN64               fsize    = 0;            // total filesize written
   ULN64               last     = fn -1;        // last sector done
   TXLN                line;                    // filename and text buffer
   TXTS                fext;                    // file extension string
   TXTIMER             stime = TxTmrGetNanoSecFromStart();
   double              elapsed;

   ENTER();
   TRACES(("flags: 0x%8.8lx\n", flags));

   TxCancelAbort();                             // cancel any pending abort (alloc!)
   if (smartuse && (dfsa->FsUnallocSmart == 0)) // no Smart alloc info available yet
   {
      TxPrint( "\nGet Smart-ALLOC info. May be aborted with <Esc> if ETA accuracy is unimportant.\n");
      DFSFNCALL( dfsa->FsAllocDisplay, 0, 0, "@", NULL); // single line ALLOC display / info gathering

      if (TxAbort())
      {
         TxPrint( "Gathering Smart-ALLOC info was aborted. ETA calculations will be less accurate!\n");
         TxCancelAbort();                       // cancel any pending abort (alloc!)
      }
   }
   if (TxaOption( 'q') == FALSE)                // unless we want quiet
   {
      dfsX10( "\nSave image  start : ", fn, "", " ");
      dfsX10( "size ", size, "", " to ");
      TxPrint( "%s\n", fspec);
   }

   strcpy( line, fspec);                        // first filename
   do
   {
      rc = dfsSaveSingleImageFile( start, todo, line, flags, filenr, limit,
                                  &fsize, &last, &fail, &skipped);
      if (rc == DFS_PENDING)
      {
         start = last +1;                       // start sector next file
         todo  = fn + size - start;             // number of sectors todo
         sprintf( fext, ".%3.3u", ++filenr);    // new file number/extension
         TxStripExtension( line);
         strcat( line, fext);                   // new full filename

         if (removabl)                          // prompt for media change
         {
            if (!dfsChangeWriteMedium( line, limit))
            {
               rc = ERROR_WRONG_DISK;
            }
         }
      }
   } while ((rc == DFS_PENDING) && !TxAbort()); // more files to be done

   if (TxaOption( 'q') == FALSE)                // unless we want quiet
   {
      strcpy(     line, "");
      dfstrX10(   line, "Last sector done  : ", last, CNN, " ");
      dfstrSz64(  line, "size ",   last - fn +1, " ");
      dfstr64XiB( line, "to ", fsize, "");
      if (filenr > 1)
      {
         TxPrint( "%s, %u files\n", line, filenr);
      }
      else                                      // keep output compact :-)
      {
         TxPrint( "%s, 1 file\n", line);
      }

      if ((smartuse) && (skipped != 0) && (skipped < (last - fn)))
      {
         TxPrint( "SmartUse optimize : %4.1lf %%   ", ((double) 100.0 * (double) skipped / (double) (last - fn)));
         dfsSz64( "  skipped ", skipped, "\n");
      }
      if (fail != 0)
      {
         dfsSz64( "Read failures  on : ", fail, " sectors, GARBAGE in resulting image.\n");
      }
      elapsed = dfsDisplayThroughput( stime, last - fn +1, dfsGetSectorSize());
      dfsDisByteThroughput( elapsed, fsize, "Image-file write speed");
   }

   if (rc == NO_ERROR)
   {
      if (flags & DFSI_CR_INDEX)
      {
         dfsImzGenerateIndex( fspec, TRUE, TRUE);
      }
   }
   else
   {
      dfsa->explain = TRUE;                     // try to explain the RC
   }
   RETURN(rc);
}                                               // end 'dfsCreateImage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS save a single imagefile with optional compression
/*****************************************************************************/
static ULONG dfsSaveSingleImageFile             // RET   result code
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    remaining sectors todo
   char               *fspec,                   // IN    destination filespec
   ULONG               flags,                   // IN    compress, smart etc
   ULONG               sequence,                // IN    file sequence number
   ULN64               limit,                   // IN    max filesize in bytes
   ULN64              *filesize,                // INOUT size of imagefile
   ULN64              *lastsn,                  // OUT   last sectornr handled
   ULN64              *failed,                  // OUT   sect with read-failure
   ULN64              *totalSkipped             // OUT   total smart skipped sectors
)
{
   ULONG               rc = NO_ERROR;
   TXHFILE             fh = 0;                  // filehandle
   BYTE               *data = NULL;             // ptr to data to write
   ULN64               i;                       // number of sectors
   ULONG               j;                       // sectors in buffer
   ULN64               sn = fn;                 // Data LSN
   USHORT              bps = dfsGetSectorSize(); // bytes per sector
   ULONG               sz;                      // nr of bytes to handle
   ULONG               handled  = 0;            // nr of bytes handled
   ULONG               bs       = 0;            // sectors to r/w/compress
   BYTE               *ibuf  = NULL;            // imaging disk i/o buffer
   BYTE               *cbuf  = NULL;            // compression buffer
   BYTE               *ubuf  = NULL;            // uncompress  buffer (verify)
   USHORT              info;                    // 16-bit value to write
   USHORT              ssize;                   // short size value
   ULONG               csize;                   // compressed size
   ULONG               usize;                   // uncompressed size  (verify)
   TXLN                text;                    // text buffer
   DFSUL_HANDLE        ulh;                     // compression handle
   BOOL                rawimage =  (flags & DFSI_RAWIMAGE); // don't add header
   BOOL                nopritem =  (flags & DFSI_NOPRITEM); // no progress item
   BOOL                appendto =  (flags & DFSI_APPENDTO); // append to image
   BOOL                compress =  (flags & DFSI_COMPRESS); // use compression
   BOOL                usecrc32 = !(flags & DFSI_NOCRCUSE); // no CRC used  6.xx
   BOOL                fullbuf2 = !(flags & DFSI_NOFULBUF); // no 2-byte FB 13.x
   BOOL                smartuse =  (flags & DFSI_SMARTUSE); // zero unused sects
   BOOL                verbose  =  (flags & DFSOPT_VERBOSE);
   ULN64               imgpos   = 0;            // file-position in image
   BOOL               *sectused = NULL;         // usage array
   ULONG               bufusage = 0;            // sector used in this buffer
   ULONG               crc32;
   ULONG               blocknr = 0;             // block-sequence nr in file
   BOOL                RleCompr = FALSE;
   USHORT              RleValue = 0;
   ULONG               bads;                    // bad sectors in Read
   ULONG               iosizeErrors = 0;        // IO size error on verify, count
   ULONG               decodeErrors = 0;        // Decode  error on verify, count
   ULONG               verifyErrors = 0;        // CRC     error on verify, count
   DFSIM_HEADER        hdr;                     // DFSee image header

   ENTER();

   TRACES(( "Start:%llx todo:%llx seq:%u limit:%llx flags:%8.8x file:'%s'\n", fn, size, sequence, limit, flags, fspec));
   TRACES(( "Smart:%d Compressed:%d Rawimage:%d\n", smartuse, compress, rawimage));

   //- Use optimal-for-OS size by default, allow upto maximum for IMZ (indexing limit)
   bs = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFIMZ);
   if (bs <= RBUFSECTORS)                       // fits in the standard I/O buffer
   {
      ibuf = rbuf;                              // always used with DOS!
   }
   else
   {
      ibuf = TxAlloc( bs, bps);                 // large buffer, non-DOS only!
   }
   if (compress)
   {
      cbuf = TxAlloc( bs, bps);                 // same size as i/o buffer
      ubuf = TxAlloc( bs, bps);                 // same size as i/o buffer
   }

   TRACES(("ibuf at: %p  cbuf at: %p\n", ibuf, cbuf));

   if ((ibuf != NULL) && ((compress == FALSE) || (cbuf != NULL)))
   {
      if ((rc = TxFileOpenForWrite( fspec, appendto, &fh)) == NO_ERROR)
      {
         if (rawimage == FALSE)                 // Add DFSee image header
         {
            time_t        now = time( &now);

            memset( &hdr, 0, sizeof(hdr));      // set to all ZEROES

            hdr.sequence = sequence;            // first file in sequence
            hdr.offset   = TXu64lo( fn);        // offset for this file
            hdr.hi_offs  = TXu64hi( fn);
            hdr.flags    = flags;               // imaging flags used
            hdr.sectors  = TXu64lo( size);      // total nr of sectors
            hdr.hi_sect  = TXu64hi( size);
            hdr.bufsize  = bps * bs;            // buffersize used
            hdr.objbase  = TXu64lo( dfstLSN2Psn(DFSTORE,0)); //- current base
            hdr.hi_base  = TXu64hi( dfstLSN2Psn(DFSTORE,0));
            hdr.objsize  = TXu64lo( dfsGetLogicalSize());    //- current size
            hdr.hi_size  = TXu64hi( dfsGetLogicalSize());
            hdr.bpsector = bps;                 // bytes per sector (8.09)

            memcpy( hdr.signature, DFSIM_SIGNATURE, DFSIM_SSIZE);

            strftime( text, TXMAXTM, "%Y-%m-%d %H:%M:%S", localtime( &now));
            sprintf( hdr.asciitext,
                     "%4.4u on %s\r\n%s; version %s %s\r\n%s\r\n%s\r\n"
                     "Base at : 0x0%llx Size:0x%llx "
                     "Offs:0x0%llx Sect:0x0%llx\r\nReg ",
                     sequence, text, DFS_N, DFS_V, DFS_C,
                     dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE),
                     dfstLSN2Psn(DFSTORE,0), dfsGetLogicalSize(), fn, size);

            #if defined (REGISTRATION)
               dfsRegistrationValid( NULL, text); // get reg status
            #else
               strcpy( text, "OEM licensed version");
            #endif
            strcat( hdr.asciitext, text);
            strcat( hdr.asciitext, "\r\n\032"); // add reginfo and EOF

            rc = TxWrite( fh, &hdr, sizeof(hdr), &handled);
            imgpos += handled;                  // for verification purposes
         }
         if ((compress) && (rc == NO_ERROR))
         {
            rc = dfsUlRegister( &ulh);          // start compression session
            if (rc == NO_ERROR)
            {
               if (hdr.bufsize < 65536)         // DOS/old-version compatible small buffer
               {
                  TRACES(("Write compressed-file header, small: %s\n", DFSUL_FILESIG));
                  rc = TxWrite( fh, DFSUL_FILESIG, DFSUL_SIG_LEN, &handled);
               }
               else                             // non-DOS or new-version large buffer
               {
                  TRACES(("Write compressed-file header, large: %s\n", DFSUL_FILEBIG));
                  rc = TxWrite( fh, DFSUL_FILEBIG, DFSUL_SIG_LEN, &handled);
               }
               if (rc == NO_ERROR)
               {
                  imgpos += handled;            // for verification purposes
                  sz = bs * bps;                // buffer size

                  if (hdr.bufsize < 65536)      // DOS/old-version compatible small buffer
                  {
                     ssize   = (USHORT) sz;     // write max buffer size
                     rc = TxWrite( fh, &ssize, sizeof(ssize), &handled);
                  }
                  else                          // non-DOS or new-version large buffer
                  {
                     rc = TxWrite( fh, &sz, sizeof(sz), &handled);
                  }
                  if (rc == NO_ERROR)
                  {
                     imgpos += handled;         // for verification purposes
                  }
               }
            }
         }
         if ((smartuse) && (rc == NO_ERROR))
         {
            if ((sectused = TxAlloc( bs, sizeof(BOOL))) == NULL)
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
         if (sequence == 1)                     // details on 1st file only
         {
            if (TxaOption( 'q') == FALSE)       // unless we want quiet
            {
               strcpy(    text, "");
               dfstrSize( text, "sectors = ", bs, ",");
               TxPrint("File compression  : %s     Buffering: %u %s %s-image\n",
                       (compress) ?  (hdr.bufsize < 65536) ? DFSUL_FILESIG : DFSUL_FILEBIG : "none  ", bs, text,
                       (smartuse) ? "Smart" : "Full");
            }
         }
         if ((dfsa->verbosity != TXAO_QUIET) && (nopritem == FALSE)) // want progress
         {
            if (sequence == 1)                  // first file
            {
               dfsProgressInit( fn, size, (smartuse) ? dfsa->FsUnallocSmart : 0,
                                "Sector:", "imaged", DFSP_BARS, DFSIM_SMART_FAST_FACTOR);

               dfsProgressShow( 0, 0, 0, NULL); // 0 of 0 done (if first is bad :-)
            }
            else                                // not 1st, signal new file
            {
               sprintf( text, "File %3.3u", sequence);
               dfsProgressResume( text);
            }
         }

         TRACES(("usecrc32: %s   fullbuf2: %s\n", (usecrc32) ? "TRUE" : "FALSE", (fullbuf2) ? "TRUE" : "FALSE"));

         for ( i = 0,        sn  = fn;
              (i < size) && (rc == NO_ERROR);
               i += bs,      sn += bs)
         {
            if ((size - i) < bs)                // if almost at end
            {
               bs = size - i;                   // remaining sectors
            }
            if (smartuse)
            {
               for (bufusage = 0, j = 0; j < bs; j++)
               {
                  sectused[j] = (DFSFNCALL( dfsa->FsLsnAllocated, fn + i + j, 0, NULL, NULL) != 0);
                  if (sectused[j] != FALSE)
                  {
                     bufusage++;                // count number of sectors
                  }                             // that are really in-use
               }
               TRACES(("Sector: 0x%llx bufusage %u out of %u\n", i, bufusage, bs));
            }
            if ((smartuse) && (bufusage == 0))  // skip reading (performance)
            {
               //- only count these completely unused buffers as 'fast smart' area
               //- since these will not be read from disk, and not compressed either.

               *totalSkipped += bs;

               memset( ibuf, 0, bs * bps);      // ZERO whole buffer
               RleCompr = TRUE;
               RleValue = (fullbuf2) ? DFSIM_SMARTFB : (usecrc32) ? DFSIM_SMARTCRC : DFSIM_SMARTBUF;
            }
            else
            {
               RleCompr = FALSE;
               rc = dfstBadSectorRead( DFSTORE, sn, bs, NULL, &bads, ibuf);
               if (rc == NO_ERROR)
               {
                  #if defined (NEVER)
                  //- Do NOT zero, keep original 'garbage' to make sure compare will work
                  //- correctly, even when buffer overlaps alloc-Area's and the compare
                  //- can not reliably determine allocation for each sector
                  if (smartuse)
                  {
                     for (j = 0; j < bs; j++)
                     {
                        if (sectused[j] == FALSE)  //- not allocated, ZERO sector
                        {                          //- to achieve better compression
                           memset( ibuf + (j * bps), 0, bps);
                        }
                     }
                  }
                  #endif
               }
               else                             // some read error
               {
                  sprintf( text, "Read  error on %5u sector(s) at sector 0x%16.16llx.", bads, sn);
                  switch (dfsa->eStrategy)
                  {
                     case TXAE_QUIT:            // no action, keep bad rc
                        TxPrint( "%s  Aborting ...\n", text);
                        break;

                     case TXAE_CONFIRM:
                        if (dfsa->batch)        // CONFIRM not possible, but do NOT ignore errors!
                        {
                           TxPrint( "%s  Aborting (batch) ...\n", text);
                        }
                        else
                        {
                           dfsProgressSuspend(); // allow regular screen output
                           if (TxConfirm( 5211, "%s  Do you want to write dummy-data (0xFE pattern) "
                                                "to the image and continue ? [Y/N]: ", text))
                           {
                              if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                              {
                                 dfsa->eStrategy = TXAE_IGNORE;
                                 sprintf( text, "Ignoring further errors, continue saving, file %u", sequence);
                              }
                              else
                              {
                                 sprintf( text, "Continue saving, file %u", sequence);
                              }
                              dfsProgressResume( text);
                              rc = NO_ERROR;
                           }
                        }
                        break;

                     case TXAE_IGNORE:
                     default:                   // ignore the error
                        TxPrint( "%s  Ignore, continue ...\n", text);
                        rc = NO_ERROR;
                        break;
                  }
                  *failed += bs;
               }
            }

            if (rc == NO_ERROR)
            {
               sz = bs * bps;                   // default whole buffer size
               if (compress)
               {
                  if (usecrc32)
                  {
                     if (RleCompr)
                     {
                        crc32 = DFSIM_FIXEDCRC; // Use fixed value, guards the
                     }                          // SMART type/length value
                     else
                     {
                        crc32 = TxCrc32( ibuf, sz); // over uncompressed data area
                     }
                  }
                  else
                  {
                     crc32 = 0;                 // for -v display :-)
                  }
                  if (RleCompr == FALSE)        // not zeroed yet (smart)
                  {
                     RleCompr  = dfsRleCompressable( ibuf, bs * bps);
                     RleValue  = (fullbuf2) ? DFSIM_FULLBUF : (usecrc32) ? DFSIM_RLEBCRC : DFSIM_RLEBYTE;
                     RleValue |= *ibuf;
                  }
                  if (RleCompr)                 // RLE compressable now ?
                  {
                     info = RleValue;
                     TRACES(("Write RLE %4.4hx compressed-block starting at %llx\n", info, i));
                     rc = TxWrite( fh, &info, sizeof(info), &handled);
                     if (rc == NO_ERROR)
                     {
                        imgpos += handled;      // for verification purposes
                        ssize   = sz;

                        if (fullbuf2 == FALSE)  // don't write length for FULLBUF
                        {
                           TRACES(("Write RLE/SMART compressed-block header, bufsize: 0x%x\n", sz));

                           //- to be refined, should be 32-bit when hdr.bufsize > 65535

                           if (hdr.bufsize < 65536) // DOS/old-version compatible small buffer
                           {
                              rc = TxWrite( fh, &ssize, sizeof(ssize), &handled);
                           }
                           else                 // non-DOS or new-version large buffer
                           {
                              rc = TxWrite( fh, &sz, sizeof(sz), &handled);
                           }
                           if (rc == NO_ERROR)
                           {
                              imgpos += handled; // for verification purposes
                           }
                        }
                     }
                     sz      = 0;               // no real data to write
                  }
                  else                          // need real LZW compression
                  {
                     TRACES(( "CRC32 over %u bytes: %8.8x\n", sz, TxCrc32( ibuf, sz)));
                     TRHEXS( 800,  ibuf, sz, "uncompressed data IN, before compress");
                     rc = dfsUlBufCompress( ulh, DFSUL_MLZW_12, ibuf, sz, cbuf, sz, &csize);
                     if (rc == NO_ERROR)        // compressed OK
                     {
                        ULONG           vResult;
                        ULONG           vCrc32;

                        TRACES(( "CRC32 over %u bytes: %8.8x\n", sz, TxCrc32( cbuf, sz)));
                        TRHEXS( 800,  cbuf, csize, "compressed data OUT, after compress");

                        //- Verify decompress to work OK, catch decode-stack error; (P#1816)
                        //- and another 'uncompressed size too large' error         (P#1839)
                        vResult = dfsUlBufUncompress( ulh,  DFSUL_MLZW_12, cbuf, csize, ubuf, sz, &usize);
                        vCrc32  = TxCrc32( ubuf, sz); // verification CRC, check against input buffer
                        TRACES(("rc: %u  sz: %u  usize: %u  crc:%8.8x\n", vResult, sz, usize, vCrc32));
                        if ((vResult == NO_ERROR) && (usize == sz) && (vCrc32 == crc32))
                        {
                           data  = cbuf;        // compressed data
                           sz    = csize;       // compressed size
                           info  = (usecrc32) ? DFSIM_LZW_CRC : DFSUL_MLZW_12;
                        }
                        else                    // Decode-stack overflow, or size mismatch
                        {
                           //- Keep count of errors for the whole session
                           if (usize > sz)      // will be SMALLER on a decode-stack overflow
                           {
                              iosizeErrors++;   // size mismatch on uncompress
                           }
                           else if (vCrc32 != crc32)
                           {
                              verifyErrors++;   // crc error on uncompressed data
                           }
                           else
                           {
                              decodeErrors++;   // must be decode stack overflow
                           }
                           data  = ibuf;        // write uncompressed
                           info  = (usecrc32) ? DFSIM_NON_CRC : DFSIM_NOCOMPR;
                           TRACES(("Verify decompress failed!\n"));
                        }
                     }
                     else                       // no compression possible
                     {
                        data  = ibuf;           // write uncompressed
                        info  = (usecrc32) ? DFSIM_NON_CRC : DFSIM_NOCOMPR;
                     }
                     TRACES(("Write IMZ compressed-block header, info: %4.4hx\n", info));
                     rc = TxWrite( fh, &info, sizeof(info), &handled);
                     if (rc == NO_ERROR)
                     {
                        imgpos += handled;      // for verification purposes

                        if (hdr.bufsize < 65536) // DOS/old-version compatible small buffer
                        {
                           ssize   = (USHORT) sz; // write block-length
                           TRACES(("Write IMZ compressed-block header, SMALL buffer: 0x%4.4hx\n", ssize));
                           rc = TxWrite( fh, &ssize, sizeof(ssize), &handled);
                        }
                        else                    // non-DOS or new-version large buffer
                        {
                           TRACES(("Write IMZ compressed-block header, LARGE buffer: 0x%x\n", sz));
                           rc = TxWrite( fh, &sz, sizeof(sz), &handled);
                        }
                        if (rc == NO_ERROR)
                        {
                           imgpos += handled;   // for verification purposes
                        }
                     }
                  }

                  //- write CRC32 for all types except RLE/SMART FULLBUF, when CRCs are enabled
                  if (usecrc32 && (rc == NO_ERROR) && ((RleCompr == FALSE) || (fullbuf2 == FALSE)))
                  {
                     TRACES(("TxWrite write 4 bytes CRC: %8.8X  at imgpos:%llx\n", crc32, imgpos));
                     rc = TxWrite( fh, &crc32, sizeof(crc32), &handled);
                     if (rc == NO_ERROR)
                     {
                        imgpos += handled;      // for verification purposes
                     }
                  }
               }
               else                             // RAW image write ...
               {
                  info = DFSIM_NOCOMPR;
                  sz   = bs * bps;
                  data = ibuf;
                  crc32 = 0;
               }
               if (verbose)
               {
                  ULONG  l = sz + 4;

                  if (usecrc32)
                  {
                     l += 4;
                  }
                  TxPrint( "\nS#:%7u=%11.11llx LSN:%12.12llx; %s L:%6.6x=% 8u CRC:%8.8x ",
                              ++blocknr, imgpos, sn, dfsImageCode2String( info), l, l, crc32);
               }
               if (sz != 0)                     // data to write ?
               {
                  TRACES(("TxWrite write %u bytes from %8.8X\n", sz, data));
                  rc = TxWrite( fh, data, sz, &handled);
                  if (rc == NO_ERROR)
                  {
                     imgpos += handled;         // for verification purposes
                  }
               }
               else                             // no data
               {
                  handled = 0;
               }
               if (rc == NO_ERROR)
               {
                  if (handled == sz)
                  {
                     if ((dfsa->verbosity != TXAO_QUIET) && (nopritem == FALSE))
                     {
                        strcpy( text, "");
                        dfstr64XiB( text, (compress) ? "IMZ " : "IMG ", *filesize + imgpos, "");
                        dfsProgressShow( sn + bs, *totalSkipped, DFSP_STRONLY, text);
                     }
                  }
                  else
                  {
                     ULONG   sectorsWritten = (ULONG) (imgpos / bps);

                     dfsProgressSuspend();      // allow regular screen output

                     sprintf( text, "Error writing to '%s', written %u bytes ", fspec, (ULONG) handled);
                     TxPrint( "%s", dfstr64XiB( text, "after ", imgpos, "\n\nThe "));

                     if (((sectorsWritten)      <= DFSECT_2GIB) &&
                         ((sectorsWritten + bs) >  DFSECT_2GIB)  )
                     {
                        TxPrint( "destination filesystem may not "
                                 "support files larger than 2 GiB!\n"
                                 "Use the '-m' option to automatically"
                                 "split into multiple imagefiles.\nOr, the ");
                     }
                     else if (((sectorsWritten)      <= DFSECT_4GIB) &&
                              ((sectorsWritten + bs) >  DFSECT_4GIB)  )
                     {
                        TxPrint( "destination filesystem may not "
                                 "support files larger than 4 GiB!\n"
                                 "Use the '-m' option to automatically"
                                 "split into multiple imagefiles.\nOr, the ");
                     }
                     TxPrint( "destination drive may be full, out of diskspace!\n");
                     rc = ERROR_WRITE_FAULT;
                  }
               }
            }
            TRACES(( "imgpos:%11.11Lx  Check limit:%11.11x\n", imgpos, limit));
            if ( (limit != 0) &&   ((i + bs)  < size) && // limit set & exceeded
                ((imgpos + (bs * bps)) >= limit)) // but not finished yet
            {
               if (compress)
               {
                  TRACES(("Write multiple-compressed-file NEXTFILE marker\n"));
                  RleValue = DFSIM_NEXTFILE;    // signal RLE nextfile
                  rc = TxWrite( fh, &RleValue, sizeof(RleValue), &handled);
                  imgpos += handled;            // for verification purposes
                  info = (USHORT) sequence;     // write current file nr
                  rc = TxWrite( fh, &info, sizeof(info), &handled);
                  imgpos += handled;            // for verification purposes
               }
               rc = DFS_PENDING;                // OK, but start new file
            }

            if (TxAbort())                      // user abort/escape requested
            {
               TXTM s0;                         // text buffer

               sprintf(   s0, "Imaged sofar is : %4.1lf %%", ((double) 100.0 *
                                          (double) (i) / (double) (size)));
               dfstrSz64( s0, " = ", i,  "\n\n");

               dfsProgressSuspend();            // allow regular screen output
               if ((dfsa->batch) || (TxConfirm( 5219,
                   "%sAborting a imaging operation will leave an incomplete"
                   " imagefile, NOT very useful for restore or browsing!\n\n"
                   "Are you sure you want to ABORT the imaging ? [Y/N] : ", s0)))
               {
                  rc = DFS_USER_ABORT;
               }
               else
               {
                  TxCancelAbort();
                  dfsProgressResume( "");
               }
            }
         }
         if (rc == DFS_PENDING)
         {
            dfsProgressSuspend();               // allow regular screen output
         }
         else
         {                                      // finish the progress BAR and
            dfsProgressTerm();                  // allow regular screen output
         }
         if (compress)
         {
            if ((decodeErrors != 0) || (iosizeErrors != 0) || (verifyErrors != 0))
            {
               TxPrint( "No compression on : %u buffers (stack:%u size:%u crc:%u) on auto-verify!\n",
                         decodeErrors + iosizeErrors + verifyErrors, decodeErrors, iosizeErrors, verifyErrors);
            }
            dfsUlTerminate( ulh);               // end compression session
         }
         if (smartuse)
         {
            TxFreeMem( sectused);               // free smart-usage array
         }
         if ((rc == DFS_PSN_LIMIT) && (sn != fn)) // reached the limit
         {
            rc = NO_ERROR;
         }
         *filesize += imgpos;                   // total size written
         *lastsn    = sn -1;

         TxClose( fh);
      }
      if (ibuf != rbuf)                         // when not existing buffer
      {
         TxFreeMem( ibuf);
      }
      if (compress)
      {
         TxFreeMem( cbuf);
         TxFreeMem( ubuf);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(( "Total imagesize:%11.11Lx  thisfile:%10.10u  lastsn:%llx\n", *filesize, imgpos, *lastsn));
   RETURN(rc);
}                                               // end 'dfsSaveSingleImageFile'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS write sectors to volume/partition from saved imagefile(s)
/*****************************************************************************/
ULONG dfsRestoreImage
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    nr of sectors to write
   ULONG               flags,                   // IN    imaging options
   char               *fspec,                   // IN    source filespec
   ULONG               fileCount                // IN    #files, same as last file#
)
{
   ULONG               rc = NO_ERROR;
   ULONG               filenr   = 1;            // file sequence number
   ULN64               total    = size;         // total size to be imaged
   ULN64               todo     = total;        // max size todo for next file
   ULN64               start    = fn;           // next sector to start with
   ULN64               last     = 0;            // last sector done
   ULN64               startimz = fn;           // startsector, from IMZ hdr
   ULN64               fsize    = 0;            // total filesize read sofar
   TXLN                info;                    // addional conform info
   TXLN                line;                    // filename and text buffer
   TXTS                fext;                    // file extension string
   TXTIMER             stime = TxTmrGetNanoSecFromStart();

   ENTER();

   if (TxaOption( 'q') == FALSE)                // unless we want quiet
   {
      dfsX10( "\nImage startsector : ", fn, "", " ");
      dfsX10( "size ", size, "", " from ");
      TxPrint( "%s\n", fspec);
   }

   TxFileSize( fspec, &fsize);                  // pass filesize TODO to first restore

   strcpy( line, fspec);                        // first filename
   do
   {
      rc = dfsRestoreSingleFile( start, todo, flags, line, filenr, fileCount,
                                 &fsize, &startimz, &last, &total);
      if (rc == DFS_PENDING)
      {
         if (strstr( line, ".img") != NULL)     // RAW image, add some info
         {
            strcpy( info, "Either the disk/partition this image is restored to "
                          "is LARGER than the original (and the image), or this "
                          "imagefile was the first of a multi-file image set.  "
                          "When restoring to a larger partition, you can simply "
                          "[Cancel] the search for additional imagefiles ..\n\n   ");
         }
         else
         {
            strcpy( info, "Imagefile: ");
         }
         start = last +1;                       // start sector next file
         todo  = fn + total - start;            // number of sectors todo
         sprintf( fext, ".%3.3u", ++filenr);    // new file number/extension
         TxStripExtension( line);
         strcat( line, fext);                   // new full filename

         if (!dfsChangeReadMedium( info, line)) // change removable for read
         {
            rc = ERROR_FILE_NOT_FOUND;          // Canceled or not found
         }
      }
   } while ((rc == DFS_PENDING) && !TxAbort()); // more files to be done

   dfsProgressTerm();                           // probably done, but better be safe!

   if ((rc != DFS_CMD_FAILED)    &&             // pipe client failure
       (rc != DFS_BAD_STRUCTURE) &&             // buffersize mismatch
       (rc != DFS_ST_MISMATCH)   &&             // sectorsize mismatch
       (TxaOption( 'q') == FALSE) )             // unless we want quiet
   {
      double           elapsed;

      strcpy(     line, "");
      dfstrX10(   line, "Last sector done  : ", last, CNN, " ");
      dfstrSz64(  line, "size ",   last - startimz +1, " ");
      dfstr64XiB( line, "from ", fsize, " in");
      if (filenr > 1)
      {
         TxPrint( "%s %u files\n", line, filenr);
      }
      else                                      // keep output compact :-)
      {
         TxPrint( "%s 1 file\n", line);
      }
      elapsed = dfsDisplayThroughput( stime, last - fn +1, dfsGetSectorSize());
      dfsDisByteThroughput( elapsed, fsize, "Image-file  read speed");
   }
   RETURN(rc);
}                                               // end 'dfsRestoreImage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS restore sectors to volume/partition from a previously saved image
/*****************************************************************************/
static ULONG dfsRestoreSingleFile
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    nr of sectors to write
   ULONG               flags,                   // IN    imaging options
   char               *fspec,                   // IN    source filespec
   ULONG               sequence,                // IN    file sequence number
   ULONG               fileCount,               // IN    #files, same as last file#
   ULN64              *filepos,                 // INOUT imagefile size read
   ULN64              *startsn,                 // OUT   real start, from IMZ
   ULN64              *lastsn,                  // OUT   last sectornr handled
   ULN64              *totalsize                // OUT   total size  (IMZ header)
)                                               //       not changed otherwise.
{
   ULONG               rc = NO_ERROR;
   TXHFILE             fh = 0;                  // filehandle
   ULN64               i;                       // number of sectors
   ULN64               sn;                      // Data LSN
   ULONG               sz;                      // nr of bytes to handle
   ULONG               handled;                 // nr of bytes handled
   ULONG               bs   = 0;                // sectors to r/w/compress
   BYTE               *ibuf  = NULL;            // imaging disk i/o buffer
   BYTE               *cbuf  = NULL;            // compression buffer
   ULONG               error_blocks = 0;        // nr of blocks in error
   BOOL                compress = FALSE;        // auto-detected uncompress
   BOOL                test     = (flags & DFSOPT_TEST);
   BOOL                compare  = (flags & DFSOPT_COMPARE);
   BOOL                verbose  = (flags & DFSOPT_VERBOSE);
   ULN64               sizetodo = size;         // size to be restored
   ULN64               startsec = fn;           // start sector for restore
   BOOL                smartuse = FALSE;        // derive from image header
   ULN64               imgpos   = 0;            // file-pos in image (progress)
   USHORT              ssize;                   // 16-bit size value
   ULONG               uncompressed;            // uncompressed IMZ buffer
   USHORT              info;                    // compress info
   USHORT              bps = dfsGetSectorSize();
   ULONG               csize;
   ULONG               crc32;
   BOOL                usecrc32 = FALSE;        // CRC32 value present ?
   ULONG               blocknr = 0;             // block-sequence nr in file
   DFSUL_HANDLE        ulh;
   ULONG               rlen;                    // read-length
   TXLN                text;                    // text buffer
   DFSIM_HEADER        hdr;                     // DFSee image header

   ENTER();

   //- Allocate maximum buffer by default, compatible with as much as possible images
   //- a used -b option is used for read sizes only, not memory allocation
   bs = DFSMAXBUFIMZ;                           // ALWAYS get the maximum buffer
   if (bs <= RBUFSECTORS)                       // fits in the standard I/O buffer
   {
      ibuf = rbuf;                              // always used with DOS!
   }
   else
   {
      ibuf = TxAlloc( bs, bps);                 // large buffer, non-DOS only!
   }
   cbuf = TxAlloc( bs, bps);                    // same size as i/o buffer, always needed

   TRACES(("ibuf at: %p  cbuf at: %p\n", ibuf, cbuf));

   if ((ibuf != NULL) && (cbuf != NULL))
   {
      sz   = bs * bps;                          // default buffersize

      if ((rc = TxFileOpenReadOnly( fspec, &fh)) == NO_ERROR)
      {
         memset( &hdr, 0, sizeof(hdr));         // set to all ZEROES

         TRACES(("Check for DFSee-specific header and/or compression\n"));

         rc = TxRead( fh, &hdr, sizeof(hdr), &handled);
         if (memcmp( hdr.signature, DFSIM_SIGNATURE, DFSIM_SSIZE) == 0)
         {
            imgpos  += handled;                 // for verification purposes
            smartuse = ((hdr.flags & DFSI_SMARTUSE) == DFSI_SMARTUSE);

            if (hdr.bpsector == 0)              // older header version
            {
               hdr.bpsector = SECTORSIZE;       // set to fixed 512
            }
            if (sequence == 1)                  // first file of a set
            {
               TxRepl( hdr.asciitext, '\r', 0); // terminate after 1st line
               if (TxaOption( 'q') == FALSE)    // unless we want quiet
               {
                  TxPrint( "DFSee imagefile # : %s\n", hdr.asciitext);
               }
               if (flags & DFSW_IMZ_SIZE)       // take size/start from header
               {
                  startsec   = TXmku64( hdr.offset, hdr.hi_offs);  //- adjust restore startsector
                  *startsn   = startsec;                           //- for correct reporting ...
                  sizetodo   = TXmku64( hdr.sectors, hdr.hi_sect); //- adjust internal total size
                  *totalsize = sizetodo;                           //- and for the caller too ...

                  strcpy( text, (test) ? (compare) ? "Size to compare   : " :
                                                     "Size to verify    : " :
                                                     "Size to restore   : ");
                  if ((startsec + sizetodo) > (dfsGetLogicalSize()))
                  {
                     ULN64 base = dfstLSN2Psn( DFSTORE, 0);

                     switch (dfstStoreType(DFSTORE))
                     {
                        case DFST_RAWIMAGE:     // update base values
                           dfstLogicalLimits( DFSTORE, base, base + startsec + sizetodo -1);
                           dfsSz64(  "Expected RAW size : ",           sizetodo,     " (from image file)\n");
                           break;

                        default:
                           dfsSz64Bps( text,             sizetodo, hdr.bpsector,     " (from image file)\n");
                           dfsSz64( "Truncated to      : ",  dfsGetLogicalSize(),    " (object size)\n");
                           break;
                     }
                  }
                  else                          // fits in existing object size
                  {
                     if (TxaOption( 'q') == FALSE) // unless we want quiet
                     {
                        dfsSz64( text,                      sizetodo,             "    (from image file)\n");
                     }
                  }
                  if (TxaOption( 'q') == FALSE) // unless we want quiet
                  {
                     dfsX10( "To first sector   : ", startsec, CBY, "                  (from image file)\n");
                  }
               }
            }
            TRACES(("File# : %u  start:0x%llx remaining-size:0x%llx  Smart:%s\n",
                     hdr.sequence, startsec, sizetodo, (smartuse) ? "YES" : "NO"));
         }
         else                                   // no header, rewind file
         {
            TxFileSeek( fh, 0, SEEK_SET);
         }

         TRACES(("Check for compression header\n"));
         rlen = DFSUL_SIG_LEN;
         rc = TxRead( fh, ibuf, rlen, &handled); // read-ahead rlen bytes!
         TRACES(( "TxRead rc:%u handled: %u sig: '%6.6s'\n", rc, handled, ibuf));
         if (rc == NO_ERROR)
         {
            ibuf[ DFSUL_SIG_LEN] = 0;           // null-terminate the string!
            imgpos += handled;                  // for verification purposes
            if ((memcmp(ibuf, DFSUL_FILESIG, DFSUL_SIG_LEN) == 0) ||
                (memcmp(ibuf, DFSUL_FILEBIG, DFSUL_SIG_LEN) == 0)  )
            {
               if (hdr.bufsize < 65536)         // DOS/old-version compatible small buffer
               {
                  rc = TxRead( fh, &ssize, sizeof(ssize), &handled);
                  uncompressed = ssize;
               }
               else                             // non-DOS or new-version large buffer
               {
                  rc = TxRead( fh, &uncompressed, sizeof(uncompressed), &handled);
               }

               //- to be refined, abort if read uncompressed size larger than hdr.bufsize (DOS) ?

               TRACES(( "TxRead rc:%u handled: %u   bufsize: 0x%x\n", rc, handled, uncompressed));
               if (rc == NO_ERROR)
               {
                  imgpos += handled;            // for verification purposes
                  if (uncompressed > sz)        // almost impossible now that we always allocate 2 MiB!
                  {
                     static BOOL alreadyConfirmed = FALSE;

                     sprintf( text, "Buffersize in the IMZ : %u, is larger than current destination buffer : %u\n", uncompressed, sz);

                     if ((dfsa->batch) || (dfsa->expertui == FALSE) || alreadyConfirmed ||
                         (TxConfirm( 5213, // batch, BASIC mode, already confirmed, or confirmed now
                         "%s\nForce buffersize to match the IMZ? [Y/N]: ", text)))
                     {
                        TxPrint( "\nBuffersize set to : %u bytes to match buffer in IMZ image\n", uncompressed);

                        alreadyConfirmed = TRUE; // don't ask again
                        bs = uncompressed / bps;
                        sz = uncompressed;
                     }
                     else
                     {
                        TxPrint( "\nBuffersize in IMZ : %u is larger than current buffer: %u\n"
                                   "Adjust target geometry before restoring, or use the -b:%u option.\n\n",
                                    uncompressed, sz, (uncompressed / bps));
                        rc = DFS_BAD_STRUCTURE; // too large, abort now
                     }
                  }
                  if (rc == NO_ERROR)
                  {
                     rc = dfsUlRegister( &ulh); // start compression session
                     if (rc == NO_ERROR)
                     {
                        compress = TRUE;        // valid compress header
                     }
                  }
               }
            }
         }
         if (compress)
         {
            if (hdr.bpsector != bps)            // sectorsize mismatch
            {
               TxPrint( "\nSectorsize in IMZ : %u does not match current sectorsize: %hu\n"
                          "Restore of the image is not possible!\n\n", hdr.bpsector, bps);
               rc = DFS_ST_MISMATCH;
            }
         }
         else                                   // uncompressed (.img)
         {
            uncompressed = bs * bps;            // buffersize used (display)
         }
      }
      if (rc == NO_ERROR)                       // OK sofar (rlen bytes read!)
      {
         ULN64  lastw = startsec;

         if (sequence == 1)
         {
            if (TxaOption( 'q') == FALSE)       // unless we want quiet
            {
               TxPrint("File compression  : %s    %sAvailable buffer: %u, in image: %u sectors.\n",
                       (compress) ?  (char *) ibuf             : "none  ",
                       (test)     ? "Test mode, no write. " : "", DFSMAXBUFIMZ, uncompressed / bps);
            }
         }
         if (dfsa->verbosity != TXAO_QUIET)     // we want progress info
         {
            if (sequence == 1)                  // the first file (IMZ)
            {
               dfsProgressInit( 0, *filepos / 1024, 0, "KByte:",
                                (test) ? (compare) ? "compared" : "verified" : "restored",
                                DFSP_BARS | DFSP_VDEC, 0);

               TRACES(("Start progress on first file\n"));
               dfsProgressShow( 0, 0, 0, NULL);

               *filepos = 0;                    // size handled sofar!
            }
            else
            {
               TRACES(("Resume progress on file %u\n", sequence));
               sprintf( text, "File %3.3u", sequence);
               dfsProgressResume( text);
            }
         }

         if (compress)                          // read, uncompress, write
         {                                      // by compressed-block size
            ULONG         ws  = bs;             // uncompressed sectors


            TRACES(("Start of decompress loop, ws == bs = %u", ws));

            for ( i = 0,   sn  = startsec;
                 (i < sizetodo) && (rc == NO_ERROR) && (!TxAbort());
                  i += ws, sn += ws)
            {
               if ((sizetodo - i) < bs)         // if almost at end
               {
                  bs = sizetodo - i;            // remaining (max) sectors
               }
               rc = TxRead( fh, &info, sizeof(info), &handled);
               TRACES(( "TxRead rc:%u handled: %u first: %4.4hx\n", rc, handled, info));
               if (rc == NO_ERROR)              // read compression method OK
               {
                  if (handled == sizeof(info))  // size is correct
                  {
                     imgpos += handled;         // for verification purposes
                     if (((info & 0xff00) != DFSIM_FULLBUF) && // RLE or SMART full-buffer
                          (info           != DFSIM_SMARTFB)  ) // has NO length field, no CRC!
                     {
                        if (hdr.bufsize < 65536) // DOS/old-version compatible small buffer
                        {
                           rc = TxRead( fh, &ssize, sizeof(ssize), &handled);
                           rlen = ssize;
                        }
                        else                    // non-DOS or new-version large buffer
                        {
                           rc = TxRead( fh, &rlen, sizeof(rlen), &handled);
                        }
                        if (rc == NO_ERROR)
                        {
                           imgpos += handled;   // for verification purposes
                        }
                        TRACES(( "TxRead rc:%u handled: %u rlen : %8.8x\n", rc, handled, rlen));
                     }
                     else                       // need correct rlen set for SMART (buffer size)
                     {
                        rlen = uncompressed;
                     }
                     if (rc == NO_ERROR)
                     {
                        if (( info           == DFSIM_LZW_CRC)  ||
                            ( info           == DFSIM_NON_CRC)  ||
                            ( info           == DFSIM_SMARTCRC) ||
                            ((info & 0xff00) == DFSIM_RLEBCRC)   )
                        {
                           usecrc32 = TRUE;
                           rc = TxRead( fh, &crc32, sizeof(crc32), &handled);
                           TRACES(( "TxRead rc:%u handled: %u crc32: %8.8x\n", rc, handled, crc32));
                           if (rc == NO_ERROR)
                           {
                              imgpos += handled; // for verification purposes
                           }
                        }
                        else
                        {
                           usecrc32 = FALSE;
                           crc32   = 0;
                        }
                     }
                  }
                  else
                  {
                     rc = DFS_BAD_STRUCTURE;    // imagefile too short
                  }
               }
               if (rc == NO_ERROR)              // compressed block-size read OK
               {
                  if (verbose)
                  {
                     ULONG   l = 4;             // minimum size, type + len

                     if (info < DFSIM_SMARTBUF) // NO RLE type block, read data
                     {
                        l += rlen;              // add real-data length
                     }
                     if (usecrc32)
                     {
                        l += 4;                 // add CRC length
                     }
                     TxPrint( "\nS#:%7u=%11.11llx LSN:%12.12llx; %s L:%6.6x=% 8u CRC:%8.8x ",
                                 ++blocknr, imgpos, sn, dfsImageCode2String( info), l, l, crc32);
                  }
                  if (info < DFSIM_SMARTBUF)    // NO RLE type block, read data
                  {
                     if ((info == DFSUL_MLZW_12) || // LZW compr, regular
                         (info == DFSIM_LZW_CRC) || // LZW compr, CRC32
                         (info == uncompressed)  || // NON compr, missing 2 bytes
                         (info == DFSIM_NON_CRC) || // NON compr, including CRC32
                         (info == DFSIM_NOCOMPR)  ) // NON compr, OK or missing 1
                     {
                        if ((  info == DFSIM_NOCOMPR) && // test 'missing-bytes' bug P#928
                            (  rlen               != uncompressed) &&
                            (((rlen & 0xff) << 8) == uncompressed)  )
                        {
                           dfsProgressSuspend(); // allow regular screen output
                           TxPrint( "Missing 1 byte  in NON-compressed data near LSN: 0x0%llx, recovering;\n", sn);
                           rlen = uncompressed;
                           TxFileSeek( fh, -1, SEEK_CUR);
                           imgpos -= 1;         // for verification purposes
                           error_blocks++;
                           dfsProgressResume( "");
                        }
                        else if (info == uncompressed)
                        {
                           dfsProgressSuspend(); // allow regular screen output
                           TxPrint( "Missing 2 bytes in NON-compressed data near LSN: 0x0%llx, recovering;\n", sn);
                           rlen = uncompressed; // assume full buffer size
                           info = DFSIM_NOCOMPR; // reset to uncompressed type
                           TxFileSeek( fh, -2, SEEK_CUR);
                           imgpos -= 2;         // for verification purposes
                           error_blocks++;
                           dfsProgressResume( "");
                        }
                        if (rc == NO_ERROR)
                        {
                           if (rlen <= uncompressed) // not too large ?
                           {
                              TRACES(("About to read 0x%8.8x = %u bytes to cbuf: %p\n", rlen, rlen, cbuf));
                              rc = TxRead( fh, cbuf, rlen, &handled);
                              TRACES(( "\nTxRead rc:%u handled: %u\n", rc, handled));
                              imgpos += handled; // for verification purposes
                           }
                           else
                           {
                              TxPrint( "\nInvalid data-length %u, compressed image seems damaged.\n", rlen);
                              rc = DFS_CMD_FAILED;
                           }
                        }
                     }
                     else
                     {
                        TxPrint( "\nInvalid data-type %4.4hx, compressed image seems damaged.\n", info);
                        rc = DFS_CMD_FAILED;
                     }
                  }
                  else
                  {
                     handled = rlen;
                  }
                  if (rc == NO_ERROR)
                  {
                     if (rlen == handled)
                     {
                        BYTE    *data = NULL;   // ptr to data to write

                        TRACES(("Data compression type: %4.4hx = '%s'\n", info, dfsImageCode2String( info)));
                        if ((info != DFSIM_NOCOMPR) && // was it compressed ?
                            (info != DFSIM_NON_CRC)  )
                        {
                           if (((info & 0xff00) == DFSIM_RLEBYTE) ||
                               ((info & 0xff00) == DFSIM_RLEBCRC) ||
                               ((info & 0xff00) == DFSIM_FULLBUF)  )
                           {
                              csize = rlen;     // uncompressed size
                              memset( ibuf, (info & 0xff), csize);
                              data  = ibuf;
                              TRACES(("Handle RLE '%4.4hx' coded block, csize: %u bytes\n", info, csize));
                           }
                           else if ((info == DFSIM_SMARTBUF) ||
                                    (info == DFSIM_SMARTCRC) ||
                                    (info == DFSIM_SMARTFB )  )
                           {
                              data  = NULL;     // nothing to write
                              csize = rlen;     // uncompressed size of zeroes
                              TRACES(("Handle SMART '%4.4hx' coded block, csize: %u bytes\n", info, csize));
                           }
                           else if (info == DFSIM_NEXTFILE)
                           {
                              dfsProgressSuspend(); // allow regular screen output
                              rc = DFS_PENDING;
                           }
                           else                 // LZW compression used
                           {
                              rc = dfsUlBufUncompress( ulh,
                                                       (ULONG)  DFSUL_METHOD(info),
                                                       (BYTE *) cbuf, rlen,
                                                       (BYTE *) ibuf, sz,
                                                       &csize);
                              data = ibuf;
                              if ((rc == TX_ABORTED) || (csize > sz)) // possible decode or size error
                              {
                                 BYTE *cdata = cbuf;

                                 dfsProgressSuspend(); // allow regular screen output
                                 if ((cdata[rlen -1] == DFSUL_MLZW_12) ||
                                     (cdata[rlen -1] == DFSIM_LZW_CRC)  )
                                 {  //- check if last byte was our info-word
                                    //- seek -1, bypassing 'missing byte' bug P#928
                                    TxPrint( "Missing 1 byte  in LZW-compressed data near LSN: 0x0%llx, recovering;\n", sn);
                                    TxFileSeek( fh, -1, SEEK_CUR);
                                    imgpos -= 1; // for verification purposes
                                    error_blocks++;
                                    dfsProgressResume("");
                                    rc = NO_ERROR;
                                 }
                                 else
                                 {
                                    TxPrint( "Invalid values in LZW-compressed  data near LSN: 0x0%llx, non-recoverable!\n", sn);
                                 }
                              }
                           }
                        }
                        else                    // just write out the buffer
                        {
                           TRACES(("Data chunk was NOT compressed!\n"));
                           data  = cbuf;
                           csize = rlen;        // use whole length read
                        }
                        if (usecrc32)
                        {
                           ULONG check = DFSIM_FIXEDCRC; // default for Smart

                           if (data != NULL)    // FIXEDCRC, SMART
                           {
                              check = TxCrc32( data, csize); // over uncompressed data
                           }
                           if (crc32 != check)
                           {
                              dfsProgressSuspend(); // allow regular screen output
                              TxPrint( "CRC mismatch in imagefile near offset 0x%11.11Lx, LSN: 0x0%llx\n", imgpos, sn);
                              error_blocks++;
                              dfsProgressResume("");
                           }
                        }
                        if (rc == NO_ERROR)
                        {
                           ws = csize / bps;    // sectors to do
                           if (ws > bs)         // destination too small, usualy because target area
                           {                    // is not an exact multiple of the buffersize used
                              ws = bs;          // just clip to available target area
                           }
                           if (data != NULL)    // data available (no Smart)
                           {
                              if (test == FALSE) // Write, restore to object
                              {
                                 rc = dfsWrite( sn, ws, data);
                                 if (rc != NO_ERROR)
                                 {
                                    sprintf( text, "Write error on %5u sector(s) at sector 0x%16.16llx.", ws, sn);
                                    switch (dfsa->eStrategy)
                                    {
                                       default:
                                       case TXAE_QUIT: // no action, keep bad rc
                                          TxPrint( "%s  Aborting ...\n", text);
                                          break;

                                       case TXAE_CONFIRM:
                                          if (dfsa->batch) // CONFIRM not possible, but do NOT ignore errors!
                                          {
                                             TxPrint( "%s  Aborting (batch) ...\n", text);
                                          }
                                          else
                                          {
                                             dfsProgressSuspend(); // allow regular screen output
                                             if (TxConfirm( 5212, "%s\nContinue the restore ? [Y/N]: ", text))
                                             {
                                                if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                                                {
                                                   dfsa->eStrategy = TXAE_IGNORE;
                                                   sprintf( text, "Ignoring further errors, continue restoring, file %u", sequence);
                                                }
                                                else
                                                {
                                                   sprintf( text, "Continue restoring, file %u", sequence);
                                                }
                                                dfsProgressResume( text);
                                                rc = NO_ERROR;
                                             }
                                          }
                                          break;

                                       case TXAE_IGNORE: // ignore the error
                                          TxPrint( "%s  Ignore, continue ...\n", text);
                                          rc = NO_ERROR;
                                          break;
                                    }
                                 }
                              }
                              else if (compare) // Read and compare with image
                              {
                                 TRHEXS( 800,  data, csize, "uncompressed data OUT");
                                 crc32 = TxCrc32( data, csize); // over uncompressed data
                                 if ((rc = dfsRead( sn, ws, ibuf)) == NO_ERROR) // overwrites data!
                                 {
                                    ULONG   check;

                                    #if defined (NEVER)
                                    //- Do NOT zero, original 'garbage' was in image, and is either restored
                                    //- to a new target, or is still there in the original location
                                    ULONG   j;

                                    if (smartuse) // ZERO unused sectors before
                                    {           // comparing the CRC values!
                                       for (j = 0; j < ws; j++)
                                       {
                                          if ((DFSFNCALL( dfsa->FsLsnAllocated, sn + j, 0, NULL, NULL) == 0))
                                          {
                                             TRACES(("Smart wipe rel-sector: %u\n", j));
                                             memset( ibuf + (j * bps), 0, bps);
                                          }
                                       }
                                       TRHEXS( 800,  data, csize, "smart-use wiped data");
                                    }
                                    #endif
                                    check = TxCrc32( ibuf, csize); // over read data

                                    TRACES(( "sn:%llx data:%8.8x ibuf:%8.8x csize:%u ws:%u crc32:%8.8x check:%8.8x\n",
                                                                    sn, data, ibuf, csize, ws, crc32, check));
                                    if (crc32 != check)
                                    {
                                       dfsProgressSuspend(); // allow regular screen output
                                       TxPrint( "Image data at offset %lld (bytes) does not match object at LSN: 0x0%llx\n", imgpos, sn);
                                       rc = DFS_ST_MISMATCH;
                                    }
                                    else
                                    {
                                       TRACES(("COMPARE data IMZ against disk-data was OK!\n"));
                                    }
                                 }
                              }
                           }
                           if (dfsa->verbosity != TXAO_QUIET) // progress based on IMZ, not target!
                           {
                              dfsProgressShow( (*filepos + imgpos) / 1024, 0, 0, NULL);
                           }
                           lastw = sn + ws -1;
                        }
                     }
                     else
                     {
                        rc = DFS_BAD_STRUCTURE; // imagefile too short
                     }
                  }
               }
            }
            dfsUlTerminate( ulh);               // end compression session
         }
         else                                   // uncompressed
         {
            ULONG         todo = bs;            // sectors todo now

            for ( i = 0,   sn  = startsec;
                 (i < sizetodo) && (rc == NO_ERROR) && (!TxAbort());
                  i += bs, sn += bs)
            {
               if ((i + bs) > sizetodo)         // last, incomplete chunk
               {
                  todo = sizetodo - i;          // #sectors
               }
               sz = todo * bps;                 // size in bytes

               rc = TxRead( fh, ibuf + rlen, sz - rlen, &handled);
               imgpos += handled;               // for verification purposes

               TRACES(("Rd, sz:%hu  rlen:%u  handled:%hu\n", sz, rlen, handled));

               if (rc == NO_ERROR)
               {
                  if ((sz - rlen) == handled)   // read specified size ?
                  {
                     if (test == FALSE)         // if not testing ...
                     {
                        rc = dfsWrite( sn, todo, ibuf);
                     }
                     else if (compare)          // Read and compare with image
                     {
                        crc32 = TxCrc32( ibuf, sz); // over image data
                        if ((rc = dfsRead( sn, todo, ibuf)) == NO_ERROR)
                        {
                           if (crc32 != TxCrc32( ibuf, sz))
                           {
                              dfsProgressSuspend(); // allow regular screen output
                              TxPrint( "Image data at offset %lld (bytes) does not match object at LSN: 0x0%llx\n", imgpos, sn);
                              rc = DFS_ST_MISMATCH;
                           }
                        }
                     }
                     if (dfsa->verbosity != TXAO_QUIET)
                     {
                        dfsProgressShow( (*filepos + imgpos) / dfsGetSectorSize(), 0, 0, NULL);
                     }
                     lastw = sn + todo -1;
                  }
                  else
                  {
                     if (handled == 0)          // nothing read
                     {                          // could be regular end of multi-file
                        dfsProgressSuspend();   // allow regular screen output
                        rc = DFS_PENDING;
                     }
                     else
                     {
                        rc = DFS_BAD_STRUCTURE; // imagefile too short
                     }
                  }
               }
               rlen = 0;                        // no read-ahead anymore
            }
         }

         if (sequence == fileCount)             // this was the last one
         {
            dfsProgressTerm();                  // Finish progress BAR, allow messages
         }

         if (rc == DFS_BAD_STRUCTURE)
         {
            if (test)                           // Premature end is OK when
            {                                   // verifying, target is most
               rc = NO_ERROR;                   // likely not same size ...
            }
            else
            {
               TxPrint("Minor warning     : Imagefile ended after 0x0%llx of 0x0%llx sectors\n", i, sizetodo);
               rc = DFS_CMD_WARNING;
            }
         }
         if (compare)
         {
            TxPrint( "The imagefile and selected-object are %s%sidentical%s\n",
                     (rc == DFS_ST_MISMATCH) ?  CBR   :     CBG,
                     (rc == DFS_ST_MISMATCH) ? "NOT " : "", CNN );
         }
         if (error_blocks != 0)
         {
            TxPrint( "Major warning!    : %u blocks in the restored image had errors, data on\n"
                     "                    those areas", error_blocks);
            dfsUlDot13( " (max ", uncompressed, " bytes each) is unreliable.\n");
            if (rc == NO_ERROR)                 // if no other errors ...
            {
               rc = DFS_CMD_WARNING;
            }
         }
         else if (!compare && test)
         {
            TxPrint( "Imagefile '%s' verified without errors.\n", fspec);
         }
         *filepos += imgpos;                    // total size read from image
         *lastsn   = lastw;
      }
      if (fh != 0)
      {
         TxClose( fh);
         fh = 0;
      }
      if (ibuf != rbuf)                         // when not existing buffer
      {
         TxFreeMem( ibuf);
      }
      if (compress)
      {
         TxFreeMem( cbuf);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsRestoreSingleFile'
/*---------------------------------------------------------------------------*/

