//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// HPFS disk structure definitions
// Based on information from Doug Azzarito's "HPFS internals" lectures
//
// Author: J. van Wijk
//
// JvW  17-06-1995   Initial version, split off from DHPFS.C
// JvW  24-07-1995   Moved sector definitions from DHPFS.C
// JvW  14-08-1995   Moved LSN constants from DHPFS.C
// JvW  16-10-1995   Added some more structure fields  (2nd HPFS Internals)
// JvW  14-12-1996   Added DASD limit threshold fields (3rd HPFS Internals)
// JvW  28-12-1996   Changed to HPFS definition portion for DFS
// JvW  04-04-2000   Fixed traps in HPFS CodePage & BMGR names
// JvW  08-03-2001   Added definitions for spare-block statusflag
// JvW  13-02-2002   Added definition for deleted FNODE signature on HPFS386
//
#ifndef    DFSHPFS_H
   #define DFSHPFS_H

// FS size limit, OS/2 implementation limit due to shared cache (32 volumes)
// leaving just 27 bits for the sector addressing
#define HPFS_LIMIT         0x07FFFFFF

#define HPLDR_LSN          ((ULONG)  0x01)      // LSN of first HPFS LDR sector
#define HPLDR_SIZE           5                  // SIZE in sectors for HPFS LDR
#define HPLDR_IMGNAME       "dfshpldr"          // Default imagefile name

#define LSN_SUPER          ((ULONG)  0x10)
#define LSN_SPARE          ((ULONG)  0x11)
#define LSN_RESRV          ((ULONG)  0x12)      // not sure about this one...
#define LEN_RESRV          ((ULONG)  0x02)
#define LSN_BMIDX          ((ULONG)  0x14)      // 1st Bitmap index sector

#define HPFS_SP_DIRTY      ((BYTE) 0x01)        // volume dirty bit
#define HPFS_SP_SPARDIR    ((BYTE) 0x02)        // spare directories used
#define HPFS_SP_HOTFIXS    ((BYTE) 0x04)        // hotfixes active
#define HPFS_SP_CORRUPT    ((BYTE) 0x08)        // volume corrupted
#define HPFS_SP_BADBITM    ((BYTE) 0x10)        // bad bitmaps
#define HPFS_SP_FASTFMT    ((BYTE) 0x20)        // fast formatted
#define HPFS_SP_RESERVE    ((BYTE) 0x40)        // reserved
#define HPFS_SP_OLDUTIL    ((BYTE) 0x80)        // old HPFS utility used

#define HPFS_MAX_ACL_EA    (0x13c)              // max size internal EA/ACL

#define HPFS_DIRBL_SECTS   ((ULONG) 4)
#define HPFS_BTMAP_SECTS   ((ULONG) 4)

#define HPFS_FNAME_LEN     15                   // length of FNODE recovery filename

typedef struct s_sptr                           // size & pointer
{
   ULONG                size;
   ULONG                ptr;
} SPTR;                                         // end of struct "s_sptr"

#define SG_SUPER        8
#define SV_SUPER        (char)0x49,(char)0xe8,(char)0x95,(char)0xf9, \
                        (char)0xc5,(char)0xe9,(char)0x53,(char)0xfa
typedef struct s_super                          // HPFS super-block
{
   char                 Signature[SG_SUPER];
   BYTE                 VerMajor;
   BYTE                 VerMinor;
   BYTE                 Reserved[2];
   ULONG                RootFnode;
   ULONG                TotalSec;
   ULONG                BadSec;
   ULONG                BitmapInd;
   ULONG                BitmapIndRsp;
   ULONG                BadList;
   ULONG                BadListRsp;
   ULONG                ChkdskDate;
   ULONG                OptimDate;
   ULONG                DirBandSec;
   ULONG                DirBand;
   ULONG                DirBandEnd;
   ULONG                DirBandMap;
   char                 VolumeName[0x20];
   ULONG                UserIdTable;
} S_SUPER;                                      // end of struct "s_super"

#define SG_SPARE        8
#define SV_SPARE        (char)0x49,(char)0x18,(char)0x91,(char)0xf9, \
                        (char)0xc5,(char)0x29,(char)0x52,(char)0xfa
typedef struct s_spare                          // HPFS spare-block
{
   char                 Signature[SG_SPARE];
   BYTE                 StatusFlag;
   BYTE                 HPFS386Flag;
   BYTE                 MMContiguityFactor;
   BYTE                 Reserved;
   ULONG                HotFixList;
   ULONG                HotFixes;
   ULONG                HotFixMax;
   ULONG                SpareDirUsed;
   ULONG                SpareDirMax;
   ULONG                CodePageInfo;
   ULONG                CodePages;
   ULONG                SuChecksum;
   ULONG                SpChecksum;
   BYTE                 Reserv_2[0x3c];
   ULONG                SpareDir[0x65];
} S_SPARE;                                      // end of struct "s_spare"


#define HPFS_BADSECS    ((ULONG) 0x1ff)
typedef struct s_badsel
{
   ULONG                Next;                   // LSN of next bad-sector elem
   ULONG                Bads[ HPFS_BADSECS];    // Bad sectors LSN's
} S_BADSEL;                                     // end of struct "s_alleaf"

#define HPFS_P_FNODE    0x20                    // parent is an Fnode, not DirBlock
#define HPFS_AL_NODE    0x80                    // allocation is for a NODE

typedef struct s_alblk                          // leader for FILE allocation
{
   BYTE                 Flag;                   // 0x80 == allocation Node
   BYTE                 Reserved[3];
   BYTE                 FreeEntries;
   BYTE                 UsedEntries;
   USHORT               OfsFirstFree;
} S_ALBLK;                                      // end of struct "s_alblk"

typedef struct s_aldir                          // start of directory B-tree
{
   ULONG                Zero2;                  // must be zero too
   ULONG                Zero;                   // must be zero
   ULONG                DirBlock;               // LSN of B-tree root DirBlock
} S_ALDIR;                                      // end of struct "s_aldir"

typedef struct s_alleaf                         // Array element FILE extents
{
   ULONG                RelSecData;             // start sector in datastream
   ULONG                LengthData;             // nr of sectors in this block
   ULONG                DataBlock;              // LSN of first block sector
} S_ALLEAF;                                     // end of struct "s_alleaf"

typedef struct s_alnode                         // Array elemet FILE tree nodes
{
   ULONG                RelSecNext;             // start sector next child-Node
   ULONG                ChildBlock;             // LSN of first block sector
} S_ALNODE;                                     // end of struct "s_alnode"

typedef union u_aldata                          // FILE Allocation / DIR block
{
   S_ALDIR              Directory;              // With just the DirBlock LSN
   S_ALLEAF             Leaf[0x08];             // either 8 blocks of leaf data
   S_ALNODE             Node[0x0c];             // or 12 blocks of node data
} U_ALDATA;                                     // end of union "u_aldata"

#define SG_FNODE        4
#define SV_FNODE        (char)0xae,(char)0x0a,(char)0xe4,(char)0xf7
#define SG_FNDEL        4
#define SV_FNDEL        (char)0x00,(char)0x00,(char)0x00,(char)0x00
#define SG_FD386        4
#define SV_FD386        (char)0x00,(char)0x0a,(char)0xe4,(char)0xf7
typedef struct s_fnode                          // Fnode, dir or file
{
   char                 Signature[SG_FNODE];    // 00
   BYTE                 Reserved[8];            // 04
   BYTE                 NameLength;             // 0C
   char                 Name[ HPFS_FNAME_LEN];  // 0D Recovery filename (truncated)
   ULONG                ParentDir;              // 1C Fnode LSN for parent DIR
   SPTR                 AclExtern;              // 20
   USHORT               AclFnLength;            // 28
   BYTE                 AclFlag;                // 2A
   BYTE                 Reserv_2;               // 2B
   SPTR                 EAsExtern;              // 2C
   USHORT               EAsFnLength;            // 34
   BYTE                 EAsFlag;                // 36
   BYTE                 DirFlag;                // 37
   S_ALBLK              AlInfo;                 // 38 Info on ALDATA (files only)
   U_ALDATA             Al;                     // 40 File allocation OR DirBlock
   ULONG                FileLength;             // A0 Valid Data Length written
   ULONG                RequiredEAs;            // A4
   BYTE                 Reserv_3[0x10];         // A8 reserved UserId
   USHORT               OfsAclControl;          // B8 Offset to Acl/EA from start
   BYTE                 FirstDasdAlertThresh;   // BA percentage used 1st alert
   BYTE                 DeltaDasdAlertThresh;   // BB percentage delta next alerts
   ULONG                DasdLimit;              // BC HPFS386, limit sectors
   ULONG                DasdUsage;              // C0 HPFS386, usage sectors
   BYTE                 EasOrAcl[ HPFS_MAX_ACL_EA];
} S_FNODE;                                      // end of struct "s_fnode"

#define HPFS_DF_BGDIR      ((BYTE) 0x01)        // begin-of-directory
#define HPFS_DF_BDOWN      ((BYTE) 0x04)        // Btree down pointer entry
#define HPFS_DF_EODIR      ((BYTE) 0x08)        // end-of-directory
#define HPFS_DF_BGEND      ((BYTE) 0x09)        // Begin or End DIR entry

typedef struct s_dirent
{
   USHORT               Length;                 // length of directory entry
   BYTE                 Flag;                   // DirEntry flag bits
   BYTE                 FatAttrib;              // FAT Attributes
   ULONG                Fnode;                  // LSN of FNODE
   ULONG                LastModDate;            // Last Modification date
   ULONG                FileSize;               // Reported file size
   ULONG                LastAccessTime;         // Last Access Time
   ULONG                CreationTime;           // Creation Time
   ULONG                EALength;               // Extended Atribute length
   BYTE                 FlexFlag;               // Flex Area Flag
   BYTE                 CodePageIndex;          // Code Page index in info
   BYTE                 NameLength;             // Length of filename
   char                 FileName[1];            // FileName, variable length
} S_DIRENT;                                     // end of struct "s_dirent"

#define SG_DIRBL        4
#define SV_DIRBL        (char)0xae,(char)0x0a,(char)0xe4,(char)0x77
typedef struct s_dirbl                          // Directory block
{
   char                 Signature[SG_DIRBL];
   ULONG                OfsFirstFree;
   ULONG                ChangeCount;
   ULONG                ParentDir;
   ULONG                This;
   S_DIRENT             FirstEntry;             // Variable length DirEntries
} S_DIRBL;                                      // end of struct "s_dirbl"

#define SG_ALLOC        4
#define SV_ALLOC        (char)0xae,(char)0x0a,(char)0xe4,(char)0x37
typedef struct s_alloc                          // Allocation sector
{
   char                 Signature[SG_ALLOC];
   ULONG                This;
   ULONG                Parent;
   S_ALBLK              AlInfo;
   U_ALDATA             Al;
   BYTE                 Reserved[0x0C];
} S_ALLOC;                                      // end of struct "s_alloc"

typedef struct s_cpib
{
   USHORT               Country;                // country code
   USHORT               CodePageId;             // Codepage number
   ULONG                CpChecksum;             // Codepage checksum
   ULONG                CpData;                 // Codepage data sector
   USHORT               VolCpId;                // Volume-specific Id
   USHORT               DbcsRanges;             // nr of DBCS ranges in Cp
} S_CPIB;                                       // end of struct "s_cpib"

#define S_MAX_CPIB      0x1f
#define SG_CPINF        4
#define SV_CPINF        (char)0xf7,(char)0x21,(char)0x45,(char)0x49
typedef struct s_cpinf                          // Codepage info
{
   char                 Signature[SG_CPINF];
   ULONG                CpCount;                // codepages in this sector
   ULONG                FirstCpib;              // first CP info block
   ULONG                NextCpInfo;             // next codepage info sector
   S_CPIB               Cpib[S_MAX_CPIB];       // Array of Cp info blocks
} S_CPINF;                                      // end of struct "s_cpinf"

typedef struct s_dbcs                           // DBCS range pairs
{
   BYTE                 start;
   BYTE                 end;
} S_DBCS;                                       // end of struct "s_dbcs"

typedef struct s_cpdb
{
   USHORT               Country;                // country code
   USHORT               CodePageId;             // Codepage number
   USHORT               DbcsRanges;             // nr of DBCS ranges in Cp
   char                 Table[0x80];            // conversion table
   S_DBCS               Dbcs[1];                // DBCS range pairs
} S_CPDB;                                       // end of struct "s_cpdb"

#define SG_CPDAT        4
#define SV_CPDAT        (char)0xf7,(char)0x21,(char)0x45,(char)0x89
typedef struct s_cpdat                          // Codepage data table
{
   char                 Signature[SG_CPDAT];
   USHORT               CpCount;                // codepages in this sector
   USHORT               FirstCpdb;              // first CP data block
   ULONG                Checksum[3];            // CpData checksums
   USHORT               OfsCpdb[3];             // CpData offsets
   S_CPDB               Cpdb;                   // Cpdb data blocks (1st !!)
} S_CPDAT;                                      // end of struct "s_cpdat"

typedef struct s_uinf
{
   BYTE                 Byte1;                  // UserId byte 1
   BYTE                 Byte2;                  // UserId byte 2
   BYTE                 Byte3;                  // UserId byte 3
   BYTE                 Byte4;                  // UserId byte 4
} S_UINF;                                       // end of struct "s_uinf"

#define SG_UITAB        4
#define SV_UITAB        (char)0xff,(char)0xff,(char)0xfa,(char)0xfa
typedef struct s_uitab                          // UserId table (HPFS-386)
{
   ULONG                UiCount;                // Userid entry count (+1)
   char                 Signature[SG_UITAB];
   S_UINF               UiInfo[0x3e];           // Array of UserId data
} S_UITAB;                                      // end of struct "s_uitab"


typedef struct s_aclblk                         // UI info in Fnode
{
   ULONG               access;                  // access info, RXWCD flags?
   S_UINF              userid;                  // HPFS386 userid key
   ULONG               reserv_1;                // unknown, usualy zero
   ULONG               reserv_2;                // unknown, usualy zero
   ULONG               reserv_3;                // unknown, usualy zero
} S_ACLBLK;                                     // end of struct "s_aclblk"


#endif
