//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// HFS disk structure definitions
// Based on information from "Unix filesystems" by Steve D Pate
// and the Apple Technical Note 1150 (HFS+ On-Disk structures) plus
// other Internet info, and a lot of reverse-engineering with DFSee
//
// Author: J. van Wijk
//
// JvW  16-07-2007 Initial version, derived from RSR
// JvW  31-03-2012 Updated with actual Apple info like Technical Note 1150
// JvW  06-03-2018 Added several catalog related definitions (HFS+ only!)
// JvW  09-05-2019 Implemented hardlink support for files and directories
//
#ifndef    DFSHFS_H
   #define DFSHFS_H

//- First Superblock, at sect# 2 for 512 BPS, 0 for 4096 (at offset 1024)
//- 2nd is at last-sector - 2
#define HFS_LSNSUP1 ((ULN64) (1024 / dfsGetSectorSize()))
#define HFS_SUPSIZE   4096                      // size of the super 'block'

// Signature values
#define HFS_CLAS_SIGNATURE  ((USHORT) 0x4244)   // 'BD' classic HFS (unsupported in DFSee!)
#define HFS_PLUS_SIGNATURE  ((USHORT) 0x482B)   // 'H+' standard, case insenstitive
#define HFS_XTND_SIGNATURE  ((USHORT) 0x4858)   // 'HX' extended, case sensitive

// Version values
#define HFS_PLUS_VERSION    ((USHORT) 0x0004)   // 'H+' version 4 only
#define HFS_XTND_VERSION    ((USHORT) 0x0005)   // 'HX' version 5 or higher

// Last mounted version
#define HFS_OS89_MOUNTED      0x382E3130        // '8.10' HFS+ Mac OS 8.1 up to 9.2.2
#define HFS_PLUS_MOUNTED      0x31302E30        // '10.0' HFS+ Mac OS X, non journaled
#define HFS_PJRN_MOUNTED      0x4846534A        // 'HFSJ' HFS+ Mac OS X, journaled
#define HFS_PFSK_MOUNTED      0x46534B21        // 'FSK!' HFS+ Failed journal replay
#define HFS_CERD_MOUNTED      0x63657264        // 'cerd' HFS+ used in (some) disk images

// MAC classic fork types
#define HFS_FORK_DATA         0x00
#define HFS_FORK_RESOURCE     0xFF

// Extent density in forks
#define HFS_EXT_PER_REC       8

// Volume attribute masks

#define HFS_VA_HARDWARE_LOCK               ((ULONG) 0x00000080)
#define HFS_VA_VOL_UNMOUNTED               ((ULONG) 0x00000100)
#define HFS_VA_SPARED_BLOCKS               ((ULONG) 0x00000200)
#define HFS_VA_NO_CACHE_REQD               ((ULONG) 0x00000400)
#define HFS_VA_BOOTVOL_INCON               ((ULONG) 0x00000800)
#define HFS_VA_NODEID_REUSED               ((ULONG) 0x00001000)
#define HFS_VA_HAS_A_JOURNAL               ((ULONG) 0x00002000)
#define HFS_VA_VOL_INCONSIST               ((ULONG) 0x00004000)
#define HFS_VA_SOFTWARE_LOCK               ((ULONG) 0x00008000)
#define HFS_VA_UNUSEDNODEFIX               ((ULONG) 0x80000000)

// allow 5 million files/directories in cache (40 Mb + all directory names)
// when CnIDs are reused so the nextCatalogID is not reliable as a maximum value
#define HFS_NP_CACHE_MAXSIZE               50000000

// Conversion from 1-jan-1904 to 1-jan-1970 time bases
#define HFS_DT_OFFSET_POSIX                ((ULONG) 0x7c25b080) // to be refined
#define HfsDate2Tm(hdt)  ((time_t) ((hdt) - HFS_DT_OFFSET_POSIX))


//================================ B-TREE =====================================

#define BT_SANE_LEVELS 9                        // reaches 2^32 nodes with 12 records/node
                                                // allowing 255 Unicode chars per key
#define BT_MAX_KEY_LEN

// Kind of nodes
#define BT_LeafNode     ((BYTE) 0xFF)
#define BT_IndexNode    ((BYTE) 0)
#define BT_HeaderNode   ((BYTE) 1)
#define BT_MapNode      ((BYTE) 2)

//- Start of each and every node
typedef struct s_bt_node_desc                   // descriptor at start each node
{
   ULONG               lNext;                   // 000 link to Next at this level
   ULONG               lPrev;                   // 004 link to Prev at this level
   BYTE                nKind;                   // 008 leaf, index, header, map
   BYTE                nHeight;                 // 009 header/map == 0; child + 1
   USHORT              nRecords;                // 00a number of records
   USHORT              unused;                  // 00c reserved, zero
} S_BT_NODE_DESC;                               // 00e end of struct "s_bt_node_desc"

//- first record in B-Tree, after node_desc
typedef struct s_bt_header_rec                  // first record in B-tree header
{                                               // offsets relative to start of the node block!
   USHORT              bthDepth;                // 00e maximum depth in tree
   ULONG               bthRootNode;             // 010 node-number of root
   ULONG               bthLeafCount;            // 014 number of leaf records
   ULONG               bthFirstLeaf;            // 018 node-number first leaf
   ULONG               bthLastLeaf;             // 01C node-number first leaf
   USHORT              bthNodeSize;             // 020 node size in bytes
   USHORT              bthMaxKeyLength;         // 022 reserved
   ULONG               bthTotalNodes;           // 024 # nodes in tree
   ULONG               bthFreeNodes;            // 028 # free/unused nodes
   USHORT              bthUnused;               // 02c reserved
   ULONG               bthClumpSize;            // 02e reserved
   BYTE                bthTreeType;             // 032 reserved
   BYTE                bthCompareType;          // 033 key compare type (case sensitivity)
   ULONG               bthAttributes;           // 034 persistent tree attributes
   ULONG               bthReserved[ 16];        // 038 reserved
} S_BT_HEADER_REC;                              // 078 end of struct "s_bt_header_rec"

#define HFS_CaseFolding   0xCF  // case folding (case-insensitive)
#define HFS_BinaryCompare 0xBC                  // binary compare (case-sensitive)

#define BTH_BadCloseMask           0x00000001   // reserved
#define BTH_BigKeysMask            0x00000002   // key length field is 16 bits
#define BTH_VariableIndexKeysMask  0x00000004   // keys in index nodes are variable length

#define BTH_USER_REC_SIZE                 128   // fixed size of user-record

typedef char A_BT_USER_REC[ BTH_USER_REC_SIZE]; // user record, usually zeroes


typedef struct s_bt_hdr_node                    // combined HEADER node definitions
{
   S_BT_NODE_DESC      dsc;                     // 000 node description
   S_BT_HEADER_REC     hdr;                     // 00e tree header record
   A_BT_USER_REC       user;                    // 078 user record
   BYTE                bTreeMap[0xf00];         // 0f8 node usage map, upto indexes
   USHORT              nodeSentinel;            // ff8 start of indexes (value 'self' = 0x0ff8)
   USHORT              nodeIndexes[3];          // ffa indexes for 3 records   (reversed order)
} S_BT_HDR_NODE;                                // end of struct "s_bt_hdr_node"

// Note that the above example hdr-node is for a nodesize of 4096 (0x1000) being one block
// Most trees use a nodesize of 8192 (0x200) so the sentinel is at 0x1ff8 instead of 0x0ff8
// and the TreeMap size is 0x1f00 in size (7936 bytes = 63488 node-usage bits)
// When more nodes are needed, there will be additional MAP nodes with a node_desc and map record

//================================ EXTENT and FORK ============================

typedef struct s_hfs_extendkey
{
   USHORT              length;                  // key length
   BYTE                forkType;                // data or resource
   BYTE                reserved;                // padding
   ULONG               fileID;                  // Unique file identifier
   ULONG               startBlock;              // first block in this extent
} S_HFS_EXTENDKEY;                              // end of struct "s_hfs_extendkey"

typedef struct s_hfs_extenddesc
{
   ULONG               startBlock;              // first allocation block
   ULONG               blockCount;              // number of blocks in extent
} S_HFS_EXTENDDESC;                             // end of struct "s_hfs_extenddesc"

// Extent records, multiple extents in a single record (array)
typedef S_HFS_EXTENDDESC S_HFS_EXTENDREC[ HFS_EXT_PER_REC];

typedef struct s_hfs_forkdata
{
   ULN64               logicalSize;             // Fork size in bytes
   ULONG               clumpSize;               // Clump size in bytes
   ULONG               totalBlocks;             // total blocks in the fork
   S_HFS_EXTENDREC     extents;                 // initial extent set
} S_HFS_FORKDATA;                               // end of struct "s_hfs_forkdata"



//================================ CATALOG ====================================

// catalog predefined CnID values
#define HFS_ID_RootParent         0x0001        // Parent ID of root folder
#define HFS_ID_RootFolder         0x0002        // Folder ID of root folder
#define HFS_ID_Extents            0x0003        // ID of the extents file
#define HFS_ID_Catalog            0x0004        // ID of the catalog file
#define HFS_ID_BadBlock           0x0005        // ID of the bad block file
#define HFS_ID_Allocation         0x0006        // ID of the allocation file (HFSP)
#define HFS_ID_Startup            0x0007        // ID of the startup file    (HFSP)
#define HFS_ID_Attributes         0x0008        // ID of the attribute file  (HFSP)
#define HFS_ID_RepairCatalog      0x000e        // ID of rebuil Catalog B-tree
#define HFS_ID_BogusExtent        0x000f        // ID of temporary, while exchanging extents
#define HFS_ID_FirstUser          0x0010        // ID first user file in catalog

// catalog record types
#define HFS_CR_Folder             0x0001        // Folder record
#define HFS_CR_File               0x0002        // File record
#define HFS_CR_Dthread            0x0003        // Folder thread record
#define HFS_CR_Fthread            0x0004        // File thread record

// catalog record flags
#define HFS_RF_FileLocked         0x0001        // File is locked       (readonly)
#define HFS_RF_ThreadExists       0x0002        // Thread-record exists
#define HFS_RF_HasAttributes      0x0004        // Has extended attribs
#define HFS_RF_HasSecurity        0x0008        // Has security info    (ACL)
#define HFS_RF_HasFolderCount     0x0010        // Has HX foldercount   (Dirs + hardlinked Dirs)
#define HFS_RF_HasLinkChain       0x0020        // Has a hardlink chain (inode or link)
#define HFS_RF_HasChildLink       0x0040        // Folder has child that is a Dir link

//- Hard/Symbolic Link special values for FILE type/creator (defined little-endian, no convert!)
#define HFS_FT_SYMLINK        0x6b6e6c73        // 'slnk' on-disk
#define HFS_FC_SYMLINK        0x70616872        // 'rhap' on-disk
#define HFS_FT_HRDLINK        0x6b6e6c68        // 'hlnk' on-disk
#define HFS_FC_HRDLINK        0x2b736668        // 'hfs+' on-disk
#define HFS_FT_FDALIAS        0x70726466        // 'fdrp' on-disk (alias also known as shortcut)
#define HFS_FC_FDALIAS        0x5343414d        // 'MACS' on-disk (or bookmark, varying in size)

#define HFS_SD_HARDLINK "....HFS+ Private Data" // Hardlink special directory name (0x00 as '.')

// To resolve a hardlinked FILE, locate its file-record (aka Inode), it should have the above
// magic values xxx_HRDLINK in the type/creator fields. Then take the hardlink reference-number
// from the crBsdInfo.iNodeNum field, and construct the filename: 'iNodeNNN' where NNN is
// the DECIMAL representation of that iNodeNum, without leading zeroes. Finally locate this
// file (record/inode) by searching it in the directory HFS_SD_HARDLINK in the root
// Take the file specifics like date/time, size and allocation from that.


#define HFS_SIGN_ALIAS        0x73696c61        // 'alis' on-disk   (little-endian, no convert!)

#define HFS_SD_FLDALIAS ".HFS+ Private Directory Data\r" // Folder-Alias special directory name

// To resolve a hardlinked DIRECTORY (folder Alias), locate its file-record (aka Inode), it should
// have the magic values xxx_FDALIAS in the type/creator fields. It will have a filesize of 0 (empty)
// and should have a resource fork (typically 464 bytes in size) to be located through the resource
// fork allocation info; Read that resource fork, check 'alis' signature at offset 0x1BC, then
// take the alias reference-number from offset 0x176 (Big Endian!) Construct the filename: 'dir_NNN'
// where NNN is the DECIMAL representation of that reference-number, without leading zeroes.
// Finally locate this directory (record/inode) by searching it in the directory HFS_SD_FLDALIAS
// in the root. Use this directory (contents) instead of the original alias

// Most contents unknown, not documented, info based on reverse-engineering TimeMachine disks
typedef struct s_hfs_alis_fork                  // resource-fork contents DIR alias (hardlink)
{
   BYTE                al_fill1[ 0x12e];        // 000 start of the structure, mostly empty
   USHORT              al_hfsplus;              // 12E 'H+' signature, in BIG-endian format
   BYTE                al_fill2[ 0x046];        // 130 middle of the structure, mostly empty
   ULONG               al_dirRef;               // 176 Directory Reference Number -> 'dir_NNN'
   BYTE                al_fill3[ 0x042];        // 17A more of the structure, various values
   ULONG               al_signature;            // 1BC Signature value, ascii (or LE ULONG :)
   BYTE                al_fill4[ 0x010];        // 1C0 end of the structure, upto size 446 bytes
} S_HFS_ALIS_FORK;                              // 1D0 end of struct "s_hfs_alis_fork"


//====================================== CATALOG definitions ====================================

typedef struct s_hfs_catalog_key
{
   USHORT              keyLength;               // key length in bytes (excluding THIS field!)
   ULONG               parentId;                // parent folder ID
   S_MACUNI_STRING     name;                    // pascal-style, big-endian Unicode[255] name
} S_HFS_CATALOG_KEY;                            // end of struct "s_hfs_catalog_key"

typedef struct s_hfs_fldrinfo
{
   struct
   {
      USHORT           top;                     // 000
      USHORT           left;                    // 002
      USHORT           bottom;                  // 004
      USHORT           right;                   // 006
   }                   frRect;                  // 000 finder window position
   USHORT              frFlags;                 // 008 finder flags
   struct
   {
      USHORT           v;                       // 00a
      USHORT           h;                       // 00c
   }                   frLocation;              // 00a finder window location
   short               frOpaque;                // 00e
} S_HFS_FLDRINFO;                               // 010 end of struct "s_hfs_fldrinfo"

typedef struct s_hfs_fileinfo
{
   ULONG               type;                    // 000 file type    (and sym/hard/folder link magic!)
   ULONG               creator;                 // 004 file creator (and sym/hard/folder link magic!)
   USHORT              frFlags;                 // 008 finder flags
   struct
   {
      USHORT           v;                       // 00a
      USHORT           h;                       // 00c
   }                   frLocation;              // 00a finder window location
   short               frOpaque;                // 00e
} S_HFS_FILEINFO;                               // 010 end of struct "s_hfs_fileinfo"


typedef struct s_hfs_finderinfo
{
   BYTE                finderData[16];          // additional FINDER info
} S_HFS_FINDERINFO;                             // end of struct "s_hfs_finderinfo"

typedef struct s_hfs_bsd_info                   // user/group, mode and special info
{
   ULONG               ownerId;                 // 000
   ULONG               groupId;                 // 004
   BYTE                adminFlags;              // 008
   BYTE                ownerFlags;              // 009
   USHORT              fileMode;                // 00a type and permissions
   union
   {
      ULONG            iNodeNum;                // 00c indirect node (hard links only)
      ULONG            linkCount;               // 00c #links that refer to this node
      ULONG            rawDevice;               // 00c special device type FBLK/FCHR
   };                                           // 010 end of union ""
} S_HFS_BSD_INFO;                               // 010 end of struct "s_hfs_bsd_info"

// The DIRINFO structure is the first 80 bytes of a FOLDER or FILE catalog record
typedef struct s_hfs_dirinfo                    // HFS PLUS catalog directoryinfo (80)
{
   USHORT              crRecType;               // 000 record type (1 == folder 2==file)
   USHORT              crFlags;                 // 002 flags
   ULONG               crItems;                 // 004 item count (folder only)
   ULONG               crCnID;                  // 008 file/folder ID
   ULONG               crCreTime;               // 00c create timestamp
   ULONG               crModContTime;           // 010 modify timestamp, contents
   ULONG               crModAttrTime;           // 014 modify timestamp, attributes
   ULONG               crAccTime;               // 018 access timestamp, macOS only
   ULONG               crBacTime;               // 01c backup timestamp
   S_HFS_BSD_INFO      crBsdInfo;               // 020 Unix style info,  macOS only
   union
   {
      S_HFS_FLDRINFO   d;                       // 030 Folder info
      S_HFS_FILEINFO   f;                       // 030 File info
   };
   S_HFS_FINDERINFO    crFdrInfo;               // 040 Opaque Finder info
} S_HFS_DIRINFO;                                // 050 end of struct "s_hfs_dirinfo"


typedef struct s_hfs_folder                     // HFS PLUS catalog folder record (88)
{
   S_HFS_DIRINFO       dir;                     // 000 Directory information
   ULONG               crTextEncoding;          // 050 hint for translation
   ULONG               crFolderCount;           // 054 # enclosed folders (if bit set)
} S_HFS_FOLDER;                                 // 058 end of struct "s_hfs_folder"


typedef struct s_hfs_file                       // HFS PLUS catalog file record (248)
{
   S_HFS_DIRINFO       dir;                     // 000 Directory information
   ULONG               crTextEncoding;          // 050 hint for translation
   ULONG               crReserved;              // 054 reserved
   S_HFS_FORKDATA      crDataFork;              // 058 data fork          (16 + 8 * 8)
   S_HFS_FORKDATA      crRsrcFork;              // 0a8 Rsrc fork          (16 + 8 * 8)
} S_HFS_FILE;                                   // 0f8 end of struct "s_hfs_file"

typedef struct s_hfs_thread                     // HFS PLUS catalog THREAD record
{
   USHORT              crRecType;               // 000 record type (3 == folder 4==file)
   USHORT              crReserved;              // 002 reserved, 0
   ULONG               parentId;                // 004 parent folder ID
   S_MACUNI_STRING     name;                    // 008 pascal-style, big-endian Unicode[255] name
} S_HFS_THREAD;                                 // end of struct "s_hfs_thread"

typedef union u_hfs_cat_record                  // any HFS catalog record
{
   S_HFS_DIRINFO       d;                       // dir info (folder or file)
   S_HFS_FOLDER        folder;                  // folder
   S_HFS_FILE          file;                    // file
   S_HFS_THREAD        t;                       // thread
} U_HFS_CAT_RECORD;                             // end of union "u_hfs_cat_record"

//- used for quick&dirty access to the ROOT-NODE, to get the LABEL (== root name)
typedef struct s_bt_rootnode                    // combined ROOT-NODE definitions
{
   S_BT_NODE_DESC      dsc;                     // 000 node description
   S_HFS_CATALOG_KEY   Label;                   // 00e root-folder name, aka LABEL
   S_HFS_FOLDER        RootFolderInfo;          // xxx directly follows variable length name
   BYTE                nodeRecordData[1];       // yyy node records, freespace and indexes
} S_BT_ROOTNODE;                                // end of struct "s_bt_rootnode"


//================================ SUPERBLOCK =================================
typedef struct s_hfs_super                      // Apple: Volume Header HFS+
{
   USHORT              Signature;               // 000
   USHORT              Version;                 // 002
   ULONG               Attributes;              // 004 volume attributes
   ULONG               LastMountedBy;           // 008 for recovery purposes
   ULONG               JournalInfoBlock;        // 00c block addr of journal info or 0

   ULONG               CreateDate;              // 010 volume creation
   ULONG               ModifyDate;              // 014 last modification
   ULONG               BackupDate;              // 018 last backup
   ULONG               CheckedDate;             // 01c last disk check

   ULONG               FileCount;               // 020 number of files in volume
   ULONG               DirCount;                // 024 number of directories in volume

   ULONG               BlockSize;               // 028 size in bytes of allocation blocks
   ULONG               TotalBlocks;             // 02c allocation blocks in volume
   ULONG               FreeBlocks;              // 030 number of unused allocation blocks

   ULONG               NextAllocation;          // 034 start of next allocation search
   ULONG               RsrcClumpSize;           // 038 default resource fork clump size
   ULONG               DataClumpSize;           // 03c default data fork clump size
   ULONG               NextCatalogID;           // 040 next unused catalog node ID

   ULONG               WriteCount;              // 044 volume write count
   ULN64               EncodingsBitmap;         // 048 which encodings have been used

   BYTE                FinderInfo[32];          // 050 information used by the Finder

   S_HFS_FORKDATA      AllocationFile;          // 070 allocation bitmap file
   S_HFS_FORKDATA      ExtentsFile;             // 0c0 extents overflow B-tree file
   S_HFS_FORKDATA      CatalogFile;             // 110 catalog B-tree file
   S_HFS_FORKDATA      AttributesFile;          // 160 extended attributes B-tree file
   S_HFS_FORKDATA      Startupfile;             // 1b0 boot file (secondary loader)
} S_HFS_SUPER;

#endif
