//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee HEX (sector) EDITOR, ASCII-viewer and disassembler view, implementation
//
// Author: J. van Wijk
//
// JvW  21-06-2006 Initial version
// JvW  30-07-2007 Added disassembler view, using same interfaces
// JvW  11-10-2016 Added pure ASCII viewer, with auto-view and toggle options
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfsupart.h>                           // PART utility functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfshexed.h>                           // DFS  hex editor interface
#include <dfswin.h>                             // DFS  windowing defs
#include <dfsdgen.h>                            // DFS  generic dialogs

#include <distorm.h>                            // Disassembler lib interface

#define DFS_DID_NEXT_VIEW      0xDF2            // Dialog result on <F2>, next view

#define DFSHEV_WM_REFRESH    TXWM_USER + 7      // reload, new contents or resize

#define DFSDISTACKS               99            // location stack size

#define DFSDIS_WWIDTH             92            // default window width
#define DFSDIS_WLINES             21            // default window length

typedef struct dfsdisasmdata                    // data for DISASM view
{
   LONG                position;                // position in flatCode
   ULONG               halfSize;                // half size of flat code
   BYTE               *flatCode;                // linear copy of code-area
   ULONG               maxLines;                // max nr of disassembly lines
   ULONG               lines;                   // number of disassembly lines
   _DecodedInst       *code;                    // disassembly data array
   ULONG               stPtr;                   // stack pointer in stack
   ULN64              *stack;                   // call/ret push/pop stack
   char              **text;                    // disassembled text lines
} DFSDISASMDATA;                                // end of struct "dfsdisasmdata"

static _DecodeType     daBits   = Decode16Bits; // code bitness 16/32/64
static _OffsetType     daOrigin = 0;            // origin for first code byte
                                                // at flatCode start

#if defined (UNIX)
   #define DFSHE_UNDO_KEY "Alt-Z"
#else
   #define DFSHE_UNDO_KEY "Ctrl-Z/Alt-Z"
#endif


#define DFSTXV_WWIDTH             92            // default window width
#define DFSTXV_WLINES             21            // default window length


#define DFSAF_NONE     0x00                     // purity flag none
#define DFSAF_PURE     0x01                     // purity flag pure ASCII
#define DFSAF_SPLIT    0x02                     // purity show-split-lines flag
#define DFSAF_MASK     0x03                     // purity mask, maximum value

static char   *dfstxv_style[4] =
{
   "Rough",                                     // Allow most values, rough
   "Clean",                                     // Allow pure ASCII and whitespace
   "R-8<-",                                     // Rough, with split lines
   "C-8<-"                                      // Clean, with split lines
};

typedef struct dfstxviewdata                    // data for ASCII text view
{
   BOOL                offsets;                 // include byte offsets in display
   USHORT              style;                   // filtering style ASCII in text
   USHORT              strings;                 // strings filter length 1 .. 9
   ULONG               textSize;                // allocated text ptrs/linebuffers
   ULONG               lines;                   // number of text lines in window
   char              **text;                    // disassembled text lines
} DFSTXVIEWDATA;                                // end of struct "dfstxviewdata"


typedef struct dfshexeddata
{
   ULN64               initialLsn;              // initial LSN for CURR buffer
   USHORT              initialPos;              // initial position in buffer
   ULN64               currentLsn;              // actual  LSN for CURR buffer
   ULONG               sectors;                 // nr of sectors per buffer
   ULONG               written;                 // nr of sectors written
   DFSHEXEDINFO       *info;                    // additional object info
   void               *altData;                 // alternate format data
} DFSHEXEDDATA;                                 // end of struct "dfshexeddata"


// Callback function to service HEXEDIT service requests
static ULONG dfsHexEdObjCallback
(
   ULONG               cmd,                     // IN    action command
   TXHEXEDIT          *dat                      // INOUT hexedit data
);

// Determine most suitable view format based on current sector data
static USHORT dfsHexEdSetAutoView               // RET   best view ID
(
   TXHEXEDIT          *dat,                     // IN    hexedit data
   BOOL                inclHex                  // IN    HEX allowed too
);

// Writeback current buffer after user-confirmation popup, incl SPACE support
static ULONG dfsHexEdWriteBack
(
   TXHEXEDIT          *dat                      // INOUT hexedit data
);

// Position line, zero-based or with cursor centered and largest area visible
static void dfsPositionCursorLine
(
   USHORT              offset,                  // IN    offset in buffer
   TXHEXEDIT          *dat                      // INOUT hexedit data
);

// Set Edit-data values and read prev/curr/next data buffers in one go
static ULONG dfsReadAllBuffers
(
   ULN64               newSn,                   // IN    LSN for CURR buffer
   ULONG               sectors,                 // IN    sectors per buffer
   BOOL                fakeNew,                 // IN    no read, fake 1 byte
   TXHEXEDIT          *dat                      // INOUT hexedit data
);

// Read buffer from specified LSN when valid, fill buffer structure and descr
static ULONG dfsReadObjectBuffer
(
   ULN64               rsn,                     // IN    relative sn of 1st sector
   ULONG               size,                    // IN    size in sectors
   BOOL                fakeNew,                 // IN    no read, fake 1 byte
   TXHEXEDIT          *dat,                     // IN    HEXEDIT data
   TXHEBUF            *buf                      // INOUT buffer data
);


// Set description string for a data buffer
static void dfsSetBufferDescr
(
   ULN64               lsn,                     // IN    lsn of 1st sector
   TXHEXEDIT          *dat,                     // IN    HEXEDIT data
   TXHEBUF            *buf                      // INOUT buffer data
);

// Display Text-viewer ASCII window in lower right of desktop
static ULONG dfsTxViewPopup
(
   TXHEXEDIT          *dat                      // IN    binary editor data
);

// Dialog window procedure, for the Text-viewer popup window dialog
static ULONG dfsTxViewDlgProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

// Fill all buffers and convert raw sector contents into ASCII array/title
static ULONG dfsReadTxView                      // RET   result
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + plugin data
   BOOL                readBuf,                 // IN    read buffers too
   short               with                     // IN    clipping with
);

// Prepare line buffer for a new line, adding optional byte offset number
static char *dfsTxViewNewLine                   // RET   prepared line
(
   char               *lineBuf,                 // INOUT linebuffer to prepare
   BOOL                offsets,                 // IN    add offset number
   BOOL                decimal,                 // IN    number in decimal
   LONG                byteOffset               // IN    current byte offset
);

// Allocate memory for a single line to be added to the text-viewer text-array
static char *dfsTxViewAddLine
(
   char               *newLine                  // IN    line to be added
);

// Update text-viewer window title/footer to reflect current position & filter
static void dfsTxViewTitle
(
   DFSHEXEDDATA       *bd,                      // IN    DFS Binary-Editor data
   TXTM                wt,                      // OUT   Window title
   TXTM                wf                       // OUT   Window footer
);


// Display text-view disassembler window in lower right of desktop
static ULONG dfsDisAsmPopup
(
   TXHEXEDIT          *dat,                      // IN    binary editor data
   LONG               *ilen                     // OUT   current instr length
);

// Dialog window procedure, for the Disassembler popup window dialog
static ULONG dfsDisAsmDlgProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

// Fill all buffers and dis-assemble flat input buffer into ASCII array/title
static ULONG dfsReadDisAsm                      // RET   result
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + disasm data
   BOOL                readBuf,                 // IN    read buffers too
   short              *markLen,                 // OUT   length first instruction
   TXTM                title                    // OUT   Window title
);

// Dis-assemble flat input buffer into ASCII array of lines, update title
static ULONG dfsDisAssemble                     // RET   result
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + disasm data
   ULONG               lines,                   // IN    number of lines to do
   short              *markLen,                 // OUT   length first instruction
   TXTM                title                    // OUT   Window title
);

// Update the disassembly window marked-area when needed, and repaint window
// When current mark is NOT the FIRST-instruction mark, leave it alone
static void dfsDisasmSetMarkRepaint
(
   TXWHANDLE           hwnd,                    // IN    current window
   short               iMarkSize                // IN    instr mark size or 0
);

// Update disasembly window title to reflect current position
static void dfsDisAsmTitle
(
   DFSHEXEDDATA       *bd,                      // IN    DFS Binary-Editor data
   TXTM                wt                       // OUT   Window title
);

// Save the current disassembly buffer (1..n lines) as a plain ASCII .asm file
static ULONG dfsSaveAssembly
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + disasm data
   char               *fname                    // IN    name of output file
);


/*****************************************************************************/
// HEX sector editor, working on current DFSee object (store)
/*****************************************************************************/
ULONG dfsHexObjectEditor
(
   ULN64               lsn,                     // IN    starting LSN
   ULN64               size,                    // IN    item size, sect (def 8)
   ULONG               mstart,                  // IN    initial mark start
   ULONG               msize,                   // IN    initial mark size or 0
   DFSHEXEDINFO       *info,                    // IN    additional object info
   USHORT              startView,               // IN    start in specified view
   USHORT              pos,                     // IN    start pos in 1st item
   USHORT              rows,                    // IN    nr of rows or 0
   USHORT              cols                     // IN    nr of cols or 0
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXTM                title;
   TXHEBUF             pre4;
   TXHEBUF             pre3;
   TXHEBUF             pre2;
   TXHEBUF             prev;
   TXHEBUF             curr;
   TXHEBUF             next;
   TXHEBUF             nex2;
   TXHEBUF             nex3;
   TXHEBUF             nex4;
   TXHEXEDIT           he;
   DFSHEXEDDATA        dfshedat;
   ULN64               sn;

   ENTER();
   TRACES(("lsn:%llx size:%u rows:%3hu cols:%3hu byteSize:%llu name:'%s'\n",
            lsn, size, rows, cols, (info) ? info->al.byteSize : 0, (info) ? info->name : ""));

   dfshedat.initialPos = pos;
   dfshedat.initialLsn = lsn;
   dfshedat.currentLsn = lsn;
   dfshedat.sectors    = (size) ? size : max( 1, (4096 / dfsGetSectorSize())); // will be fixed up for resident data!
   dfshedat.written    = 0;
   dfshedat.info       = info;
   dfshedat.altData    = NULL;

   memset( &pre4, 0, sizeof( TXHEBUF));
   memset( &pre3, 0, sizeof( TXHEBUF));
   memset( &pre2, 0, sizeof( TXHEBUF));
   memset( &prev, 0, sizeof( TXHEBUF));         // no data attached yet ...
   memset( &curr, 0, sizeof( TXHEBUF));         // first data will be read on
   memset( &next, 0, sizeof( TXHEBUF));         // WM_CREATE/INIT_POS callback
   memset( &nex2, 0, sizeof( TXHEBUF));
   memset( &nex3, 0, sizeof( TXHEBUF));
   memset( &nex4, 0, sizeof( TXHEBUF));

   he.posCurBuf      = 0;                       // will be refined by edit dlg
   he.posCursor      = 0;                       // will be refined by edit dlg
   he.hexLsb         = 0;
   he.rows           = rows;
   he.cols           = (cols <= 256) ? cols : 256;
   he.autosize       = FALSE;                   // will become auto, if cols = 0
   he.ascii          = FALSE;
   he.decimal        = FALSE;
   he.pre4           = &pre4;
   he.pre3           = &pre3;
   he.pre2           = &pre2;
   he.prev           = &prev;                   // to be filled by ObjRead
   he.curr           = &curr;
   he.next           = &next;
   he.nex2           = &nex2;
   he.nex3           = &nex3;
   he.nex4           = &nex4;
   he.diff           = NULL;                    // will be handled by edit dlg
   he.diffSize       = 0;                       // will be handled by edit dlg
   he.currCrc        = 0;                       // will be handled by edit dlg
   he.markDump       = FALSE;                   // default mark, a byte-range, not hexDump
   he.markStart      = mstart;
   he.markSize       = (msize) ? msize : (mstart) ? 1 : 0;
   he.markBase       = (msize) ? (ULN64) (lsn * dfsGetSectorSize()) : 0;
   he.userdata       = &dfshedat;
   he.actionCallback = dfsHexEdObjCallback;
   he.wFlags         = TXHE_MOVEABLE;           // window control flags
   he.altView        = startView;               // start view specified
   strcpy( he.altDispText, "AltView");          // <F2> = allternative view

   if (TxaOption(TXA_O_SCREEN))                 // full 'screen' option
   {
      he.wFlags     |= TXHE_MAXIMIZE;           // maximize HEX window on start
   }
   strcpy( title, "Sector edit: ");
   switch (dfstStoreType(  DFSTORE))
   {
      case DFST_INVALID:
      case DFST_UNUSED:   rc = DFS_CMD_FAILED;                             break;
      case DFST_RAWIMAGE: strcat( title, dfstStoreDesc2( DFSTORE) + 10);   break;
      default:            strcat( title, dfstStoreDesc1( DFSTORE) + 10);   break;
   }
   if (rc == DFS_CMD_FAILED)
   {
      TxNamedMessage( !dfsa->batch, 0, " ERROR: No object opened ",
                      "No object is opened in the current store, to allow sector editing");
   }
   else
   {
      if (info && info->al.space)               // SPACE, alloc or resident
      {
         TRACES(("Sspace byteSize: %u\n", info->al.byteSize));
         if (info->al.chunks == 0)              // resident, display meta-lsn
         {
            //- set sectors/buffer to 4, to allow up to 2 KiB resident data
            dfshedat.sectors = 4;

            dfsX10(  "Editor initial SN : ", info->al.meta, "", "   ");
            if (info->al.xatype == DFSXA_NONE)
            {
               TxPrint( "(Meta, %llu bytes at offset %hu)\n", info->al.byteSize, info->al.offset);

            }
            else                                // some kind of Xattr
            {
               TxPrint( "(%s, %llu bytes)\n", dfsXattrDescription( info->al.xatype), info->al.byteSize);
            }
         }
         else                                   // space allocation
         {
            sn = dfsSspaceRsn2Lsn( info->al.chunks, info->al.space, lsn);
            dfsX10( "Editor initial SN : ", sn,  "", "   ");
            dfsX10( "Block: ",     dfsSn2Cl(sn), "", "   (RSN:0 for allocated file space)\n");
         }
      }
      else                                      // regular disk editing
      {
         dfsX10( "Editor initial SN : ", lsn,  "", "   ");
         dfsX10( "Block: ",     dfsSn2Cl(lsn), "", "   ");
         dfsX10( "(PSN: ", dfstLSN2Psn( DFSTORE, lsn), "", " on disk)\n");
      }
      rc = txwHexEditor( TXHWND_DESKTOP, TXHWND_DESKTOP, &he, title, DFS_H_HEXED);

      if ((dfshedat.currentLsn != lsn) ||       // current LSN has changed
          (dfshedat.written    != 0  )  )       // or sectors modified
      {
         ULN64         xLsn;

         if (info && info->al.space)            // SPACE, alloc or resident
         {
            if (info->al.chunks == 0)           // resident, display meta-lsn
            {
               xLsn = info->al.meta + dfshedat.currentLsn; // close approximation
            }
            else                                // space allocation
            {
               xLsn = dfsSspaceRsn2Lsn( info->al.chunks, info->al.space, dfshedat.currentLsn);
            }
         }
         else
         {
            xLsn = dfshedat.currentLsn;
         }
         dfsX10( "Editor exit at SN : ", xLsn,  "", "   ");
         dfsX10( "Block: ",     dfsSn2Cl(xLsn), "", "   ");
         if (dfshedat.written  != 0)
         {
            TxPrint( "%u items modified or written back", dfshedat.written);
         }
         TxPrint( "\n");
         if (dfshedat.currentLsn != lsn)
         {
            nav.down = dfshedat.currentLsn;     // exit sector on ENTER ...
            nav.this = dfshedat.currentLsn;
            nav.xtra = lsn;                     // X to initial LSN
         }
      }
      TxFreeMem( pre4.data);
      TxFreeMem( pre3.data);
      TxFreeMem( pre2.data);
      TxFreeMem( prev.data);
      TxFreeMem( curr.data);
      TxFreeMem( next.data);
      TxFreeMem( nex2.data);
      TxFreeMem( nex3.data);
      TxFreeMem( nex4.data);
   }
   TxCancelAbort();                             // avoid propagated aborting
   RETURN (rc);
}                                               // end 'dfsHexObjectEditor'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Callback function to service HEXEDIT service requests
/*****************************************************************************/
static ULONG dfsHexEdObjCallback
(
   ULONG               cmd,                     // IN    action command
   TXHEXEDIT          *dat                      // INOUT hexedit data
)
{
   ULONG               rc     = NO_ERROR;         // function return
   DFSHEXEDDATA       *bd     = dat->userdata;
   ULN64               lsn    = bd->currentLsn;
   ULN64               size   = bd->sectors;
   LONG                index  = dat->posCursor + dat->posCurBuf;
   LONG                wbytes = dat->rows * dat->cols;
   ULN64               objectSectors;

   ENTER();
   TRACES(( "cmd: %u  on lsn: %llx\n", cmd, lsn));

   if (bd && bd->info && bd->info->al.space)    // we have S_SPACE allocation info
   {
      if (bd->info->al.chunks == 0)             // resident, no real SPACE
      {
         objectSectors = ((bd->info->al.byteSize - 1) / dfsGetSectorSize()) + 1;
      }
      else
      {
         objectSectors = dfsSspaceSectors( TRUE, bd->info->al.chunks, bd->info->al.space);
      }
   }
   else
   {
      objectSectors = dfsGetLogicalSize();
   }

   switch (cmd)
   {
      case TXHE_CB_UPD_DESC:
         dfsSetBufferDescr( lsn - (size * 4), dat, dat->pre4);
         dfsSetBufferDescr( lsn - (size * 3), dat, dat->pre3);
         dfsSetBufferDescr( lsn - (size * 2), dat, dat->pre2);
         dfsSetBufferDescr( lsn -  size,      dat, dat->prev);
         dfsSetBufferDescr( lsn,              dat, dat->curr);
         dfsSetBufferDescr( lsn +  size,      dat, dat->next);
         dfsSetBufferDescr( lsn + (size * 2), dat, dat->nex2);
         dfsSetBufferDescr( lsn + (size * 3), dat, dat->nex3);
         dfsSetBufferDescr( lsn + (size * 4), dat, dat->nex4);
         break;

      case TXHE_CB_INIT_POS:                    // read all buffer contents
         lsn = bd->initialLsn;                  // using initial sector nr
         rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
         dfsPositionCursorLine(  bd->initialPos, dat);
         break;

      case TXHE_CB_INSERT_1:                    // insert or delete byte
      case TXHE_CB_DELETE_1:                    // at the current position
      case TXHE_CB_DELTOEND:                    // delete to end of file
         if ((SINF->stFlags & DFSTFL_EXPLICITSIZE) == 0)
         {
            if (cmd == TXHE_CB_DELTOEND)
            {
               if ((dat->curr->size != 0) && (dat->curr->size > index +1))
               {
                  if ((dfsa->batch) || (TxConfirm( 5672,
                     "Delete all bytes AFTER the current cursor position ? [Y/N] : ")))
                  {
                     dat->curr->size = index +1;
                     dfstSetByteSize( DFSTORE, dat->curr->start + dat->curr->size);

                     rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
                     if (rc == NO_ERROR)
                     {
                        rc = TX_PENDING;        // signal buffers have changed
                     }
                  }
               }
            }
            else                                // insert or delete 1 byte
            {
               if (((index == (dat->curr->size  - 1)   || // at end of object
                              (dat->curr->size == 0))) && // or empty object
                             ((dat->next == NULL)      ||
                              (dat->next->data == NULL)))
               {
                  ULONG   bufsize = size * dfsGetSectorSize();

                  if (cmd == TXHE_CB_INSERT_1)
                  {
                     if (dat->curr->size != 0)
                     {
                        dat->posCursor++;
                     }
                     if (dat->posCursor >= wbytes)
                     {
                        dat->posCursor -= dat->cols;
                        dat->posCurBuf += dat->cols;
                     }
                     if (dat->curr->size < bufsize)
                     {
                        dat->curr->size++;
                        dfstSetByteSize( DFSTORE, dat->curr->start + dat->curr->size);
                     }
                     else                       // need to get a NEXT buffer
                     {
                        if (dat->currCrc != TxCrc32( dat->curr->data, dat->curr->size))
                        {
                           dfsHexEdWriteBack( dat); // to be refined, Cancel ?
                        }
                        lsn += size;
                        rc = dfsReadAllBuffers( lsn, size, TRUE, dat);
                        dfstSetByteSize( DFSTORE, dat->curr->start +1);
                        if (rc == NO_ERROR)
                        {
                           rc = TX_PENDING;     // signal buffers have changed
                        }
                        dat->posCurBuf = dat->curr->size - dat->posCursor -1;
                     }
                  }
                  else                          // delete 1 byte
                  {
                     if (dat->curr->size != 0)
                     {
                        if (dat->posCursor == 0)
                        {
                           dat->posCursor += dat->cols;
                           dat->posCurBuf -= dat->cols;
                        }
                        dat->posCursor--;
                        dat->curr->size--;
                        dfstSetByteSize( DFSTORE, dat->curr->start + dat->curr->size);
                        if (dat->curr->size == 0)
                        {
                           if (lsn >= size)
                           {
                              lsn -= size;
                              rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
                              if (rc == NO_ERROR)
                              {
                                 rc = TX_PENDING; // signal buffers have changed
                              }
                              dat->posCurBuf = dat->curr->size - dat->posCursor -1;
                           }
                           else
                           {
                              dat->posCursor = 0;
                              dat->posCurBuf = 0;
                           }
                        }
                     }
                  }
               }
               else
               {
                  TxNamedMessage( !dfsa->batch, 0, " ERROR: Not at end of object ",
                                  "Insert/Delete is only possible at the very end of the object! (use Ctrl-End)");
               }
            }
         }
         else
         {
            TxNamedMessage( !dfsa->batch, 0, " ERROR: Fixed size object ",
                            "Edited object has a FIXED size, Insert/Delete is not possible.");
         }
         break;

      case TXHE_CB_NEX4_BUF:
         if ((lsn + size) < objectSectors)
         {
            lsn += size;
         }
      case TXHE_CB_NEX3_BUF:
         if ((lsn + size) < objectSectors)
         {
            lsn += size;
         }
      case TXHE_CB_NEX2_BUF:
         if ((lsn + size) < objectSectors)
         {
            lsn += size;
         }
      case TXHE_CB_NEXT_BUF:
         if ((lsn + size) < objectSectors)
         {
            lsn += size;
         }
         rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
         break;

      case TXHE_CB_PRE4_BUF:
         if (lsn > 0)
         {
            lsn -= size;
         }
         else
         {
            lsn = 0;
         }
      case TXHE_CB_PRE3_BUF:
         if (lsn > 0)
         {
            lsn -= size;
         }
         else
         {
            lsn = 0;
         }
      case TXHE_CB_PRE2_BUF:
         if (lsn > 0)
         {
            lsn -= size;
         }
         else
         {
            lsn = 0;
         }
      case TXHE_CB_PREV_BUF:
         if (lsn > size)
         {
            lsn -= size;
         }
         else
         {
            lsn = 0;
         }
         rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
         break;

      case TXHE_CB_TO_START:
         lsn  = 0;
         rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
         break;

      case TXHE_CB_TO_FINAL:
         if (objectSectors > size)
         {
            lsn  = objectSectors - size;
         }
         else
         {
            lsn = 0;
         }
         rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
         break;

      case TXHE_CB_GOTOITEM:
         rc = dfsGetGotoLsnValue( "Sector edit, goto new location", &lsn);
         if ((rc == NO_ERROR) && (lsn < (objectSectors - size)))
         {
            rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
            dat->posCursor = 0;
            dat->posCurBuf = 0;
         }
         break;

      case TXHE_CB_FINDDATA:
      case TXHE_CB_FNDAGAIN:
         while (index >= dfsGetSectorSize())    // adjust sector boundary
         {
            index -= dfsGetSectorSize();
            lsn++;
         }
         if (dfsFindDialog( TRUE, TRUE, (cmd == TXHE_CB_FNDAGAIN),
                            lsn, index) == NO_ERROR)
         {
            USHORT     offset;

            dfsX10( "Search hit at LSN : ", nav.this, "", "    ");
            TxPrint( "at byte offset: 0x%2.2x\n", nav.offset);

            if (dfsGetClusterSize() != 1)       // adjust to cluster boundary
            {
               ULONG    diff = nav.this  %   dfsGetClusterSize();

               lsn    = nav.this   -  diff;
               offset = nav.offset + (diff * dfsGetSectorSize());
            }
            else
            {
               lsn    = nav.this;
               offset = nav.offset;             // set by find dialog
            }
            dat->markStart = offset;
            dat->markSize  = dfsa->lastFindLen;
            dat->markBase  = (ULN64) (lsn * dfsGetSectorSize());
            rc = dfsReadAllBuffers( lsn, size, FALSE, dat);
            dfsPositionCursorLine(  offset, dat);
         }
         else
         {
            TxNamedMessage( TRUE, 0, " ERROR: Not found, or canceled ",
                            "Find canceled, or searched item could not be found in the opened object");
         }
         break;

      case TXHE_CB_ALT_DISP:                    // alternate display, DFSee edit view
         {
            ULONG               cycleCount;     // alternate view cycle count
            ULONG               currentView;

            if ((dat->altView == DFSH_AUTOVIEW) ||
                (dat->altView == DFSH_HEX_VIEW)  )
            {
               currentView  = dfsHexEdSetAutoView( dat, (dat->altView == DFSH_AUTOVIEW));
               dat->altView = DFSH_HEX_VIEW;    // only determine it once per session
            }
            else
            {
               currentView  = dat->altView;     // start with specified
            }
            for (cycleCount = 0; cycleCount < DFSH_MAX_VIEW; cycleCount++)
            {
               switch (currentView)
               {
                  case DFSH_DIS_VIEW:
                     rc = dfsDisAsmPopup( dat, &dat->markSize);
                     if (rc == NO_ERROR)
                     {
                        lsn   = bd->currentLsn; // may have changed ...
                        index = dat->posCursor + dat->posCurBuf;
                        dat->markStart = index;
                        dat->markBase  = (ULN64) (lsn * dfsGetSectorSize());
                        dfsPositionCursorLine( index, dat);
                     }
                     break;

                  case DFSH_ASC_VIEW:
                     rc = dfsTxViewPopup( dat);
                     if (rc == NO_ERROR)
                     {
                        lsn   = bd->currentLsn; // may have changed ...
                        index = dat->posCursor + dat->posCurBuf;
                        dfsPositionCursorLine( index, dat);
                     }
                     break;

                  default:                      // undefined, or HEX, return OK
                     break;
               }

               if (rc == DFS_DID_NEXT_VIEW)     // set next view to be used
               {
                  if (++currentView > DFSH_MAX_VIEW)
                  {
                     currentView = DFSH_1ST_VIEW;
                  }
               }
               else                             // exist with OK or an error
               {
                  break;
               }
            }
         }
         break;

      case TXHE_CB_WRITEBUF:
         rc = dfsHexEdWriteBack( dat);
         break;

      default:
         TRACES(("Unknown callback function number %u requested!\n", cmd));
         break;
   }
   RETURN (rc);
}                                               // end 'dfsHexEdObjCallback'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine most suitable view format based on current sector data
/*****************************************************************************/
static USHORT dfsHexEdSetAutoView               // RET   best view ID
(
   TXHEXEDIT          *dat,                     // IN    hexedit data
   BOOL                inclHex                  // IN    HEX allowed too
)
{
   USHORT              rc      = DFSH_ASC_VIEW; // function return
   DFSHEXEDDATA       *bd      = dat->userdata;
   char                c;
   int                 i;
   int                 limit   = dfsGetSectorSize() / 2; // default minimum ASCII size

   if (bd->info && bd->info->al.byteSize && (bd->info->al.byteSize < limit))
   {
      limit = bd->info->al.byteSize;
   }

   if (dat->curr->data == NULL)                 // no data read yet
   {
      rc = dfsReadAllBuffers( bd->currentLsn, bd->sectors, FALSE, dat);
   }

   if (dat->curr->data)                         // could still be NULL (empty file)
   {
      //- First see if (start of) current buffer is all ASCII (or up to EOF char)
      for (i = 0; i < limit; i++)
      {
         c = dat->curr->data[i];
         if (c == 0x26)                         // ASCII EOF
         {
            break;                              // should be considered valid ASCII
         }
         else if (c < TXk_TAB)                  // very unlikely to be ASCII
         {
            TRACES(("Not ASCII caused by char: 0x%2.2hhx at offset:0x%3.3x\n", c, i));
            rc = DFSH_DIS_VIEW;
            break;
         }
      }

      if (rc == DFSH_DIS_VIEW)
      {
         //- Use ASM, unless HEX allowed, and first byte is NOT a known ASM instruction
         if (inclHex)
         {
            c = dat->curr->data[0];
            if ((c != DFS_M_OPCO_1     ) &&     // start is NOT like PC-ASM
                (c != DFS_M_OPCO_2     ) &&
                (c != DFS_B_OPCO_1     ) &&
                (c != DFS_B_OPCO_2     ) &&
                (c != DFS_B_OPCNOP     ) &&
                (c != DFS_B_OPCJE      ) &&
                (c != DFS_B_OPCJNE     ) &&
                (c != DFS_B_OPCJMPS    )  )
            {
               TRACES(("Not DISASM because of 1st byte is NOT a likely opcode\n"));
               if ((memcmp( dat->curr->data, "MZ",    2) == 0) || // OS/2 or WIN EXE/DLL
                   (memcmp( dat->curr->data, "LX",    2) == 0) || // OS/2 DLL or executable
                   (memcmp( dat->curr->data, "PK",    2) == 0) || // Classic ZIP file
                   (memcmp( dat->curr->data, "HSP",   3) == 0) || // OS/2 INF or HLP
                   (memcmp( dat->curr->data, "%PDF",  4) == 0)  ) // PDF files
               {
                  TRACES(("ASCII because of known prefix bytes\n"));
                  rc = DFSH_ASC_VIEW;           // Force the ASCII editor
               }
               else
               {
                  rc = DFSH_HEX_VIEW;           // revert to the HEX editor, unless ...
               }
            }
         }
         else                                   // no HEX allowed, re-evaluate ASCII
         {
            if ((memcmp( dat->curr->data, "MZ",    2) == 0) || // OS/2 or WIN EXE/DLL
                (memcmp( dat->curr->data, "LX",    2) == 0) || // OS/2 DLL or executable
                (memcmp( dat->curr->data, "PK",    2) == 0) || // Classic ZIP file
                (memcmp( dat->curr->data, "HSP",   3) == 0) || // OS/2 INF or HLP
                (memcmp( dat->curr->data, "%PDF",  4) == 0)  ) // PDF files
            {
               TRACES(("ASCII because of known prefix bytes\n"));
               rc = DFSH_ASC_VIEW;              // Force the ASCII editor
            }
            else                                // test ASCII percentage first sector
            {
               for (limit = 0, i = 0; i < dfsGetSectorSize(); i++)
               {
                  c = dat->curr->data[i];
                  if (TxIsPureAscii(c))
                  {
                     limit++;
                  }
               }
               if (limit > (dfsGetSectorSize() * 9 / 10)) // more than 90% pure ASCII
               {
                  TRACES(("ASCII because %d bytes are pure ASCII\n", limit));
                  rc = DFSH_ASC_VIEW;           // Force the ASCII editor
               }
            }
         }
      }
   }
   else                                         // could be an empty file
   {
      rc = DFSH_HEX_VIEW;                       // revert to the HEX editor
   }
   RETURN (rc);
}                                               // end 'dfsHexEdSetAutoView'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Writeback current buffer after user-confirmation popup, incl SPACE support
/*****************************************************************************/
static ULONG dfsHexEdWriteBack
(
   TXHEXEDIT          *dat                      // INOUT hexedit data
)
{
   ULONG               rc      = NO_ERROR;      // function return
   DFSHEXEDDATA       *bd      = dat->userdata;
   ULN64               sn      = bd->currentLsn; // lsn or SPACE RSN
   ULN64               lsn     = sn;
   ULN64               size    = bd->sectors;
   ULONG               changes = 0;
   ULONG               i;
   TXLN                errormsg;

   ENTER();

   if (dat->currCrc != TxCrc32( dat->curr->data, dat->curr->size))
   {
      if (dat->diff && dat->diffSize)
      {
         for (i = 0; i < dat->curr->size; i++) // only REAL size in CURR
         {
            if (dat->diff[ i] != dat->curr->data[ i])
            {
               changes++;
            }
         }
      }
      else
      {
         changes = dat->diffSize;         // worst case :-)
      }
   }
   if (bd && bd->info && bd->info->al.space)    // we have S_SPACE allocation info
   {
      if (bd->info->al.chunks != 0)             // regular allocated file data
      {
         //- assumes bd->sectors size is always in a single chunk! (block/cluster)
         lsn = dfsSspaceRsn2Lsn( bd->info->al.chunks, bd->info->al.space, sn);
      }
      else                                      // resident or Xattr/EA data
      {
         if (bd->info->al.xatype == DFSXA_NONE)
         {
            if (dfsa->FsWriteMetaSpace != NULL)
            {
               if (TxConfirm( 5670, "WRITE %llu byte resident data area, with %u changed "
                                    "byte(s) back to Meta-data-Lsn: 0x0%llx - offset: %hu\n\n"
                                    "(to UNDO the changes, reply [No], then use " DFSHE_UNDO_KEY ")\n\n"
                                    "Are you sure you want to WRITE this item back ? [Y/N] : ",
                      bd->info->al.byteSize, changes, bd->info->al.meta, bd->info->al.offset))
               {
                  dfsX10(  "Write Meta at LSN : ", bd->info->al.meta, "", "  - ");
                  TxPrint( "offset: %hu, changed bytes: %u\n", bd->info->al.offset, changes);

                  //- Copy actual modified data from current buffer to resident al.space
                  memcpy( bd->info->al.space, dat->curr->data, bd->info->al.byteSize);

                  if ((rc = DFSFNCALL(dfsa->FsWriteMetaSpace, 0, 0, NULL, &bd->info->al)) == NO_ERROR)
                  {
                     static BOOL giveRebootHint = TRUE;

                     if (giveRebootHint)
                     {
                        TxNamedMessage( TRUE, 0, " INFO: reboot recommended ",
                           "Resident (meta) data successfully written. When the filesystem being worked "
                           "on is mounted (accessible outside DFSee), make sure NOT to access the related "
                           "file in any other way, before the next REBOOT, or the changes may get lost!\n\n"
                           "In case of doubt, REBOOT NOW");
                        giveRebootHint = FALSE;
                     }
                     bd->written++;             // count sectors written
                  }
                  else
                  {
                     TxStripAnsiCodes( dfsGetRcString( rc, errormsg));
                     dfsX10(  "Error writing LSN : ", bd->info->al.meta, "", "    changes lost!");
                     TxPrint( "%s\n", errormsg); // for LOG
                     TxNamedMessage( TRUE, 0, " ERROR: write back failed ",
                        "Error writing back the resident data to Meta-data-Lsn: 0x0%llx\n%s%s",
                         bd->info->al.meta, errormsg,
                        (changes) ? "\n\nYou can use " DFSHE_UNDO_KEY " to undo changes and move on" : "");
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
            else
            {
               TxNamedMessage( TRUE, 0, " INFO: Not implemented ",
                                   "Writing back resident data not supported for Mode=%s\n\n"
                                   "Manual update: meta-data-lsn: 0x0%llx, at offset: %hu",
                                    SINF->afsys, bd->info->al.meta, bd->info->al.offset);
            }
         }
         else                                   // some kind of Xattr/EA data
         {
            TxNamedMessage( TRUE, 0, " INFO: Not implemented ",
                            "Writing back Xattr/EA data is not supported for Mode=%s", SINF->afsys);
         }
         lsn = L64_NULL;                        // and no 'regular' write ...
      }
   }
   if (lsn < LSN_SPARSE)                         // regular data-area to be written
   {
      if (TxConfirm( 5670, "WRITE item at Lsn: 0x0%llx with %u changed "
                           "byte(s) back to object:\n%s\n%s\n\n"
                           "(to UNDO the changes, reply [No], then use " DFSHE_UNDO_KEY ")\n\n"
                           "Are you sure you want to WRITE this item back ? [Y/N] : ",
                            lsn, changes, dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE)))
      {
         dfsX10(  "Write item at LSN : ", sn, "", "    ");
         TxPrint( "changed bytes: %u\n", changes);

         rc = dfsWrite( lsn, size, dat->curr->data);
         if (rc == NO_ERROR)
         {
            bd->written++;                      // count sectors written
         }
         else
         {
            TxStripAnsiCodes( dfsGetRcString( rc, errormsg));
            dfsX10(  "Error writing LSN : ", lsn, CBR, "    changes lost!");
            TxPrint( "%s\n", errormsg);         // for LOG
            TxNamedMessage( TRUE, 0, " ERROR: write back failed ",
               "Error writing back the current item to opened object:\n%s\n%s\n%s%s",
               dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE), errormsg,
              (changes) ? "\n\nYou can use " DFSHE_UNDO_KEY " to undo changes and move on" : "");
         }
      }
      else
      {
         rc = DFS_NO_CHANGE;
      }
   }
   else
   {
      rc = ERROR_WRITE_FAULT;
   }
   RETURN (rc);
}                                               // end 'dfsHexEdWriteBack'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Position line, zero-based or with cursor centered and largest area visible
/*****************************************************************************/
static void dfsPositionCursorLine
(
   USHORT              offset,                  // IN    offset in buffer
   TXHEXEDIT          *dat                      // INOUT hexedit data
)
{
   LONG                wbytes = dat->rows * dat->cols;

   ENTER();
   TRACES(( "offset: %hu  wbytes: %d  dat->rows: %hu\n", offset, wbytes, dat->rows));

   dat->posCursor = offset;
   dat->posCurBuf = 0;

   if (dat->posCursor >= wbytes)                // zero-based not possible
   {
      TRACES(( "Adjust needed, object size is: %u bytes\n", dat->curr->size));
      while ( (dat->posCursor > (wbytes  / 2)) &&
             ((dat->posCurBuf +  wbytes) < dat->curr->size))
      {
         dat->posCursor -= dat->cols;           // adjust to get cursor in
         dat->posCurBuf += dat->cols;           // visible window area
         TRACES(( "Adjusted to posCurBuf: %d  posCursor: %d\n",
                          dat->posCurBuf, dat->posCursor));
      }
   }
   VRETURN ();
}                                               // end 'dfsPositionCursorLine'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set Edit-data values and read prev/curr/next data buffers
/*****************************************************************************/
static ULONG dfsReadAllBuffers
(
   ULN64               newSn,                   // IN    SN for CURR buffer
   ULONG               sectors,                 // IN    sectors per buffer
   BOOL                fakeNew,                 // IN    no read, fake 1 byte
   TXHEXEDIT          *dat                      // INOUT hexedit data
)
{
   ULONG               rc  = NO_ERROR;          // function return
   DFSHEXEDDATA       *bd  = dat->userdata;

   ENTER();
   TRACES(( "newSn:%llx sectors:%u\n", newSn, sectors));

   bd->sectors    = sectors;
   bd->currentLsn = newSn;

   dfsReadObjectBuffer( newSn - (sectors * 4), sectors, FALSE,   dat, dat->pre4);
   dfsReadObjectBuffer( newSn - (sectors * 3), sectors, FALSE,   dat, dat->pre3);
   dfsReadObjectBuffer( newSn - (sectors * 2), sectors, FALSE,   dat, dat->pre2);
   dfsReadObjectBuffer( newSn -  sectors,      sectors, FALSE,   dat, dat->prev);
   rc = dfsReadObjectBuffer(       newSn,      sectors, fakeNew, dat, dat->curr);
   dfsReadObjectBuffer( newSn +  sectors,      sectors, FALSE,   dat, dat->next);
   dfsReadObjectBuffer( newSn + (sectors * 2), sectors, FALSE,   dat, dat->nex2);
   dfsReadObjectBuffer( newSn + (sectors * 3), sectors, FALSE,   dat, dat->nex3);
   dfsReadObjectBuffer( newSn + (sectors * 4), sectors, FALSE,   dat, dat->nex4);

   RETURN (rc);
}                                               // end 'dfsReadAllBuffers'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read buffer from specified LSN when valid, fill buffer structure and descr
/*****************************************************************************/
static ULONG dfsReadObjectBuffer
(
   ULN64               rsn,                     // IN    relative sn of 1st sector
   ULONG               size,                    // IN    size in sectors
   BOOL                fakeNew,                 // IN    no read, fake 1 byte
   TXHEXEDIT          *dat,                     // IN    HEXEDIT data
   TXHEBUF            *buf                      // INOUT buffer data
)
{
   ULONG               rc  = NO_ERROR;          // function return
   DFSHEXEDDATA       *bd  = dat->userdata;
   ULN64               lsn = rsn;               // SN, relative to opened object
   ULN64               objectSectors;           // size of object in sectors
   ULN64               objectSize;              // exact size of object in bytes
   ULONG               bufAllocSize;            // allocated buffer size
   BOOL                resident = FALSE;

   ENTER();
   TRACES(("size:%u rsn: 0x%llx\n", size, rsn));

   if (buf != NULL)
   {
      TxFreeMem( buf->data);                    // free existing data

      if (bd && bd->info && bd->info->al.space) // we have S_SPACE allocation info
      {
         objectSize    = bd->info->al.byteSize;
         if (bd->info->al.chunks == 0)          // resident data, allocated to space
         {
            lsn = 0;
            objectSectors = (objectSize) ? ((objectSize - 1) / dfsGetSectorSize()) + 1: 0;
            resident      = TRUE;
         }
         else
         {
            lsn = dfsSspaceRsn2Lsn( bd->info->al.chunks, bd->info->al.space, rsn);
            objectSectors = dfsSspaceSectors( TRUE, bd->info->al.chunks, bd->info->al.space);
         }
      }
      else
      {
         objectSize    = dfstGetByteSize( DFSTORE);
         objectSectors = dfsGetLogicalSize();
      }
      bufAllocSize = (ULONG) size * dfsGetSectorSize();
      buf->size    = bufAllocSize;
      buf->start   = (ULN64) rsn  * dfsGetSectorSize(); // start is RELATIVE too!
      buf->bps     = dfsGetSectorSize();        // for relative LSN/PSN indicators

      if (buf->start < objectSize)              // buffer starts before end-of-object
      {
         if ((buf->data = TxAlloc( size, dfsGetSectorSize())) != NULL)
         {
            TRACES(("lsn:%llx rsn:%llx start:%12.12llx size:%12llu bytes, sectors: 0x%8.8x%\n",
                     lsn,     rsn,  buf->start,  objectSize,        objectSectors));

            if (!fakeNew)
            {
               if (rsn < objectSectors)         // valid rsn, within source data area
               {
                  if (((buf->start            ) < objectSize) && // fixup exact size
                      ((buf->start + buf->size) > objectSize)  ) // (last, partial chunk)
                  {
                     buf->size = (ULONG) objectSize - buf->start;
                  }
                  if (resident)                 // copy (part of) the data to buffer
                  {
                     BYTE *data = ((BYTE *) bd->info->al.space) + buf->start;

                     TRACES(("Meta:%llx, get resident: %u bytes from offset: %u\n",
                              bd->info->al.meta, buf->size, buf->start));
                     memcpy( buf->data, data, buf->size);
                  }
                  else                          // read using S_SPACE alloc
                  {
                     if (lsn != LSN_SPARSE)     // regular LSN, or out of bounds
                     {
                        if ((rc = dfsRead( lsn, size, buf->data)) != NO_ERROR)
                        {
                           buf->size = 0;
                           TxFreeMem( buf->data); // free buffer data
                        }
                        else if (buf->size < bufAllocSize) // clear excess of last sector(s)
                        {
                           memset( buf->data + buf->size, 0, bufAllocSize - buf->size);
                        }
                     }
                     else                       // SPARSE, fill zeroes
                     {
                        memset( buf->data, 0, bufAllocSize);
                     }
                  }
               }
               else
               {
                  buf->size = 0;
                  TxFreeMem( buf->data);        // free buffer data
                  rc = DFS_PSN_LIMIT;
               }
            }
            else                                // fake 1-byte new buffer
            {
               dfstLogicalLimits( DFSTORE, 0, lsn); // new start/final PSN
               memset( buf->data, 0xfe, buf->size);
               buf->size = 1;                   // single-byte new buffer
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         buf->size = 0;
         rc = DFS_PSN_LIMIT;
      }
      dfsSetBufferDescr( rsn, dat, buf);
   }
   RETURN (rc);
}                                               // end 'dfsReadObjectBuffer'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set description string for a data buffer
/*****************************************************************************/
static void dfsSetBufferDescr
(
   ULN64               sn,                      // IN    lsn/rsn of 1st sector
   TXHEXEDIT          *dat,                     // IN    HEXEDIT data
   TXHEBUF            *buf                      // INOUT buffer data
)
{
   DFSHEXEDDATA       *bd = dat->userdata;
   BYTE                st = ST_UDATA;
   TXTT                ascii;
   TXTT                info;                    // additional item info
   ULONG               c,h,s;                   // chs numbering

   ENTER();

   if (buf)
   {
      if (buf->data)
      {
         st = dfsIdentifySector(sn, 0, buf->data);
         dfsSectorTypeAsAscii( (BYTE)(st & ~ST__INFO), ascii);
      }
      else
      {
         strcpy( ascii, "No data, EMPTY!  ");
      }
      if (bd && bd->info && bd->info->al.space) // SPACE alloc or resident data
      {
         if (bd->info->al.chunks != 0) // we have S_SPACE allocation info
         {
            //- note that passed lsn really is the RSN!
            ULN64 lsn = dfsSspaceRsn2Lsn( bd->info->al.chunks, bd->info->al.space, sn);

            sprintf( buf->desc, (dat->decimal) ?
                     "%s at LSN:%12llu RSN=%12llu " :
                     "%s at LSN:0x%10.10llx RSN=0x%10.10llx ", ascii, lsn & DFS_SN10, sn & DFS_SN10);
         }
         else                                   // resident or Xattr/EA data
         {
            if (bd->info->al.xatype == DFSXA_NONE)
            {
               sprintf( buf->desc, "%s Resident, Meta-sector:0x%10.10llx ",  ascii, bd->info->al.meta & DFS_SN10);
            }
            else
            {
               sprintf( buf->desc, "%s Xattr-format: %s, ", ascii, dfsXattrDescription( bd->info->al.xatype));
            }
         }
         dfstr64XiB( buf->desc, "Size: ", bd->info->al.byteSize, " '");
         if (strlen( bd->info->name))
         {
            strcat(  buf->desc, bd->info->name);
            strcat(  buf->desc, "'");
         }
      }
      else                                      // regular object sector area
      {
         ULN64      psn = dfstLSN2Psn( DFSTORE, sn);

         if ((sn != psn) && (dfsGetClusterSize() != 1))
         {
            sprintf( info,    " Block:0x%10.10llx ", dfsSn2Cl( sn) & DFS_SN10);
         }
         else
         {
            dfsGeoPsn2Chs( psn, dfstGeoHeads(   DFSTORE),
                                dfstGeoSectors( DFSTORE), &c, &h, &s);
            sprintf( info,      "CHS:%7u %3u %-3u", c,  h,  s);
         }
         sprintf( buf->desc, (dat->decimal) ?
                  "%s at LSN:%12.12llu PSN=%12.12llu  %s Size: %u" :
                  "%s at LSN:0x%10.10llx PSN=0x%10.10llx  %s  Size: %u",
                  ascii,  sn & DFS_SN10, psn & DFS_SN10, info, buf->size);
      }
   }
   else
   {
      strcpy( buf->desc, "No data available, no object (Disk, Partition, File) opened ?");
   }
   TRACES(("desc: %s\n", buf->desc));
   VRETURN();
}                                               // end 'dfsSetBufferDescr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display Text-viewer ASCII window in lower right of desktop
/*****************************************************************************/
static ULONG dfsTxViewPopup
(
   TXHEXEDIT          *dat                      // IN    binary editor data
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (txwIsWindow( TXHWND_DESKTOP))
   {
      DFSHEXEDDATA    *bd = dat->userdata;
      TXRECT           position;                // reference size/position
      TXWHANDLE        tview;                   // view window & frame
      TXWINDOW         window;                  // view window data
      DFSTXVIEWDATA    dt;                      // text-viewer data
      TXTM             title;                   // dynamic window title
      TXTM             footer;                  // dynamic window footer
      USHORT           ww;                      // window width
      ULONG            lineNr;                  // line number in result

      txwQueryWindowRect( TXHWND_DESKTOP, FALSE, &position);
      TRECTA( "pos ", (&position));

      bd->altData    = &dt;                     // attach text-viewer data

      if (dat->altView == DFSH_ASC_VIEW)        // start in maximized view
      {
         ww       = position.right;
         dt.lines = position.bottom -2;         // nr of lines in window
      }
      else                                      // start as window in LR
      {
         ww       = min(  position.right,       DFSTXV_WWIDTH);
         dt.lines = max( (position.bottom - 8), DFSTXV_WLINES);
      }

      dt.offsets  = TRUE;                       // show offsets by default
      dt.style    = DFSAF_PURE;                 // pure SASCII, no split-lines
      dt.strings  = 3;                          // minimal ASCII string length 3
      dt.textSize = bd->sectors * dfsGetSectorSize(); // total buffer / 3

      strcpy( title,  "");                      // filled in by ReadTxView()
      strcpy( footer, "");

      TRACES(( "textSize:%u lines:%u\n", dt.textSize, dt.lines));

      if ((dt.text = TxAlloc( dt.textSize + 1,  sizeof( char *))) != NULL)
      {
         position.left   = position.right -ww; // start as smallish window
         position.right  =                 ww; // in lower right corner
         position.top    = position.bottom - dt.lines -2;
         position.bottom =                   dt.lines +2;

         txwSetupWindowData(
            position.top, position.left, position.bottom, position.right,
            TXWS_FRAMED       |                 // borders (scroll indicator)
            TXCS_CLOSE_BUTTON |                 // include close button [X]
            TXWS_MOVEABLE,                      // resize
            DFS_H_TXVIEW,
            ' ', ' ', TXWSCHEME_COLORS,
            title, footer,
            &window);
         window.dlgFocusID = 0;                 // focus to dlg window itself
         window.tv.topline = 0;
         window.tv.leftcol = 0;
         window.tv.maxtop  = TXW_INVALID;
         window.tv.maxcol  = TXW_INVALID;
         window.tv.buf     = dt.text;
         tview = txwCreateWindow( TXHWND_DESKTOP, TXW_TEXTVIEW, 0, 0, &window, NULL);

         rc = txwDlgBox( TXHWND_DESKTOP, 0, dfsTxViewDlgProc, tview, dat);
         if (rc <= TXDID_CANCEL)                // on OK or Cancel (Esc)
         {
            rc = NO_ERROR;
         }
         for (lineNr = 0; lineNr <= dt.textSize; lineNr++)
         {
            if (dt.text[lineNr])
            {
               TRACES(("Want to Free line nr: %u, now: %p\n", lineNr, dt.text[lineNr]));
            }
         }
         for (lineNr = 0; lineNr <= dt.textSize; lineNr++)
         {
            TRACES(("Free line nr: %u, now: %p\n", lineNr, dt.text[lineNr]));
            TxFreeMem( dt.text[ lineNr]);       // free and set to NULL
         }
         TxFreeMem( dt.text);                   // free array, including lines
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN (rc);
}                                               // end 'dfsTxViewPopup'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the Text-viewer popup window dialog
/*****************************************************************************/
static ULONG dfsTxViewDlgProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXHEXEDIT          *dat = (TXHEXEDIT *) txwQueryWindowPtr( hwnd, TXQWP_USER);

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if ((hwnd != 0) && (dat != NULL))
   {
      TXWINDOW        *win = txwWindowData( hwnd);
      DFSHEXEDDATA    *bd  = dat->userdata;
      DFSTXVIEWDATA   *dt  = bd->altData;       // ASCII text-viewer data
      short            sx  = win->client.right  - win->client.left;
      ULN64            objectSectors;
      ULN64            objectSize;           // exact size of object in bytes

      if (bd && bd->info && bd->info->al.space) // we have S_SPACE allocation info
      {
         objectSize    = bd->info->al.byteSize;
         objectSectors = dfsSspaceSectors( TRUE, bd->info->al.chunks, bd->info->al.space);
         if (objectSize)
         {
            ULN64     usedSectors = ((objectSize - 1) / dfsGetSectorSize() + 1);

            if (usedSectors < objectSectors)
            {
               objectSectors = usedSectors;     // limit to real used sectors
            }
         }
      }
      else
      {
         objectSize    = dfstGetByteSize( DFSTORE);
         objectSectors = dfsGetLogicalSize();
      }

      switch (msg)
      {
         case TXWM_INITDLG:
            txwDefDlgProc( hwnd, msg, mp1, mp2); // set focus and activate
            txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) (dat->curr->data == NULL), 0);
            break;

         case DFSHEV_WM_REFRESH:
            rc = dfsReadTxView( dat, ((BOOL) mp1), sx);
            dfsTxViewTitle( bd, win->title, win->footer);
            txwInvalidateWindow( hwnd, TRUE, TRUE);
            break;

         case TXWM_CHAR:
            switch ((ULONG) mp2)
            {
               case TXk_F2:
                  txwDismissDlg( hwnd, DFS_DID_NEXT_VIEW);
                  break;

               case TXk_F3:                     // avoid application quit :-)
                  txwDismissDlg( hwnd, TXDID_EXIT); // exit editor instead
                  break;

               case TXk_F4:                     // Toggle offset display
                  dt->offsets = !dt->offsets;
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) FALSE, 0);
                  break;

               case TXk_F5:                     // Toggle PURE filtering
                  dt->style   = (dt->style + 1) & DFSAF_MASK;
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) FALSE, 0);
                  break;

               case TXk_F6:                     // Toggle 'strings' length
                  dt->strings = (dt->strings + 1);
                  if (dt->strings >= 10)
                  {
                     dt->strings = 1;
                  }
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) FALSE, 0);
                  break;

               case TXc_HOME:                   // back to initial sector
                  bd->currentLsn  = bd->initialLsn;
                  win->tv.topline = 0;          // and start of that sector
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                  break;

               case TXc_END:                   // go to last sector
                  if (objectSectors > bd->sectors)
                  {
                     bd->currentLsn  = objectSectors - bd->sectors;
                  }
                  else
                  {
                     bd->currentLsn = 0;
                  }
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                  break;

               case TXk_PGDN:
               case TXa_PGDN:
                  if ((bd->currentLsn + bd->sectors) < objectSectors)
                  {
                     bd->currentLsn += bd->sectors;
                  }
                  else if (objectSectors > bd->sectors)
                  {
                     bd->currentLsn  = objectSectors - bd->sectors;
                  }
                  else
                  {
                     bd->currentLsn = 0;
                  }
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                  break;

               case TXk_PGUP:
               case TXa_PGUP:
                  if (bd->currentLsn > bd->sectors)
                  {
                     bd->currentLsn -= bd->sectors;
                  }
                  else
                  {
                     bd->currentLsn  = 0;       // to first sector
                     win->tv.topline = 0;       // and start of that sector
                  }
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                  break;

               case TXa_F5:                     // maximize/toggle
               case TXa_F10:                    // adapt to new window
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2); // resize
                  txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) FALSE, 0);
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);

                  //- redraw, fixing up possible screen corruption by owner
                  txwInvalidateWindow( hwnd, TRUE, TRUE);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsTxViewDlgProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Fill all buffers and convert raw sector contents into ASCII array/title
/*****************************************************************************/
static ULONG dfsReadTxView                      // RET   result
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + plugin data
   BOOL                readBuf,                 // IN    read buffers too
   short               with                     // IN    clipping with
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *tripleBuf = NULL;        // Single binary buffer, from 3
   TX1K                line;                    // pure ASCII line contents
   TXTM                split;                   // split indicator line
   TXTT                add;                     // a few characters to add
   ULONG               asciiCount = 0;          // number of ascii-printables
   ULONG               splitCount = 0;          // number of non-printables
   ULONG               bufSize;                 // size of single buffer
   ULONG               lineNr = 0;              // line number in result
   int                 i;                       // byte index in triple buffer
   char                c;
   TXTS                candidate;               // ASCII candidate / flag
   TXTS                temp;
   LONG                baseOffset;              // byte offset for start buffer

   ENTER();

   if (dat && dat->userdata)
   {
      DFSHEXEDDATA    *bd  = dat->userdata;
      DFSTXVIEWDATA   *dt  = bd->altData;       // ASCII text-viewer data

      baseOffset = (bd->currentLsn - bd->initialLsn) * dfsGetSectorSize();

      if (readBuf)
      {
         rc = dfsReadAllBuffers( bd->currentLsn, bd->sectors, FALSE, dat);
      }
      if (dt && (rc == NO_ERROR))               // Create text-array from 3 buffers
      {                                         // starting from CURRENT
         for (lineNr = 0; lineNr <= dt->textSize; lineNr++)
         {
            TxFreeMem( dt->text[ lineNr]);            // free and set to NULL
         }
         bufSize = bd->sectors * dfsGetSectorSize();
         if (dat->curr->data)
         {
            if ((tripleBuf = TxAlloc( 3, bufSize)) != NULL)
            {
               //- copy data into buf, then create dynamic texts, filling dt->text
               memcpy( tripleBuf, dat->curr->data, bufSize);
               if (dat->next->data)
               {
                  memcpy( tripleBuf + bufSize, dat->next->data, bufSize);
                  if (dat->nex2->data)
                  {
                     memcpy( tripleBuf + bufSize + bufSize, dat->nex2->data, bufSize);
                  }
               }
               dfsTxViewNewLine( line, dt->offsets, dat->decimal, baseOffset);

               strcpy( candidate, "");          // no string yet
               for ( i = 0,                 lineNr = 0;
                    (i < (3 * bufSize)) && (lineNr < dt->textSize); i++)
               {
                  c = tripleBuf[ i];            // next char to handle

                  if ((dt->style & DFSAF_PURE) && !TxIsPureAscii(c))
                  {
                     c = 0;                     // make it an unprintable one
                  }
                  if ((c == 0x0a) || (strlen( line) >= with))
                  {
                     if (asciiCount)            // LF not FIRST after unprintable
                     {
                        dt->text[ lineNr++] = dfsTxViewAddLine( line);
                        dfsTxViewNewLine( line, dt->offsets, dat->decimal,
                                          baseOffset + ((c == 0x0a) ? i + 1 : i));
                        asciiCount = 1;         // stay in ASCII mode! (multiple LF, enpty lines)
                     }
                  }

                  strcpy( add, "");             // default nothing to add
                  switch (c)
                  {
                     case 0x00: case 0x01: case 0x02: case 0x03: // always filter 'dangerous' ones out
                     case 0x04: case 0x05: case 0x06: case 0x07:
                     case 0x08: case 0x0b: case 0x0c:
                     case 0x1a: case 0x1b:
                        if (strlen( candidate)) // pending candidate, discard as unprintable
                        {
                           splitCount += strlen( candidate);
                           strcpy( candidate, "");
                        }
                        if (asciiCount)
                        {
                           TRACES(("Add on asciiCount:%u  '%s'", asciiCount, line));
                           dt->text[ lineNr++] = dfsTxViewAddLine( line);
                           dfsTxViewNewLine( line, dt->offsets, dat->decimal, baseOffset + i);
                           asciiCount = 0;
                        }
                        splitCount++;
                        break;

                     default:
                        if (splitCount)
                        {
                           if (strlen( candidate) < dt->strings) // not long enough yet
                           {
                              sprintf( temp, "%c", c);
                              strcat( candidate, temp); // add to candidate string
                              c = 0;
                              TRACES(("candidate now: '%s'\n", candidate));
                           }
                           else                 // long enough, split line, switch to ASCII
                           {
                              if (dt->style & DFSAF_SPLIT)
                              {
                                 sprintf( split, "------------8<---------- %5u", splitCount);
                                 strcat(  line, split);
                                 TRACES(("Add on splitCount:%u  '%s'", splitCount, line));
                                 dt->text[ lineNr++] = dfsTxViewAddLine( line);
                              }
                              splitCount = 0;
                              dfsTxViewNewLine( line, dt->offsets, dat->decimal,
                                                baseOffset + i - strlen( candidate));
                           }
                        }
                        if (c != 0)
                        {
                           if (strlen( candidate) >= dt->strings) // pending candidate, comes first
                           {
                              char     *s;

                              for (s = candidate; *s; s++)
                              {
                                 if ((*s != 0x0a) && (*s != 0x0d)) // don't copy CR/LF to text
                                 {
                                    if (*s == 0x09) // expand TAB to 4 spaces
                                    {
                                       strcpy(  temp, "    ");
                                    }
                                    else
                                    {
                                       sprintf( temp, "%c", *s);
                                    }
                                    strcat( add, temp);
                                    asciiCount++;
                                 }
                              }
                              strcat( line, add);
                              strcpy( add,       "");
                              strcpy( candidate, "");
                           }
                           if ((c != 0x0a) && (c != 0x0d)) // don't copy CR/LF to text
                           {
                              if (c == 0x09)   // expand TAB to 4 spaces
                              {
                                 strcpy(  add, "    ");
                              }
                              else
                              {
                                 sprintf( add, "%c", c);
                              }
                              asciiCount++;
                           }
                           strcat( line, add);
                        }
                        break;
                  }
               }
               if      (asciiCount)             // pending ASCII text
               {
                  TRACES(("Add pending asciiCount:%u  '%s'", asciiCount, line));
                  dt->text[ lineNr++] = dfsTxViewAddLine( line);
               }
               else if (splitCount)             // pending unprintables
               {
                  if (dt->style & DFSAF_SPLIT)
                  {
                     sprintf( split, "------------8<---------- %5u", splitCount);
                  }
                  else if (splitCount >= (3 * bufSize))
                  {
                     sprintf( split, "------< Sectors have unprintable characters only >------");
                  }
                  else
                  {
                     splitCount = 0;            // ignore other cases
                  }
                  if (splitCount)
                  {
                     strcat(  line, split);
                     TRACES(("Add pending splitCount:%u  '%s'", splitCount, line));
                     dt->text[ lineNr++] = dfsTxViewAddLine( line);
                  }
               }
               TRACES(("#Lines now: %u\n", lineNr));
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
      }
   }
   RETURN( rc);
}                                               // end 'dfsReadTxView'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Prepare line buffer for a new line, adding optional byte offset number
/*****************************************************************************/
static char *dfsTxViewNewLine                   // RET   prepared line
(
   char               *lineBuf,                 // INOUT linebuffer to prepare
   BOOL                offsets,                 // IN    add offset number
   BOOL                decimal,                 // IN    number in decimal
   LONG                byteOffset               // IN    current byte offset
)
{
   if (offsets)
   {
      if (decimal)
      {
         sprintf( lineBuf, "%6.6d | ", byteOffset);
      }
      else                                      // HEX, allowing for negatives
      {
         sprintf( lineBuf, "%6.6x | ", byteOffset & 0xFFFFFF);
      }
   }
   else
   {
      strcpy( lineBuf, "");
   }
   return lineBuf;
}                                               // end 'dfsTxViewNewLine'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate memory for a single line to be added to the text-viewer text-array
/*****************************************************************************/
static char *dfsTxViewAddLine
(
   char               *newLine                  // IN    line to be added
)
{
   char               *aLine = NULL;            // allocated line buffer

   if ((aLine = TxAlloc( 1, strlen( newLine) + 1)) != NULL)
   {
      strcpy( aLine, newLine);
   }
   TRACES(("AddLine ptr: %p\n", aLine));
   return aLine;
}                                               // end 'dfsTxViewAddLine'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update text-viewer window title/footer to reflect current position & filter
/*****************************************************************************/
static void dfsTxViewTitle
(
   DFSHEXEDDATA       *bd,                      // IN    DFS Binary-Editor data
   TXTM                wt,                      // OUT   Window title
   TXTM                wf                       // OUT   Window footer
)
{
   ENTER();

   if (bd)
   {
      if (wt)
      {
         if (bd->info)                          // additional info there
         {
            sprintf( wt, " ASCII, Sect:%3lld, ", bd->currentLsn - bd->initialLsn);
            dfstr64XiB( wt, "Size: ", bd->info->al.byteSize, " '");
            if (strlen( bd->info->name))
            {
               strcat(  wt, bd->info->name);
               strcat(  wt, "' ");
            }
         }
         else
         {
            sprintf(    wt, " ASCII view, at relative sector: %2lld ", bd->currentLsn - bd->initialLsn);
         }
         TRACES(("Title : '%s'\n", wt));
      }
      if (wf && bd->altData)
      {
         DFSTXVIEWDATA *td = (DFSTXVIEWDATA *) bd->altData;
         sprintf( wf, "F1=Help F2=NextView F4=Offsets F5=Style:%s F6=Strings:%hu F12=Min Alt+F10=Maximize",
                       dfstxv_style[ td->style], td->strings);
         TRACES(("Footer: '%s'", wf));
      }
   }
   VRETURN();
}                                               // end 'dfsTxViewTitle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display text-view disassembler window in lower right of desktop
/*****************************************************************************/
static ULONG dfsDisAsmPopup
(
   TXHEXEDIT          *dat,                     // IN    binary editor data
   LONG               *ilen                     // OUT   current instr length
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (txwIsWindow( TXHWND_DESKTOP))
   {
      DFSHEXEDDATA    *bd = dat->userdata;
      TXRECT           position;                // reference size/position
      TXWHANDLE        aview;                   // view window & frame
      TXWINDOW         window;                  // view window data
      DFSDISASMDATA    da;                      // disassembler data
      TXTM             title;                   // dynamic window title
      TXTM             footer;                  // dynamic window footer
      TXLN            *textArray = NULL;
      USHORT           ww;                      // window width
      int              i;

      txwQueryWindowRect( TXHWND_DESKTOP, FALSE, &position);
      TRECTA( "pos ", (&position));

      bd->altData    = &da;                     // attach disassembler data

      if (dat->altView == DFSH_DIS_VIEW)        // start in maximized view
      {
         ww       = position.right;
         da.lines = position.bottom -2;         // nr of lines in window
      }
      else                                      // start as window in LR
      {
         ww       = min(  position.right,       DFSDIS_WWIDTH);
         da.lines = max( (position.bottom - 8), DFSDIS_WLINES);
      }

      da.position    = dat->posCursor  + dat->posCurBuf; // position in flatCode
      da.halfSize    = bd->sectors * dfsGetSectorSize(); // flat code size
      da.flatCode    = NULL;                    // linear copy of code-area
      da.maxLines    = position.bottom;         // max nr of lines maximized
      da.code        = NULL;
      da.stPtr       = 0;                       // empty stack
      da.stack       = NULL;
      da.text        = NULL;

      strcpy( title,  "");                      // filled in by Disassemble()
      strcpy( footer, "F1=Help F2=NextView F4=Save F5=Step F6=Bits F12=Minimize Alt+F10=Maximize");

      TRACES(( "halfSize:%u maxLines:%u lines:%u\n", da.halfSize, da.maxLines, da.lines));

      if (((da.flatCode = TxAlloc( da.halfSize,  2)                  ) != NULL) &&
          ((da.code     = TxAlloc( da.halfSize, sizeof(_DecodedInst))) != NULL) &&
          ((da.stack    = TxAlloc( DFSDISTACKS,      sizeof( ULN64 ))) != NULL) &&
          ((da.text     = TxAlloc( da.halfSize + 1,  sizeof( char *))) != NULL) &&
          ((textArray   = TxAlloc( da.halfSize, TXMAXLN)             ) != NULL)  )
      {
         for (i = 0; i < da.halfSize; i++)
         {                                      // couple the char** to each
            da.text[i] = textArray[i];          // line in the text array
         }
         da.text[da.halfSize] = NULL;           // and terminate the text array

         position.left   = position.right -ww; // start as smallish window
         position.right  =                 ww; // in lower right corner
         position.top    = position.bottom - da.lines -2;
         position.bottom =                   da.lines +2;

         txwSetupWindowData(
            position.top, position.left, position.bottom, position.right,
            TXWS_FRAMED       |                 // borders (scroll indicator)
            TXCS_CLOSE_BUTTON |                 // include close button [X]
            TXWS_MOVEABLE,                      // resize
            DFS_H_DISASM,
            ' ', ' ', TXWSCHEME_COLORS,
            title, footer,
            &window);
         window.dlgFocusID   = 0;               // focus to dlg window itself
         window.tv.topline   = 0;
         window.tv.leftcol   = 0;
         window.tv.maxtop    = TXW_INVALID;
         window.tv.maxcol    = TXW_INVALID;
         window.tv.markLine  = 0;
         window.tv.markLines = 1;
         window.tv.markCol   = (daBits == Decode64Bits) ? 14 : 10;
         window.tv.markSize  = 2;               // will be set at each update
         window.tv.buf       = da.text;
         aview = txwCreateWindow( TXHWND_DESKTOP, TXW_TEXTVIEW, 0, 0, &window, NULL);

         rc = txwDlgBox( TXHWND_DESKTOP, 0, dfsDisAsmDlgProc, aview, dat);
         if (rc <= TXDID_CANCEL)                // on OK or Cancel (Esc)
         {
            dat->posCursor = da.position - dat->posCurBuf;  //- new pos in hexed
            *ilen  = (LONG)  da.code->size;                 //- this instr length
            rc = NO_ERROR;
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
      TxFreeMem( da.flatCode);                  // free any allocated memory
      TxFreeMem( da.stack);
      TxFreeMem( da.code);
      TxFreeMem( da.text);
      TxFreeMem( textArray);
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN (rc);
}                                               // end 'dfsDisAsmPopup'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the Disassembler popup window dialog
/*****************************************************************************/
static ULONG dfsDisAsmDlgProc                   // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXHEXEDIT          *dat = (TXHEXEDIT *) txwQueryWindowPtr( hwnd, TXQWP_USER);

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if ((hwnd != 0) && (dat != NULL))
   {
      TXWINDOW        *win = txwWindowData( hwnd);
      DFSHEXEDDATA    *bd  = dat->userdata;
      DFSDISASMDATA   *dd  = bd->altData;       // disassembler data
      short            sy  = win->client.bottom - win->client.top;
      short            i;
      short            instrLength = 0;         // instruction text length to mark
      LONG             move;
      LLONG            abs = 0;                 // absolute address
      LLONG            rel = 0;                 // relative address
      TXTM             strvalue;
      ULONG            objectSectors;
      ULONG            lsn;

      if ((bd != NULL) && (dd != NULL))         // better safe then sorry
      {
         if (bd->info && bd->info->al.space)    // we have S_SPACE allocation info
         {
            objectSectors = dfsSspaceSectors( TRUE, bd->info->al.chunks, bd->info->al.space);
         }
         else
         {
            objectSectors = dfsGetLogicalSize();
         }
         switch (msg)
         {
            case TXWM_INITDLG:
               txwDefDlgProc( hwnd, msg, mp1, mp2); // set focus and activate
               txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) (dat->curr->data == NULL), 0);
               break;

            case DFSHEV_WM_REFRESH:
               win->tv.markSize = 0;            // regular mark invalid after moving
               rc = dfsReadDisAsm( dat, ((BOOL) mp1), &instrLength, win->title);
               dfsDisasmSetMarkRepaint( hwnd, instrLength);
               break;

            case TXWM_CHAR:
               switch ((ULONG) mp2)
               {
                  case TXk_F2:
                     txwDismissDlg( hwnd, DFS_DID_NEXT_VIEW);
                     break;

                  case TXk_F3:                  // avoid application quit :-)
                     txwDismissDlg( hwnd, TXDID_EXIT); // exit editor instead
                     break;

                  case TXc_G:                   // Jump to abs/rel location
                     sprintf( strvalue, "0x0%llx", daOrigin + dd->position +
                             (bd->currentLsn - bd->initialLsn) * dfsGetSectorSize());

                     if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                         "Specify an absolute, hexadecimal location to jump to",
                         " Jump to absolute location ", 0,
                           TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                           50, strvalue) != TXDID_CANCEL)
                     {
                        sscanf( strvalue, "%llx", &abs);
                        rel = abs - daOrigin - bd->initialLsn * dfsGetSectorSize();
                        lsn = rel / dd->halfSize;

                        if ((rel / dd->halfSize) < objectSectors)
                        {
                           bd->currentLsn = rel / dd->halfSize;
                           dd->position   = rel % dd->halfSize;
                           txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                        }
                        else
                        {
                           TxNamedMessage( TRUE, 0, " ERROR: invalid location ",
                                           "Location is BEYOND the end of the opened object\n\n"
                                           "Can not change to location 0x0%llx", abs);
                        }
                     }
                     break;

                  case TXc_O:                   // Set new origin
                     sprintf( strvalue, "0x0%llx", daOrigin);

                     if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                         "Specify an absolute, hexadecimal value to use as the new origin",
                         " Set assembler origin ", 0,
                           TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                           50, strvalue) != TXDID_CANCEL)
                     {
                        sscanf( strvalue, "%llx", &daOrigin);

                        rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                        dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     }
                     break;

                  case TXk_F4:                  // Save to file
                     strcpy( strvalue, "*.asm");
                     if (txwSaveAsFileDialog( strvalue, NULL, "disasm", DFSC_SAVE, NULL, NULL,
                         " Specify file for saving the disassembly to ", strvalue))
                     {
                        TxFnameExtension( strvalue, "asm"); // default extension
                        rc = dfsSaveAssembly( dat, strvalue);
                     }
                     break;

                  case 'p':
                  case 'P':                     // Manual Push location on stack
                     if (dd->stPtr < DFSDISTACKS)
                     {
                        dd->stack[ dd->stPtr++] = dd->code->offset;
                        dfsDisAsmTitle( bd, win->title);
                        txwInvalidateWindow(  hwnd, TRUE, TRUE);
                     }
                     else
                     {
                        TxNamedMessage( TRUE, 0, " ERROR: location stack full ",
                                        "Location stack is FULL!\n\nCan not push location "
                                            "0x0%llx for <Alt>+<P> pop/return", dd->code->offset);
                     }
                     break;

                  case TXa_P:                   // Manual Pop location from stack
                  case TXk_F5:                  // Step, 'execute' one instruction
                     if (        (         dd->code->mnemonic.p[0]  == 'J')  ||
                         !strncmp((char *) dd->code->mnemonic.p, "CALL", 4)  ||
                         !strncmp((char *) dd->code->mnemonic.p, "RET",  3)  ||
                                                                ((ULONG) mp2 == TXa_P))
                     {
                        if ((dd->code->mnemonic.p[0] == 'R') || ((ULONG) mp2 == TXa_P))
                        {
                           if (dd->stPtr > 0)   // something on the stack ?
                           {
                              abs = dd->stack[ --dd->stPtr];
                           }
                           else
                           {
                              TxNamedMessage( TRUE, 0, " ERROR: location stack empty ",
                                              "Location stack is empty!\n\n"
                                              "No <F5> on previous CALL instruction "
                                              "or a manual <Ctrl>+<P> was issued");
                              abs = -1LL;
                           }
                        }
                        else                    // Jump or Call
                        {
                           if ((strchr((char *) dd->code->operands.p, ':') == NULL ) &&
                               (strchr((char *) dd->code->operands.p, ',') == NULL ) &&
                               (strchr((char *) dd->code->operands.p, '[') == NULL ) &&
                               (sscanf((char *) dd->code->operands.p, "%llx", &abs) == 1))
                           {
                              if (dd->code->mnemonic.p[0] == 'C') // push RET location
                              {
                                 if (dd->stPtr < DFSDISTACKS)
                                 {
                                    dd->stack[ dd->stPtr++] = dd->code->offset +
                                                              dd->code->size;
                                 }
                                 else
                                 {
                                    TxNamedMessage( TRUE, 0, " ERROR: location stack full ",
                                                        "Location stack is FULL!\n\n"
                                                        "Can not push the return address "
                                                        "0x0%llx for <F5> on RET instruction",
                                                         dd->code->offset + dd->code->size);
                                 }
                              }
                           }
                           else
                           {
                              TxNamedMessage( TRUE, 0, " ERROR: invalid operand ",
                                              "Unsupported operand format:\n\n"
                                              "      '%s'", dd->code->operands.p);
                              abs = -1LL;
                           }
                        }
                        if (abs != -1LL)        // we have a new location
                        {
                           rel = abs - daOrigin - bd->initialLsn * dfsGetSectorSize();
                           bd->currentLsn = rel / dd->halfSize;
                           dd->position   = rel % dd->halfSize;

                           txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                        }
                     }
                     else                       // just jump to the NEXT instr
                     {                          // allow code walking with <F5>
                        txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_DOWN);
                     }
                     break;

                  case TXk_F6:                  // Bits
                     switch (daBits)
                     {
                        case Decode16Bits: daBits = Decode32Bits; break;
                        case Decode32Bits: daBits = Decode64Bits; break;
                        default:           daBits = Decode16Bits; break;
                     }
                     win->tv.markCol    = (daBits == Decode64Bits) ? 14 : 10;
                     rc = dfsDisAssemble(  dat, dd->maxLines, &instrLength, win->title);
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  case TXc_HOME:                // back to initial sector
                     win->tv.markSize = 0;      // regular mark invalid after moving
                     bd->currentLsn = bd->initialLsn;
                     dd->position   = 0;
                     txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                     break;

                  case TXk_HOME:                // to start of sector
                     win->tv.markSize = 0;      // regular mark invalid after moving
                     dd->position   = 0;
                     rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  case TXk_END:                 // to end of sector
                     win->tv.markSize = 0;      // regular mark invalid after moving
                     dd->position = dd->halfSize -1;
                     rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  case TXa_PGDN:
                     if (bd->currentLsn < (objectSectors - bd->sectors))
                     {
                        bd->currentLsn += bd->sectors;
                     }
                     else
                     {
                        bd->currentLsn  = objectSectors - bd->sectors;
                     }
                     txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                     break;

                  case TXa_PGUP:
                     if (bd->currentLsn > bd->sectors)
                     {
                        bd->currentLsn -= bd->sectors;
                     }
                     else
                     {
                        bd->currentLsn  = 0;
                     }
                     txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                     break;

                  case TXk_PGUP:                // scroll up until current 1st
                  case TXk_UP:                  // instr is on set target line
                     {
                        TXTM    refLine;        // reference line contents
                        LONG    refTarget = ((ULONG) mp2 == TXk_UP) ? 1 : dd->lines -1;
                        LONG    upLimit   = 15 * refTarget;
                        LONG    startPos  = dd->position;

                        TRACES(("refTarget:%d  upLimit:%d  dd-Text:%s\n", refTarget, upLimit, dd->text[0]));
                        strcpy( refLine, dd->text[0]);

                        for (move = refTarget, i = 0; i < upLimit; i++, move = 1)
                        {
                           TRACES(("move:%d  i:%hd  dd->position:%d\n", move, i, dd->position));
                           if (dd->position >= move)
                           {
                              dd->position -= move;
                              rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                           }
                           else
                           {
                              if (bd->currentLsn == 0) // already at startsector
                              {
                                 if (dd->position <= 0) // and at start offset
                                 {
                                    break;      // scrolled to the very start
                                 }              // of the object
                                 else
                                 {
                                    dd->position = 0;
                                    rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                                 }
                              }
                              else
                              {
                                 dd->position += dd->halfSize;
                                 dd->position -= move;
                                 if (bd->currentLsn >= bd->sectors)
                                 {
                                    bd->currentLsn -= bd->sectors;
                                 }
                                 else
                                 {
                                    bd->currentLsn  = 0;
                                 }
                                 rc = dfsReadDisAsm( dat, TRUE, &instrLength, win->title);
                              }
                           }
                           if (strcmp( refLine, dd->text[ refTarget]) == 0)
                           {
                              TRACES(("At target line"));
                              break;            // scrolled to the destination,
                           }                    // ref instr is at target line
                        }
                        if (startPos != dd->position) // really moved now ?
                        {
                           win->tv.markSize = 0; // regular mark invalid after moving
                        }
                     }
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  case TXk_PGDN:
                  case TXk_DOWN:
                     win->tv.markSize = 0;      // regular mark invalid after moving
                     if ((ULONG) mp2 == TXk_PGDN)
                     {
                        for (move = 0, i = 0; i < sy; i++)
                        {
                           move += dd->code[i].size;
                        }
                     }
                     else
                     {
                        move = dd->code[0].size; // just next instruction
                     }
                     if ((dd->position + move) < dd->halfSize)
                     {
                        dd->position += move;
                        rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                     }
                     else if (bd->currentLsn < (objectSectors - bd->sectors))
                     {
                        dd->position   += move ;
                        dd->position   -= dd->halfSize;
                        bd->currentLsn += bd->sectors;
                        rc = dfsReadDisAsm( dat, TRUE, &instrLength, win->title);
                     }
                     else                       // go to end of (last) sector
                     {
                        bd->currentLsn  = objectSectors - bd->sectors;
                        dd->position    = dd->halfSize -1;
                        rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                     }
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  case TXk_RIGHT:
                     win->tv.markSize = 0;      // regular mark invalid after moving
                     if (dd->position < (dd->halfSize -1))
                     {
                        dd->position++;
                        rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                     }
                     else if (bd->currentLsn < (objectSectors - bd->sectors))
                     {
                        dd->position    = 0;
                        bd->currentLsn += bd->sectors;
                        rc = dfsReadDisAsm( dat, TRUE, &instrLength, win->title);
                     }
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  case TXk_LEFT:
                     win->tv.markSize = 0;      // regular mark invalid after moving
                     if (dd->position > 0)
                     {
                        dd->position--;
                        rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                     }
                     else
                     {
                        dd->position = dd->halfSize -1;
                        if (bd->currentLsn >= bd->sectors)
                        {
                           bd->currentLsn -= bd->sectors;
                        }
                        else
                        {
                           bd->currentLsn  = 0;
                        }
                        rc = dfsReadDisAsm(  dat, TRUE, &instrLength, win->title);
                     }
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  case TXa_F5:                  // maximize/toggle
                  case TXa_F10:                 // adapt to new window
                     rc = txwDefDlgProc( hwnd, msg, mp1, mp2); // resize

                     txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                     break;

                  case TXa_U:                   // UNMARK, set FIRST-instruction mark
                     win->tv.markSize = 0;
                     rc = dfsDisAssemble( dat, dd->maxLines, &instrLength, win->title);
                     dfsDisasmSetMarkRepaint( hwnd, instrLength);
                     break;

                  default:
                     rc = txwDefDlgProc( hwnd, msg, mp1, mp2);

                     //- redraw, fixing up possible screen corruption by owner
                     txwInvalidateWindow( hwnd, TRUE, TRUE);
                     break;
               }
               break;

            case TXWM_BUTTONDBLCLK:
               //- if (txwMouseButtonDown( hwnd, msg, mp1, mp2) == TX_PENDING)
               {
                  short  row = TXMOUSEROW()  - win->client.top;   //- relative to the window!

                  TRACES(( "DBLCLK - row:%4hd  dd->position:%4x\n", row, dd->position));

                  abs = dd->code[ row].offset;  // target address directly from dissassembly
                  rel = abs - daOrigin - bd->initialLsn * dfsGetSectorSize();
                  lsn = rel / dd->halfSize;

                  if ((rel / dd->halfSize) < objectSectors)
                  {
                     bd->currentLsn = rel / dd->halfSize;
                     dd->position   = rel % dd->halfSize;
                     txwPostMsg( hwnd, DFSHEV_WM_REFRESH, (TXWMPARAM) TRUE, 0);
                  }
                  else
                  {
                     TxNamedMessage( TRUE, 0, " ERROR: invalid location ",
                                     "Location is BEYOND the end of the opened object\n\n"
                                     "Can not change to location 0x0%llx", abs);
                  }
               }
               break;

            default:
               rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
               break;
         }
      }
      else
      {
         TRACES(("Invalid pointers, bd:%p  dd:%p\n", bd, dd));
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsDisAsmDlgProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Fill all buffers and dis-assemble flat input buffer into ASCII array/title
/*****************************************************************************/
static ULONG dfsReadDisAsm                      // RET   result
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + disasm data
   BOOL                readBuf,                 // IN    read buffers too
   short              *markLen,                 // OUT   length first instruction
   TXTM                title                    // OUT   Window title
)
{
   ULONG               rc  = NO_ERROR;

   ENTER();

   if (dat != NULL)
   {
      DFSHEXEDDATA    *bd  = dat->userdata;
      DFSDISASMDATA   *dd  = bd->altData;       // disassembler data

      if (readBuf)
      {
         rc = dfsReadAllBuffers( bd->currentLsn, bd->sectors, FALSE, dat);
      }
      if (rc == NO_ERROR)                       // disassembly
      {
         if (dat->curr->data)
         {
            memcpy( dd->flatCode,                dat->curr->data, dd->halfSize);
         }
         if (dat->next->data)
         {
            memcpy( dd->flatCode + dd->halfSize, dat->next->data, dd->halfSize);
         }
         rc = dfsDisAssemble( dat, dd->maxLines, markLen, title);
      }
   }
   RETURN( rc);
}                                               // end 'dfsReadDisAsm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dis-assemble flat input buffer into ASCII array of lines, update title
/*****************************************************************************/
static ULONG dfsDisAssemble                     // RET   result
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + disasm data
   ULONG               lines,                   // IN    number of lines to do
   short              *markLen,                 // OUT   length first instruction
   TXTM                title                    // OUT   Window title
)
{
   ULONG               rc  = DFS_BAD_STRUCTURE;

   ENTER();

   if (dat != NULL)
   {
      DFSHEXEDDATA    *bd = dat->userdata;
      DFSDISASMDATA   *da = bd->altData;        // disassembler data
      _DecodeResult    res;
      unsigned int     instructions;
      unsigned int     i;
      _OffsetType      offsetCodeStart = (bd->currentLsn  - bd->initialLsn) * dfsGetSectorSize() +
                                          daOrigin        + da->position;

      res = distorm_decode( offsetCodeStart, da->flatCode + da->position, da->halfSize * 2,
                            daBits,          da->code,      lines,       &instructions);
      #if defined (DUMP)
      if (TxTrLevel >= 700)                      // detailed only
      {
         for (i = 0; i < instructions; i++)
         {
            _DecodedInst *di = &(da->code[i]);

            TRACES(("i:%3u offset:%16.16llx  size:%3u   mnem: '%s'\n", i, di->offset, di->size, di->mnemonic.p));
         }
      }
      #endif

      if ((res == DECRES_SUCCESS) ||            // end of flatCode bytes reached
          (res == DECRES_MEMORYERR))            // end of code array reached
      {
         LONG         codeIndex;                // BYTE index in flatCode for instruction

         for (i = 0; i < lines; i++)
         {
            if (i < instructions)               // decoded data available
            {
               _DecodedInst *di = &(da->code[i]);
               int           b;                 // byte index in instruction
               TXTT          ascii;

               codeIndex = di->offset - offsetCodeStart + da->position;

               //- avoid access outside of our buffer, in case of DISTORM errors
               if ((((LONG) di->size)        < TXMAXTT) &&
                   (((LONG) di->size + codeIndex) >= 0) &&
                   (((LONG) di->size + codeIndex) < (da->halfSize * 2)))
               {
                  for (b = 0; b < di->size; b++) // create printable ASCII string for instr
                  {
                     char    c = (char) da->flatCode[ codeIndex + b];

                     ascii[ b] = TxPrintSafe( c);
                  }
                  ascii[ di->size] = 0;

                  TRACES(("i:%5u length:%2u  instr: %-15s  ascii:'%s'\n",
                           i, di->instructionHex.length, di->instructionHex.p, ascii));

                  sprintf( da->text[i], "%0*llx  %-24s %-10s %-35s ;%s",
                          (daBits == Decode64Bits) ? 12 : 8, di->offset,
                           di->instructionHex.p, di->mnemonic.p, di->operands.p, ascii);
                  if ((i == 0) && markLen)      // mark length for topline, instruction mark,
                  {
                     *markLen = (short) di->instructionHex.length;
                  }
               }
               else
               {
                  strcpy( da->text[i], "-error!-"); // invalid output from decoder
               }
            }
            else
            {
               strcpy( da->text[i], "");        // clear rest of window
            }                                   // if not enough data
            rc = NO_ERROR;
         }
      }
      dfsDisAsmTitle( bd, title);
   }
   RETURN( rc);
}                                               // end 'dfsDisAssemble'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update the disassembly window marked-area when needed, and repaint window
// When current mark is NOT the FIRST-instruction mark, leave it alone
/*****************************************************************************/
static void dfsDisasmSetMarkRepaint
(
   TXWHANDLE           hwnd,                    // IN    current window
   short               iMarkSize                // IN    instr mark size or 0
)
{
   ENTER();

   if ((hwnd != 0) && (iMarkSize != 0))
   {
      TXWINDOW        *win = txwWindowData( hwnd);

      if ((win->tv.markSize  == 0) ||           // no mark present, or
         ((win->tv.markLine  == 0) &&           // a 'FIRST instruction' one
          (win->tv.markLines == 1) &&
          (win->tv.markSize  < 25) &&
          (win->tv.markCol   == ((daBits == Decode64Bits) ? 14 : 10))))
      {
         win->tv.markSize   = iMarkSize;

         win->tv.markLine   = 0;                // make it 'FIRST instruction'
         win->tv.markLines  = 1;
         win->tv.markCol    = (daBits == Decode64Bits) ? 14 : 10;
      }
   }
   txwInvalidateWindow( hwnd, TRUE, TRUE);
   VRETURN ();
}                                               // end 'dfsDisasmSetMarkRepaint'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update disasembly window title to reflect current position
/*****************************************************************************/
static void dfsDisAsmTitle
(
   DFSHEXEDDATA       *bd,                      // IN    DFS Binary-Editor data
   TXTM                wt                       // OUT   Window title
)
{
   ENTER();

   if ((bd != NULL) && (wt != NULL))
   {
      DFSDISASMDATA   *da = bd->altData;       // disassembler data

      sprintf( wt, " %u bits disassembly; LSN 0x%10.10llx offset 0x%4.4x Stack:%2u ",
              (daBits == Decode16Bits) ? 16 : (daBits == Decode32Bits) ? 32 : 64,
               bd->currentLsn, da->position, da->stPtr);
   }
   VRETURN();
}                                               // end 'dfsDisAsmTitle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Save the current disassembly buffer (1..n lines) as a plain ASCII .asm file
/*****************************************************************************/
static ULONG dfsSaveAssembly
(
   TXHEXEDIT          *dat,                     // INOUT TX hexed + disasm data
   char               *fname                    // IN    name of output file
)
{
   ULONG               rc = NO_ERROR;
   FILE               *df;                      // destination file ptr

   ENTER();

   if ((df = fopen( fname, "w" TXFMODE)) != NULL) // create new text file
   {
      DFSHEXEDDATA    *bd = dat->userdata;
      DFSDISASMDATA   *da = bd->altData;        // disassembler data
      ULONG            lines = da->maxLines;
      TXTM             strvalue;
      TXLN             prompt;
      int              i;

      sprintf( strvalue, "%u", lines);
      sprintf( prompt, "Specify number of lines to be disassembled into file '%s',\n"
                       "range 1 .. %u", fname, da->halfSize);
      if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL, prompt,
          " Lines to disassmble ", 0, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
            50, strvalue) != TXDID_CANCEL)
      {
         sscanf( strvalue, "%u", &lines);
         if (lines > da->halfSize)
         {
            lines = da->halfSize;
         }
      }
      fprintf( df, ";Disassembly : %u-bit x86 code at LSN 0x%10.10llx offset:0x%4.4x\n",
                   (daBits == Decode16Bits) ? 16 : (daBits == Decode32Bits) ? 32 : 64,
                    bd->currentLsn, da->position);
      switch (dfstStoreType(  DFSTORE))
      {
         case DFST_RAWIMAGE: fprintf( df, ";From object : %s\n", dfstStoreDesc2( DFSTORE) + 10); break;
         default:            fprintf( df, ";From Object : %s\n", dfstStoreDesc1( DFSTORE) + 10); break;
      }
      fprintf( df, ";Generated by: %s %s; %s\n;\n", DFS_N, DFS_V, DFS_C);

      if (lines > da->maxLines)                 // need more lines disassembled
      {
         rc = dfsDisAssemble( dat, lines, NULL, NULL);
      }
      for (i = 0; i < lines; i++)
      {
         if (strlen(da->text[i]))
         {
            if (fprintf( df, "%s\n", da->text[i]) < 0)
            {
               TxNamedMessage( !dfsa->batch, 0, " ERROR: write file failed ", "Error writing to '%s', disk might be full\n", fname);
               rc = ERROR_WRITE_FAULT;
               break;
            }
         }
      }
      fclose( df);
   }
   else
   {
      TxNamedMessage( !dfsa->batch, 0, " ERROR: file create failed ", "Assembly file '%s' cannot be created\n", fname);
      rc = TX_INVALID_FILE;
   }
   RETURN (rc);
}                                               // end 'dfsSaveAssembly'
/*---------------------------------------------------------------------------*/

