//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee major functions: searching ...
//
// Author: J. van Wijk
//
// JvW  07-01-2004   Initial version, split off from DFSUTIL.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS   navigation and defs
#include <dfsdgen.h>                            // DFS   generic dialogs
#include <dfsver.h>                             // DFS   version and naming
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsfind.h>                            // CMD   find sector data
#include <dfsutil.h>                            // DFS   utility functions
#include <dfsspace.h>                           // DFS   file-space interface
#include <dfsupart.h>                           // PART  utility functions
#include <dfsufdsk.h>                           // FDSK  utility functions
#include <dfscfdsk.h>                           // FDSK  command functions
#include <dfstable.h>                           // SLT   utility functions
#include <dfsfat.h>                             // FAT   directory definitions
#include <dfsefat.h>                            // EFAT  directory definitions
#include <dfshpfs.h>                            // HPFS  structure defs

#include <dfsaext.h>                            // EXT  superblock identify
#include <dfsahpfs.h>                           // HPFS superblock identify
#include <dfsahfs.h>                            // HFS  superblock identify
#include <dfsaapfs.h>                           // APFS superblock identify
#include <dfsajfs.h>                            // JFS  superblock identify
#include <dfsantfs.h>                           // NTFS superblock identify
#include <dfsarsr.h>                            // RSR  superblock identify
#include <dfsaxfs.h>                            // XFS  superblock identify



static  char       *cmd_find[] =
{
   "Find sectors of specified types(s), optionaly containing a string",
   "",
   "usage: find [-o:Opts] [-t:Types] [-i:n] [-a:'asc' | -h:'hex' | -u:'uni' | Str]",
   " Opts:",
   "    F      = Find individual directory-entries for FAT types 'D' or 'd'",
   "      Fd   = find directory entries for deleted files (0xe5 or '' start)",
   "      Fe   = find normal files by extension only, not part of full name",
   "      Ff   = entries for normal (not deleted) files, using partial name",
   "      FD   = entries for normal DIRECTORIES only, using partial name",
   "      Fx   = find deleted or renamed entries on EFAT, cleared 'used' bit",
   "    M      = Multiple hits within one sector will all be found/reported",
   "    Q      = Quiet search, do NOT display the found sectors (faster!)",
   "    S      = Search in restricted SPACE only (usually DIR sectors for FS)",
   "    *      = Repeat, keep on searching, building a list   (find all ...)",
   "    %      = No sector-spanning, read single sectors => fast text-search",
   "    [      = Search in filesystem freespace only     => fast undelete",
   "    ]      = Search filesystem allocated space only  => fast file-grep",
   "    !      = Less verbose, just list found ones, faster & better logging",
   "    +      = Verbose multiline output even when repeating the search (*)",
   "    -      = Backward search, from current towards lower sector numbers",
   "    ~      = Search sector 1/7/last, head 0/1 only, cylinder-boundaries",
   "    $      = Use a case-insensitive string compare for ASCII or UNICODE",
   "    /      = Assume UNICODE strings, can be combined with '$'",
   "    ^      = Inverted search, find sectors NOT containing ...",
   "    @[pos] = Compare inside the sector at position [pos] only",
   "",
   " Types     : Get actual list using '?\?\?' command; '*' is any type",
   "",
   " -a:'asc'  : ASCII only find-pattern value, string",
   " -A:'asc'    2nd 'AND' ASCII find-pattern value, string",
   " -h:'hex'  : HEX find-pattern value, sequence of hex pairs",
   " -H:'hex'  : 2nd 'AND' HEX find-pattern value, hex pairs",
   " -i:n      : Nr of sectors to step, default is 1 (search each sector)",
   " -m:'mix'  : Mixed ASCII/HEX/UNICODE find-pattern",
   " -M:'mix'  : 2nd 'AND' Mixed ASCII/HEX/UNICODE find-pattern",
   " -P        : Prompt, allowing editing of the specified value",
   "             with one of the 'ahmu' or 'AHMU' options.",
   " -u:'uni'  : UNICODE only find-pattern value, string",
   " -U:'uni'  : 2nd 'AND' UNICODE find-pattern value, string",
   "",
   " -first    : Force first sector (MBR/Boot) to be 'found' and always listed",
   "",
   " Str       : String built from a mix of ASCII, UNICODE and HEX fragments",
   "",
   NULL
};

// Next Chs/MiB LSN that has to be processed (use whichever is LOWER)
static ULONG          nextMibLsn = 0;
static ULONG          nextCylLsn = 0;

// Return next/previous sector of interest, cylinder aligned AND MiB multiple
static ULN64 dfsIterateCylAndMiB                // RET   LSN to work on next
(
   ULN64               current,                 // IN    Current LSN
   BOOL                backward,                // IN    towards LOWER next LSN
   BOOL                superblk                 // IN    Incl possible super-blocks
);

// Return next/previous sector of interest, cylinder aligned on default STORE
static ULN64 dfsIterateCylAligned               // RET   LSN to work on next
(
   ULN64               current,                 // IN    Current LSN
   BOOL                backward,                // IN    towards LOWER next LSN
   BOOL                superblk                 // IN    Incl possible super-blocks
);

// Perform additional match checks on found FAT directory-entry
static BOOL dfsFtcFilterFatFound                // RET   Entry is valid match
(
   ULONG               entrypos,                // IN    pos of DIR-entry
   USHORT              matchlen,                // IN    match string length
   char                findtype                 // IN    FAT search-type
);


/*****************************************************************************/
// High level FIND command logic, iterating for multiple search ...
/*****************************************************************************/
ULONG dfsFindSectorData
(
   char               *rawcmd,                  // IN    unparsed commandline
   char               *repeat                   // IN    default repeat 1 or *
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXTS                findtype;
   TXTM                findopts;
   TXLN                andPattern;
   TXLN                ms;                      // mixed string buffer
   USHORT              ml  = 0;                 // mixed string 1 length
   USHORT              ml2 = 0;                 // mixed string 2 length
   BYTE                st  = 0;                 // sector type, default none
   ULONG               ei  = 0;                 // entries in sectorlist
   ULN64               sn;                      // current sector number (RSN!)
   ULN64               foundsn;                 // found sector number   (LSN!)
   USHORT              index;
   ULONG               offset = 0;
   ULN64               size;
   TXTM                s0, s1;                  // temporary string space

   ENTER();
   TRACES(("rawcmd: '%s'\n", rawcmd));

   strcpy( findtype, "*");                      // default all types
   strcpy( findopts, repeat);                   // default repeat option
   TxaOptAsString( 't', TXMAXTS, findtype);
   TxaOptAsString( 'o', TXMAXTM, findopts);

   ml = dfsGetMixedString( rawcmd, "find-pattern", FALSE, FALSE, ms);
   if (TxaOption('u'))                          // unicode, transform mixed
   {                                            // string back to ASCII and
      strcat( findopts, "/");                   // add 'UNICODE' to options
      if (ml > 2)
      {
         char       *a = ms;
         char       *u = ms;

         for (index = 0; index < ml; index += 2, u += 2, a++)
         {
            *a = *u;
         }
      }
      ml /= 2;                                  // ASCII is half UNICODE size
   }
   dfsa->lastFindLen = ml;
   if (TxaOptSet('A') || TxaOptSet('H') || TxaOptSet('U') || TxaOptSet('M'))
   {
      ml2 = dfsGetMixedString( "", "AND-pattern", FALSE, TRUE, andPattern);
   }
   if ((ml != 0) || (ml2 != 0) || (findtype[0] != '*')) // mandatory
   {
      TXLN       text;
      ULN64      last     = nav.this;               // sn last displayed size
      ULN64      snlimit;
      BOOL       bootsgeo = (strchr(findopts, '~') != NULL);
      BOOL       fatentry = (strchr(findopts, 'F') != NULL);
      BOOL       s_spaced = (strchr(findopts, 'S') != NULL);
      BOOL       superblk = (strchr(findtype, 's') != NULL);
      ULONG      info;                              // additional info with sn
      TXTIMER    stime = TxTmrGetNanoSecFromStart();

      if (strchr( findopts, '!') != NULL)       // fast search, no full progress
      {
         TxPrint( "Fast search-mode! : Less verbose output. Do NOT press <Esc>, except to abort ...\n");
      }

      if ((s_spaced) && (dfsa->findSpace.space != NULL)) // findSpace is available
      {
         sn       = 0;                          // RSN for nav.this ???
         nav.this = dfsSspaceRsn2Lsn( dfsa->findSpace.chunks, dfsa->findSpace.space, sn);
         snlimit  = dfsSspaceSectors( TRUE, dfsa->findSpace.chunks, dfsa->findSpace.space);

         dfsProgressInit( 0, snlimit, 0, "SPACE:", "searched", DFSP_STAT, 0);
      }
      else
      {
         if (s_spaced)
         {
            TxPrint( "Space restriction : No SPACE info available for current filesystem/mode!\n");
            s_spaced = FALSE;
         }
         //- set start position to search FROM (default is THIS)
         dfsOptionSizeSectors( 'f', DFSTORE, 's', nav.this, &sn);
         nav.this = sn;                         // set THIS to start of search
         snlimit  = dfsGetLogicalSize();

         dfsProgressInit( 0, snlimit, 0, "Sector:", "searched", DFSP_STAT, 0);
      }
      strcpy( s0, "Found next at LSN : ");

      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery

      //- force MBR in result-list, even when not 'found' (no signature etc)
      if ((TxaOption( TXA_O_FIRST)) && (sn == 0) && ((strchr(findtype, 'b')) || (strchr(findtype, 'r'))))
      {
         ei++;                                  // count the search hit (dummy here)
         TxPrint( ".00000            : 0x00000000 = First sector, MBR or Bootsector (autofind -first)\n");
         dfsAddSI2List( sn, 0);                 // add to sector list
         sn++;                                  // start search at sector AFTER first
      }

      if (TxaOptSet('n'))                       // offset or next/prev sector
      {
         offset = TxaOptNum('n', NULL, dfsGetSectorSize());
      }
      while ((dfsFindTypeContaining( findopts, sn, TxaOptNum( 'i', NULL, 1),
              findtype, ml, ms, ml2, andPattern, snlimit, &foundsn, &st, &offset)) &&
             !TxAbort() && DFSNL_ROOM)
      {
         TRACES(("foundsn: 0x%llx\n", foundsn));
         ei++;                                  // count the search hit (real one here)
         if (fatentry)
         {
            info = ((offset >> 5)   | DFSSNINFO);   //- add coded FAT DIR entry
         }
         else if (bootsgeo && superblk && (dfsa->FsModeId == DFS_FS_FDISK))
         {
            info = (st | DFSSNINFO);                //- add st == FSMODE for FS
         }
         else
         {
            info = 0;
         }
         dfsAddSI2List( foundsn, info);           // add to sector list
         if (strchr( findopts, '*') != NULL)    // repeating find
         {
            if      (strchr( findopts, 'Q') != NULL) // reporting disabled
            {
            }
            else if (strchr( findopts, '+') != NULL) // verbose reporting
            {
               DFSNL_ACTIVE(FALSE);             // don't allow snlist update
               dfsX10( s0, foundsn, CBG, "\n");
               st = 0;                          // use autodetect
               rc = dfsReadAnDisplay(foundsn, 0, &st);
               DFSNL_ACTIVE(TRUE);
            }
            else                                // single line of output
            {
               if (strchr( findopts, '!') == NULL) // make one-line
               {
                  strcpy( text, CU1);           // overwrite type-id line(s)
                  if (ml)
                  {
                     strcat( text, CU1);        // overwrite result message
                     strcat( text, CU1);        // overwrite hex-dump intro
                     for (index = 0; index < ml; index += 16)
                     {
                        strcat( text, CU1);     // overwrite hex-dump line(s)
                     }
                  }
                  if (ml2)
                  {
                     strcat( text, CU1);        // overwrite hex-dump intro
                     for (index = 0; index < ml2; index += 16)
                     {
                        strcat( text, CU1);     // overwrite hex-dump line(s)
                     }
                  }
               }
               else
               {
                  strcpy( text, "");
               }
               if (bootsgeo)                    // boot sectors
               {
                  size = (foundsn > last) ? (foundsn - last) : (last -foundsn);
                  strcpy( s1, " ");
                  if (size > 205)               // more than 0.1 MiB
                  {
                     dfstrSz64XiB( s1, "", size, dfsGetSectorSize(), "");
                     last = foundsn;            // set last value to 'this'
                  }
                  sprintf( s0, ".%5.5u%12.12s: ",     ei-1, s1);
               }
               else
               {
                  sprintf( s0, ".%5.5u = %12.12llX : ",  ei-1, foundsn);
               }
               strcat( text, s0);

               if ((!bootsgeo) && (dfsa->FsFileInformation) &&    //- File info on
                   ((fatentry) || (strchr("fyzD", st) != NULL)))  //- FAT or I/Fnode
               {
                  rc = DFSFNCALL( dfsa->FsFileInformation, foundsn, info, "", text);
               }
               else                             // display using SN and type only
               {
                  dfsDisplaySnType( text, foundsn, 0, st);
               }
            }
         }
         else
         {
            nav.this   = foundsn;               // LSN    of found result
            nav.offset = offset;                // offset of found result
            if (strchr( findopts, 'Q') == NULL) // reporting not disabled
            {
               dfsX10( s0, foundsn, CBG, "\n");
               st = 0;                          // use autodetect
               rc = dfsReadAnDisplay( foundsn, 0, &st);
            }
            break;                              // end the find-loop
         }
         if (strchr(findopts, 'M'))             // when multiple hits/sector
         {
            offset += (ml) ? ml : sizeof(S_FATDIR); // next possible hit
         }
         else
         {
            offset = dfsGetSectorSize();        // force iterate sector
         }

         if (s_spaced)                          // determine next starting sector (RSN!)
         {
            sn = dfsSspaceLsn2Rsn( dfsa->findSpace.chunks, dfsa->findSpace.space, foundsn);
         }
         else
         {
            sn = foundsn;
         }
         TRACES(("Next (r)sn to search: 0x%llX\n", sn));
      }
      if (bootsgeo)                             // boot sectors
      {
         if (foundsn >= dfsGetLogicalSize())
         {
            foundsn = dfsGetLogicalSize() -1;
         }
         size = (foundsn > last) ? (foundsn - last) : (last -foundsn);
         strcpy( s1, "");
         if (size > 205)                        // more than 0.1 MiB
         {
            dfstrSz64XiB( s1, "", size, dfsGetSectorSize(), "");
         }
         sprintf( s0, " %s%12.12s: ", (foundsn > last) ? "ToEnd" : "Begin", s1);
         if (size != 0)
         {
            dfsDisplaySnType( s0, foundsn, 0, ST_FINAL);
         }
      }
      if (foundsn == L64_NULL)
      {
         TxPrint("\n%u search-hits within restricted SPACE (DIR/MFT/INODE/FNODE areas)\n", ei);
      }
      else
      {
         TxPrint("\n%u search-hits upto SN 0x%llx, search %s.\n",
                    ei, foundsn, TxAbort() ? "aborted" : "finished");
      }

      dfsProgressTerm();

      if (s_spaced)                             // calculate sectors searched ...
      {
         //- to be refined, s_spaced now always starts at RSN 0
         size = dfsSspaceLsn2Rsn( dfsa->findSpace.chunks, dfsa->findSpace.space, foundsn);
      }
      else
      {
         size = foundsn - nav.this +1;
      }
      dfsDisplayThroughput( stime, size, dfsGetSectorSize());

      if (ei == 0)                              // nothing found at all
      {
         rc = DFS_NOT_FOUND;
      }
      else
      {
         rc = NO_ERROR;
      }
   }
   else                                         // give some usage and help
   {
      #if defined (USEWINDOWING)
         if (TxaOption('?'))                    // explicit help request
         {
            TxShowTxt( cmd_find);
         }
         else                                   // present interactive dialog
         {
            dfsFindDialog( FALSE, FALSE, FALSE, L64_NULL, 0);
         }
      #else
         TxShowTxt( cmd_find);
      #endif
   }
   RETURN (rc);
}                                               // end 'dfsFindSectorData'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find next sector of specified type with optional specified contents
// Caller must use ProgressInit to setup proper progress reporting!
/*****************************************************************************/
BOOL dfsFindTypeContaining                      // OUT   type/contents found
(
   char               *options,                 // IN    search options
   ULN64               start,                   // IN    starting sector     (RSN!)
   ULONG               step,                    // IN    nr of sectors to step
   char               *types,                   // IN    sector types wanted
                                                // OUT   sector type found [0]
   USHORT              mlen,                    // IN    match string length
   char               *mstr,                    // IN    match string
   USHORT              mlen2,                   // IN    match string-2 length
   char               *mstr2,                   // IN    match string-2 (AND)
   ULN64               limit,                   // IN    limit sector nr     (RSN!)
   ULN64              *lsn,                     // OUT   found sector number (LSN!)
   BYTE               *fst,                     // OUT   found sector type
   ULONG              *pos                      // INOUT found position in sect
)                                               //   sector-size starts at next
{
   BOOL                found = FALSE;
   ULONG               rc    = NO_ERROR;
   BYTE                stype = types[0];        // primary sector type wanted
   BYTE                id    = ST_UDATA;        // type of sector
   ULONG               i;
   ULONG               matchpos = 0;
   ULONG               foundpos = 0;
   char               *sp       =   strchr(options, '@');
   char               *s;
   BOOL                inContext;
   BOOL                position =  (sp                   != NULL);
   BOOL                backward =  (strchr(options, '-') != NULL);
   BOOL                asstring =  (strchr(options, '$') != NULL);
   BOOL                unicoded =  (strchr(options, '/') != NULL);
   BOOL                insector =  (strchr(options, '%') != NULL);
   BOOL                in_frees =  (strchr(options, '[') != NULL);
   BOOL                in_alloc =  (strchr(options, ']') != NULL);
   BOOL                inverted =  (strchr(options, '^') != NULL);
   BOOL                pdisplay =  (strchr(options, '!') == NULL);
   BOOL                mdisplay =  (strchr(options, 'M') != NULL);
   BOOL                s_spaced = ((strchr(options, 'S') != NULL) && (dfsa->findSpace.space != NULL));
   BOOL                bootsgeo =  (strchr(options, '~') != NULL);
   BOOL                superblk =  (strchr(types,   's') != NULL);
   char               *fatentry =   strchr(options, 'F');
   ULONG               bps      = dfsGetSectorSize();  //- bytes per sector
   ULONG               sectors  = (mlen && !position && !insector) ? 2 :1;
   ULN64               frsn     = start;        // Current sector (RSN!)
   ULN64               frm      = start;        // Current sector (LSN!)
   ULONG               startpos = 0;
   size_t              splen;                   // remaining length in buffer

   ENTER();

   TRARGS(("opt:'%s' frm:%llX types:'%s' mlen:%2.2hu lsn:%llX pos:%3.3x lim:%llx SPACE:%s\n",
            options, frm,     types,     mlen, *lsn, *pos, limit, (s_spaced) ? "YES" : "NO"));

   frm = (s_spaced) ? dfsSspaceRsn2Lsn( dfsa->findSpace.chunks, dfsa->findSpace.space, frsn) : frsn;

   nextCylLsn = start;                          // starting point for each of the two
   nextMibLsn = start;                          // alternatives for MiB/Cyl boundaries

   if (position && isdigit(sp[1]))              // decimal only!!
   {
      sscanf( sp+1,"%u", &matchpos);
   }
   else                                         // default value
   {
      switch (stype)                            // default based on type
      {
         case 'f':                              // ST_FNODE:
         case 'y':                              // ST_FNDDI:
         case 'z':                              // ST_FNDEL: and ST_FNDIR
         case 'D': matchpos = 0xd;    break;    // offset name-field
         default:  matchpos = 0;      break;    // start of sector
      }
   }
   TRACES(("backward: %s asstring: %s insector: %s starting at: %u %s\n",
            (backward) ? "Y" : "N",
            (asstring) ? "Y" : "N",
            (insector) ? "Y" : "N",
             matchpos, (sp) ? "fixed-field" : "whole sector"));

   *fst = stype;                                // most likely found type
   nav.xtra = start;                            // remember start position

   if (pdisplay)                                // Verbose display progress
   {
      TXLN             text;
      TXTM             tbuf;

      if (mlen)
      {
         if ((fatentry != NULL) && (mlen > FAT_N)) // too long for FAT DIR
         {
            //- to be refined, different for EFAT
            mlen = FAT_N;                       // adjust to maximum
         }
         sprintf( text, "Searching pattern : %s - %s",
                  (unicoded) ? "UNICODE" : "ASCII",
                  (asstring) ? "ignoring-case" : "case-sensitive");
         if (position)
         {
            sprintf( tbuf, ", at pos 0x%3.3X", matchpos);
            strcat(  text, tbuf);
         }
         else if ((*pos) < bps)                 // start inside 1st sector
         {
            sprintf( tbuf, ", from : 0x%3.3X", *pos);
            strcat(  text, tbuf);
         }
         TxPrint( "%s, Hex-ascii:%s\n%s", text, CGE, CGE); // 2* clear to EOL
         TxDisplHexDump((BYTE *) mstr, mlen);
      }
      if (mlen2)                                // 2nd AND search string
      {
         if ((fatentry != NULL) && (mlen2 > FAT_N)) // too long for FAT DIR
         {
            mlen2 = FAT_N;                      // adjust to maximum
         }
         TxPrint( "AND  2nd  pattern : AS-IS - case-sensitive, Hex-ascii:%s\n", CGE);
         TxDisplHexDump((BYTE *) mstr2, mlen2);
      }
      sprintf( text, "Search sectortype : %s%-4.4s%s = %s", CBM, types, CNN, CBM);
      dfsSectorTypeAsAscii( stype, tbuf);
      strcat(   text, tbuf);
      sprintf(  tbuf, "%s options: %s%-8.8s%s", CNN, CBC, options, CNN);
      TxPrint( "%s%s", text, tbuf);
   }
   do
   {
      if (*pos < bps)                           // start WITHIN start-sector
      {
         foundpos = startpos = *pos;
         *pos     = bps;                        // but not for next sectors ...
         TRACES(("start/continue search at sector offset: 0x%3.3x\n", startpos));
      }
      else                                      // start with next in sequence
      {
         foundpos = startpos = 0;
         if (bootsgeo)                          // use bootsector geometry
         {                                      // specific sectors only
            if (step == ((ULONG) 0x100000 / dfsGetSectorSize())) // Include MiB boundaries
            {
               frm = dfsIterateCylAndMiB(  frm, backward, superblk);
            }
            else
            {
               frm = dfsIterateCylAligned( frm, backward, superblk);
            }
            frsn = frm;                         // always same for bootsgeo search
         }
         else                                   // simple iterator, use step as set
         {                                      // The default is 1 sector
            if (backward)
            {
               frsn -= step;                    // RSN iterate
               frm  -= step;                    // default LSN iterate
            }
            else
            {
               frsn += step;
               frm  += step;
            }
            if (s_spaced)
            {
               frm = dfsSspaceRsn2Lsn( dfsa->findSpace.chunks, dfsa->findSpace.space, frsn);
            }
         }
         TRACES(("start search at BEGIN of sector LSN:%llx = RSN:%llx limit:%llx\n", frm, frsn, limit));
      }

      inContext = TRUE;                         // default search every one
      if (((in_alloc) && (!in_frees)) ||        // just in allocated sectors
          ((in_frees) && (!in_alloc)))          // or just in free ones
      {
         if (DFSFNCALL( dfsa->FsLsnAllocated, frm, 0, NULL, NULL))
         {
            inContext = in_alloc;
         }
         else
         {
            inContext = in_frees;
         }
      }
      if (inContext)                            // free/allocated or all
      {
         if ((rc = dfsRead(frm, sectors, rbuf)) == NO_ERROR)
         {
            if (stype != ST_WILDC)
            {
               id = dfsIdentifySector(frm, 0, rbuf);
               for (i = 0; !found && (i < strlen(types)); i++)
               {
                  stype = types[i];
                  if ( (stype == id) ||         // wanted specific type, or
                      ((stype == ST_KNOWN) &&   // want any known
                       (id    != ST_UDATA) ))   // and found is not unknown
                  {
                     found    = TRUE;           // type found
                     *fst     = id;             // return found sector-type
                  }
               }
               if (!found && bootsgeo && superblk && (dfsa->FsModeId == DFS_FS_FDISK))
               {
                  //- Try multiple FS specific superblock matches,
                  if      (dfsHpfsIsSuperBlock( rbuf)) {*fst = ST_SHPFS; found = TRUE;}
                  else if (dfsNtfsIsSuperBlock( rbuf)) {*fst = ST_SNTFS; found = TRUE;}
                  else if (dfsJfsIsSuperBlock(  rbuf)) {*fst = ST_SJFS;  found = TRUE;}
                  else if (dfsExtIsSuperBlock(  rbuf)) {*fst = ST_SEXT;  found = TRUE;}
                  else if (dfsRsrIsSuperBlock(  rbuf)) {*fst = ST_SRSR;  found = TRUE;}
                  else if (dfsXfsIsSuperBlock(  rbuf)) {*fst = ST_SXFS;  found = TRUE;}
                  else if (dfsHfsIsSuperBlock(  rbuf)) {*fst = ST_SHFS;  found = TRUE;}
                  else if (dfsApfsIsSuperBlock( rbuf)) {*fst = ST_SAPFS; found = TRUE;}
               }
            }
            else                                // type allways OK for wildcard
            {
               found = TRUE;
            }
            if (found && mlen)                  // match string specified
            {
               found = FALSE;                   // not realy found yet!
               if (position)                    // start-position for match
               {
                  sp = (char *) rbuf + matchpos;
                  if (asstring)                 // case-insensitive string
                  {
                     if (unicoded)              // sector is in UNICODE
                     {
                        found = (TxUnicStrnicmp( sp, mstr, mlen) == 0);
                     }
                     else                       // sector is ASCII too
                     {
                        found = (strncasecmp( sp, mstr, mlen) == 0);
                     }
                  }
                  else                          // exact match
                  {
                     if (unicoded)              // sector is in UNICODE
                     {
                        found = (TxUnicStrncmp( sp, mstr, mlen) == 0);
                     }
                     else                       // sector is ASCII too
                     {
                        found = (memcmp( sp, mstr, mlen) == 0);
                     }
                  }
                  foundpos = matchpos;
               }
               else                             // search whole buffer
               {
                  for (sp = (char *) rbuf + startpos; sp != NULL && !found;)
                  {
                     TRACES(("frm:%llX rbuf:%8.8X sp:%8.8X "
                             "*sp=%2.2x mstr:[%-*.*s]\n",
                              frm, rbuf, sp, *sp, mlen, mlen, mstr));

                     splen = (size_t) (((char *) rbuf + (bps * sectors)) - sp);
                     if (asstring)
                     {
                        s  = memchr(sp, ((*mstr) | 0x20), splen);
                        sp = memchr(sp, ((*mstr) & 0xdf), splen);
                        if (s && (!sp || (s < sp)))
                        {
                           sp = s;              // get the first or only hit
                        }
                     }
                     else
                     {
                        sp = memchr(sp, *mstr, splen);
                        #if defined (DOS32)
                           TRACES(("rbuf:%8.8x  sp:%8.8x\n", rbuf, sp))
                           if ((sp - ((char *) rbuf)) > bps * sectors)
                           {
                              TRACES(("False alarm! SKIP\n"));
                              sp = NULL;
                           }
                        #endif
                     }
                     if (sp)
                     {
                        if (asstring)           // case-insensitive string
                        {
                           if (unicoded)        // sector is in UNICODE
                           {
                              found = (TxUnicStrnicmp( sp, mstr, mlen) == 0);
                           }
                           else                 // sector is ASCII too
                           {
                              found = (strncasecmp( sp, mstr, mlen) == 0);
                           }
                        }
                        else                    // exact match
                        {
                           if (unicoded)        // sector is in UNICODE
                           {
                              found = (TxUnicStrncmp( sp, mstr, mlen) == 0);
                           }
                           else                 // sector is ASCII too
                           {
                              found = (memcmp( sp, mstr, mlen) == 0);
                           }
                        }
                        TRACES(("Verified: sp=%8.8x sp:[%-*.*s] mstr:[%-*.*s] match:%u\n",
                                 sp, mlen, mlen, sp, mlen, mlen, mstr, found));

                        if (found)              // perform additional checks
                        {
                           foundpos = (ULONG) ((BYTE *) sp - rbuf);
                           TRHEXS(70, rbuf, 1024, "FIND 2-sector buffer");

                           if (fatentry)        // special FAT DIR handling
                           {
                              found = dfsFtcFilterFatFound( foundpos, mlen, fatentry[1]);
                           }
                        }
                        if (!found)             // inc search-pointer only
                        {                       // when not found to avoid
                           sp++;                // off-by-one error ...
                        }
                     }
                  }
               }
               if (inverted)                    // search NOT containing ...
               {
                  found = !found;               // invert the search result
               }
            }
            if (found && mlen2)                 // 2nd AND string specified
            {                                   // search whole buffer, exact
               found = FALSE;                   // not realy found yet!
               for (sp = (char *) rbuf; sp != NULL && !found;)
               {
                  TRACES(("frm:%llX rbuf:%8.8X sp:%8.8X "
                          "*sp=%2.2x mstr2:[%-*.*s]\n",
                           frm, rbuf, sp, *sp, mlen2, mlen2, mstr2));

                  splen = (size_t) (((char *) rbuf + (bps * sectors)) -sp);
                  if ((sp = memchr(sp, *mstr2, splen)) != NULL)
                  {
                     found = (memcmp( sp, mstr2, mlen2) == 0);
                     TRACES(("Verified: sp=%8.8x sp:[%-*.*s] mstr2:[%-*.*s] match:%u\n",
                              sp, mlen2, mlen2, sp, mlen2, mlen2, mstr2, found));

                     if (!found)                // inc search-pointer only
                     {                          // when not found to avoid
                        sp++;                   // off-by-one error ...
                     }
                  }
               }
               if (inverted)                    // search NOT containing ...
               {
                  found = !found;               // invert the search result
               }
            }
            if (found && fatentry &&            // found a Directory
                (mlen == 0) && (mlen2 == 0))    // no search strings
            {
               found = dfsFtcFilterFatFound( foundpos, mlen, fatentry[1]);
               if (!found)
               {
                  *pos = foundpos + sizeof(S_FATDIR); //- iterate to next
               }                                      //- entry, not sector
            }
            dfsProgressShow( frsn + 1, 0, dfsa->snlist[0], "Found:");
         }
         else if (frsn < limit)                 // some error, handle/ignore
         {
            switch (dfsa->eStrategy)
            {
               case TXAE_QUIT:                  // no action, keep bad rc
                  break;

               case TXAE_CONFIRM:
                  if (dfsa->batch)              // CONFIRM not possible, but do NOT ignore errors!
                  {
                  }
                  else
                  {
                     if (TxConfirm( 5191, "Read error at sector 0x0%llx.\n"
                                          "Do you want to continue searching ? [Y/N]: ", frm))
                     {
                        if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                        {
                           dfsa->eStrategy = TXAE_IGNORE;
                        }
                        rc = NO_ERROR;
                     }
                  }
                  break;

               case TXAE_IGNORE:
               default:                         // ignore the error
                  rc = NO_ERROR;
                  break;
            }
         }
      }
      else                                      // progress update while out of context
      {
         dfsProgressShow( frsn + 1, 0, dfsa->snlist[0], "Found:");
      }

      //- Explicit check on boundary if not reading, or if RSN's are used, do always ...
      if ((frsn >= limit) || (frsn >= ((ULONG) 0 - step)))
      {
         //- to be refined, limit should work for backward search too!
         //- it is now fixed to being ZERO (testing for reaching negative)
         //- perhaps 'limit' as passed should be the lower boundary then
         rc = DFS_PSN_LIMIT;
      }
   } while (!found && !TxAbort() && (rc == NO_ERROR));
   nav.down = frm;                              // remember Abort position
   if (found)
   {
      if (pdisplay || mdisplay)                 // Verbose display progress
      {                                         // or Multiple/Match position
         if (!fatentry)
         {
            TXRNOLN();                          // EOLN when not tracing
            if ((mlen != 0) && (!inverted))
            {
               TxPrint("Matched in sector : 0x0%llX at position: 0x%3.3X\n", frm, foundpos);
            }
         }
      }
      *pos = foundpos;
   }
   else                                         // make sure next will iterate
   {
      *pos = bps;
   }
   *lsn = frm;                                  // even when not found!

   TRACES(("Types:'%s' at lsn: 0x%llX = rsn: 0x%llX  fst:0x%2.2hx, pos: %3.3x\n", types, frm, frsn, *fst, foundpos));
   BRETURN (found);
}                                               // end 'dfsFindTypeContaining'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Insert all cylinder-aligned special sectornumbers into the sectorlist
/*****************************************************************************/
ULONG dfsCylinderSectorList
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               sn;
   BOOL                superblk = TxaOption('s');
   BOOL                mib_incl = TxaOption('m');
   TXTIMER             stime = TxTmrGetNanoSecFromStart();

   ENTER();

   dfsInitList(0, "-s", "-b");

   //- set start position to dump FROM (default is THIS)
   dfsOptionSizeSectors( 'f', DFSTORE, 's', nav.this, &sn);
   nav.this = sn;                               // set THIS to start of dump

   nextCylLsn = sn;                             // starting point for each of the two
   nextMibLsn = sn;                             // alternatives for MiB/Cyl boundaries

   dfsProgressInit( sn, dfsGetLogicalSize() - sn, 0, "Sector:", "searched", DFSP_STAT, 0);

   while ((sn < dfsGetLogicalSize()) && !TxAbort() && DFSNL_ROOM)
   {
      dfsAdd2SectorList( sn);                   // add to sector list
      dfsProgressShow( sn, 0, dfsa->snlist[0], "Listed:");

      if (mib_incl)
      {
         sn = dfsIterateCylAndMiB(  sn, FALSE, superblk);
      }
      else
      {
         sn = dfsIterateCylAligned( sn, FALSE, superblk);
      }
   }

   dfsProgressTerm();
   dfsDisplayThroughput( stime, sn - nav.this +1, dfsGetSectorSize());

   RETURN (rc);
}                                               // end 'dfsCylinderSectorList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return next/previous sector of interest, cylinder aligned AND MiB multiple
/*****************************************************************************/
static ULN64 dfsIterateCylAndMiB                // RET   LSN to work on next
(
   ULN64               current,                 // IN    Current LSN
   BOOL                backward,                // IN    towards LOWER next LSN
   BOOL                superblk                 // IN    Incl possible super-blocks
)
{
   ULN64               rc;                      // function return
   ULN64               MiBsect;
   ULN64               MiBmask;

   ENTER();

   if (backward)                                // backward, ignore MiB for now
   {
      rc = dfsIterateCylAligned( current, TRUE, superblk);
   }
   else                                         // forward, MiB or Cyl-CHS based
   {
      MiBsect = 0x100000ULL / dfsGetSectorSize();    //- For 512b: 0x0000000000000800
      MiBmask = 0xffffffffffffffffULL - MiBsect +1;  //- For 512b: 0xfffffffffffff800

      //- Advance to next MiB value, when current is equal or beyond
      if (current >= nextMibLsn)
      {
         nextMibLsn = ((nextMibLsn + MiBsect) & MiBmask); // align to NEXT MiB boundary
         TRACES(("Mib incremented to:0x%llx\n", nextMibLsn));
      }

      //- Advance to next Cyl value, when current is equal or beyond
      if (current >= nextCylLsn)
      {
         nextCylLsn = dfsIterateCylAligned( nextCylLsn, FALSE, superblk);
         TRACES(("Cyl incremented to:0x%llx\n", nextCylLsn));
      }
      if (nextMibLsn < nextCylLsn)              // current not used, not reentrant!
      {                                         // uses STATIC marker-values only!
         rc = nextMibLsn;
      }
      else
      {
         rc = nextCylLsn;
      }
      TRACES(("current:0x%8.8x Mib:0x%llx Cyl:0x%llx\n", current, nextMibLsn, nextCylLsn));
   }
   RETN64 (rc);
}                                               // end 'dfsIterateCylAndMiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return next/previous sector of interest, cylinder aligned on default STORE
/*****************************************************************************/
static ULN64 dfsIterateCylAligned               // RET   LSN to work on next
(
   ULN64               current,                 // IN    Current LSN
   BOOL                backward,                // IN    towards LOWER next LSN
   BOOL                superblk                 // IN    Incl possible super-blocks
)
{
   ULN64               rc;                      // function return
   ULONG               c, h, sec;               // CHS intermediate values
   ULONG               maxH = dfstGeoHeads(   DFSTORE) -1;
   ULONG               maxS = dfstGeoSectors( DFSTORE);

   ENTER();

   dfstPsn2Chs( DFSTORE, dfstLSN2Psn( DFSTORE, current), &c, &h, &sec);

   /*- CHS table, visit MBR/EBR, PBR, LVM-info and LVM-signature                                 */
   /*- JvW 20120727: updated to find JFS superblocks on 127/255 sectors per track                */
   /*- JvW 20150123: updated to find JFS superblocks on  64/32  sectors per track (1 MiB)        */
   /*- Cyl    Head    Sect  possible content                                                     */
   /*- prev   max     max   LVM signature   sector                                               */
   /*- -----+-------+-----+----------------------------------------                              */
   /*- this   0       1     MBR or EBR sector, or a boot primary                                 */
   /*- this   0       2     NTLDR 1st sector on non 1st primary   S                              */
   /*- this   0       3     EXT2  superblock       on a primary   S                              */
   /*- this   0       7     FAT32 spare bootsector on a primary                                  */
   /*- this   0       17    HPFS superblock on a primary          S                              */
   /*- this   0       65    JFS super on primary, if > 63 S/track S                              */
   /*- this   0       max   LVM information sector                                               */
   /*- this   1       1     Bootsector for filesystem on logical                                 */
   /*- this   1       2     JFS  super on primary, NTLDR logical  S                              */
   /*- this   1       3     EXT2  superblock       on a logical   S                              */
   /*- this   1       7     FAT32 spare bootsector on a logical                                  */
   /*- this   1       17    HPFS superblock on a logical/1st prim S                              */
   /*- this   1       65    JFS super on primary, if > 63 S/track S                              */
   /*- this   2       2     JFS  superblock on a logical/1st prim S                              */
   /*- this   2       3     REISER superblock on a primary        S                              */
   /*- this   3       1     JFS  super on log/1st prim on 64/32   S                              */
   /*- this   3       3     REISER superblock on logical/1st prim S                              */
   /*- this   max     max   LVM BBR sector or NTFS spare                                         */
   /*- -----+-------+-----+----------------------------------------                              */
   /*- next   0       1     MBR of EBR sector with part table                                    */

   if (backward)                                // backward through table:
   {
      if      ((h == 0) && (sec == 1))          { h = maxH; sec = maxS; c--; }
      else if ((h == 0) && (sec == 2))          {           sec = 1;         }
      else if ((h == 0) && (sec == 3))          {           sec = 2;         }
      else if ((h == 0) && (sec == 7))          {           sec = 3;         }
      else if ((h == 0) && (sec ==17))          {           sec = 7;         }
      else if ( h == 0               )          {           sec =17;         }
      else if ((h == 1) && (sec == 1))          { h = 0;    sec = maxS;      }
      else if ((h == 1) && (sec == 2))          {           sec = 1;         }
      else if ((h == 1) && (sec == 3))          {           sec = 2;         }
      else if ((h == 1) && (sec == 7))          {           sec = 3;         }
      else if ((h == 1) && (sec ==17))          {           sec = 7;         }
      else if ((h == 2) && (sec == 2))          { h = 1;    sec =17;         }
      else if ((h == 2) && (sec == 3))          {           sec = 2;         }
      else if ((h == 3) && (sec == 1))          { h = 2;    sec = 3;         }
      else if ((h == 3) && (sec == 3))          {           sec = 1;         }
      else                                      { h = 3;    sec = 3;         }
   }
   else                                         // forward through table:
   {
      //- Note: when it gets more complex, it may be easier and faster to just
      //-       search the first 300 (decimal) sectors in each track to cover
      //-       all possible superblock location on primary or logical

      if (superblk)                             // full table
      {
         if      ((h == 0) && (sec == 1))       {           sec = 2;         }
         else if ((h == 0) && (sec == 2))       {           sec = 7;         }
         else if ((h == 0) && (sec == 3))       {           sec = 7;         }
         else if ((h == 0) && (sec == 7))       {           sec = 17;        }
         else if ((h == 0) && (sec ==17))
         {
            if   (maxS < 65)                    {           sec = maxS;      }
            else                                {           sec = 65;        }
         }
         else if ((h == 0) && (sec ==65))       {           sec = maxS;      }
         else if ( h == 0               )       { h = 1;    sec = 1;         }
         else if ((h == 1) && (sec == 1))       {           sec = 2;         }
         else if ((h == 1) && (sec == 2))       {           sec = 3;         }
         else if ((h == 1) && (sec == 3))       {           sec = 7;         }
         else if ((h == 1) && (sec == 7))       {           sec = 17;        }
         else if ((h == 1) && (sec ==17))
         {
            if   (maxS < 65)                    { h = 2;    sec = 2;         }
            else                                {           sec = 65;        }
         }
         else if ((h == 1) && (sec ==65))       { h = maxH; sec = maxS;      }
         else if ((h == 2) && (sec == 2))       {           sec = 3;         }
         else if ((h == 2) && (sec == 3))       { h = 3;    sec = 1;         }
         else if ((h == 3) && (sec == 1))       {           sec = 3;         }
         else if ((h == 3) && (sec == 3))       { h = maxH; sec = maxS;      }
         else                                   { h = 0;    sec = 1;    c++; }
      }
      else                                      // faster and cleaner run ...
      {
         if      ((h == 0) && (sec == 1))       {           sec = 7;         }
         else if ((h == 0) && (sec == 7))       {           sec = maxS;      }
         else if ( h == 0               )       { h = 1;    sec = 1;         }
         else if ((h == 1) && (sec == 1))       {           sec = 7;         }
         else if ((h == 1) && (sec == 7))       { h = maxH; sec = maxS;      }
         else                                   { h = 0;    sec = 1;    c++; }
      }
   }
   rc = dfstPsn2LSN( DFSTORE, dfstChs2Psn( DFSTORE, c, h, sec));

   RETN64 (rc);
}                                               // end 'dfsIterateCylAligned'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Perform additional match checks on found FAT directory-entry in rbuf
/*****************************************************************************/
static BOOL dfsFtcFilterFatFound                // RET   Entry is valid match
(
   ULONG               entrypos,                // IN    pos of DIR-entry
   USHORT              matchlen,                // IN    match string length
   char                findtype                 // IN    FAT search-type
)
{
   BOOL                rc   = TRUE;             // function return
   S_FATDIR           *dent = (S_FATDIR *) (rbuf + (entrypos & 0xffe0));

   ENTER();
   TRACES(( "entrypos: 0x%4.4x  matchlen:%hu  type:'%c'  dent: 0x%8.8x holding 0x%2.2hx\n",
             entrypos,          matchlen, findtype, dent, *dent));

   if (dent->Name[0] == FAT_DIRFREE)
   {
      rc = FALSE;
      TRACES(("Skip, free entry\n"));
   }
   else if (dfsa->FsModeId == DFS_FS_E_FAT)
   {
      S_EFATDIR       *ex = (S_EFATDIR *) dent;

      if (findtype == 'x')                      // find deleted entries only
      {
         if (ex->EntryType & EFAT_DIRBIT_IN_USE)
         {
            rc = FALSE;
            TRACES(("Skip, not a deleted-file on EFAT\n"));
         }
         else
         {
            if (matchlen == 0)                  // no filename fragment
            {
               if (((ex->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE) ||
                    (ex->fl.SecCount  < 2))     // no (valid) FILE entry
               {
                  rc = FALSE;
                  TRACES(("Skip, non FILE, EFAT, EMPTY, DEL\n"));
               }
            }
            else
            {
               if ((ex->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_ANY_FILENAME)
               {
                  rc = FALSE;
                  TRACES(("Skip, non NAME, EFAT, str, DEL\n"));
               }
            }
         }
      }
      else
      {
         if ((ex->EntryType & EFAT_DIRBIT_IN_USE) == 0)
         {
            rc = FALSE;
            TRACES(("Skip, deleted file/dir on EFAT\n"));
         }
         else
         {
            if (matchlen == 0)                  // no filename fragment
            {
               if (((ex->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE) ||
                    (ex->fl.SecCount  < 2))      // no (valid) FILE entry
               {
                  rc = FALSE;
                  TRACES(("Skip, non FILE, EFAT, EMPTY, NORMAL\n"));
               }
            }
            else
            {
               if ((ex->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_ANY_FILENAME)
               {
                  rc = FALSE;
                  TRACES(("Skip, non NAME, EFAT, str, NORMAL\n"));
               }
            }
         }
      }
   }
   else                                         // regular FAT 12/16/32
   {
      S_VFSLOT        *vf = (S_VFSLOT *)  dent;

      if ( (vf->FatAttrib == VFAT_ATTRIB) &&
           (vf->clust     == VFAT_CLUSTR) &&
           (vf->Vzero     == 0          )  )
      {
         //- to be refined, may enhance this to allow searching on long-filenames
         rc = FALSE;
         TRACES(("Skip, LFN entry\n"));
      }
      else
      {
         if (((entrypos % sizeof(S_FATDIR)) + matchlen) > FAT_N)
         {
            rc = FALSE;
            TRACES(("Skip, not-in-name field\n"));
         }
         else
         {
            TRACES(( "Search: '%c' - file '%*.*s' at entry:%u\n",
                      findtype, FAT_N, FAT_N, dent->Name, entrypos >> 5));

            if (dent->Name[0] == FAT_DIRDEL)    // first char in name
            {
               if ((findtype != 'd') ||         // want non-deleted ?
                   (dent->FatAttrib & FATTR_DIRECTORY))
               {
                  rc = FALSE;
                  TRACES(("Skip, deleted-file or DIR\n"));
               }
            }
            else                                // file is not deleted
            {
               if (findtype == 'd')             // searching for deleted ?
               {
                  rc = FALSE;
                  TRACES(("Skip, not a deleted-file\n"));
               }
               else
               {
                  if (dent->FatAttrib & FATTR_DIRECTORY) // directory ?
                  {
                     if (findtype != 'D')       // not searching for a DIR ?
                     {
                        rc = FALSE;
                        TRACES(("Skip, dont want a DIR\n"));
                     }
                  }
                  else                          // entry is a regular file
                  {
                     if      (findtype == 'e')  // searching for EXTENSION ?
                     {
                        if ((entrypos % 32) != 8)
                        {
                           rc = FALSE;
                           TRACES(("Skip, not an extension\n"));
                        }
                     }
                     else if (findtype == 'D')  // searching for DIRECTORY ?
                     {
                        rc = FALSE;
                        TRACES(("Skip, not a DIR\n"));
                     }
                  }
               }
            }
         }
      }
   }
   BRETURN (rc);
}                                               // end 'dfsFtcFilterFatFound'
/*---------------------------------------------------------------------------*/

