//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FAT formatting functions
//
// Author: J. van Wijk
//
// JvW  29-12-2005 Initial version
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfswipe.h>                            // wipe functions
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface

#include <dfsupart.h>                           // FDISK partition functions

#include <dfsafat.h>                            // FAT analysis functions
#include <dfsufat.h>                            // FAT utility functions
#include <dfsffat.h>                            // FAT formatting functions
#include <dfsbfat.h>                            // FAT bootsector functions


/*****************************************************************************/
// Format current volume/partition using specified parameters
/*****************************************************************************/
ULONG dfsFatFormat
(
   FAT_FMT_INFO       *ffi                      // IN    FAT format information
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                str;
   TXTT                tmp;
   LLONG               rawbytes;
   ULONG               rawclust;                // approximate #clusters
   ULONG               wipeSize;
   BYTE               *sector = NULL;           // sector buffer, varible size
   USHORT              fs = ffi->desiredBits;
   USHORT              cs = ffi->desiredCsize / (dfsGetSectorSize() / SECTORSIZE);
                                                // desiredCsize is in units of 512 bytes!
   ENTER();

   strcpy(    str, "\n");
   dfstrSize( str, "Format FAT object : " , ffi->sectors, " as ");
   sprintf(   tmp, "FAT%hu, ",              ffi->desiredBits);
   strcat(    str, tmp);
   dfstrSize( str, "clustersize ",          ffi->desiredCsize, "");
   TxPrint( "%s   Label:%s\n", str,         ffi->label);

   // nr of root sectors for fat12/16
   fat->Roots      = ffi->desiredRsize / (dfsGetSectorSize() / sizeof(S_FATDIR));

   //- calculate nr of clusters to determine size of FAT-areas
   rawbytes = (LLONG) (ffi->sectors - (fat->Roots) ) * dfsGetSectorSize();
   rawclust = (ULONG) (rawbytes / ((cs * dfsGetSectorSize()) + (fs /4)) +3);

   fat->Sect       = ffi->sectors;
   fat->FatBits    = fs;
   fat->ClustSize  = cs;
   fat->FatSectors = max( cs, (rawclust / ((dfsGetSectorSize() * 8) / fs)) +1);
   fat->Fat1       = ffi->reservedSect;
   fat->Fat2       = fat->Fat1 + fat->FatSectors;
   fat->Root       = fat->Fat2 + fat->FatSectors;

   dfstSetClusterSize( DFSTORE, cs);            // global cluster size ...

   if (fs == 32)
   {
      fat->ClTwo   = fat->Root;
      fat->Roots   = cs;
   }
   else
   {
      fat->ClTwo   = fat->Root + fat->Roots;
   }
   wipeSize        = (ffi->longFormat) ? fat->Sect : (fat->Root + fat->Roots);
   fat->RawClust   = (fat->Sect - fat->ClTwo) / fat->ClustSize  + 2;

   dfsFatSetFatInUse( fat->Fat1);
   fat->CacheA.First   = FAT_NO_CLUST;          // invalidate FAT caches
   fat->CacheB.First   = FAT_NO_CLUST;
   fat->CachedClNumber = 0;
   fat->ExpandSize     = 0;
   dfsa->FsExpandSize  = 0;

   if (dfsa->verbosity >= TXAO_VERBOSE)         // verbose
   {
      TxPrint( "RAW cluster count : 0x%8.8lx   Exact #:0x%8.8lx\n", rawclust, fat->RawClust);
      TxPrint( "Sectors per FAT   : 0x%8.8lx   ", fat->FatSectors);
      dfsX10(            "Fat1 at :",             fat->Fat1, "", "   ");
      dfsX10(            "Fat2 at :",             fat->Fat1, "",  "\n");
      dfsX10(  "Root directory at : ",            fat->Root, "", "   ");
      TxPrint(              "size : 0x%4.4lx\n",     fat->Roots);
      TxPrint( "Size to be zeroed : 0x%8.8lx   sectorsize: %hu\n", wipeSize, dfsGetSectorSize());
   }
   if (!TxaOption('t'))                         // unless we are just testing
   {
      SEC512 pattern;

      memset( pattern, 0, SECTORSIZE);
      if ((!ffi->longFormat) &&                 // if not a LONG format,
           !TxaOptSet('b'))                     // and not overruled
      {
         TxaOptSetItem( "-b:1");                // use 1-sector buffer
      }
      rc  = dfsWipeArea( 0, wipeSize, DFSP_BARS,
                         (ffi->longFormat) ? "LONG formatting FAT" : "QUICK formatting FAT",
                         TRUE, pattern, SECTORSIZE);
      if (rc == NO_ERROR)
      {
         if ((sector = TxAlloc(1, dfsGetSectorSize())) == NULL)
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      if (rc == NO_ERROR)
      {
         S_FAT32  *fatSector = (S_FAT32 *) sector;

         memset( sector, 0, dfsGetSectorSize());
         fatSector->MediaType = (fat->Sect > 0xfff) ? 0xf8 : 0xf0; // HD/diskette
         switch (fs)
         {
            case 12: memcpy( fatSector->Signature, sg_fat12, SG_FAT12); break;
            case 16: memcpy( fatSector->Signature, sg_fat16, SG_FAT16); break;
            default: memcpy( fatSector->Signature, sg_fat32, SG_FAT32);
               fatSector->FatValue[0] = FAT_MINEOFCL; // FAT entry ROOT dir
               break;
         }
         if ((rc = dfsWrite( fat->Fat1, 1, sector)) == NO_ERROR)
         {
            if ((rc = dfsWrite( fat->Fat2, 1, sector)) == NO_ERROR)
            {
               //- write label as only directory-entry in ROOT
               S_FATDIR   *root = (S_FATDIR *) sector;
               time_t      now  = time( NULL);
               struct tm  *fdt  = localtime( &now);

               memset( sector, 0, dfsGetSectorSize());

               root->date.d.day     = fdt->tm_mday;
               root->date.d.month   = fdt->tm_mon  + 1;
               root->date.d.year    = fdt->tm_year - 80;
               root->time.t.hours   = fdt->tm_hour;
               root->time.t.minutes = fdt->tm_min;
               root->time.t.twosecs = fdt->tm_sec  / 2;

               memcpy( root->Name, ffi->label, FAT_N);
               root->FatAttrib = FATTR_LABEL;
               if ((rc = dfsWrite( fat->Root, 1, sector)) == NO_ERROR)
               {
                  rc = dfsFatMkBootRec( ffi->os[0], FALSE, ffi->desiredRsize,
                                        ffi->serial,       ffi->label);
                  if (rc == NO_ERROR)
                  {
                     if (fs == 32)              // FAT32 2nd/3rd boot & spares
                     {
                        FAT32B2      *f32 = (FAT32B2 *) sector;

                        memset( sector, 0,  dfsGetSectorSize());
                        f32->Signature = SV_BOOTR;
                        rc = dfsWrite( 2, 1, sector); // 3rd bootsector
                        if (rc == NO_ERROR)
                        {
                           f32->Signatur1    = SV_BOOT21;
                           f32->Signatur2    = SV_BOOT22;
                           f32->FreeClusters = fat->RawClust -3; // 0,1 and root
                           f32->NextSearchCl = 2;
                           rc = dfsWrite( 1, 1, sector); // 2nd bootsector
                        }
                     }
                     if (rc == NO_ERROR)
                     {
                        if (dfsRead( LSN_BOOTR, 3, brec) == NO_ERROR)
                        {
                           if (fs == 32)        // write SPARE copies (6/7)
                           {
                              rc = dfsWrite( LSN_FAT32SPARE, 3, brec);
                           }
                           //- write extra spare copy at LAST sector like NTFS
                           //- will also invalidate any OLD LVM-sig sector
                           dfsWrite( fat->Sect -1, 1, brec);
                        }
                     }
                     if (rc == NO_ERROR)        // all well, show statistics
                     {
                        ULONG   cl    = fat->RawClust - ((fs == 32) ? 3 : 2);
                        ULONG   csize = ((ULONG) cs) * dfsGetSectorSize();
                        LLONG   bytes = ((LLONG) cl) * csize;

                        TxPrint("%15llu bytes available on disk\n\n",        bytes);
                        TxPrint("%15hu bytes in each allocation unit.\n",    csize);
                        TxPrint("%15lu available allocation units on disk\n\n", cl);
                     }
                     else                       // some error creating 2nd/3rd
                     {                          // or spare bootsector copies
                        TxPrint( "\nError writing (spare) bootsector!\n");
                     }
                  }
               }
            }
         }
         TxFreeMem(sector);
      }
      else                                      // write verify error
      {
         TxPrint( "\nVerify error (%u) : object is not writable, or the data gets corrupted!\n", rc);
      }
   }
   RETURN (rc);
}                                               // end 'dfsFatFormat'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate optimum clustersize based on Bits & other selections
/*****************************************************************************/
USHORT dfsFatFmtClusterSize
(
   FAT_FMT_INFO       *ffi                  /* IN    FAT format information*/
)
{
   USHORT              minsize;                 // minimum cluster size
   USHORT              desired = ffi->desiredCsize;

   ENTER();

   switch (ffi->desiredBits)
   {
      case 12: minsize = ffi->minimumCs_12; break;
      case 16: minsize = ffi->minimumCs_16; break;
      default: minsize = ffi->minimumCs_32; break;
   }
   if (desired == 0)
   {
      desired = (ffi->desiredBits == 32) ? ffi->defaultCsize : minsize;
   }
   else if (desired < minsize)
   {
      desired = minsize;
   }
   if      (desired <=  1) desired =  1;
   else if (desired <=  2) desired =  2;
   else if (desired <=  4) desired =  4;
   else if (desired <=  8) desired =  8;
   else if (desired <= 16) desired = 16;
   else if (desired <= 32) desired = 32;
   else                    desired = 64;

   RETURN (desired);
}                                             /* end 'dfsFatFmtClusterSize'*/
/*---------------------------------------------------------------------------*/

