//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// EXT2, EXT3 and EXT4 disk structure definitions
// Based on information from "Unix filesystems" by Steve D Pate
// Internet 'https://ext4.wiki.kernel.org/index.php'
// and a lot of reverse-engineering with DFSee
//
// Author: J. van Wijk
//
// JvW  29-03-2004   Initial version
// JvW  09-06-2016 Updated for specific EXT4 stuff and superblock size
//
#ifndef    DFSEXT_H
   #define DFSEXT_H

//- First Superblock BG 0, at sect# 2 for 512 BPS, 0 for 4096 (at offset 1024)
#define LSN_EXTSUP ((ULN64) (1024 / dfsGetSectorSize()))


#define EXT_SYS_PARENT         0x01             // Parent Inode# for sys inodes in cache
#define EXT_TXT_RESIZE     "RESIZE-system-inode-7"
#define EXT_TXT_JOURNAL    "Journal-File"


// Special inode numbers,
#define EXT_I_BAD             1                 // Invalid, bad inode
#define EXT_I_ROOT            2                 // Root directory
#define EXT_I_BOOT_LOADER     5                 // Boot loader
#define EXT_I_UNDEL_DIR       6                 // Undelete directory
#define EXT_I_RESIZE          7                 // Reserved group descriptors
#define EXT_I_JOURNAL         8                 // Journal file

#define EXT_I_FIRST          11                 // Classic 1st userfile inode

#define SV_EXTSUP        0xef53                 // USHORT signature value

#define EXT_OS_LINUX          0                 // Linux
#define EXT_OS_HURD           1                 // GNU Unix (server)
#define EXT_OS_MASIX          2                 // Masix
#define EXT_OS_FREEBSD        3                 // FreeBSD (and MAC ?)
#define EXT_OS_LITES4         4                 // Lites

#define EXT_CLASSIC           0                 // clasic EXT2
#define EXT_DYNAMIC           1                 // dynamic INODE size

#define EXT_BLOCKS_1K         0                 // Logical Block size
#define EXT_BLOCKS_2K         1
#define EXT_BLOCKS_4K         2

#define EXT_USUAL_BLOCKSPERGROUP 0x8000

#define EXT_STATE_JDATA          0x00000001     // journaled data exists
#define EXT_STATE_NEW            0x00000002     // inode is newly created


#define EXT_FS_VIRGIN            0x0000         // Never mounted, virgin
#define EXT_FS_CLEAN             0x0001         // Unmounted cleanly
#define EXT_FS_ERROR             0x0002         // Errors detected
#define EXT_FS_ORPHAN            0x0004         // Orphans being recovered
#define EXT_FS_MAXSTATE          0x0007         // maximum state value

// Superblock Compatible, ReadOnly, and Incompatible feature flag definitions
#define EXT_FCO_DIR_PREALLOC     0x00000001
#define EXT_FCO_IMAGIC_INODES    0x00000002
#define EXT_FCO_HAS_JOURNAL      0x00000004
#define EXT_FCO_EXT_ATTR         0x00000008
#define EXT_FCO_RESIZE_INODE     0x00000010
#define EXT_FCO_DIR_INDEX        0x00000020     // Hashed B-trees DIR       (EXT3/4)
#define EXT_FCO_SPARSE_SUPER2    0x00000200     // 2 BG's with backup SB      (EXT4)
//----------------------------------------- +
#define EXT_FCO_KNOWN_FLAGS      0x0000023F

#define EXT_FRO_SPARSE_SUPER     0x00000001
#define EXT_FRO_LARGE_FILE       0x00000002
#define EXT_FRO_BTREE_DIR        0x00000004
#define EXT_FRO_HUGE_FILE        0x00000008     // Fsize in blocks            (EXT4)
#define EXT_FRO_GDT_CSUM         0x00000010     // GrpDesc have checksum      (EXT4)
#define EXT_FRO_DIR_NLINK        0x00000020     // no 32000 SUBDIR limit      (EXT4)
#define EXT_FRO_EXTRA_ISIZE      0x00000040     // large Inodes exist         (EXT4)
#define EXT_FRO_BIGALLOC         0x00000200     // alloc Clusters, not blocks (EXT4)
#define EXT_FRO_METADATA_CSUM    0x00000400     // FS supports Metadata CSUM  (EXT4)
#define EXT_FRO_REPLICA          0x00000800     // FS supports Replicas       (EXT4)
#define EXT_FRO_READONLY         0x00001000     // FS must be mounted as RO   (EXT4)
#define EXT_FRO_PROJECT          0x00002000     // FS tracks project quota    (EXT4)
//----------------------------------------- +
#define EXT_FRO_KNOWN_FLAGS      0x00003E7F

#define EXT_FIN_COMPRESSION      0x00000001
#define EXT_FIN_FILETYPE         0x00000002
#define EXT_FIN_RECOVER          0x00000004     // Needs recovery
#define EXT_FIN_JOURNAL_DEV      0x00000008     // Journal device
#define EXT_FIN_META_BG          0x00000010     // Meta BlockGroups used      (EXT4)
#define EXT_FIN_EXTENTS          0x00000040     // Files may use extents      (EXT4)
#define EXT_FIN_64BIT            0x00000080     // Blocnumbers upto 2^64      (EXT4)
#define EXT_FIN_MMP              0x00000100     // Multi Mount Protect        (EXT4)
#define EXT_FIN_FLEX_BG          0x00000200     // Flexible Block Grps        (EXT4)
#define EXT_FIN_EA_INODE         0x00000400     // Inode OK for large EA      (EXT4)
#define EXT_FIN_DIRDATA          0x00001000     // Data in Dir-entry (NI)     (EXT4)
#define EXT_FIN_CSUM_SEED        0x00002000     // Meta-CS seed is in SB      (EXT4)
#define EXT_FIN_LARGEDIR         0x00004000     // Dir > 2GB or 3levels htree (EXT4)
#define EXT_FIN_INLINE_DATA      0x00008000     // Data present in Inode      (EXT4)
#define EXT_FIN_ENCRYPT          0x00010000     // Encrypted Inodes present   (EXT4)
//----------------------------------------- +
#define EXT_FIN_KNOWN_FLAGS      0x0001F7DF

#define EXT_LEN_UUID             FS_UUID_LENGTH
#define EXT_LEN_LBL              16
#define EXT_LEN_MNT              64

typedef struct s_ext_super
{
   ULONG               Inodes;                  // 000 Inodes count
   ULONG               Blocks;                  // 004 Blocks count
   ULONG               ReservedBlocks;          // 008 Reserved blocks count
   ULONG               FreeBlocks;              // 00C Free blocks count
   ULONG               FreeInodes;              // 010 Free inodes count
   ULONG               FirstDataBlock;          // 014 First Data Block
   ULONG               LogBlockSize;            // 018 Block size (2^10+LBS) bytes
   LONG                LogClustSize;            // 01C Cluster size (2^LCS) blocks
   ULONG               BlocksPerGroup;          // 020 # Blocks per group
   ULONG               ClustsPerGroup;          // 024 # Clusters per group
   ULONG               InodesPerGroup;          // 028 # Inodes per group
   ULONG               Mtime;                   // 02C Mount time
   ULONG               Wtime;                   // 030 Write time
   USHORT              Mounts;                  // 034 # Mounts since last CHK
   USHORT              MaxMounts;               // 036 Maximal mount count
   USHORT              Signature;               // 038 Signature value
   USHORT              State;                   // 03A File system state
   USHORT              Errors;                  // 03C Behaviour on errors
   USHORT              ExtMinorRev;             // 03E minor revision level
   ULONG               LastCheck;               // 040 time of last check
   ULONG               CheckInterval;           // 044 max. time between checks
   ULONG               CreatorOS;               // 048 OS identification
   ULONG               ExtRevision;             // 04C Revision level EXT-FS
   USHORT              UidReserved;             // 050 uid for reserved blocks
   USHORT              GidReserved;             // 052 gid for reserved blocks
                                                //     DYNAMIC inodes only (rev 1)
   ULONG               FirstInode;              // 054 First non-reserved inode
   USHORT              InodeSize;               // 058 size of inode structure
   USHORT              BlockGroupNr;            // 05A block group# this superblock
   ULONG               FeatureCompat;           // 05C compatible feature set
   ULONG               FeatureInComp;           // 060 incompatible feature set
   ULONG               FeatureRoComp;           // 064 readonly-compatible features
   BYTE                Uuid[EXT_LEN_UUID];      // 068 128-bit uuid (4-2-2-2-6)
   char                VolumeName[EXT_LEN_LBL]; // 078 volume name
   char                LastMounted[EXT_LEN_MNT];// 088 dir where last mounted
   ULONG               AlgorithmUsageBitmap;    // 0C8 For compression
   BYTE                PreallocBlocks;          // 0Cc #blocks to preallocate
   BYTE                PreallocDirblocks;       // 0Cd Nr to preallocate for dirs
   USHORT              Padding1;                // 0Ce filler
   BYTE                JrnUuid[EXT_LEN_UUID];   // 0D0 journal superblock uuid
   ULONG               JrnInode;                // 0E0 journal inode number
   ULONG               JrnDevice;               // 0E4 journal device number
   ULONG               LastOrphan;              // 0E8 list of inodes to delete
   ULONG               HashSeed[4];             // 0EC Htree hash seed
   BYTE                DefHasVersion;           // 0FC default has version
   BYTE                JnlBackupType;           // 0FD Journal Inode backup type
   USHORT              DescSize;                // 0FE Size of group descriptors
   ULONG               DefaultMountOpts;        // 100 Default Mount Options
   ULONG               FirstMetaBg;             // 104 First Metablockgroup
   ULONG               MkfsTime;                // 108 Timestamp of FS creation
   ULONG               JnlBlocks[17];           // 10C Backup copy of journal i_blocks
                                                //     FEATURE_COMPAT_64BIT only:
   ULONG               BlocksCountHi;           // 150 Block count, HI 32 bits
   ULONG               ReservedBlocksCountHi;   // 154 Reserved Block Count HI
   ULONG               FreeBlocksCountHi;       // 158 Free Block count     HI
   USHORT              MinExtraIsize;           // 15C All inodes have at least # bytes
   USHORT              WantExtraIsize;          // 15E New inodes should reserve # bytes
   ULONG               MiscFlags;               // 160 Miscellaneous Flags (DIR/Develop)
   ULONG               RaidStride;              // 164 Raid, log blocks read per disk
   ULN64               MmpBlock;                // 168 Multi-Mount Block number
   ULONG               RaidStripeWidth;         // 170 RAID stripe width in blocks
   BYTE                LogBgPerFlex;            // 174 Size flexible log group 2^n
   BYTE                ChecksumType;            // 175 Metadata checksum type
   USHORT              ReservedPad;
   LLONG               KBytesWritten;           // 178 Nr of KiB written over livetime
   ULONG               SnapshotIno;             // 180 Inode of active snapshot
   ULONG               SnapshotId;              // 184 Sequential ID active snapshot
   LLONG               SnapShotRblocksCount;    // 188 Reserved Blocks for snapshot
   ULONG               SnapShotListInode;       // 190 Inode# for snapshot list
   ULONG               ErrorCount;              // 194 Number of errors seen
   ULONG               FirstErrorTime;          // 198 First time error happended
   ULONG               FirstErrorIno;           // 19C First error inode
   LLONG               FirstErrorBlock;         // 1A0 First error block
   BYTE                FirstErrorFunc[32];      // 1A8 First error function name
   ULONG               FirstErrorLine;          // 1C8 First error Linenumber
   ULONG               LastErrorTime;           // 1CC Last time error happended
   ULONG               LastErrorIno;            // 1D0 Last error inode
   ULONG               LastErrorLine;           // 1D4 Last error Linenumber
   ULN64               LastErrorBlock;          // 1D8 Last error block
   BYTE                LastErrorFunc[32];       // 1E0 Last error function name
   BYTE                MountOptions[64];        // 200 ASCIIZ string of mount options
   ULONG               UsrQuotaIno;             // 240 User  quota file ino#
   ULONG               GrpQuotaIno;             // 244 Group quota file ino#
   ULONG               OverheadBlocks;          // 248 Overhead blocks (always 0)
   ULONG               BackupSuperBgs[2];       // 24C SUP backup  BGs (sparse_super2)
   BYTE                EncryptAlgos[4];         // 254 Encryption algorithms in use
   BYTE                EncryptPwSalt[16];       // 258 Encryption Salt (string2key)
   ULONG               LostFoundIno;            // 268 Lost+Found Inode #
   ULONG               PrjQuotaIno;             // 26C Project Quota Inode #
   ULONG               ChecksumSeed;            // 270 Checksum seed for metadata CS
   ULONG               Reserved[98];            // 274 Padd to end of structure
   ULONG               SupeChecksum;            // 3FC Superblock checksum
} S_EXT_SUPER;                                  // end of 's_ext_super'        1 KiB


//- Block Group specific flags (as used with Lazy BG init, and GDT_CSUM)
#define EXT_BGF_INODE_UNINIT       0x0001       // Inode table and bitmap not initialized
#define EXT_BGF_BLOCK_UNINIT       0x0002       // Block bitmap not initialized, calculate!
#define EXT_BGF_INODE_ZEROED       0x0004       // Inode table IS zeroed (lazy, on demand)

typedef struct s_ext_groupdescr                 // Group descriptors
{
  ULONG                BlockBitmap;             // 00 Block# for block bitmap
  ULONG                InodeBitmap;             // 04 Block# for inode bitmap
  ULONG                InodeTable;              // 08 Block# for inode table
  USHORT               FreeBlocks;              // 0C nr of free blocks
  USHORT               FreeInodes;              // 0E nr of free inodes
  USHORT               Directories;             // 10 inodes that are directories
  USHORT               bgFlags;                 // 12 Block Group Flags         (EXT4)
  ULONG                ExclBitmap;              // 14 Exclude bitmap        low (EXT4)
  USHORT               BlockBmCsum;             // 18 Block bitmap checksum low (EXT4)
  USHORT               InodeBmCsum;             // 1A Inode bitmap checksum low (EXT4)
  USHORT               UnusedInodes;            // 1C Unused Inode count,   low (EXT4)
  USHORT               GdCheckSum;              // 1E Grp-descriptor checksum   (EXT4)
} S_EXT_GROUPDESCR;                             // end of struct "s_ext_groupdescr"


#define EXT_BLOCKPTRS            12             // # of direct block pointers

                                                // Inode flags
#define EXT_FL_SECRM             0x00000001     // Requires secure delete       (not impl)
#define EXT_FL_UNRM              0x00000002     // Preserve on undelete         (not impl)
#define EXT_FL_COMPR             0x00000004     // File is compressed           (not impl)
#define EXT_FL_SYNC              0x00000008     // All writes must be synchronous
#define EXT_FL_IMMUTABLE         0x00000010     // File is immutable  (don't move blocks!)
#define EXT_FL_APPEND            0x00000020     // File can only be appended
#define EXT_FL_NODUMP            0x00000040     // do not dump file             (dumputil)
#define EXT_FL_NOATIME           0x00000080     // do not update access time
#define EXT_FL_DIRTY             0x00000100     // Dirty compressed file        (not impl)
#define EXT_FL_COMPRBLK          0x00000200     // File has compressed clusters (not impl)
#define EXT_FL_NOCOMPR           0x00000400     // Don't compress this file
#define EXT_FL_ENCRYPT           0x00000800     // Inode is encrypted
#define EXT_FL_INDEX             0x00001000     // Directory has hash-indexes
#define EXT_FL_IMAGIC            0x00002000     // AFS magic directory
#define EXT_FL_JRNL_DATA         0x00004000     // File data must be be journaled
#define EXT_FL_NO_TAIL           0x00008000     // File tail should not be merged
#define EXT_FL_DIR_SYNC          0x00010000     // Dir data must be written synchronously
#define EXT_FL_TOP_DIR           0x00020000     // Top of directory hierarchy (root)
#define EXT_FL_HUGE_FILE         0x00040000     // Huge file,
#define EXT_FL_EXTENTS           0x00080000     // Inode uses extents for allocation
#define EXT_FL_LARGE_EA          0x00200000     // Inode for a large extended attribute
#define EXT_FL_EOFBLOCKS         0x00400000     // File has blocks past EOF   (deprecated)
#define EXT_FL_SNAPSHOT          0x01000000     // Inode is a Snapshot
#define EXT_FL_SNAP_DEL          0x04000000     // Snapshot is being deleted
#define EXT_FL_SNAPSHRUNK        0x08000000     // Snapshot shrink has completed
#define EXT_FL_INLINEDATA        0x10000000     // Inode has small INLINE data
#define EXT_FL_PROJ_INHRT        0x20000000     // Create childeren with same project
#define EXT_FL_RESERVED          0x80000000     // Reserved for EXT4 library
#define EXT_FL_USER_VISIBLE      0x004BDFFF     // User visible flags
#define EXT_FL_USER_MODIFIABLE   0x004B80FF     // User modifiable flags


#define EXT_EAH_MAGIC  0xEA020000               // XATTR/EA header magic value
#define DFS_EXT_ML     70                       // Max LFN name XATTR length (to be refined)
#define DFS_EXT_M_OFF  0                        // SLT magic filename OFF
#define DFS_EXT_MFILE  1                        // for filenames only
#define DFS_EXT_MDIRS  2                        // for directories
#define DFS_EXT_MBOTH (DFS_EXT_MFILE | DFS_EXT_MDIRS)

typedef struct s_ext_xattr_entry
{
   BYTE                eNameLength;             // to be refined
} S_EXT_XATTR_ENTRY;                            // end of struct "s_ext_xattr_entry"

typedef struct s_ext_dfsmagic                   // Inline EA, layout for DFSee recover
{
   ULONG               iEaMagicNumber;          // 00 Inline Inode Magic number
   S_EXT_XATTR_ENTRY   iEaEntryLfName;          // 04 EA entry for filename  (recover)
   S_EXT_XATTR_ENTRY   iEaEntryLnkIno;          // 18 EA entry for 1st parent Inode#
   ULONG               iEaEntryEndLst;          // 2C EA end of entry-list (must be 0)
   ULONG               iEaValueLnkIno;          // 30 Value of the LnkIno attribute
   char                iEaValueLfName[48];      // 34 Value of the LfName attribute
} S_EXT_DFSMAGIC;                               // end of struct "s_ext_dfsmagic"


//- Special allocation-structure for GDT-reserved, points to a single block of unknown info
//- Not really documented, to be refined
typedef struct s_ext_gdtreserved
{
   ULONG               Block[EXT_BLOCKPTRS];    // 00 should all be ZERO
   ULONG               Reserved1;               // 30 Should be ZERO
   ULONG               infoBlock;               // 34 Block# GDT reserved areas
   ULONG               Reserved2;               // 38 Should be ZERO
} S_EXT_GDTRESERVED;                            // end of struct "s_ext_blkindirect"


//- Classic block+indirect-block allocation structure for the Inode
typedef struct s_ext_blkindirect
{
   ULONG               Block[EXT_BLOCKPTRS];    // 00 ptr direct to data blocks
   ULONG               Indirect1;               // 30 ptr 1st indirection level
   ULONG               Indirect2;               // 34 ptr 2nd indirection level
   ULONG               Indirect3;               // 38 ptr 3rd indirection level
} S_EXT_BLKINDIRECT;                            // end of struct "s_ext_blkindirect"

#define EXT_EXTENT_MAGIC         0xF30A

//- Set of structures defining the EXTENTS allocation scheme in EXT4
typedef struct s_ext_extheader
{
   USHORT              ehMagic;                 // magic number 0xF30A
   USHORT              ehEntries;               // number of valid entries
   USHORT              ehMaxEntr;               // maximum nr of entries
   USHORT              ehDepth;                 // Depth THIS node (leaf is 0)
   ULONG               ehGeneration;            // Tree generation (unused now)
} S_EXT_EXTHEADER;                              // end of struct "s_ext_extheader"

typedef struct s_ext_extindex
{
   ULONG               eiRelBlock;              // 1st Relative block covered
   ULONG               eiBlockLo;               // Block# lower level leaf/idx LO
   USHORT              eiBlockHi;               // Block# lower level leaf/idx HI
   USHORT              eiUnused;
} S_EXT_EXTINDEX;                               // end of struct "s_ext_extindex"

typedef struct s_ext_extleaf
{
   ULONG               elRelBlock;              // 1st Relative block covered
   USHORT              elLength;                // Number of blocks, value coded!
   USHORT              elStartHi;               // Block# actual data extent,  HI
   ULONG               elStartLo;               // Block# actual data extent,  LO
} S_EXT_EXTLEAF;                                // end of struct "s_ext_extleaf"

typedef struct s_ext_exttail                    // last 4 bytes of NODEs outside Inode
{
   ULONG               etChecksum;
} S_EXT_EXTENT_TAIL;                            // end of struct "s_ext_extent_tail"

typedef union  u_ext_extentry
{
   S_EXT_EXTINDEX      ix;                      // index node, extents of more nodes
   S_EXT_EXTLEAF       lf;                      // leaf node,  extents of data blocks
} U_EXT_EXTENTRY;                               // end of union "u_ext_extentry"

typedef struct s_ext_extnode                    // Extent node, as used in Inode
{
   S_EXT_EXTHEADER     hdr;                     // header, also defines real size
   U_EXT_EXTENTRY      ent[4];                  // max 4 entries (as used in Inode)
} S_EXT_EXTNODE;                                // end of struct "s_ext_extnode"

typedef union u_ext_alloc
{
   S_EXT_GDTRESERVED    gdt;                    // gdt-reserved area info block
   S_EXT_BLKINDIRECT    bi;                     // block-indirect allocation info
   S_EXT_EXTNODE        ex;                     // extent-tree root, allocation info
} U_EXT_ALLOC;                                  // end of union "u_ext_alloc"

//- Note: BlocksLo/Hi sometimes seems to be slightly larger that actually allocated
typedef struct s_ext_inode                      // EXT2/3/4 INODE structure
{
   USHORT              Mode;                    // 00 File mode
   USHORT              UidLow;                  // 02 Low 16 bits of User-id
   ULONG               SizeLo;                  // 04 Size in bytes, low 32 bits
   ULONG               Atime;                   // 08 Access time
   ULONG               Itime;                   // 0C Inode Change time
   ULONG               Mtime;                   // 10 Modification time
   ULONG               Dtime;                   // 14 Deletion Time
   USHORT              GidLow;                  // 18 Low 16 bits of Group-id
   USHORT              Links;                   // 1A #Links
   ULONG               BlocksLo;                // 1C #Blocks, low 32 bits
   ULONG               Flags;                   // 20 File flags
   ULONG               VersionLo;               // 24 Version Low  (generation)
   U_EXT_ALLOC         Alloc;                   // 28 Allocation structure (60)
   ULONG               Generation;              // 64 NFS file version
   ULONG               EaBlockLo;               // 68 EA, low  32 bits (File-ACL)
   ULONG               SizeHi;                  // 6C Size in bytes, high 32 bits
   ULONG               FragmAddr;               // 70 Fragment address (obsolete)
   USHORT              BlocksHi;                // 74 #Blocks, high 16 bits
   USHORT              EaBlockHi;               // 76 EA, high 16 bits (File-ACL)
   USHORT              UidHigh;                 // 78 High 16 bits of User-id
   USHORT              GidHigh;                 // 7A High 16 bits of Group-id
   USHORT              IchecksumLo;             // 7C Checksum, low  16 bits
   USHORT              Ireserved;               // 7E Unused
                                                // 80 GOOD_OLD_EXT2_INODE 128 bytes

   USHORT              extraSize;               // 80 Size of THIS inode, - 128
   USHORT              IchecksumHi;             // 82 Checksum, high 16 bits
   ULONG               ItExtra;                 // 88 Inode Change time, extra bits
   ULONG               MtExtra;                 // 8C Modification time, extra bits
   ULONG               AtExtra;                 // 84 Access time, extra bits
   ULONG               Ctime;                   // 90 File Creation time
   ULONG               CtExtra;                 // 94 File Creation time, extra bits
   ULONG               VersionHi;               // 98 Version High (generation)
   ULONG               ProjectId;               // 9C Project ID (when FRO bit set!)

   ULONG               InlineEAs[24];           //    Filler, can be used for EA's

} S_EXT_INODE;                                  // end of 's_ext_inode'



#define EXT_FT_UNKNOWN           0x0000                // Directory filetypes
#define EXT_FT_REG_FILE          0x0001
#define EXT_FT_DIR               0x0002
#define EXT_FT_CHRDEV            0x0003
#define EXT_FT_BLKDEV            0x0004
#define EXT_FT_FIFO              0x0005
#define EXT_FT_SOCK              0x0006
#define EXT_FT_SYMLINK           0x0007


typedef struct s_ext_direntry
{
   ULONG               Inode;                   // Inode number
   USHORT              RecLen;                  // Directory entry length
   BYTE                NameLen;                 // Name length
   BYTE                FileType;
   char                Name[4];                 // File name, size for DOT/DOTDOT
} S_EXT_DIRENTRY;



#define EXT_DH_LEGACY_H          0              // original hash
#define EXT_DH_HALF_MD4          1              // half-MD4 (current EXT4)
#define EXT_DH_TEA_HASH          2
#define EXT_UH_LEGACY_H          3              // unsigned variants
#define EXT_UH_HALF_MD4          4
#define EXT_UH_TEA_HASH          5

typedef struct s_ext_dxentry                    // DIR index entry
{
   ULONG               hashValue;               // hash value for
   ULONG               relBlockDir;             // relative block for hash in DIR
} S_EXT_DXENTRY;                                // end of struct "s_ext_dxentry"

//- Note: DOT and DOTDOT are regular/linear DIR entries, DOTDOT length will
//-       skip OVER the index stuff to the end of this block, to next block
typedef struct s_ext_dxroot                     // DIR index root block
{
   S_EXT_DIRENTRY      dotEntry;                // the DOT    directory entry
   S_EXT_DIRENTRY      ddtEntry;                // the DOTDOT directory entry
   ULONG               fakeInode;               // reserved, zero
   BYTE                hashVersion;             // has algorithm used 0..5
   BYTE                infoLength;              // length of info, fixed 8
   BYTE                indirectLevels;          // depth of tree, leaf = 0
   BYTE                unusedFlags;             // reserved flags, fixed 0
   USHORT              maxEntries;              // max nr of DX recs in rootblock
   USHORT              usedEntries;             // used DX entries (+ 1 ?)
   ULONG               relBlock1st;             // relative block for lowest hash
   S_EXT_DXENTRY       entries[1];              // DX entries upto end (or CSUM)
} S_EXT_DXROOT;                                 // end of struct "s_ext_dxroot"

typedef struct s_ext_dxnode                     // DIR index internal node block
{
   ULONG               fakeInode;               // reserved, zero
   BYTE                hashVersion;             // has algorithm used 0..5
   BYTE                infoLength;              // length of info, fixed 8
   BYTE                indirectLevels;          // depth of tree, leaf = 0
   BYTE                unusedFlags;             // reserved flags, fixed 0
   USHORT              maxEntries;              // max nr of DX recs in rootblock
   USHORT              usedEntries;             // used DX entries (+ 1 ?)
   ULONG               relBlock1st;             // relative block for lowest hash
   S_EXT_DXENTRY       entries[1];              // DX entries upto end (or CSUM)
} S_EXT_DXNODE;                                 // end of struct "s_ext_dxnode"

#endif
