//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// EXFAT disk structure definitions
//
// Author: J. van Wijk
//
// JvW  14-06-2014   Initial version, derived from DFSFAT
//
#ifndef    DFSEXFT_H
   #define DFSEXFT_H

#define EXFAT_FREECLUS  ((ULONG) 0x00000000)    // free cluster indicator
#define EXFAT_INVALIDC  ((ULONG) 0x00000001)    // invalid cluster value
#define EXFAT_MAXCLUST  ((ULONG) 0xfffffff5)    // maximum CL for EXFAT
#define EXFAT_BADCLUST  ((ULONG) 0xfffffff7)    // bad cluster indicator
#define EXFAT_MEDIA_CL  ((ULONG) 0xfffffff8)    // first EOF / media type
#define EXFAT_EOFCLUST  ((ULONG) 0xffffffff)    // default EOF cluster value

#define EXFAT_BOOTSIZE           12

#define EXFAT_MAINBOOT    ((LSN) 0x00000000)
#define EXFAT_BACKUPBOOT  ((LSN) 0x0000000c)

#define EXFAT_BOOTSECTOR  ((LSN) 0x00000000)
#define EXFAT_BTEXTENDED  ((LSN) 0x00000001)
#define EXFAT_OEM_PARAMS  ((LSN) 0x00000009)
#define EXFAT_BTRESERVED  ((LSN) 0x00000010)
#define EXFAT_BTCHECKSUM  ((LSN) 0x00000011)

#define EXFAT_VF_FATMASK         0x0001
#define EXFAT_FAT1ACTIVE         0x0000
#define EXFAT_FAT2ACTIVE         0x0001

#define EXFAT_VF_DCMASK          0x0002
#define EXFAT_VOL_CLEAN          0x0000
#define EXFAT_VOL_DIRTY          0x0002

#define EXFAT_VF_BADMASK         0x0004
#define EXFAT_NO_PENDING         0x0000
#define EXFAT_PENDINGBAD         0x0004




#define SG_EXFAT       7
#define SV_EXFAT       (char)0xff,(char)0xff,(char)0xff, \
                       (char)0xff,(char)0xff,(char)0xff,(char)0xff
typedef struct s_exfat                         // FAT disk structure 32-bit
{
   BYTE                 MediaType;
   char                 Signature[SG_EXFAT];
   ULONG                FatValue[126];          // first will be for cluster 2
} S_EXFAT;                                      // end of struct "s_fat32"




//=================== DIR ENTRY TYPE INFO (1st byte of entry ==========================
//- Normal files usually consist of an ET 0x85,
//- followed by a 0xC0 data stream, with start cluster and size
//- followed by one or more 0xC1 with the filename in Unicode
//- On DELETE, the in-use bit will be cleared, leaving 0x05, 0x40 en 0x41 entries

// EXFAT directory entry info (each entry is 32 bytes in size)
#define EXFAT_DIRBIT_IN_USE   ((BYTE) 0x80)     // usage,     1 = in-use,    0 = deleted
#define EXFAT_DIRBIT_SECOND   ((BYTE) 0x40)     // catagory,  1 = secondary, 0 = prim
#define EXFAT_DIRBIT_OPTION   ((BYTE) 0x20)     // importance 1 = optional,  0 = critical
#define EXFAT_DIRBIT_C_MASK   ((BYTE) 0x1F)     // Entry code mask
#define EXFAT_DIRBITSUNUSED   ((BYTE) 0x18)     // Entry code bits that are NOT used (yet?)

// Base DIR entry codes (DEC) actually used
#define EXFAT_DEC_BITMAP      ((BYTE) 0x01)     // critical file, one bit/cluster   (1)
#define EXFAT_DEC_UPCASE      ((BYTE) 0x02)     // critical file, upper case tables (1)
#define EXFAT_DEC_VLABEL      ((BYTE) 0x03)     // critical entry, volume label     (1)
#define EXFAT_DEC_FILE        ((BYTE) 0x05)     // regular  entry, file             (3+)

#define EXFAT_DEC_VOLGUID     ((BYTE) 0x00)     // optional entry, Vol GUID
#define EXFAT_DEC_TEXFATPAD   ((BYTE) 0x01)     // optional file, TexFAT padding
#define EXFAT_DEC_WINCEACT    ((BYTE) 0x02)     // optional file, access table

#define EXFAT_DEC_STREAM      ((BYTE) 0x00)     // secondary entry, data stream
#define EXFAT_DEC_FILENAME    ((BYTE) 0x01)     // secondary entry, filename fragment


// Combined DIR entry types (DIR) and deleted entries (DEL)
#define EXFAT_DIR_EODIR       ((BYTE) 0x00)     // end of directory, rest is unused

#define EXFAT_DIR_BITMAP      (EXFAT_DEC_BITMAP    | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DIR_UPCASE      (EXFAT_DEC_UPCASE    | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DIR_VLABEL      (EXFAT_DEC_VLABEL    | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DIR_FILE        (EXFAT_DEC_FILE      | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DEL_FILE        (EXFAT_DEC_FILE)

#define EXFAT_DIR_VOLGUID     (EXFAT_DEC_VOLGUID   | EXFAT_DIRBIT_OPTION | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DIR_TEXFATPAD   (EXFAT_DEC_TEXFATPAD | EXFAT_DIRBIT_OPTION | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DIR_WINCEACT    (EXFAT_DEC_WINCEACT  | EXFAT_DIRBIT_OPTION | EXFAT_DIRBIT_IN_USE)

#define EXFAT_DIR_STREAM      (EXFAT_DEC_STREAM    | EXFAT_DIRBIT_SECOND | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DEL_STREAM      (EXFAT_DEC_STREAM    | EXFAT_DIRBIT_SECOND)
#define EXFAT_ANY_STREAM      (EXFAT_DEC_STREAM    | EXFAT_DIRBIT_SECOND)
#define EXFAT_DIR_FILENAME    (EXFAT_DEC_FILENAME  | EXFAT_DIRBIT_SECOND | EXFAT_DIRBIT_IN_USE)
#define EXFAT_DEL_FILENAME    (EXFAT_DEC_FILENAME  | EXFAT_DIRBIT_SECOND)
#define EXFAT_ANY_FILENAME    (EXFAT_DEC_FILENAME  | EXFAT_DIRBIT_SECOND)


typedef struct s_exftd_raw                      // RAW entry
{
   BYTE                EntryByte[ 31];          // Directory entry type
} S_EXFTD_RAW;                                  // end of struct "s_exftd_raw"

#define EXFAT_LABEL_SIZE     11
typedef struct s_exftd_label                    // Volume label
{
   BYTE                Length;                  // Length of the label string
   USHORT              Name[ EXFAT_LABEL_SIZE]; // Label name in Unicode
   BYTE                fil1[  8];
} S_EXFTD_LABEL;                                // end of struct "s_exftd_label"

#define EXFAT_1ST_BITMAP              0x00
#define EXFAT_2ND_BITMAP              0x01
#define EXFAT_BITMAP_MASK             0x01
typedef struct s_exftd_bitmap                   // Allocation bitmap
{
   BYTE                Flags;                   // 0 = 1st bitmap, 1 = 2nd
   BYTE                fil1[ 18];               // Reserved
   ULONG               Cluster;                 // First Cl in bitmap alloc
   LLONG               Length;                  // Length of bitmap in bytes
} S_EXFTD_BITMAP;                               // end of struct "s_exftd_bitmap"

typedef struct s_exftd_upcase                   // Upcase tables
{
   BYTE                fil1[ 3];                // Reserved
   ULONG               Checksum;                // Table checksum
   BYTE                fil2[12];                // Reserved
   ULONG               Cluster;                 // First Cl in upcase alloc
   LLONG               Length;                  // Length of upcase in bytes
} S_EXFTD_UPCASE;                               // end of struct "s_exftd_upcase"

typedef struct s_exftd_vguid                    // Volume GUID
{
   BYTE                SecCount;                // Secondary count (always 0)
   USHORT              Checksum;                // Fileset checksum
   USHORT              GenFlags;                // General flags (allocation)
   BYTE                VolumeGuid[16];          // Reserved
   BYTE                fil2[10];                // Reserved
} S_EXFTD_VGUID;                                // end of struct "s_exftd_vguid"

#define MAX_S_MS10     199                      // upto 2 seconds of 10 msec units
typedef struct s_exftd_file                     // File, attributes and timestamps
{
   BYTE                SecCount;                // Secondary count (nonzero!)  0x01
   USHORT              Checksum;                // Fileset checksum            0x02
   USHORT              Attrib;                  // File attributes (FATTR_**)  0x04
   BYTE                fil1[ 2];                // Reserved                    0x06
   S_DATIM             TmCreate;                // Create timestamp  TIME      0x08
   S_DATIM             DtCreate;                // Create timestamp  DATE
   S_DATIM             TmModify;                // Modify timestamp  TIME      0x0c
   S_DATIM             DtModify;                // Modify timestamp  DATE
   S_DATIM             TmAccess;                // Access timestamp  TIME      0x10
   S_DATIM             DtAccess;                // Access timestamp  DATE
   BYTE                MsCreate;                // Create 10ms  incr. 0..199   0x14
   BYTE                MsModify;                // Modify 10ms  incr. 0..199   0x15
   char                TzCreate;                // Create 15min +/-  TZ diff   0x16
   char                TzModify;                // Modify 15min +/-  TZ diff
   char                TzAccess;                // Access 15min +/-  TZ diff   0x18
   BYTE                fil2[ 7];
} S_EXFTD_FILE;                                 // end of struct "s_exftd_file"

#define EXFAT_ALLOC_POSSIBLE      0x01
#define EXFAT_CHAIN_INVALID       0x02
typedef struct s_exftd_stream                   // File, data stream, allocation
{
   BYTE                GenFlags;                // General flags (allocation)
   BYTE                fil1[ 1];                // Reserved
   BYTE                NameLength;              // Length in all name fragments
   USHORT              NameHash;                // Name Hash for quick search
   BYTE                fil2[ 2];                // Reserved
   ULN64               ValidLength;             // Valid Data Length (allocated)
   BYTE                fil3[ 4];                // Reserved
   ULONG               Cluster;                 // First Cl in file allocation
   ULN64               Length;                  // Length of filedata in bytes
} S_EXFTD_STREAM;                               // end of struct "s_exftd_stream"

#define EXFAT_NAME_SIZE      15
typedef struct s_exftd_name                     // File, name fragment, unicode
{
   BYTE                GenFlags;                // General flags (allocation)
   USHORT              Name[EXFAT_NAME_SIZE];   // Name fragment for file
} S_EXFTD_NAME;                                 // end of struct "s_exftd_name"

typedef struct s_exftdir
{
   BYTE                EntryType;               // Directory entry type
   union
   {
      S_EXFTD_RAW      rw;
      S_EXFTD_LABEL    lb;                      // Volume label
      S_EXFTD_BITMAP   bm;                      // Allocation bitmap
      S_EXFTD_UPCASE   up;                      // Upcase tables
      S_EXFTD_VGUID    vg;                      // Volume GUID
      S_EXFTD_FILE     fl;                      // File, attributes and timestamps
      S_EXFTD_STREAM   st;                      // File, data stream, allocation
      S_EXFTD_NAME     nm;                      // File, name fragment, unicode
   };
} S_EXFTDIR;                                    // end of struct "s_exftdir"



#endif
