//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// IBM Bootmanager/SETBOOT related functions, used by DFSee and SETBOOT (OEMSB)
//
// Author: J. van Wijk
//
// Avoid the bugs in IBM setboot with BootManager in other than first primary

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK sector types
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsetbt.h>                            // DFSee setboot function


#define SETB_CLASSIC        1                   // use IBM SETBOOT.EXE layout
#define SETB_STANDARD       0                   // use DFSee standard  layout

#if defined (OEMSB)
   #define SWCH "/"
#else
   #define SWCH "-"
#endif

// Set specified setboot structure to option value
static BOOL dfsFdskSetbName                     // RET   set name allowed
(
   BMINFOSEC          *sb,                      // IN    BM setboot info sec
   ULONG               si,                      // IN    system-index
   char                opt,                     // IN    option letter
   char               *descr                    // OUT   set description
);


// Update bootmanager setup for BMGR on specified disk (setboot operation)
static ULONG dfsFdskSetBootUpdate
(
   USHORT              disknr,                  // IN    disk with a BM part
   BMINFOSEC          *infosec,                 // IN    updates to BM info
   char               *descr                    // IN    confirmation descr
);                                              //       unchanged have 0xEE


// Implement the 'BmUpdate' callback operation on the bootmanager info
static ULONG dfsFdskBmUpdate                    // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM info
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
);


// Display a bootmanager info sector
// Note: fields with value 0xEE should be considered 'empty' and not displayed
static void dfsFdskBmInfoDisplay
(
   BMINFOSEC          *bmi                      // IN    Bootmgr info sector
);

#if defined (OEMSB)
// Display a bootmanager info sector in CLASSIC setboot.exe format (LVM level)
static void dfsFdskBmInfoClassic
(
   BMINFOSEC          *bmi                      // IN    Bootmgr info sector
);
#endif

// Show current bootmanager setup for BMGR on specified disk (setboot query)
static ULONG dfsFdskSetBootQuery
(
   USHORT              disknr                   // IN    disk with a BM part
);

// Implement the 'BmQuery' callback operation on the bootmanager info
static ULONG dfsFdskBmQuery                     // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM info
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
);



/*****************************************************************************/
// Specific implementation for the SETBOOT functionality (DFSAFDSK "setb")
/*****************************************************************************/
ULONG setbCommand
(
   void
)
{
   ULONG               rc = NO_ERROR;
   BMINFOSEC           new;                     // new BM info values
   TXLN                s0;                      // string assembly buffer
   BOOL                changes = FALSE;         // changes to new BM info
   BOOL                reboot  = FALSE;         // reboot requested
   char               *value;                   // argument value
   char                option;                  // case-insensitive
   USHORT              disknr;                  // disknr to work on

   ENTER();

   //- Note: options/switches made case-insensitive like original SETBOOT.EXE

   strcpy(  s0, "");                            // empty additional description

   if ( TxaOption('?') ||                       // explicit help, or
       (TxaOptCount() == 0) ||                  // no options specified
       (TxaArgCount() > 1))                     // parameters specified
   {
      #if defined (OEMSB)
         #if   defined (UNIX)
            TxPrint( "\n Note: Requires ROOT privileges! Use 'sudo' or 'su'\n");
         #elif defined (WIN32)
            TxPrint( "\n Note: Requires 'administrator' or 'backup' rights!\n");
            if (TxOsVersion( NULL) >= TXVER_WINNT_VISTA) // need 'start as admin'
            {
               TxPrint( "       Use context menu 'start as administrator' to start the program!\n");
            }
         #endif
         if (TxaOption('?'))                    // explicit help request
         {
            TxPrint("\n IBM setboot.exe compatible: ");
         }
         else                                   // assume parameter error
      #endif
         {
            TxPrint("\nUsage:  SETBOOT  ");
         }
      TxPrint(       SWCH "switch[:value]     (names are case sensitive)\n\n");
      TxPrint( "        " SWCH "q:[disknr] = query Bootmanager settings (on disknr)\n");
      TxPrint( "        " SWCH "t:timeout  = menu timeout, number of seconds or NO \n");
      TxPrint( "        " SWCH "m:mode     = set menu display to normal or advanced\n");
      TxPrint( "        " SWCH "x:number   = set system-index to specified number  \n");
      TxPrint( "   " SWCH "[0..5]:BM-name  = set system 0..5 to a Bootmanager name \n");
      TxPrint( "      " SWCH "ibd:d[:]     = reboot from the drive letter specified\n");
      TxPrint( "      " SWCH "iba:BM-name  = reboot from Bootmanager name specified\n");
      #ifndef OEMSB
         TxPrint(     "        -d:d[:]     = set next-boot to this driveletter (d:)\n");
      #endif
      TxPrint( "        " SWCH "b          = shutdown filesystems and reboot now   \n");
      #ifndef OEMSB
         #if defined   (DEV32)
            TxPrint(  "        -disk:nr    = Operate on Bootmanager on specified disk\n");
            TxPrint(  "                      . = current, * = ALL, bmgr = BMGR-disk\n");
            TxPrint(  "        -F-         = no buffer flush, show PM popup message\n");
         #elif defined (WIN32)
         #else
            TxPrint(  "        -F-         = no forced disk-buffer flush on reboots\n");
         #endif
         TxPrint(     "        -list       = List available names on Bootmanager menu\n");
      #endif
      if (TxaArgCount() > 1)
      {
         TxPrint( "\nWarning, supplied argument(s) unexpected: %s %s %s %s\n",
                  TxaArgValue(1), TxaArgValue(2), TxaArgValue(3), TxaArgValue(4));
      }
   }

   if (TxaOptSet( TXA_O_DISK))
   {
      disknr = dfsGetDiskNumber( TXA_O_DISK, ".", NULL, FALSE); // specified disk or FDSK_ANY
   }
   else                                         // when not specified, do ALL disks with a BMGR
   {
      disknr = FDSK_ANY;
   }

   TRACES(("SETB resolved disk-number: %hu = 0x%4.4hx\n", disknr, disknr));

   if (TxaOption(TXA_O_LIST))                   // 'list' option/switch
   {
      USHORT           parts = dfsPartitions();
      USHORT           pid;
      DFSPARTINFO     *p;

      TxPrint( "\nBootmanager names available for booting (case sensitive!):\n\n");
      for (pid = 1; pid <= parts; pid++)
      {
         if (((p = dfsGetPartInfo( pid)) != NULL) && (strlen(p->blabel) > 0))
         {
            ULONG        ulSpc = (p->geoHeads * p->geoSecs); // sectors/cylinder
            DFSDISKINFO *d     = dfsGetDiskInfo( p->disknr);

            if (d->Removable)
            {
               strcpy( s0, "(Removable, will NOT be on BMGR menu)");
            }
            else if (ulSpc && ((p->basePsn / ulSpc) > 1023))
            {
               strcpy( s0, "  (requires I13X capable bootmanager)");
            }
            else
            {
               strcpy( s0, "");
            }
            TxPrint( " %2.2s  %-20.20s  %2.2hx  %-8.8s   %s\n",
                     p->drive, p->blabel, p->partent.PartitionType, p->fsform, s0);
         }
      }
      TxPrint( "\n");
   }

   memset( &new, 0xEE, sizeof(new));            // fill with special 'empty'

   if (TxaOptSet('0'))                          // set system n to name
   {
      changes = dfsFdskSetbName( &new, 0, '0', s0);
      if (!changes) rc = DFS_NO_CHANGE;
   }
   if (TxaOptSet('1'))                          // set system n to name
   {
      changes = dfsFdskSetbName( &new, 1, '1', s0);
      if (!changes) rc = DFS_NO_CHANGE;
   }
   if (TxaOptSet('2'))                          // set system n to name
   {
      changes = dfsFdskSetbName( &new, 2, '2', s0);
      if (!changes) rc = DFS_NO_CHANGE;
   }
   if (TxaOptSet('3'))                          // set system n to name
   {
      changes = dfsFdskSetbName( &new, 3, '3', s0);
      if (!changes) rc = DFS_NO_CHANGE;
   }
   if (TxaOptSet('4'))                          // set system n to name
   {
      changes = dfsFdskSetbName( &new, 4, '4', s0);
      if (!changes) rc = DFS_NO_CHANGE;
   }
   if (TxaOptSet('5'))                          // set system n to name
   {
      changes = dfsFdskSetbName( &new, 5, '5', s0);
      if (!changes) rc = DFS_NO_CHANGE;
   }
   #if defined (OEMSB)
   if (TxaOption('b') || TxaOption('B'))        // reboot (/B)
   {
      if (dfsa->batch || TxConfirm( 0, "Reboot the system now [Y/N] : "))
      {
         reboot  = TRUE;
      }
      else
      {
         rc = DFS_NO_CHANGE;
      }
   }
   #else
   if (TxaOption('b'))                          // uppercase 'B' is Batch!
   {
      reboot  = TRUE;                           // reboot, no confirm
   }
   #endif
   if (TxaOptSet('d') || TxaOptSet('D'))        // drive next boot
   {                                            // try lowercase first
      option = (TxaOptSet('d')) ? 'd' : 'D';
      value  = TxaOptStr( option, "Reboot/IBD ", "");
      new.forceboot = (BYTE) (strlen(value) ? toupper(value[0]) : 0);
      sprintf( s0, "Set next reboot to driveletter %c: ?", toupper(value[0]));
      changes = TRUE;
   }
   if (TxaOptSet('i') || TxaOptSet('I'))        // IBx reboot (/IBD or /IBA)
   {
      option = (TxaOptSet('i')) ? 'i' : 'I';
      value  = TxaOptStr( option, "Reboot/IBx ", "C");
      if ((strlen( value) <= 1) ||              // drive letter => /IBD
          (value[1] == ':'))                    // driveletter + colon
      {
         new.forceboot = (BYTE)       (strlen(value) ? toupper(value[0]) : 0);
         sprintf( s0, "Reboot to driveletter '%c:' ?", toupper(value[0]));
         changes = TRUE;
         reboot  = TRUE;
      }
      else                                      // BMGR name => /IBA
      {
         if ((changes = dfsFdskSetbName( &new, BMSYS_FORCEDB, option, NULL)) != FALSE)
         {
            sprintf( s0, "Reboot to '%s' ?", TxaOptStr( option, NULL, ""));
            reboot  = TRUE;
         }
         else
         {
            rc = DFS_NO_CHANGE;
         }
      }
   }
   if (TxaOptSet('m') || TxaOptSet('M'))        // mode advanced or normal
   {
      option = (TxaOptSet('m')) ? 'm' : 'M';
      value  = TxaOptStr( option, "Mode A - N ", "A");
      if (toupper(value[0]) == 'A')
      {
         new.advanced = (BYTE) 1;
      }
      else                                      // normal
      {
         new.advanced = (BYTE) 0;
      }
      sprintf( s0, "Set BM-menu-mode to %s ?",
                       (new.advanced == 1) ? "advanced" : "normal");
      changes = TRUE;
   }
   if (TxaOptSet('t') || TxaOptSet('T'))        // timeout value
   {
      option = (TxaOptSet('t')) ? 't' : 'T';
      new.timeout = (USHORT) (TxaOptNum( option, NULL, 0));
      if (new.timeout == 0)
      {
         new.timeout = BMSYS_TIME_NO;
         strcpy( s0, "Set BM-menu to NO-timeout (no menu!) ?");
      }
      else
      {
         sprintf( s0, "Set BM-menu to a %hu second timeout ?", new.timeout);
         new.timeout *= BMSYS_TIMEOUT;
      }
      changes = TRUE;
   }
   if (TxaOptSet('x') || TxaOptSet('X'))        // system index
   {
      option = (TxaOptSet('x')) ? 'x' : 'X';
      new.defsys = (BYTE) (TxaOptNum( option, "Syst-Index ", 0)  & 0x07);
      sprintf( s0, "Set default-system to %hhu ?", new.defsys);
      changes = TRUE;
   }
   if ((rc == NO_ERROR) && (changes))           // need to update BM on disk
   {
      if (disknr == FDSK_ANY)
      {
         USHORT        disk;
         DFSDISKINFO  *d;

         for (disk = 1; (disk <= dfsPartitionableDisks()) && (rc == NO_ERROR); disk++)
         {
            if ((d = dfsGetDiskInfo( disk)) != NULL)
            {
               TRACES(("Disk:%hu Primaries:%hu Total:%hu BMGR-PSN:0x%llx\n", disk, d->primaries, d->totalpart, d->bmgrPsn));
               if (d->bmgrPsn != L64_NULL)
               {
                  rc = dfsFdskSetBootUpdate( disk, &new, s0);
               }
            }
         }
      }
      else
      {
         rc = dfsFdskSetBootUpdate( disknr, &new, s0);
      }
   }
   if ((rc == NO_ERROR) && (changes))           // show resulting values
   {
      TxPrint( "\nBootmanager values have been updated, current values are:\n");
      (void) dfsFdskSetBootQuery( disknr);
   }
   if (TxaOptSet('q') || TxaOptSet('Q') || TxaOptSet(TXA_O_QUERY))
   {
      if (disknr == 0)                          // not set yet, try option value
      {
         option = (TxaOptSet('q')) ? 'q' : (TxaOptSet('Q')) ? 'Q' : TXA_O_QUERY;
         disknr = (USHORT) TxaOptNum( option, NULL, 0);
      }
      (void) dfsFdskSetBootQuery( disknr);
   }
   if ((rc == NO_ERROR) && (reboot))            // need to reboot
   {
      #if defined (OEMSB)                       // other default, no external
         TxReboot( TxaOption('f') || TxaOption('F'));
         rc = SETB_NO_DOS_SYS;                  // DOS.SYS probably not there
      #else                                     // -F- does NOT flush buffers
         dfsRebootSystem( !TxaOptUnSet('F'));   // but allows the PM-based
         rc = DFS_CMD_FAILED;
      #endif                                    // popup to appear
   }
   RETURN ( rc);
}                                               // end 'setbCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set specified setboot structure to option value
/*****************************************************************************/
static BOOL dfsFdskSetbName                     // RET   set name allowed
(
   BMINFOSEC          *sb,                      // IN    BM setboot info sec
   ULONG               si,                      // IN    system-index
   char                opt,                     // IN    option letter
   char               *descr                    // OUT   set description
)
{
   BOOL                rc = TRUE;               // function return
   char               *value;                   // compound argument value
   TXLN                message;
   TXLN                name;                    // BM name
   USHORT              parts = dfsPartitions();
   USHORT              pid;
   DFSPARTINFO        *p;

   ENTER();

   value = TxaOptStr( opt, NULL, "");
   memset( name, (strlen(value)) ? ' ' : 0,  sizeof( name));
   if (strlen(value))                           // setting a real name
   {
      memcpy( name, value, strlen(value));      // copy in space-padded name

      for (rc = FALSE, pid = 1; pid <= parts; pid++)
      {
         if (((p = dfsGetPartInfo( pid)) != NULL) && (strlen(p->blabel) > 0))
         {
            if (strcmp( value, p->blabel) == 0) // match
            {
               rc = TRUE;
               break;
            }
         }
      }
      if (rc == FALSE)                          // not found
      {
         sprintf( message,  "Warning: '%s' not present on the Bootmanager menu,\n"
                            "          setting it may result in a failure to boot!\n"
                            "          Use the %slist option to list available ones\n",
                             value, SWCH);
         if (dfsa->batch)
         {
            TxPrint( "\n%s", message);
            rc = TRUE;                          // allow anyway ...
         }
         else                                     // interactive/confirm mode
         {
            strcat( message, "\nAre you sure you want to set this name [Y/N : ");
            rc = TxConfirm( 5052, message);
         }
      }
   }
   if (rc)                                      // update allowed ?
   {
      memcpy( &sb->system[si].BmSysShort, name, BMSYS_SHORT);
      memcpy( &sb->syslng[si].BmSysLong,  name, BMSYS_LONG);

      if (descr != NULL)                        // set final confirmation text
      {
         sprintf( descr, "Set system-index %u to '%s' ?", si, value);
      }
   }
   BRETURN (rc);
}                                               // end 'dfsFdskSetbName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update bootmanager setup for BMGR on specified disk (setboot operation)
/*****************************************************************************/
static ULONG dfsFdskSetBootUpdate
(
   USHORT              disknr,                  // IN    disk with a BM part
   BMINFOSEC          *infosec,                 // IN    updates to BM info
   char               *descr                    // IN    confirmation descr
)                                               //       unchanged have 0xEE
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // bmgr disk info
   FDSK_CB_INFO        cbs;                     // info for 'sub' operations

   ENTER();

   memset( &cbs, 0, sizeof(cbs));               // initial cb info
   cbs.disknr    = ((disknr != 0) && (disknr != FDSK_ANY)) ? disknr : dfsa->bmgrDisk;

   if (((d = dfsGetDiskInfo( cbs.disknr)) != NULL) && (d->bmgrPsn != L64_NULL))
   {
      TxPrint("Updating IBM Bootmanager info on disk number: %hu\n", disknr);

      cbs.sn        = d->bmgrPsn +
                      BMINFOSECTN;
      cbs.number    = BMINFOSECTS;

      cbs.more      = infosec;

      strcpy( cbs.string, descr);

      rc = dfsExecOnSectors( dfsFdskBmUpdate, &cbs);
   }
   else
   {
      TxPrint("There is no bootmanager present on ");
      if (cbs.disknr == dfsa->bmgrDisk)
      {
         TxPrint("any disk\n");
      }
      else
      {
         TxPrint("disk %u\n", cbs.disknr);
      }
      rc = DFS_NOT_FOUND;
   }
   RETURN (rc);
}                                               // end 'dfsFdskSetBootUpdate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'BmUpdate' callback operation on the bootmanager info
/*****************************************************************************/
static ULONG dfsFdskBmUpdate                    // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM info
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return
   BMINFOSEC          *bmi = (BMINFOSEC *) sec; // Bootmgr info sector
   BMINFOSEC          *new;
   int                 i;
   char               *name;

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if ((sec != NULL) && (cbp->more != NULL))    // info available
   {
      new = (BMINFOSEC *) cbp->more;            // updates
      switch (phase)
      {
         case FDSK_PREPARE:
            {
               TXLN    confirmation;

               sprintf( confirmation, "BMGR on disk: %hu, %s%s [Y/N] : ", cbp->disknr, cbp->string,
                        (strchr( cbp->string, '?')) ? "" : "Update Bootmanager info ?");

               TxPrint("\nBootmanager values to be updated:\n");
               dfsFdskBmInfoDisplay( new);
               if (!dfsa->batch && !cbp->confirmed &&
                   !TxConfirm( 5051, confirmation))
               {
                  rc = DFS_CMD_FAILED;          // aborted
               }
            }
            break;

         case FDSK_PERFORM:
            if (new->timeout != 0xEEEE)         // NULL (no update) value
            {
               bmi->timeout      = new->timeout;
               dfsa->bmgrTimeout = new->timeout / BMSYS_TIMEOUT;
            }
            if (new->defsys != 0xEE)            // NULL (no update) value
            {
               bmi->defsys = new->defsys;
            }
            if (new->advanced != 0xEE)          // NULL (no update) value
            {
               bmi->advanced = new->advanced;
            }
            if (new->forceboot != 0xEE)        // NULL (no update) value
            {
               bmi->forceboot = new->forceboot;
            }
            for (i = 0; i <= BMSYS_LASTONE; i++)
            {
               name = new->system[i].BmSysShort;
               if ((*name & 0xff) != 0xEE)      // NULL (no update) value
               {
                  memcpy( bmi->system[i].BmSysShort, name, BMSYS_SHORT);
               }
            }
            for (i = 0; i <= BMSYS_LASTONE; i++)
            {
               name = new->syslng[i].BmSysLong;
               if ((*name & 0xff) != 0xEE)      // NULL (no update) value
               {
                  memcpy( bmi->syslng[i].BmSysLong, name, BMSYS_LONG);
               }
            }
            rc = DFS_PENDING;                   // request write-back
            break;

         case FDSK_PENDING:
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskBmUpdate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display a bootmanager info sector
// Note: fields with value 0xEE should be considered 'empty' and not displayed
/*****************************************************************************/
static void dfsFdskBmInfoDisplay
(
   BMINFOSEC          *bmi                      // IN    Bootmgr info sector
)
{
   int                 i;
   char               *name;

   ENTER();
   if (bmi != NULL)                             // info available
   {
      TxPrint( "\n");
      if (bmi->timeout != 0xEEEE)               // NULL (no update) value
      {
         TxPrint( "Timeout value sec : ");
         if (bmi->timeout == BMSYS_TIME_NO)
         {
            TxPrint( "None%*.*s= Wait for selection\n", 34, 34, "");
         }
         else
         {
            TxPrint( "%u\n",  bmi->timeout / BMSYS_TIMEOUT);
         }
      }
      if (bmi->defsys != 0xEE)                  // NULL (no update) value
      {
         TxPrint( "System boot index : %u\n", bmi->defsys);
      }
      if (bmi->advanced != 0xEE)                // NULL (no update) value
      {
         TxPrint( "Menu display mode : %s\n",
                    (bmi->advanced) ? "Advanced" : "Normal");
      }
      if (bmi->forceboot != 0xEE)               // NULL (no update) value
      {
         TxPrint( "Forced boot drive : ");
         if (bmi->forceboot == 0)
         {
            TxPrint( "None%*.*s= Not set (by /ibd:d)\n", 34, 34, "");
         }
         else
         {
            TxPrint( "%s%c%s\n", CBG, bmi->forceboot, CNN);
         }
      }

      for (i = 0; i <= BMSYS_LASTONE; i++)
      {
         if (((bmi->system[i].BmSysShort[0] & 0xff) != 0xEE) ||
             ((bmi->syslng[i].BmSysLong[ 0] & 0xff) != 0xEE)  )
         {
            TxPrint( "System %u, classic : ", i);
            name = bmi->system[i].BmSysShort;

            if ((*name & 0xff) != 0xEE)         // NULL (no update) value
            {
               TxPrint( "%-*.*s  ", BMSYS_SHORT, BMSYS_SHORT,
                         (bmi->system[i].BmSysShort[0]) ?
                          bmi->system[i].BmSysShort : "None");
            }
            else
            {
               TxPrint( "--same--  ");
            }
            name = bmi->syslng[i].BmSysLong;
            if ((*name & 0xff) != 0xEE)         // NULL (no update) value
            {
               TxPrint( " LVM: %-*.*s  ", BMSYS_LONG, BMSYS_LONG,
                         (bmi->syslng[i].BmSysLong[0]) ?
                          bmi->syslng[i].BmSysLong : "None");
            }
            else
            {
               TxPrint( " LVM:  No change requested  ");
            }
            switch (i)
            {
               case BMSYS_DEFAULT: TxPrint( "= Default system");       break;
               case BMSYS_FORCEDB: TxPrint( "= Forced boot (/iba:)");  break;
               case BMSYS_LASTONE: TxPrint( "= Last booted system");   break;
               default:
                  break;
            }
            TxPrint("\n");
         }
      }
   }
   VRETURN();
}                                               // end 'dfsFdskBmInfoDisplay'
/*---------------------------------------------------------------------------*/


#if defined (OEMSB)
/*****************************************************************************/
// Display a bootmanager info sector in CLASSIC setboot.exe format (LVM level)
// Note: Excluding the bug/feature to skip system-4 and show 5 instead ... :-)
/*****************************************************************************/
static void dfsFdskBmInfoClassic
(
   BMINFOSEC          *bmi                      // IN    Bootmgr info sector
)
{
   int                 i;

   ENTER();
   if (bmi != NULL)                             // info available
   {
      TxPrint( "Timeout         ");
      if (bmi->timeout == BMSYS_TIME_NO)
      {
         TxPrint( "No\n");
      }
      else
      {
         TxPrint( "%u\n",  bmi->timeout / BMSYS_TIMEOUT);
      }
      TxPrint( "System index    %u\n",  bmi->defsys);
      TxPrint( "Mode            %s\n", (bmi->advanced) ? "Advanced" : "Normal");
      for (i = 0; i <= BMSYS_LASTONE; i++)
      {
         TxPrint( "System %u       %-*.*s\n", i, BMSYS_LONG, BMSYS_LONG,
                                 (bmi->syslng[i].BmSysLong[0]) ?
                                  bmi->syslng[i].BmSysLong : "None");
      }
   }
   VRETURN();
}                                               // end 'dfsFdskBmInfoClassic'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Show current bootmanager setup for BMGR on specified disk (setboot query)
/*****************************************************************************/
static ULONG dfsFdskSetBootQuery
(
   USHORT              disknr                   // IN    disk with a BM part
)                                               //       when 0, any disk, and
{                                               //       use SETBOOT.EXE format
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // bmgr disk info
   FDSK_CB_INFO        cbs;                     // info for 'sub' operations

   ENTER();

   memset( &cbs, 0, sizeof(cbs));               // initial cb info
   cbs.nowrite   = TRUE;                        // no write back
   cbs.confirmed = TRUE;                        // no prompting
   cbs.disknr    = ((disknr != 0) && (disknr != FDSK_ANY)) ? disknr : dfsa->bmgrDisk;

   if (((d = dfsGetDiskInfo( cbs.disknr)) != NULL) && (d->bmgrPsn != L64_NULL))
   {
      cbs.sn        = d->bmgrPsn +
                      BMINFOSECTN;
      cbs.number    = BMINFOSECTS;

      #if defined (OEMSB)
      if (disknr == 0)                          // automatic BMGR disk select
      {
         cbs.cbOptNum1 = SETB_CLASSIC;
      }
      else                                      // specific disk requested
      #endif
      {
         cbs.cbOptNum1 = SETB_STANDARD;
      }
      rc = dfsExecOnSectors( dfsFdskBmQuery, &cbs);
   }
   else
   {
      TxPrint("There is no bootmanager present on ");
      if (cbs.disknr == dfsa->bmgrDisk)
      {
         TxPrint("any disk\n");
      }
      else
      {
         TxPrint("disk %u\n", cbs.disknr);
      }
      rc = DFS_NOT_FOUND;
   }
   RETURN (rc);
}                                               // end 'dfsFdskSetBootQuery'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Implement the 'BmQuery' callback operation on the bootmanager info
/*****************************************************************************/
static ULONG dfsFdskBmQuery                     // RET PENDING  => call again
(                                               //     NO_ERROR => all done
   BYTE               *sec,                     // INOUT sectors with BM info
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *cbp                      // IN    operation info
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("Phase:%u Related:%u Confirmed:%u Option:%u Flag:%u  Disk:%hu thisSn:%llx\n",
            phase, cbp->related, cbp->confirmed, cbp->option, cbp->flag, cbp->disknr, cbp->thisSn));
   if (sec != NULL)                             // info available
   {
      switch (phase)
      {
         case FDSK_PREPARE:
            break;

         case FDSK_PERFORM:
            #if defined (OEMSB)
            if (cbp->cbOptNum1 == SETB_CLASSIC)
            {
               dfsFdskBmInfoClassic((BMINFOSEC *) sec);
            }
            else
            #endif
            {
               TxPrint( "\nBootmanager disk  : %hu", cbp->disknr);
               dfsFdskBmInfoDisplay((BMINFOSEC *) sec);
            }
            break;

         case FDSK_PENDING:
            break;

         default:
            rc = DFS_CMD_FAILED;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskBmQuery'
/*---------------------------------------------------------------------------*/

