//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee eComStation major functions: genpart
//
// Author: J. van Wijk
//
// JvW  05-04-2005   Initial version, modelled after DFSVOEM.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfseoem.h>                            // DFS  eCS major functions
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsupart.h>                           // PART utility functions
#include <dfsafdsk.h>                           // FDSK command functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfsdgen.h>                            // DFS  dialog interfaces

#define DFSE_DEFAULT_BASENAME  "genpart"


static  char       *cmd_lvmredo[] =
{
   "",
   "Refresh LVM information sectors (DLAT) for specified disk(s)",
   "",
   " Usage: LVMREDO  disknr | . | *  [-C]",
   "",
   "   disknr    = disk-number, '.' for current or '*' for all disks",
   "",
   "       -C    = Create new LVM-info for partitions that don't have any yet,",
   "               Includes a partition and volume name, but no driveletter.",
   "",
   NULL
};


static  char       *cmd_genpart[] =
{
   "",
   "Generate DFSee SCRIPT to create partitions for specified disk(s)",
   "",
   " Usage: GENPART disknr | . | * [filename | -b:'filename'] [descr] [-s] [-f-]",
   "",
   "   disknr    = disk-number, '.' for current or '*' for all disks",
   "",
   "   filename  = base filename for the script to be created, the actual",
   "               disknumber will be appended to this, and a default file",
   "               extension of '.dfs' will be added.",
   "",
   "   descr     : Optional description string that will be added to the",
   "               confirmation message when the script is RUN later.",
   "",
   "   -f        : Include all existing freespace areas as-is   (default)",
   "   -f-       : Do NOT include freespace areas, by allowing automatic",
   "               placement of the new partitions by DFsee (no location)",
   "",
   "   -s-       : Use MiB based size and location values       (default)",
   "   -s        : Use SECTOR based size and location values (exact copy)",
   "               instead of regular cylinder and megabyte values.",
   "",
   "   -!        : Force interactive dialog to specify/confirm options",
   "",
   NULL
};

// first  part of script, parameter: 2 x USHORT, disk number
static char genpart_header[] =
   ";Script generated by GENPART, recreate partitions\n"
   "; ~ param1 = [disk-number]~~Default is the original disk number %hu~\n"
   ";\n"
   ";;defaultparam 1 %hu\n"
   ";\n";

// second part of script, parameter: 2 x string, base script name, description
static char genpart_prolog[] =
   ";\n"
   "log  %s$1\n"
   "about -P-\n"
   ";\n"
   "confirm Partitions will be recreated on disk $1~~%s%sPress a key to start\n"
   ";\n"
   "fdisk disk $1\n"
   ";\n"
   "; show begin situation\n"
   ";\n"
   "part -m -n -e -d\n"
   "part -d"
   ";\n"
   "newmbr  -d  -clean  ;start with empty disk!\n"
   ";\n"
   "batch on ;no more confirmations from here ...\n"
   ";\n";


// last   part of script, parameter: none
static char genpart_finish[] =
   ";\n"
   "batch off\n"
   ";\n"
   "; show result\n"
   ";\n"
   "screen off\n"
   "pt * -d -r\n"
   "part -m -n -e -d\n"
   "screen on\n"
   "part -d\n"
   ";\n"
   "log\n"
   ";\n"
   "confirm Recreate partitions completed, press a key to exit the DFSee program ...\n"
   "q\n";


// Generate a CREATE-PARTITIONS script for the specified disk
static ULONG dfseGenPartDisk
(
   USHORT              disk,                    // IN    disknumber
   char               *fbase,                   // IN    base script filename
   char               *descr,                   // IN    system/disk description
   BOOL                freesp,                  // IN    keep freespace areas
   BOOL                sectors                  // IN    pos/size in sectors
);


// Rewrite LVM DLAT information sectors on one disk
static ULONG dfseLvmRedoDisk
(
   USHORT              disk,                    // IN    disknumber
   BOOL                create                   // IN    create LVM-info if
);                                              //       not yet present

/*****************************************************************************/
// Generate DFSee SCRIPT to create partitions for specified MBR/GPT disk(s)
/*****************************************************************************/
ULONG dfseGenPartScript
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                fstring;                 // filename string buffer
   TXLN                descr;                   // description
   TXSETVALUE          freespace = TX_UNKNOWN;
   TXSETVALUE          sectors   = TX_UNKNOWN;
   USHORT              index;

   ENTER();

   if (TxaOptSet( 'f'))
   {
      freespace = (TxaOption( 'f')) ? TX_CHECKED : TX_UNCHECKED;
   }
   if (TxaOptSet( 's'))
   {
      sectors   = (TxaOption( 's')) ? TX_CHECKED : TX_UNCHECKED;
   }

   index = (TxaArgCount() > 1) ? dfsParseDiskSpec( TxaArgValue(1), NULL) : FDSK_ANY;

   if ((TxaArgCount() > 1) && !TxaOption('?') && !TxaOption('!'))
   {
      USHORT        first;
      USHORT        last;

      if (!TxaOption('b'))                      // no script name option
      {
         // get basename from the 2nd parameter
         strcpy( fstring, TxaArgValue(2));
         if (strlen(fstring) > 0)               // we have a script name
         {
            TxStripExtension(  fstring);
         }
         else
         {
            strcpy( fstring, DFSE_DEFAULT_BASENAME);
         }
         TxaGetArgString( TXA_CUR, 3, 0, TXMAXLN, descr); // description from arg 3
      }
      else
      {
         //- get basename from -b option or use the default
         strcpy( fstring, TxaOptStr( 'b', NULL, DFSE_DEFAULT_BASENAME));
         TxaGetArgString( TXA_CUR, 2, 0, TXMAXLN, descr); // description from arg 2
      }
      TxPrint( "\n");

      first = dfsParseDiskSpec( TxaArgValue(1), &last);   //- request as a range
      if (last != 0)                                      //- valid single disk or range
      {
         for (index = first; (index <= last) && (rc == NO_ERROR); index++)
         {
            if (dfsDiskAccessible( index))
            {
               rc = dfseGenPartDisk( index, fstring, descr, (freespace != TX_UNCHECKED), (sectors == TX_CHECKED));
            }
         }
      }
      else
      {
         TxPrint( "Disknr is invalid : %u\n", first);
         rc = DFS_VALUE_ERROR;
      }
   }
   else
   {
      #if defined (USEWINDOWING)
         if (( TxaOption('?')) ||               // explicit help request
             (!txwIsWindow( TXHWND_DESKTOP)))   // or no desktop there
         {
            TxShowTxt( cmd_genpart);            // give usage
         }
         else                                   // present interactive dialog
         {
            dfsGenPartDialog( index, TxaArgValue(2), TxaOptStr( TXA_O_LABEL, NULL, ""), sectors, freespace);
         }
      #else
         TxShowTxt( cmd_genpart);               // give usage
      #endif
   }
   RETURN (rc);
}                                               // end 'dfseGenPartScript'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Generate a CREATE-PARTITIONS script for the specified disk
/*************************************************************************************************/
static ULONG dfseGenPartDisk
(
   USHORT              disk,                    // IN    disknumber
   char               *fbase,                   // IN    base script filename (may include path!)
   char               *descr,                   // IN    system/disk description
   BOOL                freesp,                  // IN    keep freespace areas
   BOOL                sectors                  // IN    pos/size in sectors
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                text;                    // text buffer
   FILE               *fh;
   USHORT              parts;                   // number of partitions
   USHORT              part;                    // current partition
   DFSPARTINFO        *p;
   DFSDISKINFO        *d;
   TXTM                option;

   ENTER();
   TRACES(( "disk:%hu  fbase:'%s'  descr:'%s'  free:%s  sectors:%s\n",
             disk, fbase, descr, (freesp) ? "YES" : "NO", (sectors) ? "YES" : "NO"));

   sprintf( text, "%s%hu.dfs", fbase, disk);    // full script name

   if ((d = dfsGetDiskInfo(disk)) != NULL)
   {
      if ((fh = fopen(text, "w")) == 0)
      {
         TxPrint("Error opening script : '%s' for write\n", text);
         rc = ERROR_OPEN_FAILED;
      }
      else
      {
         time_t        tt = time( &tt);         // current date/time

         TxPrint("Generating script : '%s', for disk %hu\n", text, disk);
         fprintf( fh, genpart_header, disk, disk);

         strftime( text, TXMAXLN, "%Y-%m-%d %H:%M:%S", localtime( &tt));
         fprintf( fh, ";------- %s -------\n;GENPART %s disk:%u = '%s' on %s Geo CHS:%7u %3u %-3u\n;%s; version %s %s\n",
                  descr, d->pStyle, disk, d->DiskName, text, d->geoCyls, d->geoHeads, d->geoSecs, DFS_N, DFS_V, DFS_C);

         fprintf( fh, genpart_prolog, TxGetBaseName(fbase), descr, (strlen( descr) != 0) ? "~~" : "");

         if (!sectors)                          // force geometry to be same
         {                                      // as when creating the script
            fprintf( fh, ";Set explicit disk geometry for created partitions\n");
            fprintf( fh, "geo %u %u %u\n;\n", d->geoCyls, d->geoHeads, d->geoSecs);
         }
         if (d->gptPartitions != 0)             // it is a GPT style disk!
         {
            fprintf( fh, ";Create the GPT guard, without trying to recover existing GPT tables\n");
            fprintf( fh, "cr gpt -C\n;\n");
         }

         parts = dfsPartitions();
         for ( part = 1; part <= parts; part++)
         {
            p = dfsGetPartInfo( part);
            if (p && (p->disknr == disk))
            {
               if (p->tablenr == GPT_STYLE)
               {
                  strcpy( text, dfsFsUuidValueString(p->typeGuid) + 1); // copy UUID str without leading {
                  fprintf( fh, "cr gpt '%36.36s' -name:'%s' ", text, p->lvm.VoluName); // name is in lvm info
               }
               else                             // regular MBR style partition
               {
                  fprintf( fh, "cr %s -t:0x%2.2hhx ", (p->primary) ? "pri" : "log", p->partent.PartitionType);
               }

               TRACES(( "PID:%hu sectors:%llu last:%llx base:%llx part:%llx\n", part, p->sectors, p->lastPsn, p->basePsn, p->partPsn));

               if (sectors)             // use HEX sectors units for size, on logicals, incl EBR track!
               {
                  fprintf( fh, "-s:0x%llx,s", (p->lastPsn - ((p->primary) ? p->basePsn :  p->partPsn) +1));
               }
               else                             // use decimal megabytes for size
               {
                  fprintf( fh, "-s:%.0lf,m", TXSMIB( p->sectors, p->bpsector) - (double) 0.5);
               }
               if (freesp)
               {
                  if (sectors)
                  {
                     fprintf( fh, " -a:0x%llx,s", (p->primary) ? p->basePsn : p->partPsn);
                  }
                  else
                  {
                     ULONG         c,h,s;

                     dfsGeoPsn2Chs( p->basePsn, p->geoHeads, p->geoSecs, &c, &h, &s);
                     sprintf( option, "%u,c", c);
                     fprintf( fh, " -a:%-10.10s", option);
                  }
               }
               fprintf( fh, " -S:%hu", p->partnr);              //- prefered slot for partition
               if ((p->primary == FALSE) && (d->ebrHead) && (p->ebrChain))
               {
                  fprintf( fh, " -X:%hu", d->ebrHead->partnr);  //- prefered slot MBR container
                  fprintf( fh, " -Y:%hu", p->ebrChain->partnr); //- prefered slot EBR container
               }

               if (p->lvmPresent)
               {
                  TxCopy( option, p->lvm.VoluName, LVM_NAME_L);
                  fprintf( fh, " -L:\"-v:'%s'", option);
                  TxCopy( option, p->lvm.PartName, LVM_NAME_L);
                  fprintf( fh, " -p:'%s'", option);
                  if (p->lvm.letter)
                  {
                     fprintf( fh, " -l:'%c'", p->lvm.letter);
                  }
                  if (p->lvm.OnBmMenu)
                  {
                     fprintf( fh, " -m");
                  }
                  fprintf( fh, "\"");
               }
               if ((p->primary) && (p->partent.Status & 0x80))
               {
                  fprintf( fh, " -F");          // make recreated ACTIVE too
               }
               fprintf( fh, "\n");
            }
         }
         if (d->lvmPartitions != 0)             // some LVM info present
         {
            fprintf( fh, ";\nlvm -d -n:'%s'\n", d->DiskName);
         }
         fprintf( fh, "%s", genpart_finish);
         fclose(  fh);
      }
   }
   RETURN (rc);
}                                               // end 'dfseGenPartDisk'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Rewrite LVM DLAT information sectors to clear up inconsistencies or damage
/*****************************************************************************/
ULONG dfseLvmRedoDlat
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              index;

   ENTER();

   if ((TxaArgCount() > 1) && !TxaOption('?'))
   {
      index = (TxaArgCount() > 1) ? dfsParseDiskSpec( TxaArgValue(1), NULL) : FDSK_ANY;
      #if defined (USEWINDOWING)
         if ((dfsa->batch) || TxConfirm( 5022,
              "Refresh all LVM DLAT sectors, while keeping names "
              "and driveletters, for %s %s ? [Y/N] : ",
               (index == FDSK_ANY) ? "ALL"   : "disk",
               (index == FDSK_ANY) ? "disks" :  TxaArgValue(1)))
      #endif
         {
            USHORT        first;
            USHORT        last;

            first = dfsParseDiskSpec( TxaArgValue(1), &last); // request as a range
            if (last != 0)                      // valid single disk or range
            {
               for (index = first; (index <= last) && (rc == NO_ERROR); index++)
               {
                  if (dfsDiskAccessible( index))
                  {
                     rc = dfseLvmRedoDisk( index, (TxaOption('C')));
                  }
               }
            }
            else
            {
               TxPrint( "Disknr is invalid : %u\n", first);
               rc = DFS_VALUE_ERROR;
            }
         }
      #if defined (USEWINDOWING)
         else
         {
            rc = DFS_NO_CHANGE;
         }
      #endif
   }
   else
   {
      TxShowTxt( cmd_lvmredo);                  // give usage
   }
   RETURN (rc);
}                                               // end 'dfseLvmRedoDlat'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Rewrite LVM DLAT information sectors on one disk
/*************************************************************************************************/
static ULONG dfseLvmRedoDisk
(
   USHORT              disk,                    // IN    disknumber
   BOOL                create                   // IN    create LVM-info if
)                                               //       not yet present
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                s1;                      // command buffer
   USHORT              parts;                   // number of partitions
   USHORT              part;                    // current partition
   DFSPARTINFO        *p;
   DFSDISKINFO        *d;

   ENTER();
   TRACES(( "disk:%hu\n", disk));

   if ((d = dfsGetDiskInfo(disk)) != NULL)
   {
      TxPrint("Redo LVM for disk : %hu, ", disk);
      if (d->pStyle[0] == DFSP_MBR)
      {
         TxPrint( "in progress\n");

         parts = dfsPartitions();
         for (part = 1; part <= parts; part++)
         {
            p = dfsGetPartInfo( part);
            if (p && (p->disknr == disk))
            {
               if (p->partent.PartitionType == DFS_P_WARP_LVM)
               {
                  sprintf( s1, "lvm -B %hu -D", part); //- redo from BBR area
                  rc = dfsMultiCommand( s1, 0, TRUE, FALSE, FALSE);
               }
               else                                    //- redo from known info
               {
                  if (p->lvmPresent)
                  {
                     sprintf( s1, "lvm -B %hu -D -s -n:'%s' "
                                  "-v:'%s' -p:'%s' -l:%c -m%c",
                              part,                 d->DiskName,
                              p->lvm.VoluName,      p->lvm.PartName,
                             (p->lvm.letter != 0) ? p->lvm.letter : '-',
                             (p->lvm.OnBmMenu)    ? ' '           : '-');
                     rc = dfsMultiCommand( s1, 0, TRUE, FALSE, FALSE);
                  }
                  else
                  {
                     if (create)
                     {
                        sprintf( s1, "lvm -B %hu -V", part); //- create default
                        rc = dfsMultiCommand( s1, 0, TRUE, FALSE, FALSE);
                     }
                     else
                     {
                        TxPrint( "Partition %hu does NOT have any LVM "
                                 "information defined yet.\n", part);
                     }
                  }
               }
            }
         }
      }
      else
      {
         TxPrint( "skipped, disk style '%s' is not compatible with LVM!\n", d->pStyle);
      }
   }
   RETURN (rc);
}                                               // end 'dfseLvmRedoDisk'
/*-----------------------------------------------------------------------------------------------*/

