//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS EJECT, functions to eject and test for removable media (like USB)
//
// Author: J. van Wijk
//
// JvW  05-02-2006 Initial version, supporting USB media on OS/2
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfseject.h>                           // DFS removable/eject support

#if   defined (WIN32)
#elif defined (DOS32)
#elif defined (UNIX)
#else

#define DFS_DC_BLOCKR         0x00              // block device removable
#define DFS_DC_GETBPB         0x00              // get device bpb info

#define DFS_DC_UNLOCK         0x00              // unlock logical drive
#define DFS_DC_LOCK           0x01              // lock logical drive
#define DFS_DC_EJECT          0x02              // eject removable
#define DFS_DC_LOAD           0x03              // load removable

typedef struct drivecmd
{
   BYTE                cmd;                     // 0=unlock 1=lock 2=eject
   BYTE                drv;                     // 0=A, 1=B 2=C ...
} DRIVECMD;                                     // end of struct "drivecmd"

#define DFS_BPB_REMOVABLE     0x08              // BPB attribute for removable

typedef struct drivebpb
{
   TXFS_EBPB           ebpb;                    // extended BPB
   BYTE                reserved[6];
   USHORT              cyls;
   BYTE                type;
   USHORT              attributes;              // device attributes
   BYTE                fill[6];                 // documented for IOCtl
} DRIVEBPB;                                     // end of struct "drivebpb"

#endif

/*****************************************************************************/
// Eject a removable medium specified by driveletter
/*****************************************************************************/
ULONG dfsEjectRemovable
(
   char               *drive                    // IN    Driveletter to eject
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      DRIVECMD         IOCtl;
      ULONG            DataLen;
      ULONG            ParmLen;
   #endif

   ENTER();
   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups

   TRACES(("Drive: %s\n", drive));
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      ParmLen   = sizeof(DRIVECMD);
      IOCtl.cmd = DFS_DC_EJECT;
      IOCtl.drv = toupper(drive[0]) - 'A';
      DataLen   = 0;

      rc = DosDevIOCtl((HFILE) -1, IOCTL_DISK,
                       DSK_UNLOCKEJECTMEDIA,
                       &IOCtl, ParmLen, &ParmLen,
                       NULL,   DataLen, &DataLen);
   #endif
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   RETURN (rc);
}                                               // end 'dfsEjectRemovable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine if a driveletter represents a removable medium/device
/*****************************************************************************/
BOOL dfsIsRemovable                             // RET   drive is removable
(
   char               *drive                    // IN    Driveletter to test
)
{
   BOOL                rc = FALSE;
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      DRIVECMD         IOCtl;
      DRIVEBPB         RemAt;
      ULONG            DataLen;
      ULONG            ParmLen;
      BYTE             NoRem;
   #endif

   ENTER();
   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups

   TRACES(("Drive: %s\n", drive));
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      ParmLen   = sizeof(IOCtl);
      IOCtl.cmd = DFS_DC_BLOCKR;
      IOCtl.drv = toupper(drive[0]) - 'A';
      DataLen   = sizeof(NoRem);

      if (DosDevIOCtl((HFILE) -1, IOCTL_DISK,
                                  DSK_BLOCKREMOVABLE,
                                  &IOCtl, ParmLen, &ParmLen,
                                  &NoRem, DataLen, &DataLen) == NO_ERROR)
      {
         if (NoRem)                             // non-removable sofar, check
         {                                      // BPB as well (USB devices)
            ParmLen   = sizeof(IOCtl);
            IOCtl.cmd = DFS_DC_GETBPB;
            IOCtl.drv = toupper(drive[0]) - 'A';
            DataLen   = sizeof(RemAt);

            if (DosDevIOCtl((HFILE) -1, IOCTL_DISK,
                                        DSK_GETDEVICEPARAMS,
                                        &IOCtl, ParmLen, &ParmLen,
                                        &RemAt, DataLen, &DataLen) == NO_ERROR)

            {
               if (RemAt.attributes & DFS_BPB_REMOVABLE)
               {
                  TRACES(( "BPB removable, like USB\n"));
                  rc = TRUE;                    // removable, probably USB
               }
            }
         }
         else
         {
            TRACES(( "Block removable\n"));
            rc = TRUE;                          // removable block device
         }
      }
   #endif
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   BRETURN (rc);
}                                               // end 'dfsIsRemovable'
/*---------------------------------------------------------------------------*/

