//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// EFAT disk structure definitions
//
// Author: J. van Wijk
//
// JvW  14-06-2014   Initial version, derived from DFSFAT
//
#ifndef    DFSEFAT_H
   #define DFSEFAT_H

#define EFAT_FREECLUS    ((ULONG) 0x00000000)     // free cluster indicator
#define EFAT_INVALIDC    ((ULONG) 0x00000001)     // invalid cluster value
#define EFAT_MAXCLUST    ((ULONG) 0xfffffff5)     // maximum CL for EFAT
#define EFAT_BADCLUST    ((ULONG) 0xfffffff7)     // bad cluster indicator
#define EFAT_MEDIA_CL    ((ULONG) 0xfffffff8)     // first EOF / media type
#define EFAT_EOFCLUST    ((ULONG) 0xffffffff)     // default EOF cluster value

#define EFAT_BOOTSIZE             12

#define EFAT_MAINBOOT    ((ULN64) 0x00000000)
#define EFAT_BACKUPBOOT  ((ULN64) 0x0000000c)

#define EFAT_BOOTSECTOR  ((ULN64) 0x00000000)
#define EFAT_BTEXTENDED  ((ULN64) 0x00000001)
#define EFAT_OEM_PARAMS  ((ULN64) 0x00000009)
#define EFAT_BTRESERVED  ((ULN64) 0x00000010)
#define EFAT_BTCHECKSUM  ((ULN64) 0x00000011)

#define EFAT_VF_FATMASK           0x0001
#define EFAT_FAT1ACTIVE           0x0000
#define EFAT_FAT2ACTIVE           0x0001

#define EFAT_VF_DCMASK            0x0002
#define EFAT_VOL_CLEAN            0x0000
#define EFAT_VOL_DIRTY            0x0002

#define EFAT_VF_BADMASK           0x0004
#define EFAT_NO_PENDING           0x0000
#define EFAT_PENDINGBAD           0x0004




#define SG_EFAT       7
#define SV_EFAT       (char)0xff,(char)0xff,(char)0xff, \
                      (char)0xff,(char)0xff,(char)0xff,(char)0xff
typedef struct s_efat                           // FAT disk structure 32-bit
{
   BYTE                 MediaType;
   char                 Signature[SG_EFAT];
   ULONG                FatValue[126];          // first will be for cluster 2
} S_EFAT;                                       // end of struct "s_efat"




//=================== DIR ENTRY TYPE INFO (1st byte of entry ==========================
//- Normal files usually consist of an ET 0x85,
//- followed by a 0xC0 data stream, with start cluster and size
//- followed by one or more 0xC1 with the filename in Unicode
//- On DELETE, the in-use bit will be cleared, leaving 0x05, 0x40 en 0x41 entries

// EFAT directory entry info (each entry is 32 bytes in size)
#define EFAT_DIRBIT_IN_USE   ((BYTE) 0x80)      // usage,     1 = in-use,    0 = deleted
#define EFAT_DIRBIT_SECOND   ((BYTE) 0x40)      // catagory,  1 = secondary, 0 = prim
#define EFAT_DIRBIT_OPTION   ((BYTE) 0x20)      // importance 1 = optional,  0 = critical
#define EFAT_DIRBIT_C_MASK   ((BYTE) 0x1F)      // Entry code mask
#define EFAT_DIRBITSUNUSED   ((BYTE) 0x18)      // Entry code bits that are NOT used (yet?)

// Base DIR entry codes (DEC) actually used
#define EFAT_DEC_BITMAP      ((BYTE) 0x01)      // critical file, one bit/cluster   (1)
#define EFAT_DEC_UPCASE      ((BYTE) 0x02)      // critical file, upper case tables (1)
#define EFAT_DEC_VLABEL      ((BYTE) 0x03)      // critical entry, volume label     (1)
#define EFAT_DEC_FILE        ((BYTE) 0x05)      // regular  entry, file             (3+)

#define EFAT_DEC_VOLGUID     ((BYTE) 0x00)      // optional entry, Vol GUID
#define EFAT_DEC_TEFATPAD    ((BYTE) 0x01)      // optional file, TeFAT padding
#define EFAT_DEC_WINCEACT    ((BYTE) 0x02)      // optional file, access table

#define EFAT_DEC_STREAM      ((BYTE) 0x00)      // secondary entry, data stream
#define EFAT_DEC_FILENAME    ((BYTE) 0x01)      // secondary entry, filename fragment


// Combined DIR entry types (DIR) and deleted entries (DEL)
#define EFAT_DIR_EODIR       ((BYTE) 0x00)      // end of directory, rest is unused

#define EFAT_DIR_BITMAP      (EFAT_DEC_BITMAP    | EFAT_DIRBIT_IN_USE)
#define EFAT_DIR_UPCASE      (EFAT_DEC_UPCASE    | EFAT_DIRBIT_IN_USE)
#define EFAT_DIR_VLABEL      (EFAT_DEC_VLABEL    | EFAT_DIRBIT_IN_USE)
#define EFAT_DIR_FILE        (EFAT_DEC_FILE      | EFAT_DIRBIT_IN_USE)
#define EFAT_DEL_FILE        (EFAT_DEC_FILE)

#define EFAT_DIR_VOLGUID     (EFAT_DEC_VOLGUID   | EFAT_DIRBIT_OPTION | EFAT_DIRBIT_IN_USE)
#define EFAT_DIR_TEFATPAD    (EFAT_DEC_TEFATPAD  | EFAT_DIRBIT_OPTION | EFAT_DIRBIT_IN_USE)
#define EFAT_DIR_WINCEACT    (EFAT_DEC_WINCEACT  | EFAT_DIRBIT_OPTION | EFAT_DIRBIT_IN_USE)

#define EFAT_DIR_STREAM      (EFAT_DEC_STREAM    | EFAT_DIRBIT_SECOND | EFAT_DIRBIT_IN_USE)
#define EFAT_DEL_STREAM      (EFAT_DEC_STREAM    | EFAT_DIRBIT_SECOND)
#define EFAT_ANY_STREAM      (EFAT_DEC_STREAM    | EFAT_DIRBIT_SECOND)
#define EFAT_DIR_FILENAME    (EFAT_DEC_FILENAME  | EFAT_DIRBIT_SECOND | EFAT_DIRBIT_IN_USE)
#define EFAT_DEL_FILENAME    (EFAT_DEC_FILENAME  | EFAT_DIRBIT_SECOND)
#define EFAT_ANY_FILENAME    (EFAT_DEC_FILENAME  | EFAT_DIRBIT_SECOND)


typedef struct s_efatd_raw                      // RAW entry
{
   BYTE                EntryByte[ 31];          // Directory entry type
} S_EFATD_RAW;                                  // end of struct "s_efatd_raw"

#define EFAT_LABEL_SIZE     11
typedef struct s_efatd_label                    // Volume label
{
   BYTE                Length;                  // Length of the label string
   USHORT              Name[ EFAT_LABEL_SIZE];  // Label name in Unicode
   BYTE                fil1[  8];
} S_EFATD_LABEL;                                // end of struct "s_efatd_label"

#define EFAT_1ST_BITMAP              0x00
#define EFAT_2ND_BITMAP              0x01
#define EFAT_BITMAP_MASK             0x01
typedef struct s_efatd_bitmap                   // Allocation bitmap
{
   BYTE                Flags;                   // 001 value 0 = 1st bitmap, 1 = 2nd
   BYTE                fil1[ 18];               // 002 Reserved
   ULONG               Cluster;                 // 014 First Cl in bitmap alloc
   ULN64               FileLength;              // 018 Length of bitmap in bytes
} S_EFATD_BITMAP;                               // 020 end of struct "s_efatd_bitmap"

typedef struct s_efatd_upcase                   // Upcase tables
{
   BYTE                fil1[ 3];                // 001 Reserved
   ULONG               Checksum;                // 004 Table checksum
   BYTE                fil2[12];                // 008 Reserved
   ULONG               Cluster;                 // 014 First Cl in upcase alloc
   ULN64               FileLength;              // 018 Length of upcase in bytes
} S_EFATD_UPCASE;                               // 020 end of struct "s_efatd_upcase"

typedef struct s_efatd_vguid                    // Volume GUID
{
   BYTE                SecCount;                // Secondary count (always 0)
   USHORT              Checksum;                // Fileset checksum
   USHORT              GenFlags;                // General flags (allocation)
   BYTE                VolumeGuid[16];          // Reserved
   BYTE                fil2[10];                // Reserved
} S_EFATD_VGUID;                                // end of struct "s_efatd_vguid"

#define MAX_S_MS10     199                      // upto 2 seconds of 10 msec units
typedef struct s_efatd_file                     // File, attributes and timestamps
{
   BYTE                SecCount;                // Secondary count (nonzero!)  0x01
   USHORT              Checksum;                // Fileset checksum            0x02
   USHORT              Attrib;                  // File attributes (FATTR_**)  0x04
   BYTE                fil1[ 2];                // Reserved                    0x06
   S_DATIM             TmCreate;                // Create timestamp  TIME      0x08
   S_DATIM             DtCreate;                // Create timestamp  DATE
   S_DATIM             TmModify;                // Modify timestamp  TIME      0x0c
   S_DATIM             DtModify;                // Modify timestamp  DATE
   S_DATIM             TmAccess;                // Access timestamp  TIME      0x10
   S_DATIM             DtAccess;                // Access timestamp  DATE
   BYTE                MsCreate;                // Create 10ms  incr. 0..199   0x14
   BYTE                MsModify;                // Modify 10ms  incr. 0..199   0x15
   char                TzCreate;                // Create 15min +/-  TZ diff   0x16
   char                TzModify;                // Modify 15min +/-  TZ diff
   char                TzAccess;                // Access 15min +/-  TZ diff   0x18
   BYTE                fil2[ 7];
} S_EFATD_FILE;                                 // end of struct "s_efatd_file"

#define EFAT_ALLOC_POSSIBLE      0x01
#define EFAT_CHAIN_INVALID       0x02
typedef struct s_efatd_stream                   // File, data stream, allocation
{
   BYTE                GenFlags;                // 001 General flags (allocation)
   BYTE                fil1[ 1];                // 002 Reserved
   BYTE                NameLength;              // 003 Length in all name fragments
   USHORT              NameHash;                // 004 Name Hash for quick search
   BYTE                fil2[ 2];                // 006 Reserved
   ULN64               ValidLength;             // 008 Valid Data Length (allocated)
   BYTE                fil3[ 4];                // 010 Reserved
   ULONG               Cluster;                 // 014 First Cl in file allocation
   ULN64               FileLength;              // 018 Length of filedata in bytes
} S_EFATD_STREAM;                               // 020 end of struct "s_efatd_stream"

#define EFAT_NAME_SIZE      15
typedef struct s_efatd_name                     // File, name fragment, unicode
{
   BYTE                GenFlags;                // General flags (allocation)
   USHORT              Name[EFAT_NAME_SIZE];    // Name fragment for file
} S_EFATD_NAME;                                 // end of struct "s_efatd_name"

typedef struct s_efatdir
{
   BYTE                EntryType;               // Directory entry type
   union
   {
      S_EFATD_RAW      rw;
      S_EFATD_LABEL    lb;                      // Volume label
      S_EFATD_BITMAP   bm;                      // Allocation bitmap
      S_EFATD_UPCASE   up;                      // Upcase tables
      S_EFATD_VGUID    vg;                      // Volume GUID
      S_EFATD_FILE     fl;                      // File, attributes and timestamps
      S_EFATD_STREAM   st;                      // File, data stream, allocation
      S_EFATD_NAME     nm;                      // File, name fragment, unicode
   };
} S_EFATDIR;                                    // end of struct "s_efatdir"



#endif
