//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS STORE, physical/logical/virtual sector read/write (sector-IO)
//
// Author: J. van Wijk
//
// JvW  27-12-96   Initial version, loosely derived from QDISK and DHPFSERV
// JvW  01-05-97   Ported to WIN32
// JvW  21-01-98   Finished port to DOS and completed set of Write API's
// JvW  01-01-99   Added extended Int13 support for DFSDOS on disks > 1024 cyl
// JvW  06-08-99   Added physical geo display for OS/2, when supported (Aurora)
// JvW  07-12-99   More selective use of ext-int13 (avoiding some BIOS bugs)
// JvW  06-01-2000 Faster CountDisks for NT (start at 1, not 31)
// JvW  02-07-2000 Major restructure, introducing DFS stores and memory stores
// JvW  09-10-2001 Fully implemented store concept now (TODO: UNDO/REDO :-)
// JvW  03-05-2002 Fixed passing of volume letter for DOS (in vfHandle)
// JvW  28-05-2002 Finished port to DPMI version with WATCOM/Causeway extender
// JvW  16-08-2004 Linux Open/read/write plus generic device abstraction-layer
// JvW  27-07-2006 Split off sector-IO to dfsector.c, rest to dfstore.c
//

#include <txlib.h>                              // TX library interfaces

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfsupart.h>                           // FDISK partition functions
#include <dfstore.h>                            // Store and sector I/O
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS navigation and defs
#include <dfsimage.h>                           // RAW/Compressed image utils
#include <dfsutil.h>                            // DFS utility functions
#include <dfsectio.h>                           // DFS store I/O private defs
#include <dfsimzci.h>                           // DFS IMZ cache/index functions
#include <dfsvdici.h>                           // DFS VDI cache/index functions

#if !defined (OEMSB)
#include <dfsmdisk.h>                           // Disk store memoryl disks
#endif

//- Note: following struct and defs used in DFSDOS, real extended Int-13
//-       Start of struct is identical to OS/2 IOCtl OEMHLP_QUERYDISKINFO
//-       structure OEMHLPDISKINFO as defined in bsedev.h in newer toolkits

#define DFS_I13EDD_DMA    0x0001                // DMA boundary errors handled
#define DFS_I13EDD_CHS    0x0002                // CHS info is valid
#define DFS_I13EDD_REM    0x0004                // removable
#define DFS_I13EDD_WRV    0x0008                // write with verify supported
#define DFS_I13EDD_CHG    0x0010                // change-line supported
#define DFS_I13EDD_LCK    0x0020                // drive-locking supported
#define DFS_I13EDD_MAX    0x0040                // CHS values not current media
                                                //     but maximum supported

typedef struct _dfs_int13_edd                   // Int 13 extended drive data
{                                               // supporting 1.x .. 3.0 info!
   USHORT           size;                       // size of params (1a for 1.x)
   USHORT           flags;                      // informational flags
   ULONG            c;                          // nr of cylinders
   ULONG            h;                          // nr of heads
   ULONG            s;                          // nr of sectors per track
   ULN64            i13Sectors;                 // 64-bit total nr of sectors
   USHORT           bps;                        // bytes per sector
   ULONG            eddConfig;                  // EDD config params. (2.x)
                                                // ---- next is 3.x spec only
   USHORT           dpiSignature;               // sign 'BEDD' = DPI present
   BYTE             dpiLength;                  // length Device Path Info incl
                                                // signature and this (24h 3.0)
   BYTE             reserved[3];                // Reserved (0).
   char             busName[4];                 // name of bus (`ISA' or `PCI')
   char             interfaceName[8];           // name of interface type (ATA,
                                                // ATAPI, SCSI, USB, 1394, FIBRE)
   char             interfacePath[8];           // Interface Path.
   char             devicePath[8];              // Device Path.
   BYTE             reserve2;                   // Reserved (0).
   BYTE             checksum;                   // Checksum of bytes 1Eh-40h (2's complement
                                                // of sum, which makes the 8 bit sum of bytes
                                                // 1Eh-41h equal to 00h).
} DFS_INT13_EDD;                                // Int13 extention drive data


#if   defined (WIN32)
   #if _WIN32_WINNT < 0x0500
      #error Watcom ..\h\nt\SDKDDVER.H needs _WIN32_WINNT as 0x0500 for GEOMETRY_EX!
   #endif
   #include <winioctl.h>
   #include <dfsntreg.h>                        // DFS NT registry / error reporting

#elif defined (DOS32)

   typedef struct _dfs_ioctl_rw
   {
      BYTE             special;
      USHORT           h;
      USHORT           c;
      USHORT           s;
      USHORT           sectors;
      BYTE            *databuf;                 // transfer area
      BYTE             reserved[19];            // avoid stack overwrites ...
   } DFS_IOCTL_RW;

   typedef struct _dfs_int13_erw                // Int 13 extended R/W data
   {
      BYTE             size;                    // size of packet == 0x10
      BYTE             reserved;
      USHORT           sectors;                 // nr of sectors to transfer
      BYTE            *databuf;                 // transfer area
      ULN64            i13Psn;                  // 64-bit start sector-number
   } DFS_INT13_ERW;                             // Int13 extention packet

   #define DFS_I13_MAGIC_I   0x55aa             // magic value on entry
   #define DFS_I13_MAGIC_O   0xaa55             // magic value on exit

   #define DFS_I13VER_1_X    0x01
   #define DFS_I13VER_2_0    0x20
   #define DFS_I13VER_2_1    0x21
   #define DFS_I13VER_3_0    0x30               // Phoenix EDD spec

   #define DFS_I13API_EXT    0x0001             // extended Int13 available
   #define DFS_I13API_REM    0x0002             // removable Int13 support
   #define DFS_I13API_EDD    0x0004             // EDD / get params supported

   #define DFS_I13_E_TEST    0x41               // INT13 extention availability
   #define DFS_I13_E_READ    0x42               // INT13 extended read
   #define DFS_I13_E_WRITE   0x43               // INT13 extended write
   #define DFS_I13_E_PARAM   0x48               // INT13 extended get params

   static   TXDX_RMINFO      txdx_rmi;

#elif defined (LINUX)
                                                // see Linux asm/ioctl.h
   #define HDIO_GETGEO       0x0301             // GEO in 16+8+8 bit values
   #define BLKGETSIZE        0x1260             // blockdevice size in 512 byte multiples
   #define BLKGETSIZE64      0x1272             // blockdevice size in bytes (LLONG)
   #define BLKSSZGET         0x1268             // logical blocksize
   #define BLKPBSZGET        0x127B             // physical block size
   #define BLKGETLASTSECT    0x126C             // get last sect (sector number == size?)
   #define BLKSETLASTSECT    0x126D

   typedef struct linux_hd_geo
   {
      BYTE             heads;
      BYTE             sectors;
      USHORT           cylinders;
      ULONG            start;
   } LINUX_HD_GEO;

#elif defined (DARWIN)

   #include <sys/disk.h>

#else

   //- Maximum numbers of sectors (buffer) in a single read/write physical track call
   //- Larger counts will cause inconsistent behaviour and data corruption!
   #define DFS_OS2_BUFSIZE_LIMIT      127

                                                // DosPhysicalDisk
   #define INFO_GET_HANDLE    2                 // Obtain handle
   #define INFO_FREE_HANDLE   3                 // Release handle

// #ifndef OEMHLP_QUERYDISKINFO
//    #define OEMHLP_QUERYDISKINFO    0x0E      // query phys-disk geo (Aurora)
// #endif

   typedef struct _DFS_DRIVEREQ                 // parameter struct for above
   {
      UCHAR Infotype;                           // fixed, zero
      UCHAR DriveUnit;                          // index of log volume 1..n
   } DFS_DRIVEREQ;


   typedef struct _DFS_LOGDEVPARAM              // Logical device params
   {
      TXFS_EBPB           eb;                   // extended BPB
      BYTE                PhysDriveNr;          // Drive number (80 for HD 1)
      BYTE                FsysValue;            // head / CHKDSK/SCAN required
      BYTE                Signature;            // bpb signature, 28, 29, 80
      BYTE                fill[3];              // reserved
      USHORT              lCylinders;
      BYTE                lDevType;             // devicetype
      USHORT              lDevAttributes;       // device attributes
   } DFS_LOGDEVPARAM;

   #define FSCTL_SECTORIO       0x9014          // HPFS sectorio function
   #define MAGIC_SECTORIO       0xdeadface      // HPFS sectorio magic value
#endif

typedef struct _DFS_DPBLOCK                     // device parameter block
{
   USHORT reserved1;
   USHORT cCylinders;
   USHORT cHeads;
   USHORT cSectors;
   USHORT reserved2;
   USHORT reserved3;
   USHORT reserved4;
   USHORT reserved5;
} DFS_DPBLOCK;

#define DFS_IMGDISK 0                           // fake disknr for images

#if   defined (WIN32)
   #define TXHDISK     HANDLE
#elif defined (UNIX)
   #define TXHDISK     int
#endif

// Position to specified sector offset for given handle (logical/physical disk)
#if defined (WIN32) || defined (UNIX)
static ULONG dfstSeekSector
(
   DFSTOREINFO        *si,                      // IN    DFS store to be used
   TXHDISK             dh,                      // IN    handle
   ULN64               sn                       // IN    sector nr
);
#endif

// Read sector(s) from volume or imagefile in buffer starting at specified LSN
static ULONG dfstReadLog
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    Start LSN
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // OUT   Sector buffer
);

// Write sector(s) to volume or imagefile in buffer starting at specified LSN
static ULONG dfstWriteLog
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    Start LSN
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // IN    Sector buffer
);


#if defined (DOS32) || defined (DEV32)

// Read 1 .. tracksize sectors in chunks of maximum 127 (OS/2 64 KB buffer bug)
static ULONG dfstReadSectors
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    nr of sectors
   BYTE               *buffer                   // OUT   Sector buffer
);

// Read 1 .. tracksize physical sectors using Cylinder/Head/Sector address
static ULONG dfstReadSectChs
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector,                  // IN    first sector number
   ULONG               nr,                      // IN    nr of sectors
   BYTE               *buffer                   // OUT   Sector buffer
);


// Write 1 .. tracksize sectors in chunks of maximum 127 (OS/2 64 KB buffer bug)
static ULONG dfstWriteSectors
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    nr of sectors
   BYTE               *buffer                   // IN    Sector buffer
);

// Write 1 .. tracksize physical sectors using Cylinder/Head/Sector address
static ULONG dfstWriteSectChs
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector,                  // IN    first sector number
   ULONG               nr,                      // IN    nr of sectors
   BYTE               *buffer                   // IN    Sector buffer
);
#endif

#if defined (DEV32)
// Open specified volume in direct mode using persistent handle
static ULONG dfstOpenVolLock
(
   DFSTOREINFO        *si,                      // IN    DFS store to be used
   char               *drive                    // IN    Drive specification
);
#endif

#if defined (DOS32)
// Check INT13 extention availability
static BOOL dfsInt13Ext
(
   USHORT              physdnr,                 // IN    phys disk number
   BOOL                verbose                  // IN    show details
);

/*****************************************************************************/
// Check INT13 extention availability
/*****************************************************************************/
static BOOL dfsInt13Ext
(
   USHORT              physdnr,                 // IN    phys disk number
   BOOL                verbose                  // IN    show details
)
{
   BOOL                rc = FALSE;              // DOS rc
   union  REGS         regs;

   ENTER();

   memset( &regs, 0, sizeof(regs));
   regs.h.ah = DFS_I13_E_TEST;                  // Check Int13 ext availability
   regs.h.dl = (BYTE) (0x7f + physdnr);
   TXWORD.bx = DFS_I13_MAGIC_I;
   int386( DFS_I13, &regs, &regs);              // Execute BIOS int-13

   if (dfsa->warnLevel > 1)
   {
      TxPrint( "INT13 API on disk : %hu, API version is 0x%2.2hx, "
               "cflag %hu bx: %4.4hx cx %4.4hx\n",
                physdnr, (short) regs.h.ah, regs.x.cflag, TXWORD.bx, TXWORD.cx);
   }
   TRACES(( "Test INT13 on disk %hu, version: %hu, "
            "cflag %hu bx: %4.4hx cx %4.4hx\n",
             physdnr, (short) regs.h.ah, regs.x.cflag, TXWORD.bx, TXWORD.cx));
   if ((regs.x.cflag == 0) &&                   // no error reported
       (TXWORD.bx    == DFS_I13_MAGIC_O) )      // magic value is OK
   {
      if (verbose)
      {
         TxPrint( "Ext Int13 version : %hu.%hu  flags: %4.4hx "
                  "Support for GetParam, Read and Write: %s\n",
              (USHORT)(regs.h.ah / 16), (USHORT)(regs.h.ah & 0xf), TXWORD.cx,
                       (TXWORD.cx  &  DFS_I13API_EXT) ? "yes" : "no");
      }
      if (TXWORD.cx  &  DFS_I13API_EXT)         // most extentions available
      {                                         // GET PARAM + READ & WRITE
         rc = TRUE;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsInt13Ext'
/*---------------------------------------------------------------------------*/
#endif


#ifndef OEMSB
#if !defined (DFS_PHYSDISK_ONLY)
/*****************************************************************************/
// Open specified volume in direct and if possible, sector-read mode
/*****************************************************************************/
ULONG dfstOpenVolume
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   char               *drive,                   // IN    Drive specification
   char               *fsys,                    // OUT   File system info
   BOOL                verbose                  // IN    Give descriptive msgs
)
{
   ULONG               rc   = NO_ERROR;         // DOS rc
   TXHFILE             dh   = 0;                // Direct Filehandle
   DFSTOREINFO        *si = &(sti[store]);
   ULN64               base = 0;                // logical base for access
   TXTM                info;                    // detailed fsys info
   #if   defined (WIN32)
      BOOL             isPartition = FALSE;     // on a harddisk
      TXTM             dspec;
   #elif defined (DOS32)
      BOOL             isPartition = FALSE;     // on a harddisk
      TXTM             dspec;
      union  REGS      regs;
      BYTE             lett;                    // driveletter
      struct SREGS     sreg;
      TXFS_GPARM      *dta = (TXFS_GPARM *) xpar;  // DTA is DPMI compatible mem
   #elif defined (DARWIN)
   #elif defined (LINUX)
      LINUX_HD_GEO     lgeo;
   #else
      BOOL             isPartition = FALSE;     // on a harddisk
      TXTM             dspec;
      ULONG            act;                     // action taken
      ULONG            rdl = 0;                 // returned datalength
      ULONG            pwd = MAGIC_SECTORIO;    // HPFS sectorio password
      PBYTE            ppwd = (PBYTE) &pwd;     // parameter list, ptr to pw
      ULONG            lpwd = sizeof(ppwd);     // size of parameter list
      ULONG            DataLen;
      ULONG            ParmLen;
      DFS_LOGDEVPARAM  LogDevParam;             // device parameter block
   #endif

   ENTER();
   TRARGS(("store:%lu  drive:'%s'\n", store, drive));

   dfstClose( store);                           // modifies SINF->drive!
   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups
   TxFsType(drive, fsys, info);                 // get filesystem info
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler

   si->Sectors = 0;                             // Size not known yet
   si->Geo.B   = 0;                             // Bps  not known yet
   si->Geo.C   = 0;                             // Geo  not known yet

   TRACES(( "1 base: %llx  si->Sectors: %llx, FSYS:%s\n", base, si->Sectors, fsys));

   #if defined (WIN32)
      sprintf( dspec, "\\\\.\\%c:", drive[0]);  // logical volume specification
      dh = CreateFile( dspec,
                       GENERIC_READ    | GENERIC_WRITE,
                       FILE_SHARE_READ | FILE_SHARE_WRITE,
                       NULL,                    // default security info
                       OPEN_EXISTING,
                       FILE_ATTRIBUTE_NORMAL,
                       NULL);
      if (dh == INVALID_HANDLE_VALUE)           // try read-only access
      {
         dh = CreateFile( dspec,
                          GENERIC_READ,
                          FILE_SHARE_READ | FILE_SHARE_WRITE,
                          NULL,                 // default security info
                          OPEN_EXISTING,
                          FILE_ATTRIBUTE_NORMAL,
                          NULL);
         if (dh == INVALID_HANDLE_VALUE)
         {
            TRACES(( "CreateFile error on '%s': %s\n", dspec, txNtLastError()));
            rc = DFS_NO_DEVICE;
         }
         else
         {
            if (verbose)
            {
               TxPrint("Read-only access  : %s, writing not possible!\n", drive);
            }
            si->ReadOnly = TRUE;
         }
      }
      if (rc == NO_ERROR)                       // OK sofar
      {
         DISK_GEOMETRY geo;                     // opened disk geometry
         ULONG         received;                // nr of bytes received

         if (DeviceIoControl(dh,
                             IOCTL_DISK_GET_DRIVE_GEOMETRY,
                             NULL, 0,           // no input needed
                             &geo, sizeof(geo),
                             &received, NULL))
         {
            si->Geo.B = (USHORT) geo.BytesPerSector;
            si->Geo.H = (ULONG)  geo.TracksPerCylinder;
            si->Geo.S = (ULONG)  geo.SectorsPerTrack;
            si->Geo.C = (ULONG)  geo.Cylinders.QuadPart;
         }
         if (verbose)
         {
            switch (geo.MediaType)
            {
               case F5_1Pt2_512:    strcpy( dspec, "1.2 MB 5.25\" diskette"); break;
               case F3_720_512:     strcpy( dspec, "720 KB  3.5\" diskette"); break;
               case F3_1Pt44_512:   strcpy( dspec, "1.44 MB 3.5\" diskette"); break;
               case F3_2Pt88_512:   strcpy( dspec, "2.88 MB 3.5\" diskette"); break;
               case RemovableMedia: strcpy( dspec, "Removable media CD/DVD"); break;
               case FixedMedia:     strcpy( dspec, "Fixed disk (partition)");
                                                    isPartition = TRUE;       break;
               default:             strcpy( dspec, "120 MB 3.5\" / unknown"); break;
            }
            TxPrint( "Volume mediumtype : %hu = %s\n", geo.MediaType, dspec);
         }
      }
   #elif defined (DOS32)

      lett = (BYTE) (drive[0]-'A'+1);           // 1=A, 2=B etc

      memset( dta, 0, sizeof(TXFS_GPARM));
      memset( &regs,  0, sizeof(regs));

      memset( &sreg,  0, sizeof(sreg));

      memset( &txdx_rmi, 0, sizeof(txdx_rmi));
      txdx_rmi.eax = 0x440d;                    // IOctl generic block dev
      txdx_rmi.ecx = 0x0860;                    // cat Log-disk; get devparam
      txdx_rmi.ebx = lett;                      // drive in bl
      txdx_rmi.ds  = txDpmiSegment(dta);        // output buffer (ds:dx)

      TRHEXS( 500,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

      regs.w.ax    = TXDX_DPMI_RMINT;           // simulate realmode INT
      regs.h.bl    = DFS_I21;                   // DOS interrupt 21
      regs.x.edi   = FP_OFF( &txdx_rmi);        // real mode register struct
      sreg.es      = FP_SEG( &txdx_rmi);

      txDpmiCall( &regs, &sreg);

      if ((regs.x.cflag == 0)          &&           // drive exists
          (dta->cylinders     != 0)    &&
          (dta->eb.LogGeoHead != 0)    &&
          (dta->eb.LogGeoSect != 0)     )           // and has a valid geometry
      {
         TRHEXS( 500,  dta,  sizeof(TXFS_GPARM), "TXFS_GPARM");
         si->Geo.C  = dta->cylinders;
         si->Geo.H  = dta->eb.LogGeoHead;
         si->Geo.S  = dta->eb.LogGeoSect;
         //- base   = dta->eb.HiddenSectors;    // base MUST be ZERO!
         dh = (TXHFILE) lett;                   // faked handle; holds letter!

         if (verbose)
         {
            switch (dta->devtype)
            {
               case 0: strcpy( dspec, "360 MB 5.25\" diskette"); break;
               case 1: strcpy( dspec, "1.2 MB 5.25\" diskette"); break;
               case 2: strcpy( dspec, "720 KB  3.5\" diskette"); break;
               case 3: strcpy( dspec, "S-Dens 8 inch diskette"); break;
               case 4: strcpy( dspec, "D-Dens 8 inch diskette"); break;
               case 6: strcpy( dspec, "48 txi 5 inch diskette"); break;
               case 8: strcpy( dspec, "R/W optical disk, WORM"); break;
               case 9: strcpy( dspec, "2.88 MB 3.5\" diskette"); break;
               case 5: strcpy( dspec, "Fixed disk (partition)");
                                       isPartition = TRUE;       break;
               default:
                  if ((si->Geo.C <= 80) && (si->Geo.H <= 2))
                  {
                     strcpy( dspec, "1.44 MB 3.5\" diskette");
                  }
                  else
                  {
                     strcpy( dspec, "Unknown (DVD or CDROM)");
                  }
                  break;
            }
            if ((dta->devattr & 0x1) == 0)
            {
               strcat( dspec, ", removable");
            }
            TxPrint( "Volume type, attr : %hu, 0x%2.2hx = %s\n",
                      dta->devtype, dta->devattr, dspec);

            if ((dta->devattr & 0x1) == 0x1)    // non-removable, notify RO
            {
               TxNamedMessage( !dfsa->batch, 5020, " INFO: Read-Only media ",
                  "Access to this (partitioned) medium is limited to READONLY "
                  "to avoid some known DOS bugs. You CAN enable writing by using "
                  "the 'store -R- %lu' command, but make sure you are seeing the "
                  "right volume first (checking bootsectors etc).", store);
            }
         }

         if ((dta->devattr & 0x1) == 0x1)       // non-removable, make RO!
         {
            si->ReadOnly = TRUE;
         }

         TRACES(("Geometry for disk : %hu  CHS:%7lu %3lu %-3lu Size:%8.1lf MB\n",
                   (USHORT) lett, si->Geo.C,  si->Geo.H,  si->Geo.S,
                         TXSMIB(( si->Geo.C * si->Geo.H * si->Geo.S), si->Geo.B)));
      }
      else
      {
         if (regs.x.cflag == 0)
         {
            rc = ERROR_INVALID_DRIVE;           // bad geometry
         }
         else                                   // reported an error
         {
            rc = TXWORD.ax;
         }
      }
   #elif defined (UNIX)
      if ((dh = open( drive, O_RDWR | O_LARGEFILE)) == -1)
      {
         if ((dh = open( drive, O_RDONLY | O_LARGEFILE)) == -1)
         {
            rc = TxRcFromErrno( errno);
         }
         else
         {
            if (verbose)
            {
               TxPrint("Read-only access  : %s, writing not possible!\n", drive);
            }
            si->ReadOnly = TRUE;
         }
      }
      #if defined (LINUX)
         if (rc == NO_ERROR)                    // OK sofar
         {
            if (ioctl( dh, HDIO_GETGEO, &lgeo) == 0)
            {
               si->Geo.H = (ULONG)  lgeo.heads;
               si->Geo.S = (ULONG)  lgeo.sectors;
               si->Geo.C = (ULONG)  lgeo.cylinders;
            }
            else                                // error getting geo
            {
               TRACES(( "HDIO_GETGEO ioctl handle %lu: %s\n", dh, strerror(errno)));
            }
         }
      #endif
   #else

      TRACES(("drive: '%s'\n", drive));

      if    (strcasecmp( fsys, "FAT32") == 0)   // FAT32 fails R/W
      {                                         // fake readonly for FAT32
         rc = ERROR_ACCESS_DENIED;              // (not supported anyway)
      }
      else                                      // try ReadWrite access to IFS
      {
         if (dfsa->os2api->DosOpenL)
         {
            rc = (dfsa->os2api->DosOpenL)((PSZ) drive,
                         &dh,                   // file handle
                         &act,                  // action taken
                         0,                     // filesize
                         FILE_NORMAL,           // no attributes
                         FILE_OPEN,             // OpenFlag
                         OPEN_FLAGS_DASD       | // OpenMode (direct)
                         OPEN_ACCESS_READWRITE | // both read and write wanted
                         OPEN_SHARE_DENYNONE,   // OpenMode (allow sharing)
                         0);                    // reserved
         }
         else
         {
            rc = DosOpen((PSZ) drive,
                         &dh,                   // file handle
                         &act,                  // action taken
                         0,                     // filesize
                         FILE_NORMAL,           // no attributes
                         FILE_OPEN,             // OpenFlag
                         OPEN_FLAGS_DASD       | // OpenMode (direct)
                         OPEN_ACCESS_READWRITE | // both read and write wanted
                         OPEN_SHARE_DENYNONE,   // OpenMode (allow sharing)
                         0);                    // reserved
         }
      }

      if (rc == ERROR_ACCESS_DENIED)            // No writing allowed ?
      {
         if (dfsa->os2api->DosOpenL)
         {
            rc = (dfsa->os2api->DosOpenL)((PSZ) drive,
                      &dh,                      // file handle
                      &act,                     // action taken
                      0,                        // filesize
                      FILE_NORMAL,              // no attributes
                      FILE_OPEN,                // OpenFlag
                      OPEN_FLAGS_DASD       |   // OpenMode (direct)
                      OPEN_ACCESS_READONLY  |   // try just reading it
                      OPEN_SHARE_DENYNONE,      // OpenMode (allow sharing)
                      0);                       // reserved
         }
         else
         {
            rc = DosOpen((PSZ) drive,
                         &dh,                   // file handle
                         &act,                  // action taken
                         0,                     // filesize
                         FILE_NORMAL,           // no attributes
                         FILE_OPEN,             // OpenFlag
                         OPEN_FLAGS_DASD      | // OpenMode (direct)
                         OPEN_ACCESS_READONLY | // try just reading it
                         OPEN_SHARE_DENYNONE,   // OpenMode (allow sharing)
                         0);                    // reserved
         }
         if (rc == NO_ERROR)
         {
            if (verbose)
            {
               TxPrint("Read-only access  : %s, writing not possible!\n", drive);
            }
            si->ReadOnly = TRUE;
         }
      }
      if (rc == NO_ERROR)                       // OK, get geometry
      {
         dspec[0]  = GET_BPB_FROM_MEDIUM;
         ParmLen   = 1;
         DataLen   = sizeof( LogDevParam);

         rc = DosDevIOCtl(dh,
                          IOCTL_DISK,
                          DSK_GETDEVICEPARAMS,
                          dspec,        ParmLen, &ParmLen,
                          &LogDevParam, DataLen, &DataLen);
         if (rc == NO_ERROR)
         {
            if (LogDevParam.lCylinders != ((USHORT) -1))
            {
               si->Geo.C = (ULONG) LogDevParam.lCylinders;
               si->Geo.H = (ULONG) LogDevParam.eb.LogGeoHead;
               si->Geo.S = (ULONG) LogDevParam.eb.LogGeoSect;
            }
            if (verbose)
            {
               switch (LogDevParam.lDevType)
               {
                  case 0: strcpy( dspec, "360 KB 5.25\" diskette"); break;
                  case 1: strcpy( dspec, "1.2 MB 5.25\" diskette"); break;
                  case 2: strcpy( dspec, "720 KB  3.5\" diskette"); break;
                  case 3: strcpy( dspec, "S-Dens 8 inch diskette"); break;
                  case 4: strcpy( dspec, "D-Dens 8 inch diskette"); break;
                  case 6: strcpy( dspec, "48 txi 5 inch diskette"); break;
                  case 8: strcpy( dspec, "R/W optical disk, WORM"); break;
                  case 9: strcpy( dspec, "2.88 MB 3.5\" diskette"); break;
                  case 5: strcpy( dspec, "Fixed disk (partition)");
                                          isPartition = TRUE;       break;
                  default:
                     if ((si->Geo.C >   0) &&
                         (si->Geo.C <= 80) &&
                         (si->Geo.H <= 2))
                     {
                        strcpy( dspec, "1.44 MB 3.5\" diskette");
                     }
                     else
                     {
                        strcpy( dspec, "Unknown (DVD or CDROM)");
                     }
                     break;
               }
               if ((LogDevParam.lDevAttributes & 0x1) == 0)
               {
                  strcat( dspec, ", removable");
               }
               TxPrint( "Volume type, attr : %hu, 0x%2.2hx = %s\n",
                  LogDevParam.lDevType, LogDevParam.lDevAttributes, dspec);
            }
         }
         else if (verbose)
         {
            if (rc == ERROR_NOT_DOS_DISK)
            {
               TxPrint( "Device with assigned driveletter '%s' (removable) Media unknown!\n", drive);
            }
            else
            {
               TxPrint( "IOctl DISK GETDEVICEPARAMS failed for '%s', rc: %lu\n", drive, rc);
            }
         }
      }
      TRACES(( "2 base: %llx  si->Sectors: %llx\n", base, si->Sectors));
      if (rc == NO_ERROR)
      {
         rc = DosFSCtl( NULL,                   // ptr to data area (dummy)
                        0,                      // datalength on input
                        &rdl,                   // returned datalength
                        ppwd,                   // ptr to parameter list
                        sizeof(ppwd),           // max size of parameter list
                        &lpwd,                  // size of parameter list
                        FSCTL_SECTORIO,         // FCTL function code
                        NULL,                   // RoutName
                        dh,                     // file handle
                        FSCTL_HANDLE);          // Routing method, handle
         if (rc == NO_ERROR)                    // SectorIO supported
         {
            TRACES(("Handle %lu set to sectorIO instead of ByteIO\n", dh));
            si->SectorIO = TRUE;                // global indicator, allowing
         }                                      // non-std (HPFS) drivers
         else
         {
            si->SectorIO = FALSE;               // no sector-IO
            switch (rc)
            {
               case ERROR_INVALID_FUNCTION:
               case ERROR_NOT_SUPPORTED:
                  if ((verbose) && (si->Geo.C > 255) &&
                      (dfsa->os2api->DosOpenL == NULL))
                  {
                     TxPrint("SectorIO support  : not available, access "
                                "limited to first 2GB of the volume\n");
                  }
                  TRACES(("No SectorIO support; using ByteIO\n"));
                  rc = NO_ERROR;
                  break;

               default:
                  if (verbose)
                  {
                     TxPrint("DosFSCtl failure  : %lu on drive '%s'\n", (ULONG) rc, drive);
                  }
                  dfstClose( store);            // close
                  break;
            }
         }
      }
      else
      {
         switch (rc)
         {
            case ERROR_INVALID_DRIVE:
               if (verbose)
               {
                  TxPrint("Invalid volume    : %s\n", drive);
               }
               break;

            case ERROR_PATH_NOT_FOUND:          // returned on network
            default:
               if (verbose)
               {
                  TxPrint("Open error on FS  : %s, not supported (yet)\n", fsys);
               }
               break;
         }
      }
   #endif
   if (rc == NO_ERROR)                          // opened successfully
   {
      ULN64            sect = 0;                // number of sectors on volume
      ULN64            free = 0;                // number of free sectors
      USHORT           bps  = 0;

      TRACES(( "3 base: %llx  si->Sectors: %llx\n", base, si->Sectors));

      TxFsSpace( drive, &free, &sect, &bps);    // available filesystem info ...

      si->vfHandle  = dh;
      si->Type      = DFST_VOLUME;              // actual type of store
      si->is.object = DFSO_VOLD;                // volume or device

      if (si->Geo.B == 0)                       // not set, get from FsSpace
      {
         si->Geo.B = (bps) ? bps : SECTORSIZE;
      }
      if (si->Geo.C == 0)                       // not set, create default
      {
         if (si->Sectors == 0)                  // not set, use FsSpace value
         {
            si->Sectors = sect;
         }
         si->Geo.S = 32;
         si->Geo.H = 32768 / si->Geo.B;         // 1 MiB cylinder size
         si->Geo.C = (si->Sectors == 0) ? 64000 :
                   (((si->Sectors -1) / (si->Geo.H * si->Geo.S))+1);
         TRACES(( "Set Default GEO: C:%lu H:%lu S:%lu bps:%hu\n",
                   si->Geo.H, si->Geo.H, si->Geo.S, si->Geo.B));
      }
      if (si->Sectors == 0)                     // still not set, get from GEO
      {
         si->Sectors  = (ULONG) si->Geo.C * si->Geo.H * si->Geo.S;
         TRACES(( "Calculated size based on geometry: %llx\n", si->Sectors));
      }
      TRACES(( "4 base: %llx  si->Sectors: %llx\n", base, si->Sectors));

      #if !defined (UNIX)
      {
         TRACES(("Logical Volume '%s' opened, handle: %lu, sect:0x%llx\n", drive, dh, sect));

         //- Note: nr of sectors from FsSpace is often a bit too low, depending
         //-       on the type of filesystem and environment

         if ((isPartition) && (si->Sectors > sect)) // correction might be needed
         {
            //- to be refined, different for each version (DOS/WIN/OS2) ??
            //- below is optimized/tested on OS/2

            if ((si->Sectors - sect) >= ((ULONG) si->Geo.H * si->Geo.S))
            {                                   // primary, #cyl is unreliable
               si->Geo.C    =       ((sect -1) / (si->Geo.H * si->Geo.S)) +1;
               si->Sectors  = (ULONG) si->Geo.C * si->Geo.H * si->Geo.S;
            }
            else                                // probably a logical, adjust
            {                                   // for bootsector offset
               si->Sectors -= si->Geo.S;        // works for OS/2 only
            }
            TRACES(( "Corrected size for this partition: %llx\n", si->Sectors));
         }
      }
      #endif
      TRACES(( "5 base: %llx  si->Sectors: %llx\n", base, si->Sectors));
      strcpy( si->Name, drive);
      dfstGeo2Sys( si);                         // sync Original Geo with new
      TRACES(( "6 base: %llx  si->Sectors: %llx\n", base, si->Sectors));
      dfstDiskGeometry(  store, base, 0, 0, 0, 0, FALSE, verbose); //- set Geo
      TRACES(( "7 base: %llx  si->Sectors: %llx\n", base, si->Sectors));
      dfstLogicalLimits( store, base, si->Sectors + base -1);      //- set size
   }
   RETURN (rc);
}                                               // end 'dfstOpenVolume'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Open specified Raw or Compressed image file(s) as source for analysis
/*****************************************************************************/
ULONG dfstOpenImageFile
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   char               *file,                    // IN    File specification
   BOOL                append,                  // IN    Append to file allowed
   BOOL                forceRAW,                // IN    always open RAW (IMZ/VDI)
   BOOL                verbose,                 // IN    Display geometry & size
   ULONG              *fcount                   // OUT   #of files (multi) or NULL
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   TXHFILE             dh = 0;                  // Direct Filehandle
   DFSTOREINFO        *si = &(sti[store]);
   ULN64               imStart = 0;             // Image start (IMZ header)
   ULN64               sectors = 0;             // Image size in sectors
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      ULONG  act;                               // action taken
   #endif
   ULONG     imzBuf;
   ULONG     imzBps;

   ENTER();

   dfstClose( store);

   TRACES(( "Image file '%s', append: %s  forceRAW: %s\n", file, (append) ? "YES" : "NO", (forceRAW) ? "YES" : "NO"));

   #if defined (WIN32)
      dh = CreateFile( file,
                       GENERIC_READ    | GENERIC_WRITE,
                       FILE_SHARE_READ | FILE_SHARE_WRITE,
                       NULL,                    // default security info
                       OPEN_EXISTING,
                       FILE_ATTRIBUTE_NORMAL,
                       NULL);
      if (dh == INVALID_HANDLE_VALUE)           // try read-only access
      {
         dh = CreateFile( file,
                          GENERIC_READ,
                          FILE_SHARE_READ | FILE_SHARE_WRITE,
                          NULL,                 // default security info
                          OPEN_EXISTING,
                          FILE_ATTRIBUTE_NORMAL,
                          NULL);
         if (dh == INVALID_HANDLE_VALUE)
         {
            TRACES(( "CreateFile error on '%s': %s\n", file, txNtLastError()));
            rc = DFS_NO_DEVICE;
         }
         else
         {
            TxPrint("Read-only access  : %s, writing not possible!\n", file);
            si->ReadOnly = TRUE;
         }
      }
   #elif defined (DOS32)
      dh = fopen( file, "rb+");                 // open binary read/write
      if (dh == 0)
      {
         dh = fopen( file, "rb");               // open binary read-only
         if (dh == 0)
         {
            rc = DFS_NO_DEVICE;                 // to be refined
         }
         else
         {
            TxPrint("Read-only access  : %s, writing not possible!\n", file);
            si->ReadOnly = TRUE;
         }
      }
   #elif defined (UNIX)
      if ((dh = open( file, O_RDWR | O_LARGEFILE)) == -1)
      {
         if ((dh = open( file, O_RDONLY | O_LARGEFILE)) == -1)
         {
            rc = TxRcFromErrno( errno);
         }
         else
         {
            TxPrint("Read-only access  : %s, writing not possible!\n", file);
            si->ReadOnly = TRUE;
         }
      }
   #else
      if (dfsa->os2api->DosOpenL)
      {
         rc = (dfsa->os2api->DosOpenL)((PSZ) file,
                      &dh,                      // file handle
                      &act,                     // action taken
                      0,                        // filesize
                      FILE_NORMAL,              // no attributes
                      OPEN_ACTION_OPEN_IF_EXISTS,
                      OPEN_ACCESS_READWRITE |
                      OPEN_SHARE_DENYNONE,      // OpenMode (allow sharing)
                      0);                       // reserved
      }
      else
      {
         rc = DosOpen((PSZ) file,
                      &dh,                      // file handle
                      &act,                     // action taken
                      0,                        // filesize and attributes
                      FILE_NORMAL,              // no attributes
                      OPEN_ACTION_OPEN_IF_EXISTS,
                      OPEN_ACCESS_READWRITE |
                      OPEN_SHARE_DENYNONE,      // OpenMode (allow sharing)
                      0);                       // reserved
      }
      if (rc == ERROR_ACCESS_DENIED)            // No writing allowed ?
      {
         if (dfsa->os2api->DosOpenL)
         {
            rc = (dfsa->os2api->DosOpenL)((PSZ) file,
                         &dh,                   // file handle
                         &act,                  // action taken
                         0,                     // filesize
                         FILE_NORMAL,           // no attributes
                         OPEN_ACTION_OPEN_IF_EXISTS,
                         OPEN_ACCESS_READONLY  | // try just reading it
                         OPEN_SHARE_DENYNONE,   // OpenMode (allow sharing)
                         0);                    // reserved
         }
         else
         {
            rc = DosOpen((PSZ) file,
                         &dh,                   // file handle
                         &act,                  // action taken
                         0,                     // filesize
                         FILE_NORMAL,           // no attributes
                         OPEN_ACTION_OPEN_IF_EXISTS,
                         OPEN_ACCESS_READONLY  | // try just reading it
                         OPEN_SHARE_DENYNONE,   // OpenMode (allow sharing)
                         0);                    // reserved
         }
         if (rc == NO_ERROR)
         {
            TxPrint("Read-only access  : %s, writing not possible!\n", file);
            si->ReadOnly = TRUE;
         }
      }
      if (rc == NO_ERROR)
      {
         si->SectorIO = FALSE;                  // no sector-IO
      }
      else
      {
         switch (rc)
         {
            case ERROR_OPEN_FAILED:
            case ERROR_INVALID_DRIVE:
            case ERROR_FILE_NOT_FOUND:
            case ERROR_PATH_NOT_FOUND:          // returned on network
               TxPrint("Invalid file specification '%s'.\n", file);
               break;

            default:
               TxPrint("Error %lu on DosOpen '%s'.\n", (ULONG) rc, file);
               break;
         }
      }
   #endif
   if (rc == NO_ERROR)                          // opened successfully
   {
      ULONG            vdiSectors;              // 32bit sector count

      TRACES(("Image file '%s' opened, handle: %lu\n", file, dh));
      strcpy( si->Name, file);
      *fcount = 1;                              // default file count

      //- set correct GEO, will be 'inherrited' by Image-disks
      if (forceRAW)                             // open RAW, even if IMZ/VDO or ...
      {
         rc = DFS_PENDING;                      // open RAW, even if IMZ/VDO or ...
      }
      else if (dfsGetVdiImageInfo( file, dh, NULL, NULL, NULL, NULL, NULL, &vdiSectors, &imzBuf, &imzBps))
      {
         //- Note: Buffer is more than one track, typically 1 Mib (use 64/32)
         si->Type   = DFST_RAWIMAGE;            // store type, will be discarded for PM
         si->Geo.S  = DFS_1MIB_SECT;            // fixed sectors/track
         si->Geo.H  = imzBuf / (si->Geo.S * imzBps); // should divide well :)
         si->Geo.B  = imzBps;
         sectors    = (ULN64) vdiSectors;
         si->ByteSize = sectors * imzBps;
      }
      else if (dfsGetImzImageInfo( file, dh, TRUE, &si->ByteSize, fcount, &imStart, &sectors, &imzBuf, &imzBps, NULL))
      {
         //- Note: Buffer is often exactly one track (created on DOS or OS/2), or it is huge RBUF == 256KiB
         si->Type   = DFST_IMZIMAGE;            // actual type of store
         si->Geo.S  = imzBuf / imzBps;          // buffersize to sectors/track
         if (si->Geo.S > DFS_2TB_SECT)          // huge buffer was used, invalid sectors/track
         {
            if ((sectors & DFS_1MIB_MASK) == 0) // looks like 1MiB multiple, use 1MiB geo
            {
               si->Geo.S = DFS_1MIB_SECT;
            }
            else
            {
               si->Geo.S = DFS_STD_SECT;
            }
         }
         si->Geo.H  = (si->Geo.S == DFS_1MIB_SECT) ? DFS_1MIB_HEAD : DFS_STD_HEAD;
         si->Geo.B  = imzBps;
         rc = dfsImzInitCacheIndex( dh, file, verbose, &si->accessInfo); // IM as well as PM-disk
         TRACES(("IMZ '%s', got info at: 0x%8.8lx\n", file, si->accessInfo));
      }
      else
      {
         rc = DFS_PENDING;                      // use RAW for all unrecognized ones
      }
      if (rc == DFS_PENDING)
      {
         si->Type   = DFST_RAWIMAGE;            // actual type of store
         si->Geo.S  = DFS_STD_SECT;
         si->Geo.H  = DFS_STD_HEAD;
         si->Geo.B  = SECTORSIZE;
         TxHandle2FileSize( dh, &si->ByteSize); // get exact size for RAW image
         sectors    = (si->ByteSize + si->Geo.B - 1) / si->Geo.B; // allow partial last sector!
         TxFileSeek( dh, 0, SEEK_SET);          // make sure we are at start of file
         rc = NO_ERROR;                         // and all is well ...
      }
      if (verbose)
      {
         dfsSz64Bps( "Size opened image : ", sectors, SECTORSIZE,  "");
         dfsUllDot20(", total file size : ", si->ByteSize, " bytes\n");
      }

      if (sectors == 0)
      {
         si->Geo.C  = 0;                        // minimal size
      }
      else
      {
         si->Geo.C  = ((sectors -1) / (si->Geo.H * si->Geo.S)) +1;
      }
      si->DiskId    = DFS_IMGDISK;              // disknr for img Geo
      si->is.object = DFSO_IMGF;                // Image file
      si->vfHandle  = dh;                       // file handle
      si->Sectors   = sectors;                  // size of image
      dfstGeo2Sys( si);                         // sync Original Geo

      if (append == FALSE)                      // no append allowed
      {
         SINF->stFlags |= DFSTFL_EXPLICITSIZE;
      }
      //- to be refined, get better GEO from contents, geo sniffing ...

      dfstDiskGeometry(  store, 0, 0, 0, 0, 0, FALSE, verbose);
      dfstLogicalLimits( store, imStart, ((sectors) ? sectors -1 : 0) + imStart);
   }
   RETURN (rc);
}                                               // end 'dfstOpenImageFile'
/*---------------------------------------------------------------------------*/
#endif
#endif

/*****************************************************************************/
// Open a partitionable medium: phys/virt/image using DFSee disk-id range 1..n
/*****************************************************************************/
ULONG dfstOpenDisk
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   USHORT              diskid,                  // IN    DFSee disk id
   BOOL                verbose,                 // IN    show disk geo
   BOOL                reset                    // IN    reset disk geo
)
{
   ULONG               rc = NO_ERROR;
   USHORT              disknr = dfsDid2RealDiskNr(diskid);
   DFSTOREINFO        *si     = &(sti[store]);
   BYTE               *mbr    = NULL;
   DFSGEODISK         *dskgeo = dfsGetDiskGeoInfo( diskid);
   DFSMEDIATYPE        mtype  = dfsDid2DiskType(diskid);

   #if defined (WIN32)
      TXTM             dspec;
   #elif defined (DOS32)
      union  REGS      regs;
      struct SREGS     sreg;
   #elif defined (UNIX)
      char            *devname = dfsDid2DeviceName(diskid);
      ULONG            bpsec   = 0;             // bytes per sector/block
      #if defined (LINUX)
         LINUX_HD_GEO  lgeo;
         ULONG         totalSize = 0;           // nr of sectors on the disk
         ULN64         geoSize;
      #endif
   #else
      TXTM             dspec;
      USHORT           dpdHandle = 0;           // USHORT handle value
      ULONG            DataLen;
      ULONG            ParmLen;
      DFS_DPBLOCK      DevParam;                // device parameter block
   #endif

   ENTER();

   TRARGS(("Reset:%s Verbose:%s disk:%hu disknr:%hu store: %lu DiskId: %hu bps:%hu size:0x%llx\n",
           (reset) ? "YES" : "NO ", (verbose) ? "YES" : "NO ",
            diskid, disknr, store, si->DiskId, si->Geo.B, si->Sectors));
   if ((si->DiskId != diskid) || (diskid == 0) || reset) // other disk or reset needed
   {
      dfstClose( store);
      switch (mtype)
      {
#if !defined (DFS_PHYSDISK_ONLY)
         case DFSD_VIRT:
            si->is.object = DFSO_DISK;          // partitionable medium
            si->Type      = DFST_MEMORY;        // actual type of store
            si->LockIgn   = TRUE;               // Do not prompt when writing
            TRACES(("Create link from store %lu to disk %hu = VD%hu\n", store, diskid, disknr));

            rc = dfsGetMdiskGeometry( disknr, &si->Sectors, &si->Geo.C, &si->Geo.H, &si->Geo.S, &si->Geo.B);
            if (rc == NO_ERROR)
            {
               dfstGeo2Sys( si);                // sync Original Geo with new
               TRACES(("Memory disk %hu opened for store %lu\n", diskid, store));
            }
            break;

         case DFSD_IRAW:                        // RAW disk image
         case DFSD_IMZD:                        // Compressed disk image, IMZ
         case DFSD_VBOX:                        // VirtualBox disk image, VDI
            si->is.object = DFSO_DISK;          // partitionable medium
            si->Type      = (mtype == DFSD_IRAW) ? DFST_PMIRAW : DFST_PMIMZD;
            si->LockIgn   = TRUE;               // Do not prompt when writing
            si->Geo.B     = SECTORSIZE;         // to be refined!
            switch (mtype)
            {
               default:
               case DFSD_IRAW: si->Type = DFST_PMIRAW; break;
               case DFSD_IMZD: si->Type = DFST_PMIMZD; break;
               case DFSD_VBOX: si->Type = DFST_PMVBOX; break;
            }
            TRACES(("Create link from store %lu to disk %hu = IM%hu\n", store, diskid, disknr));

            rc = dfsGetImageDiskInfo( disknr, si->Name,
                                      &si->vfHandle, &si->accessInfo,
                                      &si->Geo.C, &si->Geo.H, &si->Geo.S);
            if (rc == NO_ERROR)
            {
               if (mtype == DFSD_IMZD)          // IMZ can NOT be written too (for now)
               {
                  si->ReadOnly = TRUE;
               }
               dfstGeo2Sys( si);                // sync Original Geo with new
               TRACES(("Image disk %hu opened for store %lu\n", diskid, store));
            }
            break;
#endif
         case DFSD_PHYS:
            si->is.object = DFSO_DISK;          // partitionable medium
            si->Type      = DFST_PHDISK;        // actual type of store
            si->di    = &(PhysDisk[ disknr]);   // disk handle & useCount
            TRARGS(("Store:%lu disk:%hu  use:%lu  handle:%8.8lx\n",
                     store, si->DiskId, DI->useCount, DI->pdHandle));
            TRACES(("Create link from store %lu to disk %hu = PD%hu\n", store, diskid, disknr));
            #if defined (WIN32)
               if (DI->useCount == 0)           // first open
               {
                  sprintf( dspec, "\\\\.\\PHYSICALDRIVE%u", disknr -1);
                  DI->pdHandle = CreateFile( dspec,
                                           GENERIC_READ    | GENERIC_WRITE,
                                           FILE_SHARE_READ | FILE_SHARE_WRITE,
                                           NULL,   // default security info
                                           OPEN_EXISTING,
                                           FILE_ATTRIBUTE_NORMAL,
                                           NULL);
               }
               if (DI->pdHandle != INVALID_HANDLE_VALUE)
               {
                  DISK_GEOMETRY_EX *dgex = (DISK_GEOMETRY_EX *) rbuf;  //- use large buffer
                  DISK_GEOMETRY geo;                                   //- opened disk geometry
                  ULONG         received;                              //- nr of bytes received

                  if (TxOsVersion( NULL) >= TXVER_WINNT_WINXP)         //- has _EX ioctl
                  {
                     if (DeviceIoControl(DI->pdHandle,
                                         IOCTL_DISK_GET_DRIVE_GEOMETRY_EX,
                                         NULL, 0, // no input needed
                                         dgex, (RBUFBYTES),
                                         &received, NULL))
                     {
                        TRACES(("Received %lu bytes from GEOMETRY_EX, size: 0x%llx\n",
                                 received, dgex->DiskSize.QuadPart));
                        TRHEXS( 500, dgex, 128, "GEOMETRY_EX");

                        si->Geo.B   = (USHORT) dgex->Geometry.BytesPerSector;
                        si->Geo.H   = (ULONG)  dgex->Geometry.TracksPerCylinder;
                        si->Geo.S   = (ULONG)  dgex->Geometry.SectorsPerTrack;
                        si->Geo.C   = (ULONG)  dgex->Geometry.Cylinders.QuadPart;
                        si->Sectors = (ULN64)  dgex->DiskSize.QuadPart / si->Geo.B;
                        dfstGeo2Sys( si);       // sync Original Geo with new
                     }
                     else
                     {
                        TRACES(("GEOMETRY_EX, error: '%s' on disk: %hu\n", txNtLastError(), disknr));
                        rc = DFS_PENDING;
                     }
                  }
                  else                          // older versions have no _EX at all
                  {
                     rc = DFS_PENDING;
                  }
                  if (rc == DFS_PENDING)
                  {
                     if (DeviceIoControl(DI->pdHandle,
                                         IOCTL_DISK_GET_DRIVE_GEOMETRY,
                                         NULL, 0, // no input needed
                                         &geo, sizeof(geo),
                                         &received, NULL))
                     {
                        si->Geo.B = (USHORT) geo.BytesPerSector;
                        si->Geo.H = (ULONG)  geo.TracksPerCylinder;
                        si->Geo.S = (ULONG)  geo.SectorsPerTrack;
                        si->Geo.C = (ULONG)  geo.Cylinders.QuadPart;
                     }
                     else                       // probably a reserved removable (USB)
                     {
                        si->Geo.B = SECTORSIZE;
                        si->Geo.C = DFS_STD_CYLS;
                        si->Geo.H = DFS_STD_HEAD;
                        si->Geo.S = DFS_STD_SECT;
                        dfstGeo2Sys( si);       // sync Original Geo with new
                        TRACES(("GEOMETRY,    error: '%s' on disk: %hu\n", txNtLastError(), disknr));
                     }
                     rc = NO_ERROR;             // for now, may fail on MBR read
                  }
                  if ((si->Geo.B == 0) || (si->Geo.B > 4096))
                  {
                     if (verbose)
                     {
                        TxPrint("\n Disk %2hu  WARNING : Reported bytes/sector value %s%hu%s is non-standard\n"
                                    "                    and can cause interpretation and other problems.\n"
                                    "                    Standard sectorsize 512 bytes will be used instead!\n"
                                    "                    Use the 'geo -bps:%hu' command to set this to the\n"
                                    "                    detected value if you are sure it is correct.\n",
                                                         diskid, CBR, si->Geo.B, CNN, si->Geo.B);
                     }
                     si->Geo.B = SECTORSIZE;
                  }
               }
               else
               {
                  TRACES(( "CreateFile error on '%s': %s\n", dspec, txNtLastError()));
                  rc = DFS_NO_DEVICE;           // to be refined
               }
            #elif defined (DOS32)
               memset( &regs, 0, sizeof(regs));
               regs.h.ah = DFS_I13_S_PARAM;     // Get Current Drive params
               regs.h.dl = (BYTE) (0x7f + disknr);
               TRACES(( "Get drive params for disk: 0x%hx\n", regs.h.dl));
               TRHEXS( 500, &regs, sizeof(regs), "INT13_REGS");
               int386( DFS_I13, &regs, &regs);  // Execute BIOS int-13
               TRACES(( "Get drive params C-flag  : 0x%hx\n", regs.x.cflag));
               if ((regs.x.cflag == 0) || (dfsa->win9x)) // drive exists
               {
                  DI->pdHandle = (ULONG) (0x7f + disknr); // faked nr for Int-13
                  si->Geo.B  = SECTORSIZE;      // to be refined (DOS f 36 ?)
                  si->Geo.H  = regs.h.dh + 1;
                  si->Geo.S  = DFSC2SECTOR(TXWORD.cx);
                  si->Geo.C  = DFSC2CYLIND(TXWORD.cx) +1;
                  TRACES(("CHS: %lu %lu %lu\n", si->Geo.C, si->Geo.H, si->Geo.S));

                  dfstGeo2Sys( si);             // sync Original Geo with new

                  if (verbose && dfsa->warnLevel > 1)
                  {
                     TxPrint( "\nNormal INT13 open : disk %hu = PD%hu, geo: %hu %hu %hu",
                                 diskid, disknr, si->Geo.C, si->Geo.H, si->Geo.S);
                  }
                  if ((TxaExeSwitch('I') || !TxaExeSwitchSet('I')) && // no -I-
                      (dfsInt13Ext( disknr, verbose)) ) // and ext-I13
                  {
                     ULONG  eCylinders = 1;     // recalculated nr of cylinders
                     DFS_INT13_EDD   *pi13;     // Int 13 extended drive data

                     pi13 = (DFS_INT13_EDD *) xpar;

                     memset(pi13, 0, sizeof(DFS_INT13_EDD));
                     pi13->size   =  sizeof(DFS_INT13_EDD);

                     memset( &sreg,  0, sizeof(sreg));
                     memset( &regs,  0, sizeof(regs));

                     memset( &txdx_rmi, 0, sizeof(txdx_rmi));
                     txdx_rmi.eax = DFS_I13_E_PARAM << 8;
                     txdx_rmi.edx = 0x7f + disknr;
                     txdx_rmi.ds  = txDpmiSegment(pi13); // params (ds:si)

                     TRHEXS( 500,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

                     regs.w.ax    = TXDX_DPMI_RMINT; // simulate realmode INT
                     regs.h.bl    = DFS_I13;
                     regs.x.edi   = FP_OFF( &txdx_rmi);
                     sreg.es      = FP_SEG( &txdx_rmi);

                     txDpmiCall( &regs, &sreg);

                     //- use total-sectors and (logical) head+sect count
                     //- to re-calculate the logical number of cylinders

                     if (regs.x.cflag == 0)
                     {
                        TRHEXS( 500, pi13, sizeof(DFS_INT13_EDD), "DFS_INT13_EDD");
                        TRACES(("flag: %4.4hx C:%lu H:%lu S:%lu total:0x0%llx\n",
                           pi13->flags, pi13->c, pi13->h, pi13->s, pi13->i13Sectors));

                        si->Sectors = pi13->i13Sectors;

                        if ((si->Geo.H) && (si->Geo.S))
                        {
                           //- to be refined, might use returned C/H/S values from
                           //- the call to determine #sectors if sectors.lo == 0
                           eCylinders = (ULONG) (pi13->i13Sectors / (si->Geo.H * si->Geo.S));
                        }
                        //- to be refined, issue WARNING if si->Geo.H and .S is
                        //- not equal to the pi13-> values (XBIOS != BIOS)
                        si->Geo.B = pi13->bps;
                        if ((si->Geo.B == 0) || (si->Geo.B > 4096))
                        {
                           if (verbose)
                           {
                              TxPrint("\n Disk %2hu  WARNING : Reported bytes/sector value %s%hu%s is non-standard\n"
                                          "                    and can cause interpretation and other problems.\n"
                                          "                    Standard sectorsize 512 bytes will be used instead!\n"
                                          "                    Use the 'geo -bps:%hu' command to set this to the\n"
                                          "                    detected value if you are sure it is correct.\n",
                                                               diskid, CBR, si->Geo.B, CNN, si->Geo.B);
                           }
                           si->Geo.B = SECTORSIZE;
                        }
                        if ((!TxaExeSwitch('I')) && // not forced to use std I13
                            (eCylinders >= 1024)) // and indeed more than 1024
                        {
                           si->Geo.C      = eCylinders;
                           si->Extended13 = TRUE; // signal extended Int13
                                                // read and write possible
                           dfstGeo2Sys( si);    // sync Original Geo with new
                        }
#ifndef OEMSB
                        if (verbose)
                        {
                           TXLN            geo;
                           ULN64           size = pi13->i13Sectors;

                           TxPrint(   "\nExtended Int-13   : Present");
                           if (si->Extended13)  // use extentions ...
                           {
                              TxPrint(", will be used on each cylinder >= 1024)");
                           }
                           else
                           {
                              TxPrint(", will not be used    (<1024 or -I used)");
                           }
                           if (size == 0)       // invalid nr of sectors
                           {
                              size = (ULN64) pi13->c * pi13->h * pi13->s;
                           }
                           sprintf( geo, "\n BIOS-Geo D%s%2u%s Cyl :%8lu H:%3lu S:%-3lu Bps:%-4lu  ",
                                    CBC, diskid, CNN, pi13->c, pi13->h, pi13->s, si->Geo.B);
                           TxPrint( dfstrSz64Bps( geo, "Size:", size, si->Geo.B, ""));
                        }
#endif
                     }
                  }
                  else if (verbose)
                  {
                     TxPrint( "\nDisk %hu Extended INT13 reported unavailable", disknr);
                  }
               }
               else
               {
                  TxPrint( "\nDisk %hu INT13-get-param Failure status: %2.2hx", disknr, regs.h.ah);
                  rc = DFS_NO_DEVICE;           // to be refined
               }
            #elif defined (UNIX)
               if (DI->useCount == 0)           // first open
               {
                  if ((DI->pdHandle = open( devname, O_RDWR | O_LARGEFILE)) == -1)
                  {
                     #if !defined (OEM_BRANDED)
                        if (verbose)
                        {
                           TxPrint("Read-write access : error '%s'!\n", strerror(errno));
                        }
                     #endif
                     if ((DI->pdHandle = open( devname, O_RDONLY | O_LARGEFILE)) == -1)
                     {
                        DI->pdHandle = PDHNULL;
                        rc = TxRcFromErrno( errno);
                     }
                     else
                     {
                        #if !defined (OEM_BRANDED)
                           if (verbose)
                           {
                              TxPrint("Read-only access  : disk %hu, writing not possible!\n", diskid);
                           }
                        #endif
                        si->ReadOnly = TRUE;
                     }
                  }
               }
               TRACES(("DI-useCount: %lu, DI-pdHandle: %8.8lx\n", DI->useCount, DI->pdHandle));

               //- to be refined, geometry sniffing, or other API on DARWIN ??
               if (rc == NO_ERROR)              // determine geometry etc
               {
                  #if defined (DARWIN)
                     u_int64_t  blocks;

                     if (ioctl( DI->pdHandle, DKIOCGETBLOCKCOUNT, &blocks) == 0)
                     {
                        if ((ioctl( DI->pdHandle, DKIOCGETBLOCKSIZE, &bpsec) == 0) &&
                            (bpsec != 0) && (bpsec <= 4096))
                        {
                           si->Geo.B = (USHORT) bpsec; // use reported value
                        }
                        else
                        {
                           if (verbose)
                           {
                              TxPrint( " Disk %2hu  WARNING : Reported bytes/sector value %s%lu%s is non-standard\n"
                                         "                    and can cause interpretation and other problems.\n"
                                         "                    Standard sectorsize 512 bytes will be used instead!\n"
                                         "                    Use the 'geo -bps:%lu' command to set this to the\n"
                                         "                    detected value if you are sure it is correct.\n",
                                                              diskid, CBR, bpsec, CNN, bpsec);
                           }
                           si->Geo.B = SECTORSIZE; // fixed value!
                        }
                        si->Sectors = blocks;
                        si->Geo.H   = DFS_1MIB_HEAD / (si->Geo.B / SECTORSIZE); //- Corrected for non 512 byte
                        si->Geo.S   = DFS_1MIB_SECT;
                        si->Geo.C   = si->Sectors / (si->Geo.H * si->Geo.S);    //- real 1 MiB cylinders
                        dfstGeo2Sys( si);       // sync Original Geo with new
                     }
                     else
                     {
                        TRACES(( "Error getting SIZE using ioctl on fd:%8.8lx\n", DI->pdHandle));
                     }
                  #else
                     LLONG  seekOffset = 0;

                     if (ioctl( DI->pdHandle, HDIO_GETGEO, &lgeo) == 0)
                     {
                        //- Note: ioctl might not be reliable, it does NOT work
                        //- correctly on Slackware-7, giving 4096 reported 1024
                        //- 20160514 JvW: allow non-512 again, seems OK now

                        if ((ioctl( DI->pdHandle, BLKSSZGET, &bpsec) == 0) &&
                            (bpsec != 0) && (bpsec <= 4096))
                        {
                           si->Geo.B = (USHORT) bpsec; // use reported value
                        }
                        else
                        {
                           if (verbose)
                           {
                              TxPrint("\n Disk %2hu  WARNING : Reported bytes/sector value %s%lu%s is non-standard\n"
                                          "                    and can cause interpretation and other problems.\n"
                                          "                    Standard sectorsize 512 bytes will be used instead!\n"
                                          "                    Use the 'geo -bps:%lu' command to set this to the\n"
                                          "                    detected value if you are sure it is correct.\n",
                                                               diskid, CBR, bpsec, CNN, bpsec);
                           }
                           si->Geo.B = SECTORSIZE; // fixed value!
                        }
                        si->Geo.H = (ULONG)  lgeo.heads;
                        si->Geo.S = (ULONG)  lgeo.sectors;
                        si->Geo.C = (ULONG)  lgeo.cylinders;

                        //- Seek to END sets position to exact SIZE of the device (in bytes)
                        //- Would be same value as BLKGETSIZE64, but that is defined inconsistently (ioctl nr)
                        if (( _llseek( DI->pdHandle, 0, 0, &seekOffset, SEEK_END) == -1) || (seekOffset <= 0))
                        {
                           TRACES(("llseek END failure, returned offset: %lld\n", seekOffset));
                           //- use BLKGETSIZE as fallback (max 2Tb, will truncate beyond that!)
                           if (ioctl( DI->pdHandle, BLKGETSIZE, &totalSize) != 0)
                           {
                              TRACES(( "BLKGETSIZE ioctl handle %lu: %s\n", DI->pdHandle, strerror(errno)));
                              totalSize = 0;
                           }
                           else                 // convert from fixed 512 multiples to actual sectorsize
                           {
                              TRACES(( "BLKGETSIZE ioctl handle %lu: Truncated 512b blocks: 0x%lx\n", DI->pdHandle, totalSize));
                              si->Sectors = (ULN64) totalSize / (si->Geo.B / SECTORSIZE);
                           }
                        }
                        else                    // calculate totalSize from seek offset
                        {
                           si->Sectors = (ULN64) (seekOffset / si->Geo.B);
                           si->Geo.C   = si->Sectors / (si->Geo.H * si->Geo.S); // calculate C but keep H/S from HDIO_GETGEO
                           dfstGeo2Sys( si);    // sync Original Geo with new
                           TRACES(("seekOffset: 0x%llx  bpsec:%lu  sectors: 0x%llx\n", seekOffset, si->Geo.B, si->Sectors));
                        }
                        _llseek( DI->pdHandle, 0, 0, &seekOffset, SEEK_SET); // back to first sector

                        if (si->Sectors == 0)     // detect / pass ioctl error
                        {
                           si->Sectors = totalSize;

                           // Calculate margin, almost 1 cyl above geo size
                           geoSize  = (ULN64) si->Geo.C * si->Geo.H * si->Geo.S;
                           TRACES(("Size HDIO_GETGEO: 0x%lx,  Calculated: 0x%llx\n", totalSize, geoSize));
                           if ((si->Sectors > (geoSize + (si->Geo.H * si->Geo.S) -1)))
                           {
                              si->Geo.C = si->Sectors / (si->Geo.H * si->Geo.S);
                              if (verbose)
                              {
                                 TxPrint( "\n Disk %2hu  Warning : Geo size 0x0%llx smaller than size in sectors 0x0%llx,\n"
                                              "                    Cylinder count recalculated and set to %lu.",
                                                                   diskid, geoSize, si->Sectors, si->Geo.C);
                                 TRACES(("\n"));
                              }
                           }
                           TRACES(( "Forced GEO on ioctl: C:%lu H:%lu S:%lu bps:%lu\n",
                                     dskgeo->Forced.C, dskgeo->Forced.H, dskgeo->Forced.S, dskgeo->Forced.B));
                        }
                        dfstGeo2Sys( si);       // sync Original Geo with new
                     }
                     else                       // error getting geo
                     {
                        TRACES(( "HDIO_GETGEO ioctl handle %lu: %s\n", DI->pdHandle, strerror(errno)));
                     }
                  #endif
               }
            #else                               // OS/2
               if (DI->useCount == 0)           // first open
               {
                  sprintf( dspec, "%u:", disknr);
                  ParmLen = strlen( dspec)+1;
                  DataLen = 2;
                  rc = DosPhysicalDisk(INFO_GET_HANDLE,
                                       &dpdHandle,
                                       DataLen,
                                       (void *) dspec,
                                       ParmLen);
                  if (rc == NO_ERROR)
                  {
                     DI->pdHandle = (ULONG) dpdHandle;
                  }
                  else
                  {
                  }
               }
               if (rc == NO_ERROR)
               {
                  dspec[0]  = '\0';
                  ParmLen   = 1;
                  DataLen   = sizeof( DevParam);

                  rc = DosDevIOCtl(DI->pdHandle,
                                   IOCTL_PHYSICALDISK,
                                   PDSK_GETPHYSDEVICEPARAMS,
                                   dspec,     ParmLen, &ParmLen,
                                   &DevParam, DataLen, &DataLen);
                  if (rc == NO_ERROR)
                  {
                     ULONG          dr = 0;
                     OEMHLPDISKINFO param;      // QUERYDISKINFO data
                     ULONG          act;        // action taken
                     TXHFILE        oemh;

                     dr = DosOpen((PSZ) "OEMHLP$",
                                  &oemh,        // file handle
                                  &act,         // action taken
                                  0,            // filesize and attributes
                                  FILE_NORMAL,  // no attributes
                                  OPEN_ACTION_OPEN_IF_EXISTS,
                                  OPEN_ACCESS_READONLY |
                                  OPEN_SHARE_DENYNONE, // OpenMode (allow sharing)
                                  0);           // reserved
                     if (dr == NO_ERROR)
                     {
                        memset( &param, 0, sizeof(param));
                        dspec[0]  = (char) (0x7f + disknr);
                        ParmLen   = 1;
                        DataLen   = sizeof( param);
                        if ((dr  = DosDevIOCtl( oemh,
                                         IOCTL_OEMHLP,
                                         OEMHLP_QUERYDISKINFO,
                                         dspec,  ParmLen, &ParmLen,
                                         &param, DataLen, &DataLen)) == NO_ERROR)
                        {
                           TXLN  geo;
                           ULN64 bSize = param.ODI_PhysSectors_low + ((ULN64) param.ODI_PhysSectors_high << 32);
                           ULONG cSC   = param.ODI_Heads * param.ODI_SectorsPerTrack;

                           if (bSize == 0)       // invalid nr of sectors
                           {
                              bSize = (ULN64) param.ODI_Cylinders * cSC; // use calculated value
                           }
                           //- Check against native reported size to catch disk order
                           //- problems with BIOS versus OS (setting boot order etc)
                           //- and to show size of disks larger than 2TB (64-bit size)
                           //- Restrict BIOS-Geo size difference to a single cylinder
                           //- and native cylindercount NOT being 65535 (OS/2 limit)
                           if (rc == NO_ERROR) // have native info
                           {
                              ULONG  nativeCylS = ((ULONG) DevParam.cHeads * (ULONG) DevParam.cSectors);
                              ULN64  nativeSize = ((ULN64) DevParam.cCylinders * nativeCylS);

                              if ((bSize < (nativeSize + nativeCylS)) &&
                                  (nativeSize < (bSize + nativeCylS))  )
                              {
                                 si->Sectors = bSize; // use exact reported nr of sectors
                              }
#ifndef OEMSB
                              else if ((verbose) && (bSize > nativeCylS)) // exclude DUMMY (USB) disks
                              {
                                 TxPrint(    "\nDisk %hu ", disknr);
                                 dfsSz64Bps( "BIOS size ",  bSize,      param.ODI_SectorSize, " ");
                                 dfsSz64Bps( "versus OS: ", nativeSize, SECTORSIZE, (DevParam.cCylinders == 0xffff) ?
                                                                                ", likely 65535 OS/2 cyl. limit!\n" :
                                                                                ", inconsistent disk-ordering ?\n");
                              }
#endif
                           }
#ifndef OEMSB
                           if (verbose)
                           {
                              if (bSize != 0)    // don't show dummy disks (USB)
                              {
                                 sprintf( geo, "\n BIOS-Geo D%s%2u%s Cyl :%8lu H:%3lu S:%-3lu Bps:%-4lu  Size:", CBC, diskid, CNN,
                                          param.ODI_Cylinders, param.ODI_Heads, param.ODI_SectorsPerTrack, SECTORSIZE);
                                 dfsSz64( geo, bSize, "");

                                 #ifndef OEMSB // check for 'sane' head/sectors sizes to avoid bogus error on weird (mem) disks
                                 if ((bSize > ((ULN64) 65535 * DevParam.cHeads * DevParam.cSectors)) &&
                                     (DevParam.cHeads >= 64)  && (DevParam.cSectors >= 32) &&  (DevParam.cSectors <= 63))
                                 {
                                    ULONG       os2Limit = dfs16bitCylLimitGiB( DevParam.cHeads, DevParam.cSectors);

                                    TxPrint("\n Disk %2hu     INFO : Size exceeds the normal OS/2 65535 cylinder = %lu GiB limit!\n"
                                             "%20.20sRequires 127 or 255 sect/track in L-Geo (DANIS506/AHCI ADD)\n"
                                             "%20.20sto use the full disk size, or limits OS/2 to first %lu GiB.\n",
                                              diskid, os2Limit, "", "", os2Limit);
                                 }
                                 #endif
                              }
                              if (dfsa->warnLevel > 1) // additional info
                              {
                                 TxPrint("\n  ODI sectors low : 0x%8.8lx = %lu\n",
                                              param.ODI_PhysSectors_low, param.ODI_PhysSectors_low);

                                 TxPrint( "  ODI sectors high: 0x%8.8lx bps: 0x%4.4hx flags: 0x%4.4hx\n",
                                             param.ODI_PhysSectors_high, param.ODI_SectorSize, param.ODI_Flags);

                                 TxPrint( "  ODI I/O port nr : 0x%4.4hx    Ctrl: 0x%4.4hx HR-UN: 0x%02.2hx\n",
                                             param.ODI_IOPort, param.ODI_ControlPort, param.ODI_HeadRegisterUN);

                                 TxPrint( "  ODI DMA-info    : 0x%2.2hx      PIO : 0x%2.2hx   IRQ  : 0x%02.2hx = %hu\n",
                                                      (USHORT) param.ODI_DMAInfo, (USHORT) param.ODI_PIOInfo,
                                                      (USHORT) param.ODI_IRQ, (USHORT) param.ODI_IRQ);
                              }
                           }
#endif
                        }
                        DosClose( oemh);
                     }
                  }
               }
               switch (rc)
               {
                  case NO_ERROR:
                     //- si->Sectors will be calculated from CHS later (stays 0 now)
                     si->Geo.C = (ULONG) DevParam.cCylinders;
                     si->Geo.H = (ULONG) DevParam.cHeads;
                     si->Geo.S = (ULONG) DevParam.cSectors;
                     si->Geo.B = SECTORSIZE;    // to be refined, unknown ...

                     dfstGeo2Sys( si);          // sync Original Geo with new

                     si->tlSize = ((DevParam.cSectors -1) * 2 *
                                    sizeof(USHORT)) + sizeof(TRACKLAYOUT);

                     if ((si->TrackLayout = TxAlloc( 1, si->tlSize)) != NULL)
                     {
                        USHORT      i;

                        TRACES(("tlSize:%lu for %hu sectors, bps:%hu\n",
                                 si->tlSize, si->Geo.S, si->Geo.B));
                        for (i = 0; i < si->Geo.S; i++)
                        {
                           si->TrackLayout->TrackTable[i].usSectorNumber = i +1;
                           si->TrackLayout->TrackTable[i].usSectorSize   = si->Geo.B;
                        }
                     }
                     else
                     {
                        rc = DFS_ALLOC_ERROR;
                     }
                     break;

                  case ERROR_NOT_READY:
                     TxPrint("\nPhysical disk %u = PD%hu is not ready (removable ?)\n",
                                diskid, disknr);
                     break;

                  case ERROR_ACCESS_DENIED:
                     TxPrint("\nPhysical disk %u = PD%hu is not accessible (locked)\n",
                                diskid, disknr);
                     break;

                  default:
                     TxPrint("\nPhysical disk %u = PD%hu not opened, rc = %ld\n",
                                diskid, disknr, (LONG) rc);
                     break;
               }
            #endif
            break;

         default:                               // invalid disk-number or an
            rc = DFS_NO_DEVICE;                 // unsupported type 'disk'
            break;
      }
      if (rc == NO_ERROR)                       // Force & report geometry
      {
         if (((si->Sectors == DFS_MAX_PSN) || (si->Sectors == 0)) && (si->Geo.C != 0))
         {
            si->Sectors  = si->Geo.C * si->Geo.H * si->Geo.S;
            TRACES(("Calculated nr of sectors 0x%llx from C/H/S\n", si->Sectors));
         }
         si->DiskId   = diskid;

         TRACES(("si-> geometry: C:%6lu  H:%3lu  S:%2lu  bps:%lu, size:0x%llx\n",
                  si->Geo.C, si->Geo.H, si->Geo.S, si->Geo.B, si->Sectors));

         if (si->di != NULL)                    // phys disk
         {
            TRACES(("Physical disk %hu = PD%hu opened for store %lu, handle %8.8lx\n",
                              diskid, disknr, store, (ULONG) DI->pdHandle));
            DI->useCount++;                     // count use of the handle
         }

         TRACES(("Forced geometry now: C:%lu H:%lu S:%lu bps:%hu\n",
                  dskgeo->Forced.C, dskgeo->Forced.H, dskgeo->Forced.S, dskgeo->Forced.B));

         if ((dfsa->geoCalc) &&                                         //- Calc wanted
             (dfstCalculateGeometry( store, (verbose && dfsa->geoCalc), //- Calculated OK
                 &(dskgeo->Calculated)) == NO_ERROR))
         {
            BOOL       forced = ( (dskgeo->Reason[0] != 0) &&
                                 ((dskgeo->Forced.C  != 0) ||
                                  (dskgeo->Forced.H  != 0) ||
                                  (dskgeo->Forced.S  != 0) ||
                                  (dskgeo->Forced.B  != 0) ));

            if ((reset && forced) || (dskgeo->Reason[0] == 0))
            {
               if ((si->Geo.C     != dskgeo->Calculated.C) ||
                   (si->Geo.H     != dskgeo->Calculated.H) ||
                   (si->Geo.S     != dskgeo->Calculated.S) ||
                   (si->Geo.B     != dskgeo->Calculated.B)  )
               {
                  dskgeo->Forced.C = dskgeo->Calculated.C;
                  dskgeo->Forced.H = dskgeo->Calculated.H;
                  dskgeo->Forced.S = dskgeo->Calculated.S;
                  dskgeo->Forced.B = dskgeo->Calculated.B;
                  strcpy( dskgeo->Reason, "");
               }
               else
               {
                  dfsResetForcedGeo( diskid);
               }
            }
         }
         else
         {
            if (reset)                          // reset forced geometry
            {
               dfsResetForcedGeo( diskid);
            }
         }
         dfstDiskGeometry( store, 0, dskgeo->Forced.C, dskgeo->Forced.H,
                                     dskgeo->Forced.S, dskgeo->Forced.B,
                                     FALSE,  verbose);

         if ((dfsa->warnLevel > 1) && ((si->Geo.S > DFS_STD_SECT)
             #if defined (DEV32) // allow 0x7F and 0xFF (DANIS506 > 500 GB)
                       && ((si->Geo.S & DFS_STD_SECT) != DFS_STD_SECT)
             #endif
            ))
         {
            TxPrint( "\n Disk %2hu  WARNING : Sectors/track value %s%hu%s is non-standard and can\n"
                         "                    cause CHS related interpretation problems.\n",
                                              diskid, CBR, si->Geo.S, CNN);
         }

         if ((mbr = TxAlloc( 1, si->Geo.B)) != NULL)
         {
            rc = dfstReadPsn(store, 0, 1, mbr); // catch not-ready errors
            si->is.noAccess = rc;               // keep MBR status for filters
            #if !defined (OEMSB)
            if (verbose)                        // verbose reporting ?
            {
               if (rc == NO_ERROR)
               {
                  if (TxaOption('m'))           // show MBR identification ?
                  {
                     #if !defined (DARWIN)
                        if (dfsa->warnLevel > 1) // unless pedantic mode set,
                        {
                           TxPrint( " BIOS Int13 limit : ");
                           if (si->Geo.C < 1024)
                           {
                              TxPrint( "none, no limit for this disk\n");
                           }
                           else
                           {
                              dfsSizeXiB( "1024, I13X support needed beyond :",
                                          (1024 * si->Geo.H * si->Geo.S -1), si->Geo.B, "\n");
                           }
                        }
                     #endif
                     dfsMbrEbrIdentifyDisplay( mbr, " MBR crc");
                  }
               }
               else
               {
                  TxPrint( "IGNORE disk %hu PD%hu : MBR read failed, "
                           "Reserved (USB) DUMMY, locked or BAD-sectors\n", diskid, disknr);
               }
            }
            #endif
            TxFreeMem( mbr);
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else                                      // re-init to unused/closed
      {
         dfstInitSingleStore( store);
      }
   }
   RETURN (rc);
}                                               // end 'dfstOpenDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Close open store, and re-initialize
/*****************************************************************************/
ULONG dfstClose
(
   DFST_HANDLE         store                    // IN    DFS store to be used
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      ULONG            ParmLen;
   #endif

   ENTER();
   TRARGS(( "Close store %lu, type: %d\n", store, si->Type));

   #if !defined (OEMSB)
   if (si->LockCnt)                             // store lock pending
   {
      si->LockCnt = 1;                          // force final unlock
      rc = dfstLocking( store, DFS_UNLOCK, FALSE, FALSE);
   }
   #endif
   switch (si->Type)
   {
      case DFST_PHDISK:
         if (si->di != NULL)                    // phys disk
         {
            TRARGS(("Store:%lu disk:%hu  use:%lu  handle:%8.8lx\n",
                     store, si->DiskId, DI->useCount, DI->pdHandle));

            if (si->is.partid != 0)
            {
               dfsa->autoPid  = si->is.partid;
               TRACES(("autoPid set to %2.2hu\n", dfsa->autoPid));
            }
            if ((DI->useCount == 1) && (DI->pdHandle != PDHNULL))
            {
               #if   defined (WIN32)
                  rc = TxClose( DI->pdHandle);
               #elif defined (DOS32)
               #elif defined (UNIX)
                  if (close( DI->pdHandle) == -1)
                  {
                     rc = TxRcFromErrno( errno);
                  }
               #else
                  ParmLen=2;
                  rc = DosPhysicalDisk(INFO_FREE_HANDLE,
                                       0,
                                       0,
                                       &DI->pdHandle,
                                       ParmLen);
               #endif
               TRACES(("Physical disk, handle %8.8lx closed\n", (ULONG) DI->pdHandle));
               DI->useCount = 0;
               DI->pdHandle = PDHNULL;
            }
            else if (DI->useCount > 0)
            {
               DI->useCount--;                  // update use count for handle
            }
            TRACES(("Physical disk %hu, useCount: %lu  handle: %8.8lx\n",
                     si->DiskId, DI->useCount, DI->pdHandle));
            TRACES(("Remove link from store %lu to disk %hu\n", store, si->DiskId));
            si->di = NULL;
         }
         break;

      case DFST_VOLUME:
         #if defined (DOS32)
         #else
            TxClose( si->vfHandle);
         #endif
         break;

      case DFST_RAWIMAGE:
         if (si->vfHandle != 0)                 // allow 0 after 'steal' handle
         {
            if ((SINF->stFlags & DFSTFL_EXPLICITSIZE) == 0) // filesize may have changed ?
            {
               rc = TxSetFileSize( si->vfHandle, si->ByteSize);
            }
            TxClose( si->vfHandle);
         }
         break;

#if !defined (OEMSB)
      case DFST_IMZIMAGE:
         if (si->vfHandle != 0)                 // allow 0 after 'steal' handle
         {
            rc = dfsImzFreeCacheIndex( &si->accessInfo); // free cache/index data
            TxClose( si->vfHandle);
         }
         break;
#endif

      case DFST_PMVBOX:
         TRACES(("Partitioned VirtualBox disk Image\n"));
         break;

      case DFST_PMIMZD:
         TRACES(("Partitioned Compressed IMZ disk Image\n"));
         break;

      case DFST_PMIRAW:
         TRACES(("Partitioned RAW Image\n"));
         break;

      case DFST_MEMORY:
         TRACES(("Memory-Disk store\n"));
         break;

      case DFST_UNUSED:
         TRACES(("Store not in use, empty\n"));
         break;

      default:
         TRACES(("Invalid store type!\n"));
         break;
   }
   dfstInitSingleStore( store);                 // reset all values to default
   RETURN (rc);
}                                               // end 'dfstClose'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read sectors using PSN, and count number of sectors actually read
// Will always return PSN_LIMIT when psn is beyond last sector of object now
// (previously allowed reading whole disk, even if just one partition selected)
/*****************************************************************************/
ULONG dfstPsnReadCount
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG              *done,                    // OUT   sectors done, or NULL
   BYTE               *buffer                   // OUT   Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // function return code
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               sc = nrs;                // sector count

   ENTER();
   TRLEVX(200,("Base start: %llx, Final: %llX sectorsize:%hu\n", si->Start, si->Final, si->Geo.B));
   TRARGS(("Store:%lu, read %lu sects from LSN %llx = PSN %llX to: %8.8lX\n", store, nrs, dfstPsn2LSN(store, psn), psn, buffer));

   //- Allow reading before the START but NOT beyond the END of the current object
   //- allows reading LVM, or GRUB references, when a partition is selected
   if (psn <= si->Final)                        // psn itself within limit
   {
      if ((psn + nrs -1) > si->Final)           // last sector beyond limit
      {
         sc = si->Final - psn +1;               // corrected nr of sectors
         TRACES(("Nr of sectors to do restricted to: %lu (base limit)\n", sc));
      }
   }
   else
   {
      rc = DFS_PSN_LIMIT;                       // don't even attempt to read
   }
   if (rc == NO_ERROR)                          // within limits or partitioned
   {
      memset(buffer, DFS_BADSEC_PATTERN,        // recognizable contents on error
             sc   *  si->Geo.B);                // makes spotting bad-sectors easy

      switch (si->Type)
      {
         case DFST_PHDISK:
            if (si->di != NULL)
            {
               TRLEVX(200,( "pdhandle: %8.8lx\n", DI->pdHandle));
               #if defined (WIN32)
                  {
                     ULONG      dummy;          // new lsn
                     int        ra;             // read attempts

                     rc = dfstSeekSector( si, DI->pdHandle, psn);
                     if (rc == NO_ERROR)
                     {
                        for ( ra = 0,                      rc  = ERROR_READ_FAULT;
                             (ra < dfsa->readAttempts) && (rc != NO_ERROR);
                              ra++)
                        {
                           rc = TxRead( DI->pdHandle, buffer, sc * si->Geo.B, &dummy);
                           if (rc != NO_ERROR)
                           {
                              si->Retries++;    // total retry statistics
                              TRACES(( "ReadFile error: '%s', attempt %u of %lu, total retries %lu\n",
                                        txNtLastError(), ra+1, dfsa->readAttempts, si->Retries));
                           }
                        }
                     }
                  }
               #elif defined (UNIX)
                  {
                     int        ra;             // read attempts

                     rc = dfstSeekSector( si, DI->pdHandle, psn);
                     if (rc == NO_ERROR)
                     {
                        for ( ra = 0,                      rc  = ERROR_READ_FAULT;
                             (ra < dfsa->readAttempts) && (rc != NO_ERROR);
                              ra++)
                        {
                           if (read( DI->pdHandle, buffer, sc * si->Geo.B) != -1)
                           {
                              rc = NO_ERROR;
                           }
                           else                 // retry useful ?
                           {
                              TRACES(( "read() errno: %d\n", errno));
                              if ((errno != EIO) && (errno != EINTR))
                              {
                                 break;         // no retrying on other errors
                              }
                              else
                              {
                                 //- to be refined, should the Seek be repeated
                                 //- too to account for partial reads ? (EINTR)

                                 si->Retries++; // total retry statistics
                              }
                           }
                        }
                     }
                  }
               #elif defined (DOS32)
                  {
                     rc = dfstReadSectors( store, psn, sc, buffer);
                  }
               #else
                  {
                     ULONG   chunkSectors;
                     ULONG   chunkOffset  = 0;  // start PSN for chunk
                     ULONG   sectorsTodo  = sc; // number of sectors left to do

                     while ((sectorsTodo > 0) && (rc == NO_ERROR)) // exit when PSN limit!
                     {
                        if (sectorsTodo > DFS_OS2_BUFSIZE_LIMIT)
                        {
                           chunkSectors = DFS_OS2_BUFSIZE_LIMIT;
                           sectorsTodo -= DFS_OS2_BUFSIZE_LIMIT;
                        }
                        else
                        {
                           chunkSectors = sectorsTodo;
                           sectorsTodo  = 0;
                        }
                        rc = dfstReadSectors( store, psn +  chunkOffset,  chunkSectors,
                                                  buffer + (chunkOffset * si->Geo.B));
                        chunkOffset += chunkSectors;
                     }
                  }
               #endif
            }
            else
            {
               rc = DFS_NO_DEVICE;
            }
            break;

#if !defined (DFS_PHYSDISK_ONLY)
         case DFST_VOLUME:
         case DFST_PMIRAW:
         case DFST_RAWIMAGE:
            #if defined (DOS32)                 // one sector at a time
               {
                  ULONG i;

                  for (i = 0; i < sc && rc == NO_ERROR; i++)
                  {
                     rc = dfstReadLog( store, psn +i, 1, buffer + (i * si->Geo.B));
                  }
               }
            #else
               rc = dfstReadLog( store, psn, sc, buffer);
            #endif
            break;

         case DFST_PMIMZD:
         case DFST_IMZIMAGE:
            {
               ULONG   i;

               for (i = 0; i < sc && rc == NO_ERROR; i++)
               {
                  rc = dfsImzReadSector( psn    +  i,  si->Geo.B, si->accessInfo,
                                         buffer + (i * si->Geo.B));
               }
            }
            break;

         case DFST_PMVBOX:
            {
               ULONG   i;

               for (i = 0; i < sc && rc == NO_ERROR; i++)
               {
                  rc = dfsVdiReadSector( psn    +  i,  si->Geo.B, si->accessInfo,
                                         buffer + (i * si->Geo.B));
               }
            }
            break;

         case DFST_MEMORY:
            {
               ULONG   i;

               for (i = 0; i < sc && rc == NO_ERROR; i++)
               {
                  rc = dfsReadMdiskSector( dfsDid2RealDiskNr(si->DiskId),
                                           psn    +  i,  si->Geo.B,
                                           buffer + (i * si->Geo.B));
               }
            }
            break;
#endif
         default:
            rc = DFS_NO_DEVICE;
            break;
      }
   }
   #if defined (DUMP)                           // bad sector simulation
      if (rc == NO_ERROR)
      {
         ULN64  badsect = TxaExeSwitchNum( TXA_O_SIMUL, NULL, 0);

         if ((badsect != 0) && (psn != 0) && ((psn % badsect) == 0))
         {
            TRACES(( "Simulating a BAD SECTOR read!\n"));

            TxSleep( 500);                      // simulate retry delay
            memset(buffer, DFS_BADSEC_PATTERN,  // destroy read data
                   sc   *  si->Geo.B);

            rc = ERROR_CRC;
         }
      }
   #endif

   if (psn != DFS_MAX_PSN)                      // keep LSN2Psn value when limited
   {
      si->LastPSN = psn;                        // remember last Read PSN
      if (rc == NO_ERROR)
      {
         si->LastPSN += (sc -1);                // last of whole block
      }
   }
   if (rc == DFS_PSN_LIMIT)
   {
      sc = 0;                                   // signal nothing read!
   }
   if (done != NULL)
   {
      *done = sc;
   }
   TRLEVX(200,("sectors done: %lu\n", sc));
   TRLEVX(500,( "CRC32 over %lu bytes: %8.8lx\n", (sc * si->Geo.B), TxCrc32( buffer, (sc * si->Geo.B))));
   TRHEXS( 500,  buffer, 0x100, "START of buffer data");
   TRHEXS( 900,  buffer, sc * si->Geo.B, "ALL buffer data");
   RETURN (rc);
}                                               // end 'dfstPsnReadCount'
/*---------------------------------------------------------------------------*/


#if !defined (DFS_PHYSDISK_ONLY)
/*****************************************************************************/
// Read sector(s) from volume or imagefile in buffer starting at specified LSN
/*****************************************************************************/
static ULONG dfstReadLog
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    Start LSN
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // OUT   Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   int                 ra;                      // read attempts
   ULONG               rd = 0;                  // sectors read
   ULONG               rs = nrs;                // read size
   LLONG               so = (LLONG) lsn;        // seek offset
   #if   defined (WIN32)
   #elif defined (DOS32)
      union  REGS      regs;
      struct SREGS     sreg;
      DFS_IOCTL_RW    *ios = (DFS_IOCTL_RW *) xpar;
   #elif defined (UNIX)
   #else
      LLONG            nl = 0;                  // new lsn
   #endif

   ENTER();
   TRACES(("% 2lu sectors at LSN %llx buffer at %p\n", nrs, lsn, buffer));

   #if defined (WIN32) || defined (UNIX)
      rc = dfstSeekSector( si, si->vfHandle, (ULONG) so);
   #endif
   if (si->SectorIO == FALSE)
   {
      so *= si->Geo.B;
      rs *= si->Geo.B;
   }
   #if   defined (WIN32)
   #elif defined (DOS32)
   if (si->Type == DFST_VOLUME)                 // direct disk access
   {
      ULONG            c,h,s;

      rc = dfstPsn2Sys( store, lsn, &c, &h, &s);
      if ((rc == NO_ERROR) && (c <= 0xffff))
      {
         //- to be refined 64bit, is DOS restricted to 65536 cylinders? (500Gb)

         BYTE       *dpmiBuf = NULL;            // DPMI compatible buffer ptr

         if (buffer > TXDX_REAL_LIMIT)          // above 1MB
         {
            if (nrs > XBUFS)
            {
               nrs = XBUFS;                     // limit to xbuf buffersize
               TxPrint( "\nInternal ERROR    : ReadLog exceeds XBUFS (%hu)\n", XBUFS);
            }
            dpmiBuf = xbuf;                     // use DPMI compatible buffer
         }                                      // copy data after read!
         else
         {
            dpmiBuf = buffer;                   // use supplied buffer directly
         }
         for ( ra = 0,                      rc  = ERROR_READ_FAULT;
              (ra < dfsa->readAttempts) && (rc != NO_ERROR);
               ra++)
         {
            memset( &regs,  0, sizeof(regs));
            memset( ios,    0, sizeof(DFS_IOCTL_RW)); // start clean

            ios->c       = (USHORT) c;
            ios->h       = (USHORT) h;
            ios->s       = (USHORT) s;
            ios->databuf = txDpmi16ptr16( dpmiBuf);
            ios->s      -= 1;                   // sectors start at 0!
            ios->sectors = 1;

            TRACES(("dpmiBuf: %p  ios: %p  ios->databuf: %8.8lx\n", dpmiBuf, ios, ios->databuf));
            TRHEXS( 500,  ios, sizeof(DFS_IOCTL_RW), "DFS_IOCTL_RW");

            memset( &sreg,     0, sizeof(sreg));
            memset( &txdx_rmi, 0, sizeof(txdx_rmi));
            txdx_rmi.eax = 0x440d;              // IOctl generic block dev
            txdx_rmi.ecx = 0x0861;              // logical disk, read track
            txdx_rmi.ebx = (BYTE) si->vfHandle; // 1=A, 2=B etc
            txdx_rmi.ds  = txDpmiSegment(ios);  // IO struct (ds:dx)

            TRHEXS( 500,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

            regs.w.ax    = TXDX_DPMI_RMINT;     // simulate realmode INT
            regs.h.bl    = DFS_I21;             // DOS function
            regs.x.edi   = FP_OFF( &txdx_rmi);
            sreg.es      = FP_SEG( &txdx_rmi);

            txDpmiCall( &regs, &sreg);
            if (regs.x.cflag != 0)              // error
            {
               si->Retries++;                   // total retry statistics
               rc = TXWORD.ax;
               TRACES(( "Read fail, attempt %u of %lu, total retries %lu\n", ra+1, dfsa->readAttempts, si->Retries));
            }
            else
            {
               rc = NO_ERROR;
            }
         }

         TRACES(("IORW structure at %p  buffer: %8.8lx\n", ios, ios->databuf));
         TRHEXS( 500, ios, sizeof(DFS_IOCTL_RW), "DFS_IOCTL_RW");
         TRACES(("Disk %u lsn:%llX ==> cyl:%lu, h:%lu, s:%lu\n", (USHORT) si->vfHandle, lsn, ios->c, ios->h, ios->s));

         if (regs.x.cflag == 0)                 // no error
         {
            if (buffer > TXDX_REAL_LIMIT)       // above 1MB
            {
               memcpy( buffer, dpmiBuf, nrs * si->Geo.B);
            }
            rc = DFS_PENDING;                   // function done, RC pending
         }
      }
      else
      {
         rc = DFS_PSN_LIMIT;
      }
   }
   else                                         // file-image access (DFST_RAWIMAGE)
   {
      rc = fseek( si->vfHandle, (LONG) so, SEEK_SET);
   }
   #elif defined (UNIX)
   #else
   TRACES(( "DosSetFilePtr handle: %lu, offset: %llu  FILE_BEGIN\n", si->vfHandle, so));
   if (dfsa->os2api->DosSeekL)
   {
      rc = (dfsa->os2api->DosSeekL)(si->vfHandle,  so, FILE_BEGIN, &nl);
   }
   else
   {
      if (so < 0x8000000)                       // within 2GB
      {
         rc = DosSetFilePtr( si->vfHandle, (LONG) so, FILE_BEGIN, (ULONG *) &nl);
      }
      else
      {
         rc = DFS_PSN_LIMIT;
      }
   }
   if ((rc == ERROR_SEEK) && (so == 0) && (si->ReadOnly)) // FAT32 kludge
   {                                            // will ignore read error on
      rc = NO_ERROR;                            // first read to bootsector
   }                                            // in drive correlation :-)
   #endif
   switch (rc)
   {
      case NO_ERROR:                            // actual read not done yet!
         if (si->Type == DFST_RAWIMAGE)         // File (image) access
         {
            TRACES(( "ByteSize:0x%llx so:0x%llx\n", si->ByteSize, so));
            if (so >= si->ByteSize)             // read beyond end ?
            {
               rc = ERROR_READ_FAULT;           // not allowed ...
            }
         }
         if (rc == NO_ERROR)
         {
            for ( ra = 0, rc  = ERROR_READ_FAULT;
                 (ra < dfsa->readAttempts) && (rc != NO_ERROR);
                  ra++)
            {
               rc = TxRead( si->vfHandle, buffer, rs, &rd);
               if (rc != NO_ERROR)
               {
                  si->Retries++;                // total retry statistics
                  TRACES(( "Read fail, attempt %u of %lu, total retries %lu\n", ra+1, dfsa->readAttempts, si->Retries));
               }
            }
            if (rc == NO_ERROR)
            {
               if (rd == 0)
               {
                  rc = ERROR_READ_FAULT;
               }
            }
            TRACES(("Read %u (= %lu sectors), h: %lu at LSN %llX; got: %u\n", (USHORT) rs, nrs, (ULONG) si->vfHandle, lsn, rd));
         }
         break;

      case DFS_PENDING:
         rc = NO_ERROR;                         // already done
         break;

      case ERROR_INVALID_DRIVE:                 // Usually caught on Open but
         rc = NO_ERROR;                         // sometimes returned by DOS
         break;                                 // on the bootsector scan

      default:
         if ((dfsa->warnLevel > 1) ||           // unless pedantic mode set,
             ((lsn != L64_NULL) && (lsn != 0))) // avoid many messages for the
         {                                      // bootsector (FAT32 on OS/2)
            TxPrint("Error %3u seeking : 0x%llX for read - %s\n", rc, lsn, dfstStoreDesc1( store));
         }
         break;
   }
   RETURN (rc);
}                                               // end 'dfstReadLog'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Write sectors using PSN, and count number of sectors actually written
/*****************************************************************************/
ULONG dfstPsnWriteCount
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    Number of sectors
   ULONG              *done,                    // OUT   sectors done, or NULL
   BYTE               *buffer                   // IN    Sector buffer
)
{
   ULONG               rc = 0;                  // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               sc = nrs;                // sector count

   ENTER();

   TRLEVX(200,("Base start: %llx, Final: %llX, sectorsize:%lu\n", si->Start, si->Final, si->Geo.B));
   TRARGS(("%lu sectors to store %lu, PSN %llX from: %8.8lX\n", nrs, store, psn, buffer));

   TRLEVX(200,( "CRC32 over %lu bytes: %8.8lx\n", (sc * si->Geo.B), TxCrc32( buffer, (sc * si->Geo.B))));
   TRHEXS( 800,  buffer, sc * si->Geo.B, "buffer data");

   //- Allow write beyond limits only for expandable file
   if ((psn >= si->Start) && ((psn <= si->Final) || (si->Type == DFST_RAWIMAGE)))
   {
      if (si->Type != DFST_RAWIMAGE)
      {
         if ((psn + nrs -1) > si->Final)        // last sector beyond limit
         {
            sc = si->Final - psn +1;            // corrected nr of sectors
            TRACES(("Nr of sectors to do restricted to: %lu (base limit)\n", sc));
         }
      }

      if (si->ReadOnly == FALSE)
      {
         switch (si->Type)
         {
            case DFST_PHDISK:
               if (si->di != NULL)
               {
                  #if   defined (WIN32)
                     {
                        ULONG         dummy;

                        rc = dfstSeekSector( si, DI->pdHandle, psn);
                        if (rc == NO_ERROR)
                        {
                           if (TxWrite( DI->pdHandle, buffer, sc * si->Geo.B, &dummy) != 0)
                           {
                              TRACES(( "WriteFile error: %s\n", txNtLastError()));
                              rc = GetLastError();
                           }
                        }
                     }
                  #elif defined (UNIX)
                  {
                     rc = dfstSeekSector( si, DI->pdHandle, psn);
                     if (rc == NO_ERROR)
                     {
                        if (write( DI->pdHandle, buffer, sc * si->Geo.B) == -1)
                        {
                           rc = TxRcFromErrno( errno);
                        }
                     }
                  }
                  #elif defined (DOS32)
                     {
                        rc = dfstWriteSectors( store, psn, sc, buffer);
                     }
                  #else
                     {
                        ULONG   chunkSectors;
                        ULONG   chunkOffset  = 0;  // start PSN for chunk
                        ULONG   sectorsTodo  = sc; // number of sectors left to do

                        while ((sectorsTodo > 0) && (rc == NO_ERROR)) // exit when PSN limit!
                        {
                           if (sectorsTodo > DFS_OS2_BUFSIZE_LIMIT)
                           {
                              chunkSectors = DFS_OS2_BUFSIZE_LIMIT;
                              sectorsTodo -= DFS_OS2_BUFSIZE_LIMIT;
                           }
                           else
                           {
                              chunkSectors = sectorsTodo;
                              sectorsTodo  = 0;
                           }
                           rc = dfstWriteSectors( store, psn +  chunkOffset,  chunkSectors,
                                                      buffer + (chunkOffset * si->Geo.B));
                           chunkOffset += chunkSectors;
                        }
                     }
                  #endif
               }
               else
               {
                  rc = DFS_NO_DEVICE;
               }
               break;

#if !defined (DFS_PHYSDISK_ONLY)
            case DFST_VOLUME:
            case DFST_PMIRAW:
            case DFST_RAWIMAGE:
               #if defined (DOS32)              // one sector at a time
                  {
                     ULONG   i;

                     for (i = 0; i < sc && rc == NO_ERROR; i++)
                     {
                        rc = dfstWriteLog( store, psn +i, 1, buffer + (i * si->Geo.B));
                     }
                  }
               #else
                  rc = dfstWriteLog( store, psn, sc, buffer);
               #endif
               break;

            case DFST_PMIMZD:
            case DFST_IMZIMAGE:
               rc = DFS_READ_ONLY;              // for the moment, to be refined
               break;

            case DFST_PMVBOX:
               {
                  ULONG   i;

                  for (i = 0; i < sc && rc == NO_ERROR; i++)
                  {
                     rc = dfsVdiWriteSector( psn    +  i,  si->Geo.B, si->accessInfo,
                                             buffer + (i * si->Geo.B));
                  }
               }
               break;

            case DFST_MEMORY:
               {
                  ULONG   i;

                  for (i = 0; i < sc && rc == NO_ERROR; i++)
                  {
                     rc = dfsWriteMdiskSector( dfsDid2RealDiskNr(si->DiskId),
                                               psn +i,       si->Geo.B,
                                               buffer + (i * si->Geo.B));
                  }
               }
               break;
#endif
            default:
               rc = DFS_NO_DEVICE;
               break;
         }
      }
      else
      {
         rc = DFS_READ_ONLY;
      }
   }
   else
   {
      sc = 0;                                   // nothing written!
      rc = DFS_PSN_LIMIT;
   }
   if (psn != DFS_MAX_PSN)                      // keep LSN2Psn value when limited
   {
      si->LastPSN = psn;                        // remember last Write PSN
      if (rc == NO_ERROR)
      {
         si->LastPSN += (sc -1);                // last of whole block
      }
   }
   if (done != NULL)
   {
      *done = sc;
   }
   TRACES(("sectors done: %lu\n", sc));
   RETURN (rc);
}                                               // end 'dfstPsnWriteCount'
/*---------------------------------------------------------------------------*/


#if !defined (DFS_PHYSDISK_ONLY)
/*****************************************************************************/
// Write sector(s) to logical volume in buffer starting at specified LSN
/*****************************************************************************/
static ULONG dfstWriteLog
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               lsn,                     // IN    Start LSN
   ULONG               nrs,                     // IN    Number of sectors
   BYTE               *buffer                   // IN    Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               wr = 0;                  // bytes written
   ULONG               ws = nrs;                // write size, in bytes
   ULN64               so = (ULN64) lsn;        // seek offset (64 bit)
   #if   defined (WIN32)
   #elif defined (DOS32)
      union  REGS      regs;
      struct SREGS     sreg;
      DFS_IOCTL_RW    *ios = (DFS_IOCTL_RW *) xpar;
   #elif defined (UNIX)
   #else
      LLONG            nl = 0;                  // new lsn
   #endif

   ENTER();
   TRACES(("% 2lu sectors at LSN %llx buffer at %p\n", nrs, lsn, buffer));

   #if defined (WIN32) || defined (UNIX)
      rc = dfstSeekSector( si, si->vfHandle, (ULONG) so);
   #endif
   if (si->SectorIO == FALSE)
   {
      so *= si->Geo.B;                          // byte start of write-area
      ws *= si->Geo.B;                          // nr of bytes to be written
   }
   #if   defined (WIN32)
   #elif defined (DOS32)
   if (si->Type == DFST_VOLUME)                 // direct disk access
   {
      ULONG            c,h,s;

      rc = dfstPsn2Sys( store, lsn, &c, &h, &s);
      if ((rc == NO_ERROR) && (c <= 0xffff))
      {
         BYTE       *dpmiBuf;                   // DPMI compatible buffer ptr

         memset( &regs,  0, sizeof(regs));
         memset( ios,    0, sizeof(DFS_IOCTL_RW)); // start clean


         if (buffer > TXDX_REAL_LIMIT)          // above 1MB
         {
            if (nrs > XBUFS)
            {
               nrs = XBUFS;                     // limit to xbuf buffersize
               TxPrint( "\nInternal ERROR    : WriteLog exceeds XBUFS (%hu)\n", XBUFS);
            }
            dpmiBuf = xbuf;                     // use DPMI compatible buffer
            memcpy( dpmiBuf, buffer, nrs * si->Geo.B);
         }                                      // copy data before write!
         else
         {
            dpmiBuf = buffer;                   // use supplied buffer directly
         }
         ios->c       = (USHORT) c;
         ios->h       = (USHORT) h;
         ios->s       = (USHORT) s;
         ios->databuf = txDpmi16ptr16( dpmiBuf);
         ios->s      -= 1;                      // sectors start at 0!
         ios->sectors = 1;

         TRACES(("dpmiBuf: %p  ios: %p  ios->databuf: %8.8lx\n",
                  dpmiBuf,     ios,     ios->databuf));
         TRHEXS( 500,  ios, sizeof(DFS_IOCTL_RW), "DFS_IOCTL_RW");

         memset( &sreg,     0, sizeof(sreg));
         memset( &txdx_rmi, 0, sizeof(txdx_rmi));
         txdx_rmi.eax = 0x440d;                 // IOctl generic block dev
         txdx_rmi.ecx = 0x0841;                 // logical disk, write track
         txdx_rmi.ebx = (BYTE) si->vfHandle;    // 1=A, 2=B etc
         txdx_rmi.ds  = txDpmiSegment(ios);     // IO struct (ds:dx)

         TRHEXS( 500,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

         regs.w.ax    = TXDX_DPMI_RMINT;        // simulate realmode INT
         regs.h.bl    = DFS_I21;                // DOS function
         regs.x.edi   = FP_OFF( &txdx_rmi);
         sreg.es      = FP_SEG( &txdx_rmi);

         txDpmiCall( &regs, &sreg);

         TRACES(("IORW structure at %p  buffer: %8.8lx\n", ios, ios->databuf));
         TRHEXS( 500, ios, sizeof(DFS_IOCTL_RW), "DFS_IOCTL_RW");
         TRACES(("Disk %u lsn:%llX ==> cyl:%u, h:%u, s:%u\n", (USHORT) si->vfHandle, lsn, ios->c, ios->h, ios->s));

         if (regs.x.cflag != 0)                 // error
         {
            rc = TXWORD.ax;
         }
         else
         {
            rc = DFS_PENDING;                   // function done, RC pending
         }
      }
      else
      {
         rc = DFS_PSN_LIMIT;
      }
   }
   else                                         // file-image access (DFST_RAWIMAGE)
   {
      rc = fseek( si->vfHandle, (LONG) so, SEEK_SET);
   }
   #elif defined (UNIX)
   #else
   if (dfsa->os2api->DosSeekL)
   {
      rc = (dfsa->os2api->DosSeekL)(si->vfHandle,  (LLONG) so, FILE_BEGIN, &nl);
   }
   else
   {
      if (so < 0x8000000)                       // within 2GB
      {
         rc = DosSetFilePtr( si->vfHandle, (LONG) so, FILE_BEGIN, (ULONG *) &nl);
      }
      else
      {
         rc = DFS_PSN_LIMIT;
      }
   }
   #endif
   switch (rc)
   {
      case NO_ERROR:
         if (si->Type == DFST_RAWIMAGE)         // File (image) access
         {
            TRACES(( "ByteSize:0x%llx so:0x%llx ws:%lu\n", si->ByteSize, so, ws));
            if ((so + ws) > si->ByteSize)       // write beyond end ?
            {
               if (SINF->stFlags & DFSTFL_EXPLICITSIZE) // no grow allowed
               {
                  if (so < si->ByteSize)
                  {
                     ws = si->ByteSize - so;    // limit size to write
                  }
                  else
                  {
                     rc = ERROR_WRITE_FAULT;    // completely beyond ...
                  }
               }
               else                             // update exact bytesize
               {                                // in the store info
                  si->ByteSize = so + ws;
               }
            }
         }
         if (rc == NO_ERROR)
         {
            if ((rc = TxWrite( si->vfHandle, buffer, ws, &wr)) == NO_ERROR)
            {
               if (wr == 0)
               {
                  rc = ERROR_WRITE_FAULT;
               }
            }
         }
         TRACES(("Write %lu (= %lu sectors), h: %lu at LSN 0x%llX; written: %u\n",
                  ws, nrs, (ULONG) si->vfHandle, lsn, wr));
         break;

      case ERROR_INVALID_DRIVE:                 // Usually caught on Open but
         rc = NO_ERROR;                         // sometimes returned by DOS
         break;                                 // on the bootsector scan

      case DFS_PENDING:
         rc = NO_ERROR;                         // already done
         break;

      default:
         if (lsn != L64_NULL)
         {
            TxPrint("Error %3u seeking : 0x%llX for write - %s\n", rc, lsn, dfstStoreDesc1( store));
         }
         break;
   }
   RETURN (rc);
}                                               // end 'dfstWriteLog'
/*---------------------------------------------------------------------------*/
#endif

#if defined (WIN32) || defined (UNIX)
/*****************************************************************************/
// Position to specified sector offset for given handle (logical/physical disk)
/*****************************************************************************/
static ULONG dfstSeekSector
(
   DFSTOREINFO        *si,                      // IN    DFS store to be used
   TXHDISK             dh,                      // IN    handle
   ULN64               sn                       // IN    sector nr
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   int                 BitPs;
   #if defined (UNIX)
      LLONG            ll;                      // seek offset/result
   #endif
   #if !defined (DARWIN)
      ULONG            hi = 0;                  // High part of seek offset
      ULONG            lo = 0;                  // Low  part of seek offset
   #endif


   switch (si->Geo.B)                           // sectorsize
   {
      case 1024: BitPs = 10; break;
      case 2048: BitPs = 11; break;
      case 4096: BitPs = 12; break;
      default:   BitPs =  9; break;
   }

   #if defined (DARWIN)
      ll = ((LLONG) sn) << BitPs;               // offset in bytes, not sectors
      TRLEVX(500,("Seek handle: %8.8lx ll:0x%llX for sector:0x%llX\n", dh, ll, sn));
   #else
      lo =  sn << BitPs;
      hi = (sn & (L64_NULL << (32 - BitPs))) >> (32 - BitPs);
      TRLEVX(500,("Seek handle: 0x%lx Hi:%08.8lX Lo:0x%lX for sector:0x%llX\n", dh, hi, lo, sn));
   #endif


   #if   defined (WIN32)
      if (SetFilePointer( dh, lo, (PLONG) &hi, FILE_BEGIN) == L32_NULL)
      {
         TRLEVX(500,( "SetFilePointer error: %s\n", txNtLastError()));
         rc = GetLastError();
      }
   #elif defined (LINUX)
      if ( _llseek( dh, hi, lo, &ll, SEEK_SET) == -1)
      {
         rc = TxRcFromErrno( errno);
      }
   #elif defined (DARWIN)
      if (lseek( dh, ll, SEEK_SET) == -1LL)
      {
         rc = TxRcFromErrno( errno);
      }
   #else
   #endif
   return (rc);
}                                               // end 'dfstSeekSector'
/*---------------------------------------------------------------------------                    */
#endif

#if defined (DOS32) || defined (DEV32)
/*****************************************************************************/
// Read 1 .. tracksize sectors in chunks of maximum 127 (OS/2 64 KB buffer bug)
/*****************************************************************************/
static ULONG dfstReadSectors
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    nr of sectors
   BYTE               *buffer                   // OUT   Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               fc, fh, fs;              // first sector CHS
   ULONG               lc, lh, ls;              // last  sector CHS
   ULONG               ns = si->Sys.S;          // use real system geometry!
   ULONG               nh = si->Sys.H;
   ULONG               nr = 0;                  // nr of sectors in run
   ULONG               rs = 0;                  // relative sector nr
   ULONG               c, h;

   ENTER();
   TRARGS(("Store:%lu psn:0x%llx  nrs:%lu  buffer:0x%8.8lx\n", store, psn, nrs, buffer));

   if ((rc = dfstPsn2Sys( store, psn, &fc, &fh, &fs)) == NO_ERROR)
   {
      if ((rc = dfstPsn2Sys( store, psn + nrs -1, &lc, &lh, &ls)) == NO_ERROR)
      {
         if (fs != 1)                           // not at start of track
         {
            nr = ns - fs +1;
            if ((ULONG) nr > nrs)
            {
               nr = (USHORT) nrs;               // limit to #sectors wanted
            }
            rc = dfstReadSectChs( store, fc, fh, fs, nr , buffer + (rs * si->Geo.B));

            rs  += nr;                          // next relative sector todo

            h = fh +1;                          // next head for FULL track
         }
         else
         {
            h = fh;                             // nothing done yet
         }
         if ((rs < nrs) && (rc == NO_ERROR))
         {
            c = fc;
            if (h >= nh)                        // next head on next cyl !
            {
               h = 0;
               c++;
            }
            while ((c < lc) || ((c == lc) && (h < lh)) && (rc == NO_ERROR))
            {
               rc = dfstReadSectChs( store, c, h, 1, ns, buffer + (rs * si->Geo.B));
               rs  += ns;                       // next relative sector todo

               if (++h >= nh)
               {
                  h = 0;                        // back to first head
                  c++;                          // on next cylinder
               }
            }
            if ((rs < nrs) && (rc == NO_ERROR))
            {
               rc = dfstReadSectChs( store, lc, lh, 1, ls , buffer + (rs * si->Geo.B));
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfstReadSectors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read 1 .. tracksize physical sectors using Cylinder/Head/Sector address
/*****************************************************************************/
static ULONG dfstReadSectChs
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector,                  // IN    first sector number
   ULONG               nrs,                     // IN    nr of sectors
   BYTE               *buffer                   // OUT   Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   int                 ra;                      // read attempts
   #if defined (DOS32)
      union   REGS     regs;
      struct SREGS     sreg;
      BYTE            *dpmiBuf;                 // DPMI compatible buffer ptr
   #elif defined (UNIX)
   #else
      ULONG            ParmLengthInOut;
      ULONG            DataLengthInOut;
   #endif

   ENTER();
   TRLEVX(200,("Store:%lu disk:%hu  use:%lu  handle:%8.8lx\n", store, si->DiskId, DI->useCount, DI->pdHandle));
   TRARGS(("LSN: %llx = CHS:%lu %lu %lu = PSN: %llx  nrs:%hu\n",
            dfstPsn2LSN( store, dfsGeoChs2Psn( si->Sys.H, si->Sys.S, cylinder, head, sector)),
                                                                     cylinder, head, sector,
                                dfsGeoChs2Psn( si->Sys.H, si->Sys.S, cylinder, head, sector), nrs));

   if (DI->pdHandle != PDHNULL)
   {
      #if defined (DOS32)

         if (buffer > TXDX_REAL_LIMIT)          // above 1MB
         {
            if (nrs > XBUFS)
            {
               nrs = XBUFS;                     // limit to xbuf buffersize
               TxPrint( "\nInternal ERROR    : ReadSect exceeds XBUFS (%hu)\n", XBUFS);
            }
            dpmiBuf = xbuf;                     // use DPMI compatible buffer
         }                                      // copy data after read!
         else
         {
            dpmiBuf = buffer;                   // use supplied buffer directly
         }

         for ( ra = 0,                      rc  = ERROR_READ_FAULT;
              (ra < dfsa->readAttempts) && (rc != NO_ERROR);
               ra++)
         {
            memset( &regs, 0, sizeof(regs));
            memset( &sreg, 0, sizeof(sreg));

            if ((si->Extended13)  && (cylinder >= 1024) ) // Ext-Int13 needed
            {
               DFS_INT13_ERW    *pi13;          // int13 parameter pointer

               pi13 = (DFS_INT13_ERW *) xpar;   // use DPMI compatible area

               memset(pi13, 0, sizeof(DFS_INT13_ERW));
               pi13->size    = sizeof(DFS_INT13_ERW);
               pi13->sectors = nrs;
               pi13->i13Psn  = dfstChs2Psn( store, cylinder, head, sector);

               pi13->databuf = txDpmi16ptr16( dpmiBuf);

               TRACES(("dpmiBuf: %p  pi13: %p  pi13->databuf: %8.8lx\n",
                        dpmiBuf,     pi13,     pi13->databuf));
               TRHEXS( 500,  pi13, sizeof(DFS_INT13_ERW), "DFS_INT13_ERW");
               TRHEXS( 500,  dpmiBuf,      64, "DFS_INT13_ERW buffer pre");

               memset( &txdx_rmi, 0, sizeof(txdx_rmi));
               txdx_rmi.eax = DFS_I13_E_READ  << 8;
               txdx_rmi.edx = (BYTE) DI->pdHandle;
               txdx_rmi.ds  = txDpmiSegment(pi13); // params (ds:si)

               TRHEXS( 500,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

               regs.w.ax    = TXDX_DPMI_RMINT;  // simulate realmode INT
               regs.h.bl    = DFS_I13;
               regs.x.edi   = FP_OFF( &txdx_rmi);
               sreg.es      = FP_SEG( &txdx_rmi);

               txDpmiCall( &regs, &sreg);
            }
            else                                // below cylinder 1024
            {
               memset( &txdx_rmi, 0, sizeof(txdx_rmi));
               txdx_rmi.eax = (DFS_I13_S_READ  << 8) + nrs;
               txdx_rmi.edx = (head            << 8) + DI->pdHandle;
               txdx_rmi.ecx = DFSCOMBINE(sector,cylinder);
               txdx_rmi.es  = txDpmiSegment(dpmiBuf); // buffer (es:bx)

               regs.w.ax    = TXDX_DPMI_RMINT;  // simulate realmode INT
               regs.h.bl    = DFS_I13;
               regs.x.edi   = FP_OFF( &txdx_rmi);
               sreg.es      = FP_SEG( &txdx_rmi);

               txDpmiCall( &regs, &sreg);
            }
            if (regs.x.cflag != 0)              // some error
            {
               si->Retries++;                   // total retry statistics
               rc = TXWORD.ax;
               TRACES(( "Read fail, attempt %u of %lu, total retries %lu\n",
                         ra+1, dfsa->readAttempts, si->Retries));
            }
            else
            {
               rc = NO_ERROR;
            }
         }
         if ((rc == NO_ERROR) &&                // read successful
             (buffer > TXDX_REAL_LIMIT))        // and above 1MB
         {                                      // copy data to buffer
            memcpy( buffer, dpmiBuf, nrs * si->Geo.B);
         }
      #elif defined (UNIX)
      #else                                     // 32-bit OS/2
         if (si->TrackLayout != NULL)
         {
            if (cylinder <= 0xffff)
            {
               for ( ra = 0,                      rc  = ERROR_READ_FAULT;
                    (ra < dfsa->readAttempts) && (rc != NO_ERROR);
                     ra++)
               {
                  if (sector == 1)
                  {
                     si->TrackLayout->bCommand = 0x01; // starting at 1
                  }
                  else
                  {
                     si->TrackLayout->bCommand = 0x00; // not starting at 1
                  }
                  si->TrackLayout->usCylinder    = (USHORT) cylinder;
                  si->TrackLayout->usHead        = (USHORT) head;
                  si->TrackLayout->usFirstSector = (USHORT) sector -1; // first table entry
                  si->TrackLayout->cSectors      = nrs;

                  ParmLengthInOut = si->tlSize;
                  DataLengthInOut = si->Geo.B * nrs;

                  TRLEVX(200,( "IOCTL TrackLayout Cmd:%hu C:% 4hu H:% 3hu S:% 2hu sects:%hu\n",
                            si->TrackLayout->bCommand,
                            si->TrackLayout->usCylinder,
                            si->TrackLayout->usHead,
                            si->TrackLayout->usFirstSector,
                            si->TrackLayout->cSectors));
                  TRLEVX(200,( "IOCTL Handle:%lu ParmLen:%lu DataLen:%lu buffer addr:%8.8lx\n",
                            DI->pdHandle, ParmLengthInOut, DataLengthInOut, buffer));
                  TRHEXS(800,  si->TrackLayout, si->tlSize, "trackLayout");

                  rc = DosDevIOCtl( DI->pdHandle,
                                    IOCTL_PHYSICALDISK,
                                    PDSK_READPHYSTRACK,
                                    si->TrackLayout,
                                    si->tlSize,
                                    &ParmLengthInOut,
                                    buffer,
                                    si->Geo.B * nrs,
                                    &DataLengthInOut);

                  TRLEVX(200,( "IOCTL rc:%lu  ParmLen:%lu DataLen:%lu\n",
                                      rc, ParmLengthInOut, DataLengthInOut));
                  if (rc != NO_ERROR)
                  {
                     si->Retries++;             // total retry statistics
                     TRACES(( "Read fail, attempt %u of %lu, total retries %lu\n",
                               ra+1, dfsa->readAttempts, si->Retries));
                  }
               }
            }
            else
            {
               rc = DFS_PSN_LIMIT;
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      #endif
   }
   else
   {
      rc = DFS_NO_DEVICE;
   }
   TRHEXS( 800,  buffer, nrs * si->Geo.B, "track data");
   RETURN (rc);
}                                               // end 'dfstReadSectChs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write 1 .. tracksize sectors in chunks of maximum 127 (OS/2 64 KB buffer bug)
/*****************************************************************************/
static ULONG dfstWriteSectors
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               psn,                     // IN    phys. sector nr
   ULONG               nrs,                     // IN    nr of sectors
   BYTE               *buffer                   // IN    Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   ULONG               fc, fh, fs;              // first sector CHS
   ULONG               lc, lh, ls;              // last  sector CHS
   ULONG               ns = si->Sys.S;          // use real system geometry!
   ULONG               nh = si->Sys.H;
   ULONG               nr = 0;                  // nr of sectors in run
   ULONG               rs = 0;                  // relative sector nr
   ULONG               c, h;

   ENTER();
   TRARGS(("Store:%lu psn:0x%llx  nrs:%lu  buffer:0x%8.8lx\n", store, psn, nrs, buffer));

   if ((rc = dfstPsn2Sys( store, psn, &fc, &fh, &fs)) == NO_ERROR)
   {
      if ((rc = dfstPsn2Sys( store, psn + nrs -1, &lc, &lh, &ls)) == NO_ERROR)
      {
         if (fs != 1)                           // not at start of track
         {
            nr = ns - fs +1;
            if ((ULONG) nr > nrs)
            {
               nr = (USHORT) nrs;                // limit to #sectors wanted
            }
            rc = dfstWriteSectChs( store, fc, fh, fs, nr , buffer + (rs * si->Geo.B));

            rs  += nr;                          // next relative sector todo

            h = fh +1;                          // next head for FULL track
         }
         else
         {
            h = fh;                             // nothing done yet
         }
         if ((rs < nrs) && (rc == NO_ERROR))
         {
            c = fc;
            if (h >= nh)                        // next head is next cyl !
            {
               h = 0;
               c++;
            }
            while ((c < lc) || ((c == lc) && (h < lh)) && (rc == NO_ERROR))
            {
               rc = dfstWriteSectChs( store, c, h, 1, ns, buffer + (rs * si->Geo.B));
               rs  += ns;                       // next relative sector todo

               if (++h >= nh)
               {
                  h = 0;                        // back to first head
                  c++;                          // on next cylinder
               }
            }
            if ((rs < nrs) && (rc == NO_ERROR))
            {
               rc = dfstWriteSectChs( store, lc, lh, 1, ls , buffer + (rs * si->Geo.B));
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfstWriteSectors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write 1 .. tracksize physical sectors using Cylinder/Head/Sector address
/*****************************************************************************/
static ULONG dfstWriteSectChs
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector,                  // IN    first sector number
   ULONG               nrs,                     // IN    nr of sectors
   BYTE               *buffer                   // IN    Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   #if defined (DOS32)
      union   REGS     regs;
      struct SREGS     sreg;
      BYTE            *dpmiBuf;                 // DPMI compatible buffer ptr
   #elif defined (UNIX)
   #else
      ULONG            ParmLengthInOut;
      ULONG            DataLengthInOut;
   #endif

   ENTER();
   TRLEVX(200,("Store:%lu disk:%hu  use:%lu  handle:%8.8lx\n", store, si->DiskId, DI->useCount, DI->pdHandle));
   TRARGS(("LSN: %llx = CHS:%lu %lu %lu = PSN: %llx  nrs:%hu\n",
            dfstPsn2LSN( store, dfstChs2Psn( store, cylinder, head, sector)),
                                                    cylinder, head, sector,
                                dfstChs2Psn( store, cylinder, head, sector), nrs));
   TRHEXS( 800,  buffer, nrs * si->Geo.B, "track data");

   if (DI->pdHandle != PDHNULL)
   {
      #if defined (DOS32)
         memset( &regs, 0, sizeof(regs));
         memset( &sreg, 0, sizeof(sreg));

         if (buffer > TXDX_REAL_LIMIT)          // above 1MB
         {
            if (nrs > XBUFS)
            {
               nrs = XBUFS;                     // limit to xbuf buffersize
               TxPrint( "\nInternal ERROR    : WriteSect exceeds XBUFS (%hu)\n", XBUFS);
            }
            dpmiBuf = xbuf;                     // use DPMI compatible buffer
            memcpy( dpmiBuf, buffer, nrs * si->Geo.B);
         }                                      // copy data before write!
         else
         {
            dpmiBuf   = buffer;                 // use supplied buffer directly
         }

         if ((si->Extended13)  && (cylinder >= 1024) ) // Int13 extention needed
         {
            DFS_INT13_ERW    *pi13;             // int13 parameter pointer

            pi13 = (DFS_INT13_ERW *) xpar;      // use DPMI compatible area

            memset(pi13, 0, sizeof(DFS_INT13_ERW));
            pi13->size    = sizeof(DFS_INT13_ERW);
            pi13->sectors = nrs;
            pi13->i13Psn  = dfstChs2Psn( store, cylinder, head, sector);

            pi13->databuf = txDpmi16ptr16( dpmiBuf);

            TRACES(("dpmiBuf: %p  pi13: %p  pi13->databuf: %8.8lx\n",
                     dpmiBuf,     pi13,     pi13->databuf));
            TRHEXS( 500,  pi13, sizeof(DFS_INT13_ERW), "DFS_INT13_ERW");
            TRHEXS( 500,  dpmiBuf,      64, "DFS_INT13_ERW buffer pre");

            memset( &txdx_rmi, 0, sizeof(txdx_rmi));
            txdx_rmi.eax = DFS_I13_E_WRITE << 8;
            txdx_rmi.edx = (BYTE) DI->pdHandle;
            txdx_rmi.ds  = txDpmiSegment(pi13); // params (ds:si)

            TRHEXS( 500,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

            regs.w.ax    = TXDX_DPMI_RMINT;     // simulate realmode INT
            regs.h.bl    = DFS_I13;
            regs.x.edi   = FP_OFF( &txdx_rmi);
            sreg.es      = FP_SEG( &txdx_rmi);

            txDpmiCall( &regs, &sreg);
         }
         else                                   // below cylinder 1024
         {
            memset( &txdx_rmi, 0, sizeof(txdx_rmi));
            txdx_rmi.eax = (DFS_I13_S_WRITE << 8) + nrs;
            txdx_rmi.edx = (head            << 8) + DI->pdHandle;
            txdx_rmi.ecx = DFSCOMBINE(sector,cylinder);
            txdx_rmi.es  = txDpmiSegment(dpmiBuf); // buffer (es:bx)

            regs.w.ax    = TXDX_DPMI_RMINT;     // simulate realmode INT
            regs.h.bl    = DFS_I13;
            regs.x.edi   = FP_OFF( &txdx_rmi);
            sreg.es      = FP_SEG( &txdx_rmi);

            txDpmiCall( &regs, &sreg);
         }
         if (regs.x.cflag != 0)                 // some error
         {
            rc = TXWORD.ax;
         }
      #elif defined (UNIX)
      #else                                     // 32-bit OS/2
         if (si->TrackLayout != NULL)
         {
            if (cylinder <= 0xffff)
            {
               if (sector == 1)
               {
                  si->TrackLayout->bCommand = 0x01; // starting at 1
               }
               else
               {
                  si->TrackLayout->bCommand = 0x00; // not starting at 1
               }
               si->TrackLayout->usCylinder    = (USHORT) cylinder;
               si->TrackLayout->usHead        = (USHORT) head;
               si->TrackLayout->usFirstSector = (USHORT) sector -1; // first entry in table
               si->TrackLayout->cSectors      = nrs;

               ParmLengthInOut = si->tlSize;
               DataLengthInOut = si->Geo.B * nrs;

               TRLEVX(200,( "IOCTL TrackLayout Cmd:%hu C:% 4hu H:% 3hu S:% 2hu sects:%hu\n",
                         si->TrackLayout->bCommand,
                         si->TrackLayout->usCylinder,
                         si->TrackLayout->usHead,
                         si->TrackLayout->usFirstSector,
                         si->TrackLayout->cSectors));

               TRLEVX(200,( "IOCTL Handle:%lu ParmLen:%lu DataLen:%lu buffer addr:%8.8lx\n",
                         DI->pdHandle, ParmLengthInOut, DataLengthInOut, buffer));
               TRHEXS( 800,  si->TrackLayout, si->tlSize, "trackLayout structure");

               rc = DosDevIOCtl( DI->pdHandle,
                                 IOCTL_PHYSICALDISK,
                                 PDSK_WRITEPHYSTRACK,
                                 si->TrackLayout,
                                 si->tlSize,
                                 &ParmLengthInOut,
                                 buffer,
                                 si->Geo.B * nrs,
                                 &DataLengthInOut);

               TRLEVX(200,( "IOCTL rc:%lu  ParmLen:%lu DataLen:%lu\n",
                                   rc, ParmLengthInOut, DataLengthInOut));
            }
            else
            {
               rc = DFS_PSN_LIMIT;
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      #endif
   }
   else
   {
      rc = DFS_NO_DEVICE;
   }
   RETURN (rc);
}                                               // end 'dfstWriteSectChs'
/*---------------------------------------------------------------------------*/
#endif

#if !defined (OEMSB)
/*****************************************************************************/
// Lock or unlock physical-disk or volume, use persistent handle variables
/*****************************************************************************/
ULONG dfstLocking                               // RET   result
(
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULONG               lockmode,                // IN    LOCK / UNLOCK
   BOOL                ignore,                  // IN    ignore failure
   BOOL                verbose                  // IN    report lock status
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   DFSTOREINFO        *si = &(sti[store]);
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      TXHFILE          devhandle = 0;
      ULONG            dcatagory;
      TXLN             locktext  = {0};
      TXTS             dspec;
      ULONG            DataLen;
      ULONG            ParmLen;
      DFS_DPBLOCK      DevParam;                // device parameter block
      BOOL             wasLocked = (si->LockCnt != 0);
   #endif

   ENTER();

   TRACES(("DiskId: %hu, LogLock: %lu, LockCnt: %lu\n",
            si->DiskId,  si->LogLock,  si->LockCnt))
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      switch (si->Type)
      {
         case DFST_PHDISK:
            if (si->Start != 0)                 // partition opened,
            {                                   // try to lock volume
               if (isupper(SINF->drive[0]))     // associated volume letter
               {
                  if ((lockmode != DFS_UNLOCK) && (si->LogLock == 0))
                  {
                     dfstOpenVolLock( si, SINF->drive); // get volume lock handle
                  }
                  if (si->LogLock != 0)
                  {
                     devhandle = si->LogLock;
                     dcatagory = IOCTL_DISK;
                     sprintf( locktext, "Partition %s", SINF->drive);
                  }
               }
            }
            else                                // whole disk, check large floppy
            {
               DFSDISKINFO    *d  = dfsGetDiskInfo( SINF->disknr);

               if (d && (d->flags & DFS_F_FSYSTEMONLY) && isalpha(d->ddrive[0]))
               {
                  if ((lockmode != DFS_UNLOCK) && (si->LogLock == 0))
                  {
                     dfstOpenVolLock( si, d->ddrive); // get volume lock handle
                  }
                  if (si->LogLock != 0)
                  {
                     devhandle = si->LogLock;
                     dcatagory = IOCTL_DISK;
                     sprintf( locktext, "LargeFlop %s", d->ddrive);
                  }
               }
            }
            if (devhandle == 0)                 // whole physical disk open, or
            {                                   // part is not a mounted volume
               devhandle = DI->pdHandle;
               dcatagory = IOCTL_PHYSICALDISK;
               sprintf( locktext, "Physical disk %u", SINF->disknr);
            }
            break;

         case DFST_VOLUME:
            devhandle = si->vfHandle;
            dcatagory = IOCTL_DISK;
            sprintf( locktext, "Logical volume %s", SINF->drive);
            break;

         case DFST_RAWIMAGE:
            sprintf( locktext, "Unsupported: Image file %s", SINF->drive);
            rc = DFS_NO_DEVICE;                 // force warning message
            break;

         case DFST_MEMORY:                      // no need to lock a virtual
         default:
            break;
      }

      if (devhandle != 0)                       // actual lock attempt ...
      {
         if (((lockmode == DFS_UNLOCK) && (si->LockCnt == 1)) ||
             ((lockmode == DFS_LOCK  ) && (si->LockCnt == 0)) )
         {
            dspec[0]  = '\0';
            DataLen   = 1;
            ParmLen   = sizeof( DevParam);

            rc = DosDevIOCtl(devhandle,
                             dcatagory,
                             lockmode,
                             dspec,
                             DataLen, &DataLen,
                             &DevParam,
                             ParmLen, &ParmLen);
            TRACES(("IOCTL func %lu, cat %lu, handle %lu, rc:%ld\n",
                     lockmode, dcatagory, (ULONG) devhandle, (LONG) rc));
         }
         if (rc == NO_ERROR)
         {
            if (lockmode == DFS_UNLOCK)
            {
               if (si->LockCnt)
               {
                  si->LockCnt--;
               }
               if (si->LockCnt == 0)
               {
                  if (si->LogLock != 0)
                  {
                     TxClose( si->LogLock);
                     si->LogLock = 0;
                  }
                  if (si->Type != DFST_MEMORY)
                  {
                     si->LockIgn = dfsa->LockIgn; // ignore lock-failures
                  }
               }
            }
            else
            {
               si->LockCnt++;
            }
         }
         if ((verbose) && (dfsa->batch == FALSE))
         {
            TxPrint("Object lock-count : %lu, %s %s %slocked %s\n", si->LockCnt, locktext, (rc) ? "NOT"    : "",
                   (lockmode == DFS_UNLOCK) ? (si->LockCnt) ? "still " : (wasLocked)   ? "un" : "was not " : "",
                   (lockmode == DFS_LOCK  ) ? "exclusively" : "");
         }
      }
   #endif
   if ((rc != NO_ERROR) && (ignore))
   {
      si->LockIgn = TRUE;                       // Do not prompt when writing
      TxPrint("Ignoring lock-status from now on, write without prompting\n");
   }
   TRACES(("DiskId: %hu, LogLock: %lu, LockCnt: %lu\n", si->DiskId,  si->LogLock,  si->LockCnt))
   RETURN (rc);
}                                               // end 'dfstLocking'
/*---------------------------------------------------------------------------*/

#if defined (DEV32)
/*****************************************************************************/
// Open specified volume in direct mode using persistent handle
/*****************************************************************************/
static ULONG dfstOpenVolLock
(
   DFSTOREINFO        *si,                      // IN    DFS store to be used
   char               *drive                    // IN    Drive specification
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   TXHFILE             dh = 0;                  // Direct Filehandle
   #if   defined (WIN32)
      TXTS             dspec;                   // drive specification string
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      ULONG            act;                     // action taken
   #endif

   ENTER();

   #if defined (WIN32)
      sprintf( dspec, "\\\\.\\%c:", drive[0]);  // logical volume specification
      dh = CreateFile( dspec,
                       GENERIC_READ    | GENERIC_WRITE,
                       FILE_SHARE_READ | FILE_SHARE_WRITE,
                       NULL,                    // default security info
                       OPEN_EXISTING,
                       FILE_ATTRIBUTE_NORMAL,
                       NULL);
      if (dh == INVALID_HANDLE_VALUE)
      {
         TRACES(( "CreateFile error on '%s': %s\n", dspec, txNtLastError()));
         rc = DFS_NO_DEVICE;                    // to be refined
      }
   #elif defined (DOS32)
   #elif defined (UNIX)
   #else
      rc = DosOpen((PSZ) drive,
                   &dh,                         // file handle
                   &act,                        // action taken
                   0,                           // filesize
                   FILE_NORMAL,                 // no attributes
                   FILE_OPEN,                   // OpenFlag
                   OPEN_FLAGS_DASD      |       // OpenMode (direct)
                   OPEN_SHARE_DENYNONE,         // OpenMode (allow sharing)
                   0);                          // reserved
   #endif
   if (rc == NO_ERROR)                          // opened successfully
   {
      TRACES(("Logical Volume '%s' lock handle: %lu\n", drive, dh));
      si->LogLock = dh;
   }
   RETURN (rc);
}                                               // end 'dfstOpenVolLock'
/*---------------------------------------------------------------------------*/
#endif

#endif
