//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS disk structure definitions
//
// Author: J. van Wijk
//
// JvW  17-06-1995 Initial version, split off from HPFSDEF.H
// JvW  21-03-2012 Added GPT disk partitioning definitions
// JvW  03-06-2014 Added OS/2 DUMPFS definitions
//
#ifndef    DFSDISK_H
   #define DFSDISK_H

#ifndef SECTORSIZE
#define SECTORSIZE  512
#endif

// Mapping macro's from DFS SecCyl USHORT to sector and cylinder values
#define DFSC2SECTOR(sc) (ULONG)((sc)&0x3F)
#define DFSC2CYLIND(sc) (ULONG)(HIBYTE(sc)+(((USHORT)(LOBYTE(sc)&0xC0))<<2))
#define DFSCOMBINE(s,c) (ULONG)((((s) > 0x3f) || ((c) > 0x3ff)) ? 0xffff :   \
                       ((((c) & 0x00ff) << 8) | (((c) & 0x0300) >> 2) | (s)))

// Number of sectors to ULONG MiB value
#define DFSECT2MIB(sz,ss)  (ULONG) ((sz) / (ULN64)((ULN64) ((ULN64)1024*(ULN64)1024) / (ULN64) (ss)))

// (ULN64) Mb or Kb value to sectors
#define DFSMB2SECTS(mb,ss) (ULN64) ((mb) * (ULN64)((ULN64) ((ULN64)1024*(ULN64)1024) / (ULN64) (ss)))
#define DFSKB2SECTS(kb,ss) (ULN64) ((kb) * (ULN64)((ULN64) ((ULN64)1024            ) / (ULN64) (ss)))

typedef BYTE   SEC512[SECTORSIZE];

#define DFS32MAX           0xffffffffUL
#define DFS64MAX           0xffffffffffffffffULL
#define L32_NULL           DFS32MAX
#define L64_NULL           DFS64MAX

#define DFSECT_2GIB         (0x3fffff)          // #sects in 2   GiB, minus 1
#define DFSECT_4GIB         (0x7fffff)          // #sects in 4   GiB, minus 1
#define DFSECT_CD80          (1331200)          // #sects in 650 MiB, 80 min CD
#define DFSECT_BIGF         (0x040000)          // #sects in 128 MiB, big file (alloc progress)

#define LSN_BOOTR            0x00
#define LEN_BOOTR            0x40

#pragma pack(1)                                 // byte packing
typedef struct _DFSC
{                                               // gives sector position
   BYTE                Head;                    // Read/Write head
   USHORT              SecCyl;                  // Sector & cylinder number
} DFSC;

typedef struct _DFSPARTENTRY                    // partition table entry
{
   BYTE                Status;                  // partition status
   DFSC                FirstSector;             // First sector
   BYTE                PartitionType;           // Partition type
   DFSC                LastSector;              // Last sector
   ULONG               BootSectorOffset;        // Boot sector Offset
   ULONG               NumberOfSectors;         // Number of sectors
} DFSPARTENTRY;

typedef enum dfs_os_boot
{
   DFS_OS,                                      // generic OS boot record
   DFS_NT,                                      // NT type OS boot record
   DFS_F2,                                      // FAT32 type boot record
   DFS_EX,                                      // EFAT  type boot record
   DFS_G1                                       // GRUB1 type boot record
} DFS_OS_BOOT;                                  // end of enum "dfs_os_boot"

//- FsysValue used for different things by different filesystems
//- Originally documented as a Head value, it was usually 0
//- For later windows versions is signals a CHKDSK required status
//- For bootable HPFS and JFS on OS2 it is the driveletter (0x80 == C:)

#define BT_LABL_L      11
#define BT_TYPE_L       8
typedef struct dfsosinfo                        // generic OS bootrec info
{                                               // at offset 0x24 in bootsector
   BYTE                PhysDriveNr;             // 024 Drive number (80 for HD 1)
   BYTE                FsysValue;               // 025 head/CHKDSK required/drive
   BYTE                Signature;               // 026 bpb signature, 28, 29, 80
   ULONG               SerialNr;                // 027 volume serial number
   char                Label[BT_LABL_L];        // 02b volume label
   char                Type[BT_TYPE_L];         // 036 volume type
   USHORT              CodeStart;               // 03e Bootcode, two bytes
   BYTE                Code[11];                // 040 Code start (memcpy)
   BYTE                BmDriveNr1;              // 04b drive number BMGR 1st
   BYTE                BmDataHead;              // 04c Head-nr BM data sect
   USHORT              BmDataSecCyl;            // 04d Sec/Cyl BM data sect
   BYTE                BmValue1;                // 04f Unknown, usually v-2
   BYTE                BmValue2;                // 050 Unknown, usually 2
   USHORT              BmBootSecCyl;            // 051 Sec/Cyl BM boot sect
   BYTE                BmDriveNr2;              // 053 drive number BMGR 2nd
   BYTE                BmBootHead;              // 054 Head-nr BM boot sector
   BYTE                RestCode[309];           // 055 Rest Bootcode, bytes
} DFSOSINFO;                                    // 18a end of "dfsosinfo"

// Note JvW: Above gives warnings in GCC, since the 'Code' array is just 11 bytes
//           and stuff is copied into it until the end (Including RestCode)




#define DFSF2_NO_FAT_MIRROR        0x80         // 00=mirror (std) 80=no-mirror
#define DFSF2_FAT_MIRROR_MASK      0x0F         // bit-mask for ACTIVE FAT 0..F

typedef struct dfsf2info                        // FAT32 bootrec info
{                                               // at offset 0x24 in bootsector
   ULONG               FatSectors;              // 024 nr of sectors per FAT
   USHORT              extFlags;                // 028 F32 extended flags   (== 0)
   USHORT              fsVersion;               // 02A filesystem version   (== 0)
   ULONG               RootCluster;             // 02C Cluster nr Root Dir  (~~ 2)
   USHORT              FsInfoLsn;               // 030 Freespace info   LSN (== 1)
   USHORT              SpBootLsn;               // 032 Spare bootsector LSN (== 6)
   ULONG               longzero1;               // 034
   ULONG               longzero2;               // 038
   ULONG               longzero3;               // 03C
   BYTE                PhysDriveNr;             // 040 Drive number (80 for HD 1)
   BYTE                FsysValue;               // 041 head/CHKDSK required/drive
   BYTE                Signature;               // 042 bpb signature, 28, 29, 80
   ULONG               SerialNr;                // 043 volume serial number   @0x43
   char                Label[BT_LABL_L];        // 047 volume label
   char                Type[BT_TYPE_L];         // 052 volume type      (BPB end)
   USHORT              CodeStart;               // 05A Bootcode, start
   ULONG               Code[75];                // 05C Bootcode, in ULONG's
   USHORT              CodeEnd;                 // 188 Bootcode, upto BootMgr
} DFSF2INFO;                                    // 18A end of "dfsf2info"


// CodeEnd value as seen on EFAT formatted by MAC OSX, other values are Windows
#define DFSEX_MACOSX_CODE_END           0xf4f4

typedef struct dfsexinfo                        // EFAT bootrec info
{                                               // at offset 0x24 in bootsector
   ULONG               fil1[7];                 // 024 filler to real info
   ULN64               PartitionOffset;         // 040 from diskstart! or 0
   ULN64               VolumeLength;            // 048 FS size in sectors
   ULONG               FatOffset;               // 050 Start FAT, sectors
   ULONG               FatLength;               // 054 Size  FAT, sectors
   ULONG               ClHeapOffset;            // 058 Clusterheap offset
   ULONG               ClusterCount;            // 05c Cluster count sectors
   ULONG               RootCluster;             // 060 Cl-nr for Root-DIR
   ULONG               SerialNr;                // 064 volume serial number
   BYTE                FsRevMinor;              // 068 FS minor version (00)
   BYTE                FsRevMajor;              // 069 FS major version (01)
   USHORT              VolumeFlags;             // 06A Volume flag bits
   BYTE                BpsShift;                // 06C Bytes per sector shift
   BYTE                SpcShift;                // 06D Sectors/Cluster  shift
   BYTE                NrOfFats;                // 06E Nr of FATs/Bitmaps (1)
   BYTE                BiosDrive;               // 06F Ext-int-13 drive 0x80
   BYTE                PercentUsed;             // 070 Inuse percentage 0..99
   BYTE                fil2[7];                 // 071 filler to bootcode
   USHORT              Code[70];                // 078 Bootcode, in ULONG's
   USHORT              CodeEnd;                 // 188 Bootcode, upto BootMgr
} DFSEXINFO;                                    // 18a end of "dfsexinfo"


typedef struct dfsntinfo                        // specific NT bootrec info
{                                               // at offset 0x24 in bootsector
   BYTE                PhysDriveNr;             // 024 Drive number (80 for HD 1)
   BYTE                FsysValue;               // 025 head/CHKDSK required/drive
   BYTE                Signature;               // 026 bpb signature, 28, 29, 80
   BYTE                fil1;                    // 027 unknown
   ULN64               VolSize;                 // 028 volume size, in sectors
   ULN64               MftClus;                 // 030 MFT cluster
   ULN64               MftCopy;                 // 038 MFT copy cluster
   ULONG               MftSize;                 // 040 MFT record size (clust/byte)
   ULONG               DirSize;                 // 044 DIR (INDX) size (clust/byte)
   ULN64               SerialNr;                // 048 Volume serial, 64 bit
   ULONG               Checksum;                // 050 bootsector checksum   (ZERO)
   char                fil4[307];               // 054 filler, upto BootMgr info
} DFSNTINFO;                                    // 18A end of "dfsntinfo"

// Note: MftSize and DirSize are estimated guesses, it is likely that the
//       value is clusters/record if positive and indicates a power of 2
//       when negative. example 0x000000f6 ==> minus 10 ==> 2^10 == 1024 bytes


typedef struct dfsg1info                        // specific GRUB stage 1
{                                               // at offset 0x24 in bootsector
   BYTE                grFill[26];              // 024 Zeroed in GRUB bootsectors
   BYTE                grMajor;                 // 03E version GRUB installer
   BYTE                grMinor;                 // 03F
   BYTE                grBootDisk;              // 040 boot disk, 0xFF = BIOS
   BYTE                grForceLBA;              // 041 stage1 use LBA
   USHORT              grStage2Addr;            // 042 stage2 load address
   ULONG               grStage2Psn;             // 044 stage2 sector number
   USHORT              grStage2Segm;            // 048 stage2 segment number
   char                fil4[320];               // 04A filler, upto BootMgr
} DFSG1INFO;                                    // 18A end of "dfsg1info"

#define GR_MSG_STAGE   "Loading stage"
#define GR_IDN_STAGE15  0xffffffff

#define GR_SEG_STAGE15 0x220                    // GRUB seg# to load stage 1.5
#define GR_SEG_STAGE2  0x820                    // GRUB seg# to load stage 2

typedef struct dfsg2info                        // specific GRUB stage 1.5 or 2
{
   BYTE                grCode1[504];            // 000 first sector of code + filler
   ULONG               grStage2Self;            // 1f8 PSN of rest, next sect
   USHORT              grStage2Sect;            // 1fc Number of stage2 sects
   USHORT              grStageXSegm;            // 1fe Segment# for loading
   BYTE                grFill[6];               // 200 unknown
   BYTE                grMajor;                 // 206 version GRUB installer
   BYTE                grMinor;                 // 207
   ULONG               grInstPart;              // 208 install partition
   ULONG               grSavedEnt;              // 20c saved entry number
   BYTE                grStage2Id;              // 210 ID byte (0x10 ?)
   BYTE                grFill2;                 // 211
   char                grVersion[5];            // 212 GRUB version null-term
   char                grStage2name[89];        // Stage2 names, and 1.5 part#
   BYTE                grCode2[400];            // rest of code sector2   @x270
} DFSG2INFO;                                    // end of struct "dfsg2info"

//- Note: For stage 1.5, usually located directly after the MBR sector, the 3rd
//-       byte after the grVersion string the partition number in GRUB form,
//-       so '06' would be /dev/xda7 where the GRUB config files are located

typedef struct dfswapblock1                     // Linux SWAP area, type 2
{                                               // type 1 only has signature
   BYTE                swLead[1024];            // leader, empty or bootloader
   ULONG               swVersion;               // version (01) little-endian
   ULONG               swBlocks;                // net swap space in blocks
   ULONG               swBadCount;              // nr of bad blocks in swap
   ULONG               swPad[125];              // empty upto bad-block array
   ULONG               swBadBlock[637];         // bad-block array, block nrs
   USHORT              swTail;                  // fill upto signature
   BYTE                swSignature[10];         // SWAP signature bytes   @xff6
} DFSWAPBLOCK1;                                 // end of struct "dfswapblock1"


#define BT_BM_BOOT      0x01                    // bootable flag old BMGR
#define BT_BM_INST      0x04                    // installable flag BMGR
#define BT_BM_L         8
#define BT_SYST_L       8
#define SV_BOOTR        (USHORT) 0xaa55
typedef struct s_bootr                          // Boot sector
{
   BYTE                Instruction[3];          // 000 first bootcode instruction
   char                OpSys[BT_SYST_L];        // 003 source operating system
   TXFS_EBPB           eb;                      // 00B extended BPB
   union                                        // 024 OS or FS specific info
   {
      DFSOSINFO        os;                      // 024 generic  OS    info
      DFSNTINFO        nt;                      // 024 specific NT    info
      DFSF2INFO        f2;                      // 024 specific FAT32 info
      DFSEXINFO        ex;                      // 024 specific EFAT info
      DFSG1INFO        g1;                      // 024 specific GRUB  stage1
   };
   char                BootMgrFlag;             // 18A Part for EBR on menu
   char                BootMgrName[BT_BM_L +1]; // 18B Null terminated BootMgr name
   char                BootMgrNext[BT_BM_L +1]; // 184 Null term next EBR PSN ascii
   char                fil3[27];                // 18D filler, upto part table
   ULONG               NTsignature;             // 1B8 NT Disk Admin sign.
   USHORT              reserved;                // 1BC unknown
   DFSPARTENTRY        PartitionTable[4];       // 1BE partition table
   USHORT              Signature;               // 1FE
} S_BOOTR;                                      // end of struct "s_bootr"

// byte just before Signature is used by the eCS 2.0 Bootable JFS code
// Can access that using the sector as a byte array, and the correct index
#define JFSBOOTALTERNATELETTER    0x1fd

//- Note: Bootmanager data in EBR may actually be a 4-item array with
//-       a 1 character Flag and 8 byte name, but since just 2 entries
//-       are used, and the flag for the 'Next' link is always zero,
//-       the above simplified definitions works too, and has the advantage
//        that the name strings are zero terminated


#define SV_BOOT21      0x41615252               // 1st boot2 signature: RRaA
#define SV_BOOT22      0x61417272               // 2nd boot2 signature: rrAa
typedef struct fat32b2
{
   ULONG               Signatur1;
   ULONG               fill1[ 120];             // 480 bytes
   ULONG               Signatur2;
   ULONG               FreeClusters;            // nr of free clusters
   ULONG               NextSearchCl;            // to start searching ...
   USHORT              fill2[ 7];               // 14 bytes
   USHORT              Signature;               // AA55 signature
} FAT32B2;                                      // end of struct "fat32b2"


#define SV_DUMPS1      0x00000000               // 1st boot signature: 0
#define SV_DUMPS2      0x00000001               // 2nd boot signature: 1
#define SV_DUMPS3      0x0000000b               // 3rd boot signature: B
#define SV_DUMPS4      0x00000000               // 4th boot signature: 0
typedef struct dumpfsinfo
{
   ULONG               Signatur1;               // Startaddress, must be zero
   ULONG               DumpSize1;               // End = dumpfile size in bytes
   ULONG               Signatur2;               // flag? value 0x00000001
   ULONG               Signatur3;               // flag? value 0x0000000b
   BYTE                tmMinutes;
   BYTE                tmHours;                 // time values in BCD
   BYTE                tmZero;                  // show using %2.2hx
   BYTE                tmSeconds;
   BYTE                dtYears;
   BYTE                dtCentury;               // date values in BCD
   BYTE                dtDay;                   // show using %2.2hx
   BYTE                dtMonth;
   ULONG               PartSects;               // Partition Sectors, minus 1
   ULONG               DumpSize2;               // dumpfile size in bytes
   ULONG               fill1[ 119];             // 476 bytes
   ULONG               Signatur4;               // must be zero
} DUMPFSINFO;                                   // end of struct "dumpfsinfo"

typedef struct s_eablk                          // Generic definition of EA
{                                               // for HPFS, FAT and JFS
   BYTE                Flag;                    // EA flag, type of EA
   BYTE                NameLength;              // length of name excl \0
   USHORT              DataLength;              // length of EA data area
   char                Name[1];                 // ASCIIZ EA name
                                                // Followed by EA data area
} S_EABLK;                                      // end of struct "s_eablk"

//- structure compatible with the OS/2 toolkit FEA2, including field names
typedef struct s_fea2
{
   ULONG               oNextEntryOffset;        // offset to next EABLK
   BYTE                fEA;
   BYTE                cbName;
   USHORT              cbValue;
   char                szName[1];
} S_FEA2;                                       // end of struct "s_fea2"

//- structure compatible with the OS/2 toolkit FEA2LIST/PFEA2LIST, including field names
typedef struct s_fea2list
{
   ULONG               cbList;                  // total size of EA-list, bytes
   S_FEA2              fea2[1];                 // array of EA's
} S_FEA2LIST;                                   // end of struct "s_fea2list"


#define DFS_GUID_LENGTH 16                      // length of GUID fields

typedef BYTE           DFS_GUID[ DFS_GUID_LENGTH];

#endif
