//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Generic Dialog functions, called from the menu and generic commands
//
// Author: J. van Wijk
//
// JvW  26-01-2005 Initial version
//

#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfsupart.h>                           // FS partition utilities
#include <dfsufgpt.h>                           // GPT utility functions
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsver.h>                             // DFS version and naming
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfsectio.h>                           // DFS store I/O private defs
#include <dfstable.h>                           // SLT utility functions
#include <dfswin.h>                             // window and help definitions
#include <dfstdlg.h>                            // generic dialogs definitions
#include <dfsdgen.h>                            // generic dialogs interface

//- Window styles used with the Dialog Widgets
//- DFSDM is 'moving' DFSDS' is sizing and DFSD2 is half-sizing, all horizontal

static TXTM        fdescr = "";                 // FROM object/store description
static TXTM        tdescr = "";                 // TO   object/store description


// Callback signaling a change in the target path (including driveletter/vol)
static void dfsImageCbTargetPathChange
(
   char              **newPath                  // IN    ptr to new full PATH string
);


// Dialog window procedure, for the Restore/Compare/Verify button widgets
static ULONG dfsRcvWinProc                      // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Dialog window procedure, for the clone from-list widget field
static ULONG dfsCloneWinProc                    // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Dialog window procedure, for the SLT-display dialog field; update status
static ULONG dfsSltWinProc                      // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Dialog window procedure, for the GEOMETRY dialog, set dependant fields
static ULONG dfsGeoWinProc                      // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Dialog window procedure, for the UUID/GUID dialog, set dependant fields
static ULONG dfsGuidUuidWinProc                 // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

// Dialog window procedure, for UID dialog fields, fill out fields to length
static ULONG dfsUidFieldsWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

// Calculate GEO values to be displayed from RADIO settings and global GEO
static void dfsGeoSetControls
(
   void
);

// Determine RADIO button to be the selected one for current geometry
static void dfsGeoSetRadios
(
   void
);

/*========================== FIND ===============================================================*/
static BOOL fdwr1  = TRUE;                      // Radio Every sector
static BOOL fdwr2  = FALSE;                     // Radio Cylinder boundaries
static BOOL fdwr3  = FALSE;                     // Radio Freespace only
static BOOL fdwr4  = FALSE;                     // Radio Allocated only

static BOOL fdwr5  = TRUE;                      // Radio Ascii
static BOOL fdwr6  = FALSE;                     // Radio Unicode
static BOOL fdwr7  = FALSE;                     // Radio Hex
static BOOL fdwr8  = FALSE;                     // Radio Mixed

static BOOL fdwrS  = FALSE;                     // Radio Single shot
static BOOL fdwrR  = TRUE;                      // Radio Repeat search
static BOOL fdwrM  = FALSE;                     // Radio Multiple hits/sector

static BOOL fdwc1  = FALSE;                     // Check Case-sensitive
static BOOL fdwc2  = TRUE;                      // Check No sector span
static BOOL fdwc3  = FALSE;                     // Check Search backward
static BOOL fdwc4  = FALSE;                     // Check NOT containing
static BOOL fdwc5  = FALSE;                     // Check Verbose output
static BOOL fdwc6  = FALSE;                     // Check Start at NEXT
static BOOL fdwc7  = TRUE;                      // Show search args

static TXTS fdwe0  = "Current";                 // startsector for find -f:
static TXTS fdwe1  = "";                        // position for 1st argument
static TXTS fdwe2  = "";                        // sector types
static TXTS fdwe3  = "";                        // sector step value
static TXTM fdwe4  = "";                        // string for 2nd argument

/*
(*) Ascii or UTF-8 (*) Repeat, 1 hit/sector  (*) Search in every sector
( ) Unicode  16bit ( ) Repeat multiple hits  ( ) On Cylinder boundaries
( ) Hex pairs      ( ) Search once, display  ( ) In freespace (undelete)
( ) Mixed string   [ ] Case-sensitive match  ( ) In allocated (filegrep)
[ ] Verbose output [ ] Search backwards      [ ] Start at NEXT/PREV sect
[x] Show arguments [ ] NOT containing ...    [x] No sector span (faster)

 StartOffset  @Position  Types    Step    2ndary string, case-sensitive
[...........][.........][.....][........][.............................]
*/
#define   DFSDGFINDWIDGETS 23
#define   DFSDGFINDCVERBOS  4
#define   DFSDGFINDRREPEAT  6
#define   DFSDGFINDRRMULTI  7

static TXWIDGET  dfsDgFindWidgets[DFSDGFINDWIDGETS] =  // order determines TAB-order!
{
   {0,  0, 1, 18, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr5, "Ascii or UTF-8")},
   {1,  0, 1, 18, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr6, "Unicode  16bit")},
   {2,  0, 1, 18, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr7, "Hex pairs")},
   {3,  0, 1, 18, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr8, "Mixed string")},
   {4,  0, 1, 18, 0, 5, 0, TXWS_AUTOCHK, 0, TXStdButton( &fdwc5, "Verbose output")},
   {5,  0, 1, 18, 0, 5, 0, TXWS_AUTOCHK, 0, TXStdButton( &fdwc7, "Show arguments")},

   {0, 19, 1, 25, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwrR, "Repeat, 1 hit/sector")},
   {1, 19, 1, 25, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwrM, "Repeat multiple hits")},
   {2, 19, 1, 25, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwrS, "Search once, display")},
   {3, 19, 1, 25, 0, 6, 0, TXWS_AUTOCHK, 0, TXStdButton( &fdwc1, "Case-sensitive match")},
   {4, 19, 1, 25, 0, 6, 0, TXWS_AUTOCHK, 0, TXStdButton( &fdwc3, "Search backwards")},
   {5, 19, 1, 25, 0, 6, 0, TXWS_AUTOCHK, 0, TXStdButton( &fdwc4, "NOT containing ...")},


   {0, 45, 1, 28, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr1, "Search in every sector" )},
   {1, 45, 1, 28, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr2, "On Cylinder boundaries" )},
   {2, 45, 1, 28, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr3, "In freespace (undelete)")},
   {3, 45, 1, 28, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &fdwr4, "In allocated (filegrep)")},
   {4, 45, 1, 28, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &fdwc6, "Start at NEXT/PREV sect")},
   {5, 45, 1, 28, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &fdwc2, "No sector span (faster)")},

   {7,  0, 2, 13, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fdwe0, TXMAXTS, "StartOffset")},
   {7, 13, 2, 11, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fdwe1, TXMAXTS, "@Position")},
   {7, 24, 2,  7, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fdwe2, TXMAXTS, "Types")},
   {7, 31, 2, 10, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( fdwe3, TXMAXTS, "  Step")},
   {7, 41, 2, 31, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD_SIZE, 0,
                           TXStdEntryf( fdwe4, TXMAXTM, "2ndary string, case-sensitive")}
};

static TXGW_DATA dfsDgFindDlg =
{
   DFSDGFINDWIDGETS,                            // number of widgets
   DFSC_FINDTC,                                 // dialog help
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgFindWidgets                             // array of widgets
};

static TXLN   findarg1 = "";                    // first search argument


/*************************************************************************************************/
// Present FIND options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsFindDialog
(
   BOOL                single,                  // IN    allow single-shot only
   BOOL                quiet,                   // IN    no result sector display
   BOOL                again,                   // IN    skip dialog if valid args
   ULN64               start,                   // IN    starting LSN or L64_NULL
   ULONG               spos                     // IN    start offset or 0
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXTS                optstr;
   TXLN                escaped;                 // quote-escaped copy
   TXLN                escvars;                 // $-escaped copy (variables)
   BOOL                repeat = again && (strlen(findarg1) ||
                                          strlen(fdwe4)    ||
                                          strlen(fdwe2));
   ENTER();

   if (single)
   {
      fdwrS  = TRUE;                            // Radio Single shot
      fdwrR  = FALSE;                           // Radio Repeat search
      fdwrM  = FALSE;                           // Radio Multiple hits/sector

      dfsDgFindWidgets[ DFSDGFINDCVERBOS].flags |=  TXWI_DISABLED;
      dfsDgFindWidgets[ DFSDGFINDRREPEAT].flags |=  TXWI_DISABLED;
      dfsDgFindWidgets[ DFSDGFINDRRMULTI].flags |=  TXWI_DISABLED;
   }
   else                                         // enable radio buttons
   {
      dfsDgFindWidgets[ DFSDGFINDCVERBOS].flags &= ~TXWI_DISABLED;
      dfsDgFindWidgets[ DFSDGFINDRREPEAT].flags &= ~TXWI_DISABLED;
      dfsDgFindWidgets[ DFSDGFINDRRMULTI].flags &= ~TXWI_DISABLED;
   }
   if (start != L64_NULL)
   {
      sprintf( fdwe0, "0x%llx", start);
   }

   dfsBEGINWORK();                              // signal work starting
   while (repeat || txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &dfsDgFindDlg,
         "Specify the primary ASCII, UNICODE, HEX or Mixed string "
         "to search the sectors for. Adjust the search options to "
         "your liking, for usage consult online-help with <F1> or "
         "the 'FIND' command documentation\n\n"
         " Primary search string, format and case-sensitivity set by the options",
         " Specify search string(s) and options ", DFSC_FINDTC,
         TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER, 72, findarg1)
      != TXDID_CANCEL)
   {
      if (strlen(findarg1) || strlen(fdwe4) || strlen(fdwe2))
      {
         strcpy( command, "find -o:");

         //- Use 'escaped' $ to avoid variable substitute, and no ASCII compare if HEX!
         if (!fdwc1 && !fdwr7) strcat( command, "$$");
         if ( fdwc2)           strcat( command, "%");
         if ( fdwc3)           strcat( command, "-");
         if ( fdwc4)           strcat( command, "^");
         if ( quiet)
         {
                               strcat( command, "Q"); // no auto sector display
         }
         else
         {
            if      ( fdwrS)   strcat( command, "+"); // always verbose
            else if ( fdwc5)   strcat( command, "+"); // optional verbose for repeat
         }
         if (!single)
         {
            if ( fdwrR)        strcat( command, "*");
            if ( fdwrM)        strcat( command, "*!M");
         }
         if ( fdwr2)           strcat( command, "~");
         if (!fdwc7)           strcat( command, "!");
         if ( fdwr3)           strcat( command, "[");
         if ( fdwr4)           strcat( command, "]");

         if ((strlen(fdwe1) != 0) && (isdigit(fdwe1[0]))) // valid position
         {
            strcat( command, "@");
            strcat( command, fdwe1);
         }
         if (fdwc6)
         {
            strcat( command, " -n");
         }
         if (strlen(fdwe2) != 0)                // sector types specified
         {
            strcat( command, " -t:'");
            strcat( command, fdwe2);
            strcat( command, "'");
         }
         if (strlen(fdwe3) != 0)                // sector step value
         {
            strcat( command, " -i:");
            strcat( command, fdwe3);
         }
         if (strlen(fdwe4))                     // 2nd search string
         {
            TxRepStr( fdwe4,   '"', "\\\"", escaped, TXMAXLN); // escape double quotes
            TxRepStr( escaped, '$', "$$",   escvars, TXMAXLN); // escape variable $
            if ( fdwr5) strcat( command, " -A:\"");
            if ( fdwr6) strcat( command, " -U:\"");
            if ( fdwr7) strcat( command, " -H:\"");
            if ( fdwr8) strcat( command, " -M:\"");
            strcat(             command, escvars);
            strcat(             command, "\"");
         }
         if (strlen(findarg1))                  // 1st search string
         {
            TxRepStr( findarg1, '"', "\\\"", escaped, TXMAXLN); // escape double quotes
            TxRepStr( escaped,  '$', "$$",   escvars, TXMAXLN); // escape variable $
            if ( fdwr5) strcat( command, " -a:\"");
            if ( fdwr6) strcat( command, " -u:\"");
            if ( fdwr7) strcat( command, " -h:\"");
            if ( fdwr8) strcat( command, " -m:\"");
            strcat(             command, escvars);
            strcat(             command, "\"");
         }
         if ((strlen( fdwe0) != 0) &&           // Start value specified
             (isxdigit( fdwe0[0]))  )           // and it is a number
         {
            TxStrip( fdwe0, fdwe0, ' ', ' ');
            strcat(  command, " -f:");
            strcat(  command, fdwe0);
         }
         if (repeat)                            // start from next sector
         {
            strcat(  command, " -n");
         }
         else if (spos != 0)                      // explicit start offset
         {
            sprintf( optstr, " -n:0x%x", (fdwc3) ? spos -1 : spos +1);
            strcat(  command, optstr);
         }
         if (quiet)
         {
            rc = dfsMultiCommand( command, 0, TRUE, FALSE, FALSE);
         }
         else
         {
            TxCancelAbort();
            txwAdd2History( dfsa->cmdwindow, command);
            rc = dfsMultiCommand( command, 0, TRUE, TRUE, TRUE);
         }
         break;                                 // out of while ...
      }
      else
      {
         TxNamedMessage( TRUE, DFSC_FINDTC, " ERROR: No search argument ",
                    "You must specify at least one search argument or a sector Type ...");
      }
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsFindDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== RUN SCRIPT =========================================================*/
static BOOL rsVerbose = FALSE;                  // Verbose
static BOOL rsStep    = FALSE;                  // SingleStep

/*
[ ] Verbose, display each script line when executed
[ ] Single step, confirm before executing each line
*/
#define   DFSDGRSWIDGETS 2
static TXWIDGET  dfsDgRsWidgets[DFSDGRSWIDGETS] = // order determines TAB-order!
{
   {0,  0, 1, 58, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( &rsVerbose,
                          "Verbose, display each script line when executed")},
   {1,  0, 1, 58, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( &rsStep,
                          "Single step, confirm before executing each line")}
};

static TXGW_DATA dfsDgRsDlg =
{
   DFSDGRSWIDGETS,                              // number of widgets
   DFSC_RUNS,                                   // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgRsWidgets                               // array of widgets
};

/*************************************************************************************************/
// Present Run-script file-dialog with options and execute resulting command
/*************************************************************************************************/
ULONG dfsRunScriptDialog
(
   char               *firstParam,              // IN    path/scriptname, or empty
   char               *scriptInfo               // OUT   scriptname + parameters
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                params;
   TXLN                fspec;
   TX1K                dlgText;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   // Handle input options when specified
   if (TxaOptSet('s'))                          // single-step option used
   {
      rsStep = TxaOption('s');
   }
   if (TxaOptSet('v'))                          // verbose option used
   {
      rsVerbose = TxaOption('v');
   }

   strcpy( fspec, firstParam);                  // Specified (partial) name
   if (strchr( fspec, '.') == NULL)             // no extension
   {
      strcat( fspec, "*");                      // add wildcard
   }
   TxFnameExtension( fspec, "dfs");             // add default extension

   if (txwOpenFileDialog( fspec, NULL, NULL, DFSC_RUNS, NULL,
      (dfsa->expertui) ? &dfsDgRsDlg : NULL, " Select DFSee script file to RUN ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      sprintf(scriptInfo, "%s%s'%s'", (rsVerbose) ? "-v " : "",
                                      (rsStep)    ? "-s " : "", fspec);

      TxsValidateScript( fspec, NULL, params, NULL); // get description in params
      if (strstr( params, "no-parameters") == NULL)
      {
         if (strlen( params) != 0)
         {
            sprintf( dlgText, "%s\n\nParameters enclosed in [] are "
                     "optional, others are mandatory.\n%s", fspec, params);
         }
         else
         {
            sprintf( dlgText, "%s\n\nSpecify additional parameters for "
                     "the script or just leave as is ...", fspec);
         }
                                    // Get the parameters specified on the commandline, for editing
         TxaGetArgString( TXA_CUR, 2, TXA_ALL, TXMAXLN, params);
         if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
               dlgText, " Specify parameter(s) for the script ", DFSC_RUNS,
               TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
               50, params) != TXDID_CANCEL)
         {
            strcat( scriptInfo, " ");
            strcat( scriptInfo, params);
         }
         else                                   // ESC on parameter prompt
         {                                      // cancel script execution
            strcpy( scriptInfo, "");
         }
      }
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsRunScriptDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== IMPORT =============================================================*/
static BOOL impdwc1 = FALSE;                    // Include data sectors

/*
[ ] Import each DATA sector from .BIN/.BN? file too
*/
#define   DFSDGIMPWIDGETS 1
static TXWIDGET  dfsDgImpWidgets[DFSDGIMPWIDGETS] = // order determines TAB-order!
{
   {0,  0, 1, 58, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( &impdwc1,
                          "Import each DATA sector from .BIN/.BN? file too")}
};

static TXGW_DATA dfsDgImpDlg =
{
   DFSDGIMPWIDGETS,                             // number of widgets
   DFSC_IMPLIST,                                // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgImpWidgets                              // array of widgets
};

/*************************************************************************************************/
// Present Import options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsImportDialog
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;

   ENTER();

   dfsBEGINWORK();                              // signal work starting
   strcpy( fspec, "*.sn?;*.lsn;*.lst");
   if (txwOpenFileDialog( fspec, NULL, NULL, DFSC_IMPLIST, NULL, &dfsDgImpDlg,
       " Select the sector-list file to be imported ", fspec))
   {
      sprintf(    command, "import \"%s\"%s", fspec, (impdwc1) ? " -d" : "");
      dfsExecCmd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsImportDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== EXPORT =============================================================*/
static BOOL expdwc1 = FALSE;                    // Include data sectors

/*
[ ] Export each DATA sector to a .BIN/.BN? file too
*/
#define   DFSDGEXPWIDGETS 1
static TXWIDGET  dfsDgExpWidgets[DFSDGEXPWIDGETS] = // order determines TAB-order!
{
   {0,  0, 1, 58, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( &expdwc1,
                          "Export each DATA sector to a .BIN/.BN? file too")}
};

static TXGW_DATA dfsDgExpDlg =
{
   DFSDGEXPWIDGETS,                             // number of widgets
   DFSC_EXPLIST,                                // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgExpWidgets                              // array of widgets
};

/*************************************************************************************************/
// Present Export options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsExportDialog
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;

   ENTER();

   dfsBEGINWORK();                              // signal work starting
   strcpy( fspec, "*.sn?;*.lsn");
   if (txwSaveAsFileDialog( fspec, NULL, NULL, DFSC_EXPLIST, NULL, &dfsDgExpDlg,
       " Specify file for exporting the sector-list to ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      sprintf(    command, "export \"%s\"%s", fspec, (expdwc1) ? " -d" : "");
      dfsExecCmd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsExportDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== IMAGE ==============================================================*/
#if defined (DOS32)
   static BOOL     simr1  = FALSE;              // Single image-file
   static BOOL     simr2  = TRUE;               // Multi file, max size
#else
   static BOOL     simr1  = TRUE;               // Single image-file
   static BOOL     simr2  = FALSE;              // Multi file, max size
#endif
static BOOL        simr3  = FALSE;              // Multi file removable

static BOOL        simr4  = TRUE;               // Smart compr
static BOOL        simr5  = FALSE;              // Compressed
static BOOL        simr6  = FALSE;              // RAW image

static BOOL        simr7  = TRUE;               // Buffer default, opt speed
static BOOL        simr8  = FALSE;              // Buffer max DOS compatible
static BOOL        simr9  = FALSE;              // Buffer max for IMZ index

static BOOL        simc1  = TRUE;               // Generate index file
static BOOL        simc2  = FALSE;              // Ignore errors
static BOOL        simc3  = TRUE;               // HEX sector values

static TXTS        sime1  = "2047";             // Multiple max size
static BOOL        simrk  = FALSE;              // KiB
static BOOL        simrm  = TRUE;               // MiB
static BOOL        simrg  = FALSE;              // GiB

static TXTS        sime2  = "";                 // From
static TXTS        sime3  = "";                 // Size
static BOOL        simrK  = FALSE;              // KiB
static BOOL        simrM  = FALSE;              // MiB
static BOOL        simrG  = FALSE;              // GiB
static BOOL        simrS  = TRUE;               // Sec

/*
Create FROM
Partition :[01 /dev/sda1    ARCAOS      D: Pri 0x07 JFS        2047.0 MiB V]

 From Part: [04 /dev/hda7    ECS12       F: Log 07 HPFS        917.7 MiB ]

[x] Generate browse INDEX   [ ] Ignore errors       Set BUFFER size for:
(*) Single file, unlimited  (*) Smart compress  (*) Speed + SMART efficiency
( ) MultiFile on same disk  ( ) Compressed      ( ) Small IDX DOS compatible
( ) MultiFile media change  ( ) RAW image       ( ) Disk geometry track size

 First sector     Image size       [x] Sector values are HEX, not decimal
[0.............] [Whole Area    ]  (*) Sectors  ( ) KiB  ( ) MiB  ( ) GiB
                  Filesize limit : [650.......] ( ) KiB  () MiB  ( ) GiB
*/
#define   DFSDGSIMWIDGETS 28
#define   DFSDGSIMWBASICS 12
#define   DFSDGSIMT1      1                     // index of description
#define   DFSDGSIMT2      2                     // index of leader
#define   DFSDGSIML1      3                     // index of list-1
#define   DFSDGSIMR1      6                     // index of simr1
#define   DFSDGSIMR2      7                     // index of simr2
#define   DFSDGSIMR3      8                     // index of simr3
#define   DFSDGSIIGN      5                     // index of Ignore errors
static TXWIDGET  dfsDgSimWidgets[DFSDGSIMWIDGETS] =    // order determines TAB-order!
{
   {0,  0, 1, 11, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "Create FROM")},

   {1, 12, 1, 65, 1, 0, 1, DFSDSOUTPUT,  0, TXStdStline( fdescr)},
   {1,  0, 1, 11, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( NULL)},
   {1, 11, 1, 65, 0, 0, 0, DFSDSLISTBOX, 0, TXStdLstBox( NULL, 0, NULL)},

   {3,  0, 1, 25, 0, 6, 0, TXWS_AUTOCHK, 0, TXStdButton( &simc1, "Generate browse INDEX")},
   {3, 28, 1, 18, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &simc2, "Ignore errors")},

   {4,  0, 1, 28, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &simr1, "Single file, unlimited")} ,
   {5,  0, 1, 28, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &simr2, "MultiFile on same disk")},
   {6,  0, 1, 28, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &simr3, "MultiFile media change")},

   {4, 28, 1, 18, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &simr4, "Smart compress")},
   {5, 28, 1, 18, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &simr5, "Compressed")},
   {6, 28, 1, 18, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &simr6, "RAW image")},

   {3, 52, 1, 25, 0, 0, 1, DFSDSOUTPUT,  0, TXStdStline( "Set BUFFER size for:")},
   {4, 48, 1, 30, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &simr7, "Speed + SMART efficiency")},
   {5, 48, 1, 30, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &simr8, "Small IDX DOS compatible")},
   {6, 48, 1, 30, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &simr9, "Disk geometry track size")},

   {8, 35, 1, 38, 0, 0, 0, DFSDMAUTOCHK, 0, TXStdButton( &simc3, "Sector values are HEX, not decimal")},
   {8,  0, 2, 16, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD2SIZE,
                                         0, TXStdEntryf( sime2, TXMAXTS, "First sector")},
   {8, 17, 2, 16, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD2SIZE |  TXWS_HCHILD2MOVE,
                                         0, TXStdEntryf( sime3, TXMAXTS, "Image size")},

   {9, 35, 1, 11, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &simrS, "Sectors")},
   {9, 48, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &simrK, "KiB")},
   {9, 57, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &simrM, "MiB")},
   {9, 66, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &simrG, "GiB")},

  {10, 18, 1, 16, 0, 0, 0, DFSD2OUTPUT,  0, TXStdStline( "Filesize limit :")},
  {10, 35, 1, 12, 0, 0, 0, TXWS_ENTRYB   |  TXWS_HCHILD2SIZE |  TXWS_HCHILD2MOVE,
                                         0, TXStdEntryf( sime1, TXMAXTS, "")},
  {10, 48, 1,  7, 0, 5, 0, DFSDMAUTORAD, 0, TXStdButton( &simrk, "KiB")},
  {10, 57, 1,  7, 0, 5, 0, DFSDMAUTORAD, 0, TXStdButton( &simrm, "MiB")},
  {10, 66, 1,  7, 0, 5, 0, DFSDMAUTORAD, 0, TXStdButton( &simrg, "GiB")}
};

static TXGW_DATA dfsDgSimDlg =
{
   DFSDGSIMWIDGETS,                             // number of widgets
   DFSC_IMAGDLG,                                // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgSimWidgets                              // array of widgets
};


/*************************************************************************************************/
// Present IMAGE options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsImageDialog
(
   DFSOBJECT           source,                  // IN    Source for the image
   char               *image,                   // IN    filename, "" or NULL
   BOOL                media,                   // IN    set media-change
   ULONG               mfsize,                  // IN    max filesize in MiB
   ULN64               tsector,                 // IN    To   sector number
   ULN64               sizetodo                 // IN    Size to image (0 = all)
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;
   TXTT                option;
   TXSELIST           *list = NULL;
   TXTM                text;

   ENTER();

   TRACES(( "IMAGE to:'%s'  mfsize: %u start position:0x%llx  size:0x%llx\n",
            (image) ? image : "NULL", mfsize, tsector, sizetodo));

   dfsBEGINWORK();                              // signal work starting

   if (dfsa->expertui)                          // set available widgets Expert/Basic
   {
      dfsDgSimDlg.count = DFSDGSIMWIDGETS;                  //- show ALL controls
      dfsDgSimWidgets[DFSDGSIIGN].flags  &= ~TXWI_DISABLED; //- enable 'ignore errors'
   }
   else
   {
      dfsDgSimDlg.count = DFSDGSIMWBASICS;                  //- remove all after 'RAW'
      dfsDgSimWidgets[DFSDGSIIGN].flags  |=  TXWI_DISABLED; //- disable 'ignore errors'
   }

   if (media)
   {
      simr1 = FALSE;
      simr2 = FALSE;
      simr3 = TRUE;                             // force media-change set
   }
   if (mfsize != 0)                             // maximum filesize specified
   {
      sprintf( sime1, "%u", mfsize);            // value in ASCII
      simrk  = FALSE;                           // KiB
      simrm  = TRUE;                            // MiB
      simrg  = FALSE;                           // GiB
   }
   sprintf( sime2, "%llx", tsector);            // set start sector number
   if (sizetodo != 0)
   {
      sprintf( sime3, "%llx", sizetodo);        // set Size to process (0 = all)
   }
   else
   {
      strcpy(  sime3, "Whole Area");
   }

   dfsDgSimWidgets[DFSDGSIMT1].flags  |=  TXWI_DISABLED; // text field OFF
   dfsDgSimWidgets[DFSDGSIML1].flags  &= ~TXWI_DISABLED; // list field ON
   switch (source)
   {
      case DFSO_VOLD:
         TRACES(( "Volume, list: %8.8x\n", dfsa->slVolumes));
         dfsDgSimWidgets[DFSDGSIMT2].sl.buf = "Dev/Volume:";
         dfsDgSimWidgets[DFSDGSIML1].title  = " Create image from volume ";
         if (dfsa->slVolumes == NULL)
         {
            dfsa->slVolumes = TxFsDriveSelist( TXFSV_HD | TXFSV_CD | TXFSV_FLOP, FALSE);
         }
         list = dfsa->slVolumes;
         TxSelStringSelect( list,  SINF->drive, 2);
         break;

      case DFSO_DISK:
         TRACES(( "Disk, list: %8.8x\n", dfsa->slDiskOne));
         dfsDgSimWidgets[DFSDGSIMT2].sl.buf = "Whole disk:";
         dfsDgSimWidgets[DFSDGSIML1].title  = " Create image from disk ";
         list = dfsa->slDiskOne;
         sprintf( text, "%2hu",    SINF->disknr);
         TxSelStringSelect( list,  text, 2);
         break;

      case DFSO_PART:
         TRACES(( "Disk, list: %8.8x\n", dfsa->slPartOne));
         dfsDgSimWidgets[DFSDGSIMT2].sl.buf = "Partition :";
         dfsDgSimWidgets[DFSDGSIML1].title  = " Create image from partition ";
         list = dfsa->slPartOne;
         sprintf( text, "%2.2hu", SINF->partid);
         TxSelStringSelect( list,  text, 2);
         break;

      default:
         if (source == DFSO_IMGF)
         {
            strcpy( fdescr, dfstStoreDesc2( DFSTORE) + 10);
            dfsDgSimWidgets[DFSDGSIMT2].sl.buf  = "(img) File:";
         }
         else                                   // other types
         {
            strcpy( fdescr, dfstStoreDesc1( DFSTORE) + 10);
            dfsDgSimWidgets[DFSDGSIMT2].sl.buf  = "Current   :";
         }
         dfsDgSimWidgets[DFSDGSIMT1].flags  &= ~TXWI_DISABLED; // text field ON
         dfsDgSimWidgets[DFSDGSIML1].flags  |=  TXWI_DISABLED; // list field OFF
         break;
   }
   dfsDgSimWidgets[DFSDGSIML1].lb.list = list;

   strcpy( fspec, "*.i??");
   if (txwSaveAsFileDialog( fspec, NULL, image, DFSC_IMAGDLG, dfsImageCbTargetPathChange, &dfsDgSimDlg,
       " Specify options and name for (1st) imagefile to CREATE ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      TxStrip( fspec, fspec, ' ', ' ');
      TxStrip( sime2, sime2, ' ', ' ');
      sprintf( command, "image \"%s\" %s%s",
               fspec, (simc3) ? "0x" : "", (strlen(sime2)) ? sime2 : "0");

      if ((strlen( sime3) != 0) &&              // Size value specified
          (isxdigit( sime3[0]))  )              // and it is a number
      {
         TxStrip( sime3, sime3, ' ', ' ');
         sprintf( option, " %s%s,%s", (simc3 && simrS) ? "0x" : "", sime3,
                 (simrK) ? "k" : (simrM) ? "m" : (simrG) ? "g" : "s");
         strcat(  command, option);
      }
      if (list)
      {
         strcpy(  text, list->items[ list->selected]->text);
         TxStrip( text, text, ' ', ' ');        // strip leading spaces
         TxRepl(  text, ' ', 0);                // replace next space by 0
         switch (source)                        // select proper source
         {
            case DFSO_VOLD: sprintf( option, " -V:\"%s\"", text);  break;
            case DFSO_DISK: sprintf( option, " -D:%s",     text);  break;
            case DFSO_PART: sprintf( option, " -P:%s",     text);  break;
            default:        strcpy(  option, "");                  break;
         }
         strcat( command, option);
      }
      if (!simr1)                               // not a single file
      {
         strcat( command, " -m:");
         if (strlen( sime1) != 0)               // size specified
         {
            TxStrip( sime1, sime1, ' ', ' ');
            sprintf( option, "%s,%s", sime1, (simrk) ? "k" :
                                             (simrm) ? "m" :
                                                       "g");
            strcat( command, option);
         }
      }
      if ( simc2)         strcat( command, " -E:i");
      if ( simr6)         strcat( command, " -R");
      if ( simr5 | simr4) strcat( command, " -z");
      if ( simc1)         strcat( command, " -x");
      if ( simr4)         strcat( command, " -S");
      if ( simr3)         strcat( command, " -M");
      if ( simr8)
      {
         sprintf( option, " -b:%u", RBUFSECTORS); // DOS max, 256 KiB
         strcat(  command, option);
      }
      else if (simr9)
      {
         sprintf( option, " -b:%u", dfstGeoSectors( DFSTORE));
         strcat(  command, option);
      }
      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsImageDialog'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Callback signaling a change in the target path (including driveletter/vol)
/*****************************************************************************/
static void dfsImageCbTargetPathChange
(
   char              **newPath                  // IN    ptr to new full PATH string
)
{
   BOOL                upd = FALSE;
   BOOL                multiFile = FALSE;       // 2TB limit, requires multi-file

   ENTER();
   TRACES(("New target Path: '%s'\n", *newPath));

   #if defined (LINUX)
      {
         TXTM          cmd;
         char         *fstype;

         sprintf( cmd, "df --output=fstype %s 2>&1 | grep -v -i 'Type'", *newPath);
         txcExecRedirectIO( cmd, NULL, &fstype);

         if ((strncasecmp( fstype, "VFAT", 4) == 0) || (strncasecmp( fstype, "HPFS", 4) == 0))
         {
            multiFile = TRUE;                   // Default to multi on HPFS or any FAT
         }
         TxFreeMem( fstype);
      }
   #elif defined (DARWIN)
   #else                                        // determine FSYS from the volume/driveletter
      {
         TXTS          drive;
         TXTS          fstype;

         sprintf(  drive, "%2.2s", *newPath);   // terminate after driveletter
         TxFsType( drive, fstype, NULL);        // determine FS type for driveletter

         if ((strncasecmp( fstype, "FAT", 3) == 0) || (strncasecmp( fstype, "HPFS", 4) == 0))
         {
            multiFile = TRUE;                   // Default to multi on HPFS or any FAT
         }
      }
   #endif

   if (multiFile == TRUE)
   {
      if (simr1 == TRUE)
      {
         simr1 = FALSE;
         simr2 = TRUE;                          // multi-file, no mediachange
         simr3 = FALSE;
         upd = TRUE;                            // widgets have changed, repaint
      }
   }
   else                                         // default to unlimited single file
   {
      if (simr1 == FALSE)
      {
         simr1 = TRUE;                          // Single-file, unlimited size
         simr2 = FALSE;
         simr3 = FALSE;                         // force media-change set
         upd = TRUE;                            // widgets have changed, repaint
      }
   }
   if (upd)
   {
      txwInvalidateWindow( dfsDgSimWidgets[ DFSDGSIMR1].hwnd, FALSE, FALSE); // repaint this widget
      txwInvalidateWindow( dfsDgSimWidgets[ DFSDGSIMR2].hwnd, FALSE, FALSE); // repaint this widget
      txwInvalidateWindow( dfsDgSimWidgets[ DFSDGSIMR3].hwnd, FALSE, FALSE); // repaint this widget
   }
   TRACES(("Default to multiple files: %s\n", (simr2) ? "YES" : "NO"));
   VRETURN();
}                                               // end 'dfsImageCbTargetPathChange'
/*---------------------------------------------------------------------------*/



/*========================== RESTORE ============================================================*/
static TXTS wrimrcv = "";                       // Restore/Compare/Verify

static BOOL wrimr1  = TRUE;                     // Restore
static BOOL wrimr2  = FALSE;                    // Verify
static BOOL wrimr3  = FALSE;                    // Compare

static BOOL wrimc2  = FALSE;                    // Exclude, -Lvm
static BOOL wrimc4  = TRUE;                     // HEX sector values
static BOOL wrimc6  = TRUE;                     // Use size from image file

static TXTS wrime2  = "";                       // From
static TXTS wrime3  = "";                       // Size
static BOOL wrimrK  = FALSE;                    // KiB
static BOOL wrimrM  = FALSE;                    // MiB
static BOOL wrimrG  = FALSE;                    // GiB
static BOOL wrimrS  = TRUE;                     // Sec

/*
Restore TO
Partition :[01 /dev/sda1    ARCAOS      D: Pri 0x07 JFS        2047.0 MiB V]

[ ] Use size/start from image file      () Restore, write to object
                                        ( ) Compare imagefile to object
[ ] Exclude LVM area                    ( ) Verify image file integrity

 First sector     Image size      [] Sector values are HEX, not decimal
[Start of Area ] [Whole Area    ] () Sectors  ( ) KiB  ( ) MiB  ( ) GiB
*/
#define   DFSDGWRIMWIDGETS 16
#define   DFSDGWRIMWBASICS  6
#define   DFSDGWRIMRCV      1                   // index of RCV string
#define   DFSDGWRIMT1       1                   // index of description
#define   DFSDGWRIMT2       2                   // index of leader
#define   DFSDGWRIML1       3                   // index of list-1
#define   DFSDGWRIMXL       5                   // index of Exclude LVM
static TXWIDGET  dfsDgWrimWidgets[DFSDGWRIMWIDGETS] =  // order determines TAB-order!
{
   {0,  0, 1, 11, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( wrimrcv)},

   {1, 12, 1, 65, 1, 0, 0, DFSDSOUTPUT,  0, TXStdStline( tdescr)},
   {1,  0, 1, 11, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( NULL)},
   {1, 11, 1, 65, 0, 0, 0, DFSDSLISTBOX, 0, TXStdLstBox( NULL, 0, NULL)},

   {3,  0, 1, 34, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &wrimc6, "Get size/start from image file")},
   {5,  0, 1, 34, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &wrimc2, "Exclude LVM area")},

   {3, 40, 1, 32, 0, 2, 0, DFSDMAUTORAD, 0, TXWprButton( &wrimr1, dfsRcvWinProc, "Restore, write to object")},
   {4, 40, 1, 32, 0, 2, 0, DFSDMAUTORAD, 0, TXWprButton( &wrimr3, dfsRcvWinProc, "Compare imagefile to object")},
   {5, 40, 1, 32, 0, 2, 0, DFSDMAUTORAD, 0, TXWprButton( &wrimr2, dfsRcvWinProc, "Verify image file integrity")},

   {7, 36, 1, 38, 0, 0, 0, DFSDMAUTOCHK, 0, TXStdButton( &wrimc4, "Sector values are HEX, not decimal")},
   {7,  0, 2, 16, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD2SIZE,
                                         0, TXStdEntryf( wrime2, TXMAXTS, "First sector")},
   {7, 17, 2, 16, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD2SIZE  |  TXWS_HCHILD2MOVE,
                                         0, TXStdEntryf( wrime3, TXMAXTS, "Image size")},
   {8, 36, 1, 11, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &wrimrS, "Sectors")},
   {8, 49, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &wrimrK, "KiB")},
   {8, 58, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &wrimrM, "MiB")},
   {8, 67, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &wrimrG, "GiB")}
};

#define WID_IMGDLG           810
static TXGW_DATA dfsDgWrimDlg =
{
   DFSDGWRIMWIDGETS,                            // number of widgets
   DFSC_WRIMAGE,                                // help, widget overrules
   WID_IMGDLG,                                  // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgWrimWidgets                             // array of widgets
};


/*************************************************************************************************/
// Present RESTORE options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsRestoreDialog
(
   DFSOBJECT           destination,             // IN    Destination for image
   char               *image,                   // IN    filename, "" or NULL
   BOOL                verify,                  // IN    verify only
   ULN64               tsector,                 // IN    To   sector number
   ULN64               sizetodo                 // IN    Size to restore (0 = all)
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;
   TXTT                option;
   TXTM                title;
   TXGW_DATA          *widgets = NULL;
   TXSELIST           *list    = NULL;
   TXTM                text;

   ENTER();
   TRACES(( "%s  from:'%s'  start position:0x%llx  size:0x%llx\n",
          (verify) ? "RESTORE" : "VERIFY ", tsector, sizetodo));

   dfsBEGINWORK();                              // signal work starting

   if (tsector != 0)
   {
      sprintf( wrime2, "%llx", tsector);        // Offset to process (0 = 0)
   }
   else
   {
      strcpy(  wrime2, "Start of Area");
   }
   if (sizetodo != 0)
   {
      sprintf( wrime3, "%llx", sizetodo);       // set Size to process (0 = all)
   }
   else
   {
      strcpy(  wrime3, "Whole Area");
   }
   if (TxaOption('S'))
   {
      wrimc6 = TRUE;                            // force checkbox when option set
   }

   if (verify)                                  // VERIFY only
   {
      wrimr1  = FALSE;                          // Restore
      wrimr2  = TRUE;                           // Verify
      wrimr3  = FALSE;                          // Compare
      strcpy( title, " Select an imagefile to be checked for file damage ");
      strcpy( fspec, "*.imz");
   }
   else
   {
      if (wrimr2 == TRUE)                       // last was verify
      {
         wrimr1  = TRUE;                        // Restore
         wrimr2  = FALSE;                       // Verify
         wrimr3  = FALSE;                       // Compare
      }
      dfsDgWrimWidgets[DFSDGWRIMT1].flags  |=  TXWI_DISABLED; // text field OFF
      dfsDgWrimWidgets[DFSDGWRIML1].flags  &= ~TXWI_DISABLED; // list field ON
      strcpy( title, " Select a file and options for the image RESTORE ");
      if (dfsa->expertui)
      {
         strcat( title, "/ COMPARE ");
      }
      switch (destination)
      {
         case DFSO_VOLD:
            dfsDgWrimWidgets[DFSDGWRIMT2].sl.buf = "Dev/Volume:";
            dfsDgWrimWidgets[DFSDGWRIML1].title  = " Restore image to volume ";
            if (dfsa->slVolumes == NULL)
            {
               dfsa->slVolumes = TxFsDriveSelist( TXFSV_HD | TXFSV_CD |
                                                  TXFSV_FLOP, FALSE);
            }
            list = dfsa->slVolumes;
            TxSelStringSelect( list,  SINF->drive, 2);
            break;

         case DFSO_DISK:
            dfsDgWrimWidgets[DFSDGWRIMT2].sl.buf = "Whole disk:";
            dfsDgWrimWidgets[DFSDGWRIML1].title  = " Restore image to disk ";
            list = dfsa->slDiskOne;
            sprintf( text, "%2hu",    SINF->disknr);
            TxSelStringSelect( list,  text, 2);
            break;

         case DFSO_PART:
            dfsDgWrimWidgets[DFSDGWRIMT2].sl.buf = "Partition :";
            dfsDgWrimWidgets[DFSDGWRIML1].title  = " Restore image to partition ";
            list = dfsa->slPartOne;
            sprintf( text, "%2.2hu", SINF->partid);
            TxSelStringSelect( list,  text, 2);
            break;

         default:
            if (destination == DFSO_IMGF)
            {
               strcpy( tdescr, dfstStoreDesc2( DFSTORE) + 10);
               dfsDgWrimWidgets[DFSDGWRIMT2].sl.buf  = "(img) File:";
            }
            else                                // other types
            {
               strcpy( tdescr, dfstStoreDesc1( DFSTORE) + 10);
               dfsDgWrimWidgets[DFSDGWRIMT2].sl.buf  = "Current   :";
            }
            dfsDgWrimWidgets[DFSDGWRIMT1].flags  &= ~TXWI_DISABLED; // text field ON
            dfsDgWrimWidgets[DFSDGWRIML1].flags  |=  TXWI_DISABLED; // list field OFF
            break;
      }
      dfsDgWrimWidgets[DFSDGWRIML1].lb.list = list;
      strcpy( fspec, "*.i??");

      if (dfsa->expertui)                       // set available widgets Expert/Basic
      {
         dfsDgWrimDlg.count = DFSDGWRIMWIDGETS;
         dfsDgWrimWidgets[DFSDGWRIMXL].flags  &= ~TXWI_DISABLED;
      }
      else
      {
         dfsDgWrimDlg.count = DFSDGWRIMWBASICS;
         dfsDgWrimWidgets[DFSDGWRIMXL].flags  |=  TXWI_DISABLED;
      }
      widgets = &dfsDgWrimDlg;
   }
   strcpy( wrimrcv, (wrimr1) ? "Restore TO" : (wrimr2) ? "Verify  TO" : "Compare TO");

   if (txwOpenFileDialog( fspec, NULL, image, DFSC_WRIMAGE, NULL, widgets, title, fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      TxStrip( fspec, fspec, ' ', ' ');
      sprintf( command, "restore \"%s\" ", fspec);
      TxStrip( wrime2, wrime2, ' ', ' ');
      if ((strlen( wrime2) != 0) &&             // Start value specified
          (isxdigit( wrime2[0]))  )             // and it is a number
      {
         sprintf( option, "%s%s", (wrimc4) ? "0x" : "", wrime2);
         strcat( command, option);
      }
      else
      {                                         // mandatory if size given,
         strcat( command, "0");                 // positional parameter!
      }
      TxStrip( wrime3, wrime3, ' ', ' ');
      if ((strlen( wrime3) != 0) &&             // Size value specified
          (isxdigit( wrime3[0]))  )             // and it is a number
      {
         sprintf( option, " %s%s,%s", (wrimc4 && wrimrS) ? "0x" : "", wrime3,
                 (wrimrK) ? "k" : (wrimrM) ? "m" : (wrimrG) ? "g" : "s");
         strcat(  command, option);
         wrimc6 = FALSE;                        // ignore 'size from image'
      }
      if (list)
      {
         strcpy(  text, list->items[ list->selected]->text);
         TxStrip( text, text, ' ', ' ');        // strip leading spaces
         TxRepl(  text, ' ', 0);                // replace next space by 0
         switch (destination)                   // select proper destination
         {
            case DFSO_VOLD: sprintf( option, " -V:\"%s\"", text);  break;
            case DFSO_DISK: sprintf( option, " -D:%s",     text);  break;
            case DFSO_PART: sprintf( option, " -P:%s",     text);  break;
            default:        strcpy(  option, "");                  break;
         }
         strcat( command, option);
      }
      if ( wrimr2 | verify) strcat( command, " -test");
      if ( wrimr3)          strcat( command, " -test -c");
      if ( wrimc2)          strcat( command, " -L");
      if ( wrimc6)          strcat( command, " -S");

      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsRestoreDialog'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the RESTORE/COMPARE/VERIFY button widgets
/*****************************************************************************/
static ULONG dfsRcvWinProc                      // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      //- does not work, perhaps incorrect ID ?
      //-HANDLE     rcvText = txwWindowFromID( 0, WID_IMGDLG + DFSDGWRIMRCV);
      TXWHANDLE     owner = txwQueryWindow( hwnd, TXQW_OWNER); // get dialog info

      TRCMSG( hwnd, msg, mp1, mp2);

      rc = txwDefWindowProc( hwnd, msg, mp1, mp2); // default handling first, sets variable!
      if (msg == TXWM_CHAR)                     // catch the simulated SPACE character
      {
         strcpy( wrimrcv, (wrimr1) ? "Restore TO" : (wrimr2) ? "Verify  TO" : "Compare TO");
         //-InvalidateWindow( rcvText, FALSE, TRUE); // whole dlg
         txwInvalidateWindow( owner, FALSE, TRUE); // whole dlg
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsRcvWinProc'
/*---------------------------------------------------------------------------*/


/*========================== CLONE ==============================================================*/
static BOOL        cloc1  = FALSE;              // Merge on error
static BOOL        cloc2  = FALSE;              // Ignore errors
static BOOL        cloc3  = FALSE;              // Exclude LVMsig
static BOOL        cloc4  = TRUE;               // HEX sector values
static BOOL        cloc5  = FALSE;              // COMPARE only
static BOOL        cloc6  = TRUE;               // SMART sector skip
static BOOL        cloc7  = FALSE;              // REVERSE High-to-Low

static BOOL        clocN  = FALSE;              // Update LVM diskname
static TXTT        cloeN  = "";                 // New LVM diskname entry

static TXTS        cloe1  = "";                 // From sector
static TXTS        cloe2  = "";                 // To   sector
static TXTS        cloe3  = "";                 // Size
static TXTS        cloe4  = "";                 // SkipBads
static BOOL        clorK  = FALSE;              // KiB
static BOOL        clorM  = FALSE;              // MiB
static BOOL        clorG  = FALSE;              // GiB
static BOOL        clorS  = TRUE;               // Sec
static BOOL        cloOK  = TRUE;               // OK button, dummy variable
static BOOL        cloCA  = FALSE;              // Cancel button dummy

/*
 [ ] VERIFY only, compare FROM against TO         [ ] High-to-Low sectors
 [ ] Skip unused areas in cloning (smart)         [ ] Ignore errors
 [ ] Merge original data on read errors           [ ] Exclude LVM area
 [ ] Change the LVM diskname on 'To' disk clone to:  [CurrentNameYYYYMMDD]

 From Part: [04 /dev/hda6    ECS12       F: Log 07 HPFS       917.7 MiB V]


 To   Part: [17 /dev/hdb5                   Log 07 -          917.7 MiB V]

                   Nr. of sectors skipped              Ŀ  Ŀ
 From 1st sector   on any bad-sector area                OK    Cancel
[...............] [..............]                       

 To   1st sector   Size to do      [] Sector values are HEX, not decimal
[...............] [Whole Area    ] () Sectors  ( ) KiB  ( ) MiB  ( ) GiB
*/
#define   DFSDGCLOWIDGETS 27                    // For Expert UI
#define   DFSDGCLOWBASICS 16                    // For Basic  UI
#define   DFSDGCLOCV       0                    // index of compare/verify
#define   DFSDGCLOHL       1                    // index of High to Low
#define   DFSDGCLOME       4                    // index of Merge Errors
#define   DFSDGCLOXL       5                    // index of exclude LVM
#define   DFSDGCLOUN       6                    // index of Update LVM diskname
#define   DFSDGCLONE       7                    // index of LVM diskname entry
#define   DFSDGCLOD1       8                    // index of from description
#define   DFSDGCLOT1       9                    // index of from leader
#define   DFSDGCLOL1      10                    // index of from list-1
#define   DFSDGCLOD2      11                    // index of to description
#define   DFSDGCLOT2      12                    // index of to leader
#define   DFSDGCLOL2      13                    // index of to list-2
#define   DFSDGCLOOK      14                    // index of OK button
static TXWIDGET  dfsDgCloWidgets[DFSDGCLOWIDGETS] =    // order determines TAB-order!
{
   {0,  0, 1, 42, 0, 6, 0, TXWS_AUTOCHK, 0, TXStdButton( &cloc5, "COMPARE only, verify FROM against TO")},
   {0, 42, 1, 32, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &cloc7, "High-to-Low sector numbers")},
   {1,  0, 1, 42, 0, 6, 0, TXWS_AUTOCHK, 0, TXStdButton( &cloc6, "Skip unused areas in cloning (SMART)")},
   {1, 42, 1, 32, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &cloc2, "Ignore errors, auto continue")},
   {2,  0, 1, 42, 0, 6, 0, TXWS_AUTOCHK, 0, TXStdButton( &cloc1, "Merge original data on read errors")},
   {2, 42, 1, 32, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &cloc3, "Exclude LVM area at end")},

   {3,  0, 1, 50, 0, 8, 0, TXWS_AUTOCHK, 0, TXStdButton( &clocN, "Change the LVM diskname on 'To' disk clone to:")},
   {3, 52, 1, 21, 0, 0, 0, TXWS_ENTRYB   |  TXWS_HCHILD2SIZE,
                                         0, TXStdEntryf( cloeN, LVM_NAME_L -1, "")},

   {5, 11, 1, 65, 1, 0, 0, DFSDSOUTPUT,  0, TXStdStline( fdescr)},
   {5,  0, 1, 11, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( NULL)},
   {5, 10, 1, 65, 0, 0, 0, DFSDSLISTBOX, 0, TXWprLstBox( NULL, 0, dfsCloneWinProc, NULL)},

   {8, 11, 1, 65, 1, 0, 0, DFSDSOUTPUT,  0, TXStdStline( tdescr)},
   {8,  0, 1, 11, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( NULL)},
   {8, 10, 1, 65, 0, 0, 0, DFSDSLISTBOX, 0, TXStdLstBox( NULL, 0, NULL)},

   {10,55, 3,  8, 0, 0, TXDID_OK,
                           DFSDMPBUTTON, 0, TXStdButton( &cloOK, " OK ")},
   {10,67, 3,  8, 0, 0, TXDID_CANCEL,
                           DFSDMPBUTTON, 0, TXStdButton( &cloCA, "Cancel")},

   {10,18, 1, 24, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "Nr. of sectors skipped")},
   {11,18, 1, 24, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "on any bad-sector area")},

   {11, 0, 2, 17, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD2SIZE,
                                         0, TXStdEntryf( cloe1, TXMAXTS, "From 1st sector")},

   {12,17, 1, 17, 0, 0, 0, TXWS_ENTRYB   |  TXWS_HCHILD2SIZE,
                                         0, TXStdEntryf( cloe4, TXMAXTS, "")},

   {14, 0, 2, 17, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD2SIZE,
                                         0, TXStdEntryf( cloe2, TXMAXTS, "To   1st sector")},
   {14,17, 2, 17, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD2SIZE |  TXWS_HCHILD2MOVE,
                                         0, TXStdEntryf( cloe3, TXMAXTS, "Size to do")},
   {14,37, 1, 38, 0, 0, 0, DFSDMAUTOCHK, 0, TXStdButton( &cloc4, "Sector values are HEX, not decimal")},

   {15,37, 1, 11, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &clorS, "Sectors")},
   {15,50, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &clorK, "KiB")},
   {15,59, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &clorM, "MiB")},
   {15,68, 1,  7, 0, 4, 0, DFSDMAUTORAD, 0, TXStdButton( &clorG, "GiB")}
};

static TXGW_DATA dfsDgCloDlg =
{
   DFSDGCLOWIDGETS,                             // number of widgets
   DFSC_CLONDLG,                                // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgCloWidgets                              // array of widgets
};


/*************************************************************************************************/
// Present CLONE options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsCloneDialog
(
   BOOL                clone,                   // IN    Clone, not VERIFY
   DFSOBJECT           fobject,                 // IN    From object reference
   ULN64               fsector,                 // IN    From sector number
   DFSOBJECT           tobject,                 // IN    To   object reference
   ULN64               tsector,                 // IN    To   sector number
   ULN64               sizetodo,                // IN    Size to clone (0 = all)
   ULONG               skipbads                 // IN    sectors to skip on BADS
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               fstart = 0;              // from start
   ULN64               tstart = 0;              // to   start
   ULONG               fselected;               // from selected
   ULONG               tselected;               // to   selected
   USHORT              focus;                   // field to get focus
   TXSELIST           *flist  = NULL;           // from list
   TXSELIST           *tlist  = NULL;           // to   list
   DFST_HANDLE         tstore = DFSTORE;        // DFSee STORE ids
   DFST_HANDLE         fstore = (tstore == 1) ? 2 : 1;
   DFSISTORE          *fsi;                     // FROM store info struct
   TXLN                command;
   TXTT                option;
   TXTM                text;

   ENTER();
   TRACES(( "%s  from:0x%llx  at:0x%llx  size:0x%llx\n",
          (clone) ? "CLONE " : "VERIFY", fsector, tsector, sizetodo));

   dfsBEGINWORK();                              // signal work starting

   if (dfstQueryStoreType( fstore, &fsi, NULL) == DFST_UNUSED)
   {
      fsi = SINF;                               // when not set yet, default to
   }                                            // same as the CURRENT to-store
   dfsDgCloWidgets[DFSDGCLOD1].flags  |=  TXWI_DISABLED; // disable static
   dfsDgCloWidgets[DFSDGCLOL1].flags  &= ~TXWI_DISABLED; // enable list

   if ((fobject == DFSO_DISK) && (tobject == DFSO_DISK))
   {
      dfsDgCloWidgets[DFSDGCLOUN].flags  &= ~TXWI_DISABLED; // enable update-name
      dfsDgCloWidgets[DFSDGCLONE].flags  &= ~TXWI_DISABLED; // enable name entry
   }
   else                                         // not disk-to-disk
   {
      dfsDgCloWidgets[DFSDGCLOUN].flags  |=  TXWI_DISABLED; // disable update-name
      dfsDgCloWidgets[DFSDGCLONE].flags  |=  TXWI_DISABLED; // disable name entry
   }

   switch (fobject)
   {
      case DFSO_VOLD:
         dfsDgCloWidgets[DFSDGCLOT1].sl.buf = "From vol.:";
         dfsDgCloWidgets[DFSDGCLOL1].title  = " Clone from volume ";
         if (dfsa->slVolumes == NULL)
         {
            dfsa->slVolumes = TxFsDriveSelist( TXFSV_HD | TXFSV_CD |
                                               TXFSV_FLOP, FALSE);
         }
         rc = TxSelistDuplicate( dfsa->slVolumes, FALSE, TRUE, &flist);
         TxSelStringSelect( flist,  fsi->drive, 2);
         break;

      case DFSO_DISK:
         dfsDgCloWidgets[DFSDGCLOT1].sl.buf = "From disk:";
         dfsDgCloWidgets[DFSDGCLOL1].title  = " Clone from disk ";
         rc = TxSelistDuplicate( dfsa->slDiskOne, FALSE, TRUE, &flist);
         sprintf( text, "%2hu",     fsi->disknr);
         TxSelStringSelect( flist,  text, 2);
         break;

      case DFSO_PART:
         dfsDgCloWidgets[DFSDGCLOT1].sl.buf = "From part:";
         dfsDgCloWidgets[DFSDGCLOL1].title  = " Clone from partition ";
         rc = TxSelistDuplicate( dfsa->slPartOne, FALSE, TRUE, &flist);
         sprintf( text, "%2.2hu",  fsi->partid);
         TxSelStringSelect( flist,  text, 2);
         break;

      default:
         if (fobject == DFSO_IMGF)
         {
            strcpy( fdescr, dfstStoreDesc2( fstore) + 10);
            dfsDgCloWidgets[DFSDGCLOT1].sl.buf  = "From file:";
         }
         else                                   // other types
         {
            strcpy( fdescr, dfstStoreDesc1( fstore) + 10);
            dfsDgCloWidgets[DFSDGCLOT1].sl.buf  = "From Curr:";
         }
         dfsDgCloWidgets[DFSDGCLOD1].flags  &= ~TXWI_DISABLED;
         dfsDgCloWidgets[DFSDGCLOL1].flags  |=  TXWI_DISABLED;
         break;
   }
   dfsDgCloWidgets[DFSDGCLOL1].lb.list = flist;

   dfsDgCloWidgets[DFSDGCLOD2].flags  |=  TXWI_DISABLED; // disable static
   dfsDgCloWidgets[DFSDGCLOL2].flags  &= ~TXWI_DISABLED; // enable list
   switch (tobject)
   {
      case DFSO_VOLD:
         dfsDgCloWidgets[DFSDGCLOT2].sl.buf = "To   vol.:";
         dfsDgCloWidgets[DFSDGCLOL2].title  = " Clone to volume ";
         if (dfsa->slVolumes == NULL)
         {
            dfsa->slVolumes = TxFsDriveSelist( TXFSV_HD | TXFSV_CD |
                                               TXFSV_FLOP, FALSE);
         }
         tlist = dfsa->slVolumes;
         TxSelStringSelect( tlist,  SINF->drive, 2);
         break;

      case DFSO_DISK:
         dfsDgCloWidgets[DFSDGCLOT2].sl.buf = "To   disk:";
         dfsDgCloWidgets[DFSDGCLOL2].title  = " Clone to disk ";
         tlist = dfsa->slDiskOne;
         sprintf( text, "%2hu",    SINF->disknr);
         TxSelStringSelect( tlist,  text, 2);
         break;

      case DFSO_PART:
         dfsDgCloWidgets[DFSDGCLOT2].sl.buf = "To   part:";
         dfsDgCloWidgets[DFSDGCLOL2].title  = " Clone to partition ";
         tlist = dfsa->slPartOne;
         sprintf( text, "%2.2hu", SINF->partid);
         TxSelStringSelect( tlist,  text, 2);
         break;

      default:
         if (tobject == DFSO_IMGF)
         {
            strcpy( tdescr, dfstStoreDesc2( tstore) + 10);
            dfsDgCloWidgets[DFSDGCLOT2].sl.buf  = "To   file:";
         }
         else                                   // other types
         {
            strcpy( tdescr, dfstStoreDesc1( tstore) + 10);
            dfsDgCloWidgets[DFSDGCLOT2].sl.buf  = "To   Curr:";
         }
         dfsDgCloWidgets[DFSDGCLOD2].flags  &= ~TXWI_DISABLED;
         dfsDgCloWidgets[DFSDGCLOL2].flags  |=  TXWI_DISABLED;
         break;
   }
   dfsDgCloWidgets[DFSDGCLOL2].lb.list = tlist;

   if      (flist != NULL) focus = DFSDGCLOL1;
   else if (tlist != NULL) focus = DFSDGCLOL2;
   else                    focus = DFSDGCLOOK;

   cloc5 = (clone == FALSE);                    // set COMPARE checkbox
   cloc4 = TRUE;
   sprintf( cloe1, "%llx", fsector);
   sprintf( cloe2, "%llx", tsector);            // set From/To sector numbers
   if (sizetodo != 0)
   {
      sprintf( cloe3, "%llx", sizetodo);        // set Size to clone  (0 = all)
   }
   else
   {
      strcpy( cloe3, "Whole Area");
   }
   sprintf( cloe4, "%x", skipbads);             // HEX value, usually 0
   if (skipbads)                                // when non-zero, disable SMART/ignore-errors
   {
      cloc2  = TRUE;                            // ignore errors unless overriden
      cloc6  = FALSE;                           // no SMART, unless overridden
   }

   if (dfsa->expertui)                          // set available widgets Expert/Basic
   {
      dfsDgCloDlg.count = DFSDGCLOWIDGETS;
      dfsDgCloWidgets[DFSDGCLOCV].flags  &= ~TXWI_DISABLED;
      dfsDgCloWidgets[DFSDGCLOHL].flags  &= ~TXWI_DISABLED;
      dfsDgCloWidgets[DFSDGCLOME].flags  &= ~TXWI_DISABLED;
   }
   else
   {
      dfsDgCloDlg.count = DFSDGCLOWBASICS;
      dfsDgCloWidgets[DFSDGCLOCV].flags  |=  TXWI_DISABLED;
      dfsDgCloWidgets[DFSDGCLOHL].flags  |=  TXWI_DISABLED;
      dfsDgCloWidgets[DFSDGCLOME].flags  |=  TXWI_DISABLED;
   }

   while ((rc == NO_ERROR) && (txwWidgetDialog( TXHWND_DESKTOP, TXHWND_DESKTOP,
           NULL, " Specify cloning options and 'From' plus 'To' objects ",
           TXWD_MOVEABLE | TXWD_HCENTER | TXWD_VCENTER, focus, &dfsDgCloDlg)
        != TXDID_CANCEL))
   {
      sscanf( cloe1, "%llx", &fstart);
      sscanf( cloe2, "%llx", &tstart);          // get to/from start value

      fselected = (flist) ? flist->selected : 11111; // different values, always
      tselected = (tlist) ? tlist->selected : 22222; // larger than list sizes

      if ((fobject != tobject) || (fselected != tselected) || (fstart != tstart))
      {
         sprintf( command, "%s %s%s", (cloc5)         ? "comp" : "clone",
                                      (cloc4)         ? "0x"   : "",
                                      (strlen(cloe2)) ?  cloe2 : "0");
         if ((strlen( cloe3) != 0) &&           // Size value specified
             (isxdigit( cloe3[0]))  )           // and it is a number
         {
            TxStrip( cloe3, cloe3, ' ', ' ');
            sprintf( option, " %s%s,%s", (cloc4) ? "0x" : "", cloe3,
                    (clorK) ? "k" : (clorM) ? "m" : (clorG) ? "g" : "s");
            strcat(  command, option);
         }

         if ((strlen( cloe1) != 0)     &&       // From start sector
             (strcmp( cloe1, "0") != 0) )       // and not zero
         {
            TxStrip( cloe1, cloe1, ' ', ' ');
            sprintf( option, " -f:%s%s,%s", (cloc4) ? "0x" : "", cloe1,
                    (clorK) ? "k" : (clorM) ? "m" : (clorG) ? "g" : "s");
            strcat(  command, option);
         }

         if ((strlen( cloe4) != 0)     &&       // SkipBads sector value
             (strcmp( cloe4, "0") != 0) )       // and not zero
         {
            TxStrip( cloe4, cloe4, ' ', ' ');
            sprintf( option, " -skipbads:%s%s", (cloc4) ? "0x" : "", cloe4);
            if (strchr( cloe4, ',') == NULL)    // no explicit unit given
            {
               strcat( option, ",");
               strcat( option, (clorM) ? "m" : (clorG) ? "g" : "s");
            }
            strcat(  command, option);
         }

         if (flist)
         {
            strcpy(  text, flist->items[ flist->selected]->text);
            TxStrip( text, text, ' ', ' ');     // strip leading spaces
            TxRepl(  text, ' ', 0);             // replace next space by 0
            switch (fobject)                    // select proper source
            {
               case DFSO_VOLD: sprintf( option, " -v:\"%s\"", text);  break;
               case DFSO_DISK: sprintf( option, " -d:%s",     text);  break;
               case DFSO_PART: sprintf( option, " -p:%s",     text);  break;
               default:        strcpy(  option, "");                  break;
            }
            strcat( command, option);
         }

         if (tlist)
         {
            strcpy(  text, tlist->items[ tlist->selected]->text);
            TxStrip( text, text, ' ', ' ');     // strip leading spaces
            TxRepl(  text, ' ', 0);             // replace next space by 0
            switch (tobject)                    // select proper destination
            {
               case DFSO_VOLD: sprintf( option, " -V:\"%s\"", text);  break;
               case DFSO_DISK: sprintf( option, " -D:%s",     text);  break;
               case DFSO_PART: sprintf( option, " -P:%s",     text);  break;
               default:        strcpy(  option, "");                  break;
            }
            strcat( command, option);
         }

         if ( cloc1)         strcat( command, " -M");
         if ( cloc2)         strcat( command, " -E:i");
         if ( cloc3)         strcat( command, " -L");
         if ( cloc6)         strcat( command, " -S");
         if ( cloc7)         strcat( command, " -R");
         if ( clocN)
         {
            TxStrip( cloeN, cloeN, ' ', ' ');
            sprintf( text,  " -name:\"%s\"", cloeN);
            strcat(  command, text);
         }

         dfsExecEnd( command);
         break;
      }
      else
      {
         TxNamedMessage( TRUE, DFSC_CLONING, " ERROR: Can not CLONE to self ",
                         "The selected FROM and TO objects/First sector seem to be the same!\n\n"
                         "You must specify a 'FROM' location that is DIFFERENT from "
                         "the 'TO' location to be able to clone between them ...");
      }
   }
   if (flist)                                   // free Duplicated list
   {
      txSelDestroy( &flist);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsCloneDialog'
/*-----------------------------------------------------------------------------------------------*/



/*****************************************************************************/
// Dialog window procedure, for the CLONE from-list widget field
/*****************************************************************************/
static ULONG dfsCloneWinProc                    // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;
   TXSELIST           *flist  = NULL;           // from list
   static USHORT       fromDiskId = 0;          // Disk-ID D2D FROM disk to prevent
                                                // unwanted LVM-name change on TAB
   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      TXWHANDLE     owner = txwQueryWindow( hwnd, TXQW_OWNER); // get dialog info

      TRCMSG( hwnd, msg, mp1, mp2);

      switch (msg)
      {
         case TXWM_SETFOCUS:                    // on loosing focus
            flist = dfsDgCloWidgets[DFSDGCLOL1].lb.list;

            TRACES(("flist: 0x%8.8x\n", flist));
            if (flist)
            {
               char    listType = 0;
               char   *listDescription = (char *) flist->miscInfo;

               if (listDescription && *listDescription)
               {
                  listType = toupper( *listDescription);
                  TRACES(("List type %c for descr: %s\n", listType, listDescription));

                  if (listType == 'D')          // list of disks
                  {
                     DFSDISKINFO *d = dfsGetDiskInfo( atoi(flist->items[ flist->selected]->text));

                     if ((d != NULL) && (d->lvmPartitions != 0))
                     {
                        time_t    tt = time( &tt); // current date/time
                        TXTS      now;          // date string buffer

                        if (d->disknr != fromDiskId)  //- only ONCE for selected disk
                        {                             //- to allow manual editing too
                           fromDiskId  = d->disknr;

                           strftime( now, TXMAXTS, "%Y%m%d", localtime( &tt));
                           strcpy( cloeN, d->DiskName); // start with current FROM name
                           cloeN[ 11] = 0;      // maximum prefix length
                           strcat( cloeN, now); // append 8 character date
                        }
                        clocN = TRUE;
                        txwEnableWindow( dfsDgCloWidgets[DFSDGCLOUN].hwnd, TRUE);
                        txwEnableWindow( dfsDgCloWidgets[DFSDGCLONE].hwnd, TRUE);
                        txwShowWindow(   dfsDgCloWidgets[DFSDGCLOUN].hwnd, TRUE);
                        txwShowWindow(   dfsDgCloWidgets[DFSDGCLONE].hwnd, TRUE);
                     }
                     else
                     {
                        clocN = FALSE;
                        txwEnableWindow( dfsDgCloWidgets[DFSDGCLOUN].hwnd, FALSE);
                        txwEnableWindow( dfsDgCloWidgets[DFSDGCLONE].hwnd, FALSE);
                        txwShowWindow(   dfsDgCloWidgets[DFSDGCLOUN].hwnd, FALSE);
                        txwShowWindow(   dfsDgCloWidgets[DFSDGCLONE].hwnd, FALSE);
                     }
                     txwEnableWindow( dfsDgCloWidgets[DFSDGCLOXL].hwnd, FALSE); // disable exclude LVM
                     txwShowWindow(   dfsDgCloWidgets[DFSDGCLOXL].hwnd, FALSE);
                  }
                  else if (listType == 'P')     // list of partitions
                  {
                     DFSPARTINFO *p = dfsGetPartInfo( atoi(flist->items[ flist->selected]->text));

                     if ((p != NULL) && p->lvmPresent && dfsa->expertui &&
                         (p->partent.PartitionType == DFS_P_WARP_LVM))
                     {
                        txwEnableWindow( dfsDgCloWidgets[DFSDGCLOXL].hwnd, TRUE); // enable exclude LVM
                        txwShowWindow(   dfsDgCloWidgets[DFSDGCLOXL].hwnd, TRUE);
                     }
                     else
                     {
                        cloc3 = FALSE;          // disable exclude LVM
                        txwEnableWindow( dfsDgCloWidgets[DFSDGCLOXL].hwnd, FALSE);
                        txwShowWindow(   dfsDgCloWidgets[DFSDGCLOXL].hwnd, FALSE);
                     }
                     clocN = FALSE;             // disable LVM name change
                     txwEnableWindow( dfsDgCloWidgets[DFSDGCLOUN].hwnd, FALSE);
                     txwEnableWindow( dfsDgCloWidgets[DFSDGCLONE].hwnd, FALSE);
                     txwShowWindow(   dfsDgCloWidgets[DFSDGCLOUN].hwnd, FALSE);
                     txwShowWindow(   dfsDgCloWidgets[DFSDGCLONE].hwnd, FALSE);
                  }
                  else                          // other clone types, disable all specials
                  {
                     clocN = FALSE;
                     txwEnableWindow( dfsDgCloWidgets[DFSDGCLOUN].hwnd, FALSE);
                     txwEnableWindow( dfsDgCloWidgets[DFSDGCLONE].hwnd, FALSE);
                     txwShowWindow(   dfsDgCloWidgets[DFSDGCLOUN].hwnd, FALSE);
                     txwShowWindow(   dfsDgCloWidgets[DFSDGCLONE].hwnd, FALSE);

                     txwEnableWindow( dfsDgCloWidgets[DFSDGCLOXL].hwnd, FALSE); // disable exclude LVM
                     txwShowWindow(   dfsDgCloWidgets[DFSDGCLOXL].hwnd, FALSE);
                  }
                  txwInvalidateWindow( owner, FALSE, TRUE); // whole dialog
               }
               else
               {
                  TRACES(("List description (userinfo) missing!\n"));
               }
            }
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2); // default handling too!
            break;

         default:
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsCloneWinProc'
/*---------------------------------------------------------------------------*/



/*========================== GENPART ============================================================*/
static BOOL        genc1  = TRUE;               // Sector based size/location
static BOOL        genc2  = FALSE;              // No location/freespace

static TXTM        gene1  = "";                 // Description

/*
    From   disk:  1 Maxtor 114.5 GiB      117239.9 MiB          ]

    Description:[                                                ]

[x] Use sector based sizes and location, for exact copy on any GEO
[ ] Do NOT include freespace areas, use default location on create

    Specify base filename only, path is added, extension stripped
*/
#define   DFSDGGENWIDGETS 7
#define   DFSDGGENL1      1                     // index of list-1
static TXWIDGET  dfsDgGenWidgets[DFSDGGENWIDGETS] =    // order determines TAB-order!
{
   {0,  4, 1, 12, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "From   disk:")},
   {0, 16, 1, 50, 0, 0, 0, DFSDSLISTBOX, 0, TXStdLstBox( NULL, 0, NULL)},

   {2,  4, 1, 12, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "Description:")},
   {2, 16, 1, 50, 0, 0, 0, TXWS_ENTRYB   |  TXWS_HCHILD2SIZE |  TXWS_HCHILD2MOVE,
                                         0, TXStdEntryf( gene1, TXMAXTM, "")},

   {4,  0, 1, 66, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &genc1,
                          "Use sector based sizes and location, for exact copy on any GEO")},
   {5,  0, 1, 66, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &genc2,
                          "Do NOT include freespace areas, use default location on create")},

   {7,  4, 1, 62, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline(
                          "Specify base filename only, path is added, extension stripped")},
};

static TXGW_DATA dfsDgGenDlg =
{
   DFSDGGENWIDGETS,                             // number of widgets
   DFSC_GENPDLG,                                // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgGenWidgets                              // array of widgets
};


/*************************************************************************************************/
// Present GENPART options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsGenPartDialog
(
   USHORT              disk,                    // IN    disk number
   char               *base,                    // IN    base filename
   char               *descr,                   // IN    description string
   TXSETVALUE          sectors,                 // IN    use sector values
   TXSETVALUE          dofree                   // IN    include freespace
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;
   TXSELIST           *list = NULL;
   ULONG               item = 0;                // default to ALL disks

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   if (sectors != TX_UNKNOWN)
   {
      genc1 = (sectors == TX_CHECKED);
   }
   if (dofree  != TX_UNKNOWN)
   {
      genc2 = (dofree  == TX_CHECKED);
   }
   if (descr && strlen(descr))
   {
      strcpy( gene1, descr);
   }

   list = dfsa->slDiskAll;
   if ((disk != 0) && (disk != FDSK_ANY))
   {
      sprintf( command, "%2hu", disk);
      TxSelStringSelect( list, command, 2);
   }
   else
   {
      list->selected = 0;                       // default to 'all disks'
   }
   dfsDgGenWidgets[DFSDGGENL1].lb.list = list;
   dfsDgGenWidgets[DFSDGGENL1].title   = " Generate partition-script for disk ";

   strcpy( fspec, "*.dfs");
   if (txwSaveAsFileDialog( fspec, NULL, base, DFSC_GENPDLG, NULL, &dfsDgGenDlg,
       " Specify GENPART options and basename for the scriptfile ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      if (list)                                 // get selected disk nr
      {
         item = list->items[ list->selected]->value -TXDID_MAX;
      }
      sprintf( command, "genpart %u \"%s\" ", item, fspec);
      strcat( command, (genc1) ? "-s " : "-s- ");
      strcat( command, (genc2) ? "-f- " : "-f ");
      if (strlen( gene1) != 0)                  // Description specified
      {
         strcat(  command, gene1);
      }
      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsGenPartDialog'
/*-----------------------------------------------------------------------------------------------*/



/*========================== DFSDISK ============================================================*/
static BOOL        dskrA  = FALSE;              // Search ALL sectors (slow!)
static BOOL        dskrB  = TRUE;               // on MiB and Cyl boundaries
static BOOL        dskrC  = FALSE;              // Search Cyl only    (fast!)
static BOOL        dskrM  = FALSE;              // Search MiB only
static BOOL        dskr2  = FALSE;              // Partition, BR and LVM only
static BOOL        dskr3  = TRUE;               // Part, BR, LVM + ANY superblock
static BOOL        dskr4  = FALSE;              // Include HPFS   superblocks
static BOOL        dskr5  = FALSE;              // Include JFS    superblocks
static BOOL        dskr6  = FALSE;              // Include NTFS   NTLDR sectors
static BOOL        dskr7  = FALSE;              // Include HFS+   superblocks
static BOOL        dskj5  = FALSE;              // Include EXT2/3 superblocks
static BOOL        dskj6  = FALSE;              // Include REISER superblocks
static BOOL        dskj7  = FALSE;              // Include XFS    superblocks

static TXTS        dske1  = "calculated";       // #Cylinders
static TXTS        dske2  = "current";          // #Heads
static TXTS        dske3  = "current";          // #Sectors

/*
   FULL Analyse disk: [1 Maxtor 114.5 GiB      117239.9 MiB           ]

() Partitioning and LVM only    #Cylinders     #Heads      #Sect/track
( ) Search HPFS   structures    [calculated]   [current]   [current    ]
( ) Search NTFS   structures
( ) Search JFS    structures   ( ) On MiB only  () On Cyl only (fastest)
( ) Search EXT2   structures   ( ) Search on Cylinder and MiB boundaries
( ) Search REISER structures   ( ) Search ALL sectors           (slowest)

*/
#define   DFSDISKWIDGETS 18
#define   DFSDISKL1       1                     // index of list-1
static TXWIDGET  dfsDiskWidgets[ DFSDISKWIDGETS] = // order determines TAB-order!
{
   {0,  2, 1, 18, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "Analyse disk:")},
   {0, 20, 1, 51, 0, 0, 0, DFSDSLISTBOX, 0, TXStdLstBox( NULL, 0, NULL)},

   {2, 31, 2, 12, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( dske1, TXMAXTS, "#Cylinders")},
   {2, 46, 2,  9, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( dske2, TXMAXTS, "#Heads")},
   {2, 58, 2, 13, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( dske3, TXMAXTS, "#Sect/track")},

   {2,  0, 1, 32, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskr2, "Partitioning and LVM only")},
   {3,  0, 1, 32, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskr3, "Part/LVM + ANY superblock")},
   {4,  0, 1, 32, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskr4, "+ HPFS superblocks only")},
   {5,  0, 1, 13, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskr5, "+ JFS")},
   {6,  0, 1, 13, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskr6, "+ NTFS")},
   {7,  0, 1, 13, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskr7, "+ HFS+")},
   {5, 14, 1, 13, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskj5, "+ EXT2")},
   {6, 14, 1, 13, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskj6, "+ REISER")},
   {7, 14, 1, 13, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &dskj7, "+ XFS")},

   {5, 30, 1, 16, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &dskrM, "On MiB only")},
   {5, 47, 1, 26, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &dskrC,                  "On Cyl only (fastest)")},
   {6, 30, 1, 44, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &dskrB, "Search on Cylinder and MiB boundaries")},
   {7, 30, 1, 44, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &dskrA, "Search ALL sectors           (slowest)")},
};

static TXGW_DATA dfsDiskDlg =
{
   DFSDISKWIDGETS,                              // number of widgets
   DFSC_DISKDLG,                                // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDiskWidgets                               // array of widgets
};


/*************************************************************************************************/
// Present DFSDISK options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsDfsDiskDialog
(
   USHORT              disk,                    // IN    disk number string
   char               *base                     // IN    base filename
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;
   TXSELIST           *list = NULL;
   ULONG               item = 0;                // default to ALL disks
   TXTM                title;
   TXGW_DATA          *widgets = NULL;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   list = dfsa->slDiskAll;
   if ((disk != 0) && (disk != FDSK_ANY))
   {
      sprintf( command, "%2hu", disk);
      TxSelStringSelect( list, command, 2);
   }
   else
   {
      list->selected = 0;                       // default to 'all disks'
   }
   dfsDiskWidgets[ DFSDISKL1].lb.list = list;
   dfsDiskWidgets[ DFSDISKL1].title   = " Generate DFSDISK analysis for disk ";

   if (dfsa->expertui)
   {
      strcpy( title, " Specify DFSDISK options and basename for the report files ");
      widgets = &dfsDiskDlg;
   }
   else
   {
      strcpy( title, " Specify a base filename for the analysis report files ");
   }

   strcpy( fspec, "*.sk?");
   if (txwSaveAsFileDialog( fspec, NULL, base, DFSC_DISKDLG, NULL, widgets, title, fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      if (list)                                 // get selected disk nr
      {
         item = list->items[ list->selected]->value -TXDID_MAX;
      }
      sprintf( command, "dfsdisk %u -b:\"%s\" ", item, fspec);
      if      ( dskr3)           strcat( command, "-s:ANY ");
      if      ( dskr4)           strcat( command, "-s:HPFS ");
      if      ( dskr5)           strcat( command, "-s:JFS ");
      if      ( dskr6)           strcat( command, "-s:NTFS ");
      if      ( dskr7)           strcat( command, "-s:HFS ");
      if      ( dskj5)           strcat( command, "-s:EXT2 ");
      if      ( dskj6)           strcat( command, "-s:REISER ");
      if      ( dskj7)           strcat( command, "-s:XFS ");
      if      ( dskrA)
      {
         strcat( command, "ALL");               // ALL sectors, very slow
      }
      else if ( dskrM)
      {
         strcat( command, "MiB");               // MiB boundaries only
      }
      else                                      // Cyl or Both, need GEO stuff
      {
         if   ( dskrC)
         {
            strcat( command, "CYL ");           // Cylinders only
         }
         else                                   // default, do Both
         {
            strcat( command, "BOTH ");          // Cyl and MiB boundaries
         }
         if ((dske1[0] != 'c') || (dske2[0] != 'c') || (dske3[0] != 'c'))
         {
            strcat( command, (dske1[0] != 'c') ? dske1 : "?");
            strcat( command, " ");
            strcat( command, (dske2[0] != 'c') ? dske2 : "");
            strcat( command, " ");
            strcat( command, (dske3[0] != 'c') ? dske3 : "");
         }
      }
      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsDfsDiskDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== DFSFAST ============================================================*/

/*
   FAST Analyse disk: [1 Maxtor 114.5 GiB      117239.9 MiB           ]

*/
#define   DFSFASTWIDGETS  2
#define   DFSFASTL1       1                     // index of list-1
static TXWIDGET  dfsFastWidgets[DFSFASTWIDGETS] = // order determines TAB-order!
{
   {0,  2, 1, 18, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "FAST Analyse disk:")},
   {0, 20, 1, 51, 0, 0, 0, DFSDSLISTBOX, 0, TXStdLstBox( NULL, 0, NULL)},
};

static TXGW_DATA dfsFastDlg =
{
   DFSFASTWIDGETS,                              // number of widgets
   DFSC_FASTDLG,                                // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsFastWidgets                               // array of widgets
};


/*************************************************************************************************/
// Present DFSFAST options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsDfsFastDialog
(
   USHORT              disk,                    // IN    disk number string
   char               *base                     // IN    base filename
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;
   TXSELIST           *list = NULL;
   ULONG               item = 0;                // default to ALL disks
   TXTM                title;
   TXGW_DATA          *widgets = NULL;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   list = dfsa->slDiskAll;
   if ((disk != 0) && (disk != FDSK_ANY))
   {
      sprintf( command, "%2hu", disk);
      TxSelStringSelect( list, command, 2);
   }
   else
   {
      list->selected = 0;                       // default to 'all disks'
   }
   dfsFastWidgets[ DFSFASTL1].lb.list = list;
   dfsFastWidgets[ DFSFASTL1].title   = " Generate DFSFAST analysis for disk ";

   if (dfsa->expertui)
   {
      strcpy( title, " Specify DFSFAST options and basename for the report files ");
      widgets = &dfsFastDlg;
   }
   else
   {
      strcpy( title, " Specify a base filename for the analysis report files ");
   }

   strcpy( fspec, "*.sk?");
   if (txwSaveAsFileDialog( fspec, NULL, base, DFSC_FASTDLG, NULL, widgets, title, fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      if (list)                                 // get selected disk nr
      {
         item = list->items[ list->selected]->value -TXDID_MAX;
      }
      sprintf( command, "dfsfast %u -b:\"%s\" ", item, fspec);

      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsDfsFastDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== LOG ================================================================*/
static TXLN        logo1;                       // Output field user message
static BOOL        logc1;                       // Log-reopen at each line

/*
  Specify a filename on a WRITABLE volume/driveletter for logging
  All new screen output will be APPENDED to the specified file.")

  Create logfile for the session, or use Cancel/Esc for no logging

[ ] Close and re-open the logfile after writing each line (slow!)
*/
#define   DFSDGLOGWIDGETS 4
static TXWIDGET  dfsDgLogWidgets[DFSDGLOGWIDGETS] =    // order determines TAB-order!
{
   {0,  2, 1, 65, 0, 0, 1, DFSDSOUTPUT,  0, TXStdStline( fdescr)},
   {1,  2, 1, 65, 0, 0, 1, DFSDSOUTPUT,  0, TXStdStline( tdescr)},
   {3,  2, 1, 65, 0, 0, 1, DFSDSOUTPUT,  0, TXStdStline( logo1)},

   {5,  0, 1, 65, 0, 7, 0, TXWS_AUTOCHK, 0, TXStdButton( &logc1,
   "Close and reopen file after writing each line (slow!)")},
};

static TXGW_DATA dfsDgLogDlg =
{
   DFSDGLOGWIDGETS,                             // number of widgets
   DFSC_LOGF,                                   // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgLogWidgets                              // array of widgets
};


/*************************************************************************************************/
// Present LOG/TRACE options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsLogDialog
(
   char               *logname,                 // IN    default name or NULL
   ULONG               helpid,                  // IN    specific help-id
   BOOL                reopen,                  // IN    reopen logfile
   char               *message                  // IN    extra message or NULL
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   dfsDgLogDlg.helpid = helpid;
   logc1 = reopen;

   strcpy( fdescr, "Specify a filename on a WRITABLE volume/driveletter for logging");
   strcpy( tdescr, "All new screen output will be APPENDED to the specified file.");

   if (message && strlen( message))
   {
      strcpy( logo1, message);
   }
   else
   {
      strcpy( logo1, "Start logfile for the session, or use Cancel/Esc for no logging");
   }

   strcpy( fspec, "*.log");
   if (txwSaveAsFileDialog( fspec, NULL, logname, helpid, NULL, &dfsDgLogDlg,
       " Specify filename for logging this session to ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      if (helpid == DFSC_LOGF)                  // regular LOG
      {
         sprintf( command, "log \"%s\"", fspec); // allow space/single-quote
      }
      else
      {
         sprintf( command, "trace \"%s\" -m:999,k -f:9 -t", fspec); // trace
      }
      if (logc1)
      {
         strcat( command, " -r");               // reopen option
      }
      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsLogDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== HELP ===============================================================*/
static TXTM        hlogo;                       // Output field user message
static TXLN        descr;                       // Input field for optional description

/*
   Description for selected additional help file (defaults to path+filename)

  [                                                                         ]

*/
#define   DFSDGHELPWIDGETS 2
static TXWIDGET  dfsDgHelpWidgets[DFSDGHELPWIDGETS] =    // order determines TAB-order!
{
   {0,  1, 1, 76, 0, 0, 1, DFSDSOUTPUT,  0, TXStdStline( hlogo)},
   {2,  0, 1, 74, 0, 0, 0, TXWS_ENTRYB,  0, TXStdEntryf( descr, TXMAXTM, "")},
};

static TXGW_DATA dfsDgHelpDlg =
{
   DFSDGHELPWIDGETS,                            // number of widgets
   DFSC_H_LOAD,                                 // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgHelpWidgets                             // array of widgets
};


/*************************************************************************************************/
// Present HELP load-section from file dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsLoadHelpDialog
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   strcpy( hlogo, "Description for selected additional help file (defaults to path+filename)");

   strcpy( fspec, "*.txt");
   if (txwOpenFileDialog( fspec, NULL, NULL, DFSC_H_LOAD, NULL, &dfsDgHelpDlg,
       " Specify filename to load additional help from ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      TxStrip( descr, descr, ' ', ' ');
      if (descr[0] == 0)                        // empty description ?
      {
         strcpy( descr, fspec);                 // use full path and filename
      }
      sprintf( command, "loadhelp \"%s\" \"%s\"", fspec, descr);

      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsLoadHelpDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== Open Image =========================================================*/
static BOOL bReadOnly = TRUE;                   // Open image in Readonly mode
static BOOL bAppendOk = FALSE;                  // Allow append on RAW, when R/W
static BOOL bInitFsys = TRUE;                   // Initialize found filesystem

/*
[x] Open the image in Read-Only mode, no changes allowed
[ ] Allow appending to RAW image if R/W, changing the size
[x] Switch to found filesystem (Mode=xxx), when recognised
*/
#define   DFSDGIMAGEWIDGETS 3
static TXWIDGET  dfsDgImageWidgets[DFSDGIMAGEWIDGETS] = // order determines TAB-order!
{
   {0,  0, 1, 58, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( &bReadOnly, "Open the image in Read-Only mode, no changes allowed  ")},
   {1,  0, 1, 58, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( &bAppendOk, "Allow appending to RAW image if R/W, changing the size")},
   {2,  0, 1, 58, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( &bInitFsys, "Switch to found filesystem (Mode=xxx), when recognised")}
};

static TXGW_DATA dfsDgImageDlg =
{
   DFSDGIMAGEWIDGETS,                           // number of widgets
   DFSC_OIMAG,                                  // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgImageWidgets                            // array of widgets
};


/*************************************************************************************************/
// Present File-open dialog for images, and execute resulting IM/IMZ command to open it
/*************************************************************************************************/
ULONG dfsImageSelectDialog
(
   char               *spec,                    // IN    dir or wildcard specification
   DFSMEDIATYPE        mtype,                   // IN    mediatype (determines cmd to execute)
   char               *ext                      // IN    default file extension wanted
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;
   TXTS                option = {0};
   char               *path  = NULL;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   if (TxaOptSet('R'))
   {
      bReadOnly = TxaOption('R');
   }
   if (TxaOptSet('A'))
   {
      bAppendOk = TxaOption('A');
   }
   if (TxaOptSet('f'))
   {
      bInitFsys = TxaOption('f');
   }

   sprintf(  fspec, "*.%s", ext);
   if (spec[ strlen(spec) -1] == FS_PATH_SEP)   // path specified
   {
      path = spec;
   }
   else if (strlen( spec))
   {
      strcpy( fspec, spec);
   }
   if (txwOpenFileDialog( fspec, path, NULL, DFSC_OIMAG, NULL, (mtype == DFSD_IRAW) ? &dfsDgImageDlg : NULL,
                         " Specify image filename to be opened for analysis or BROWSE ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      switch (mtype)
      {
         case DFSD_VBOX:
            sprintf( command, "vdi \"%s\"", fspec);
            break;

         case DFSD_IMZD:
            if (TxaOptSet( 'p'))                // preserve pre-selected partition number
            {
               sprintf( option, " -p:%llu", TxaOptNum( 'p', NULL, 1)); // default FIRST
            }
            sprintf( command, "imz %s \"%s\"", option, fspec);
            break;

         case DFSD_IRAW:
         default:
            sprintf( command, "im \"%s\"",  fspec);
            if (!bReadOnly)
            {
               strcat( command, " -R-");
               if (bAppendOk)
               {
                  strcat( command, " -A-");
               }
            }
            if (!bInitFsys)
            {
               strcat( command, " -f-");
            }
            break;
      }
      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsImageSelectDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== PRESTORE ===========================================================*/
static TXTS  pree1 = "";                        // Disknr to restore to
static TXTS  pree2 = "reGglLb";                 // Sectortypes to restore (ALL)

static BOOL  prec1 = FALSE;                     // Prompt for each sector to do
static BOOL  prec2 = FALSE;                     // List only, no sector restore
static BOOL  prec3 = FALSE;                     // Display full sector contents

/*
[SAME] Disknr to restore .PDx to  [reGglLb]  Restrict to specified types
'SAME' gets disknr from filename             Empty to restore *ANY* type

                                   r = MBR sectors      e = EBR sectors
[ ] Prompt for each sector to do   G = GPT headers      g = GPT entries
[ ] List only, no sector restore   l = LVM info, DLAT   s = LVM sig, BBR
[ ] Display full sector contents   b = Partition boot sectors
*/
#define   DFSDGPREWIDGETS 13
static TXWIDGET  dfsDgPreWidgets[DFSDGPREWIDGETS] = // order determines TAB-order!
{
   {0,  0, 1,  6, 0, 0, 0, TXWS_ENTRYB,  0, TXStdEntryf( pree1, TXMAXTS, "")},
   {0,  7, 1, 25, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline(         "Disknr to restore .PDx to")},
   {1,  0, 1, 32, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline(  "'SAME' gets disknr from filename")},

   {0, 34, 1,  9, 0, 0, 0, TXWS_ENTRYB,  0, TXStdEntryf( pree2, TXMAXTS, "")},
   {0, 45, 1, 28, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "Restrict to specified types")},
   {1, 45, 1, 28, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "Empty to restore *ANY* type")},
   {3, 35, 1, 38, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "r = MBR sectors      e = EBR sectors")},
   {4, 35, 1, 38, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "G = GPT headers      g = GPT entries")},
   {5, 35, 1, 38, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "l = LVM info, DLAT   L = LVM sig, BBR")},
   {6, 35, 1, 38, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "b = Boot sectors         (+ features)")},

   {4,  0, 1, 32, 0, 1, 0, TXWS_AUTOCHK, 0, TXStdButton( &prec1, "Prompt for each sector to do")},
   {5,  0, 1, 32, 0, 1, 0, TXWS_AUTOCHK, 0, TXStdButton( &prec2, "List only, no sector restore")},
   {6,  0, 1, 32, 0, 1, 0, TXWS_AUTOCHK, 0, TXStdButton( &prec3, "Display full sector contents")}
};

static TXGW_DATA dfsDgPreDlg =
{
   DFSDGPREWIDGETS,                             // number of widgets
   DFSC_RESTPD,                                 // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgPreWidgets                              // array of widgets
};

/*************************************************************************************************/
// Present PRESTORE options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsPrestoreDialog
(
   USHORT              disknr                   // IN    disknr or 0
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   if (disknr == 0)
   {
      strcpy( pree1, "SAME");
   }
   else
   {
      sprintf( pree1, "%hu", disknr);
   }
   strcpy( fspec, "*.pd?");
   if (txwOpenFileDialog( fspec, NULL, NULL, DFSC_RESTPD, NULL, &dfsDgPreDlg,
       " Select .PDx file to be restored, and set options ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      sprintf( command, "prestore %s \"%s\" %s",
               (isdigit( pree1[0])) ?  pree1 : "*",
                fspec,   pree2);

      if (!prec1) strcat( command, " -c-");     // no prompting
      if ( prec2) strcat( command, " -l");      // list only
      if ( prec3) strcat( command, " -v");      // verbose

      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsPrestoreDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== PSAVE ==============================================================*/
static TXTM  psae1 = "Partition info backup selected from DFSee menu";

/*
Partition info backup selected from DFSee menu
 Description to be added to the .PDx BACKUP files
[                                                  ]
*/
#define   DFSDGPSAWIDGETS 1
static TXWIDGET  dfsDgPsaWidgets[DFSDGPSAWIDGETS] = // order determines TAB-order!
{
   {0,  0, 2, 50, 0, 0, 0, TXWS_ENTRYBT  |  TXWS_HCHILD_SIZE, 0, TXStdEntryf( psae1, TXMAXTM,
                          "Description to be added to the .PDx BACKUP files")}
};

static TXGW_DATA dfsDgPsaDlg =
{
   DFSDGPSAWIDGETS,                             // number of widgets
   DFSC_SAVEPD,                                 // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgPsaWidgets                              // array of widgets
};

/*************************************************************************************************/
// Present PSAVE options dialog and execute resulting command
/*************************************************************************************************/
ULONG dfsPsaveDialog
(
   USHORT              disknr                   // IN    disknr or 0
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   TXLN                fspec;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   strcpy( fspec, "*.pd?");
   if (txwSaveAsFileDialog( fspec, NULL, DFS_X, DFSC_SAVEPD, NULL, &dfsDgPsaDlg,
       " Specify filename and description for partition info BACKUP ", fspec))
   {
      TxRepl( fspec, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
      sprintf( command, "psave %hu \"%s\" %s", disknr, fspec, psae1);

      dfsExecEnd( command);
   }
   dfsENDWORK();                                // signal work done
   RETURN (rc);
}                                               // end 'dfsPsaveDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== LIST/RECOVER files =================================================*/
static BOOL flsr1  = TRUE;                      // Radio Minimum
static BOOL flsr2  = FALSE;                     // Radio Maximum
static BOOL flsrK  = FALSE;                     // KiB
static BOOL flsrM  = FALSE;                     // MiB
static BOOL flsrG  = FALSE;                     // GiB
static BOOL flsrS  = TRUE;                      // Sec
static BOOL flsrN  = TRUE;                      // Normal
static BOOL flsrF  = FALSE;                     // Files
static BOOL flsrB  = FALSE;                     // Browse

static TXTS flse1  = "100";                     // percentage value
static TXTS flse2  = "0";                       // minimum filesize
static TXTS flse3  = "*";                       // maximum filesize

static TXTS flse4  = "Anytime, no restriction"; // Modified timestamp value (0 = 1970 ...)

/*
    Recovery reliability                          File size  () KiB
(*) minimum  percentage                  minimum [.........] ( ) MiB
( ) maximum [..........] (0 = no limit!) maximum [.........] ( ) GiB
                                                             ( ) sectors
    Filtering
(*) Browse: filter on Files, but keep all Directories in the list
( ) Files : filter on Files, discard all Directories
( ) Normal: filter on Files and Directories
                                                  #Days-ago or Date/Time
    Restrict to files created or modified after: [Anytime, no restriction]
*/

#define   DFSDGFLISTWIDGETS 20
static TXWIDGET  dfsDgFlistWidgets[DFSDGFLISTWIDGETS] =  // order determines TAB-order!
{
   {0,  4, 1, 22, 0, 0, 0, DFSDSOUTPUT,     0, TXStdStline(         "Recovery reliability")},
   {1,  0, 1, 12, 0, 1, 0, TXWS_AUTORAD,    0, TXStdButton( &flsr1, "minimum")},
   {2,  0, 1, 12, 0, 1, 0, TXWS_AUTORAD,    0, TXStdButton( &flsr2, "maximum")},
   {1, 12, 2, 12, 0, 0, 0, TXWS_ENTRYBT,    0, TXStdEntryf( flse1, TXMAXTS, "percentage")},

   {2, 25, 1, 16, 0, 0, 0, DFSDSOUTPUT,     0, TXStdStline( "(0 = no limit!)")},

   {0, 50, 1, 11, 0, 0, 0, DFSDSOUTPUT,     0, TXStdStline(         "File size")},
   {1, 41, 1,  8, 0, 0, 0, DFSDSOUTPUT,     0, TXStdStline(         "minimum")},
   {1, 49, 1, 11, 0, 0, 0, TXWS_ENTRYB,     0, TXStdEntryf( flse2, TXMAXTS, "")},
   {2, 41, 1,  8, 0, 0, 0, DFSDSOUTPUT,     0, TXStdStline(         "maximum")},
   {2, 49, 1, 11, 0, 0, 0, TXWS_ENTRYB,     0, TXStdEntryf( flse3, TXMAXTS, "")},

   {0 ,61, 1,  7, 0, 4, 0, DFSDMAUTORAD,    0, TXStdButton( &flsrK, "KiB")},
   {1 ,61, 1,  7, 0, 4, 0, DFSDMAUTORAD,    0, TXStdButton( &flsrM, "MiB")},
   {2 ,61, 1,  7, 0, 4, 0, DFSDMAUTORAD,    0, TXStdButton( &flsrG, "GiB")},
   {3, 61, 1, 11, 0, 4, 0, DFSDMAUTORAD,    0, TXStdButton( &flsrS, "Sectors")},

   {4,  4, 1, 22, 0, 0, 0, DFSDSOUTPUT,     0, TXStdStline(         "Filtering")},
   {5,  0, 1, 70, 0, 2, 0, TXWS_AUTORAD,    0, TXStdButton( &flsrB, "Browse: filter on Files, but keep all Directories in the list")},
   {6,  0, 1, 70, 0, 2, 0, TXWS_AUTORAD,    0, TXStdButton( &flsrF, "Files : filter on Files, discard all Directories")},
   {7,  0, 1, 45, 0, 2, 0, TXWS_AUTORAD,    0, TXStdButton( &flsrN, "Normal: filter on Files and Directories")},

   {8, 49, 2, 25, 0, 0, 0, TXWS_ENTRYBT, 5209, TXStdEntryf( flse4, TXMAXTS, "#Days-ago or Date/Time")},
   {9,  4, 1, 45, 0, 0, 0, DFSDSOUTPUT,     0, TXStdStline(         "Restrict to files created or modified after:")}
};

#define DFSDGFLIST_HELP    5208

static TXGW_DATA dfsDgFlistDlg =
{
   DFSDGFLISTWIDGETS,                           // number of widgets
   DFSDGFLIST_HELP,                             // dialog help
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgFlistWidgets                            // array of widgets
};

static TXLN   wildcard = "";                    // wildcard string


/*************************************************************************************************/
// Present LIST/RECOVER/BROWSE file selection dialog and return compound selection string
/*************************************************************************************************/
ULONG dfsFileListDialog                         // RET   result
(
   char               *title,                   // IN    dialog title
   ULONG               count,                   // IN    listed files, info only
   BOOL                browse,                  // IN    Browse, unless overruled
   TXTM                select                   // INOUT file selection string
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                prompt;                  // prompt text
   ULONG               threshold = 0;           // threshold percentage
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse ...
   time_t              modTstamp = 0;           // Modify timestamp value, or 0

   ENTER();

   itemType = (browse) ? 'B' : 'F';             // default when not set by select string
   if (strlen( select))                         // string on input ?
   {
      dfsParseFileSelection( select, wildcard, &flsr1, &threshold, &minS, &maxS, &modTstamp, &itemType);
      sprintf( flse1, "%u", threshold);
      sprintf( flse2, "%u", minS);
      sprintf( flse3, "%u", maxS);
      flsrS = TRUE;                             // always SECTORS from parsed string!
      switch (itemType)
      {
         case 'B': flsrB = TRUE; flsrN = FALSE; flsrF = FALSE; break;
         case 'F': flsrF = TRUE; flsrN = FALSE; flsrB = FALSE; break;
         case 'N':
         default:  flsrN = TRUE; flsrF = FALSE; flsrB = FALSE; break;
      }
   }
   sprintf( prompt, "The sector list currently contains "
                    "%u potential filenames.\n\n"
                    "Specify a wildcard for PATH+filename "
                    "or leave blank to do entire list.", count);

   if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &dfsDgFlistDlg,
         prompt, title, DFSDGFLIST_HELP,
         TXPB_MOVEABLE | TXPB_HCENTER |  TXPB_VCENTER,
         TXMAXTM - TXMAXTS, wildcard) != TXDID_CANCEL)
   {
      TxStrip( wildcard, wildcard, ' ', ' ');   // strip leading/trailing spaces
      if (strlen( wildcard))
      {
         strcpy( select, wildcard);
      }
      else
      {
         strcpy( select, "*");                  // explicit 'everything'
      }
      strcat( select, "%");
      if (strlen( flse1))
      {
         if ( flsr2) strcat( select, "-");
         strcat(             select, flse1);
      }
      strcat( select, "%");
      TxStrip( flse2, flse2, ' ', ' ');
      strcat( select, flse2);
      if (strcmp( flse2, "0") != 0)
      {
         if ( flsrK) strcat( select, ",k");
         if ( flsrM) strcat( select, ",m");
         if ( flsrG) strcat( select, ",g");
         if ( flsrS) strcat( select, ",s");
      }
      strcat( select, "%");
      TxStrip( flse3, flse3, ' ', ' ');
      strcat( select, flse3);
      if (strcmp( flse3, "*") != 0)
      {
         if ( flsrK) strcat( select, ",k");
         if ( flsrM) strcat( select, ",m");
         if ( flsrG) strcat( select, ",g");
         if ( flsrS) strcat( select, ",s");
      }
      strcat( select, "%");
      if ( flsrN) strcat( select, "N");
      if ( flsrF) strcat( select, "F");
      if ( flsrB) strcat( select, "B");

      strcat( select, "%");                     // always include datetime value
      TxStrip( flse4, flse4, ' ', ' ');         // empty will be ALL (1970 ...)
      strcat( select, flse4);
   }
   else
   {
      rc = DFS_NO_CHANGE;
   }
   TRACES(("select: '%s'\n", select));
   RETURN (rc);
}                                               // end 'dfsFileListDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== SLT display ========================================================*/
static BOOL sltr1  = TRUE;                      // Radio every
static BOOL sltr2  = FALSE;                     // Radio error-only
static BOOL sltr3  = FALSE;                     // Radio error-value

static BOOL sltr4  = FALSE;                     // Radio default format
static BOOL sltr5  = FALSE;                     // Radio verbose
static BOOL sltr6  = TRUE;                      // Radio verbose 1-liner

static BOOL sltr7  = TRUE;                      // Radio start
static BOOL sltr8  = FALSE;                     // Radio to-end
static BOOL sltr9  = FALSE;                     // Radio at-sector

static BOOL sltc1  = FALSE;                     // Force rebuild

static TXTS slte0  = "*";                       // sector type
static TXTS slte1  = "";                        // error mask
static TXTS slte2  = "";                        // slt start index
static TXTS slte3  = "this";                    // at hex sector

static TXTM slto1  = "";                        // SLT status string

/*
SLT Areas:  4132      Ctrl+R refreshes  status: Being built now

[ ] Rebuild SLT                Display areas for sector type: [..]

() One-line incl filenames   () Display every SLT entry
( ) Multiple-lines, verbose   ( ) Errors only  (CHECK)
( ) Default area info only    ( ) For hex ERROR flags: [.........]

() Show SLT from default or a specified start index:  [.........]
( ) Show SLT area exactly to end of table/filesystem
( ) Show SLT area containing a specified hex sector:   [.........]
*/
#define   DFSDGSLTWIDGETS 16
static TXWIDGET  dfsDgSltWidgets[DFSDGSLTWIDGETS] =  // order determines TAB-order!
{
   {0,  0, 1, 66, 0, 0, 0, DFSDSOUTPUT,  0, "", TXW_STLINE, 0, dfsSltWinProc, TXWgStline(slto1)},

   {2,  0, 1, 16, 0, 0, 0, DFSDMAUTOCHK, 0, TXStdButton( &sltc1, "Rebuild SLT")},
   {2, 31, 1, 30, 0, 0, 0, DFSDSOUTPUT,  0, TXStdStline( "Display areas for sector type:")},
   {2, 62, 1,  4, 0, 0, 0, TXWS_ENTRYB,  0, TXStdEntryf(  slte0,  TXMAXTS, "")},

   {4,  0, 1, 28, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr6, "One-line incl filenames")},
   {5,  0, 1, 28, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr5, "Multiple-lines, verbose")},
   {6,  0, 1, 28, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr4, "Default area info only ")},
   {4, 30, 1, 36, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr1, "Display every SLT entry")},
   {5, 30, 1, 26, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr2, "Errors only  (CHECK)")},
   {6, 30, 1, 26, 0, 2, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr3, "For hex ERROR flags:")},
   {6, 55, 1, 11, 0, 0, 0, TXWS_ENTRYB,  0, TXStdEntryf(  slte1,  TXMAXTS, "")},


   {8,  0, 1, 54, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr7, "Show SLT from default or a specified start index:")},
   {9,  0, 1, 54, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr8, "Show SLT area exactly to end of table/filesystem")},
   {10, 0, 1, 54, 0, 3, 0, TXWS_AUTORAD, 0, TXStdButton( &sltr9, "Show SLT area containing a specified hex sector:")},
   {8, 55, 1, 11, 0, 0, 0, TXWS_ENTRYB,  0, TXStdEntryf(  slte2,  TXMAXTS, "")},
   {10,55, 1, 11, 0, 0, 0, TXWS_ENTRYB,  0, TXStdEntryf(  slte3,  TXMAXTS, "")},
};

#define DFSDGSLT_HELP      5555

static TXGW_DATA dfsDgSltDlg =
{
   DFSDGSLTWIDGETS,                             // number of widgets
   DFSDGSLT_HELP,                               // SLT dialog help
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgSltWidgets                              // array of widgets
};

static TXTT   pagesize = "";                    // SLT pagesize

/*************************************************************************************************/
// Present SLT display option dialog and execute resulting SLT command
/*************************************************************************************************/
ULONG dfsSltDisplayDialog                       // RET   result
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;

   ENTER();

   if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &dfsDgSltDlg,
         "Specify number of lines to display, '*' to show ALL\n"
         "or leave empty for a default page size of one screen ...",
         " Specify options for displaying the Sector Lookup Table ",
         DFSDGSLT_HELP, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
         TXMAXTT, pagesize) != TXDID_CANCEL)
   {
      TxStrip( slte2, slte2, ' ', ' ');
      sprintf( command, "slt %s %s", strlen(slte2) ? slte2 : ".", pagesize);

      if (sltr2) strcat( command, " -m");       // errors only
      if (sltr3)
      {
         TxStrip( slte1, slte1, ' ', ' ');
         strcat( command, " -m:0x");            // specifies errors only
         strcat( command, slte1);
      }

      if (sltr4) strcat( command, " -v- -1-");  // simple
      if (sltr5) strcat( command, " -v  -1-");  // verbose
      if (sltr6) strcat( command, " -v  -1");   // verbose 1-liners

      if (sltr8) strcat( command, " -e");       // upto end
      if (sltr9)
      {
         TxStrip( slte3, slte3, ' ', ' ');
         strcat( command, " -a:");              // at specified LSN
         strcat( command, slte3);
      }
      if (slte0[0] != '*')
      {
         strcat( command, " -t:");              // specific type
         strcat( command, slte0);
      }
      if      (sltc1) strcat( command, " -r");  // force rebuild

      #if defined (HAVETHREADS)
         if (dfsSlTableStatus(NULL) == SLT_UPDATE) // not updating
         {
            TxNamedMessage( TRUE, 5556, " INFO: SLT not ready yet ",
               "      The Sector Lookup Table is not ready yet.\n\n"
               "Results will be displayed automatically once complete.\n"
               "While waiting, the <Esc> key will return you to the menu\n"
               "or cmdline to do other work on the same partition/volume");
         }
      #endif
      dfsProgressSuspend();
      dfsExecEnd( command);
   }
   else
   {
      rc = DFS_NO_CHANGE;
   }
   RETURN (rc);
}                                               // end 'dfsSltDisplayDialog'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the SLT-display dialog field; update status
/*****************************************************************************/
static ULONG dfsSltWinProc                      // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      switch (msg)
      {
         case TXWM_CREATE:                      // setup event hook
            txwAttachEventHook( DFS_EHK_SLTBUILD, hwnd);
            break;

         case TXWM_HOOKEVENT:                   // force a repaint ...
            txwInvalidateWindow( hwnd, FALSE, FALSE);
            break;

         case TXWM_DESTROY:                     // cancel event hook
            txwDetachEventHook( DFS_EHK_SLTBUILD, hwnd);
            break;

         case TXWM_PAINT:                       // update before each paint
            dfsGetSltStatusString( slto1, "Ctrl+R refreshes");
         default:
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsSltWinProc'
/*---------------------------------------------------------------------------*/


/*========================== CHS-STYLE ==========================================================*/
static BOOL        chsrI  = FALSE;              // IBM
static BOOL        chsrP  = FALSE;              // PQ
static BOOL        chsrM  = FALSE;              // MS
static BOOL        chsOK  = FALSE;              // OK button, dummy variable
static BOOL        chsCA  = FALSE;              // Cancel button dummy
static BOOL        chsc1  = FALSE;              // make default, skip dialog

/*
( ) IBM (FDISK, LVM) and default DFSee style
( ) PowerQuest (Partition Magic, Ghost, DriveImage
( ) Microsoft Windows old standard style

[] Make this default and skip the dialog from now on

        Ŀ              Ŀ
          OK                 Cancel 
                      
*/

#define   DFSDGCHSOK       4                    // index of OK button
#define   DFSDGCHSWIDGETS  6
static TXWIDGET  DfsDgChsWidgets[DFSDGCHSWIDGETS] =    // order determines TAB-order!
{
   {0, 0,  1, 51, 0, 1, 0, DFSDMAUTORAD, 0, TXStdButton( &chsrI, "IBM (FDISK, LVM) and default DFSee style")},
   {1, 0,  1, 51, 0, 1, 0, DFSDMAUTORAD, 0, TXStdButton( &chsrP, "PowerQuest (Partition Magic, Ghost, DriveImage)")},
   {2, 0,  1, 51, 0, 1, 0, DFSDMAUTORAD, 0, TXStdButton( &chsrM, "Microsoft Windows old standard style")},

   {4, 0,  1, 54, 0, 2, 0, TXWS_AUTOCHK, 0, TXStdButton( &chsc1, "Make this default and skip the dialog from now on")},

   {6, 10, 3,  8, 0, 0, TXDID_OK,     DFSDMPBUTTON, 0, TXStdButton( &chsOK, " OK ")},
   {6, 30, 3,  8, 0, 0, TXDID_CANCEL, DFSDMPBUTTON, 0, TXStdButton( &chsCA, "Cancel")}
};

#define DFSDGCHS_HELP      5560

static TXGW_DATA DfsDgChsDlg =
{
   DFSDGCHSWIDGETS,                             // number of widgets
   DFSDGCHS_HELP,                               // help, widget overrules
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   DfsDgChsWidgets                              // array of widgets
};

static ULONG  defaultChsStyle = DFS_CHSTYLE_NONE;

/*************************************************************************************************/
// Present CHS style selection dialog and return the selected value, default dfsa->chsStyle
/*************************************************************************************************/
ULONG dfsChsStyleDialog                         // RET   result
(
   BOOL                reset,                   // IN    Reset default style
   ULONG              *style                    // INOUT CHS style value
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (reset)
   {
      defaultChsStyle = DFS_CHSTYLE_NONE;
   }
   if (defaultChsStyle == DFS_CHSTYLE_NONE)     // not made default yet ...
   {
      if ((!chsrI) && (!chsrP) && (!chsrM))     // no dialog history yet
      {
         switch (dfsa->chsStyle)
         {
            case DFS_CHSTYLE_PQ: chsrP = TRUE; break;
            case DFS_CHSTYLE_MS: chsrM = TRUE; break;
            default:             chsrI = TRUE; break;
         }
      }

      if (txwWidgetDialog( TXHWND_DESKTOP, TXHWND_DESKTOP,
              NULL, " Specify the CHS (dummy) style to be used ",
              TXWD_MOVEABLE | TXWD_HCENTER | TXWD_VCENTER,
              DFSDGCHSOK,   &DfsDgChsDlg) != TXDID_CANCEL)
      {
         if      (chsrP) *style = DFS_CHSTYLE_PQ;
         else if (chsrM) *style = DFS_CHSTYLE_MS;
         else            *style = DFS_CHSTYLE_IBM;

         if (chsc1)                             // set this as default too
         {
            defaultChsStyle = *style;           // for this dialog
            dfsa->chsStyle  = *style;           // and other ChsStyle usage
         }
      }
      else
      {
         rc = DFS_NO_CHANGE;
      }
   }
   else                                         // just return the value that
   {                                            // was made the default
      *style = defaultChsStyle;
   }
   RETURN (rc);
}                                               // end 'dfsChsStyleDialog'
/*-----------------------------------------------------------------------------------------------*/


/*================== PROMPT entryfield plus 1 to 5 options ======================================*/
/*
[x] Description for checkbox 0
[x] Description for checkbox 1
[x] Description for checkbox 2
[x] Description for checkbox 3
[x] Description for checkbox 4
*/
#define   DFSDGPROPTWIDGETS  5
static TXWIDGET  dfsDgPrOptWidgets[DFSDGPROPTWIDGETS] =  // order determines TAB-order!
{
   {0,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {1,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {2,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {3,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {4,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)}
};

static TXGW_DATA dfsDgPrOptDlg =
{
   0,                                           // number of widgets, dynamic
   0,                                           // dialog help, dynamic
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgPrOptWidgets                            // array of widgets
};


/*************************************************************************************************/
// Present PromptBox for a string value, preceeded by 1 to 5 option checkboxes
/*************************************************************************************************/
ULONG dfsPromptOptDialog                        // RET   result
(
   char               *title,                   // IN    dialog title
   char               *message,                 // IN    prompt message
   ULONG               helpid,                  // IN    dialog help
   short               length,                  // IN    entryfield length
   char               *field,                   // INOUT entryfield string
   BOOL               *opt0val,                 // INOUT Option-0 value
   char               *opt0txt,                 // IN    Option-0 text
   BOOL               *opt1val,                 // INOUT Option-1 value or NULL
   char               *opt1txt,                 // IN    Option-1 text  or NULL
   BOOL               *opt2val,                 // INOUT Option-2 value or NULL
   char               *opt2txt,                 // IN    Option-2 text  or NULL
   BOOL               *opt3val,                 // INOUT Option-3 value or NULL
   char               *opt3txt,                 // IN    Option-3 text  or NULL
   BOOL               *opt4val,                 // INOUT Option-4 value or NULL
   char               *opt4txt                  // IN    Option-4 text  or NULL
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   dfsDgPrOptDlg.helpid = helpid;

   dfsDgPrOptDlg.count  = 1;
   dfsDgPrOptDlg.widget[0].bu.text                = opt0txt;
   dfsDgPrOptDlg.widget[0].bu.checked             = opt0val;
   if (opt1val)
   {
      dfsDgPrOptDlg.count = 2;
      dfsDgPrOptDlg.widget[1].bu.text             = opt1txt;
      dfsDgPrOptDlg.widget[1].bu.checked          = opt1val;
      if (opt2val)
      {
         dfsDgPrOptDlg.count = 3;
         dfsDgPrOptDlg.widget[2].bu.text          = opt2txt;
         dfsDgPrOptDlg.widget[2].bu.checked       = opt2val;
         if (opt3val)
         {
            dfsDgPrOptDlg.count = 4;
            dfsDgPrOptDlg.widget[3].bu.text       = opt3txt;
            dfsDgPrOptDlg.widget[3].bu.checked    = opt3val;
            if (opt4val)
            {
               dfsDgPrOptDlg.count = 5;
               dfsDgPrOptDlg.widget[4].bu.text    = opt4txt;
               dfsDgPrOptDlg.widget[4].bu.checked = opt4val;
            }
         }
      }
   }

   if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &dfsDgPrOptDlg,
                     message, title, helpid,
                     TXPB_MOVEABLE | TXPB_HCENTER |  TXPB_VCENTER,
                     length, field) == TXDID_CANCEL)
   {
      rc = DFS_NO_CHANGE;
   }
   RETURN (rc);
}                                               // end 'dfsPromptOptDialog'
/*-----------------------------------------------------------------------------------------------*/


/*================== CONFIRMATION dialog plus 1 to 5 options ====================================*/
/*
[x] Description for checkbox 0
[x] Description for checkbox 1
[x] Description for checkbox 2
[x] Description for checkbox 3
[x] Description for checkbox 4
*/
#define   DFSDGCFOPTWIDGETS  5
static TXWIDGET  dfsDgCfOptWidgets[DFSDGCFOPTWIDGETS] =  // order determines TAB-order!
{
   {0,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {1,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {2,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {3,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {4,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)}
};

static TXGW_DATA dfsDgCfOptDlg =
{
   0,                                           // number of widgets, dynamic
   0,                                           // dialog help, dynamic
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgCfOptWidgets                            // array of widgets
};


/*************************************************************************************************/
// Present Confirmation dialog with fixed text, preceeded by 1 to 5 option checkboxes
/*************************************************************************************************/
BOOL dfsConfirmOptDialog                        // RET   confirmed
(
   ULONG               helpid,                  // IN    dialog help
   char               *message,                 // IN    prompt message
   BOOL               *opt0val,                 // INOUT Option-0 value
   char               *opt0txt,                 // IN    Option-0 text
   BOOL               *opt1val,                 // INOUT Option-1 value or NULL
   char               *opt1txt,                 // IN    Option-1 text  or NULL
   BOOL               *opt2val,                 // INOUT Option-2 value or NULL
   char               *opt2txt,                 // IN    Option-2 text  or NULL
   BOOL               *opt3val,                 // INOUT Option-3 value or NULL
   char               *opt3txt,                 // IN    Option-3 text  or NULL
   BOOL               *opt4val,                 // INOUT Option-4 value or NULL
   char               *opt4txt                  // IN    Option-4 text  or NULL
)
{
   BOOL                rc = NO_ERROR;           // function return

   ENTER();

   dfsDgCfOptDlg.helpid = helpid;

   dfsDgCfOptDlg.count  = 1;
   dfsDgCfOptDlg.widget[0].bu.text                = opt0txt;
   dfsDgCfOptDlg.widget[0].bu.checked             = opt0val;
   if (opt1val)
   {
      dfsDgCfOptDlg.count = 2;
      dfsDgCfOptDlg.widget[1].bu.text             = opt1txt;
      dfsDgCfOptDlg.widget[1].bu.checked          = opt1val;
      if (opt2val)
      {
         dfsDgCfOptDlg.count = 3;
         dfsDgCfOptDlg.widget[2].bu.text          = opt2txt;
         dfsDgCfOptDlg.widget[2].bu.checked       = opt2val;
         if (opt3val)
         {
            dfsDgCfOptDlg.count = 4;
            dfsDgCfOptDlg.widget[3].bu.text       = opt3txt;
            dfsDgCfOptDlg.widget[3].bu.checked    = opt3val;
            if (opt4val)
            {
               dfsDgCfOptDlg.count = 5;
               dfsDgCfOptDlg.widget[4].bu.text    = opt4txt;
               dfsDgCfOptDlg.widget[4].bu.checked = opt4val;
            }
         }
      }
   }
   rc = TxConfirmWidgets( helpid, &dfsDgCfOptDlg, message);

   BRETURN (rc);
}                                               // end 'dfsConfirmOptDialog'
/*-----------------------------------------------------------------------------------------------*/


/*========================== DIRSELECT + CHECKBOX ===============================================*/
/*
[x] Description for checkbox 0
[x] Description for checkbox 1
[x] Description for checkbox 2
[x] Description for checkbox 3
[x] Description for checkbox 4
*/
#define   DFSDGDIRSWIDGETS 5
static TXWIDGET  dfsDgDirsWidgets[DFSDGDIRSWIDGETS] =  // order determines TAB-order!
{
   {0,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {1,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {2,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {3,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)},
   {4,  0, 1, TXMAXTM, 0, 0, 0, TXWS_AUTOCHK, 0, TXStdButton( NULL, NULL)}
};

static TXGW_DATA dfsDgDirsDlg =
{
   0,                                           // number of widgets, dynamic
   0,                                           // dialog help, dynamic
   810,                                         // base window ID
   NULL,                                        // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgDirsWidgets                             // array of widgets
};


/*************************************************************************************************/
// Present Directory select dialog, with one to seven optional Checkboxes
/*************************************************************************************************/
BOOL dfsDirSelectDialog
(
   char               *dlgTitle,                // IN    dialog title
   ULONG               helpid,                  // IN    dialog help
   char               *dirPath,                 // INOUT dir path to select
   BOOL               *opt0val,                 // INOUT Option-0 value
   char               *opt0txt,                 // IN    Option-0 text
   BOOL               *opt1val,                 // INOUT Option-1 value or NULL
   char               *opt1txt,                 // IN    Option-1 text  or NULL
   BOOL               *opt2val,                 // INOUT Option-2 value or NULL
   char               *opt2txt,                 // IN    Option-2 text  or NULL
   BOOL               *opt3val,                 // INOUT Option-3 value or NULL
   char               *opt3txt,                 // IN    Option-3 text  or NULL
   BOOL               *opt4val,                 // INOUT Option-4 value or NULL
   char               *opt4txt,                 // IN    Option-4 text  or NULL
   BOOL               *opt5val,                 // INOUT Option-5 value or NULL
   char               *opt5txt,                 // IN    Option-5 text  or NULL
   BOOL               *opt6val,                 // INOUT Option-6 value or NULL
   char               *opt6txt                  // IN    Option-6 text  or NULL
)
{
   BOOL                rc;                      // function return

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   dfsDgDirsDlg.helpid = helpid;
   dfsDgDirsDlg.count  = 1;
   dfsDgDirsDlg.widget[0].bu.text                 = opt0txt;
   dfsDgDirsDlg.widget[0].bu.checked              = opt0val;
   if (opt1val)
   {
      dfsDgDirsDlg.count = 2;
      dfsDgDirsDlg.widget[1].bu.text              = opt1txt;
      dfsDgDirsDlg.widget[1].bu.checked           = opt1val;
      if (opt2val)
      {
         dfsDgDirsDlg.count = 3;
         dfsDgDirsDlg.widget[2].bu.text           = opt2txt;
         dfsDgDirsDlg.widget[2].bu.checked        = opt2val;
         if (opt3val)
         {
            dfsDgDirsDlg.count = 4;
            dfsDgDirsDlg.widget[3].bu.text        = opt3txt;
            dfsDgDirsDlg.widget[3].bu.checked     = opt3val;
            if (opt4val)
            {
               dfsDgDirsDlg.count = 5;
               dfsDgDirsDlg.widget[4].bu.text     = opt4txt;
               dfsDgDirsDlg.widget[4].bu.checked  = opt4val;
            }
         }
      }
   }
   TRACES(("widget count: %u\n", dfsDgDirsDlg.count));

   rc = txwSelDirFileDialog( dirPath, helpid, NULL, &dfsDgDirsDlg, dlgTitle, dirPath);

   dfsENDWORK();                                // signal work done
   BRETURN (rc);
}                                               // end 'dfsDirSelectDialog'
/*-----------------------------------------------------------------------------------------------*/


/*===================================== GEO =====================================================*/
static BOOL        georReset;                   // Reset to system geometry
static BOOL        georCl255;                   // Classic 255/63  (desktop)
static BOOL        georCl240;                   // Classic 240/63  (laptops)
static BOOL        georOS255;                   // SSD/USB 255/32  (USB-sticks/SSD OS/2)
static BOOL        georMbCyl;                   //  1 MiB   64/32  (SSD/4K sects)
static BOOL        georBgSsd;                   // Big SSD 255/240 (1.9 TB limit)
static BOOL        geor1Tos2;                   // OS2/eCS 255/127 (1.0 TB limit)
static BOOL        geor2Tos2;                   // OS2/eCS 255/255 (2.0 TB limit)
static BOOL        georCustm;                   // Custom geometry, values:

static TXTM        geooX  = "";                 // Size as reported in 8 positions

static TXTM        geosX  = "";                 // Size for geo in 8 positions
static TXTS        geosC  = "";                 // #Cylinders, system geo
static TXTS        geosH  = "";                 // #Heads
static TXTS        geosS  = "";                 // #Sectors

static TXTM        geolX  = "";                 // Size for geo in 8 positions
static TXTS        geolC  = "";                 // #Cylinders, logical geo
static TXTS        geolH  = "";                 // #Heads
static TXTS        geolS  = "";                 // #Sectors

static TXTS        geoeC  = "";                 // #Cylinders
static TXTS        geoeH  = "";                 // #Heads
static TXTS        geoeS  = "";                 // #Sectors

static TXTM        geocS  = "";                 // Cylinder size in 8 positions
static TXTM        geotS  = "";                 // Track    size in 8 positions

static BOOL        geoOK  = TRUE;               // OK button, dummy variable
static BOOL        geoCA  = FALSE;              // Cancel button dummy

/*
 Geometry info for disk: [04 ECS12       F: Log 07 HPFS        917.7 MiB ]

  xxxxxx MiB as reported by the OS     (includes partial cylinder)
  yyyyyy MiB for system  geometry:    12345678       255          63
  zzzzzz MiB for logical geometry:    12345678       255          63

 ( ) Reset to system geometry         #Cylinders     #Heads      #Sect/track
 () Custom geometry, values:        [calculated]   [current]   [current    ]

 ( ) Classic desktop/3.5"  255/63   OS/2 limit  502 GB \   for other OSes
 ( ) Classic laptops/2.5"  240/63   OS/2 limit  472 GB  \  no real limit
     USB/SSD/4Kb-sectors   255/32   OS/2 limit  267 GB  /
 ( ) Std SSD/4Kb-sectors    64/32   OS/2 limit   64 GB /
 ( ) Big SSD/4Kb-sectors  255/240   OS/2 limit 1912 GB \   for other OSes
 ( ) Huge rotating disk   255/127   OS/2 limit 1012 GB  >  no limits, but
 ( ) Huge rotating disk   255/255   OS/2 limit 2032 GB /   not recommended

 Ŀ   Resulting Cylinder size:  yyy.y MiB    Ŀ
  Set Geometry                                            Cancel/Done 
    Alignment / Track  size:  xxx.x KiB    

*/
#define   DFSDGGEOWIDGETS 27
#define   DFSDGGEOT1       0                    // index of from leader
#define   DFSDGGEOD1       1                    // index of from description
#define   DFSDGGEOSIZEOS   2
#define   DFSDGGEOSYSTXT   3
#define   DFSDGGEOSYSCYL   4
#define   DFSDGGEOSYSHEAD  5
#define   DFSDGGEOSYSSECT  6
#define   DFSDGGEOLOGTXT   7
#define   DFSDGGEOCUSTOM  11
#define   DFSDGGEORESET   12
#define   DFSDGGEOCL255   13
#define   DFSDGGEOCL240   14
#define   DFSDGGEOOS255   15
#define   DFSDGGEOMIBCYL  16
#define   DFSDGGEOBIGSSD  17
#define   DFSDGGEO1TBOS2  18
#define   DFSDGGEO2TBOS2  19
#define   DFSDGGEOCE      20                    // cylinder entry field

#define   DFSDGGEOFOCUS   23                    // Focus, on [Set Geometry] button

static TXWIDGET  dfsDgGeoWidgets[DFSDGGEOWIDGETS] =    // order determines TAB-order!
{
   {0,  0, 1, 20, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( "Geometry info for : ")},
   {0, 24, 1, 48, 1, 0, 0, DFSDSOUTPUT,  0, TXStdStline( fdescr)},

   {2,  0, 1, 70, 1, 0, 0, DFSDSOUTPUT,  0, TXStdStline( geooX)},

   {3,  0, 1, 34, 1, 0, 0, DFSDSOUTPUT,  0, TXStdStline( geosX)},
   {3, 36, 1,  9, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geosC)},
   {3, 49, 1,  3, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geosH)},
   {3, 59, 1,  3, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geosS)},

   {4,  0, 1, 34, 1, 0, 0, DFSDSOUTPUT,  0, TXStdStline( geolX)},
   {4, 36, 1,  9, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geolC)},
   {4, 49, 1,  3, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geolH)},
   {4, 59, 1,  3, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geolS)},

   { 7, 0, 1, 34, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &georCustm, "Custom geometry, using values:")},
   { 8, 0, 1, 32, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &georReset, "Reset to system geometry")},
   { 9, 0, 1, 74, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &georCl255, "Classic desktop/3.5\"  255/63   OS/2 limit  502 GB \\   for other OSes")},
   {10, 0, 1, 74, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &georCl240, "Classic laptops/2.5\"  240/63   OS/2 limit  472 GB  \\  no real limit  ")},
   {11, 0, 1, 74, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &georOS255, "USB/SSD/4Kb-sectors   255/32   OS/2 limit  267 GB  /                 ")},
   {12, 0, 1, 74, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &georMbCyl, "Std SSD/4Kb-sectors    64/32   OS/2 limit   67 GB /                  ")},
   {13, 0, 1, 74, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &georBgSsd, "Big SSD/4Kb-sectors  255/240   OS/2 limit 1912 GB \\   for other OSes")},
   {14, 0, 1, 74, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &geor1Tos2, "Huge rotating disk   255/127   OS/2 limit 1012 GB  >  no limits, but ")},
   {15, 0, 1, 74, 0, 1, 0, TXWS_AUTORAD, 0, TXStdButton( &geor2Tos2, "Huge rotating disk   255/255   OS/2 limit 2032 GB /   not recommended")},

   {6, 35, 2, 12, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( geoeC, TXMAXTS, "#Cylinders")},
   {6, 48, 2,  9, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( geoeH, TXMAXTS, "#Heads")},
   {6, 58, 2, 13, 0, 0, 0, TXWS_ENTRYBT, 0, TXStdEntryf( geoeS, TXMAXTS, "#Sect/track")},

   {17, 0, 3, 16, 0, 0, TXDID_OK,
                           DFSDMPBUTTON, 0, TXStdButton( &geoOK, "Set Geometry")},
   {17,58, 3, 15, 0, 0, TXDID_CANCEL,
                           DFSDMPBUTTON, 0, TXStdButton( &geoCA, "Cancel/Done")},

   {17,19, 1, 36, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geocS)},
   {19,19, 1, 36, 0, 0, 0, TXWS_OUTPUT,  0, TXStdStline( geotS)}
};


#define WID_GEODLG           810
#define DFSGEODLG_RECREATE  (TXDID_MAX -1)      // recreate the dialog

static TXGW_DATA dfsDgGeoDlg =
{
   DFSDGGEOWIDGETS,                             // number of widgets
   DFSC_GEODLG,                                 // help, widget overrules
   WID_GEODLG,                                  // base window ID
   dfsGeoWinProc,                               // widget window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgGeoWidgets                              // array of widgets
};


/*************************************************************************************************/
// Present set GEOmetry options dialog and execute resulting command(s) until done
/*************************************************************************************************/
void dfsGeoDialog
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                command;
   ULONG               geoC = 0;                // 0 = calculate
   ULONG               geoH = 0;                // default for sys RESET
   ULONG               geoS = 0;
   TXTS                geoSetC;
   TXTS                geoSetH;
   TXTS                geoSetS;

   ENTER();

   dfsBEGINWORK();                              // signal work starting

   strcpy( fdescr, dfstStoreDesc1( DFSTORE) + 10);
   dfsDgGeoWidgets[DFSDGGEOD1].flags &= ~TXWI_DISABLED; // text field ON
   dfsDgGeoWidgets[DFSDGGEOSIZEOS].flags &= ~TXWI_DISABLED; // text field ON
   dfsDgGeoWidgets[DFSDGGEOLOGTXT].flags &= ~TXWI_DISABLED; // text field ON

   dfsGeoSetRadios();                           // set RADIOs to match geometry

   dfsGeoSetControls();                         // initial value other fields

   while ((rc = txwWidgetDialog( TXHWND_DESKTOP, TXHWND_DESKTOP,
           NULL, " Select pre-defined geometries, or set a custom one ",
           TXWD_MOVEABLE | TXWD_HCENTER | TXWD_VCENTER, DFSDGGEOFOCUS, &dfsDgGeoDlg))
        != TXDID_CANCEL)
   {
      if (rc != DFSGEODLG_RECREATE)
      {
         geoC = dfsGetMcsNumber( geoeC, 0);     // allow hex values too
         geoH = dfsGetMcsNumber( geoeH, 0);
         geoS = dfsGetMcsNumber( geoeS, 0);

         TRACES(("C='%s' => %u  H='%s' => %u  S='%s' => %u\n", geoeC, geoC, geoeH, geoH, geoeS, geoS));

         if (geoC || geoH || geoS)
         {
            sprintf( geoSetC, "%u", geoC);
            sprintf( geoSetH, "%u", geoH);
            sprintf( geoSetS, "%u", geoS);
            sprintf( command, "geo %s %s %s", (geoC) ? geoSetC : "?",
                                              (geoH) ? geoSetH : ".",
                                              (geoS) ? geoSetS : ".");
            dfsExecEnd( command);

            dfsGeoSetRadios();                  // set RADIOs to match geometry
         }
      }
      dfsGeoSetControls();                      // update other fields and visibility

      if (rc == DFSGEODLG_RECREATE)             // update cyl/track size indicators
      {
         geoH = dfsGetMcsNumber( geoeH, 0);     // possibly updated by SetControls
         geoS = dfsGetMcsNumber( geoeS, 0);

         strcpy(    geocS, "Resulting Cylinder size:");
         dfstrSXiB( geocS, "", (geoH * geoS), "");

         strcpy(    geotS, "Alignment / Track  size:");
         dfstrSXiB( geotS, "",         geoS , "");
      }
   }
   dfsENDWORK();                                // signal work done
   VRETURN();
}                                               // end 'dfsGeoDialog'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the GEOMETRY dialog, set dependant fields
/*****************************************************************************/
static ULONG dfsGeoWinProc                      // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      switch (msg)
      {
         case TXWM_COMMAND:
            switch ((ULONG) mp2)                // command source
            {
               case TXCMDSRC_RADIOBUTTON:
                  switch ((ULONG) mp1)          // Window-ID
                  {
                     case WID_GEODLG + DFSDGGEORESET:
                     case WID_GEODLG + DFSDGGEOCL255:
                     case WID_GEODLG + DFSDGGEOCL240:
                     case WID_GEODLG + DFSDGGEOOS255:
                     case WID_GEODLG + DFSDGGEOMIBCYL:
                     case WID_GEODLG + DFSDGGEOBIGSSD:
                     case WID_GEODLG + DFSDGGEO1TBOS2:
                     case WID_GEODLG + DFSDGGEO2TBOS2:
                     case WID_GEODLG + DFSDGGEOCUSTOM:
                        txwDismissDlg( hwnd, DFSGEODLG_RECREATE);
                        break;

                     default:
                        rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                        break;
                  }
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         //- to be refined, catch update to disk-selection list ?

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsGeoWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate GEO values to be displayed from RADIO settings and global GEO
/*****************************************************************************/
static void dfsGeoSetControls
(
   void
)
{
   DFSTOREINFO        *si = &(sti[DFSTORE]);
   BOOL                sgeoDifferent = FALSE;
   ULONG               geoC  = 0;                // 0 = calculate
   ULONG               geoH  = si->Sys.H;        // default for sys RESET
   ULONG               geoS  = si->Sys.S;
   ULONG               sSize = si->Sys.C * si->Sys.H * si->Sys.S;
   ULONG               lSize = si->Geo.C * si->Geo.H * si->Geo.S;

   ENTER();

   if ((si->Geo.C != si->Sys.C) ||
       (si->Geo.H != si->Sys.H) ||
       (si->Geo.S != si->Sys.S) ||              // forced geometry in effect
       (si->Geo.B != si->Sys.B)  )
   {
      sgeoDifferent = TRUE;
      strcpy(    geosX, "");
      dfstrSXiB( geosX, "", sSize, " for  system geometry :");
      sprintf(   geosC, "%9u", si->Sys.C);
      sprintf(   geosH, "%3u", si->Sys.H);
      sprintf(   geosS, "%3u", si->Sys.S);
   }

   dfsdShowWg( dfsDgGeoWidgets[ DFSDGGEOSYSTXT  ], (sgeoDifferent == TRUE));
   dfsdShowWg( dfsDgGeoWidgets[ DFSDGGEOSYSCYL  ], (sgeoDifferent == TRUE));
   dfsdShowWg( dfsDgGeoWidgets[ DFSDGGEOSYSHEAD ], (sgeoDifferent == TRUE));
   dfsdShowWg( dfsDgGeoWidgets[ DFSDGGEOSYSSECT ], (sgeoDifferent == TRUE));
   dfsdShowWg( dfsDgGeoWidgets[ DFSDGGEORESET   ], (sgeoDifferent == TRUE));

   strcpy(    geolX, "");
   dfstrSXiB( geolX, "", lSize, " for logical geometry :");
   sprintf(   geolC, "%9u", si->Geo.C);
   sprintf(   geolH, "%3u", si->Geo.H);
   sprintf(   geolS, "%3u", si->Geo.S);

   strcpy(    geooX, "");
   dfstrSXiB( geooX, "", si->Sectors,
                        (si->Sectors <= sSize)         ? " as reported by the OS" :
              (dfstStoreType( DFSTORE) == DFST_MEMORY) ? " raw memory disk size " :
                          " as reported by the OS    (includes partial cylinder)");

   //- Set custom GEO fields based on active Radio button

   if      (georReset)   { geoC = si->Sys.C;                   }
   else if (georCl255)   { geoH = 255;       geoS =  63;       }
   else if (georCl240)   { geoH = 240;       geoS =  63;       }
   else if (georOS255)   { geoH = 255;       geoS =  32;       }
   else if (georMbCyl)   { geoH =  64;       geoS =  32;       }
   else if (georBgSsd)   { geoH = 255;       geoS = 240;       }
   else if (geor1Tos2)   { geoH = 255;       geoS = 127;       }
   else if (geor2Tos2)   { geoH = 255;       geoS = 255;       }
   else                  { geoH = si->Geo.H; geoS = si->Geo.S; }

   if (geoC == 0)                               // calculate
   {
      strcpy(  geoeC, "calculated");
   }
   else
   {
      sprintf( geoeC, "%9u", geoC);
   }
   sprintf(    geoeH, "%3u", geoH);
   sprintf(    geoeS, "%3u", geoS);

   VRETURN ();
}                                               // end 'dfsGeoSetControls'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine RADIO button to be the selected one for current geometry
/*****************************************************************************/
static void dfsGeoSetRadios
(
   void
)
{
   DFSTOREINFO        *si = &(sti[DFSTORE]);
   ULONG               geoH  = si->Geo.H;        // default for sys RESET
   ULONG               geoS  = si->Geo.S;

   ENTER();

   strcpy(    geocS, "Resulting Cylinder size:");
   dfstrSXiB( geocS, "", (geoH * geoS), "");

   strcpy(    geotS, "Alignment / Track  size:");
   dfstrSXiB( geotS, "",         geoS , "");

   georReset  = FALSE;
   georCl255  = FALSE;
   georCl240  = FALSE;
   georMbCyl  = FALSE;
   georBgSsd  = FALSE;
   geor1Tos2  = FALSE;
   geor2Tos2  = FALSE;
   georCustm  = FALSE;

   if      ((geoH == 255) && (geoS ==  63)) georCl255  = TRUE;
   else if ((geoH == 240) && (geoS ==  63)) georCl240  = TRUE;
   else if ((geoH ==  64) && (geoS ==  32)) georMbCyl  = TRUE;
   else if ((geoH == 255) && (geoS == 240)) georBgSsd  = TRUE;
   else if ((geoH == 255) && (geoS == 127)) geor1Tos2  = TRUE;
   else if ((geoH == 255) && (geoS == 255)) geor2Tos2  = TRUE;
   else                                     georCustm  = TRUE;

   VRETURN ();
}                                               // end 'dfsGeoSetRadios'
/*---------------------------------------------------------------------------*/


/*=============================== GUID/UUID ================================================*/
static TXTS        uuid1  = "00000000";         // 1st sequence,  8 hex digits
static TXTS        uuid2  = "0000";             // 2nd sequence,  4 hex digits
static TXTS        uuid3  = "0000";             // 3rd sequence,  4 hex digits
static TXTS        uuid4  = "0000";             // 4th sequence,  4 hex digits
static TXTS        uuid5  = "000000000000";     // 5th sequence, 12 hex digits

static TXTS        okTxt;                       // Dynamic text for OK button

/*
Specify an 128 bits UUID/GUID value in 5 hex groups")},

    Hex (x8)   (x4)   (x4)   (x4)     (x12)
  {[01234567]-[0123]-[0123]-[0123]-[0123456789AB]}

 Ŀ  Ŀ     Ŀ
     OK         Cancel       Create Unique 
        
   Write Back

*/
#define   DFSDGUIDWIDGETS  15                   // maximum number of widgets
#define   DFSDGUIDFOCUS     2                   // widget index to get focus

#define   WID_UIDDLG           810
#define   DFSDG_UIDGEN        (WID_UIDDLG + 1)  // UUID generate button

static TXWIDGET  dfsDgUidWidgets[DFSDGUIDWIDGETS] =    // order determines TAB-order!
{

   {0, 0, 1, 51, 0, 0, 0, TXWS_OUTPUT,   DFSC_EDITUID, TXStdStline( "Specify an 128 bits UUID/GUID value in 5 hex groups")},
   {3, 2, 1,  1, 0, 0, 0, TXWS_OUTPUT,   DFSC_EDITUID, TXStdStline( "{")},
   {2, 3, 2, 10, 0, 0, 0, TXWS_ENTRYHEX, DFSC_EDITUID, TXWprEntryf( uuid1,  8, dfsUidFieldsWinProc, "Hex (x8)")},
   {3,13, 1,  1, 0, 0, 0, TXWS_OUTPUT,   DFSC_EDITUID, TXStdStline( "-")},
   {2,14, 2,  6, 0, 0, 0, TXWS_ENTRYHEX, DFSC_EDITUID, TXWprEntryf( uuid2,  4, dfsUidFieldsWinProc,     "(x4)")},
   {3,20, 1,  1, 0, 0, 0, TXWS_OUTPUT,   DFSC_EDITUID, TXStdStline( "-")},
   {2,21, 2,  6, 0, 0, 0, TXWS_ENTRYHEX, DFSC_EDITUID, TXWprEntryf( uuid3,  4, dfsUidFieldsWinProc,     "(x4)")},
   {3,27, 1,  1, 0, 0, 0, TXWS_OUTPUT,   DFSC_EDITUID, TXStdStline( "-")},
   {2,28, 2,  6, 0, 0, 0, TXWS_ENTRYHEX, DFSC_EDITUID, TXWprEntryf( uuid4,  4, dfsUidFieldsWinProc,     "(x4)")},
   {3,34, 1,  1, 0, 0, 0, TXWS_OUTPUT,   DFSC_EDITUID, TXStdStline( "-")},
   {2,35, 2, 14, 0, 0, 0, TXWS_ENTRYHEX, DFSC_EDITUID, TXWprEntryf( uuid5, 12, dfsUidFieldsWinProc, "   (x12)")},
   {3,49, 1,  1, 0, 0, 0, TXWS_OUTPUT,   DFSC_EDITUID, TXStdStline( "}")},

   {5,  1, 3, 14, 0, 0, TXDID_OK,     DFSDMPBUTTON, 0, TXStdButton( &chsOK,  okTxt)},
   {5, 19, 3, 10, 0, 0, TXDID_CANCEL, DFSDMPBUTTON, 0, TXStdButton( &chsCA, "Cancel")},
   {5, 34, 3, 17, 0, 0, DFSDG_UIDGEN, DFSDMPBUTTON, 0, TXStdButton( &chsCA, "Create Unique")}
};



static TXGW_DATA dfsDgUidDlg =
{
   DFSDGUIDWIDGETS,                             // number of widgets (set dynamically!)
   0,                                           // help on help, or default
   WID_UIDDLG,                                  // base window ID
   dfsGuidUuidWinProc,                          // dialog window procedure
   NULL,                                        // persistent position TXRECT
   dfsDgUidWidgets                              // array of widgets
};


/*************************************************************************************************/
// Present dialog to specify or Edit a GUID/UUID STRING value in 5 separate hex fields
/*************************************************************************************************/
ULONG dfsGuidUuidDialog
(
   char               *title,                   // IN    Dialog title, UID purpose
   char               *okText,                  // IN    OK button text (max 10 chars)
   USHORT              genId,                   // IN    ID for cr unique, 0 to disable
   ULONG               helpid,                  // IN    Generic (caller) help on help
   char               *uidStr                   // INOUT Guid/Uuid string {...}
)
{
   ULONG               rc = NO_ERROR;           // function return
   BOOL                oldInsert = txwSetInsertMode( FALSE); // force REPLACE mode

   ENTER();

   strcpy( okTxt, (okText && strlen( okText)) ? okText : "OK");
   dfsDgUidDlg.count  = (genId) ? DFSDGUIDWIDGETS : DFSDGUIDWIDGETS -1;
   dfsDgUidDlg.helpid = helpid;                 // set help-on-help from caller

   if (dfsUidStringIsValid( uidStr))            // valid UID as input (edit)
   {
      TxCopy( uuid1, uidStr +  1, 9);
      TxCopy( uuid2, uidStr + 10, 5);
      TxCopy( uuid3, uidStr + 15, 5);
      TxCopy( uuid4, uidStr + 20, 5);
      TxCopy( uuid5, uidStr + 25,13);
   }
   if ((rc = txwWidgetDialog( TXHWND_DESKTOP, TXHWND_DESKTOP, &genId, title,
        TXWD_MOVEABLE | TXWD_HCENTER | TXWD_VCENTER, DFSDGUIDFOCUS, &dfsDgUidDlg)) != TXDID_CANCEL)
   {
      sprintf( uidStr, "{%s-%s-%s-%s-%s}", uuid1, uuid2, uuid3, uuid4, uuid5);
   }
   txwSetInsertMode( oldInsert);                // restore INSERT mode setting

   TRACES(("GUID out: '%s'\n", uidStr));
   RETURN( rc);
}                                               // end 'dfsGuidUuidDialog'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the UUID/GUID dialog, set dependant fields
/*****************************************************************************/
static ULONG dfsGuidUuidWinProc                 // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      DFS_GUID         uuidBin;
      TXTT             uuidStr;
      USHORT          *id = (USHORT *) txwQueryWindowPtr( hwnd, TXQWP_USER);

      switch (msg)
      {
         case TXWM_COMMAND:
            switch ((ULONG) mp2)                // command source
            {
               case TXCMDSRC_PUSHBUTTON:
                  switch ((ULONG) mp1)          // Window-ID
                  {
                     case DFSDG_UIDGEN:
                        dfsGptMkGuid( *id, uuidBin);
                        strcpy( uuidStr, dfsFsUuidValueString( uuidBin));
                        TxCopy( uuid1, uuidStr +  1, 9);
                        TxCopy( uuid2, uuidStr + 10, 5);
                        TxCopy( uuid3, uuidStr + 15, 5);
                        TxCopy( uuid4, uuidStr + 20, 5);
                        TxCopy( uuid5, uuidStr + 25,13);
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // whole dlg
                        break;

                     default:
                        rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                        break;
                  }
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsGuidUuidWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for UID dialog fields, fill out fields to length
/*****************************************************************************/
static ULONG dfsUidFieldsWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;
   TXWINDOW           *win;
   TXENTRYFIELD       *dat;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      win = txwWindowData( hwnd);
      switch (msg)
      {
         case TXWM_CHAR:
            //- Note: must be regular WindowProc, DlgProc will close the field itself!
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            if (rc == NO_ERROR)
            {
               dat = &win->ef;

               //- Right-fill with '0' to length, and replace any spaces by '0'
               while (strlen( dat->buf) <  dat->rsize) strcat( dat->buf, "0"); TxRepl( dat->buf, ' ', '0');

               txwInvalidateWindow( hwnd, FALSE, FALSE);
            }
            break;

         default:
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsUidFieldsWinProc'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Dialog to Edit a binary GUID/UUID value in 5 separate hex fields, update in-place, or string IN
/*************************************************************************************************/
ULONG dfsGuidUuidEditor
(
   char               *title,                   // IN    Dialog title, UID purpose
   char               *okText,                  // IN    OK button text (max 10 chars)
   USHORT              genId,                   // IN    ID for cr unique (PID/DSK-nr)
   ULONG               helpid,                  // IN    Generic (caller) help on help
   char               *uidStr,                  // IN    Optional begin value for UID
   BOOL                swap,                    // IN    perform endian swaps
   DFS_GUID            uidBin                   // INOUT 16 Byte binary GUID/UUID value
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXTT                uidString;               // string representation of uidBin

   ENTER();

   if (uidStr && (dfsUidStringIsValid( uidStr)))
   {
      strcpy( uidString, uidStr);               // use supplied string value of UID
   }
   else                                         // use existing binary value (in-place)
   {
      if (swap)
      {
         strcpy( uidString, dfsFsUuidValueString( uidBin));
      }
      else
      {
         strcpy( uidString, dfsFsUuidValueNoSwap( uidBin));
      }
   }

   if (dfsGuidUuidDialog( title, okText, genId, helpid, uidString) != TXDID_CANCEL)
   {
      rc = dfsUidStringToBinary( uidString, uidBin, swap);
   }
   else
   {
      rc = DFS_USER_ABORT;
   }
   RETURN( rc);
}                                               // end 'dfsGuidUuidEditor'
/*-----------------------------------------------------------------------------------------------*/


