//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK MBR Dialog and windowing functions
//
// Author: J. van Wijk
//
// JvW  21-09-2015 Initial version, splitt off from DFSDFDSK
//


#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfsdfmbr.h>                           // FDISK MBR dialog functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfswin.h>                             // window and help definitions
#include <dfsver.h>                             // version and naming info

#if defined (USEWINDOWING)


// PCREATE dialog specific definitions

#define DFSD_WID_PCRDLG      700                // window id dialog frame
#define DFSD_WID_CRBLOG      701                // radio button Logical
#define DFSD_WID_CRBPRI      702                // radio button Primary
#define DFSD_WID_CEOFFS      703                // entryfield   Offset
#define DFSD_WID_CRBSTA      704                // radio button At-Start
#define DFSD_WID_CRBEND      705                // radio button At-End
#define DFSD_WID_CSLOTT      706                // entry SLOT description
#define DFSD_WID_CSLOTS      707                // entry SLOT S partition
#define DFSD_WID_CSLOTX      708                // entry SLOT X MBR container
#define DFSD_WID_CSLOTY      709                // entry SLOT Y EBR container
#define DFSD_WID_CCBALN      710                // check-box    no Align
#define DFSD_WID_CRBREC      711                // radio-box    Recovery
#define DFSD_WID_CRBCLR      712                // radio-box    Clear BS
#define DFSD_WID_CCBLVM      713                // check-box    Lvm
#define DFSD_WID_CCBACT      714                // check-box    Set ACTIVE
#define DFSD_WID_CCBMUL      715                // check-box    Multiple P
#define DFSD_WID_CSPINV      716                // spin-value type list
#define DFSD_WID_CESIZE      717                // entryfield create-size
#define DFSD_WID_CCBMAX      718                // check-box    Use MAX size
#define DFSD_WID_CBUTTN      719                // buttons Ok and Cancel

#define DFSH_PM             (DFSH_FDISK + 500)
#define DFSH_PM_CRBLOG      (DFSH_PM + 1)
#define DFSH_PM_CRBPRI      (DFSH_PM + 2)
#define DFSH_PM_CEOFFS      (DFSH_PM + 3)
#define DFSH_PM_CRBSTA      (DFSH_PM + 4)
#define DFSH_PM_CRBEND      (DFSH_PM + 5)
#define DFSH_PM_CSLOTT      (DFSH_PM + 7)
#define DFSH_PM_CSLOTS      (DFSH_PM + 7)
#define DFSH_PM_CSLOTX      (DFSH_PM + 8)
#define DFSH_PM_CSLOTY      (DFSH_PM + 9)
#define DFSH_PM_CRBREC      (DFSH_PM +10)
#define DFSH_PM_CRBCLR      (DFSH_PM +11)
#define DFSH_PM_CCBLVM      (DFSH_PM +12)
#define DFSH_PM_CCBALN      (DFSH_PM +13)
#define DFSH_PM_CCBACT      (DFSH_PM +14)
#define DFSH_PM_CCBMUL      (DFSH_PM +15)
#define DFSH_PM_CSPINV      (DFSH_PM +16)
#define DFSH_PM_CESIZE      (DFSH_PM +17)
#define DFSH_PM_CCBMAX      (DFSH_PM +18)
#define DFSH_PM_CBUTTN      (DFSH_PM +19)

#define DFSH_PM_FRAME       (DFSH_PM +20)       // shown help-on-help (2nd)


#define DFSD_PCREAT_WC          73              // horizontal size in columns
#define DFSD_PCREAT_WL          23              // vertical   size in lines

#define PCTP                ((ULONG) 256)
#define PFTP                DFS_PTYPE_AUTOFAT

TXSitem(pcFatx,PFTP,DFSH_PM_CSPINV,0, 1,"FAT using automatic type select" ,"FAT, type selected by DFSee based on size, position etc.");
TXSitem(pcNtfs,0x07,DFSH_PM_CSPINV,0, 6,"IFS, NTFS, Windows-NT/W2K/XP 07" ,"IFS, Installable filesystem, on Windows most likely NTFS");
TXSitem(pcHpfs,0x07,DFSH_PM_CSPINV,0, 6,"IFS, HPFS, OS/2 & eCS native 07" ,"IFS, Installable filesystem, on OS2 or eCS probably HPFS");
TXSitem(pcBjfs,0x07,DFSH_PM_CSPINV,0, 6,"IFS, Bootable JFS eCS   type 07" ,"JFS, eCS 2.x Journaled filesystem with bootable JFS.IFS ");
TXSitem(pcLjfs,0x35,DFSH_PM_CSPINV,0, 1,"JFS, LVM,  OS/2 & eCS   type 35" ,"JFS, OS/2 4.5x Journaled filesystem on LVM-type volume  ");
TXSitem(pcBmgr,0x0a,DFSH_PM_CSPINV,0, 6,"IBM  BootManager, 1-Cyl type 0A" ,"IBM  BootManager used with OS/2, eCS and other systems  ");
TXSitem(pcFat1,0x01,DFSH_PM_CSPINV,0,19,"FAT, 12-bit: max 16 MiB type 01" ,"FAT, 12-bit as used on diskettes or partitions  < 16 MiB");
TXSitem(pcFat4,0x04,DFSH_PM_CSPINV,0,19,"FAT, 16-bit: max 32 MiB type 04" ,"FAT, 16-bit as used on partitions from  16 upto   32 MiB");
TXSitem(pcFat6,0x06,DFSH_PM_CSPINV,0,19,"FAT, 16-bit: max  2 GiB type 06" ,"FAT, 16-bit as used on partitions from  32 upto 2048 MiB");
TXSitem(pcFatb,0x0b,DFSH_PM_CSPINV,0,19,"FAT, 32-bit: < Cyl 1024 type 0B" ,"FAT, 32-bit-standard, when Part-end below cylinder 1024 ");
TXSitem(pcFatc,0x0c,DFSH_PM_CSPINV,0,19,"FAT, 32-bit: > Cyl 1023 type 0C" ,"FAT, 32-bit-extended, when Part-end above cylinder 1023 ");
TXSitem(pcFate,0x0e,DFSH_PM_CSPINV,0,19,"FAT, 16-bit: > Cyl 1023 type 0E" ,"FAT, 16-bit-extended, when Part-end above cylinder 1023 ");
TXSitem(pcSwap,0x82,DFSH_PM_CSPINV,0, 1,"Linux, SWAP virtual memory   82" ,"Linux swap space, to create disk-based virtual memory   ");
TXSitem(pcNatv,0x83,DFSH_PM_CSPINV,0, 1,"Linux, EXT2/3/4, REISER, JFS 83" ,"EXT2, EXT3, EXT4, ReiserFS, JFS or other native Linux FS");
TXSitem(pcLlin,0x8e,DFSH_PM_CSPINV,0, 1,"Linux, LVM-style partition   8E" ,"Linux style Logical Volume Manager (LVM) partition      ");
TXSitem(pcMhfs,0xaf,DFSH_PM_CSPINV,0, 1,"macOS, HFS+ data partition   AF" ,"macOS style HFS+ partition, journaled filesystem, Apple ");
TXSitem(pcGptg,0xee,DFSH_PM_CSPINV,0, 1,"EFI / GPT guard  partition   EE" ,"EFI-GPT partitioning, guard partition in protective MBR ");
TXSitem(pcTmpi,0xd7,DFSH_PM_CSPINV,0, 1,"Temporary/Backup no warnings D7" ,"UNDEFINED partition type, ignored by DFSee, no warnings ");
TXSitem(pcType,PCTP,DFSH_PM_CSPINV,0, 9,"Specify Custom type value (HEX)" ,"Allow input of a HEX system type value between 00 and ff");
static TXS_ITEM *pct[] = {&pcFatx, &pcNtfs, &pcHpfs,
                          &pcBjfs, &pcLjfs, &pcBmgr,
                          &pcSwap, &pcNatv, &pcLlin,
                          &pcMhfs,
                          &pcFat1, &pcFat4, &pcFat6, &pcFate, &pcFatb, &pcFatc,
                          &pcGptg,
                          &pcTmpi,
                          &pcType};
TXSlist(dfspct,19,19,pct);


typedef struct dfs_pcmbrdata                    // MBR partition-create data
{
   DFSPARTINFO     *fsp;                        // freespace info (geo, size)
   TXTT             tpsbuf;                     // titlefield create-size
   TXTT             vpsbuf;                     // entryfield create-size
   double           vpsize;                     // floating size
   double           fssize;                     // freespace area size (max)
   TXTT             vpobuf;                     // entryfield offset
   TXTT             vesbuf;                     // entryfield entry slot
   TXTT             vexbuf;                     // entryfield entry MBR container
   TXTT             veybuf;                     // entryfield entry EBR container
   double           vpoffs;                     // floating offset
   ULONG            vstype;                     // system-type value
   BOOL             vrbLog;                     // radio button Logical
   BOOL             vrbPri;                     // radio button Primary
   BOOL             vrbSta;                     // radio button At-Start
   BOOL             vrbEnd;                     // radio button At-End
   BOOL             vcbLvm;                     // check-box    Lvm
   BOOL             vcbMax;                     // check-box Use Maximum size
   BOOL             vcbAln;                     // check-box do NOT align offset/size
   BOOL             vcbClr;                     // check-box    Clear BS
   BOOL             vcbAct;                     // check-box    set ACTIVE
   BOOL             vcbMul;                     // check-box    Multiple P
} DFS_PCMBRDATA;                                // end of struct "dfs_pcmbrdata"

static char dfsdf_bt_ok[]       = " OK ";
static char dfsdf_bt_cancel[]   = "Cancel";

static char        *pcrMbrHelpText[] =
{
   "",
   " This dialog allows specification of all the information needed to",
   " create a new MBR-style partition, once the freespace is selected.",
   "",
   TXHELPITEM(001, "Logical partition button")
   " When set, this button indicates that a LOGICAL partition",
   " will be created.",
   "",
   " Depending on the type of freespace, there could be choice of",
   " Logical and Primary by activating the corresponding button.",
   "",
   " When no choice is possible, the mandatory one is displayed",
   " only and cannot be changed by the user.",
   "",
   " The radio-button is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(002, "Primary partition button")
   " When set, this button indicates that a PRIMARY partition",
   " will be created.",
   "",
   " Depending on the type of freespace, there could be choice of",
   " Logical and Primary by activating the corresponding button.",
   "",
   " When no choice is possible, the mandatory one is displayed",
   " only and cannot be changed by the user.",
   "",
   " For a 'Partial Cylinder' it will be the default (ON), but",
   " can be changed to logical when desired.",
   "",
   " The radio-button is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(003, "Offset in freespace")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This allows specification of the offset from the beginning or",
   " end of the selected freespace area (depending on selection).",
   " It allows you to reserve some freespace there for future use.",
   "",
   " It will be interpreted as a decimal value in megabytes (MiB).",
   " As an alternative, you can use the mcs-number formats to",
   " specify the offset in gigabytes (,g), cylinders (,c) etc.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(004, "From start freespace button")
   " When set, this button indicates that the new partition will",
   " be created at the START of the selected freespace area.",
   " (possibly leaving some reserved space, defined by 'offset')",
   "",
   " The radio-button is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(005, "From end freespace button")
   " When set, this button indicates that the new partition will",
   " be created at the END of the selected freespace area.",
   " (possibly leaving some reserved space, defined by 'offset')",
   "",
   " The radio-button is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(007, "Prefered table entry, for partition itself")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This allows specification of the prefered table entry (0..3)",
   " to be used for the partition itself.",
   "",
   " This will be the entry in the MBR table for primary partitions,",
   " or an entry in the EBR table for a logical partition.",
   "",
   " Specifying a prefered entry may help in exactly recreating a",
   " certain partitioning on another disk, or manipulating the way",
   " partitions are numbered on some operating systems.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(008, "Prefered table entry, for the MBR container")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This allows specification of the prefered table entry (0..3)",
   " to be used for the single extended-partition CONTAINER that",
   " is specified in the MBR table (type 0x05 or 0x0F).",
   "",
   " Specifying a prefered entry may help in exactly recreating a",
   " certain partitioning on another disk, or manipulating the way",
   " partitions are numbered on some operating systems.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(009, "Prefered table entry, for a logical next-link")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This allows specification of the prefered table entry (0..3)",
   " to be used for the 'next-link' with a logical partition.",
   "",
   " This will be an entry in the EBR table for a logical partition",
   " when there are more logical partitions following this one in",
   " the extended container. The type would be 0x05 or 0x0F.",
   "",
   " Specifying a prefered entry may help in exactly recreating a",
   " certain partitioning on another disk, or manipulating the way",
   " partitions are numbered on some operating systems.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(010, "Recovery mode, keep existing bootsector and LVM info intact")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When marked, this option will make sure the bootsector for the ",
   " newly created partition and any related LVM information is kept",
   " intact so it will be re-used instead of created new.",
   "",
   " This is MANDATORY if you want to RECOVER and existing filesystem",
   " without having to re-create that information again later.",
   "",
   " So, if you are RE-CREATING the partition for RECOVERY purposes,",
   " DO NOT check this since it will wipe out the bootsector with",
   " important size, format and label-information!",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(011, "Create NEW mode, clear existing bootsector and old LVM info")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When marked, this option will cause the bootsector for the ",
   " newly created partition to be CLEARED using the 0xF6 pattern",
   " just like any standard FDISK tool would do.",
   "",
   " This can be useful to make sure any old formatting information",
   " does not interfere with the new desired usage of the partition.",
   "",
   " If you are RE-CREATING the partition for RECOVERY purposes,",
   " DO NOT check this since it will wipe out the bootsector",
   " with important size, format and label-information!",
   "",
   " When creating LVM information too, selecting this option will",
   " cause any existing LVM information on the disk to be discarded",
   " and cleared. (required on change LVM to Compatibility volume)."
   "",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(012, "Include LVM information")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When marked, this option will cause an additional dialog to be",
   " presented to specify/update LVM-information for the partition.",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(013, "Do NOT Align to Cylinder boundaries")
   " When checked, this will cause the values for offset and size to",
   " be UNALIGNED to the regular cylinder boundaries (CHS).",
   "",
   " This may allow for slighter larger partitions at the start and",
   " end of the disk, at the expense of non standard alignment.",
   "",
   " It is also the default when the selected freespace is for a",
   " 'Partial Cylinder'",
   "",
   TXHELPITEM(014, "Set as ACTIVE partition")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When marked, this option will cause the new partition to be",
   " set as the ACTIVE partition, meaning the FLAG byte in the",
   " partition table will be set to 0x80",
   "",
   " This would cause the BIOS to use this partition as the one to",
   " be started at system-startup time. Note that there should be",
   " only ONE partition per physical disk marked ACTIVE!",
   "",
   " This setting is only allowed on PRIMARY partitions, trying",
   " to check it for a logical, will result in an error message.",
   " Changing the partition type to logical with this checkbox",
   " marked, will cause it to be unmarked automatically.",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(015, "Allow multiple primaries")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " When marked, this option will allow multiple primaries to be",
   " visible after creating this new one. it will NOT hide other",
   " prumary partitions.",
   "",
   " This setting is only allowed on PRIMARY partitions, trying",
   " to check it for a logical, will result in an error message.",
   " Changing the partition type to logical with this checkbox",
   " marked, will cause it to be unmarked automatically.",
   "",
   " The check-box is toggled using the <Space-bar> or the mouse",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(016, "Partition system type")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This is the 'system-type' for the partition, a single byte that",
   " often represents the kind of filesystem or the operating system",
   " using this partition.",
   "",
   " The most used values can be selected from the list directly.",
   " Any other (custom) value can be specified as a hexadecimal value",
   " by selecting the last entry 'Specify custom type value (HEX)'.",
   "",
   " A full list of types recognized by DFSee can be obtained by using",
   " the command 'type' from the commandline (use <F10> to get there).",
   " A specific (hexadecimal) value can be shown using 'type nn',",
   "",
   " Selecting the 'Custom value' entry in that popup-list will result",
   " in a another popup window requesting a HEXadecimal type value.",
   "",
   TXHELPITEM(017, "Partition size field")
   " For more help on the complete dialog, press <F1> again ...",
   "",
   " This field allows specification of the desired size for the",
   " new partition. It can be given as a decimal value that is",
   " interpreted as megabytes (MiB) by default.",
   "",
   " As an alternative, you can use the mcs-number formats to",
   " specify the size in gigabytes (,g), cylinders (,c) etc.",
   "",
   " Using the <Enter> key here will end the dialog and execute a ",
   " CREATE command based on all specified values.",
   "",
   TXHELPITEM(018, "Use MAX size: xxxx.yy MiB")
   " When checked, this will force the size to be used to the MAXIMUM",
   " available size in the slected freespace area, considering the",
   " current offset value.",
   "",
   " The entryfield wil be disabled (readonly), showing either the",
   " exact size in sectors when not-aligned, or '*' when aligned.",
   "",
   " When unchecked, the size entryfield is enabled to allow input",
   " of a desired size in Mib or sectors.",
   "",
   " It is the default when the selected freespace is for a",
   " 'Partial Cylinder', so the size will be MAX, in sectors",
   " because it will NOT be aligned",
   "",
   TXHELPITEM(019, "OK and Cancel buttons")
   " When all values are set to the desired values, the [OK]",
   " button will end the dialog and execute a CREATE command.",
   "",
   " Actually, using <Enter> on ANY field in the dialog EXCEPT the",
   " spin-value control with partition-types will do the same.",
   "",
   " The [Cancel] button will end the dialog without creating",
   " a new partition, just like the 'Escape' key would.",
   "",
   TXHELPITEM(020, "MBR Partition Create Dialog window")
   " This dialog allows specification of all the information needed",
   " to create a new MBR partition, once the freespace is selected.",
   "",
   " Using the <Enter> key on ANY field in the dialog EXCEPT the",
   " spin-value control with partition-types will end the dialog",
   " and execute a CREATE command based on all specified values.",
   "",
   " There will be a final confirmation popup in the CREATE, so",
   " you will have another chance of checking all values and",
   " CANCEL the create if any wrong values have been used.",
   "",
   " The dialog ends with the following keys / actions:",
   "",
   "  - <ENTER> End dialog, and CREATE the partition as specified",
   "  - <Esc>   End dialog, do not create a partition",
   "",
   "",
   "",
   NULL
};


// Dialog window procedure, for the MBR-style CreatePart Dialog
static ULONG dfsMbrCrPartDlgWinProc             // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Dialog window procedure, for the CREATE-MBR partition dialog fields
static ULONG dfsMbrCreateWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


// Calculate exact maximum MBR partitionsize based on pri/log, position and GEO
static void dfsMbrCpGetMaxSize
(
   DFS_PCMBRDATA      *pcd                      // IN    Part Create Data
);


/*****************************************************************************/
// Initialize help system for Create MBR-part Dialog (accessible from startup)
/*****************************************************************************/
void dfsCreateMbrInitHelp
(
   void
)
{
   ENTER();

   txwRegisterHelpText( DFSH_PM, "dfsmbrcreate", "MBR-style Partition-Create", pcrMbrHelpText);

   VRETURN ();
}                                               // end 'dfsCreateMbrInitHelp'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Present an MBR-style Create-Partition Dialog, with freespace info as input
/*****************************************************************************/
BOOL txwCreateMbrPartDialog
(
   DFSPARTINFO        *f,                       // IN    freespace information
   char               *create                   // OUT   Create command
)
{
   BOOL                rc = FALSE;              // function return
   ULONG               result = 0;

   ENTER();

   if (f != NULL)                               // MUST have a freespace area!
   {
      if (txwIsWindow( TXHWND_DESKTOP))         // is there a desktop ?
      {
         TXRECT        position;                // reference size/position
         TXWHANDLE     cframe;                  // file-dialog frame
         TXWHANDLE     centry;                  // entryfield create-size
         TXWHANDLE     centof;                  // entryfield offset
         TXWHANDLE     cente1;                  // entryfield entry text 1
         TXWHANDLE     centes;                  // entryfield entry slot
         TXWHANDLE     centex;                  // entryfield entry MBR container
         TXWHANDLE     centey;                  // entryfield entry EBR container
         TXWHANDLE     cspinv;                  // spin-value type list
         TXWHANDLE     cbtok;                   // OK button
         TXWHANDLE     cbtcan;                  // CANCEL button
         TXWHANDLE     crbLog;                  // radio button Logical
         TXWHANDLE     crbPri;                  // radio button Primary
         TXWHANDLE     crbSta;                  // radio button At-Start
         TXWHANDLE     crbEnd;                  // radio button At-End
         TXWHANDLE     ccbMax;                  // Checkbox Use-MAX-size
         TXWHANDLE     ccbAln;                  // Checkbox Do-NOT-align
         TXWHANDLE     ccbRec;                  // radio-box    Recovery
         TXWHANDLE     ccbClr;                  // radio-box    Clear BS
         TXWHANDLE     ccbLvm;                  // check-box    Lvm
         TXWHANDLE     ccbAct;                  // check-box    Set ACTIVE
         TXWHANDLE     ccbMul;                  // check-box    Multiple P
         TXWINDOW      window;                  // setup window data
         ULONG         style;
         USHORT        phsize;                  // parent window width
         USHORT        pvsize;                  // parent window height
         short         ww;                      // dialog window width
         short         wc;                      // std column with
         DFS_PCMBRDATA cpp;                     // private create-part data
         BOOL          logOnly = FALSE;         // logical allowed only
         BOOL          priOnly = FALSE;         // primary allowed only
         static BOOL   recovery = FALSE;        // default NO recovery mode
         ULONG         sectors;                 // a size in sectors
         BYTE          unit = 'm';              // unit for size (k,m,g,c,h,s)

         switch (f->partent.PartitionType)
         {
            case DFS_FSP_ELOG :
            case DFS_FSP_HLOG :
            case DFS_FSP_TLOG :
            case DFS_FSP_LOG  : logOnly = TRUE; break;
            case DFS_FSP_PRIM : priOnly = TRUE; break;
            default           :                 break;
         }
         memset( &cpp, 0, sizeof(DFS_PCMBRDATA));  // start empty

         cpp.fsp = f;                           // attach fsp reference

         if (logOnly)                           // no BMGR allowed
         {
            pcBmgr.flags |= TXSF_DISABLED;
         }

         cpp.vstype = 0;                        // custom system-type value

         if (f->partent.PartitionType == DFS_FSP_ENDC) // Partial Cylinder
         {
            cpp.vcbAln = TRUE;                  // do NOT align
            cpp.vcbMax = TRUE;                  // and use maximum size
         }
         if ((priOnly) || (f->partent.PartitionType == DFS_FSP_ENDC)) // Partial Cylinder
         {
            cpp.vrbLog = FALSE;                 // radio button Logical
            cpp.vrbPri = TRUE;                  // radio button Primary
         }
         else
         {
            cpp.vrbLog = TRUE;                  // radio button Logical
            cpp.vrbPri = FALSE;                 // radio button Primary
         }
         cpp.vrbSta = TRUE;                     // radio button At-Start
         cpp.vrbEnd = FALSE;                    // radio button At-End
         cpp.vcbLvm = dfsa->lvmPresent;         // check-box    Lvm
         cpp.vcbAct = FALSE;                    // check-box    Set ACTIVE
         cpp.vcbMul = TRUE;                     // check-box    Multiple P
         cpp.vcbClr = !recovery;                // mutual exclusive value
                                                // but remember previous :-)

         strcpy(  cpp.vpobuf, "0");             // offset value
         dfsMbrCpGetMaxSize( &cpp);             // calculate max sizes
         TRACES(( "fsp-info:%8.8x, id:%2.2hu size:%9.1lf\n", f, f->id, cpp.vpsize));

         strcpy(  cpp.vesbuf, "0");             // prefered entry partition
         strcpy(  cpp.vexbuf, "3");             // prefered entry MBR container
         strcpy(  cpp.veybuf, "1");             // prefered entry EBR container

         txwQueryWindowPos( TXHWND_DESKTOP, FALSE, &position);
         phsize = position.right  - position.left;
         pvsize = position.bottom - position.top;

         ww = min( DFSD_PCREAT_WC,  (phsize - 5));
         wc = 32;

         position.left   = (phsize - ww) /2;    // always center the dialog
         position.right  = ww;
         position.top    = (pvsize < 26) ? 0 : 1;
         position.bottom = DFSD_PCREAT_WL;

         TRECTA( "pos/size", (&position));

         style = TXWS_DIALOG | TXWS_DISABLED | TXWS_MOVEABLE | TXCS_CLOSE_BUTTON;
         txwSetupWindowData(
            position.top, position.left, position.bottom, position.right,
            style | TXWS_CAST_SHADOW,           // window frame style
            DFSH_PM_FRAME,                      // dialog help
            ' ', ' ',                           // window & border char
            TXWSCHEME_COLORS,                   // std scheme colors
            " Specify MBR-style partition properties ",
            txwstd_footer,
            &window);
         window.st.buf      = NULL;             // NO artwork attached
         window.dlgFocusID  = DFSD_WID_CESIZE;
         cframe = txwCreateWindow( TXHWND_DESKTOP, TXW_CANVAS, 0, 0, &window, NULL);
         txwSetWindowUShort( cframe, TXQWS_ID, DFSD_WID_PCRDLG);

         if (!priOnly)
         {
            style = TXWS_RADIOB | TXWS_HCHILD2SIZE;
            if (logOnly)
            {
               style |= TXWS_DISABLED;
            }
            txwSetupWindowData(
               1, 1, 1, wc,
               style, DFSH_PM_CRBLOG,           // style + help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "", "",                          // title & footer text
               &window);
            window.bu.text    = "Create a logical partition";
            window.bu.checked = &cpp.vrbLog;
            crbLog = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, NULL);
            txwSetWindowUShort( crbLog, TXQWS_ID, DFSD_WID_CRBLOG);
            txwSetWindowUShort( crbLog, TXQWS_GROUP, 1);
         }


         if (!logOnly)
         {
            style = TXWS_RADIOB | TXWS_HCHILD2SIZE;
            if (priOnly)
            {
               style |= TXWS_DISABLED;
            }
            txwSetupWindowData(
               2, 1, 1, wc,
               style, DFSH_PM_CRBPRI,           // style + help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "", "",                          // title & footer text
               &window);
            window.bu.text    = "Create a primary partition";
            window.bu.checked = &cpp.vrbPri;
            crbPri = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, NULL);
            txwSetWindowUShort( crbPri, TXQWS_ID, DFSD_WID_CRBPRI);
            txwSetWindowUShort( crbPri, TXQWS_GROUP, 1);
         }


         style = TXWS_DSPINBT     | TXLS_DROP_ENTER  | TXWS_LEFTJUSTIFY |
                 TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            1, wc +4, 2, ww - wc -7,
            style, DFSH_PM_CSPINV,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "Select MBR partition-type", "",
            &window);
         window.lb.list = &dfspct;              // spin-value type list
         window.lb.cpos = dfspct.selected - dfspct.top;
         cspinv = txwCreateWindow(  cframe, TXW_LISTBOX, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( cspinv, TXQWS_ID, DFSD_WID_CSPINV);


         style = TXWS_ENTRYBT | TXWS_VCHILD2MOVE; // entryfield with title, allow move
         txwSetupWindowData(
            5, 1, 2,  9,
            style, DFSH_PM_CEOFFS,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "Offset", "",                       // title & footer text
            &window);
         window.ef.leftcol  = 0;
         window.ef.maxcol   = TXW_INVALID;
         window.ef.curpos   = 0;
         window.ef.markCol  = 0;
         window.ef.markSize = 0;
         window.ef.rsize    = TXMAXTT;
         window.ef.buf      = cpp.vpobuf;
         window.ef.history  = NULL;
         centof = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( centof, TXQWS_ID, DFSD_WID_CEOFFS);


         style = TXWS_RADIOB | TXWS_HCHILD2SIZE | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            5, 11, 1, wc -9,
            style, DFSH_PM_CRBSTA,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "From start freesp.";
         window.bu.checked = &cpp.vrbSta;
         crbSta = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( crbSta, TXQWS_ID, DFSD_WID_CRBSTA);
         txwSetWindowUShort( crbSta, TXQWS_GROUP, 2);


         style = TXWS_RADIOB | TXWS_HCHILD2SIZE | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            6, 11, 1, wc -9,
            style, DFSH_PM_CRBEND,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "From end freespace";
         window.bu.checked = &cpp.vrbEnd;
         crbEnd = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( crbEnd, TXQWS_ID, DFSD_WID_CRBEND);
         txwSetWindowUShort( crbEnd, TXQWS_GROUP, 2);


         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            5, wc +4, 1, ww - wc -7,
            style, DFSH_PM_CCBMAX,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = cpp.tpsbuf;
         window.bu.checked = &cpp.vcbMax;
         ccbMax = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( ccbMax, TXQWS_ID, DFSD_WID_CCBMAX);


         style = TXWS_ENTRYB      |             // entryfield without title, allow size and move
                 TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE | TXWS_VCHILD2MOVE;
         txwSetupWindowData(
            6, wc +4, 1,  ww - wc -7,
            style, DFSH_PM_CESIZE,              // style and help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.ef.leftcol  = 0;
         window.ef.maxcol   = TXW_INVALID;
         window.ef.curpos   = 0;
         window.ef.markCol  = 0;
         window.ef.markSize = 0;
         window.ef.rsize    = TXMAXTT;
         window.ef.buf      = cpp.vpsbuf;
         window.ef.history  = NULL;
         centry = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( centry, TXQWS_ID, DFSD_WID_CESIZE);


         if (dfsa->expertui)
         {
            style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               8, 1, 1, ww,
               style, DFSH_PM_CCBALN,           // style + help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "", "",                          // title & footer text
               &window);
            window.bu.text    = "Do NOT align offset and size values to standard boundaries";
            window.bu.checked = &cpp.vcbAln;
            ccbAln = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
            txwSetWindowUShort( ccbAln, TXQWS_ID, DFSD_WID_CCBALN);
         }


         style = TXWS_AUTORAD | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            10, 1, 1, ww,
            style, DFSH_PM_CRBREC,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Recovery mode, keep existing bootsector and LVM info intact";
         window.bu.checked = &recovery;
         ccbRec = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( ccbRec, TXQWS_ID, DFSD_WID_CRBREC);
         txwSetWindowUShort( ccbRec, TXQWS_GROUP, 4);


         style = TXWS_AUTORAD | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            11, 1, 1, ww,
            style, DFSH_PM_CRBCLR,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Create NEW mode, clear existing bootsector and old LVM info";
         window.bu.checked = &cpp.vcbClr;
         ccbClr = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( ccbClr, TXQWS_ID, DFSD_WID_CRBCLR);
         txwSetWindowUShort( ccbClr, TXQWS_GROUP, 4);


         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            13, 1, 1, wc,
            style, DFSH_PM_CCBLVM,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Include LVM information";
         window.bu.checked = &cpp.vcbLvm;
         ccbLvm = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( ccbLvm, TXQWS_ID, DFSD_WID_CCBLVM);
         txwSetWindowUShort( ccbLvm, TXQWS_GROUP, 3);


         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            14, 1, 1, wc,
            style, DFSH_PM_CCBACT,              // style + help
            ' ', ' ',                           // window/border chars
            TXWSCHEME_COLORS,                   // std scheme colors
            "", "",                             // title & footer text
            &window);
         window.bu.text    = "Set as ACTIVE partition";
         window.bu.checked = &cpp.vcbAct;
         ccbAct = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
         txwSetWindowUShort( ccbAct, TXQWS_ID, DFSD_WID_CCBACT);
         txwSetWindowUShort( ccbAct, TXQWS_GROUP, 3);


         if (dfsa->expertui)
         {
            style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               15, 1, 1, wc,
               style, DFSH_PM_CCBMUL,           // style + help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "", "",                          // title & footer text
               &window);
            window.bu.text    = "Allow multiple primaries";
            window.bu.checked = &cpp.vcbMul;
            ccbMul = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window, dfsMbrCreateWinProc);
            txwSetWindowUShort( ccbMul, TXQWS_ID, DFSD_WID_CCBMUL);
            txwSetWindowUShort( ccbMul, TXQWS_GROUP, 3);


            //- Table entry description field, output only
            style = TXWS_OUTPUTT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               14, wc +7, 2,  36,
               style, 0,                        // style and help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // scheme colors
                            "Prefered table entry (slot)", "",
               &window);
            window.sl.buf = "range 0..3, to be used for:";
            cente1 = txwCreateWindow( cframe, TXW_STLINE, cframe, 0, &window, dfsMbrCreateWinProc);

            style = TXWS_ENTRYBT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
            txwSetupWindowData(
               17, wc +5, 2,  11,
               style, DFSH_PM_CSLOTS,           // style and help
               ' ', ' ',                        // window/border chars
               TXWSCHEME_COLORS,                // std scheme colors
               "Partition", "",                 // title & footer text
               &window);
            window.ef.leftcol  = 0;
            window.ef.maxcol   = TXW_INVALID;
            window.ef.curpos   = 0;
            window.ef.markCol  = 0;
            window.ef.markSize = 0;
            window.ef.rsize    = TXMAXTT;
            window.ef.buf      = cpp.vesbuf;
            window.ef.history  = NULL;
            centes = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsMbrCreateWinProc);
            txwSetWindowUShort( centes, TXQWS_ID, DFSD_WID_CSLOTS);

            if (!priOnly)
            {
               style = TXWS_ENTRYBT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
               txwSetupWindowData(
                  17, wc +16, 2,  10,
                  style, DFSH_PM_CSLOTX,        // style and help
                  ' ', ' ',                     // window/border chars
                  TXWSCHEME_COLORS,             // std scheme colors
                  "MBR cont", "",               // title & footer text
                  &window);
               window.ef.leftcol  = 0;
               window.ef.maxcol   = TXW_INVALID;
               window.ef.curpos   = 0;
               window.ef.markCol  = 0;
               window.ef.markSize = 0;
               window.ef.rsize    = TXMAXTT;
               window.ef.buf      = cpp.vexbuf;
               window.ef.history  = NULL;
               centex = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsMbrCreateWinProc);
               txwSetWindowUShort( centex, TXQWS_ID, DFSD_WID_CSLOTX);

               style = TXWS_ENTRYBT | TXWS_HCHILD_MOVE | TXWS_VCHILD_MOVE;
               txwSetupWindowData(
                  17, wc +27, 2,  10,
                  style, DFSH_PM_CSLOTY,        // style and help
                  ' ', ' ',                     // window/border chars
                  TXWSCHEME_COLORS,             // std scheme colors
                  "EBR cont", "",               // title & footer text
                  &window);
               window.ef.leftcol  = 0;
               window.ef.maxcol   = TXW_INVALID;
               window.ef.curpos   = 0;
               window.ef.markCol  = 0;
               window.ef.markSize = 0;
               window.ef.rsize    = TXMAXTT;
               window.ef.buf      = cpp.veybuf;
               window.ef.history  = NULL;
               centey = txwCreateWindow(   cframe, TXW_ENTRYFIELD, cframe, 0, &window, dfsMbrCreateWinProc);
               txwSetWindowUShort( centey, TXQWS_ID, DFSD_WID_CSLOTY);
            }
         }

         style = TXWS_PBUTTON | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            17, 5,                              // position
            3, strlen( dfsdf_bt_ok) + 4,        // vertical/horizontal size
            style, DFSH_PM_CBUTTN,              // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_ok;
         cbtok = txwCreateWindow(   cframe, TXW_BUTTON, cframe, 0, &window,
                                    txwDefWindowProc);
         txwSetWindowUShort( cbtok, TXQWS_ID, TXMBID_OK); // rc == ID


         style = TXWS_PBUTTON | TXWS_HCHILD2MOVE | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            17, wc - 13,                        // position
            3, strlen( dfsdf_bt_cancel) + 4,    // vertical/horizontal size
            style, DFSH_PM_CBUTTN,              // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_cancel;
         cbtcan = txwCreateWindow(  cframe, TXW_BUTTON, cframe, 0, &window,
                                    txwDefWindowProc);
         txwSetWindowUShort( cbtcan, TXQWS_ID, TXMBID_CANCEL); // rc == ID


         result = txwDlgBox( TXHWND_DESKTOP, TXHWND_DESKTOP,
                             dfsMbrCrPartDlgWinProc, cframe, &cpp);

         if (result != TXDID_CANCEL)
         {
            if (create != NULL)
            {
               ULONG   ltype = dfspct.items[dfspct.selected]->value;
               double  size  = 0.0;
               int     slot;
               TXTM    optstr;                  // option string

               if (ltype == DFS_PTYPE_AUTOFAT)  // automatic FAT type
               {
                  strcpy( optstr, "FAT");
               }
               else if (ltype == PCTP)          // custom specified type
               {
                  if (cpp.vstype == 0)          // but no value given yet
                  {
                     TXTT s1 = {0};

                     if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                           "Specify desired system-type as a hexadecimal"
                           "value 00 .. ff",
                           " Specify HEX system-type ", DFSH_PM_CSPINV,
                           TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                           5, s1) != TXDID_CANCEL)
                     {
                        sscanf( s1,  "%X", &(cpp.vstype));
                     }
                  }
                  sprintf( optstr, "0x%2.2x", (cpp.vstype & 0xff));
               }
               else
               {
                  sprintf( optstr, "0x%2.2x", ltype);
               }
               TRACES(( "size: %lf  vpsize: %lf  vstype: %u  type: '%s' offset:%s\n",
                         size,  cpp.vpsize,  ltype, optstr, cpp.vpobuf));

               if (ltype == DFS_P_BOOTMGR)
               {
                  sprintf( create, "cr pri bmgr -c -s:1,c -f:%hu", f->id);
               }
               else if (ltype == DFS_P_EFI_GPT)
               {
                  sprintf( create, "cr pri gpt -f:%hu", f->id);
               }
               else
               {
                  sprintf( create, "cr %s %s -c -f:%hu", (cpp.vrbLog) ? "log" : "pri", optstr, f->id);
               }

               dfsMbrCpGetMaxSize( &cpp);       // calculate max/derived sizes
               if (!cpp.vcbMax)                 // Use MAX size, so do NOT specify!
               {
                  if (ltype == DFS_P_BOOTMGR)
                  {
                     size = 1.0;
                     cpp.vrbLog = FALSE;        // do not allow logical BMGR
                  }
                  else if (ltype == DFS_P_EFI_GPT)
                  {
                     cpp.vrbLog = FALSE;        // do not allow logical GPT
                     cpp.vcbLvm = FALSE;        // no LVM info on GPT guard
                  }
                  else
                  {
                     sectors = (ULONG) TxaParseNumber( cpp.vpsbuf, DFSRADIX_SIZE, &unit);
                     sectors = (ULONG) dfsApplyNumberUnit( sectors, (unit == TXA_DFUNIT) ? 'm' : unit, DFSTORE);
                     if (sectors)
                     {
                        size = (double) (DFSECT2MIB(sectors, dfsGetSectorSize())) + 0.001;
                     }
                  }
                  if ((size <= cpp.fssize) && (size != 0)) // given size acceptable
                  {
                     cpp.vpsize = size;
                  }
                  if (unit == 's')              // size in sectors, use field 'as is'
                  {
                     sprintf( optstr, " -s:%s", cpp.vpsbuf);
                  }
                  else
                  {
                     sprintf( optstr, " -s:%.0lf", (double) (cpp.vpsize - 0.5));
                  }
                  strcat(  create, optstr);
               }

               if (cpp.vcbAln)                  // no alignment wanted
               {
                  strcat( create, " -align-");
               }
               slot = atoi(cpp.vesbuf);
               if ((slot < 4) && (slot != 0))
               {
                  sprintf( optstr, " -S:%d", slot);
                  strcat( create, optstr);
               }
               slot = atoi(cpp.vexbuf);
               if ((slot < 4) && (slot != 3))
               {
                  sprintf( optstr, " -X:%d", slot);
                  strcat( create, optstr);
               }
               slot = atoi(cpp.veybuf);
               if ((slot < 4) && (slot != 1))
               {
                  sprintf( optstr, " -Y:%d", slot);
                  strcat( create, optstr);
               }

               sectors = TxaParseNumber( cpp.vpobuf, DFSRADIX_SIZE, &unit);
               sectors = dfsApplyNumberUnit( sectors, (unit == TXA_DFUNIT) ? 'm' : unit, DFSTORE);
               if (sectors == 0)                // empty or zero
               {
                  if (cpp.vrbEnd)
                  {
                     strcat( create, " -e");
                  }
               }
               else
               {
                  if (cpp.vrbEnd)
                  {
                     strcat( create, " -e:");
                  }
                  else
                  {
                     strcat( create, " -r:");
                  }
                  strcat( create, cpp.vpobuf);
               }
               if (cpp.vcbLvm)
               {
                  strcat( create, " -L");
               }
               if (cpp.vcbClr)
               {
                  strcat( create, " -C");
               }
               if (cpp.vrbPri)
               {
                  if (cpp.vcbAct)
                  {
                     strcat( create, " -F");
                  }
                  if (cpp.vcbMul)
                  {
                     strcat( create, " -M");
                  }
               }
               if (ltype == DFS_P_BOOTMGR)      // just made a BMGR
               {
                  dfspct.selected = 0;          // reset to 1st list item
                  dfspct.top      = 0;          // being automatic-FAT
               }
            }
            rc = TRUE;
         }
         pcBmgr.flags &= ~TXSF_DISABLED;        // enable BMGR again
      }
   }
   BRETURN (rc);
}                                               // end 'txwCreateMbrPartDialog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the MBR-style CreatePart Dialog
/*****************************************************************************/
static ULONG dfsMbrCrPartDlgWinProc             // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);
   if (hwnd != 0)
   {
      DFS_PCMBRDATA   *cpd  = txwQueryWindowPtr(  hwnd, TXQWP_USER);
      TXSELIST        *list = NULL;
      TXS_ITEM        *item = NULL;
      TXWHANDLE        ctrl = NULL;             // handle for a child control window

      switch (msg)
      {
         case TXWM_COMMAND:
            switch ((ULONG) mp2)                // command source
            {
               case TXCMDSRC_RADIOBUTTON:
                  switch ((ULONG) mp1)
                  {
                     case DFSD_WID_CRBLOG:
                     case DFSD_WID_CRBPRI:
                        cpd->vrbPri =  cpd->vrbLog;
                        cpd->vrbLog = !cpd->vrbLog;

                        if (cpd->vrbPri == FALSE)
                        {
                           cpd->vcbAct = FALSE; // no 'set ACTIVE' for logical
                           cpd->vcbMul = FALSE; // no 'Multiple P' for logical
                        }
                        dfsMbrCpGetMaxSize( cpd); // calculate max sizes
                        break;

                     case DFSD_WID_CRBSTA:
                     case DFSD_WID_CRBEND:
                        cpd->vrbSta =  cpd->vrbEnd;
                        cpd->vrbEnd = !cpd->vrbEnd;
                        break;

                     default:
                        break;
                  }
                  txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                  break;

               case TXCMDSRC_CHECKBOX:
                  switch ((ULONG) mp1)
                  {
                     case DFSD_WID_CCBACT:      // ACTIVE checkbox
                        if ((cpd->vcbAct) && (cpd->vrbPri == FALSE))
                        {
                           cpd->vcbAct = FALSE; // no 'set ACTIVE' for logical

                           TxNamedMessage( TRUE, DFSH_PM_CCBACT, " ERROR: Not a primary partition ",
                              "Only PRIMARY partitions can be set to be the ACTIVE one!");

                           txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        }
                        break;

                     case DFSD_WID_CCBMUL:      // Multiple Primaries checkbox
                        if ((cpd->vcbMul) && (cpd->vrbPri == FALSE))
                        {
                           cpd->vcbMul = FALSE; // no 'set ACTIVE' for logical

                           TxNamedMessage( TRUE, DFSH_PM_CCBMUL, " ERROR: Not a primary partition ",
                              "Only PRIMARY partitions affect other primaries visibility!");

                           txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        }
                        break;

                     case DFSD_WID_CCBMAX:
                     case DFSD_WID_CCBALN:
                        dfsMbrCpGetMaxSize( cpd); // calculate max sizes
                        txwInvalidateWindow( hwnd, FALSE, TRUE); // dlg incl controls
                        break;

                     default:
                        rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                        break;
                  }
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         case TXWM_CONTROL:
            if ((list = (TXSELIST *) mp2) != NULL)
            {
               item = list->items[list->selected];
            }
            TRACES(("WM_CONTROL list:%8.8x item:%8.8x, selected %u = 0x%02hx\n",
                    list, item, list->selected, list->items[list->selected]->value));
            if (item != NULL)
            {
               if (TXSH2FROMMP(mp1) == TXLN_SELECT)
               {
                  txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) item->desc, 0);
               }
               else if (TXSH2FROMMP(mp1) == TXLN_ENTER)
               {
                  TXTT s1 = {0};

                  if (item->value == PCTP)      // Partition custom type value
                  {
                     if (cpd->vstype != 0)      // show existing value
                     {
                        sprintf( s1, "0x%2.2x", cpd->vstype);
                     }
                     if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                           "Specify desired system-type as a hexadecimal "
                           "value 00 .. ff",
                           " Specify HEX system-type ", DFSH_PM_CSPINV,
                           TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER | TXPB_HEXNUM_ONLY,
                           7, s1) != TXDID_CANCEL)
                     {
                        sscanf( s1, "%X", &(cpd->vstype));
                     }
                  }
               }
            }
            break;

         case TXWM_CHAR:
            switch ((ULONG) mp2)
            {
               case TXk_F10:                    // popup (list) requested
                  ctrl = txwWindowFromID( hwnd, DFSD_WID_CSPINV);
                  txwPostMsg( ctrl, TXWM_CHAR, 0, (TXWMPARAM) TXk_ENTER); // ENTER on list, popup
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsMbrCrPartDlgWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the CREATE-MBR partition dialog fields
/*****************************************************************************/
static ULONG dfsMbrCreateWinProc               // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if (hwnd != 0)
   {
      TXWHANDLE       owner  = txwQueryWindow( hwnd, TXQW_OWNER); // get dialog info
      DFS_PCMBRDATA  *cpd    = txwQueryWindowPtr(  owner, TXQWP_USER);
      USHORT          wid    = txwQueryWindowUShort( hwnd, TXQWS_ID);
      TXS_ITEM       *item   = NULL;

      switch (msg)
      {
         case TXWM_SETFOCUS:
            item = dfspct.items[dfspct.selected];
            TRACES(( "selected: %u = 0x%02hx\n", dfspct.selected, item->value));
            if ( (item->value  == DFS_P_BOOTMGR) &&
                ((item->flags  &  TXSF_DISABLED) == 0) &&
                ((cpd->vrbPri  == FALSE) || (cpd->vpsize > 1.0)))
            {
               TRACES(("Force BMGR size/type on SETFOCUS, hwnd:%8.8x\n", hwnd));
               cpd->vrbPri = TRUE;              // force PRImary radiobutton ON
               cpd->vrbLog = FALSE;
               cpd->vpsize = 1.0;
               strcpy( cpd->vpsbuf, "1,c");     // update input-field itself

               txwInvalidateWindow( owner, FALSE, TRUE); // dlg incl controls
            }
            if ( (item->value  == DFS_P_EFI_GPT) &&
                ((item->flags  &  TXSF_DISABLED) == 0) &&
                 (cpd->vrbPri  == FALSE))
            {
               TRACES(("Force GPT guard to be PRIMARY\n"));
               cpd->vcbLvm = FALSE;             // no LVM info on GPT guard
               cpd->vrbPri = TRUE;              // force PRImary radiobutton ON
               cpd->vrbLog = FALSE;
               cpd->vpsize = 0.0;
               strcpy( cpd->vpsbuf, "*");       // update input-field itself

               txwInvalidateWindow( owner, FALSE, TRUE); // dlg incl controls
            }
            if ((wid == DFSD_WID_CEOFFS) && (((BOOL) mp1) == FALSE))
            {
               TRACES(("Offset lost focus, value: '%s'\n", cpd->vpobuf));
               dfsMbrCpGetMaxSize( cpd);        // calculate max sizes
               txwInvalidateWindow( owner, FALSE, TRUE); // dlg incl controls
            }
         default:                               // do standard processing too
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsMbrCreateWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate exact maximum partitionsize based on pri/log, position and GEO
/*****************************************************************************/
static void dfsMbrCpGetMaxSize
(
   DFS_PCMBRDATA      *pcd                      // IN    Part Create Data
)
{
   ULONG               ptroom   = 0;            // minimal room for part-tables
   ULONG               offset   = 0;            // specified offset in sectors
   ULONG               sectors  = 0;            // available net size in sectors
   ULONG               size     = 0;            // field size in sectors
   BYTE                s_unit;                  // unit for size (k,m,g,c,h,s)
   BYTE                o_unit;                  // unit for offset value
   ULONG               alignSec;

   ENTER();

   //- get current alignment in sectors
   alignSec = dfsAlignmentSectors( (pcd->vrbLog) ? 'L' : 'P', pcd->fsp);

   if (pcd->fsp->basePsn == 0)                  // at start disk
   {
      if (pcd->vrbLog)                          // logical at start, need MBR room
      {
         ptroom = (pcd->vcbAln) ? 1 : alignSec; // 1 sector or proper alignment
      }
   }
   if ((pcd->vrbLog) || (pcd->fsp->basePsn == 0)) // first track for MBR/EBR tables
   {
      ptroom += dfsDefaultGapSectors( (pcd->vrbLog) ? 'L' : 'P', pcd->fsp); // add GAP
   }

   offset = (ULONG) TxaParseNumber( pcd->vpobuf, DFSRADIX_SIZE, &o_unit);
   offset = (ULONG) dfsApplyNumberUnit( offset, (o_unit == TXA_DFUNIT) ? 'm' : o_unit, DFSTORE);
   if (offset < ptroom)
   {
      offset  = ptroom;
   }
   TRACES(("alignSec: 0x0%x ptroom: 0x0%x  offset: 0x0%x\n", alignSec, ptroom, offset));

   if ((o_unit == 's') || pcd->vcbAln)          // specified in sectors, or No-Align
   {                                            // force SIZE to maximum, in sectors
      if (pcd->fsp->sectors > offset)           // offset within freespace area
      {
         sectors = pcd->fsp->sectors - offset;  // raw size in sectors
      }
   }
   else                                         // no sectors, round to alignment
   {
      if (offset != ptroom)                     // round offset only for an explicit value
      {
         if (!pcd->vcbAln)                      // align only when NOT checked
         {
            offset  = ((offset + alignSec -1) / alignSec) * alignSec;
         }
         if (pcd->vrbLog)                       // any logical partition
         {
            offset += dfsDefaultGapSectors( 'L', pcd->fsp); // add EBR GAP
         }
         sprintf( pcd->vpobuf, "%.0lf", TXSMIB( offset, pcd->fsp->bpsector) - 0.5);
      }

      sectors = pcd->fsp->sectors - offset;
      if ((!pcd->vcbAln) && (sectors > alignSec)) // Align only when NOT checked
      {
         //- to be refined, works correct only when start is aligned ???
         sectors -= (pcd->fsp->sectors % alignSec);
      }
   }
   pcd->fssize = TXSMIB( sectors, pcd->fsp->bpsector);
   pcd->vpsize = pcd->fssize;                   // show max in MiB, 2 decimals not rounded!
   sprintf( pcd->tpsbuf, "Use MAX size: %.2lf MiB", pcd->vpsize);

   //- handle the specified size field contents, adjust to MAX when too large
   size = (ULONG) TxaParseNumber( pcd->vpsbuf, DFSRADIX_SIZE, &s_unit);
   size = (ULONG) dfsApplyNumberUnit( size, (s_unit == TXA_DFUNIT) ? 'm' : s_unit, DFSTORE);

   //- When not specified, or (close to the) MAX for offset alignment, redo the SIZE field
   if ((pcd->vcbMax) || (pcd->vpsbuf[0] == '*') ||
       (size == 0)   || ((size + max( 0x1000, (2*alignSec))) > sectors))
   {
      //- Use 's when specified and NOT at MAX size, or when no-alignment is used
      if (((s_unit == 's') && (size < sectors)) || pcd->vcbAln)
      {
         sprintf( pcd->vpsbuf, "0x%8.8x,s", sectors);   //- new size value, raw
      }
      else
      {
         if (pcd->vcbMax)
         {
            strcpy(  pcd->vpsbuf, "*");
         }
         else
         {
            if (pcd->vpsize > 0.5)
            {
               sprintf( pcd->vpsbuf, "%.0lf", pcd->vpsize - 0.5); // new size, MiB rounded down
            }
            else
            {
               strcpy(  pcd->vpsbuf, "0");
            }
         }
      }
   }
   TRACES(("MaxSize in sectors: 0x0%x, field value: '%s'\n", sectors, pcd->vpsbuf));
   VRETURN();
}                                               // end 'dfsMbrCpGetMaxSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Prompt for (new) MBR-style partition type using custom prompt string
/*****************************************************************************/
BOOL dfsPromptMbrPartitionType
(
   char               *prompt,                  // IN    specific prompt string
   ULONG               helpid,                  // IN    specific help-id
   ULONG              *ptype                    // OUT   type code 00..ff
)
{
   BOOL                rc = TRUE;               // function return
   ULONG               listcode = TXDID_OK;
   TXRECT              where = {18,2,0,0};      // fixed position
   TXTT                s1    = {0};             // type string, empty

   ENTER();

   listcode = txwListBox( TXHWND_DESKTOP, TXHWND_DESKTOP, &where,
                          prompt, "", helpid, TXLB_MOVEABLE,
                          cSchemeColor, cSchemeColor, // std list colors
                          &dfspct);
   switch (listcode)
   {
      case PCTP:
         if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
               "Specify desired system-type as a hexadecimal"
               "value 00 .. ff",
               " Specify HEX system-type ", helpid,
               TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER | TXPB_HEXNUM_ONLY,
               5, s1) != TXDID_CANCEL)
         {
            sscanf( s1, "%X", ptype);
         }
         else
         {
            rc = FALSE;
         }
         break;

      case TXDID_CANCEL:
         rc = FALSE;
         break;

      default:
         *ptype = listcode;
         break;
   }
   BRETURN (rc);
}                                           // end 'dfsPromptMbrPartitionType'
/*---------------------------------------------------------------------------*/

#endif                                          // USEWINDOWING

