//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK LVM Dialog and windowing functions
//
// Author: J. van Wijk
//
// JvW  30-10-2015 Initial version, split off from DFSDFDSK
//


#include <txlib.h>                              // TX library interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK callback and commands
#include <dfsdflvm.h>                           // FDISK LVM dialog functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfswin.h>                             // window and help definitions
#include <dfsosapi.h>                           // OS specific stuff

#if defined (USEWINDOWING)

// LVMSET dialog specific definitions

#define DFSD_WID_LVMDLG     690                 // window id dialog frame
#define DFSD_WID_TEXT       691                 // window id info text
#define DFSD_WID_EVOL       692                 // window id entry volume
#define DFSD_WID_EPAR       693                 // window id entry partition
#define DFSD_WID_ELET       694                 // window id entry driveletter
#define DFSD_WID_CMEN       695                 // window id checkbox BM menu
#define DFSD_WID_CINS       696                 // window id checkbox install
#define DFSD_WID_CSIZ       697                 // window id checkbox size sync
#define DFSD_WID_CSNC       698                 // window id checkbox sync LVM

#define DFSH_LS             (DFSH_FDISK + 580)
#define DFSH_LS_EVOL        (DFSH_LS + 1)
#define DFSH_LS_EPAR        (DFSH_LS + 1)
#define DFSH_LS_ELET        (DFSH_LS + 1)       // same help chapter for all
#define DFSH_LS_CMEN        (DFSH_LS + 1)
#define DFSH_LS_CINS        (DFSH_LS + 1)
#define DFSH_LS_CSIZ        (DFSH_LS + 1)
#define DFSH_LS_CSNC        (DFSH_LS + 1)

#define DFSH_LS_FRAME       (DFSH_LS + 9)       // shown help-on-help (2nd)

#define DFSD_LVMSET_WC          56              // horizontal size in columns
#define DFSD_LVMSET_WL          19              // vertical   size in lines

static char dfsdf_bt_ok[]       = " OK ";
static char dfsdf_bt_cancel[]   = "Cancel";
static char dfsdf_bt_prev[]     = "Prev";
static char dfsdf_bt_next[]     = "Next";

static char  dfsdf_footer[] = "F1=Help F2/5/8=HEX F4=OK PgDn=Next PgUp=Prev  F12=Minimize Alt+F10=Maximize";

static char           *lvmSetHelpText[] =
{
   "",
TXHELPITEM(001,"LVM Dialog fields and buttons")
   " This dialog allows verification and modification of the",
   " LVM 'Drive Letter Assignment Table' or DLAT used by LVM.",
   "",
   " To identify the involved partition, some information is",
   " displayed at the top of the dialog, this includes the label,",
   " filesystem, size of the partition, disk nr and name and the",
   " starting cylinder for the partition.",
   "",
   " LVM information is partition specific, defining:",
   "",
   "  - Volumename, when the partition is part of a volume",
   "    To delete the volume (but not the partition!) specify '-'",
   "",
   "  - Partition name, to identify this partition",
   "    To remove the name (make it anonymous) specify '-'",
   "",
   "  - Driveletter, when the volume is visible for OS/2",
   "    To hide the volume from OS/2 specify '-' or '--'",
   "    You can also specify '*' to let OS/2 assign the first available",
   "    driveletter on attaching the device (most useful on removables)",
   "",
#if defined (DEV32)
   "  - Sync driveletter assignment with live LVM engine",
   "    This will cause DFSee to attempt to communicate any changes",
   "    in driveletters, either assigning a new/different letter or",
   "    hiding the volume, to the LVM engine. This will allow a new",
   "    volume to be accessible without the need to reboot",
   "",
   "    DO NOT USE THIS OPTION WITH REMOVALE MEDIA (like USB disks).",
   "    Due to bugs in the LVM engine, it is quite likely that the",
   "    system will TRAP when doing that, or fail to assign the letter",
   "    It should work without problems with fixed disks.",
   "",
   "    Because of this the option is NOT RECOMMENDED!",
   "",
#endif
   "  - Sync LVM start/size, synchronize the start-sector and size of",
   "    the partition in the LVM info with the partition-tables.",
   "    (This choice is present only when there IS a difference)",
   "",
   "  - On-BM-menu, when the volume is on the boot menu",
   "    Check this to add the volume to the IBM BootManager menu",
   "",
   "  - Installable, volume is target for installation",
   "    Check this to make the volume a target for OS/2 installation",
   "",
   "",
   " For the entry-fields, the CURRENT value is displayed in green,",
   " just below the field when it has the focus",
   "",
   "",
   "",
TXHELPITEM(009,"LVM Dialog window")
   "",
   " The dialog is presented when prompting is enabled (no -P- switch)",
   "",
   " It is used to show the current values of the information, and",
   " the possibly new values specified as options on the LVM command.",
   "",
   " When closing the dialog, the resulting changes will be displayed",
   " and a final confirmation to update the info is presented.",
   "",
   " The dialog ends with the following keys / actions:",
   "",
   "  - <ENTER> End dialog, if changes, ask confirmation and write back",
   "  - <PgDn>  Like ENTER, then display dialog for NEXT partition",
   "  - <PgUp>  Like ENTER, then display dialog for PREVIOUS partition",
   "  - <Esc>   End dialog, do not commit or write back any changes",
   "",
   "  - <F2>    HEX-edit DriveLetter Assignment Table, LVM-info   (DLAT)"
   "  - <F5>    HEX-edit First sector for partition, bootsector    (PBR)",
   "  - <F8>    HEX-edit Last  sector for partition, LVM-signature (BBR)",
   "",
   NULL
};


// Dialog window procedure, for the FDISK LVM NEXT/PREV type Dialog
static ULONG dfsFdskLvmDlgWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);


/*****************************************************************************/
// Initialize help system for the LVM Set Dialog (accessible from startup)
/*****************************************************************************/
void dfsFdskLvmInitHelp
(
   void
)
{
   ENTER();

   txwRegisterHelpText( DFSH_LS, "dfslvmdialog", "LVM Set Dialog help", lvmSetHelpText);

   VRETURN ();
}                                               // end 'dfsFdskLvmInitHelp'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM dialog with default values and allow changes
/*****************************************************************************/
ULONG dfsFdskLvmSetDialog                       // RET   dialog result
(
   FDSK_CB_INFO       *cbp,                     // INOUT LVMSET parameters
   S_LVINF            *lvm,                     // LVM sector
   S_LVPART           *lp                       // LVM part info
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   if (txwIsWindow( TXHWND_DESKTOP))
   {
      TXRECT           deskpos;                 // desktop size/position
      static TXRECT    position = {0,0,0,0};    // reference size/position
      TXWHANDLE        dframe;                  // dialog frame
      TXWHANDLE        dtext;                   // informational text
      TXWHANDLE        drlet;                   //
      TXTM             drlet_v;                 // entryfield driveletter
      TXTM             drlet_o;                 // original driveletter
      TXWHANDLE        dvolu;                   //
      TXTM             dvolu_v;                 // entryfield volume name
      TXTM             dvolu_o;                 // original volume name
      TXWHANDLE        dpart;                   //
      TXTM             dpart_v;                 // entryfield partition name
      TXTM             dpart_o;                 // original partition name
      TXWHANDLE        dmenu;                   //
      BOOL             dmenu_v;                 // checkbox on-bm-menu
      BOOL             dmenu_o;                 // original on-bm-menu
      TXTM             dmenu_t;                 // checkbutton text
      TXWHANDLE        dinst;                   //
      BOOL             dinst_v;                 // checkbox installable
      BOOL             dinst_o;                 // original installable
      TXTM             dinst_t;                 // checkbutton text
      TXWHANDLE        dsize;                   //
      BOOL             dsize_v;                 // checkbox size sync
      #if defined (DEV32)
         TXWHANDLE     dsync;
      #endif
      TXWHANDLE        dbtok;                   // OK button
      TXWHANDLE        dbtcan;                  // CANCEL button
      TXWHANDLE        dbtprev;                 // PREV   button
      TXWHANDLE        dbtnext;                 // NEXT   button
      TXWINDOW         window;                  // setup window data
      ULONG            style;                   // window style
      USHORT           phsize;                  // parent window width
      USHORT           pvsize;                  // parent window height
      short            eflength = LVM_NAME_L-1; // length of entryfields (19!)
      TXLN             title;
      TXTS             mibsize;
      char           **mText;
      int              lines;                   // nr of lines
      int              ll = DFSD_LVMSET_WC - 4; // real max line length
      DFSPARTINFO     *p = (DFSPARTINFO *) cbp->more;
      ULONG            size = p->sectors;

      strcpy(  mibsize, "");
      sprintf( title, "Drive: %2.2s = %8.8s %11.11s  Size : %s\n"
                      "Disk : %2hu = %-*.*s   Cyl : %5u",
                       p->drive, p->fsform, p->plabel,
                       dfstrSizeXiB( mibsize, "", lp->sectors, p->bpsector, ""),
                       p->disknr, LVM_NAME_L, LVM_NAME_L, lvm->DiskName,
                       dfstPsn2Cyl( DFSTORE, lp->partPsn));

      mText = txString2Text( title, &ll, &lines);

      sprintf( title, " SET LVM information for partition: %2.2hu ", p->id);

      sprintf( drlet_o, "%c%c", (lp->letter != 0) ? lp->letter : '-',
                                (lp->letter != 0) ? ':'        : '-');
      sprintf( dvolu_o, "%-*.*s", LVM_NAME_L, LVM_NAME_L, lp->VoluName);
      TxStrip( dvolu_o, dvolu_o, 0, ' ');       // remove trailing spaces
      sprintf( dpart_o, "%-*.*s", LVM_NAME_L, LVM_NAME_L, lp->PartName);
      TxStrip( dpart_o, dpart_o, 0, ' ');       // remove trailing spaces

      strcpy( drlet_v, (strlen(cbp->bname) == 0) ? drlet_o :
                       (cbp->bname[0] != '-') ? cbp->bname : "- hidden -");

      strcpy( dvolu_v, (strlen(cbp->vname) == 0) ? dvolu_o :
                       (cbp->vname[0] != '-') ? cbp->vname : "- none -");
      if ((strlen(lp->VoluName)) && (cbp->vname[0] == '-'))
      {
         strcat( dvolu_v, " (delete)");
      }

      strcpy( dpart_v, (strlen(cbp->pname) == 0) ? dpart_o :
                       (cbp->pname[0] != '-') ? cbp->pname : "- none -");
      if ((strlen(lp->PartName)) && (cbp->pname[0] == '-'))
      {
         strcat( dpart_v, " (delete)");
      }

      dmenu_o = (lp->OnBmMenu != 0);
      dmenu_v = (cbp->ntype != 0xff) ? (cbp->ntype != 0)   : dmenu_o;
      sprintf( dmenu_t, "On BM menu   (%s)", (dmenu_o) ? "on"   : "off");

      dinst_o = (lp->Installable != 0);
      dinst_v = (cbp->stype != 0xff) ? (cbp->stype != 0)   : dinst_o;
      sprintf( dinst_t, "Installable  (%s)", (dinst_o) ? "on"   : "off");

      dsize_v = cbp->cbOption1;

      txwQueryWindowPos( TXHWND_DESKTOP, FALSE, &deskpos);
      phsize = deskpos.right  - deskpos.left;
      pvsize = deskpos.bottom - deskpos.top;

      if (position.right == 0)                  // not initialized yet ?
      {                                         // start with desktop position
         if (deskpos.left + DFSD_LVMSET_WC < phsize)
         {
            position.left   = (phsize - DFSD_LVMSET_WC) / 2;
            position.right  = DFSD_LVMSET_WC;
            if (position.left > 4)
            {
               position.left -= 1;              // a little left from center
            }
         }
         if (DFSD_LVMSET_WL + 4 < pvsize)
         {
            position.top += 2;                  // move dialog a bit down
         }
         if (position.top   + DFSD_LVMSET_WL < pvsize)
         {
            position.bottom = DFSD_LVMSET_WL;
         }
      }
      else
      {
         position.right  = DFSD_LVMSET_WC;      // fixed size of start Dlg
         position.bottom = DFSD_LVMSET_WL;      // (but keeps position)
      }
      TRECTA( "pos/size", (&position));

      style = TXWS_DFRAME | TXWS_CAST_SHADOW | TXCS_CLOSE_BUTTON;
      txwSetupWindowData(
         position.top,                          // upper left corner
         position.left,
         position.bottom,                       // vertical size
         position.right,                        // horizontal size
         style, DFSH_LS_FRAME,                  // style & help
         ' ', ' ',                              // window & border char
         TXWSCHEME_COLORS,                      // std scheme colors
         title, dfsdf_footer,
         &window);
      window.dlgFocusID = DFSD_WID_EVOL;        // volume gets focus
      window.st.buf     = NULL;                 // no artwork attached
      dframe = txwCreateWindow( TXHWND_DESKTOP, TXW_CANVAS, 0, 0, &window,
                                NULL);          // will be set by CreateDlg
      txwSetWindowUShort( dframe, TXQWS_ID, DFSD_WID_LVMDLG);
      txwSetWindowPtr(    dframe, TXQWP_DATA, &position); // for exit position

      //- add child windows to dframe ...

      txwSetupWindowData(
         position.bottom - 18,                  // UL corner, line   (relative)
         1, 2, ll,                              // UL column + vert/hor size
         TXWS_OUTPUT, 0,                        // window frame style + help
         ' ', ' ',                              // window/border chars
         TXWSCHEME_COLORS,                      // std scheme colors
         "", "",                                // title/footer text
         &window);
      window.tv.topline = 0;
      window.tv.leftcol = 0;
      window.tv.maxtop  = TXW_INVALID;
      window.tv.maxcol  = TXW_INVALID;
      window.tv.buf     = mText;
      dtext = txwCreateWindow( dframe, TXW_TEXTVIEW, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dtext, TXQWS_ID, DFSD_WID_TEXT);


      style = TXWS_ENTRYBT | TXWS_HCHILD2SIZE;
      txwSetupWindowData(
         position.bottom - 15,                  // UL corner, line   (relative)
         1, 2, eflength +2,                     // UL column + vert/hor size
         style, DFSH_LS_EVOL,                   // style + help
         ' ', ' ',
         TXWSCHEME_COLORS,                      // std scheme colors
         "Volume name", "",                     // title and footer text
         &window);
      window.ef.leftcol  = 0;
      window.ef.maxcol   = TXW_INVALID;
      window.ef.curpos   = 0;
      window.ef.markCol  = 0;
      window.ef.markSize = 0;
      window.ef.rsize    = eflength;
      window.ef.buf      = dvolu_v;
      window.ef.history  = NULL;
      dvolu = txwCreateWindow( dframe, TXW_ENTRYFIELD, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dvolu, TXQWS_ID, DFSD_WID_EVOL);


      style = TXWS_ENTRYBT | TXWS_HCHILD2SIZE;
      txwSetupWindowData(
         position.bottom - 15,                  // UL corner, line   (relative)
         30, 2, eflength +2,                    // UL column + vert/hor size
         style, DFSH_LS_EPAR,                   // style + help
         ' ', ' ',
         TXWSCHEME_COLORS,                      // std scheme colors
         "Partition name", "",
         &window);
      window.ef.leftcol  = 0;
      window.ef.maxcol   = TXW_INVALID;
      window.ef.curpos   = 0;
      window.ef.markCol  = 0;
      window.ef.markSize = 0;
      window.ef.rsize    = eflength;
      window.ef.buf      = dpart_v;
      window.ef.history  = NULL;
      dpart = txwCreateWindow( dframe, TXW_ENTRYFIELD, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dpart, TXQWS_ID, DFSD_WID_EPAR);


      style = TXWS_ENTRYBT | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 11,                  // UL corner, line   (relative)
         1, 2, 13,                              // UL column + vert/hor size
         style, DFSH_LS_ELET,                   // style + help
         ' ', ' ',
         TXWSCHEME_COLORS,                      // std scheme colors
         "Driveletter", "",
         &window);
      window.ef.leftcol  = 0;
      window.ef.maxcol   = TXW_INVALID;
      window.ef.curpos   = 0;
      window.ef.markCol  = 0;
      window.ef.markSize = 0;
      window.ef.rsize    = 3;
      window.ef.buf      = drlet_v;
      window.ef.history  = NULL;
      drlet = txwCreateWindow( dframe, TXW_ENTRYFIELD, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( drlet, TXQWS_ID, DFSD_WID_ELET);

      if ((lp->sectors != size) || (lp->partPsn != p->basePsn))
      {                                         // only when start/size differs
         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            position.bottom - 12,               // UL corner, line   (relative)
            30, 1, 25,                          // UL column + vert/hor size
            style, DFSH_LS_CSIZ,                // style + help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.bu.text    = "Sync LVM size/start";
         window.bu.checked = &dsize_v;
         dsize = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                                  txwDefWindowProc);
         txwSetWindowUShort( dsize, TXQWS_ID, DFSD_WID_CSIZ);
         txwSetWindowUShort( dsize, TXQWS_GROUP, 1);
      }

      if (dfsa->expertui)
      {
         style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
         txwSetupWindowData(
            position.bottom - 11,               // UL corner, line   (relative)
            30, 1, 25,                          // UL column + vert/hor size
            style, DFSH_LS_CINS,                // style + help
            ' ', ' ', TXWSCHEME_COLORS, "", "",
            &window);
         window.bu.text    = dinst_t;
         window.bu.checked = &dinst_v;
         dinst = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                                  txwDefWindowProc);
         txwSetWindowUShort( dinst, TXQWS_ID, DFSD_WID_CINS);
         txwSetWindowUShort( dinst, TXQWS_GROUP, 1);
      }

      style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 10,                  // UL corner, line   (relative)
         30, 1, 25,                             // UL column + vert/hor size
         style, DFSH_LS_CMEN,                   // style + help
         ' ', ' ', TXWSCHEME_COLORS, "", "",
         &window);
      window.bu.text    = dmenu_t;
      window.bu.checked = &dmenu_v;
      dmenu = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                               txwDefWindowProc);
      txwSetWindowUShort( dmenu, TXQWS_ID, DFSD_WID_CMEN);
      txwSetWindowUShort( dmenu, TXQWS_GROUP, 1);

      #if defined (DEV32)                       // OS2 feature only, LVM sync
         if ((dfsa->expertui) &&                // but only when in Expert mode
             (dfsLvmEnginePresent()))           // and when LVM Engine is there
         {
            style = TXWS_AUTOCHK | TXWS_HCHILD2SIZE | TXWS_HCHILD2MOVE;
            txwSetupWindowData(
               position.bottom - 8,             // UL corner, line   (relative)
               1,  1, 52,                       // UL column + vert/hor size
               style, DFSH_LS_CSNC,             // style + help
               ' ', ' ', TXWSCHEME_COLORS, "", "",
               &window);
            window.bu.text    = "Sync driveletter assignment with live LVM engine";
            window.bu.text    = "Sync driveletter to LVM engine, NOT RECOMMENDED!";
            window.bu.checked = &cbp->lvmSync;
            dsync = txwCreateWindow( dframe, TXW_BUTTON, dframe, 0, &window,
                                     txwDefWindowProc);
            txwSetWindowUShort( dsync, TXQWS_ID, DFSD_WID_CSNC);
         }
      #endif

      if (p->id > 1)                            // is there a previous ?
      {
         style = TXWS_PBUTTON | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            position.bottom - 6,                // UL corner, line   (relative)
            3,                                  // column
            3, strlen( dfsdf_bt_prev) + 4,      // vertical/horizontal size
            style, DFSH_LS,                     // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_prev;
         dbtprev = txwCreateWindow(  dframe, TXW_BUTTON, dframe, 0, &window,
                                     txwDefWindowProc);
         txwSetWindowUShort( dbtprev, TXQWS_ID, DFSDLG_PREV); // rc == ID
      }

      if (p->id < dfsPartitions())              // is there a next ?
      {
         style = TXWS_PBUTTON | TXWS_VCHILD_MOVE;
         txwSetupWindowData(
            position.bottom - 6,                // UL corner, line   (relative)
            13,                                 // column
            3, strlen( dfsdf_bt_next) + 4,      // vertical/horizontal size
            style, DFSH_LS,                     // style and help
            ' ', ' ', TXWSCHEME_COLORS, "",  "",
            &window);
         window.bu.text = dfsdf_bt_next;
         dbtnext = txwCreateWindow(  dframe, TXW_BUTTON, dframe, 0, &window,
                                     txwDefWindowProc);
         txwSetWindowUShort( dbtnext, TXQWS_ID, DFSDLG_NEXT); // rc == ID
      }

      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         position.right - 25,                   // column
         3, strlen( dfsdf_bt_ok) + 4,           // vertical/horizontal size
         style, DFSH_LS,                        // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_ok;
      dbtok = txwCreateWindow(   dframe, TXW_BUTTON, dframe, 0, &window,
                                 txwDefWindowProc);
      txwSetWindowUShort( dbtok, TXQWS_ID, TXMBID_OK); // rc == ID


      style = TXWS_PBUTTON | TXWS_VCHILD_MOVE | TXWS_HCHILD2MOVE;
      txwSetupWindowData(
         position.bottom - 6,                   // UL corner, line   (relative)
         position.right - 15,                   // column
         3, strlen( dfsdf_bt_cancel) + 4,       // vertical/horizontal size
         style, DFSH_LS,                        // style and help
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.bu.text = dfsdf_bt_cancel;
      dbtcan = txwCreateWindow(  dframe, TXW_BUTTON, dframe, 0, &window,
                                 txwDefWindowProc);
      txwSetWindowUShort( dbtcan, TXQWS_ID, TXMBID_CANCEL); // rc == ID

      rc = txwDlgBox( TXHWND_DESKTOP, TXHWND_DESKTOP,
                      dfsFdskLvmDlgWinProc, dframe, NULL);

      if (rc != TXDID_CANCEL)                   // copy changed values back
      {
         if (toupper(drlet_v[0]) != drlet_o[0]) // driveletter change
         {
            cbp->bname[0] = (isalpha(drlet_v[0]) || (drlet_v[0] == '*'))
                                           ? toupper(drlet_v[0]) : '-';
            cbp->cbOptNum2++;
         }
         if (strcmp( dvolu_o, dvolu_v) != 0)    // volumename change
         {
            TxStrip( dvolu_v, dvolu_v, 0, ' '); // strip trailing spaces
            strcpy( cbp->vname, (isgraph(dvolu_v[0])) ? dvolu_v : "-");
            cbp->cbOptNum2++;
         }
         if (strcmp( dpart_o, dpart_v) != 0)    // partition name change
         {
            TxStrip( dpart_v, dpart_v, 0, ' '); // strip trailing spaces
            strcpy( cbp->pname, (isgraph(dpart_v[0])) ? dpart_v : "-");
            cbp->cbOptNum2++;
         }
         if (dmenu_o  !=  dmenu_v)              // on-menu change
         {
            cbp->ntype = (dmenu_v) ? 1 : 0;
            cbp->cbOptNum2++;
         }
         if (dinst_o  !=  dinst_v)              // installable change
         {
            cbp->stype = (dinst_v) ? 1 : 0;
            cbp->cbOptNum2++;
         }
         if ((cbp->cbOption1 = dsize_v) != FALSE) // size synchronize ?
         {
            cbp->cbOptNum2++;                   // count as change
         }
         switch (rc)                            // dialog result
         {
            case DFSDLG_NEXT:                     // request next
            case DFSDLG_PREV:                     // request prev
            case DFSDLG_REQ5:
            case DFSDLG_REQ6:
            case DFSDLG_REQ7:
            case DFSDLG_REQ8:
            case DFSDLG_EDIT:                     // request editor
               cbp->cbOptNum1 = rc;
               break;

            default:
               cbp->cbOptNum1 = DFSDLG_NONE;
               break;
         }
         rc = NO_ERROR;
      }
      else                                      // cancel / escape
      {
         cbp->cbOptNum1 = DFSDLG_NONE;            // stop partition iteration
      }
      txFreeText( mText);                       // free the text memory
   }
   RETURN (rc);
}                                               // end 'dfsFdskLvmSetDialog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dialog window procedure, for the FDISK LVM NEXT/PREV type Dialog
/*****************************************************************************/
static ULONG dfsFdskLvmDlgWinProc                // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc    = NO_ERROR;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);
   if (hwnd != 0)
   {
      switch (msg)
      {
         case TXWM_CHAR:
            switch ((ULONG) mp2)
            {
               case TXk_F3:
                  txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
                  break;

               case TXk_F4:
                  txwDismissDlg( hwnd, DFSDLG_NONE); // save, end dlg
                  break;

               case TXk_PGDN:
                  txwDismissDlg( hwnd, DFSDLG_NEXT); // save, to next table
                  break;

               case TXk_PGUP:
                  txwDismissDlg( hwnd, DFSDLG_PREV); // save, to prev table
                  break;

               case DFSDLG_EDIT:                  // allow several Fkeys for
               case DFSDLG_REQ5:                  // editing, higher level
               case DFSDLG_REQ6:                  // may edit at different
               case DFSDLG_REQ7:                  // sectors for each one
               case DFSDLG_REQ8:
                  txwDismissDlg( hwnd, (ULONG) mp2); // save, start editor
                  break;

               default:
                  rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         default:
            rc = txwDefDlgProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'dfsFdskLvmDlgWinProc'
/*---------------------------------------------------------------------------*/


#endif                                          // USEWINDOWING
